/*
    BundleProjectType.m

    Implementation of the BundleProjectType class for the
    ProjectManager application.

    Copyright (C) 2005  Saso Kiselkov

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#import "BundleProjectType.h"

#import <Foundation/NSString.h>
#import <Foundation/NSBundle.h>
#import <Foundation/NSDictionary.h>
#import <Foundation/NSArray.h>
#import <Foundation/NSSet.h>

#import <AppKit/NSImage.h>

#import "../../ProjectDocument.h"

#import "BundleAttributes.h"

static inline void
InsertMakefileVariable(NSMutableString * makefileString,
                       NSString * name,
                       NSString * value)
{
  [makefileString appendString: [NSString stringWithFormat:
    @"\n"
    @"%@ = %@\n", name, value]];
}

static void
InsertMakefileEnumerationVariable(NSMutableString * makefileString,
                                  NSString * name,
                                  NSArray * values)
{
  NSEnumerator * e;
  NSString * value;

  [makefileString appendString: [NSString stringWithFormat:
    @"\n"
    @"%@ = ", name]];
  e = [values objectEnumerator];
  while ((value = [e nextObject]) != nil)
    {
      [makefileString appendString: [NSString stringWithFormat: @"%@ \\\n",
        value]];
    }
}

/**
 * Returns a string containg `string' prefixed with a '/' character.
 */
static inline NSString *
PreSlash(NSString * string)
{
  return [NSString stringWithFormat: @"/%@", string];
}

static NSString
  * const ClassFilesCategoryName = @"Class Files",
  * const HeaderFilesCategoryName = @"Header Files",
  * const OtherSourceFilesCategoryName = @"Other Source Files",
  * const InterfaceFilesCategoryName = @"Interfaces",
  * const ResourceFilesCategoryName = @"Resource Files",
  * const LocalizedResourceFilesCategoryName = @"Localized Resource Files",
  * const SupportingFilesCategoryName = @"Supporting Files";

@interface BundleProjectType (Private)

+ (void) loadCategoryIcons;

- (BOOL) prepareMakefile;
- (BOOL) prepareInfoFile;

@end

static NSImage * ClassFilesIcon = nil,
               * HeaderFilesIcon = nil,
               * InterfaceFilesIcon = nil,
               * ResourceFilesIcon = nil,
               * LocalizedResourceFilesIcon = nil,
               * SupportingFilesIcon = nil;

@implementation BundleProjectType (Private)

+ (void) loadCategoryIcons
{
  NSBundle * myBundle = [NSBundle bundleForClass: self];

  ClassFilesIcon = [[NSImage alloc] initByReferencingFile:
    [myBundle pathForResource: @"ClassFiles" ofType: @"tiff"]];
  HeaderFilesIcon = [[NSImage alloc] initByReferencingFile:
    [myBundle pathForResource: @"HeaderFiles" ofType: @"tiff"]];
  InterfaceFilesIcon = [[NSImage alloc] initByReferencingFile:
    [myBundle pathForResource: @"InterfaceFiles" ofType: @"tiff"]];
  ResourceFilesIcon = [[NSImage alloc] initByReferencingFile:
    [myBundle pathForResource: @"ResourceFiles" ofType: @"tiff"]];
  LocalizedResourceFilesIcon = [[NSImage alloc] initByReferencingFile:
    [myBundle pathForResource: @"LocalizedResourceFiles" ofType: @"tiff"]];
  SupportingFilesIcon = [[NSImage alloc] initByReferencingFile:
    [myBundle pathForResource: @"SupportingFiles" ofType: @"tiff"]];
}

- (BOOL) prepareMakefile
{
  NSString * buildDir = [owner projectDirectory];
  NSMutableString * makefileString = [NSMutableString string];
  NSString * projectName = [owner projectName];
  NSMutableSet * languages;
  NSMutableSet * localizedResourceFiles;

  [makefileString appendString:
    @"\n"
    @"include $(GNUSTEP_MAKEFILES)/common.make\n"];

  // prepend some definitions to turn on compiler warnings as necessary
  [makefileString appendString:
    @"\n"
    @"ifeq ($(warnings), yes)\n"
    @"ADDITIONAL_OBJCFLAGS += -W\n"
    @"ADDITIONAL_OBJCPPFLAGS += -W\n"
    @"ADDITIONAL_CFLAGS += -W\n"
    @"ADDITIONAL_CPPFLAGS += -W\n"
    @"endif\n"
    @"ifeq ($(allwarnings), yes)\n"
    @"ADDITIONAL_OBJCFLAGS += -Wall\n"
    @"ADDITIONAL_OBJCPPFLAGS += -Wall\n"
    @"ADDITIONAL_CFLAGS += -Wall\n"
    @"ADDITIONAL_CPPFLAGS += -Wall\n"
    @"endif\n"];

  InsertMakefileVariable(makefileString, @"BUNDLE_NAME", projectName);
  if (bundleExtension != nil)
    {
      InsertMakefileVariable(makefileString,
        @"BUNDLE_EXTENSION",
        // automatically prefix it with "."
        [@"." stringByAppendingString: bundleExtension]);
    }

  // OBJC_FILES, OBJCC_FILES and C_FILES
  {
    NSArray * sourceFiles = [[owner allFilesUnderCategory:
      PreSlash(ClassFilesCategoryName)]
      arrayByAddingObjectsFromArray: [owner allFilesUnderCategory:
      PreSlash(OtherSourceFilesCategoryName)]];
    unsigned int n = [sourceFiles count];
    NSMutableArray * objcFiles = [NSMutableArray arrayWithCapacity: n],
                   * objccFiles = [NSMutableArray arrayWithCapacity: n],
                   * cFiles = [NSMutableArray arrayWithCapacity: n];
    NSEnumerator * e;
    NSString * filename;

    // filter out files by extension
    e = [sourceFiles objectEnumerator];
    while ((filename = [e nextObject]) != nil)
      {
        if ([[filename pathExtension] caseInsensitiveCompare: @"m"] ==
          NSOrderedSame)
          {
            [objcFiles addObject: filename];
          }
        else if ([[filename pathExtension] caseInsensitiveCompare: @"mm"] ==
          NSOrderedSame)
          {
            [objccFiles addObject: filename];
          }
        else if ([[filename pathExtension] caseInsensitiveCompare: @"c"] ==
          NSOrderedSame)
          {
            [cFiles addObject: filename];
          }
      }

    InsertMakefileEnumerationVariable(makefileString,
      [NSString stringWithFormat: @"%@_OBJC_FILES", projectName],
      objcFiles);
    InsertMakefileEnumerationVariable(makefileString,
      [NSString stringWithFormat: @"%@_OBJCC_FILES", projectName],
      objccFiles);
    InsertMakefileEnumerationVariable(makefileString,
      [NSString stringWithFormat: @"%@_C_FILES", projectName],
      cFiles);
  }

  // RESOURCE_FILES
  {
    NSArray * files = [owner allFilesUnderCategory:
      PreSlash(ResourceFilesCategoryName)];
    NSMutableArray * resourceFiles = [NSMutableArray arrayWithCapacity:
      [files count]];
    NSEnumerator * e;
    NSString * filename;

    e = [files objectEnumerator];
    while ((filename = [e nextObject]) != nil)
      {
        [resourceFiles addObject: [@"Resources"
          stringByAppendingPathComponent: filename]];
      }

    InsertMakefileEnumerationVariable(makefileString,
      [NSString stringWithFormat: @"%@_RESOURCE_FILES", projectName],
      resourceFiles);
  }

  // LANGUAGES
  languages = [[NSMutableSet new] autorelease];

  [languages addObjectsFromArray: [owner subcategoriesInCategory:
    PreSlash(LocalizedResourceFilesCategoryName)]];
  [languages addObjectsFromArray: [owner subcategoriesInCategory:
    PreSlash(InterfaceFilesCategoryName)]];

  InsertMakefileEnumerationVariable(makefileString,
    [NSString stringWithFormat: @"%@_LANGUAGES", projectName],
    [languages allObjects]);

  // LOCALIZED_RESOURCE_FILES
  localizedResourceFiles = [[NSMutableSet new] autorelease];

  [localizedResourceFiles addObjectsFromArray: [owner allFilesUnderCategory:
    PreSlash(LocalizedResourceFilesCategoryName)]];
  [localizedResourceFiles addObjectsFromArray: [owner allFilesUnderCategory:
    PreSlash(InterfaceFilesCategoryName)]];

  InsertMakefileEnumerationVariable(makefileString,
    [NSString stringWithFormat: @"%@_LOCALIZED_RESOURCE_FILES", projectName],
    [localizedResourceFiles allObjects]);

  if (principalClass != nil)
    {
      InsertMakefileVariable(makefileString,
        [NSString stringWithFormat: @"%@_PRINCIPAL_CLASS", projectName],
        principalClass);
    }

  // SUBPROJECTS
  {
    NSArray * subprojectNames = [[owner subprojects] allKeys];
    NSEnumerator * e = [subprojectNames objectEnumerator];
    NSString * subprojectName;
    NSMutableArray * subprojects = [NSMutableArray arrayWithCapacity:
      [subprojectNames count]];

    while ((subprojectName = [e nextObject]) != nil)
      {
        [subprojects addObject: [[self pathToSubprojectsDirectory]
          stringByAppendingPathComponent: subprojectName]];
      }

    InsertMakefileEnumerationVariable(makefileString,
                                      @"SUBPROJECTS",
                                      subprojects);
  }

  [makefileString appendString:
    @"\n"
    @"-include GNUmakefile.preamble\n"
    @"include $(GNUSTEP_MAKEFILES)/bundle.make\n"
    @"include $(GNUSTEP_MAKEFILES)/aggregate.make\n"
    @"-include GNUmakefile.postamble\n"];

  return [makefileString writeToFile: 
    [buildDir stringByAppendingPathComponent: @"GNUmakefile"]
                          atomically: NO];
}

- (BOOL) prepareInfoFile
{
  NSString * infoPath = [[owner projectDirectory]
    stringByAppendingPathComponent: [[[owner projectName]
    stringByAppendingString: @"Info"]
    stringByAppendingPathExtension: @"plist"]];
  NSMutableDictionary * infoDict;

  // try reading in an already existant <Project>Info.plist file
  infoDict = [[[NSDictionary dictionaryWithContentsOfFile: infoPath]
    mutableCopy] autorelease];
  if (infoDict == nil)
    {
      infoDict = [NSMutableDictionary dictionary];
    }

  if (authors != nil)
    {
      [infoDict setObject: authors forKey: @"Authors"];
    }
  if (bundleDescription != nil)
    {
      [infoDict setObject: bundleDescription forKey: @"Description"];
    }
  if (copyright != nil)
    {
      [infoDict setObject: copyright forKey: @"Copyright"];
    }
  if (copyrightDescription != nil)
    {
      [infoDict setObject: copyrightDescription
                   forKey: @"CopyrightDescription"];
    }

  return [infoDict writeToFile: infoPath atomically: YES];
}

@end

@implementation BundleProjectType

+ (NSString *) projectTypeID
{
  return @"Bundle";
}

+ (NSString *) humanReadableProjectTypeName
{
  return _(@"Bundle");
}

+ (NSString *) projectTypeDescription
{
  return _(@"A bundle of resources and/or code.");
}

+ (NSImage *) projectTypeIcon
{
  return nil;
}

+ (int) projectCapabilities
{
  return FilesProjectCapability |
         BuildProjectCapability |
         SubprojectsProjectCapability;
}

+ (NSDictionary *) projectTemplateDescriptions
{
  NSBundle * myBundle = [NSBundle bundleForClass: self];

  return [NSDictionary dictionaryWithObjectsAndKeys:
    _(@"An empty project with no additional files."), _(@"Empty"),
    nil];
}

+ (NSString *) pathToProjectTemplate: (NSString *) templateName
{
  return [[NSBundle bundleForClass: self]
    pathForResource: @"Empty" ofType: @"template"];
}

- initWithOwner: (ProjectDocument *) anOwner
 infoDictionary: (NSDictionary *) infoDict
{
  if ([self init])
    {
      owner = anOwner;

      ASSIGN(principalClass, [infoDict objectForKey: @"PrincipalClass"]);
      ASSIGN(authors, [NSArray arrayWithArray:
        [infoDict objectForKey: @"Authors"]]);
      ASSIGN(bundleDescription, [infoDict objectForKey: @"BundleDescription"]);
      ASSIGN(copyright, [infoDict objectForKey: @"Copyright"]);
      ASSIGN(copyrightDescription, [infoDict objectForKey:
        @"CopyrightDescription"]);
      ASSIGN(bundleExtension, [infoDict objectForKey: @"BundleExtension"]);

      return self;
    }
  else
    {
      return nil;
    }
}

- (NSDictionary *) infoDictionary
{
  NSMutableDictionary * dict = [NSMutableDictionary dictionary];

  if (principalClass != nil)
    {
      [dict setObject: principalClass forKey: @"PrincipalClass"];
    }
  [dict setObject: authors forKey: @"Authors"];
  if (bundleDescription != nil)
    {
      [dict setObject: bundleDescription forKey: @"BundleDescription"];
    }
  if (copyright != nil)
    {
      [dict setObject: copyright forKey: @"Copyright"];
    }
  if (copyrightDescription != nil)
    {
      [dict setObject: copyrightDescription forKey: @"CopyrightDescription"];
    }
  if (bundleExtension != nil)
    {
      [dict setObject: bundleExtension forKey: @"BundleExtension"];
    }

  return dict;
}

- (void) dealloc
{
  TEST_RELEASE(principalClass);
  TEST_RELEASE(authors);
  TEST_RELEASE(bundleDescription);
  TEST_RELEASE(copyright);
  TEST_RELEASE(copyrightDescription);

  TEST_RELEASE(attributes);

  [super dealloc];
}

- (BOOL) prepareForBuild
{
  if (![self prepareMakefile])
    {
      return NO;
    }

  if (![self prepareInfoFile])
    {
      return NO;
    }

  return YES;
}


- (NSString *) pathToFileTemplatesDirectoryForCategory: (NSString *) category
{
  NSArray * pathComponents = [category pathComponents];

  if ([pathComponents count] <= 1)
    {
      return nil;
    }
  else
    {
      // the name of the templates directory is the same as the category
      // name, but with ' ' substituted for '_' (as gnustep-make doesn't
      // allow for spaces in resource filenames)

      return [[NSBundle bundleForClass: [self class]]
        pathForResource: [[pathComponents objectAtIndex: 1]
        stringByReplacingString: @" " withString: @"_"]
                 ofType: @"templates"];
    }
}

- (BOOL) canAddCategoriesToCategory: (NSString *) category
{
  return YES;
}

- (BOOL) canAddFilesToCategory: (NSString *) category
{
  if ([category isEqualToString: @"/"] ||
    [category isEqualToString: PreSlash(InterfaceFilesCategoryName)] ||
    [category isEqualToString: PreSlash(LocalizedResourceFilesCategoryName)])
    {
      return NO;
    }
  else
    {
      return YES;
    }
}

- (BOOL) canDeleteCategory: (NSString *) category
{
  category = [category stringByStandardizingPath];

  if ([category isEqualToString: PreSlash(ClassFilesCategoryName)] ||
    [category isEqualToString: PreSlash(HeaderFilesCategoryName)] ||
    [category isEqualToString: PreSlash(OtherSourceFilesCategoryName)] ||
    [category isEqualToString: PreSlash(InterfaceFilesCategoryName)] ||
    [category isEqualToString: PreSlash(ResourceFilesCategoryName)] ||
    [category isEqualToString: PreSlash(LocalizedResourceFilesCategoryName)] ||
    [category isEqualToString: PreSlash(SupportingFilesCategoryName)])
    {
      return NO;
    }
  else
    {
      return YES;
    }
}

- (NSString *) pathToFile: (NSString *) fileName
               inCategory: (NSString *) category
{
  NSArray * pathComponents;
  NSString * path;

  category = [category stringByStandardizingPath];

  if ([category hasPrefix: PreSlash(ClassFilesCategoryName)] ||
      [category hasPrefix: PreSlash(HeaderFilesCategoryName)] ||
      [category hasPrefix: PreSlash(OtherSourceFilesCategoryName)] ||
      [category hasPrefix: PreSlash(SupportingFilesCategoryName)])
    {
      path = [owner projectDirectory];
    }
  else if ([category hasPrefix: PreSlash(ResourceFilesCategoryName)])
    {
      path = [[owner projectDirectory]
        stringByAppendingPathComponent: @"Resources"];
    }
  else if ([category hasPrefix: PreSlash(InterfaceFilesCategoryName)] ||
           [category hasPrefix: PreSlash(LocalizedResourceFilesCategoryName)])
    {
      NSArray * pathComponents = [category pathComponents];

        // we need at least 3 path components here - the last one indicates
        // the language
      if ([pathComponents count] >= 3)
        {
          path = [[owner projectDirectory]
            stringByAppendingPathComponent:
            [NSString stringWithFormat: @"%@.lproj", [pathComponents
            objectAtIndex: 2]]];
        }
      else
        {
          path = nil;
        }
    }
  else
    {
      path = [[owner projectDirectory]
        stringByAppendingPathComponent: @"OtherFiles"];
    }

  if (fileName != nil)
    {
      path = [path stringByAppendingPathComponent: fileName];
    }

  return path;
}

- (NSString *) pathToSubprojectsDirectory
{
  return [[owner projectDirectory]
    stringByAppendingPathComponent: @"Subprojects"];
}

- (NSImage *) iconForCategory: (NSString *) category
{
  if (ClassFilesIcon == nil)
    {
      [[self class] loadCategoryIcons];
    }

  category = [category stringByStandardizingPath];

  if ([category isEqualToString: PreSlash(ClassFilesCategoryName)])
    {
      return ClassFilesIcon;
    }
  else if ([category isEqualToString: PreSlash(HeaderFilesCategoryName)])
    {
      return HeaderFilesIcon;
    }
  else if ([category isEqualToString: PreSlash(InterfaceFilesCategoryName)])
    {
      return InterfaceFilesIcon;
    }
  else if ([category isEqualToString: PreSlash(ResourceFilesCategoryName)])
    {
      return ResourceFilesIcon;
    }
  else if ([category isEqualToString:
    PreSlash(LocalizedResourceFilesCategoryName)])
    {
      return LocalizedResourceFilesIcon;
    }
  else if ([category isEqualToString: PreSlash(SupportingFilesCategoryName)])
    {
      return SupportingFilesIcon;
    }
  else
    {
      return nil;
    }
}

- (NSDictionary *) projectAttributeViews
{
  if (attributes == nil)
    {
      attributes = [[BundleAttributes alloc] initWithOwner: self];
    }

  return [NSDictionary dictionaryWithObject: [attributes view]
                                     forKey: _(@"Bundle Info")];
}

- (NSArray *) permissibleFileTypesInCategory: (NSString *) category
{
  if ([category hasPrefix: PreSlash(ClassFilesCategoryName)])
    {
      return [NSArray arrayWithObjects: @"m", @"mm", @"cc", @"cpp", nil];
    }
  else if ([category hasPrefix: PreSlash(OtherSourceFilesCategoryName)])
    {
      return [NSArray arrayWithObjects: @"c", @"m", @"mm", @"cc", @"cpp", nil];
    }
  else if ([category hasPrefix: PreSlash(HeaderFilesCategoryName)])
    {
      return [NSArray arrayWithObject: @"h"];
    }
  else if ([category hasPrefix: PreSlash(InterfaceFilesCategoryName)])
    {
      return [NSArray arrayWithObjects: @"gorm", @"nib", @"gsmarkup",
        @"gmodel", nil];
    }
  else
    {
      return nil;
    }
}

- (NSString *) pathToProjectBinaryOfType: (int) type
{
  return nil;
}

- (NSString *) debuggerType
{
  return nil;
}

- (NSArray *) fixedFrameworks
{
  return nil;
}

- (void) setPrincipalClass: (NSString *) aClass
{
  ASSIGN(principalClass, aClass);
  [owner updateChangeCount: NSChangeDone];
}

- (NSString *) principalClass
{
  return principalClass;
}

- (void) setAuthors: (NSArray *) anArray
{
  ASSIGNCOPY(authors, anArray);
  [owner updateChangeCount: NSChangeDone];
}

- (NSArray *) authors
{
  return authors;
}

- (void) setBundleDescription: (NSString *) aDescription
{
  ASSIGN(bundleDescription, aDescription);
  [owner updateChangeCount: NSChangeDone];
}

- (NSString *) bundleDescription
{
  return bundleDescription;
}

- (void) setCopyright: (NSString *) aCopyright
{
  ASSIGN(copyright, aCopyright);
  [owner updateChangeCount: NSChangeDone];
}

- (NSString *) copyright
{
  return copyright;
}

- (void) setCopyrightDescription: (NSString *) aCopyrightDescription
{
  ASSIGN(copyrightDescription, aCopyrightDescription);
  [owner updateChangeCount: NSChangeDone];
}

- (NSString *) copyrightDescription
{
  return copyrightDescription;
}

- (void) setBundleExtension: (NSString *) anExtension
{
  ASSIGN(bundleExtension, anExtension);
  [owner updateChangeCount: NSChangeDone];
}

- (NSString *) bundleExtension
{
  return bundleExtension;
}

@end
