/*
 * WizardPanelInspector.m
 *
 * Implementation of the WizardPanelInspector class for the
 * WizardPalette palette
 *
 * Copyright (c) 2006, by Saso Kiselkov
 *
 * For license details please see the file COPYING included with this
 * source distribution package.
 */

#import "WizardPanelInspector.h"

#import <Foundation/NSValue.h>
#import <Foundation/NSException.h>
#import <Foundation/NSNotification.h>

#import <AppKit/NSNibLoading.h>
#import <AppKit/NSTableView.h>
#import <AppKit/NSTableColumn.h>
#import <AppKit/NSButtonCell.h>
#import <AppKit/NSForm.h>
#import <AppKit/NSFormCell.h>

#import "GormWizardPanel.h"

@implementation WizardPanelInspector

- init
{
  if ((self = [super init]) != nil)
    {
      NSButtonCell * cell;

      [NSBundle loadNibNamed: [self className] owner: self];

      cell = [[NSButtonCell new] autorelease];
      [cell setButtonType: NSSwitchButton];
      [cell setTarget: self];
      [cell setAction: @selector(changeInitialStage:)];
      [cell setImagePosition: NSImageOnly];

      [[stages tableColumnWithIdentifier: @"Initial"] setDataCell: cell];

      [[NSNotificationCenter defaultCenter]
        addObserver: self
           selector: @selector(titleChanged:)
               name: NSControlTextDidEndEditingNotification
             object: titleForm];
      title = [titleForm cellAtIndex: 0];
    }

  return self;
}

- (NSView*) initialFirstResponder
{
  return titleForm;
}

- (void) addStage: (id)sender
{
  NSString * stageName = _(@"New Stage");
  NSString * oldInitialStage = [[object initialStage] retain];
  NSMutableArray * newStages = [NSMutableArray arrayWithArray:
    [object stages]];
  int i;

  for (i = 1; [newStages containsObject: stageName]; i++)
    {
      stageName = [NSString stringWithFormat: _(@"New Stage %i"), i];
    }

  [newStages addObject: stageName];
  [object setStages: newStages];

  [object setInitialStage: oldInitialStage];
  [oldInitialStage release];

  [stages reloadData];

  [self ok: self];
}


- (void) changeCenter: (id)sender
{
  [object setCentersBeforeActivating: [center state]];

  [self ok: self];
}


- (void) changeClosable: (id)sender
{
  if ([closable state] == YES)
    {
      [object setStyleMask: [object styleMask] | NSClosableWindowMask];
    }
  else
    {
      [object setStyleMask: [object styleMask] & (~NSClosableWindowMask)];
    }

  [self ok: self];
}

- (void) changeInitialStage: sender
{
  int row = [stages clickedRow];

  if (row >= 0)
    {
      [object setInitialStage: [[object stages] objectAtIndex: row]];

      [stages reloadData];
    }

  [self ok: self];
}

- (void) changeModal: (id)sender
{
  [object setRunsInModalSession: [modal state]];

  [self ok: self];
}


- (void) changeResizeBar: (id)sender
{
  if ([resizeBar state] == YES)
    {
      [object setStyleMask: [object styleMask] | NSResizableWindowMask];
    }
  else
    {
      [object setStyleMask: [object styleMask] & (~NSResizableWindowMask)];
    }

  [self ok: self];
}


- (void) changeTitled: (id)sender
{
  if ([resizeBar state] == YES)
    {
      [object setStyleMask: [object styleMask] | NSTitledWindowMask];
    }
  else
    {
      [object setStyleMask: [object styleMask] & (~NSTitledWindowMask)];
    }

  [self ok: self];
}


- (void) removeStage: (id)sender
{
  int rowIndex = [stages selectedRow];

  if (rowIndex >= 0)
    {
      NSMutableArray * newStages = [NSMutableArray arrayWithArray:
        [object stages]];
      NSString * oldInitialStage;

      if ([[object initialStage] isEqualToString: [newStages objectAtIndex:
        rowIndex]])
        {
          oldInitialStage = nil;
        }
      else
        {
          oldInitialStage = [[object initialStage] retain];
        }

      [newStages removeObjectAtIndex: rowIndex];
      [object setStages: newStages];

      [object setInitialStage: oldInitialStage];
      [oldInitialStage release];

      [stages reloadData];

      [self ok: self];
    }
}

- (void) titleChanged: (NSNotification *) notif
{
  [object setTitle: [title stringValue]];

  [self ok: [notif object]];
}

- (int) numberOfRowsInTableView: (NSTableView *)aTableView
{
  return [[object stages] count];
}

- (id) tableView: (NSTableView *)aTableView 
objectValueForTableColumn: (NSTableColumn *)aTableColumn 
             row: (int)rowIndex
{
  NSString * identifier = [aTableColumn identifier];

  if ([identifier isEqualToString: @"Stage"])
    {
      return [[object stages] objectAtIndex: rowIndex];
    }
  else
    {
      NSString * initialStage = [object initialStage];

      if (initialStage == nil)
        {
          if (rowIndex == 0)
            {
              return [NSNumber numberWithBool: YES];
            }
          else
            {
              return [NSNumber numberWithBool: NO];
            }
        }
      else
        {
          if ([[[object stages] objectAtIndex: rowIndex] isEqualToString:
            initialStage])
            {
              return [NSNumber numberWithBool: YES];
            }
          else
            {
              return [NSNumber numberWithBool: NO];
            }
        }
    }
}

- (void) tableView: (NSTableView *)aTableView 
    setObjectValue: (id)anObject 
    forTableColumn: (NSTableColumn *)aTableColumn
               row: (int)rowIndex
{
  NSMutableArray * newStages = [NSMutableArray arrayWithArray:
    [object stages]];
  NSString * oldInitialStage;

  if ([newStages containsObject: anObject])
    {
      NSRunAlertPanel(_(@"Stage already present"),
        _(@"Cannot set stage name \"%@\" - such a stage\n"
          @"already is present, and stage names must be unique."),
        nil, nil, nil, anObject);

      [stages reloadData];

      return;
    }

  if ([[object initialStage] isEqualToString: [newStages
    objectAtIndex: rowIndex]])
    {
      oldInitialStage = nil;
    }
  else
    {
      oldInitialStage = [[object initialStage] retain];
    }

  [newStages replaceObjectAtIndex: rowIndex withObject: anObject];
  [object setStages: newStages];

  [object setInitialStage: oldInitialStage];
  [oldInitialStage release];

  [stages reloadData];

  [self ok: self];
}

- (void) setObject: anObject
{
  unsigned int styleMask = [anObject styleMask];

  [super setObject: anObject];

  [title setStringValue: [object title]];

  [closable setState: (styleMask & NSClosableWindowMask) > 0];
  [resizeBar setState: (styleMask & NSResizableWindowMask) > 0];
  [titled setState: (styleMask & NSTitledWindowMask) > 0];
  [center setState: [object centersBeforeActivating]];

  [modal setState: [anObject runsInModalSession]];
}

@end
