/***************************************************************************
		   gui_playlist.cpp  -  description
			 -------------------
begin                : Thu Mar 7 2002
copyright            : (C) 2001 by Holger Sattel
email                : hsattel@rumms.uni-mannheim.de

This file contributed by Tim Lee ;)
****************************************************************************/

/***************************************************************************
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
**************************************************************************/

#include "gui_playlist.h"
#include "gui.h"
#include "jobmanager.h"
#include "job_modifyplaylists.h"
#include "job_modifyplaylist_tracks.h"
#include "job_queryplaylist_tracks.h"
#include "job_querydatabase.h"
#include "job_generateplaylist.h"
#include "pixmapcache.h"

#ifdef HAVE_MEXTRAS
#include "job_callmusicextras.h"
#include "job_getextradata.h"
#endif /* HAVE_MEXTRAS */

#include "job_importplaylist.h"
#include "database.h"

#ifdef HAVE_MEXTRAS
#include "musicextraswrapper.h"
#include "mextras_log.h"
#endif /* HAVE_MEXTRAS */

#ifdef HAVE_MIXXX
#include "mixxxclient.h"
#endif /* HAVE_MIXXX */

#include <qlayout.h>
#include <qframe.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qcombobox.h>
#include <qgroupbox.h>
#include <qpushbutton.h>
#include <qptrlist.h>
#include <qlistview.h>
#include <qpopupmenu.h>
#include <qmessagebox.h>
#include <qinputdialog.h>
#include <qtooltip.h>
#include <qiconview.h>
#include <qlistview.h>
#include <qpalette.h>
#include <qfont.h>
#include <qfiledialog.h>
#include <qurloperator.h>
#include <qtabdialog.h>

#include <iostream>
#include <prokyon.h>

#include <assert.h>

GUI_Playlist::GUI_Playlist(QWidget *parent, const char *name)
  : QTabWidget(parent, name), firsttime( true ) 
#ifdef HAVE_MEXTRAS
    ,coverIconViewItem(0), artistIconViewItem(0), 
    last_coverID(0), last_artistID(0), last_bkgID(0)
#endif /* HAVE_MEXTRAS */
{

  vsplit = new QSplitter(QSplitter::Vertical, this);
  vsplit->setOpaqueResize( true );

  pl_selframe = new QFrame( vsplit );
  QVBoxLayout *pl_selblay = new QVBoxLayout(pl_selframe);

  playlist_list = new ExtendedDropListView(pl_selframe);
  QString temps = _( "Change playlists\nRight click to play, burn CD or export.");
  QToolTip::add( playlist_list, temps );
  QToolTip::add( playlist_list->viewport(), temps );
  playlist_list->addColumn(pixmapcache->getFadeIconSet("lvi_playlist.png"), _("Playlists"),220);
  playlist_list->setItemMargin(1);
  playlist_list->setSelectionMode(QListView::Single);
  playlist_list->setSorting(0);
  playlist_list->setHScrollBarMode(QScrollView::AlwaysOff);
  playlist_list->setColumnWidthMode(0,QListView::Manual);
  playlist_list->setShowToolTips(true);
  playlist_list->setAcceptDrops( true );
  playlist_list->setShowSortIndicator(true);

  connect(playlist_list, SIGNAL(selectionChanged(QListViewItem*)),
          this, SLOT(slot_playlistSelectionChanged(QListViewItem*)));
  connect(playlist_list, SIGNAL(doubleClicked(QListViewItem*)),
          this, SLOT(slot_playlistDoubleClicked(QListViewItem*)));
  connect(playlist_list, SIGNAL(contextMenuRequested(QListViewItem*, const QPoint&, int)),
          this, SLOT(slot_contextMenu(QListViewItem*, const QPoint&, int)));

  pl_selblay->addWidget(playlist_list,1);

  QFrame *pl_butframe = new QFrame(pl_selframe);
  QGridLayout *pl_butgrid = new QGridLayout(pl_butframe, 1, 2);
  buttonNew = new QPushButton(pixmapcache->getFadeIconSet("filenew.png"), _("New"), pl_butframe);
  buttonNew->setMinimumWidth(20);
  pl_butgrid->addWidget(buttonNew, 0, 0);
  buttonImport = new QPushButton(pixmapcache->getFadeIconSet("fileimport.png"), _("Import"), pl_butframe);
  buttonImport->setMinimumWidth(20);
  pl_butgrid->addWidget(buttonImport, 0, 1);

  connect(buttonNew, SIGNAL(clicked()), this, SLOT(slot_newButton()));
  connect(buttonImport, SIGNAL(clicked()), this, SLOT(slot_importPlaylist()));

  pl_selblay->addWidget(pl_butframe);

  QFrame *pl_selframe_low = new QFrame(vsplit);
  QVBoxLayout *pl_selblay_low = new QVBoxLayout(pl_selframe_low);

  tracklist = new ExtendedDropListView(pl_selframe_low);
  //2 lines required even if  HAVE_MIXXX in undefined
  int mixxxStatusColumn = tracklist->addColumn("", -1);
  tracklist->hideColumn(mixxxStatusColumn);
  //
  tracklist->addColumn(pixmapcache->getFadeIconSet("stock_music-library.png"), _("Tracks in Playlist"),220);
  tracklist->setItemMargin(0);
  tracklist->setSelectionMode(QListView::Extended);
  tracklist->setSorting(-1,true);
  tracklist->setHScrollBarMode(QScrollView::AlwaysOff);
  tracklist->setColumnWidthMode(0,QListView::Manual);
  tracklist->setShowToolTips(true);
  tracklist->setAcceptDrops( true );
  tracklist->setAllColumnsShowFocus(true);

  connect(tracklist, SIGNAL(contextMenuRequested(QListViewItem*, const QPoint&, int)),
          this, SLOT(slot_contextMenu(QListViewItem*, const QPoint&, int)));
  connect(tracklist, SIGNAL(selectionChanged()),
	  this, SLOT(slot_trackSelectionChanged()));
  connect(tracklist, SIGNAL(doubleClicked(QListViewItem*)),
	  this, SLOT(slot_playlistTrackDoubleClicked(QListViewItem*)));

  pl_selblay_low->addWidget(tracklist);

  QFrame *pl_titframe = new QFrame(pl_selframe_low);
  QGridLayout *pl_titgrid = new QGridLayout(pl_titframe, 1, 3);
  QPushButton *buttonUp = new QPushButton(pixmapcache->getFadeIconSet("up.png"), _("Up"), pl_titframe);
  buttonUp->setMinimumWidth(20);
  pl_titgrid->addWidget(buttonUp, 0, 0);
  QPushButton *buttonDown = new QPushButton(pixmapcache->getFadeIconSet("down.png"), _("Down"), pl_titframe);
  buttonDown->setMinimumWidth(20);
  pl_titgrid->addWidget(buttonDown, 0, 1);
  QPushButton *buttonSave = new QPushButton(pixmapcache->getFadeIconSet("filesave.png"), _("Save"), pl_titframe);
  buttonSave->setMinimumWidth(20);
  pl_titgrid->addWidget(buttonSave, 0, 2);

  connect(buttonUp, SIGNAL(clicked()), this, SLOT(slot_movePlaylistTrackUp()));
  connect(buttonDown, SIGNAL(clicked()), this, SLOT(slot_movePlaylistTrackDown()));
  connect(buttonSave, SIGNAL(clicked()), this, SLOT(slot_savePlaylistTracks()));

  pl_selblay_low->addWidget(pl_titframe);

  QFrame *pl_infoframe = new QFrame(pl_selframe_low);
  QGridLayout *pl_infogrid = new QGridLayout(pl_infoframe, 1, 6);
  /* TRANSLATORS: Tr = short form for tracks */
  QLabel *pl_trackslab = new QLabel(QString(_("Tr")) + ": ", pl_infoframe);
  pl_infogrid->addWidget(pl_trackslab, 0, 0);
  pl_tracks = new QLabel("0", pl_infoframe);
  pl_infogrid->addWidget(pl_tracks, 0, 1);
  /* TRANSLATORS: Ti = short form for Playtime */
  QLabel *pl_timelab = new QLabel(QString(_(" Ti")) + ": ", pl_infoframe);
  pl_infogrid->addWidget(pl_timelab, 0, 2);
  pl_time = new QLabel("0", pl_infoframe);
  pl_infogrid->addWidget(pl_time, 0, 3);
  /* TRANSLATORS: Sz = short form for size */
  QLabel *pl_sizelab = new QLabel(QString(_(" Sz")) + ": ", pl_infoframe);
  pl_infogrid->addWidget(pl_sizelab, 0, 4);
  pl_size = new QLabel("0 MB", pl_infoframe);
  pl_infogrid->addWidget(pl_size, 0, 5);
  
  pl_selblay_low->addWidget(pl_infoframe);
  
  current_tracklist = new QListView();
  current_tracklist->addColumn("#",-1);
  current_tracklist->addColumn(_("Artist"),120);
  current_tracklist->addColumn(_("Title"),160);
  current_tracklist->setAllColumnsShowFocus(true);
  current_tracklist->setSelectionMode(QListView::Extended);
  current_tracklist->setSorting(-1,true);

  connect(current_tracklist, SIGNAL(contextMenuRequested(QListViewItem*, const QPoint&, int)),
	  this, SLOT(slot_contextMenu(QListViewItem*, const QPoint&, int)));
  connect(current_tracklist, SIGNAL(selectionChanged()),
	  this, SLOT(slot_trackSelectionChanged()));

  tabGenerator = new QFrame(this);
  QToolTip::add( tabGenerator, _( "Generate playlists by randomily picking tracks,\n within user-defined constraints" ) );
  
  QGridLayout *generatorLay = new QGridLayout(tabGenerator, 8, 2);
  generatorLay->setRowStretch(7,1);
  
  QLabel *labelMinTrackLength = new QLabel(QString(_("Min Track Length")) + ": ", tabGenerator);
  labelMinTrackLength->setTextFormat( Qt::RichText );
  generatorLay->addWidget(labelMinTrackLength, 0, 0);
  generatorSpinMinTrackLength = new QSpinBox(0, 600, 10, tabGenerator);
  generatorSpinMinTrackLength->setSizePolicy(QSizePolicy( QSizePolicy::Maximum, QSizePolicy::Preferred));
  generatorSpinMinTrackLength->setButtonSymbols(QSpinBox::PlusMinus);
  generatorLay->addWidget(generatorSpinMinTrackLength, 0, 1);

  QLabel *labelMaxTrackLength = new QLabel(QString(_("Max Track Length")) + ": ", tabGenerator);
  labelMaxTrackLength->setTextFormat( Qt::RichText );
  generatorLay->addWidget(labelMaxTrackLength, 1, 0);
  generatorSpinMaxTrackLength = new QSpinBox(0, 600, 10, tabGenerator);
  generatorSpinMaxTrackLength->setSizePolicy(QSizePolicy( QSizePolicy::Maximum, QSizePolicy::Preferred));
  generatorSpinMaxTrackLength->setButtonSymbols(QSpinBox::PlusMinus);
  generatorLay->addWidget(generatorSpinMaxTrackLength, 1, 1);

  /******************************** work in progress....
  QLabel *labelArtists = new QLabel(QString(_("Artists")) + ": ", tabGenerator);
  generatorLay->addWidget(labelArtists, row, 0);
  generatorComboArtists = new QComboBox(false, tabGenerator);
  generatorLay->addWidget(generatorComboArtists, row++, 1);
  *********************************/

  QLabel *labTracks = new QLabel(QString(_("Number of Tracks")) + ": ", tabGenerator);
  labTracks->setTextFormat( Qt::RichText );
  generatorLay->addWidget(labTracks, 2, 0);
  generatorSpinTracks = new QSpinBox(10, 100, 10, tabGenerator);
  generatorSpinTracks->setSizePolicy(QSizePolicy( QSizePolicy::Maximum, QSizePolicy::Preferred));
  generatorSpinTracks->setButtonSymbols(QSpinBox::PlusMinus);
  generatorLay->addWidget(generatorSpinTracks, 2, 1);

  QLabel *labRating = new QLabel(QString(_("Min Rating")) + ": ", tabGenerator);
  labRating->setTextFormat( Qt::RichText );
  generatorLay->addWidget(labRating, 3, 0);
  generatorSpinRating = new QSpinBox(0, 5, 1, tabGenerator);
  generatorSpinRating->setSizePolicy(QSizePolicy( QSizePolicy::Maximum, QSizePolicy::Preferred));
  generatorSpinRating->setButtonSymbols(QSpinBox::PlusMinus);
  generatorLay->addWidget(generatorSpinRating, 3, 1);

  QLabel *labFavouriteArtists = new QLabel(QString(_("Only fav. Artists")) + ": ", tabGenerator);
  labFavouriteArtists->setTextFormat( Qt::RichText );
  generatorLay->addWidget(labFavouriteArtists, 4, 0);
  generatorCheckBoxOnlyFavouriteArtists = new QCheckBox(tabGenerator);    
  generatorLay->addWidget(generatorCheckBoxOnlyFavouriteArtists, 4, 1);

  QLabel *labMedium = new QLabel(QString(_("Only tracks on hardisk")) + ": ", tabGenerator);
  labMedium->setTextFormat( Qt::RichText );
  generatorLay->addWidget(labMedium, 5, 0);
  generatorCheckBoxOnlyHarddisk = new QCheckBox(tabGenerator);
  generatorCheckBoxOnlyHarddisk->setChecked(true);
  generatorLay->addWidget(generatorCheckBoxOnlyHarddisk, 5, 1);

  QPushButton *buttonReset = new QPushButton(_("Reset"), tabGenerator);
  generatorLay->addWidget(buttonReset, 6, 0);
  buttonReset->setMinimumWidth(20);

  QPushButton *buttonGenerate = new QPushButton(_("Go"), tabGenerator);
  generatorLay->addWidget(buttonGenerate, 6, 1);
  buttonGenerate->setMinimumWidth(20);

  QLabel *labLogo = new QLabel(tabGenerator);
  labLogo->setPixmap(pixmapcache->getFaded("prokyon_logo.png"));
  //labLogo->setScaledContents(true);
  labLogo->setAlignment(Qt::AlignRight + Qt::AlignBottom);
  generatorLay->addMultiCellWidget(labLogo, 7, 7, 0, 1);

  slot_resetButton();
  connect(generatorSpinMinTrackLength, SIGNAL(valueChanged(int)), this, SLOT(slot_generatorMinTrackLength(int)));
  connect(generatorSpinMaxTrackLength, SIGNAL(valueChanged(int)), this, SLOT(slot_generatorMaxTrackLength(int)));
  connect(buttonReset, SIGNAL(clicked()), this, SLOT(slot_resetButton()));
  connect(buttonGenerate, SIGNAL(clicked()), this, SLOT(slot_generateButton()));

#ifdef HAVE_MEXTRAS
  extrasSplit = new QSplitter(QSplitter::Vertical, this);
  extrasSplit->setOpaqueResize( true );

  QFrame *imageFrame = new QFrame( extrasSplit );
  QVBoxLayout *imageFrameLayout = new QVBoxLayout(imageFrame);

  images = new QIconView( imageFrame );
  images->setAutoArrange(true);
  images->setArrangement(QIconView::TopToBottom);
  images->setItemsMovable(false);
  QToolTip::add( images->viewport(), _( "Display artist and cover pictures retrieved by Musicextras" ) );
  
  QPushButton *buttonRefreshFromInternet = new QPushButton(pixmapcache->getFadeIconSet("html.png"), _("WWW Scan"), imageFrame);
  connect(buttonRefreshFromInternet, SIGNAL(clicked()), this, SLOT(slot_refreshFromInternetButton()));
  QToolTip::add( buttonRefreshFromInternet, _("Scan WWW, using Musicextras, for images, lyrics... ") );

  buttonAbortRefreshFromInternet = new QPushButton(pixmapcache->getFadeIconSet("action_discard.png"), _("Abort"), imageFrame);
  connect(buttonAbortRefreshFromInternet, SIGNAL(clicked()), this, SLOT(slot_abortRefreshFromInternetButton()));
  QToolTip::add( buttonAbortRefreshFromInternet, _("Abort WWW Scan ") );
  buttonAbortRefreshFromInternet->setEnabled(false);
  imageFrameLayout->addWidget(images);

  QHBoxLayout *buttonLayout = new QHBoxLayout(imageFrameLayout);
  buttonLayout->addWidget(buttonRefreshFromInternet, 3);
  buttonLayout->addWidget(buttonAbortRefreshFromInternet, 1);

  images->show();

  tabmex = new QTabWidget(extrasSplit);
  
  QFrame *tabLyrics = new QFrame(extrasSplit);
  QGridLayout *lyricsLay = new QGridLayout(tabLyrics, 3, 2);
  QLabel *artistLabelLyrics = new QLabel(QString(_("Artist")) + ": " , tabLyrics);
  artistValueLyrics = new QLabel(_("Unknown") , tabLyrics);
  artistValueLyrics->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
  QLabel *titleLabelLyrics = new QLabel(QString(_("Title")) + ": " , tabLyrics);
  titleValueLyrics = new QLabel(_("Unknown") , tabLyrics);
  titleValueLyrics->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
  lyrics = new QTextEdit(tabLyrics);
  QToolTip::add(lyrics, _( "Displays track lyrics\nSingle left click to enlarge window") );
  lyrics->setReadOnly(true);
  lyrics->setWordWrap(QTextEdit::NoWrap);
  lyrics->setText(_("Lyrics will be displayed here"));
  connect( lyrics, SIGNAL(clicked(int,int)), this, SLOT(slot_ClickLyrics()));
  lyricsLay->addWidget(artistLabelLyrics, 0, 0);
  lyricsLay->addWidget(artistValueLyrics, 0, 1);
  lyricsLay->addWidget(titleLabelLyrics, 1, 0);
  lyricsLay->addWidget(titleValueLyrics, 1, 1);
  lyricsLay->addMultiCellWidget(lyrics, 2, 2, 0, 1);

  QFrame *tabReview = new QFrame(extrasSplit);
  QGridLayout *reviewLay = new QGridLayout(tabReview, 3, 2);
  QLabel *artistLabelReview = new QLabel(QString(_("Artist")) + ": " , tabReview);
  artistValueReview = new QLabel(_("Unknown") , tabReview);
  artistValueReview->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
  QLabel *albumLabelReview = new QLabel(QString(_("Album")) + ": " , tabReview);
  albumValueReview = new QLabel(_("Unknown") , tabReview);
  albumValueReview->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
  review = new QTextEdit(tabReview);
  QToolTip::add( review, _( "Displays album review\nSingle left click to enlarge window") );
  review->setReadOnly(true);
  review->setWordWrap(QTextEdit::WidgetWidth);
  review->setText(_("Album review will be displayed here"));
  connect( review, SIGNAL(clicked(int,int)), this, SLOT(slot_ClickReview()));
  reviewLay->addWidget(artistLabelReview, 0, 0);
  reviewLay->addWidget(artistValueReview, 0, 1);
  reviewLay->addWidget(albumLabelReview, 1, 0);
  reviewLay->addWidget(albumValueReview, 1, 1);
  reviewLay->addMultiCellWidget(review, 2, 2, 0, 1);

  QFrame *tabTracks = new QFrame(extrasSplit);
  QGridLayout *tracksLay = new QGridLayout(tabTracks, 3, 2);
  QLabel *artistLabelTracks = new QLabel(QString(_("Artist")) + ": " , tabTracks);
  artistValueTracks = new QLabel(_("Unknown") , tabTracks);
  artistValueTracks->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
  QLabel *albumLabelTracks = new QLabel(QString(_("Album")) + ": " , tabTracks);
  albumValueTracks = new QLabel(_("Unknown") , tabTracks);
  albumValueTracks->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
  tracks = new QTextEdit(tabTracks);
  QToolTip::add( tracks, _( "Displays album tracks\nSingle left click to enlarge window") );
  tracks->setReadOnly(true);
  tracks->setWordWrap(QTextEdit::WidgetWidth);
  tracks->setText(_("Album tracks will be displayed here"));
  connect( tracks, SIGNAL(clicked(int,int)), this, SLOT(slot_ClickTracks()));
  tracksLay->addWidget(artistLabelTracks, 0, 0);
  tracksLay->addWidget(artistValueTracks, 0, 1);
  tracksLay->addWidget(albumLabelTracks, 1, 0);
  tracksLay->addWidget(albumValueTracks, 1, 1);
  tracksLay->addMultiCellWidget(tracks, 2, 2, 0, 1);

  QFrame *tabBiography = new QFrame(extrasSplit);
  QGridLayout *biographyLay = new QGridLayout(tabBiography, 4, 2);
  QLabel *artistLabelBiography = new QLabel(QString(_("Artist")) + ": ", tabBiography);
  artistValueBiography = new QLabel(_("Unknown") , tabBiography);
  artistValueBiography->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
  QLabel *yearsActiveLabelBiography = new QLabel(QString(_("Years active")) + ": " , tabBiography);
  yearsActiveValueBiography = new QLabel(_("Unknown") , tabBiography);
  yearsActiveValueBiography->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
  biography = new QTextEdit(tabBiography);
  QToolTip::add( biography, _( "Displays artist's biography\nSingle left click to enlarge window") );
  biography->setReadOnly(true);
  biography->setWordWrap(QTextEdit::WidgetWidth);
  biography->setText(_("Biography will be displayed here"));
  connect( biography, SIGNAL(clicked(int,int)), this, SLOT(slot_ClickBiography()));
  biographyLay->addWidget(artistLabelBiography, 0, 0);
  biographyLay->addWidget(artistValueBiography, 0, 1);
  biographyLay->addWidget(yearsActiveLabelBiography, 1, 0);
  biographyLay->addWidget(yearsActiveValueBiography, 1, 1);
  biographyLay->addMultiCellWidget(biography, 2, 2, 0, 1);
  
  mexlog = new QTextEdit(this);
  QToolTip::add( mexlog, _( "Displays musicextras log\nSingle left click to enlarge window") );
  mexlog->setTextFormat(LogText);
  QFont f( "Sans", 7);
  mexlog->setFont( f );
  connect( mexlog, SIGNAL(clicked(int,int)), this, SLOT(slot_ClickMexlog()));

  tabmex->addTab(tabLyrics, _("Lyrics"));
  tabmex->addTab(tabReview, _("Album Review"));
  tabmex->addTab(tabTracks, _("Album Tracks"));
  tabmex->addTab(tabBiography, _("Biography"));
  tabmex->addTab(mexlog, _("Log"));

  addTab(extrasSplit, _("Extras"));
#endif /* HAVE_MEXTRAS */

  addTab(vsplit, _("Playlists"));
  addTab(tabGenerator, _("Generator"));

  connect(this, SIGNAL(currentChanged(QWidget*)), this, SLOT(slot_tabChange(QWidget*)));

  displayed_playlist = NULL;
  fileCopyInProgress = false;
  numberOfFilesToCopy = 0;
}

GUI_Playlist::~GUI_Playlist()
{
}

void GUI_Playlist::slot_setConnectionState(bool state)
{
  isConnected = state;
  if(!isConnected) clear();
}

#ifdef HAVE_MIXXX
void GUI_Playlist::setMixxxConnectionState(bool state)
{
  isMixxxConnected = state;
}
#endif /* HAVE_MIXXX */

#ifdef HAVE_MEXTRAS
void GUI_Playlist::setArtistPixmap(const QIDPixmap &pixmap)
{
  if ( pixmap.ID != last_artistID) {
    //    qWarning( "artist_ID from:%i to:%i",last_artistID,pixmap.ID);
    last_artistID = pixmap.ID;

    if (artistIconViewItem) {
      delete artistIconViewItem;
      artistIconViewItem=0;
    } 
  
    if (!pixmap.image.isNull()) {
      artistIconViewItem = new QIconViewItem( images, "Artist");
      artistIconViewItem->setPixmap(pixmap.image);
    }
  }
}

void GUI_Playlist::setCoverPixmap(const QIDPixmap &pixmap)
{
  if ( pixmap.ID != last_coverID) {
    //    qWarning( "Cover_ID from:%i to:%i",last_coverID,pixmap.ID);
    last_coverID = pixmap.ID;

    if (coverIconViewItem) {
      delete coverIconViewItem;
      coverIconViewItem=0;
    } 
  
    if (!pixmap.image.isNull()) {
      coverIconViewItem = new QIconViewItem( images, "Cover");
      coverIconViewItem->setPixmap(pixmap.image);
    }
  }
}

void GUI_Playlist::setLyrics(const QString &text)
{
  lyrics->setText(text);
}

void GUI_Playlist::setYearsActive(const QString &text)
{
  yearsActiveValueBiography->setText(text);
}

void GUI_Playlist::setArtist(const QString &text)
{
  artistValueLyrics->setText(text);
  artistValueReview->setText(text);
  artistValueTracks->setText(text);
  artistValueBiography->setText(text);
}

void GUI_Playlist::setAlbum(const QString &text)
{
  albumValueReview->setText(text);
  albumValueTracks->setText(text);
}

void GUI_Playlist::setTitle(const QString &text)
{
  titleValueLyrics->setText(text);
}

void GUI_Playlist::setAlbumReview(const QString &text)
{
  review->setText(text);
}

void GUI_Playlist::setAlbumTracks(const QString &text)
{
  tracks->setText(text);
}

void GUI_Playlist::setBiography(const QString &text)
{
  biography->setText(text);
}
#endif /* HAVE_MEXTRAS */

void GUI_Playlist::notifyNewPlaylistBasis(QList<PLAYLIST> *play_list)
{
  if(!isConnected) return;

  playlist_list->clear();

  for(PLAYLIST *curr = play_list->first();curr != 0; curr = play_list->next())
    notifyNewPlaylist(curr, false);
}

void GUI_Playlist::notifyNewPlaylistTracksBasis(QList<PLAYLIST_TRACK> *tracks_list, int playlistID)
{
  if(!isConnected) return;
  if(playlistID < 0) return;

  if(playlistID == 0) {
    current_tracklist->clear();
    mapCurrent.clear();
  }
  else {
    tracklist->clear();
    mapEditor.clear();
  }
  selectedItems.clear();
 
  for(PLAYLIST_TRACK *curr = tracks_list->first(); curr != 0; curr = tracks_list->next())
    notifyNewPlaylistTrack(curr,playlistID);
}

void GUI_Playlist::notifyNewPlaylist(PLAYLIST *playlist, bool isNewCreated)
{
  if(!isConnected) return;

  LVI_Playlist *item;
  item = new LVI_Playlist(playlist_list, playlist);
  item->setText(0,playlist->name);
  if(isNewCreated) playlist_list->setSelected(item, true);
}

void GUI_Playlist::notifyNewPlaylistTrack(PLAYLIST_TRACK *pl_track, int playlistID)
{
  if(!isConnected) return;

  LVI_Playlist *checklvi = static_cast<LVI_Playlist*>(playlist_list->selectedItem());
  if((playlistID != 0) && ((checklvi == NULL) || (checklvi->getID() != playlistID)))
    return;

  if((playlistID == 0 &&
      mapCurrent.find(pl_track->track_id) != mapCurrent.end()) ||
     (playlistID > 0 &&
      mapEditor.find(pl_track->track_id) != mapEditor.end()))
    return;

  QListView *temp_listview;
  if(playlistID == 0)
    temp_listview = current_tracklist;
  else
    temp_listview = tracklist;

  LVI_PlaylistTrack *item;
  item = new LVI_PlaylistTrack(temp_listview, pl_track, mapType[pl_track->medium_id]);
  item->setAvailability(mapAvail[pl_track->medium_id]);

  if(playlistID == 0)
    mapCurrent[pl_track->track_id] = item;
  else 
    mapEditor[pl_track->track_id] = item;

  temp_listview->setSorting(-1);
  
  adjustPlaylistStatistics();
}

void GUI_Playlist::notifyRemovePlaylist(LVI_Playlist *playlist)
{
  if(!isConnected) return;

  if (displayed_playlist == playlist) {
    if(playlist->itemAbove())
      displayed_playlist = dynamic_cast<LVI_Playlist*>(playlist->itemAbove());
    else
      displayed_playlist = dynamic_cast<LVI_Playlist*>(playlist->itemBelow());

    if(displayed_playlist != NULL) {
      tracklist->setSelected(dynamic_cast<QListViewItem*>(playlist),true);
      tracklist->setCurrentItem(dynamic_cast<QListViewItem*>(playlist));
    }
    tracklist->clear();
    selectedItems.clear();
    resetPlaylistStatistics();
  }

  delete playlist;
}

void GUI_Playlist::notifyRemovePlaylistTracks(QList<LVI_PlaylistTrack> *pl_track_list)
{
  for(LVI_PlaylistTrack *lvi = pl_track_list->first(); lvi != 0; lvi = pl_track_list->next()) {
    if(lvi->getPlaylistID() == 0) {
      mapCurrent.remove(lvi->getTrackID());
      current_tracklist->takeItem(static_cast<QListViewItem*>(lvi));
    }
    else {
      mapEditor.remove(lvi->getTrackID());
      tracklist->takeItem(static_cast<QListViewItem*>(lvi));
    }
    delete lvi;
  }
  adjustPlaylistStatistics();
}

void GUI_Playlist::notifyRenamePlaylist(LVI_Playlist *playlist, QString newname)
{
  playlist->setText(0, newname);
  playlist->setName(newname);
  playlist_list->sort();
}

void GUI_Playlist::notifyMovePlaylistTrack()
{
/*
  if(current_tracklist->hasFocus())
    current_tracklist->sort();
  else
    tracklist->sort();
*/
}

void GUI_Playlist::notifyNewMediumBasis(QList<MEDIUM> *list)
{
  for(MEDIUM *medium = list->first(); medium != 0; medium = list->next()) notifyNewMedium(medium);
}

void GUI_Playlist::notifyNewMedium(MEDIUM *medium)
{
  mapType[medium->id] = medium->type;
  mapAvail[medium->id] = (medium->type == MEDIUM_HARDDISK ? true : false);
}

void GUI_Playlist::notifyMediumRemoved(int id)
{
  mapType.remove(id);
  mapAvail.remove(id);
}

void GUI_Playlist::notifyMediumAvailabilityChanged(int mediumID, QString path, bool avail)
{
  if(avail == true) mapPath[mediumID] = path;

  mapAvail[mediumID] = avail;
  for(LVI_PlaylistTrack *lvi = dynamic_cast<LVI_PlaylistTrack*>(current_tracklist->firstChild());
      lvi != 0; lvi = dynamic_cast<LVI_PlaylistTrack*>(lvi->nextSibling())) {
    if(lvi->getMediumID() == mediumID) lvi->setAvailability(avail);
  }

  for(LVI_PlaylistTrack *lvi = dynamic_cast<LVI_PlaylistTrack*>(tracklist->firstChild());
      lvi != 0; lvi = dynamic_cast<LVI_PlaylistTrack*>(lvi->nextSibling())) {
    if(lvi->getMediumID() == mediumID) lvi->setAvailability(avail);
  }
}

#ifdef HAVE_MEXTRAS
void GUI_Playlist::notifyNewExtraData(EXTRADATA_GUI *data)
{
  if (data->artist_pixmap)
    setArtistPixmap(*data->artist_pixmap);
  else 
    setArtistPixmap(QIDPixmap());
  
  setLyrics(data->lyrics);
  setYearsActive(data->years_active);
  setAlbumReview(data->album_review);
  setAlbumTracks(data->album_tracks);
  setBiography(data->biography);

  LVI_Track *item = static_cast<LVI_Track*>(gui->getListing()->list()->currentItem());
  if (item) {
    TRACK *track = new TRACK(*item->getOriginalTrack());
    setArtist(track->artist);
    setAlbum(track->album);
    setTitle(track->title);
  }

  if (!config->mextras_bgnd()) {
    if (data->cover_pixmap) 
      setCoverPixmap(*data->cover_pixmap);
    else 
      setCoverPixmap(QIDPixmap());
  } else {
    if (data->cover_pixmap && !data->cover_pixmap->image.isNull() ) {
      if ( data->cover_pixmap->ID != last_bkgID ) {
	//	qWarning( "bkg_ID from:%i to:%i",last_bkgID ,data->cover_pixmap->ID);
	last_bkgID = data->cover_pixmap->ID;
	QImage blended_img = data->cover_pixmap->image.convertToImage();
	gui->getListing()->list()->setPaletteBackgroundPixmap( GUI::blend( paletteBackgroundColor (), blended_img, 0.6) );
      }
    }
    else {
      gui->getListing()->list()->unsetPalette();
      //      qWarning( "bkg_ID from:%i to:%i", last_bkgID, 0);
      last_bkgID = 0;
    }
  }
}
#endif /* HAVE_MEXTRAS */

void GUI_Playlist::clear()
{
  tracklist->clear();
  current_tracklist->clear();
  playlist_list->clear();
  selectedItems.clear();
  mapCurrent.clear();
  mapEditor.clear();
  resetPlaylistStatistics();
}

int GUI_Playlist::getDisplayedPlaylistID()
{
  if(displayed_playlist)
    return displayed_playlist->getID();

  return -1;
}

// ********************
// *** PRIVATE PART ***
// ********************

void GUI_Playlist::getSelectedItems(bool current)
{
  selectedItems.clear();
  QListViewItem *curr;
  curr = (current) ? current_tracklist->firstChild() : tracklist->firstChild();
  while(curr) {
    if(curr->isSelected())
      selectedItems.append(curr);

    curr = curr->nextSibling();
  }
}

bool GUI_Playlist::existsPlaylist(const QString &s)
{
  if(!isConnected) return false;

  return (playlist_list->findItem(s,0,ExactMatch) != 0);
}

// #################################################
// # callbacks for new/rename/remove buttons
// #################################################
void GUI_Playlist::slot_newButton()
{
  if(!isConnected) return;

  bool ok = false, done;
  QString name;
  do {
    name = QInputDialog::getText(_("New playlist"), QString(_("Enter name of new playlist")) + ":", QLineEdit::Normal, name, &ok);
    if(ok) {
      done = (!name.isEmpty() && !existsPlaylist(name));
      if(!done) {
	    QMessageBox::information(gui, _("Message"),
				 _("Playlist name is empty\nOR\nPlaylist with that name already exists!"),
				 QMessageBox::Ok);
      }
    } else 
      done = true;
  } while(!done);

  if(ok) 
    jobman->addJob(new Job_ModifyPlaylists(APPEND_PLAYLIST, name));
}

void GUI_Playlist::slot_menuRenamePlaylist()
{
  if(!isConnected) return;

  QListViewItem *item;
  item = playlist_list->selectedItem();
  if(item == 0) {
    return;
  }

  bool ok = false, done;
  QString name;
  do {
    name = QInputDialog::getText(_("Rename playlist"), QString(_("Enter new name of playlist")) + ":", QLineEdit::Normal, name, &ok);
    if(ok) {
      done = (!name.isEmpty() && !existsPlaylist(name));
      if(!done) {
	QMessageBox::information(gui, _("Message"),
				 _("Playlist name is empty\nOR\nPlaylist with that name already exists!"),
				 QMessageBox::Ok);
      }
    } else 
      done = true;
  } while(!done);

  if(ok) {
    jobman->addJob(new Job_ModifyPlaylists(RENAME_PLAYLIST, dynamic_cast<LVI_Playlist*>(item), name));
  }
}

void GUI_Playlist::slot_menuRemovePlaylist()
{
  if(!isConnected) return;

  QListViewItem *item;
  item = playlist_list->selectedItem();
  if(item == 0)
    return;
  
  int result =  QMessageBox::warning(gui, 
                     _("Remove Playlist"),
                     /* TRANSLATORS: %1 = name of playlist */
                     QString(_("OK to remove playlist '%1' from database?")).arg(item->text(0)),
                     QMessageBox::Yes, QMessageBox::No);

  if (result != QMessageBox::Yes) return;

  jobman->addJob(new Job_ModifyPlaylists(DELETE_PLAYLIST, dynamic_cast<LVI_Playlist*>(item)));
}

void GUI_Playlist::slot_refreshFromInternetButton()
{
#ifdef HAVE_MEXTRAS
  QListViewItem *curr = gui->getListing()->list()->firstChild();
  tabmex->showPage( mexlog );  // ensure log is visible... 
  while(curr) {
    if(curr->isSelected()) {
      LVI_Track *item = dynamic_cast<LVI_Track*>(curr);
      if (!item ) assert( 0 && "Software error, cannot cast to LVI_Track*");
      TRACK *track = new TRACK(*item->getOriginalTrack());
      database->lock();
      database->zapExtraData( track );   // will remove all musicextras data
      gui->getPlaylisting()->mexlog->append( QDateTime::currentDateTime().toString() );
      jobmusicextras = new Job_CallMusicextras(*track, true);
      jobman->addJob(jobmusicextras);
      database->unlock();
    }
    curr = curr->nextSibling();
  }
#endif /* HAVE_MEXTRAS */
}

void GUI_Playlist::slot_abortRefreshFromInternetButton()
{
#ifdef HAVE_MEXTRAS
  jobman->terminateMusicextrasJob();
#endif /* HAVE_MEXTRAS */
}

void GUI_Playlist::slot_resetButton()
{
  generatorSpinMinTrackLength->setValue(120);
  generatorSpinMaxTrackLength->setValue(300);
  generatorSpinTracks->setValue(10);
  generatorCheckBoxOnlyFavouriteArtists->setChecked(false);
  generatorCheckBoxOnlyHarddisk->setChecked(true);
  generatorSpinRating->setValue(0);
}

void GUI_Playlist::slot_generateButton()
{
  if(!isConnected) return;

  jobman->addJob(new Job_GeneratePlaylist(generatorSpinTracks->value(), generatorSpinMinTrackLength->value(), generatorSpinMaxTrackLength->value(), generatorCheckBoxOnlyFavouriteArtists->isChecked(), generatorCheckBoxOnlyHarddisk->isChecked(), generatorSpinRating->value()));
  
  setCurrentPage(indexOf(vsplit));
}

void GUI_Playlist::slot_generatorMinTrackLength(int length)
{
  generatorSpinMaxTrackLength->setMinValue(length);
}

void GUI_Playlist::slot_generatorMaxTrackLength(int length)
{
  generatorSpinMinTrackLength->setMaxValue(length);
}

// #################################################
// # callbacks for move track up/down buttons
// #################################################
void GUI_Playlist::slot_movePlaylistTrackUp()
{
	if(!isConnected) return;
	
	QListViewItem *item;
	item = tracklist->currentItem();
	if(item == 0)
		return;
		
	if(item == tracklist->firstChild()) return;

	if(item->itemAbove() == tracklist->firstChild()) tracklist->firstChild()->moveItem(item);
	else item->moveItem(item->itemAbove()->itemAbove());

	slot_savePlaylistTracks();

}

void GUI_Playlist::slot_movePlaylistTrackDown()
{
  if(!isConnected) return;
  
  QListViewItem *item;
  item = tracklist->currentItem();
  if(item == 0)
  	return;
  
  if(item->itemBelow() == 0) return;
  
  item->moveItem(item->itemBelow());

  slot_savePlaylistTracks();

}

void GUI_Playlist::slot_savePlaylistTracks()
{
  if(!displayed_playlist || tracklist->firstChild() == 0) return;

  QList<int> *poslist = new QList<int>;
  for(QListViewItem *curr = tracklist->firstChild(); curr != 0; curr = curr->nextSibling()) {
	LVI_PlaylistTrack *lvi = dynamic_cast<LVI_PlaylistTrack*>(curr);
	int *id = new int;
	*id = lvi->getTrackID();
	poslist->append(id);
  }
  jobman->addJob(new Job_ModifyPlaylist_Tracks(SAVE_POSITIONS, displayed_playlist->getID(), poslist));
}


// #################################################
// # callback for context menu
// #################################################
void GUI_Playlist::slot_contextMenu(QListViewItem *lvi, const QPoint &p, int column)
{
  if(!isConnected) return;
  
  qApp->unlock();  // Allows update from pending threads
  // see http://lists.trolltech.com/qt-interest/2003-08/thread00316-0.html
  QPopupMenu *menu = new QPopupMenu(this);
  qApp->lock();
  
  int i=0;
  if(lvi->listView() == playlist_list) {
//    menu->insertItem("ADD this playlist to current", this, SLOT(slot_menuAddPlaylistToCurrent()),0,i++);
#ifdef HAVE_MIXXX
    if(config->RegularPlayerEnabled()) {
#endif /* HAVE_MIXXX */
      menu->insertItem(pixmapcache->get("stock_media-play.png"), _("Play playlist with regular player"), this, SLOT(slot_menuPlayPlaylist()),0,i++);
      menu->insertItem(pixmapcache->get("stock_music-library.png"), _("Enqueue playlist at regular player"), this, SLOT(slot_menuEnqueuePlaylist()),0,i++);
      menu->insertSeparator();
#ifdef HAVE_MIXXX
    }
    if(config->MixxxPlayerEnabled() && isMixxxConnected) {
      menu->insertItem(pixmapcache->get("stock_music-library.png"), _("mixxx: enqueue playlist at channel 1"), this, SLOT(slot_menuEnqueuePlaylistAtChannel1()),0,i++);
      menu->insertItem(pixmapcache->get("stock_music-library.png"), _("mixxx: enqueue playlist at channel 2"), this, SLOT(slot_menuEnqueuePlaylistAtChannel2()),0,i++);
      menu->insertSeparator();
    }
#endif /* HAVE_MIXXX */
	menu->insertItem(pixmapcache->get("action_updateonly.png"), _("Playlist to Selection"), this, SLOT(slot_menuPlaylistToSelection()),0,i++);
    menu->insertItem(pixmapcache->get("lvi_hdd.png"), _("Export playlist"), this, SLOT(slot_menuExport()),0,i++);
    menu->insertItem(pixmapcache->get("lvi_hdd.png"), _("Copy Files To ..."), this, SLOT(slot_menuCopyFiles()),0,i++);
    // prevent concurrent copy jobs
    menu->setItemEnabled(i-1, !fileCopyInProgress);
    menu->insertSeparator();
	menu->insertItem(pixmapcache->get("action_burncd.png"), _("Burn AudioCD from playlist using prokburn"), this, SLOT(slot_menuBurnCD()),0,i++);
    menu->insertItem(pixmapcache->get("action_burncd.png"), _("Burn AudioCD from playlist using k3b"), this, SLOT(slot_menuBurnK3bAudioCD()),0,i++);
    menu->insertItem(pixmapcache->get("action_burncd.png"), _("Burn DataCD from playlist using k3b"), this, SLOT(slot_menuBurnK3bDataCD()),0,i++);
    menu->insertItem(pixmapcache->get("action_burncd.png"), _("Burn DataDVD from playlist using k3b"), this, SLOT(slot_menuBurnK3bDataCD()),0,i++);
    menu->insertSeparator();
    menu->insertItem(_("Rename"), this, SLOT(slot_menuRenamePlaylist()),0,i++);
    menu->insertItem(pixmapcache->get("action_remove.png"), _("Remove"), this, SLOT(slot_menuRemovePlaylist()),0,i++);
  }
  else if(lvi->listView() == tracklist || lvi->listView() == current_tracklist) {
	if(dynamic_cast<LVI_PlaylistTrack*>(lvi)->getIsAvailable()) {
#ifdef HAVE_MIXXX
      if(config->RegularPlayerEnabled()) {
#endif /* HAVE_MIXXX */
        menu->insertItem(pixmapcache->get("stock_media-play.png"), _("Play track(s) with regular player"), this, SLOT(slot_menuPlayPlaylistTracks()),0,i++);
        menu->insertItem(pixmapcache->get("stock_music-library.png"), _("Enqueue track(s) at regular player"), this, SLOT(slot_menuEnqueuePlaylistTracks()),0,i++);
        menu->insertSeparator();
#ifdef HAVE_MIXXX
      }
      if(config->MixxxPlayerEnabled() && isMixxxConnected) {
        menu->insertItem(pixmapcache->get("play-off-Ch1.png"), _("mixxx: load track to channel 1"), this, SLOT(slot_mixxxLoadAtChannel1()),0,i++);
        menu->insertItem(pixmapcache->get("play-off-Ch2.png"), _("mixxx: load track to channel 2"), this, SLOT(slot_mixxxLoadAtChannel2()),0,i++);
        menu->insertItem(pixmapcache->get("play-on-Ch1.png"), _("mixxx: load track to channel 1 and play"), this, SLOT(slot_mixxxLoadAndPlayAtChannel1()),0,i++);
        menu->insertItem(pixmapcache->get("play-on-Ch2.png"), _("mixxx: load track to channel 2 and play"), this, SLOT(slot_mixxxLoadAndPlayAtChannel2()),0,i++);
        menu->insertItem(pixmapcache->get("stock_music-library.png"), _("mixxx: enqueue at channel 1"), this, SLOT(slot_mixxxEnqueueTracksAtChannel1()),0,i++);
        menu->insertItem(pixmapcache->get("stock_music-library.png"), _("mixxx: enqueue at channel 2"), this, SLOT(slot_mixxxEnqueueTracksAtChannel2()),0,i++);
        menu->insertSeparator();
      }
#endif /* HAVE_MIXXX */
	}
    menu->insertItem(pixmapcache->get("action_remove.png"), _("remove track(s) from playlist"), this, SLOT(slot_menuRemovePlaylistTracks()),0,i++);
  }
  menu->exec(p,0);
}

void GUI_Playlist::slot_menuBurnCD()
{
  QString command = QString( "xterm -e %1" ).arg( config->getBurnCDCmd().stripWhiteSpace() );
  
  for (QListViewItem *curr = tracklist->firstChild(); curr != 0; curr = curr->nextSibling())
    {
      LVI_PlaylistTrack *lvi = dynamic_cast<LVI_PlaylistTrack*>(curr);
      PLAYLIST_TRACK *track = lvi->getPlaylistTrack();
      QString prepath="";
      if(mapPath[lvi->getMediumID()] != 0) prepath = mapPath[lvi->getMediumID()];

      if(lvi->getIsAvailable()) 
	    command.append( QString(" ") + adjustStringCommandLine(prepath + track->path + "/" + track->filename) );
    }

  QString listfiles("<b>" + QString(_("List of files to burn")) + ":</b><br><br><font color=blue>");
  int totaltime=0;
 
  QList<TRACK>* ltrack  = database->queryTracksByPlaylist(displayed_playlist->getID());
  for(TRACK *cur = ltrack->first(); cur != 0; cur = ltrack->next()) {
    listfiles.append(QString(cur->path + "/" + cur->filename +"<br>").stripWhiteSpace() );
    totaltime += cur->length + 2;  // 2 second blank between tracks
    if (cur->samplerate != 44100) {
      // TRANSLATORS: %1 = Path, %2 = Filename
      QMessageBox::critical(this, _("Message"), QString(_("<font color=red><b>File %1/%2</font><br>"
							  "uses a samplerate different from 44100Hz!</b><br><br>"
							  "It cannot be burned to an audio CD without special conversion."
							  "Remove it from the playlist if you want to proceed.<br>")
							).arg(cur->path).arg(cur->filename));
      return;
    }
  }


  int userIsSure = QMessageBox::warning(gui, _("Burning a CD"), listfiles + QString("</font><br><br><center><b><u>" + QString(_("Total CD length will be:")) + "</u> %1 " /* TRANSLATORS: minutes */ + QString(_("min")) + " %2 " + /* TRANSLATORS: seconds */ QString(_("sec")) + "</center></b><br><br>").arg(totaltime/60).arg(totaltime%60) +
					QString(_("Are you really sure you want to proceed?")) + "<br>", 
					_("&Yes"), _("&No"));
  if(userIsSure == 0) {
    command.append(" &");
    //    qWarning( command.local8Bit().data()); 
    system(command.local8Bit().data());
  }
}

void GUI_Playlist::slot_menuBurnK3bAudioCD()
{
  burnK3bCD("audiocd");
}

void GUI_Playlist::slot_menuBurnK3bDataCD()
{
  burnK3bCD("datacd");
}

void GUI_Playlist::slot_menuBurnK3bDataDVD()
{
  burnK3bCD("datadvd");
}

void GUI_Playlist::burnK3bCD(QString type)
{
  QString command = QString("%1 --%2").arg( config->getK3bCmd().stripWhiteSpace() ).arg( type );
  
  for (QListViewItem *curr = tracklist->firstChild(); curr != 0; curr = curr->nextSibling())
  {
      LVI_PlaylistTrack *lvi = dynamic_cast<LVI_PlaylistTrack*>(curr);
      PLAYLIST_TRACK *track = lvi->getPlaylistTrack();
      QString prepath="";
      if(mapPath[lvi->getMediumID()] != 0) prepath = mapPath[lvi->getMediumID()];

      if(lvi->getIsAvailable()) 
        command.append( " " + adjustStringCommandLine( prepath + track->path + "/" + track->filename ) );
  }

  command.append(" &");
  //  qWarning( command.local8Bit().data()); 
  system(command.local8Bit().data());
}

void GUI_Playlist::slot_menuAddPlaylistToCurrent()
{
  QList<LVI_PlaylistTrack> *temp_list = new QList<LVI_PlaylistTrack>;
  for(QListViewItem *curr = tracklist->firstChild(); curr != 0; curr = curr->nextSibling()) {
	LVI_PlaylistTrack *lvi = dynamic_cast<LVI_PlaylistTrack*>(curr);
	temp_list->append(lvi);
  }
  jobman->addJob(new Job_ModifyPlaylist_Tracks(APPEND_TO_CURRENT,temp_list));
}

void GUI_Playlist::slot_menuPlaylistToSelection()
{
    qWarning("displayed_playlist->getID(): %d", displayed_playlist->getID() );
    jobman->addJob(new Job_QueryDatabase(QUERY_BY_PLAYLIST, displayed_playlist->getID()));
}

void GUI_Playlist::slot_menuPlayPlaylist()
{
  callRegularPlayerWithPlaylist(false);
}

void GUI_Playlist::slot_menuEnqueuePlaylist()
{
  callRegularPlayerWithPlaylist(true);
}

void GUI_Playlist::callRegularPlayerWithPlaylist(bool enqueue)
{
  QStringList fileList;
  
  for (QListViewItem *curr = tracklist->firstChild(); curr != 0; curr = curr->nextSibling())
  {
    LVI_PlaylistTrack *lvi = dynamic_cast<LVI_PlaylistTrack*>(curr);
    PLAYLIST_TRACK *track = lvi->getPlaylistTrack();
    QString prepath="";
    if(mapPath[lvi->getMediumID()] != 0) prepath = mapPath[lvi->getMediumID()];
    if(lvi->getIsAvailable())
      fileList.append(QString(prepath + track->path + "/" + track->filename));
  }
  
  commandLineCallToRegularPlayer(fileList, enqueue);
}

void GUI_Playlist::slot_menuEnqueuePlaylistAtChannel1()
{
#ifdef HAVE_MIXXX
  enqueuePlaylistAtChannel(1);
#endif /* HAVE_MIXXX */
}

void GUI_Playlist::slot_menuEnqueuePlaylistAtChannel2()
{
#ifdef HAVE_MIXXX
  enqueuePlaylistAtChannel(2);
#endif /* HAVE_MIXXX */
}

#ifdef HAVE_MIXXX
void GUI_Playlist::enqueuePlaylistAtChannel(int channel)
{
  QList<LVI_PlaylistTrack> list;
  
  for (QListViewItem *curr = tracklist->firstChild(); curr != 0; curr = curr->nextSibling())
  {
    LVI_PlaylistTrack *lvi = dynamic_cast<LVI_PlaylistTrack*>(curr);
    if(lvi->getIsAvailable())
      list.append(lvi);
  }
  
  gui->getListing()->getMixxxQueue()->enqueueAtChannel(list, channel);
}
#endif /* HAVE_MIXXX */

void GUI_Playlist::slot_menuRemovePlaylistTracks()
{
  QList<LVI_PlaylistTrack> *temp_list = new QList<LVI_PlaylistTrack>;
  for(QListViewItem *curr = selectedItems.first(); curr != 0; curr = selectedItems.next()) {
    LVI_PlaylistTrack *lvi = dynamic_cast<LVI_PlaylistTrack*>(curr);
    temp_list->append(lvi);
  }
  jobman->addJob(new Job_ModifyPlaylist_Tracks(DELETE_PLAYLIST_TRACK,temp_list));
}

void GUI_Playlist::slot_menuPlayPlaylistTracks()
{
  callRegularPlayerWithTracks(false);
}

void GUI_Playlist::slot_menuEnqueuePlaylistTracks()
{
  callRegularPlayerWithTracks(true);
}

void GUI_Playlist::callRegularPlayerWithTracks(bool enqueue)
{
  commandLineCallToRegularPlayer(getSelectedFileList(), enqueue);
}

QStringList GUI_Playlist::getSelectedFileList()
{
  QStringList fileList;
  
  for (QListViewItem *curr = selectedItems.first(); curr != 0; curr = selectedItems.next())
  {
    LVI_PlaylistTrack *lvi = dynamic_cast<LVI_PlaylistTrack*>(curr);
    PLAYLIST_TRACK *track = lvi->getPlaylistTrack();
    QString prepath="";
    if(mapPath[lvi->getMediumID()] != 0) prepath = mapPath[lvi->getMediumID()];
    if(lvi->getIsAvailable())
      fileList.append(QString(prepath + track->path + "/" + track->filename));
  }
  
  return fileList;
}

QString GUI_Playlist::getCurrentFile()
{
  QString file;
  
  LVI_PlaylistTrack *lvi = dynamic_cast<LVI_PlaylistTrack*>(tracklist->currentItem());
  PLAYLIST_TRACK *track = lvi->getPlaylistTrack();
  QString prepath="";
  if(mapPath[lvi->getMediumID()] != 0) prepath = mapPath[lvi->getMediumID()];
  if(lvi->getIsAvailable()) {
    file = prepath + track->path + "/" + track->filename;
  } else {
    QMessageBox::information(this, "Message", _("File(s) not available for playing"), QMessageBox::Ok);
  }
  
  return file;
}


void GUI_Playlist::slot_mixxxLoadAtChannel1()
{
#ifdef HAVE_MIXXX
  mixxxLoadAtChannel(1);
#endif /* HAVE_MIXXX */
}

void GUI_Playlist::slot_mixxxLoadAtChannel2()
{
#ifdef HAVE_MIXXX
  mixxxLoadAtChannel(2);
#endif /* HAVE_MIXXX */
}

#ifdef HAVE_MIXXX
void GUI_Playlist::mixxxLoadAtChannel(int channel)
{
  mixxxClient->loadAtChannel(getCurrentFile(), channel);
}
#endif /* HAVE_MIXXX */

void GUI_Playlist::slot_mixxxLoadAndPlayAtChannel1()
{
#ifdef HAVE_MIXXX
  mixxxLoadAndPlayAtChannel(1);
#endif /* HAVE_MIXXX */
}

void GUI_Playlist::slot_mixxxLoadAndPlayAtChannel2()
{
#ifdef HAVE_MIXXX
  mixxxLoadAndPlayAtChannel(2);
#endif /* HAVE_MIXXX */
}

#ifdef HAVE_MIXXX
void GUI_Playlist::mixxxLoadAndPlayAtChannel(int channel)
{
  mixxxClient->loadAndPlayAtChannel(getCurrentFile(), channel);
}
#endif /* HAVE_MIXXX */

void GUI_Playlist::slot_mixxxEnqueueTracksAtChannel1()
{
#ifdef HAVE_MIXXX
    mixxxEnqueueTracksAtChannel(1);
#endif /* HAVE_MIXXX */
}

void GUI_Playlist::slot_mixxxEnqueueTracksAtChannel2()
{
#ifdef HAVE_MIXXX
    mixxxEnqueueTracksAtChannel(2);
#endif /* HAVE_MIXXX */
}

#ifdef HAVE_MIXXX
void GUI_Playlist::mixxxEnqueueTracksAtChannel(int channel)
{
  QList<LVI_PlaylistTrack> list;
  
  for (QListViewItem *curr = selectedItems.first(); curr != 0; curr = selectedItems.next())
  {
    LVI_PlaylistTrack *lvi = dynamic_cast<LVI_PlaylistTrack*>(curr);
    if(lvi->getIsAvailable())
      list.append(lvi);
  }
  
  gui->getListing()->getMixxxQueue()->enqueueAtChannel(list, channel);
}
#endif /* HAVE_MIXXX */

void GUI_Playlist::slot_importPlaylist()
{
  if(!isConnected) return;

  int proceed = QMessageBox::No;
  QString fileName;
  while( proceed == QMessageBox::No )
  {
      fileName = QFileDialog::getOpenFileName(".", QString(_("playlist")) + " (*.m3u *.pls)", this, "Import Playlist", _("Import Playlist"));
      if( fileName.isEmpty() ) {
        // cancel button
        return;
      }
      QFileInfo info( fileName );
      QString extension = info.extension( /*complete*/ false );

      if( "m3u" != extension && "pls" != extension ) {
        QMessageBox::critical(this, _("Error"), QString(_("Mime type not supported:")) + "\n" + extension, QMessageBox::Ok, 0);
        continue;
      }

      if( existsPlaylist(info.baseName()) ) {
        proceed = QMessageBox::warning(this, 
                          "prokyon3 - " + QString(_("Import Playlist")),
                          info.baseName() + "\n" + QString(_("The playlist already exists!\n"
                          "Do you want to overwrite it?")),
                          QMessageBox::Yes, QMessageBox::No);
        if( proceed != QMessageBox::Yes ) continue;
      } else {
        proceed = QMessageBox::Yes;
      }
  }
  
  jobman->addJob(new Job_ImportPlaylist(fileName));
}

void GUI_Playlist::slot_menuExport()
{
  LVI_Playlist *playlist = static_cast<LVI_Playlist*>(playlist_list->selectedItem());
  QString defaultName = playlist->getName();
  // replace characters that cause trouble, p.e. see generated playlists
  defaultName.replace(" ", "_");
  defaultName.replace(":", "_");
  defaultName.replace("/", "_");
  
  QFileDialog* fd = new QFileDialog(".", QString(_("playlist")) + " (*.m3u *.pls);;html (*.htm *.html)", this, _("Export Playlist"), /*modal*/ true);
  fd->setCaption( "prokyon3 - " + QString(_("Export Playlist")) );
  fd->setMode( QFileDialog::AnyFile );
  fd->setSelection(defaultName + ".m3u");
  int proceed = QMessageBox::No;
  while( proceed == QMessageBox::No )
  {
    if( fd->exec() == QDialog::Accepted )
    {
      QString fileName = fd->selectedFile();
      
      QFile file( fileName );
      QFileInfo info( file );
      if( info.exists() ) {
        proceed = QMessageBox::warning(NULL, 
                          "prokyon3 - " + QString(_("Export Playlist")),
                          fileName + "\n" + QString(_("The file already exists!\n"
                          "Do you want to overwrite it?")),
                          QMessageBox::Yes, QMessageBox::No);
        if( proceed != QMessageBox::Yes ) continue;
      }
      QString extension = info.extension( /*complete*/ false );
      if( "m3u" != extension && "pls" != extension ) {
        QMessageBox::critical(this, _("Error"), QString(_("Mime type not supported:")) + "\n" + extension, QMessageBox::Ok, 0);
        proceed = QMessageBox::No;
        continue;
      }
      
      if( file.open( IO_WriteOnly ) ) {
        QTextStream stream( &file );
        
        if( "m3u" == extension )      stream << "#EXTM3U\n";
        else if( "pls" == extension ) stream << "[playlist]\nNumberOfEntries=" << tracklist->childCount() << "\n";
        
        unsigned int i = 1;
        for (QListViewItem *curr = tracklist->firstChild(); curr != 0; curr = curr->nextSibling())
        {
          LVI_PlaylistTrack *lvi = dynamic_cast<LVI_PlaylistTrack*>(curr);
          PLAYLIST_TRACK *track = lvi->getPlaylistTrack();
          if( "m3u" == extension ) stream << "#EXTINF:" << lvi->getLength() << "," << lvi->text(0) << "\n";
          if( "pls" == extension ) stream << "File" << i++ << "=";
          QString prepath = "";
          if(mapPath[lvi->getMediumID()] != 0) prepath = mapPath[lvi->getMediumID()];
          if(lvi->getIsAvailable()) {
            stream << QString(prepath + track->path + "/" + track->filename).local8Bit().data() << "\n";
          }
        }
        file.close();
        proceed = QMessageBox::Yes;
      } else {
        QMessageBox::critical(this, _("Error"), QString(_("Could not write File:")) + "\n" + fileName, QMessageBox::Ok, 0);
      }
    } else {
      // cancel button
      return;
    }
  }
}

void GUI_Playlist::slot_menuCopyFiles()
{
    LVI_Playlist *playlist = static_cast<LVI_Playlist*>(playlist_list->selectedItem());
     
    QFileDialog* fd = new QFileDialog(".", "", this, _("Select a directory"), /*modal*/ true);
    fd->setCaption( "prokyon3 - " + QString(_("Copy Files To ...")) );
    fd->setMode( QFileDialog::DirectoryOnly );
    if( fd->exec() == QDialog::Accepted )
    {
        QString directory = fd->selectedFile();
        QFileInfo info( directory );
        if( !info.isDir() ) {
            QMessageBox::critical(NULL, 
                        "prokyon3 - " + QString(_("Copy Files To ...")),
                        /* TRANSLATORS: %1 = file name */
                        QString(_("%1 is not a directory!")).arg(directory),
                        QMessageBox::Ok, 0);
            return;
        }
    
        QStringList files;
    
        for (QListViewItem *curr = tracklist->firstChild(); curr != 0; curr = curr->nextSibling()) {
            LVI_PlaylistTrack *lvi = dynamic_cast<LVI_PlaylistTrack*>(curr);
            PLAYLIST_TRACK *track = lvi->getPlaylistTrack();
            QString prepath = "";
            if(mapPath[lvi->getMediumID()] != 0) prepath = mapPath[lvi->getMediumID()];
            if(lvi->getIsAvailable()) {
                files.append(QString("file:" + prepath + track->path + "/" + track->filename));
            }
        }
    
        if(!fileCopyInProgress && files.count() >= 1) {
            fileCopyInProgress = true;
            numberOfFilesToCopy = files.count();
            
            copyFilesProgressDialogScroll = new ProgressDialogScroll("", _("Hide"), numberOfFilesToCopy - 1, this , "progress" ); 
            
            QUrlOperator *op = new QUrlOperator();
            connect( op, SIGNAL(startedNextCopy(const QPtrList<QNetworkOperation>&)), this, SLOT(slot_startedNextCopy(const QPtrList<QNetworkOperation>&)));
            
            // this operation always runs in background -> no need to create a job (thread)
            op->copy( files, "file:" + directory );
        }
    }
}

void GUI_Playlist::slot_startedNextCopy(const QPtrList<QNetworkOperation> &lst)
{
    if(!copyFilesProgressDialogScroll->wasCanceled()) {
        QNetworkOperation *op = lst.getFirst();
        copyFilesProgressDialogScroll->TextEdit->append(op->arg(0));
        copyFilesProgressDialogScroll->setProgress( copyFilesProgressDialogScroll->progress() + 1 );
    }

    --numberOfFilesToCopy;
    if(numberOfFilesToCopy <= 0) {
        fileCopyInProgress = false;
    }
}

// #################################################
// # callback for playlist selection change
// #################################################
void GUI_Playlist::slot_playlistSelectionChanged(QListViewItem *lvi)
{
  if(!isConnected) return;

  if (lvi == NULL) {
  	displayed_playlist = NULL;	
  	return;
  }
  
  LVI_Playlist *item = dynamic_cast<LVI_Playlist*>(lvi);
  if (item == NULL) {
  	displayed_playlist = NULL;
  	return;
  }

  selectedItems.clear();
  jobman->addJob(new Job_QueryPlaylist_Tracks(item->getID()));
  displayed_playlist = item;
}

// #################################################
// # callback for playlist track selection change
// #################################################
void GUI_Playlist::slot_trackSelectionChanged()
{
  if(!isConnected) return;

  if(current_tracklist->hasFocus())
    getSelectedItems(true);
  else
    getSelectedItems(false);
}

// #################################################
// # callback for selected tab changed
// #################################################
void GUI_Playlist::slot_tabChange(QWidget *widget)
{
  if(widget == current_tracklist) {
    selectedItems.clear();
    getSelectedItems(true);
    current_tracklist->setFocus();
    if (isConnected) {
      QList<PLAYLIST_TRACK> *tracks_list_dummy;
      notifyNewPlaylistTracksBasis(tracks_list_dummy, 0);
    }
  }
  else {
    selectedItems.clear();
    getSelectedItems(false);
    vsplit->setFocus();
  }
}

// #################################################
// # callback for doubleClicked
// #################################################
void GUI_Playlist::slot_playlistDoubleClicked(QListViewItem *lvi)
{
  slot_menuPlayPlaylist();
}

void GUI_Playlist::slot_playlistTrackDoubleClicked(QListViewItem *lvi)
{
  slot_menuPlayPlaylistTracks();
}

void GUI_Playlist::customEvent( QCustomEvent * e )
{
#ifdef HAVE_MEXTRAS
  if ( e->type() == LOGEVENT_CODE ) {  // It must be a LogEvent from a MusicextrasWrapper thread
    LogEvent* le = static_cast<LogEvent*>( e );
    mexlog->append(le->message());
  } else if ( e->type() == ALLMUSICEXTRASJOBSDONEEVENT_CODE ) {  // 
    buttonAbortRefreshFromInternet->setEnabled(false);
  } else if ( e->type() == NEWMUSICEXTRASJOBEVENT_CODE ) {  // 
    buttonAbortRefreshFromInternet->setEnabled(true);
  } else if ( e->type() == EXTRAEVENT_CODE ) {  // It must be an ExtraEvent from a Get_ExtraData  thread
    ExtraEvent* xe = static_cast<ExtraEvent*>( e );
    notifyNewExtraData( xe->data() );
    delete xe->data();
  } else 
#endif /* HAVE_MEXTRAS */
    if ( e->type() == STATUSEVENT_CODE ) {  // It must be an StatusEvent from a thread
    StatusEvent* se = static_cast<StatusEvent*>( e );
    gui->setStatusInfo( se->data()->message, se->data()->pos);
    delete se->data();
  } else if ( e->type() == PLAYLISTIMPORTEVENT_CODE ) {
    PlaylistImportEvent* pe = static_cast<PlaylistImportEvent*>( e );
    QTabDialog *td = new QTabDialog( this, "prokyon3 - " + QCString(_("playlist import result")), true);
    td->setCaption("prokyon3 - " + QString(_("playlist import result")));
    QLabel *result = new QLabel(pe->result(), td);
    QTextEdit *notFound = new QTextEdit(td);
    notFound->setReadOnly( true );
    notFound->setWordWrap( QTextEdit::NoWrap );
    notFound->setText(pe->notFound().join("\n"));
    QTextEdit *notSynced = new QTextEdit(td);
    notSynced->setReadOnly( true );
    notSynced->setWordWrap( QTextEdit::NoWrap );
    notSynced->setText(pe->notSynced().join("\n"));
    td->addTab(result, _("Result"));
    td->addTab(notFound, _("Files not found"));
    td->addTab(notSynced, _("Files not synchronized"));
    td->setOkButton(_("Ok"));
    td->exec();
  }
}

void GUI_Playlist::adjustPlaylistStatistics()
{
  long tracks = 0;
  long time   = 0;
  double size = 0;
  
  for (QListViewItem *curr = tracklist->firstChild(); curr != 0; curr = curr->nextSibling())
  {
    LVI_PlaylistTrack *lvi = dynamic_cast<LVI_PlaylistTrack*>(curr);
    tracks++;
    time += lvi->getLength();
    size += lvi->getSize()/1048576;
  }
  
  pl_tracks->setText(QString::number(tracks));
  pl_time->setText(QString().sprintf("%d:%02d", time/60, time%60));
  pl_size->setText(QString().sprintf("%.2f MB", size));
}

void GUI_Playlist::resetPlaylistStatistics()
{
  pl_tracks->setText("0");
  pl_time->setText("0");
  pl_size->setText("0 MB");
}

void GUI_Playlist::slot_ClickLyrics() {
#ifdef HAVE_MEXTRAS
  View_Dialog* view = new View_Dialog( this, lyrics->text() );
  view->exec();
#endif /* HAVE_MEXTRAS */
}

void GUI_Playlist::slot_ClickReview() {
#ifdef HAVE_MEXTRAS
  View_Dialog* view = new View_Dialog( this, review->text() );
  view->exec();
#endif /* HAVE_MEXTRAS */
}

void GUI_Playlist::slot_ClickTracks() {
#ifdef HAVE_MEXTRAS
  View_Dialog* view = new View_Dialog( this, tracks->text() );
  view->exec();
#endif /* HAVE_MEXTRAS */
}

void GUI_Playlist::slot_ClickBiography() {
#ifdef HAVE_MEXTRAS
  View_Dialog* view = new View_Dialog( this, biography->text() );
  view->exec();
#endif /* HAVE_MEXTRAS */
}

void GUI_Playlist::slot_ClickMexlog() {
#ifdef HAVE_MEXTRAS
  View_Dialog* view = new View_Dialog( this, mexlog->text(), Qt::LogText );
  view->exec();
#endif /* HAVE_MEXTRAS */
}

#ifdef HAVE_MEXTRAS
void GUI_Playlist::clear_IconView() {
  last_coverID = 0; 
  last_artistID = 0;
  last_bkgID = 0;
  delete artistIconViewItem;
  artistIconViewItem=0;
  delete coverIconViewItem;
  coverIconViewItem=0;
}
#endif /* HAVE_MEXTRAS */


// #################################################
// # drag and drop support
// # 
// #  We assume that internal drag is enabled 
// #      only from QListView's of valid LVI_Tracks.  
// #  All selected Tracks are added to the play_list
// #################################################

ExtendedDropListView::ExtendedDropListView(QWidget *parent, const char *name)
  : QListView(parent, name)
{}

void ExtendedDropListView::dragEnterEvent(QDragEnterEvent* event)
{ 
  event->accept( true );
}

void ExtendedDropListView::dropEvent(QDropEvent* event)
{  

  QList<TRACK> *templist = new QList<TRACK>;
  int ID;

  QListView * listdrop = dynamic_cast<QListView*>( event->source() );
  if ( listdrop ) {   
    for(QListViewItem *item = listdrop->firstChild(); item != 0; item = item->nextSibling()) {
      if(item->isSelected() && dynamic_cast<LVI_Track*>(item)->getOriginalTrack() )  // robustness only
	templist->append(dynamic_cast<LVI_Track*>(item)->getOriginalTrack());
    }

    // First check if user dropped to a playlist name
    QPoint pt = mapToGlobal(event->pos());
    LVI_Playlist *pl = dynamic_cast<LVI_Playlist*>( itemAt( viewport()->mapFromGlobal( pt)  ));
    if ( pl) {
      if ( pl->getID()>0 && templist->count()>0)  // robustness only
	jobman->addJob(new Job_ModifyPlaylist_Tracks(APPEND_PLAYLIST_TRACK, templist, pl->getID()));
    }  else 
      // otherwise add to the currently selected playlist, if any.
      if((ID = gui->getPlaylisting()->getDisplayedPlaylistID()) > 0) 
	jobman->addJob(new Job_ModifyPlaylist_Tracks(APPEND_PLAYLIST_TRACK, templist, ID));
  }
}

