#ifndef LINUX_MODEL_H_INCLUDED
#define LINUX_MODEL_H_INCLUDED /* allow multiple inclusions */

/* 
 * Prospect: a developer's system profiler.
 *
 * COPYRIGHT (C) 2001-2004 Hewlett-Packard Company
 *
 * Author: Alex Tsariounov, HP
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* $Id: linux_model.h,v 1.18 2004/01/09 20:29:28 type2 Exp $ */

/*
 * This is the Linux model header file.  The global definitions are included
 * here for files to include.  The vars actually live in linux_model.c
 */

/*
 * ---------------------------- Includes --------------------------------
 */
#include <sys/times.h>
#include "f_sym_name.h"

/*
 * ----------------------------- Defines ---------------------------------
 */

/* Linux timeshare priority */
#define REALTIME_OFF   0

/* Maximum number of processors */
#define MAX_PROCS   64

/* Array access help */
#define getproc_by_pid(pid)   getproc_by_pid_cached(pid, 0)

/* Function name for crosscheck of /proc/ksyms and System.map
 * Note that this needs to be a data symbol since on ia64 the text symbols are
 * different from System.map and /proc/ksysms.  The symbol high_memory would
 * work just as well here.
 */
#define CROSSCHECKSTR  "num_physpages"

/*
 * ----------------------- Global type definitions --------------------------
 */

/* symbols are not found for these reasons */
typedef enum _ss_noSymReasons {
       cSS_INIT,                    /* the init state                      */
       cSS_FOUND_SYMS,              /* found the syms, all ok              */
       cSS_NO_SYMBOLS,              /* catch-all no symbols don't know why */
       cSS_STRIPPED_FILE,           /* file has been stripped, som only    */
       cSS_CONFUSED,                /* confusion, should not show up       */
       cSS_JUMP_TABLE,              /* hit below symbols in jump table     */
       cSS_SELF_GEN_CODE,           /* self-generated(modified) code       */
       cSS_FILE_NOT_FOUND,          /* file open failed - filesystem code  */
       cSS_FOUND_FILE               /* intemediate, found the file         */
} tNOSYMREASONS;

/* process pr_birthBy flags */
typedef enum _pr_birthByFlag {
       cBIRTH_not         = 0, /* not fork, not vfork and  not exec */
       cBIRTH_fork        = 1,
       cBIRTH_vfork       = 2,
       cBIRTH_exec        = 3,
       cBIRTH_spontaneous = 4, /* usually a kernel thread */
       cBIRTH_preExist    = 5
} tPRBIRTHBYFLAG;

/* process pr_endBy flags */
typedef enum _pr_endByFlag {
       cEND_not          =  0,   /* if not exec and not exit */
       cEND_exit         =  1,
       cEND_exec         =  2
} tPRENDBYFLAG;

/* memory mapped region structure: user and kernel */
typedef struct _region_data {
    struct _region_data *rd_fwd;    /* Foward link _region_data */
    struct _region_data *rd_bck;    /* Backward link _region_data */
    int                  rd_freed;  /* This region has been freed */
    pid_t                rd_pid;    /* Owner pid */
    char                *rd_name;   /* Short name of mapped file */
    char                *rd_path;   /* Name of mapped file */
    char                *rd_start;  /* Virtual start for region in mem */
    char                *rd_end;    /* Virtual end for region in mem */
    unsigned long        rd_length; /* Virtual length for region in mem */
    unsigned long        rd_offset; /* Offset for region in file */

    syms_key_t rd_symkey;           /* Symbols control block for this region */
    int        rd_syminit;          /* Flag to not reopen symbol file */
    int        rd_kerntext;         /* Is this kernel text region? */
} region_t;

/* virtual address space head - list of mapped regions: user and kernel */
typedef struct _vas_data {
    region_t        *vh_fwd;         /* Forward link _region_data lo_vaddr */
    region_t        *vh_bck;         /* Backward link _region_data hi_vaddr */  
    struct _process *vh_proc;        /* Owner process struct */
    pid_t            vh_pid;         /* Owner pid */
    int              vh_entries;     /* How many maps */
} vh_t;

/* profile array of symbols structure */
typedef struct _symb_struct {
       region_t     *ss_regptr;      /* region data pointer      */
       char         *ss_sym_ptr;     /* Pointer to symbol info   */
       unsigned int  ss_cnt;         /* Hits to symbol name      */
       unsigned int  ss_icount;      /* Unique PC's hit          */
       unsigned long ss_value;       /* Value of symbol          */
       tNOSYMREASONS ss_noSymReason; /* Why are there no symbols */
} symb_t;

/* process list head */
typedef struct _proc_list_type {
    void        *pl_procs;            /* list of processes */
    unsigned int pl_numProcs;         /* total num of processes detected */
    unsigned int pl_numKthreads;      /* total num of kernel threads detected */
} proc_list_t;

/* symbol region names singly linked list */
typedef struct _name_list {
    char              *name;
    struct _name_list *next;
} name_list_t;

/* the process type */
typedef struct _process {
    char         *pr_path;           /* name of this proc */
    char         *pr_myKname;        /* cmd name assigned by kernel */
    pid_t         pr_myPid;          /* my pid */
    pid_t         pr_myParentPid;    /* my parent pid */
    pid_t         pr_myGroupPid;     /* my group pid number */
    pid_t         pr_mySid;          /* my session pid number */
    unsigned long pr_flags;          /* my flags */
    int           pr_isKthread;      /* is this a kernel thread */

    struct _process *pr_parent;      /* _process struct for parent */
    tPRBIRTHBYFLAG pr_birthBy;       /* Birth by exec, fork, spontanious */
    tPRENDBYFLAG  pr_endBy;          /* End by exec, exit,  still running */

    vh_t         *pr_vasHead;        /* the list of regions defining the vas */

    void         *pr_profile;        /* the list of pc hits */
    unsigned int  pr_profUniq;       /* unique pc hits */
    unsigned int  pr_profTot;        /* total hits */
    double        pr_usrRunMs;       /* extrapolated user run time */
    
    void         *pr_sysProfile;     /* kernel hits for this process */
    unsigned int  pr_sysProfUniq;    /* unique kernel hits for this proc */
    unsigned int  pr_sysProfTot;     /* total kernel hits for this proc */
    double        pr_sysRunMs;       /* extrapolated system run time */

    symb_t       *pr_syms_array;     /* base address of extracted syms */
    unsigned int  pr_syms_numb;      /* total entries in above */
    unsigned int  pr_syms_uniq;      /* unique entries in above */
    vh_t         *pr_syms_vh;        /* ptr list of regions defining the vas */
    unsigned int  pr_syms_nm_glob;   /* total global symbols to print */
    unsigned int  pr_syms_hits;      /* total pc hits in extracted profile */
    region_t     *pr_syms_region;    /* current region for symbols */
    name_list_t  *pr_syms_name_lst;  /* pointer to list of reg names */

    unsigned int  pr_exec_times;     /* how many times did this one exec */
    int           pr_is32bit;        /* is this process 32 or 64 bit */

    int           pr_is_mt;          /* Is this process multi-threaded? */
    void         *pr_threads;        /* List  of threads in this process */
    unsigned int  pr_nr_t_counted;   /* num of threads counted in this proc */
    unsigned int  pr_nr_t_created;   /* num of threads created by this proc */
    unsigned int  pr_nr_t_destroyed; /* num of threads destroyed by this proc */
    unsigned int  pr_nr_t_active;    /* num of threads currently active here */

    unsigned int   pr_ucpus_used;    /* how may cpus ran this thread user */
    unsigned long *pr_ucpu_log;      /* array of cpu and # of trace rec user */
    unsigned int   pr_kcpus_used;    /* how may cpus ran this thread kern */
    unsigned long *pr_kcpu_log;      /* array of cpu and # of trace rec kern */

} process_t;

/* kernel profile holding struct */
typedef struct _kernel_prof_type {
    void            *kp_profile;          /* Total kernel profile */
    unsigned int     kp_profUniq;
    unsigned int     kp_profTot; 
    unsigned int     kp_tot_cpus_used;
    unsigned long   *kp_tot_cpu_log;

    void            *kp_intrProfile;      /* profile due to interrupt context */
    unsigned int     kp_intrProfUniq;
    unsigned int     kp_intrProfTot; 
    unsigned int     kp_intr_cpus_used;
    unsigned long   *kp_intr_cpu_log;

    void            *kp_kthreadProfile;   /* profile due to kernel threads */
    unsigned int     kp_kthreadProfUniq;
    unsigned int     kp_kthreadProfTot; 
    unsigned int     kp_kthread_cpus_used;
    unsigned long   *kp_kthread_cpu_log;
    
    void            *kp_usrProfile;       /* profile due to user processes */
    unsigned int     kp_usrProfUniq;
    unsigned int     kp_usrProfTot; 
    unsigned int     kp_usr_cpus_used;
    unsigned long   *kp_usr_cpu_log;
} kernel_prof_t;

/* The kernel information place */
typedef struct _kernel_struct_type {
    vh_t            k_vas;          /* kernel text and module regions */
    kernel_prof_t   k_prof;         /* the profile dtrees */
    char           *k_path;         /* Which kernel are we running */
} kernel_struct_t;

/*
 * Stats enumeration for stats array.
 */
enum {
    PRO_BUFFERS_READ,
    PRO_FLUSHES,
    PRO_NOTIFICATIONS,
    PRO_UNKNOWN_NOTES,
    PRO_EXECS,
    PRO_FORKS,
    PRO_MAPS,
    PRO_INCOMPLETE_MAPS,
    PRO_DROP_MODULES,
    PRO_EXITS,
    PRO_SAMPLES,
    PRO_NUM_SAMPLE_TRACES,
    PRO_SYS_HITS,
    PRO_MOD_HITS,
    PRO_USR_HITS,
    PRO_SELF_HITS,
    PRO_UNKNOWN_HITS,
    PRO_GATE_HITS,
    PRO_MAX_STATS
};

/*
 * ------------------------ Exported global data ----------------------------
 */
extern double gFirstTrace_ms;         /* Time stamp of first trace seen */
extern double gPrevTrace_ms;          /* Time stamp of previous trace seen */
extern double gCurrentTrace_ms;       /* Time stamp of the current trace */

extern proc_list_t gProclist;         /* The process list */
extern kernel_struct_t gK;            /* The kernel place */

extern sigset_t gSigblockmask;         /* Signals we handle and want to blk */
extern int gCurrcpu;                   /* Current Spu for the buffer */

extern clock_t gBeforepoint;           /* Point of child exec */
extern struct tms gBeforetimes;        /* usr/sys time before child exec */
extern clock_t gAfterpoint;            /* Point of child exit */
extern struct tms gAftertimes;         /* usr/sys times after child exit */
  
extern int gDoGettimeofday;            /* Use gettimeofday for timestamps */
extern unsigned long gPstats[PRO_MAX_STATS];  /* Prospect stats counters */

extern unsigned long gKernelVmOffset;  /* Virtual memory offset for kernel */

/*
 * ------------------------------ Prototypes --------------------------------
 */
#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

process_t * getproc_by_pid_cached(pid_t pid, int refresh);
void putproc_by_pid(pid_t pid, process_t *p);
process_t * alloc_proc(pid_t pid);
void alloc_k_prof(void);
int read_proc_for_pid(process_t *p);
int read_system_map(void);
unsigned long read_proc_ksyms(void);
region_t * get_kr_by_name(char *name);
unsigned int read_uint_from_proc(char *filen);
void write_uint_to_proc(char *filen, unsigned int val);
int make_realtime(int pri);
int try_fixup_module_path(region_t *r);

#ifdef __cplusplus
}
#endif /* __cplusplus */



#endif  /* !LINUX_MODEL_H_INCLUDED */

