/*
 * iconaction.cpp - the QAction subclass that uses Icons and supports animation
 * Copyright (C) 2003  Michail Pishchagin
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include "iconaction.h"

#include "iconwidget.h"
#include "iconset.h"

#include <qpopupmenu.h>

//----------------------------------------------------------------------------
// IconAction
//----------------------------------------------------------------------------

class IconAction::Private : public QObject
{
	Q_OBJECT
public:
	QPtrList<IconToolButton> buttons;
	Icon *icon;
	IconAction *action;

	Private(IconAction *act) {
		icon = 0;
		action = act;
	}

public slots:
	void menuClicked() {
		action->setOn( !action->isOn() );
		emit action->setOn( action->isOn() );
		emit action->activated(); // required for our lovely status change actions
	}
};

IconAction::IconAction(const QString &text, const QString &icon, const QString &menuText, QKeySequence accel, QObject *parent, const char *name, bool toggle)
: QAction(text, menuText, accel, parent, name, toggle)
{
	d = new Private(this);

	setIcon(icon);
}

IconAction::IconAction(const QString &text, const QString &menuText, QKeySequence accel, QObject *parent, const char *name, bool toggle)
: QAction(text, menuText, accel, parent, name, toggle)
{
	d = new Private(this);
}

IconAction::~IconAction()
{
	// delete the buttons list before our own destruction
	d->buttons.setAutoDelete(true);
	d->buttons.clear();

	delete d;
}

void IconAction::setIcon(const Icon *i)
{
	if ( d->icon ) {
		disconnect(d->icon, 0, this, 0 );
		d->icon->stop();
		delete d->icon;
		d->icon = 0;
	}

	QIconSet is;
	if ( i ) {
		d->icon = new Icon(*i);
		connect(d->icon, SIGNAL(iconModified(const QPixmap &)), SLOT(iconUpdated(const QPixmap &)));
		d->icon->activated(true);

		is = d->icon->iconSet();
	}

	QAction::setIconSet( is );

	for ( QPtrListIterator<IconToolButton> it(d->buttons); it.current(); ++it ) {
		IconToolButton *btn = it.current();
		btn->setIcon ( d->icon );
	}
}

void IconAction::setIcon(const QString &name)
{
	setIcon( IconsetFactory::iconPtr(name) );
}

const QString &IconAction::iconName() const
{
	if ( d->icon )
		return d->icon->name();
	return QString::null;
}

bool IconAction::addTo(QWidget *w)
{
	if ( w->inherits("QToolBar") ) {
		QCString bname = name() + QCString("_action_button");
		IconToolButton *btn = new IconToolButton ( w, bname );
		d->buttons.append ( btn );

		btn->setToggleButton ( isToggleAction() );
		btn->setOn( isOn() );
		btn->setTextLabel ( text() );
		btn->setIcon ( d->icon, false );
		btn->setEnabled ( isEnabled() );
		addingToolButton(btn);

		connect(btn, SIGNAL(clicked()), this, SIGNAL(activated()));
		connect(btn, SIGNAL(toggled(bool)), this, SLOT(setOn(bool)));

		connect(btn, SIGNAL(destroyed()), SLOT(objectDestroyed()));
	}
	else if ( w->inherits("QPopupMenu") ) { // i have added this because of many mysterious segfaults
		if ( !isVisible() )
			return true;

		QPopupMenu *popup = (QPopupMenu *)w;
		int id = popup->insertItem (iconSet(), menuText());
		popup->setItemEnabled(id, isEnabled());
		popup->setWhatsThis(id, whatsThis());
		if ( isToggleAction() ) {
			popup->setCheckable( true );
			popup->setItemChecked(id, isOn());
			popup->connectItem(id, d, SLOT(menuClicked()));
		}
		else
			popup->connectItem(id, this, SIGNAL(activated()));

		if ( !accel().isEmpty() )
			popup->changeItem(id, iconSet(), menuText() + '\t' + (QString)accel());

	}
	else
		return QAction::addTo(w);

	return true;
}

void IconAction::objectDestroyed()
{
	const QObject *obj = sender();
	d->buttons.removeRef((IconToolButton *)obj);
}

void IconAction::setOn(bool b)
{
	QAction::setOn(b);
	for ( QPtrListIterator<IconToolButton> it(d->buttons); it.current(); ++it ) {
		IconToolButton *btn = it.current();
		btn->setOn(b);
	}
}

void IconAction::toolButtonToggled(bool b)
{
	setOn(b);
}

void IconAction::setEnabled(bool e)
{
	QAction::setEnabled(e);
	for ( QPtrListIterator<IconToolButton> it(d->buttons); it.current(); ++it ) {
		IconToolButton *btn = it.current();
		btn->setEnabled (e);
	}
}

void IconAction::setText(const QString &t)
{
	QAction::setText(t);
	for ( QPtrListIterator<IconToolButton> it(d->buttons); it.current(); ++it ) {
		IconToolButton *btn = it.current();
		btn->setTextLabel(t);
	}
}

QPtrList<IconToolButton> IconAction::buttonList()
{
	return d->buttons;
}

void IconAction::iconUpdated(const QPixmap &)
{
	QAction::setIconSet( d->icon->iconSet() );
}

#include "iconaction.moc"
