/*
 * mainwin_p.cpp - classes used privately by the main window.
 * Copyright (C) 2001-2003  Justin Karneges, Michail Pishchagin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include "common.h"
#include "mainwin_p.h"

#include <qstyle.h>
#include <qtoolbar.h>
#include <qtimer.h>
#include <qsignalmapper.h>
#include <qobjectlist.h>
#include "psiaccount.h"
#include "iconwidget.h"
#include "alerticon.h"

//----------------------------------------------------------------------------
// PopupActionButton
//----------------------------------------------------------------------------

class PopupActionButton : public QPushButton
{
	Q_OBJECT
public:
	PopupActionButton(QWidget *parent = 0, const char *name = 0);
	~PopupActionButton();

	void setIcon(Icon *, bool showText);

private slots:
	void pixmapUpdated(const QPixmap &);

private:
	void drawButtonLabel(QPainter *p);
	bool hasToolTip;
	Icon *icon;
	bool showText;
};

PopupActionButton::PopupActionButton(QWidget *parent, const char *name)
: QPushButton(parent, name)
{
	setWFlags(getWFlags() | WRepaintNoErase);

	hasToolTip = false;
	icon = 0;
}

PopupActionButton::~PopupActionButton()
{
	if (icon)
		icon->stop();
}

void PopupActionButton::setIcon(Icon *i, bool st)
{
	if ( icon ) {
		icon->stop();
		disconnect (icon, 0, this, 0);
		icon = 0;
	}

	icon = i;

	showText = st;
	pixmapUpdated(i->pixmap());

	connect(icon, SIGNAL(pixmapChanged(const QPixmap &)), SLOT(pixmapUpdated(const QPixmap &)));
	icon->activated();
}

void PopupActionButton::pixmapUpdated(const QPixmap &pix)
{
	if ( showText )
		setIconSet(pix);
	else
		setPixmap(pix);
}

void PopupActionButton::drawButtonLabel(QPainter *p)
{
	// code by Justin Karneges :-)
	// crazy code ahead!  watch out for potholes and deer.

	// this gets us the width of the "text area" on the button.
	// adapted from qt/src/styles/qcommonstyle.cpp and qt/src/widgets/qpushbutton.cpp
	QRect r = style().subRect(QStyle::SR_PushButtonContents, this);
	if(isMenuButton())
		r.setWidth(r.width() - style().pixelMetric(QStyle::PM_MenuButtonIndicator, this));
	if(iconSet() && !iconSet()->isNull())
		r.setWidth(r.width() - (iconSet()->pixmap(QIconSet::Small, QIconSet::Normal, QIconSet::Off).width() + 4));

	// font metrics
	QFontMetrics fm(font());

	// w1 = width of button text, w2 = width of text area
	int w1 = fm.width(text());
	int w2 = r.width();

	// backup original text
	QString oldtext = text();

	// button text larger than what will fit?
	if(w1 > w2) {
		if( !hasToolTip ) {
			QToolTip::add(this, text());
			hasToolTip = TRUE;
		}

		// make a string that fits
		bool found = FALSE;
		QString newtext;
		int n;
		for(n = oldtext.length(); n > 0; --n) {
			if(fm.width(oldtext, n) < w2) {
				found = TRUE;
				break;
			}
		}
		if(found)
			newtext = oldtext.mid(0, n);
		else
			newtext = "";

		// set the new text that fits.  updates must be off, or we recurse.
		setUpdatesEnabled(FALSE);
		QButton::setText(newtext);
		setUpdatesEnabled(TRUE);
	}
	else {
		if( hasToolTip ) {
			QToolTip::remove(this);
			hasToolTip = FALSE;
		}
	}

	// draw!
	QPushButton::drawButtonLabel(p);

	// restore original button text now that we are done drawing.
	setUpdatesEnabled(FALSE);
	QButton::setText(oldtext);
	setUpdatesEnabled(TRUE);
}

//----------------------------------------------------------------------------
// PopupAction -- the IconButton with popup or QPopupMenu
//----------------------------------------------------------------------------

class PopupAction::Private : public QObject
{
public:
	QPopupMenu *menu;
	QSizePolicy size;
	QPtrList<PopupActionButton> buttons;
	Icon *icon;
	bool showText;

	Private (QPopupMenu *_menu, QObject *parent)
	: QObject (parent)
	{
		menu = _menu;
		icon = 0;
		showText = true;
	}

	~Private()
	{
		buttons.clear();
		if (icon)
			delete icon;
	}
};

PopupAction::PopupAction (const QString &label, QPopupMenu *_menu, QObject *parent, const char *name)
: QAction (label, label, 0, parent, name)
{
	d = new Private (_menu, this);
}

void PopupAction::setSizePolicy (const QSizePolicy &p)
{
	d->size = p;
}

void PopupAction::setAlert (const Icon *icon)
{
	setIcon(icon, d->showText, true);
}

void PopupAction::setIcon (const Icon *icon, bool showText, bool alert)
{
	Icon *oldIcon = 0;
	if ( d->icon ) {
		oldIcon = d->icon;
		d->icon = 0;
	}

	d->showText = showText;

	if ( icon ) {
		if ( !alert )
			d->icon = new Icon(*icon);
		else
			d->icon = new AlertIcon(icon);

		QAction::setIconSet(*icon);
	}
	else {
		d->icon = 0;
		QAction::setIconSet(QIconSet());
	}

	for ( QPtrListIterator<PopupActionButton> it(d->buttons); it.current(); ++it ) {
		PopupActionButton *btn = it.current();
		btn->setIcon (d->icon, showText);
	}

	if ( oldIcon ) {
		delete oldIcon;
	}
}

void PopupAction::setText (const QString &text)
{
	QAction::setText (text);
	for ( QPtrListIterator<PopupActionButton> it(d->buttons); it.current(); ++it ) {
		PopupActionButton *btn = it.current();
		btn->setText (text);
	}
}

bool PopupAction::addTo (QWidget *w)
{
	if ( w->inherits("QPopupMenu") ) {
		QPopupMenu *m = (QPopupMenu*)w;
		m->insertItem (iconSet(), menuText(), d->menu);
	}
	else if ( w->inherits("QToolBar") ) {
		QCString bname = name() + QCString("_action_button");
		PopupActionButton *btn = new PopupActionButton ( (QToolBar*)w, bname );
		d->buttons.append ( btn );
		btn->setPopup ( d->menu );
		btn->setText ( text() );
		btn->setIcon ( d->icon, d->showText );
		btn->setSizePolicy ( d->size );
		btn->setEnabled ( isEnabled() );

		connect( btn, SIGNAL( destroyed() ), SLOT( objectDestroyed() ) );
	}
	else
		return false;

	return true;
}

void PopupAction::objectDestroyed ()
{
	const QObject *obj = sender();
	d->buttons.removeRef( (PopupActionButton *) obj );
}

void PopupAction::setEnabled (bool e)
{
	QAction::setEnabled (e);
	for ( QPtrListIterator<PopupActionButton> it(d->buttons); it.current(); ++it ) {
		PopupActionButton *btn = it.current();
		btn->setEnabled (e);
	}
}

//----------------------------------------------------------------------------
// MLabel -- a clickable label
//----------------------------------------------------------------------------

MLabel::MLabel(QWidget *parent, const char *name)
:QLabel(parent, name)
{
	setMinimumWidth(48);
	setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Preferred);
	setFrameStyle(QFrame::Panel | QFrame::Sunken);
}

void MLabel::mouseReleaseEvent(QMouseEvent *e)
{
	emit clicked(e->button());
	e->ignore();
}

void MLabel::mouseDoubleClickEvent(QMouseEvent *e)
{
	if(e->button() == LeftButton)
		emit doubleClicked();

	e->ignore();
}

//----------------------------------------------------------------------------
// MTray
//----------------------------------------------------------------------------

class MTray::Private
{
public:
	Private() {
		icon	= 0;
		ti	= 0;
	}

	~Private() {
		if ( icon )
			delete icon;

		delete ti;
	}

	TrayIcon *ti;
	Icon *icon;

	QPixmap makeIcon(const QImage &_in);
};

MTray::MTray(const QImage &icon, const QString &tip, QPopupMenu *popup, QObject *parent)
:QObject(parent)
{
	d = new Private;

	d->ti = new TrayIcon(d->makeIcon(icon), tip, popup);
	d->ti->setWMDock(option.isWMDock);
	connect(d->ti, SIGNAL(clicked(const QPoint &, int)), SIGNAL(clicked(const QPoint &, int)));
	connect(d->ti, SIGNAL(doubleClicked(const QPoint &)), SIGNAL(doubleClicked(const QPoint &)));
	connect(d->ti, SIGNAL(closed()), SIGNAL(closed()));
	d->ti->show();
}

MTray::~MTray()
{
	delete d;
}

void MTray::setToolTip(const QString &str)
{
	d->ti->setToolTip(str);
}

void MTray::setIcon(const Icon *icon, bool alert)
{
	if ( d->icon ) {
		disconnect(d->icon, 0, this, 0 );
		d->icon->stop();

		delete d->icon;
		d->icon = 0;
	}

	if ( icon ) {
		if ( !alert )
			d->icon = new Icon(*icon);
		else
			d->icon = new AlertIcon(icon);

		connect(d->icon, SIGNAL(pixmapChanged(const QPixmap &)), SLOT(animate()));
		d->icon->activated();
	}
	else
		d->icon = new Icon();

	animate();
}

void MTray::setAlert(const Icon *icon)
{
	setIcon(icon, true);
}

bool MTray::isAnimating() const
{
	return d->icon->isAnimated();
}

bool MTray::isWMDock()
{
	return d->ti->isWMDock();
}

void MTray::show()
{
	d->ti->show();
}

void MTray::hide()
{
	d->ti->hide();
}

QPixmap MTray::Private::makeIcon(const QImage &_in)
{
#ifdef Q_WS_X11
	/*if(1) {
		QImage real = wmdock[0];
		real.detach();
		QImage in = _in;
		in.detach();

		// make sure it is no bigger than 16x16
		if(in.width() > 16 || in.height() > 16)
			in = in.smoothScale(16,16);

		int xo = 40;
		int yo = 40;

		int n, n2;

		// draw a border
		for(n = 0; n < 22; ++n) {
			real.setPixel(n + xo, yo, qRgb(0x80,0x80,0x80));
			real.setPixel(n + xo, yo + 21, qRgb(0x00,0x00,0x00));
			real.setPixel(xo, yo + n, qRgb(0x80,0x80,0x80));
			real.setPixel(xo + 21, yo + n, qRgb(0x00,0x00,0x00));
		}

		xo += 3;
		yo += 3;

		// draw a dropshadow
		for(n2 = 0; n2 < in.height(); ++n2) {
			for(n = 0; n < in.width(); ++n) {
				if(qAlpha(_in.pixel(n,n2))) {
					real.setPixel(n + xo + 2, n2 + yo + 2, qRgb(0x80,0x80,0x80));
				}
			}
		}
		for(n2 = 0; n2 < in.height(); ++n2) {
			for(n = 0; n < in.width(); ++n) {
				if(qAlpha(_in.pixel(n,n2))) {
					real.setPixel(n + xo, n2 + yo, _in.pixel(n,n2));
				}
			}
		}

		QPixmap icon;
		icon.convertFromImage(real);
		return icon;
	}
	else {*/
		// on X11, the KDE dock is 22x22.  let's make our icon "seem" bigger.
		QImage real(22,22,32);
		//QImage in = _in.convertToImage();
		QImage in = _in;
		in.detach();
		real.fill(0);
		real.setAlphaBuffer(true);

		// make sure it is no bigger than 16x16
		if(in.width() > 16 || in.height() > 16)
			in = in.smoothScale(16,16);

		int xo = (real.width() - in.width()) / 2;
		int yo = (real.height() - in.height()) / 2;

		int n, n2;

		// draw a border
		/*for(n = 0; n < 22; ++n) {
			real.setPixel(n,  0, qRgb(0x80,0x80,0x80));
			real.setPixel(n, 21, qRgb(0x00,0x00,0x00));
			real.setPixel( 0, n, qRgb(0x80,0x80,0x80));
			real.setPixel(21, n, qRgb(0x00,0x00,0x00));
		}*/

		// draw a dropshadow
		for(n2 = 0; n2 < in.height(); ++n2) {
			for(n = 0; n < in.width(); ++n) {
				if(qAlpha(in.pixel(n,n2))) {
					int x = n + xo + 2;
					int y = n2 + yo + 2;
					real.setPixel(x, y, qRgba(0x80,0x80,0x80,0xff));
				}
			}
		}
		// draw the image
		for(n2 = 0; n2 < in.height(); ++n2) {
			for(n = 0; n < in.width(); ++n) {
				if(qAlpha(in.pixel(n,n2))) {
					QRgb c = in.pixel(n, n2);
					real.setPixel(n + xo, n2 + yo, qRgba(qRed(c), qGreen(c), qBlue(c), 0xff));
				}
			}
		}
		// create the alpha layer
		for(n2 = real.height()-2; n2 >= 0; --n2) {
			for(n = 0; n < real.width(); ++n) {
				uint c = real.pixel(n, n2);
				if(c > 0) {
					QRgb old = real.pixel(n, n2);
					real.setPixel(n, n2, qRgba(qRed(old), qGreen(old), qBlue(old), 0xff));
				}
			}
		}

		QPixmap icon;
		icon.convertFromImage(real);
		return icon;
	//}
#else
	QPixmap icon;
	icon.convertFromImage(_in);
	return icon;
#endif
}

void MTray::animate()
{
	d->ti->setIcon(d->makeIcon(d->icon->image()));
}

//----------------------------------------------------------------------------
// MAction
//----------------------------------------------------------------------------

class MAction::Private : public QObject
{
public:
	int id;
	PsiCon *psi;
	QSignalMapper *sm;

	Private (int _id, PsiCon *_psi, QObject *parent)
	: QObject (parent)
	{
		id = _id;
		psi = _psi;
		sm = new QSignalMapper(this, "MAction::Private::SignalMapper");
	}

	QPopupMenu *subMenu(QWidget *p)
	{
		QPopupMenu *pm = new QPopupMenu (p);
		uint i = 0;
		for ( PsiAccountListIt it(psi->accountList(TRUE)); it.current(); ++it, i++ )
		{
			PsiAccount *acc = it.current();
			pm->insertItem( acc->name(), parent(), SLOT(itemActivated(int)), 0, id*1000 + i );
			pm->setItemParameter ( id*1000 + i, i );
		}
		return pm;
	}
};

MAction::MAction(Icon i, const QString &s, int id, PsiCon *psi, QObject *parent)
: IconAction(s, s, 0, parent)
{
	init (i, id, psi);
}

MAction::MAction(const QString &s, int id, PsiCon *psi, QObject *parent)
: IconAction(s, s, 0, parent)
{
	init (Icon(), id, psi);
}

void MAction::init(Icon i, int id, PsiCon *psi)
{
	d = new Private(id, psi, this);
	setIcon (&i);
	connect(psi, SIGNAL(accountCountChanged()), SLOT(numAccountsChanged()));
	setEnabled ( !d->psi->accountList(TRUE).isEmpty() );
	connect (d->sm, SIGNAL(mapped(int)), SLOT(itemActivated(int)));
}

bool MAction::addTo(QWidget *w)
{
	if ( w->inherits("QPopupMenu") )
	{
		QPopupMenu *menu = (QPopupMenu*)w;
		if ( d->psi->accountList(TRUE).count() < 2 ) {
			menu->insertItem ( iconSet(), menuText(), this, SLOT(itemActivated(int)), 0, d->id*1000 + 0 );
			menu->setItemEnabled (d->id*1000 + 0, isEnabled());
			menu->setItemParameter ( d->id*1000 + 0, 0 );
		}
		else
			menu->insertItem(iconSet(), menuText(), d->subMenu(w));
	}
	else
		return IconAction::addTo(w);

	return true;
}

void MAction::addingToolButton(IconToolButton *btn)
{
	connect (btn, SIGNAL(clicked()), d->sm, SLOT(map()));
	d->sm->setMapping (btn, 0);

	if ( d->psi->accountList(TRUE).count() >= 2 )
		btn->setPopup ( d->subMenu(btn) );
	else
		btn->setPopup (0);

	btn->setPopupDelay (1); // the popup will be displayed immediately

	QString tt, str = menuText();
	for (int i = 0; i < (int)str.length(); i++)
		if ( str[i] == '&' && str[i+1] != '&' )
			continue;
		else
			tt += str[i];

	QToolTip::add (btn, tt);
}

void MAction::itemActivated(int n)
{
	PsiAccountList list = (PsiAccountList)d->psi->accountList(TRUE);

	if (n >= (int)list.count()) // just in case
		return;

	emit activated((PsiAccount *)list.at(n), d->id);
}

void MAction::numAccountsChanged()
{
	setEnabled ( !d->psi->accountList(TRUE).isEmpty() );

	QPtrList<IconToolButton> btns = buttonList();
	for ( QPtrListIterator<IconToolButton> it(btns); it.current(); ++it ) {
		QToolButton *btn = it.current();

		if ( btn->popup() )
			delete btn->popup();
		btn->setPopup (0);

		if ( d->psi->accountList(TRUE).count() >= 2 )
			btn->setPopup ( d->subMenu(btn) );
	}
}

//----------------------------------------------------------------------------
// SpacerAction
//----------------------------------------------------------------------------

class SpacerAction::Private
{
public:
	Private() { }

	class StretchWidget : public QWidget
	{
	public:
		StretchWidget(QWidget *parent)
		: QWidget(parent)
		{
			setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Preferred);
		}
	};
};

SpacerAction::SpacerAction(QObject *parent, const char *name)
: QAction(parent, name)
{
	d = new Private;
	setText(tr("<Spacer>"));
	setMenuText(tr("<Spacer>"));
}

SpacerAction::~SpacerAction()
{
	delete d;
}

bool SpacerAction::addTo(QWidget *w)
{
	if ( w->inherits("QToolBar") ) {
		new Private::StretchWidget(w);
		return true;
	}

	return false;
}

//----------------------------------------------------------------------------
// EventNotifierAction
//----------------------------------------------------------------------------

class EventNotifierAction::Private
{
public:
	Private() { }

	QPtrList<MLabel> labels;
	bool hide;
};

EventNotifierAction::EventNotifierAction(QObject *parent, const char *name)
: QAction(parent, name)
{
	d = new Private;
	setMenuText(tr("<Event notifier>"));
	d->hide = true;
}

EventNotifierAction::~EventNotifierAction()
{
	delete d;
}

bool EventNotifierAction::addTo(QWidget *w)
{
	if ( w->inherits("QToolBar") ) {
		MLabel *label = new MLabel(w, "EventNotifierAction::MLabel");
		label->setText(text());
		d->labels.append(label);
		connect(label, SIGNAL(destroyed()), SLOT(objectDestroyed()));
		connect(label, SIGNAL(doubleClicked()), SIGNAL(activated()));
		connect(label, SIGNAL(clicked(int)), SIGNAL(clicked(int)));

		if ( d->hide )
			hide();

		return true;
	}

	return false;
}

void EventNotifierAction::setText(const QString &t)
{
	QAction::setText("<nobr>" + t + "</nobr>");

	QPtrListIterator<MLabel> it ( d->labels );
	for ( ; it.current(); ++it) {
		MLabel *label = it.current();
		label->setText(text());
	}
}

void EventNotifierAction::objectDestroyed()
{
	MLabel *label = (MLabel *)sender();
	d->labels.removeRef(label);
}

void EventNotifierAction::hide()
{
	d->hide = true;

	QPtrListIterator<MLabel> it ( d->labels );
	for ( ; it.current(); ++it) {
		MLabel *label = it.current();
		label->hide();
		QToolBar *toolBar = (QToolBar *)label->parent();

		QObjectList *l = toolBar->queryList( "QWidget" );
		int found = 0;

		QObjectListIt it( *l );
		QObject *obj;
		for ( ; (obj = it.current()); ++it) {
			if ( QCString(obj->name()).left(3) != "qt_" ) // misc internal Qt objects
				found++;
		}
		delete l;

		if ( found == 1 ) // only MLabel is on ToolBar
			toolBar->hide();
	}
}

void EventNotifierAction::show()
{
	d->hide = false;

	QPtrListIterator<MLabel> it ( d->labels );
	for ( ; it.current(); ++it) {
		MLabel *label = it.current();
		QToolBar *toolBar = (QToolBar *)label->parent();
		toolBar->show();
		label->show();
	}
}

#include "mainwin_p.moc"
