require 'puppet/provider/parsedfile'

tab = case Facter.value(:operatingsystem)
    when "Solaris": :suntab
    else
        :crontab
    end


Puppet::Type.type(:cron).provide(:crontab,
    :parent => Puppet::Provider::ParsedFile,
    :default_target => ENV["USER"] || "root",
    :filetype => tab
) do
    commands :crontab => "crontab"

    text_line :comment, :match => %r{^#}, :post_parse => proc { |hash|
        if hash[:line] =~ /Puppet Name: (.+)\s*$/
            hash[:name] = $1
        end
    }

    text_line :blank, :match => %r{^\s*$}

    text_line :environment, :match => %r{^\w+=}

    record_line :freebsd_special, :fields => %w{special command},
        :match => %r{^@(\w+)\s+(.+)$}, :pre_gen => proc { |hash|
            hash[:special] = "@" + hash[:special]
        }

    crontab = record_line :crontab, :fields => %w{minute hour monthday month weekday command},
        :optional => %w{minute hour weekday month monthday}, :absent => "*"

    class << crontab
        def numeric_fields
            fields - [:command]
        end
        # Do some post-processing of the parsed record.  Basically just
        # split the numeric fields on ','.
        def post_parse(details)
            numeric_fields.each do |field|
                if val = details[field] and val != :absent
                    details[field] = details[field].split(",")
                end
            end
        end

        # Join the fields back up based on ','.
        def pre_gen(details)
            numeric_fields.each do |field|
                if vals = details[field] and vals.is_a?(Array)
                    details[field] = vals.join(",")
                end
            end
        end


        # Add name and environments as necessary.
        def to_line(details)
            str = ""
            if details[:name]
                str = "# Puppet Name: %s\n" % details[:name]
            end
            if details[:environment] and details[:environment] != :absent
                details[:environment].each do |env|
                    str += env + "\n"
                end
            end

            str += join(details)
            str
        end
    end


    # Return the header placed at the top of each generated file, warning
    # users that modifying this file manually is probably a bad idea.
    def self.header
%{# HEADER This file was autogenerated at #{Time.now} by puppet.
# HEADER While it can still be managed manually, it is definitely notrecommended.
# HEADER Note particularly that the comments starting with 'Puppet Name' should
# HEADER not be deleted, as doing so could cause duplicate cron jobs.\n}
    end

    # See if we can match the hash against an existing cron job.
    def self.match(hash)
        model.find_all { |obj|
            obj.value(:user) == hash[:user] and obj.value(:command) == hash[:command]
        }.each do |obj|
            # we now have a cron job whose command exactly matches
            # let's see if the other fields match

            # First check the @special stuff
            if hash[:special]
                next unless obj.value(:special) == hash[:special]
            end

            # Then the normal fields.
            matched = true
            record_type(hash[:record_type]).fields().each do |field|
                next if field == :command
                if hash[field] and ! obj.value(field)
                    Puppet.info "Cron is missing %s: %s and %s" %
                        [field, hash[field].inspect, obj.value(field).inspect]
                    matched = false
                    break
                end

                if ! hash[field] and obj.value(field)
                    Puppet.info "Hash is missing %s: %s and %s" %
                        [field, obj.value(field).inspect, hash[field].inspect]
                    matched = false
                    break
                end

                # Yay differing definitions of absent.
                next if (hash[field] == :absent and obj.value(field) == "*")

                # Everything should be in the form of arrays, not the normal text.
                next if (hash[field] == obj.value(field))
                Puppet.info "Did not match %s: %s vs %s" %
                    [field, obj.value(field).inspect, hash[field].inspect]
                matched = false 
                break
            end
            next unless matched
            return obj
        end

        return false
    end

    # Collapse name and env records.
    def self.prefetch_hook(records)
        name = nil
        envs = []
        records.each { |record|
            case record[:record_type]
            when :comment:
                if record[:name]
                    name = record[:name]
                    record[:skip] = true
                end
            when :environment:
                if name
                    envs << record[:line]
                    record[:skip] = true
                end
            else
                if name
                    record[:name] = name
                    name = nil
                end
                unless envs.empty?
                    record[:environment] = envs
                    envs = []
                end
            end
        }.reject { |record| record[:skip] }
    end

    def self.to_file(records)
        text = super
        # Apparently Freebsd will "helpfully" add a new TZ line to every
        # single cron line, but not in all cases (e.g., it doesn't do it
        # on my machine).  This is my attempt to fix it so the TZ lines don't
        # multiply.
        if text =~ /(^TZ=.+\n)/
            tz = $1 
            text.sub!(tz, '')
            text = tz + text
        end
        return text
    end

    def user=(user)
        @property_hash[:user] = user
        @property_hash[:target] = user
    end

    def user
        @property_hash[:user] || @property_hash[:target]
    end
end

# $Id: crontab.rb 2440 2007-04-30 19:29:15Z luke $
