/* PureAdmin
 * Copyright (C) 2003 Isak Savo
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * Functions for communication with PureFTPd server
 * Some parts is taken from the pure-ftpwho utility distributed with PureFTPd,
 * copyright is in these cases held by their respective author(s)
 *
 * Copyright (C) 2003 Isak Savo
 */
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <netdb.h>
#include <sys/mman.h>
#include <dirent.h>
#include <string.h>
#include <errno.h>
#include <time.h>

#include "srv_comm.h"
#include "helper.h"
#include "cfg.h"
#include "globals.h"


gint32 _list_generate_unique_id (void);

static GList *cur_activities = NULL; 

void srv_terminate (void)
{
   GList *node = g_list_first (cur_activities);
   if (!node)
     return;
   do
   {
      g_free (node->data);
   } while ((node = node->next));

   g_list_free (cur_activities);
   cur_activities = NULL;
   return;
}

/* Returns the same data as srv_get_activities, only that the filesystem isn't checked for 
 * new or updated connections. This is used for actions on currently known connections (close connection etc.)
 */
GList *srv_get_activities_in_memory (void)
{
   return cur_activities;
}

gboolean srv_try_get_activities ()
{
   gchar *output, *cmd, *p;
   gint ret;
   gint exit_status;

   if (srv_vars.cmd_pure_ftpwho[0] == '\0')
   {
      pur_log_err ("Command 'pure-ftpwho' not found");
      return FALSE;
   }
   cmd = g_strdup_printf ("%s -s -H", srv_vars.cmd_pure_ftpwho);
   ret = g_spawn_command_line_sync (cmd,
				    &output,
				    NULL,
				    &exit_status,
				    NULL);
   pur_log_dbg ("Running command: [%s]", cmd);
   if (!ret)
   {
      pur_log_wrn ("Unable to execute '%s'", cmd);
      g_free (cmd);
      g_free (output);
      return FALSE;
   }
   g_free (cmd);
   if (exit_status)
   {
      pur_log_wrn ("Pure-ftpwho exited abnormally: %d", exit_status);
      pur_log_wrn ("Output was: [%s]", output);
      g_free (output);
      return FALSE;
   }
   g_free (output);
   return TRUE;
}


gchar **ftpwho_parse_line (gchar *str)
{
   gchar *ptr, **retval, *utf8_str;
   gchar tmp[1024];
   gint i = 0, entry_count = 0;

   ptr = str;
   
   if (!str)
     return NULL;
   retval = (gchar **) g_malloc0 (13 * sizeof (gchar **));
   memset (retval, 0, 13 * sizeof (gchar **));
   memset (tmp, '\0', sizeof (tmp));
   while (*ptr && entry_count < 12)
   {
      if (i >= sizeof (tmp))
      {
	 utf8_str = g_locale_to_utf8 (tmp, -1, NULL, NULL, NULL);
	 if (!utf8_str)
	   retval[entry_count++] = g_strdup (tmp);
	 else
	   retval[entry_count++] = utf8_str;

	 i = 0;
	 memset (tmp, 0, sizeof (tmp));
	 /* Increase ptr to the next entry in the string */
	 while (*ptr)
	 {
	    if (*ptr == '\\')
	      ptr++; /* Do not stop if we encounter an escaped '|' */
	    ptr++;
	    if (*ptr == '|')
	    {
	       ptr++;
	       break;
	    }
	 }
	 continue;
      }
      switch (*ptr)
      {
	 case '\\':
	   tmp[i++] = *(ptr + 1);
	   ptr++;
	   break;
	 case '|':
	   utf8_str = g_locale_to_utf8 (tmp, -1, NULL, NULL, NULL);
	   if (!utf8_str)
	     retval[entry_count++] = g_strdup (tmp);
	   else
	     retval[entry_count++] = utf8_str;
	   i = 0;
	   memset (tmp, '\0', sizeof(tmp));
	   break;
	 default:
	   tmp[i++] = *ptr;
	   break;
      }
      ptr++;
   }
   if (*tmp && i) /* Add the last entry to the array */
   {
      utf8_str = g_locale_to_utf8 (tmp, -1, NULL, NULL, NULL);
      if (!utf8_str)
	retval[entry_count++] = g_strdup (tmp);
      else
	retval[entry_count++] = utf8_str;
      
   }
   
   return retval;
}

GList *srv_get_activities (gboolean *error)
{
   gchar *c_stdout, *c_stderr;
   gchar **lines, **entries;
   gchar *cmd;
   gint exit_status, lineno;
   gboolean ret;
   PActivity *activity;
   
   if (error)
     *error = FALSE;

   /* Free the list first */
   g_list_foreach (cur_activities, (GFunc) g_free, NULL);
   g_list_free (cur_activities);
   cur_activities = NULL;

   c_stdout = c_stderr = NULL;
   cmd = g_strdup_printf ("%s -s %s", srv_vars.cmd_pure_ftpwho, cfg.resolve_hostnames ? "" : "-H");
   
   ret = g_spawn_command_line_sync (cmd,
				    &c_stdout,
				    NULL,
				    &exit_status,
				    NULL);
   if (!ret)
   {
      pur_log_wrn ("Unable to execute child: [%s]", cmd);
      if (error)
	*error = TRUE;
      g_free (cmd);
      return NULL;
   }
   g_free (cmd);
   /* Parse output */
   c_stdout = g_strchomp (c_stdout);
   
   lines = g_strsplit (c_stdout, "\n", 0);
   g_free (c_stdout);
   lineno = 0;
   while (lines[lineno])
   {
      entries = ftpwho_parse_line (lines[lineno]);
      if (!entries)
      {
	 lineno++;
	 continue;
      }
      /* Add new found entries to global list */
      activity = (PActivity *) g_malloc0 (sizeof (PActivity));
      
      if (entries[0])
	activity->pid = atoi (entries[0]);
      strncpy (activity->username, entries[1], sizeof (activity->username));
      activity->online_since = atoi (entries[2]);
      if (strncmp (entries[3], "IDLE", 4) == 0)
	activity->state = FTPWHO_STATE_IDLE;
      else if (strncmp (entries[3], " DL ", 4) == 0)
	activity->state = FTPWHO_STATE_DOWNLOAD;
      else if (strncmp (entries[3], " UL ", 4) == 0)
	activity->state = FTPWHO_STATE_UPLOAD;
      strncpy (activity->filename, entries[4], sizeof (activity->filename));
      strncpy (activity->remote_addr, entries[5], sizeof (activity->remote_addr));
      if (entries[8])
	activity->download_current_size = atoi (entries[8]);
      if (entries[9])
	activity->download_total_size = atoi (entries[9]);
      if (entries[10])
	activity->percentage = atoi (entries[10]);
      if (entries[11])
	activity->speed = atoi (entries[11]);
      activity->id = _list_generate_unique_id ();
      cur_activities = g_list_append (cur_activities, activity);

      g_strfreev (entries);
      lineno++;
   }
   g_strfreev (lines);
   return cur_activities;
}

PActivity *srv_get_activity_with_id (gint32 id)
{
   GList *node = g_list_first (cur_activities);
   PActivity *act;
   gboolean found = FALSE;

   while (node && !found)
   {
      act = (PActivity *) node->data;
      if (act->id == id)
	found = TRUE;
      node = g_list_next (node);
   }
   if (found)
     return act;
   else
     return NULL;
}

GList *srv_get_pids_for_user_host (const gchar *user, const gchar *host)
{
   GList *node = g_list_first (cur_activities);
   GList *ret = NULL;
   PActivity *act;
   
   if (!user || !host ||
       *user == '\0' || *host == '\0')
     return NULL;
   while (node)
   {
      act = (PActivity *) node->data;
      if (misc_str_is_equal (act->username, user) && 
	  misc_str_is_equal (act->remote_addr, host))
      {
	 ret = g_list_append (ret, GINT_TO_POINTER (act->pid));
      }
      node = g_list_next (node);
   }
   return ret;
}
    
gint lst_find_user (gconstpointer listdata, gconstpointer searchdata)
{
   gchar *haystack = (listdata ? ((POnlineUser *) listdata)->username : NULL);
   gchar *needle = (searchdata ? ((PActivity *) searchdata)->username : NULL);
   gint namematch, hostmatch;

   if (!haystack || !needle ||
       *haystack == 0 || *needle == 0)
     return 0;

   /* This is simple string matching */
   namematch = strncmp (haystack, (gchar *) needle, strlen (haystack));

   haystack = ((POnlineUser *) listdata)->remote_addr;
   needle = ((PActivity *) searchdata)->remote_addr;
   hostmatch = strncmp (haystack, needle, strlen (haystack));
   
   if (hostmatch == namematch && namematch == 0)
     return 0;		/* Absolute match */
   else if (hostmatch == 0)
     return namematch;   /* Same host, different user */
   else
     return hostmatch;  /* Different host, same user */
}

GList *srv_get_connected_users (GList *activities)
{
   GList *retval = NULL, *node;
   POnlineUser *user = NULL;
   
   activities = g_list_first (activities);
   while (activities)
   {
      if ((node = g_list_find_custom (retval, activities->data, (GCompareFunc) lst_find_user)))
	((POnlineUser *) node->data)->num_connections++;
      else
      {
	 user = (POnlineUser *) g_malloc0 (sizeof (POnlineUser));
	 memcpy (user->username, ((PActivity *) activities->data)->username, 
		 sizeof (((PActivity *) activities->data)->username));
	 memcpy (user->remote_addr, ((PActivity *) activities->data)->remote_addr, 
		 sizeof (((PActivity *) activities->data)->remote_addr));
	 user->num_connections = 1;
	 retval = g_list_append (retval, user);
      }
      activities = g_list_next (activities);
   }
   return g_list_first (retval);
}

/* Generates a random unique ID and returns it.
 * Theoreticly this could lead to a eternal loop, but it shouldn't
 * in practice! (Unless there are a couple of billion entries in the list)
 */
gint32 _list_generate_unique_id (void)
{
   GList *node;
   GRand *number;
   gint32 r_num, found;
   
   number = g_rand_new ();
   do
   {
      found = TRUE;
      /* generate a random number between 0 and 10 million */
      r_num = g_rand_int_range (number, 0, 10000000);
      node = g_list_first (cur_activities);
      while (node || !found)
      {
	 if (((PActivity *) node->data)->id == r_num)
	   found = FALSE;
	 node = g_list_next (node);
      }
   } while (!found);
   g_rand_free (number);
   return r_num;
}
