#-*- coding:utf-8 -*-
# cython: profile=False

#  Pybik -- A 3 dimensional magic cube game.
#  Copyright © 2009-2012  B. Clausius <barcc@gmx.de>
#
#  This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.

# Ported from GNUbik
# Original filename: glarea-common.c
# Original copyright and license: 2003  John Darrington, GPL3+

# no unicode_literals for compiled modules
from __future__ import print_function, division

# This line makes cython happy
global __name__, __package__    # pylint: disable=W0604
#px/__compiled = True
__compiled = False

# pylint: disable=W0614,W0401
#px/from gl cimport *
from OpenGL.GL import *
#px/from glu cimport *
from OpenGL.GLU import *
#px/from libc.math cimport cos, sin, tan, atan2, M_PI
from math import cos, sin, tan, atan2, pi as M_PI
#px/from _gldraw cimport *
from gldraw import *
# pylint: enable=W0614,W0401

from .debug import debug, DEBUG_DRAW

debug('Importing module:', __name__)
debug('  from package:', __package__)
debug('  compiled:', __compiled)


#px/cdef Matrix rotation_matrix
rotation_matrix = Matrix()

# jitter8 distribution for antialiasing
#     -7-5-3-1 1 3 5 7
#    +----------------+
#  7 |*1              |
#  5 |          *3    |
#  3 |    *2          |
#  1 |              *5|
# -1 |        *0      |
# -3 |  *7            |
# -5 |            *4  |
# -7 |      *6        |
#    +----------------+
jitter1 = [(0, 0)]
jitter2 = [(-1/4, 1/4), (1/4, -1/4)]
jitter3 = [( 0.0033922635, 0.3317967229),
           ( 0.2806016275,-0.2495619123),
           (-0.2738171062,-0.0868446388)]
# pylint: disable=C0324
jitter5 = [(0, 0), (-1/5,-2/5), ( 1/5, 2/5), ( 2/5,-1/5), (-2/5, 1/5)]
jitter8 = [( 1/16,-1/16), (-7/16, 7/16), (-3/16, 3/16), ( 3/16, 5/16),
           ( 5/16,-5/16), ( 7/16, 1/16), (-1/16,-7/16), (-5/16,-3/16)]
# pylint: enable=C0324
jitters = [jitter1, jitter2, jitter3, jitter5, jitter8]

#px/cdef struct Terrain:
class terrain: pass     # pylint: disable=W0232, C0321, R0903
    #px+float red
    #px+float green
    #px+float blue
    #px+bint lighting
    #px+int width
    #px+int height
#px+cdef Terrain terrain

#px/cdef struct Frustrum:
class frustrum:     # pylint: disable=W0232, R0903
    #px+double bounding_sphere_radius
    #px+double zoom
    #px+double cp_near
    #px+double cp_far
    #px/double proj_rect[4]
    proj_rect = [None]*4
    #px+int antialiasing
    #px+int njitter
    #px/double proj_rects_jitter[8][4]
    proj_rects_jitter = [[None]*4 for __i in range(8)]
    #px+bint multisample
#px+cdef Frustrum frustrum


### module state

def init_module():
    terrain.red = 0.0
    terrain.green = 0.0
    terrain.blue = 0.0
    terrain.lighting = False
    terrain.width = 1
    terrain.height = 1
    
    frustrum.bounding_sphere_radius = 1.
    frustrum.zoom = 1.
    frustrum.cp_near = 1.
    frustrum.cp_far = 2.
    #frustrum.proj_rect in _resize
    frustrum.antialiasing = 0
    frustrum.njitter = 0
    #frustrum.proj_rects_jitter in _resize
    frustrum.multisample = 0

#px/cdef void _update_projection():
def _update_projection():
    #px+cdef int min_dim
    #px+cdef double dx, dy, wsize
    min_dim = min(terrain.width, terrain.height)
    if terrain.width < terrain.height:
        aspectx = 1.
        aspecty = terrain.height / terrain.width
    else:
        aspectx = terrain.width / terrain.height
        aspecty = 1.
    frustrum.proj_rect[0] = -frustrum.zoom * aspectx
    frustrum.proj_rect[1] = +frustrum.zoom * aspectx
    frustrum.proj_rect[2] = -frustrum.zoom * aspecty
    frustrum.proj_rect[3] = +frustrum.zoom * aspecty
    for jitter in xrange(frustrum.njitter):
        wsize = 2 * frustrum.zoom
        dx = -jitters[frustrum.antialiasing][jitter][0] * wsize / min_dim
        dy = -jitters[frustrum.antialiasing][jitter][1] * wsize / min_dim
        frustrum.proj_rects_jitter[jitter][0] = dx - frustrum.zoom * aspectx
        frustrum.proj_rects_jitter[jitter][1] = dx + frustrum.zoom * aspectx
        frustrum.proj_rects_jitter[jitter][2] = dy - frustrum.zoom * aspecty
        frustrum.proj_rects_jitter[jitter][3] = dy + frustrum.zoom * aspecty
        
def set_frustrum(bounding_sphere_radius, zoom):
    fovy = 33.0  # field of view angle
    frustrum.bounding_sphere_radius = bounding_sphere_radius
    frustrum.zoom = frustrum.bounding_sphere_radius / zoom
    frustrum.cp_near = frustrum.zoom / tan(fovy * M_PI / 360.0)
    frustrum.cp_far = frustrum.cp_near + 2 * frustrum.bounding_sphere_radius
    _update_projection()
    
#px/cpdef set_lighting(int enable):
def set_lighting(enable):
    terrain.lighting = enable
        
def set_background_color(red, green, blue):
    terrain.red = red
    terrain.green = green
    terrain.blue = blue
    
def set_antialiasing(antialiasing, multisample=None):
    if antialiasing is not None:
        frustrum.antialiasing = antialiasing
    if frustrum.antialiasing < 0:
        frustrum.antialiasing = 0
    elif frustrum.antialiasing >= len(jitters):
        frustrum.antialiasing = len(jitters) - 1
    frustrum.njitter = len(jitters[frustrum.antialiasing])
    if multisample is not None:
        frustrum.multisample = multisample
    _update_projection()
    
#px/cdef void _set_rotation_matrix(p_Vector M, float radiansx, float radiansy):
def _set_rotation_matrix(M, radiansx, radiansy):
    #px+cdef float sx, sy, cx, cy
    #px+cdef float m00, m11, m12, m20
    
    sx = sin(radiansx/2)
    sy = sin(radiansy/2)
    cx = cos(radiansx/2)
    cy = cos(radiansy/2)
    
    m00 = 2*cx*cx - 1.
    m11 = 2*cy*cy - 1.
    m12 = 2*sy*cy
    m20 = 2*sx*cx
    # pylint: disable=C0321
    M[0][0] =  m00;         M[1][0] = 0.;   M[2][0] =  m20
    M[0][1] =  m12 * m20;   M[1][1] = m11;  M[2][1] = -m00 * m12
    M[0][2] = -m11 * m20;   M[1][2] = m12;  M[2][2] =  m00 * m11
    # py lint: enable=C0321
    
    M[0][3] = M[1][3] = M[2][3] = 0.
    M[3][0] = M[3][1] = M[3][2] = 0.
    M[3][3] = 1.
    
def set_rotation_xy(x, y):
    x %= 360
    # pylint: disable=C0321
    if y < -90:     y = -90
    elif y > 90:    y = 90
    _set_rotation_matrix(rotation_matrix, M_PI * x / 180.0, M_PI * y / 180.0)
    return x, y
    
### GL state

#px/cdef void _init_lighting():
def _init_lighting():
    #px/cdef float *light_ambient = [.2, .2, .2, 1.]
    light_ambient = [.2, .2, .2, 1.]
    #px/cdef float *light_diffuse = [1., 1., 1., 1.]
    light_diffuse = [1., 1., 1., 1.]
    ##px/cdef float *mat_diffuse = [.8, .8, .8, 1.]
    #mat_diffuse = [.8, .8, .8, 1.]
    # Controls intensity and color of specular reflections
    #px/cdef float *mat_specular = [.5, .6, .5, 1.]
    mat_specular = [.5, .6, .5, 1.]
    # Controls SPREAD of specular reflections (128.0 = small highlight)
    #px+cdef float mat_shininess
    mat_shininess = 60.
    # Directional light, at infinity (parameter 4, w=0.0), shining towards -Z
    #px/cdef float *light_position = [0., 0., 1., 0.]
    light_position = [0., 0., 1., 0.]
    #px/cdef float *lmodel_ambient = [.6, .6, .6, 1.]
    lmodel_ambient = [.6, .6, .6, 1.]
    
    glShadeModel(GL_SMOOTH)
    
    glLightfv(GL_LIGHT0, GL_AMBIENT, light_ambient)
    glLightfv(GL_LIGHT0, GL_DIFFUSE, light_diffuse)
    glLightfv(GL_LIGHT0, GL_POSITION, light_position)
    
    glLightModelfv(GL_LIGHT_MODEL_AMBIENT, lmodel_ambient)
    # Trying to light the insides of the cubies is a waste of time
    glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_FALSE)
    
    # This gives a rolling-sheen effect to the specularity.
    # Not all stickers on a face light up at once.
    glLightModeli(GL_LIGHT_MODEL_LOCAL_VIEWER, GL_TRUE)
    
    glEnable(GL_LIGHT0)
    
    glShadeModel(GL_FLAT)
    
    glMaterialfv(GL_FRONT, GL_SPECULAR, mat_specular)
    #glMaterialfv(GL_FRONT, GL_DIFFUSE, mat_diffuse)
    glMaterialf(GL_FRONT, GL_SHININESS, mat_shininess)
    glColorMaterial(GL_FRONT, GL_AMBIENT_AND_DIFFUSE)
    glEnable(GL_COLOR_MATERIAL)
    
    glEnable(GL_NORMALIZE)
    
def init_glarea():
    debug('GL Version:',
                #px+<char*>
                glGetString(GL_VERSION))
    glClearColor(0., 0., 0., 1.)
    glEnable(GL_DEPTH_TEST)
    glEnable(GL_CULL_FACE) # Rasterise only the exterior faces,  to speed things up
    glMatrixMode(GL_MODELVIEW)
    glLoadIdentity()
    _init_lighting()
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT)
    glClear(GL_ACCUM_BUFFER_BIT)
    glAccum(GL_RETURN, 1.0)
    
def resize(width, height):
    terrain.width = width
    terrain.height = height
    glViewport(0, 0, terrain.width, terrain.height)
    _update_projection()
    
### display functions

#px/cdef void _init_projection(double* proj_rect):
def _init_projection(proj_rect):
    glMatrixMode(GL_PROJECTION)
    glLoadIdentity()
    glFrustum(proj_rect[0], proj_rect[1],
              proj_rect[2], proj_rect[3],
              frustrum.cp_near, frustrum.cp_far)
            
#px/cdef void _init_modelview():
def _init_modelview():
    # Update viewer position in modelview matrix
    glMatrixMode(GL_MODELVIEW)
    glLoadIdentity()
    gluLookAt(0, 0, frustrum.bounding_sphere_radius + frustrum.cp_near,
              0, 0, 0,          # center
              0.0, 1.0, 0.0)    # up
    #px/glMultMatrixf(<float*>rotation_matrix)
    glMultMatrixf(rotation_matrix)
    
def display():
    #px+cdef int jitter
    
    if terrain.lighting:
        glEnable(GL_LIGHTING)
    else:
        glDisable(GL_LIGHTING)
    if frustrum.multisample:
        glEnable(GL_MULTISAMPLE)
    else:
        glDisable(GL_MULTISAMPLE)
    glClearColor(terrain.red, terrain.green, terrain.blue, 1.)
    _init_modelview()
    if frustrum.antialiasing:
        glClear(GL_ACCUM_BUFFER_BIT)
        for jitter in xrange(frustrum.njitter):
            glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT)
            _init_projection(frustrum.proj_rects_jitter[jitter])
            glMatrixMode(GL_MODELVIEW)
            draw_cube()
            glAccum(GL_ACCUM, 1.0/frustrum.njitter)
        glAccum(GL_RETURN, 1.0)
    else:
        glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT)
        _init_projection(frustrum.proj_rect)
        glMatrixMode(GL_MODELVIEW)
        draw_cube()
    if DEBUG_DRAW:
        draw_cube_debug()
    

### picking functions

def pick_polygons(x, y, granularity):
    #px+cdef int viewport[4]
    #px+cdef unsigned char pixel[3]
    
    #px/glGetIntegerv(GL_VIEWPORT, viewport)
    viewport = glGetIntegerv(GL_VIEWPORT)
    if not (0 <= x - viewport[0] < viewport[2] and 0 <= y - viewport[1] < viewport[3]):
        return 0
        
    glDisable(GL_LIGHTING)
        
    glMatrixMode(GL_PROJECTION)
    glPushMatrix()
    glLoadIdentity()
    gluPickMatrix(x, y, granularity, granularity, viewport)
    glFrustum(frustrum.proj_rect[0], frustrum.proj_rect[1],
              frustrum.proj_rect[2], frustrum.proj_rect[3],
              frustrum.cp_near, frustrum.cp_far)
    glClearColor(0., 0., 0., 1.)
    _init_modelview()
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT)
    
    pick_cube()
    #px/glReadPixels(x, y, 1, 1, GL_RGB, GL_UNSIGNED_BYTE, pixel)
    pixel = glReadPixels(x, y, 1, 1, GL_RGB, GL_UNSIGNED_BYTE, [[[0, 0, 0]]])[0][0]
    index = pixel[0]<<4 | pixel[1] | pixel[2]>>4
    
    glMatrixMode(GL_PROJECTION)
    glPopMatrix()
    
    return index
    
#px/cdef _modelview_to_viewport(int *viewport, Matrix proj, Matrix view, vvect):
def _modelview_to_viewport(viewport, proj, view, vvect):
    #px+cdef float u0, u1, u2, v0, v1, v3
    #px+cdef int w0, w1
    
    # u = V^T * vvect
    u0 = view[0][0]*vvect[0] + view[1][0]*vvect[1] + view[2][0]*vvect[2] + view[3][0]
    u1 = view[0][1]*vvect[0] + view[1][1]*vvect[1] + view[2][1]*vvect[2] + view[3][1]
    u2 = view[0][2]*vvect[0] + view[1][2]*vvect[1] + view[2][2]*vvect[2] + view[3][2]
    #u3 = 1.
    
    # v = P * u
    v0 = proj[0][0] * u0 + proj[1][0] * u1 + proj[2][0] * u2 + proj[3][0] #* u3
    v1 = proj[0][1] * u0 + proj[1][1] * u1 + proj[2][1] * u2 + proj[3][1] #* u3
    #v2 = proj[0][2] * u0 + proj[1][2] * u1 + proj[2][2] * u2 + proj[3][2] * u3
    v3 = proj[0][3] * u0 + proj[1][3] * u1 + proj[2][3] * u2 + proj[3][3] #* u3
    
    w0 = int((v0 / v3 + 1) / 2 * viewport[2])
    w1 = int((v1 / v3 + 1) / 2 * viewport[3])
    return w0, w1
    
def get_cursor_angle(point1, point2):
    '''The result is the angle (on the screen)
    at which the mouse cursor needs to be drawn.'''
    #px+cdef int viewport[4]
    #px+cdef Matrix view
    #px+cdef Matrix proj
    #px+cdef float v1,v2, w1,w2, angle
    
    #px/glGetIntegerv(GL_VIEWPORT, viewport)
    viewport = glGetIntegerv(GL_VIEWPORT)
    #px/glGetFloatv(GL_PROJECTION_MATRIX, <float*>proj)
    proj = glGetFloatv(GL_PROJECTION_MATRIX)
    #px/glGetFloatv(GL_MODELVIEW_MATRIX, <float*>view)
    view = glGetFloatv(GL_MODELVIEW_MATRIX)
    
    v1, v2 = _modelview_to_viewport(viewport, proj, view, point1)
    w1, w2 = _modelview_to_viewport(viewport, proj, view, point2)
    
    angle = atan2((v1-w1), (v2-w2)) * 180.0 / M_PI
    return angle, ((point1, point2), ((v1, v2), (w1, w2)))
    

