#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-
#
# TestThreadedINotify.py - tests cases
# Copyright (C) 2006  Sbastien Martini <sebastien.martini@gmail.com>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
# 02111-1307, USA.

import os, unittest, shutil, sys
import colorize
import coverage
import time

# coverage init
# early initialization, before the import of pyinotify
# (avoid to miss something)
coverage.erase()
coverage.start()

try:
    import autopath
    import src.pyinotify.pyinotify
    from src.pyinotify.pyinotify import *
    module_ = src.pyinotify.pyinotify
except ImportError:
    import pyinotify
    from pyinotify import *
    module_ = pyinotify

# verbose
module_.VERBOSE = True

# watch manager instance
WM = WatchManager()
# notifier instance and init
NOTIFIER = ThreadedNotifier(WM)
# start notifier's thread
NOTIFIER.start()


def all(seq, cond):
    """
    seq: iterable sequence
    cond: x -> bool
    """
    for i in seq:
        if not cond(i):
            return False
    return True


def all_positive(d):
    """
    d = {key: int}
    """
    return all(d, lambda x: d[x] > 0)


def all_true(d):
    """
    d = {key: int}
    """
    return all(d, lambda x: d[x])


def all_false(d):
    """
    d = {key: int}
    """
    return all(d, lambda x: not d[x])



class TestThreadedNotifier(unittest.TestCase):
    """
    test ThreadedNotifier class
    """

    def setUp(self):
        # initialization
        self._tempdir = os.path.join('/tmp/', 'test-pyinotify')
        # clean first
        self.tearDown()
        # directories creations
        os.mkdir(self._tempdir)
        # nested directories
        for i in range(10):
            outter = os.path.join(self._tempdir, 'out%d' % i)
            os.mkdir(outter)
            for j in range(10):
                os.mkdir(os.path.join(outter, 'in%d' % j))
        # temp files
        open(os.path.join(self._tempdir, 'out0ino.txt'), 'w').close()


    def tearDown(self):
        # removes temporaries
        try:
            shutil.rmtree(self._tempdir)
        except OSError, err:
            sys.stderr.write('%s\n' % err)


    def test_add_watch(self):
        """
        test add_watch method
        """
        mask_all = EventsCodes.ALL_EVENTS

        w1 = WM.add_watch(self._tempdir, mask_all)
        self.assertEqual(len(w1), 1)
        self.assert_(all_positive(w1))

        w2 = WM.add_watch([os.path.join(self._tempdir, 'out0'),
                           os.path.join(self._tempdir, 'out1/'),
                           os.path.join(self._tempdir, 'don-t-exist/'),
                           os.path.join(self._tempdir, 'out2')],
                          mask_all)
        self.assertEqual(len(w2), 3)

        w3 = WM.add_watch(os.path.join(self._tempdir,
                                       'tmp-dont-exist'),
                          mask_all)
        self.assertEqual(len(w3), 0)

        w4 = WM.add_watch(os.path.join(self._tempdir,
                                       'tmp-dont-exist'),
                          mask_all, rec=True)
        self.assertEqual(len(w4), 0)

        w5 = WM.add_watch(os.path.join(self._tempdir, 'out3'),
                          mask_all, rec=True)
        self.assertEqual(len(w5), 11)
        self.assert_(all_positive(w5))

        w6 = WM.add_watch(os.path.join(self._tempdir, 'out5'),0)
        self.assertEqual(len(w6), 1)
        self.assert_(w6[os.path.join(self._tempdir,'out5')] < 0)

        w7 = WM.rm_watch(WM.get_wd(self._tempdir),
                         rec=True)
        self.assert_(all_true(w7))


    def test_update_watch(self):
        """
        test update_watch method
        """
        mask_all = EventsCodes.ALL_EVENTS

        w1 = WM.add_watch(self._tempdir, mask_all)
        w2 = WM.update_watch(w1[self._tempdir])
        self.assert_(all_positive(w2))

        w3 = WM.update_watch(w1[self._tempdir], mask=1)
        self.assert_(all_true(w3))

        w4 = WM.update_watch(w1[self._tempdir],
                             proc_fun=ProcessEvent())
        self.assert_(all_true(w4))

        w5 = WM.update_watch(w1[self._tempdir],
                             proc_fun=ProcessEvent(),
                             mask=1)
        self.assert_(all_true(w5))

        w6 = WM.update_watch(4242, mask_all)
        self.assert_(all_false(w6))

        w7p = os.path.join(self._tempdir, 'out0ino.txt')
        w7 = WM.add_watch(w7p, mask=mask_all, rec=True)
        self.assertEqual(len(w7), 1)
        self.assert_(all_positive(w7))

        w8 = WM.update_watch(w7[w7p], mask=mask_all, rec=True)
        self.assertEqual(len(w8), 1)
        self.assert_(all_true(w8))

        w9 = WM.rm_watch(w7[w7p], rec=True)
        self.assertEqual(len(w9), 1)
        self.assert_(all_true(w9))

        # delete event may not have been already processed, thus the
        # watch can still be in the watch manager dict.
        time.sleep(2)
        w10 = WM.update_watch(w7[w7p], mask=mask_all, rec=True)
        self.assertEqual(len(w10), 0)

        w10b = WM.update_watch(w7[w7p], mask=mask_all)
        self.assertEqual(len(w10b), 1)
        self.assert_(all_false(w10b))

        w11p = os.path.join(self._tempdir, 'out3/')
        w11 = WM.add_watch(w11p, mask=mask_all, rec=True)
        self.assertEqual(len(w11), 11)
        self.assert_(all_positive(w11))

        w12 = WM.update_watch(WM.get_wd(os.path.join(w11p, 'in5/')),
                              mask=mask_all, rec=True)
        self.assertEqual(len(w12), 1)
        self.assert_(all_true(w12))

        w13 = WM.update_watch(WM.get_wd(w11p), rec=True)
        self.assertEqual(len(w13), 11)
        self.assert_(all_true(w13))

        w14 = WM.update_watch(WM.get_wd(w11p),
                              mask=1,
                              rec=True)
        self.assertEqual(len(w14), 11)
        self.assert_(all_true(w14))



        w15 = WM.update_watch(w11.values()+ \
                              [-100, 4525, 77545475745454, 0],
                              mask=1)
        self.assertEqual(len(w15), 15)
        self.assert_(not all_true(w15))
        del w15[-100]
        del w15[4525]
        del w15[77545475745454]
        del w15[0]
        self.assert_(all_true(w15))

        w16 = WM.update_watch(w11.values()+ \
                              [-100, 4525, 77545475745454, 0],
                              mask=1, rec=True)
        self.assertEqual(len(w16), 11)
        self.assert_(all_true(w16))

        w17 = WM.rm_watch(WM.get_wd(w11p), rec=True)
        self.assertEqual(len(w17), 11)
        self.assert_(all_true(w17))

        time.sleep(2)
        w18 = WM.update_watch(w17.keys(),
                              proc_fun=ProcessEvent())
        time.sleep(2)
        self.assertEqual(len(w18), 11)
        self.assert_(all_false(w18))

        w19 = WM.add_watch(self._tempdir, mask_all, rec=True)
        self.assert_(all_positive(w19))

        w20 = WM.rm_watch(WM.get_wd(self._tempdir),
                          rec=True)
        self.assert_(all_true(w20))


    def test_rm_watch(self):
        """
        test rm_watch method
        """
        mask_all = EventsCodes.ALL_EVENTS

        w1 = WM.rm_watch(4242)
        self.assertEqual(len(w1), 1)
        self.assert_(all_false(w1))

        w2 = WM.add_watch(self._tempdir, mask_all, rec=True)

        time.sleep(2)
        w3 = WM.rm_watch(WM.get_wd(os.path.join(self._tempdir, 'out0')), False)
        time.sleep(2)
        self.assertEqual(len(w3), 1)
        self.assert_(all_true(w3))
        self.assertEqual(WM.get_wd(os.path.join(self._tempdir, 'out0')), None) # fixme , too fast

        w4 = WM.rm_watch(WM.get_wd(os.path.join(self._tempdir, 'out1')), True)
        self.assertEqual(len(w4), 11)
        self.assert_(all_true(w4))

        w5 = WM.rm_watch(WM.get_wd(os.path.join(self._tempdir, 'out2/in0/')), False)
        self.assertEqual(len(w5), 1)
        self.assert_(all_true(w5))

        w6 = WM.rm_watch(WM.get_wd(os.path.join(self._tempdir, 'out2/in1/')), True)
        self.assertEqual(len(w6), 1)
        self.assert_(all_true(w6))

        w7 = WM.rm_watch(WM.get_wd(os.path.join(self._tempdir, 'out3')), True)
        self.assertEqual(len(w7), 11)
        self.assert_(all_true(w7))

        w8 = WM.add_watch(os.path.join(self._tempdir, 'out3'),
                          mask_all, rec=True)
        self.assertEqual(len(w8), 11)
        self.assert_(all_positive(w8))

        time.sleep(2)
        w9 = WM.rm_watch(w8.values(), rec=True)
        time.sleep(2)
        self.assertEqual(len(w9), 11)
        #self.assert_(all_true(w9)) #fixme

        w10 = WM.add_watch(os.path.join(self._tempdir, 'out3'),
                           mask_all, rec=True)
        self.assertEqual(len(w10), 11)
        self.assert_(all_positive(w10))

        w11 = WM.rm_watch(w10.values()[:5]+\
                          [-100, 4525, 77545475745454, 0]+\
                          w10.values()[5:])
        self.assertEqual(len(w11), 15)
        self.assert_(not all_true(w11))
        del w11[-100]
        del w11[4525]
        del w11[77545475745454]
        del w11[0]
        self.assert_(all_true(w11))

        w10b = WM.add_watch(os.path.join(self._tempdir, 'out3'),
                            mask_all, rec=True)
        self.assertEqual(len(w10b), 11)
        self.assert_(all_positive(w10b))

        w11b = WM.rm_watch(w10b.values()[:5]+\
                           [-100, 4525, 77545475745454, 0]+\
                           w10b.values()[5:], rec=True)
        self.assertEqual(len(w11b), 11)
        self.assert_(all_true(w11b))

        w12 = WM.rm_watch([-100, 4525, 77545475745454, 0], rec=True)
        self.assertEqual(len(w12), 0)

        w12b = WM.rm_watch([-100, 4525, 77545475745454, 0])
        self.assertEqual(len(w12b), 4)
        self.assert_(all_false(w12b))

        w131 = WM.add_watch(os.path.join(self._tempdir, 'out0ino.txt'),
                            mask_all, rec=True)
        self.assertEqual(len(w131), 1)
        self.assert_(all_positive(w131))

        w13 = WM.rm_watch(WM.get_wd(os.path.join(self._tempdir, 'out0ino.txt')), True)
        self.assertEqual(len(w13), 1)
        self.assert_(all_true(w13))

        w14 = WM.add_watch(os.path.join(self._tempdir, 'out0ino.txt'),
                           mask_all)
        self.assertEqual(len(w14), 1)
        self.assert_(all_positive(w14))

        w15 = WM.rm_watch(WM.get_wd(os.path.join(self._tempdir, 'out0ino.txt')))
        self.assertEqual(len(w15), 1)
        self.assert_(all_true(w15))

        w16 = WM.rm_watch(WM.get_wd(self._tempdir), True)
        self.assert_(all_true(w16))

        # phy rm then rm watch
#         w17 = WM.add_watch(os.path.join(self._tempdir,'out0/in0/'),
#                                       mask_all)
#         self.assertEqual(len(w17), 1)
#         self.assert_(all_positive(w17))

#         os.rmdir(os.path.join(self._tempdir,'out0/in0/'))

#         w19 = WM.rm_watch(w17.values()[0])
#         self.assertEqual(len(w19), 1)
#         self.assert_(all_false(w19))

#         os.mkdir(os.path.join(self._tempdir,'out0/in0/'))


    def test_symlinks(self):
        """
        test symlinks
        """
        mask_all = EventsCodes.ALL_EVENTS
        # creates symlink
        src = os.path.join(self._tempdir, 'out0')
        dst = os.path.join(self._tempdir, 'symlink')
        os.symlink(src, dst)

        w1 = WM.add_watch(dst, mask_all)
        self.assertEqual(len(w1), 1)
        self.assert_(all_positive(w1))

        w2 = WM.update_watch(WM.get_wd(dst), mask=1)
        self.assertEqual(len(w2), 1)
        self.assert_(all_true(w2))

        w3 = WM.rm_watch(WM.get_wd(dst))
        self.assertEqual(len(w3), 1)
        self.assert_(all_true(w3))

        w4 = WM.add_watch(self._tempdir, mask_all, rec=True)
        self.assert_(all_positive(w4))

        w5 = WM.add_watch(dst, mask_all)
        self.assertEqual(len(w5), 1)
        self.assert_(all_positive(w5))
        self.assertEqual(w5[dst], w4[src])

        time.sleep(2)
        w6 = WM.rm_watch(WM.get_wd(self._tempdir), True)
        time.sleep(2)
        self.assert_(all_true(w6))

        w7 = WM.rm_watch(w5[dst], True)
        self.assertEqual(len(w7), 0)

        w8 = WM.add_watch(dst, mask_all, rec=True)
        self.assertEqual(len(w8), 1)
        self.assert_(all_positive(w8))

        w9 = WM.add_watch(src, mask_all, rec=True)
        self.assertEqual(len(w9), 11)
        self.assert_(all_positive(w9))
        self.assertEqual(w9[src], w8[dst])

        # fail, should work
        #w58 = WM.rm_watch('/tmp/inot2/symlink', True)
        #check(w58, 1)

        w12 = WM.update_watch(w8[dst], rec=True)
        self.assertEqual(len(w12), 11)
        self.assert_(all_true(w12))

        w13 = WM.rm_watch(WM.get_wd(src), True)
        self.assertEqual(len(w13), 11)
        self.assert_(all_true(w13))

        w14 = WM.rm_watch(w8[dst], True)
        self.assertEqual(len(w14), 0)


def testsuite():
    suite = unittest.TestSuite()
    suite.addTests([TestThreadedNotifier('test_add_watch'),
                    TestThreadedNotifier('test_update_watch'),
                    TestThreadedNotifier('test_rm_watch'),
                    TestThreadedNotifier('test_symlinks')])
    return suite


def test_it():
    try:
        runner = unittest.TextTestRunner(verbosity=2)
        runner.run(testsuite())
    finally:
        print 'stop monitoring...'
        NOTIFIER.stop()

    # coverage analysis
    coverage.stop()
    # colorization
    f, s, m, mf = coverage.analysis(module_)
    fo = file(os.path.basename(f)+'.html', 'wb')
    colorize.colorize_file(f, outstream=fo, not_covered=mf)

    coverage.report(module_)
    coverage.erase()


if __name__ == '__main__':
    test_it()
