#  Ei!, emacs, this is -*-Python-*- mode
########################################################################
#
#       License: BSD
#       Created: September 21, 2002
#       Author:  Francesc Altet - faltet@carabos.com
#
#       $Id: hdf5Extension.pyx 1153 2005-09-06 11:32:10Z faltet $
#
########################################################################

"""Pyrex interface between several PyTables classes and HDF5 library.

Classes (type extensions):

    File
    AttributeSet
    Group
    Array
    VLArray
    IndexArray
    UnImplemented

Functions:

Misc variables:

    __version__
"""

import sys
import os
import warnings
import cPickle

import numarray
from numarray import records, strings

from tables.exceptions import HDF5ExtError
from tables.enum import Enum
from tables.utils import checkFileAccess

from tables.utilsExtension import  \
     enumToHDF5, enumFromHDF5, getTypeEnum, \
     convertTime64, getLeafHDF5Type, isHDF5File

from definitions cimport import_libnumarray, NA_getBufferPtrAndSize


__version__ = "$Revision: 1153 $"


#-------------------------------------------------------------------

# C funtions and variable declaration from its headers

# Type size_t is defined in stdlib.h
cdef extern from "stdlib.h":
  #ctypedef int size_t
  # The correct correspondence between size_t and a basic type is *long*
  # instead of int, because they are the same size even for 64-bit platforms
  # F. Altet 2003-01-08
  ctypedef long size_t
  void *malloc(size_t size)
  void free(void *ptr)
  double atof(char *nptr)

cdef extern from "time.h":
  ctypedef int time_t

# The next has been substituted by equivalents in Python, so that these
# functions could be accessible in Windows systems
# Thanks to Shack Toms for this!
# F. Altet 2004-10-01
# cdef extern from "math.h":
#   double nextafter(double x, double y)
#   float nextafterf(float x, float y)

# Funtions for printing in C
cdef extern from "stdio.h":
  int sprintf(char *str,  char *format, ...)
  int snprintf(char *str, size_t size, char *format, ...)

cdef extern from "string.h":
  char *strcpy(char *dest, char *src)
  char *strncpy(char *dest, char *src, size_t n)
  int strcmp(char *s1, char *s2)
  char *strdup(char *s)
  void *memcpy(void *dest, void *src, size_t n)

# Some helper routines from the Python API
cdef extern from "Python.h":

  # To access tuples
  object PyTuple_New(int)
  int PyTuple_SetItem(object, int, object)
  object PyTuple_GetItem(object, int)
  int PyTuple_Size(object tuple)

  # To access integers
  object PyInt_FromLong(long)
  long PyInt_AsLong(object)
  object PyLong_FromLongLong(long long)

  # To access double
  object PyFloat_FromDouble(double)

  # To access strings
  object PyString_FromStringAndSize(char *s, int len)
  char *PyString_AsString(object string)
  object PyString_FromString(char *)

  # To release global interpreter lock (GIL) for threading
  void Py_BEGIN_ALLOW_THREADS()
  void Py_END_ALLOW_THREADS()

  # To access to Memory (Buffer) objects presents in numarray
  object PyBuffer_FromMemory(void *ptr, int size)
  object PyBuffer_FromReadWriteMemory(void *ptr, int size)
  object PyBuffer_New(int size)
  int PyObject_CheckReadBuffer(object)
  int PyObject_AsReadBuffer(object, void **rbuf, int *len)
  int PyObject_AsWriteBuffer(object, void **rbuf, int *len)

# Structs and functions from numarray
cdef extern from "numarray/numarray.h":

  ctypedef enum NumarrayType:
    tAny
    tBool
    tInt8
    tUInt8
    tInt16
    tUInt16
    tInt32
    tUInt32
    tInt64
    tUInt64
    tFloat32
    tFloat64
    tComplex32
    tComplex64
    tObject
    tDefault
    tLong

# Functions and types from HDF5
cdef extern from "hdf5.h":
  int H5F_ACC_TRUNC, H5F_ACC_RDONLY, H5F_ACC_RDWR, H5F_ACC_EXCL
  int H5F_ACC_DEBUG, H5F_ACC_CREAT
  int H5P_DEFAULT, H5S_ALL
  int H5P_FILE_CREATE, H5P_FILE_ACCESS
  int H5FD_LOG_LOC_WRITE, H5FD_LOG_ALL
  int H5I_INVALID_HID

  ctypedef int hid_t  # In H5Ipublic.h
  ctypedef int hbool_t
  ctypedef int herr_t
  ctypedef int htri_t
  ctypedef long long hsize_t
  ctypedef signed long long hssize_t

  ctypedef struct hvl_t:
    size_t len                 # Length of VL data (in base type units) */
    void *p                    # Pointer to VL data */

  ctypedef enum H5G_obj_t:
    H5G_UNKNOWN = -1,           # Unknown object type          */
    H5G_LINK,                   # Object is a symbolic link    */
    H5G_GROUP,                  # Object is a group            */
    H5G_DATASET,                # Object is a dataset          */
    H5G_TYPE,                   # Object is a named data type  */
    H5G_RESERVED_4,             # Reserved for future use      */
    H5G_RESERVED_5,             # Reserved for future use      */
    H5G_RESERVED_6,             # Reserved for future use      */
    H5G_RESERVED_7              # Reserved for future use      */

  cdef struct H5G_stat_t:
    unsigned long fileno[2]
    unsigned long objno[2]
    unsigned nlink
    H5G_obj_t type  # new in HDF5 1.6
    time_t mtime
    size_t linklen
    #H5O_stat_t ohdr           # Object header information. New in HDF5 1.6

  cdef enum H5T_class_t:
    H5T_NO_CLASS         = -1,  #error                                      */
    H5T_INTEGER          = 0,   #integer types                              */
    H5T_FLOAT            = 1,   #floating-point types                       */
    H5T_TIME             = 2,   #date and time types                        */
    H5T_STRING           = 3,   #character string types                     */
    H5T_BITFIELD         = 4,   #bit field types                            */
    H5T_OPAQUE           = 5,   #opaque types                               */
    H5T_COMPOUND         = 6,   #compound types                             */
    H5T_REFERENCE        = 7,   #reference types                            */
    H5T_ENUM             = 8,   #enumeration types                          */
    H5T_VLEN             = 9,   #Variable-Length types                      */
    H5T_ARRAY            = 10,  #Array types                                */
    H5T_NCLASSES                #this must be last                          */

  # The difference between a single file and a set of mounted files
  cdef enum H5F_scope_t:
    H5F_SCOPE_LOCAL     = 0,    # specified file handle only
    H5F_SCOPE_GLOBAL    = 1,    # entire virtual file
    H5F_SCOPE_DOWN      = 2     # for internal use only

  cdef enum H5T_sign_t:
    H5T_SGN_ERROR        = -1,  #error                                      */
    H5T_SGN_NONE         = 0,   #this is an unsigned type                   */
    H5T_SGN_2            = 1,   #two's complement                           */
    H5T_NSGN             = 2    #this must be last!                         */

  cdef enum H5G_link_t:
    H5G_LINK_ERROR      = -1,
    H5G_LINK_HARD       = 0,
    H5G_LINK_SOFT       = 1


  # Native types
  cdef enum:
    H5T_NATIVE_CHAR
    H5T_NATIVE_SCHAR
    H5T_NATIVE_UCHAR
    H5T_NATIVE_SHORT
    H5T_NATIVE_USHORT
    H5T_NATIVE_INT
    H5T_NATIVE_UINT
    H5T_NATIVE_LONG
    H5T_NATIVE_ULONG
    H5T_NATIVE_LLONG
    H5T_NATIVE_ULLONG
    H5T_NATIVE_FLOAT
    H5T_NATIVE_DOUBLE
    H5T_NATIVE_LDOUBLE

# Functions from HDF5
cdef extern from "H5public.h":
  hid_t  H5Fcreate(char *filename, unsigned int flags,
                   hid_t create_plist, hid_t access_plist)

  hid_t  H5Fopen(char *name, unsigned flags, hid_t access_id)

  herr_t H5Fclose (hid_t file_id)

  herr_t H5Fflush(hid_t object_id, H5F_scope_t scope )

  hid_t  H5Dopen (hid_t file_id, char *name)

  herr_t H5Dclose (hid_t dset_id)

  herr_t H5Dread (hid_t dset_id, hid_t mem_type_id, hid_t mem_space_id,
                  hid_t file_space_id, hid_t plist_id, void *buf)

  hid_t H5Dget_type (hid_t dset_id)

  hid_t H5Dget_space (hid_t dset_id)

  hid_t  H5Gcreate(hid_t loc_id, char *name, size_t size_hint )

  herr_t H5Gget_objinfo(hid_t loc_id,
                        char *name,
                        hbool_t follow_link,
                        H5G_stat_t *statbuf )

  hid_t  H5Gopen(hid_t loc_id, char *name )

  herr_t H5Gclose(hid_t group_id)

  herr_t H5Glink (hid_t file_id, H5G_link_t link_type,
                  char *current_name, char *new_name)

  herr_t H5Gunlink (hid_t file_id, char *name)

  herr_t H5Gmove(hid_t loc_id, char *src, char *dst)

  herr_t H5get_libversion(unsigned *majnum, unsigned *minnum,
                          unsigned *relnum )

  herr_t H5check_version(unsigned majnum, unsigned minnum,
          unsigned relnum )

  herr_t H5Adelete(hid_t loc_id, char *name )

  herr_t H5Tclose(hid_t type_id)

  herr_t H5Tget_sign(hid_t type_id)

  #hid_t H5Pcreate(H5P_class_t type )  # Wrong in documentation!
  hid_t H5Pcreate(hid_t type )

  herr_t H5Pset_cache(hid_t plist_id, int mdc_nelmts, int rdcc_nelmts,
                      size_t rdcc_nbytes, double rdcc_w0 )

  herr_t H5Pset_sieve_buf_size( hid_t fapl_id, hsize_t size )

  herr_t H5Pset_fapl_log( hid_t fapl_id, char *logfile,
                          unsigned int flags, size_t buf_size )

  int H5Sget_simple_extent_ndims(hid_t space_id)

  int H5Sget_simple_extent_dims(hid_t space_id, hsize_t dims[],
                                hsize_t maxdims[])

  herr_t H5Sclose(hid_t space_id)

  # Functions for enumeration handling:
  hid_t  H5Tget_super(hid_t type)
  H5T_class_t H5Tget_class(hid_t type_id)
  int    H5Tget_nmembers(hid_t type_id)
  hid_t  H5Tget_member_type(hid_t type_id, int membno)
  char*  H5Tget_member_name(hid_t type_id, int membno)
  herr_t H5Tget_member_value(hid_t type_id, int membno, void *value)

# Functions from HDF5 HL Lite
cdef extern from "H5LT.h":

  herr_t H5LT_get_attribute_disk(hid_t loc_id, char *attr_name, void *attr_out)

  herr_t H5LTget_attribute_ndims( hid_t loc_id,
                                  char *obj_name,
                                  char *attr_name,
                                  int *rank )
  herr_t H5LTget_attribute_info( hid_t loc_id, char *obj_name, char *attr_name,
                                 hsize_t *dims, H5T_class_t *class_id,
                                 size_t *type_size, hid_t *type_id)

  herr_t H5LTset_attribute_string( hid_t loc_id, char *obj_name,
                                   char *attr_name, char *attr_data )

  herr_t H5LTset_attribute_numerical_NAarray( hid_t loc_id,
                                              char *obj_name,
                                              char *attr_name,
                                              size_t rank,
                                              hsize_t *dims,
                                              hid_t type_id,
                                              void *data )
  herr_t H5LTget_attribute( hid_t loc_id,
                            char *obj_name,
                            char *attr_name,
                            hid_t mem_type_id,
                            void *data )

  herr_t H5LTget_attribute_string( hid_t loc_id, char *obj_name,
                                   char *attr_name, char *attr_data )

  herr_t H5LTget_attribute_int( hid_t loc_id,
                                char *obj_name,
                                char *attr_name,
                                int *data )

  herr_t H5LT_set_attribute_numerical( hid_t loc_id,
                                       char *obj_name,
                                       char *attr_name,
                                       hid_t type_id,
                                       void *data )

  herr_t H5LT_find_attribute(hid_t loc_id, char *attr_name )


# Functions from HDF5 ARRAY (this is not part of HDF5 HL; it's private)
cdef extern from "H5ARRAY.h":

  herr_t H5ARRAYmake( hid_t loc_id, char *dset_name, char *class_,
                      char *title, char *flavor, char *obversion,
                      int rank, hsize_t *dims, int extdim,
                      hid_t type_id, hsize_t *dims_chunk, void *fill_data,
                      int complevel, char  *complib, int shuffle,
                      int fletcher32, void *data)

  herr_t H5ARRAYappend_records( hid_t loc_id, char *dset_name,
                                int rank, hsize_t *dims_orig,
                                hsize_t *dims_new, int extdim, void *data )

  herr_t H5ARRAYwrite_records( hid_t loc_id,  char *dset_name,
                               int rank, hsize_t *start, hsize_t *step,
                               hsize_t *count, void *data )

  herr_t H5ARRAYtruncate( hid_t loc_id, char *dset_name,
                          int extdim, hsize_t size)

  herr_t H5ARRAYread( hid_t loc_id, char *dset_name,
                      hsize_t start,  hsize_t nrows, hsize_t step,
                      int extdim, void *data )

  herr_t H5ARRAYreadSlice( hid_t loc_id, char *dset_name,
                           hsize_t *start, hsize_t *stop,
                           hsize_t *step, void *data )

  herr_t H5ARRAYreadIndex( hid_t loc_id, char *dset_name, int notequal,
                           hsize_t *start, hsize_t *stop, hsize_t *step,
                           void *data )

  herr_t H5ARRAYget_ndims( hid_t loc_id, char *dset_name, int *rank )

  hid_t H5ARRAYget_info( hid_t loc_id, char *dset_name, hsize_t *dims,
                         hsize_t *maxdims, hid_t *super_type_id,
                         H5T_class_t *super_class_id, char *byteorder)

# Functions for optimized operations for ARRAY
cdef extern from "H5ARRAY-opt.h":

  herr_t H5ARRAYOopen_readSlice( hid_t *dataset_id,
                                 hid_t *space_id,
                                 hid_t *type_id,
                                 hid_t loc_id,
                                 char *dset_name)

  herr_t H5ARRAYOread_readSlice( hid_t dataset_id,
                                 hid_t space_id,
                                 hid_t type_id,
                                 hsize_t irow,
                                 hsize_t start,
                                 hsize_t stop,
                                 void *data )

  herr_t H5ARRAYOclose_readSlice(hid_t dataset_id,
                                 hid_t space_id,
                                 hid_t type_id)

# Functions for VLEN Arrays
cdef extern from "H5VLARRAY.h":

  hid_t H5VLARRAYmake( hid_t loc_id, char *dset_name, char *class_, char *title,
                       char *flavor, char *obversion, int rank, int scalar,
                       hsize_t *dims, hid_t type_id, hsize_t chunk_size,
                       void *fill_data, int complevel, char *complib,
                       int shuffle, int flecther32, void *data)

  herr_t H5VLARRAYappend_records( hid_t loc_id, char *dset_name,
                                  int nobjects, hsize_t nrecords,
                                  void *data )

  herr_t H5VLARRAYmodify_records( hid_t loc_id, char *dset_name,
                                  hsize_t nrow, int nobjects,
                                  void *data )

  herr_t H5VLARRAYread( hid_t loc_id, char *dset_name,
                        hsize_t start,  hsize_t nrows, hsize_t step,
                        hvl_t *rdata, hsize_t *rdatalen )

  herr_t H5VLARRAYget_ndims( hid_t loc_id, char *dset_name, int *rank )

  herr_t H5ARRAYget_chunksize( hid_t loc_id, char *dset_name,
                               int rank, hsize_t *dims_chunk)

  hid_t H5VLARRAYget_info( hid_t loc_id, char *dset_name,
                           hsize_t *nrecords, hsize_t *base_dims,
                           hid_t *base_type_id, char *base_byteorder)


# Funtion to compute the HDF5 type from a numarray enum type
cdef extern from "arraytypes.h":

  hid_t convArrayType(int fmt, size_t size, char *byteorder)
  size_t getArrayType(hid_t type_id, int *fmt)


# Helper routines
cdef extern from "utils.h":
  herr_t set_cache_size(hid_t file_id, size_t cache_size)
  object _getTablesVersion()
  #object getZLIBVersionInfo()
  object getHDF5VersionInfo()
  object createNamesTuple(char *buffer[], int nelements)
  object Giterate(hid_t parent_id, hid_t loc_id, char *name)
  object Aiterate(hid_t loc_id)
  object H5UIget_info(hid_t loc_id, char *name, char *byteorder)

  object get_attribute_string_sys(hid_t loc_id, char *obj_name,
                                  char *attr_name)


#-----------------------------------------------------------------------------

# Local variables

# CharArray type
CharType = numarray.records.CharType

# Conversion tables from/to classes to the numarray enum types
naEnumToNAType = {
  tBool: numarray.Bool,  # Boolean type added
  tInt8: numarray.Int8,    tUInt8: numarray.UInt8,
  tInt16: numarray.Int16,  tUInt16: numarray.UInt16,
  tInt32: numarray.Int32,  tUInt32: numarray.UInt32,
  tInt64: numarray.Int64,  tUInt64: numarray.UInt64,
  tFloat32: numarray.Float32,  tFloat64: numarray.Float64,
  tComplex32: numarray.Complex32,  tComplex64: numarray.Complex64,
  # Special cases:
  ord('a'): CharType,  # For strings.
  ord('t'): numarray.Int32,  ord('T'): numarray.Float64}  # For times.

naTypeToNAEnum = {
  numarray.Bool: tBool,
  numarray.Int8: tInt8,    numarray.UInt8: tUInt8,
  numarray.Int16: tInt16,  numarray.UInt16: tUInt16,
  numarray.Int32: tInt32,  numarray.UInt32: tUInt32,
  numarray.Int64: tInt64,  numarray.UInt64: tUInt64,
  numarray.Float32: tFloat32,  numarray.Float64: tFloat64,
  numarray.Complex32: tComplex32,  numarray.Complex64: tComplex64,
  # Special cases:
  numarray.records.CharType: ord('a')}  # For strings.

NATypeToHDF5AtomicType = {
                            numarray.Int8      : H5T_NATIVE_SCHAR,
                            numarray.Int16     : H5T_NATIVE_SHORT,
                            numarray.Int32     : H5T_NATIVE_INT,
                            numarray.Int64     : H5T_NATIVE_LLONG,

                            numarray.UInt8     : H5T_NATIVE_UCHAR,
                            numarray.UInt16    : H5T_NATIVE_USHORT,
                            numarray.UInt32    : H5T_NATIVE_UINT,
                            numarray.UInt64    : H5T_NATIVE_ULLONG,

                            numarray.Float32   : H5T_NATIVE_FLOAT,
                            numarray.Float64   : H5T_NATIVE_DOUBLE
                        }

# Conversion from numarray int codes to strings
naEnumToNASType = {
  tBool:'Bool',  # Boolean type added
  tInt8:'Int8',    tUInt8:'UInt8',
  tInt16:'Int16',  tUInt16:'UInt16',
  tInt32:'Int32',  tUInt32:'UInt32',
  tInt64:'Int64',  tUInt64:'UInt64',
  tFloat32:'Float32',  tFloat64:'Float64',
  tComplex32:'Complex32',  tComplex64:'Complex64',
  # Special cases:
  ord('a'):'CharType',  # For strings.
  ord('t'):'Time32',  ord('T'):'Time64',  # For times.
  ord('e'):'Enum'}  # For enumerations.


#----------------------------------------------------------------------------

# Initialization code

# The numarray API requires this function to be called before
# using any numarray facilities in an extension module.
import_libnumarray()

#---------------------------------------------------------------------------

# Main functions

## What is the point in opening an already closed node to flush it?
## ivb(2005-07-13)
##
## def flush_leaf(where, name):
##   "Flush the buffers of a Leaf"
##   cdef hid_t   parent_id, dataset_id
##   cdef char    *dataset_name
## 
##   # Get the object ID. Do this until a standard objectID is implemented
##   parent_id = where._v_objectID
##   # Open the dataset
##   dataset_name = strdup(name)
##   dataset_id = H5Dopen( parent_id, dataset_name )
##   # Flush the leaf
##   H5Fflush(dataset_id, H5F_SCOPE_GLOBAL)
##   H5Dclose( dataset_id )

#------------------------------------------------------------------------

# Type extensions declarations (these are subclassed by PyTables
# Python classes)

cdef class File:
  cdef hid_t   file_id
  cdef hid_t   access_plist
  cdef char    *name


  def __new__(self, char *name, char *mode, char *title, int new,
              object trTable, char *root, int isPTFile, object filters,
              size_t cache_size):
    # Create a new file using default properties
    self.name = name
    self.mode = pymode = mode

    # After this check we can be quite sure that the file or directory
    # exists and permissions are right.
    checkFileAccess(name, mode)

    assert pymode in ('r', 'r+', 'a', 'w'), """\
an invalid mode string ``%s`` passed the ``checkFileAccess()`` test; \
please report this to the authors""" % pymode

    # After this check we know that the file exists and is an HDF5 file
    # when needed.
    exists = os.path.exists(name)
    if pymode in ('r', 'r+') or (pymode == 'a' and exists):
      if not isHDF5File(name):
        raise IOError("file ``%s`` exists but it is not an HDF5 file"
                      % self.name)

    if pymode == 'r':
      # Just a test for disabling metadata caching.
      ## access_plist = H5Pcreate(H5P_FILE_ACCESS)
      ## H5Pset_cache(access_plist, 0, 0, 0, 0.0)
      ## H5Pset_sieve_buf_size(access_plist, 0)
      ##self.file_id = H5Fopen(name, H5F_ACC_RDONLY, access_plist)
      self.file_id = H5Fopen(name, H5F_ACC_RDONLY, H5P_DEFAULT)
    elif pymode == 'r+':
      self.file_id = H5Fopen(name, H5F_ACC_RDWR, H5P_DEFAULT)
    elif pymode == 'a':
      if exists:
        # A test for logging.
        ## access_plist = H5Pcreate(H5P_FILE_ACCESS)
        ## H5Pset_cache(access_plist, 0, 0, 0, 0.0)
        ## H5Pset_sieve_buf_size(access_plist, 0)
        ## H5Pset_fapl_log (access_plist, "test.log", H5FD_LOG_LOC_WRITE, 0)
        ## self.file_id = H5Fopen(name, H5F_ACC_RDWR, access_plist)
        self.file_id = H5Fopen(name, H5F_ACC_RDWR, H5P_DEFAULT)
      else:
        self.file_id = H5Fcreate(name, H5F_ACC_TRUNC, H5P_DEFAULT, H5P_DEFAULT)
    elif pymode == 'w':
      self.file_id = H5Fcreate(name, H5F_ACC_TRUNC, H5P_DEFAULT, H5P_DEFAULT)

    # Set the cache size (only for HDF5 1.8.x)
    set_cache_size(self.file_id, cache_size)

  # Accessor definitions
  def _getFileId(self):
    return self.file_id

  def _flushFile(self, scope):
    # Close the file
    H5Fflush(self.file_id, scope)

  def _closeFile(self):
    # Close the file
    H5Fclose( self.file_id )
    self.file_id = 0    # Means file closed

  # This method is moved out of scope, until we provide code to delete
  # the memory booked by this extension types
  def __dealloc__(self):
    cdef int ret
    #print "Destroying object File in Extension"
    if self.file_id:
      #print "Closing the HDF5 file", name," because user didn't do that!."
      ret = H5Fclose(self.file_id)
      if ret < 0:
        raise HDF5ExtError("Problems closing the file '%s'" % self.name)


cdef class AttributeSet:
  cdef hid_t   parent_id
  cdef char    *name
  cdef object  node

  def _g_new(self, node):
    self.node = node
    # Initialize the C attributes of Node object
    self.name =  PyString_AsString(node._v_hdf5name)
    # The parent group id of the node
    self.parent_id = node._v_parent._v_objectID

  def _g_listAttr(self):
    cdef object attrlist
    cdef hid_t loc_id

    if isinstance(self.node, Group):  # To match hdf5Extension.Group
      # Return a tuple with the attribute list
      attrlist = Aiterate(self.node._v_objectID)
    else:
      # Get the dataset ID (the Leaf objects are always closed)
      loc_id = H5Dopen(self.parent_id, self.name)
      # Keep the object ID in the objectID attribute in the parent dataset
      # The existing datsets are always opened here, so this would
      # be enough to get the objectID for existing datasets
      self.node._v_objectID = loc_id
      if loc_id < 0:
        raise HDF5ExtError("Cannot open the dataset '%s'" % self.name)
      attrlist = Aiterate(loc_id)
      # Close this dataset
      ret = H5Dclose(loc_id)
      if ret < 0:
        raise HDF5ExtError("Cannot close the dataset '%s'" % self.name)

    return attrlist

  def _g_setAttr(self, char *name, object value):
    cdef int ret
    cdef int valint
    cdef double valdouble
    cdef hid_t type_id
    cdef size_t rank
    cdef hsize_t *dims
    cdef void* data
    cdef long buflen
    cdef int i

    # Check for scalar attributes (if file.format_version < "1.4")
    if hasattr( self._v_node._v_file, "format_version"):
      format_version = self._v_node._v_file.format_version
    else:
      format_version = None

    if format_version is not None and format_version < "1.4" and \
       (isinstance(value, int) or isinstance(value, float)):
      value = numarray.asarray(value, shape=(1,))

    ret = 0
    # Append this attribute on disk
    if isinstance(value, str):
      ret = H5LTset_attribute_string(self.parent_id, self.name, name, value)
    elif isinstance(value, numarray.numarraycore.NumArray):
      if value.type() in NATypeToHDF5AtomicType.keys():
        type_id = NATypeToHDF5AtomicType[value.type()]

        rank = value.rank
        dims = <hsize_t *>malloc(rank * sizeof(hsize_t))

        for i from 0 <= i < rank:
          dims[i] = value.shape[i]

        buflen = NA_getBufferPtrAndSize(value._data, 1, &data)
        ret = H5LTset_attribute_numerical_NAarray(self.parent_id,
                                                  self.name, name, rank,
                                                  dims, type_id, data)
        free(<void *>dims)
      else:   # One should add complex support for numarrays
        pickledvalue = cPickle.dumps(value, 0)
        self._g_setAttrStr(name, pickledvalue)
    else:
      # Convert this object to a null-terminated string
      # (binary pickles are not supported at this moment)
      pickledvalue = cPickle.dumps(value, 0)
      self._g_setAttrStr(name, pickledvalue)

    if ret < 0:
      raise HDF5ExtError("Can't set attribute '%s' in node:\n %s." %
                         (name, self.node))

  def _g_setAttrStr(self, char *attrname, char *attrvalue):
    cdef int ret

    ret = H5LTset_attribute_string(self.parent_id, self.name,
                                   attrname, attrvalue)
    if ret < 0:
      raise HDF5ExtError("Can't set attribute '%s' in node:\n %s." %
                         (attrname, self.node))

  def _g_getAttr(self, char *attrname):
    cdef object attrvalue
    cdef hid_t loc_id
    if isinstance(self.node, Group):
      attrvalue = self._g_getNodeAttr(self.parent_id, self.node._v_objectID,
                                      self.name, attrname)
    else:
      # Get the dataset ID
      loc_id = H5Dopen(self.parent_id, self.name)
      if loc_id < 0:
        raise HDF5ExtError("Cannot open the dataset '%s' in node '%s'" % \
                           (self.name, self._v_parent._v_name))

      attrvalue = self._g_getNodeAttr(self.parent_id, loc_id,
                                      self.name, attrname)

      # Close this dataset
      ret = H5Dclose(loc_id)
      if ret < 0:
        raise HDF5ExtError("Cannot close the dataset '%s'" % self.name)

    return attrvalue

  # Get a system attribute (they should be only strings)
  def _g_getSysAttr(self, char *attrname):
    ret = get_attribute_string_sys(self.parent_id, self.name, attrname)
    return ret

  # This function is useful to retrieve system attributes of Leafs
  def _g_getChildSysAttr(self, char *dsetname, char *attrname):
    ret = get_attribute_string_sys(self.node._v_objectID, dsetname, attrname)
    return ret

  def _g_getChildAttr(self, char *dsetname, char *attrname):
    cdef object attrvalue
    cdef hid_t loc_id
    # Get the dataset ID
    loc_id = H5Dopen(self.node._v_objectID, dsetname)
    if loc_id < 0:
      raise HDF5ExtError("Cannot open the child '%s' of node '%s'" % \
                         (dsetname, self.name))

    attrvalue = self._g_getNodeAttr(self.node._v_objectID, loc_id,
                                    dsetname, attrname)
    # Close this dataset
    ret = H5Dclose(loc_id)
    if ret < 0:
      raise HDF5ExtError("Cannot close the dataset '%s'" % dsetname)

    return attrvalue

  # Get attributes
  def _g_getNodeAttr(self, hid_t parent_id, hid_t loc_id,
                     char *dsetname, char *attrname):
    cdef hsize_t *dims, nelements
    cdef H5T_class_t class_id
    cdef size_t type_size
    cdef char  *attrvaluestr
    cdef short  attrvalueshort
    cdef long  attrvaluelong
    cdef float  attrvaluefloat
    cdef double  attrvaluedouble
    cdef long long attrvaluelonglong
    cdef object retvalue
    cdef hid_t mem_type
    cdef int rank
    cdef int ret, i
    cdef int enumtype
    cdef void *rbuf
    cdef long buflen
    cdef hid_t type_id
    cdef H5T_sign_t sign #H5T_SGN_ERROR (-1), H5T_SGN_NONE (0), H5T_SGN_2 (1)

    # Check if attribute exists
    if H5LT_find_attribute(loc_id, attrname) <= 0:
      # If the attribute does not exists, return None
      # and do not even warn the user
      return None

    ret = H5LTget_attribute_ndims(parent_id, dsetname, attrname, &rank )
    if ret < 0:
      raise HDF5ExtError("Can't get ndims on attribute %s in node %s." %
                             (attrname, dsetname))

    # Allocate memory to collect the dimension of objects with dimensionality
    if rank > 0:
      dims = <hsize_t *>malloc(rank * sizeof(hsize_t))
    else:
      dims = NULL

    ret = H5LTget_attribute_info(parent_id, dsetname, attrname,
                                 dims, &class_id, &type_size, &type_id)
    if ret < 0:
      raise HDF5ExtError("Can't get info on attribute %s in node %s." %
                               (attrname, dsetname))
    # Get the attribute type
    type_size = getArrayType(type_id, &enumtype)
    if type_size < 0:
      raise TypeError, "HDF5 class %d not supported. Sorry!" % class_id
    sign = H5Tget_sign(type_id)
    H5Tclose(type_id)    # Release resources

    # Get the array shape
    shape = []
    for i from 0 <= i < rank:
      # The <int> cast avoids returning a Long integer
      shape.append(<int>dims[i])
    shape = tuple(shape)

    retvalue = None
    dtype = naEnumToNAType.get(enumtype)
    if dtype and class_id in (H5T_INTEGER, H5T_FLOAT):
      retvalue = numarray.array(None, type=dtype, shape=shape)
      # Get the pointer to the buffer data area
      buflen = NA_getBufferPtrAndSize(retvalue._data, 1, &rbuf)

    if class_id == H5T_INTEGER:
      if sign == H5T_SGN_2:
        if type_size == 1:
          ret = H5LTget_attribute(parent_id, dsetname,
                                  attrname, H5T_NATIVE_SCHAR, rbuf)
        if type_size == 2:
          ret = H5LTget_attribute(parent_id, dsetname,
                                  attrname, H5T_NATIVE_SHORT, rbuf)
        if type_size == 4:
          ret = H5LTget_attribute(parent_id, dsetname,
                                  attrname, H5T_NATIVE_INT, rbuf)
        if type_size == 8:
          ret = H5LTget_attribute(parent_id,dsetname,
                                  attrname, H5T_NATIVE_LLONG, rbuf)
      elif sign == H5T_SGN_NONE:
        if type_size == 1:
          ret = H5LTget_attribute(parent_id, dsetname,
                                  attrname, H5T_NATIVE_UCHAR, rbuf)
        if type_size == 2:
          ret = H5LTget_attribute(parent_id, dsetname,
                                  attrname, H5T_NATIVE_USHORT, rbuf)
        if type_size == 4:
          ret = H5LTget_attribute(parent_id, dsetname,
                                  attrname, H5T_NATIVE_UINT, rbuf)
        if type_size == 8:
          ret = H5LTget_attribute(parent_id,dsetname,
                                  attrname, H5T_NATIVE_ULLONG, rbuf)
      else:
        warnings.warn("""\
Type of attribute '%s' in node '%s' is not supported. Sorry about that!"""
                      % (attrname, dsetname))
        return None

    elif class_id == H5T_FLOAT:
      if type_size == 4:
        ret = H5LTget_attribute(parent_id, dsetname,
                                attrname, H5T_NATIVE_FLOAT, rbuf)
      if type_size == 8:
        ret = H5LTget_attribute(parent_id, dsetname,
                                attrname, H5T_NATIVE_DOUBLE, rbuf)

    elif class_id == H5T_STRING:
      attrvaluestr = <char *>malloc(type_size * sizeof(char))
      ret = H5LTget_attribute_string(parent_id, dsetname,
                                     attrname, attrvaluestr)
      retvalue = PyString_FromString(attrvaluestr)
      free(<void *> attrvaluestr)   # To avoid memory leak!

    else:
      warnings.warn("""\
Type of attribute '%s' in node '%s' is not supported. Sorry about that!"""
                    % (attrname, dsetname))
      return None

    if dims:
      free(<void *> dims)

    # Check the return value of H5LTget_attribute_* call
    if ret < 0:
      raise HDF5ExtError("Attribute %s exists in node %s, but can't get it."\
                         % (attrname, dsetname))

    # Check for multimensional attributes (if file.format_version > "1.4")
    if hasattr( self._v_node._v_file, "format_version"):
      format_version = self._v_node._v_file.format_version
    else:
      format_version = None

    if format_version is not None:
      if format_version < "1.4":
        if rank > 0 and shape[0] > 1:
          warnings.warn("""\
Multi-dimensional attribute '%s' in node '%s' is not supported in file format version %s.
Loaded anyway."""
                        % (attrname, dsetname, format_version))

        elif class_id == H5T_INTEGER:
          # return as 'int' built-in type
          if shape == ():
            retvalue = int(retvalue[()])
          else:
            retvalue = int(retvalue[0])

        elif class_id == H5T_FLOAT:
          # return as 'float' built-in type
          if shape == ():
            retvalue = float(retvalue[()])
          else:
            retvalue = float(retvalue[0])

        # Just if one wants to convert a scalar int a Python scalar
        # instead of an numarray scalar. But I think it is better and more
        # consistent a numarray scalar.
#       elif format_version >= "1.4" and rank == 0:
#         if class_id == H5T_INTEGER or class_id == H5T_FLOAT:
#           # A scalar will be converted to a Python scalar from format 1.4 on
#           retvalue = retvalue[()]

    return retvalue

  def _g_remove(self, attrname):
    cdef int ret
    cdef hid_t loc_id

    if isinstance(self.node, Group):
      ret = H5Adelete(self.node._v_objectID, attrname )
      if ret < 0:
        raise HDF5ExtError("Attribute '%s' exists in node '%s', but cannot be deleted." \
                         % (attrname, dsetname))
    else:
      # Open the dataset
      loc_id = H5Dopen(self.parent_id, self.name)
      if loc_id < 0:
        raise HDF5ExtError("Cannot open the dataset '%s' in node '%s'" % \
                           (dsetname, self.name))

      ret = H5Adelete(loc_id, attrname )
      if ret < 0:
        raise HDF5ExtError("Attribute '%s' exists in node '%s', but cannot be deleted." \
                         % (attrname, self.name))

      # Close this dataset
      ret = H5Dclose(loc_id)
      if ret < 0:
        raise HDF5ExtError("Cannot close the dataset '%s'" % self.name)

  def __dealloc__(self):
    cdef int ret
    #print "Destroying object AttributeSet in Extension"
    self.node = None
    self.parent_id = 0


cdef class Group:
  cdef hid_t   group_id
  cdef hid_t   parent_id
  cdef char    *name

  def _g_new(self, where, name):
    # Initialize the C attributes of Group object
    self.name = strdup(name)
    # The parent group id for this object
    self.parent_id = where._v_objectID

  def _g_createGroup(self):
    cdef hid_t ret

    # Create a new group
    ret = H5Gcreate(self.parent_id, self.name, 0)
    if ret < 0:
      raise HDF5ExtError("Can't create the group %s." % self.name)
    self.group_id = ret
    return self.group_id

  def _g_openGroup(self):
    cdef hid_t ret

    ret = H5Gopen(self.parent_id, self.name)
    if ret < 0:
      raise HDF5ExtError("Can't open the group: '%s'." % self.name)
    self.group_id = ret
    return self.group_id

  def _g_openIndex(self):
    cdef hid_t ret

    ret = H5Gopen(self.parent_id, self.name)
    if ret < 0:
      raise HDF5ExtError("Can't open the index: '%s'." % self.name)
    self.group_id = ret
    return self.group_id

  def _g_listGroup(self, hid_t parent_id, hid_t loc_id, char *name):
    # Return a tuple with the objects groups and objects dsets
    return Giterate(parent_id, loc_id, name)

  def _g_getGChildAttr(self, char *group_name, char *attr_name):
    """Return an attribute of a child Group.

    When successful, returns the format version string, for TRUE, or 0
    (zero), for FALSE. Otherwise returns a negative value.

    """

    cdef hid_t gchild_id
    cdef herr_t ret
    cdef char attr_value[256]

    # Check if attribute exists
    # Open the group
    gchild_id = H5Gopen(self.group_id, group_name)
    strcpy(attr_value, "unknown")  # Default value
    if H5LT_find_attribute(gchild_id, attr_name):
      # Read the attr_name attribute
      ret = H5LT_get_attribute_disk(gchild_id, attr_name, attr_value)
      if ret < 0:
        strcpy(attr_value, "unknown")

    # Close child group
    H5Gclose(gchild_id)
    return attr_value

  def _g_getAttr(self, char *attr_name):
    """Return an attribute of a child Group.

    When successful, returns the format version string, for TRUE, or 0
    (zero), for FALSE. Otherwise returns a negative value.

    """
    cdef herr_t ret
    cdef char attr_value[256]

    # Check if attribute exists
    strcpy(attr_value, "unknown")  # Default value
    if H5LT_find_attribute(self.group_id, attr_name):
      # Read the attr_name attribute
      ret = H5LT_get_attribute_disk(self.group_id, attr_name, attr_value)
      if ret < 0:
        strcpy(attr_value, "unknown")

    return attr_value

  def _g_flushGroup(self):
    # Close the group
    H5Fflush(self.group_id, H5F_SCOPE_GLOBAL)

  def _g_closeGroup(self):
    cdef int ret

    ret = H5Gclose(self.group_id)
    if ret < 0:
      raise HDF5ExtError("Problems closing the Group %s" % self.name )
    self.group_id = 0  # indicate that this group is closed

  def _g_renameNode(self, char *oldname, char *newname):
    cdef int ret

    ret = H5Gmove(self.group_id, oldname, newname)
    if ret < 0:
      raise HDF5ExtError("Problems renaming the node %s" % oldname )
    return ret

  def _g_moveNode(self, char *oldname, char *newname):
    cdef int ret

    ret = H5Gmove(self.group_id, oldname, newname)
    if ret < 0:
      raise HDF5ExtError("Problems moving the node %s to %s" %
                         (oldname, newname) )
    return ret

  def _g_deleteGroup(self):
    cdef int ret

    # Delete this group
    ret = H5Gunlink(self.parent_id, self.name)
    if ret < 0:
      raise HDF5ExtError("Problems deleting the Group %s" % self.name )
    return ret

  def _g_deleteLeaf(self, char *dsetname):
    cdef int ret

    # Delete the leaf child
    ret = H5Gunlink(self.group_id, dsetname)
    if ret < 0:
      raise HDF5ExtError("Problems deleting the Leaf '%s'" % dsetname )
    return ret

  def __dealloc__(self):
    cdef int ret
    # print "Destroying object Group in Extension"
    if self.group_id <> 0 and 0:
      print "Group open: ", self.name
    free(<void *>self.name)


cdef class Array:
  # Instance variables
  cdef hid_t   parent_id, type_id
  cdef char    *name
  cdef int     rank
  cdef hsize_t *dims
  cdef hsize_t *maxdims
  cdef hsize_t *dims_chunk
  cdef hsize_t stride[2]

  def _g_new(self, where, name):
    # Initialize the C attributes of Group object (Very important!)
    self.name = strdup(name)
    # The parent group id for this object
    self.parent_id = where._v_objectID

  def _createArray(self, object naarr, char *title):
    cdef int i
    cdef herr_t ret
    cdef hid_t oid
    cdef void *rbuf
    cdef long buflen
    cdef int enumtype, itemsize, offset
    cdef char *byteorder
    cdef char *flavor, *complib, *version, *class_
    cdef object type

    if isinstance(naarr, strings.CharArray):
      type = CharType
      enumtype = naTypeToNAEnum[CharType]
    else:
      type = naarr._type
      try:
        enumtype = naTypeToNAEnum[naarr._type]
      except KeyError:
        raise TypeError, \
      """Type class '%s' not supported right now. Sorry about that.
      """ % repr(naarr._type)

    # String types different from Numarray types are still not allowed
    # in regular Arrays.
    stype = str(type)

    itemsize = naarr._itemsize
    byteorder = PyString_AsString(self.byteorder)
    self.type_id = convArrayType(enumtype, itemsize, byteorder)
    if self.type_id < 0:
      raise TypeError, \
        """type '%s' is not supported right now. Sorry about that.""" \
    % type

    # Get the pointer to the buffer data area
    # the second parameter means whether the buffer is read-only or not
    buflen = NA_getBufferPtrAndSize(naarr._data, 1, &rbuf)
    # Correct the start of the buffer with the _byteoffset
    offset = naarr._byteoffset
    rbuf = <void *>(<char *>rbuf + offset)

    # Allocate space for the dimension axis info
    self.rank = len(naarr.shape)
    self.dims = <hsize_t *>malloc(self.rank * sizeof(hsize_t))
    # Fill the dimension axis info with adequate info (and type!)
    for i from  0 <= i < self.rank:
      self.dims[i] = naarr.shape[i]

    # Save the array
    flavor = PyString_AsString(self.flavor)
    complib = PyString_AsString(self.filters.complib)
    version = PyString_AsString(self._v_version)
    class_ = PyString_AsString(self._c_classId)
    oid = H5ARRAYmake(self.parent_id, self.name, class_, title,
                      flavor, version, self.rank, self.dims, self.extdim,
                      self.type_id, NULL, NULL,
                      self.filters.complevel, complib,
                      self.filters.shuffle, self.filters.fletcher32,
                      rbuf)
    if oid < 0:
      raise HDF5ExtError("Problems creating the %s." % self.__class__.__name__)
    self._v_objectID = oid
    H5Tclose(self.type_id)    # Release resources

    return (type, stype)

  def _createEArray(self, char *title):
    cdef int i, enumtype
    cdef herr_t ret
    cdef hid_t oid
    cdef void *rbuf
    cdef char *byteorder
    cdef char *flavor, *complib, *version, *class_
    cdef void *fill_value
    cdef int itemsize

    atom = self.atom
    itemsize = atom.itemsize
    
    try:
      # Since Time columns have no Numarray type of their own,
      # a special case is made for them.
      stype = atom.stype
      if stype == 'Time32':
        enumtype = ord('t')
      elif stype == 'Time64':
        enumtype = ord('T')
      else:
        enumtype = naTypeToNAEnum[self.type]
    except KeyError:
      raise TypeError, \
            """Type class '%s' not supported right now. Sorry about that.
            """ % repr(self.type)

    if stype == 'Enum':
      self.type_id = enumToHDF5(atom, self.byteorder)
    else:
      byteorder = PyString_AsString(self.byteorder)
      self.type_id = convArrayType(enumtype, itemsize, byteorder)
      if self.type_id < 0:
        raise TypeError, \
          """type '%s' is not supported right now. Sorry about that.""" \
      % self.type

    fill_value = NULL

    self.rank = len(self.shape)
    self.dims = <hsize_t *>malloc(self.rank * sizeof(hsize_t))
    if self._v_chunksize:
      self.dims_chunk = <hsize_t *>malloc(self.rank * sizeof(hsize_t))
    # Fill the dimension axis info with adequate info
    for i from  0 <= i < self.rank:
      self.dims[i] = self.shape[i]
      if self._v_chunksize:
        self.dims_chunk[i] = self._v_chunksize[i]

    rbuf = NULL   # The data pointer. We don't have data to save initially
    # Manually convert some string values that can't be done automatically
    flavor = PyString_AsString(atom.flavor)
    complib = PyString_AsString(self.filters.complib)
    version = PyString_AsString(self._v_version)
    class_ = PyString_AsString(self._c_classId)
    fill_value = <void *>malloc(<size_t> itemsize)
    if(fill_value):
      for i from  0 <= i < itemsize:
        (<char *>fill_value)[i] = 0
    else:
      raise HDF5ExtError("Unable to allocate memory for fill_value.")
    # Create the EArray
    oid = H5ARRAYmake(self.parent_id, self.name, class_, title,
                      flavor, version, self.rank, self.dims, self.extdim,
                      self.type_id, self.dims_chunk, fill_value,
                      self.filters.complevel, complib,
                      self.filters.shuffle, self.filters.fletcher32,
                      rbuf)
    if oid < 0:
      raise HDF5ExtError("Problems creating the %s." % self.__class__.__name__)
    self._v_objectID = oid

    # Release resources
    H5Tclose(self.type_id)
    if(fill_value): free(fill_value)
    return


  def _loadEnum(self):
    """_loadEnum() -> (Enum, naType)
    Load enumerated type associated with this array.

    This method loads the HDF5 enumerated type associated with this
    array.  It returns an `Enum` instance built from that, and the
    Numarray type used to encode it.
    """

    cdef hid_t typeId, enumId

    # Get the type of the array.
    typeId = getLeafHDF5Type(self.parent_id, self.name)
    enumId = getTypeEnum(typeId)
    # close `typeId`
    H5Tclose(typeId)

    # Get the Enum and Numarray types and close the HDF5 type.
    try:
      return enumFromHDF5(enumId)
    finally:
      # (Yes, the ``finally`` clause *is* executed.)
      if H5Tclose(enumId) < 0:
        raise HDF5ExtError("failed to close HDF5 enumerated type")


  def _openArray(self):
    cdef object shape
    cdef size_t type_size, type_precision
    cdef H5T_class_t class_id
    cdef H5T_sign_t sign
    cdef char byteorder[16]  # "non-relevant" fits easily here
    cdef int i, enumtype
    cdef hid_t oid
    cdef int extdim
    cdef char flavor[256]

    # Get the rank for this array object
    if H5ARRAYget_ndims(self.parent_id, self.name, &self.rank) < 0:
        raise HDF5ExtError("Problems getting ndims!")
    # Allocate space for the dimension axis info
    self.dims = <hsize_t *>malloc(self.rank * sizeof(hsize_t))
    self.maxdims = <hsize_t *>malloc(self.rank * sizeof(hsize_t))
    # Get info on dimensions, class and type size
    oid = H5ARRAYget_info(self.parent_id, self.name, self.dims, self.maxdims,
                          &self.type_id, &class_id, byteorder)
    if oid < 0:
      raise HDF5ExtError("Unable to get array info.")

    self._v_objectID = oid
    strcpy(flavor, "NumArray")  # Default value
    self.extdim = -1  # default is non-chunked Array
#     if self._v_file._isPTFile:
#       if (hasattr(self._v_attrs,"EXTDIM")):
#         # For EArray, EXTDIM attribute exists
#         H5LTget_attribute_int(self.parent_id, self.name, "EXTDIM", &extdim)
#         self.extdim = extdim
#     else:
    # Case of a non-pytables possible native chunked dataset
    # It is better to always use this, so that the _v_attrs instance
    # shouldn't need to be created. This saves quite a lot of time during
    # file openings.
    # F. Altet 2005-06-05
    for i from 0 <= i < self.rank:
      if self.maxdims[i] == -1:
        self.extdim = i
        break

    if self._v_file._isPTFile:
      H5LTget_attribute_string(self.parent_id, self.name, "FLAVOR", flavor)
    self.flavor = flavor  # Gives class visibility to flavor

    # Allocate space for the dimension chunking info
    self.dims_chunk = <hsize_t *>malloc(self.rank * sizeof(hsize_t))
    if ( (H5ARRAYget_chunksize(self.parent_id, self.name,
                               self.rank, self.dims_chunk)) < 0):
      #H5ARRAYget_chunksize frees dims_chunk
      self.dims_chunk = NULL
      if self.extdim >= 0 or self.__class__.__name__ == 'CArray':
        raise HDF5ExtError, "Problems getting the chunksizes!"
    # Get the array type
    type_size = getArrayType(self.type_id, &enumtype)
    if type_size < 0:
      raise TypeError, "HDF5 class %d not supported. Sorry!" % class_id

    H5Tclose(self.type_id)    # Release resources

    # We had problems when creating Tuples directly with Pyrex!.
    # A bug report has been sent to Greg Ewing and here is his answer:
    """

    It's impossible to call PyTuple_SetItem and PyTuple_GetItem
    correctly from Pyrex, because they don't follow the standard
    reference counting protocol (PyTuple_GetItem returns a borrowed
    reference, and PyTuple_SetItem steals a reference).

    It's best to use Python constructs to create tuples if you
    can. Otherwise, you could create wrapppers for these functions in
    an external C file which provide standard reference counting
    behaviour.

    """
    # So, I've decided to create the shape tuple using Python constructs
    shape = []
    chunksizes = []
    for i from 0 <= i < self.rank:
      shape.append(self.dims[i])
      if self.dims_chunk:
        chunksizes.append(<int>self.dims_chunk[i])
    shape = tuple(shape)
    chunksizes = tuple(chunksizes)

    return (naEnumToNAType.get(enumtype, None), naEnumToNASType[enumtype],
            shape, type_size, byteorder, chunksizes)

  def _convertTypes(self, object naarr, int sense):
    """Converts Time64 elements in 'naarr' between Numarray and HDF5 formats.

    Numarray to HDF5 conversion is performed when 'sense' is 0.
    Otherwise, HDF5 to Numarray conversion is performed.
    The conversion is done in place, i.e. 'naarr' is modified.
    """

    # This should be generalised to support other type conversions.
    if self.stype == 'Time64':
      convertTime64(naarr, len(naarr), sense)

  def _append(self, object naarr):
    cdef int ret
    cdef hsize_t *dims_arr
    cdef void *rbuf
    cdef long offset
    cdef int buflen
    cdef object shape
    cdef int extdim

    # Allocate space for the dimension axis info
    dims_arr = <hsize_t *>malloc(self.rank * sizeof(hsize_t))
    # Fill the dimension axis info with adequate info (and type!)
    for i from  0 <= i < self.rank:
        dims_arr[i] = naarr.shape[i]

    # Get the pointer to the buffer data area
    # Both methods do the same
    buflen = NA_getBufferPtrAndSize(naarr._data, 1, &rbuf)
    # Correct the start of the buffer with the _byteoffset
    offset = naarr._byteoffset
    rbuf = <void *>(<char *>rbuf + offset)

    # Convert some Numarray types to HDF5 before storing.
    self._convertTypes(naarr, 0)

    # Append the records:
    extdim = self.extdim
    Py_BEGIN_ALLOW_THREADS
    ret = H5ARRAYappend_records(self.parent_id, self.name, self.rank,
                                self.dims, dims_arr, extdim, rbuf)
    Py_END_ALLOW_THREADS
    if ret < 0:
      raise HDF5ExtError("Problems appending the elements")

    free(dims_arr)
    # Update the new dimensionality
    shape = list(self.shape)
    shape[self.extdim] = self.dims[self.extdim]
    self.shape = tuple(shape)
    self.nrows = self.dims[self.extdim]

  def _modify(self, object startl, object stepl, object countl,
              object naarr):
    cdef int ret
    cdef void *rbuf, *temp
    cdef hsize_t *start, *step, *count
    cdef long buflen, offset

    # Get the pointer to the buffer data area
    buflen = NA_getBufferPtrAndSize(naarr._data, 1, &rbuf)
    # Correct the start of the buffer with the _byteoffset
    offset = naarr._byteoffset
    rbuf = <void *>(<char *>rbuf + offset)

    # Get the start, step and count values
    buflen = NA_getBufferPtrAndSize(startl._data, 1, <void **>&start)
    buflen = NA_getBufferPtrAndSize(stepl._data, 1, <void **>&step)
    buflen = NA_getBufferPtrAndSize(countl._data, 1, <void **>&count)

    # Convert some Numarray types to HDF5 before storing.
    self._convertTypes(naarr, 0)

    # Modify the elements:
    Py_BEGIN_ALLOW_THREADS
    ret = H5ARRAYwrite_records(self.parent_id, self.name, self.rank,
                               start, step, count, rbuf)
    Py_END_ALLOW_THREADS
    if ret < 0:
      raise HDF5ExtError("Internal error modifying the elements (H5ARRAYwrite_records returned errorcode -%i)"%(-ret))

  def _truncateArray(self, hsize_t size):
    cdef hsize_t extdim
    cdef hsize_t ret

    extdim = self.extdim
    if size >= self.shape[extdim]:
      return self.shape[extdim]

    ret = H5ARRAYtruncate(self.parent_id, self.name, extdim, size)
    if ret < 0:
      raise HDF5ExtError("Problems truncating the EArray node.")

    # Update the new dimensionality
    shape = list(self.shape)
    shape[self.extdim] = size
    self.shape = tuple(shape)
    self.nrows = size

    return

  def _readArray(self, hsize_t start, hsize_t stop, hsize_t step,
                 object naarr):
    cdef herr_t ret
    cdef void *rbuf
    cdef long buflen
    cdef hsize_t nrows
    cdef int extdim

    # Get the pointer to the buffer data area
    buflen = NA_getBufferPtrAndSize(naarr._data, 1, &rbuf)

    # Number of rows to read
    nrows = ((stop - start - 1) / step) + 1  # (stop-start)/step  do not work
    if hasattr(self, "extdim"):
      extdim = self.extdim
    else:
      exdim = -1
    Py_BEGIN_ALLOW_THREADS
    ret = H5ARRAYread(self.parent_id, self.name, start, nrows, step,
                      extdim, rbuf)
    Py_END_ALLOW_THREADS
    if ret < 0:
      raise HDF5ExtError("Problems reading the array data.")

    # Convert some HDF5 types to Numarray after reading.
    self._convertTypes(naarr, 1)

    return

  def _g_readSlice(self, startl, stopl, stepl, bufferl):
    cdef herr_t ret
    cdef long ndims, buflen
    cdef void *startlb, *stoplb, *steplb, *rbuflb
    cdef long offset

    # Get the pointer to the buffer data area of startl, stopl and stepl arrays
    ndims = NA_getBufferPtrAndSize(startl._data, 1, &startlb)
    ndims = NA_getBufferPtrAndSize(stopl._data, 1, &stoplb)
    ndims = NA_getBufferPtrAndSize(stepl._data, 1, &steplb)
    # Get the pointer to the buffer data area
    buflen = NA_getBufferPtrAndSize(bufferl._data, 1, &rbuflb)
    # Correct the start of the buffer with the _byteoffset
    offset = bufferl._byteoffset
    rbuflb = <void *>(<char *>rbuflb + offset)
    # Do the physical read
    ret = H5ARRAYreadSlice(self.parent_id, self.name,
                           <hsize_t *>startlb, <hsize_t *>stoplb,
                           <hsize_t *>steplb, rbuflb)
    if ret < 0:
      raise HDF5ExtError("Problems reading the array data.")

    # Convert some HDF5 types to Numarray after reading.
    self._convertTypes(bufferl, 1)

    return

  def __dealloc__(self):
    #print "Destroying object Array in Extension"
    free(<void *>self.dims)
    free(<void *>self.name)
    if self.maxdims:
      free(<void *>self.maxdims)
    if self.dims_chunk:
      free(self.dims_chunk)


cdef class IndexArray(Array):
  """Homogeneous dataset for keeping sorted and index values"""
  cdef void    *rbuflb, *vrbufR, *vrbufA
  cdef hid_t   dataset_id, type_id2, space_id
  # It is necessary for arrRel and arrAbs to be accessible in IndexArray,
  # so let's this commented out
  #cdef object arrRel, arrAbs

  def _initIndexSlice(self, maxCoords):
    "Initialize the structures for doing a binary search"
    cdef long buflen

    # Get the pointer to the buffer data area
    self.arrRel = numarray.zeros(type="Int32",shape=(maxCoords,))
    self.arrAbs = numarray.zeros(type="Int64",shape=(maxCoords,))
    buflen = NA_getBufferPtrAndSize(self.arrRel._data, 1, &self.vrbufR)
    buflen = NA_getBufferPtrAndSize(self.arrAbs._data, 1, &self.vrbufA)

    # Open the array for reading
    if (H5ARRAYOopen_readSlice(&self.dataset_id, &self.space_id,
                               &self.type_id2, self.parent_id, self.name) < 0):
      raise HDF5ExtError("Problems opening the sorted array data.")

  def _g_readIndex(self, hsize_t irow, hsize_t start, hsize_t stop,
                   long offsetl):
    cdef herr_t ret
    cdef long buflen
    cdef int *rbufR
    cdef long long *rbufA
    cdef long long offset
    cdef long j, len

    # Correct the start of the buffer with offsetl
    rbufR = <int *>self.vrbufR + offsetl
    rbufA = <long long *>self.vrbufA + offsetl
    # Do the physical read
    Py_BEGIN_ALLOW_THREADS
    ret = H5ARRAYOread_readSlice(self.dataset_id, self.space_id, self.type_id2,
                                 irow, start, stop, rbufR)
    Py_END_ALLOW_THREADS
    if ret < 0:
      raise HDF5ExtError("Problems reading the array data.")

    # Now, compute the absolute coords for table rows by adding the offset
    len = stop-start
    offset = irow*self.nelemslice
    for j from 0 <= j < len:
      rbufA[j] = rbufR[j] + offset

    return

  def _destroyIndexSlice(self):
    # Close the array for reading
    if H5ARRAYOclose_readSlice(self.dataset_id, self.space_id,
                               self.type_id2) < 0:
      raise HDF5ExtError("Problems closing the sorted array data.")

  def _initSortedSlice(self, int bufsize):
    "Initialize the structures for doing a binary search"
    cdef long ndims
    cdef int buflen

    # Create the buffer for reading sorted data chunks
    if str(self.type) == "CharType":
      self.bufferl = strings.array(None, itemsize=self.itemsize, shape=bufsize)
    else:
      self.bufferl = numarray.array(None, type=self.type, shape=bufsize)
      # Set the same byteorder than on-disk
      self.bufferl._byteorder = self.byteorder

    # Get the pointer to the buffer data area
    buflen = NA_getBufferPtrAndSize(self.bufferl._data, 1, &self.rbuflb)

    # Open the array for reading
    if (H5ARRAYOopen_readSlice(&self.dataset_id, &self.space_id,
                               &self.type_id2, self.parent_id, self.name) < 0):
      raise HDF5ExtError("Problems opening the sorted array data.")

  def _readSortedSlice(self, hsize_t irow, hsize_t start, hsize_t stop):
    "Read the sorted part of an index"

    Py_BEGIN_ALLOW_THREADS
    ret = H5ARRAYOread_readSlice(self.dataset_id, self.space_id, self.type_id2,
                                 irow, start, stop, self.rbuflb)
    Py_END_ALLOW_THREADS
    if ret < 0:
      raise HDF5ExtError("Problems reading the array data.")

    return self.bufferl

  def _destroySortedSlice(self):
    del self.bufferl
    # Close the array for reading
    if H5ARRAYOclose_readSlice(self.dataset_id, self.space_id,
                               self.type_id2) < 0:
      raise HDF5ExtError("Problems closing the sorted array data.")

# This has been copied from the standard module bisect.
# Checks for the values out of limits has been added at the beginning
# because I forsee that this should be a very common case.
# 2004-05-20
  def _bisect_left(self, a, x, int hi):
    """Return the index where to insert item x in list a, assuming a is sorted.

    The return value i is such that all e in a[:i] have e < x, and all e in
    a[i:] have e >= x.  So if x already appears in the list, i points just
    before the leftmost x already there.

    """
    cdef int lo, mid

    lo = 0
    if x <= a[0]: return 0
    if a[-1] < x: return hi
    while lo < hi:
        mid = (lo+hi)/2
        if a[mid] < x: lo = mid+1
        else: hi = mid
    return lo

  def _bisect_right(self, a, x, int hi):
    """Return the index where to insert item x in list a, assuming a is sorted.

    The return value i is such that all e in a[:i] have e <= x, and all e in
    a[i:] have e > x.  So if x already appears in the list, i points just
    beyond the rightmost x already there.

    """
    cdef int lo, mid

    lo = 0
    if x < a[0]: return 0
    if a[-1] <= x: return hi
    while lo < hi:
      mid = (lo+hi)/2
      if x < a[mid]: hi = mid
      else: lo = mid+1
    return lo

  def _interSearch_left(self, int nrow, int chunksize, item, int lo, int hi):
    cdef int niter, mid, start, result, beginning

    niter = 0
    beginning = 0
    while lo < hi:
      mid = (lo+hi)/2
      start = (mid/chunksize)*chunksize
      buffer = self._readSortedSlice(nrow, start, start+chunksize)
      #buffer = xrange(start,start+chunksize) # test
      niter = niter + 1
      result = self._bisect_left(buffer, item, chunksize)
      if result == 0:
        if buffer[result] == item:
          lo = start
          beginning = 1
          break
        # The item is at left
        hi = mid
      elif result == chunksize:
        # The item is at the right
        lo = mid+1
      else:
        # Item has been found. Exit the loop and return
        lo = result+start
        break
    return (lo, beginning, niter)

  def _interSearch_right(self, int nrow, int chunksize, item, int lo, int hi):
    cdef int niter, mid, start, result, ending

    niter = 0
    ending = 0
    while lo < hi:
      mid = (lo+hi)/2
      start = (mid/chunksize)*chunksize
      buffer = self._readSortedSlice(nrow, start, start+chunksize)
      niter = niter + 1
      result = self._bisect_right(buffer, item, chunksize)
      if result == 0:
        # The item is at left
        hi = mid
      elif result == chunksize:
        if buffer[result-1] == item:
          lo = start+chunksize
          ending = 1
          break
        # The item is at the right
        lo = mid+1
      else:
        # Item has been found. Exit the loop and return
        lo = result+start
        break
    return (lo, ending, niter)

# This is coded in python space as well, but the improvement in speed
# here is very little. So, it's better to let _searchBin live there.
# F. Altet 2004-12-27
#   def _searchBin(self, int nrow, object item, object rangeValues):
#     cdef int hi, chunksize, item1done, item2done
#     cdef int result1, result2, nchunk, lbounds
#     cdef object item1, item2, begin, end, bounds, chunk

#     item1, item2 = item
#     hi = self.shape[1]
#     chunksize = self.chunksize # Number of elements/chunksize
#     item1done = 0; item2done = 0

#     # First, look at the beginning of the slice
#     begin = rangeValues[nrow,0]
#     # Look for items at the beginning of sorted slices
#     if item1 <= begin:
#       result1 = 0
#       item1done = 1
#     if item2 < begin:
#       result2 = 0
#       item2done = 1
#     if item1done and item2done:
#       #print "done 1"
#       return (result1, result2)
#     # Then, look for items at the end of the sorted slice
#     end = rangeValues[nrow,1]
#     if not item1done:
#       if item1 > end:
#         result1 = hi
#         item1done = 1
#     if not item2done:
#       if item2 >= end:
#         result2 = hi
#         item2done = 1
#     if item1done and item2done:
#       #print "done 2"
#       return (result1, result2)
#     # Finally, do a lookup for item1 and item2 if they were not found
#     # Lookup in the middle of slice for item1
#     bounds = self._v_parent.bounds[nrow]
#     lbounds = len(bounds)
#     if not item1done:
#       # Search the appropriate chunk in bounds cache
#       nchunk = self._bisect_left(bounds, item1, lbounds)
#       chunk = self._readSortedSlice(nrow, chunksize*nchunk,
#                                     chunksize*(nchunk+1))
#       result1 = self._bisect_left(chunk, item1, chunksize)
#       result1 = result1 + chunksize*nchunk
#     # Lookup in the middle of slice for item2
#     if not item2done:
#       # Search the appropriate chunk in bounds cache
#       nchunk = self._bisect_right(bounds, item2, lbounds)
#       chunk = self._readSortedSlice(nrow, chunksize*nchunk,
#                                     chunksize*(nchunk+1))
#       result2 = self._bisect_right(chunk, item2, chunksize)
#       result2 = result2 + chunksize*nchunk
#     return (result1, result2)


cdef class Index:
  """Container for sorted and indices datasets"""
  # Instance variables

  # Methods
  pass
#   def search(self, item, notequal):
#     """Do a binary search in this index for an item"""
#     cdef long ntotaliter, tlen, bufsize, i, start, stop, niter

#     #t1=time.time()
#     ntotaliter = 0; tlen = 0
#     self.starts = []; self.lengths = []
#     bufsize = self.sorted._v_chunksize[1] # number of elements/chunksize
#     self.nelemslice = self.sorted.nelemslice   # number of columns/slice
#     self.sorted._initSortedSlice(bufsize)
#     # Do the lookup for values fullfilling the conditions
#     #for i in xrange(self.sorted.nrows):
#     for i from 0 <= i < self.sorted.nrows:
#       (start, stop, niter) = self.sorted._searchBin(i, item)
#       self.starts.append(start)
#       self.lengths.append(stop - start)
#       ntotaliter = ntotaliter + niter
#       tlen = tlen + (stop - start)
#     self.sorted._destroySortedSlice()
#     #print "time reading indices:", time.time()-t1
#     return tlen

#   def getCoords(self, long long startCoords, long maxCoords):
#     """Get the coordinates of indices satisfiying the cuts"""
#     cdef long len1, len2, leni, stop, relCoords, irow, startl, stopl

#     #t1=time.time()
#     len1 = 0; len2 = 0;
#     stop = 0; relCoords = 0
#     for irow from  0 <= irow < self.sorted.nrows: 
#       leni = self.lengths[irow]; len2 = len2 + leni
#       if (leni > 0 and len1 <= startCoords < len2):
#         startl = self.starts[irow] + (startCoords-len1)
#         if maxCoords >= leni - (startCoords-len1):
#           # Values fit on buffer
#           stopl = startl + leni
#         else:
#           # Stop after this iteration
#           stopl = startl + maxCoords
#           stop = 1
#         self.indices._g_readIndex(irow, startl, stopl, relCoords)
#         relCoords = relCoords + stopl - startl
#         if stop:
#           break
#         maxCoords = maxCoords - (leni - (startCoords-len1))
#         startCoords = startCoords + (leni - (startCoords-len1))
#       len1 = len1 + leni
#     selections = numarray.sort(self.indices.arrAbs[:relCoords])
#     #selections = self.indices.arrAbs[:relCoords]
#     #print "time doing revIndexing:", time.time()-t1
#     return selections


cdef class VLArray:
  # Instance variables
  cdef hid_t   parent_id
  cdef hid_t   oid
  cdef char    *name
  cdef int     rank
  cdef hsize_t *dims
  cdef hid_t   type_id
  cdef hsize_t nrecords
  cdef int     scalar

  def _g_new(self, where, name):
    # Initialize the C attributes of Group object (Very important!)
    self.name = strdup(name)
    # The parent group id for this object
    self.parent_id = where._v_objectID

  def _createArray(self, char *title):
    cdef int i, enumtype
    cdef herr_t ret
    cdef void *rbuf
    cdef char *byteorder
    cdef char *flavor, *complib, *version, *class_

    atom = self.atom
    type_ = atom.type
    stype = atom.stype
    try:
      # Since Time columns have no Numarray type of their own,
      # a special case is made for them.
      if stype == 'Time32':
        enumtype = ord('t')
      elif stype == 'Time64':
        enumtype = ord('T')
      else:
        enumtype = naTypeToNAEnum[type_]
    except KeyError:
      raise TypeError, \
            """Type class '%s' not supported right now. Sorry about that.
            """ % repr(type_)

    if stype == 'Enum':
      self.type_id = enumToHDF5(atom, self.byteorder)
    else:
      byteorder = PyString_AsString(self.byteorder)
      # Get the HDF5 type id
      self.type_id = convArrayType(enumtype, atom.itemsize, byteorder)
      if self.type_id < 0:
        raise TypeError, \
          """type '%s' is not supported right now. Sorry about that.""" \
      % type_

    # Allocate space for the dimension axis info
    if isinstance(atom.shape, int):
      self.rank = 1
      self.scalar = 1
    else:
      self.rank = len(atom.shape)
      self.scalar = 0

    self.dims = <hsize_t *>malloc(self.rank * sizeof(hsize_t))
    # Fill the dimension axis info with adequate info
    for i from  0 <= i < self.rank:
      if isinstance(atom.shape, int):
        self.dims[i] = atom.shape
      else:
        self.dims[i] = atom.shape[i]

    rbuf = NULL   # We don't have data to save initially

    # Manually convert some string values that can't be done automatically
    flavor = PyString_AsString(atom.flavor)
    complib = PyString_AsString(self.filters.complib)
    version = PyString_AsString(self._v_version)
    class_ = PyString_AsString(self._c_classId)
    # Create the vlarray
    oid = H5VLARRAYmake(self.parent_id, self.name, class_, title,
                        flavor, version, self.rank, self.scalar,
                        self.dims, self.type_id, self._v_chunksize, rbuf,
                        self.filters.complevel, complib,
                        self.filters.shuffle, self.filters.fletcher32,
                        rbuf)
    if oid < 0:
      raise HDF5ExtError("Problems creating the VLArray.")
    self._v_objectID = oid
    self.nrecords = 0  # Initialize the number of records saved


  def _loadEnum(self):
    """_loadEnum() -> (Enum, naType)
    Load enumerated type associated with this array.

    This method loads the HDF5 enumerated type associated with this
    array.  It returns an `Enum` instance built from that, and the
    Numarray type used to encode it.
    """

    cdef hid_t typeId, rowTypeId, enumId

    # Get the array type of the array.
    typeId = getLeafHDF5Type(self.parent_id, self.name)

    # Get the enumerated type.
    rowTypeId = H5Tget_super(typeId)
    enumId = getTypeEnum(rowTypeId)  # `rowTypeId` will be closed
    # close `typeId` and `rowTypeId`
    H5Tclose(typeId)
    H5Tclose(rowTypeId)

    # Get the Enum and Numarray types and close the HDF5 type.
    try:
      return enumFromHDF5(enumId)
    finally:
      # (Yes, the ``finally`` clause *is* executed.)
      if H5Tclose(enumId) < 0:
        raise HDF5ExtError("failed to close HDF5 enumerated type")


  def _openArray(self):
    cdef object shape
    cdef char byteorder[16]  # "non-relevant" fits easily here
    cdef int i, enumtype
    cdef hid_t oid
    cdef herr_t ret
    cdef hsize_t nrecords[1]
    cdef char flavor[256]

    # Get the rank for the atom in the array object
    ret = H5VLARRAYget_ndims(self.parent_id, self.name, &self.rank)
    # Allocate space for the dimension axis info
    if self.rank:
      self.dims = <hsize_t *>malloc(self.rank * sizeof(hsize_t))
    else:
      self.dims = NULL;
    # Get info on dimensions, class and type size
    oid = H5VLARRAYget_info(self.parent_id, self.name, nrecords,
                            self.dims, &self.type_id, byteorder)
    self._v_objectID = oid
    ret = H5LTget_attribute_string(self.parent_id, self.name, "FLAVOR", flavor)
    if ret < 0:
      strcpy(flavor, "unknown")

    # Give to these variables class visibility
    self.flavor = flavor
    self.byteorder = byteorder

    # Get the array type
    self._basesize = getArrayType(self.type_id, &enumtype)
    if self._basesize < 0:
      raise TypeError, "The HDF5 class of object does not seem VLEN. Sorry!"

    # Get the type of the atomic type
    self._atomictype = naEnumToNAType.get(enumtype, None)
    self._atomicstype = naEnumToNASType[enumtype]
    # Get the size and shape of the atomic type
    self._atomicsize = self._basesize
    if self.rank:
      shape = []
      for i from 0 <= i < self.rank:
        shape.append(self.dims[i])
        self._atomicsize = self._atomicsize * self.dims[i]
      shape = tuple(shape)
    else:
      # rank zero means a scalar
      shape = 1

    self._atomicshape = shape
    self.nrecords = nrecords[0]  # Initialize the number of records saved
    return nrecords[0]

  def _convertTypes(self, object naarr, int sense):
    """Converts Time64 elements in 'naarr' between Numarray and HDF5 formats.

    Numarray to HDF5 conversion is performed when 'sense' is 0.
    Otherwise, HDF5 to Numarray conversion is performed.
    The conversion is done in place, i.e. 'naarr' is modified.
    """

    # This should be generalised to support other type conversions.
    if self._atomicstype == 'Time64':
      convertTime64(naarr, len(naarr), sense)

  def _append(self, object naarr, int nobjects):
    cdef int ret
    cdef void *rbuf
    cdef long buflen, offset

    # Get the pointer to the buffer data area
    if nobjects:
      buflen = NA_getBufferPtrAndSize(naarr._data, 1, &rbuf)
      # Correct the start of the buffer with the _byteoffset
      offset = naarr._byteoffset
      rbuf = <void *>(<char *>rbuf + offset)

      # Convert some Numarray types to HDF5 before storing.
      self._convertTypes(naarr, 0)
    else:
      rbuf = NULL

    # Append the records:
    Py_BEGIN_ALLOW_THREADS
    ret = H5VLARRAYappend_records(self.parent_id, self.name,
                                  nobjects, self.nrecords,
                                  rbuf)
    Py_END_ALLOW_THREADS
    if ret < 0:
      raise HDF5ExtError("Problems appending the records.")

    self.nrecords = self.nrecords + 1

  def _modify(self, hsize_t nrow, object naarr, int nobjects):
    cdef int ret
    cdef void *rbuf
    cdef long buflen, offset

    # Get the pointer to the buffer data area
    buflen = NA_getBufferPtrAndSize(naarr._data, 1, &rbuf)
    # Correct the start of the buffer with the _byteoffset
    offset = naarr._byteoffset
    rbuf = <void *>(<char *>rbuf + offset)

    if nobjects:
      # Convert some Numarray types to HDF5 before storing.
      self._convertTypes(naarr, 0)

    # Append the records:
    Py_BEGIN_ALLOW_THREADS
    ret = H5VLARRAYmodify_records(self.parent_id, self.name,
                                  nrow, nobjects, rbuf)
    Py_END_ALLOW_THREADS
    if ret < 0:
      raise HDF5ExtError("Problems modifying the record.")

    return nobjects

  def _readArray(self, hsize_t start, hsize_t stop, hsize_t step):
    cdef herr_t ret
    cdef hvl_t *rdata   # Information read in
    cdef size_t vllen
    cdef hsize_t rdatalen
    cdef object rbuf, naarr, shape, datalist
    cdef int i
    cdef hsize_t nrows

    # Compute the number of rows to read
    nrows = ((stop - start - 1) / step) + 1  # (stop-start)/step  do not work
    rdata = <hvl_t *>malloc(<size_t>nrows*sizeof(hvl_t))
    Py_BEGIN_ALLOW_THREADS
    ret = H5VLARRAYread(self.parent_id, self.name, start, nrows, step, rdata,
                        &rdatalen)
    Py_END_ALLOW_THREADS
    if ret < 0:
      raise HDF5ExtError("Problems reading the array data.")

    datalist = []
    for i from 0 <= i < nrows:
      # Number of atoms in row
      vllen = rdata[i].len
      # Get the pointer to the buffer data area
      if vllen > 0:
        rbuf = PyBuffer_FromReadWriteMemory(rdata[i].p, vllen*self._atomicsize)
      else:
        # Case where there is info with zero lentgh
        rbuf = None
      # Compute the shape for the read array
      if (isinstance(self._atomicshape, tuple)):
        shape = list(self._atomicshape)
        shape.insert(0, vllen)  # put the length at the beginning of the shape
      elif self._atomicshape > 1:
        shape = (vllen, self._atomicshape)
      else:
        # Case of scalars (self._atomicshape == 1)
        shape = (vllen,)
      # Create an array to keep this info
      # It seems that an array made from a buffer cannot be
      # pickled. We do a copy of the buffer so that the objects would
      # be able to get pickled without problems.
      # However this give problems (!) Return to the initial point :-/
      # F. Altet 2004-11-16
      if str(self._atomictype) == "CharType":
        naarr = strings.array(rbuf, itemsize=self._basesize, shape=shape)
      else:
        naarr = numarray.array(rbuf, type=self._atomictype, shape=shape)

      # Convert some HDF5 types to Numarray after reading.
      self._convertTypes(naarr, 1)

      datalist.append(naarr)

    # Release resources
    free(rdata)

    return datalist

  def __dealloc__(self):
    #print "Destroying object VLArray in Extension"
    H5Tclose(self.type_id)    # To avoid memory leaks
    if self.dims:
      free(<void *>self.dims)
    free(<void *>self.name)


cdef class UnImplemented:
  # Instance variables
  cdef hid_t   parent_id
  cdef char    *name

  def _g_new(self, where, name):
    # Initialize the C attributes of Group object (Very important!)
    self.name = strdup(name)
    # The parent group id for this object
    self.parent_id = where._v_objectID

  def _openUnImplemented(self):
    cdef object shape
    cdef char byteorder[16]  # "non-relevant" fits easily here

    # Get info on dimensions
    shape = H5UIget_info(self.parent_id, self.name, byteorder)
    return (shape, byteorder)

  def __dealloc__(self):
    free(<void *>self.name)
