########################################################################
#
#       License: BSD
#       Created: September 4, 2002
#       Author:  Francesc Altet - faltet@carabos.com
#
#       $Source: /cvsroot/pytables/pytables/tables/Table.py,v $
#       $Id: Table.py 1186 2005-09-13 11:18:02Z faltet $
#
########################################################################

"""Here is defined the Table class.

See Table class docstring for more info.

Classes:

    Table
    Cols
    Column

Functions:


Misc variables:

    __version__


"""

import sys
import warnings
import re

import numarray
import numarray.records as records
import nestedrecords

try:
    import Numeric
    Numeric_imported = True
except ImportError:
    Numeric_imported = False

import tables.TableExtension as TableExtension
from tables.utils import calcBufferSize, processRange, \
     processRangeRead, joinPath
from tables.Leaf import Leaf
from tables.Index import Index, IndexProps
from tables.IsDescription import \
     IsDescription, Description, Col, StringCol, EnumCol
from tables.VLArray import Atom, StringAtom
from tables.Group import IndexesTableG, IndexesDescG
from tables.exceptions import NodeError, PerformanceWarning
from tables.constants import MAX_COLUMNS

__version__ = "$Revision: 1186 $"


# 2.2: Added support for complex types. Introduced in version 0.9.
# 2.2.1: Added suport for time types.
# 2.3: Changed the indexes naming schema.
# 2.4: Get rid of FIELD_%d_NAME attributes and changed indexes naming schema.
obversion = "2.4"  # The Table VERSION number


# Map Numarray record codes to Numarray types.
# This is extended with additional dataypes used by PyTables.
codeToNAType = records.numfmt.copy()
codeToNAType['t4'] = 'Time32'  # 32 bit integer time value
codeToNAType['t8'] = 'Time64'  # 64 bit real time value
codeToNAType['e'] = 'Enum'  # enumerated value, must also find base type


# Paths and names for hidden nodes related with indexes.
_indexName   = '_i_%s'  # %s -> encoded table path

# Compile a regular expression for expressions like '(2,2)Int8'
prog = re.compile(r'([\(\),\d\s]*)([A-Za-z]+[0-9]*)')

def _getEncodedTableName(tablename):
    return _indexName % tablename

def _getIndexTableName(parent, tablename):
    return joinPath(parent._v_pathname, _getEncodedTableName(tablename))

def _getIndexColName(parent, tablename, colname):
    return joinPath(_getIndexTableName(parent, tablename), colname)


class Table(TableExtension.Table, Leaf):
    """Represent a table in the object tree.

    It provides methods to create new tables or open existing ones, as
    well as to write/read data to/from table objects over the
    file. A method is also provided to iterate over the rows without
    loading the entire table or column in memory.

    Data can be written or read both as Row instances, numarray
    (NumArray or RecArray) objects or NestedRecArray objects.

    Methods:

        __getitem__(key)
        __iter__()
        __setitem__(key, value)
        append(rows)
        col(name)
        flushRowsToIndex()
        iterrows(start, stop, step)
        itersequence(sequence)
        modifyRows(start, rows)
        modifyColumn(columns, names, [start] [, stop] [, step])
        modifyColumns(columns, names, [start] [, stop] [, step])
        read([start] [, stop] [, step] [, field [, flavor]])
        reIndex()
        reIndexDirty()
        removeRows(start [, stop])
        removeIndex(column)
        where(condition [, start] [, stop] [, step])
        whereAppend(dstTable, condition [, start] [, stop] [, step])
        getWhereList(condition [, flavor])

    Instance variables:

        description -- the metaobject describing this table
        row -- a reference to the Row object associated with this table
        nrows -- the number of rows in this table
        rowsize -- the size, in bytes, of each row
        cols -- accessor to the columns using a natural name schema
        colnames -- the field names for the table (list)
        coltypes -- the type class for the table fields (dictionary)
        colstypes -- the string type for the table fields (dictionary)
        colshapes -- the shapes for the table fields (dictionary)
        colindexed -- whether the table fields are indexed (dictionary)
        indexed -- whether or not some field in Table is indexed
        indexprops -- properties of an indexed Table

    """

    # Class identifier.
    _c_classId = 'TABLE'

    # <undo-redo support>
    _c_canUndoCreate = True  # Can creation/copying be undone and redone?
    _c_canUndoRemove = True  # Can removal be undone and redone?
    _c_canUndoMove   = True  # Can movement/renaming be undone and redone?
    # </undo-redo support>

    def __init__(self, description = None, title = "",
                 filters = None, expectedrows = 10000):
        """Create an instance Table.

        Keyword arguments:

        description -- A IsDescription subclass or a dictionary where
            the keys are the field names, and the values the type
            definitions. And it can be also a RecArray or NestedRecArray
            object. If None, the table metadata is read from disk, else,
            it's taken from previous parameters.

        title -- Sets a TITLE attribute on the HDF5 table entity.

        filters -- An instance of the Filters class that provides
            information about the desired I/O filters to be applied
            during the life of this object.

        expectedrows -- An user estimate about the number of rows
            that will be on table. If not provided, the default value
            is appropiate for tables until 1 MB in size (more or less,
            depending on the record size). If you plan to save bigger
            tables try providing a guess; this will optimize the HDF5
            B-Tree creation and management process time and memory
            used.

        """

        # Common variables
        self._v_new_title = title
        self._v_new_filters = filters
        self._v_expectedrows = expectedrows
        # Initialize the number of rows to a default
        self.nrows = 0
        # Initialize the possible cuts in columns
        self.ops = []
        self.opsValues = []
        self.opsColnames = []
        # indexed is False initially
        self.indexed = False


        # Initialize this object in case is a new Table
        if isinstance(description, dict):
            # Dictionary case
            self.description = Description(description)
        elif isinstance(description, records.RecArray):
            # RecArray object case
            self._newRecArray(description)
            # Provide a better guess for the expected number of rows
            # But beware with the small recarray lengths!
            # Commented out until a better approach is found
            #if self._v_expectedrows == expectedrows:
            #    self._v_expectedrows = self.nrows
        elif (type(description) == type(IsDescription) and
              issubclass(description, IsDescription)):
            # IsDescription subclass case
            descr = description()
            self.description = Description(descr.columns)
        elif isinstance(description, Description):
            # It is a Description instance already
            self.description = description
        elif description is not None:
            raise TypeError("""\
``description`` argument is not of a supported type: \
``IsDescription`` subclass, dictionary, ``RecArray``or \
``NestedRecArray`` instance""")

        # Flag that tells if this table is new or it must be read from disk.
        self._v_new = description is not None

    def _newBuffer(self, init=1):
        """Create a new recarray buffer for I/O purposes"""

        colnames = self.description._v_nestedNames
        formats = self.description._v_nestedFormats
        #print "nestednames-->", colnames
        #print "formats-->", formats
        recarr = nestedrecords.array(None, formats=formats,
                                     shape=(self._v_maxTuples,),
                                     names = colnames)
        # Initialize the recarray with the defaults in description
        recarr._fields = recarr._get_fields()
        if init:
            for objcol in self.description._v_walk("Col"):
                colname = objcol._v_pathname
                recarr._fields[colname][:] =  objcol.dflt
        return recarr

    def _descrFromNRA(self, nra):
        "Get a description dictionary from a NestedRecArray"

        fields = {}
        i = 0
        for (colname, format) in nra.descr:
            if isinstance(format, str):
                # Column case
                shape, type = prog.search(format).groups()
                if shape == "":
                    shape = 1   # No shape. Put it to 1
                elif len(shape) == 1:
                    shape = int(shape)
                else:  # '(n, m, ...)'
                    # The next is safer and faster than eval(shape)
                    shape = tuple([ int(c) for c in shape[1:-1].split(',')
                                    if c.strip().isdigit() ])
                if type[0] == "a":
                    itemsize = int(type[1:])
                    type = "a"
                else:
                    type = numarray.typeDict[type]
                    itemsize = type.bytes
                #print "format-->", format
                #print "shape, type, isize-->", shape, type, itemsize
                # Special case for strings
                if type == 'a':
                    fields[colname] =  StringCol(length=itemsize,
                                                 dflt=None,
                                                 shape=shape,
                                                 pos=i)
                else:
                    fields[colname] = Col(dtype=type,
                                          shape=shape,
                                          pos=i)  # Position matters
            else:
                # Nested column
                fields[colname] = self._descrFromNRA(nra.field(colname))
                fields[colname]["_v_pos"] = i
            i += 1
        return fields

    def _descrFromRA(self, recarr):
        "Get a description dictionary from a RecArray"

        fields = {}
        self.colnames = recarr._names
        for i in xrange(len(self.colnames)):
            colname = self.colnames[i]
            # Special case for strings
            if isinstance(recarr._fmt[i], records.Char):
                fields[colname] =  StringCol(length=recarr._itemsizes[i],
                                             dflt=None,
                                             shape=recarr._repeats[i],
                                             pos=i)
            else:
                fields[colname] = Col(dtype=recarr._fmt[i],
                                      shape=recarr._repeats[i],
                                      pos=i)  # Position matters
        return fields

    def _newRecArray(self, recarr):
        """Save a recarray to disk, and map it as a Table object

        This method is aware of byteswapped and non-contiguous recarrays
        """

        # Check if recarray is discontigous:
        if not recarr.iscontiguous():
            # Make a copy to ensure that it is contiguous
            # We always should make a copy because I think that
            # HDF5 does not support strided buffers, but just offsets
            # between fields
            recarr = recarr.copy()
        # Initialize the number of rows
        self.nrows = len(recarr)
        # If self._v_recarray exists, and has data, it would be marked as
        # the initial buffer
        if self.nrows > 0:
            self._v_recarray = recarr
        if hasattr(recarr, "descr"):  # Quacks like a NestedRecArray
            fields = self._descrFromNRA(recarr)
        else:
            fields = self._descrFromRA(recarr)
        # Set the byteorder
        fields['_v_byteorder'] = recarr._byteorder
        # Create an instance description to host the record fields
        self.description = Description(fields)
        # The rest of the info is automatically added when self.create()
        # is called

    def _getTime64ColNames(self):
        """Returns a list containing 'Time64' column names."""

        # This should be generalised into some infrastructure to support
        # other kinds of columns to be converted.
        # ivilata(2004-12-21)

        return [ colobj._v_pathname
                 for colobj in self.description._v_walk('Col')
                 if colobj.stype == 'Time64' ]

    def _getEnumMap(self):
        """Return mapping from enumerated column names to `Enum` instances."""

        enumMap = {}
        for colobj in self.description._v_walk('Col'):
            if colobj.stype == 'Enum':
                enumMap[colobj._v_pathname] = colobj.enum
        return enumMap

    def _createIndexesTable(self, igroup):
        itgroup=IndexesTableG("Indexes container for table "+self._v_pathname)
        setattr(igroup, _getEncodedTableName(self._v_name), itgroup)
        # Assign the pathname table to this Group
        itgroup._v_attrs._g__setattr('PATHNAME', self._v_pathname)
        # Delete the FILTERS_INDEX attribute from table, so that
        # we don't have to syncronize it
        self._v_attrs._g__delattr('FILTERS_INDEX')
        return itgroup

    def _createIndexesDescr(self, igroup, dname, iname, filters):
        idgroup=IndexesDescG("Indexes container for sub-description "+dname,
                             filters=filters)
        setattr(igroup, iname, idgroup)
        # Assign the pathname table to this Group
        pathname = "%s.cols.%s" % (self._v_pathname, dname)
        idgroup._v_attrs._g__setattr('PATHNAME', pathname)
        return idgroup

    def _create(self):
        """Create a new table on disk."""

        # Initialize the shape attribute
        self.shape = (self.nrows,)
        # The size of this record
        self.rowsize = self.description._v_totalsize
        # Protection against too large row sizes
        # Set to a 512 KB limit (just because banana 640 KB limitation)
        if self.rowsize > 512*1024:
            raise ValueError, \
"""Row size too large. Maximum size is 512 Kbytes, and you are asking
for a row size of %s bytes.""" % (self.rowsize)
        # Get the byte order
        self.byteorder = self.description._v_byteorder

        # Find Time64 column names. (This should be generalised.)
        self._time64colnames = self._getTime64ColNames()
        # Get a mapping of enumerated columns to their `Enum` instances.
        self._colenums = self._getEnumMap()

        # Compute some values for buffering and I/O parameters
        (self._v_maxTuples, self._v_chunksize) = \
                            calcBufferSize(self.rowsize, self._v_expectedrows)

        # Create the table on disk
        self._createTable(self._v_new_title, self.filters.complib, obversion)
        # Warning against assigning too much columns...
        # F. Altet 2005-06-05
        self.colnames = tuple(self.description._v_nestedNames)
        if (len(self.colnames) > MAX_COLUMNS):
            warnings.warn("""\
table ``%s`` is exceeding the recommended maximum number of columns (%d);\
be ready to see PyTables asking for *lots* of memory and possibly slow I/O"""
                          % (self._v_pathname, MAX_COLUMNS),
                          PerformanceWarning)
        # Create the Row object helper
        # This is a property now
        #self.row = TableExtension.Row(self)

        # Compute some important parameters for createTable
        self.coltypes = {}      # The column types
        self.colstypes = {}     # The column string types
        self.colshapes = {}     # The column shapes
        self.colitemsizes = {}  # The column itemsizes
        self.colindexed = {}    # Is the key column indexed?
        self.indexed = False    # Are there any indexed columns?
        colobjects = self.description._v_colObjects
        for colobj in self.description._v_walk(type="Col"):
            colname = colobj._v_pathname
            # Get the column types and string types
            self.coltypes[colname] = colobj.type
            self.colstypes[colname] = colobj.stype
            # Extract the shapes and itemsizes for columns
            self.colshapes[colname] = colobj.shape
            self.colitemsizes[colname] = colobj.itemsize
            # Indexed?
            colindexed = colobj.indexed
            self.colindexed[colname] = colindexed
            if colindexed:
                self.indexed = True

        # We have to define indexprops here in order to propagate
        # index properties for eventual future index creation
        self.indexprops = getattr(self.description, '_v_indexprops',
                                  IndexProps())
        # Save AUTOMATIC_INDEX and REINDEX flags as attributes
        setAttr = self._v_attrs._g__setattr
        setAttr('AUTOMATIC_INDEX', self.indexprops.auto)
        setAttr('REINDEX', self.indexprops.reindex)
        # Filters is saved here until one index is created.
        setAttr('FILTERS_INDEX', self.indexprops.filters)
        if self.indexed:
            self._indexedrows = 0
            self._unsaved_indexedrows = 0

        # Create a cols accessor
        self.cols = Cols(self, self.description)

    def _open(self):
        """Opens a table from disk and read the metadata on it.

        Creates an user description on the flight to easy the access to
        the actual data.

        """
        # Get table info
        description = self._getInfo()
        if self._v_file._isPTFile:
            # Checking validity names for fields is not necessary
            # when opening a PyTables file
            # Do this nested!
            description['__check_validity__'] = 0
        # Create an instance description to host the record fields
        self.description = Description(description)
        # Add info for indexed columns
        for objcol in self.description._v_walk(type="Col"):
            colname = objcol._v_pathname
            indexname = _getIndexColName(self._v_parent, self._v_name, colname)
            colindexed = indexname in self._v_file._allNodes
            objcol.indexed = colindexed
        # Get the row size
        self.rowsize = self.description._v_totalsize
        # Update the shape attribute
        self.shape = (self.nrows,)
        # The expectedrows would be the actual number
        self._v_expectedrows = self.nrows
        # Get the byteorder
        self.byteorder = self.description._v_byteorder

        # Extract the coltypes, colstypes, shapes and itemsizes
        # self.colnames, coltypes, col*... should be removed?
        self.colnames = self.description._v_nestedNames

        # Find Time64 column names. (This should be generalised.)
        self._time64colnames = self._getTime64ColNames()
        # Get a mapping of enumerated columns to their `Enum` instances.
        self._colenums = self._getEnumMap()

        # Compute buffer size
        (self._v_maxTuples, self._v_chunksize) = \
              calcBufferSize(self.rowsize, self.nrows)

        # Get info about columns
        self.colnames = tuple(self.description._v_nestedNames)
        self.coltypes = {}      # The column types
        self.colstypes = {}     # The column string types
        self.colshapes = {}     # The column shapes
        self.colitemsizes = {}  # The column itemsizes
        for colobj in self.description._v_walk(type="Col"):
            colname = colobj._v_pathname
            # Get the column types and string types
            self.coltypes[colname] = colobj.type
            self.colstypes[colname] = colobj.stype
            # Extract the shapes and itemsizes for columns
            self.colshapes[colname] = colobj.shape
            self.colitemsizes[colname] = colobj.itemsize
        # Create a cols accessor
        self.cols = Cols(self, self.description)

    # Define row as a property
    def _getRow(self):
        mydict = self.__dict__
        if 'row' in mydict:
            return mydict['row']
        else:
            mydict['row'] = row = TableExtension.Row(self)
            return row

    row = property(_getRow, None, None, "Row accessor")

    def _g_afterOpen(self):
        """Tidy object tree after its creation."""

        # We are putting here the index-related issues
        # as well as filling general info for table
        # This is needed because we need first the index objects created
        self.colindexed = {}  # Is the specified column indexed?
        self.indexed = False      # Are there any indexed columns?

        # Do the indexes group exist?
        indexesGroupPath = _getIndexTableName(self._v_parent, self._v_name)
        igroup = indexesGroupPath in self._v_file._allNodes
        for colobj in self.description._v_walk(type="Col"):
            colname = colobj._v_pathname
            # Is this column indexed?
            if igroup:
                indexname = _getIndexColName(self._v_parent, self._v_name,
                                             colname)
                indexed = indexname in self._v_file._allNodes
                self.colindexed[colname] = indexed
                if indexed:
                    indexobj = self.cols._f_col(colname).index
            else:
                indexed = False
                self.colindexed[colname] = False
            if indexed:
                self.indexed = True

        # Create an index properties object.
        # It does not matter to which column 'indexobj' belongs,
        # since their respective index objects share
        # the same filters and number of elements.
        autoindex = getattr(self.attrs, 'AUTOMATIC_INDEX', None)
        reindex = getattr(self.attrs, 'REINDEX', None)
        if self.indexed:
            filters = indexobj.filters
        else:
            filters = getattr(self.attrs, 'FILTERS_INDEX', None)
        self.indexprops = IndexProps(auto=autoindex, reindex=reindex,
                                     filters=filters)
        if self.indexed:
            self._indexedrows = indexobj.nelements
            self._unsaved_indexedrows = self.nrows - self._indexedrows


    def _checkColumn(self, colname):
        """
        Check that the column named `colname` exists in the table.

        If it does not exist, a ``KeyError`` is raised.
        """

        for colobj in self.description._v_walk(type="All"):
            cname = colobj._v_pathname
            if colname == cname:
                break
        if colname <> cname:
            raise KeyError(
                "table ``%s`` does not have a column named ``%s``"
                % (self._v_pathname, colname))
        return colobj

    def where(self, condition, start=None, stop=None, step=None):
        """
        Iterate over values fulfilling a `condition`.

        This method returns an iterator yielding `Row` instances built
        from rows in the table that satisfy the given `condition` over a
        column.  If that column is indexed, its index will be used in
        order to accelerate the search.  Else, the *in-kernel* iterator
        (with has still better performance than standard Python
        selections) will be chosen instead.

        Moreover, if a range is supplied (i.e. some of the `start`,
        `stop` or `step` parameters are passed), only the rows in that
        range *and* fullfilling the `condition` are returned.  The
        meaning of the `start`, `stop` and `step` parameters is the same
        as in the ``range()`` Python function, except that negative
        values of `step` are *not* allowed.  Moreover, if only `start`
        is specified, then `stop` will be set to ``start+1``.

        You can mix this method with standard Python selections in order
        to have complex queries.  It is strongly recommended that you
        pass the most restrictive condition as the parameter to this
        method if you want to achieve maximum performance.

        Example of use::

            passvalues=[]
            for row in table.where(0 < table.cols.col1 < 0.3, step=5):
                if row['col2'] <= 20:
                    passvalues.append(row['col3'])
            print "Values that pass the cuts:", passvalues

        """

        if not isinstance(condition, Column):
            raise TypeError("""\
Wrong 'condition' parameter type. Only Column instances are suported.""")

        if not condition.shape in [1, (1,)]:
            raise NotImplementedError, "You cannot use in-kernel or indexed searches along multimensional columns. Use the regular table iterator for that."

        colindex = condition.index
        if colindex and not condition.dirty and colindex.nelements > 0:
            # Call the indexed version method
            return self._whereIndexed(condition, start, stop, step)
        # Fall back to in-kernel selection method
        return self._whereInRange(condition, start, stop, step)

    def _whereInRange(self, condition, start=None, stop=None, step=None):
        """
        Iterate over values fulfilling a `condition` avoiding indexes.

        This method is completely equivalent to `where()`, but it avoids
        the usage of the indexing capabilites on the column affected by
        the `condition` (whether indexed or not).

        This method is mainly for avoiding some pathological corner
        cases where automatically using indexation yields a poorer
        performance.  In that case, please contact the developers and
        explain your case.
        """

        if not isinstance(condition, Column):
            raise TypeError("""\
Wrong 'condition' parameter type. Only Column instances are suported.""")

        self.whereColname = condition.pathname   # Flag for Row.__iter__
        (start, stop, step) = processRangeRead(self.nrows, start, stop, step)
        if start < stop:
            # call row with coords=None and ncoords=-1 (in-kernel selection)
            #return self.row(start, stop, step, coords=None, ncoords=-1)
            row = TableExtension.Row(self)
            return row(start, stop, step, coords=None, ncoords=-1)
        # Fall-back action is to return an empty RecArray
        return iter([])

    def _whereIndexed(self, condition, start=None, stop=None, step=None):
        """
        Iterate over values fulfilling a `condition` using indexes.

        This method is completely equivalent to `where()`, but it forces
        the usage of the indexing capabilities on the column affected by
        the `condition`.  If the column is not indexed, a ``ValueError``
        is raised.
        """

        if not isinstance(condition, Column):
            raise TypeError("""\
Wrong 'condition' parameter type. Only Column instances are suported.""")
        if condition.index is None:
            raise ValueError("""\
This method is intended only for indexed columns.""")
        if condition.dirty:
            raise ValueError("""\
This method is intended only for indexed columns, but this column has a dirty index. Try re-indexing it in order to put the index in a sane state.""")
        if condition.index.nelements == 0:
            raise ValueError("""\
This method is intended only for indexed columns, but this column has not a minimum entries (%s) to be indexed.""" % condition.index.nelemslice)

        self.whereColname = condition.pathname   # Flag for Row.__iter__
        # Get the coordinates to lookup
        ncoords = condition.index.getLookupRange(condition)
        if ncoords > 0:
            # Call the indexed version of Row iterator (coords=None,ncoords>=0)
            (start, stop, step) = processRangeRead(self.nrows, start, stop,
                                                   step)
            #return self.row(start, stop, step, coords=None, ncoords=ncoords)
            row = TableExtension.Row(self)
            return row(start, stop, step, coords=None, ncoords=ncoords)
        else:
            # Fall-back action is to return an empty iterator
            self.ops = []
            self.opsValues = []
            self.opsColnames = []
            self.whereColname = None
            return iter([])

    def readIndexed(self, condition):
        """Returns a NestedRecArray fulfilling the 'condition' param.

        condition can be used to specify selections along a column in the
        form:

        condition=(0<table.cols.col1<0.3)

        This method is only intended to be used for indexed columns.
        """

        if not isinstance(condition, Column):
            raise TypeError(
                "``condition`` argument is not an instance of ``Column``")
        if condition.index is None:
            raise ValueError(
                "the column referenced by ``condition`` is not indexed")
        if condition.dirty:
            raise ValueError("""\
the column referenced by ``condition`` has a dirty index; \
please reindex the table to put the index in a sane state""")

        self.whereColname = condition.pathname   # Flag for Row.__iter__
        # Get the coordinates to lookup
        nrecords = condition.index.getLookupRange(condition)
        recarr = nestedrecords.array(None,
                                     formats=self.description._v_nestedFormats,
                                     shape=(nrecords,),
                                     names = self.colnames)
        if nrecords > 0:
            # Read the contents of a selection in a recarray
            condition.index.indices._initIndexSlice(nrecords)
            coords = condition.index.getCoords(0, nrecords)
            recout = self._read_elements_ra(recarr, coords)
            condition.index.indices._destroyIndexSlice()
        # Delete indexation caches
        self.ops = []
        self.opsValues = []
        self.opsColnames = []
        self.whereColname = None
        return recarr

    def whereAppend(self, dstTable, condition, start=None, stop=None, step=None):
        """
        Append rows fulfulling the `condition` to the `dstTable` table.

        `dstTable` must be capable of taking the rows resulting from the
        query, i.e. it must have columns with the expected names and
        compatible types.  The meaning of the other arguments is the
        same as in the `where()` method.

        The number of rows appended to `dstTable` is returned as a
        result.
        """

        # Check that file is not in read-only mode
        if self._v_file.mode == 'r':
            raise IOError("""\
Attempt to write over a file opened in read-only mode.""")

        colNames = self.colnames
        dstRow = dstTable.row
        nrows = 0
        try:
            for srcRow in self.where(condition, start, stop, step):
                for colName in colNames:
                    dstRow[colName] = srcRow[colName]
                dstRow.append()
                nrows += 1
            dstTable.flush()       # flush and close (at HDF5 level) dstTable
        except (KeyError, TypeError):
            srcRow._close_read()  # Make sure source is closed
            (type, value, traceback) = sys.exc_info()
            # re-raise the error
            raise type, value
        return nrows


    def getWhereList(self, condition, flavor="List"):
        """Get the row coordinates that fulfill the 'condition' param

        'condition' can be used to specify selections along a column
        in the form:

        condition=(0<table.cols.col1<0.3)

        'flavor' is the desired type of the returned list. It can take
        the 'List', 'Tuple' or 'NumArray' values.

        """

        if not isinstance(condition, Column):
            raise TypeError("""\
Wrong 'condition' parameter type. Only Column instances are suported.""")

        supportedFlavors = ['NumArray', 'List', 'Tuple']
        if flavor not in supportedFlavors:
            raise ValueError("""\
Specified 'flavor' value is not one of %s.""" % (supportedFlavors,))

        # Take advantage of indexation, if present
        if condition.index is not None:
            # get the number of coords and set-up internal variables
            ncoords = condition.index.getLookupRange(condition)
            # create buffers for indices
            condition.index.indices._initIndexSlice(ncoords)
            # get the coordinates that passes the selection cuts
            coords = condition.index.getCoords(0, ncoords)
            # Remove buffers for indices
            condition.index.indices._destroyIndexSlice()
            # get the remaining rows from the table
            start = condition.index.nelements
            remainCoords = [p.nrow() for p in \
                            self._whereInRange(condition, start, self.nrows, 1)]
            nremain = len(remainCoords)
            # append the new values to the existing ones
            coords.resize(ncoords+nremain)
            coords[ncoords:] = remainCoords
        else:
            coords = [p.nrow() for p in self.where(condition)]
            coords = numarray.array(coords, type=numarray.Int64)
        # re-initialize internal selection values
        self.ops = []
        self.opsValues = []
        self.opsColnames = []
        self.whereColname = None
        # do some conversion (if needed)
        if flavor == "List":
            coords = coords.tolist()
        elif flavor == "Tuple":
            coords = tuple(coords.tolist())
        return coords

    def itersequence(self, sequence, sort=True):
        """Iterate over a list of row coordinates.

        sort means that sequence will be sorted so that I/O would
        perform better. If your sequence is already sorted or you
        don't want to sort it, put this parameter to 0. The default is
        to sort the sequence.
        """

        if not hasattr(sequence, '__getitem__'):
            raise TypeError("""\
Wrong 'sequence' parameter type. Only sequences are suported.""")

        coords = numarray.array(sequence, type=numarray.Int64)
        # That would allow the retrieving on a sequential order
        # so, given better speed in the general situation
        if sort:
            coords.sort()
        #return self.row(coords=coords, ncoords=-1)
        row = TableExtension.Row(self)
        return row(coords=coords, ncoords=-1)

    def iterrows(self, start=None, stop=None, step=None):
        """Iterate over all the rows or a range.

        Specifying a negative value of step is not supported yet.

        """
        (start, stop, step) = processRangeRead(self.nrows, start, stop, step)
        if start < stop:
            #return self.row(start, stop, step, coords=None, ncoords=-1)
            row = TableExtension.Row(self)
            return row(start, stop, step, coords=None, ncoords=-1)
        # Fall-back action is to return an empty iterator
        return iter([])

    def __iter__(self):
        """Iterate over all the rows."""

        return self.iterrows()

    def read(self, start=None, stop=None, step=None,
             field=None, flavor="numarray", coords = None):
        """Read a range of rows and return an in-memory object.

        If "start", "stop", or "step" parameters are supplied, a row
        range is selected. If "field" is specified, only this "field" is
        returned as a NumArray object. If "field" is not supplied all
        the fields are selected and a NestedRecArray is returned.  If
        both "field" and "flavor" are provided, an additional conversion
        to an object of this flavor is made. "flavor" must have any of
        the next values: "numarray", "Numeric", "Tuple" or "List".

        If coords is specified, only the indices in coords that are in
        the range of (start, stop) are returned. If coords is
        specified, step only can be assigned to be 1, otherwise an
        error is issued.

        """

        if field:
            if field.find('/') > -1:
                raise NotImplementedError, \
"You cannot use Table.read to select deepen columns. Use Table.cols accessor instead."
            self._checkColumn(field)
        if flavor is None:
            flavor = "numarray"
        if flavor == "Numeric" and field is None:
            raise ValueError, \
                  """Numeric does not support heterogeneous datasets yet. You cannot specify a Numeric flavor without specifying a field."""

        (start, stop, step) = processRangeRead(self.nrows, start, stop, step)

        if coords is not None:
            # Check step value.
            if len(coords) and step != 1:
                raise NotImplementedError("""\
``step`` must be 1 when the ``coords`` parameter is specified""")
            # Turn coords into an array of 64-bit indexes,
            # as expected by _read().
            if not (isinstance(coords, numarray.NumArray)
                    and coords.type() != numarray.Int64):
                coords = numarray.array(coords, type=numarray.Int64)

        if flavor == "numarray":
            return self._read(start, stop, step, field, coords)
        else:
            arr = self._read(start, stop, step, field, coords)
            # Convert to Numeric, tuple or list if needed
            if flavor == "Numeric":
                if Numeric_imported:
                    # This works for both numeric and chararrays
                    # arr=Numeric.array(arr, typecode=arr.typecode())
                    # The next is 10 times faster (for tolist(),
                    # we should check for tostring()!)
                    if arr.__class__.__name__ == "CharArray":
                        arrstr = arr.tostring()
                        shape = list(arr.shape)
                        shape.append(arr.itemsize())
                        arr=Numeric.reshape(Numeric.array(arrstr), shape)
                    else:
                        if str(arr.type()) == "Bool":
                            # Typecode boolean does not exist on Numeric
                            typecode = "1"
                        else:
                            typecode = arr.typecode()
                        if arr.shape <> ():
                            shape = arr.shape
                            arr=Numeric.fromstring(arr._data,
                                                   typecode=arr.typecode())
                            arr.shape = shape
                        else:
                            # This works for rank-0 arrays
                            # (but is slower for big arrays)
                            arr=Numeric.array(arr, typecode=arr.typecode())
                else:
                    # Warn the user
                    warnings.warn( \
"""You are asking for a Numeric object, but Numeric is not installed locally.
  Returning a numarray object instead!.""")
            elif flavor == "Tuple":
                # Fixes bug #972534
                arr = tuple(self.tolist(arr))
            elif flavor == "List":
                # Fixes bug #972534
                arr = self.tolist(arr)
            else:
                raise ValueError, \
"""You are asking for an unsupported flavor (%s). Supported values are:
"Numeric", "Tuple" and "List".""" % (flavor)

        return arr


    def readCoordinates(self, coords, field=None, flavor='numarray'):
        """
        Read a set of rows given their indexes into an in-memory object.

        This method works much like the `read()` method, but it uses a
        sequence (`coords`) of row indexes to select the wanted columns,
        instead of a column range.

        It returns the selected rows in a ``NestedRecArray`` object.  If
        both `field` and `flavor` are provided, an additional conversion
        to an object of this flavor is made, just as in `read()`.  """

        # Convert coords into a numarray of Int64 indices
        try:
            coords = numarray.array(coords, type=numarray.Int64)
        except:  #XXX
            (typerr, value, traceback) = sys.exc_info()
            raise ValueError, \
"coords parameter cannot be converted into a Int64 numarray object. The error was: <%s>" % (str(self), value)

        return self.read(field=field, flavor=flavor, coords=coords)


    def tolist(self, arr):
        """Converts a NestedRecArray or NestedRecord to a list of rows"""
        outlist = []
        if isinstance(arr, records.Record):
            for i in xrange(arr.array._nfields):
                outlist.append(arr.array.field(i)[arr.row])
            outlist = tuple(outlist)  # return a tuple for records
        elif isinstance(arr, records.RecArray):
            for j in xrange(arr.nelements()):
                tmplist = []
                for i in xrange(arr._nfields):
                    tmplist.append(arr.field(i)[j])
                outlist.append(tuple(tmplist))
        # Fixes bug #991715
        else:
            # Other objects are passed "as is"
            outlist = list(arr)
        return outlist

    def _read(self, start, stop, step, field=None, coords=None):
        """Read a range of rows and return an in-memory object.
        """

        select_field = None
        if field:
            if field not in self.coltypes:
                if field in self.description._v_names:
                    # Remember to select this field
                    select_field = field
                    field = None
                else:
                    raise KeyError, "Field %s not found in table %s" % \
                          (field, self)
            else:
                # The column hangs directly from the top
                typeField = self.coltypes[field]

        # Return a rank-0 array if start > stop
        if start >= stop:
            if field == None:
                nra = nestedrecords.array(
                    None,
                    formats=self.description._v_nestedFormats,
                    shape=(0,),
                    names = self.colnames)
                return nra
            elif isinstance(typeField, records.Char):
                return numarray.strings.array(shape=(0,), itemsize = 0)
            else:
                return numarray.array(shape=(0,), type=typeField)

        if coords is None:
            # (stop-start)//step  is not enough
            nrows = ((stop - start - 1) // step) + 1
        else:
            assert isinstance(coords, numarray.NumArray)
            assert coords.type() == numarray.Int64
            # I should test for stop and start values as well
            nrows = len(coords)

        # Compute the shape of the resulting column object
        if field and coords is None:  # coords handling expects a RecArray
            shape = self.colshapes[field]
            itemsize = self.colitemsizes[field]
            if type(shape) in (int,long):
                if shape == 1:
                    shape = (nrows,)
                else:
                    shape = (nrows, shape)
            else:
                shape2 = [nrows]
                shape2.extend(shape)
                shape = tuple(shape2)

            # Create the resulting recarray
            if isinstance(typeField, records.Char):
                # String-column case
                result = numarray.strings.array(shape=shape, itemsize=itemsize)
            else:
                # Non-string column case
                result = numarray.array(shape=shape, type=typeField)
        else:
            # Recarray case
            colnames = self.description._v_nestedNames
            formats = self.description._v_nestedFormats
            result = nestedrecords.array(None, formats=formats,
                                         shape=(nrows,),
                                         names = colnames)

        # Handle coordinates separately.
        if coords is not None:
            if len(coords) > 0:
                self._open_read(result)
                self._read_elements(result, coords)
                self._close_read()
            if field:
                # result is always a RecArray
                return result.field(field)
            return result

        # Call the routine to fill-up the resulting array
        if step == 1 and not field:
            # This optimization works three times faster than
            # the row._fillCol method (up to 170 MB/s on a pentium IV @ 2GHz)
            self._open_read(result)
            self._read_records(result, start, stop-start)
            self._close_read()  # Close the table
        # Warning!: _read_field_name should not be used until
        # H5TBread_fields_name in TableExtension will be finished
        # F. Altet 2005/05/26
        elif field and step > 15 and 0:
            # For step>15, this seems to work always faster than row._fillCol.
            self._read_field_name(result, start, stop, step, field)
        else:
            self.row._fillCol(result, start, stop, step, field)

        if select_field:
            return result.field(select_field)
        else:
            return result


    def getEnum(self, colname):
        """
        Get the enumerated type associated with the named column.

        If the column named `colname` (a string) exists and is of an
        enumerated type, the corresponding `Enum` instance is returned.
        If it is not of an enumerated type, a ``TypeError`` is raised.
        If the column does not exist, a ``KeyError`` is raised.
        """

        self._checkColumn(colname)

        try:
            return self._colenums[colname]
        except KeyError:
            raise TypeError(
                "column ``%s`` of table ``%s`` is not of an enumerated type"
                % (colname, self._v_pathname))


    def col(self, name):
        """
        Get a column from the table.

        If a column called `name` exists in the table, it is read and
        returned as a ``numarray.NumArray`` object, or as a
        ``numarray.strings.CharArray`` object (whatever is more
        appropriate).  If it does not exist, a ``KeyError`` is raised.

        Example of use::

            narray = table.col('var2')

        That statement is equivalent to::

            narray = table.read(field='var2')

        Here you can see how this method can be used as a shorthand for
        the `read()` method.
        """
        return self.read(field=name)


    def __getitem__(self, key):
        """
        Get a row or a range of rows from the table.

        If the `key` argument is an integer, the corresponding table row
        is returned as a ``tables.nestedrecords.NestedRecord`` object.
        If `key` is a slice, the range of rows determined by it is
        returned as a ``tables.nestedrecords.NestedRecArray`` object.

        Using a string as `key` to get a column is supported but
        deprecated.  Please use the `col()` method.

        Example of use::

            record = table[4]
            recarray = table[4:1000:2]

        Those statements are equivalent to::

            record = table.read(start=4)[0]
            recarray = table.read(start=4, stop=1000, step=2)

        Here you can see how indexing and slicing can be used as
        shorthands for the `read()` method.
        """

        if type(key) in (int,long):
            # Index out of range protection
            if key >= self.nrows:
                raise IndexError, "Index out of range"
            if key < 0:
                # To support negative values
                key += self.nrows
            (start, stop, step) = processRange(self.nrows, key, key+1, 1)
            # For the scalar case, convert the Record and return it as a tuple
            # Fixes bug #972534
            # Reverted to return a numarray.records.Record in order
            # to support better the nested datatypes
            # return self.tolist(self._read(start, stop, step, None, None)[0])
            return self._read(start, stop, step, None, None)[0]
        elif isinstance(key, slice):
            (start, stop, step) = processRange(self.nrows,
                                               key.start, key.stop, key.step)
            return self._read(start, stop, step, None, None)
        elif isinstance(key, str):
            warnings.warn(DeprecationWarning("""\
``table['colname']`` is deprecated; please use ``table.col('colname')``"""),
                          stacklevel=2)
            return self.col(key)
        else:
            raise TypeError("invalid index or slice: %r" % (key,))


    def __setitem__(self, key, value):
        """Sets a table row or table slice.

        It takes different actions depending on the type of the 'key'
        parameter:

        If 'key' is an integer, the corresponding table row is set to
        'value' (List or Tuple). If 'key' is a slice, the row slice
        determined by key is set to value (a NestedRecArray or list of
        rows).

        """

        if self._v_file.mode == 'r':
            raise IOError("""\
Attempt to write over a file opened in read-only mode.""")

        if type(key) in (int,long):
            # Index out of range protection
            if key >= self.nrows:
                raise IndexError, "Index out of range"
            if key < 0:
                # To support negative values
                key += self.nrows
            return self.modifyRows(key, key+1, 1, [value])
        elif isinstance(key, slice):
            (start, stop, step) = processRange(self.nrows,
                                               key.start, key.stop, key.step)
            return self.modifyRows(start, stop, step, value)
        else:
            raise ValueError, "Non-valid index or slice: %s" % key

    def append(self, rows):
        """Append a series of rows to the end of the table

        rows can be either a recarray or a structure that is able to
        be converted to a recarray compliant with the table format.

        It raises an 'ValueError' in case the rows parameter could not
        be converted to an object compliant with table description.

        """

        if self._v_file.mode == 'r':
            raise IOError("""\
Attempt to write over a file opened in read-only mode.""")

        # Try to convert the object into a recarray
        try:
            # This always makes a copy of the original,
            # so the resulting object is safe to in-place conversion.
            recarray = nestedrecords.array(
                rows,
                formats=self.description._v_nestedFormats,
                names=self.colnames)
        except:  #XXX
            (typerr, value, traceback) = sys.exc_info()
            raise ValueError, \
"rows parameter cannot be converted into a recarray object compliant with table '%s'. The error was: <%s>" % (str(self), value)
        lenrows = recarray.shape[0]
        self._open_append(recarray)
        self._append_records(recarray, lenrows)
        self._close_append()
        # Update the number of saved rows
        self.nrows += lenrows
        self.shape = (self.nrows,)
        # Save indexedrows
        if self.indexed:
            # Update the number of unsaved indexed rows
            self._unsaved_indexedrows += lenrows
            if self.indexprops.auto:
                self.flushRowsToIndex(lastrow=0)

    def _saveBufferedRows(self):
        """Save buffered table rows"""
        # Save the records on disk
        # Data is copied below to the buffer,
        # so it's safe to do an in-place conversion.
        # Open and close the table before and after appending to let
        # the dataset_id and other variables in a safe state for other
        # reading calls. Fixes #1186892
        # F. Altet 2005-04-05
        self._open_append(self._v_wbuffer)
        self._append_records(self._v_wbuffer, self.row._getUnsavedNRows())
        self._close_append()
        # Update the number of saved rows in this buffer
        self.nrows += self.row._getUnsavedNRows()
        # Reset the buffer unsaved counter and the buffer read row counter
        self.row._setUnsavedNRows(0)
        # Set the shape attribute (the self.nrows may be less than the maximum)
        self.shape = (self.nrows,)
        if self.indexed and self.indexprops.auto:
            # Flush the unindexed rows (this needs to read table)
            self.flushRowsToIndex(lastrow=0)
        # Get a fresh copy of the default values
        # This copy seems to make the writing with compression a 5%
        # faster than if the copy is not made. Why??
        if hasattr(self, "_v_wbuffercpy"):
            self._v_wbuffer[:] = self._v_wbuffercpy[:]
        return

    def modifyRows(self, start=None, stop=None, step=1, rows=None):
        """Modify a series of rows in the slice [start:stop:step]

        `rows` can be either a recarray or a structure that is able to
        be converted to a recarray compliant with the table format.

        Returns the number of modified rows.

        It raises an 'ValueError' in case the rows parameter could not
        be converted to an object compliant with table description.

        It raises an 'IndexError' in case the modification will exceed
        the length of the table.

        """

        if rows is None:      # Nothing to be done
            return
        if start is None:
            start = 0

        if start < 0:
            raise ValueError("'start' must have a positive value.")
        if step < 1:
            raise ValueError("'step' must have a value greater or equal than 1.")
        if stop is None:
            # compute the stop value. start + len(rows)*step does not work
            stop = start + (len(rows)-1)*step + 1

        (start, stop, step) = processRange(self.nrows, start, stop, step)
        if stop > self.nrows:
            raise IndexError, \
"This modification will exceed the length of the table. Giving up."
        # Compute the number of rows to read. (stop-start)/step does not work
        nrows = ((stop - start - 1) / step) + 1
        if len(rows) < nrows:
            raise ValueError, \
           "The value has not enough elements to fill-in the specified range"
        # Try to convert the object into a recarray
        try:
            # This always makes a copy of the original,
            # so the resulting object is safe to in-place conversion.
            recarray = nestedrecords.array(
                rows,
                formats=self.description._v_nestedFormats,
                names=self.colnames)
            # records.array does not seem to change the names
            # attibute in case rows is a recarray.
            # Change it manually and report this
            # 2004-08-08
            recarray._names = self.colnames
        except:  #XXX
            (typerr, value, traceback) = sys.exc_info()
            raise ValueError, \
"rows parameter cannot be converted into a recarray object compliant with table format '%s'. The error was: <%s>" % (str(self.description._v_nestedFormats), value)
        lenrows = len(recarray)
        if start + lenrows > self.nrows:
            raise IndexError, \
"This modification will exceed the length of the table. Giving up."
        self._modify_records(start, stop, step, recarray)
        # Redo the index if needed
        if self.indexed:
            # Mark all the indexes as dirty
            for (colname, colindexed) in self.colindexed.iteritems():
                if colindexed:
                    indexcol = self.cols._f_col(colname)
                    indexcol.dirty = 1
            if self.indexprops.reindex:
                self._indexedrows = self.reIndex()
                self._unsaved_indexedrows = self.nrows - self._indexedrows
        return lenrows

    def modifyColumn(self, start=None, stop=None, step=1,
                     column=None, colname=None):
        """Modify one single column in the row slice [start:stop:step]

        column can be either a 'NestedRecArray', 'RecArray', 'NumArray',
        list or tuple that is able to be converted into a
        'NestedRecArray' compliant with the specified colname column of
        the table.

        colname specifies the column name of the table to be modified.

        Returns the number of modified rows.

        It raises an 'ValueError' in case the columns parameter could
        not be converted into an object compliant with the column
        description.

        It raises an 'IndexError' in case the modification will exceed
        the length of the table.

        """

        if not isinstance(colname, str):
            raise TypeError("""\
The 'colname' parameter must be a string.""")

        if column is None:      # Nothing to be done
            return 0
        if start is None:
            start = 0

        if start < 0:
            raise ValueError("'start' must have a positive value.")
        if step < 1:
            raise ValueError("'step' must have a value greater or equal than 1.")
        # Get the column format to be modified:
        objcol = self._checkColumn(colname)
        if isinstance(objcol, Description):
            selcolname = objcol._v_nestedNames
        else:
            selcolname = objcol._v_parent._v_nestedNames[objcol._v_pos]
        format = objcol._v_parent._v_nestedFormats[objcol._v_pos]
        # Try to convert the column object into a recarray
        try:
            if isinstance(column, records.RecArray):
                recarray = nestedrecords.array(column, formats=format,
                                               names=selcolname)
            else:
                recarray = nestedrecords.fromarrays([column], formats=format,
                                                    names=selcolname)
            recarray._fields = recarray._get_fields()  # Refresh the cache
        except:  #XXX
            (typerr, value, traceback) = sys.exc_info()
            raise ValueError, \
"column parameter cannot be converted into a recarray object compliant with specified column '%s'. The error was: <%s>" % (str(column), value)

        if stop is None:
            # compute the stop value. start + len(rows)*step does not work
            stop = start + (len(recarray)-1)*step + 1
        (start, stop, step) = processRange(self.nrows, start, stop, step)
        if stop > self.nrows:
            raise IndexError, \
"This modification will exceed the length of the table. Giving up."
        # Compute the number of rows to read. (stop-start)/step does not work
        nrows = ((stop - start - 1) / step) + 1
        if len(recarray) < nrows:
            raise ValueError, \
           "The value has not enough elements to fill-in the specified range"
        # Now, read the original values:
        mod_recarr = self.read(start, stop, step)
        mod_recarr._fields = mod_recarr._get_fields()  # Refresh the cache
        # Modify the appropriate column in the original recarray
        if isinstance(objcol, Description):
            mod_recarr.field(colname)[:] = recarray
        else:
            # recarray should have one one field
            mod_recarr.field(colname)[:] = recarray.field(0)
        # save this modified rows in table
        self._modify_records(start, stop, step, mod_recarr)
        # Redo the index if needed
        if self.indexed:
            # First, mark the modified indexes as dirty
            for (colname2, colindexed) in self.colindexed.iteritems():
                if colindexed and colname2 == colname:
                    col = getattr(self.cols, colname)
                    col.dirty = 1
            # Then, reindex if needed
            if self.indexprops.reindex:
                self._indexedrows = self.reIndex()
                self._unsaved_indexedrows = self.nrows - self._indexedrows
        return nrows

    def modifyColumns(self, start=None, stop=None, step=1,
                      columns=None, names=None):
        """Modify a series of columns in the row slice [start:stop:step]

        columns can be either a 'NestedRecArray', 'RecArray' or a list
        of arrays, lists or tuple (the columns) that is able to be
        converted into a 'NestedRecArray' compliant with the specified
        column names subset of the table format.

        names specifies the column names of the table to be modified.

        Returns the number of modified rows.

        It raises an 'ValueError' in case the columns parameter could
        not be converted into an object compliant with table
        description.

        It raises an 'IndexError' in case the modification will exceed
        the length of the table.

        """

        if type(names) not in (list, tuple):
            raise TypeError("""\
The 'names' parameter must be a list of strings.""")

        if columns is None:      # Nothing to be done
            return 0
        if start is None:
            start = 0
        if start < 0:
            raise ValueError("'start' must have a positive value.")
        if step < 1:
            raise ValueError("'step' must have a value greater or equal than 1.")        # Get the column formats to be modified:
        formats = []
        selcolnames = []
        for colname in names:
            objcol = self._checkColumn(colname)
            selcolnames.append(objcol._v_parent._v_nestedNames[objcol._v_pos])
            formats.append(objcol._v_parent._v_nestedFormats[objcol._v_pos])
        # Try to convert the columns object into a recarray
        try:
            if isinstance(columns, records.RecArray):
                recarray = nestedrecords.array(columns, formats=formats,
                                               names=selcolnames)
            else:
                recarray = nestedrecords.fromarrays(columns, formats=formats,
                                                    names=selcolnames)
            recarray._fields = recarray._get_fields()  # Refresh the cache
        except:  #XXX
            (typerr, value, traceback) = sys.exc_info()
            raise ValueError, \
"columns parameter cannot be converted into a recarray object compliant with table '%s'. The error was: <%s>" % (str(self), value)

        if stop is None:
            # compute the stop value. start + len(rows)*step does not work
            stop = start + (len(recarray)-1)*step + 1
        (start, stop, step) = processRange(self.nrows, start, stop, step)
        if stop > self.nrows:
            raise IndexError, \
"This modification will exceed the length of the table. Giving up."
        # Compute the number of rows to read. (stop-start)/step does not work
        nrows = ((stop - start - 1) / step) + 1
        if len(recarray) < nrows:
            raise ValueError, \
           "The value has not enough elements to fill-in the specified range"
        # Now, read the original values:
        mod_recarr = self.read(start, stop, step)
        mod_recarr._fields = mod_recarr._get_fields()  # Refresh the cache
        # Modify the appropriate columns in the original recarray
        for name in recarray._names:
            mod_recarr._fields[name][:] = recarray._fields[name]
        # save this modified rows in table
        self._modify_records(start, stop, step, mod_recarr)
        # Redo the index if needed
        if self.indexed:
            # First, mark the modified indexes as dirty
            for (colname, colindexed) in self.colindexed.iteritems():
                if colindexed and colname in names:
                    col = getattr(self.cols, colname)
                    col.dirty = 1
            # Then, reindex if needed
            if self.indexprops.reindex:
                self._indexedrows = self.reIndex()
                self._unsaved_indexedrows = self.nrows - self._indexedrows
        return nrows

    def flushRowsToIndex(self, lastrow=1):
        "Add remaining rows in buffers to non-dirty indexes"
        rowsadded = 0
        if self.indexed:
            # Update the number of unsaved indexed rows
            start = self._indexedrows
            nrows = self._unsaved_indexedrows
            for (colname, colindexed) in self.colindexed.iteritems():
                if colindexed:
                    col = self.cols._f_col(colname)
                    if nrows > 0 and not col.dirty:
                        rowsadded = col._addRowsToIndex(start, nrows, lastrow)
            self._unsaved_indexedrows -= rowsadded
            self._indexedrows += rowsadded
        return rowsadded

    def removeRows(self, start, stop=None):
        """Remove a range of rows.

        If only "start" is supplied, this row is to be deleted.
        If "start" and "stop" parameters are supplied, a row
        range is selected to be removed.

        """

        (start, stop, step) = processRangeRead(self.nrows, start, stop, 1)
        nrows = stop - start
        if nrows >= self.nrows:
            raise NotImplementedError, \
"""You are trying to delete all the rows in table "%s". This is not supported right now due to limitations on the underlying HDF5 library. Sorry!""" % self._v_pathname
        nrows = self._remove_row(start, nrows)
        self.nrows -= nrows    # discount the removed rows from the total
        self.shape = (self.nrows,)    # update to the new shape
        # removeRows is a invalidating index operation
        if self.indexed:
            if self.indexprops.reindex:
                self._indexedrows = self.reIndex()
                self._unsaved_indexedrows = self.nrows - self._indexedrows
            else:
                # Mark all the indexes as dirty
                for (colname, colindexed) in self.colindexed.iteritems():
                    if colindexed:
                        col = self.cols._f_col(colname)
                        col.dirty = 1

        return nrows

    def _g_move(self, newParent=None, newName=None):
        """
        Move this node in the hierarchy.

        This overloads the Node._g_move() method.
        """

        oldparent = self._v_parent
        oldname = self._v_name

        # First, move the table to the new location.
        super(Table, self)._g_move(newParent, newName)
        # update the HDF5 memory type id for the new node
        self._g_updateTypeId()

        # Then move the associated indexes (if any)
        if self.indexed:
            itgroup = self._v_file.getNode(_getIndexTableName(oldparent,
                                                              oldname))
            oldiname = itgroup._v_name
            newigroup = self._v_parent
            newiname = _getEncodedTableName(self._v_name)
            itgroup._g_move(newigroup, newiname)

    def _g_remove(self, recursive=False):
        parent = self._v_parent
        # Remove the associated indexes (if they exists!)
        if self.indexed:
            itgroup = self._v_file.getNode(_getIndexTableName(parent,
                                                              self.name))
            itgroup._f_remove(recursive=True)
        parent._g_deleteLeaf(self._v_name)
        self.close(flush=0)

    def removeIndex(self, index):
        "Remove the index associated with the specified column"

        # Check that file is not in read-only mode
        if self._v_file.mode == 'r':
            raise IOError("""\
Attempt to write over a file opened in read-only mode.""")

        if not isinstance(index, Index):
            raise TypeError("""\
Wrong 'index' parameter type. Only Index instances are accepted.""")
        index.column.removeIndex()

    def reIndex(self):
        """Recompute the existing indexes in table"""
        for (colname, colindexed) in self.colindexed.iteritems():
            if colindexed:
                indexcol = self.cols._f_col(colname)
                indexedrows = indexcol.reIndex()
        return indexedrows

    def reIndexDirty(self):
        """Recompute the existing indexes in table if they are dirty"""
        for (colname, colindexed) in self.colindexed.iteritems():
            if colindexed:
                indexcol = self.cols._f_col(colname)
                indexedrows = indexcol.reIndexDirty()
        return indexedrows

    def _g_copyRows(self, object, start, stop, step):
        "Copy rows from self to object"
        (start, stop, step) = processRangeRead(self.nrows, start, stop, step)
        nrowsinbuf = self._v_maxTuples
        recarray = self._newBuffer(init=0)
        object._open_append(recarray)
        nrowsdest = object.nrows
        for start2 in xrange(start, stop, step*nrowsinbuf):
            # Save the records on disk
            stop2 = start2+step*nrowsinbuf
            if stop2 > stop:
                stop2 = stop
            #object.append(self[start2:stop2:step])
            # Optimized version (it saves some conversions)
            nrows = ((stop2 - start2 - 1) // step) + 1
            self.row._fillCol(recarray, start2, stop2, step, None)
            # The recarray is created anew,
            # so the operation is safe to in-place conversion.
            object._append_records(recarray, nrows)
            nrowsdest += nrows
        object._close_append()
        # Update the number of saved rows in this buffer
        object.nrows = nrowsdest
        # Set the shape attribute (the self.nrows may be less than the maximum)
        object.shape = (nrowsdest,)
        return

    # This is an optimized version of copy
    def _g_copyWithStats(self, group, name, start, stop, step, title, filters):
        "Private part of Leaf.copy() for each kind of leaf"
        # Build the new Table object
        description = self.description
        # Checking validity names for fields in destination is not necessary
        description.__dict__['__check_validity__'] = 0
        # Ensure backward compatibility with old index format
        if hasattr(self.attrs, "VERSION") and self.attrs.VERSION < "2.3":
            # Set the appropriate indexes properties for these old indexes
            autoindex = getattr(self.attrs, 'AUTOMATIC_INDEX', None)
            reindex = getattr(self.attrs, 'REINDEX', None)
            self.indexprops = IndexProps(auto=autoindex, reindex=reindex)
            for colname in self.colnames:
                indexname = "_i_%s_%s" % (self.name, colname)
                indexpathname = joinPath(self._v_parent._v_pathname, indexname)
                try:
                    index = self._v_file.getNode(indexpathname)
                    # Get the filters values
                    self.indexprops.filters = self._g_getFilters()
                    getattr(description, colname).indexed = 1
                except NodeError:
                    getattr(description, colname).indexed = 0

        # Add a possible IndexProps property to that
        if hasattr(self, "indexprops"):
            description.__dict__["_v_indexprops"] = self.indexprops

        object = self._v_file.createTable(
            group, name, description, title=title, filters=filters,
            expectedrows=self.nrows, _log = False)
        # Now, fill the new table with values from the old one
        self._g_copyRows(object, start, stop, step)
        nbytes=self.nrows*self.rowsize
        if object.indexed:
            warnings.warn( \
"Regenerating indexes for destination table %s:%s. This may take while, be patient please." % (object._v_file.filename, object._v_pathname))
            object._indexedrows = 0
            object._unsaved_indexedrows = object.nrows
            if object.indexprops.auto:
                object.flushRowsToIndex(lastrow=1)
        return (object, nbytes)

    def flush(self):
        """Flush the table buffers."""
        # Flush any unsaved row
        if hasattr(self, 'row'):
            if self.row._getUnsavedNRows() > 0:
                self._saveBufferedRows()
            ## What is the point in opening a closed node to flush it?
            ## ivb(2005-07-13)
            ##
            ## # Flush the data to disk
            ## # Call the H5Fflush with this Leaf just in case of buffered I/O
            ## hdf5Extension.flush_leaf(self._v_parent, self._v_hdf5name)
        if hasattr(self, "indexed") and self.indexed and self.indexprops.auto:
            # Flush any unindexed row
            rowsadded = self.flushRowsToIndex(lastrow=1)
        # Close a possible opened table for append
        self._close_append()
        # Clean the Row instance
        # In some situations, this maybe undefined (When?)
        if hasattr(self, "row"):
            # If not call row._cleanup()
            # the memory consumption seems to increase instead of decrease (!)
            # However, the reading/write speed seems to improve a bit (!!)
            # I choose to favor speed vs memory consumption
            self.row._cleanup()
            pass

    def _f_close(self, flush = True):
        # Close the Table
        super(Table, self)._f_close(flush)
        if hasattr(self, "cols"):
            self.cols._f_close()
            self.cols = None
        # We must delete the row object, as this make a back reference to Table
        # In some situations, this maybe undefined
        if self.__dict__.has_key("row"):
            # To deal with some situations where exceptions has been raised
            # in the middle of read iterators
            if self.row._in_riterator():
                self._close_read()
            del self.__dict__['row']
        # Free the description class!
        #self.description._f_close()
        del self.description
        if hasattr(self, "indexprops"):
            del self.indexprops

        # After the objects are disconnected, destroy the
        # object dictionary using the brute force
        # This should help to the garbage collector
        # This is not really necessary 2005-06-05
        #self.__dict__.clear()

    def __repr__(self):
        """This provides column metainfo in addition to standard __str__"""

        if hasattr(self, "indexed") and self.indexed == 1:
            return \
"""%s
  description := %r
  indexprops := %r
  byteorder := %s""" % \
        (str(self), self.description, self.indexprops, self.byteorder)
        else:
            return "%s\n  description := %r\n  byteorder := %s" % \
                   (str(self), self.description, self.byteorder)


class Cols(object):
    """This is a container for columns in a table

    It provides methods to get Column objects that gives access to the
    data in the column.

    Like with Group instances and AttributeSet instances, the natural
    naming is used, i.e. you can access the columns on a table like if
    they were normal Cols attributes.

    Instance variables:

        _v_colnames -- List with all column names hanging from cols
        _v_colpathnames -- List with all column names hanging from cols
        _v_table -- The parent table instance
        _v_desc -- The associated Desciption instance

    Methods:

        __getitem__(slice)
        __f_col(colname)
        __len__()

    """

    def __init__(self, table, desc):
        """Create the container to keep the column information.
        """

        self.__dict__["_v_table"] = table
        self.__dict__["_v_desc"] = desc
        self.__dict__["_v_colnames"] = desc._v_names
        self.__dict__["_v_colpathnames"] = table.description._v_pathnames
        # Put the column in the local dictionary
        for name in desc._v_names:
            if name in desc._v_types:
                self.__dict__[name] = Column(table, name, desc)
            else:
                self.__dict__[name] = Cols(table, desc._v_colObjects[name])

    def __len__(self):
        return len(self._v_colnames)

    def _f_col(self, colname):
        """Return the column named "colname"."""

        if not isinstance(colname, str):
            raise TypeError, \
"Parameter can only be an string. You passed object: %s" % colname
        if ((colname.find('/') > -1 and
             not colname in self._v_colpathnames) and
            not colname in self._v_colnames):
                raise KeyError(
"Cols accessor ``%s.cols%s`` does not have a column named ``%s``"
        % (self._v_table._v_pathname, self._v_desc._v_pathname, colname))

        # Get the Column or Description object
        inames = colname.split('/')
        cols = self
        for iname in inames:
            cols = cols.__dict__[iname]
        return cols

    def __getitem__(self, key):
        """
        Get a row or a range of rows from a (nested) column.

        If the `key` argument is an integer, the corresponding nested
        type row is returned as a ``tables.nestedrecords.NestedRecord``
        object.  If `key` is a slice, the range of rows determined by it
        is returned as a ``tables.nestedrecords.NestedRecArray`` object.

        Using a string as `key` to get a column is supported but
        deprecated.  Please use the `_f_col()` method.

        Example of use::

            record = table.cols[4]  # equivalent to table[4]
            recarray = table.cols.Info[4:1000:2]

        Those statements are equivalent to::

            nrecord = table.read(start=4)[0]
            nrecarray = table.read(start=4, stop=1000, step=2).field('Info')

        Here you can see how a mix of natural naming, indexing and
        slicing can be used as shorthands for the `Table.read()` method.

        """

        nrows = self._v_table.nrows
        if type(key) in (int,long):
            # Index out of range protection
            if key >= nrows:
                raise IndexError, "Index out of range"
            if key < 0:
                # To support negative values
                key += nrows
            (start, stop, step) = processRange(nrows, key, key+1, 1)
            colgroup = self._v_desc._v_pathname
            if colgroup == "":  # The root group
                return self._v_table._read(start, stop, step, None, None)[0]
            else:
                crecord = self._v_table._read(start, stop, step, None, None)[0]
                return crecord.field(colgroup)
        elif isinstance(key, slice):
            (start, stop, step) = processRange(nrows,
                                               key.start, key.stop, key.step)
            colgroup = self._v_desc._v_pathname
            if colgroup == "":  # The root group
                return self._v_table._read(start, stop, step, None, None)
            else:
                crecarray = self._v_table._read(start, stop, step, None, None)
                return crecarray.field(colgroup)
        elif isinstance(key, str):
            warnings.warn(DeprecationWarning("""\
``table.cols['colname']`` is deprecated; please use ``table.cols._f_col('colname')``"""),
                          stacklevel=2)
            return self._f_col(key)
        else:
            raise TypeError("invalid index or slice: %r" % (key,))

    def _f_close(self):
        # First, close the columns (ie possible indices open)
        for col in self._v_colnames:
            colobj = self._f_col(col)
            if isinstance(colobj, Column):
                colobj.close()
                # Delete the reference to column
                del self.__dict__[col]
            else:
                colobj._f_close()
        # delete back references:
        self._v_table == None
        self._v_colnames = None
        self._v_colpathnames = None
        self._v_desc = None
        # Delete all the columns references
        self.__dict__.clear()

    # Cols does not accept comparisons
    def __lt__(self, other):
        raise TypeError, "Cols object can't be used in comparisons"

    def __le__(self, other):
        raise TypeError, "Cols object can't be used in comparisons"

    def __gt__(self, other):
        raise TypeError, "Cols object can't be used in comparisons"

    def __ge__(self, other):
        raise TypeError, "Cols object can't be used in comparisons"

    def __eq__(self, other):
        raise TypeError, "Cols object can't be used in comparisons"

    def __ne__(self, other):
        raise TypeError, "Cols object can't be used in comparisons"

    def __str__(self):
        """The string representation for this object."""
        # The pathname
        tablepathname = self._v_table._v_pathname
        descpathname = self._v_desc._v_pathname
        # Get this class name
        classname = self.__class__.__name__
        # The number of columns
        ncols = len(self._v_colnames)
        return "%s.cols.%s (%s), %s columns" % \
               (tablepathname, descpathname, classname, ncols)

    def __repr__(self):
        """A detailed string representation for this object."""

        out = str(self) + "\n"
        for name in self._v_colnames:
            # Get this class name
            classname = getattr(self, name).__class__.__name__
            # The shape for this column
            shape = self._v_desc._v_shapes[name]
            # The type
            if name in self._v_desc._v_types:
                tcol = self._v_desc._v_types[name]
            else:
                tcol = "Description"
            if shape == 1:
                shape = (1,)
            out += "  %s (%s%s, %s)" % (name, classname, shape, tcol) + "\n"
        return out


class Column(object):
    """This is an accessor for the actual data in a table column

    Instance variables:

        table -- the parent table instance
        name -- the name of the associated column
        pathname -- the complete pathname of the column (the same as `name`
                    if column is non-nested)
        descr -- the parent description object
        type -- the type of the column
        shape -- the shape of the column
        index -- the Index object (None if doesn't exists)
        dirty -- whether the index is dirty or not (property)

    Methods:
        __getitem__(key)
        __setitem__(key, value)
        createIndex()
        reIndex()
        reIndexDirty()
        removeIndex()
    """

    def __init__(self, table, name, descr):
        """Create the container to keep the column information.

        Parameters:

        table -- The parent table instance
        name -- The name of the column that is associated with this object
        descr -- The parent description object

        """
        self.table = table
        self.name = name
        self.pathname = descr._v_colObjects[name]._v_pathname
        self.descr = descr
        self.type = descr._v_types[name]
        self.shape = descr._v_shapes[name]
        # Check whether an index exists or not
        indexname = _getIndexColName(table._v_parent, table._v_name,
                                     self.pathname)
        try:
            self.index = table._v_file.getNode(indexname)
            self.index.__dict__['column'] = self # points to this column
        except NodeError:
            self.index = None
            if (hasattr(table, "colindexed") and
                table.colindexed[self.pathname]):
                # The user wants to indexate this column,
                # but it doesn't exists yet. Create it without a warning.
                self.createIndex(warn=0)         # self.index is assigned here

    # Define dirty as a property
    def _get_dirty(self):
        if self.index and hasattr(self.index._v_attrs, "DIRTY"):
            return getattr(self.index._v_attrs, "DIRTY")
        else:
            return 0

    def _set_dirty(self, dirty):
        # Only set the index column as dirty if it exists
        if self.index:
            setattr(self.index._v_attrs,"DIRTY", dirty)

    # Define a property.  The 'delete this attribute'
    # method is defined as None, so the attribute can't be deleted.
    dirty = property(_get_dirty, _set_dirty, None, "Column dirtyness")

    def __len__(self):
        return self.table.nrows

    def __getitem__(self, key):
        """Returns a column element or slice

        It takes different actions depending on the type of the 'key'
        parameter:

        If 'key' is an integer, the corresponding element in the
        column is returned as a NumArray/CharArray, or a scalar
        object, depending on its shape. If 'key' is a slice, the row
        slice determined by this slice is returned as a NumArray or
        CharArray object (whatever is appropriate).

        """

        if type(key) in (int,long):
            # Index out of range protection
            if key >= self.table.nrows:
                raise IndexError, "Index out of range"
            if key < 0:
                # To support negative values
                key += self.table.nrows
            (start, stop, step) = processRange(self.table.nrows, key, key+1, 1)
            return self.table._read(start, stop, step, self.pathname, None)[0]
        elif isinstance(key, slice):
            (start, stop, step) = processRange(self.table.nrows, key.start,
                                               key.stop, key.step)
            return self.table._read(start, stop, step, self.pathname, None)
        else:
            raise TypeError, "'%s' key type is not valid in this context" % \
                  (key)

    def __setitem__(self, key, value):
        """Sets a column element or slice.

        It takes different actions depending on the type of the 'key'
        parameter:

        If 'key' is an integer, the corresponding element in the
        column is set to 'value' (scalar or NumArray/CharArray,
        depending on column's shape). If 'key' is a slice, the row
        slice determined by 'key' is set to 'value' (a
        NumArray/CharArray or list of elements).

        """

        if self.table._v_file.mode == 'r':
            raise IOError("""\
Attempt to write over a file opened in read-only mode.""")

        if type(key) in (int,long):
            # Index out of range protection
            if key >= self.table.nrows:
                raise IndexError, "Index out of range"
            if key < 0:
                # To support negative values
                key += self.table.nrows
            return self.table.modifyColumns(key, key+1, 1,
                                            [[value]], names=[self.pathname])
        elif isinstance(key, slice):
            (start, stop, step) = processRange(self.table.nrows,
                                               key.start, key.stop, key.step)
            return self.table.modifyColumns(start, stop, step,
                                            [value], names=[self.pathname])
        else:
            raise ValueError, "Non-valid index or slice: %s" % key

    def _addComparison(self, noper, other):
        self.table.ops.append(noper)
        self.table.opsValues.append(other)
        self.table.opsColnames.append(self.pathname)

    def __lt__(self, other):
        self._addComparison(1, other)
        return self

    def __le__(self, other):
        self.table.ops.append(2)
        self.table.opsValues.append(other)
        return self

    def __gt__(self, other):
        self.table.ops.append(3)
        self.table.opsValues.append(other)
        return self

    def __ge__(self, other):
        self.table.ops.append(4)
        self.table.opsValues.append(other)
        return self

    def __eq__(self, other):
        self.table.ops.append(5)
        self.table.opsValues.append(other)
        return self

    def __ne__(self, other):
        self.table.ops.append(6)
        self.table.opsValues.append(other)
        return self

    def _addLogical(self, noper):
        self.table.ops.append(noper)
        self.table.opsValues.append(None)
        self.table.opsColnames.append(None)

    def __and__(self, other):
        self._addLogical(10)
        return self

    def __or__(self, other):
        self._addLogical(11)
        return self

    def __xor__(self, other):
        self._addLogical(12)
        return self

    def createIndex(self, warn=1, testmode=0):
        """Create an index for this column"""

        # Check that file is not in read-only mode
        if self.table._v_file.mode == 'r':
            raise IOError("""\
Attempt to write over a file opened in read-only mode.""")

        # Warn if the index already exists
        if self.index:
            raise ValueError, \
"%s for column '%s' already exists. If you want to re-create it, please, try with reIndex() method better" % (str(self.index), str(self.pathname))

        if self.descr._v_shapes[self.name] != 1:
            raise ValueError("Only scalar columns can be indexed.")

        # Get the indexes group for table, and if not exists, create it
        getNode = self.table._v_file.getNode
        try:
            itgroup = getNode(_getIndexTableName(self.table._v_parent,
                                                 self.table._v_name))
        except NodeError:
            itgroup = self.table._createIndexesTable(self.table._v_parent)

        # Get the filters from indexprops
        filters = self.table.indexprops.filters

        # Create the necessary intermediate groups for descriptors
        idgroup = itgroup
        dname = ""
        pathname = self.descr._v_pathname
        if pathname <> "":
            inames = pathname.split('/')
            for iname in inames:
                if dname == "":
                    dname = iname
                else:
                    dname += '/'+iname
                try:
                    idgroup = getNode(self.table._v_parent._v_name+'/'+dname)
                except NodeError:
                    idgroup = self.table._createIndexesDescr(
                        idgroup, dname, iname, filters)

        # Create the atom
        atomtype = self.descr._v_types[self.name]
        if str(atomtype) == "CharType":
            atom = StringAtom(shape=self.descr._v_shapes[self.name],
                              length=self.descr._v_itemsizes[self.name])
        else:
            atom = Atom(dtype=atomtype,
                        shape=self.descr._v_shapes[self.name])

        # Get the index colum name
        iname = _getIndexColName(self.table._v_parent, self.table._v_name,
                                 self.name)
        # Create the index itself
        self.index = Index(atom=atom, column=self, name=iname,
                           title="Index for "+self.name+" column",
                           filters=filters,
                           expectedrows=self.table._v_expectedrows,
                           testmode=testmode)
        setattr(idgroup, self.name, self.index)
        # Feed the index with values
        nelemslice = self.index.nelemslice
        if self.table.nrows < nelemslice:
            if warn:
                # print "Debug: Not enough info for indexing"
                warnings.warn( \
"Not enough rows for indexing. You need at least %s rows and table has just %s." % (nelemslice, self.table.nrows))
            return 0
        # Add rows to the index
        indexedrows = self._addRowsToIndex(0, self.table.nrows, lastrow=1)
        self.dirty = 0
        # Set some flags in table parent
        self.table.indexed = True
        self.table.colindexed[self.pathname] = True
        # If the user has not defined properties, assign the default
        self.table.indexprops = getattr(
            self.table.description, '_v_indexprops', IndexProps())
        self.table._indexedrows = indexedrows
        self.table._unsaved_indexedrows = self.table.nrows - indexedrows
        return indexedrows

    def _addRowsToIndex(self, start, nrows, lastrow):
        """Add more elements to the existing index """
        nelemslice = self.index.nelemslice
        #assert self.table.nrows >= self.index.sorted.nelemslice
#         indexedrows = 0
#         for i in xrange(start, start+nrows-nelemslice+1, nelemslice):
#             self.index.append(self[i:i+nelemslice])
#             indexedrows += nelemslice
        # The next loop does not rely on xrange so that it can
        # deal with long ints (i.e. more than 32-bit integers)
        # This allows to index columns with more than 2**31 rows
        # F. Altet 2005-05-09
        indexedrows = 0
        i = start
        stop = start+nrows-nelemslice+1
        while i < stop:
            self.index.append(self[i:i+nelemslice])
            indexedrows += nelemslice
            i += nelemslice
        # index the remaining rows
        nremain = nrows - indexedrows
        if nremain > 0 and lastrow:
            #self.index.appendLastRow(self[indexedrows:nrows], self.table.nrows)
            #indexedrows += nremain
            pass
        return indexedrows

    def reIndex(self):
        """Recompute the existing index"""

        # Check that file is not in read-only mode
        if self.table._v_file.mode == 'r':
            raise IOError("""\
Attempt to write over a file opened in read-only mode.""")

        if self.index is not None:
            # Delete the existing Index
            self.index._remove()
            self.index = None
            # Create a new Index without warnings
            return self.createIndex(warn=0)
        else:
            return 0  # The column is not intended for indexing

    def reIndexDirty(self):
        """Recompute the existing index only if it is dirty"""

        # Check that file is not in read-only mode
        if self.table._v_file.mode == 'r':
            raise IOError("""\
Attempt to write over a file opened in read-only mode.""")

        if self.index is not None and self.dirty:
            # Delete the existing Index
            self.index._remove()
            # Create a new Index without warnings
            return self.createIndex(warn=0)
        else:
            # The column is not intended for indexing or is not dirty
            return 0

    def removeIndex(self):
        """Delete the associated column's index"""

        # Check that file is not in read-only mode
        if self.table._v_file.mode == 'r':
            raise IOError("""\
Attempt to write over a file opened in read-only mode.""")

        # delete some references
        if self.index:
            self.index._remove()
            self.index = None
            self.table.colindexed[self.name] = 0
        else:
            return  # Do nothing

    def close(self):
        """Close this column"""
        if self.index:
            self.index._f_close()  # Close the associated index
            self.index = None
        # delete some back references
        self.table = None
        self.type = None
        self.descr = None
        # After the objects are disconnected, destroy the
        # object dictionary using the brute force ;-)
        # This should help to the garbage collector
        self.__dict__.clear()

    def __str__(self):
        """The string representation for this object."""
        # The pathname
        tablepathname = self.table._v_pathname
        pathname = self.pathname.replace('/', '.')
        # Get this class name
        classname = self.__class__.__name__
        # The shape for this column
        shape = self.descr._v_shapes[self.name]
        if shape == 1:
            shape = (1,)
        # The type
        tcol = self.descr._v_types[self.name]
        return "%s.cols.%s (%s%s, %s, idx=%s)" % \
               (tablepathname, pathname, classname, shape, tcol, self.index)

    def __repr__(self):
        """A detailed string representation for this object."""
        return str(self)



## Local Variables:
## mode: python
## py-indent-offset: 4
## tab-width: 4
## fill-column: 72
## End:
