#include "domlette.h"

long g_nodeCount = 0;

/** Node C API *********************************************************/

/* Allocates memory for a new node object of the given type and initializes
   part of it.
*/
PyNodeObject *_Node_New(PyTypeObject *type, PyObject *ownerDocument)
{
  PyNodeObject *node = PyObject_GC_New(PyNodeObject, type);
  if (node != NULL) {
#ifdef DEBUG_NODE_CREATION
    printf("Created %s node at 0x%p\n", type->tp_name, node);
#endif
    node->parentNode = Py_None;
    Py_INCREF(Py_None);
    
    node->ownerDocument = ownerDocument;
    Py_INCREF(ownerDocument);

    g_nodeCount++;
  }

  return node;
}

void _Node_Del(PyNodeObject *node)
{
#ifdef DEBUG_NODE_CREATION
  printf("Destroyed %s node at 0x%p\n", node->ob_type->tp_name, node);
#endif

  g_nodeCount--;

  Py_XDECREF(node->parentNode);
  node->parentNode = NULL;

  Py_XDECREF(node->ownerDocument);
  node->ownerDocument = NULL;

  PyObject_GC_Del((PyObject *)node);
}

PyNodeObject *Node_RemoveChild(PyNodeObject *self, PyNodeObject *removeChild) {

  /* Remove the child from its parent list.
   * Assuming the child is actually a child of the parent, find out its index
   * then, remove it from our list of children.  
   * DECREF the child once to remove the dependency of the parent to the child
   * DECREF the parent once to remove the dependency of the child to the parent
   * set the child's parent Node to Py_None and INCREF Py_None
   */

  PyObject *children;
  int index, ctr;

  if (!PyNode_Check(self) || !PyNode_Check(removeChild)) {
    PyErr_BadInternalCall();
    return NULL;
  }
  
#ifdef DEBUG_NODE_REMOVE_CHILD
  printf("Remove Child initial state\n");
  printf("self\n");
  _PyObject_Dump(self);
  printf("------------------\n");
  printf("remove child\n");
  _PyObject_Dump(removeChild);
  printf("------------------\n");
#endif

  if (PyDocument_Check(self)) {
    children = PyDocument_CHILDNODES(self); 
  }
  else if (PyElement_Check(self)){
    children = PyElement_CHILDNODES(self); 
  }
  else if (PyDocumentFragment_Check(self)){
    children = ((PyDocumentFragmentObject *)self)->childNodes; 
  } 
  else {
    DOMException_HierarchyRequestErr("Not allowed to have children");
    return NULL;
  }

#ifdef DEBUG_NODE_REMOVE_CHILD
  printf("children\n");
  _PyObject_Dump(children);
  printf("------------------\n");
#endif
  
  /*Find the index of the child to be removed*/
  index = -1;
  for (ctr=0;ctr <  PyList_GET_SIZE(children);ctr++) {
    if (PyList_GET_ITEM(children, ctr) == (PyObject *)removeChild) {
      index = ctr;
      break;
    }
  }

  if (index == -1) {
    DOMException_NotFoundErr("Child not found");
    return NULL;
  }
  
#ifdef DEBUG_NODE_REMOVE_CHILD
  printf("index: %d\n",index);
#endif

  /*We have an index, call PySequence_DelItem which will DECREF child*/
  /*Temporarily INCREF the remove child incase the list has the last reference to it*/
  Py_INCREF(removeChild);

  PySequence_DelItem(children, index);

#ifdef DEBUG_NODE_REMOVE_CHILD
  printf("removeChild after delItem\n");
  _PyObject_Dump(removeChild);
  printf("------------------\n");
#endif
 
  /*Reset the parentNode on the child to Py_None, DECREF parent Node and INCREF Py_None*/
  Py_DECREF(removeChild->parentNode);
  removeChild->parentNode = Py_None;
  Py_INCREF(Py_None);

#ifdef DEBUG_NODE_REMOVE_CHILD
  printf("self after parent change\n");
  _PyObject_Dump(self);
  printf("------------------\n");
#endif

  /*Undo the temp INCREF*/
  Py_DECREF(removeChild);

#ifdef DEBUG_NODE_REMOVE_CHILD
  printf("Remove Child final state\n");
  printf("self\n");
  _PyObject_Dump(self);
  printf("------------------\n");
  printf("remove child\n");
  _PyObject_Dump(removeChild);
  printf("------------------\n");
  printf("children\n");
  _PyObject_Dump(children);
  printf("------------------\n");
#endif
  
  return removeChild;
}


PyNodeObject *Node_AppendChild(PyNodeObject *self, PyNodeObject *child)
{
  PyObject *children;
  PyObject *dfChildren;

  if (!PyNode_Check(self) || !PyNode_Check(child)) {
    PyErr_BadInternalCall();
    return NULL;
  }

#ifdef DEBUG_NODE_APPEND_CHILD
  printf("Append Child initial state\n");
  printf("self\n");
  _PyObject_Dump(self);
  printf("------------------\n");
  printf("child\n");
  _PyObject_Dump(child);
  printf("------------------\n");
#endif

  if (PyDocument_Check(self)) {
    children = PyDocument_CHILDNODES(self); 
  }
  else if (PyElement_Check(self)){
    children = PyElement_CHILDNODES(self); 
  }
  else if (PyDocumentFragment_Check(self)){
    children = ((PyDocumentFragmentObject *)self)->childNodes; 
  } 
  else {
    DOMException_HierarchyRequestErr("Not allowed to have children");
    return NULL;
  }

#ifdef DEBUG_NODE_APPEND_CHILD
  printf("children\n");
  _PyObject_Dump(children);
  printf("------------------\n");
#endif

  /* Perform special processing if the child is a DocumentFragment */
  if (PyDocumentFragment_Check(child)) {
    int i, size;
#ifdef DEBUG_NODE_APPEND_CHILD
    printf("Append a Document Fragment\n");
#endif
    /*Loop and add them all Removeing them from the df*/
    dfChildren = ((PyDocumentFragmentObject *)child)->childNodes;
    size = PyList_GET_SIZE(dfChildren);
    for (i = 0; i < size; i++) {
      /*Call append child on each of the children, append child does a remove if needed*/
      PyObject *dfChild = PyList_GET_ITEM(dfChildren, 0);
#ifdef DEBUG_NODE_APPEND_CHILD
      printf("dfChild %d before sub append\n",i);
      _PyObject_Dump(dfChild);
      printf("------------------\n");
#endif
      Node_AppendChild(self,(PyNodeObject *)dfChild);
#ifdef DEBUG_NODE_APPEND_CHILD
      printf("dfChild %d after sub append\n",i);
      _PyObject_Dump(dfChild);
      printf("------------------\n");
#endif
    }
  } 
  else {
    /* Do a simple append
       If the node already has a parent node, we need to remove it from its parent.
       Set its parent node to us
       Add it to the end of our child list

       Always incref the child once to create our own reference to it
       Always incref ourselves once because of the parent node relationship
    */

    /* Add the node to our children */
    PyList_Append(children, (PyObject *)child);

#ifdef DEBUG_NODE_APPEND_CHILD
    printf("child after list append\n");
    _PyObject_Dump(child);
    printf("------------------\n");
#endif

    /* Remove the child from its previous parent if it had one */
    if (child->parentNode == Py_None) {
      /*No current parent, decref Py_None*/
      Py_DECREF(Py_None);
    } else {
      /*Child has a parent, remove the child from the parent, remove Child does a decref of parent*/
      Node_RemoveChild((PyNodeObject *)child->parentNode, child);
    }

    /*Set the parent node to us, old parent already DECREFed*/
    child->parentNode = (PyObject *)self;
    Py_INCREF(self);
  }

#ifdef DEBUG_NODE_APPEND_CHILD
  printf("Append Child final state\n");
  printf("self\n");
  _PyObject_Dump(self);
  printf("------------------\n");
  printf("child\n");
  _PyObject_Dump(child);
  printf("------------------\n");
  printf("children\n");
  _PyObject_Dump(children);
  printf("------------------\n");
#endif

  return child;
}



PyNodeObject *Node_InsertBefore(PyNodeObject *self, PyNodeObject *newChild,
				PyNodeObject *refChild)
{
  PyObject *children;
  PyObject *dfChildren, *dfChild;
  int index;

  if (!PyNode_Check(self) || !PyNode_Check(newChild)) {
    PyErr_BadInternalCall();
    return NULL;
  }

#ifdef DEBUG_NODE_INSERT_BEFORE
  printf("Starting Insert Before\n");
  printf("new child\n");
  _PyObject_Dump(newChild);
  printf("---------------\n");
  printf("ref child\n");
  _PyObject_Dump(refChild);
  printf("---------------\n");
  printf("self\n");
  _PyObject_Dump(self);
  printf("---------------\n");
#endif

  if (PyDocument_Check(self)) {
    children = PyDocument_CHILDNODES(self);
  }
  else if (PyElement_Check(self)){
    children = PyElement_CHILDNODES(self); 
  }
  else if (PyDocumentFragment_Check(self)){
    children = ((PyDocumentFragmentObject *)self)->childNodes; 
  } 
  else {
    DOMException_HierarchyRequestErr("Not allowed to have children");
    return NULL;
  }

#ifdef DEBUG_NODE_INSERT_BEFORE
  printf("children\n");
  _PyObject_Dump(children);
  printf("---------------\n");
#endif

  if ((PyObject *)refChild == Py_None) {
#ifdef DEBUG_NODE_INSERT_BEFORE
    printf("Doing an append\n");
#endif
    if (Node_AppendChild(self, newChild) == NULL)
      return NULL;
  } else {
    /* Find the index of the reference node */
    for (index = 0; index <  PyList_GET_SIZE(children); index++) {
      if (PyList_GET_ITEM(children, index) == (PyObject *)refChild)
        break;
    }
    if (index == PyList_GET_SIZE(children)) {
      DOMException_NotFoundErr("Reference Node not found");
      return NULL;
    }
#ifdef DEBUG_NODE_INSERT_BEFORE
    printf("Reference node index: %d\n",index);
#endif
    
    if (PyDocumentFragment_Check(newChild)) {
      int i, size;
      /*Loop and add them all Removeing them from the df*/
      dfChildren = ((PyDocumentFragmentObject *)newChild)->childNodes;
      size = PyList_GET_SIZE(dfChildren);
      for (i = 0; i < size; i++) {
#ifdef DEBUG_NODE_INSERT_BEFORE
	printf("*****Start insert of DF child %d*****\n",i);
#endif
	dfChild = PyList_GET_ITEM(dfChildren, 0);
	Node_InsertBefore(self, (PyNodeObject *)dfChild, refChild);
#ifdef DEBUG_NODE_INSERT_BEFORE
	printf("*****End insert of DF child %d*****\n",i);
#endif
      }
      
    } else {
      /*If the node has a parent node that is not Py_None, we need to remove them from
	that parent.
	Then, insert them into our list, doing a single INCREF to the new child.
	
      */

      /* Add the newChild to our list of children */
      PyList_Insert(children, index, (PyObject *)newChild);

      if (newChild->parentNode == Py_None) {
	/*No current parent, decref Py_None*/
	Py_DECREF(Py_None);
      } else {
	/*Child has a parent, remove the child from the parent, remove Child does a decref of parent*/
	Node_RemoveChild((PyNodeObject *)newChild->parentNode, newChild);
      }

#ifdef DEBUG_NODE_INSERT_BEFORE
      printf("newChild after list append\n");
      _PyObject_Dump(newChild);
      printf("------------------\n");
#endif
      /*Set the parent node to us, old parent already DECREFed*/
      newChild->parentNode = (PyObject *)self;
      Py_INCREF(self);
    }
  }


#ifdef DEBUG_NODE_INSERT_BEFORE
  printf("Ending Insert Before\n");
  printf("new child\n");
  _PyObject_Dump(newChild);
  printf("---------------\n");
  printf("ref child\n");
  _PyObject_Dump(refChild);
  printf("---------------\n");
  printf("self\n");
  _PyObject_Dump(self);
  printf("---------------\n");
#endif
  return newChild;
}


PyNodeObject *Node_CloneNode(PyObject *node, int deep,
			     PyNodeObject *newOwnerDocument)
{
  PyObject *obj;
  int node_type;

  /* Get the nodeType as a plain integer */
  obj = PyObject_GetAttrString(node, "nodeType");
  if (!obj) return NULL;

  node_type = PyInt_AsLong(obj);
  Py_DECREF(obj);

  switch (node_type) {
  case ELEMENT_NODE:
    return (PyNodeObject *)Element_CloneNode(node, deep, newOwnerDocument);
  case ATTRIBUTE_NODE:
    return (PyNodeObject *)Attr_CloneNode(node, deep, newOwnerDocument);
  case TEXT_NODE:
    return (PyNodeObject *)Text_CloneNode(node, deep, newOwnerDocument);
  case COMMENT_NODE:
    return (PyNodeObject *)Comment_CloneNode(node, deep, newOwnerDocument);
  case DOCUMENT_FRAGMENT_NODE:
    return (PyNodeObject *)DocumentFragment_CloneNode(node, deep,
						      newOwnerDocument);
  case PROCESSING_INSTRUCTION_NODE:
    return (PyNodeObject *)ProcessingInstruction_CloneNode(node, deep,
							   newOwnerDocument);
  default:
    /* FIXME: DOMException */
    DOMException_NotSupportedErr("cloneNode: unknown nodeType %d");
    return NULL;
  }
}

/*
    Start the external interface

*/

/** Node Python API ****************************************************/

PyObject *PyNode_normalize(PyObject *self, PyObject *args)
{
  PyObject *children;
  int ctr;
  PyObject *current, *next, *temp;

  if (!PyArg_ParseTuple(args, ":normalize")) 
    return NULL;

#ifdef DEBUG_NODE_NORMALIZE
  printf("normalize initial state\n");
  printf("self\n");
  _PyObject_Dump(self);
  printf("------------------\n");
#endif

  if (PyDocument_Check(self)) {
    children = PyDocument_CHILDNODES(self);
  }
  else if (PyElement_Check(self)) {
    children = PyElement_CHILDNODES(self);
  }
  else if (PyDocumentFragment_Check(self)){
    children = ((PyDocumentFragmentObject *)self)->childNodes; 
  } 
  else {
    Py_INCREF(Py_None);
    return Py_None;
  }

  if (PyList_GET_SIZE(children) < 2) {
    Py_INCREF(Py_None);
    return Py_None;
  }

#ifdef DEBUG_NODE_NORMALIZE
  printf("children\n");
  _PyObject_Dump(children);
  printf("------------------\n");
#endif

  ctr = 0;
  /*Count from 0 to 1 minus the length (the last node cannot be normalized with anything*/
  while (ctr < PyList_GET_SIZE(children)-1) {
    /*See if the current node is a text node*/
    current = PyList_GET_ITEM(children,ctr);
#ifdef DEBUG_NODE_NORMALIZE
    printf("Current Node\n");
    _PyObject_Dump(current);
    printf("------------------\n");
#endif
    if (PyText_Check(current)) {
      next = PyList_GET_ITEM(children,ctr+1);
#ifdef DEBUG_NODE_NORMALIZE
      printf("Next Node\n");
      _PyObject_Dump(next);
      printf("------------------\n");
#endif
      if (PyText_Check(next)) {
	/*Merge them*/
	temp = PySequence_Concat((PyObject *)((PyTextObject *)current)->nodeValue, (PyObject *)((PyTextObject *)next)->nodeValue);
#ifdef DEBUG_NODE_NORMALIZE
	printf("Merged nodeValue\n");
	_PyObject_Dump(temp);
	printf("------------------\n");
#endif
	
	Py_DECREF(((PyTextObject *)current)->nodeValue);
	((PyTextObject *)current)->nodeValue = temp;
	
	/*Remove it*/
	Node_RemoveChild((PyNodeObject *)self,(PyNodeObject *)next);
#ifdef DEBUG_NODE_NORMALIZE
	printf("self after merge\n");
	_PyObject_Dump(self);
	printf("------------------\n");
	printf("current after merge\n");
	_PyObject_Dump(current);
	printf("------------------\n");
#endif
      } else {
	ctr ++;
      }
    } else {
      ctr++;
    }
  }

#ifdef DEBUG_NODE_NORMALIZE
  printf("normalize final state\n");
  printf("self\n");
  _PyObject_Dump(self);
  printf("------------------\n");
  printf("children\n");
  _PyObject_Dump(children);
  printf("------------------\n");
#endif

  Py_INCREF(Py_None);
  return Py_None;
}

PyObject *PyNode_hasChildNodes(PyObject * self, PyObject * args) {

  PyObject *rt = Py_False;

  if (!PyArg_ParseTuple(args, ":hasChildNodes")) 
    return NULL;

  if (PyDocument_Check(self) && 
      PyList_GET_SIZE(PyDocument_CHILDNODES(self))) {
    rt = Py_True;
  } 
  else if (PyElement_Check(self) && 
	   PyList_GET_SIZE(PyElement_CHILDNODES(self))) {
    rt = Py_True;
  }

  Py_INCREF(rt);
  return rt;
}

PyObject *PyNode_removeChild(PyObject *self, PyObject *args) {

  PyObject *removeChild;
  PyNodeObject *result;

  if (!PyArg_ParseTuple(args,"O:removeChild", &removeChild)) 
    return NULL;

  result = Node_RemoveChild((PyNodeObject *)self, (PyNodeObject *)removeChild);

  Py_XINCREF(result);
  return (PyObject *)result;
}


PyObject *PyNode_isSameNode(PyObject *self, PyObject *args)
{
  PyObject *other;

  if(!PyArg_ParseTuple(args,"O:isSameNode", &other)) 
    return NULL;

  return PyInt_FromLong(self == other);
}

PyObject *PyNode_xpath(PyObject *self, PyObject *args, PyObject *kw)
{
  PyObject *expr_text, *xpath_module, *eval, *result;
  PyObject *explicit_nss = NULL;
  static char *kwlist[] = {"expr", "explicitNss", NULL};

  if (!PyArg_ParseTupleAndKeywords(args, kw, "O|O:xpath", kwlist, 
                                   &expr_text, &explicit_nss))
    return NULL;

  if (explicit_nss == NULL){
    explicit_nss = Py_None;
    /* No need to Py_INCREF in these circumstances */
  }
  xpath_module = PyImport_ImportModule("Ft.Xml.XPath.Util");
  if (xpath_module == NULL) return NULL;
  eval = PyObject_GetAttrString(xpath_module, "SimpleEvaluate");
  if (eval == NULL) return NULL;
  result = PyObject_CallFunction(eval, "OOO", expr_text, self, explicit_nss);
  if (result == NULL) return NULL;
  /*_PyObject_Dump(result);*/
  return result;
}

PyObject *PyNode_appendChild(PyObject *self, PyObject *args)
{
  PyObject *newChild;
  PyNodeObject *oldChild;

  if(!PyArg_ParseTuple(args,"O:appendChild", &newChild)) 
    return NULL;

  /*Append child adds a ref to the child to signify the new relationship*/
  oldChild = Node_AppendChild((PyNodeObject *)self,(PyNodeObject *)newChild);
  if (oldChild == NULL) return NULL;
  /*Add a ref to the child for the return value*/
  Py_INCREF(oldChild);
  return (PyObject *)oldChild;
}


PyObject *PyNode_insertBefore(PyObject *self, PyObject *args)
{
  PyObject *newChild, *refChild;

  if(!PyArg_ParseTuple(args,"OO:insertBefore", &newChild,&refChild)) 
    return NULL;

  if (!Node_InsertBefore((PyNodeObject *)self, (PyNodeObject *)newChild,
			 (PyNodeObject *)refChild))
    return NULL;

  /*INCREF newChild for the return value*/
  Py_INCREF(newChild);
  return newChild;
}


PyObject *PyNode_replaceChild(PyObject *self, PyObject *args)
{
  PyObject *newChild, *oldChild, *sibling;

  if(!PyArg_ParseTuple(args,"OO:replaceChild", &newChild,&oldChild)) 
    return NULL;

#ifdef DEBUG_NODE_REPLACE_CHILD
  printf("replace child initial state\n");
  printf("self\n");
  _PyObject_Dump(self);
  printf("------------------\n");
  printf("newChild\n");
  _PyObject_Dump(newChild);
  printf("------------------\n");
  printf("oldChild\n");
  _PyObject_Dump(oldChild);
  printf("------------------\n");
#endif

  sibling = PyObject_GetAttrString(oldChild, "nextSibling");

#ifdef DEBUG_NODE_REPLACE_CHILD
  printf("sibling\n");
  _PyObject_Dump(sibling);
  printf("------------------\n");
#endif

  /*INC the return value before we call remove incase this is the last reference*/
  Py_INCREF(oldChild);

  if (!Node_RemoveChild((PyNodeObject *)self,(PyNodeObject *)oldChild)) {
    return NULL;
  }

#ifdef DEBUG_NODE_REPLACE_CHILD
  printf("oldChild after remove\n");
  _PyObject_Dump(oldChild);
  printf("------------------\n");
#endif

  /*Now insert it before the sibling (sibling could be Py_None but that is OK)*/
  if (!Node_InsertBefore((PyNodeObject *)self, (PyNodeObject *)newChild,
			 (PyNodeObject *)sibling))
    return NULL;

#ifdef DEBUG_NODE_REPLACE_CHILD
  printf("newChild after insert\n");
  _PyObject_Dump(oldChild);
  printf("------------------\n");
#endif

  /*DECREF the sibling (getattr added one)*/
  Py_DECREF(sibling);

#ifdef DEBUG_NODE_REPLACE_CHILD
  printf("replace child final state\n");
  printf("self\n");
  _PyObject_Dump(self);
  printf("------------------\n");
  printf("newChild\n");
  _PyObject_Dump(newChild);
  printf("------------------\n");
  printf("oldChild\n");
  _PyObject_Dump(oldChild);
  printf("------------------\n");
  printf("sibling\n");
  _PyObject_Dump(oldChild);
  printf("------------------\n");
#endif
  return oldChild;
}


PyObject *PyNode_cloneNode(PyObject *self, PyObject *args)
{
  PyObject *boolean_deep = Py_False;
  int deep;
  PyNodeObject *result;

  if (!PyArg_ParseTuple(args,"|O:cloneNode", &boolean_deep))
    return NULL;

  deep = PyObject_IsTrue(boolean_deep);
  if (deep == -1)
    return NULL;

  if (PyDocument_Check(self)) {
    PyErr_SetString(PyExc_TypeError,"cloneNode not allowed on documents");
    return NULL;
  } 

  result = Node_CloneNode(self, deep, 
                          (PyNodeObject *) PyNode_OWNER_DOCUMENT(self));

  return (PyObject *) result;
}


/*Used by the all of the types as helpers */

int node_clear(PyNodeObject *self)
{
#ifdef DEBUG_NODE_CREATION
  printf("Clearing %s node at 0x%p\n", self->ob_type->tp_name, self);
#endif

  Py_XDECREF(self->parentNode);
  self->parentNode = NULL;

  Py_XDECREF(self->ownerDocument);
  self->ownerDocument = NULL;

  return 0;
}


int node_traverse(PyNodeObject *node, visitproc visit, void *arg)
{
  int rt;
#ifdef DEBUG_NODE_CREATION
  printf("Traversing %s node at 0x%p\n", node->ob_type->tp_name, node);
#endif
  if (node->parentNode != NULL) {
    rt = visit(node->parentNode, arg);
    if (rt) return rt;
  }

  if (node->ownerDocument != NULL) {
    rt = visit(node->ownerDocument, arg);
    if (rt) return rt;
  }
  return 0;
}


PyObject *node_getattr(PyNodeObject *self, char *name, 
		       struct PyMethodDef methods[])
{
  PyObject * rt = NULL;
  int temp;
  PyObject *parentNode = NULL;
  int parentNodeIndex = -1;
  PyObject *childNodes;

  switch(name[0]) {
  case 'a':
    {
      if (!strcmp(name,"attributes")) {
	rt = Py_None;
      }
      break;
    }
  case 'c':
    {
      if (!strcmp(name,"childNodes")) {
	return PyList_New(0);
      }
      break;
    }
  case 'f':
    {
      if (!strcmp(name,"firstChild")) {
        rt = Py_None;
      }
      break;
    }
  case 'l':
    {
      if (!strcmp(name,"localName")) {
        rt = Py_None;
      }
      else if (!strcmp(name,"lastChild")) {
        rt = Py_None;
      }
      break;
    }
  case 'n':
    {
      if (!strcmp(name,"nodeValue")) {
        rt = Py_None;
      }
      else if (!strcmp(name,"namespaceURI")) {
        rt = Py_None;
      }
      else if (!strcmp(name,"nextSibling")) {
	/*Calculate this*/
	parentNode = self->parentNode;
	if (PyDocument_Check(parentNode)) {
	  childNodes = PyDocument_CHILDNODES(parentNode);
	} else if (PyElement_Check(parentNode)) {
	  childNodes = PyElement_CHILDNODES(parentNode);
	} else if (PyDocumentFragment_Check(parentNode)) {
	  childNodes = ((PyDocumentFragmentObject *)parentNode)->childNodes;
	} else {
	  Py_INCREF(Py_None);
	  return Py_None;
	}
	for (temp = 0;temp < PyList_GET_SIZE(childNodes);temp ++) {
	  if ((PyNodeObject *)PyList_GET_ITEM(childNodes,temp) == self) {
	    parentNodeIndex = temp;
	    break;
	  }
	}
	if (parentNodeIndex == -1) {
	  /*Through error????*/
	  Py_INCREF(Py_None);
	  return Py_None;
	}
	if (parentNodeIndex == PyList_GET_SIZE(childNodes) - 1) {
	  /*End of the list*/
	  rt = Py_None;
	} else {
	  rt = PyList_GET_ITEM(childNodes, parentNodeIndex+1);
	}
      }
      break;
    }
  case 'o':
    {
      if (!strcmp(name,"ownerDocument")) {
        rt = self->ownerDocument;
      }
      break;
    }
  case 'p':
    {
      if (!strcmp(name,"prefix")) {
        rt = Py_None;
      }
      else if (!strcmp(name,"parentNode")) {
        rt = self->parentNode;
      }
      else if (!strcmp(name,"previousSibling")) {
        /*Calculate this*/
        parentNode = self->parentNode;
        if (PyDocument_Check(parentNode)) {
          childNodes = PyDocument_CHILDNODES(parentNode);
        } else if (PyElement_Check(parentNode)) {
          childNodes = PyElement_CHILDNODES(parentNode);
        } else if (PyDocumentFragment_Check(parentNode)) {
          childNodes = ((PyDocumentFragmentObject *)parentNode)->childNodes;
        } else {
          Py_INCREF(Py_None);
          return Py_None;
        }
        for (temp = 0;temp < PyList_GET_SIZE(childNodes);temp ++) {
          if ((PyNodeObject *)PyList_GET_ITEM(childNodes,temp) == self) {
            parentNodeIndex = temp;
            break;
          }
        }
        if (parentNodeIndex == -1) {
          /*Through error????*/
          Py_INCREF(Py_None);
          return Py_None;
        }
        if (parentNodeIndex == 0) {
          /*Beginning of the list*/
          rt = Py_None;
        } else {
          rt = PyList_GET_ITEM(childNodes, parentNodeIndex-1);
        }
      }
      break;
    }
    /* handle node type constants */
  case 'A': 
    {
      if (!strcmp(name, "ATTRIBUTE_NODE")) {
        return PyInt_FromLong(ATTRIBUTE_NODE);
      }
      break;
    }
  case 'C':
    {
      if (!strcmp(name, "CDATA_SECTION_NODE")) {
        return PyInt_FromLong(CDATA_SECTION_NODE);
      }
      else if (!strcmp(name, "COMMENT_NODE")) {
        return PyInt_FromLong(COMMENT_NODE);
      }
      break;
    }
  case 'D':
    {
      if (!strcmp(name, "DOCUMENT_NODE")) {
        return PyInt_FromLong(DOCUMENT_NODE);
      }
      else if (!strcmp(name, "DOCUMENT_TYPE_NODE")) {
        return PyInt_FromLong(DOCUMENT_TYPE_NODE);
      }
      else if (!strcmp(name, "DOCUMENT_FRAGMENT_NODE")) {
        return PyInt_FromLong(DOCUMENT_FRAGMENT_NODE);
      }
      break;
    }
  case 'E':
    {
      if (!strcmp(name, "ELEMENT_NODE")) {
        return PyInt_FromLong(ELEMENT_NODE);
      }
      else if (!strcmp(name, "ENTITY_REFERENCE_NODE")) {
        return PyInt_FromLong(ENTITY_REFERENCE_NODE);
      }
      else if (!strcmp(name, "ENTITY_NODE")) {
        return PyInt_FromLong(ENTITY_NODE);
      }
      break;
    }
  case 'N': 
    {
      if (!strcmp(name, "NOTATION_NODE")) {
        return PyInt_FromLong(NOTATION_NODE);
      }
      break;
    }
  case 'P': 
    {
      if (!strcmp(name, "PROCESSING_INSTRUCTION_NODE")) {
        return PyInt_FromLong(PROCESSING_INSTRUCTION_NODE);
      }
      break;
    }
  case 'T': 
    {
      if (!strcmp(name, "TEXT_NODE")) {
        return PyInt_FromLong(TEXT_NODE);
      }
      break;
    }

  default:
    {
      if (!strcmp(name, "rootNode")) {
	rt = self->ownerDocument;
      }
      else if (!strcmp(name, "docIndex")) {
        return PyLong_FromLong(self->docIndex);
      }
      /* DOM3 baseURI is calculated according to XML Base */
      else if (!strcmp(name, "baseURI") || !strcmp(name, "xmlBase")) {
        if (PyDocument_Check(self)) {
          rt = PyDocument_BASE_URI(self);
        } else if (PyElement_Check(self)) {
          rt = PyElement_BASE_URI(self);
        } else if (PyDocumentFragment_Check(self)) {
          /* FIXME: Check that this is what we want */
          rt = Py_None;
        } else {
          parentNode = self->parentNode;
          if (PyDocument_Check(parentNode)) {
            rt = PyDocument_BASE_URI(parentNode);
          } else if (PyElement_Check(parentNode)) {
            rt = PyElement_BASE_URI(parentNode);
          } else if (PyDocumentFragment_Check(parentNode)) {
            /* FIXME: Check that this is what we want */
            rt = Py_None;
          }
        }
      }
      break;
    }
    break;
  }
  if (rt != NULL) {
    Py_INCREF(rt);
    return rt;
  }

  return Py_FindMethod(methods, (PyObject *)self, name);
}


int node_setattr(PyNodeObject *self, char *name, PyObject *v)
{
  /* Set attribute 'name' to value 'v'. v==NULL means delete */
  if (v == NULL) {
    PyErr_Format(PyExc_AttributeError, 
                 "Cannot delete attribute '%.400s' on '%.50s' object",
                 name, self->ob_type->tp_name);
  } else {
    PyErr_Format(PyExc_AttributeError,
                 "Cannot set attribute '%.400s' on '%.50s' object", 
                 name, self->ob_type->tp_name);
  }
  return -1;
}


int node_test_ref_counts(PyObject *tester,PyNodeObject *node,long *childCtr,PyObject *internCtr,int base) {

  char buf[256];
  /*Inc the child ctr*/
  (*childCtr)++;

  if (PyElement_Check(node)) {
    return element_test_ref_counts(tester,(PyElementObject *)node,childCtr,internCtr,base);
  } else if (PyText_Check(node)) {
    return text_test_ref_counts(tester,(PyTextObject *)node,childCtr,internCtr,base);
  } else if (PyComment_Check(node)) {
    return comment_test_ref_counts(tester,(PyCommentObject *)node,childCtr,internCtr,base);
  } else if (PyAttr_Check(node)) {
    return attr_test_ref_counts(tester,(PyAttrObject *)node,childCtr,internCtr,base);
  } else if (PyProcessingInstruction_Check(node)) {
    return pi_test_ref_counts(tester,(PyProcessingInstructionObject *)node,childCtr,internCtr,base);
  } else {
    sprintf(buf,"Untested type: %s",node->ob_type->tp_name);
    PyObject_CallMethod(tester,"warning","s",buf);
  }

  return 1;
}
