import tempfile, os

from Xml.Xslt import test_harness

from Ft.Lib import Uri
from Ft.Xml import InputSource, Domlette
from Ft.Xml.XPath import Context
from Ft.Xml.Xslt.Exslt import Strings

SRC1 = """<dummy/>"""

SRC2 = """\
<?xml version="1.0"?>
<foo>
  <a/>
  <b/>
  <c/>
</foo>"""

FUNCS = {}

def NotTested(tester, title):
    tester.startTest(title)
    tester.warning('Not tested')
    tester.testDone()
    return

def ApplyTest(tester, src, stylesheets, expected, topLevelParams=None,
              compareFunc=None, title=None):

    base = os.getcwd()
    if base[-1] != os.sep:
        base += os.sep
    base = Uri.OsPathToUri(base)
    source = test_harness.FileInfo(string=src)
    sheets = [ test_harness.FileInfo(string=sheet, baseUri=base) for sheet in stylesheets ]
    test_harness.XsltTest(tester, source, sheets, expected,
                          topLevelParams=topLevelParams,
                          compareFunc=compareFunc, title=title)
    return


###########################
# EXSLT - Common
###########################

FUNCS['Common'] = []

def TestNodeSet(tester):
    sty = """\
<?xml version="1.0"?>
<xsl:stylesheet
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
  xmlns:common="http://exslt.org/common"
  version="1.0"
>

  <xsl:output method="text"/>

  <xsl:template match="/">
    <xsl:variable name='spam'>
      <eggs/>
    </xsl:variable>
    <xsl:value-of select='count(common:node-set($spam)/*)'/>
    <xsl:text> </xsl:text>
    <xsl:value-of select='name(common:node-set($spam)/*)'/>
  </xsl:template>

</xsl:stylesheet>
"""
    result = ApplyTest(tester, SRC1, [sty], "1 eggs", compareFunc=cmp,
                       title='node-set')
    return

FUNCS['Common'].append(TestNodeSet)


# 4XSLT used to die a horrible death when computing the union of
# two node-sets that were created with exslt:node-set().
def TestNodeSetUnion(tester):
    sty1 = """\
<?xml version="1.0" encoding="utf-8"?>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
version="1.0"
xmlns:exslt="http://exslt.org/common"
exclude-result-prefixes="exslt">

  <xsl:output method="xml" indent="yes"/>

  <xsl:template match="/">

    <xsl:variable name="rtf1">
      <a/><b/><c/>
    </xsl:variable>

    <xsl:variable name="rtf2">
      <x/><y/><z/>
    </xsl:variable>

    <xsl:variable name="set1" select="$rtf1"/>
    <xsl:variable name="set2" select="$rtf2"/>

    <results>
      <xsl:variable name="rtf-union" select="exslt:node-set($rtf1)|exslt:node-set($rtf2)"/>
      <result info="$rtf1">
        <xsl:copy-of select="$rtf1"/>
      </result>
      <result info="$rtf2">
        <xsl:copy-of select="$rtf2"/>
      </result>
      <result info="$rtf-union">
        <xsl:for-each select="$rtf-union/*">
          <xsl:sort select="local-name()"/>
          <xsl:copy/>
        </xsl:for-each>
      </result>
    </results>

  </xsl:template>

</xsl:stylesheet>"""

    sty2 = """\
<?xml version="1.0" encoding="utf-8"?>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
version="1.0"
xmlns:exslt="http://exslt.org/common"
exclude-result-prefixes="exslt">

  <xsl:output method="xml" indent="yes"/>

  <xsl:template match="/">

    <xsl:variable name="rtf1">
      <a/><b/><c/>
    </xsl:variable>

    <xsl:variable name="rtf2">
      <x/><y/><z/>
    </xsl:variable>

    <xsl:variable name="set1" select="exslt:node-set($rtf1)"/>
    <xsl:variable name="set2" select="exslt:node-set($rtf2)"/>

    <results>
      <xsl:variable name="set-union" select="$set1|$set2"/>
      <result info="$set1">
        <xsl:copy-of select="$set1"/>
      </result>
      <result info="$set2">
        <xsl:copy-of select="$set2"/>
      </result>
      <result info="$set-union">
        <xsl:for-each select="$set-union/*">
          <xsl:sort select="local-name()"/>
          <xsl:copy/>
        </xsl:for-each>
      </result>
    </results>

  </xsl:template>

</xsl:stylesheet>"""

    sty3 = """\
<?xml version="1.0" encoding="utf-8"?>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
version="1.0"
xmlns:exslt="http://exslt.org/common"
exclude-result-prefixes="exslt">

  <xsl:output method="xml" indent="yes"/>

  <xsl:template match="/">

    <xsl:variable name="rtf2">
      <x/><y/><z/>
    </xsl:variable>

    <xsl:variable name="set1" select="/foo/*"/>
    <xsl:variable name="set2" select="exslt:node-set($rtf2)"/>

    <results>
      <xsl:variable name="set-union" select="$set1|$set2"/>
      <result info="$set1">
        <xsl:copy-of select="$set1"/>
      </result>
      <result info="$set2">
        <xsl:copy-of select="$set2"/>
      </result>
      <result info="$set-union">
        <xsl:for-each select="$set-union|$set-union/*">
          <xsl:sort select="local-name()"/>
          <xsl:copy/>
        </xsl:for-each>
      </result>
    </results>

  </xsl:template>

</xsl:stylesheet>"""

    expected_1 = """<?xml version="1.0" encoding="UTF-8"?>
<results>
  <result info="$rtf1">
    <a/>
    <b/>
    <c/>
  </result>
  <result info="$rtf2">
    <x/>
    <y/>
    <z/>
  </result>
  <result info="$rtf-union">
    <a/>
    <b/>
    <c/>
    <x/>
    <y/>
    <z/>
  </result>
</results>"""

    expected_2 = """<?xml version="1.0" encoding="UTF-8"?>
<results>
  <result info="$rtf1">
    <a/>
    <b/>
    <c/>
  </result>
  <result info="$rtf2">
    <x/>
    <y/>
    <z/>
  </result>
  <result info="$set-union">
    <a/>
    <b/>
    <c/>
    <x/>
    <y/>
    <z/>
  </result>
</results>"""

    expected_3 = expected_2

    ApplyTest(tester, SRC1, [sty1], expected_1, title='node-set union 1')
    ApplyTest(tester, SRC1, [sty2], expected_2, title='node-set union 2')
    ApplyTest(tester, SRC2, [sty3], expected_3, title='node-set union 3')
    return

FUNCS['Common'].append(TestNodeSetUnion)

def TestObjectType(tester):
    sty = """\
<?xml version="1.0"?>
<xsl:stylesheet
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
  xmlns:common="http://exslt.org/common"
  version="1.0"
>

  <xsl:output method="text"/>

  <xsl:template match="/">
    <xsl:variable name='spam'>
      <eggs/>
    </xsl:variable>
    <xsl:value-of select='common:object-type($spam)'/>
    <xsl:text> </xsl:text>
    <xsl:value-of select='common:object-type(.)'/>
    <xsl:text> </xsl:text>
    <xsl:value-of select='common:object-type("1")'/>
    <xsl:text> </xsl:text>
    <xsl:value-of select='common:object-type(1)'/>
    <xsl:text> </xsl:text>
    <xsl:value-of select='common:object-type(1=1)'/>
  </xsl:template>

</xsl:stylesheet>
"""
    ApplyTest(tester, SRC1, [sty], "RTF node-set string number boolean",
              compareFunc=cmp, title="object-type")
    return

FUNCS['Common'].append(TestObjectType)

def TestDocument(tester):
    sty = """\
<?xml version="1.0" encoding="ISO-8859-1"?>
<xsl:stylesheet
    xmlns:xsl="http://www.w3.org/1999/XSL/Transform" version="1.0"
    xmlns:common="http://exslt.org/common"
    extension-element-prefixes="common">

<xsl:output method="xml" indent="yes" encoding="ISO-8859-1"/>
<xsl:param name="filebase"/>

<xsl:template match="/">
  <xsl:apply-templates select="test"/>
</xsl:template>

<xsl:template match="test">
  <testout>
    <xsl:apply-templates select="data"/>
  </testout>
</xsl:template>

<xsl:template match="data">
  <xsl:variable name="file" select="concat($filebase, ., '.xml')"/>
  <datafile>
    <xsl:text>Writing data file </xsl:text>
    <xsl:value-of select="."/>
  </datafile>
  <common:document href="{$file}" method="xml" indent="yes" encoding="ISO-8859-1">
        <datatree>
          <name><xsl:value-of select="."/></name>
          <what>test</what>
        </datatree>
  </common:document>
</xsl:template>

</xsl:stylesheet>"""

    src = """\
<?xml version="1.0" encoding="ISO-8859-1"?>

<test>
  <data>11</data>
  <data>12</data>
  <data>13</data>
  <data>14</data>
  <data>15</data>
</test>
"""

    expected = """<?xml version="1.0" encoding="ISO-8859-1"?>
<testout>
  <datafile>Writing data file 11</datafile>
  <datafile>Writing data file 12</datafile>
  <datafile>Writing data file 13</datafile>
  <datafile>Writing data file 14</datafile>
  <datafile>Writing data file 15</datafile>
</testout>"""

    file_expected = """<?xml version="1.0" encoding="ISO-8859-1"?>
<datatree>
  <name>%s</name>
  <what>test</what>
</datatree>"""

    fileName = tempfile.mktemp()
    fileUri = Uri.OsPathToUri(fileName)
    tester.startGroup("common:document")
    ApplyTest(tester, src, [sty], expected,
              topLevelParams={'filebase' : fileUri},
              title='document')
    tester.startTest("document() results")
    for data in range(11, 16):
        file = '%s%d.xml' % (fileName, data)
        tester.compare(1, os.path.exists(file))
        fileData = open(file,'r').read()
        tester.compare(file_expected % data, fileData)
        if os.path.exists(file):
            os.unlink(file)
    tester.testDone()
    tester.groupDone()
    return

FUNCS['Common'].append(TestDocument)

###########################
# EXSLT - Sets
###########################

L = FUNCS['Sets'] = []

def TestDifference(tester):
    sty = """<?xml version="1.0"?>
<xsl:stylesheet
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
  xmlns:set="http://exslt.org/sets"
  xmlns:x="http://uche.ogbuji.net/dummy"
  version="1.0"
>

  <xsl:output method="text"/>

  <x:spam/>
  <x:eggs/>

  <xsl:variable name='self' select="document('')/*"/>

  <xsl:template match="/">
    <xsl:variable name='result' select='set:difference($self/x:*, $self/x:spam)'/>
    <xsl:value-of select='count($result)'/>
    <xsl:text> </xsl:text>
    <xsl:value-of select='name($result)'/>
  </xsl:template>

</xsl:stylesheet>
"""
    result = ApplyTest(tester, SRC1, [sty], "1 x:eggs", compareFunc=cmp,
                       title='difference')
    return

L.append(TestDifference)


def TestHasSameNode(tester):
    test = "Test set:has-same-node"
    sty = """\
<?xml version="1.0"?>
<xsl:stylesheet
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
  xmlns:set="http://exslt.org/sets"
  xmlns:x="http://uche.ogbuji.net/dummy"
  version="1.0"
>

  <xsl:output method="text"/>

  <x:spam/>
  <x:eggs/>

  <xsl:variable name='self' select='document("")/*'/>
  <xsl:variable name='empty' select='/parent::*'/>

  <xsl:template match="/">
    <xsl:variable name='r1' select='set:has-same-node($self/x:*, $self/x:spam)'/>
    <xsl:choose>
      <xsl:when test='$r1'>
        <xsl:text>works</xsl:text>
      </xsl:when>
      <xsl:otherwise>
        <xsl:text>broken</xsl:text>
      </xsl:otherwise>
    </xsl:choose>
    <xsl:text> </xsl:text>
    <xsl:variable name='r2' select='set:has-same-node($self/x:*, $empty)'/>
    <xsl:choose>
      <xsl:when test='$r2'>
        <xsl:text>broken</xsl:text>
      </xsl:when>
      <xsl:otherwise>
        <xsl:text>works</xsl:text>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

</xsl:stylesheet>
"""

    ApplyTest(tester, SRC1, [sty], "works works", compareFunc=cmp,
              title='has-same-node')
    return

L.append(TestHasSameNode)


def TestIntersection(tester):
    sty = """\
<?xml version="1.0"?>
<xsl:stylesheet
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
  xmlns:set="http://exslt.org/sets"
  xmlns:x="http://uche.ogbuji.net/dummy"
  version="1.0"
>

  <xsl:output method="text"/>

  <x:spam/>
  <x:eggs/>

  <xsl:variable name='self' select='document("")/*'/>
  <xsl:variable name='empty' select='/parent::*'/>

  <xsl:template match="/">
    <xsl:variable name='result' select='set:intersection($self/x:*, $self/x:spam)'/>
    <xsl:value-of select='count($result)'/>
    <xsl:text> </xsl:text>
    <xsl:value-of select='name($result)'/>
  </xsl:template>

</xsl:stylesheet>
"""

    ApplyTest(tester, SRC1, [sty], "1 x:spam", compareFunc=cmp,
              title='intersection')
    return


L.append(TestIntersection)


def TestDistinct(tester):
    sty = """\
<?xml version="1.0"?>
<xsl:stylesheet
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
  xmlns:set="http://exslt.org/sets"
  xmlns:x="http://uche.ogbuji.net/dummy"
  version="1.0"
>

  <xsl:output method="text"/>

  <x:node pos='1'>spam</x:node>
  <x:node pos='2'>eggs</x:node>
  <x:node pos='3'>spam</x:node>

  <xsl:variable name='self' select='document("")/*'/>

  <xsl:template match="/">
    <xsl:variable name='result' select='set:distinct($self/x:*)'/>
    <xsl:value-of select='count($result)'/>
    <xsl:for-each select='$result'>
      <xsl:text> </xsl:text>
      <xsl:value-of select='@pos'/>
      <xsl:value-of select='text()'/>
    </xsl:for-each>
  </xsl:template>

</xsl:stylesheet>
"""

    ApplyTest(tester, SRC1, [sty], "2 1spam 2eggs", compareFunc=cmp,
              title='distinct')
    return

L.append(TestDistinct)


def TestLeading(tester):
    sty = """\
<?xml version="1.0"?>
<xsl:stylesheet
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
  xmlns:set="http://exslt.org/sets"
  xmlns:x="http://uche.ogbuji.net/dummy"
  version="1.0"
>

  <xsl:output method="text"/>

  <x:spam/>
  <x:eggs/>
  <x:bread/>
  <x:eggs/>
  <x:spam/>

  <xsl:variable name='self' select='document("")/*'/>

  <xsl:template match="/">
  <xsl:variable name='result' select='set:leading($self/x:*, "name()=&apos;x:bread&apos;")'/>
    <xsl:value-of select='count($result)'/>
    <xsl:for-each select='$result'>
      <xsl:text> </xsl:text>
      <xsl:value-of select='name()'/>
    </xsl:for-each>
  </xsl:template>

</xsl:stylesheet>
"""

    ApplyTest(tester, SRC1, [sty], "0", compareFunc=cmp, title='leading')
    return

L.append(TestLeading)


def TestTrailing(tester):
    sty = """\
<?xml version="1.0"?>
<xsl:stylesheet
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
  xmlns:set="http://exslt.org/sets"
  xmlns:x="http://uche.ogbuji.net/dummy"
  version="1.0"
>

  <xsl:output method="text"/>

  <x:spam/>
  <x:eggs/>
  <x:bread/>
  <x:eggs/>
  <x:spam/>

  <xsl:variable name='self' select='document("")/*'/>

  <xsl:template match="/">
  <xsl:variable name='result' select='set:trailing($self/x:*, "name()=&apos;x:bread&apos;")'/>
    <xsl:value-of select='count($result)'/>
    <xsl:for-each select='$result'>
      <xsl:text> </xsl:text>
      <xsl:value-of select='name()'/>
    </xsl:for-each>
  </xsl:template>

</xsl:stylesheet>
"""

    ApplyTest(tester, SRC1, [sty], "0", compareFunc=cmp, title='trailing')
    return

L.append(TestTrailing)

###########################
# EXSLT - Math
###########################

L = FUNCS['Math'] = []

def TestMax(tester):
    sty = """\
<?xml version="1.0"?>
<xsl:stylesheet
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
  xmlns:math="http://exslt.org/math"
  xmlns:common="http://exslt.org/common"
  xmlns:x="http://uche.ogbuji.net/dummy"
  version="1.0"
>

  <xsl:output method="text"/>

  <x:spam>2</x:spam>
  <x:eggs>12.0</x:eggs>
  <x:bread>100</x:bread>

  <xsl:variable name='self' select='document("")/*'/>

  <xsl:template match="/">
    <xsl:variable name='result' select='math:max($self/x:*)'/>
    <xsl:value-of select='common:object-type($result)'/>
    <xsl:text> </xsl:text>
    <xsl:value-of select='$result'/>
  </xsl:template>

</xsl:stylesheet>
"""
    ApplyTest(tester, SRC1, [sty], "number 100", compareFunc=cmp, title='max')
    return

L.append(TestMax)


def TestMin(tester):
    sty = """\
<?xml version="1.0"?>
<xsl:stylesheet
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
  xmlns:math="http://exslt.org/math"
  xmlns:common="http://exslt.org/common"
  xmlns:x="http://uche.ogbuji.net/dummy"
  version="1.0"
>

  <xsl:output method="text"/>

  <x:spam>2</x:spam>
  <x:eggs>12.0</x:eggs>
  <x:bread>100</x:bread>

  <xsl:variable name='self' select='document("")/*'/>

  <xsl:template match="/">
    <xsl:variable name='result' select='math:min($self/x:*)'/>
    <xsl:value-of select='common:object-type($result)'/>
    <xsl:text> </xsl:text>
    <xsl:value-of select='$result'/>
  </xsl:template>

</xsl:stylesheet>
"""
    ApplyTest(tester, SRC1, [sty], "number 2", compareFunc=cmp, title='min')
    return

L.append(TestMin)


def TestHighest(tester):
    sty = """\
<?xml version="1.0"?>
<xsl:stylesheet
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
  xmlns:math="http://exslt.org/math"
  xmlns:common="http://exslt.org/common"
  xmlns:x="http://uche.ogbuji.net/dummy"
  version="1.0"
>

  <xsl:output method="text"/>

  <x:spam>2</x:spam>
  <x:bread>100</x:bread>
  <x:eggs>12.0</x:eggs>

  <xsl:variable name='self' select='document("")/*'/>

  <xsl:template match="/">
    <xsl:variable name='result' select='math:highest($self/x:*)'/>
    <xsl:value-of select='name($result)'/>
  </xsl:template>

</xsl:stylesheet>
"""
    ApplyTest(tester, SRC1, [sty], "x:bread", compareFunc=cmp, title='highest')
    return

L.append(TestHighest)


def TestLowest(tester):
    sty = """\
<?xml version="1.0"?>
<xsl:stylesheet
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
  xmlns:math="http://exslt.org/math"
  xmlns:common="http://exslt.org/common"
  xmlns:x="http://uche.ogbuji.net/dummy"
  version="1.0"
>

  <xsl:output method="text"/>

  <x:spam>2</x:spam>
  <x:bread>100</x:bread>
  <x:eggs>12.0</x:eggs>

  <xsl:variable name='self' select='document("")/*'/>

  <xsl:template match="/">
    <xsl:variable name='result' select='math:lowest($self/x:*)'/>
    <xsl:value-of select='name($result)'/>
  </xsl:template>

</xsl:stylesheet>
"""
    ApplyTest(tester, SRC1, [sty], "x:spam", compareFunc=cmp, title='lowest')
    return

L.append(TestLowest)

###########################
# EXSLT - Functions
###########################

L = FUNCS['Functions'] = []

def TestFunction(tester):

    src1 = """<?xml version="1.0"?><foo/>"""

    sty1 = """\
<?xml version="1.0" encoding="utf-8"?>
<xsl:stylesheet version="1.0"
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
  xmlns:myns="http://stuff.foo"
  xmlns:func="http://exslt.org/functions"
  extension-element-prefixes="func"
  exclude-result-prefixes="myns">

  <xsl:output method="xml"/>

  <func:function name="myns:toUpperCase">
    <xsl:param name="stringToConvert"/>
    <func:result select="translate($stringToConvert,
                         'abcdefghijklmnopqrstuvwxyz',
                         'ABCDEFGHIJKLMNOPQRSTUVWXYZ')"/>
  </func:function>

  <xsl:template match="/">
    <result><xsl:value-of select="myns:toUpperCase('Hello World')"/></result>
  </xsl:template>

</xsl:stylesheet>"""

    expected1 = """<?xml version="1.0" encoding="UTF-8"?>
<result>HELLO WORLD</result>"""

    sty2 = """\
<?xml version="1.0"?>
<xsl:stylesheet
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
  xmlns:x="http://uche.ogbuji.net/dummy"
  xmlns:func="http://exslt.org/functions"
  extension-element-prefixes="func"
  version="1.0"
>

  <func:function name="x:count-elements">
    <xsl:for-each select="(//*)[1]">
      <func:result select="count(//*)" />
    </xsl:for-each>
  </func:function>

  <xsl:template match="/">
    <out>
      <xsl:value-of select="x:count-elements()" />
    </out>
  </xsl:template>

</xsl:stylesheet>
"""

    src2 = """\
<doc>
                              <section index="section1"
                                       index2="atr2val">
                                 <section index="subSection1.1">
                                    <p>Hello</p>
                                    <p>Hello again.</p>
                                 </section>
                              </section>
                              <section index="section2">
                                 <p>Hello2</p>
                                 <section index="subSection2.1">
                                    <p>Hello</p>
                                    <p>Hello again.</p>
                                 </section>
                                 <section index="subSection2.2">
                                    <p>Hello</p>
                                    <p>Hello again.</p>
                                 </section>
                                 <p>Hello2 again.</p>
                                 <section index="subSection2.3">
                                    <p>Hello</p>
                                    <p>Hello again.</p>
                                 </section>
                              </section>
                           </doc>
                           """
    expected2 = """<?xml version="1.0" encoding="UTF-8"?>
<out xmlns:x="http://uche.ogbuji.net/dummy">17</out>"""

    # SF #707131
    src3 = "<?xml version='1.0'?><test>echo me</test>"
    sty3 = """<?xml version='1.0'?>
<xsl:transform version='1.0'
  xmlns:xsl='http://www.w3.org/1999/XSL/Transform'
  xmlns:a='http://a' xmlns:b='http://b'
  exclude-result-prefixes='a b'>

  <xsl:include href='Xml/Xslt/Core/function-problem-a.xslt'/>
  <xsl:include href='Xml/Xslt/Core/function-problem-b.xslt'/>

  <xsl:template match='/'>
    <result>
      <xsl:value-of select='b:b-echo(a:a-echo(.))'/>
    </result>
  </xsl:template>
</xsl:transform>
"""
    expected3 = """<?xml version="1.0" encoding="UTF-8"?>
<result>echo me</result>"""

    ApplyTest(tester, src1, [sty1], expected1, title='function 1')
    ApplyTest(tester, src2, [sty2], expected2, title='function 2')
    ApplyTest(tester, src3, [sty3], expected3, title='function 3')
    return

L.append(TestFunction)

###########################
# EXSLT - Date Time
###########################

DATETIME = ['add',
            'add-duration',
            'day-abbreviation',
            'day-in-month',
            'day-in-week',
            'day-in-year',
            'day-name',
            'day-of-week-in-month',
            'difference',
            'duration',
            'format-date',
            'hour-in-day',
            'leap-year',
            'minute-in-hour',
            'month-abbreviation',
            'month-in-year',
            'month-name',
            'parse=date',
            'second-in-minute',
            'seconds',
            'sum',
            'week-in-month',
            'week-in-year',
            'year',
            ]

L = FUNCS['Dates and Times'] = []

for title in DATETIME:
    dummy = lambda tester, title=title: NotTested(tester, title)
    L.append(dummy)

def TestDateTimeFuncs(tester):
    import time, re
    from Ft.Xml.Xslt.Exslt import DateTime
    reader = Domlette.NonvalidatingReader
    node = reader.parseString(SRC1, 'urn:dummy')
    context = Context.Context(node)

    # date:date-time()
    tester.startTest('date:date-time() syntax')
    result = DateTime.DateTime(context)
    pat = r'^\d\d\d\d-[01]\d-[0-3]\dT[012]\d:[0-5]\d:[0-6]\d(Z|[+\-]\d\d:?\d\d)?$'
    matched = re.search(pat, result) is not None
    tester.compare(1, matched, msg='Result %r does not match regexp %r' % (result, pat))
    tester.testDone()

    # date:date()
    tester.startTest('date:date() syntax')
    result = DateTime.Date(context)
    pat = r'^\d\d\d\d-[01]\d-[0-3]\d$'
    matched = re.search(pat, result) is not None
    tester.compare(1, matched, msg='Result %r does not match regexp %r' % (result, pat))
    tester.testDone()
    # Our implementation only supports ISO 8601 date, time, and date/time formats for input & output.
    # xs:dateTime values are the same as ISO 8601 date/time strings.
    # xs:date values can have an optional time zone on the end, unlike ISO 8601 date strings.
    # xs:time values do not start with 'T', unlike ISO 8601 time strings.
    DATETIMES = [
                  (u'2003-09-12T23:59:59-0400', u'2003-09-12'),
                  (u'2003-09-12T23:59:59-04:00', u'2003-09-12'), # offset in extended format
                  (u'2001-01-01T00:00:00Z', u'2001-01-01'),
                  (u'2000-01-01T00:00:00', u'2000-01-01'),
                  (u'2005-05-05', u'2005-05-05'),
                  (u'Jan 01, 2001', u''),
                  #(u'2005-05-05+0100', u'2005-05-05+0100'), # tz in = tz out? spec unclear
                ]
    for DATE, expected in DATETIMES:
        tester.startTest("date:date('%s')" % DATE)
        result = DateTime.Date(context, unicode(DATE))
        tester.compare(expected, result)
        tester.testDone()

    # date:time()
    tester.startTest('date:time() syntax')
    result = DateTime.Time(context)
    pat = r'^[012]\d:[0-5]\d:[0-6]\d(Z|[+\-]\d\d:?\d\d)?$'
    matched = re.search(pat, result) is not None
    tester.compare(1, matched, msg='Result %r does not match regexp %r' % (result, pat))
    tester.testDone()
    # see note above re: ISO 8601 vs XSD
    # Our implementation always returns with a timezone.
    # It's not supposed to, if none was given in the input.
    DATETIMES = [
                  (u'2003-09-12T23:59:59-0400', [u'23:59:59-04:00', u'23:59:59-0400', u'03:59:59Z']),
                  (u'2003-09-12T23:59:59-04:00', [u'23:59:59-04:00', u'23:59:59-0400', u'03:59:59Z']),
                  (u'2001-01-01T00:00:00Z', u'00:00:00Z'),
                  (u'2000-01-01T00:00:00', u'00:00:00Z'),
                  (u'00:00:00', u'00:00:00'), # xs:date input good!
                  (u'T00:00:00', u''),        # ISO 8601 date input bad!
                  (u'02:22:22 PM', u''),
                ]
    for TIME, expected in DATETIMES:
        tester.startTest("date:time('%s')" % TIME)
        result = DateTime.Time(context, unicode(TIME))
        if type(expected) == type([]):
            tester.compareIn(expected, result)
        else:
            tester.compare(expected, result)
        tester.testDone()

    return

L.append(TestDateTimeFuncs)

###########################
# EXSLT - Strings
###########################

STRINGS = ['align',
           'concat',
           'decode-uri',
           'padding',
           'tokenize',
           ]

L = FUNCS['Strings'] = []

for title in STRINGS:
    dummy = lambda tester, title=title: NotTested(tester, title)
    L.append(dummy)

def TestEncodeUri(tester):
    strings = { 0: u'hello world',
                1: u'hello%20world',
                2: u'1 & 2 are < 3',
                3: u'100% OK?',
                4: u'\u00a1Hola!',
                5: u'\u4eca\u65e5\u306f',
              }

    encodings = { 1: u'utf-8',
                  2: u'iso-8859-1',
                  3: u'utf-16le',
                }

    escape_res = { 1: u'true()',
                   0: u'false()',
                 }

    expected = { #(str, encoding, escape_reserved): expected
                 (0, 1, 0): u'hello%20world',
                 (0, 1, 1): u'hello%20world',
                 (0, 2, 0): u'hello%20world',
                 (0, 2, 1): u'hello%20world',
                 (0, 3, 0): u'hello%20world',
                 (0, 3, 1): u'hello%20world',
                 (1, 1, 0): u'hello%20world',
                 (1, 1, 1): u'hello%20world',
                 (1, 2, 0): u'hello%20world',
                 (1, 2, 1): u'hello%20world',
                 (1, 3, 0): u'hello%20world',
                 (1, 3, 1): u'hello%20world',
                 (2, 1, 0): u'1%20&%202%20are%20%3C%203',
                 (2, 1, 1): u'1%20%26%202%20are%20%3C%203',
                 (2, 2, 0): u'1%20&%202%20are%20%3C%203',
                 (2, 2, 1): u'1%20%26%202%20are%20%3C%203',
                 (2, 3, 0): u'1%20&%202%20are%20%3C%203',
                 (2, 3, 1): u'1%20%26%202%20are%20%3C%203',
                 (3, 1, 0): u'100%25%20OK?',
                 (3, 1, 1): u'100%25%20OK%3F',
                 (3, 2, 0): u'100%25%20OK?',
                 (3, 2, 1): u'100%25%20OK%3F',
                 (3, 3, 0): u'100%25%20OK?',
                 (3, 3, 1): u'100%25%20OK%3F',
                 (4, 1, 0): u'%C2%A1Hola!',
                 (4, 1, 1): u'%C2%A1Hola!',
                 (4, 2, 0): u'%A1Hola!',
                 (4, 2, 1): u'%A1Hola!',
                 (4, 3, 0): u'%A1%00Hola!',
                 (4, 3, 1): u'%A1%00Hola!',
                 (5, 1, 0): u'%E4%BB%8A%E6%97%A5%E3%81%AF',
                 (5, 1, 1): u'%E4%BB%8A%E6%97%A5%E3%81%AF',
                 (5, 2, 0): u'%3F%3F%3F',
                 (5, 2, 1): u'%3F%3F%3F',
                 (5, 3, 0): u'%CA%4E%E5%65%6F%30',
                 (5, 3, 1): u'%CA%4E%E5%65%6F%30',
               }

    reader = Domlette.NonvalidatingReader
    node = reader.parseString(SRC1, 'urn:dummy')
    context = Context.Context(node)

    for s_key in strings.keys():
        s = strings[s_key]
        for enc_key in encodings.keys():
            encoding = encodings[enc_key]
            for esc_key in escape_res.keys():
                escape = escape_res[esc_key]
                test_title = u"str:encode-uri(%r, %s, '%s')" % (s, escape, encoding)
                tester.startTest(test_title)
                if expected.has_key((s_key, enc_key, esc_key)):
                    expected_str = expected[(s_key, enc_key, esc_key)]
                    result = Strings.EncodeUri(context, s, esc_key, encoding)
                    tester.compare(expected_str, result)
                else:
                    tester.warning('Not tested; expected result unknown')
                tester.testDone()
    return

L.append(TestEncodeUri)

def TestStringReplace(tester):

    sty1 = """<?xml version="1.0" encoding="utf-8"?>
<xsl:stylesheet version="1.0"
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
  xmlns:exsl="http://exslt.org/common"
  xmlns:str="http://exslt.org/strings"
  extension-element-prefixes="exsl str">

  <xsl:output method="xml" indent="yes"/>

  <xsl:include href="Xml/Xslt/Core/str.replace.function.xsl"/>
  <xsl:include href="Xml/Xslt/Core/str.replace.template.xsl"/>

  <xsl:template match="/">
    <result>
      <!-- just strings -->
      <func>
        <xsl:copy-of select="str:replace('kill me','ll','ss')"/>
      </func>
      <efun>
        <xsl:copy-of select="str:replacef('kill me','ll','ss')"/>
      </efun>
      <tmpl>
        <xsl:call-template name="str:replacet">
          <xsl:with-param name="string" select="'kill me'" />
          <xsl:with-param name="search" select="'ll'" />
          <xsl:with-param name="replace" select="'ss'" />
        </xsl:call-template>
      </tmpl>

      <func>
        <xsl:copy-of select="str:replace('ha-ha silly','ha','bye')"/>
      </func>
      <efun>
        <xsl:copy-of select="str:replacef('ha-ha silly','ha','bye')"/>
      </efun>
      <tmpl>
        <xsl:call-template name="str:replacet">
          <xsl:with-param name="string" select="'ha-ha silly'" />
          <xsl:with-param name="search" select="'ha'" />
          <xsl:with-param name="replace" select="'bye'" />
        </xsl:call-template>
      </tmpl>

      <func>
        <xsl:copy-of select="str:replace('ha-ha silly','boo','bye')"/>
      </func>
      <efun>
        <xsl:copy-of select="str:replacef('ha-ha silly','boo','bye')"/>
      </efun>
      <tmpl>
        <xsl:call-template name="str:replacet">
          <xsl:with-param name="string" select="'ha-ha silly'" />
          <xsl:with-param name="search" select="'boo'" />
          <xsl:with-param name="replace" select="'bye'" />
        </xsl:call-template>
      </tmpl>

      <!-- getting fancy -->
      <xsl:variable name="s" select="'four score and seven years ago, our forefathers'"/>
      <xsl:variable name="search">
        <substring>e</substring>
        <substring>seven</substring>
        <substring>our</substring>
        <substring>the</substring>
      </xsl:variable>
      <xsl:variable name="searchNodes" select="exsl:node-set($search)/node()"/>
      <xsl:variable name="replace">
        <E/>
        <xsl:comment>HI</xsl:comment>
        <xsl:text>OO</xsl:text>
      </xsl:variable>
      <xsl:variable name="replNodes" select="exsl:node-set($replace)/node()"/>
      <func>
        <xsl:copy-of select="str:replace($s,$searchNodes,$replNodes)"/>
      </func>
      <efun>
        <xsl:copy-of select="str:replacef($s,$searchNodes,$replNodes)"/>
      </efun>
      <tmpl>
        <xsl:call-template name="str:replacet">
          <xsl:with-param name="string" select="$s" />
          <xsl:with-param name="search" select="$searchNodes" />
          <xsl:with-param name="replace" select="$replNodes" />
        </xsl:call-template>
      </tmpl>

      <!-- exceptional empty string case -->
      <xsl:variable name="replace2">
        <br/>
      </xsl:variable>
      <xsl:variable name="replNodes2" select="exsl:node-set($replace2)/node()"/>
      <func>
        <xsl:copy-of select="str:replace($s,'',$replNodes2)"/>
      </func>
      <efun>
        <xsl:copy-of select="str:replacef($s,'',$replNodes2)"/>
      </efun>
      <tmpl>
        <xsl:call-template name="str:replacet">
          <xsl:with-param name="string" select="$s" />
          <xsl:with-param name="search" select="''" />
          <xsl:with-param name="replace" select="$replNodes2" />
        </xsl:call-template>
      </tmpl>

      <!-- attrs and namespace nodes in replacements are treated as empty strings -->
      <xsl:variable name="replace3">
        <foo bar="baz"/>
      </xsl:variable>
      <xsl:variable name="replNodes3" select="exsl:node-set($replace3)/foo/@bar"/>
      <func>
        <xsl:copy-of select="str:replace($s,'years',$replNodes3)"/>
      </func>
      <efun>
        <xsl:copy-of select="str:replacef($s,'years',$replNodes3)"/>
      </efun>
      <tmpl>
        <xsl:call-template name="str:replacet">
          <xsl:with-param name="string" select="$s" />
          <xsl:with-param name="search" select="'years'" />
          <xsl:with-param name="replace" select="$replNodes3" />
        </xsl:call-template>
      </tmpl>


      <!-- string to node -->
      <xsl:variable name="s3" select="'line one&#10;line two&#10;line three'"/>
      <func>
        <xsl:copy-of select="str:replace($s3,'&#10;',$replNodes2)"/>
      </func>
      <efun>
        <xsl:copy-of select="str:replacef($s3,'&#10;',$replNodes2)"/>
      </efun>
      <tmpl>
        <xsl:call-template name="str:replacet">
          <xsl:with-param name="string" select="$s3" />
          <xsl:with-param name="search" select="'&#10;'" />
          <xsl:with-param name="replace" select="$replNodes2" />
        </xsl:call-template>
      </tmpl>
<oleg>
 <xsl:variable name="srch">
   <substr>$name</substr>
   <substr>$id</substr>
 </xsl:variable>
 <xsl:variable name="repl-rtf">
   <name>phd</name>
   <id>1234</id>
 </xsl:variable>
 <xsl:variable name="repl" select="exsl:node-set($repl-rtf)/node()"/>
 <xsl:copy-of select="str:replace('Patient: $name ($id)',$srch,$repl)"/>
</oleg>

    </result>
  </xsl:template>

</xsl:stylesheet>"""

    expected1 = """<?xml version="1.0" encoding="UTF-8"?>
<result>
  <func>kiss me</func>
  <efun>kiss me</efun>
  <tmpl>kiss me</tmpl>
  <func>bye-bye silly</func>
  <efun>bye-bye silly</efun>
  <tmpl>bye-bye silly</tmpl>
  <func>ha-ha silly</func>
  <efun>ha-ha silly</efun>
  <tmpl>ha-ha silly</tmpl>
  <func>fOO scor<E/> and <!--HI--> y<E/>ars ago, OO for<E/>fars</func>
  <efun>fOO scor<E/> and <!--HI--> y<E/>ars ago, OO for<E/>fars</efun>
  <tmpl>fOO scor<E/> and <!--HI--> y<E/>ars ago, OO for<E/>fars</tmpl>
  <func>four score and seven years ago, our forefathers</func>
  <efun>four score and seven years ago, our forefathers</efun>
  <tmpl>four score and seven years ago, our forefathers</tmpl>
  <func>four score and seven  ago, our forefathers</func>
  <efun>four score and seven  ago, our forefathers</efun>
  <tmpl>four score and seven  ago, our forefathers</tmpl>
  <func>line one<br/>line two<br/>line three</func>
  <efun>line one<br/>line two<br/>line three</efun>
  <tmpl>line one<br/>line two<br/>line three</tmpl>
  <oleg>Patient: $name ($id)</oleg>
</result>"""

    ApplyTest(tester, SRC1, [sty1], expected1, title='str:replace')
    return

L.append(TestStringReplace)

def TestReplace(tester):
    sty1 = """<?xml version="1.0" encoding="utf-8"?>
<xsl:stylesheet version="1.0"
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
  xmlns:str="http://exslt.org/strings"
  exclude-result-prefixes="str">

  <xsl:output method="xml" indent="yes"/>

  <xsl:template match="/">
    <result>
      <one>
        <xsl:value-of select="str:replace('ha-ha silly','ha','bye')"/>
      </one>
      <two>
        <xsl:value-of select="str:replace('ha-ha silly','boo','bye')"/>
      </two>
    </result>
  </xsl:template>

</xsl:stylesheet>"""

    expected_1 = """<?xml version="1.0" encoding="UTF-8"?>
<result>
  <one>bye-bye silly</one>
  <two>ha-ha silly</two>
</result>"""

    tester.startGroup("str:replace()")
    ApplyTest(tester, SRC1, [sty1], expected_1)
    tester.groupDone()
    return

L.append(TestReplace)


def TestSplit(tester):
    sty1 = """<?xml version="1.0" encoding="utf-8"?>
<xsl:stylesheet version="1.0"
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
  xmlns:str="http://exslt.org/strings">

  <xsl:output method="xml" indent="yes"/>

  <xsl:template match="/">
    <result>
      <one>
        <xsl:copy-of select="str:split('a, simple, string',', ')"/>
      </one>
      <two>
        <xsl:copy-of select="str:split('date math str')"/>
      </two>
      <three>
        <xsl:copy-of select="str:split('foo','')"/>
      </three>
    </result>
  </xsl:template>

</xsl:stylesheet>"""

    expected_1 = """<?xml version="1.0" encoding="UTF-8"?>
<result xmlns:str="http://exslt.org/strings">
  <one>
    <token>a</token>
    <token>simple</token>
    <token>string</token>
  </one>
  <two>
    <token>date</token>
    <token>math</token>
    <token>str</token>
  </two>
  <three>
    <token>f</token>
    <token>o</token>
    <token>o</token>
  </three>
</result>"""

    ApplyTest(tester, SRC1, [sty1], expected_1, title='str:split')
    return

L.append(TestSplit)


###########################
# EXSLT - Regular Expressions
###########################

REGEXP = ['match',
          'replace',
          'test',
          ]

L = FUNCS['Regular Expressions'] = []

for title in REGEXP:
    dummy = lambda tester, title=title: NotTested(tester, title)
    L.append(dummy)


def Test(tester):
    for (group, tests) in FUNCS.items():
        tester.startGroup('EXSLT - %s' % group)
        for test in tests:
            test(tester)
        tester.groupDone()
    return
