########################################################################
# $Header: /var/local/cvsroot/4Suite/Ft/Server/Server/Http/Basic.py,v 1.118 2006/03/29 19:40:47 cogbuji Exp $
"""
4Suite API module for the HTTP Handler

Copyright 2005 Fourthought, Inc. (USA).
Detailed license and copyright information: http://4suite.org/COPYRIGHT
Project home, documentation, distributions: http://4suite.org/
"""

__all__ = ['BasicHttpHandler', 'XmlBody']

import re, urlparse, os, time, cgi, sha, cgi
try:
    # Python 2.3+
    from types import basestring as StringTypes
except ImportError:
    from types import StringTypes

import BaseRequestHandler
import HeaderDict
import Status
import Session

from Ft.Lib import Time, Uri
from Ft.Server import RESERVED_NAMESPACE
from Ft.Server.Common import Schema, ResourceTypes, Util
from Ft.Server.Common.ResourceTypes import *
from Ft.Server.Server import FtServerServerException
from Ft.Server.Server import Error as ServerError
from Ft.Server.Server import SCore
from Ft.Server.Server.Http import FtServerHttpException, Error
from Ft.Server.Server.SCore import GetRepository
from Ft.Xml.Lib.XmlString import IsQName
from Ft.Server.Server.Drivers import FtssInputSource

REPL_PAT = re.compile(r"(?<!\\)\\([0-9])")

class BasicHttpHandler(BaseRequestHandler.BaseRequestHandler):

    isSoap = False
    isXmlBody = False
    repo = None
    commit = False

    def do_GET(self):
        try:
            try:
                self._process_request(self.args)
            except FtServerServerException, e:
                self._handle_exception(e)
            except:
                self.commit = False
                raise
        finally:
            if self.repo is not None:
                if self.commit:
                    self.repo.txCommit()
                else:
                    self.repo.txRollback()
        return

    def do_POST(self):
        content_type = self.headers_in['Content-Type'][0]
        #SOAP 1.2, strictly
        self.isSoap = content_type == 'application/soap'
        self.isXmlBody = re.match(r'(?:text|application)/.*\+?xml',
                                  content_type) is not None
        try:
            try:
                self._process_request(self.args)
            except FtServerServerException, e:
                self._handle_exception(e)
            except:
                self.commit = False
                raise
        finally:
            if self.repo is not None:
                if self.commit:
                    self.repo.txCommit()
                else:
                    self.repo.txRollback()
        return

    def do_PUT(self):
        try:
            try:
                self._process_request(self.args)
            except FtServerServerException, e:
                self._handle_exception(e)
            except:
                self.commit = False
                raise
        finally:
            if self.repo is not None:
                if self.commit:
                    self.repo.txCommit()
                else:
                    self.repo.txRollback()
        return
    do_DELETE = do_PUT

    def _handle_exception(self, e):
        errorLog = self.server.errorLog
        self.commit = False
        if e.errorCode in [ServerError.UNKNOWN_PATH, ServerError.INVALID_PATH]:
            errorLog.debug("Access to unknown path %s", e.params['path'])
            self.send_error(Status.HTTP_NOT_FOUND, uri=self.path)
            return
        elif e.errorCode == ServerError.PERMISSION_DENIED:
            if not self.authChecker(mandatory=True):
                errorLog.warning('Attempted unauthorized %s access to %s',
                                 e.params['level'], e.params['path'])
                if self.session_method and \
                       self.server.session_permission_denied_uri:
                    self.sendRedirect(
                        self.server.session_permission_denied_uri)
                else:
                    # send_error will determine if the response should be
                    # HTTP_FORBIDDEN or HTTP_UNAUTHORIZED
                    self.send_error(Status.HTTP_FORBIDDEN, uri=self.path)
            return
        raise

    def _process_request(self, args):
        # make it local to avoid excessive attribute lookups
        errorLog = self.server.errorLog

        # Process configuration options from query arguments into
        # instance variables.
        self.load_config(args)

        # Get a connection to the repository.  There are 4 possible options:
        #  1. an existing session
        #  2. a new session
        #  3. a 403 authentication
        #  4. an anonymous connection
        if self.getRepo():
            # error response already sent
            return

        # Commit all changes unless told otherwise (via XSLT extensions)
        self.commit = True

        #First of all, if there are no query args, try to match a
        #processing rule
        if not args:
            resource, chain, args = self.apply_rules()
        else:
            resource = None
        if resource is None:
            #For PUT/DELETE requests, If no stylesheets are registered by rule, the default actions are:
            #PUT: update the resource identified by request-uri
            #DELETE: delete the resource identified by request-uri
            if self.method.lower() in ['put','delete']:
                path = self.filename
                if self.method.lower() == 'put':
                    if not self.repo.hasResource(path):
                        #Resource doen't exist.. Create.
                        #From RFC2616 / 9.6:
                        #The recipient of the entity MUST NOT ignore any Content-* (e.g. Content-Range) headers that it does not understand or implement
                        #and MUST return a 501 (Not Implemented) response in such cases.
                        understoodHeaders=['content-type','content-length']
                        nonUnderstoodHeaders = [header for header in self.headers_in.keys() if header.lower().find('content-')+1 and header.lower() not in understoodHeaders]
                        if nonUnderstoodHeaders:
                            self.status = Status.HTTP_NOT_IMPLEMENTED
                            return
                        isXml = False
                        newImt = 'text/plain'
                        if 'Content-Type' in self.headers_in:
                            content_type=self.headers_in['Content-Type'][0]
                            if re.match(r'(?:text|application)/.*\+?xml',content_type) is not None:
                                #Create as XML Document
                                isXml = True
                            else:
                                #Create as RawFile
                                newImt = content_type
                        else:
                            #Behavior is unclear in this case... Do we assume it's XML and invoke createDocument on the repository
                            #or parse it to determine if it's XML (use createDocument) / nonXML (use createRawFile)?
                            from Ft.Xml.Domlette import NonvalidatingReader
                            try:
                                NonvalidatingReader.parseString(self.body, path)
                                isXml = True
                            except:
                                pass
                        if isXml:
                            self.repo.createDocument(path, self.body)
                        else:
                            self.repo.createRawFile(path, newImt, self.body)
                        self.server.errorLog.info("Created new resource: %s\n", str(path))
                        self.status = Status.HTTP_CREATED
                        return
                    else:
                        #Update an existing resource via setContent
                        res=self.repo.fetchResource(path)
                        res.setContent(self.body)
                        self.server.errorLog.info("Updated Resource: %s\n", str(path))
                else:
                    res=self.repo.fetchResource(path)
                    res.delete()
                    self.server.errorLog.info("Deleted Resource: %s\n", str(path))

                #self.wfile.write(body)
                self.status = Status.HTTP_OK
                return
            else:
                #If they specify an 'xslt' arg, use it, else, if there is a
                #default for this directory or resource type use that.
                resource = self.load_resource(not args)
                if resource is None:
                    # error response already sent
                    return
                chain = self.load_stylesheets(resource)

        # If there are stylesheets to process, use them to get the content
        # to return.  Otherwise, just return the content of the resource.
        if chain:
            # prepare stylesheet params (usable by the stylesheet; not
            # modifiable outside of XSLT processing)
            document_root = self.server.documentRoot
            server_port = self.connection.local_port
            request_uri = self.hostname + self.unparsed_uri
            user_name = self.user or BaseRequestHandler.ANONYMOUS_USER_NAME

            # FIXME - This is a hack.  We really need a parameter that
            #   properly represents the resource ID of the source doc
            resource_id = resource.getAbsolutePathAsUri()
            source_uri = resource.getAbsolutePath()[len(document_root):]

            # The URI of the highest precedence stylesheet in that is
            # first in the chain of stylesheets to process.
            stylesheet_path = (isinstance(chain[0], (tuple, list)) and
                               chain[0][0] or chain[0]).getAbsolutePath()
            xslt_uri = stylesheet_path[len(document_root):]
            params = {
                (RESERVED_NAMESPACE, 'uri-path') : self.path,
                (RESERVED_NAMESPACE, 'http-host') : self.hostname,
                (RESERVED_NAMESPACE, 'server-name') : self.hostname,
                (RESERVED_NAMESPACE, 'server-port') : server_port,
                (RESERVED_NAMESPACE, 'absolute-request-uri') : request_uri,
                # deprecated; remove for 1.0 final
                (RESERVED_NAMESPACE, 'request-uri') : request_uri,
                (RESERVED_NAMESPACE, 'request') : self.unparsed_uri,
                (RESERVED_NAMESPACE, 'user-name') : user_name,
                (RESERVED_NAMESPACE, 'document-root') : document_root,
                (RESERVED_NAMESPACE, 'resource-id') : resource_id,
                (RESERVED_NAMESPACE, 'source-uri') : source_uri,
                (RESERVED_NAMESPACE, 'xslt-uri') : xslt_uri,
                # deprecated; remove for 1.0 final
                (RESERVED_NAMESPACE, 'stylesheet-uri') : xslt_uri,
                }

            if self.isSoap:
                soap_action = self.headers_in.get('SOAPAction', [''])[0]
                params[(RESERVED_NAMESPACE, 'soap-action')] = soap_action

            if self.session:
                params[(RESERVED_NAMESPACE, 'sessionId')] = self.session.id

            # query args are stylesheet params too
            for name, value in args.items():
                # colons and characters not allowed in XSLT parameter
                # names become hyphens.
                name = name.replace(u':', u'-')
                if not IsQName(name):
                    for i in xrange(len(name)):
                        if not IsQName(name[:i+1]):
                            name = name.replace(name[i], u'-')

                # Treat given parameters the same as 4XSLT does
                # (if multiple values for same param, only use
                # the first value)
                if len(value) > 1:
                    params[name] = value
                else:
                    params[name] = value[0]

            # prepare extension params for extension elements and
            # functions to use (they belong to the processor and can
            # be modified)
            ext_params = {
                (RESERVED_NAMESPACE, 'commit'): 0,
                (RESERVED_NAMESPACE, 'handler'): self,
                (RESERVED_NAMESPACE, 'error-log'): errorLog,
                (RESERVED_NAMESPACE, 'access-log'): self.server.accessLog,
                (RESERVED_NAMESPACE, 'queryArgs'): args,
            }

            errorLog.debug("Begin Rendering")
            #chain is a list of lists.  The outer list is the chain
            #of stylesheets.  Each inner list represents all
            #the transforms that make up each link in the chain
            body, imt = resource.applyXslt(chain, params,
                                           ignorePis=(not self.enable_pis),
                                           extParams=ext_params,
                                           imtParams=True)
            errorLog.debug("End Rendering")

            if ext_params.get((RESERVED_NAMESPACE, 'rollback')):
                errorLog.info('Extension element/function requested'
                              ' rollback\n')
                self.commit = False

            login = ext_params.get((RESERVED_NAMESPACE, 'login'))
            if login:
                redirect = self._session_login(*login)
            else:
                redirect = ext_params.get((RESERVED_NAMESPACE, 'response-uri'))
            if redirect:
                errorLog.info("Redirecting to: %s\n", redirect)
                if self.session:
                    headers = HeaderDict.HeaderDict()
                    self.session.updateHeaders(self.repo, headers, errorLog)
                else:
                    headers = None
                return self.sendRedirect(redirect, headers)

            if self.session:
                self.session.updateHeaders(self.repo, self.headers_out,
                                           errorLog)
        else:
            # If the browser requested an entity only "If-Modified-Since"
            # sometime after or equal to the entity's last modified date
            # in 4SS, then respond with a 304 ("Not Modified").

            #FIXME: rfc2068 sction 14.26 seems to indicate that
            #we must handle "If-None-Match: *", even though that
            #header would only really make sense for PUTs

            last_modified = resource.getLastModifiedDate()
            check_last_mod = self.headers_in.terse().get('If-Modified-Since')
            if check_last_mod:
                lmd = Time.FromISO8601(last_modified)
                # Netscape 4 sends If-Modified-Since headers that have the
                # full weekday name and "; extra crap" on the end
                clm = Time.FromRFC822(check_last_mod.split(';')[0])
                if lmd <= clm:
                    #Do not add cookie headers for requests that
                    #don't involve them, since these do not concern
                    #app logic, and the 'expires' Set-Cookie field
                    #prevents browsers from caching.
                    self.status = Status.HTTP_NOT_MODIFIED
                    return
            #Set the Last-Modified Header, e.g.
            # Mon, 17 Dec 2001 19:29:08 GMT
            # and ensure it doesn't exceed the Date header
            # as per RFC 2616 sec. 14.19.
            dt = Time.FromISO8601(last_modified)
            if dt.asPythonTime() > time.mktime(time.gmtime(self.request_time)):
                dt = Time.FromPythonTime(self.request_time)

            # Always add a Last-Modified HTTP header to the response
            # to help browsers cache entities.
            self.headers_out['Last-Modified'] = dt.asRFC822DateTime()

            body = resource.getContent()
            imt = resource.getImt()

        # If we reached this point, there were no errors getting the resource
        # or transform result from the repo and no status has been set.
        # Update the headers, add the body to the output and set the HTTP
        # status to OK.
        self.headers_out['Content-Type'] = imt
        self.wfile.write(body)
        self.status = Status.HTTP_OK
        return


    def load_config(self, queryArgs):
        # make it local to avoid excessive attribute lookups
        config = self.server
        errorLog = config.errorLog

        logging = {}

        ## URI Query Path
        # The URI query path option allows us to override what path is used
        # as the source object in this request.  There are two options
        # associated with it:
        #   1. the path to be specified as the request URL to initiate
        #      this processing
        #   2. the query arg used to define the new source path
        self.uri_path = queryArgs.get(config.uri_param, [None])[0]
        if self.uri_path:
            logging['uri_path'] = self.uri_path

        ## XSLT Param
        # The xslt paramater is used to specify what stylesheet(s) are
        # applied to the source object.  It consists of a name of the
        # query paramater, and a list of stylesheets.
        # Ignore PIS tells us the param to send to the processor
        self.stylesheets = queryArgs.get(config.xslt_param, [])
        self.enable_pis = config.enable_pis_param in queryArgs
        if self.stylesheets:
            logging['stylesheets'] = self.stylesheets
            logging['enable_pis'] = self.enable_pis

        ## Sessions
        # To define how sessions work, there are many options.  First is
        # what type of session are supported.  Then, where is the username
        # and password for sesion found.  How is anon session created.
        # What are the uris to redirect to on an invalid and valid login
        # lastly, what page is displayed when a session is invalidated
        self.session_method = config.session_method
        if self.session_method:
            self.session_user_name = queryArgs.get(config.session_user_name,
                                                  [None])[0]
            logging['session_user_name'] = self.session_user_name

            self.session_password = queryArgs.get(config.session_password,
                                                  [None])[0]
            logging['session_password'] = self.session_password

            self.session_anonymous = config.session_anonymous_flag in queryArgs
            logging['session_anonymous'] = self.session_anonymous

            if config.session_login_uri in queryArgs:
                self.session_post_login_uri = \
                    queryArgs[config.session_login_uri][0]
                logging['session_post_login_uri'] = self.session_post_login_uri
            else:
                self.session_post_login_uri = None

            if config.session_invalid_login_uri in queryArgs:
                self.session_post_invalid_login_uri = \
                    queryArgs[config.session_invalid_login_uri][0]
                logging['session_post_invalid_login_uri'] = \
                    self.session_post_invalid_login_uri
            else:
                self.session_post_invalid_login_uri = None

        ## Actions
        # Give us a hint as to what to do at a high level
        if config.action_param in queryArgs:
            self.action = queryArgs[config.action_param][0]
            logging['action'] = self.action
        else:
            self.action = 'pass'

        # Make a copy as this mapping MAY be modified
        self.redirects = config.redirects.copy()

        # Disable XSLT defaults is a query param that will override default processing
        if config.disable_default_xslt_param in queryArgs:
            self.defaultXslt = {}
            logging['disable_default_xslt'] = True
        else:
            self.defaultXslt = config.defaultXslt

        # Log all the loaded config options
        if logging:
            items = logging.keys()
            items.sort()
            errorLog.debug("Loaded options from request query args:\n")
            for item in items:
                errorLog.debug('    %s: %s\n', item, logging[item])
        return

    def getRepo(self):
        """
        Return 1 if a response was already sent
        """
        # make it local to avoid excessive attribute lookups
        errorLog = self.server.errorLog

        self.repo = None
        self.session = None

        if self.action == 'login':
            ##Perform a login.  Get the query args user-name and password
            #Create a new session, then return the response uri

            if not self.session_method:
                raise FtServerHttpException(Error.SESSIONS_NOT_ENABLED,
                    login_query_arg=self.server.action_param)

            if self.session_anonymous or (not self.session_user_name and not self.session_password):
                self.user = None
                self.password = None
                errorLog.debug("Session-based login requested for anonymous user")
            else:
                self.user = self.session_user_name
                self.password = sha.new(self.session_password or '').hexdigest()
                errorLog.debug("Session-based login requested for user %s", self.user)

            try:
                # sendError=False will prevent a failed login from generating an error response
                # that would normally be for HTTP authentication; we will generate our own
                # response, if needed
                repo = self.getRepository(sendError=False)
            except FtServerServerException, e:
                if e.errorCode == ServerError.INVALID_LOGIN:
                    if not self.session_post_invalid_login_uri:
                        raise FtServerHttpException(Error.BAD_LOGIN_FAILURE_REDIRECT)

                    #Send back a 302
                    errorLog.error("Repository access denied for user %s; will redirect", self.user)
                    self.sendRedirect(self.session_post_invalid_login_uri)
                    return True
                raise
            sid = repo.createSession('', self.server.session_ttl)
            self.session = Session.Create(sid, self.session_method, errorLog)

            if self.session_post_login_uri:
                errorLog.info("Login successful; new session created; will redirect")
                if self.session:
                    headers = HeaderDict.HeaderDict()
                    self.session.updateHeaders(repo, headers, errorLog)
                else:
                    headers = None
                self.sendRedirect(self.session_post_login_uri, headers)
                repo.txCommit()
                return True
            #If not, then they want to do more with this
            errorLog.info("Login successful; new session created; no redirect")
            self.repo = repo
            return False
        elif self.session_method:
            # Try to use the request headers to associate the request
            # with a session that has been stored in the repo.
            # This call will raise an exception if there is a sessionId
            # or login info in the request but no session in the repo.
            try:
                session = Session.Retrieve(self.args, self.headers_in,
                                           self.session_method, errorLog)
                if session:
                    self.session = session
                    #FIXME: Get client address and check for exceptions
                    self.repo = SCore.RetrieveSession(session.id, '', errorLog,
                                                      self.server.properties)
                    errorLog.info("Retrieved Session for user %s",
                                  self.repo.getUserName())
                    return False
            except FtServerServerException, e:
                redirect = self.server.session_invalid_uri
                if session:
                    uri = session.userAgentVariables.get('timeout-redirect')
                    if uri:
                        redirect = uri

                # in case of bogus or expired sessionId, redirect
                if e.errorCode == ServerError.INVALID_SESSION and redirect:
                    errorLog.info("Session invalid; redirecting to %s",
                                  redirect)
                    headers = HeaderDict.HeaderDict()
                    self.session.updateHeaders(None, headers, errorLog)
                    self.sendRedirect(redirect, headers)
                    return True
                elif e.errorCode != ServerError.INVALID_SESSION:
                    from Ft.Server.Server import MessageSource
                    errorLog.error("Error (%d) during session retrieval: %s",
                                   e.errorCode, e.message)

        #If we are here, then there is no login or session (except maybe 403 auth)
        #Try the old-fashioned way
        self.repo = self.getRepository()
        return self.repo is None

    def getHostHeader(self):
        #Apache reverse proxies add X-Forwarded-Host and munge Host
        headers = self.headers_in.terse()
        host_port = '%s:%s' % (self.server.hostname, self.server.port)
        host_port = headers.get('X-Forwarded-Host',
                                headers.get('Host', host_port))
        return host_port

    def sendRedirect(self, response_uri, headers=None, permanent=False):
        """Generates in the response a Location: header from the given
        response_uri, and sets the response code to 302 or 301, depending
        on whether permanent was 0 or 1. Defaults to 302 (temporary).
        The Location header value must be an absolute URI, optionally
        with a fragment. In the event of a relative URI, the base URI is
        calculated from the request's Host header, if any, or from the
        server's configured hostname and port."""
        # fragment allowed in Location header per RFC 2616 errata
        # (see http://skrb.org/ietf/http_errata.html#location-fragments )
        self.server.errorLog.debug("Redirect to '%s' requested", response_uri)
        base_uri = 'http://' + self.getHostHeader() + self.path
        response_uri = Uri.Absolutize(response_uri, base_uri)
        self.server.errorLog.info("Preparing response to redirect to %s",
                                  response_uri)
        self.headers_out['Location'] = response_uri
        if headers:
            self.headers_out.update(headers)
        if permanent:
            self.status = Status.HTTP_MOVED_PERMANENTLY
        else:
            self.status = Status.HTTP_MOVED_TEMPORARILY
        return ''


    def load_resource(self, allowRedirect):
        if self.isXmlBody or self.isSoap:
            return XmlBody(self.body, self.repo)

        # if the requested resource path is the magic path that indicates that we want
        # to use a query argument value as the resource's path, then try to use it
        #
        # example:  /resourcebyuri?uri=/ftss/data/us-states.xml
        #
        if self.path == self.server.uri_query_path:
            path = self.uri_path
            if not uri_arg:
                # Malformed
                msg = _("Requests for %s must have the query argument '%s'" %
                         (self.server.uri_query_path, self.server.uri_param))
                self.send_error(Status.HTTP_BAD_REQUEST, error=msg)
                return
        else:
            path = self.filename
        self.server.errorLog.info("Loading Resource: %s\n", str(path))
        res = self.repo.fetchResource(path)
        #If resource is a container and the resource path in the request did
        #not have a trailing slash, prepare to redirect to the container with the slash
        #NOTE: use unparsed_path because path can contain ;no-traverse
        perm = False
        if self.unparsed_path[-1] != '/' and res.resourceType == ResourceType.CONTAINER:
            allowRedirect = True
            perm = True
            newURI = '%s/' % self.unparsed_path
            # ('', '', '/buyerbase', '', 'foo=bar', '')
            if self.unparsed_params:
                newURI += ';%s' % self.unparsed_params
            if self.unparsed_args:
                newURI += '?%s' % self.unparsed_args
            self.redirects[res.getAbsolutePath()] = newURI
            self.server.errorLog.debug("Container resource requested without trailing '/'. Redirecting.")

        # handle redirects.
        # if there were no query args, allowRedirect will be false.
        if res.getAbsolutePath() in self.redirects and allowRedirect:
            if self.session:
                headers = HeaderDict.HeaderDict()
                self.session.updateHeaders(self.repo, headers,
                                           self.server.errorLog)
            else:
                headers = None
            self.sendRedirect(self.redirects[res.getAbsolutePath()], headers,
                              perm)
            return None
        return res

    def load_stylesheets(self, resource):
        """Load all available stylesheets"""
        if not self.stylesheets:
            # See if there is a default
            default = self.defaultXslt.get(resource.resourceType)
            if default:
                self.stylesheets.append(default)

        stys = []
        for sty in self.stylesheets:
            #Raise to let the exception handling take action
            s = resource.fetchResource(sty)
            stys.append(s)
            self.server.errorLog.info("Loaded Stylesheet: %s\n",
                                      str(s.getAbsolutePath()))

        return stys

    def apply_rules(self):
        for pattern, src, xforms, args, chain, method in self.server.processing_rules:
            #path = res.getPath().absolutePath
            m = pattern.match(self.path)
            # only patterns matching the entire path are considered
            # e.g. pattern "/a/b/c" would match on "/a/b/c/d"
            #NOTE:  Patterns match by HTTP method as well
            if m and m.group() == self.path and (self.method.lower() in [meth for meth in method.split()] or not method):
                self.server.errorLog.info(
                    "Applying rule matching pattern %r\n", pattern.pattern)
                try:
                    res = self.repo.fetchResource(self.filename)
                except:
                    if self.isXmlBody:
                        res = XmlBody(self.body, self.repo)
                    else:
                        res = self.repo

                # regular expression back-reference replacement function
                # e.g. <Rule pattern="a(b)c" xslt-source="\1"/> would
                #  produce a source of "b"
                backref_replace = lambda sm, pm=m: pm.group(int(sm.group(1)))
                if src:
                    # Handle back-references in xslt-source
                    src = REPL_PAT.sub(backref_replace, src)
                    res = res.fetchResource(src)
                if args:
                    # Handle back-references in extra-args
                    args = REPL_PAT.sub(backref_replace, args)

                stys = []
                for xform in xforms.split():
                    # Handle back-references in xslt-transform
                    xform = REPL_PAT.sub(backref_replace, xform)
                    stys.append(res.fetchResource(xform))

                if chain:
                    stys = [ (i,) for i in stys ]
                new_args = cgi.parse_qs(args, keep_blank_values=True)
                matched_args = m.groupdict("")
                for k in matched_args:
                    new_args[k] = [matched_args[k]]
                return (res, stys, new_args)
        return (None, [], {})

    def get_form_encoding(self, form):
        if self.form_encoding: return
        import cgi
        encoding = 'ISO-8859-1'
        fields = cgi.parse_qs(form)
        if self.server.http_post_encoding_var in fields:
            encoding = fields[self.server.http_post_encoding_var][0]
        self.form_encoding = encoding

    def _render_request(self, resource, chain, args):
        return body, imt

    def _session_login(self, user, password, ttl, successUri, failureUri,
                       timeoutUri):
        # make it local to avoid excessive attribute lookups
        errorLog = self.server.errorLog

        errorLog.debug("Session-based login requested via extension "
                       "parameters for user %s", user)

        # Use *current* repo to check credentials because they may have added
        # a user or changed a password in this tx
        failed = not self.repo.checkLogin(user, password)
        if failed:
            errorLog.error("Repository access denied for user %s;"
                           " will redirect", user)
            return failureUri

        if self.repo.hasSession():
            self.repo.invalidateSession()

        sid = self.repo.createSession('', ttl or self.server.session_ttl,
                                      overriddenLogin=(user, password))
        self.session = Session.Create(sid, self.session_method, errorLog)
        errorLog.info("Login successful; new session created;"
                      " will redirect")
        self.session.userAgentVariables['timeout-redirect'] = timeoutUri
        return successUri


class XmlBody:
    resourceType = ResourceTypes.ResourceType.XML_DOCUMENT
    def __init__(self, text, repo):
        self.body = text
        self.repo = repo
        return

    def fetchResource(self, path):
        return self.repo.fetchResource(path)

    def getAbsolutePath(self):
        return ''

    def getAbsolutePathAsUri(self):
        return ''

    def applyXslt(self, stylesheets, params=None, ignorePis=True,
                  extParams=None, extModules=None, imtParams=False):
        """
        applies the specified stylesheets
        (with the specified transformation parameters)
        on the given string
        """
        import Ft.Xml.Xslt.Processor
        from Ft.Server.Common import ResourceTypes, Schema, XmlLib
        from Ft.Server.Server import FtServerServerException, Error
        from Ft.Xml.InputSource import InputSourceFactory

        extension_modules = ['Ft.Server.Server.Xslt']
        if extModules:
            extension_modules.extend(extModules)

        #First map the stylesheets
        if not isinstance(stylesheets, (list, tuple)):
            stylesheets = (stylesheets,)

        p = Ft.Xml.Xslt.Processor.Processor()
        p.extensionParams = extParams or {}
        p._repository = self.repo
        p.registerExtensionModules(extension_modules)
        p.setDocumentReader(FtssInputSource.NonvalidatingReader)
        for s in stylesheets:
            if isinstance(s,tuple):
                s=s[0]
            if isinstance(s, StringTypes):
                p.appendStylesheet(p.inputSourceFactory.fromUri(s))
            elif hasattr(s,'toStylesheet'):
                #A document reference
                p.appendStylesheetInstance(s.toStylesheet(self.repo.getRoot()))
            elif hasattr(s,'asStylesheet'):
                #A reference to a document in the system
                p.appendStylesheetInstance(s.asStylesheet())

        p.inputSourceFactory = FtssInputSource._FtssInputSourceFactory(
            self.repo._driver)
        rt = p.run(p.inputSourceFactory.fromString(self.body,'ftss:///'),
                   ignorePis=ignorePis, topLevelParams=params)
        if extParams is not None and hasattr(p, 'extensionParams'):
            extParams.update(p.extensionParams)

        imt = p._lastOutputParams.mediaType
        if imtParams:
            enc = p._lastOutputParams.encoding
            if enc:
                imt = '%s; charset=%s' % (imt, enc)
        return rt, imt


