
##############################################################################
#
# Copyright (c) 2003-2016 by The University of Queensland
# http://www.uq.edu.au
#
# Primary Business: Queensland, Australia
# Licensed under the Apache License, version 2.0
# http://www.apache.org/licenses/LICENSE-2.0
#
# Development until 2012 by Earth Systems Science Computational Center (ESSCC)
# Development 2012-2013 by School of Earth Sciences
# Development from 2014 by Centre for Geoscience Computing (GeoComp)
#
##############################################################################

from __future__ import print_function, division

__copyright__="""Copyright (c) 2003-2016 by The University of Queensland
http://www.uq.edu.au
Primary Business: Queensland, Australia"""
__license__="""Licensed under the Apache License, version 2.0
http://www.apache.org/licenses/LICENSE-2.0"""
__url__="https://launchpad.net/escript-finley"

"""
basic tests for functions in util.py effecting the spatial distribution 

it is assumed that the domain is the usint square/cube 

not all these test will run for all domains. check the doc string for the assumptions of a particular test

:var __author__: name of author
:var __copyright__: copyrights
:var __license__: licence agreement
:var __url__: url entry point on documentation
:var __version__: version
:var __date__: date of the version
"""
__author__="Lutz Gross, l.gross@uq.edu.au"

import esys.escriptcore.utestselect as unittest
from esys.escript import *
from numpy import array


class Test_Util_SpatialFunctions_noGradOnBoundary_noContact(unittest.TestCase):
   RES_TOL=1.e-8
   def test_x_ofDomain(self):
     """
     test getX() of the domain to be in the [0,1]^dim box
     """
     dim=self.domain.getDim()
     x=self.domain.getX()
     self.assertEqual(x.getShape(),(dim,),"wrong shape of result.")
     self.assertAlmostEqual(inf(x[0]),0.,int(-log10(self.RES_TOL)),"min x0 wrong")
     self.assertAlmostEqual(sup(x[0]),1.,int(-log10(self.RES_TOL)),"max x0 wrong")
     self.assertAlmostEqual(inf(x[1]),0.,int(-log10(self.RES_TOL)),"min x1 wrong")
     self.assertAlmostEqual(sup(x[1]),1.,int(-log10(self.RES_TOL)),"max x1 wrong")
     if dim>2:
       self.assertAlmostEqual(inf(x[2]),0.,int(-log10(self.RES_TOL)),"min x2 wrong")
       self.assertAlmostEqual(sup(x[2]),1.,int(-log10(self.RES_TOL)),"max x2 wrong")

   def test_SolutionOrder(self):
      """
      test the approximation order 
      """
      self.assertTrue(self.order == Solution(self.domain).getApproximationOrder(), "wrong order (Solution)")
      self.assertTrue(self.order == ContinuousFunction(self.domain).getApproximationOrder(), "wrong order (continuous function)")
      self.assertTrue(1 == ReducedSolution(self.domain).getApproximationOrder(), "wrong order (ReducedSolution)")
      self.assertTrue(1 == ReducedContinuousFunction(self.domain).getApproximationOrder(), "wrong order (Reduced continuous function)")
      for i in range(self.domain.getDim()):
         for k in range(Function(self.domain).getApproximationOrder()+1):
             self.assertAlmostEqual(integrate(Function(self.domain).getX()[i]**k),1./(k+1),8,"wrong integral (i=%s, order = %s)"%(i,k))
         for k in range(ReducedFunction(self.domain).getApproximationOrder()+1):
             self.assertAlmostEqual(integrate(ReducedFunction(self.domain).getX()[i]**k),1./(k+1),8,"wrong integral (i=%s, order = %s (reduced))"%(i,k))


   def test_normal_FunctionOnBoundary(self):
     """
     test getNormal() on boundary

     assumptions: FunctionOnBoundary(self.domain) exists
     """
     dim=self.domain.getDim()
     f=FunctionOnBoundary(self.domain)
     x=f.getX()
     ref=Vector(0.,what=f)
     if dim==3:
         ref.setTaggedValue(200,[0,0,1])
         ref.setTaggedValue(100,[0,0,-1])
         ref.setTaggedValue(20,[0,1,0])
         ref.setTaggedValue(10,[0,-1,0])
         ref.setTaggedValue(2,[1,0,0])
         ref.setTaggedValue(1,[-1,0,0])
     else:
         ref.setTaggedValue(2,[1,0])
         ref.setTaggedValue(1,[-1,0])
         ref.setTaggedValue(20, [0,1])
         ref.setTaggedValue(10, [0,-1])

     res=f.getNormal()
     self.assertEqual(res.getShape(),(dim,),"wrong shape of result.")
     self.assertEqual(res.getFunctionSpace(),f,"wrong functionspace of result.")
     self.assertTrue(Lsup(ref-res)<=self.RES_TOL,"wrong result")

   def test_normal_ReducedFunctionOnBoundary(self):
     """
     test getNormal() on boundary

     assumptions: FunctionOnBoundary(self.domain) exists
     """
     dim=self.domain.getDim()
     f=ReducedFunctionOnBoundary(self.domain)
     x=f.getX()
     ref=Vector(0.,what=f)
     if dim==3:
         ref.setTaggedValue(200,[0,0,1])
         ref.setTaggedValue(100,[0,0,-1])
         ref.setTaggedValue(20,[0,1,0])
         ref.setTaggedValue(10,[0,-1,0])
         ref.setTaggedValue(2,[1,0,0])
         ref.setTaggedValue(1,[-1,0,0])
     else:
         ref.setTaggedValue(2,[1,0])
         ref.setTaggedValue(1,[-1,0])
         ref.setTaggedValue(20, [0,1])
         ref.setTaggedValue(10, [0,-1])

     res=f.getNormal()
     self.assertEqual(res.getShape(),(dim,),"wrong shape of result.")
     self.assertEqual(res.getFunctionSpace(),f,"wrong functionspace of result.")
     self.assertTrue(Lsup(ref-res)<=self.RES_TOL,"wrong result")

   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunction_fromData_ContinuousFunction_rank0(self):
      """
      tests integral of rank 0 Data on the Function

      assumptions: ContinuousFunction(self.domain) exists
                   self.domain supports integral on Function
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=Function(self.domain)
      w=ContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(),w)
      if dim==2:
        arg=(0.465976690369)*x[0]**o+(-0.25136070196)*x[0]+(0.409576841566)*x[1]**o+(0.726707772754)*x[1]
        ref=(0.875553531935)/(o+1.)+(0.237673535397)
      else:
        arg=(0.567785895928)*x[0]**o+(0.928477128396)*x[0]+(-0.625720943088)*x[1]**o+(-0.681460841298)*x[1]+(-0.866801010069)*x[2]**o+(0.159594300637)*x[2]
        ref=(-0.924736057229)/(o+1.)+(0.203305293867)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunction_fromData_ContinuousFunction_rank1(self):
      """
      tests integral of rank 1 Data on the Function

      assumptions: ContinuousFunction(self.domain) exists
                   self.domain supports integral on Function
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=Function(self.domain)
      w=ContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(2,),w)
      ref=numpy.zeros((2,),numpy.float_)
      if dim==2:
        arg[0]=(-0.839020088668)*x[0]**o+(-0.880288035177)*x[0]+(0.260951734364)*x[1]**o+(-0.624208000982)*x[1]
        arg[1]=(0.985421831314)*x[0]**o+(0.660399229451)*x[0]+(-0.514003848134)*x[1]**o+(0.0312162082593)*x[1]
        ref[0]=(-0.578068354304)/(o+1.)+(-0.75224801808)
        ref[1]=(0.47141798318)/(o+1.)+(0.345807718855)
      else:
        arg[0]=(-0.627307316929)*x[0]**o+(0.500432411035)*x[0]+(0.233895908313)*x[1]**o+(-0.125048135432)*x[1]+(0.0695114009632)*x[2]**o+(0.995170977074)*x[2]
        arg[1]=(0.760033715785)*x[0]**o+(0.48895905139)*x[0]+(0.240372023176)*x[1]**o+(0.33946689896)*x[1]+(-0.774638489419)*x[2]**o+(-0.0777240871074)*x[2]
        ref[0]=(-0.323900007653)/(o+1.)+(0.685277626338)
        ref[1]=(0.225767249542)/(o+1.)+(0.375350931622)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunction_fromData_ContinuousFunction_rank2(self):
      """
      tests integral of rank 2 Data on the Function

      assumptions: ContinuousFunction(self.domain) exists
                   self.domain supports integral on Function
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=Function(self.domain)
      w=ContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5),w)
      ref=numpy.zeros((4, 5),numpy.float_)
      if dim==2:
        arg[0,0]=(-0.844492048883)*x[0]**o+(0.727132679659)*x[0]+(-0.862576284722)*x[1]**o+(0.502937561417)*x[1]
        arg[0,1]=(-0.587157419309)*x[0]**o+(0.00327068097107)*x[0]+(0.422592349683)*x[1]**o+(0.167479615369)*x[1]
        arg[0,2]=(0.771156665672)*x[0]**o+(-0.371091138833)*x[0]+(-0.867750378658)*x[1]**o+(-0.744394290185)*x[1]
        arg[0,3]=(0.669718598855)*x[0]**o+(-0.0825909409121)*x[0]+(0.868872948791)*x[1]**o+(0.141891982204)*x[1]
        arg[0,4]=(-0.441072764653)*x[0]**o+(-0.929654649265)*x[0]+(-0.35883498816)*x[1]**o+(-0.84045480381)*x[1]
        arg[1,0]=(0.555668196622)*x[0]**o+(0.94306681043)*x[0]+(0.586512446245)*x[1]**o+(0.702007276327)*x[1]
        arg[1,1]=(0.792083771005)*x[0]**o+(-0.865830139617)*x[0]+(0.0929971631615)*x[1]**o+(0.620446289121)*x[1]
        arg[1,2]=(-0.201978375472)*x[0]**o+(-0.0777206885918)*x[0]+(0.798755459687)*x[1]**o+(-0.278086468846)*x[1]
        arg[1,3]=(0.063028419572)*x[0]**o+(-0.454786232294)*x[0]+(-0.0481393324293)*x[1]**o+(-0.612657132834)*x[1]
        arg[1,4]=(-0.743283787964)*x[0]**o+(0.854827303846)*x[0]+(0.94241635825)*x[1]**o+(0.436937526521)*x[1]
        arg[2,0]=(-0.814975695669)*x[0]**o+(0.579477783769)*x[0]+(0.00998264005965)*x[1]**o+(-0.913665878657)*x[1]
        arg[2,1]=(-0.526833976369)*x[0]**o+(-0.800386086232)*x[0]+(-0.487600053086)*x[1]**o+(0.00242562519003)*x[1]
        arg[2,2]=(-0.964119591874)*x[0]**o+(0.203212394214)*x[0]+(-0.96850837954)*x[1]**o+(-0.214886132512)*x[1]
        arg[2,3]=(-0.356249104564)*x[0]**o+(0.329769293221)*x[0]+(0.936664928754)*x[1]**o+(0.520603503283)*x[1]
        arg[2,4]=(0.255464219424)*x[0]**o+(-0.157783294516)*x[0]+(0.519451787865)*x[1]**o+(0.00960521397376)*x[1]
        arg[3,0]=(-0.117250486034)*x[0]**o+(0.192945431273)*x[0]+(-0.7580154297)*x[1]**o+(-0.308801650294)*x[1]
        arg[3,1]=(0.243931002845)*x[0]**o+(-0.626714191745)*x[0]+(-0.365870571265)*x[1]**o+(0.652257582021)*x[1]
        arg[3,2]=(-0.769756656122)*x[0]**o+(0.711106390017)*x[0]+(0.150749630653)*x[1]**o+(-0.0441309732448)*x[1]
        arg[3,3]=(0.92677837493)*x[0]**o+(0.507199859313)*x[0]+(-0.542275490828)*x[1]**o+(-0.0907270087228)*x[1]
        arg[3,4]=(-0.0306175885885)*x[0]**o+(-0.434666505018)*x[0]+(-0.815633427562)*x[1]**o+(-0.916424797903)*x[1]
        ref[0,0]=(-1.70706833361)/(o+1.)+(0.615035120538)
        ref[0,1]=(-0.164565069626)/(o+1.)+(0.08537514817)
        ref[0,2]=(-0.0965937129851)/(o+1.)+(-0.557742714509)
        ref[0,3]=(1.53859154765)/(o+1.)+(0.0296505206459)
        ref[0,4]=(-0.799907752814)/(o+1.)+(-0.885054726537)
        ref[1,0]=(1.14218064287)/(o+1.)+(0.822537043379)
        ref[1,1]=(0.885080934167)/(o+1.)+(-0.122691925248)
        ref[1,2]=(0.596777084215)/(o+1.)+(-0.177903578719)
        ref[1,3]=(0.0148890871427)/(o+1.)+(-0.533721682564)
        ref[1,4]=(0.199132570285)/(o+1.)+(0.645882415184)
        ref[2,0]=(-0.80499305561)/(o+1.)+(-0.167094047444)
        ref[2,1]=(-1.01443402945)/(o+1.)+(-0.398980230521)
        ref[2,2]=(-1.93262797141)/(o+1.)+(-0.00583686914904)
        ref[2,3]=(0.580415824189)/(o+1.)+(0.425186398252)
        ref[2,4]=(0.774916007289)/(o+1.)+(-0.074089040271)
        ref[3,0]=(-0.875265915734)/(o+1.)+(-0.0579281095106)
        ref[3,1]=(-0.12193956842)/(o+1.)+(0.0127716951382)
        ref[3,2]=(-0.619007025469)/(o+1.)+(0.333487708386)
        ref[3,3]=(0.384502884102)/(o+1.)+(0.208236425295)
        ref[3,4]=(-0.846251016151)/(o+1.)+(-0.675545651461)
      else:
        arg[0,0]=(0.596538443259)*x[0]**o+(0.479462980373)*x[0]+(-0.334469126403)*x[1]**o+(0.784201085556)*x[1]+(-0.231474140531)*x[2]**o+(-0.368096345589)*x[2]
        arg[0,1]=(0.188945383338)*x[0]**o+(-0.95033024122)*x[0]+(0.567904901047)*x[1]**o+(-0.76703296351)*x[1]+(-0.0116083690376)*x[2]**o+(-0.211176589764)*x[2]
        arg[0,2]=(0.225412552449)*x[0]**o+(-0.731615155924)*x[0]+(-0.362727501637)*x[1]**o+(-0.195442119344)*x[1]+(-0.215444456285)*x[2]**o+(0.412715796392)*x[2]
        arg[0,3]=(-0.00941287481408)*x[0]**o+(-0.207271678463)*x[0]+(0.66582433583)*x[1]**o+(-0.297871484308)*x[1]+(-0.819460465536)*x[2]**o+(-0.3316307033)*x[2]
        arg[0,4]=(-0.172265683581)*x[0]**o+(0.881658164973)*x[0]+(0.0267816673708)*x[1]**o+(-0.526757427521)*x[1]+(-0.666953449252)*x[2]**o+(0.366533690002)*x[2]
        arg[1,0]=(-0.575272180021)*x[0]**o+(-0.707236339163)*x[0]+(-0.734723292219)*x[1]**o+(-0.872019568084)*x[1]+(-0.513269190461)*x[2]**o+(-0.95077853535)*x[2]
        arg[1,1]=(0.687652562435)*x[0]**o+(0.500668143461)*x[0]+(0.127678854367)*x[1]**o+(0.135196530391)*x[1]+(0.219757431874)*x[2]**o+(0.43373842168)*x[2]
        arg[1,2]=(-0.674120744855)*x[0]**o+(-0.792769889708)*x[0]+(0.0414602844391)*x[1]**o+(-0.159887164563)*x[1]+(0.406441555189)*x[2]**o+(-0.662210077269)*x[2]
        arg[1,3]=(-0.623120612358)*x[0]**o+(0.895951757448)*x[0]+(0.957407224559)*x[1]**o+(-0.472225118848)*x[1]+(-0.294551998847)*x[2]**o+(0.503584320886)*x[2]
        arg[1,4]=(-0.304220645545)*x[0]**o+(0.812321371589)*x[0]+(0.932257196386)*x[1]**o+(0.736014913915)*x[1]+(0.119140391025)*x[2]**o+(0.910696940879)*x[2]
        arg[2,0]=(0.275470013797)*x[0]**o+(-0.903754203548)*x[0]+(-0.443211232771)*x[1]**o+(0.78815938507)*x[1]+(-0.0802934106722)*x[2]**o+(0.096532189886)*x[2]
        arg[2,1]=(-0.4505392488)*x[0]**o+(-0.668542586506)*x[0]+(0.487967999984)*x[1]**o+(-0.642310745404)*x[1]+(-0.956530771836)*x[2]**o+(0.517544404873)*x[2]
        arg[2,2]=(-0.984032885692)*x[0]**o+(-0.606688527142)*x[0]+(0.0565483776156)*x[1]**o+(-0.945228520749)*x[1]+(0.433070187921)*x[2]**o+(-0.599983370996)*x[2]
        arg[2,3]=(-0.673469008741)*x[0]**o+(0.612144012035)*x[0]+(-0.169497339443)*x[1]**o+(0.267990057152)*x[1]+(0.431716413548)*x[2]**o+(0.188904436012)*x[2]
        arg[2,4]=(0.320782309844)*x[0]**o+(-0.830068082984)*x[0]+(0.900183436056)*x[1]**o+(-0.343951186189)*x[1]+(-0.00137135137854)*x[2]**o+(-0.655764522249)*x[2]
        arg[3,0]=(-0.936193981767)*x[0]**o+(0.885260284217)*x[0]+(0.00397725971098)*x[1]**o+(-0.0864751234501)*x[1]+(0.25001166345)*x[2]**o+(-0.127859505667)*x[2]
        arg[3,1]=(-0.0841986843231)*x[0]**o+(-0.95678469673)*x[0]+(0.867049114883)*x[1]**o+(0.154499266839)*x[1]+(0.0217146474622)*x[2]**o+(-0.945774319527)*x[2]
        arg[3,2]=(-0.668161122306)*x[0]**o+(0.779140555525)*x[0]+(0.217613826157)*x[1]**o+(0.900077039607)*x[1]+(-0.251881965789)*x[2]**o+(-0.331281723354)*x[2]
        arg[3,3]=(-0.95624079226)*x[0]**o+(-0.403226291225)*x[0]+(0.303340462877)*x[1]**o+(-0.269819286355)*x[1]+(0.605804508829)*x[2]**o+(-0.461404715818)*x[2]
        arg[3,4]=(-0.933059329973)*x[0]**o+(-0.00851461368762)*x[0]+(0.0588544443918)*x[1]**o+(-0.682939702389)*x[1]+(-0.213965742261)*x[2]**o+(-0.102965977964)*x[2]
        ref[0,0]=(0.0305951763245)/(o+1.)+(0.44778386017)
        ref[0,1]=(0.745241915348)/(o+1.)+(-0.964269897247)
        ref[0,2]=(-0.352759405473)/(o+1.)+(-0.257170739438)
        ref[0,3]=(-0.163049004521)/(o+1.)+(-0.418386933036)
        ref[0,4]=(-0.812437465462)/(o+1.)+(0.360717213727)
        ref[1,0]=(-1.8232646627)/(o+1.)+(-1.2650172213)
        ref[1,1]=(1.03508884868)/(o+1.)+(0.534801547766)
        ref[1,2]=(-0.226218905227)/(o+1.)+(-0.80743356577)
        ref[1,3]=(0.0397346133545)/(o+1.)+(0.463655479743)
        ref[1,4]=(0.747176941866)/(o+1.)+(1.22951661319)
        ref[2,0]=(-0.248034629647)/(o+1.)+(-0.00953131429598)
        ref[2,1]=(-0.919102020652)/(o+1.)+(-0.396654463518)
        ref[2,2]=(-0.494414320156)/(o+1.)+(-1.07595020944)
        ref[2,3]=(-0.411249934637)/(o+1.)+(0.534519252599)
        ref[2,4]=(1.21959439452)/(o+1.)+(-0.914891895711)
        ref[3,0]=(-0.682205058606)/(o+1.)+(0.33546282755)
        ref[3,1]=(0.804565078022)/(o+1.)+(-0.874029874709)
        ref[3,2]=(-0.702429261938)/(o+1.)+(0.673967935889)
        ref[3,3]=(-0.0470958205546)/(o+1.)+(-0.567225146699)
        ref[3,4]=(-1.08817062784)/(o+1.)+(-0.39721014702)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunction_fromData_ContinuousFunction_rank3(self):
      """
      tests integral of rank 3 Data on the Function

      assumptions: ContinuousFunction(self.domain) exists
                   self.domain supports integral on Function
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=Function(self.domain)
      w=ContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(6, 2, 2),w)
      ref=numpy.zeros((6, 2, 2),numpy.float_)
      if dim==2:
        arg[0,0,0]=(0.401537399178)*x[0]**o+(-0.868937065568)*x[0]+(0.356734884346)*x[1]**o+(0.0137485089422)*x[1]
        arg[0,0,1]=(0.705383698155)*x[0]**o+(-0.884988282352)*x[0]+(-0.979185994641)*x[1]**o+(0.929576942519)*x[1]
        arg[0,1,0]=(-0.292540086272)*x[0]**o+(-0.223230840915)*x[0]+(0.717291109855)*x[1]**o+(-0.826233591275)*x[1]
        arg[0,1,1]=(0.55739919643)*x[0]**o+(0.2571407205)*x[0]+(0.00434404142287)*x[1]**o+(0.200305144451)*x[1]
        arg[1,0,0]=(0.451841947443)*x[0]**o+(-0.576279050146)*x[0]+(-0.128142118055)*x[1]**o+(0.316406960557)*x[1]
        arg[1,0,1]=(-0.07174792925)*x[0]**o+(-0.392935435009)*x[0]+(-0.343535485172)*x[1]**o+(0.619131193274)*x[1]
        arg[1,1,0]=(-0.667031669611)*x[0]**o+(-0.184806094874)*x[0]+(-0.199375296384)*x[1]**o+(0.449351889815)*x[1]
        arg[1,1,1]=(-0.51924508404)*x[0]**o+(0.870508122509)*x[0]+(-0.274149433148)*x[1]**o+(-0.762367215962)*x[1]
        arg[2,0,0]=(-0.501546909709)*x[0]**o+(0.164122129343)*x[0]+(-0.601370828214)*x[1]**o+(0.888046948888)*x[1]
        arg[2,0,1]=(0.272472486899)*x[0]**o+(-0.958776296738)*x[0]+(-0.334676045163)*x[1]**o+(0.410943596779)*x[1]
        arg[2,1,0]=(0.363478201368)*x[0]**o+(0.753492598803)*x[0]+(-0.663726768274)*x[1]**o+(0.875443816963)*x[1]
        arg[2,1,1]=(0.995788536211)*x[0]**o+(-0.123932097555)*x[0]+(0.143025171293)*x[1]**o+(0.724918059214)*x[1]
        arg[3,0,0]=(-0.578717829755)*x[0]**o+(-0.899523186929)*x[0]+(0.58523203854)*x[1]**o+(0.751620714771)*x[1]
        arg[3,0,1]=(0.179841033744)*x[0]**o+(0.268015839688)*x[0]+(0.860292991989)*x[1]**o+(0.0117969420991)*x[1]
        arg[3,1,0]=(-0.575777000777)*x[0]**o+(0.170047012345)*x[0]+(-0.418743243278)*x[1]**o+(-0.278415849566)*x[1]
        arg[3,1,1]=(-0.913076795648)*x[0]**o+(0.486330450107)*x[0]+(-0.423570363453)*x[1]**o+(0.388575474946)*x[1]
        arg[4,0,0]=(0.16695368621)*x[0]**o+(0.133497460585)*x[0]+(-0.444434659151)*x[1]**o+(-0.895229636275)*x[1]
        arg[4,0,1]=(0.432018868106)*x[0]**o+(-0.281976075625)*x[0]+(-0.0455135209769)*x[1]**o+(0.76870037059)*x[1]
        arg[4,1,0]=(0.238542836112)*x[0]**o+(-0.806931884409)*x[0]+(-0.0365533871189)*x[1]**o+(0.731336634477)*x[1]
        arg[4,1,1]=(0.512035255059)*x[0]**o+(-0.683560978902)*x[0]+(-0.953782277737)*x[1]**o+(-0.311664210765)*x[1]
        arg[5,0,0]=(-0.842399299624)*x[0]**o+(0.460676816656)*x[0]+(-0.251781603841)*x[1]**o+(0.321792359074)*x[1]
        arg[5,0,1]=(-0.900576257062)*x[0]**o+(0.842393710528)*x[0]+(0.630445446843)*x[1]**o+(-0.647745144739)*x[1]
        arg[5,1,0]=(0.136687908652)*x[0]**o+(0.0252907890641)*x[0]+(0.813054611844)*x[1]**o+(-0.459930787056)*x[1]
        arg[5,1,1]=(-0.00242518550941)*x[0]**o+(-0.558785348137)*x[0]+(0.0568659980522)*x[1]**o+(0.689293614051)*x[1]
        ref[0,0,0]=(0.758272283524)/(o+1.)+(-0.427594278313)
        ref[0,0,1]=(-0.273802296487)/(o+1.)+(0.0222943300833)
        ref[0,1,0]=(0.424751023583)/(o+1.)+(-0.524732216095)
        ref[0,1,1]=(0.561743237852)/(o+1.)+(0.228722932476)
        ref[1,0,0]=(0.323699829388)/(o+1.)+(-0.129936044795)
        ref[1,0,1]=(-0.415283414421)/(o+1.)+(0.113097879133)
        ref[1,1,0]=(-0.866406965995)/(o+1.)+(0.132272897471)
        ref[1,1,1]=(-0.793394517189)/(o+1.)+(0.0540704532731)
        ref[2,0,0]=(-1.10291773792)/(o+1.)+(0.526084539115)
        ref[2,0,1]=(-0.0622035582637)/(o+1.)+(-0.273916349979)
        ref[2,1,0]=(-0.300248566906)/(o+1.)+(0.814468207883)
        ref[2,1,1]=(1.1388137075)/(o+1.)+(0.300492980829)
        ref[3,0,0]=(0.00651420878498)/(o+1.)+(-0.073951236079)
        ref[3,0,1]=(1.04013402573)/(o+1.)+(0.139906390894)
        ref[3,1,0]=(-0.994520244055)/(o+1.)+(-0.0541844186105)
        ref[3,1,1]=(-1.3366471591)/(o+1.)+(0.437452962527)
        ref[4,0,0]=(-0.27748097294)/(o+1.)+(-0.380866087845)
        ref[4,0,1]=(0.386505347129)/(o+1.)+(0.243362147483)
        ref[4,1,0]=(0.201989448993)/(o+1.)+(-0.0377976249661)
        ref[4,1,1]=(-0.441747022678)/(o+1.)+(-0.497612594833)
        ref[5,0,0]=(-1.09418090346)/(o+1.)+(0.391234587865)
        ref[5,0,1]=(-0.270130810219)/(o+1.)+(0.0973242828949)
        ref[5,1,0]=(0.949742520497)/(o+1.)+(-0.217319998996)
        ref[5,1,1]=(0.0544408125428)/(o+1.)+(0.0652541329568)
      else:
        arg[0,0,0]=(0.411788818613)*x[0]**o+(0.324151698669)*x[0]+(0.656551349616)*x[1]**o+(0.801577401197)*x[1]+(0.690440431022)*x[2]**o+(0.776409870151)*x[2]
        arg[0,0,1]=(-0.224985947659)*x[0]**o+(-0.564703086188)*x[0]+(-0.37196336116)*x[1]**o+(-0.114979853172)*x[1]+(0.274462080499)*x[2]**o+(-0.988489021983)*x[2]
        arg[0,1,0]=(0.512331530009)*x[0]**o+(0.519617843594)*x[0]+(-0.515174861158)*x[1]**o+(-0.675953902994)*x[1]+(0.796111490716)*x[2]**o+(0.229569489858)*x[2]
        arg[0,1,1]=(0.347765144376)*x[0]**o+(-0.0110194730031)*x[0]+(-0.593936645599)*x[1]**o+(-0.372786558089)*x[1]+(-0.604691321362)*x[2]**o+(-0.888002521077)*x[2]
        arg[1,0,0]=(-0.575954903433)*x[0]**o+(0.0160235655616)*x[0]+(0.305335632724)*x[1]**o+(-0.259402553796)*x[1]+(-0.221276854741)*x[2]**o+(-0.519932341069)*x[2]
        arg[1,0,1]=(-0.674987647238)*x[0]**o+(-0.45852667934)*x[0]+(-0.758937440978)*x[1]**o+(0.80566898197)*x[1]+(-0.961619154322)*x[2]**o+(0.861564430243)*x[2]
        arg[1,1,0]=(-0.297475412437)*x[0]**o+(0.881828705604)*x[0]+(-0.823267615874)*x[1]**o+(-0.0232728101313)*x[1]+(-0.676220001453)*x[2]**o+(-0.382414036981)*x[2]
        arg[1,1,1]=(-0.31008204616)*x[0]**o+(0.805195900401)*x[0]+(-0.758209772143)*x[1]**o+(0.861379709073)*x[1]+(0.0631063866518)*x[2]**o+(0.356607080553)*x[2]
        arg[2,0,0]=(0.305214964591)*x[0]**o+(0.815106526139)*x[0]+(0.138797156005)*x[1]**o+(-0.241921896332)*x[1]+(-0.103783035058)*x[2]**o+(-0.161952044984)*x[2]
        arg[2,0,1]=(-0.925375677635)*x[0]**o+(0.0341347568201)*x[0]+(-0.706905924964)*x[1]**o+(-0.193304368451)*x[1]+(-0.449866757104)*x[2]**o+(-0.894963680869)*x[2]
        arg[2,1,0]=(0.822676224279)*x[0]**o+(-0.28758006682)*x[0]+(-0.869025918917)*x[1]**o+(0.0107602044744)*x[1]+(-0.832322765163)*x[2]**o+(0.748440319389)*x[2]
        arg[2,1,1]=(-0.415326508997)*x[0]**o+(-0.879521424817)*x[0]+(-0.971781441062)*x[1]**o+(0.527586634108)*x[1]+(0.336682669856)*x[2]**o+(-0.948414041149)*x[2]
        arg[3,0,0]=(-0.345788531715)*x[0]**o+(0.626583498807)*x[0]+(-0.186418683649)*x[1]**o+(0.3141628424)*x[1]+(-0.781119787783)*x[2]**o+(0.972708430266)*x[2]
        arg[3,0,1]=(0.175536279095)*x[0]**o+(-0.150027774295)*x[0]+(-0.506653680683)*x[1]**o+(-0.482544397029)*x[1]+(-0.59090996152)*x[2]**o+(0.618123625026)*x[2]
        arg[3,1,0]=(0.737984493124)*x[0]**o+(0.650901680297)*x[0]+(-0.361389200009)*x[1]**o+(-0.784304665396)*x[1]+(-0.954132232142)*x[2]**o+(0.684341934348)*x[2]
        arg[3,1,1]=(-0.385757579645)*x[0]**o+(0.251136919845)*x[0]+(0.95143939933)*x[1]**o+(0.501860043225)*x[1]+(-0.675673011184)*x[2]**o+(0.337315450579)*x[2]
        arg[4,0,0]=(-0.740183402139)*x[0]**o+(-0.828278760743)*x[0]+(0.88000889117)*x[1]**o+(-0.0226801022184)*x[1]+(0.6811397662)*x[2]**o+(-0.254627446498)*x[2]
        arg[4,0,1]=(-0.194128952361)*x[0]**o+(-0.524854896388)*x[0]+(0.341100326577)*x[1]**o+(-0.26072443499)*x[1]+(-0.197396331776)*x[2]**o+(-0.589037632112)*x[2]
        arg[4,1,0]=(0.261995024865)*x[0]**o+(0.214193793699)*x[0]+(0.572024600244)*x[1]**o+(0.230659327198)*x[1]+(0.420971705453)*x[2]**o+(0.333445049827)*x[2]
        arg[4,1,1]=(-0.0679525054395)*x[0]**o+(0.0756646296473)*x[0]+(-0.299865836239)*x[1]**o+(-0.828294764914)*x[1]+(0.606170745631)*x[2]**o+(0.624267827393)*x[2]
        arg[5,0,0]=(0.312565542211)*x[0]**o+(-0.368815496171)*x[0]+(-0.0615973473105)*x[1]**o+(0.205012835971)*x[1]+(0.310818553076)*x[2]**o+(-0.287393908304)*x[2]
        arg[5,0,1]=(-0.409626309069)*x[0]**o+(0.977953573562)*x[0]+(0.583013544138)*x[1]**o+(0.542907128677)*x[1]+(0.488538336366)*x[2]**o+(-0.85823611123)*x[2]
        arg[5,1,0]=(-0.899607109395)*x[0]**o+(-0.0459053040522)*x[0]+(0.938423293041)*x[1]**o+(-0.112431665317)*x[1]+(0.620609328687)*x[2]**o+(-0.663082429319)*x[2]
        arg[5,1,1]=(0.00370253144626)*x[0]**o+(0.503733497956)*x[0]+(0.362457785043)*x[1]**o+(0.417891957303)*x[1]+(0.404543677829)*x[2]**o+(0.117828669122)*x[2]
        ref[0,0,0]=(1.75878059925)/(o+1.)+(0.951069485009)
        ref[0,0,1]=(-0.32248722832)/(o+1.)+(-0.834085980672)
        ref[0,1,0]=(0.793268159567)/(o+1.)+(0.036616715229)
        ref[0,1,1]=(-0.850862822585)/(o+1.)+(-0.635904276085)
        ref[1,0,0]=(-0.49189612545)/(o+1.)+(-0.381655664652)
        ref[1,0,1]=(-2.39554424254)/(o+1.)+(0.604353366437)
        ref[1,1,0]=(-1.79696302976)/(o+1.)+(0.238070929246)
        ref[1,1,1]=(-1.00518543165)/(o+1.)+(1.01159134501)
        ref[2,0,0]=(0.340229085537)/(o+1.)+(0.205616292412)
        ref[2,0,1]=(-2.0821483597)/(o+1.)+(-0.52706664625)
        ref[2,1,0]=(-0.878672459802)/(o+1.)+(0.235810228522)
        ref[2,1,1]=(-1.0504252802)/(o+1.)+(-0.650174415929)
        ref[3,0,0]=(-1.31332700315)/(o+1.)+(0.956727385736)
        ref[3,0,1]=(-0.922027363109)/(o+1.)+(-0.00722427314892)
        ref[3,1,0]=(-0.577536939027)/(o+1.)+(0.275469474624)
        ref[3,1,1]=(-0.109991191499)/(o+1.)+(0.545156206825)
        ref[4,0,0]=(0.820965255231)/(o+1.)+(-0.55279315473)
        ref[4,0,1]=(-0.0504249575602)/(o+1.)+(-0.687308481745)
        ref[4,1,0]=(1.25499133056)/(o+1.)+(0.389149085362)
        ref[4,1,1]=(0.238352403952)/(o+1.)+(-0.0641811539369)
        ref[5,0,0]=(0.561786747976)/(o+1.)+(-0.225598284252)
        ref[5,0,1]=(0.661925571434)/(o+1.)+(0.331312295504)
        ref[5,1,0]=(0.659425512333)/(o+1.)+(-0.410709699344)
        ref[5,1,1]=(0.770703994319)/(o+1.)+(0.519727062191)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunction_fromData_ContinuousFunction_rank4(self):
      """
      tests integral of rank 4 Data on the Function

      assumptions: ContinuousFunction(self.domain) exists
                   self.domain supports integral on Function
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=Function(self.domain)
      w=ContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5, 3, 2),w)
      ref=numpy.zeros((4, 5, 3, 2),numpy.float_)
      if dim==2:
        arg[0,0,0,0]=(-0.164405890316)*x[0]**o+(-0.323303939764)*x[0]+(-0.0516829984856)*x[1]**o+(-0.220765052533)*x[1]
        arg[0,0,0,1]=(0.152172612108)*x[0]**o+(-0.166242164365)*x[0]+(0.807177808571)*x[1]**o+(0.0404817109573)*x[1]
        arg[0,0,1,0]=(0.016912808567)*x[0]**o+(0.623923316774)*x[0]+(-0.542886934905)*x[1]**o+(0.0110784560205)*x[1]
        arg[0,0,1,1]=(-0.663885786909)*x[0]**o+(-0.55563355485)*x[0]+(0.191993284382)*x[1]**o+(0.734973556043)*x[1]
        arg[0,0,2,0]=(-0.110316754183)*x[0]**o+(-0.584022783672)*x[0]+(0.56471376011)*x[1]**o+(0.846317747143)*x[1]
        arg[0,0,2,1]=(-0.767988009278)*x[0]**o+(0.440997912352)*x[0]+(0.45366348093)*x[1]**o+(0.376620990318)*x[1]
        arg[0,1,0,0]=(-0.138006508051)*x[0]**o+(0.8450564766)*x[0]+(-0.273559897633)*x[1]**o+(0.683678615645)*x[1]
        arg[0,1,0,1]=(-0.467521625474)*x[0]**o+(0.32289306962)*x[0]+(0.340974562152)*x[1]**o+(-0.58109391053)*x[1]
        arg[0,1,1,0]=(0.76867991208)*x[0]**o+(0.683527735131)*x[0]+(0.762986872389)*x[1]**o+(-0.189342734286)*x[1]
        arg[0,1,1,1]=(0.125150175351)*x[0]**o+(-0.175190872876)*x[0]+(-0.569766488076)*x[1]**o+(0.787503755206)*x[1]
        arg[0,1,2,0]=(-0.339695860559)*x[0]**o+(0.119452596206)*x[0]+(0.474378864192)*x[1]**o+(0.845982555265)*x[1]
        arg[0,1,2,1]=(0.820551334836)*x[0]**o+(-0.200024721285)*x[0]+(-0.525871855871)*x[1]**o+(-0.714639297791)*x[1]
        arg[0,2,0,0]=(-0.145788788262)*x[0]**o+(-0.695717636982)*x[0]+(0.736564332801)*x[1]**o+(0.634307016587)*x[1]
        arg[0,2,0,1]=(-0.25142006947)*x[0]**o+(0.027028767127)*x[0]+(-0.614777438028)*x[1]**o+(-0.691417785617)*x[1]
        arg[0,2,1,0]=(0.366081140839)*x[0]**o+(0.477490533426)*x[0]+(-0.116611335116)*x[1]**o+(-0.863552165712)*x[1]
        arg[0,2,1,1]=(-0.467659558308)*x[0]**o+(0.0836299545334)*x[0]+(0.903093252833)*x[1]**o+(-0.956858182533)*x[1]
        arg[0,2,2,0]=(0.870921031579)*x[0]**o+(-0.801681014465)*x[0]+(-0.951819608329)*x[1]**o+(-0.626222917084)*x[1]
        arg[0,2,2,1]=(0.414973183041)*x[0]**o+(-0.444580205703)*x[0]+(-0.346458764208)*x[1]**o+(-0.679140100745)*x[1]
        arg[0,3,0,0]=(-0.725326935158)*x[0]**o+(-0.914034759513)*x[0]+(0.702614377806)*x[1]**o+(0.819004276317)*x[1]
        arg[0,3,0,1]=(0.617099932352)*x[0]**o+(-0.155182850011)*x[0]+(-0.066428715129)*x[1]**o+(-0.286480585435)*x[1]
        arg[0,3,1,0]=(0.710942545488)*x[0]**o+(-0.734509277268)*x[0]+(-0.807474546329)*x[1]**o+(0.307631760449)*x[1]
        arg[0,3,1,1]=(-0.709966924732)*x[0]**o+(-0.462297022374)*x[0]+(0.773787774038)*x[1]**o+(0.762137020991)*x[1]
        arg[0,3,2,0]=(-0.425660932209)*x[0]**o+(-0.562130300181)*x[0]+(-0.516719014647)*x[1]**o+(-0.547832887369)*x[1]
        arg[0,3,2,1]=(0.149573176879)*x[0]**o+(-0.208999629436)*x[0]+(-0.191426033553)*x[1]**o+(-0.119046538873)*x[1]
        arg[0,4,0,0]=(-0.99932317151)*x[0]**o+(-0.832461558068)*x[0]+(-0.847073271676)*x[1]**o+(0.882046887729)*x[1]
        arg[0,4,0,1]=(0.14235557233)*x[0]**o+(-0.352572053276)*x[0]+(-0.231989519255)*x[1]**o+(0.146855135398)*x[1]
        arg[0,4,1,0]=(0.550561449012)*x[0]**o+(0.530745681425)*x[0]+(0.198889689472)*x[1]**o+(0.165503108658)*x[1]
        arg[0,4,1,1]=(0.059663401309)*x[0]**o+(0.168377411751)*x[0]+(-0.224682723925)*x[1]**o+(-0.255406907757)*x[1]
        arg[0,4,2,0]=(-0.747776894683)*x[0]**o+(-0.281763358533)*x[0]+(0.660369838923)*x[1]**o+(0.348541234687)*x[1]
        arg[0,4,2,1]=(-0.886248366716)*x[0]**o+(0.665256341312)*x[0]+(-0.118041122298)*x[1]**o+(0.192284814559)*x[1]
        arg[1,0,0,0]=(-0.212291422817)*x[0]**o+(-0.16123490382)*x[0]+(-0.884220059537)*x[1]**o+(-0.40104775897)*x[1]
        arg[1,0,0,1]=(0.389745305389)*x[0]**o+(0.544302066739)*x[0]+(-0.504770065056)*x[1]**o+(0.903371132654)*x[1]
        arg[1,0,1,0]=(0.983764499357)*x[0]**o+(0.0707766278425)*x[0]+(-0.119851167133)*x[1]**o+(0.668490934432)*x[1]
        arg[1,0,1,1]=(-0.61333738947)*x[0]**o+(-0.515837139916)*x[0]+(-0.825015725806)*x[1]**o+(-0.858420009384)*x[1]
        arg[1,0,2,0]=(0.838488008532)*x[0]**o+(0.320020900158)*x[0]+(0.265140913892)*x[1]**o+(0.493460973766)*x[1]
        arg[1,0,2,1]=(-0.957171379585)*x[0]**o+(0.911667647687)*x[0]+(0.0634495769152)*x[1]**o+(-0.244783653159)*x[1]
        arg[1,1,0,0]=(0.854312398231)*x[0]**o+(0.0177604750376)*x[0]+(-0.263135825468)*x[1]**o+(0.196452389095)*x[1]
        arg[1,1,0,1]=(0.496952473267)*x[0]**o+(-0.505581680291)*x[0]+(0.227653536528)*x[1]**o+(-0.138354150837)*x[1]
        arg[1,1,1,0]=(0.569682536816)*x[0]**o+(-0.523174667456)*x[0]+(-0.434073415962)*x[1]**o+(-0.853667257479)*x[1]
        arg[1,1,1,1]=(0.636882167235)*x[0]**o+(-0.122206427025)*x[0]+(0.934361025063)*x[1]**o+(-0.43037160177)*x[1]
        arg[1,1,2,0]=(0.0798318147504)*x[0]**o+(0.825231320185)*x[0]+(0.810701650808)*x[1]**o+(-0.544356371381)*x[1]
        arg[1,1,2,1]=(-0.426710170704)*x[0]**o+(0.200697079677)*x[0]+(0.242027752153)*x[1]**o+(0.746762119874)*x[1]
        arg[1,2,0,0]=(-0.497091856216)*x[0]**o+(0.0953816987137)*x[0]+(-0.388615292775)*x[1]**o+(0.794962452203)*x[1]
        arg[1,2,0,1]=(0.484769626964)*x[0]**o+(0.700484219762)*x[0]+(-0.230377697673)*x[1]**o+(-0.110327221848)*x[1]
        arg[1,2,1,0]=(-0.791806541129)*x[0]**o+(0.0199911517557)*x[0]+(-0.441657020836)*x[1]**o+(-0.972999930466)*x[1]
        arg[1,2,1,1]=(-0.566750983493)*x[0]**o+(0.193689535753)*x[0]+(0.659310468709)*x[1]**o+(0.209045229813)*x[1]
        arg[1,2,2,0]=(0.852553900341)*x[0]**o+(-0.869275488172)*x[0]+(-0.308429715369)*x[1]**o+(0.096335995297)*x[1]
        arg[1,2,2,1]=(0.169351018917)*x[0]**o+(0.269577471686)*x[0]+(-0.226840840616)*x[1]**o+(-0.94229540867)*x[1]
        arg[1,3,0,0]=(0.481711489234)*x[0]**o+(0.293576608201)*x[0]+(0.101998881767)*x[1]**o+(-0.10730442531)*x[1]
        arg[1,3,0,1]=(-0.824144820539)*x[0]**o+(0.174754759038)*x[0]+(-0.251803196848)*x[1]**o+(-0.389656255459)*x[1]
        arg[1,3,1,0]=(0.438310633462)*x[0]**o+(-0.268051020546)*x[0]+(0.670456978527)*x[1]**o+(-0.547393352784)*x[1]
        arg[1,3,1,1]=(-0.95082246973)*x[0]**o+(0.739061218443)*x[0]+(0.209495253219)*x[1]**o+(-0.604801908581)*x[1]
        arg[1,3,2,0]=(0.696900724687)*x[0]**o+(0.501191311274)*x[0]+(0.982952568975)*x[1]**o+(0.1345704134)*x[1]
        arg[1,3,2,1]=(0.823865642457)*x[0]**o+(0.344980106206)*x[0]+(0.0740071873515)*x[1]**o+(-0.950460516779)*x[1]
        arg[1,4,0,0]=(-0.428525532814)*x[0]**o+(0.496012936411)*x[0]+(-0.0756953876324)*x[1]**o+(0.8382068924)*x[1]
        arg[1,4,0,1]=(0.950388644852)*x[0]**o+(-0.225116488186)*x[0]+(0.515802878592)*x[1]**o+(-0.51004230039)*x[1]
        arg[1,4,1,0]=(0.868733254296)*x[0]**o+(-0.506544709529)*x[0]+(0.28935579533)*x[1]**o+(0.100277736612)*x[1]
        arg[1,4,1,1]=(-0.104533384722)*x[0]**o+(0.598995288066)*x[0]+(-0.738946772839)*x[1]**o+(-0.23245555216)*x[1]
        arg[1,4,2,0]=(-0.286232179213)*x[0]**o+(-0.31267596034)*x[0]+(0.588681250431)*x[1]**o+(-0.579857978487)*x[1]
        arg[1,4,2,1]=(0.405666376198)*x[0]**o+(0.380029842735)*x[0]+(-0.646672003588)*x[1]**o+(0.419606979519)*x[1]
        arg[2,0,0,0]=(0.183075434304)*x[0]**o+(0.0929532345426)*x[0]+(-0.742007638693)*x[1]**o+(-0.521914593386)*x[1]
        arg[2,0,0,1]=(0.292878118412)*x[0]**o+(-0.427165220789)*x[0]+(0.540033289295)*x[1]**o+(0.997945616043)*x[1]
        arg[2,0,1,0]=(0.872801856197)*x[0]**o+(0.0427577769706)*x[0]+(0.595354626662)*x[1]**o+(-0.878459740609)*x[1]
        arg[2,0,1,1]=(-0.259988866684)*x[0]**o+(0.154043636253)*x[0]+(0.0580047878991)*x[1]**o+(0.40696744056)*x[1]
        arg[2,0,2,0]=(-0.602648518466)*x[0]**o+(0.30877596654)*x[0]+(-0.268563459489)*x[1]**o+(-0.526379663585)*x[1]
        arg[2,0,2,1]=(-0.375777994931)*x[0]**o+(-0.175440011709)*x[0]+(-0.578650003477)*x[1]**o+(0.822157683321)*x[1]
        arg[2,1,0,0]=(-0.44389226502)*x[0]**o+(0.683580346916)*x[0]+(0.471857559282)*x[1]**o+(-0.869624665398)*x[1]
        arg[2,1,0,1]=(0.768272315036)*x[0]**o+(-0.709688321973)*x[0]+(0.200727881948)*x[1]**o+(0.617326752785)*x[1]
        arg[2,1,1,0]=(-0.371169228982)*x[0]**o+(-0.0248248405104)*x[0]+(-0.899904749204)*x[1]**o+(0.796779883083)*x[1]
        arg[2,1,1,1]=(0.220088670807)*x[0]**o+(0.691718163435)*x[0]+(-0.0834793048429)*x[1]**o+(-0.748110928988)*x[1]
        arg[2,1,2,0]=(0.246045293748)*x[0]**o+(-0.669660154233)*x[0]+(0.601040129345)*x[1]**o+(-0.0893792369983)*x[1]
        arg[2,1,2,1]=(0.318708374425)*x[0]**o+(-0.973048583674)*x[0]+(-0.575891882633)*x[1]**o+(0.229050058038)*x[1]
        arg[2,2,0,0]=(0.358008410645)*x[0]**o+(-0.834388697702)*x[0]+(0.0931465346237)*x[1]**o+(0.503934981947)*x[1]
        arg[2,2,0,1]=(-0.257597986495)*x[0]**o+(-0.58070568528)*x[0]+(0.735541935653)*x[1]**o+(-0.704666216263)*x[1]
        arg[2,2,1,0]=(0.0207421250729)*x[0]**o+(-0.724644198871)*x[0]+(0.637124335031)*x[1]**o+(-0.547300138808)*x[1]
        arg[2,2,1,1]=(0.280179217797)*x[0]**o+(-0.387279903944)*x[0]+(0.516275341085)*x[1]**o+(0.781250973385)*x[1]
        arg[2,2,2,0]=(-0.97771111517)*x[0]**o+(-0.140997969015)*x[0]+(-0.495112130955)*x[1]**o+(0.151523726543)*x[1]
        arg[2,2,2,1]=(-0.48638258963)*x[0]**o+(-0.251363723104)*x[0]+(-0.484911659879)*x[1]**o+(-0.841285872322)*x[1]
        arg[2,3,0,0]=(-0.70159761565)*x[0]**o+(-0.661658396735)*x[0]+(-0.380920250292)*x[1]**o+(0.682598882025)*x[1]
        arg[2,3,0,1]=(0.477056088757)*x[0]**o+(0.697397619448)*x[0]+(0.28617323153)*x[1]**o+(-0.0443690469658)*x[1]
        arg[2,3,1,0]=(-0.823038323615)*x[0]**o+(-0.742146866184)*x[0]+(0.258604766787)*x[1]**o+(-0.740368164141)*x[1]
        arg[2,3,1,1]=(0.958751077013)*x[0]**o+(0.353976218746)*x[0]+(-0.954344840728)*x[1]**o+(0.436915389196)*x[1]
        arg[2,3,2,0]=(0.432342075923)*x[0]**o+(-0.0699761039653)*x[0]+(0.701490046186)*x[1]**o+(0.0888793443125)*x[1]
        arg[2,3,2,1]=(0.612536667525)*x[0]**o+(0.107433723026)*x[0]+(-0.611874456965)*x[1]**o+(-0.253338899867)*x[1]
        arg[2,4,0,0]=(-0.0234819568317)*x[0]**o+(-0.0642964170457)*x[0]+(0.556679188908)*x[1]**o+(0.865922901373)*x[1]
        arg[2,4,0,1]=(0.669235019483)*x[0]**o+(0.744361910609)*x[0]+(0.383397302403)*x[1]**o+(0.669587709042)*x[1]
        arg[2,4,1,0]=(-0.143461051192)*x[0]**o+(-0.0272388772724)*x[0]+(0.723868767782)*x[1]**o+(0.303617835707)*x[1]
        arg[2,4,1,1]=(-0.829765013546)*x[0]**o+(-0.784193619495)*x[0]+(0.384376849236)*x[1]**o+(0.280904709362)*x[1]
        arg[2,4,2,0]=(0.38737050935)*x[0]**o+(0.749614613361)*x[0]+(-0.340848293488)*x[1]**o+(0.96291850214)*x[1]
        arg[2,4,2,1]=(0.619896872679)*x[0]**o+(0.970694589398)*x[0]+(-0.300698305892)*x[1]**o+(0.794408153506)*x[1]
        arg[3,0,0,0]=(-0.808442463698)*x[0]**o+(-0.527555255968)*x[0]+(0.246776596713)*x[1]**o+(-0.719809811547)*x[1]
        arg[3,0,0,1]=(-0.390016002326)*x[0]**o+(-0.864168521951)*x[0]+(-0.154649478359)*x[1]**o+(0.210766031283)*x[1]
        arg[3,0,1,0]=(-0.941916382966)*x[0]**o+(-0.0352551994705)*x[0]+(-0.660026082442)*x[1]**o+(-0.869241000235)*x[1]
        arg[3,0,1,1]=(-0.287992443961)*x[0]**o+(0.292551506636)*x[0]+(-0.619953648778)*x[1]**o+(0.746095811695)*x[1]
        arg[3,0,2,0]=(-0.233659077726)*x[0]**o+(-0.568647415596)*x[0]+(-0.22048065705)*x[1]**o+(0.658105381132)*x[1]
        arg[3,0,2,1]=(0.871784854897)*x[0]**o+(-0.118389479971)*x[0]+(0.143290959871)*x[1]**o+(0.753198625961)*x[1]
        arg[3,1,0,0]=(0.993988451174)*x[0]**o+(0.451849763406)*x[0]+(-0.972805640824)*x[1]**o+(0.796893857099)*x[1]
        arg[3,1,0,1]=(-0.424353439197)*x[0]**o+(-0.575799061446)*x[0]+(-0.233936106648)*x[1]**o+(-0.451864349448)*x[1]
        arg[3,1,1,0]=(0.639821887614)*x[0]**o+(0.771041944334)*x[0]+(0.420297869214)*x[1]**o+(0.671734395857)*x[1]
        arg[3,1,1,1]=(0.398121897075)*x[0]**o+(0.647679737081)*x[0]+(-0.544457308885)*x[1]**o+(0.58955287395)*x[1]
        arg[3,1,2,0]=(-0.118222351008)*x[0]**o+(-0.943807502172)*x[0]+(0.866866987138)*x[1]**o+(-0.0793340319339)*x[1]
        arg[3,1,2,1]=(-0.157286204244)*x[0]**o+(-0.789214982692)*x[0]+(0.646008960971)*x[1]**o+(0.866687256647)*x[1]
        arg[3,2,0,0]=(-0.620625598995)*x[0]**o+(-0.12874444072)*x[0]+(0.4402928966)*x[1]**o+(-0.0344340173898)*x[1]
        arg[3,2,0,1]=(-0.714597102332)*x[0]**o+(0.730206152989)*x[0]+(0.795717819655)*x[1]**o+(0.0543647593833)*x[1]
        arg[3,2,1,0]=(0.698721346048)*x[0]**o+(0.599204472096)*x[0]+(0.323529910402)*x[1]**o+(0.362678869119)*x[1]
        arg[3,2,1,1]=(0.976107824554)*x[0]**o+(0.559889392243)*x[0]+(0.240016231718)*x[1]**o+(0.343322878078)*x[1]
        arg[3,2,2,0]=(0.609695396838)*x[0]**o+(0.786331471916)*x[0]+(-0.965489152976)*x[1]**o+(0.553653504493)*x[1]
        arg[3,2,2,1]=(0.897952533418)*x[0]**o+(0.227242627929)*x[0]+(-0.121752169524)*x[1]**o+(0.844617689997)*x[1]
        arg[3,3,0,0]=(-0.249698211819)*x[0]**o+(-0.400095108874)*x[0]+(-0.44300562221)*x[1]**o+(0.387777895056)*x[1]
        arg[3,3,0,1]=(0.626469203641)*x[0]**o+(0.741893669108)*x[0]+(0.261882509899)*x[1]**o+(-0.726628233604)*x[1]
        arg[3,3,1,0]=(-0.525223500134)*x[0]**o+(0.918314547123)*x[0]+(0.426441176651)*x[1]**o+(0.338058005974)*x[1]
        arg[3,3,1,1]=(-0.31358142709)*x[0]**o+(-0.411070225036)*x[0]+(0.0902340021221)*x[1]**o+(0.404135075929)*x[1]
        arg[3,3,2,0]=(0.528842663849)*x[0]**o+(0.474768688828)*x[0]+(0.697643730083)*x[1]**o+(-0.170335125341)*x[1]
        arg[3,3,2,1]=(0.875221054674)*x[0]**o+(-0.441830492507)*x[0]+(-0.60312069438)*x[1]**o+(0.664895429753)*x[1]
        arg[3,4,0,0]=(0.563422972203)*x[0]**o+(-0.595004250424)*x[0]+(0.888927367905)*x[1]**o+(0.553640735136)*x[1]
        arg[3,4,0,1]=(0.79563891979)*x[0]**o+(-0.461626171695)*x[0]+(0.700194671524)*x[1]**o+(-0.687505010217)*x[1]
        arg[3,4,1,0]=(0.655248139354)*x[0]**o+(0.588175755435)*x[0]+(-0.29236593461)*x[1]**o+(0.575186295035)*x[1]
        arg[3,4,1,1]=(0.405730566113)*x[0]**o+(-0.161117488254)*x[0]+(0.912224633472)*x[1]**o+(0.0292423764587)*x[1]
        arg[3,4,2,0]=(-0.405819525278)*x[0]**o+(-0.362840757195)*x[0]+(0.961950188511)*x[1]**o+(0.205213542149)*x[1]
        arg[3,4,2,1]=(0.703066280254)*x[0]**o+(-0.936055321446)*x[0]+(0.861070629413)*x[1]**o+(0.0856428368256)*x[1]
        ref[0,0,0,0]=(-0.216088888802)/(o+1.)+(-0.272034496148)
        ref[0,0,0,1]=(0.959350420679)/(o+1.)+(-0.062880226704)
        ref[0,0,1,0]=(-0.525974126338)/(o+1.)+(0.317500886397)
        ref[0,0,1,1]=(-0.471892502528)/(o+1.)+(0.0896700005967)
        ref[0,0,2,0]=(0.454397005927)/(o+1.)+(0.131147481736)
        ref[0,0,2,1]=(-0.314324528348)/(o+1.)+(0.408809451335)
        ref[0,1,0,0]=(-0.411566405684)/(o+1.)+(0.764367546122)
        ref[0,1,0,1]=(-0.126547063321)/(o+1.)+(-0.129100420455)
        ref[0,1,1,0]=(1.53166678447)/(o+1.)+(0.247092500423)
        ref[0,1,1,1]=(-0.444616312725)/(o+1.)+(0.306156441165)
        ref[0,1,2,0]=(0.134683003633)/(o+1.)+(0.482717575736)
        ref[0,1,2,1]=(0.294679478965)/(o+1.)+(-0.457332009538)
        ref[0,2,0,0]=(0.590775544538)/(o+1.)+(-0.0307053101973)
        ref[0,2,0,1]=(-0.866197507498)/(o+1.)+(-0.332194509245)
        ref[0,2,1,0]=(0.249469805723)/(o+1.)+(-0.193030816143)
        ref[0,2,1,1]=(0.435433694525)/(o+1.)+(-0.436614114)
        ref[0,2,2,0]=(-0.0808985767498)/(o+1.)+(-0.713951965774)
        ref[0,2,2,1]=(0.0685144188336)/(o+1.)+(-0.561860153224)
        ref[0,3,0,0]=(-0.0227125573528)/(o+1.)+(-0.0475152415978)
        ref[0,3,0,1]=(0.550671217223)/(o+1.)+(-0.220831717723)
        ref[0,3,1,0]=(-0.0965320008409)/(o+1.)+(-0.213438758409)
        ref[0,3,1,1]=(0.0638208493064)/(o+1.)+(0.149919999309)
        ref[0,3,2,0]=(-0.942379946856)/(o+1.)+(-0.554981593775)
        ref[0,3,2,1]=(-0.0418528566735)/(o+1.)+(-0.164023084154)
        ref[0,4,0,0]=(-1.84639644319)/(o+1.)+(0.0247926648306)
        ref[0,4,0,1]=(-0.0896339469252)/(o+1.)+(-0.102858458939)
        ref[0,4,1,0]=(0.749451138484)/(o+1.)+(0.348124395042)
        ref[0,4,1,1]=(-0.165019322616)/(o+1.)+(-0.0435147480027)
        ref[0,4,2,0]=(-0.0874070557597)/(o+1.)+(0.033388938077)
        ref[0,4,2,1]=(-1.00428948901)/(o+1.)+(0.428770577935)
        ref[1,0,0,0]=(-1.09651148235)/(o+1.)+(-0.281141331395)
        ref[1,0,0,1]=(-0.115024759667)/(o+1.)+(0.723836599697)
        ref[1,0,1,0]=(0.863913332224)/(o+1.)+(0.369633781137)
        ref[1,0,1,1]=(-1.43835311528)/(o+1.)+(-0.68712857465)
        ref[1,0,2,0]=(1.10362892242)/(o+1.)+(0.406740936962)
        ref[1,0,2,1]=(-0.893721802669)/(o+1.)+(0.333441997264)
        ref[1,1,0,0]=(0.591176572763)/(o+1.)+(0.107106432066)
        ref[1,1,0,1]=(0.724606009795)/(o+1.)+(-0.321967915564)
        ref[1,1,1,0]=(0.135609120854)/(o+1.)+(-0.688420962467)
        ref[1,1,1,1]=(1.5712431923)/(o+1.)+(-0.276289014397)
        ref[1,1,2,0]=(0.890533465558)/(o+1.)+(0.140437474402)
        ref[1,1,2,1]=(-0.184682418551)/(o+1.)+(0.473729599776)
        ref[1,2,0,0]=(-0.88570714899)/(o+1.)+(0.445172075458)
        ref[1,2,0,1]=(0.254391929291)/(o+1.)+(0.295078498957)
        ref[1,2,1,0]=(-1.23346356196)/(o+1.)+(-0.476504389355)
        ref[1,2,1,1]=(0.0925594852161)/(o+1.)+(0.201367382783)
        ref[1,2,2,0]=(0.544124184972)/(o+1.)+(-0.386469746437)
        ref[1,2,2,1]=(-0.057489821699)/(o+1.)+(-0.336358968492)
        ref[1,3,0,0]=(0.583710371001)/(o+1.)+(0.0931360914458)
        ref[1,3,0,1]=(-1.07594801739)/(o+1.)+(-0.107450748211)
        ref[1,3,1,0]=(1.10876761199)/(o+1.)+(-0.407722186665)
        ref[1,3,1,1]=(-0.741327216511)/(o+1.)+(0.0671296549307)
        ref[1,3,2,0]=(1.67985329366)/(o+1.)+(0.317880862337)
        ref[1,3,2,1]=(0.897872829808)/(o+1.)+(-0.302740205287)
        ref[1,4,0,0]=(-0.504220920447)/(o+1.)+(0.667109914405)
        ref[1,4,0,1]=(1.46619152344)/(o+1.)+(-0.367579394288)
        ref[1,4,1,0]=(1.15808904963)/(o+1.)+(-0.203133486458)
        ref[1,4,1,1]=(-0.843480157561)/(o+1.)+(0.183269867953)
        ref[1,4,2,0]=(0.302449071217)/(o+1.)+(-0.446266969414)
        ref[1,4,2,1]=(-0.24100562739)/(o+1.)+(0.399818411127)
        ref[2,0,0,0]=(-0.558932204389)/(o+1.)+(-0.214480679421)
        ref[2,0,0,1]=(0.832911407707)/(o+1.)+(0.285390197627)
        ref[2,0,1,0]=(1.46815648286)/(o+1.)+(-0.417850981819)
        ref[2,0,1,1]=(-0.201984078785)/(o+1.)+(0.280505538406)
        ref[2,0,2,0]=(-0.871211977955)/(o+1.)+(-0.108801848522)
        ref[2,0,2,1]=(-0.954427998409)/(o+1.)+(0.323358835806)
        ref[2,1,0,0]=(0.0279652942615)/(o+1.)+(-0.0930221592407)
        ref[2,1,0,1]=(0.969000196983)/(o+1.)+(-0.0461807845941)
        ref[2,1,1,0]=(-1.27107397819)/(o+1.)+(0.385977521287)
        ref[2,1,1,1]=(0.136609365964)/(o+1.)+(-0.0281963827764)
        ref[2,1,2,0]=(0.847085423093)/(o+1.)+(-0.379519695616)
        ref[2,1,2,1]=(-0.257183508208)/(o+1.)+(-0.371999262818)
        ref[2,2,0,0]=(0.451154945268)/(o+1.)+(-0.165226857878)
        ref[2,2,0,1]=(0.477943949158)/(o+1.)+(-0.642685950772)
        ref[2,2,1,0]=(0.657866460103)/(o+1.)+(-0.635972168839)
        ref[2,2,1,1]=(0.796454558882)/(o+1.)+(0.19698553472)
        ref[2,2,2,0]=(-1.47282324613)/(o+1.)+(0.00526287876389)
        ref[2,2,2,1]=(-0.971294249509)/(o+1.)+(-0.546324797713)
        ref[2,3,0,0]=(-1.08251786594)/(o+1.)+(0.0104702426452)
        ref[2,3,0,1]=(0.763229320287)/(o+1.)+(0.326514286241)
        ref[2,3,1,0]=(-0.564433556829)/(o+1.)+(-0.741257515163)
        ref[2,3,1,1]=(0.00440623628526)/(o+1.)+(0.395445803971)
        ref[2,3,2,0]=(1.13383212211)/(o+1.)+(0.0094516201736)
        ref[2,3,2,1]=(0.000662210559809)/(o+1.)+(-0.0729525884203)
        ref[2,4,0,0]=(0.533197232076)/(o+1.)+(0.400813242164)
        ref[2,4,0,1]=(1.05263232189)/(o+1.)+(0.706974809826)
        ref[2,4,1,0]=(0.580407716589)/(o+1.)+(0.138189479217)
        ref[2,4,1,1]=(-0.44538816431)/(o+1.)+(-0.251644455066)
        ref[2,4,2,0]=(0.0465222158616)/(o+1.)+(0.85626655775)
        ref[2,4,2,1]=(0.319198566788)/(o+1.)+(0.882551371452)
        ref[3,0,0,0]=(-0.561665866984)/(o+1.)+(-0.623682533758)
        ref[3,0,0,1]=(-0.544665480685)/(o+1.)+(-0.326701245334)
        ref[3,0,1,0]=(-1.60194246541)/(o+1.)+(-0.452248099853)
        ref[3,0,1,1]=(-0.907946092739)/(o+1.)+(0.519323659165)
        ref[3,0,2,0]=(-0.454139734776)/(o+1.)+(0.0447289827682)
        ref[3,0,2,1]=(1.01507581477)/(o+1.)+(0.317404572995)
        ref[3,1,0,0]=(0.0211828103496)/(o+1.)+(0.624371810253)
        ref[3,1,0,1]=(-0.658289545845)/(o+1.)+(-0.513831705447)
        ref[3,1,1,0]=(1.06011975683)/(o+1.)+(0.721388170095)
        ref[3,1,1,1]=(-0.14633541181)/(o+1.)+(0.618616305516)
        ref[3,1,2,0]=(0.74864463613)/(o+1.)+(-0.511570767053)
        ref[3,1,2,1]=(0.488722756727)/(o+1.)+(0.0387361369774)
        ref[3,2,0,0]=(-0.180332702395)/(o+1.)+(-0.0815892290549)
        ref[3,2,0,1]=(0.0811207173227)/(o+1.)+(0.392285456186)
        ref[3,2,1,0]=(1.02225125645)/(o+1.)+(0.480941670608)
        ref[3,2,1,1]=(1.21612405627)/(o+1.)+(0.45160613516)
        ref[3,2,2,0]=(-0.355793756138)/(o+1.)+(0.669992488205)
        ref[3,2,2,1]=(0.776200363894)/(o+1.)+(0.535930158963)
        ref[3,3,0,0]=(-0.692703834029)/(o+1.)+(-0.00615860690903)
        ref[3,3,0,1]=(0.88835171354)/(o+1.)+(0.00763271775166)
        ref[3,3,1,0]=(-0.0987823234827)/(o+1.)+(0.628186276549)
        ref[3,3,1,1]=(-0.223347424968)/(o+1.)+(-0.00346757455392)
        ref[3,3,2,0]=(1.22648639393)/(o+1.)+(0.152216781743)
        ref[3,3,2,1]=(0.272100360294)/(o+1.)+(0.111532468623)
        ref[3,4,0,0]=(1.45235034011)/(o+1.)+(-0.0206817576441)
        ref[3,4,0,1]=(1.49583359131)/(o+1.)+(-0.574565590956)
        ref[3,4,1,0]=(0.362882204744)/(o+1.)+(0.581681025235)
        ref[3,4,1,1]=(1.31795519959)/(o+1.)+(-0.0659375558974)
        ref[3,4,2,0]=(0.556130663233)/(o+1.)+(-0.078813607523)
        ref[3,4,2,1]=(1.56413690967)/(o+1.)+(-0.42520624231)
      else:
        arg[0,0,0,0]=(0.956854727968)*x[0]**o+(-0.43263548815)*x[0]+(-0.1485543705)*x[1]**o+(-0.411060684134)*x[1]+(-0.306829513954)*x[2]**o+(0.89077924839)*x[2]
        arg[0,0,0,1]=(0.125905642347)*x[0]**o+(-0.645491335766)*x[0]+(-0.237571989123)*x[1]**o+(-0.239301942511)*x[1]+(-0.384215893518)*x[2]**o+(-0.371218179274)*x[2]
        arg[0,0,1,0]=(0.0204273419969)*x[0]**o+(0.354631489501)*x[0]+(0.235086936835)*x[1]**o+(0.647551574475)*x[1]+(-0.729066874572)*x[2]**o+(0.177733791046)*x[2]
        arg[0,0,1,1]=(-0.0938492166558)*x[0]**o+(-0.484339074939)*x[0]+(0.619333193491)*x[1]**o+(0.809157671781)*x[1]+(-0.28367956249)*x[2]**o+(-0.1758249267)*x[2]
        arg[0,0,2,0]=(-0.334944259519)*x[0]**o+(-0.126145567301)*x[0]+(-0.646972616146)*x[1]**o+(0.20287553181)*x[1]+(-0.383816067326)*x[2]**o+(-0.0265591031498)*x[2]
        arg[0,0,2,1]=(0.103308564583)*x[0]**o+(0.016108181951)*x[0]+(-0.367450237136)*x[1]**o+(0.657618788015)*x[1]+(-0.8327532511)*x[2]**o+(-0.267286876849)*x[2]
        arg[0,1,0,0]=(0.688995722152)*x[0]**o+(-0.982012541087)*x[0]+(-0.232339851099)*x[1]**o+(-0.097863461266)*x[1]+(0.25916639806)*x[2]**o+(0.0538637018789)*x[2]
        arg[0,1,0,1]=(0.446698337789)*x[0]**o+(-0.309434573629)*x[0]+(0.617004214016)*x[1]**o+(0.747666594435)*x[1]+(-0.869684246717)*x[2]**o+(0.75016186442)*x[2]
        arg[0,1,1,0]=(-0.559974353252)*x[0]**o+(0.966642794905)*x[0]+(-0.228473000256)*x[1]**o+(-0.435093965632)*x[1]+(0.426986383284)*x[2]**o+(0.131300817321)*x[2]
        arg[0,1,1,1]=(-0.696027598385)*x[0]**o+(-0.804778535506)*x[0]+(0.930171049019)*x[1]**o+(-0.900826020873)*x[1]+(0.844689166523)*x[2]**o+(0.0349718139676)*x[2]
        arg[0,1,2,0]=(0.359170897328)*x[0]**o+(0.992944582308)*x[0]+(0.579162388074)*x[1]**o+(0.743410851741)*x[1]+(-0.204325306226)*x[2]**o+(-0.992005484878)*x[2]
        arg[0,1,2,1]=(-0.883554988936)*x[0]**o+(0.486947073334)*x[0]+(-0.78078860326)*x[1]**o+(-0.134196299284)*x[1]+(-0.980852067031)*x[2]**o+(0.453023187355)*x[2]
        arg[0,2,0,0]=(0.804524524371)*x[0]**o+(0.397924403155)*x[0]+(0.712297039229)*x[1]**o+(0.713485622558)*x[1]+(-0.966783610543)*x[2]**o+(0.497869404576)*x[2]
        arg[0,2,0,1]=(-0.564706930001)*x[0]**o+(-0.482026983244)*x[0]+(0.284926154564)*x[1]**o+(0.844423322149)*x[1]+(0.444203847061)*x[2]**o+(0.333244188639)*x[2]
        arg[0,2,1,0]=(-0.630410083438)*x[0]**o+(-0.831218126629)*x[0]+(0.400237927956)*x[1]**o+(-0.70518378137)*x[1]+(-0.803754805917)*x[2]**o+(0.60059455602)*x[2]
        arg[0,2,1,1]=(0.198272547853)*x[0]**o+(0.0840356091496)*x[0]+(0.114385146155)*x[1]**o+(0.731057038288)*x[1]+(-0.976056408523)*x[2]**o+(-0.132565954304)*x[2]
        arg[0,2,2,0]=(0.983300141657)*x[0]**o+(-0.733168195162)*x[0]+(-0.948802450335)*x[1]**o+(-0.756616078178)*x[1]+(0.450302374372)*x[2]**o+(0.802115535717)*x[2]
        arg[0,2,2,1]=(-0.680853503774)*x[0]**o+(-0.910402862928)*x[0]+(0.583775613868)*x[1]**o+(0.741516645611)*x[1]+(-0.550498467533)*x[2]**o+(-0.610648481959)*x[2]
        arg[0,3,0,0]=(0.981909313325)*x[0]**o+(0.148145112359)*x[0]+(-0.989721117518)*x[1]**o+(0.0246558730659)*x[1]+(0.721439979982)*x[2]**o+(0.896052837889)*x[2]
        arg[0,3,0,1]=(0.109831691542)*x[0]**o+(-0.349555366958)*x[0]+(-0.54822642636)*x[1]**o+(0.121919022172)*x[1]+(-0.580228247481)*x[2]**o+(0.564276455713)*x[2]
        arg[0,3,1,0]=(0.106091093353)*x[0]**o+(0.410111742327)*x[0]+(-0.0669026478824)*x[1]**o+(-0.662684907574)*x[1]+(-0.351815520918)*x[2]**o+(-0.324802958038)*x[2]
        arg[0,3,1,1]=(0.857685803258)*x[0]**o+(-0.782627954043)*x[0]+(-0.555036389439)*x[1]**o+(-0.0126716685608)*x[1]+(0.260461379777)*x[2]**o+(0.117590260591)*x[2]
        arg[0,3,2,0]=(-0.987873674604)*x[0]**o+(0.0779662295512)*x[0]+(-0.928229202803)*x[1]**o+(0.508015712707)*x[1]+(0.248638760509)*x[2]**o+(-0.712158091156)*x[2]
        arg[0,3,2,1]=(-0.977973356138)*x[0]**o+(0.270015563339)*x[0]+(0.811157094127)*x[1]**o+(-0.756460384936)*x[1]+(-0.735557557504)*x[2]**o+(0.780333584929)*x[2]
        arg[0,4,0,0]=(-0.762444411646)*x[0]**o+(0.386894014691)*x[0]+(0.763898738855)*x[1]**o+(-0.422896581278)*x[1]+(0.428591445563)*x[2]**o+(-0.617626926396)*x[2]
        arg[0,4,0,1]=(-0.269524167286)*x[0]**o+(0.996895823788)*x[0]+(0.771396605337)*x[1]**o+(-0.993794940262)*x[1]+(-0.534803587481)*x[2]**o+(-0.928627946122)*x[2]
        arg[0,4,1,0]=(0.978497459584)*x[0]**o+(-0.343496507186)*x[0]+(-0.250581087884)*x[1]**o+(-0.855743445852)*x[1]+(-0.127512246124)*x[2]**o+(-0.133936486447)*x[2]
        arg[0,4,1,1]=(-0.844450818471)*x[0]**o+(0.263329730105)*x[0]+(0.411212783735)*x[1]**o+(-0.816379748952)*x[1]+(-0.443806901857)*x[2]**o+(0.1262969097)*x[2]
        arg[0,4,2,0]=(-0.326938381124)*x[0]**o+(-0.707054728551)*x[0]+(-0.725513874889)*x[1]**o+(0.655254977879)*x[1]+(0.772734170665)*x[2]**o+(-0.963856465024)*x[2]
        arg[0,4,2,1]=(0.896415185133)*x[0]**o+(-0.722431478737)*x[0]+(0.942058336843)*x[1]**o+(0.838230971246)*x[1]+(-0.279821311439)*x[2]**o+(-0.330130961206)*x[2]
        arg[1,0,0,0]=(-0.466537465363)*x[0]**o+(0.86493624537)*x[0]+(-0.943049537777)*x[1]**o+(0.658727518981)*x[1]+(-0.130355508544)*x[2]**o+(-0.158245981876)*x[2]
        arg[1,0,0,1]=(0.247046604605)*x[0]**o+(-0.287683750248)*x[0]+(0.266630203142)*x[1]**o+(0.372409648579)*x[1]+(-0.239411725166)*x[2]**o+(0.537339451105)*x[2]
        arg[1,0,1,0]=(-0.232312387253)*x[0]**o+(-0.0493954012747)*x[0]+(-0.461085479525)*x[1]**o+(-0.0744729096417)*x[1]+(-0.795163064273)*x[2]**o+(0.613075576806)*x[2]
        arg[1,0,1,1]=(-0.68570821748)*x[0]**o+(-0.492507082229)*x[0]+(-0.0673636097828)*x[1]**o+(0.569119992806)*x[1]+(-0.227653595613)*x[2]**o+(0.0337433636837)*x[2]
        arg[1,0,2,0]=(-0.124371489405)*x[0]**o+(-0.726635218442)*x[0]+(-0.835418167414)*x[1]**o+(-0.00414597707753)*x[1]+(0.8365917104)*x[2]**o+(-0.507314639615)*x[2]
        arg[1,0,2,1]=(-0.973228057052)*x[0]**o+(0.123228228923)*x[0]+(0.545712842666)*x[1]**o+(-0.451331566047)*x[1]+(0.299501907067)*x[2]**o+(-0.24898389883)*x[2]
        arg[1,1,0,0]=(-0.940690906836)*x[0]**o+(-0.357571197214)*x[0]+(0.98643983012)*x[1]**o+(-0.82113140872)*x[1]+(0.460009285424)*x[2]**o+(-0.0298714172583)*x[2]
        arg[1,1,0,1]=(-0.881543245921)*x[0]**o+(-0.640741814727)*x[0]+(-0.749244677096)*x[1]**o+(0.47161101909)*x[1]+(0.712893375828)*x[2]**o+(0.600151133959)*x[2]
        arg[1,1,1,0]=(-0.217214692405)*x[0]**o+(-0.504211076658)*x[0]+(0.857127736087)*x[1]**o+(0.0439761116172)*x[1]+(0.956380848189)*x[2]**o+(0.698482155013)*x[2]
        arg[1,1,1,1]=(-0.288969728734)*x[0]**o+(-0.63960886193)*x[0]+(-0.741293494258)*x[1]**o+(-0.125034323955)*x[1]+(-0.608855286755)*x[2]**o+(0.462427737944)*x[2]
        arg[1,1,2,0]=(0.126242043984)*x[0]**o+(0.811846484451)*x[0]+(-0.522920862127)*x[1]**o+(0.678142081935)*x[1]+(0.0392586436898)*x[2]**o+(0.403908304728)*x[2]
        arg[1,1,2,1]=(-0.290103861843)*x[0]**o+(0.336130241928)*x[0]+(-0.872641307678)*x[1]**o+(-0.372020796526)*x[1]+(0.172572077906)*x[2]**o+(-0.789431556966)*x[2]
        arg[1,2,0,0]=(-0.158114079913)*x[0]**o+(0.954919898537)*x[0]+(-0.711834968234)*x[1]**o+(-0.390487270434)*x[1]+(-0.086581514295)*x[2]**o+(0.442047084988)*x[2]
        arg[1,2,0,1]=(-0.914034594356)*x[0]**o+(0.521137372279)*x[0]+(0.12454416569)*x[1]**o+(0.627815188423)*x[1]+(0.770244577882)*x[2]**o+(-0.846691748091)*x[2]
        arg[1,2,1,0]=(0.769631824465)*x[0]**o+(0.988840796717)*x[0]+(-0.401836868027)*x[1]**o+(0.401846957832)*x[1]+(-0.877826997285)*x[2]**o+(0.857725824229)*x[2]
        arg[1,2,1,1]=(0.173539365276)*x[0]**o+(0.373206670706)*x[0]+(-0.597288730366)*x[1]**o+(-0.770220794555)*x[1]+(-0.586510454467)*x[2]**o+(-0.569370842441)*x[2]
        arg[1,2,2,0]=(0.673765155818)*x[0]**o+(-0.642914516709)*x[0]+(0.794438851788)*x[1]**o+(-0.17557754186)*x[1]+(0.997565143863)*x[2]**o+(0.722460144483)*x[2]
        arg[1,2,2,1]=(-0.878452482718)*x[0]**o+(-0.846862156166)*x[0]+(0.462351027504)*x[1]**o+(0.267435606154)*x[1]+(-0.452135086983)*x[2]**o+(0.701577407252)*x[2]
        arg[1,3,0,0]=(-0.0557389135964)*x[0]**o+(-0.781831460089)*x[0]+(0.214846092931)*x[1]**o+(0.597436260353)*x[1]+(0.713968428327)*x[2]**o+(-0.456335676572)*x[2]
        arg[1,3,0,1]=(0.288266531585)*x[0]**o+(-0.259529556521)*x[0]+(0.0286107975184)*x[1]**o+(0.465095455484)*x[1]+(-0.747648909025)*x[2]**o+(0.682866470382)*x[2]
        arg[1,3,1,0]=(-0.720339104668)*x[0]**o+(0.961100526257)*x[0]+(0.0814885381756)*x[1]**o+(-0.448775223649)*x[1]+(0.250972143198)*x[2]**o+(-0.240231439947)*x[2]
        arg[1,3,1,1]=(-0.459305786748)*x[0]**o+(0.272222143865)*x[0]+(-0.499164099401)*x[1]**o+(-0.225162547271)*x[1]+(-0.92261124167)*x[2]**o+(-0.436659790026)*x[2]
        arg[1,3,2,0]=(0.890752966329)*x[0]**o+(-0.775207922011)*x[0]+(0.715960284721)*x[1]**o+(0.843422841789)*x[1]+(0.513788850466)*x[2]**o+(-0.325733710281)*x[2]
        arg[1,3,2,1]=(0.958726172433)*x[0]**o+(-0.743672393249)*x[0]+(-0.523783133945)*x[1]**o+(0.133861976051)*x[1]+(-0.109230940116)*x[2]**o+(0.819716327751)*x[2]
        arg[1,4,0,0]=(0.54439729551)*x[0]**o+(0.0763503792087)*x[0]+(0.0634218657498)*x[1]**o+(-0.405471845044)*x[1]+(0.316327091756)*x[2]**o+(-0.539493863939)*x[2]
        arg[1,4,0,1]=(0.527330373182)*x[0]**o+(0.77531447123)*x[0]+(-0.481869315442)*x[1]**o+(0.556927692362)*x[1]+(-0.267215098309)*x[2]**o+(0.958919406465)*x[2]
        arg[1,4,1,0]=(0.540950372012)*x[0]**o+(-0.95002626611)*x[0]+(-0.523544509893)*x[1]**o+(-0.583056635057)*x[1]+(-0.00288409928193)*x[2]**o+(0.671385098427)*x[2]
        arg[1,4,1,1]=(-0.39802076092)*x[0]**o+(-0.583264022146)*x[0]+(-0.656261495961)*x[1]**o+(-0.527677078418)*x[1]+(-0.2632498932)*x[2]**o+(-0.537632951507)*x[2]
        arg[1,4,2,0]=(-0.77176416566)*x[0]**o+(0.812696440735)*x[0]+(0.0925789242005)*x[1]**o+(0.299698967601)*x[1]+(-0.0185377066587)*x[2]**o+(-0.883247318682)*x[2]
        arg[1,4,2,1]=(0.360949407452)*x[0]**o+(-0.463659754812)*x[0]+(0.799066387132)*x[1]**o+(0.0318151346916)*x[1]+(-0.380675976437)*x[2]**o+(-0.299341342112)*x[2]
        arg[2,0,0,0]=(-0.985332287141)*x[0]**o+(0.33398206098)*x[0]+(0.39204502373)*x[1]**o+(-0.586826360517)*x[1]+(0.868240161369)*x[2]**o+(-0.257897525907)*x[2]
        arg[2,0,0,1]=(-0.801369266967)*x[0]**o+(0.333496542809)*x[0]+(0.459444994717)*x[1]**o+(-0.950297568688)*x[1]+(-0.760722281765)*x[2]**o+(0.0933850221377)*x[2]
        arg[2,0,1,0]=(0.53372811812)*x[0]**o+(0.999551098511)*x[0]+(-0.761650890163)*x[1]**o+(-0.80456940786)*x[1]+(-0.452089806623)*x[2]**o+(-0.40841169158)*x[2]
        arg[2,0,1,1]=(-0.782294374468)*x[0]**o+(0.614736817827)*x[0]+(0.262995910218)*x[1]**o+(0.067349886804)*x[1]+(-0.167494656909)*x[2]**o+(-0.730462378769)*x[2]
        arg[2,0,2,0]=(0.261001857862)*x[0]**o+(0.959318765248)*x[0]+(-0.177224016298)*x[1]**o+(-0.979637664014)*x[1]+(0.858937930292)*x[2]**o+(-0.627116377798)*x[2]
        arg[2,0,2,1]=(-0.238083578675)*x[0]**o+(0.586367991705)*x[0]+(0.747378561289)*x[1]**o+(-0.540178926415)*x[1]+(-0.519781240185)*x[2]**o+(-0.42715014981)*x[2]
        arg[2,1,0,0]=(-0.21022574437)*x[0]**o+(-0.292359099348)*x[0]+(0.449614855785)*x[1]**o+(-0.416904619153)*x[1]+(0.507537288404)*x[2]**o+(0.943584716996)*x[2]
        arg[2,1,0,1]=(-0.410591525695)*x[0]**o+(-0.258796394127)*x[0]+(-0.474714731399)*x[1]**o+(-0.518288451462)*x[1]+(-0.15640903664)*x[2]**o+(0.97242574307)*x[2]
        arg[2,1,1,0]=(-0.930865834892)*x[0]**o+(0.961887630616)*x[0]+(0.688706862295)*x[1]**o+(-0.375163634404)*x[1]+(0.378954001649)*x[2]**o+(-0.747140890466)*x[2]
        arg[2,1,1,1]=(0.0454701384411)*x[0]**o+(-0.37651801332)*x[0]+(0.0808750051583)*x[1]**o+(-0.530355850075)*x[1]+(0.769675417234)*x[2]**o+(-0.767046792016)*x[2]
        arg[2,1,2,0]=(0.179573418696)*x[0]**o+(-0.0547918709025)*x[0]+(0.739792606043)*x[1]**o+(0.849619562235)*x[1]+(-0.92840796838)*x[2]**o+(-0.302941296337)*x[2]
        arg[2,1,2,1]=(0.160657807137)*x[0]**o+(0.551943403654)*x[0]+(0.649164334204)*x[1]**o+(0.298882266215)*x[1]+(0.33872360276)*x[2]**o+(-0.558910868754)*x[2]
        arg[2,2,0,0]=(-0.704330511528)*x[0]**o+(-0.326943167116)*x[0]+(0.626863819473)*x[1]**o+(-0.107147718189)*x[1]+(0.00668670393016)*x[2]**o+(0.572223662188)*x[2]
        arg[2,2,0,1]=(0.879695137799)*x[0]**o+(-0.313488700497)*x[0]+(-0.113134766565)*x[1]**o+(-0.131007759381)*x[1]+(0.55460969593)*x[2]**o+(0.898706374575)*x[2]
        arg[2,2,1,0]=(0.329932124981)*x[0]**o+(-0.624965470976)*x[0]+(0.82906774364)*x[1]**o+(-0.681403786977)*x[1]+(-0.447119491747)*x[2]**o+(0.143763601614)*x[2]
        arg[2,2,1,1]=(0.0367746207528)*x[0]**o+(0.47553259605)*x[0]+(-0.517723045973)*x[1]**o+(0.263932783195)*x[1]+(-0.131032589284)*x[2]**o+(-0.132729291841)*x[2]
        arg[2,2,2,0]=(-0.108349266564)*x[0]**o+(-0.684748849483)*x[0]+(0.203271582231)*x[1]**o+(-0.860842085896)*x[1]+(0.469684577785)*x[2]**o+(0.195736308182)*x[2]
        arg[2,2,2,1]=(0.915666799349)*x[0]**o+(0.383338953756)*x[0]+(-0.554011249579)*x[1]**o+(0.232285573367)*x[1]+(0.39288151674)*x[2]**o+(0.692592233016)*x[2]
        arg[2,3,0,0]=(0.81179207663)*x[0]**o+(0.786809236572)*x[0]+(-0.814758445611)*x[1]**o+(-0.78461653396)*x[1]+(0.171435802778)*x[2]**o+(0.660186909193)*x[2]
        arg[2,3,0,1]=(0.752694523669)*x[0]**o+(0.301473785296)*x[0]+(0.610379955695)*x[1]**o+(0.573729399495)*x[1]+(0.225983751859)*x[2]**o+(-0.349530143446)*x[2]
        arg[2,3,1,0]=(-0.647825490424)*x[0]**o+(-0.232573858642)*x[0]+(0.773519085771)*x[1]**o+(-0.224497979868)*x[1]+(-0.681726885952)*x[2]**o+(-0.217968759388)*x[2]
        arg[2,3,1,1]=(0.391888417642)*x[0]**o+(-0.240556139087)*x[0]+(-0.852397797723)*x[1]**o+(0.318342758913)*x[1]+(-0.773804983598)*x[2]**o+(-0.957590491908)*x[2]
        arg[2,3,2,0]=(0.7126582007)*x[0]**o+(0.806022164546)*x[0]+(0.232348951782)*x[1]**o+(-0.985774366344)*x[1]+(-0.365161907541)*x[2]**o+(0.0218977623299)*x[2]
        arg[2,3,2,1]=(-0.775068843872)*x[0]**o+(-0.595291330879)*x[0]+(0.681122481841)*x[1]**o+(0.152514252732)*x[1]+(0.510976191808)*x[2]**o+(-0.69539067488)*x[2]
        arg[2,4,0,0]=(-0.362761749405)*x[0]**o+(0.494347958562)*x[0]+(0.215244694522)*x[1]**o+(-0.527767969217)*x[1]+(-0.115976071059)*x[2]**o+(-0.998233415841)*x[2]
        arg[2,4,0,1]=(0.620291015098)*x[0]**o+(-0.222644358028)*x[0]+(0.0377714785847)*x[1]**o+(0.663978623913)*x[1]+(0.996481523389)*x[2]**o+(-0.689519418095)*x[2]
        arg[2,4,1,0]=(-0.0374746407394)*x[0]**o+(0.3186355021)*x[0]+(-0.704305640533)*x[1]**o+(-0.632264150766)*x[1]+(-0.166691199609)*x[2]**o+(-0.794128441334)*x[2]
        arg[2,4,1,1]=(0.117806334919)*x[0]**o+(-0.717334572919)*x[0]+(0.665341054571)*x[1]**o+(0.354208273372)*x[1]+(0.349883631586)*x[2]**o+(-0.260626035314)*x[2]
        arg[2,4,2,0]=(0.160337827685)*x[0]**o+(-0.92734430881)*x[0]+(0.872112181363)*x[1]**o+(-0.796372970466)*x[1]+(-0.776882702685)*x[2]**o+(0.246816968474)*x[2]
        arg[2,4,2,1]=(-0.317926812571)*x[0]**o+(-0.795779593996)*x[0]+(0.785287724793)*x[1]**o+(0.839033515634)*x[1]+(0.433187943714)*x[2]**o+(0.308175885606)*x[2]
        arg[3,0,0,0]=(-0.555193292291)*x[0]**o+(-0.0280452948283)*x[0]+(0.988455127227)*x[1]**o+(-0.637707060271)*x[1]+(-0.95062571916)*x[2]**o+(-0.563121211553)*x[2]
        arg[3,0,0,1]=(-0.169558534946)*x[0]**o+(0.886096637835)*x[0]+(-0.666307491933)*x[1]**o+(0.650321801985)*x[1]+(-0.268934932123)*x[2]**o+(-0.406617199906)*x[2]
        arg[3,0,1,0]=(0.497351781449)*x[0]**o+(0.768617651685)*x[0]+(0.200962956226)*x[1]**o+(0.694236078954)*x[1]+(0.0323702864321)*x[2]**o+(-0.666046265832)*x[2]
        arg[3,0,1,1]=(0.425151807387)*x[0]**o+(-0.886432438604)*x[0]+(-0.963257203569)*x[1]**o+(-0.698524427437)*x[1]+(0.0165810713355)*x[2]**o+(-0.142193988951)*x[2]
        arg[3,0,2,0]=(0.965746112788)*x[0]**o+(-0.333068915886)*x[0]+(0.331031372002)*x[1]**o+(-0.384004796757)*x[1]+(0.0144121092048)*x[2]**o+(-0.352607264387)*x[2]
        arg[3,0,2,1]=(-0.391557775789)*x[0]**o+(0.127020330586)*x[0]+(0.758832450081)*x[1]**o+(0.49669414469)*x[1]+(0.358403498634)*x[2]**o+(0.184963946351)*x[2]
        arg[3,1,0,0]=(0.978053897464)*x[0]**o+(-0.211787313494)*x[0]+(0.927706482976)*x[1]**o+(-0.451292956408)*x[1]+(0.592824645174)*x[2]**o+(0.896240907546)*x[2]
        arg[3,1,0,1]=(0.0780568875295)*x[0]**o+(0.130098698523)*x[0]+(-0.281871571918)*x[1]**o+(-0.596362849045)*x[1]+(-0.436455789199)*x[2]**o+(-0.657145053021)*x[2]
        arg[3,1,1,0]=(-0.278111817329)*x[0]**o+(-0.627384158565)*x[0]+(-0.995802044048)*x[1]**o+(0.42485189159)*x[1]+(0.483473275475)*x[2]**o+(-0.177928974458)*x[2]
        arg[3,1,1,1]=(0.127721047557)*x[0]**o+(0.93147711978)*x[0]+(-0.350992052283)*x[1]**o+(0.0993773637672)*x[1]+(0.29111744653)*x[2]**o+(-0.821710278883)*x[2]
        arg[3,1,2,0]=(-0.697941918276)*x[0]**o+(-0.585872232852)*x[0]+(0.697967779925)*x[1]**o+(-0.811186381728)*x[1]+(0.708726108998)*x[2]**o+(0.289602027081)*x[2]
        arg[3,1,2,1]=(-0.461424409147)*x[0]**o+(0.335089386385)*x[0]+(-0.964838280406)*x[1]**o+(0.828627561583)*x[1]+(-0.861416281013)*x[2]**o+(0.898354675269)*x[2]
        arg[3,2,0,0]=(0.752543705358)*x[0]**o+(0.164487088496)*x[0]+(0.756164002406)*x[1]**o+(-0.740019514032)*x[1]+(0.107245489678)*x[2]**o+(0.677707700079)*x[2]
        arg[3,2,0,1]=(0.763792557134)*x[0]**o+(0.725507550672)*x[0]+(0.171323949629)*x[1]**o+(-0.287172075258)*x[1]+(-0.94865226715)*x[2]**o+(-0.163093465336)*x[2]
        arg[3,2,1,0]=(-0.327892721048)*x[0]**o+(0.334145897353)*x[0]+(-0.936827293436)*x[1]**o+(0.0822687872593)*x[1]+(0.259794829604)*x[2]**o+(0.376908280759)*x[2]
        arg[3,2,1,1]=(0.595012545955)*x[0]**o+(0.617490014693)*x[0]+(-0.658036388462)*x[1]**o+(-0.921055393527)*x[1]+(-0.257878085344)*x[2]**o+(-0.623113089161)*x[2]
        arg[3,2,2,0]=(0.961175705937)*x[0]**o+(0.656031660842)*x[0]+(0.964101580317)*x[1]**o+(0.824033157042)*x[1]+(0.510244886658)*x[2]**o+(-0.236501162729)*x[2]
        arg[3,2,2,1]=(-0.128405588923)*x[0]**o+(-0.0327141502894)*x[0]+(0.909941485131)*x[1]**o+(0.79947258121)*x[1]+(0.635883050534)*x[2]**o+(0.841997575796)*x[2]
        arg[3,3,0,0]=(-0.906234907643)*x[0]**o+(-0.316076976009)*x[0]+(0.863208664533)*x[1]**o+(-0.968640170523)*x[1]+(0.791637381686)*x[2]**o+(-0.543353462098)*x[2]
        arg[3,3,0,1]=(-0.586709036832)*x[0]**o+(-0.299493216592)*x[0]+(-0.173966725899)*x[1]**o+(0.0971838711677)*x[1]+(0.385662922244)*x[2]**o+(0.288932625368)*x[2]
        arg[3,3,1,0]=(-0.33865322795)*x[0]**o+(0.442176445777)*x[0]+(0.706183231282)*x[1]**o+(0.62528018966)*x[1]+(-0.377746006979)*x[2]**o+(-0.261455124139)*x[2]
        arg[3,3,1,1]=(0.834855553776)*x[0]**o+(0.755005824764)*x[0]+(0.446589332183)*x[1]**o+(0.41883521757)*x[1]+(0.820177207209)*x[2]**o+(0.603811552083)*x[2]
        arg[3,3,2,0]=(-0.624282266193)*x[0]**o+(0.28109086861)*x[0]+(0.44286011655)*x[1]**o+(-0.117876187343)*x[1]+(-0.267660131821)*x[2]**o+(0.880359406159)*x[2]
        arg[3,3,2,1]=(-0.0722687033317)*x[0]**o+(-0.7013147214)*x[0]+(0.00614303249005)*x[1]**o+(-0.658138946692)*x[1]+(0.0557277088337)*x[2]**o+(0.614362766103)*x[2]
        arg[3,4,0,0]=(0.935244440586)*x[0]**o+(-0.0699305155896)*x[0]+(-0.849303362918)*x[1]**o+(0.728062537708)*x[1]+(0.404621498028)*x[2]**o+(0.611486631437)*x[2]
        arg[3,4,0,1]=(0.168117782017)*x[0]**o+(-0.0454396423181)*x[0]+(0.46021270423)*x[1]**o+(0.597343569492)*x[1]+(0.561215321486)*x[2]**o+(0.715771227777)*x[2]
        arg[3,4,1,0]=(-0.502675682461)*x[0]**o+(0.543910335185)*x[0]+(0.721732643536)*x[1]**o+(-0.862126899566)*x[1]+(0.794704104653)*x[2]**o+(-0.284248276797)*x[2]
        arg[3,4,1,1]=(-0.580751817984)*x[0]**o+(0.482838505281)*x[0]+(-0.426066109367)*x[1]**o+(-0.456724382007)*x[1]+(-0.770655426938)*x[2]**o+(-0.226232224569)*x[2]
        arg[3,4,2,0]=(0.199832861603)*x[0]**o+(-0.586239366436)*x[0]+(0.730019631541)*x[1]**o+(0.185265476259)*x[1]+(0.514243351617)*x[2]**o+(-0.757742857839)*x[2]
        arg[3,4,2,1]=(-0.941843068195)*x[0]**o+(0.211732407161)*x[0]+(-0.99705602485)*x[1]**o+(-0.198037742645)*x[1]+(0.154058794178)*x[2]**o+(0.0675962671791)*x[2]
        ref[0,0,0,0]=(0.501470843513)/(o+1.)+(0.0235415380528)
        ref[0,0,0,1]=(-0.495882240293)/(o+1.)+(-0.628005728776)
        ref[0,0,1,0]=(-0.47355259574)/(o+1.)+(0.589958427511)
        ref[0,0,1,1]=(0.241804414345)/(o+1.)+(0.0744968350706)
        ref[0,0,2,0]=(-1.36573294299)/(o+1.)+(0.0250854306796)
        ref[0,0,2,1]=(-1.09689492365)/(o+1.)+(0.203220046558)
        ref[0,1,0,0]=(0.715822269113)/(o+1.)+(-0.513006150237)
        ref[0,1,0,1]=(0.194018305088)/(o+1.)+(0.594196942613)
        ref[0,1,1,0]=(-0.361460970224)/(o+1.)+(0.331424823297)
        ref[0,1,1,1]=(1.07883261716)/(o+1.)+(-0.835316371206)
        ref[0,1,2,0]=(0.734007979175)/(o+1.)+(0.372174974585)
        ref[0,1,2,1]=(-2.64519565923)/(o+1.)+(0.402886980703)
        ref[0,2,0,0]=(0.550037953057)/(o+1.)+(0.804639715144)
        ref[0,2,0,1]=(0.164423071623)/(o+1.)+(0.347820263772)
        ref[0,2,1,0]=(-1.0339269614)/(o+1.)+(-0.46790367599)
        ref[0,2,1,1]=(-0.663398714515)/(o+1.)+(0.341263346566)
        ref[0,2,2,0]=(0.484800065694)/(o+1.)+(-0.343834368812)
        ref[0,2,2,1]=(-0.647576357439)/(o+1.)+(-0.389767349638)
        ref[0,3,0,0]=(0.713628175789)/(o+1.)+(0.534426911657)
        ref[0,3,0,1]=(-1.0186229823)/(o+1.)+(0.168320055464)
        ref[0,3,1,0]=(-0.312627075447)/(o+1.)+(-0.288688061643)
        ref[0,3,1,1]=(0.563110793595)/(o+1.)+(-0.338854681006)
        ref[0,3,2,0]=(-1.6674641169)/(o+1.)+(-0.0630880744491)
        ref[0,3,2,1]=(-0.902373819515)/(o+1.)+(0.146944381666)
        ref[0,4,0,0]=(0.430045772773)/(o+1.)+(-0.326814746492)
        ref[0,4,0,1]=(-0.0329311494305)/(o+1.)+(-0.462763531298)
        ref[0,4,1,0]=(0.600404125575)/(o+1.)+(-0.666588219743)
        ref[0,4,1,1]=(-0.877044936593)/(o+1.)+(-0.213376554573)
        ref[0,4,2,0]=(-0.279718085349)/(o+1.)+(-0.507828107848)
        ref[0,4,2,1]=(1.55865221054)/(o+1.)+(-0.107165734349)
        ref[1,0,0,0]=(-1.53994251168)/(o+1.)+(0.682708891237)
        ref[1,0,0,1]=(0.27426508258)/(o+1.)+(0.311032674718)
        ref[1,0,1,0]=(-1.48856093105)/(o+1.)+(0.244603632945)
        ref[1,0,1,1]=(-0.980725422876)/(o+1.)+(0.0551781371305)
        ref[1,0,2,0]=(-0.123197946419)/(o+1.)+(-0.619047917567)
        ref[1,0,2,1]=(-0.128013307318)/(o+1.)+(-0.288543617977)
        ref[1,1,0,0]=(0.505758208708)/(o+1.)+(-0.604287011596)
        ref[1,1,0,1]=(-0.917894547188)/(o+1.)+(0.215510169161)
        ref[1,1,1,0]=(1.59629389187)/(o+1.)+(0.119123594986)
        ref[1,1,1,1]=(-1.63911850975)/(o+1.)+(-0.15110772397)
        ref[1,1,2,0]=(-0.357420174452)/(o+1.)+(0.946948435557)
        ref[1,1,2,1]=(-0.990173091615)/(o+1.)+(-0.412661055782)
        ref[1,2,0,0]=(-0.956530562442)/(o+1.)+(0.503239856545)
        ref[1,2,0,1]=(-0.0192458507848)/(o+1.)+(0.151130406306)
        ref[1,2,1,0]=(-0.510032040848)/(o+1.)+(1.12420678939)
        ref[1,2,1,1]=(-1.01025981956)/(o+1.)+(-0.483192483145)
        ref[1,2,2,0]=(2.46576915147)/(o+1.)+(-0.0480159570431)
        ref[1,2,2,1]=(-0.868236542197)/(o+1.)+(0.0610754286198)
        ref[1,3,0,0]=(0.873075607662)/(o+1.)+(-0.320365438154)
        ref[1,3,0,1]=(-0.430771579922)/(o+1.)+(0.444216184672)
        ref[1,3,1,0]=(-0.387878423294)/(o+1.)+(0.13604693133)
        ref[1,3,1,1]=(-1.88108112782)/(o+1.)+(-0.194800096716)
        ref[1,3,2,0]=(2.12050210152)/(o+1.)+(-0.128759395251)
        ref[1,3,2,1]=(0.325712098372)/(o+1.)+(0.104952955276)
        ref[1,4,0,0]=(0.924146253016)/(o+1.)+(-0.434307664887)
        ref[1,4,0,1]=(-0.221754040568)/(o+1.)+(1.14558078503)
        ref[1,4,1,0]=(0.0145217628367)/(o+1.)+(-0.43084890137)
        ref[1,4,1,1]=(-1.31753215008)/(o+1.)+(-0.824287026036)
        ref[1,4,2,0]=(-0.697722948118)/(o+1.)+(0.114574044827)
        ref[1,4,2,1]=(0.779339818147)/(o+1.)+(-0.365592981116)
        ref[2,0,0,0]=(0.274952897958)/(o+1.)+(-0.255370912722)
        ref[2,0,0,1]=(-1.10264655401)/(o+1.)+(-0.261708001871)
        ref[2,0,1,0]=(-0.680012578665)/(o+1.)+(-0.106715000464)
        ref[2,0,1,1]=(-0.686793121159)/(o+1.)+(-0.0241878370688)
        ref[2,0,2,0]=(0.942715771856)/(o+1.)+(-0.323717638282)
        ref[2,0,2,1]=(-0.0104862575713)/(o+1.)+(-0.19048054226)
        ref[2,1,0,0]=(0.746926399818)/(o+1.)+(0.117160499248)
        ref[2,1,0,1]=(-1.04171529373)/(o+1.)+(0.0976704487401)
        ref[2,1,1,0]=(0.136795029052)/(o+1.)+(-0.0802084471266)
        ref[2,1,1,1]=(0.896020560833)/(o+1.)+(-0.836960327706)
        ref[2,1,2,0]=(-0.00904194364113)/(o+1.)+(0.245943197498)
        ref[2,1,2,1]=(1.1485457441)/(o+1.)+(0.145957400558)
        ref[2,2,0,0]=(-0.0707799881243)/(o+1.)+(0.0690663884416)
        ref[2,2,0,1]=(1.32117006716)/(o+1.)+(0.227104957348)
        ref[2,2,1,0]=(0.711880376874)/(o+1.)+(-0.58130282817)
        ref[2,2,1,1]=(-0.611981014504)/(o+1.)+(0.303368043702)
        ref[2,2,2,0]=(0.564606893452)/(o+1.)+(-0.674927313599)
        ref[2,2,2,1]=(0.754537066511)/(o+1.)+(0.654108380069)
        ref[2,3,0,0]=(0.168469433796)/(o+1.)+(0.331189805903)
        ref[2,3,0,1]=(1.58905823122)/(o+1.)+(0.262836520673)
        ref[2,3,1,0]=(-0.556033290604)/(o+1.)+(-0.337520298949)
        ref[2,3,1,1]=(-1.23431436368)/(o+1.)+(-0.439901936041)
        ref[2,3,2,0]=(0.579845244941)/(o+1.)+(-0.0789272197338)
        ref[2,3,2,1]=(0.417029829778)/(o+1.)+(-0.569083876513)
        ref[2,4,0,0]=(-0.263493125943)/(o+1.)+(-0.515826713248)
        ref[2,4,0,1]=(1.65454401707)/(o+1.)+(-0.124092576105)
        ref[2,4,1,0]=(-0.908471480882)/(o+1.)+(-0.553878545)
        ref[2,4,1,1]=(1.13303102108)/(o+1.)+(-0.31187616743)
        ref[2,4,2,0]=(0.255567306363)/(o+1.)+(-0.738450155401)
        ref[2,4,2,1]=(0.900548855936)/(o+1.)+(0.175714903622)
        ref[3,0,0,0]=(-0.517363884224)/(o+1.)+(-0.614436783326)
        ref[3,0,0,1]=(-1.104800959)/(o+1.)+(0.564900619957)
        ref[3,0,1,0]=(0.730685024108)/(o+1.)+(0.398403732404)
        ref[3,0,1,1]=(-0.521524324846)/(o+1.)+(-0.863575427496)
        ref[3,0,2,0]=(1.31118959399)/(o+1.)+(-0.534840488515)
        ref[3,0,2,1]=(0.725678172925)/(o+1.)+(0.404339210814)
        ref[3,1,0,0]=(2.49858502561)/(o+1.)+(0.116580318822)
        ref[3,1,0,1]=(-0.640270473588)/(o+1.)+(-0.561704601771)
        ref[3,1,1,0]=(-0.790440585902)/(o+1.)+(-0.190230620716)
        ref[3,1,1,1]=(0.0678464418036)/(o+1.)+(0.104572102332)
        ref[3,1,2,0]=(0.708751970646)/(o+1.)+(-0.553728293749)
        ref[3,1,2,1]=(-2.28767897057)/(o+1.)+(1.03103581162)
        ref[3,2,0,0]=(1.61595319744)/(o+1.)+(0.0510876372718)
        ref[3,2,0,1]=(-0.0135357603866)/(o+1.)+(0.137621005039)
        ref[3,2,1,0]=(-1.00492518488)/(o+1.)+(0.396661482686)
        ref[3,2,1,1]=(-0.320901927851)/(o+1.)+(-0.463339233997)
        ref[3,2,2,0]=(2.43552217291)/(o+1.)+(0.621781827577)
        ref[3,2,2,1]=(1.41741894674)/(o+1.)+(0.804378003359)
        ref[3,3,0,0]=(0.748611138576)/(o+1.)+(-0.914035304315)
        ref[3,3,0,1]=(-0.375012840487)/(o+1.)+(0.0433116399723)
        ref[3,3,1,0]=(-0.0102160036467)/(o+1.)+(0.403000755649)
        ref[3,3,1,1]=(2.10162209317)/(o+1.)+(0.888826297208)
        ref[3,3,2,0]=(-0.449082281464)/(o+1.)+(0.521787043713)
        ref[3,3,2,1]=(-0.0103979620079)/(o+1.)+(-0.372545450994)
        ref[3,4,0,0]=(0.490562575696)/(o+1.)+(0.634809326777)
        ref[3,4,0,1]=(1.18954580773)/(o+1.)+(0.633837577475)
        ref[3,4,1,0]=(1.01376106573)/(o+1.)+(-0.301232420589)
        ref[3,4,1,1]=(-1.77747335429)/(o+1.)+(-0.100059050648)
        ref[3,4,2,0]=(1.44409584476)/(o+1.)+(-0.579358374008)
        ref[3,4,2,1]=(-1.78484029887)/(o+1.)+(0.0406454658476)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5, 3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunction_fromData_Solution_rank0(self):
      """
      tests integral of rank 0 Data on the Function

      assumptions: Solution(self.domain) exists
                   self.domain supports integral on Function
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=Function(self.domain)
      w=Solution(self.domain)
      x=w.getX()
      arg=Data(0,(),w)
      if dim==2:
        arg=(-0.788336609161)*x[0]**o+(-0.46539601764)*x[0]+(0.0970136798824)*x[1]**o+(-0.120665298671)*x[1]
        ref=(-0.691322929278)/(o+1.)+(-0.293030658156)
      else:
        arg=(0.922255261003)*x[0]**o+(0.0862232364707)*x[0]+(-0.94152635039)*x[1]**o+(0.771078486789)*x[1]+(-0.487642191181)*x[2]**o+(0.971754960362)*x[2]
        ref=(-0.506913280568)/(o+1.)+(0.914528341811)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunction_fromData_Solution_rank1(self):
      """
      tests integral of rank 1 Data on the Function

      assumptions: Solution(self.domain) exists
                   self.domain supports integral on Function
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=Function(self.domain)
      w=Solution(self.domain)
      x=w.getX()
      arg=Data(0,(2,),w)
      ref=numpy.zeros((2,),numpy.float_)
      if dim==2:
        arg[0]=(-0.193071015033)*x[0]**o+(-0.281957941612)*x[0]+(0.269334552863)*x[1]**o+(0.0332262192804)*x[1]
        arg[1]=(-0.982634724425)*x[0]**o+(0.872638820737)*x[0]+(0.828028964603)*x[1]**o+(0.70464418136)*x[1]
        ref[0]=(0.0762635378296)/(o+1.)+(-0.124365861166)
        ref[1]=(-0.154605759822)/(o+1.)+(0.788641501048)
      else:
        arg[0]=(0.0746983082972)*x[0]**o+(0.281818436343)*x[0]+(-0.835321029206)*x[1]**o+(-0.349524270524)*x[1]+(-0.471905088601)*x[2]**o+(0.156327167402)*x[2]
        arg[1]=(-0.971935771255)*x[0]**o+(0.943151542057)*x[0]+(-0.60084258169)*x[1]**o+(0.458392855527)*x[1]+(-0.0173384187065)*x[2]**o+(0.538410338758)*x[2]
        ref[0]=(-1.23252780951)/(o+1.)+(0.0443106666103)
        ref[1]=(-1.59011677165)/(o+1.)+(0.969977368171)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunction_fromData_Solution_rank2(self):
      """
      tests integral of rank 2 Data on the Function

      assumptions: Solution(self.domain) exists
                   self.domain supports integral on Function
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=Function(self.domain)
      w=Solution(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5),w)
      ref=numpy.zeros((4, 5),numpy.float_)
      if dim==2:
        arg[0,0]=(0.568783264759)*x[0]**o+(0.690377280868)*x[0]+(-0.689750922806)*x[1]**o+(-0.262828324126)*x[1]
        arg[0,1]=(-0.981657284673)*x[0]**o+(0.37118874852)*x[0]+(0.733992104146)*x[1]**o+(0.350268735699)*x[1]
        arg[0,2]=(0.713336100765)*x[0]**o+(0.737674447814)*x[0]+(0.239606043833)*x[1]**o+(0.814911366356)*x[1]
        arg[0,3]=(0.171706130967)*x[0]**o+(0.56324398491)*x[0]+(0.328884588679)*x[1]**o+(0.13307521808)*x[1]
        arg[0,4]=(-0.309130885049)*x[0]**o+(0.353596646674)*x[0]+(-0.282764686556)*x[1]**o+(-0.56977508596)*x[1]
        arg[1,0]=(0.974136161109)*x[0]**o+(-0.432477120938)*x[0]+(-0.841212164751)*x[1]**o+(-0.429211937256)*x[1]
        arg[1,1]=(0.0303890146585)*x[0]**o+(-0.712865964696)*x[0]+(0.572471431129)*x[1]**o+(-0.194324510922)*x[1]
        arg[1,2]=(-0.757134870981)*x[0]**o+(-0.018314520594)*x[0]+(0.250423290169)*x[1]**o+(-0.876678823729)*x[1]
        arg[1,3]=(-0.351885929068)*x[0]**o+(-0.561258605321)*x[0]+(0.208995575986)*x[1]**o+(0.428437856056)*x[1]
        arg[1,4]=(-0.313383912526)*x[0]**o+(-0.713484404328)*x[0]+(-0.279619274966)*x[1]**o+(0.72420661494)*x[1]
        arg[2,0]=(-0.686382641827)*x[0]**o+(0.678578408675)*x[0]+(-0.696532353786)*x[1]**o+(-0.154644995634)*x[1]
        arg[2,1]=(0.368548759951)*x[0]**o+(-0.674856247769)*x[0]+(0.908734263899)*x[1]**o+(0.331173572353)*x[1]
        arg[2,2]=(-0.1279136556)*x[0]**o+(-0.161474840876)*x[0]+(-0.775009015881)*x[1]**o+(-0.689552100815)*x[1]
        arg[2,3]=(-0.638450048485)*x[0]**o+(0.755485009392)*x[0]+(-0.793363159431)*x[1]**o+(0.255883452369)*x[1]
        arg[2,4]=(-0.831891512113)*x[0]**o+(-0.811678751836)*x[0]+(-0.487866621558)*x[1]**o+(0.240951890349)*x[1]
        arg[3,0]=(0.448249735424)*x[0]**o+(0.0809467993259)*x[0]+(0.232269153692)*x[1]**o+(0.477512628661)*x[1]
        arg[3,1]=(0.494093407104)*x[0]**o+(0.196761586998)*x[0]+(-0.702155296644)*x[1]**o+(-0.748641693731)*x[1]
        arg[3,2]=(0.432370067164)*x[0]**o+(-0.365671005874)*x[0]+(0.452009309356)*x[1]**o+(-0.951328172558)*x[1]
        arg[3,3]=(-0.993290240028)*x[0]**o+(0.172534714933)*x[0]+(0.00917693958413)*x[1]**o+(0.417515657393)*x[1]
        arg[3,4]=(0.541299019375)*x[0]**o+(0.479118172057)*x[0]+(-0.502475287154)*x[1]**o+(-0.04397899716)*x[1]
        ref[0,0]=(-0.120967658047)/(o+1.)+(0.213774478371)
        ref[0,1]=(-0.247665180527)/(o+1.)+(0.360728742109)
        ref[0,2]=(0.952942144598)/(o+1.)+(0.776292907085)
        ref[0,3]=(0.500590719646)/(o+1.)+(0.348159601495)
        ref[0,4]=(-0.591895571605)/(o+1.)+(-0.108089219643)
        ref[1,0]=(0.132923996358)/(o+1.)+(-0.430844529097)
        ref[1,1]=(0.602860445788)/(o+1.)+(-0.453595237809)
        ref[1,2]=(-0.506711580812)/(o+1.)+(-0.447496672161)
        ref[1,3]=(-0.142890353081)/(o+1.)+(-0.066410374632)
        ref[1,4]=(-0.593003187492)/(o+1.)+(0.00536110530621)
        ref[2,0]=(-1.38291499561)/(o+1.)+(0.261966706521)
        ref[2,1]=(1.27728302385)/(o+1.)+(-0.171841337708)
        ref[2,2]=(-0.902922671481)/(o+1.)+(-0.425513470846)
        ref[2,3]=(-1.43181320792)/(o+1.)+(0.505684230881)
        ref[2,4]=(-1.31975813367)/(o+1.)+(-0.285363430743)
        ref[3,0]=(0.680518889117)/(o+1.)+(0.279229713993)
        ref[3,1]=(-0.20806188954)/(o+1.)+(-0.275940053366)
        ref[3,2]=(0.88437937652)/(o+1.)+(-0.658499589216)
        ref[3,3]=(-0.984113300443)/(o+1.)+(0.295025186163)
        ref[3,4]=(0.0388237322207)/(o+1.)+(0.217569587448)
      else:
        arg[0,0]=(-0.866239193113)*x[0]**o+(0.479559390215)*x[0]+(0.512920648125)*x[1]**o+(-0.504775214511)*x[1]+(0.462210441279)*x[2]**o+(0.000249131516168)*x[2]
        arg[0,1]=(-0.11397206495)*x[0]**o+(0.0762933475692)*x[0]+(0.347716642793)*x[1]**o+(0.373262578958)*x[1]+(0.965468442129)*x[2]**o+(-0.743205128452)*x[2]
        arg[0,2]=(-0.214519408107)*x[0]**o+(-0.663675446512)*x[0]+(-0.424102625804)*x[1]**o+(-0.426644021856)*x[1]+(-0.0731662128704)*x[2]**o+(-0.375185307096)*x[2]
        arg[0,3]=(0.599595484989)*x[0]**o+(-0.303187098474)*x[0]+(0.278524134467)*x[1]**o+(-0.579432425528)*x[1]+(0.006698672684)*x[2]**o+(-0.820721120517)*x[2]
        arg[0,4]=(0.538847126305)*x[0]**o+(0.58093792853)*x[0]+(-0.21189097773)*x[1]**o+(0.637101424777)*x[1]+(0.0155994636183)*x[2]**o+(0.488675370945)*x[2]
        arg[1,0]=(-0.775581481064)*x[0]**o+(-0.968381856562)*x[0]+(-0.660755920498)*x[1]**o+(0.746481830633)*x[1]+(-0.300061615221)*x[2]**o+(-0.720031101537)*x[2]
        arg[1,1]=(0.990001351802)*x[0]**o+(-0.168997300944)*x[0]+(0.0134555819367)*x[1]**o+(0.969355591009)*x[1]+(-0.347553438064)*x[2]**o+(0.648537935984)*x[2]
        arg[1,2]=(-0.327760535516)*x[0]**o+(0.517489199537)*x[0]+(-0.0589046196438)*x[1]**o+(-0.893401439507)*x[1]+(-0.0821634552571)*x[2]**o+(0.861266440343)*x[2]
        arg[1,3]=(-0.808782893446)*x[0]**o+(-0.196363777506)*x[0]+(0.623837845847)*x[1]**o+(-0.05392424847)*x[1]+(0.616011657554)*x[2]**o+(0.800682023434)*x[2]
        arg[1,4]=(-0.425073222968)*x[0]**o+(-0.635248695204)*x[0]+(0.947222621311)*x[1]**o+(0.253198813179)*x[1]+(-0.907568459712)*x[2]**o+(-0.513049744062)*x[2]
        arg[2,0]=(-0.103867107528)*x[0]**o+(0.893031214224)*x[0]+(0.151292631908)*x[1]**o+(-0.407189878657)*x[1]+(0.618120612304)*x[2]**o+(0.983992964931)*x[2]
        arg[2,1]=(0.310432646627)*x[0]**o+(0.235143842057)*x[0]+(0.729683627093)*x[1]**o+(0.891778928523)*x[1]+(-0.148927367367)*x[2]**o+(0.913403960844)*x[2]
        arg[2,2]=(0.610293692912)*x[0]**o+(-0.28497731539)*x[0]+(-0.926462472232)*x[1]**o+(-0.427657211191)*x[1]+(0.467905160751)*x[2]**o+(0.195578740767)*x[2]
        arg[2,3]=(0.854869452114)*x[0]**o+(-0.752530632831)*x[0]+(0.0128220565485)*x[1]**o+(-0.726013651836)*x[1]+(-0.0979888078519)*x[2]**o+(-0.658955703807)*x[2]
        arg[2,4]=(0.536447848289)*x[0]**o+(-0.658205406579)*x[0]+(0.66427484293)*x[1]**o+(-0.250014589496)*x[1]+(-0.0654257159073)*x[2]**o+(0.538196496913)*x[2]
        arg[3,0]=(-0.514665034974)*x[0]**o+(-0.278189747423)*x[0]+(0.402996329453)*x[1]**o+(-0.0362969716175)*x[1]+(-0.707890815004)*x[2]**o+(-0.298762865096)*x[2]
        arg[3,1]=(0.7342439327)*x[0]**o+(-0.387336714116)*x[0]+(-0.67888634151)*x[1]**o+(0.647322640846)*x[1]+(-0.0346274738413)*x[2]**o+(0.985950301913)*x[2]
        arg[3,2]=(0.687555064124)*x[0]**o+(0.345506248144)*x[0]+(0.658692258064)*x[1]**o+(0.810796031447)*x[1]+(0.102608006398)*x[2]**o+(0.349877767777)*x[2]
        arg[3,3]=(-0.456241806837)*x[0]**o+(-0.181224535865)*x[0]+(0.401842456169)*x[1]**o+(-0.413936620519)*x[1]+(0.436794604974)*x[2]**o+(0.95051042082)*x[2]
        arg[3,4]=(-0.330739393408)*x[0]**o+(-0.261876658969)*x[0]+(0.287474017623)*x[1]**o+(0.463691723118)*x[1]+(0.757597823206)*x[2]**o+(0.49336246207)*x[2]
        ref[0,0]=(0.108891896292)/(o+1.)+(-0.01248334639)
        ref[0,1]=(1.19921301997)/(o+1.)+(-0.146824600962)
        ref[0,2]=(-0.711788246781)/(o+1.)+(-0.732752387732)
        ref[0,3]=(0.88481829214)/(o+1.)+(-0.85167032226)
        ref[0,4]=(0.342555612194)/(o+1.)+(0.853357362126)
        ref[1,0]=(-1.73639901678)/(o+1.)+(-0.470965563733)
        ref[1,1]=(0.655903495675)/(o+1.)+(0.724448113024)
        ref[1,2]=(-0.468828610417)/(o+1.)+(0.242677100186)
        ref[1,3]=(0.431066609955)/(o+1.)+(0.275196998729)
        ref[1,4]=(-0.385419061368)/(o+1.)+(-0.447549813044)
        ref[2,0]=(0.665546136685)/(o+1.)+(0.734917150249)
        ref[2,1]=(0.891188906354)/(o+1.)+(1.02016336571)
        ref[2,2]=(0.151736381432)/(o+1.)+(-0.258527892907)
        ref[2,3]=(0.76970270081)/(o+1.)+(-1.06874999424)
        ref[2,4]=(1.13529697531)/(o+1.)+(-0.185011749581)
        ref[3,0]=(-0.819559520525)/(o+1.)+(-0.306624792068)
        ref[3,1]=(0.020730117348)/(o+1.)+(0.622968114321)
        ref[3,2]=(1.44885532859)/(o+1.)+(0.753090023684)
        ref[3,3]=(0.382395254306)/(o+1.)+(0.177674632218)
        ref[3,4]=(0.714332447421)/(o+1.)+(0.34758876311)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunction_fromData_Solution_rank3(self):
      """
      tests integral of rank 3 Data on the Function

      assumptions: Solution(self.domain) exists
                   self.domain supports integral on Function
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=Function(self.domain)
      w=Solution(self.domain)
      x=w.getX()
      arg=Data(0,(6, 2, 2),w)
      ref=numpy.zeros((6, 2, 2),numpy.float_)
      if dim==2:
        arg[0,0,0]=(0.779261125448)*x[0]**o+(-0.640503838419)*x[0]+(0.589085035572)*x[1]**o+(-0.0426687544445)*x[1]
        arg[0,0,1]=(-0.419361292473)*x[0]**o+(0.0850684591011)*x[0]+(-0.440758336688)*x[1]**o+(0.151015611594)*x[1]
        arg[0,1,0]=(-0.727043966709)*x[0]**o+(0.711730063365)*x[0]+(0.657759819512)*x[1]**o+(-0.210981156609)*x[1]
        arg[0,1,1]=(-0.276491768644)*x[0]**o+(-0.324865684188)*x[0]+(-0.921856877627)*x[1]**o+(-0.246833902926)*x[1]
        arg[1,0,0]=(0.720541615152)*x[0]**o+(-0.0406851820986)*x[0]+(0.33976001267)*x[1]**o+(0.937989079378)*x[1]
        arg[1,0,1]=(0.239272962875)*x[0]**o+(0.482026352856)*x[0]+(0.159148474482)*x[1]**o+(0.680831564145)*x[1]
        arg[1,1,0]=(0.55170300456)*x[0]**o+(-0.261357073238)*x[0]+(0.298185649596)*x[1]**o+(-0.540598468483)*x[1]
        arg[1,1,1]=(0.508262701362)*x[0]**o+(-0.531692643817)*x[0]+(0.151424571021)*x[1]**o+(-0.77488755481)*x[1]
        arg[2,0,0]=(0.265201389846)*x[0]**o+(0.377300848247)*x[0]+(0.0402148688087)*x[1]**o+(0.0263806695524)*x[1]
        arg[2,0,1]=(0.321397565411)*x[0]**o+(-0.277650913698)*x[0]+(0.909430556637)*x[1]**o+(-0.43244155765)*x[1]
        arg[2,1,0]=(-0.671417887851)*x[0]**o+(-0.772328065901)*x[0]+(0.0896878386442)*x[1]**o+(0.584773963567)*x[1]
        arg[2,1,1]=(0.318995094412)*x[0]**o+(-0.396578444213)*x[0]+(-0.627767362562)*x[1]**o+(-0.588717342984)*x[1]
        arg[3,0,0]=(-0.154999830633)*x[0]**o+(-0.263434767944)*x[0]+(0.0704862479101)*x[1]**o+(0.216237505495)*x[1]
        arg[3,0,1]=(0.956868643095)*x[0]**o+(-0.0331644426873)*x[0]+(-0.236340025853)*x[1]**o+(-0.814705425837)*x[1]
        arg[3,1,0]=(-0.477985064084)*x[0]**o+(-0.368896833289)*x[0]+(0.208122873207)*x[1]**o+(-0.566525593432)*x[1]
        arg[3,1,1]=(-0.299257275248)*x[0]**o+(0.937022356746)*x[0]+(0.671724793932)*x[1]**o+(0.80705820341)*x[1]
        arg[4,0,0]=(0.857855249735)*x[0]**o+(0.537756541143)*x[0]+(0.611784225987)*x[1]**o+(0.944061991186)*x[1]
        arg[4,0,1]=(0.825779290308)*x[0]**o+(-0.636461636739)*x[0]+(-0.382007450888)*x[1]**o+(0.170178262303)*x[1]
        arg[4,1,0]=(0.275666857657)*x[0]**o+(-0.404983003194)*x[0]+(0.476932065223)*x[1]**o+(0.931645726493)*x[1]
        arg[4,1,1]=(0.956182179652)*x[0]**o+(0.543482655321)*x[0]+(0.60492494312)*x[1]**o+(0.384497838341)*x[1]
        arg[5,0,0]=(0.476937368598)*x[0]**o+(-0.588434801718)*x[0]+(0.865080125217)*x[1]**o+(-0.954493375041)*x[1]
        arg[5,0,1]=(0.151850214007)*x[0]**o+(0.252651124764)*x[0]+(-0.570132867968)*x[1]**o+(-0.0885902498736)*x[1]
        arg[5,1,0]=(-0.69330471578)*x[0]**o+(0.165874518143)*x[0]+(-0.610503496045)*x[1]**o+(-0.871951964138)*x[1]
        arg[5,1,1]=(-0.539191264945)*x[0]**o+(0.558474736927)*x[0]+(0.752045569996)*x[1]**o+(-0.5305308467)*x[1]
        ref[0,0,0]=(1.36834616102)/(o+1.)+(-0.341586296432)
        ref[0,0,1]=(-0.860119629161)/(o+1.)+(0.118042035348)
        ref[0,1,0]=(-0.0692841471964)/(o+1.)+(0.250374453378)
        ref[0,1,1]=(-1.19834864627)/(o+1.)+(-0.285849793557)
        ref[1,0,0]=(1.06030162782)/(o+1.)+(0.448651948639)
        ref[1,0,1]=(0.398421437357)/(o+1.)+(0.5814289585)
        ref[1,1,0]=(0.849888654155)/(o+1.)+(-0.400977770861)
        ref[1,1,1]=(0.659687272383)/(o+1.)+(-0.653290099314)
        ref[2,0,0]=(0.305416258655)/(o+1.)+(0.2018407589)
        ref[2,0,1]=(1.23082812205)/(o+1.)+(-0.355046235674)
        ref[2,1,0]=(-0.581730049206)/(o+1.)+(-0.0937770511668)
        ref[2,1,1]=(-0.30877226815)/(o+1.)+(-0.492647893598)
        ref[3,0,0]=(-0.0845135827233)/(o+1.)+(-0.0235986312242)
        ref[3,0,1]=(0.720528617241)/(o+1.)+(-0.423934934262)
        ref[3,1,0]=(-0.269862190877)/(o+1.)+(-0.46771121336)
        ref[3,1,1]=(0.372467518684)/(o+1.)+(0.872040280078)
        ref[4,0,0]=(1.46963947572)/(o+1.)+(0.740909266165)
        ref[4,0,1]=(0.44377183942)/(o+1.)+(-0.233141687218)
        ref[4,1,0]=(0.75259892288)/(o+1.)+(0.26333136165)
        ref[4,1,1]=(1.56110712277)/(o+1.)+(0.463990246831)
        ref[5,0,0]=(1.34201749382)/(o+1.)+(-0.771464088379)
        ref[5,0,1]=(-0.418282653961)/(o+1.)+(0.0820304374451)
        ref[5,1,0]=(-1.30380821182)/(o+1.)+(-0.353038722997)
        ref[5,1,1]=(0.212854305051)/(o+1.)+(0.0139719451134)
      else:
        arg[0,0,0]=(-0.825926001774)*x[0]**o+(0.418049316138)*x[0]+(0.668695279849)*x[1]**o+(-0.7323605258)*x[1]+(-0.637717742224)*x[2]**o+(0.257065974143)*x[2]
        arg[0,0,1]=(-0.169779448443)*x[0]**o+(-0.648612931029)*x[0]+(-0.643814352995)*x[1]**o+(0.396704595574)*x[1]+(-0.248766540261)*x[2]**o+(-0.736410673282)*x[2]
        arg[0,1,0]=(-0.371929441192)*x[0]**o+(0.438661796459)*x[0]+(-0.893450426264)*x[1]**o+(0.643494077636)*x[1]+(0.252099102122)*x[2]**o+(-0.0857798075076)*x[2]
        arg[0,1,1]=(0.336251711034)*x[0]**o+(-0.317426111079)*x[0]+(-0.801221986791)*x[1]**o+(0.281123033042)*x[1]+(0.921153543163)*x[2]**o+(0.497003835923)*x[2]
        arg[1,0,0]=(0.40897311872)*x[0]**o+(0.326894801733)*x[0]+(0.216268807497)*x[1]**o+(0.946235364974)*x[1]+(-0.480995485096)*x[2]**o+(0.869772937982)*x[2]
        arg[1,0,1]=(0.108715554449)*x[0]**o+(0.0847529231253)*x[0]+(0.866120965844)*x[1]**o+(0.43989772616)*x[1]+(0.67962061792)*x[2]**o+(0.90311988632)*x[2]
        arg[1,1,0]=(0.28571484173)*x[0]**o+(0.593050819853)*x[0]+(-0.384864010146)*x[1]**o+(-0.57920566259)*x[1]+(-0.304024223769)*x[2]**o+(-0.0279518764989)*x[2]
        arg[1,1,1]=(0.249126658892)*x[0]**o+(0.243355410215)*x[0]+(0.154816978135)*x[1]**o+(-0.260343533153)*x[1]+(-0.973133787852)*x[2]**o+(0.752545508531)*x[2]
        arg[2,0,0]=(0.103612497259)*x[0]**o+(0.0919234144108)*x[0]+(0.416394965635)*x[1]**o+(-0.0132359939268)*x[1]+(-0.522775855178)*x[2]**o+(-0.985811066906)*x[2]
        arg[2,0,1]=(-0.592641488144)*x[0]**o+(0.703703144485)*x[0]+(-0.0149046721584)*x[1]**o+(0.220368283581)*x[1]+(-0.656858164281)*x[2]**o+(0.165951102113)*x[2]
        arg[2,1,0]=(0.309492848012)*x[0]**o+(-0.301037623847)*x[0]+(-0.224814259669)*x[1]**o+(-0.950794406688)*x[1]+(-0.296088293363)*x[2]**o+(-0.712930293168)*x[2]
        arg[2,1,1]=(0.0592633594166)*x[0]**o+(-0.179007619512)*x[0]+(-0.922169690831)*x[1]**o+(-0.778228075089)*x[1]+(-0.135749453777)*x[2]**o+(-0.146265051618)*x[2]
        arg[3,0,0]=(-0.71773633584)*x[0]**o+(0.471551385738)*x[0]+(-0.041415831943)*x[1]**o+(0.343380640343)*x[1]+(-0.211821446187)*x[2]**o+(-0.661201893183)*x[2]
        arg[3,0,1]=(-0.159071045836)*x[0]**o+(0.216715127143)*x[0]+(-0.846847485213)*x[1]**o+(0.662226888693)*x[1]+(0.050806337433)*x[2]**o+(0.744071347051)*x[2]
        arg[3,1,0]=(-0.363431877105)*x[0]**o+(-0.461513470653)*x[0]+(-0.130947882739)*x[1]**o+(-0.624729599981)*x[1]+(0.778572567699)*x[2]**o+(0.324584316744)*x[2]
        arg[3,1,1]=(0.395432262346)*x[0]**o+(0.826450301981)*x[0]+(0.176721657975)*x[1]**o+(-0.370954952405)*x[1]+(-0.708437370609)*x[2]**o+(-0.150075114354)*x[2]
        arg[4,0,0]=(-0.18662583335)*x[0]**o+(-0.0120831234457)*x[0]+(0.805686139762)*x[1]**o+(0.30195413136)*x[1]+(0.648067854031)*x[2]**o+(0.28073240632)*x[2]
        arg[4,0,1]=(-0.913625644854)*x[0]**o+(-0.264637827098)*x[0]+(-0.6531590441)*x[1]**o+(0.396422722623)*x[1]+(0.280591271146)*x[2]**o+(0.498346191707)*x[2]
        arg[4,1,0]=(0.208029597907)*x[0]**o+(-0.295747909396)*x[0]+(-0.541509895186)*x[1]**o+(-0.511394562575)*x[1]+(-0.459197562982)*x[2]**o+(-0.749433614666)*x[2]
        arg[4,1,1]=(-0.29700028905)*x[0]**o+(-0.590932598278)*x[0]+(0.36219700213)*x[1]**o+(0.482759242709)*x[1]+(-0.135696240971)*x[2]**o+(0.49658807849)*x[2]
        arg[5,0,0]=(-0.800345624219)*x[0]**o+(0.259709746207)*x[0]+(0.480073493816)*x[1]**o+(0.499332550127)*x[1]+(0.691794186846)*x[2]**o+(-0.319369042249)*x[2]
        arg[5,0,1]=(0.300879052118)*x[0]**o+(-0.985396817441)*x[0]+(0.971136879002)*x[1]**o+(0.806560091745)*x[1]+(0.361308859099)*x[2]**o+(0.264277089732)*x[2]
        arg[5,1,0]=(0.271857906548)*x[0]**o+(0.338260411675)*x[0]+(-0.170292665803)*x[1]**o+(0.675491566142)*x[1]+(0.897921755421)*x[2]**o+(-0.829233864443)*x[2]
        arg[5,1,1]=(0.673366845241)*x[0]**o+(0.121940550422)*x[0]+(0.116054155545)*x[1]**o+(0.0456556938689)*x[1]+(-0.978127590673)*x[2]**o+(0.180985351995)*x[2]
        ref[0,0,0]=(-0.794948464148)/(o+1.)+(-0.0286226177598)
        ref[0,0,1]=(-1.0623603417)/(o+1.)+(-0.494159504368)
        ref[0,1,0]=(-1.01328076533)/(o+1.)+(0.498188033294)
        ref[0,1,1]=(0.456183267406)/(o+1.)+(0.230350378943)
        ref[1,0,0]=(0.144246441122)/(o+1.)+(1.07145155234)
        ref[1,0,1]=(1.65445713821)/(o+1.)+(0.713885267803)
        ref[1,1,0]=(-0.403173392185)/(o+1.)+(-0.0070533596178)
        ref[1,1,1]=(-0.569190150825)/(o+1.)+(0.367778692797)
        ref[2,0,0]=(-0.00276839228366)/(o+1.)+(-0.453561823211)
        ref[2,0,1]=(-1.26440432458)/(o+1.)+(0.545011265089)
        ref[2,1,0]=(-0.21140970502)/(o+1.)+(-0.982381161851)
        ref[2,1,1]=(-0.998655785191)/(o+1.)+(-0.55175037311)
        ref[3,0,0]=(-0.97097361397)/(o+1.)+(0.0768650664488)
        ref[3,0,1]=(-0.955112193616)/(o+1.)+(0.811506681444)
        ref[3,1,0]=(0.284192807855)/(o+1.)+(-0.380829376946)
        ref[3,1,1]=(-0.136283450288)/(o+1.)+(0.152710117611)
        ref[4,0,0]=(1.26712816044)/(o+1.)+(0.285301707117)
        ref[4,0,1]=(-1.28619341781)/(o+1.)+(0.315065543616)
        ref[4,1,0]=(-0.792677860261)/(o+1.)+(-0.778288043318)
        ref[4,1,1]=(-0.0704995278906)/(o+1.)+(0.194207361461)
        ref[5,0,0]=(0.371522056443)/(o+1.)+(0.219836627043)
        ref[5,0,1]=(1.63332479022)/(o+1.)+(0.0427201820182)
        ref[5,1,0]=(0.999486996166)/(o+1.)+(0.0922590566873)
        ref[5,1,1]=(-0.188706589887)/(o+1.)+(0.174290798143)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunction_fromData_Solution_rank4(self):
      """
      tests integral of rank 4 Data on the Function

      assumptions: Solution(self.domain) exists
                   self.domain supports integral on Function
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=Function(self.domain)
      w=Solution(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5, 3, 2),w)
      ref=numpy.zeros((4, 5, 3, 2),numpy.float_)
      if dim==2:
        arg[0,0,0,0]=(0.647932589381)*x[0]**o+(-0.092686685918)*x[0]+(0.167888464541)*x[1]**o+(0.698445840627)*x[1]
        arg[0,0,0,1]=(0.413204959658)*x[0]**o+(0.772926029512)*x[0]+(-0.63702948026)*x[1]**o+(0.608392823861)*x[1]
        arg[0,0,1,0]=(0.303506932052)*x[0]**o+(0.650309664063)*x[0]+(-0.805238344485)*x[1]**o+(-0.62971680995)*x[1]
        arg[0,0,1,1]=(-0.205701190736)*x[0]**o+(-0.0834042840814)*x[0]+(-0.525839398431)*x[1]**o+(0.779973223005)*x[1]
        arg[0,0,2,0]=(-0.97954633631)*x[0]**o+(-0.298973871114)*x[0]+(0.858022652298)*x[1]**o+(0.667671796632)*x[1]
        arg[0,0,2,1]=(-0.790133712345)*x[0]**o+(-0.733743414808)*x[0]+(0.337185148145)*x[1]**o+(-0.927853051244)*x[1]
        arg[0,1,0,0]=(0.617198112458)*x[0]**o+(0.70438849416)*x[0]+(-0.861661596364)*x[1]**o+(-0.251796116219)*x[1]
        arg[0,1,0,1]=(0.37107453636)*x[0]**o+(-0.490440661153)*x[0]+(0.849252675242)*x[1]**o+(-0.896547775077)*x[1]
        arg[0,1,1,0]=(0.375080744007)*x[0]**o+(-0.609557663503)*x[0]+(-0.930846980977)*x[1]**o+(0.0623096843381)*x[1]
        arg[0,1,1,1]=(-0.809863615805)*x[0]**o+(-0.999724521803)*x[0]+(0.493027030633)*x[1]**o+(-0.672546253752)*x[1]
        arg[0,1,2,0]=(0.219294077412)*x[0]**o+(-0.7854819847)*x[0]+(-0.353026321798)*x[1]**o+(-0.965348418509)*x[1]
        arg[0,1,2,1]=(0.177236321722)*x[0]**o+(-0.860491592895)*x[0]+(-0.458724268899)*x[1]**o+(-0.476508229653)*x[1]
        arg[0,2,0,0]=(0.0760611767074)*x[0]**o+(-0.00296398964042)*x[0]+(-0.972843039436)*x[1]**o+(-0.0422057743881)*x[1]
        arg[0,2,0,1]=(-0.926737777701)*x[0]**o+(-0.895791593378)*x[0]+(-0.776529150964)*x[1]**o+(0.557115706978)*x[1]
        arg[0,2,1,0]=(0.353264702858)*x[0]**o+(0.632583933138)*x[0]+(-0.354112684469)*x[1]**o+(0.0541346174117)*x[1]
        arg[0,2,1,1]=(-0.124181989217)*x[0]**o+(-0.803262099858)*x[0]+(0.733246361984)*x[1]**o+(0.0639821744038)*x[1]
        arg[0,2,2,0]=(-0.446733845136)*x[0]**o+(-0.187373171381)*x[0]+(-0.312309043572)*x[1]**o+(0.830789380477)*x[1]
        arg[0,2,2,1]=(0.403889932995)*x[0]**o+(-0.243474625233)*x[0]+(0.35707779102)*x[1]**o+(0.29725421988)*x[1]
        arg[0,3,0,0]=(-0.856376546696)*x[0]**o+(0.448431504377)*x[0]+(-0.535187100108)*x[1]**o+(-0.90994242894)*x[1]
        arg[0,3,0,1]=(-0.321475754775)*x[0]**o+(0.0389358177737)*x[0]+(-0.0949733279735)*x[1]**o+(0.921937240884)*x[1]
        arg[0,3,1,0]=(0.591276336387)*x[0]**o+(-0.598236438303)*x[0]+(-0.829829995185)*x[1]**o+(0.539669346291)*x[1]
        arg[0,3,1,1]=(-0.944726366074)*x[0]**o+(-0.796928273271)*x[0]+(-0.424606656681)*x[1]**o+(0.35341306063)*x[1]
        arg[0,3,2,0]=(-0.850914599406)*x[0]**o+(0.791179202404)*x[0]+(-0.0220845565851)*x[1]**o+(0.594756136957)*x[1]
        arg[0,3,2,1]=(0.355862680641)*x[0]**o+(0.134261680432)*x[0]+(0.477798827075)*x[1]**o+(0.508452338499)*x[1]
        arg[0,4,0,0]=(0.481974944781)*x[0]**o+(0.259271175551)*x[0]+(0.953827692515)*x[1]**o+(0.364160709544)*x[1]
        arg[0,4,0,1]=(-0.316627315143)*x[0]**o+(-0.856616735533)*x[0]+(0.308508314695)*x[1]**o+(-0.677602758367)*x[1]
        arg[0,4,1,0]=(-0.76069824745)*x[0]**o+(0.282351651183)*x[0]+(-0.15964276552)*x[1]**o+(-0.344276428838)*x[1]
        arg[0,4,1,1]=(9.00669303738e-05)*x[0]**o+(0.948498888115)*x[0]+(0.690370267333)*x[1]**o+(0.619119188327)*x[1]
        arg[0,4,2,0]=(0.957692561039)*x[0]**o+(-0.147005436631)*x[0]+(-0.177840574215)*x[1]**o+(-0.28259452614)*x[1]
        arg[0,4,2,1]=(-0.431771943451)*x[0]**o+(-0.158132166651)*x[0]+(-0.134385316147)*x[1]**o+(0.751243590835)*x[1]
        arg[1,0,0,0]=(-0.816841420645)*x[0]**o+(0.55608373851)*x[0]+(-0.885838014201)*x[1]**o+(-0.467159163867)*x[1]
        arg[1,0,0,1]=(-0.543361117401)*x[0]**o+(0.329490019675)*x[0]+(0.942703992489)*x[1]**o+(-0.261273838803)*x[1]
        arg[1,0,1,0]=(0.865108106801)*x[0]**o+(-0.376972427024)*x[0]+(-0.0257304693919)*x[1]**o+(0.694210909726)*x[1]
        arg[1,0,1,1]=(0.428354498121)*x[0]**o+(0.962704601892)*x[0]+(-0.442636806139)*x[1]**o+(0.424724924149)*x[1]
        arg[1,0,2,0]=(-0.157277242299)*x[0]**o+(0.487588443251)*x[0]+(-0.813089096341)*x[1]**o+(0.0406316718829)*x[1]
        arg[1,0,2,1]=(0.345322702588)*x[0]**o+(0.224429042659)*x[0]+(-0.24109926586)*x[1]**o+(0.433141241931)*x[1]
        arg[1,1,0,0]=(0.956901263519)*x[0]**o+(0.48418062914)*x[0]+(0.14089030013)*x[1]**o+(-0.492352201717)*x[1]
        arg[1,1,0,1]=(-0.679781695411)*x[0]**o+(0.29857196636)*x[0]+(-0.234968984185)*x[1]**o+(0.114575919411)*x[1]
        arg[1,1,1,0]=(0.488378272934)*x[0]**o+(-0.358508327343)*x[0]+(-0.337958846849)*x[1]**o+(0.189303452912)*x[1]
        arg[1,1,1,1]=(-0.506993859574)*x[0]**o+(-0.682478021258)*x[0]+(-0.540344338899)*x[1]**o+(0.366965535976)*x[1]
        arg[1,1,2,0]=(-0.681151638572)*x[0]**o+(0.790837550009)*x[0]+(-0.810785716023)*x[1]**o+(-0.00739097962416)*x[1]
        arg[1,1,2,1]=(-0.355290573677)*x[0]**o+(-0.263294580108)*x[0]+(0.10469018402)*x[1]**o+(-0.650388283116)*x[1]
        arg[1,2,0,0]=(0.906212689905)*x[0]**o+(-0.673956103538)*x[0]+(0.305335278934)*x[1]**o+(-0.0805949628719)*x[1]
        arg[1,2,0,1]=(-0.98311373852)*x[0]**o+(-0.165316704542)*x[0]+(0.197637770603)*x[1]**o+(0.647989184342)*x[1]
        arg[1,2,1,0]=(0.415990736251)*x[0]**o+(-0.710476520822)*x[0]+(0.791512786296)*x[1]**o+(-0.10853188742)*x[1]
        arg[1,2,1,1]=(0.04337234954)*x[0]**o+(0.595650094449)*x[0]+(0.228098665648)*x[1]**o+(-0.868930899722)*x[1]
        arg[1,2,2,0]=(-0.791842482688)*x[0]**o+(-0.611246387621)*x[0]+(0.0504978789412)*x[1]**o+(0.624626300694)*x[1]
        arg[1,2,2,1]=(0.0207375595263)*x[0]**o+(0.560638072838)*x[0]+(0.44126312508)*x[1]**o+(0.169151353084)*x[1]
        arg[1,3,0,0]=(-0.0512069089279)*x[0]**o+(0.51110271913)*x[0]+(0.0785633458626)*x[1]**o+(0.836821453174)*x[1]
        arg[1,3,0,1]=(0.185737468851)*x[0]**o+(0.516071031227)*x[0]+(0.0995388802342)*x[1]**o+(0.25527909092)*x[1]
        arg[1,3,1,0]=(0.318240681435)*x[0]**o+(-0.442725217398)*x[0]+(0.533066616909)*x[1]**o+(0.679570707339)*x[1]
        arg[1,3,1,1]=(0.353367397644)*x[0]**o+(-0.396487189861)*x[0]+(-0.117773711001)*x[1]**o+(0.885897675293)*x[1]
        arg[1,3,2,0]=(-0.647645988694)*x[0]**o+(-0.755258662164)*x[0]+(0.288300257472)*x[1]**o+(0.505342385389)*x[1]
        arg[1,3,2,1]=(0.000781386924262)*x[0]**o+(0.190244787656)*x[0]+(0.589388834071)*x[1]**o+(0.669705413859)*x[1]
        arg[1,4,0,0]=(0.320107896841)*x[0]**o+(0.673508197912)*x[0]+(0.667026895874)*x[1]**o+(-0.818578971562)*x[1]
        arg[1,4,0,1]=(-0.226755915796)*x[0]**o+(0.71800207991)*x[0]+(0.850490212043)*x[1]**o+(0.261708964138)*x[1]
        arg[1,4,1,0]=(-0.0147350282361)*x[0]**o+(0.743889536542)*x[0]+(-0.898802067607)*x[1]**o+(0.711589170571)*x[1]
        arg[1,4,1,1]=(-0.816145493818)*x[0]**o+(0.405931568806)*x[0]+(0.159156576579)*x[1]**o+(0.143576116984)*x[1]
        arg[1,4,2,0]=(0.994791165013)*x[0]**o+(0.65054901122)*x[0]+(0.039218059733)*x[1]**o+(-0.662978657934)*x[1]
        arg[1,4,2,1]=(-0.792277825616)*x[0]**o+(-0.257224872111)*x[0]+(-0.244502392787)*x[1]**o+(-0.276042189495)*x[1]
        arg[2,0,0,0]=(0.133211578016)*x[0]**o+(0.10029141733)*x[0]+(-0.52480547182)*x[1]**o+(0.814652707336)*x[1]
        arg[2,0,0,1]=(0.836429340685)*x[0]**o+(-0.837709322969)*x[0]+(-0.442953015462)*x[1]**o+(-0.984757344044)*x[1]
        arg[2,0,1,0]=(0.893738853578)*x[0]**o+(-0.238050980408)*x[0]+(-0.913668527302)*x[1]**o+(0.895136777323)*x[1]
        arg[2,0,1,1]=(-0.0457361766611)*x[0]**o+(0.414814798208)*x[0]+(-0.57261759439)*x[1]**o+(-0.358969048076)*x[1]
        arg[2,0,2,0]=(-0.320394419991)*x[0]**o+(-0.990129952721)*x[0]+(0.890888895016)*x[1]**o+(0.241063301452)*x[1]
        arg[2,0,2,1]=(-0.604164679932)*x[0]**o+(-0.598939404114)*x[0]+(-0.606633388454)*x[1]**o+(-0.666879580873)*x[1]
        arg[2,1,0,0]=(0.68515547034)*x[0]**o+(0.892247437859)*x[0]+(0.642003383991)*x[1]**o+(0.569240970392)*x[1]
        arg[2,1,0,1]=(0.918713108587)*x[0]**o+(0.865547962284)*x[0]+(0.517952260633)*x[1]**o+(-0.307334010857)*x[1]
        arg[2,1,1,0]=(0.353382468404)*x[0]**o+(-0.542718692644)*x[0]+(-0.957182863451)*x[1]**o+(-0.889352632225)*x[1]
        arg[2,1,1,1]=(0.505185847817)*x[0]**o+(-0.71137143775)*x[0]+(0.853785404375)*x[1]**o+(-0.731028307335)*x[1]
        arg[2,1,2,0]=(-0.0805774093876)*x[0]**o+(0.949831911897)*x[0]+(0.511009248375)*x[1]**o+(-0.0192710206538)*x[1]
        arg[2,1,2,1]=(-0.987709862593)*x[0]**o+(-0.536077508064)*x[0]+(-0.0726882784026)*x[1]**o+(-0.27275457372)*x[1]
        arg[2,2,0,0]=(0.271001953238)*x[0]**o+(0.0446246860776)*x[0]+(0.103657462724)*x[1]**o+(0.359900072694)*x[1]
        arg[2,2,0,1]=(0.613460277623)*x[0]**o+(-0.23324730519)*x[0]+(0.188458167579)*x[1]**o+(-0.574087648909)*x[1]
        arg[2,2,1,0]=(-0.787444207567)*x[0]**o+(0.143772612473)*x[0]+(-0.720451072166)*x[1]**o+(0.936819171338)*x[1]
        arg[2,2,1,1]=(-0.219701379296)*x[0]**o+(-0.343100221149)*x[0]+(-0.972737482064)*x[1]**o+(-0.614208415137)*x[1]
        arg[2,2,2,0]=(0.0292070295438)*x[0]**o+(-0.694263454424)*x[0]+(0.188064440036)*x[1]**o+(0.0901763471697)*x[1]
        arg[2,2,2,1]=(0.407629585245)*x[0]**o+(-0.745619322197)*x[0]+(0.949044983116)*x[1]**o+(-0.316057288128)*x[1]
        arg[2,3,0,0]=(0.124212182643)*x[0]**o+(0.621703244759)*x[0]+(-0.0592136772296)*x[1]**o+(0.571981450786)*x[1]
        arg[2,3,0,1]=(-0.683989210442)*x[0]**o+(-0.318774594157)*x[0]+(0.850235115277)*x[1]**o+(-0.871075179838)*x[1]
        arg[2,3,1,0]=(0.478657634044)*x[0]**o+(-0.43702074673)*x[0]+(0.535572237843)*x[1]**o+(0.22354698199)*x[1]
        arg[2,3,1,1]=(0.0391087869133)*x[0]**o+(0.765140505702)*x[0]+(-0.66279875624)*x[1]**o+(-0.41964006149)*x[1]
        arg[2,3,2,0]=(-0.11123799186)*x[0]**o+(0.533398540082)*x[0]+(-0.931488199925)*x[1]**o+(-0.83080969281)*x[1]
        arg[2,3,2,1]=(-0.138259313941)*x[0]**o+(-0.508937684332)*x[0]+(0.583603445711)*x[1]**o+(-0.0782724872225)*x[1]
        arg[2,4,0,0]=(-0.316223936951)*x[0]**o+(0.329776289164)*x[0]+(-0.692491559115)*x[1]**o+(0.814572730294)*x[1]
        arg[2,4,0,1]=(0.74890960681)*x[0]**o+(-0.79167945549)*x[0]+(0.175328119487)*x[1]**o+(-0.253003240064)*x[1]
        arg[2,4,1,0]=(0.657827625141)*x[0]**o+(-0.268878597055)*x[0]+(0.247374286921)*x[1]**o+(0.414500041415)*x[1]
        arg[2,4,1,1]=(-0.187587046202)*x[0]**o+(-0.594983066039)*x[0]+(-0.203784406879)*x[1]**o+(0.122237566739)*x[1]
        arg[2,4,2,0]=(0.671210794607)*x[0]**o+(0.886706498415)*x[0]+(-0.368997114212)*x[1]**o+(0.0418311781187)*x[1]
        arg[2,4,2,1]=(-0.272424416171)*x[0]**o+(-0.537477247783)*x[0]+(0.0405655834868)*x[1]**o+(-0.179684833703)*x[1]
        arg[3,0,0,0]=(0.457602964193)*x[0]**o+(-0.186625111516)*x[0]+(-0.75244105343)*x[1]**o+(-0.278893445377)*x[1]
        arg[3,0,0,1]=(-0.690378721784)*x[0]**o+(0.0157608350188)*x[0]+(-0.203765434813)*x[1]**o+(0.69771096957)*x[1]
        arg[3,0,1,0]=(-0.884824688493)*x[0]**o+(0.26613451569)*x[0]+(0.226081491639)*x[1]**o+(0.597724298389)*x[1]
        arg[3,0,1,1]=(0.0372124540969)*x[0]**o+(-0.420242551812)*x[0]+(-0.625683613012)*x[1]**o+(0.279682305563)*x[1]
        arg[3,0,2,0]=(-0.570873872458)*x[0]**o+(-0.82361574923)*x[0]+(0.970204188337)*x[1]**o+(-0.242725003569)*x[1]
        arg[3,0,2,1]=(-0.714826095124)*x[0]**o+(-0.334939427182)*x[0]+(-0.730058530908)*x[1]**o+(0.287148154312)*x[1]
        arg[3,1,0,0]=(0.164234307274)*x[0]**o+(-0.388849394356)*x[0]+(0.391518896617)*x[1]**o+(0.00142750126236)*x[1]
        arg[3,1,0,1]=(-0.327700781196)*x[0]**o+(0.0442604682565)*x[0]+(-0.683834466536)*x[1]**o+(-0.665567774985)*x[1]
        arg[3,1,1,0]=(0.135979934041)*x[0]**o+(-0.822614011173)*x[0]+(-0.946061768113)*x[1]**o+(-0.585442102671)*x[1]
        arg[3,1,1,1]=(0.235568769871)*x[0]**o+(0.738233870703)*x[0]+(0.437750339594)*x[1]**o+(-0.44521724321)*x[1]
        arg[3,1,2,0]=(0.449256898185)*x[0]**o+(-0.464211755196)*x[0]+(0.616886411002)*x[1]**o+(0.885500269963)*x[1]
        arg[3,1,2,1]=(0.0851105760559)*x[0]**o+(0.888976499957)*x[0]+(0.0842441111314)*x[1]**o+(-0.366994083394)*x[1]
        arg[3,2,0,0]=(0.77120050858)*x[0]**o+(0.354835584949)*x[0]+(0.406813878866)*x[1]**o+(-0.600042478179)*x[1]
        arg[3,2,0,1]=(-0.747876879184)*x[0]**o+(0.527204926634)*x[0]+(0.134965492713)*x[1]**o+(0.497662739459)*x[1]
        arg[3,2,1,0]=(0.367550591163)*x[0]**o+(-0.700608509353)*x[0]+(0.0737696587931)*x[1]**o+(-0.164466899816)*x[1]
        arg[3,2,1,1]=(0.0908791429433)*x[0]**o+(-0.168468767791)*x[0]+(0.85978873292)*x[1]**o+(-0.785241412884)*x[1]
        arg[3,2,2,0]=(-0.296455818292)*x[0]**o+(-0.0277004420512)*x[0]+(0.409183414933)*x[1]**o+(-0.879831945543)*x[1]
        arg[3,2,2,1]=(0.971512010223)*x[0]**o+(-0.811249982262)*x[0]+(-0.846549204737)*x[1]**o+(0.689986186691)*x[1]
        arg[3,3,0,0]=(-0.800235545702)*x[0]**o+(-0.796021196346)*x[0]+(-0.0660744375539)*x[1]**o+(-0.826632192842)*x[1]
        arg[3,3,0,1]=(0.305678237361)*x[0]**o+(0.203936000462)*x[0]+(-0.583171879217)*x[1]**o+(0.924112859119)*x[1]
        arg[3,3,1,0]=(-0.937148913762)*x[0]**o+(-0.172547991203)*x[0]+(0.928491154238)*x[1]**o+(-0.502876301351)*x[1]
        arg[3,3,1,1]=(0.731413856616)*x[0]**o+(0.837213097155)*x[0]+(-0.650032764358)*x[1]**o+(0.96973961589)*x[1]
        arg[3,3,2,0]=(-0.501994458598)*x[0]**o+(-0.705938508685)*x[0]+(0.664394117366)*x[1]**o+(0.692950029135)*x[1]
        arg[3,3,2,1]=(-0.0948399323409)*x[0]**o+(-0.686744554674)*x[0]+(0.577765681583)*x[1]**o+(0.969310750204)*x[1]
        arg[3,4,0,0]=(-0.484011788432)*x[0]**o+(-0.560744139846)*x[0]+(0.072097608773)*x[1]**o+(-0.103052228904)*x[1]
        arg[3,4,0,1]=(-0.201158934695)*x[0]**o+(0.463105335811)*x[0]+(0.733365706682)*x[1]**o+(-0.330035241794)*x[1]
        arg[3,4,1,0]=(0.252379277102)*x[0]**o+(-0.110596657969)*x[0]+(0.763687257878)*x[1]**o+(-0.345598371263)*x[1]
        arg[3,4,1,1]=(0.0868412855617)*x[0]**o+(0.385098679406)*x[0]+(0.291487288978)*x[1]**o+(-0.112145846209)*x[1]
        arg[3,4,2,0]=(-0.913724685255)*x[0]**o+(-0.0348735884313)*x[0]+(0.550688528861)*x[1]**o+(0.798394152329)*x[1]
        arg[3,4,2,1]=(-0.92870523219)*x[0]**o+(0.177354002469)*x[0]+(0.350393765313)*x[1]**o+(-0.974997198741)*x[1]
        ref[0,0,0,0]=(0.815821053922)/(o+1.)+(0.302879577355)
        ref[0,0,0,1]=(-0.223824520603)/(o+1.)+(0.690659426687)
        ref[0,0,1,0]=(-0.501731412433)/(o+1.)+(0.0102964270565)
        ref[0,0,1,1]=(-0.731540589167)/(o+1.)+(0.348284469462)
        ref[0,0,2,0]=(-0.121523684012)/(o+1.)+(0.184348962759)
        ref[0,0,2,1]=(-0.4529485642)/(o+1.)+(-0.830798233026)
        ref[0,1,0,0]=(-0.244463483906)/(o+1.)+(0.226296188971)
        ref[0,1,0,1]=(1.2203272116)/(o+1.)+(-0.693494218115)
        ref[0,1,1,0]=(-0.55576623697)/(o+1.)+(-0.273623989583)
        ref[0,1,1,1]=(-0.316836585171)/(o+1.)+(-0.836135387777)
        ref[0,1,2,0]=(-0.133732244387)/(o+1.)+(-0.875415201605)
        ref[0,1,2,1]=(-0.281487947177)/(o+1.)+(-0.668499911274)
        ref[0,2,0,0]=(-0.896781862729)/(o+1.)+(-0.0225848820142)
        ref[0,2,0,1]=(-1.70326692867)/(o+1.)+(-0.1693379432)
        ref[0,2,1,0]=(-0.00084798161079)/(o+1.)+(0.343359275275)
        ref[0,2,1,1]=(0.609064372767)/(o+1.)+(-0.369639962727)
        ref[0,2,2,0]=(-0.759042888708)/(o+1.)+(0.321708104548)
        ref[0,2,2,1]=(0.760967724015)/(o+1.)+(0.0268897973234)
        ref[0,3,0,0]=(-1.3915636468)/(o+1.)+(-0.230755462281)
        ref[0,3,0,1]=(-0.416449082748)/(o+1.)+(0.480436529329)
        ref[0,3,1,0]=(-0.238553658798)/(o+1.)+(-0.029283546006)
        ref[0,3,1,1]=(-1.36933302275)/(o+1.)+(-0.221757606321)
        ref[0,3,2,0]=(-0.872999155991)/(o+1.)+(0.69296766968)
        ref[0,3,2,1]=(0.833661507716)/(o+1.)+(0.321357009466)
        ref[0,4,0,0]=(1.4358026373)/(o+1.)+(0.311715942547)
        ref[0,4,0,1]=(-0.00811900044778)/(o+1.)+(-0.76710974695)
        ref[0,4,1,0]=(-0.92034101297)/(o+1.)+(-0.0309623888275)
        ref[0,4,1,1]=(0.690460334263)/(o+1.)+(0.783809038221)
        ref[0,4,2,0]=(0.779851986824)/(o+1.)+(-0.214799981386)
        ref[0,4,2,1]=(-0.566157259598)/(o+1.)+(0.296555712092)
        ref[1,0,0,0]=(-1.70267943485)/(o+1.)+(0.0444622873215)
        ref[1,0,0,1]=(0.399342875087)/(o+1.)+(0.0341080904363)
        ref[1,0,1,0]=(0.839377637409)/(o+1.)+(0.158619241351)
        ref[1,0,1,1]=(-0.0142823080176)/(o+1.)+(0.693714763021)
        ref[1,0,2,0]=(-0.97036633864)/(o+1.)+(0.264110057567)
        ref[1,0,2,1]=(0.104223436728)/(o+1.)+(0.328785142295)
        ref[1,1,0,0]=(1.09779156365)/(o+1.)+(-0.00408578628855)
        ref[1,1,0,1]=(-0.914750679597)/(o+1.)+(0.206573942886)
        ref[1,1,1,0]=(0.150419426085)/(o+1.)+(-0.0846024372156)
        ref[1,1,1,1]=(-1.04733819847)/(o+1.)+(-0.157756242641)
        ref[1,1,2,0]=(-1.49193735459)/(o+1.)+(0.391723285192)
        ref[1,1,2,1]=(-0.250600389657)/(o+1.)+(-0.456841431612)
        ref[1,2,0,0]=(1.21154796884)/(o+1.)+(-0.377275533205)
        ref[1,2,0,1]=(-0.785475967917)/(o+1.)+(0.2413362399)
        ref[1,2,1,0]=(1.20750352255)/(o+1.)+(-0.409504204121)
        ref[1,2,1,1]=(0.271471015188)/(o+1.)+(-0.136640402636)
        ref[1,2,2,0]=(-0.741344603747)/(o+1.)+(0.00668995653618)
        ref[1,2,2,1]=(0.462000684606)/(o+1.)+(0.364894712961)
        ref[1,3,0,0]=(0.0273564369347)/(o+1.)+(0.673962086152)
        ref[1,3,0,1]=(0.285276349085)/(o+1.)+(0.385675061073)
        ref[1,3,1,0]=(0.851307298344)/(o+1.)+(0.118422744971)
        ref[1,3,1,1]=(0.235593686643)/(o+1.)+(0.244705242716)
        ref[1,3,2,0]=(-0.359345731222)/(o+1.)+(-0.124958138388)
        ref[1,3,2,1]=(0.590170220995)/(o+1.)+(0.429975100757)
        ref[1,4,0,0]=(0.987134792715)/(o+1.)+(-0.0725353868253)
        ref[1,4,0,1]=(0.623734296247)/(o+1.)+(0.489855522024)
        ref[1,4,1,0]=(-0.913537095843)/(o+1.)+(0.727739353556)
        ref[1,4,1,1]=(-0.656988917238)/(o+1.)+(0.274753842895)
        ref[1,4,2,0]=(1.03400922475)/(o+1.)+(-0.00621482335664)
        ref[1,4,2,1]=(-1.0367802184)/(o+1.)+(-0.266633530803)
        ref[2,0,0,0]=(-0.391593893803)/(o+1.)+(0.457472062333)
        ref[2,0,0,1]=(0.393476325223)/(o+1.)+(-0.911233333506)
        ref[2,0,1,0]=(-0.0199296737245)/(o+1.)+(0.328542898457)
        ref[2,0,1,1]=(-0.618353771051)/(o+1.)+(0.0279228750664)
        ref[2,0,2,0]=(0.570494475025)/(o+1.)+(-0.374533325635)
        ref[2,0,2,1]=(-1.21079806839)/(o+1.)+(-0.632909492494)
        ref[2,1,0,0]=(1.32715885433)/(o+1.)+(0.730744204126)
        ref[2,1,0,1]=(1.43666536922)/(o+1.)+(0.279106975714)
        ref[2,1,1,0]=(-0.603800395047)/(o+1.)+(-0.716035662434)
        ref[2,1,1,1]=(1.35897125219)/(o+1.)+(-0.721199872542)
        ref[2,1,2,0]=(0.430431838988)/(o+1.)+(0.465280445622)
        ref[2,1,2,1]=(-1.060398141)/(o+1.)+(-0.404416040892)
        ref[2,2,0,0]=(0.374659415961)/(o+1.)+(0.202262379386)
        ref[2,2,0,1]=(0.801918445202)/(o+1.)+(-0.403667477049)
        ref[2,2,1,0]=(-1.50789527973)/(o+1.)+(0.540295891905)
        ref[2,2,1,1]=(-1.19243886136)/(o+1.)+(-0.478654318143)
        ref[2,2,2,0]=(0.21727146958)/(o+1.)+(-0.302043553627)
        ref[2,2,2,1]=(1.35667456836)/(o+1.)+(-0.530838305162)
        ref[2,3,0,0]=(0.0649985054133)/(o+1.)+(0.596842347772)
        ref[2,3,0,1]=(0.166245904835)/(o+1.)+(-0.594924886997)
        ref[2,3,1,0]=(1.01422987189)/(o+1.)+(-0.10673688237)
        ref[2,3,1,1]=(-0.623689969327)/(o+1.)+(0.172750222106)
        ref[2,3,2,0]=(-1.04272619178)/(o+1.)+(-0.148705576364)
        ref[2,3,2,1]=(0.445344131771)/(o+1.)+(-0.293605085777)
        ref[2,4,0,0]=(-1.00871549607)/(o+1.)+(0.572174509729)
        ref[2,4,0,1]=(0.924237726297)/(o+1.)+(-0.522341347777)
        ref[2,4,1,0]=(0.905201912063)/(o+1.)+(0.0728107221801)
        ref[2,4,1,1]=(-0.391371453081)/(o+1.)+(-0.23637274965)
        ref[2,4,2,0]=(0.302213680395)/(o+1.)+(0.464268838267)
        ref[2,4,2,1]=(-0.231858832685)/(o+1.)+(-0.358581040743)
        ref[3,0,0,0]=(-0.294838089237)/(o+1.)+(-0.232759278447)
        ref[3,0,0,1]=(-0.894144156597)/(o+1.)+(0.356735902294)
        ref[3,0,1,0]=(-0.658743196854)/(o+1.)+(0.43192940704)
        ref[3,0,1,1]=(-0.588471158915)/(o+1.)+(-0.0702801231243)
        ref[3,0,2,0]=(0.399330315879)/(o+1.)+(-0.5331703764)
        ref[3,0,2,1]=(-1.44488462603)/(o+1.)+(-0.0238956364351)
        ref[3,1,0,0]=(0.555753203891)/(o+1.)+(-0.193710946547)
        ref[3,1,0,1]=(-1.01153524773)/(o+1.)+(-0.310653653364)
        ref[3,1,1,0]=(-0.810081834072)/(o+1.)+(-0.704028056922)
        ref[3,1,1,1]=(0.673319109465)/(o+1.)+(0.146508313746)
        ref[3,1,2,0]=(1.06614330919)/(o+1.)+(0.210644257384)
        ref[3,1,2,1]=(0.169354687187)/(o+1.)+(0.260991208281)
        ref[3,2,0,0]=(1.17801438745)/(o+1.)+(-0.122603446615)
        ref[3,2,0,1]=(-0.612911386471)/(o+1.)+(0.512433833047)
        ref[3,2,1,0]=(0.441320249956)/(o+1.)+(-0.432537704584)
        ref[3,2,1,1]=(0.950667875863)/(o+1.)+(-0.476855090338)
        ref[3,2,2,0]=(0.112727596641)/(o+1.)+(-0.453766193797)
        ref[3,2,2,1]=(0.124962805486)/(o+1.)+(-0.0606318977858)
        ref[3,3,0,0]=(-0.866309983256)/(o+1.)+(-0.811326694594)
        ref[3,3,0,1]=(-0.277493641857)/(o+1.)+(0.564024429791)
        ref[3,3,1,0]=(-0.00865775952435)/(o+1.)+(-0.337712146277)
        ref[3,3,1,1]=(0.0813810922576)/(o+1.)+(0.903476356523)
        ref[3,3,2,0]=(0.162399658768)/(o+1.)+(-0.00649423977499)
        ref[3,3,2,1]=(0.482925749243)/(o+1.)+(0.141283097765)
        ref[3,4,0,0]=(-0.411914179659)/(o+1.)+(-0.331898184375)
        ref[3,4,0,1]=(0.532206771987)/(o+1.)+(0.0665350470083)
        ref[3,4,1,0]=(1.01606653498)/(o+1.)+(-0.228097514616)
        ref[3,4,1,1]=(0.37832857454)/(o+1.)+(0.136476416599)
        ref[3,4,2,0]=(-0.363036156393)/(o+1.)+(0.381760281949)
        ref[3,4,2,1]=(-0.578311466877)/(o+1.)+(-0.398821598136)
      else:
        arg[0,0,0,0]=(0.860676142569)*x[0]**o+(-0.156046991162)*x[0]+(-0.0462355579656)*x[1]**o+(0.70842844961)*x[1]+(0.617299123548)*x[2]**o+(-0.865174329565)*x[2]
        arg[0,0,0,1]=(-0.284308060865)*x[0]**o+(0.788848579389)*x[0]+(0.629321669042)*x[1]**o+(0.066637849809)*x[1]+(0.114152381009)*x[2]**o+(-0.466687745846)*x[2]
        arg[0,0,1,0]=(-0.690823198517)*x[0]**o+(-0.141510804573)*x[0]+(0.969978831967)*x[1]**o+(0.307926382658)*x[1]+(0.710373009094)*x[2]**o+(-0.494828315907)*x[2]
        arg[0,0,1,1]=(0.953361413913)*x[0]**o+(-0.103742611936)*x[0]+(0.63934955554)*x[1]**o+(-0.0778114476564)*x[1]+(-0.221185122698)*x[2]**o+(-0.287432047404)*x[2]
        arg[0,0,2,0]=(0.378306380558)*x[0]**o+(-0.117743922672)*x[0]+(-0.0606376630208)*x[1]**o+(-0.59439189505)*x[1]+(0.10507320657)*x[2]**o+(0.856411423067)*x[2]
        arg[0,0,2,1]=(0.777734672779)*x[0]**o+(0.286956002007)*x[0]+(-0.931811626583)*x[1]**o+(-0.806626659661)*x[1]+(-0.420390949282)*x[2]**o+(0.854080006578)*x[2]
        arg[0,1,0,0]=(0.403426948964)*x[0]**o+(0.707099419296)*x[0]+(0.593137130456)*x[1]**o+(-0.757560771598)*x[1]+(0.369632217934)*x[2]**o+(-0.348480285397)*x[2]
        arg[0,1,0,1]=(-0.491454189782)*x[0]**o+(0.547276640565)*x[0]+(-0.955045542708)*x[1]**o+(0.529989872531)*x[1]+(-0.161086874799)*x[2]**o+(-0.181810040724)*x[2]
        arg[0,1,1,0]=(0.46282177104)*x[0]**o+(0.254346937251)*x[0]+(0.824797105546)*x[1]**o+(-0.449979162246)*x[1]+(0.22307703071)*x[2]**o+(-0.592072135668)*x[2]
        arg[0,1,1,1]=(0.329230675141)*x[0]**o+(-0.733034697145)*x[0]+(0.762195609347)*x[1]**o+(-0.639097591362)*x[1]+(0.765354927905)*x[2]**o+(0.313571493405)*x[2]
        arg[0,1,2,0]=(0.3874127072)*x[0]**o+(-0.0569429727569)*x[0]+(-0.203121198708)*x[1]**o+(-0.736379499164)*x[1]+(-0.980629375436)*x[2]**o+(0.955793660048)*x[2]
        arg[0,1,2,1]=(-0.656255469455)*x[0]**o+(0.711904808699)*x[0]+(-0.363253678399)*x[1]**o+(0.921875000807)*x[1]+(0.686394816431)*x[2]**o+(0.818500716704)*x[2]
        arg[0,2,0,0]=(-0.931555347846)*x[0]**o+(-0.249573400742)*x[0]+(0.617166716404)*x[1]**o+(-0.929587051108)*x[1]+(0.830928715064)*x[2]**o+(0.461637431613)*x[2]
        arg[0,2,0,1]=(-0.8677719945)*x[0]**o+(-0.176892180445)*x[0]+(-0.881887456464)*x[1]**o+(0.923355677513)*x[1]+(-0.978544668453)*x[2]**o+(-0.427009384287)*x[2]
        arg[0,2,1,0]=(-0.0707937433084)*x[0]**o+(-0.872892141393)*x[0]+(0.606234798381)*x[1]**o+(0.935845076847)*x[1]+(-0.97559052201)*x[2]**o+(-0.31138518209)*x[2]
        arg[0,2,1,1]=(0.450210597865)*x[0]**o+(0.842939440378)*x[0]+(-0.333422856577)*x[1]**o+(-0.492809679443)*x[1]+(-0.254728278043)*x[2]**o+(-0.0702767077952)*x[2]
        arg[0,2,2,0]=(-0.250463216903)*x[0]**o+(0.756576293763)*x[0]+(0.596902629953)*x[1]**o+(-0.887080794947)*x[1]+(0.027319948008)*x[2]**o+(0.552496782776)*x[2]
        arg[0,2,2,1]=(-0.0883102140591)*x[0]**o+(0.988759610101)*x[0]+(-0.208152392508)*x[1]**o+(0.938999108564)*x[1]+(-0.773883611417)*x[2]**o+(0.653413306338)*x[2]
        arg[0,3,0,0]=(-0.399644211606)*x[0]**o+(-0.978595776407)*x[0]+(0.99454650986)*x[1]**o+(0.143313668715)*x[1]+(0.328592750842)*x[2]**o+(-0.958804581101)*x[2]
        arg[0,3,0,1]=(-0.0667966885959)*x[0]**o+(0.332653574704)*x[0]+(-0.762025150916)*x[1]**o+(0.895044013374)*x[1]+(0.627013843362)*x[2]**o+(-0.682392672232)*x[2]
        arg[0,3,1,0]=(-0.834479453773)*x[0]**o+(-0.375930859033)*x[0]+(0.129507966861)*x[1]**o+(0.995364280408)*x[1]+(-0.126081191572)*x[2]**o+(0.0344262779491)*x[2]
        arg[0,3,1,1]=(0.796952126596)*x[0]**o+(0.310562243091)*x[0]+(0.854548623788)*x[1]**o+(-0.719100537284)*x[1]+(-0.236266783438)*x[2]**o+(-0.646647129209)*x[2]
        arg[0,3,2,0]=(0.844073289115)*x[0]**o+(0.423300345347)*x[0]+(0.126354796852)*x[1]**o+(0.180372257606)*x[1]+(-0.564712722146)*x[2]**o+(0.466988430183)*x[2]
        arg[0,3,2,1]=(0.228775824367)*x[0]**o+(0.40701094831)*x[0]+(0.223101054593)*x[1]**o+(-0.894937905383)*x[1]+(0.425392942773)*x[2]**o+(-0.844238667169)*x[2]
        arg[0,4,0,0]=(0.3363848738)*x[0]**o+(-0.985327066062)*x[0]+(-0.107847971678)*x[1]**o+(0.596121801488)*x[1]+(0.655589801663)*x[2]**o+(-0.729650731949)*x[2]
        arg[0,4,0,1]=(-0.176435679219)*x[0]**o+(0.382558711725)*x[0]+(-0.658212492337)*x[1]**o+(0.584904267612)*x[1]+(-0.534684876705)*x[2]**o+(-0.441262614915)*x[2]
        arg[0,4,1,0]=(0.281301737167)*x[0]**o+(-0.642486274791)*x[0]+(0.420753884869)*x[1]**o+(0.482835955591)*x[1]+(0.169431333788)*x[2]**o+(0.922841193825)*x[2]
        arg[0,4,1,1]=(0.235812590604)*x[0]**o+(-0.446233855396)*x[0]+(0.465015868566)*x[1]**o+(-0.780441276888)*x[1]+(0.513632761351)*x[2]**o+(-0.0576482811678)*x[2]
        arg[0,4,2,0]=(-0.617245874462)*x[0]**o+(0.183661473466)*x[0]+(-0.711148911239)*x[1]**o+(0.125228991306)*x[1]+(-0.500218603959)*x[2]**o+(0.70597577229)*x[2]
        arg[0,4,2,1]=(0.56330304117)*x[0]**o+(0.631284986628)*x[0]+(-0.222485175776)*x[1]**o+(0.0639235737228)*x[1]+(-0.276613868747)*x[2]**o+(0.407488564857)*x[2]
        arg[1,0,0,0]=(0.159032146915)*x[0]**o+(-0.716040989693)*x[0]+(0.410709172613)*x[1]**o+(-0.527667963652)*x[1]+(-0.160515156497)*x[2]**o+(0.294448742382)*x[2]
        arg[1,0,0,1]=(0.941139035835)*x[0]**o+(0.707410046094)*x[0]+(-0.172095410803)*x[1]**o+(-0.967035204304)*x[1]+(0.870395272646)*x[2]**o+(0.830870986802)*x[2]
        arg[1,0,1,0]=(0.731015285309)*x[0]**o+(0.450519048697)*x[0]+(-0.484568714675)*x[1]**o+(0.454201220108)*x[1]+(-0.161850459099)*x[2]**o+(-0.617463263396)*x[2]
        arg[1,0,1,1]=(-0.958841628847)*x[0]**o+(-0.0667813400021)*x[0]+(0.659551780082)*x[1]**o+(0.514424015304)*x[1]+(0.920583608125)*x[2]**o+(0.89564899572)*x[2]
        arg[1,0,2,0]=(0.0482630479817)*x[0]**o+(0.911326064322)*x[0]+(-0.275134422092)*x[1]**o+(0.216495232459)*x[1]+(-0.973586171195)*x[2]**o+(0.275694688611)*x[2]
        arg[1,0,2,1]=(0.769733688293)*x[0]**o+(0.771829018421)*x[0]+(-0.22434549079)*x[1]**o+(0.131966885817)*x[1]+(-0.109424705679)*x[2]**o+(-0.369585136379)*x[2]
        arg[1,1,0,0]=(-0.0160180905877)*x[0]**o+(-0.268260755267)*x[0]+(-0.213940412022)*x[1]**o+(0.665987805279)*x[1]+(0.968879828017)*x[2]**o+(-0.449435538913)*x[2]
        arg[1,1,0,1]=(0.361055351825)*x[0]**o+(0.463294478488)*x[0]+(0.692655668486)*x[1]**o+(0.151415573206)*x[1]+(0.902280461515)*x[2]**o+(0.690926934073)*x[2]
        arg[1,1,1,0]=(0.591998602299)*x[0]**o+(0.516482177709)*x[0]+(-0.298511909182)*x[1]**o+(0.281818071894)*x[1]+(0.207089953979)*x[2]**o+(0.917004722388)*x[2]
        arg[1,1,1,1]=(0.479832408236)*x[0]**o+(-0.643374914165)*x[0]+(0.370844835249)*x[1]**o+(0.560458499239)*x[1]+(-0.865852319767)*x[2]**o+(0.560572130123)*x[2]
        arg[1,1,2,0]=(0.709293526635)*x[0]**o+(-0.388347223155)*x[0]+(0.60797330273)*x[1]**o+(-0.847624783099)*x[1]+(-0.0476367756382)*x[2]**o+(0.637841988478)*x[2]
        arg[1,1,2,1]=(0.343458045422)*x[0]**o+(-0.764451449587)*x[0]+(-0.444256801761)*x[1]**o+(0.385385997921)*x[1]+(-0.722095136671)*x[2]**o+(0.677852473207)*x[2]
        arg[1,2,0,0]=(0.710372887392)*x[0]**o+(-0.628494679119)*x[0]+(-0.59513507905)*x[1]**o+(-0.527824874295)*x[1]+(-0.611189170789)*x[2]**o+(0.090651976806)*x[2]
        arg[1,2,0,1]=(-0.980160150289)*x[0]**o+(-0.0286097581598)*x[0]+(-0.407906975953)*x[1]**o+(-0.319422835951)*x[1]+(-0.489787310147)*x[2]**o+(-0.694386341354)*x[2]
        arg[1,2,1,0]=(-0.416867224376)*x[0]**o+(0.631406877489)*x[0]+(-0.93939195942)*x[1]**o+(0.266858600313)*x[1]+(0.982173221612)*x[2]**o+(0.112901364617)*x[2]
        arg[1,2,1,1]=(0.412001480348)*x[0]**o+(0.624281489887)*x[0]+(0.306558457751)*x[1]**o+(0.95864454419)*x[1]+(0.299175627954)*x[2]**o+(-0.0531991017347)*x[2]
        arg[1,2,2,0]=(-0.0342261621741)*x[0]**o+(-0.587245704486)*x[0]+(-0.976407464107)*x[1]**o+(-0.0157215030322)*x[1]+(-0.529857472095)*x[2]**o+(-0.419626697673)*x[2]
        arg[1,2,2,1]=(-0.681737301037)*x[0]**o+(-0.980349460702)*x[0]+(-0.839469631393)*x[1]**o+(-0.412417814907)*x[1]+(0.451717398947)*x[2]**o+(0.962167685987)*x[2]
        arg[1,3,0,0]=(0.283453965913)*x[0]**o+(0.204445403787)*x[0]+(-0.806150376236)*x[1]**o+(0.122187094069)*x[1]+(-0.679291367964)*x[2]**o+(-0.875353966489)*x[2]
        arg[1,3,0,1]=(0.115974475175)*x[0]**o+(0.701473497654)*x[0]+(0.298012120304)*x[1]**o+(0.567597901709)*x[1]+(0.721211831476)*x[2]**o+(-0.255389572775)*x[2]
        arg[1,3,1,0]=(0.393891431092)*x[0]**o+(-0.563382537925)*x[0]+(0.974194829106)*x[1]**o+(0.659688559688)*x[1]+(-0.394266143828)*x[2]**o+(0.875266200191)*x[2]
        arg[1,3,1,1]=(-0.456930975697)*x[0]**o+(0.334590626231)*x[0]+(-0.227176883362)*x[1]**o+(0.76775699173)*x[1]+(0.839371167887)*x[2]**o+(-0.155473916326)*x[2]
        arg[1,3,2,0]=(0.346761981787)*x[0]**o+(-0.625121706322)*x[0]+(0.340655154638)*x[1]**o+(-0.0646323546999)*x[1]+(-0.211347402677)*x[2]**o+(0.233661266919)*x[2]
        arg[1,3,2,1]=(-0.086096644956)*x[0]**o+(-0.845623653209)*x[0]+(-0.0866093549957)*x[1]**o+(-0.467286349011)*x[1]+(-0.0621455515357)*x[2]**o+(0.164779979085)*x[2]
        arg[1,4,0,0]=(0.42022138208)*x[0]**o+(0.136984451355)*x[0]+(0.664770979086)*x[1]**o+(-0.537368896193)*x[1]+(0.874256726691)*x[2]**o+(-0.58614100824)*x[2]
        arg[1,4,0,1]=(-0.805860964792)*x[0]**o+(0.758511304044)*x[0]+(-0.574686863541)*x[1]**o+(-0.410873101247)*x[1]+(0.932015171551)*x[2]**o+(0.439987195351)*x[2]
        arg[1,4,1,0]=(-0.515187049123)*x[0]**o+(0.308078404334)*x[0]+(0.940959771774)*x[1]**o+(0.466269579691)*x[1]+(-0.190617144843)*x[2]**o+(0.738172533582)*x[2]
        arg[1,4,1,1]=(0.283326036586)*x[0]**o+(-0.693631747857)*x[0]+(0.478965080319)*x[1]**o+(-0.0465389396775)*x[1]+(-0.72886286959)*x[2]**o+(-0.9346424149)*x[2]
        arg[1,4,2,0]=(-0.71539766285)*x[0]**o+(0.361875987071)*x[0]+(0.905270678876)*x[1]**o+(-0.675024725702)*x[1]+(0.378259473658)*x[2]**o+(0.0520394226711)*x[2]
        arg[1,4,2,1]=(-0.319600525923)*x[0]**o+(0.0449956327134)*x[0]+(0.707418658521)*x[1]**o+(0.210137972776)*x[1]+(-0.767860728688)*x[2]**o+(0.232340825758)*x[2]
        arg[2,0,0,0]=(0.92844243651)*x[0]**o+(0.956810319504)*x[0]+(0.762198571194)*x[1]**o+(-0.363107634629)*x[1]+(0.0750393505254)*x[2]**o+(-0.43573581041)*x[2]
        arg[2,0,0,1]=(-0.193613121742)*x[0]**o+(0.440027871839)*x[0]+(-0.607905506346)*x[1]**o+(0.894840354958)*x[1]+(0.601279436178)*x[2]**o+(0.46806560762)*x[2]
        arg[2,0,1,0]=(0.370901517567)*x[0]**o+(-0.738769905647)*x[0]+(-0.698692979664)*x[1]**o+(-0.286318840335)*x[1]+(-0.879367315688)*x[2]**o+(-0.959912132991)*x[2]
        arg[2,0,1,1]=(-0.278939533774)*x[0]**o+(0.183795552426)*x[0]+(-0.341943551466)*x[1]**o+(0.491847260291)*x[1]+(-0.141030429384)*x[2]**o+(0.256287855539)*x[2]
        arg[2,0,2,0]=(-0.135930357616)*x[0]**o+(-0.382515749949)*x[0]+(-0.43030873581)*x[1]**o+(-0.0211891159981)*x[1]+(-0.377865120002)*x[2]**o+(0.109942325029)*x[2]
        arg[2,0,2,1]=(0.288827267201)*x[0]**o+(-0.907970544468)*x[0]+(-0.622258309213)*x[1]**o+(-0.763273879894)*x[1]+(0.438715013671)*x[2]**o+(-0.93873140988)*x[2]
        arg[2,1,0,0]=(-0.963487369906)*x[0]**o+(0.899051290288)*x[0]+(-0.268257956548)*x[1]**o+(-0.229092726584)*x[1]+(0.427024112314)*x[2]**o+(0.0435138095138)*x[2]
        arg[2,1,0,1]=(0.89996926597)*x[0]**o+(-0.206500706229)*x[0]+(-0.191769123095)*x[1]**o+(-0.348924155298)*x[1]+(-0.74405653776)*x[2]**o+(0.0724946691864)*x[2]
        arg[2,1,1,0]=(0.338170397901)*x[0]**o+(-0.725761081242)*x[0]+(-0.45928272676)*x[1]**o+(0.795535326783)*x[1]+(-0.316443844644)*x[2]**o+(-0.844205401892)*x[2]
        arg[2,1,1,1]=(0.509131757681)*x[0]**o+(0.319175801535)*x[0]+(-0.807058926279)*x[1]**o+(0.63928269996)*x[1]+(0.219324073432)*x[2]**o+(0.351217517434)*x[2]
        arg[2,1,2,0]=(-0.534090721611)*x[0]**o+(0.27146424294)*x[0]+(0.250778065716)*x[1]**o+(-0.901322994167)*x[1]+(-0.57588768699)*x[2]**o+(0.405043003456)*x[2]
        arg[2,1,2,1]=(0.101699906269)*x[0]**o+(0.361960318738)*x[0]+(0.278974923593)*x[1]**o+(-0.336289352037)*x[1]+(-0.677353936215)*x[2]**o+(0.0726793347753)*x[2]
        arg[2,2,0,0]=(-0.734961283823)*x[0]**o+(-0.593268557578)*x[0]+(0.310896858565)*x[1]**o+(-0.482326137951)*x[1]+(0.886472957099)*x[2]**o+(-0.0403563476581)*x[2]
        arg[2,2,0,1]=(-0.166071725106)*x[0]**o+(-0.454245686818)*x[0]+(-0.892674508991)*x[1]**o+(0.581554190251)*x[1]+(0.995323654908)*x[2]**o+(-0.0336714433349)*x[2]
        arg[2,2,1,0]=(-0.437491848099)*x[0]**o+(-0.195039508213)*x[0]+(0.822011086215)*x[1]**o+(0.771044412863)*x[1]+(0.697534261252)*x[2]**o+(0.666510217725)*x[2]
        arg[2,2,1,1]=(-0.529148276275)*x[0]**o+(-0.093016460024)*x[0]+(0.663641010649)*x[1]**o+(0.940439359843)*x[1]+(0.754151159716)*x[2]**o+(0.852349310591)*x[2]
        arg[2,2,2,0]=(0.859523244682)*x[0]**o+(0.704825125128)*x[0]+(0.221832129598)*x[1]**o+(0.573940020456)*x[1]+(0.935673751591)*x[2]**o+(0.23235170475)*x[2]
        arg[2,2,2,1]=(0.0359047473512)*x[0]**o+(0.836816935443)*x[0]+(0.6411608273)*x[1]**o+(0.291526953908)*x[1]+(0.735564515362)*x[2]**o+(-0.465781621769)*x[2]
        arg[2,3,0,0]=(0.352740040216)*x[0]**o+(0.54224020581)*x[0]+(0.681356174115)*x[1]**o+(-0.97589117869)*x[1]+(0.880755719884)*x[2]**o+(0.562259403362)*x[2]
        arg[2,3,0,1]=(0.563708471054)*x[0]**o+(0.525523219723)*x[0]+(0.423068360374)*x[1]**o+(0.29239160821)*x[1]+(0.874360368885)*x[2]**o+(0.922265851359)*x[2]
        arg[2,3,1,0]=(0.694309936305)*x[0]**o+(-0.409224769672)*x[0]+(-0.748835864803)*x[1]**o+(-0.21841534641)*x[1]+(0.497059735624)*x[2]**o+(-0.753234694148)*x[2]
        arg[2,3,1,1]=(0.109706510352)*x[0]**o+(-0.0160720403678)*x[0]+(0.888778080744)*x[1]**o+(-0.938692716268)*x[1]+(0.529262951478)*x[2]**o+(0.481430680214)*x[2]
        arg[2,3,2,0]=(0.65368659132)*x[0]**o+(-0.988176396392)*x[0]+(0.989546739333)*x[1]**o+(0.23178715846)*x[1]+(-0.557645438328)*x[2]**o+(0.063228513147)*x[2]
        arg[2,3,2,1]=(0.286316141831)*x[0]**o+(0.668138463424)*x[0]+(-0.950430671533)*x[1]**o+(-0.978012107187)*x[1]+(-0.107618382464)*x[2]**o+(-0.684084421276)*x[2]
        arg[2,4,0,0]=(-0.63510659432)*x[0]**o+(0.264242935995)*x[0]+(0.960647241624)*x[1]**o+(0.58483379289)*x[1]+(0.7255270271)*x[2]**o+(-0.476151653985)*x[2]
        arg[2,4,0,1]=(-0.221953908705)*x[0]**o+(-0.994143786509)*x[0]+(-0.359525421944)*x[1]**o+(0.533966197668)*x[1]+(-0.29739666047)*x[2]**o+(-0.0824204124466)*x[2]
        arg[2,4,1,0]=(-0.447043625871)*x[0]**o+(-0.904485300564)*x[0]+(0.26179071803)*x[1]**o+(0.437320719437)*x[1]+(-0.56501466909)*x[2]**o+(0.129662379166)*x[2]
        arg[2,4,1,1]=(0.259809756366)*x[0]**o+(0.446788808701)*x[0]+(-0.143702396023)*x[1]**o+(0.879531090832)*x[1]+(0.410047703156)*x[2]**o+(0.755750006468)*x[2]
        arg[2,4,2,0]=(-0.730881615951)*x[0]**o+(-0.357110792705)*x[0]+(0.552722654621)*x[1]**o+(0.965623848242)*x[1]+(0.555951256049)*x[2]**o+(0.259021370764)*x[2]
        arg[2,4,2,1]=(-0.465880467377)*x[0]**o+(-0.840984822217)*x[0]+(0.665817382039)*x[1]**o+(-0.390932961529)*x[1]+(-0.0857939888315)*x[2]**o+(-0.712185917672)*x[2]
        arg[3,0,0,0]=(0.710281429223)*x[0]**o+(-0.723903639057)*x[0]+(-0.267455509065)*x[1]**o+(-0.731684188612)*x[1]+(-0.287442751768)*x[2]**o+(-0.266871749947)*x[2]
        arg[3,0,0,1]=(0.460723870663)*x[0]**o+(0.0502154310124)*x[0]+(-0.260241363193)*x[1]**o+(-0.516473088648)*x[1]+(-0.925984113039)*x[2]**o+(-0.0297616649018)*x[2]
        arg[3,0,1,0]=(0.523186648278)*x[0]**o+(0.937042293367)*x[0]+(0.811740916657)*x[1]**o+(0.0544805175015)*x[1]+(-0.543484279955)*x[2]**o+(0.244714198306)*x[2]
        arg[3,0,1,1]=(-0.69212630167)*x[0]**o+(0.868708276385)*x[0]+(0.473290025811)*x[1]**o+(0.137043602294)*x[1]+(-0.19291314113)*x[2]**o+(-0.755147036188)*x[2]
        arg[3,0,2,0]=(0.0472638421122)*x[0]**o+(0.403135551064)*x[0]+(0.290083188289)*x[1]**o+(0.96419060449)*x[1]+(-0.240544184077)*x[2]**o+(-0.183028108529)*x[2]
        arg[3,0,2,1]=(0.354253383908)*x[0]**o+(0.628816528774)*x[0]+(0.588194429756)*x[1]**o+(-0.852128734457)*x[1]+(-0.84774778925)*x[2]**o+(0.796310123562)*x[2]
        arg[3,1,0,0]=(0.528880495806)*x[0]**o+(-0.74176589205)*x[0]+(-0.447816938371)*x[1]**o+(0.611297564802)*x[1]+(0.742134198482)*x[2]**o+(-0.625840664202)*x[2]
        arg[3,1,0,1]=(0.419633746259)*x[0]**o+(-0.247742274578)*x[0]+(0.302328885303)*x[1]**o+(-0.629854480502)*x[1]+(0.493919301288)*x[2]**o+(0.902910119017)*x[2]
        arg[3,1,1,0]=(0.509160481656)*x[0]**o+(-0.388129364133)*x[0]+(-0.595185230598)*x[1]**o+(-0.289592836865)*x[1]+(0.615663908134)*x[2]**o+(-0.560687711511)*x[2]
        arg[3,1,1,1]=(0.647782473568)*x[0]**o+(0.935840881725)*x[0]+(0.440348212287)*x[1]**o+(-0.489797988572)*x[1]+(0.766592631787)*x[2]**o+(0.217399649246)*x[2]
        arg[3,1,2,0]=(0.64944026906)*x[0]**o+(0.487409918475)*x[0]+(-0.0322210593784)*x[1]**o+(0.281311337861)*x[1]+(-0.694252407571)*x[2]**o+(-0.659678386121)*x[2]
        arg[3,1,2,1]=(0.806357278285)*x[0]**o+(0.419430374567)*x[0]+(-0.0525796600116)*x[1]**o+(-0.556886341914)*x[1]+(-0.288209814683)*x[2]**o+(-0.309696771495)*x[2]
        arg[3,2,0,0]=(-0.300806388527)*x[0]**o+(-0.39334416908)*x[0]+(0.000711617540648)*x[1]**o+(-0.618539958736)*x[1]+(0.0574098223152)*x[2]**o+(-0.351595384168)*x[2]
        arg[3,2,0,1]=(-0.898675793892)*x[0]**o+(0.904479958277)*x[0]+(0.0781945046018)*x[1]**o+(-0.0461053392586)*x[1]+(0.844798127031)*x[2]**o+(-0.82306838709)*x[2]
        arg[3,2,1,0]=(-0.268742335093)*x[0]**o+(-0.532639944208)*x[0]+(0.0101227109551)*x[1]**o+(-0.438312865518)*x[1]+(0.104288648246)*x[2]**o+(-0.115669027311)*x[2]
        arg[3,2,1,1]=(0.639871308792)*x[0]**o+(0.260847628514)*x[0]+(-0.455670719167)*x[1]**o+(0.210619035618)*x[1]+(-0.401809068119)*x[2]**o+(-0.0625194517026)*x[2]
        arg[3,2,2,0]=(-0.713684223829)*x[0]**o+(0.214281604766)*x[0]+(-0.60152592324)*x[1]**o+(0.249807228316)*x[1]+(-0.939395078261)*x[2]**o+(-0.519744396388)*x[2]
        arg[3,2,2,1]=(-0.340588993443)*x[0]**o+(-0.0165543854997)*x[0]+(-0.366479789338)*x[1]**o+(0.316347630987)*x[1]+(-0.457055798679)*x[2]**o+(-0.132368055553)*x[2]
        arg[3,3,0,0]=(-0.745211736715)*x[0]**o+(0.0106787556884)*x[0]+(-0.448180943446)*x[1]**o+(0.694982281891)*x[1]+(0.924431258221)*x[2]**o+(-0.841633806168)*x[2]
        arg[3,3,0,1]=(0.368054364362)*x[0]**o+(0.19288251992)*x[0]+(0.565854068035)*x[1]**o+(0.747534544754)*x[1]+(0.473165789569)*x[2]**o+(-0.59399986706)*x[2]
        arg[3,3,1,0]=(-0.142707671693)*x[0]**o+(0.177095043535)*x[0]+(-0.348946846847)*x[1]**o+(-0.545430013244)*x[1]+(-0.671405653136)*x[2]**o+(-0.531699969702)*x[2]
        arg[3,3,1,1]=(0.783201055631)*x[0]**o+(0.899495042323)*x[0]+(0.117802924059)*x[1]**o+(-0.277400900172)*x[1]+(-0.351268362305)*x[2]**o+(0.52756480897)*x[2]
        arg[3,3,2,0]=(-0.866758006698)*x[0]**o+(0.293139875238)*x[0]+(-0.318394231317)*x[1]**o+(-0.783701622124)*x[1]+(-0.95137399655)*x[2]**o+(-0.474198736639)*x[2]
        arg[3,3,2,1]=(0.0428398560619)*x[0]**o+(-0.970180601102)*x[0]+(-0.753897573393)*x[1]**o+(-0.0873352699251)*x[1]+(-0.131440532574)*x[2]**o+(0.875241605077)*x[2]
        arg[3,4,0,0]=(0.32399493643)*x[0]**o+(0.67295092524)*x[0]+(-0.29221110671)*x[1]**o+(0.118066079866)*x[1]+(-0.566541154667)*x[2]**o+(-0.806367499079)*x[2]
        arg[3,4,0,1]=(0.278776399755)*x[0]**o+(0.472327826919)*x[0]+(-0.956871861949)*x[1]**o+(0.109236874792)*x[1]+(-0.494095116883)*x[2]**o+(-0.194313775518)*x[2]
        arg[3,4,1,0]=(0.650920037415)*x[0]**o+(0.977698516929)*x[0]+(-0.0730777837998)*x[1]**o+(-0.576543152032)*x[1]+(0.781481934657)*x[2]**o+(-0.389328840554)*x[2]
        arg[3,4,1,1]=(-0.798341529522)*x[0]**o+(-0.630245234712)*x[0]+(-0.444550323489)*x[1]**o+(-0.539189248295)*x[1]+(-0.904964423673)*x[2]**o+(-0.611560462796)*x[2]
        arg[3,4,2,0]=(-0.392991417685)*x[0]**o+(0.313599061567)*x[0]+(-0.215747337398)*x[1]**o+(-0.310316872476)*x[1]+(-0.269452617785)*x[2]**o+(0.684353794645)*x[2]
        arg[3,4,2,1]=(-0.586668781403)*x[0]**o+(-0.0711284937177)*x[0]+(0.408273850302)*x[1]**o+(0.761860299288)*x[1]+(0.902079242449)*x[2]**o+(0.70161017722)*x[2]
        ref[0,0,0,0]=(1.43173970815)/(o+1.)+(-0.156396435558)
        ref[0,0,0,1]=(0.459165989186)/(o+1.)+(0.194399341676)
        ref[0,0,1,0]=(0.989528642545)/(o+1.)+(-0.164206368911)
        ref[0,0,1,1]=(1.37152584676)/(o+1.)+(-0.234493053498)
        ref[0,0,2,0]=(0.422741924108)/(o+1.)+(0.0721378026728)
        ref[0,0,2,1]=(-0.574467903086)/(o+1.)+(0.167204674462)
        ref[0,1,0,0]=(1.36619629735)/(o+1.)+(-0.19947081885)
        ref[0,1,0,1]=(-1.60758660729)/(o+1.)+(0.447728236186)
        ref[0,1,1,0]=(1.5106959073)/(o+1.)+(-0.393852180331)
        ref[0,1,1,1]=(1.85678121239)/(o+1.)+(-0.529280397551)
        ref[0,1,2,0]=(-0.796337866944)/(o+1.)+(0.0812355940636)
        ref[0,1,2,1]=(-0.333114331422)/(o+1.)+(1.22614026311)
        ref[0,2,0,0]=(0.516540083621)/(o+1.)+(-0.358761510118)
        ref[0,2,0,1]=(-2.72820411942)/(o+1.)+(0.15972705639)
        ref[0,2,1,0]=(-0.440149466937)/(o+1.)+(-0.124216123318)
        ref[0,2,1,1]=(-0.137940536755)/(o+1.)+(0.13992652657)
        ref[0,2,2,0]=(0.373759361058)/(o+1.)+(0.210996140796)
        ref[0,2,2,1]=(-1.07034621798)/(o+1.)+(1.2905860125)
        ref[0,3,0,0]=(0.923495049095)/(o+1.)+(-0.897043344396)
        ref[0,3,0,1]=(-0.20180799615)/(o+1.)+(0.272652457923)
        ref[0,3,1,0]=(-0.831052678484)/(o+1.)+(0.326929849662)
        ref[0,3,1,1]=(1.41523396695)/(o+1.)+(-0.5275927117)
        ref[0,3,2,0]=(0.405715363821)/(o+1.)+(0.535330516568)
        ref[0,3,2,1]=(0.877269821733)/(o+1.)+(-0.666082812121)
        ref[0,4,0,0]=(0.884126703786)/(o+1.)+(-0.559427998262)
        ref[0,4,0,1]=(-1.36933304826)/(o+1.)+(0.263100182211)
        ref[0,4,1,0]=(0.871486955825)/(o+1.)+(0.381595437312)
        ref[0,4,1,1]=(1.21446122052)/(o+1.)+(-0.642161706726)
        ref[0,4,2,0]=(-1.82861338966)/(o+1.)+(0.507433118531)
        ref[0,4,2,1]=(0.0642039966469)/(o+1.)+(0.551348562604)
        ref[1,0,0,0]=(0.40922616303)/(o+1.)+(-0.474630105482)
        ref[1,0,0,1]=(1.63943889768)/(o+1.)+(0.285622914296)
        ref[1,0,1,0]=(0.0845961115353)/(o+1.)+(0.143628502705)
        ref[1,0,1,1]=(0.62129375936)/(o+1.)+(0.671645835511)
        ref[1,0,2,0]=(-1.2004575453)/(o+1.)+(0.701757992696)
        ref[1,0,2,1]=(0.435963491825)/(o+1.)+(0.26710538393)
        ref[1,1,0,0]=(0.738921325407)/(o+1.)+(-0.0258542444502)
        ref[1,1,0,1]=(1.95599148183)/(o+1.)+(0.652818492883)
        ref[1,1,1,0]=(0.500576647095)/(o+1.)+(0.857652485995)
        ref[1,1,1,1]=(-0.0151750762821)/(o+1.)+(0.238827857598)
        ref[1,1,2,0]=(1.26963005373)/(o+1.)+(-0.299065008888)
        ref[1,1,2,1]=(-0.82289389301)/(o+1.)+(0.14939351077)
        ref[1,2,0,0]=(-0.495951362447)/(o+1.)+(-0.532833788304)
        ref[1,2,0,1]=(-1.87785443639)/(o+1.)+(-0.521209467732)
        ref[1,2,1,0]=(-0.374085962184)/(o+1.)+(0.505583421209)
        ref[1,2,1,1]=(1.01773556605)/(o+1.)+(0.764863466171)
        ref[1,2,2,0]=(-1.54049109838)/(o+1.)+(-0.511296952595)
        ref[1,2,2,1]=(-1.06948953348)/(o+1.)+(-0.215299794811)
        ref[1,3,0,0]=(-1.20198777829)/(o+1.)+(-0.274360734317)
        ref[1,3,0,1]=(1.13519842695)/(o+1.)+(0.506840913294)
        ref[1,3,1,0]=(0.973820116369)/(o+1.)+(0.485786110977)
        ref[1,3,1,1]=(0.155263308828)/(o+1.)+(0.473436850817)
        ref[1,3,2,0]=(0.476069733748)/(o+1.)+(-0.228046397052)
        ref[1,3,2,1]=(-0.234851551487)/(o+1.)+(-0.574065011567)
        ref[1,4,0,0]=(1.95924908786)/(o+1.)+(-0.493262726539)
        ref[1,4,0,1]=(-0.448532656782)/(o+1.)+(0.393812699074)
        ref[1,4,1,0]=(0.235155577808)/(o+1.)+(0.756260258803)
        ref[1,4,1,1]=(0.0334282473144)/(o+1.)+(-0.837406551217)
        ref[1,4,2,0]=(0.568132489684)/(o+1.)+(-0.13055465798)
        ref[1,4,2,1]=(-0.380042596089)/(o+1.)+(0.243737215624)
        ref[2,0,0,0]=(1.76568035823)/(o+1.)+(0.0789834372326)
        ref[2,0,0,1]=(-0.20023919191)/(o+1.)+(0.901466917208)
        ref[2,0,1,0]=(-1.20715877778)/(o+1.)+(-0.992500439486)
        ref[2,0,1,1]=(-0.761913514624)/(o+1.)+(0.465965334128)
        ref[2,0,2,0]=(-0.944104213428)/(o+1.)+(-0.146881270459)
        ref[2,0,2,1]=(0.105283971659)/(o+1.)+(-1.30498791712)
        ref[2,1,0,0]=(-0.804721214141)/(o+1.)+(0.356736186609)
        ref[2,1,0,1]=(-0.0358563948854)/(o+1.)+(-0.24146509617)
        ref[2,1,1,0]=(-0.437556173502)/(o+1.)+(-0.387215578175)
        ref[2,1,1,1]=(-0.0786030951662)/(o+1.)+(0.654838009465)
        ref[2,1,2,0]=(-0.859200342885)/(o+1.)+(-0.112407873885)
        ref[2,1,2,1]=(-0.296679106354)/(o+1.)+(0.0491751507381)
        ref[2,2,0,0]=(0.462408531841)/(o+1.)+(-0.557975521593)
        ref[2,2,0,1]=(-0.0634225791893)/(o+1.)+(0.0468185300489)
        ref[2,2,1,0]=(1.08205349937)/(o+1.)+(0.621257561187)
        ref[2,2,1,1]=(0.88864389409)/(o+1.)+(0.849886105205)
        ref[2,2,2,0]=(2.01702912587)/(o+1.)+(0.755558425167)
        ref[2,2,2,1]=(1.41263009001)/(o+1.)+(0.331281133791)
        ref[2,3,0,0]=(1.91485193421)/(o+1.)+(0.0643042152414)
        ref[2,3,0,1]=(1.86113720031)/(o+1.)+(0.870090339646)
        ref[2,3,1,0]=(0.442533807126)/(o+1.)+(-0.690437405116)
        ref[2,3,1,1]=(1.52774754257)/(o+1.)+(-0.236667038211)
        ref[2,3,2,0]=(1.08558789233)/(o+1.)+(-0.346580362392)
        ref[2,3,2,1]=(-0.771732912166)/(o+1.)+(-0.49697903252)
        ref[2,4,0,0]=(1.0510676744)/(o+1.)+(0.18646253745)
        ref[2,4,0,1]=(-0.878875991119)/(o+1.)+(-0.271299000644)
        ref[2,4,1,0]=(-0.75026757693)/(o+1.)+(-0.168751100981)
        ref[2,4,1,1]=(0.526155063499)/(o+1.)+(1.041034953)
        ref[2,4,2,0]=(0.377792294719)/(o+1.)+(0.433767213151)
        ref[2,4,2,1]=(0.114142925831)/(o+1.)+(-0.972051850709)
        ref[3,0,0,0]=(0.15538316839)/(o+1.)+(-0.861229788808)
        ref[3,0,0,1]=(-0.725501605569)/(o+1.)+(-0.248009661269)
        ref[3,0,1,0]=(0.791443284981)/(o+1.)+(0.618118504587)
        ref[3,0,1,1]=(-0.411749416988)/(o+1.)+(0.125302421245)
        ref[3,0,2,0]=(0.0968028463245)/(o+1.)+(0.592149023513)
        ref[3,0,2,1]=(0.0947000244134)/(o+1.)+(0.286498958939)
        ref[3,1,0,0]=(0.823197755916)/(o+1.)+(-0.378154495725)
        ref[3,1,0,1]=(1.21588193285)/(o+1.)+(0.0126566819682)
        ref[3,1,1,0]=(0.529639159192)/(o+1.)+(-0.619204956255)
        ref[3,1,1,1]=(1.85472331764)/(o+1.)+(0.3317212712)
        ref[3,1,2,0]=(-0.0770331978898)/(o+1.)+(0.0545214351076)
        ref[3,1,2,1]=(0.46556780359)/(o+1.)+(-0.223576369421)
        ref[3,2,0,0]=(-0.242684948671)/(o+1.)+(-0.681739755992)
        ref[3,2,0,1]=(0.0243168377401)/(o+1.)+(0.0176531159642)
        ref[3,2,1,0]=(-0.154330975892)/(o+1.)+(-0.543310918519)
        ref[3,2,1,1]=(-0.217608478494)/(o+1.)+(0.204473606215)
        ref[3,2,2,0]=(-2.25460522533)/(o+1.)+(-0.0278277816531)
        ref[3,2,2,1]=(-1.16412458146)/(o+1.)+(0.0837125949668)
        ref[3,3,0,0]=(-0.26896142194)/(o+1.)+(-0.0679863842941)
        ref[3,3,0,1]=(1.40707422197)/(o+1.)+(0.173208598807)
        ref[3,3,1,0]=(-1.16306017168)/(o+1.)+(-0.450017469706)
        ref[3,3,1,1]=(0.549735617385)/(o+1.)+(0.57482947556)
        ref[3,3,2,0]=(-2.13652623456)/(o+1.)+(-0.482380241762)
        ref[3,3,2,1]=(-0.842498249905)/(o+1.)+(-0.0911371329748)
        ref[3,4,0,0]=(-0.534757324947)/(o+1.)+(-0.00767524698684)
        ref[3,4,0,1]=(-1.17219057908)/(o+1.)+(0.193625463097)
        ref[3,4,1,0]=(1.35932418827)/(o+1.)+(0.00591326217154)
        ref[3,4,1,1]=(-2.14785627668)/(o+1.)+(-0.890497472901)
        ref[3,4,2,0]=(-0.878191372868)/(o+1.)+(0.343817991868)
        ref[3,4,2,1]=(0.723684311348)/(o+1.)+(0.696170991395)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5, 3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunction_fromData_ReducedSolution_rank0(self):
      """
      tests integral of rank 0 Data on the Function

      assumptions: ReducedSolution(self.domain) exists
                   self.domain supports integral on Function
      """
      o=1
      dim=self.domain.getDim()
      w_ref=Function(self.domain)
      w=ReducedSolution(self.domain)
      x=w.getX()
      arg=Data(0,(),w)
      if dim==2:
        arg=(0.729700679932)*x[0]+(-1.38329769179)*x[1]
        ref=(0.329728333821)/(o+1.)+(-0.491662672839)
      else:
        arg=(-1.75076012766)*x[0]+(-0.0862990889349)*x[1]+(0.637429232833)*x[2]
        ref=(-0.577517867896)/(o+1.)+(-0.311056057931)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunction_fromData_ReducedSolution_rank1(self):
      """
      tests integral of rank 1 Data on the Function

      assumptions: ReducedSolution(self.domain) exists
                   self.domain supports integral on Function
      """
      o=1
      dim=self.domain.getDim()
      w_ref=Function(self.domain)
      w=ReducedSolution(self.domain)
      x=w.getX()
      arg=Data(0,(2,),w)
      ref=numpy.zeros((2,),numpy.float_)
      if dim==2:
        arg[0]=(1.0296233191)*x[0]+(-0.347985432388)*x[1]
        arg[1]=(-0.182082548052)*x[0]+(0.114822539064)*x[1]
        ref[0]=(0.876900301573)/(o+1.)+(-0.0976312074308)
        ref[1]=(-1.09276229827)/(o+1.)+(0.512751144643)
      else:
        arg[0]=(0.390938934986)*x[0]+(-0.17813109236)*x[1]+(0.92611177061)*x[2]
        arg[1]=(0.169562781554)*x[0]+(-0.149625135256)*x[1]+(0.357445984273)*x[2]
        ref[0]=(1.07308269316)/(o+1.)+(0.0329184600363)
        ref[1]=(-1.07527269336)/(o+1.)+(0.726328161966)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunction_fromData_ReducedSolution_rank2(self):
      """
      tests integral of rank 2 Data on the Function

      assumptions: ReducedSolution(self.domain) exists
                   self.domain supports integral on Function
      """
      o=1
      dim=self.domain.getDim()
      w_ref=Function(self.domain)
      w=ReducedSolution(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5),w)
      ref=numpy.zeros((4, 5),numpy.float_)
      if dim==2:
        arg[0,0]=(-0.798668686601)*x[0]+(-0.502017718729)*x[1]
        arg[0,1]=(-0.682368031728)*x[0]+(-0.915048032379)*x[1]
        arg[0,2]=(-0.0237771508029)*x[0]+(0.177441262563)*x[1]
        arg[0,3]=(0.135600942216)*x[0]+(0.496697614771)*x[1]
        arg[0,4]=(0.495272681631)*x[0]+(-0.457599050745)*x[1]
        arg[1,0]=(-0.252415627225)*x[0]+(-0.820872546406)*x[1]
        arg[1,1]=(-0.183362696688)*x[0]+(-0.721277471492)*x[1]
        arg[1,2]=(0.401101418876)*x[0]+(0.0483185737314)*x[1]
        arg[1,3]=(0.251246583055)*x[0]+(-0.837901564153)*x[1]
        arg[1,4]=(0.2984760032)*x[0]+(0.0504569221307)*x[1]
        arg[2,0]=(1.18468310151)*x[0]+(-0.545612792222)*x[1]
        arg[2,1]=(-1.08451540544)*x[0]+(0.820927137753)*x[1]
        arg[2,2]=(0.85435600905)*x[0]+(-0.309819803422)*x[1]
        arg[2,3]=(-0.184148030844)*x[0]+(0.978315441698)*x[1]
        arg[2,4]=(1.36943188922)*x[0]+(-1.3176984733)*x[1]
        arg[3,0]=(0.42189213134)*x[0]+(1.23097633986)*x[1]
        arg[3,1]=(0.0409353425792)*x[0]+(-1.15555830367)*x[1]
        arg[3,2]=(0.557942070395)*x[0]+(-0.591659744114)*x[1]
        arg[3,3]=(-1.07432449167)*x[0]+(-0.665475086161)*x[1]
        arg[3,4]=(-1.00421972544)*x[0]+(1.16737316771)*x[1]
        ref[0,0]=(-0.400434079596)/(o+1.)+(-0.450126162867)
        ref[0,1]=(-1.35836659237)/(o+1.)+(-0.11952473587)
        ref[0,2]=(0.154775934077)/(o+1.)+(-0.000555911158727)
        ref[0,3]=(-0.177525502969)/(o+1.)+(0.404912029978)
        ref[0,4]=(1.00122501632)/(o+1.)+(-0.481775692719)
        ref[1,0]=(-1.37507083806)/(o+1.)+(0.150891332214)
        ref[1,1]=(-1.37429063854)/(o+1.)+(0.234825235182)
        ref[1,2]=(1.30069236547)/(o+1.)+(-0.425636186429)
        ref[1,3]=(-0.414658324515)/(o+1.)+(-0.0859983282917)
        ref[1,4]=(0.757097138678)/(o+1.)+(-0.204082106674)
        ref[2,0]=(1.09244584366)/(o+1.)+(-0.226687767185)
        ref[2,1]=(0.109481169277)/(o+1.)+(-0.186534718481)
        ref[2,2]=(0.587125473152)/(o+1.)+(-0.0212946337622)
        ref[2,3]=(0.00878695699842)/(o+1.)+(0.392690226928)
        ref[2,4]=(0.306810290785)/(o+1.)+(-0.127538437433)
        ref[3,0]=(0.317826050636)/(o+1.)+(0.667521210281)
        ref[3,1]=(-0.276254182573)/(o+1.)+(-0.419184389261)
        ref[3,2]=(-0.753831729065)/(o+1.)+(0.360057027673)
        ref[3,3]=(-1.40505001195)/(o+1.)+(-0.167374782943)
        ref[3,4]=(0.53112066724)/(o+1.)+(-0.183983612488)
      else:
        arg[0,0]=(1.54447818842)*x[0]+(-0.560568590939)*x[1]+(-1.33749549019)*x[2]
        arg[0,1]=(-0.535544800192)*x[0]+(-0.732513980801)*x[1]+(0.309855030569)*x[2]
        arg[0,2]=(-0.45251604551)*x[0]+(-0.0933932995351)*x[1]+(-0.465724167794)*x[2]
        arg[0,3]=(0.10344455234)*x[0]+(-0.0451254971766)*x[1]+(-1.39912042607)*x[2]
        arg[0,4]=(0.617585449939)*x[0]+(1.27357810776)*x[1]+(-0.840161322302)*x[2]
        arg[1,0]=(0.650274355535)*x[0]+(0.547805408891)*x[1]+(0.314350326062)*x[2]
        arg[1,1]=(0.697200474806)*x[0]+(0.558681734309)*x[1]+(-0.0277936766536)*x[2]
        arg[1,2]=(-0.508578328229)*x[0]+(-0.513894226666)*x[1]+(-0.528417418023)*x[2]
        arg[1,3]=(-1.50391414876)*x[0]+(0.174080260823)*x[1]+(-1.48528641514)*x[2]
        arg[1,4]=(0.859470776434)*x[0]+(-0.121348493404)*x[1]+(1.69796346547)*x[2]
        arg[2,0]=(1.00593912049)*x[0]+(0.794254020467)*x[1]+(-1.02770358461)*x[2]
        arg[2,1]=(-1.64586076188)*x[0]+(-0.83927167526)*x[1]+(-1.13088449805)*x[2]
        arg[2,2]=(0.0628777004357)*x[0]+(-1.27316091149)*x[1]+(0.417655047748)*x[2]
        arg[2,3]=(-1.43661526118)*x[0]+(0.0676042964527)*x[1]+(0.174947572442)*x[2]
        arg[2,4]=(0.504347051098)*x[0]+(-1.62976096707)*x[1]+(0.61579932957)*x[2]
        arg[3,0]=(1.05212645875)*x[0]+(-1.10517003049)*x[1]+(0.902242097595)*x[2]
        arg[3,1]=(-0.997611854104)*x[0]+(-1.53611378293)*x[1]+(0.322835276897)*x[2]
        arg[3,2]=(-0.815135194015)*x[0]+(0.357734619962)*x[1]+(0.684966069635)*x[2]
        arg[3,3]=(0.4545417685)*x[0]+(0.386262464083)*x[1]+(0.131414279708)*x[2]
        arg[3,4]=(-0.222694646001)*x[0]+(-0.470499833621)*x[1]+(-0.593290862391)*x[2]
        ref[0,0]=(-0.593803934901)/(o+1.)+(0.120109021094)
        ref[0,1]=(0.0569115377867)/(o+1.)+(-0.507557644105)
        ref[0,2]=(-0.202741590274)/(o+1.)+(-0.404445961283)
        ref[0,3]=(-0.516888777015)/(o+1.)+(-0.411956296948)
        ref[0,4]=(0.52292787417)/(o+1.)+(0.264037180616)
        ref[1,0]=(-0.102685279742)/(o+1.)+(0.807557685115)
        ref[1,1]=(1.44527264276)/(o+1.)+(-0.108592055148)
        ref[1,2]=(-0.448767718864)/(o+1.)+(-0.551061127027)
        ref[1,3]=(-1.19936004685)/(o+1.)+(-0.807880128113)
        ref[1,4]=(1.09918755605)/(o+1.)+(0.668449096223)
        ref[2,0]=(-0.873708007453)/(o+1.)+(0.823098781902)
        ref[2,1]=(-1.9697776018)/(o+1.)+(-0.823119666698)
        ref[2,2]=(0.598023292545)/(o+1.)+(-0.695325727928)
        ref[2,3]=(-0.486077421455)/(o+1.)+(-0.353992985415)
        ref[2,4]=(0.65037669171)/(o+1.)+(-0.579995639058)
        ref[3,0]=(0.115401170876)/(o+1.)+(0.366898677493)
        ref[3,1]=(-0.836209920174)/(o+1.)+(-0.68734021998)
        ref[3,2]=(0.753819223079)/(o+1.)+(-0.263126863749)
        ref[3,3]=(-0.75294945942)/(o+1.)+(0.862583985856)
        ref[3,4]=(-1.44356156537)/(o+1.)+(0.0785381116785)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunction_fromData_ReducedSolution_rank3(self):
      """
      tests integral of rank 3 Data on the Function

      assumptions: ReducedSolution(self.domain) exists
                   self.domain supports integral on Function
      """
      o=1
      dim=self.domain.getDim()
      w_ref=Function(self.domain)
      w=ReducedSolution(self.domain)
      x=w.getX()
      arg=Data(0,(6, 2, 2),w)
      ref=numpy.zeros((6, 2, 2),numpy.float_)
      if dim==2:
        arg[0,0,0]=(-0.409425595197)*x[0]+(-0.992427840513)*x[1]
        arg[0,0,1]=(0.645693185569)*x[0]+(-1.38338827813)*x[1]
        arg[0,1,0]=(0.838847764041)*x[0]+(-1.01358491173)*x[1]
        arg[0,1,1]=(0.381938057912)*x[0]+(0.61979806953)*x[1]
        arg[1,0,0]=(-1.36783302742)*x[0]+(-0.132422986617)*x[1]
        arg[1,0,1]=(-0.137283745381)*x[0]+(0.633534018338)*x[1]
        arg[1,1,0]=(0.0698189930006)*x[0]+(-1.45980415637)*x[1]
        arg[1,1,1]=(-0.391725001578)*x[0]+(0.355258883756)*x[1]
        arg[2,0,0]=(-0.596091494216)*x[0]+(-0.911870026464)*x[1]
        arg[2,0,1]=(0.138007894626)*x[0]+(1.76859278751)*x[1]
        arg[2,1,0]=(-0.791803431843)*x[0]+(-0.813170959331)*x[1]
        arg[2,1,1]=(-1.05466714677)*x[0]+(-1.12822729059)*x[1]
        arg[3,0,0]=(-0.417845531677)*x[0]+(1.13689234148)*x[1]
        arg[3,0,1]=(-0.0323347813877)*x[0]+(1.14381520037)*x[1]
        arg[3,1,0]=(0.639524393892)*x[0]+(0.864696525716)*x[1]
        arg[3,1,1]=(0.593104139176)*x[0]+(0.812642260561)*x[1]
        arg[4,0,0]=(-0.0580034066457)*x[0]+(0.802938738147)*x[1]
        arg[4,0,1]=(0.722143376522)*x[0]+(0.843814429252)*x[1]
        arg[4,1,0]=(-0.574057925551)*x[0]+(-0.298232158953)*x[1]
        arg[4,1,1]=(0.422529038525)*x[0]+(0.841942812244)*x[1]
        arg[5,0,0]=(1.90272013575)*x[0]+(-1.84811498267)*x[1]
        arg[5,0,1]=(-1.52156314991)*x[0]+(-0.797833417606)*x[1]
        arg[5,1,0]=(0.723748480291)*x[0]+(0.8667659268)*x[1]
        arg[5,1,1]=(-0.70289499551)*x[0]+(-0.726595802952)*x[1]
        ref[0,0,0]=(-1.1719312491)/(o+1.)+(-0.114961093303)
        ref[0,0,1]=(0.229993657297)/(o+1.)+(-0.483844374927)
        ref[0,1,0]=(-0.630442924796)/(o+1.)+(0.227852888553)
        ref[0,1,1]=(-0.806392286115)/(o+1.)+(0.904064206778)
        ref[1,0,0]=(-1.38312106397)/(o+1.)+(-0.0585674750338)
        ref[1,0,1]=(0.595552955095)/(o+1.)+(-0.049651341069)
        ref[1,1,0]=(-1.63606241234)/(o+1.)+(0.123038624483)
        ref[1,1,1]=(-0.340463713286)/(o+1.)+(0.151998797732)
        ref[2,0,0]=(-1.31925158304)/(o+1.)+(-0.094354968822)
        ref[2,0,1]=(0.699277735512)/(o+1.)+(0.60366147331)
        ref[2,1,0]=(-0.219878898763)/(o+1.)+(-0.692547746205)
        ref[2,1,1]=(-1.46750249607)/(o+1.)+(-0.357695970647)
        ref[3,0,0]=(0.71941698906)/(o+1.)+(-0.000185089626866)
        ref[3,0,1]=(-0.526993313502)/(o+1.)+(0.819236866243)
        ref[3,1,0]=(1.51671320209)/(o+1.)+(-0.00624614123899)
        ref[3,1,1]=(1.05185683613)/(o+1.)+(0.176944781803)
        ref[4,0,0]=(1.20432711487)/(o+1.)+(-0.229695891683)
        ref[4,0,1]=(1.26863112516)/(o+1.)+(0.14866334031)
        ref[4,1,0]=(0.34377834331)/(o+1.)+(-0.608034213907)
        ref[4,1,1]=(-0.189568721058)/(o+1.)+(0.727020285914)
        ref[5,0,0]=(0.116981035747)/(o+1.)+(-0.0311879413343)
        ref[5,0,1]=(-0.638728138162)/(o+1.)+(-0.840334214679)
        ref[5,1,0]=(0.891634589919)/(o+1.)+(0.349439908586)
        ref[5,1,1]=(-0.552480077628)/(o+1.)+(-0.438505360418)
      else:
        arg[0,0,0]=(-0.662295604848)*x[0]+(1.04606412328)*x[1]+(-0.376008345205)*x[2]
        arg[0,0,1]=(1.25030158303)*x[0]+(-0.0735167236581)*x[1]+(1.31623868156)*x[2]
        arg[0,1,0]=(0.717953620717)*x[0]+(-0.700464709991)*x[1]+(0.0345381771501)*x[2]
        arg[0,1,1]=(0.225944321569)*x[0]+(-1.09510173632)*x[1]+(-0.371235420166)*x[2]
        arg[1,0,0]=(-0.0755865552996)*x[0]+(-0.168214695387)*x[1]+(-0.437427629408)*x[2]
        arg[1,0,1]=(-1.52674685128)*x[0]+(-0.408158359188)*x[1]+(0.750091649824)*x[2]
        arg[1,1,0]=(-0.572400948498)*x[0]+(-0.802826119419)*x[1]+(-1.0233358643)*x[2]
        arg[1,1,1]=(0.623156259061)*x[0]+(0.378774331134)*x[1]+(0.0419766309042)*x[2]
        arg[2,0,0]=(0.443271002813)*x[0]+(-0.930092172267)*x[1]+(-0.20199150178)*x[2]
        arg[2,0,1]=(0.408614848881)*x[0]+(-1.3007040171)*x[1]+(-0.872615408174)*x[2]
        arg[2,1,0]=(-0.174910742406)*x[0]+(1.64872582781)*x[1]+(0.701172036891)*x[2]
        arg[2,1,1]=(0.417145974895)*x[0]+(0.428983721984)*x[1]+(-0.000787616769604)*x[2]
        arg[3,0,0]=(0.655644872855)*x[0]+(-0.268612479186)*x[1]+(0.11779790898)*x[2]
        arg[3,0,1]=(-0.244670332245)*x[0]+(-0.857502890419)*x[1]+(0.111792341131)*x[2]
        arg[3,1,0]=(0.131303542183)*x[0]+(0.231230724482)*x[1]+(0.116997126513)*x[2]
        arg[3,1,1]=(0.725434968911)*x[0]+(1.00206695553)*x[1]+(-0.319715936784)*x[2]
        arg[4,0,0]=(1.30949067662)*x[0]+(-1.14809917605)*x[1]+(0.00529778192088)*x[2]
        arg[4,0,1]=(0.969626585498)*x[0]+(-0.517908035476)*x[1]+(1.0467168328)*x[2]
        arg[4,1,0]=(0.362824899886)*x[0]+(0.890586602899)*x[1]+(-0.135163953653)*x[2]
        arg[4,1,1]=(-0.10353577028)*x[0]+(-0.0997924233294)*x[1]+(0.308973949876)*x[2]
        arg[5,0,0]=(-1.37242244095)*x[0]+(-0.746928231432)*x[1]+(-1.37385481504)*x[2]
        arg[5,0,1]=(0.231730881472)*x[0]+(-0.250265629505)*x[1]+(-0.903599685792)*x[2]
        arg[5,1,0]=(-0.813783753893)*x[0]+(0.41081474894)*x[1]+(-0.541868665288)*x[2]
        arg[5,1,1]=(0.19296641571)*x[0]+(1.8056250646)*x[1]+(0.283675443031)*x[2]
        ref[0,0,0]=(0.859538687531)/(o+1.)+(-0.425889257151)
        ref[0,0,1]=(-0.0674840316504)/(o+1.)+(1.28025378629)
        ref[0,1,0]=(-0.76922088579)/(o+1.)+(0.410623986833)
        ref[0,1,1]=(-0.582981712778)/(o+1.)+(-0.328705561068)
        ref[1,0,0]=(1.25605113432)/(o+1.)+(-0.968640007206)
        ref[1,0,1]=(-1.59448994908)/(o+1.)+(0.204838194218)
        ref[1,1,0]=(-2.07497671334)/(o+1.)+(-0.161793109438)
        ref[1,1,1]=(0.199659680372)/(o+1.)+(0.422123770364)
        ref[2,0,0]=(-1.86970457224)/(o+1.)+(0.590445950504)
        ref[2,0,1]=(-1.09505237157)/(o+1.)+(-0.334826102411)
        ref[2,1,0]=(0.395146564722)/(o+1.)+(0.889920278785)
        ref[2,1,1]=(1.13315474193)/(o+1.)+(-0.143906330909)
        ref[3,0,0]=(-0.804398097072)/(o+1.)+(0.65461419986)
        ref[3,0,1]=(-0.829074327243)/(o+1.)+(-0.0806532771448)
        ref[3,1,0]=(0.126278821088)/(o+1.)+(0.176626286045)
        ref[3,1,1]=(-0.498753315826)/(o+1.)+(0.953269651743)
        ref[4,0,0]=(0.33849802718)/(o+1.)+(-0.0859043723417)
        ref[4,0,1]=(0.153172891455)/(o+1.)+(0.672631245684)
        ref[4,1,0]=(1.42803542207)/(o+1.)+(-0.154893936468)
        ref[4,1,1]=(0.605555698814)/(o+1.)+(-0.249954971274)
        ref[5,0,0]=(-1.21823919194)/(o+1.)+(-1.13748314774)
        ref[5,0,1]=(-1.13684484437)/(o+1.)+(0.107355205274)
        ref[5,1,0]=(-1.21991007491)/(o+1.)+(0.137536202333)
        ref[5,1,1]=(2.05825898341)/(o+1.)+(0.112003969962)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunction_fromData_ReducedSolution_rank4(self):
      """
      tests integral of rank 4 Data on the Function

      assumptions: ReducedSolution(self.domain) exists
                   self.domain supports integral on Function
      """
      o=1
      dim=self.domain.getDim()
      w_ref=Function(self.domain)
      w=ReducedSolution(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5, 3, 2),w)
      ref=numpy.zeros((4, 5, 3, 2),numpy.float_)
      if dim==2:
        arg[0,0,0,0]=(-0.2631296794)*x[0]+(-0.130117190301)*x[1]
        arg[0,0,0,1]=(0.635732952218)*x[0]+(-0.647555274619)*x[1]
        arg[0,0,1,0]=(1.06301179339)*x[0]+(-0.227813591181)*x[1]
        arg[0,0,1,1]=(-0.519979711376)*x[0]+(1.04747480856)*x[1]
        arg[0,0,2,0]=(0.0232818794686)*x[0]+(-1.14799687633)*x[1]
        arg[0,0,2,1]=(0.359605391616)*x[0]+(-1.32942998109)*x[1]
        arg[0,1,0,0]=(-0.555284020205)*x[0]+(0.11676202509)*x[1]
        arg[0,1,0,1]=(-1.83479976366)*x[0]+(0.413168061643)*x[1]
        arg[0,1,1,0]=(0.239736306037)*x[0]+(0.680002261596)*x[1]
        arg[0,1,1,1]=(0.0555978436642)*x[0]+(1.28910533087)*x[1]
        arg[0,1,2,0]=(0.732696470486)*x[0]+(0.538837927425)*x[1]
        arg[0,1,2,1]=(0.528018376979)*x[0]+(1.20428218411)*x[1]
        arg[0,2,0,0]=(0.863803666388)*x[0]+(0.187082373754)*x[1]
        arg[0,2,0,1]=(-1.41490739709)*x[0]+(-0.741935114995)*x[1]
        arg[0,2,1,0]=(0.0501211708116)*x[0]+(-0.370371645307)*x[1]
        arg[0,2,1,1]=(-0.0353423795999)*x[0]+(1.00911245551)*x[1]
        arg[0,2,2,0]=(1.79997397117)*x[0]+(0.240103369892)*x[1]
        arg[0,2,2,1]=(0.926052414873)*x[0]+(0.00914357056458)*x[1]
        arg[0,3,0,0]=(-1.38911305622)*x[0]+(-1.29285413766)*x[1]
        arg[0,3,0,1]=(0.464908270265)*x[0]+(-0.354824878801)*x[1]
        arg[0,3,1,0]=(0.17477491298)*x[0]+(0.882807721541)*x[1]
        arg[0,3,1,1]=(0.633177386897)*x[0]+(-0.312628707154)*x[1]
        arg[0,3,2,0]=(0.0841289229919)*x[0]+(-0.244085514309)*x[1]
        arg[0,3,2,1]=(0.198631529554)*x[0]+(-0.324387433152)*x[1]
        arg[0,4,0,0]=(-1.41503119851)*x[0]+(-0.760818812719)*x[1]
        arg[0,4,0,1]=(-0.0326236565753)*x[0]+(0.835716629913)*x[1]
        arg[0,4,1,0]=(1.84482629546)*x[0]+(-0.341557269651)*x[1]
        arg[0,4,1,1]=(-0.380600663728)*x[0]+(-0.295663906776)*x[1]
        arg[0,4,2,0]=(0.411308057157)*x[0]+(1.0109767907)*x[1]
        arg[0,4,2,1]=(-1.10494637617)*x[0]+(0.401374247774)*x[1]
        arg[1,0,0,0]=(0.437019391027)*x[0]+(0.0507256248679)*x[1]
        arg[1,0,0,1]=(-0.399457976892)*x[0]+(1.60118981524)*x[1]
        arg[1,0,1,0]=(-1.51289007675)*x[0]+(0.821406477166)*x[1]
        arg[1,0,1,1]=(0.405770274494)*x[0]+(0.637591733337)*x[1]
        arg[1,0,2,0]=(-1.4945593715)*x[0]+(1.0919176668)*x[1]
        arg[1,0,2,1]=(-0.482868528044)*x[0]+(0.578004764959)*x[1]
        arg[1,1,0,0]=(-1.44298476434)*x[0]+(0.197151370973)*x[1]
        arg[1,1,0,1]=(1.38517677391)*x[0]+(0.915352009234)*x[1]
        arg[1,1,1,0]=(1.69025511761)*x[0]+(0.0203268727501)*x[1]
        arg[1,1,1,1]=(0.615392847773)*x[0]+(-0.301873832451)*x[1]
        arg[1,1,2,0]=(0.3081860115)*x[0]+(-0.495230532585)*x[1]
        arg[1,1,2,1]=(-0.0258752746808)*x[0]+(0.441629273958)*x[1]
        arg[1,2,0,0]=(-0.947577742489)*x[0]+(0.218632343222)*x[1]
        arg[1,2,0,1]=(0.520029690788)*x[0]+(0.403806873912)*x[1]
        arg[1,2,1,0]=(1.32820278922)*x[0]+(-1.68314905373)*x[1]
        arg[1,2,1,1]=(1.30409876916)*x[0]+(-0.721819171819)*x[1]
        arg[1,2,2,0]=(1.37872038109)*x[0]+(0.0725139769178)*x[1]
        arg[1,2,2,1]=(0.35593830057)*x[0]+(0.28751298465)*x[1]
        arg[1,3,0,0]=(-1.17950311406)*x[0]+(-0.232819397686)*x[1]
        arg[1,3,0,1]=(0.17340184781)*x[0]+(-0.43329852484)*x[1]
        arg[1,3,1,0]=(-1.71125828064)*x[0]+(0.957284609667)*x[1]
        arg[1,3,1,1]=(-0.429043922461)*x[0]+(0.15749154348)*x[1]
        arg[1,3,2,0]=(-1.42626208855)*x[0]+(-1.12812746347)*x[1]
        arg[1,3,2,1]=(0.490516645689)*x[0]+(0.559039786338)*x[1]
        arg[1,4,0,0]=(0.436395831147)*x[0]+(0.382691724897)*x[1]
        arg[1,4,0,1]=(1.27903773839)*x[0]+(-0.0642376846324)*x[1]
        arg[1,4,1,0]=(0.503159792753)*x[0]+(0.392973700403)*x[1]
        arg[1,4,1,1]=(0.0881500114326)*x[0]+(0.834121935361)*x[1]
        arg[1,4,2,0]=(-0.620745962116)*x[0]+(-0.0770814791601)*x[1]
        arg[1,4,2,1]=(0.0231684675117)*x[0]+(-1.33994827808)*x[1]
        arg[2,0,0,0]=(-0.972311570714)*x[0]+(-0.235293375636)*x[1]
        arg[2,0,0,1]=(0.21633355466)*x[0]+(0.11325401613)*x[1]
        arg[2,0,1,0]=(0.922986727965)*x[0]+(-1.00516246282)*x[1]
        arg[2,0,1,1]=(-0.740778981838)*x[0]+(0.616728971608)*x[1]
        arg[2,0,2,0]=(1.73201175195)*x[0]+(-1.1634325209)*x[1]
        arg[2,0,2,1]=(1.15829744421)*x[0]+(-1.87219685708)*x[1]
        arg[2,1,0,0]=(-0.458901790255)*x[0]+(1.13812715713)*x[1]
        arg[2,1,0,1]=(1.03315725208)*x[0]+(-0.112487270341)*x[1]
        arg[2,1,1,0]=(-1.7070220527)*x[0]+(0.126058081856)*x[1]
        arg[2,1,1,1]=(0.863301459388)*x[0]+(0.226670498194)*x[1]
        arg[2,1,2,0]=(-1.0012992544)*x[0]+(-0.195604893926)*x[1]
        arg[2,1,2,1]=(-0.128048233768)*x[0]+(0.827383267773)*x[1]
        arg[2,2,0,0]=(-0.0876478859699)*x[0]+(1.69600082596)*x[1]
        arg[2,2,0,1]=(-0.616278041482)*x[0]+(-0.995548774612)*x[1]
        arg[2,2,1,0]=(0.780699602657)*x[0]+(-0.721498796198)*x[1]
        arg[2,2,1,1]=(-1.17591990575)*x[0]+(-0.66080847955)*x[1]
        arg[2,2,2,0]=(1.71259481962)*x[0]+(-1.12456334234)*x[1]
        arg[2,2,2,1]=(-0.721345747555)*x[0]+(-1.28222132706)*x[1]
        arg[2,3,0,0]=(-1.25971324146)*x[0]+(-0.00559590465243)*x[1]
        arg[2,3,0,1]=(-0.49234245798)*x[0]+(0.30900436328)*x[1]
        arg[2,3,1,0]=(1.3068919966)*x[0]+(-0.891484126193)*x[1]
        arg[2,3,1,1]=(0.319277203381)*x[0]+(-0.535113401495)*x[1]
        arg[2,3,2,0]=(0.529480518887)*x[0]+(0.00452686996008)*x[1]
        arg[2,3,2,1]=(-1.0978905592)*x[0]+(-0.421513714335)*x[1]
        arg[2,4,0,0]=(-1.27711509307)*x[0]+(-1.42208296178)*x[1]
        arg[2,4,0,1]=(-0.621292314868)*x[0]+(-1.10962518414)*x[1]
        arg[2,4,1,0]=(-0.183478331213)*x[0]+(1.0775900659)*x[1]
        arg[2,4,1,1]=(1.07354953663)*x[0]+(0.869765287054)*x[1]
        arg[2,4,2,0]=(-0.696050139647)*x[0]+(-0.120769338549)*x[1]
        arg[2,4,2,1]=(-0.895597167617)*x[0]+(1.41025466606)*x[1]
        arg[3,0,0,0]=(1.15262876611)*x[0]+(-0.505428168285)*x[1]
        arg[3,0,0,1]=(-0.277585299138)*x[0]+(0.352683243558)*x[1]
        arg[3,0,1,0]=(0.218694567952)*x[0]+(-0.170325565499)*x[1]
        arg[3,0,1,1]=(0.297325594036)*x[0]+(0.459308129811)*x[1]
        arg[3,0,2,0]=(-0.572509624935)*x[0]+(0.756188981433)*x[1]
        arg[3,0,2,1]=(0.226462444009)*x[0]+(-1.42100784266)*x[1]
        arg[3,1,0,0]=(-0.28388824767)*x[0]+(-0.306064847514)*x[1]
        arg[3,1,0,1]=(-0.805831316478)*x[0]+(0.756933495912)*x[1]
        arg[3,1,1,0]=(1.37344361233)*x[0]+(-0.377600924857)*x[1]
        arg[3,1,1,1]=(-0.0725302533884)*x[0]+(-0.425074450201)*x[1]
        arg[3,1,2,0]=(-0.066659431931)*x[0]+(1.39252394891)*x[1]
        arg[3,1,2,1]=(0.593440019224)*x[0]+(-0.945629564385)*x[1]
        arg[3,2,0,0]=(0.101850471598)*x[0]+(0.0552356415787)*x[1]
        arg[3,2,0,1]=(-0.407603005372)*x[0]+(-0.849371670514)*x[1]
        arg[3,2,1,0]=(0.435403192155)*x[0]+(0.213673097377)*x[1]
        arg[3,2,1,1]=(0.424081872219)*x[0]+(-0.0800529063085)*x[1]
        arg[3,2,2,0]=(0.189428134309)*x[0]+(-0.205813522374)*x[1]
        arg[3,2,2,1]=(-0.51449771505)*x[0]+(0.0570738169919)*x[1]
        arg[3,3,0,0]=(1.05431625119)*x[0]+(-0.0583139615539)*x[1]
        arg[3,3,0,1]=(0.0292261008496)*x[0]+(-0.569482715214)*x[1]
        arg[3,3,1,0]=(-0.155861158232)*x[0]+(0.628422800294)*x[1]
        arg[3,3,1,1]=(-1.24050157391)*x[0]+(-1.83949885089)*x[1]
        arg[3,3,2,0]=(-0.140709404689)*x[0]+(0.975529832713)*x[1]
        arg[3,3,2,1]=(-0.290824210426)*x[0]+(-0.198373932879)*x[1]
        arg[3,4,0,0]=(0.627236720719)*x[0]+(-0.306676521516)*x[1]
        arg[3,4,0,1]=(-0.964303098626)*x[0]+(-0.701698363587)*x[1]
        arg[3,4,1,0]=(-0.0447860906704)*x[0]+(0.324659175162)*x[1]
        arg[3,4,1,1]=(1.28273443124)*x[0]+(-0.764329261143)*x[1]
        arg[3,4,2,0]=(-0.023232709502)*x[0]+(-0.124231212653)*x[1]
        arg[3,4,2,1]=(0.510754136991)*x[0]+(0.570492207442)*x[1]
        ref[0,0,0,0]=(0.648764087616)/(o+1.)+(-0.521005478658)
        ref[0,0,0,1]=(-0.0127476670646)/(o+1.)+(0.000462672331599)
        ref[0,0,1,0]=(0.258601710827)/(o+1.)+(0.288298245692)
        ref[0,0,1,1]=(0.0577415806545)/(o+1.)+(0.234876758266)
        ref[0,0,2,0]=(-0.91129780757)/(o+1.)+(-0.106708594644)
        ref[0,0,2,1]=(0.0841946322525)/(o+1.)+(-0.527009610865)
        ref[0,1,0,0]=(0.111101563297)/(o+1.)+(-0.274811779206)
        ref[0,1,0,1]=(-0.504062576651)/(o+1.)+(-0.458784562683)
        ref[0,1,1,0]=(0.65560969277)/(o+1.)+(0.132064437431)
        ref[0,1,1,1]=(0.0866382135955)/(o+1.)+(0.629032480469)
        ref[0,1,2,0]=(1.36074036394)/(o+1.)+(-0.0446029830125)
        ref[0,1,2,1]=(1.16071753175)/(o+1.)+(0.285791514672)
        ref[0,2,0,0]=(0.693065154772)/(o+1.)+(0.178910442685)
        ref[0,2,0,1]=(-1.41209835364)/(o+1.)+(-0.372372079222)
        ref[0,2,1,0]=(-0.542625642859)/(o+1.)+(0.111187584182)
        ref[0,2,1,1]=(0.628714449405)/(o+1.)+(0.172527813253)
        ref[0,2,2,0]=(1.62887921583)/(o+1.)+(0.205599062617)
        ref[0,2,2,1]=(1.0640301541)/(o+1.)+(-0.0644170843301)
        ref[0,3,0,0]=(-1.09870552313)/(o+1.)+(-0.791630835375)
        ref[0,3,0,1]=(0.260706656546)/(o+1.)+(-0.0753116325413)
        ref[0,3,1,0]=(-0.0543842358149)/(o+1.)+(0.555983435168)
        ref[0,3,1,1]=(0.0369075211917)/(o+1.)+(0.141820579276)
        ref[0,3,2,0]=(-0.784250881508)/(o+1.)+(0.312147145095)
        ref[0,3,2,1]=(-0.178450092422)/(o+1.)+(0.0263470944113)
        ref[0,4,0,0]=(-0.993852999635)/(o+1.)+(-0.590998505799)
        ref[0,4,0,1]=(-0.0449039600312)/(o+1.)+(0.423998466684)
        ref[0,4,1,0]=(0.583535069477)/(o+1.)+(0.459866978166)
        ref[0,4,1,1]=(-0.251020886878)/(o+1.)+(-0.212621841813)
        ref[0,4,2,0]=(0.408688561613)/(o+1.)+(0.506798143121)
        ref[0,4,2,1]=(0.537090447747)/(o+1.)+(-0.620331288072)
        ref[1,0,0,0]=(0.37428496839)/(o+1.)+(0.0567300237522)
        ref[1,0,0,1]=(0.777672833448)/(o+1.)+(0.212029502451)
        ref[1,0,1,0]=(-0.745422608709)/(o+1.)+(0.026969504563)
        ref[1,0,1,1]=(0.466434983875)/(o+1.)+(0.288463511978)
        ref[1,0,2,0]=(0.237457788829)/(o+1.)+(-0.320049746767)
        ref[1,0,2,1]=(0.346974181328)/(o+1.)+(-0.125918972207)
        ref[1,1,0,0]=(-0.823737519034)/(o+1.)+(-0.211047937165)
        ref[1,1,0,1]=(1.00324369541)/(o+1.)+(0.648642543871)
        ref[1,1,1,0]=(0.674013158596)/(o+1.)+(0.518284415882)
        ref[1,1,1,1]=(0.403142463029)/(o+1.)+(-0.0448117238535)
        ref[1,1,2,0]=(0.0399328921703)/(o+1.)+(-0.113488706628)
        ref[1,1,2,1]=(0.111446633645)/(o+1.)+(0.152153682816)
        ref[1,2,0,0]=(-0.481646721692)/(o+1.)+(-0.123649338787)
        ref[1,2,0,1]=(0.857170000026)/(o+1.)+(0.033333282337)
        ref[1,2,1,0]=(-0.445362398126)/(o+1.)+(0.0452080668082)
        ref[1,2,1,1]=(-0.256201827386)/(o+1.)+(0.419240712362)
        ref[1,2,2,0]=(1.02991996311)/(o+1.)+(0.210657197448)
        ref[1,2,2,1]=(0.355299316895)/(o+1.)+(0.144075984162)
        ref[1,3,0,0]=(-0.679726322154)/(o+1.)+(-0.366298094796)
        ref[1,3,0,1]=(-0.67885027329)/(o+1.)+(0.20947679813)
        ref[1,3,1,0]=(-0.163081485233)/(o+1.)+(-0.295446092868)
        ref[1,3,1,1]=(-0.035768623673)/(o+1.)+(-0.117891877654)
        ref[1,3,2,0]=(-0.639368943038)/(o+1.)+(-0.957510304493)
        ref[1,3,2,1]=(0.160615066001)/(o+1.)+(0.444470683013)
        ref[1,4,0,0]=(1.0649487216)/(o+1.)+(-0.122930582777)
        ref[1,4,0,1]=(0.22766574122)/(o+1.)+(0.493567156271)
        ref[1,4,1,0]=(-0.481296537412)/(o+1.)+(0.688715015285)
        ref[1,4,1,1]=(-0.195268237132)/(o+1.)+(0.558770091963)
        ref[1,4,2,0]=(0.180618258117)/(o+1.)+(-0.439222849696)
        ref[1,4,2,1]=(0.0261012984826)/(o+1.)+(-0.671440554524)
        ref[2,0,0,0]=(-0.301101739758)/(o+1.)+(-0.453251603296)
        ref[2,0,0,1]=(0.00878006761908)/(o+1.)+(0.160403751586)
        ref[2,0,1,0]=(-0.80011300584)/(o+1.)+(0.35896863549)
        ref[2,0,1,1]=(0.00248831604776)/(o+1.)+(-0.0632691631389)
        ref[2,0,2,0]=(-0.0569220251136)/(o+1.)+(0.312750628078)
        ref[2,0,2,1]=(-0.440109525096)/(o+1.)+(-0.136894943888)
        ref[2,1,0,0]=(-0.465652661854)/(o+1.)+(0.572439014362)
        ref[2,1,0,1]=(0.0522958647723)/(o+1.)+(0.434187058485)
        ref[2,1,1,0]=(0.00194260180185)/(o+1.)+(-0.791453286324)
        ref[2,1,1,1]=(0.981996325082)/(o+1.)+(0.0539878162501)
        ref[2,1,2,0]=(-0.243334369882)/(o+1.)+(-0.476784889223)
        ref[2,1,2,1]=(1.53208640175)/(o+1.)+(-0.416375683873)
        ref[2,2,0,0]=(1.27849139532)/(o+1.)+(0.164930772337)
        ref[2,2,0,1]=(-1.4813889023)/(o+1.)+(-0.0652189568992)
        ref[2,2,1,0]=(-0.728282325831)/(o+1.)+(0.393741566145)
        ref[2,2,1,1]=(-0.537765426582)/(o+1.)+(-0.649481479359)
        ref[2,2,2,0]=(-0.266224153785)/(o+1.)+(0.427127815533)
        ref[2,2,2,1]=(-0.862349098277)/(o+1.)+(-0.570608988167)
        ref[2,3,0,0]=(-1.09340368619)/(o+1.)+(-0.0859527299593)
        ref[2,3,0,1]=(1.02842400947)/(o+1.)+(-0.605881052086)
        ref[2,3,1,0]=(0.0131810342609)/(o+1.)+(0.20111341807)
        ref[2,3,1,1]=(-1.11244447248)/(o+1.)+(0.448304137186)
        ref[2,3,2,0]=(-0.978348257673)/(o+1.)+(0.75617782326)
        ref[2,3,2,1]=(-0.140801959181)/(o+1.)+(-0.689301157175)
        ref[2,4,0,0]=(-0.733659962936)/(o+1.)+(-0.982769045955)
        ref[2,4,0,1]=(-1.6278906672)/(o+1.)+(-0.0515134159034)
        ref[2,4,1,0]=(1.52117858199)/(o+1.)+(-0.313533423654)
        ref[2,4,1,1]=(0.621221455007)/(o+1.)+(0.661046684337)
        ref[2,4,2,0]=(-0.157580028393)/(o+1.)+(-0.329619724901)
        ref[2,4,2,1]=(0.52303412622)/(o+1.)+(-0.00418831388809)
        ref[3,0,0,0]=(0.337890126112)/(o+1.)+(0.154655235856)
        ref[3,0,0,1]=(-0.100490423466)/(o+1.)+(0.0877941839427)
        ref[3,0,1,0]=(0.37549944857)/(o+1.)+(-0.163565223059)
        ref[3,0,1,1]=(-0.3212149071)/(o+1.)+(0.538924315474)
        ref[3,0,2,0]=(-0.359815373231)/(o+1.)+(0.271747364864)
        ref[3,0,2,1]=(-0.890575856923)/(o+1.)+(-0.151984770865)
        ref[3,1,0,0]=(-1.10384271628)/(o+1.)+(0.256944810546)
        ref[3,1,0,1]=(0.647425296535)/(o+1.)+(-0.348161558551)
        ref[3,1,1,0]=(1.41806770884)/(o+1.)+(-0.211112510685)
        ref[3,1,1,1]=(0.106528838317)/(o+1.)+(-0.302066770953)
        ref[3,1,2,0]=(0.917213295746)/(o+1.)+(0.204325610614)
        ref[3,1,2,1]=(-0.24738513406)/(o+1.)+(-0.0524022055505)
        ref[3,2,0,0]=(-0.0537683888373)/(o+1.)+(0.105427251007)
        ref[3,2,0,1]=(-1.24647336894)/(o+1.)+(-0.00525065347416)
        ref[3,2,1,0]=(0.187132797735)/(o+1.)+(0.230971745898)
        ref[3,2,1,1]=(0.492750417331)/(o+1.)+(-0.0743607257106)
        ref[3,2,2,0]=(0.062835017839)/(o+1.)+(-0.0396102029518)
        ref[3,2,2,1]=(-0.967272854749)/(o+1.)+(0.254924478345)
        ref[3,3,0,0]=(1.41291903854)/(o+1.)+(-0.208458374448)
        ref[3,3,0,1]=(-0.740922121718)/(o+1.)+(0.100332753677)
        ref[3,3,1,0]=(1.2839228132)/(o+1.)+(-0.405680585568)
        ref[3,3,1,1]=(-1.69516258968)/(o+1.)+(-0.69241891756)
        ref[3,3,2,0]=(0.216594952745)/(o+1.)+(0.309112737639)
        ref[3,3,2,1]=(-0.634806027047)/(o+1.)+(0.0728039418712)
        ref[3,4,0,0]=(-0.310491357088)/(o+1.)+(0.315525778146)
        ref[3,4,0,1]=(-0.751915044154)/(o+1.)+(-0.457043209029)
        ref[3,4,1,0]=(-0.339963591836)/(o+1.)+(0.309918338163)
        ref[3,4,1,1]=(0.0720250143477)/(o+1.)+(0.223190077872)
        ref[3,4,2,0]=(-0.0508554115866)/(o+1.)+(-0.0483042552841)
        ref[3,4,2,1]=(0.754037720807)/(o+1.)+(0.163604311813)
      else:
        arg[0,0,0,0]=(-1.24350959032)*x[0]+(-0.580181370761)*x[1]+(-0.174543112173)*x[2]
        arg[0,0,0,1]=(1.43450421585)*x[0]+(0.717062915298)*x[1]+(-0.91866562143)*x[2]
        arg[0,0,1,0]=(0.0188823859409)*x[0]+(1.24568141807)*x[1]+(0.0842604494481)*x[2]
        arg[0,0,1,1]=(0.871591276766)*x[0]+(0.653031319605)*x[1]+(0.037714556611)*x[2]
        arg[0,0,2,0]=(1.60473786555)*x[0]+(1.47509195705)*x[1]+(-0.103485610136)*x[2]
        arg[0,0,2,1]=(1.68030531336)*x[0]+(0.551201281257)*x[1]+(1.26951824468)*x[2]
        arg[0,1,0,0]=(0.491638014782)*x[0]+(1.26264219762)*x[1]+(-0.921297094204)*x[2]
        arg[0,1,0,1]=(0.20108233628)*x[0]+(-1.36096031252)*x[1]+(-0.878077046379)*x[2]
        arg[0,1,1,0]=(0.900775475455)*x[0]+(0.193423726606)*x[1]+(-0.139955126775)*x[2]
        arg[0,1,1,1]=(-0.0354326925578)*x[0]+(-0.464646718718)*x[1]+(1.49240476583)*x[2]
        arg[0,1,2,0]=(-1.40991971664)*x[0]+(-1.7747715584)*x[1]+(-0.96387650203)*x[2]
        arg[0,1,2,1]=(-0.476443682267)*x[0]+(0.55076135194)*x[1]+(0.190289559512)*x[2]
        arg[0,2,0,0]=(0.535490315669)*x[0]+(0.566915366497)*x[1]+(-0.41513643529)*x[2]
        arg[0,2,0,1]=(-0.18986562478)*x[0]+(-0.62347050233)*x[1]+(1.18004907973)*x[2]
        arg[0,2,1,0]=(-0.288440432828)*x[0]+(0.0375642125411)*x[1]+(0.574653643212)*x[2]
        arg[0,2,1,1]=(1.03327739612)*x[0]+(-0.0443874572943)*x[1]+(0.972142765323)*x[2]
        arg[0,2,2,0]=(-1.27131901306)*x[0]+(0.619179457229)*x[1]+(-0.436029340417)*x[2]
        arg[0,2,2,1]=(0.0573099137019)*x[0]+(0.211373076629)*x[1]+(0.859991510967)*x[2]
        arg[0,3,0,0]=(0.650974277613)*x[0]+(-0.926352792387)*x[1]+(-1.84479828618)*x[2]
        arg[0,3,0,1]=(-0.293748564815)*x[0]+(0.53683587198)*x[1]+(1.51147453744)*x[2]
        arg[0,3,1,0]=(0.837889627485)*x[0]+(0.62568225238)*x[1]+(0.365321068424)*x[2]
        arg[0,3,1,1]=(-0.671841278343)*x[0]+(-0.00822894090357)*x[1]+(-1.35180265459)*x[2]
        arg[0,3,2,0]=(0.823443588183)*x[0]+(-0.366903894987)*x[1]+(-0.878402795844)*x[2]
        arg[0,3,2,1]=(-0.856292830088)*x[0]+(-0.134584368493)*x[1]+(-0.299126849425)*x[2]
        arg[0,4,0,0]=(0.62792423481)*x[0]+(0.781414088068)*x[1]+(-1.12299220031)*x[2]
        arg[0,4,0,1]=(-0.00743646774917)*x[0]+(0.52784004028)*x[1]+(-1.44284109205)*x[2]
        arg[0,4,1,0]=(-0.855519018884)*x[0]+(-0.222306698876)*x[1]+(0.0502517275952)*x[2]
        arg[0,4,1,1]=(-0.311060622512)*x[0]+(0.321283094479)*x[1]+(-0.484147152851)*x[2]
        arg[0,4,2,0]=(-1.43220161062)*x[0]+(-0.27822988291)*x[1]+(-0.692367106541)*x[2]
        arg[0,4,2,1]=(1.3956686934)*x[0]+(-1.26426333906)*x[1]+(-1.09343602989)*x[2]
        arg[1,0,0,0]=(0.122525927098)*x[0]+(-0.484009414389)*x[1]+(-0.822036495419)*x[2]
        arg[1,0,0,1]=(-0.276748173334)*x[0]+(0.35307031133)*x[1]+(-1.23466402387)*x[2]
        arg[1,0,1,0]=(-0.479355493223)*x[0]+(-0.56368304348)*x[1]+(0.54679107219)*x[2]
        arg[1,0,1,1]=(-0.483618214285)*x[0]+(1.60438722757)*x[1]+(0.542389963596)*x[2]
        arg[1,0,2,0]=(0.130988624202)*x[0]+(-0.449468495509)*x[1]+(0.254640520023)*x[2]
        arg[1,0,2,1]=(1.13790232796)*x[0]+(-0.716218913004)*x[1]+(-0.193031264955)*x[2]
        arg[1,1,0,0]=(-0.0653152151059)*x[0]+(1.07890922508)*x[1]+(0.324995864884)*x[2]
        arg[1,1,0,1]=(0.769151345051)*x[0]+(-1.35911748672)*x[1]+(0.299326803785)*x[2]
        arg[1,1,1,0]=(-0.431128032697)*x[0]+(1.09261915758)*x[1]+(-0.029871998484)*x[2]
        arg[1,1,1,1]=(-0.922618915502)*x[0]+(1.30318141394)*x[1]+(-0.324039981399)*x[2]
        arg[1,1,2,0]=(-0.668178375077)*x[0]+(0.132203602988)*x[1]+(0.236035070297)*x[2]
        arg[1,1,2,1]=(-0.518594419598)*x[0]+(-0.0429695477263)*x[1]+(1.11877941395)*x[2]
        arg[1,2,0,0]=(0.610776644247)*x[0]+(-0.023900598958)*x[1]+(0.0480141870788)*x[2]
        arg[1,2,0,1]=(0.562736426016)*x[0]+(-0.294969624732)*x[1]+(-0.23898469143)*x[2]
        arg[1,2,1,0]=(0.645173726197)*x[0]+(-1.18365074993)*x[1]+(0.102581636468)*x[2]
        arg[1,2,1,1]=(-0.518976597355)*x[0]+(0.372817596528)*x[1]+(0.49035249397)*x[2]
        arg[1,2,2,0]=(0.0192623731103)*x[0]+(-0.241279406644)*x[1]+(-0.420015571981)*x[2]
        arg[1,2,2,1]=(0.318641827599)*x[0]+(-1.5357263392)*x[1]+(0.350150621892)*x[2]
        arg[1,3,0,0]=(-0.719401628191)*x[0]+(-1.77806447133)*x[1]+(1.19750443209)*x[2]
        arg[1,3,0,1]=(-1.00122204304)*x[0]+(0.675162216383)*x[1]+(1.1180105058)*x[2]
        arg[1,3,1,0]=(0.331149947519)*x[0]+(0.310832741708)*x[1]+(0.087866556795)*x[2]
        arg[1,3,1,1]=(-0.156343054316)*x[0]+(1.00796776488)*x[1]+(-1.53460740228)*x[2]
        arg[1,3,2,0]=(0.714240747972)*x[0]+(0.968170992322)*x[1]+(-0.962819191598)*x[2]
        arg[1,3,2,1]=(-0.200412114379)*x[0]+(-0.173781234544)*x[1]+(0.899584749309)*x[2]
        arg[1,4,0,0]=(-1.38011177642)*x[0]+(-0.316506470647)*x[1]+(0.266952180469)*x[2]
        arg[1,4,0,1]=(0.343359684584)*x[0]+(-1.13244466712)*x[1]+(-0.81840155125)*x[2]
        arg[1,4,1,0]=(0.295299129957)*x[0]+(0.199810907018)*x[1]+(0.631717407771)*x[2]
        arg[1,4,1,1]=(-0.46840784478)*x[0]+(-0.402683932206)*x[1]+(-0.782209293177)*x[2]
        arg[1,4,2,0]=(-1.8006389736)*x[0]+(0.119727360087)*x[1]+(0.482076626168)*x[2]
        arg[1,4,2,1]=(0.96553885281)*x[0]+(0.425097333238)*x[1]+(0.54533551085)*x[2]
        arg[2,0,0,0]=(0.932725602425)*x[0]+(-0.513780652569)*x[1]+(0.275742915648)*x[2]
        arg[2,0,0,1]=(1.10053578072)*x[0]+(0.787895323287)*x[1]+(-1.46544221328)*x[2]
        arg[2,0,1,0]=(0.83657269666)*x[0]+(-1.15717482853)*x[1]+(1.1090271297)*x[2]
        arg[2,0,1,1]=(1.59370135994)*x[0]+(0.452015708845)*x[1]+(0.0862230059082)*x[2]
        arg[2,0,2,0]=(-0.954832509866)*x[0]+(0.51192257473)*x[1]+(-1.48499771643)*x[2]
        arg[2,0,2,1]=(-0.556993361475)*x[0]+(0.438110051421)*x[1]+(0.237254018143)*x[2]
        arg[2,1,0,0]=(-0.318980966383)*x[0]+(-0.263071719097)*x[1]+(-0.348122442404)*x[2]
        arg[2,1,0,1]=(0.100191287326)*x[0]+(-0.422996999974)*x[1]+(-0.850702087112)*x[2]
        arg[2,1,1,0]=(-0.198942237141)*x[0]+(0.277459568173)*x[1]+(-1.66195774421)*x[2]
        arg[2,1,1,1]=(-1.62211381863)*x[0]+(-0.13417844308)*x[1]+(1.07441481356)*x[2]
        arg[2,1,2,0]=(-0.183120762554)*x[0]+(-0.158546687568)*x[1]+(1.05938565108)*x[2]
        arg[2,1,2,1]=(-0.530596736933)*x[0]+(-0.507872303289)*x[1]+(0.0437745116021)*x[2]
        arg[2,2,0,0]=(-0.430338834548)*x[0]+(-1.01989755268)*x[1]+(0.0364152729771)*x[2]
        arg[2,2,0,1]=(-0.55129175645)*x[0]+(0.283907226465)*x[1]+(0.281403899376)*x[2]
        arg[2,2,1,0]=(0.119036949181)*x[0]+(0.187265158741)*x[1]+(-0.0845814658663)*x[2]
        arg[2,2,1,1]=(1.39900884968)*x[0]+(1.42806588781)*x[1]+(0.211444691268)*x[2]
        arg[2,2,2,0]=(-0.609664666581)*x[0]+(-0.29988332429)*x[1]+(-0.28125800646)*x[2]
        arg[2,2,2,1]=(-1.77545247157)*x[0]+(0.337828315779)*x[1]+(-0.239141292616)*x[2]
        arg[2,3,0,0]=(-1.23141604656)*x[0]+(0.322998749903)*x[1]+(-1.70685281816)*x[2]
        arg[2,3,0,1]=(-1.4416737434)*x[0]+(1.49740114642)*x[1]+(0.732057753882)*x[2]
        arg[2,3,1,0]=(-1.59112622077)*x[0]+(0.665317566947)*x[1]+(0.826417253499)*x[2]
        arg[2,3,1,1]=(-1.22000830454)*x[0]+(0.618713896065)*x[1]+(1.27354984534)*x[2]
        arg[2,3,2,0]=(0.125639723221)*x[0]+(0.663241510656)*x[1]+(-0.426459288281)*x[2]
        arg[2,3,2,1]=(1.39120212155)*x[0]+(0.9115748844)*x[1]+(1.77889030223)*x[2]
        arg[2,4,0,0]=(-0.217927656387)*x[0]+(1.20391185439)*x[1]+(-0.188197734908)*x[2]
        arg[2,4,0,1]=(-0.442009082947)*x[0]+(0.399477101959)*x[1]+(-1.33974368956)*x[2]
        arg[2,4,1,0]=(0.636534809382)*x[0]+(0.872923470503)*x[1]+(-0.922286702006)*x[2]
        arg[2,4,1,1]=(-0.831840624431)*x[0]+(0.946223248622)*x[1]+(0.380300039133)*x[2]
        arg[2,4,2,0]=(0.0803088903901)*x[0]+(-0.857685204574)*x[1]+(-0.479475079431)*x[2]
        arg[2,4,2,1]=(-1.7233942516)*x[0]+(-1.38771345941)*x[1]+(0.761958201633)*x[2]
        arg[3,0,0,0]=(-0.738032075796)*x[0]+(1.33264583681)*x[1]+(-1.21101988385)*x[2]
        arg[3,0,0,1]=(1.40336327022)*x[0]+(-0.77078885213)*x[1]+(1.70960627204)*x[2]
        arg[3,0,1,0]=(0.452477629041)*x[0]+(0.235432009434)*x[1]+(1.69314009822)*x[2]
        arg[3,0,1,1]=(-0.816213143646)*x[0]+(-1.42050350309)*x[1]+(-0.0806105456758)*x[2]
        arg[3,0,2,0]=(0.0400708086023)*x[0]+(-0.336948609628)*x[1]+(-0.113687293995)*x[2]
        arg[3,0,2,1]=(0.414077492095)*x[0]+(-1.06937716207)*x[1]+(1.25972041173)*x[2]
        arg[3,1,0,0]=(-1.02975555201)*x[0]+(-0.123306171085)*x[1]+(-1.74750909033)*x[2]
        arg[3,1,0,1]=(-1.01988698009)*x[0]+(-0.142188487205)*x[1]+(0.274514515492)*x[2]
        arg[3,1,1,0]=(0.440626586141)*x[0]+(-0.0569580581917)*x[1]+(-0.77065631367)*x[2]
        arg[3,1,1,1]=(0.987903067972)*x[0]+(-1.47543187274)*x[1]+(-0.0888962732893)*x[2]
        arg[3,1,2,0]=(0.150506204416)*x[0]+(0.035476396146)*x[1]+(-0.852061690418)*x[2]
        arg[3,1,2,1]=(-1.15179800132)*x[0]+(-0.425570974979)*x[1]+(1.10092214699)*x[2]
        arg[3,2,0,0]=(1.01255188057)*x[0]+(1.24279465527)*x[1]+(1.25190246102)*x[2]
        arg[3,2,0,1]=(-1.16275538385)*x[0]+(1.13672085108)*x[1]+(0.131953731231)*x[2]
        arg[3,2,1,0]=(0.774290747173)*x[0]+(0.275410883089)*x[1]+(-0.825692266439)*x[2]
        arg[3,2,1,1]=(-1.07241838074)*x[0]+(-0.0974675737726)*x[1]+(1.1856849697)*x[2]
        arg[3,2,2,0]=(0.234209248938)*x[0]+(0.756893234577)*x[1]+(-0.365141055079)*x[2]
        arg[3,2,2,1]=(-1.6668858309)*x[0]+(-0.916769120288)*x[1]+(-0.716365681217)*x[2]
        arg[3,3,0,0]=(-0.36890215249)*x[0]+(-0.581062815398)*x[1]+(1.6280164507)*x[2]
        arg[3,3,0,1]=(-0.933634743559)*x[0]+(-0.0657657779145)*x[1]+(1.35011362804)*x[2]
        arg[3,3,1,0]=(1.13739289322)*x[0]+(0.0103208127738)*x[1]+(1.01628193014)*x[2]
        arg[3,3,1,1]=(0.037979696459)*x[0]+(-1.76926650495)*x[1]+(-1.28602572757)*x[2]
        arg[3,3,2,0]=(0.797830431467)*x[0]+(0.732108360923)*x[1]+(-0.436390867333)*x[2]
        arg[3,3,2,1]=(-0.571865447888)*x[0]+(0.895109047357)*x[1]+(-0.673234648776)*x[2]
        arg[3,4,0,0]=(-1.59667195158)*x[0]+(-1.06146105331)*x[1]+(-1.81255923783)*x[2]
        arg[3,4,0,1]=(-0.0721327214647)*x[0]+(1.23982376787)*x[1]+(1.43101011)*x[2]
        arg[3,4,1,0]=(0.117288395573)*x[0]+(-0.37678966212)*x[1]+(-1.14107285089)*x[2]
        arg[3,4,1,1]=(-0.303270532321)*x[0]+(1.33784402324)*x[1]+(-0.651624796413)*x[2]
        arg[3,4,2,0]=(0.215849013256)*x[0]+(0.247315697644)*x[1]+(0.404644392153)*x[2]
        arg[3,4,2,1]=(0.975781190478)*x[0]+(-0.252285063767)*x[1]+(-0.145694910854)*x[2]
        ref[0,0,0,0]=(-0.718934738033)/(o+1.)+(-0.639649667608)
        ref[0,0,0,1]=(1.01007662915)/(o+1.)+(0.111412440284)
        ref[0,0,1,0]=(0.827868953278)/(o+1.)+(0.26047765009)
        ref[0,0,1,1]=(1.48122352358)/(o+1.)+(0.0405568147034)
        ref[0,0,2,0]=(1.33282099436)/(o+1.)+(0.821761609054)
        ref[0,0,2,1]=(1.65298089862)/(o+1.)+(0.924021970342)
        ref[0,1,0,0]=(0.930162571453)/(o+1.)+(-0.0485897266263)
        ref[0,1,0,1]=(-0.347599158269)/(o+1.)+(-0.845177932176)
        ref[0,1,1,0]=(0.623152121663)/(o+1.)+(0.165545976811)
        ref[0,1,1,1]=(0.0506217690841)/(o+1.)+(0.470851792737)
        ref[0,1,2,0]=(-1.99181491207)/(o+1.)+(-1.0783764325)
        ref[0,1,2,1]=(-0.731065007801)/(o+1.)+(0.497836118493)
        ref[0,2,0,0]=(-0.0812249009151)/(o+1.)+(0.384247073896)
        ref[0,2,0,1]=(-0.115768505253)/(o+1.)+(0.241240728938)
        ref[0,2,1,0]=(0.152728009607)/(o+1.)+(0.0855247066589)
        ref[0,2,1,1]=(1.43086526503)/(o+1.)+(0.265083719563)
        ref[0,2,2,0]=(-0.967906369418)/(o+1.)+(-0.060131263415)
        ref[0,2,2,1]=(-0.269269238419)/(o+1.)+(0.698971869859)
        ref[0,3,0,0]=(-1.71659801402)/(o+1.)+(-0.201789393467)
        ref[0,3,0,1]=(0.265343635273)/(o+1.)+(0.744609104665)
        ref[0,3,1,0]=(1.83021875237)/(o+1.)+(-0.000662902039879)
        ref[0,3,1,1]=(-0.317529833375)/(o+1.)+(-0.857171520229)
        ref[0,3,2,0]=(-0.705757693524)/(o+1.)+(0.141947295438)
        ref[0,3,2,1]=(-1.27000495392)/(o+1.)+(-0.00999954704202)
        ref[0,4,0,0]=(0.061892522425)/(o+1.)+(0.112226800069)
        ref[0,4,0,1]=(-0.747062267148)/(o+1.)+(-0.0876876261864)
        ref[0,4,1,0]=(-1.19488448842)/(o+1.)+(0.0836552491261)
        ref[0,4,1,1]=(1.36573976292)/(o+1.)+(-0.9198322219)
        ref[0,4,2,0]=(-1.67982141697)/(o+1.)+(-0.361488591554)
        ref[0,4,2,1]=(-0.598050084547)/(o+1.)+(-0.181990295501)
        ref[1,0,0,0]=(-2.45927674426)/(o+1.)+(0.637878380777)
        ref[1,0,0,1]=(-0.575223026885)/(o+1.)+(-0.291559429493)
        ref[1,0,1,0]=(0.0871765959469)/(o+1.)+(-0.29171203023)
        ref[1,0,1,1]=(0.94050575309)/(o+1.)+(0.361326611893)
        ref[1,0,2,0]=(-0.306500880914)/(o+1.)+(0.121330764815)
        ref[1,0,2,1]=(1.60748296414)/(o+1.)+(-0.689415407072)
        ref[1,1,0,0]=(0.640401899755)/(o+1.)+(0.34909398755)
        ref[1,1,0,1]=(0.134022909144)/(o+1.)+(-0.212331123516)
        ref[1,1,1,0]=(0.248008519593)/(o+1.)+(0.191805303402)
        ref[1,1,1,1]=(0.638911732328)/(o+1.)+(-0.291194607647)
        ref[1,1,2,0]=(0.109264462855)/(o+1.)+(-0.204602082323)
        ref[1,1,2,1]=(-0.185963112656)/(o+1.)+(0.371589279643)
        ref[1,2,0,0]=(-0.590669733193)/(o+1.)+(0.61277998278)
        ref[1,2,0,1]=(-1.15856908266)/(o+1.)+(0.593675596257)
        ref[1,2,1,0]=(-0.451642187813)/(o+1.)+(0.00787340027166)
        ref[1,2,1,1]=(-0.428757501259)/(o+1.)+(0.386475497201)
        ref[1,2,2,0]=(-0.414532914271)/(o+1.)+(-0.113749845622)
        ref[1,2,2,1]=(-0.513613710163)/(o+1.)+(-0.176660089771)
        ref[1,3,0,0]=(-0.890487934256)/(o+1.)+(-0.20473686659)
        ref[1,3,0,1]=(1.64512779208)/(o+1.)+(-0.426588556466)
        ref[1,3,1,0]=(0.831527595966)/(o+1.)+(-0.0508391749725)
        ref[1,3,1,1]=(-0.718921003194)/(o+1.)+(0.0179691557385)
        ref[1,3,2,0]=(0.405959213301)/(o+1.)+(0.156816667697)
        ref[1,3,2,1]=(0.426319486322)/(o+1.)+(0.0495359570318)
        ref[1,4,0,0]=(-1.11364330874)/(o+1.)+(-0.158011378929)
        ref[1,4,0,1]=(-0.0993967101457)/(o+1.)+(-0.754044911817)
        ref[1,4,1,0]=(0.979493241598)/(o+1.)+(0.0736671015741)
        ref[1,4,1,1]=(-1.78692478393)/(o+1.)+(0.0668118568857)
        ref[1,4,2,0]=(-0.691473647622)/(o+1.)+(-0.253680669863)
        ref[1,4,2,1]=(1.16914997228)/(o+1.)+(0.383410862311)
        ref[2,0,0,0]=(0.790187638399)/(o+1.)+(-0.047749886448)
        ref[2,0,0,1]=(0.429625240932)/(o+1.)+(-0.00331817510394)
        ref[2,0,1,0]=(-0.101344997877)/(o+1.)+(0.444884997853)
        ref[2,0,1,1]=(2.15519953039)/(o+1.)+(-0.0116297278494)
        ref[2,0,2,0]=(-1.40159594866)/(o+1.)+(-0.263155851453)
        ref[2,0,2,1]=(-0.658391287862)/(o+1.)+(0.388380997976)
        ref[2,1,0,0]=(-0.773933630328)/(o+1.)+(-0.0781207487777)
        ref[2,1,0,1]=(0.0015551432788)/(o+1.)+(-0.587531471519)
        ref[2,1,1,0]=(0.122693324758)/(o+1.)+(-0.853066868967)
        ref[2,1,1,1]=(-0.219255630453)/(o+1.)+(-0.231310908847)
        ref[2,1,2,0]=(1.33176471286)/(o+1.)+(-0.307023255951)
        ref[2,1,2,1]=(-1.67471265259)/(o+1.)+(0.340009061986)
        ref[2,2,0,0]=(-0.727222063897)/(o+1.)+(-0.343299525178)
        ref[2,2,0,1]=(-1.87667590405)/(o+1.)+(0.945347636719)
        ref[2,2,1,0]=(0.776003927859)/(o+1.)+(-0.277141642902)
        ref[2,2,1,1]=(1.62850322537)/(o+1.)+(0.705008101698)
        ref[2,2,2,0]=(1.31869548331)/(o+1.)+(-1.25475074032)
        ref[2,2,2,1]=(-1.71691241115)/(o+1.)+(0.0200734813728)
        ref[2,3,0,0]=(-0.37741220744)/(o+1.)+(-1.11892895368)
        ref[2,3,0,1]=(0.684763737686)/(o+1.)+(0.0515107096073)
        ref[2,3,1,0]=(-0.0572023243047)/(o+1.)+(-0.0210945380092)
        ref[2,3,1,1]=(0.031457290932)/(o+1.)+(0.320399072966)
        ref[2,3,2,0]=(-0.71613334286)/(o+1.)+(0.539277644228)
        ref[2,3,2,1]=(2.10068005021)/(o+1.)+(0.990493628981)
        ref[2,4,0,0]=(0.830379294407)/(o+1.)+(-0.016296415655)
        ref[2,4,0,1]=(-1.05197706105)/(o+1.)+(-0.165149304749)
        ref[2,4,1,0]=(-0.0854959336547)/(o+1.)+(0.336333755767)
        ref[2,4,1,1]=(1.15845594915)/(o+1.)+(-0.331886642912)
        ref[2,4,2,0]=(-1.25248721318)/(o+1.)+(-0.00218209021539)
        ref[2,4,2,1]=(-0.991752052473)/(o+1.)+(-0.678698728452)
        ref[3,0,0,0]=(-0.262071815816)/(o+1.)+(-0.177167153508)
        ref[3,0,0,1]=(1.20598261029)/(o+1.)+(0.568099039922)
        ref[3,0,1,0]=(1.21237099375)/(o+1.)+(0.584339371474)
        ref[3,0,1,1]=(-1.04628498282)/(o+1.)+(-0.635521104799)
        ref[3,0,2,0]=(0.630843868383)/(o+1.)+(-0.520704481702)
        ref[3,0,2,1]=(1.56136208039)/(o+1.)+(-0.478470669318)
        ref[3,1,0,0]=(-0.372752943047)/(o+1.)+(-1.26390893519)
        ref[3,1,0,1]=(0.192139112377)/(o+1.)+(-0.53985003209)
        ref[3,1,1,0]=(-1.18480584023)/(o+1.)+(0.398909027255)
        ref[3,1,1,1]=(0.402069277892)/(o+1.)+(-0.489247177975)
        ref[3,1,2,0]=(0.27608763117)/(o+1.)+(-0.471083360513)
        ref[3,1,2,1]=(-0.273636427696)/(o+1.)+(-0.101405200809)
        ref[3,2,0,0]=(1.92572771773)/(o+1.)+(0.790760639563)
        ref[3,2,0,1]=(0.16264122626)/(o+1.)+(-0.0283610138982)
        ref[3,2,1,0]=(0.377561018039)/(o+1.)+(-0.076775827108)
        ref[3,2,1,1]=(-0.652466873272)/(o+1.)+(0.334132944227)
        ref[3,2,2,0]=(-0.0304721826842)/(o+1.)+(0.32821680556)
        ref[3,2,2,1]=(-1.52932569088)/(o+1.)+(-0.885347470761)
        ref[3,3,0,0]=(-0.451938162279)/(o+1.)+(0.564994822546)
        ref[3,3,0,1]=(-0.477583137662)/(o+1.)+(0.414148122113)
        ref[3,3,1,0]=(1.38037325689)/(o+1.)+(0.391811189625)
        ref[3,3,1,1]=(-1.01333159097)/(o+1.)+(-1.00199047255)
        ref[3,3,2,0]=(1.3222644334)/(o+1.)+(-0.114358254173)
        ref[3,3,2,1]=(-0.248989333346)/(o+1.)+(-0.0505008579799)
        ref[3,4,0,0]=(-1.92196761736)/(o+1.)+(-1.27436231268)
        ref[3,4,0,1]=(0.894404962457)/(o+1.)+(0.852148096971)
        ref[3,4,1,0]=(0.868763436571)/(o+1.)+(-1.134668777)
        ref[3,4,1,1]=(0.438747123022)/(o+1.)+(-0.0278992142568)
        ref[3,4,2,0]=(0.532575431402)/(o+1.)+(0.167616835826)
        ref[3,4,2,1]=(0.796744496799)/(o+1.)+(-0.109471640471)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5, 3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunction_fromData_ReducedContinuousFunction_rank0(self):
      """
      tests integral of rank 0 Data on the Function

      assumptions: ReducedContinuousFunction(self.domain) exists
                   self.domain supports integral on Function
      """
      o=1
      dim=self.domain.getDim()
      w_ref=Function(self.domain)
      w=ReducedContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(),w)
      if dim==2:
        arg=(0.729700679932)*x[0]+(-1.38329769179)*x[1]
        ref=(0.329728333821)/(o+1.)+(-0.491662672839)
      else:
        arg=(-1.75076012766)*x[0]+(-0.0862990889349)*x[1]+(0.637429232833)*x[2]
        ref=(-0.577517867896)/(o+1.)+(-0.311056057931)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunction_fromData_ReducedContinuousFunction_rank1(self):
      """
      tests integral of rank 1 Data on the Function

      assumptions: ReducedContinuousFunction(self.domain) exists
                   self.domain supports integral on Function
      """
      o=1
      dim=self.domain.getDim()
      w_ref=Function(self.domain)
      w=ReducedContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(2,),w)
      ref=numpy.zeros((2,),numpy.float_)
      if dim==2:
        arg[0]=(1.0296233191)*x[0]+(-0.347985432388)*x[1]
        arg[1]=(-0.182082548052)*x[0]+(0.114822539064)*x[1]
        ref[0]=(0.876900301573)/(o+1.)+(-0.0976312074308)
        ref[1]=(-1.09276229827)/(o+1.)+(0.512751144643)
      else:
        arg[0]=(0.390938934986)*x[0]+(-0.17813109236)*x[1]+(0.92611177061)*x[2]
        arg[1]=(0.169562781554)*x[0]+(-0.149625135256)*x[1]+(0.357445984273)*x[2]
        ref[0]=(1.07308269316)/(o+1.)+(0.0329184600363)
        ref[1]=(-1.07527269336)/(o+1.)+(0.726328161966)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunction_fromData_ReducedContinuousFunction_rank2(self):
      """
      tests integral of rank 2 Data on the Function

      assumptions: ReducedContinuousFunction(self.domain) exists
                   self.domain supports integral on Function
      """
      o=1
      dim=self.domain.getDim()
      w_ref=Function(self.domain)
      w=ReducedContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5),w)
      ref=numpy.zeros((4, 5),numpy.float_)
      if dim==2:
        arg[0,0]=(-0.798668686601)*x[0]+(-0.502017718729)*x[1]
        arg[0,1]=(-0.682368031728)*x[0]+(-0.915048032379)*x[1]
        arg[0,2]=(-0.0237771508029)*x[0]+(0.177441262563)*x[1]
        arg[0,3]=(0.135600942216)*x[0]+(0.496697614771)*x[1]
        arg[0,4]=(0.495272681631)*x[0]+(-0.457599050745)*x[1]
        arg[1,0]=(-0.252415627225)*x[0]+(-0.820872546406)*x[1]
        arg[1,1]=(-0.183362696688)*x[0]+(-0.721277471492)*x[1]
        arg[1,2]=(0.401101418876)*x[0]+(0.0483185737314)*x[1]
        arg[1,3]=(0.251246583055)*x[0]+(-0.837901564153)*x[1]
        arg[1,4]=(0.2984760032)*x[0]+(0.0504569221307)*x[1]
        arg[2,0]=(1.18468310151)*x[0]+(-0.545612792222)*x[1]
        arg[2,1]=(-1.08451540544)*x[0]+(0.820927137753)*x[1]
        arg[2,2]=(0.85435600905)*x[0]+(-0.309819803422)*x[1]
        arg[2,3]=(-0.184148030844)*x[0]+(0.978315441698)*x[1]
        arg[2,4]=(1.36943188922)*x[0]+(-1.3176984733)*x[1]
        arg[3,0]=(0.42189213134)*x[0]+(1.23097633986)*x[1]
        arg[3,1]=(0.0409353425792)*x[0]+(-1.15555830367)*x[1]
        arg[3,2]=(0.557942070395)*x[0]+(-0.591659744114)*x[1]
        arg[3,3]=(-1.07432449167)*x[0]+(-0.665475086161)*x[1]
        arg[3,4]=(-1.00421972544)*x[0]+(1.16737316771)*x[1]
        ref[0,0]=(-0.400434079596)/(o+1.)+(-0.450126162867)
        ref[0,1]=(-1.35836659237)/(o+1.)+(-0.11952473587)
        ref[0,2]=(0.154775934077)/(o+1.)+(-0.000555911158727)
        ref[0,3]=(-0.177525502969)/(o+1.)+(0.404912029978)
        ref[0,4]=(1.00122501632)/(o+1.)+(-0.481775692719)
        ref[1,0]=(-1.37507083806)/(o+1.)+(0.150891332214)
        ref[1,1]=(-1.37429063854)/(o+1.)+(0.234825235182)
        ref[1,2]=(1.30069236547)/(o+1.)+(-0.425636186429)
        ref[1,3]=(-0.414658324515)/(o+1.)+(-0.0859983282917)
        ref[1,4]=(0.757097138678)/(o+1.)+(-0.204082106674)
        ref[2,0]=(1.09244584366)/(o+1.)+(-0.226687767185)
        ref[2,1]=(0.109481169277)/(o+1.)+(-0.186534718481)
        ref[2,2]=(0.587125473152)/(o+1.)+(-0.0212946337622)
        ref[2,3]=(0.00878695699842)/(o+1.)+(0.392690226928)
        ref[2,4]=(0.306810290785)/(o+1.)+(-0.127538437433)
        ref[3,0]=(0.317826050636)/(o+1.)+(0.667521210281)
        ref[3,1]=(-0.276254182573)/(o+1.)+(-0.419184389261)
        ref[3,2]=(-0.753831729065)/(o+1.)+(0.360057027673)
        ref[3,3]=(-1.40505001195)/(o+1.)+(-0.167374782943)
        ref[3,4]=(0.53112066724)/(o+1.)+(-0.183983612488)
      else:
        arg[0,0]=(1.54447818842)*x[0]+(-0.560568590939)*x[1]+(-1.33749549019)*x[2]
        arg[0,1]=(-0.535544800192)*x[0]+(-0.732513980801)*x[1]+(0.309855030569)*x[2]
        arg[0,2]=(-0.45251604551)*x[0]+(-0.0933932995351)*x[1]+(-0.465724167794)*x[2]
        arg[0,3]=(0.10344455234)*x[0]+(-0.0451254971766)*x[1]+(-1.39912042607)*x[2]
        arg[0,4]=(0.617585449939)*x[0]+(1.27357810776)*x[1]+(-0.840161322302)*x[2]
        arg[1,0]=(0.650274355535)*x[0]+(0.547805408891)*x[1]+(0.314350326062)*x[2]
        arg[1,1]=(0.697200474806)*x[0]+(0.558681734309)*x[1]+(-0.0277936766536)*x[2]
        arg[1,2]=(-0.508578328229)*x[0]+(-0.513894226666)*x[1]+(-0.528417418023)*x[2]
        arg[1,3]=(-1.50391414876)*x[0]+(0.174080260823)*x[1]+(-1.48528641514)*x[2]
        arg[1,4]=(0.859470776434)*x[0]+(-0.121348493404)*x[1]+(1.69796346547)*x[2]
        arg[2,0]=(1.00593912049)*x[0]+(0.794254020467)*x[1]+(-1.02770358461)*x[2]
        arg[2,1]=(-1.64586076188)*x[0]+(-0.83927167526)*x[1]+(-1.13088449805)*x[2]
        arg[2,2]=(0.0628777004357)*x[0]+(-1.27316091149)*x[1]+(0.417655047748)*x[2]
        arg[2,3]=(-1.43661526118)*x[0]+(0.0676042964527)*x[1]+(0.174947572442)*x[2]
        arg[2,4]=(0.504347051098)*x[0]+(-1.62976096707)*x[1]+(0.61579932957)*x[2]
        arg[3,0]=(1.05212645875)*x[0]+(-1.10517003049)*x[1]+(0.902242097595)*x[2]
        arg[3,1]=(-0.997611854104)*x[0]+(-1.53611378293)*x[1]+(0.322835276897)*x[2]
        arg[3,2]=(-0.815135194015)*x[0]+(0.357734619962)*x[1]+(0.684966069635)*x[2]
        arg[3,3]=(0.4545417685)*x[0]+(0.386262464083)*x[1]+(0.131414279708)*x[2]
        arg[3,4]=(-0.222694646001)*x[0]+(-0.470499833621)*x[1]+(-0.593290862391)*x[2]
        ref[0,0]=(-0.593803934901)/(o+1.)+(0.120109021094)
        ref[0,1]=(0.0569115377867)/(o+1.)+(-0.507557644105)
        ref[0,2]=(-0.202741590274)/(o+1.)+(-0.404445961283)
        ref[0,3]=(-0.516888777015)/(o+1.)+(-0.411956296948)
        ref[0,4]=(0.52292787417)/(o+1.)+(0.264037180616)
        ref[1,0]=(-0.102685279742)/(o+1.)+(0.807557685115)
        ref[1,1]=(1.44527264276)/(o+1.)+(-0.108592055148)
        ref[1,2]=(-0.448767718864)/(o+1.)+(-0.551061127027)
        ref[1,3]=(-1.19936004685)/(o+1.)+(-0.807880128113)
        ref[1,4]=(1.09918755605)/(o+1.)+(0.668449096223)
        ref[2,0]=(-0.873708007453)/(o+1.)+(0.823098781902)
        ref[2,1]=(-1.9697776018)/(o+1.)+(-0.823119666698)
        ref[2,2]=(0.598023292545)/(o+1.)+(-0.695325727928)
        ref[2,3]=(-0.486077421455)/(o+1.)+(-0.353992985415)
        ref[2,4]=(0.65037669171)/(o+1.)+(-0.579995639058)
        ref[3,0]=(0.115401170876)/(o+1.)+(0.366898677493)
        ref[3,1]=(-0.836209920174)/(o+1.)+(-0.68734021998)
        ref[3,2]=(0.753819223079)/(o+1.)+(-0.263126863749)
        ref[3,3]=(-0.75294945942)/(o+1.)+(0.862583985856)
        ref[3,4]=(-1.44356156537)/(o+1.)+(0.0785381116785)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunction_fromData_ReducedContinuousFunction_rank3(self):
      """
      tests integral of rank 3 Data on the Function

      assumptions: ReducedContinuousFunction(self.domain) exists
                   self.domain supports integral on Function
      """
      o=1
      dim=self.domain.getDim()
      w_ref=Function(self.domain)
      w=ReducedContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(6, 2, 2),w)
      ref=numpy.zeros((6, 2, 2),numpy.float_)
      if dim==2:
        arg[0,0,0]=(-0.409425595197)*x[0]+(-0.992427840513)*x[1]
        arg[0,0,1]=(0.645693185569)*x[0]+(-1.38338827813)*x[1]
        arg[0,1,0]=(0.838847764041)*x[0]+(-1.01358491173)*x[1]
        arg[0,1,1]=(0.381938057912)*x[0]+(0.61979806953)*x[1]
        arg[1,0,0]=(-1.36783302742)*x[0]+(-0.132422986617)*x[1]
        arg[1,0,1]=(-0.137283745381)*x[0]+(0.633534018338)*x[1]
        arg[1,1,0]=(0.0698189930006)*x[0]+(-1.45980415637)*x[1]
        arg[1,1,1]=(-0.391725001578)*x[0]+(0.355258883756)*x[1]
        arg[2,0,0]=(-0.596091494216)*x[0]+(-0.911870026464)*x[1]
        arg[2,0,1]=(0.138007894626)*x[0]+(1.76859278751)*x[1]
        arg[2,1,0]=(-0.791803431843)*x[0]+(-0.813170959331)*x[1]
        arg[2,1,1]=(-1.05466714677)*x[0]+(-1.12822729059)*x[1]
        arg[3,0,0]=(-0.417845531677)*x[0]+(1.13689234148)*x[1]
        arg[3,0,1]=(-0.0323347813877)*x[0]+(1.14381520037)*x[1]
        arg[3,1,0]=(0.639524393892)*x[0]+(0.864696525716)*x[1]
        arg[3,1,1]=(0.593104139176)*x[0]+(0.812642260561)*x[1]
        arg[4,0,0]=(-0.0580034066457)*x[0]+(0.802938738147)*x[1]
        arg[4,0,1]=(0.722143376522)*x[0]+(0.843814429252)*x[1]
        arg[4,1,0]=(-0.574057925551)*x[0]+(-0.298232158953)*x[1]
        arg[4,1,1]=(0.422529038525)*x[0]+(0.841942812244)*x[1]
        arg[5,0,0]=(1.90272013575)*x[0]+(-1.84811498267)*x[1]
        arg[5,0,1]=(-1.52156314991)*x[0]+(-0.797833417606)*x[1]
        arg[5,1,0]=(0.723748480291)*x[0]+(0.8667659268)*x[1]
        arg[5,1,1]=(-0.70289499551)*x[0]+(-0.726595802952)*x[1]
        ref[0,0,0]=(-1.1719312491)/(o+1.)+(-0.114961093303)
        ref[0,0,1]=(0.229993657297)/(o+1.)+(-0.483844374927)
        ref[0,1,0]=(-0.630442924796)/(o+1.)+(0.227852888553)
        ref[0,1,1]=(-0.806392286115)/(o+1.)+(0.904064206778)
        ref[1,0,0]=(-1.38312106397)/(o+1.)+(-0.0585674750338)
        ref[1,0,1]=(0.595552955095)/(o+1.)+(-0.049651341069)
        ref[1,1,0]=(-1.63606241234)/(o+1.)+(0.123038624483)
        ref[1,1,1]=(-0.340463713286)/(o+1.)+(0.151998797732)
        ref[2,0,0]=(-1.31925158304)/(o+1.)+(-0.094354968822)
        ref[2,0,1]=(0.699277735512)/(o+1.)+(0.60366147331)
        ref[2,1,0]=(-0.219878898763)/(o+1.)+(-0.692547746205)
        ref[2,1,1]=(-1.46750249607)/(o+1.)+(-0.357695970647)
        ref[3,0,0]=(0.71941698906)/(o+1.)+(-0.000185089626866)
        ref[3,0,1]=(-0.526993313502)/(o+1.)+(0.819236866243)
        ref[3,1,0]=(1.51671320209)/(o+1.)+(-0.00624614123899)
        ref[3,1,1]=(1.05185683613)/(o+1.)+(0.176944781803)
        ref[4,0,0]=(1.20432711487)/(o+1.)+(-0.229695891683)
        ref[4,0,1]=(1.26863112516)/(o+1.)+(0.14866334031)
        ref[4,1,0]=(0.34377834331)/(o+1.)+(-0.608034213907)
        ref[4,1,1]=(-0.189568721058)/(o+1.)+(0.727020285914)
        ref[5,0,0]=(0.116981035747)/(o+1.)+(-0.0311879413343)
        ref[5,0,1]=(-0.638728138162)/(o+1.)+(-0.840334214679)
        ref[5,1,0]=(0.891634589919)/(o+1.)+(0.349439908586)
        ref[5,1,1]=(-0.552480077628)/(o+1.)+(-0.438505360418)
      else:
        arg[0,0,0]=(-0.662295604848)*x[0]+(1.04606412328)*x[1]+(-0.376008345205)*x[2]
        arg[0,0,1]=(1.25030158303)*x[0]+(-0.0735167236581)*x[1]+(1.31623868156)*x[2]
        arg[0,1,0]=(0.717953620717)*x[0]+(-0.700464709991)*x[1]+(0.0345381771501)*x[2]
        arg[0,1,1]=(0.225944321569)*x[0]+(-1.09510173632)*x[1]+(-0.371235420166)*x[2]
        arg[1,0,0]=(-0.0755865552996)*x[0]+(-0.168214695387)*x[1]+(-0.437427629408)*x[2]
        arg[1,0,1]=(-1.52674685128)*x[0]+(-0.408158359188)*x[1]+(0.750091649824)*x[2]
        arg[1,1,0]=(-0.572400948498)*x[0]+(-0.802826119419)*x[1]+(-1.0233358643)*x[2]
        arg[1,1,1]=(0.623156259061)*x[0]+(0.378774331134)*x[1]+(0.0419766309042)*x[2]
        arg[2,0,0]=(0.443271002813)*x[0]+(-0.930092172267)*x[1]+(-0.20199150178)*x[2]
        arg[2,0,1]=(0.408614848881)*x[0]+(-1.3007040171)*x[1]+(-0.872615408174)*x[2]
        arg[2,1,0]=(-0.174910742406)*x[0]+(1.64872582781)*x[1]+(0.701172036891)*x[2]
        arg[2,1,1]=(0.417145974895)*x[0]+(0.428983721984)*x[1]+(-0.000787616769604)*x[2]
        arg[3,0,0]=(0.655644872855)*x[0]+(-0.268612479186)*x[1]+(0.11779790898)*x[2]
        arg[3,0,1]=(-0.244670332245)*x[0]+(-0.857502890419)*x[1]+(0.111792341131)*x[2]
        arg[3,1,0]=(0.131303542183)*x[0]+(0.231230724482)*x[1]+(0.116997126513)*x[2]
        arg[3,1,1]=(0.725434968911)*x[0]+(1.00206695553)*x[1]+(-0.319715936784)*x[2]
        arg[4,0,0]=(1.30949067662)*x[0]+(-1.14809917605)*x[1]+(0.00529778192088)*x[2]
        arg[4,0,1]=(0.969626585498)*x[0]+(-0.517908035476)*x[1]+(1.0467168328)*x[2]
        arg[4,1,0]=(0.362824899886)*x[0]+(0.890586602899)*x[1]+(-0.135163953653)*x[2]
        arg[4,1,1]=(-0.10353577028)*x[0]+(-0.0997924233294)*x[1]+(0.308973949876)*x[2]
        arg[5,0,0]=(-1.37242244095)*x[0]+(-0.746928231432)*x[1]+(-1.37385481504)*x[2]
        arg[5,0,1]=(0.231730881472)*x[0]+(-0.250265629505)*x[1]+(-0.903599685792)*x[2]
        arg[5,1,0]=(-0.813783753893)*x[0]+(0.41081474894)*x[1]+(-0.541868665288)*x[2]
        arg[5,1,1]=(0.19296641571)*x[0]+(1.8056250646)*x[1]+(0.283675443031)*x[2]
        ref[0,0,0]=(0.859538687531)/(o+1.)+(-0.425889257151)
        ref[0,0,1]=(-0.0674840316504)/(o+1.)+(1.28025378629)
        ref[0,1,0]=(-0.76922088579)/(o+1.)+(0.410623986833)
        ref[0,1,1]=(-0.582981712778)/(o+1.)+(-0.328705561068)
        ref[1,0,0]=(1.25605113432)/(o+1.)+(-0.968640007206)
        ref[1,0,1]=(-1.59448994908)/(o+1.)+(0.204838194218)
        ref[1,1,0]=(-2.07497671334)/(o+1.)+(-0.161793109438)
        ref[1,1,1]=(0.199659680372)/(o+1.)+(0.422123770364)
        ref[2,0,0]=(-1.86970457224)/(o+1.)+(0.590445950504)
        ref[2,0,1]=(-1.09505237157)/(o+1.)+(-0.334826102411)
        ref[2,1,0]=(0.395146564722)/(o+1.)+(0.889920278785)
        ref[2,1,1]=(1.13315474193)/(o+1.)+(-0.143906330909)
        ref[3,0,0]=(-0.804398097072)/(o+1.)+(0.65461419986)
        ref[3,0,1]=(-0.829074327243)/(o+1.)+(-0.0806532771448)
        ref[3,1,0]=(0.126278821088)/(o+1.)+(0.176626286045)
        ref[3,1,1]=(-0.498753315826)/(o+1.)+(0.953269651743)
        ref[4,0,0]=(0.33849802718)/(o+1.)+(-0.0859043723417)
        ref[4,0,1]=(0.153172891455)/(o+1.)+(0.672631245684)
        ref[4,1,0]=(1.42803542207)/(o+1.)+(-0.154893936468)
        ref[4,1,1]=(0.605555698814)/(o+1.)+(-0.249954971274)
        ref[5,0,0]=(-1.21823919194)/(o+1.)+(-1.13748314774)
        ref[5,0,1]=(-1.13684484437)/(o+1.)+(0.107355205274)
        ref[5,1,0]=(-1.21991007491)/(o+1.)+(0.137536202333)
        ref[5,1,1]=(2.05825898341)/(o+1.)+(0.112003969962)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunction_fromData_ReducedContinuousFunction_rank4(self):
      """
      tests integral of rank 4 Data on the Function

      assumptions: ReducedContinuousFunction(self.domain) exists
                   self.domain supports integral on Function
      """
      o=1
      dim=self.domain.getDim()
      w_ref=Function(self.domain)
      w=ReducedContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5, 3, 2),w)
      ref=numpy.zeros((4, 5, 3, 2),numpy.float_)
      if dim==2:
        arg[0,0,0,0]=(-0.2631296794)*x[0]+(-0.130117190301)*x[1]
        arg[0,0,0,1]=(0.635732952218)*x[0]+(-0.647555274619)*x[1]
        arg[0,0,1,0]=(1.06301179339)*x[0]+(-0.227813591181)*x[1]
        arg[0,0,1,1]=(-0.519979711376)*x[0]+(1.04747480856)*x[1]
        arg[0,0,2,0]=(0.0232818794686)*x[0]+(-1.14799687633)*x[1]
        arg[0,0,2,1]=(0.359605391616)*x[0]+(-1.32942998109)*x[1]
        arg[0,1,0,0]=(-0.555284020205)*x[0]+(0.11676202509)*x[1]
        arg[0,1,0,1]=(-1.83479976366)*x[0]+(0.413168061643)*x[1]
        arg[0,1,1,0]=(0.239736306037)*x[0]+(0.680002261596)*x[1]
        arg[0,1,1,1]=(0.0555978436642)*x[0]+(1.28910533087)*x[1]
        arg[0,1,2,0]=(0.732696470486)*x[0]+(0.538837927425)*x[1]
        arg[0,1,2,1]=(0.528018376979)*x[0]+(1.20428218411)*x[1]
        arg[0,2,0,0]=(0.863803666388)*x[0]+(0.187082373754)*x[1]
        arg[0,2,0,1]=(-1.41490739709)*x[0]+(-0.741935114995)*x[1]
        arg[0,2,1,0]=(0.0501211708116)*x[0]+(-0.370371645307)*x[1]
        arg[0,2,1,1]=(-0.0353423795999)*x[0]+(1.00911245551)*x[1]
        arg[0,2,2,0]=(1.79997397117)*x[0]+(0.240103369892)*x[1]
        arg[0,2,2,1]=(0.926052414873)*x[0]+(0.00914357056458)*x[1]
        arg[0,3,0,0]=(-1.38911305622)*x[0]+(-1.29285413766)*x[1]
        arg[0,3,0,1]=(0.464908270265)*x[0]+(-0.354824878801)*x[1]
        arg[0,3,1,0]=(0.17477491298)*x[0]+(0.882807721541)*x[1]
        arg[0,3,1,1]=(0.633177386897)*x[0]+(-0.312628707154)*x[1]
        arg[0,3,2,0]=(0.0841289229919)*x[0]+(-0.244085514309)*x[1]
        arg[0,3,2,1]=(0.198631529554)*x[0]+(-0.324387433152)*x[1]
        arg[0,4,0,0]=(-1.41503119851)*x[0]+(-0.760818812719)*x[1]
        arg[0,4,0,1]=(-0.0326236565753)*x[0]+(0.835716629913)*x[1]
        arg[0,4,1,0]=(1.84482629546)*x[0]+(-0.341557269651)*x[1]
        arg[0,4,1,1]=(-0.380600663728)*x[0]+(-0.295663906776)*x[1]
        arg[0,4,2,0]=(0.411308057157)*x[0]+(1.0109767907)*x[1]
        arg[0,4,2,1]=(-1.10494637617)*x[0]+(0.401374247774)*x[1]
        arg[1,0,0,0]=(0.437019391027)*x[0]+(0.0507256248679)*x[1]
        arg[1,0,0,1]=(-0.399457976892)*x[0]+(1.60118981524)*x[1]
        arg[1,0,1,0]=(-1.51289007675)*x[0]+(0.821406477166)*x[1]
        arg[1,0,1,1]=(0.405770274494)*x[0]+(0.637591733337)*x[1]
        arg[1,0,2,0]=(-1.4945593715)*x[0]+(1.0919176668)*x[1]
        arg[1,0,2,1]=(-0.482868528044)*x[0]+(0.578004764959)*x[1]
        arg[1,1,0,0]=(-1.44298476434)*x[0]+(0.197151370973)*x[1]
        arg[1,1,0,1]=(1.38517677391)*x[0]+(0.915352009234)*x[1]
        arg[1,1,1,0]=(1.69025511761)*x[0]+(0.0203268727501)*x[1]
        arg[1,1,1,1]=(0.615392847773)*x[0]+(-0.301873832451)*x[1]
        arg[1,1,2,0]=(0.3081860115)*x[0]+(-0.495230532585)*x[1]
        arg[1,1,2,1]=(-0.0258752746808)*x[0]+(0.441629273958)*x[1]
        arg[1,2,0,0]=(-0.947577742489)*x[0]+(0.218632343222)*x[1]
        arg[1,2,0,1]=(0.520029690788)*x[0]+(0.403806873912)*x[1]
        arg[1,2,1,0]=(1.32820278922)*x[0]+(-1.68314905373)*x[1]
        arg[1,2,1,1]=(1.30409876916)*x[0]+(-0.721819171819)*x[1]
        arg[1,2,2,0]=(1.37872038109)*x[0]+(0.0725139769178)*x[1]
        arg[1,2,2,1]=(0.35593830057)*x[0]+(0.28751298465)*x[1]
        arg[1,3,0,0]=(-1.17950311406)*x[0]+(-0.232819397686)*x[1]
        arg[1,3,0,1]=(0.17340184781)*x[0]+(-0.43329852484)*x[1]
        arg[1,3,1,0]=(-1.71125828064)*x[0]+(0.957284609667)*x[1]
        arg[1,3,1,1]=(-0.429043922461)*x[0]+(0.15749154348)*x[1]
        arg[1,3,2,0]=(-1.42626208855)*x[0]+(-1.12812746347)*x[1]
        arg[1,3,2,1]=(0.490516645689)*x[0]+(0.559039786338)*x[1]
        arg[1,4,0,0]=(0.436395831147)*x[0]+(0.382691724897)*x[1]
        arg[1,4,0,1]=(1.27903773839)*x[0]+(-0.0642376846324)*x[1]
        arg[1,4,1,0]=(0.503159792753)*x[0]+(0.392973700403)*x[1]
        arg[1,4,1,1]=(0.0881500114326)*x[0]+(0.834121935361)*x[1]
        arg[1,4,2,0]=(-0.620745962116)*x[0]+(-0.0770814791601)*x[1]
        arg[1,4,2,1]=(0.0231684675117)*x[0]+(-1.33994827808)*x[1]
        arg[2,0,0,0]=(-0.972311570714)*x[0]+(-0.235293375636)*x[1]
        arg[2,0,0,1]=(0.21633355466)*x[0]+(0.11325401613)*x[1]
        arg[2,0,1,0]=(0.922986727965)*x[0]+(-1.00516246282)*x[1]
        arg[2,0,1,1]=(-0.740778981838)*x[0]+(0.616728971608)*x[1]
        arg[2,0,2,0]=(1.73201175195)*x[0]+(-1.1634325209)*x[1]
        arg[2,0,2,1]=(1.15829744421)*x[0]+(-1.87219685708)*x[1]
        arg[2,1,0,0]=(-0.458901790255)*x[0]+(1.13812715713)*x[1]
        arg[2,1,0,1]=(1.03315725208)*x[0]+(-0.112487270341)*x[1]
        arg[2,1,1,0]=(-1.7070220527)*x[0]+(0.126058081856)*x[1]
        arg[2,1,1,1]=(0.863301459388)*x[0]+(0.226670498194)*x[1]
        arg[2,1,2,0]=(-1.0012992544)*x[0]+(-0.195604893926)*x[1]
        arg[2,1,2,1]=(-0.128048233768)*x[0]+(0.827383267773)*x[1]
        arg[2,2,0,0]=(-0.0876478859699)*x[0]+(1.69600082596)*x[1]
        arg[2,2,0,1]=(-0.616278041482)*x[0]+(-0.995548774612)*x[1]
        arg[2,2,1,0]=(0.780699602657)*x[0]+(-0.721498796198)*x[1]
        arg[2,2,1,1]=(-1.17591990575)*x[0]+(-0.66080847955)*x[1]
        arg[2,2,2,0]=(1.71259481962)*x[0]+(-1.12456334234)*x[1]
        arg[2,2,2,1]=(-0.721345747555)*x[0]+(-1.28222132706)*x[1]
        arg[2,3,0,0]=(-1.25971324146)*x[0]+(-0.00559590465243)*x[1]
        arg[2,3,0,1]=(-0.49234245798)*x[0]+(0.30900436328)*x[1]
        arg[2,3,1,0]=(1.3068919966)*x[0]+(-0.891484126193)*x[1]
        arg[2,3,1,1]=(0.319277203381)*x[0]+(-0.535113401495)*x[1]
        arg[2,3,2,0]=(0.529480518887)*x[0]+(0.00452686996008)*x[1]
        arg[2,3,2,1]=(-1.0978905592)*x[0]+(-0.421513714335)*x[1]
        arg[2,4,0,0]=(-1.27711509307)*x[0]+(-1.42208296178)*x[1]
        arg[2,4,0,1]=(-0.621292314868)*x[0]+(-1.10962518414)*x[1]
        arg[2,4,1,0]=(-0.183478331213)*x[0]+(1.0775900659)*x[1]
        arg[2,4,1,1]=(1.07354953663)*x[0]+(0.869765287054)*x[1]
        arg[2,4,2,0]=(-0.696050139647)*x[0]+(-0.120769338549)*x[1]
        arg[2,4,2,1]=(-0.895597167617)*x[0]+(1.41025466606)*x[1]
        arg[3,0,0,0]=(1.15262876611)*x[0]+(-0.505428168285)*x[1]
        arg[3,0,0,1]=(-0.277585299138)*x[0]+(0.352683243558)*x[1]
        arg[3,0,1,0]=(0.218694567952)*x[0]+(-0.170325565499)*x[1]
        arg[3,0,1,1]=(0.297325594036)*x[0]+(0.459308129811)*x[1]
        arg[3,0,2,0]=(-0.572509624935)*x[0]+(0.756188981433)*x[1]
        arg[3,0,2,1]=(0.226462444009)*x[0]+(-1.42100784266)*x[1]
        arg[3,1,0,0]=(-0.28388824767)*x[0]+(-0.306064847514)*x[1]
        arg[3,1,0,1]=(-0.805831316478)*x[0]+(0.756933495912)*x[1]
        arg[3,1,1,0]=(1.37344361233)*x[0]+(-0.377600924857)*x[1]
        arg[3,1,1,1]=(-0.0725302533884)*x[0]+(-0.425074450201)*x[1]
        arg[3,1,2,0]=(-0.066659431931)*x[0]+(1.39252394891)*x[1]
        arg[3,1,2,1]=(0.593440019224)*x[0]+(-0.945629564385)*x[1]
        arg[3,2,0,0]=(0.101850471598)*x[0]+(0.0552356415787)*x[1]
        arg[3,2,0,1]=(-0.407603005372)*x[0]+(-0.849371670514)*x[1]
        arg[3,2,1,0]=(0.435403192155)*x[0]+(0.213673097377)*x[1]
        arg[3,2,1,1]=(0.424081872219)*x[0]+(-0.0800529063085)*x[1]
        arg[3,2,2,0]=(0.189428134309)*x[0]+(-0.205813522374)*x[1]
        arg[3,2,2,1]=(-0.51449771505)*x[0]+(0.0570738169919)*x[1]
        arg[3,3,0,0]=(1.05431625119)*x[0]+(-0.0583139615539)*x[1]
        arg[3,3,0,1]=(0.0292261008496)*x[0]+(-0.569482715214)*x[1]
        arg[3,3,1,0]=(-0.155861158232)*x[0]+(0.628422800294)*x[1]
        arg[3,3,1,1]=(-1.24050157391)*x[0]+(-1.83949885089)*x[1]
        arg[3,3,2,0]=(-0.140709404689)*x[0]+(0.975529832713)*x[1]
        arg[3,3,2,1]=(-0.290824210426)*x[0]+(-0.198373932879)*x[1]
        arg[3,4,0,0]=(0.627236720719)*x[0]+(-0.306676521516)*x[1]
        arg[3,4,0,1]=(-0.964303098626)*x[0]+(-0.701698363587)*x[1]
        arg[3,4,1,0]=(-0.0447860906704)*x[0]+(0.324659175162)*x[1]
        arg[3,4,1,1]=(1.28273443124)*x[0]+(-0.764329261143)*x[1]
        arg[3,4,2,0]=(-0.023232709502)*x[0]+(-0.124231212653)*x[1]
        arg[3,4,2,1]=(0.510754136991)*x[0]+(0.570492207442)*x[1]
        ref[0,0,0,0]=(0.648764087616)/(o+1.)+(-0.521005478658)
        ref[0,0,0,1]=(-0.0127476670646)/(o+1.)+(0.000462672331599)
        ref[0,0,1,0]=(0.258601710827)/(o+1.)+(0.288298245692)
        ref[0,0,1,1]=(0.0577415806545)/(o+1.)+(0.234876758266)
        ref[0,0,2,0]=(-0.91129780757)/(o+1.)+(-0.106708594644)
        ref[0,0,2,1]=(0.0841946322525)/(o+1.)+(-0.527009610865)
        ref[0,1,0,0]=(0.111101563297)/(o+1.)+(-0.274811779206)
        ref[0,1,0,1]=(-0.504062576651)/(o+1.)+(-0.458784562683)
        ref[0,1,1,0]=(0.65560969277)/(o+1.)+(0.132064437431)
        ref[0,1,1,1]=(0.0866382135955)/(o+1.)+(0.629032480469)
        ref[0,1,2,0]=(1.36074036394)/(o+1.)+(-0.0446029830125)
        ref[0,1,2,1]=(1.16071753175)/(o+1.)+(0.285791514672)
        ref[0,2,0,0]=(0.693065154772)/(o+1.)+(0.178910442685)
        ref[0,2,0,1]=(-1.41209835364)/(o+1.)+(-0.372372079222)
        ref[0,2,1,0]=(-0.542625642859)/(o+1.)+(0.111187584182)
        ref[0,2,1,1]=(0.628714449405)/(o+1.)+(0.172527813253)
        ref[0,2,2,0]=(1.62887921583)/(o+1.)+(0.205599062617)
        ref[0,2,2,1]=(1.0640301541)/(o+1.)+(-0.0644170843301)
        ref[0,3,0,0]=(-1.09870552313)/(o+1.)+(-0.791630835375)
        ref[0,3,0,1]=(0.260706656546)/(o+1.)+(-0.0753116325413)
        ref[0,3,1,0]=(-0.0543842358149)/(o+1.)+(0.555983435168)
        ref[0,3,1,1]=(0.0369075211917)/(o+1.)+(0.141820579276)
        ref[0,3,2,0]=(-0.784250881508)/(o+1.)+(0.312147145095)
        ref[0,3,2,1]=(-0.178450092422)/(o+1.)+(0.0263470944113)
        ref[0,4,0,0]=(-0.993852999635)/(o+1.)+(-0.590998505799)
        ref[0,4,0,1]=(-0.0449039600312)/(o+1.)+(0.423998466684)
        ref[0,4,1,0]=(0.583535069477)/(o+1.)+(0.459866978166)
        ref[0,4,1,1]=(-0.251020886878)/(o+1.)+(-0.212621841813)
        ref[0,4,2,0]=(0.408688561613)/(o+1.)+(0.506798143121)
        ref[0,4,2,1]=(0.537090447747)/(o+1.)+(-0.620331288072)
        ref[1,0,0,0]=(0.37428496839)/(o+1.)+(0.0567300237522)
        ref[1,0,0,1]=(0.777672833448)/(o+1.)+(0.212029502451)
        ref[1,0,1,0]=(-0.745422608709)/(o+1.)+(0.026969504563)
        ref[1,0,1,1]=(0.466434983875)/(o+1.)+(0.288463511978)
        ref[1,0,2,0]=(0.237457788829)/(o+1.)+(-0.320049746767)
        ref[1,0,2,1]=(0.346974181328)/(o+1.)+(-0.125918972207)
        ref[1,1,0,0]=(-0.823737519034)/(o+1.)+(-0.211047937165)
        ref[1,1,0,1]=(1.00324369541)/(o+1.)+(0.648642543871)
        ref[1,1,1,0]=(0.674013158596)/(o+1.)+(0.518284415882)
        ref[1,1,1,1]=(0.403142463029)/(o+1.)+(-0.0448117238535)
        ref[1,1,2,0]=(0.0399328921703)/(o+1.)+(-0.113488706628)
        ref[1,1,2,1]=(0.111446633645)/(o+1.)+(0.152153682816)
        ref[1,2,0,0]=(-0.481646721692)/(o+1.)+(-0.123649338787)
        ref[1,2,0,1]=(0.857170000026)/(o+1.)+(0.033333282337)
        ref[1,2,1,0]=(-0.445362398126)/(o+1.)+(0.0452080668082)
        ref[1,2,1,1]=(-0.256201827386)/(o+1.)+(0.419240712362)
        ref[1,2,2,0]=(1.02991996311)/(o+1.)+(0.210657197448)
        ref[1,2,2,1]=(0.355299316895)/(o+1.)+(0.144075984162)
        ref[1,3,0,0]=(-0.679726322154)/(o+1.)+(-0.366298094796)
        ref[1,3,0,1]=(-0.67885027329)/(o+1.)+(0.20947679813)
        ref[1,3,1,0]=(-0.163081485233)/(o+1.)+(-0.295446092868)
        ref[1,3,1,1]=(-0.035768623673)/(o+1.)+(-0.117891877654)
        ref[1,3,2,0]=(-0.639368943038)/(o+1.)+(-0.957510304493)
        ref[1,3,2,1]=(0.160615066001)/(o+1.)+(0.444470683013)
        ref[1,4,0,0]=(1.0649487216)/(o+1.)+(-0.122930582777)
        ref[1,4,0,1]=(0.22766574122)/(o+1.)+(0.493567156271)
        ref[1,4,1,0]=(-0.481296537412)/(o+1.)+(0.688715015285)
        ref[1,4,1,1]=(-0.195268237132)/(o+1.)+(0.558770091963)
        ref[1,4,2,0]=(0.180618258117)/(o+1.)+(-0.439222849696)
        ref[1,4,2,1]=(0.0261012984826)/(o+1.)+(-0.671440554524)
        ref[2,0,0,0]=(-0.301101739758)/(o+1.)+(-0.453251603296)
        ref[2,0,0,1]=(0.00878006761908)/(o+1.)+(0.160403751586)
        ref[2,0,1,0]=(-0.80011300584)/(o+1.)+(0.35896863549)
        ref[2,0,1,1]=(0.00248831604776)/(o+1.)+(-0.0632691631389)
        ref[2,0,2,0]=(-0.0569220251136)/(o+1.)+(0.312750628078)
        ref[2,0,2,1]=(-0.440109525096)/(o+1.)+(-0.136894943888)
        ref[2,1,0,0]=(-0.465652661854)/(o+1.)+(0.572439014362)
        ref[2,1,0,1]=(0.0522958647723)/(o+1.)+(0.434187058485)
        ref[2,1,1,0]=(0.00194260180185)/(o+1.)+(-0.791453286324)
        ref[2,1,1,1]=(0.981996325082)/(o+1.)+(0.0539878162501)
        ref[2,1,2,0]=(-0.243334369882)/(o+1.)+(-0.476784889223)
        ref[2,1,2,1]=(1.53208640175)/(o+1.)+(-0.416375683873)
        ref[2,2,0,0]=(1.27849139532)/(o+1.)+(0.164930772337)
        ref[2,2,0,1]=(-1.4813889023)/(o+1.)+(-0.0652189568992)
        ref[2,2,1,0]=(-0.728282325831)/(o+1.)+(0.393741566145)
        ref[2,2,1,1]=(-0.537765426582)/(o+1.)+(-0.649481479359)
        ref[2,2,2,0]=(-0.266224153785)/(o+1.)+(0.427127815533)
        ref[2,2,2,1]=(-0.862349098277)/(o+1.)+(-0.570608988167)
        ref[2,3,0,0]=(-1.09340368619)/(o+1.)+(-0.0859527299593)
        ref[2,3,0,1]=(1.02842400947)/(o+1.)+(-0.605881052086)
        ref[2,3,1,0]=(0.0131810342609)/(o+1.)+(0.20111341807)
        ref[2,3,1,1]=(-1.11244447248)/(o+1.)+(0.448304137186)
        ref[2,3,2,0]=(-0.978348257673)/(o+1.)+(0.75617782326)
        ref[2,3,2,1]=(-0.140801959181)/(o+1.)+(-0.689301157175)
        ref[2,4,0,0]=(-0.733659962936)/(o+1.)+(-0.982769045955)
        ref[2,4,0,1]=(-1.6278906672)/(o+1.)+(-0.0515134159034)
        ref[2,4,1,0]=(1.52117858199)/(o+1.)+(-0.313533423654)
        ref[2,4,1,1]=(0.621221455007)/(o+1.)+(0.661046684337)
        ref[2,4,2,0]=(-0.157580028393)/(o+1.)+(-0.329619724901)
        ref[2,4,2,1]=(0.52303412622)/(o+1.)+(-0.00418831388809)
        ref[3,0,0,0]=(0.337890126112)/(o+1.)+(0.154655235856)
        ref[3,0,0,1]=(-0.100490423466)/(o+1.)+(0.0877941839427)
        ref[3,0,1,0]=(0.37549944857)/(o+1.)+(-0.163565223059)
        ref[3,0,1,1]=(-0.3212149071)/(o+1.)+(0.538924315474)
        ref[3,0,2,0]=(-0.359815373231)/(o+1.)+(0.271747364864)
        ref[3,0,2,1]=(-0.890575856923)/(o+1.)+(-0.151984770865)
        ref[3,1,0,0]=(-1.10384271628)/(o+1.)+(0.256944810546)
        ref[3,1,0,1]=(0.647425296535)/(o+1.)+(-0.348161558551)
        ref[3,1,1,0]=(1.41806770884)/(o+1.)+(-0.211112510685)
        ref[3,1,1,1]=(0.106528838317)/(o+1.)+(-0.302066770953)
        ref[3,1,2,0]=(0.917213295746)/(o+1.)+(0.204325610614)
        ref[3,1,2,1]=(-0.24738513406)/(o+1.)+(-0.0524022055505)
        ref[3,2,0,0]=(-0.0537683888373)/(o+1.)+(0.105427251007)
        ref[3,2,0,1]=(-1.24647336894)/(o+1.)+(-0.00525065347416)
        ref[3,2,1,0]=(0.187132797735)/(o+1.)+(0.230971745898)
        ref[3,2,1,1]=(0.492750417331)/(o+1.)+(-0.0743607257106)
        ref[3,2,2,0]=(0.062835017839)/(o+1.)+(-0.0396102029518)
        ref[3,2,2,1]=(-0.967272854749)/(o+1.)+(0.254924478345)
        ref[3,3,0,0]=(1.41291903854)/(o+1.)+(-0.208458374448)
        ref[3,3,0,1]=(-0.740922121718)/(o+1.)+(0.100332753677)
        ref[3,3,1,0]=(1.2839228132)/(o+1.)+(-0.405680585568)
        ref[3,3,1,1]=(-1.69516258968)/(o+1.)+(-0.69241891756)
        ref[3,3,2,0]=(0.216594952745)/(o+1.)+(0.309112737639)
        ref[3,3,2,1]=(-0.634806027047)/(o+1.)+(0.0728039418712)
        ref[3,4,0,0]=(-0.310491357088)/(o+1.)+(0.315525778146)
        ref[3,4,0,1]=(-0.751915044154)/(o+1.)+(-0.457043209029)
        ref[3,4,1,0]=(-0.339963591836)/(o+1.)+(0.309918338163)
        ref[3,4,1,1]=(0.0720250143477)/(o+1.)+(0.223190077872)
        ref[3,4,2,0]=(-0.0508554115866)/(o+1.)+(-0.0483042552841)
        ref[3,4,2,1]=(0.754037720807)/(o+1.)+(0.163604311813)
      else:
        arg[0,0,0,0]=(-1.24350959032)*x[0]+(-0.580181370761)*x[1]+(-0.174543112173)*x[2]
        arg[0,0,0,1]=(1.43450421585)*x[0]+(0.717062915298)*x[1]+(-0.91866562143)*x[2]
        arg[0,0,1,0]=(0.0188823859409)*x[0]+(1.24568141807)*x[1]+(0.0842604494481)*x[2]
        arg[0,0,1,1]=(0.871591276766)*x[0]+(0.653031319605)*x[1]+(0.037714556611)*x[2]
        arg[0,0,2,0]=(1.60473786555)*x[0]+(1.47509195705)*x[1]+(-0.103485610136)*x[2]
        arg[0,0,2,1]=(1.68030531336)*x[0]+(0.551201281257)*x[1]+(1.26951824468)*x[2]
        arg[0,1,0,0]=(0.491638014782)*x[0]+(1.26264219762)*x[1]+(-0.921297094204)*x[2]
        arg[0,1,0,1]=(0.20108233628)*x[0]+(-1.36096031252)*x[1]+(-0.878077046379)*x[2]
        arg[0,1,1,0]=(0.900775475455)*x[0]+(0.193423726606)*x[1]+(-0.139955126775)*x[2]
        arg[0,1,1,1]=(-0.0354326925578)*x[0]+(-0.464646718718)*x[1]+(1.49240476583)*x[2]
        arg[0,1,2,0]=(-1.40991971664)*x[0]+(-1.7747715584)*x[1]+(-0.96387650203)*x[2]
        arg[0,1,2,1]=(-0.476443682267)*x[0]+(0.55076135194)*x[1]+(0.190289559512)*x[2]
        arg[0,2,0,0]=(0.535490315669)*x[0]+(0.566915366497)*x[1]+(-0.41513643529)*x[2]
        arg[0,2,0,1]=(-0.18986562478)*x[0]+(-0.62347050233)*x[1]+(1.18004907973)*x[2]
        arg[0,2,1,0]=(-0.288440432828)*x[0]+(0.0375642125411)*x[1]+(0.574653643212)*x[2]
        arg[0,2,1,1]=(1.03327739612)*x[0]+(-0.0443874572943)*x[1]+(0.972142765323)*x[2]
        arg[0,2,2,0]=(-1.27131901306)*x[0]+(0.619179457229)*x[1]+(-0.436029340417)*x[2]
        arg[0,2,2,1]=(0.0573099137019)*x[0]+(0.211373076629)*x[1]+(0.859991510967)*x[2]
        arg[0,3,0,0]=(0.650974277613)*x[0]+(-0.926352792387)*x[1]+(-1.84479828618)*x[2]
        arg[0,3,0,1]=(-0.293748564815)*x[0]+(0.53683587198)*x[1]+(1.51147453744)*x[2]
        arg[0,3,1,0]=(0.837889627485)*x[0]+(0.62568225238)*x[1]+(0.365321068424)*x[2]
        arg[0,3,1,1]=(-0.671841278343)*x[0]+(-0.00822894090357)*x[1]+(-1.35180265459)*x[2]
        arg[0,3,2,0]=(0.823443588183)*x[0]+(-0.366903894987)*x[1]+(-0.878402795844)*x[2]
        arg[0,3,2,1]=(-0.856292830088)*x[0]+(-0.134584368493)*x[1]+(-0.299126849425)*x[2]
        arg[0,4,0,0]=(0.62792423481)*x[0]+(0.781414088068)*x[1]+(-1.12299220031)*x[2]
        arg[0,4,0,1]=(-0.00743646774917)*x[0]+(0.52784004028)*x[1]+(-1.44284109205)*x[2]
        arg[0,4,1,0]=(-0.855519018884)*x[0]+(-0.222306698876)*x[1]+(0.0502517275952)*x[2]
        arg[0,4,1,1]=(-0.311060622512)*x[0]+(0.321283094479)*x[1]+(-0.484147152851)*x[2]
        arg[0,4,2,0]=(-1.43220161062)*x[0]+(-0.27822988291)*x[1]+(-0.692367106541)*x[2]
        arg[0,4,2,1]=(1.3956686934)*x[0]+(-1.26426333906)*x[1]+(-1.09343602989)*x[2]
        arg[1,0,0,0]=(0.122525927098)*x[0]+(-0.484009414389)*x[1]+(-0.822036495419)*x[2]
        arg[1,0,0,1]=(-0.276748173334)*x[0]+(0.35307031133)*x[1]+(-1.23466402387)*x[2]
        arg[1,0,1,0]=(-0.479355493223)*x[0]+(-0.56368304348)*x[1]+(0.54679107219)*x[2]
        arg[1,0,1,1]=(-0.483618214285)*x[0]+(1.60438722757)*x[1]+(0.542389963596)*x[2]
        arg[1,0,2,0]=(0.130988624202)*x[0]+(-0.449468495509)*x[1]+(0.254640520023)*x[2]
        arg[1,0,2,1]=(1.13790232796)*x[0]+(-0.716218913004)*x[1]+(-0.193031264955)*x[2]
        arg[1,1,0,0]=(-0.0653152151059)*x[0]+(1.07890922508)*x[1]+(0.324995864884)*x[2]
        arg[1,1,0,1]=(0.769151345051)*x[0]+(-1.35911748672)*x[1]+(0.299326803785)*x[2]
        arg[1,1,1,0]=(-0.431128032697)*x[0]+(1.09261915758)*x[1]+(-0.029871998484)*x[2]
        arg[1,1,1,1]=(-0.922618915502)*x[0]+(1.30318141394)*x[1]+(-0.324039981399)*x[2]
        arg[1,1,2,0]=(-0.668178375077)*x[0]+(0.132203602988)*x[1]+(0.236035070297)*x[2]
        arg[1,1,2,1]=(-0.518594419598)*x[0]+(-0.0429695477263)*x[1]+(1.11877941395)*x[2]
        arg[1,2,0,0]=(0.610776644247)*x[0]+(-0.023900598958)*x[1]+(0.0480141870788)*x[2]
        arg[1,2,0,1]=(0.562736426016)*x[0]+(-0.294969624732)*x[1]+(-0.23898469143)*x[2]
        arg[1,2,1,0]=(0.645173726197)*x[0]+(-1.18365074993)*x[1]+(0.102581636468)*x[2]
        arg[1,2,1,1]=(-0.518976597355)*x[0]+(0.372817596528)*x[1]+(0.49035249397)*x[2]
        arg[1,2,2,0]=(0.0192623731103)*x[0]+(-0.241279406644)*x[1]+(-0.420015571981)*x[2]
        arg[1,2,2,1]=(0.318641827599)*x[0]+(-1.5357263392)*x[1]+(0.350150621892)*x[2]
        arg[1,3,0,0]=(-0.719401628191)*x[0]+(-1.77806447133)*x[1]+(1.19750443209)*x[2]
        arg[1,3,0,1]=(-1.00122204304)*x[0]+(0.675162216383)*x[1]+(1.1180105058)*x[2]
        arg[1,3,1,0]=(0.331149947519)*x[0]+(0.310832741708)*x[1]+(0.087866556795)*x[2]
        arg[1,3,1,1]=(-0.156343054316)*x[0]+(1.00796776488)*x[1]+(-1.53460740228)*x[2]
        arg[1,3,2,0]=(0.714240747972)*x[0]+(0.968170992322)*x[1]+(-0.962819191598)*x[2]
        arg[1,3,2,1]=(-0.200412114379)*x[0]+(-0.173781234544)*x[1]+(0.899584749309)*x[2]
        arg[1,4,0,0]=(-1.38011177642)*x[0]+(-0.316506470647)*x[1]+(0.266952180469)*x[2]
        arg[1,4,0,1]=(0.343359684584)*x[0]+(-1.13244466712)*x[1]+(-0.81840155125)*x[2]
        arg[1,4,1,0]=(0.295299129957)*x[0]+(0.199810907018)*x[1]+(0.631717407771)*x[2]
        arg[1,4,1,1]=(-0.46840784478)*x[0]+(-0.402683932206)*x[1]+(-0.782209293177)*x[2]
        arg[1,4,2,0]=(-1.8006389736)*x[0]+(0.119727360087)*x[1]+(0.482076626168)*x[2]
        arg[1,4,2,1]=(0.96553885281)*x[0]+(0.425097333238)*x[1]+(0.54533551085)*x[2]
        arg[2,0,0,0]=(0.932725602425)*x[0]+(-0.513780652569)*x[1]+(0.275742915648)*x[2]
        arg[2,0,0,1]=(1.10053578072)*x[0]+(0.787895323287)*x[1]+(-1.46544221328)*x[2]
        arg[2,0,1,0]=(0.83657269666)*x[0]+(-1.15717482853)*x[1]+(1.1090271297)*x[2]
        arg[2,0,1,1]=(1.59370135994)*x[0]+(0.452015708845)*x[1]+(0.0862230059082)*x[2]
        arg[2,0,2,0]=(-0.954832509866)*x[0]+(0.51192257473)*x[1]+(-1.48499771643)*x[2]
        arg[2,0,2,1]=(-0.556993361475)*x[0]+(0.438110051421)*x[1]+(0.237254018143)*x[2]
        arg[2,1,0,0]=(-0.318980966383)*x[0]+(-0.263071719097)*x[1]+(-0.348122442404)*x[2]
        arg[2,1,0,1]=(0.100191287326)*x[0]+(-0.422996999974)*x[1]+(-0.850702087112)*x[2]
        arg[2,1,1,0]=(-0.198942237141)*x[0]+(0.277459568173)*x[1]+(-1.66195774421)*x[2]
        arg[2,1,1,1]=(-1.62211381863)*x[0]+(-0.13417844308)*x[1]+(1.07441481356)*x[2]
        arg[2,1,2,0]=(-0.183120762554)*x[0]+(-0.158546687568)*x[1]+(1.05938565108)*x[2]
        arg[2,1,2,1]=(-0.530596736933)*x[0]+(-0.507872303289)*x[1]+(0.0437745116021)*x[2]
        arg[2,2,0,0]=(-0.430338834548)*x[0]+(-1.01989755268)*x[1]+(0.0364152729771)*x[2]
        arg[2,2,0,1]=(-0.55129175645)*x[0]+(0.283907226465)*x[1]+(0.281403899376)*x[2]
        arg[2,2,1,0]=(0.119036949181)*x[0]+(0.187265158741)*x[1]+(-0.0845814658663)*x[2]
        arg[2,2,1,1]=(1.39900884968)*x[0]+(1.42806588781)*x[1]+(0.211444691268)*x[2]
        arg[2,2,2,0]=(-0.609664666581)*x[0]+(-0.29988332429)*x[1]+(-0.28125800646)*x[2]
        arg[2,2,2,1]=(-1.77545247157)*x[0]+(0.337828315779)*x[1]+(-0.239141292616)*x[2]
        arg[2,3,0,0]=(-1.23141604656)*x[0]+(0.322998749903)*x[1]+(-1.70685281816)*x[2]
        arg[2,3,0,1]=(-1.4416737434)*x[0]+(1.49740114642)*x[1]+(0.732057753882)*x[2]
        arg[2,3,1,0]=(-1.59112622077)*x[0]+(0.665317566947)*x[1]+(0.826417253499)*x[2]
        arg[2,3,1,1]=(-1.22000830454)*x[0]+(0.618713896065)*x[1]+(1.27354984534)*x[2]
        arg[2,3,2,0]=(0.125639723221)*x[0]+(0.663241510656)*x[1]+(-0.426459288281)*x[2]
        arg[2,3,2,1]=(1.39120212155)*x[0]+(0.9115748844)*x[1]+(1.77889030223)*x[2]
        arg[2,4,0,0]=(-0.217927656387)*x[0]+(1.20391185439)*x[1]+(-0.188197734908)*x[2]
        arg[2,4,0,1]=(-0.442009082947)*x[0]+(0.399477101959)*x[1]+(-1.33974368956)*x[2]
        arg[2,4,1,0]=(0.636534809382)*x[0]+(0.872923470503)*x[1]+(-0.922286702006)*x[2]
        arg[2,4,1,1]=(-0.831840624431)*x[0]+(0.946223248622)*x[1]+(0.380300039133)*x[2]
        arg[2,4,2,0]=(0.0803088903901)*x[0]+(-0.857685204574)*x[1]+(-0.479475079431)*x[2]
        arg[2,4,2,1]=(-1.7233942516)*x[0]+(-1.38771345941)*x[1]+(0.761958201633)*x[2]
        arg[3,0,0,0]=(-0.738032075796)*x[0]+(1.33264583681)*x[1]+(-1.21101988385)*x[2]
        arg[3,0,0,1]=(1.40336327022)*x[0]+(-0.77078885213)*x[1]+(1.70960627204)*x[2]
        arg[3,0,1,0]=(0.452477629041)*x[0]+(0.235432009434)*x[1]+(1.69314009822)*x[2]
        arg[3,0,1,1]=(-0.816213143646)*x[0]+(-1.42050350309)*x[1]+(-0.0806105456758)*x[2]
        arg[3,0,2,0]=(0.0400708086023)*x[0]+(-0.336948609628)*x[1]+(-0.113687293995)*x[2]
        arg[3,0,2,1]=(0.414077492095)*x[0]+(-1.06937716207)*x[1]+(1.25972041173)*x[2]
        arg[3,1,0,0]=(-1.02975555201)*x[0]+(-0.123306171085)*x[1]+(-1.74750909033)*x[2]
        arg[3,1,0,1]=(-1.01988698009)*x[0]+(-0.142188487205)*x[1]+(0.274514515492)*x[2]
        arg[3,1,1,0]=(0.440626586141)*x[0]+(-0.0569580581917)*x[1]+(-0.77065631367)*x[2]
        arg[3,1,1,1]=(0.987903067972)*x[0]+(-1.47543187274)*x[1]+(-0.0888962732893)*x[2]
        arg[3,1,2,0]=(0.150506204416)*x[0]+(0.035476396146)*x[1]+(-0.852061690418)*x[2]
        arg[3,1,2,1]=(-1.15179800132)*x[0]+(-0.425570974979)*x[1]+(1.10092214699)*x[2]
        arg[3,2,0,0]=(1.01255188057)*x[0]+(1.24279465527)*x[1]+(1.25190246102)*x[2]
        arg[3,2,0,1]=(-1.16275538385)*x[0]+(1.13672085108)*x[1]+(0.131953731231)*x[2]
        arg[3,2,1,0]=(0.774290747173)*x[0]+(0.275410883089)*x[1]+(-0.825692266439)*x[2]
        arg[3,2,1,1]=(-1.07241838074)*x[0]+(-0.0974675737726)*x[1]+(1.1856849697)*x[2]
        arg[3,2,2,0]=(0.234209248938)*x[0]+(0.756893234577)*x[1]+(-0.365141055079)*x[2]
        arg[3,2,2,1]=(-1.6668858309)*x[0]+(-0.916769120288)*x[1]+(-0.716365681217)*x[2]
        arg[3,3,0,0]=(-0.36890215249)*x[0]+(-0.581062815398)*x[1]+(1.6280164507)*x[2]
        arg[3,3,0,1]=(-0.933634743559)*x[0]+(-0.0657657779145)*x[1]+(1.35011362804)*x[2]
        arg[3,3,1,0]=(1.13739289322)*x[0]+(0.0103208127738)*x[1]+(1.01628193014)*x[2]
        arg[3,3,1,1]=(0.037979696459)*x[0]+(-1.76926650495)*x[1]+(-1.28602572757)*x[2]
        arg[3,3,2,0]=(0.797830431467)*x[0]+(0.732108360923)*x[1]+(-0.436390867333)*x[2]
        arg[3,3,2,1]=(-0.571865447888)*x[0]+(0.895109047357)*x[1]+(-0.673234648776)*x[2]
        arg[3,4,0,0]=(-1.59667195158)*x[0]+(-1.06146105331)*x[1]+(-1.81255923783)*x[2]
        arg[3,4,0,1]=(-0.0721327214647)*x[0]+(1.23982376787)*x[1]+(1.43101011)*x[2]
        arg[3,4,1,0]=(0.117288395573)*x[0]+(-0.37678966212)*x[1]+(-1.14107285089)*x[2]
        arg[3,4,1,1]=(-0.303270532321)*x[0]+(1.33784402324)*x[1]+(-0.651624796413)*x[2]
        arg[3,4,2,0]=(0.215849013256)*x[0]+(0.247315697644)*x[1]+(0.404644392153)*x[2]
        arg[3,4,2,1]=(0.975781190478)*x[0]+(-0.252285063767)*x[1]+(-0.145694910854)*x[2]
        ref[0,0,0,0]=(-0.718934738033)/(o+1.)+(-0.639649667608)
        ref[0,0,0,1]=(1.01007662915)/(o+1.)+(0.111412440284)
        ref[0,0,1,0]=(0.827868953278)/(o+1.)+(0.26047765009)
        ref[0,0,1,1]=(1.48122352358)/(o+1.)+(0.0405568147034)
        ref[0,0,2,0]=(1.33282099436)/(o+1.)+(0.821761609054)
        ref[0,0,2,1]=(1.65298089862)/(o+1.)+(0.924021970342)
        ref[0,1,0,0]=(0.930162571453)/(o+1.)+(-0.0485897266263)
        ref[0,1,0,1]=(-0.347599158269)/(o+1.)+(-0.845177932176)
        ref[0,1,1,0]=(0.623152121663)/(o+1.)+(0.165545976811)
        ref[0,1,1,1]=(0.0506217690841)/(o+1.)+(0.470851792737)
        ref[0,1,2,0]=(-1.99181491207)/(o+1.)+(-1.0783764325)
        ref[0,1,2,1]=(-0.731065007801)/(o+1.)+(0.497836118493)
        ref[0,2,0,0]=(-0.0812249009151)/(o+1.)+(0.384247073896)
        ref[0,2,0,1]=(-0.115768505253)/(o+1.)+(0.241240728938)
        ref[0,2,1,0]=(0.152728009607)/(o+1.)+(0.0855247066589)
        ref[0,2,1,1]=(1.43086526503)/(o+1.)+(0.265083719563)
        ref[0,2,2,0]=(-0.967906369418)/(o+1.)+(-0.060131263415)
        ref[0,2,2,1]=(-0.269269238419)/(o+1.)+(0.698971869859)
        ref[0,3,0,0]=(-1.71659801402)/(o+1.)+(-0.201789393467)
        ref[0,3,0,1]=(0.265343635273)/(o+1.)+(0.744609104665)
        ref[0,3,1,0]=(1.83021875237)/(o+1.)+(-0.000662902039879)
        ref[0,3,1,1]=(-0.317529833375)/(o+1.)+(-0.857171520229)
        ref[0,3,2,0]=(-0.705757693524)/(o+1.)+(0.141947295438)
        ref[0,3,2,1]=(-1.27000495392)/(o+1.)+(-0.00999954704202)
        ref[0,4,0,0]=(0.061892522425)/(o+1.)+(0.112226800069)
        ref[0,4,0,1]=(-0.747062267148)/(o+1.)+(-0.0876876261864)
        ref[0,4,1,0]=(-1.19488448842)/(o+1.)+(0.0836552491261)
        ref[0,4,1,1]=(1.36573976292)/(o+1.)+(-0.9198322219)
        ref[0,4,2,0]=(-1.67982141697)/(o+1.)+(-0.361488591554)
        ref[0,4,2,1]=(-0.598050084547)/(o+1.)+(-0.181990295501)
        ref[1,0,0,0]=(-2.45927674426)/(o+1.)+(0.637878380777)
        ref[1,0,0,1]=(-0.575223026885)/(o+1.)+(-0.291559429493)
        ref[1,0,1,0]=(0.0871765959469)/(o+1.)+(-0.29171203023)
        ref[1,0,1,1]=(0.94050575309)/(o+1.)+(0.361326611893)
        ref[1,0,2,0]=(-0.306500880914)/(o+1.)+(0.121330764815)
        ref[1,0,2,1]=(1.60748296414)/(o+1.)+(-0.689415407072)
        ref[1,1,0,0]=(0.640401899755)/(o+1.)+(0.34909398755)
        ref[1,1,0,1]=(0.134022909144)/(o+1.)+(-0.212331123516)
        ref[1,1,1,0]=(0.248008519593)/(o+1.)+(0.191805303402)
        ref[1,1,1,1]=(0.638911732328)/(o+1.)+(-0.291194607647)
        ref[1,1,2,0]=(0.109264462855)/(o+1.)+(-0.204602082323)
        ref[1,1,2,1]=(-0.185963112656)/(o+1.)+(0.371589279643)
        ref[1,2,0,0]=(-0.590669733193)/(o+1.)+(0.61277998278)
        ref[1,2,0,1]=(-1.15856908266)/(o+1.)+(0.593675596257)
        ref[1,2,1,0]=(-0.451642187813)/(o+1.)+(0.00787340027166)
        ref[1,2,1,1]=(-0.428757501259)/(o+1.)+(0.386475497201)
        ref[1,2,2,0]=(-0.414532914271)/(o+1.)+(-0.113749845622)
        ref[1,2,2,1]=(-0.513613710163)/(o+1.)+(-0.176660089771)
        ref[1,3,0,0]=(-0.890487934256)/(o+1.)+(-0.20473686659)
        ref[1,3,0,1]=(1.64512779208)/(o+1.)+(-0.426588556466)
        ref[1,3,1,0]=(0.831527595966)/(o+1.)+(-0.0508391749725)
        ref[1,3,1,1]=(-0.718921003194)/(o+1.)+(0.0179691557385)
        ref[1,3,2,0]=(0.405959213301)/(o+1.)+(0.156816667697)
        ref[1,3,2,1]=(0.426319486322)/(o+1.)+(0.0495359570318)
        ref[1,4,0,0]=(-1.11364330874)/(o+1.)+(-0.158011378929)
        ref[1,4,0,1]=(-0.0993967101457)/(o+1.)+(-0.754044911817)
        ref[1,4,1,0]=(0.979493241598)/(o+1.)+(0.0736671015741)
        ref[1,4,1,1]=(-1.78692478393)/(o+1.)+(0.0668118568857)
        ref[1,4,2,0]=(-0.691473647622)/(o+1.)+(-0.253680669863)
        ref[1,4,2,1]=(1.16914997228)/(o+1.)+(0.383410862311)
        ref[2,0,0,0]=(0.790187638399)/(o+1.)+(-0.047749886448)
        ref[2,0,0,1]=(0.429625240932)/(o+1.)+(-0.00331817510394)
        ref[2,0,1,0]=(-0.101344997877)/(o+1.)+(0.444884997853)
        ref[2,0,1,1]=(2.15519953039)/(o+1.)+(-0.0116297278494)
        ref[2,0,2,0]=(-1.40159594866)/(o+1.)+(-0.263155851453)
        ref[2,0,2,1]=(-0.658391287862)/(o+1.)+(0.388380997976)
        ref[2,1,0,0]=(-0.773933630328)/(o+1.)+(-0.0781207487777)
        ref[2,1,0,1]=(0.0015551432788)/(o+1.)+(-0.587531471519)
        ref[2,1,1,0]=(0.122693324758)/(o+1.)+(-0.853066868967)
        ref[2,1,1,1]=(-0.219255630453)/(o+1.)+(-0.231310908847)
        ref[2,1,2,0]=(1.33176471286)/(o+1.)+(-0.307023255951)
        ref[2,1,2,1]=(-1.67471265259)/(o+1.)+(0.340009061986)
        ref[2,2,0,0]=(-0.727222063897)/(o+1.)+(-0.343299525178)
        ref[2,2,0,1]=(-1.87667590405)/(o+1.)+(0.945347636719)
        ref[2,2,1,0]=(0.776003927859)/(o+1.)+(-0.277141642902)
        ref[2,2,1,1]=(1.62850322537)/(o+1.)+(0.705008101698)
        ref[2,2,2,0]=(1.31869548331)/(o+1.)+(-1.25475074032)
        ref[2,2,2,1]=(-1.71691241115)/(o+1.)+(0.0200734813728)
        ref[2,3,0,0]=(-0.37741220744)/(o+1.)+(-1.11892895368)
        ref[2,3,0,1]=(0.684763737686)/(o+1.)+(0.0515107096073)
        ref[2,3,1,0]=(-0.0572023243047)/(o+1.)+(-0.0210945380092)
        ref[2,3,1,1]=(0.031457290932)/(o+1.)+(0.320399072966)
        ref[2,3,2,0]=(-0.71613334286)/(o+1.)+(0.539277644228)
        ref[2,3,2,1]=(2.10068005021)/(o+1.)+(0.990493628981)
        ref[2,4,0,0]=(0.830379294407)/(o+1.)+(-0.016296415655)
        ref[2,4,0,1]=(-1.05197706105)/(o+1.)+(-0.165149304749)
        ref[2,4,1,0]=(-0.0854959336547)/(o+1.)+(0.336333755767)
        ref[2,4,1,1]=(1.15845594915)/(o+1.)+(-0.331886642912)
        ref[2,4,2,0]=(-1.25248721318)/(o+1.)+(-0.00218209021539)
        ref[2,4,2,1]=(-0.991752052473)/(o+1.)+(-0.678698728452)
        ref[3,0,0,0]=(-0.262071815816)/(o+1.)+(-0.177167153508)
        ref[3,0,0,1]=(1.20598261029)/(o+1.)+(0.568099039922)
        ref[3,0,1,0]=(1.21237099375)/(o+1.)+(0.584339371474)
        ref[3,0,1,1]=(-1.04628498282)/(o+1.)+(-0.635521104799)
        ref[3,0,2,0]=(0.630843868383)/(o+1.)+(-0.520704481702)
        ref[3,0,2,1]=(1.56136208039)/(o+1.)+(-0.478470669318)
        ref[3,1,0,0]=(-0.372752943047)/(o+1.)+(-1.26390893519)
        ref[3,1,0,1]=(0.192139112377)/(o+1.)+(-0.53985003209)
        ref[3,1,1,0]=(-1.18480584023)/(o+1.)+(0.398909027255)
        ref[3,1,1,1]=(0.402069277892)/(o+1.)+(-0.489247177975)
        ref[3,1,2,0]=(0.27608763117)/(o+1.)+(-0.471083360513)
        ref[3,1,2,1]=(-0.273636427696)/(o+1.)+(-0.101405200809)
        ref[3,2,0,0]=(1.92572771773)/(o+1.)+(0.790760639563)
        ref[3,2,0,1]=(0.16264122626)/(o+1.)+(-0.0283610138982)
        ref[3,2,1,0]=(0.377561018039)/(o+1.)+(-0.076775827108)
        ref[3,2,1,1]=(-0.652466873272)/(o+1.)+(0.334132944227)
        ref[3,2,2,0]=(-0.0304721826842)/(o+1.)+(0.32821680556)
        ref[3,2,2,1]=(-1.52932569088)/(o+1.)+(-0.885347470761)
        ref[3,3,0,0]=(-0.451938162279)/(o+1.)+(0.564994822546)
        ref[3,3,0,1]=(-0.477583137662)/(o+1.)+(0.414148122113)
        ref[3,3,1,0]=(1.38037325689)/(o+1.)+(0.391811189625)
        ref[3,3,1,1]=(-1.01333159097)/(o+1.)+(-1.00199047255)
        ref[3,3,2,0]=(1.3222644334)/(o+1.)+(-0.114358254173)
        ref[3,3,2,1]=(-0.248989333346)/(o+1.)+(-0.0505008579799)
        ref[3,4,0,0]=(-1.92196761736)/(o+1.)+(-1.27436231268)
        ref[3,4,0,1]=(0.894404962457)/(o+1.)+(0.852148096971)
        ref[3,4,1,0]=(0.868763436571)/(o+1.)+(-1.134668777)
        ref[3,4,1,1]=(0.438747123022)/(o+1.)+(-0.0278992142568)
        ref[3,4,2,0]=(0.532575431402)/(o+1.)+(0.167616835826)
        ref[3,4,2,1]=(0.796744496799)/(o+1.)+(-0.109471640471)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5, 3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunction_fromData_Function_rank0(self):
      """
      tests integral of rank 0 Data on the Function

      assumptions: Function(self.domain) exists
                   self.domain supports integral on Function
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=Function(self.domain)
      w=Function(self.domain)
      x=w.getX()
      arg=Data(0,(),w)
      if dim==2:
        arg=(-0.217470460227)*x[0]**o+(0.664132905811)*x[0]+(0.493878617743)*x[1]**o+(0.111327715714)*x[1]
        ref=(0.276408157516)/(o+1.)+(0.387730310763)
      else:
        arg=(0.371937838886)*x[0]**o+(0.653846807778)*x[0]+(0.172358058188)*x[1]**o+(-0.928017916614)*x[1]+(0.116156616174)*x[2]**o+(0.371332501889)*x[2]
        ref=(0.660452513249)/(o+1.)+(0.0485806965265)
      res=integrate(arg)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunction_fromData_Function_rank1(self):
      """
      tests integral of rank 1 Data on the Function

      assumptions: Function(self.domain) exists
                   self.domain supports integral on Function
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=Function(self.domain)
      w=Function(self.domain)
      x=w.getX()
      arg=Data(0,(2,),w)
      ref=numpy.zeros((2,),numpy.float_)
      if dim==2:
        arg[0]=(-0.329952046701)*x[0]**o+(0.12757022314)*x[0]+(0.198280362833)*x[1]**o+(0.129179854334)*x[1]
        arg[1]=(0.185895161332)*x[0]**o+(0.270740935936)*x[0]+(0.630122833594)*x[1]**o+(-0.0696403698787)*x[1]
        ref[0]=(-0.131671683869)/(o+1.)+(0.128375038737)
        ref[1]=(0.816017994925)/(o+1.)+(0.100550283029)
      else:
        arg[0]=(0.526257699936)*x[0]**o+(0.0731442796396)*x[0]+(-0.266121642119)*x[1]**o+(-0.929478242136)*x[1]+(0.743092861949)*x[2]**o+(0.22690419858)*x[2]
        arg[1]=(-0.121387039147)*x[0]**o+(-0.0580608263784)*x[0]+(0.761164059949)*x[1]**o+(0.821938972431)*x[1]+(-0.756423175962)*x[2]**o+(0.213207132353)*x[2]
        ref[0]=(1.00322891977)/(o+1.)+(-0.314714881958)
        ref[1]=(-0.11664615516)/(o+1.)+(0.488542639203)
      res=integrate(arg)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunction_fromData_Function_rank2(self):
      """
      tests integral of rank 2 Data on the Function

      assumptions: Function(self.domain) exists
                   self.domain supports integral on Function
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=Function(self.domain)
      w=Function(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5),w)
      ref=numpy.zeros((4, 5),numpy.float_)
      if dim==2:
        arg[0,0]=(-0.963318847499)*x[0]**o+(-0.184977238308)*x[0]+(0.153633231879)*x[1]**o+(-0.640028315216)*x[1]
        arg[0,1]=(0.0879103656102)*x[0]**o+(-0.928854425726)*x[0]+(-0.138566176095)*x[1]**o+(-0.670118398936)*x[1]
        arg[0,2]=(0.920856326204)*x[0]**o+(0.461047594595)*x[0]+(-0.914778320363)*x[1]**o+(-0.561644217568)*x[1]
        arg[0,3]=(0.506148289443)*x[0]**o+(-0.474044343536)*x[0]+(0.742127575836)*x[1]**o+(-0.52349314159)*x[1]
        arg[0,4]=(0.0810224543145)*x[0]**o+(-0.307992483498)*x[0]+(0.749673245291)*x[1]**o+(-0.554099873044)*x[1]
        arg[1,0]=(-0.270897099553)*x[0]**o+(-0.222178461886)*x[0]+(0.755729049844)*x[1]**o+(0.740743777894)*x[1]
        arg[1,1]=(0.0220934176819)*x[0]**o+(-0.309641658323)*x[0]+(-0.951833592348)*x[1]**o+(0.430554526918)*x[1]
        arg[1,2]=(0.176033311218)*x[0]**o+(0.728188001962)*x[0]+(-0.428357954006)*x[1]**o+(0.955059500699)*x[1]
        arg[1,3]=(0.946680422474)*x[0]**o+(-0.251968148743)*x[0]+(-0.3704600461)*x[1]**o+(0.692080922297)*x[1]
        arg[1,4]=(0.122655375591)*x[0]**o+(0.617285886533)*x[0]+(0.729184247986)*x[1]**o+(0.30369620264)*x[1]
        arg[2,0]=(-0.957780501271)*x[0]**o+(0.438290429468)*x[0]+(-0.504841431279)*x[1]**o+(0.935350834608)*x[1]
        arg[2,1]=(-0.49410765407)*x[0]**o+(-0.635142349002)*x[0]+(0.700131616572)*x[1]**o+(0.99523209861)*x[1]
        arg[2,2]=(0.0967686936303)*x[0]**o+(0.987754073381)*x[0]+(-0.161220173386)*x[1]**o+(-0.343926097784)*x[1]
        arg[2,3]=(0.560040886628)*x[0]**o+(-0.843778720323)*x[0]+(0.345354034369)*x[1]**o+(0.440584543858)*x[1]
        arg[2,4]=(0.988977815245)*x[0]**o+(0.661922598357)*x[0]+(-0.425421002549)*x[1]**o+(0.624279872655)*x[1]
        arg[3,0]=(-0.997217068098)*x[0]**o+(-0.973497091595)*x[0]+(-0.0980683851598)*x[1]**o+(0.408081557041)*x[1]
        arg[3,1]=(0.0827616522414)*x[0]**o+(0.931022772362)*x[0]+(-0.588688206358)*x[1]**o+(0.900467348992)*x[1]
        arg[3,2]=(0.0933642874383)*x[0]**o+(0.398732664158)*x[0]+(-0.437102212783)*x[1]**o+(-0.126538302466)*x[1]
        arg[3,3]=(0.251989644486)*x[0]**o+(0.765809815972)*x[0]+(-0.325455283605)*x[1]**o+(0.171819372538)*x[1]
        arg[3,4]=(0.649407682145)*x[0]**o+(0.963776897105)*x[0]+(0.671915752131)*x[1]**o+(-0.468729629153)*x[1]
        ref[0,0]=(-0.80968561562)/(o+1.)+(-0.412502776762)
        ref[0,1]=(-0.0506558104848)/(o+1.)+(-0.799486412331)
        ref[0,2]=(0.00607800584081)/(o+1.)+(-0.0502983114867)
        ref[0,3]=(1.24827586528)/(o+1.)+(-0.498768742563)
        ref[0,4]=(0.830695699605)/(o+1.)+(-0.431046178271)
        ref[1,0]=(0.484831950292)/(o+1.)+(0.259282658004)
        ref[1,1]=(-0.929740174666)/(o+1.)+(0.0604564342974)
        ref[1,2]=(-0.252324642788)/(o+1.)+(0.84162375133)
        ref[1,3]=(0.576220376373)/(o+1.)+(0.220056386777)
        ref[1,4]=(0.851839623577)/(o+1.)+(0.460491044586)
        ref[2,0]=(-1.46262193255)/(o+1.)+(0.686820632038)
        ref[2,1]=(0.206023962501)/(o+1.)+(0.180044874804)
        ref[2,2]=(-0.0644514797559)/(o+1.)+(0.321913987798)
        ref[2,3]=(0.905394920997)/(o+1.)+(-0.201597088233)
        ref[2,4]=(0.563556812695)/(o+1.)+(0.643101235506)
        ref[3,0]=(-1.09528545326)/(o+1.)+(-0.282707767277)
        ref[3,1]=(-0.505926554117)/(o+1.)+(0.915745060677)
        ref[3,2]=(-0.343737925345)/(o+1.)+(0.136097180846)
        ref[3,3]=(-0.0734656391194)/(o+1.)+(0.468814594255)
        ref[3,4]=(1.32132343428)/(o+1.)+(0.247523633976)
      else:
        arg[0,0]=(-0.512698606017)*x[0]**o+(-0.156853163498)*x[0]+(0.247997355633)*x[1]**o+(-0.363003068904)*x[1]+(-0.940097989108)*x[2]**o+(-0.661063850288)*x[2]
        arg[0,1]=(-0.886390181577)*x[0]**o+(0.724049511142)*x[0]+(0.260684592163)*x[1]**o+(0.265648885034)*x[1]+(0.123084281693)*x[2]**o+(0.181492979151)*x[2]
        arg[0,2]=(-0.570543998851)*x[0]**o+(0.0541041165388)*x[0]+(0.630994990544)*x[1]**o+(-0.89311079559)*x[1]+(0.750714073617)*x[2]**o+(-0.0684796380527)*x[2]
        arg[0,3]=(0.605486031268)*x[0]**o+(0.0387617487672)*x[0]+(-0.365027101406)*x[1]**o+(0.323275180514)*x[1]+(-0.763142660003)*x[2]**o+(-0.331069480921)*x[2]
        arg[0,4]=(-0.0779035848472)*x[0]**o+(-0.0234904624931)*x[0]+(0.591635786678)*x[1]**o+(-0.793680126332)*x[1]+(-0.874742296757)*x[2]**o+(-0.184528885157)*x[2]
        arg[1,0]=(-0.62480972544)*x[0]**o+(-0.99242215547)*x[0]+(0.634250679117)*x[1]**o+(-0.785448642692)*x[1]+(-0.013335654402)*x[2]**o+(0.52112750417)*x[2]
        arg[1,1]=(0.611541580179)*x[0]**o+(-0.203144044802)*x[0]+(-0.411277184995)*x[1]**o+(0.373487145939)*x[1]+(-0.679339345026)*x[2]**o+(-0.632413536218)*x[2]
        arg[1,2]=(-0.258927016205)*x[0]**o+(0.0242628959926)*x[0]+(-0.596909760383)*x[1]**o+(-0.295787354787)*x[1]+(-0.337930022823)*x[2]**o+(-0.825085100446)*x[2]
        arg[1,3]=(-0.0976663781507)*x[0]**o+(0.716600132838)*x[0]+(-0.682155341898)*x[1]**o+(-0.154484864167)*x[1]+(0.754686874908)*x[2]**o+(-0.102453470268)*x[2]
        arg[1,4]=(0.268714339964)*x[0]**o+(0.974234398844)*x[0]+(-0.370113158384)*x[1]**o+(0.0745228956861)*x[1]+(-0.0674370676047)*x[2]**o+(0.132605448278)*x[2]
        arg[2,0]=(-0.00750893262562)*x[0]**o+(0.507641887134)*x[0]+(0.638483932117)*x[1]**o+(-0.940295540912)*x[1]+(-0.201492519698)*x[2]**o+(0.504878595818)*x[2]
        arg[2,1]=(-0.0283522980593)*x[0]**o+(-0.76577346605)*x[0]+(0.0863472830598)*x[1]**o+(-0.904447859187)*x[1]+(-0.203390005492)*x[2]**o+(-0.741629136645)*x[2]
        arg[2,2]=(-0.0196123831667)*x[0]**o+(-0.320721372799)*x[0]+(0.0670421287749)*x[1]**o+(-0.415029682416)*x[1]+(0.450269143475)*x[2]**o+(-0.735201210502)*x[2]
        arg[2,3]=(-0.621537330758)*x[0]**o+(0.21573881897)*x[0]+(0.0364275955835)*x[1]**o+(0.873908952751)*x[1]+(-0.197046790303)*x[2]**o+(0.655654722165)*x[2]
        arg[2,4]=(0.432531007791)*x[0]**o+(-0.65605251166)*x[0]+(0.728245204833)*x[1]**o+(-0.769998936989)*x[1]+(-0.797879623133)*x[2]**o+(0.36575554332)*x[2]
        arg[3,0]=(0.512219734043)*x[0]**o+(-0.177942735603)*x[0]+(-0.307293350237)*x[1]**o+(-0.780181541911)*x[1]+(-0.510201727643)*x[2]**o+(-0.567980715525)*x[2]
        arg[3,1]=(-0.405123989366)*x[0]**o+(-0.133316566583)*x[0]+(0.552868689995)*x[1]**o+(-0.921149434884)*x[1]+(0.510851078054)*x[2]**o+(0.7207533866)*x[2]
        arg[3,2]=(0.150011788901)*x[0]**o+(-0.55947555552)*x[0]+(-0.353366762442)*x[1]**o+(0.204954197301)*x[1]+(-0.48254632377)*x[2]**o+(-0.272738900127)*x[2]
        arg[3,3]=(-0.690981207461)*x[0]**o+(0.38683432128)*x[0]+(0.655731808738)*x[1]**o+(0.162269870301)*x[1]+(-0.373468659959)*x[2]**o+(-0.841431521009)*x[2]
        arg[3,4]=(0.156455675319)*x[0]**o+(0.501690084272)*x[0]+(-0.339255791314)*x[1]**o+(0.0832260784264)*x[1]+(0.394137401365)*x[2]**o+(-0.92403906326)*x[2]
        ref[0,0]=(-1.20479923949)/(o+1.)+(-0.590460041345)
        ref[0,1]=(-0.502621307721)/(o+1.)+(0.585595687663)
        ref[0,2]=(0.811165065311)/(o+1.)+(-0.453743158552)
        ref[0,3]=(-0.522683730141)/(o+1.)+(0.0154837241797)
        ref[0,4]=(-0.361010094926)/(o+1.)+(-0.500849736991)
        ref[1,0]=(-0.00389470072479)/(o+1.)+(-0.628371646996)
        ref[1,1]=(-0.479074949842)/(o+1.)+(-0.23103521754)
        ref[1,2]=(-1.19376679941)/(o+1.)+(-0.54830477962)
        ref[1,3]=(-0.0251348451402)/(o+1.)+(0.229830899202)
        ref[1,4]=(-0.168835886025)/(o+1.)+(0.590681371404)
        ref[2,0]=(0.429482479793)/(o+1.)+(0.0361124710203)
        ref[2,1]=(-0.145395020492)/(o+1.)+(-1.20592523094)
        ref[2,2]=(0.497698889084)/(o+1.)+(-0.735476132859)
        ref[2,3]=(-0.782156525477)/(o+1.)+(0.872651246943)
        ref[2,4]=(0.362896589491)/(o+1.)+(-0.530147952664)
        ref[3,0]=(-0.305275343837)/(o+1.)+(-0.763052496519)
        ref[3,1]=(0.658595778683)/(o+1.)+(-0.166856307434)
        ref[3,2]=(-0.685901297311)/(o+1.)+(-0.313630129173)
        ref[3,3]=(-0.408718058683)/(o+1.)+(-0.146163664714)
        ref[3,4]=(0.21133728537)/(o+1.)+(-0.169561450281)
      res=integrate(arg)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunction_fromData_Function_rank3(self):
      """
      tests integral of rank 3 Data on the Function

      assumptions: Function(self.domain) exists
                   self.domain supports integral on Function
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=Function(self.domain)
      w=Function(self.domain)
      x=w.getX()
      arg=Data(0,(6, 2, 2),w)
      ref=numpy.zeros((6, 2, 2),numpy.float_)
      if dim==2:
        arg[0,0,0]=(-0.165423911829)*x[0]**o+(0.912542062064)*x[0]+(0.515840721225)*x[1]**o+(-0.855922021017)*x[1]
        arg[0,0,1]=(0.591873802546)*x[0]**o+(-0.039385167187)*x[0]+(0.0261484884274)*x[1]**o+(0.227950240434)*x[1]
        arg[0,1,0]=(-0.825833712046)*x[0]**o+(0.989606797765)*x[0]+(0.975260558791)*x[1]**o+(0.297963603984)*x[1]
        arg[0,1,1]=(-0.658054502482)*x[0]**o+(-0.865505090963)*x[0]+(0.817184084453)*x[1]**o+(-0.409588098899)*x[1]
        arg[1,0,0]=(0.538456987198)*x[0]**o+(-0.0768787509391)*x[0]+(0.424620431835)*x[1]**o+(0.799917948831)*x[1]
        arg[1,0,1]=(0.125945558199)*x[0]**o+(-0.48320049996)*x[0]+(0.105474962327)*x[1]**o+(-0.090713555582)*x[1]
        arg[1,1,0]=(0.835213375418)*x[0]**o+(-0.513407038769)*x[0]+(-0.342890811735)*x[1]**o+(-0.285057736244)*x[1]
        arg[1,1,1]=(0.762782085749)*x[0]**o+(-0.308457087352)*x[0]+(0.23756726753)*x[1]**o+(-0.909365956504)*x[1]
        arg[2,0,0]=(-0.76467980678)*x[0]**o+(-0.265552922006)*x[0]+(-0.261862355539)*x[1]**o+(-0.399407404618)*x[1]
        arg[2,0,1]=(0.79401193617)*x[0]**o+(-0.0300461495216)*x[0]+(-0.978834133508)*x[1]**o+(-0.984845382599)*x[1]
        arg[2,1,0]=(0.898033018534)*x[0]**o+(-0.184513430159)*x[0]+(0.0430634098526)*x[1]**o+(0.62213313954)*x[1]
        arg[2,1,1]=(0.126926439246)*x[0]**o+(0.671259428917)*x[0]+(-0.396062855417)*x[1]**o+(0.0461825691916)*x[1]
        arg[3,0,0]=(0.185756594279)*x[0]**o+(0.570946291973)*x[0]+(0.0673731944956)*x[1]**o+(0.367766610292)*x[1]
        arg[3,0,1]=(0.021269224946)*x[0]**o+(-0.463663279063)*x[0]+(-0.687930988068)*x[1]**o+(0.184714204849)*x[1]
        arg[3,1,0]=(0.754288639303)*x[0]**o+(-0.94612827517)*x[0]+(0.802463196709)*x[1]**o+(0.202355582077)*x[1]
        arg[3,1,1]=(0.840002857786)*x[0]**o+(-0.692207739447)*x[0]+(0.486569391181)*x[1]**o+(-0.713960793192)*x[1]
        arg[4,0,0]=(-0.205794539655)*x[0]**o+(-0.930458992533)*x[0]+(-0.62580205172)*x[1]**o+(-0.884738110673)*x[1]
        arg[4,0,1]=(-0.91433612349)*x[0]**o+(0.694961527804)*x[0]+(-0.510941334746)*x[1]**o+(-0.510725271706)*x[1]
        arg[4,1,0]=(0.852785499366)*x[0]**o+(0.870826628851)*x[0]+(-0.577529754198)*x[1]**o+(-0.583793684857)*x[1]
        arg[4,1,1]=(-0.604869226881)*x[0]**o+(0.482463572402)*x[0]+(0.829654316885)*x[1]**o+(0.597543822455)*x[1]
        arg[5,0,0]=(0.413459469749)*x[0]**o+(0.865143264083)*x[0]+(0.986197644249)*x[1]**o+(-0.2511021346)*x[1]
        arg[5,0,1]=(0.811572168213)*x[0]**o+(-0.110121334437)*x[0]+(-0.985492232242)*x[1]**o+(-0.913436084156)*x[1]
        arg[5,1,0]=(-0.660869051422)*x[0]**o+(-0.743235620427)*x[0]+(0.205393778238)*x[1]**o+(0.296069404687)*x[1]
        arg[5,1,1]=(-0.13285011267)*x[0]**o+(0.659400366834)*x[0]+(-0.889984368942)*x[1]**o+(0.584893563559)*x[1]
        ref[0,0,0]=(0.350416809396)/(o+1.)+(0.0283100205234)
        ref[0,0,1]=(0.618022290973)/(o+1.)+(0.0942825366236)
        ref[0,1,0]=(0.149426846745)/(o+1.)+(0.643785200874)
        ref[0,1,1]=(0.159129581971)/(o+1.)+(-0.637546594931)
        ref[1,0,0]=(0.963077419032)/(o+1.)+(0.361519598946)
        ref[1,0,1]=(0.231420520526)/(o+1.)+(-0.286957027771)
        ref[1,1,0]=(0.492322563683)/(o+1.)+(-0.399232387506)
        ref[1,1,1]=(1.00034935328)/(o+1.)+(-0.608911521928)
        ref[2,0,0]=(-1.02654216232)/(o+1.)+(-0.332480163312)
        ref[2,0,1]=(-0.184822197338)/(o+1.)+(-0.507445766061)
        ref[2,1,0]=(0.941096428387)/(o+1.)+(0.21880985469)
        ref[2,1,1]=(-0.269136416172)/(o+1.)+(0.358720999054)
        ref[3,0,0]=(0.253129788774)/(o+1.)+(0.469356451133)
        ref[3,0,1]=(-0.666661763122)/(o+1.)+(-0.139474537107)
        ref[3,1,0]=(1.55675183601)/(o+1.)+(-0.371886346546)
        ref[3,1,1]=(1.32657224897)/(o+1.)+(-0.703084266319)
        ref[4,0,0]=(-0.831596591375)/(o+1.)+(-0.907598551603)
        ref[4,0,1]=(-1.42527745824)/(o+1.)+(0.0921181280487)
        ref[4,1,0]=(0.275255745168)/(o+1.)+(0.143516471997)
        ref[4,1,1]=(0.224785090004)/(o+1.)+(0.540003697429)
        ref[5,0,0]=(1.399657114)/(o+1.)+(0.307020564742)
        ref[5,0,1]=(-0.173920064028)/(o+1.)+(-0.511778709296)
        ref[5,1,0]=(-0.455475273184)/(o+1.)+(-0.22358310787)
        ref[5,1,1]=(-1.02283448161)/(o+1.)+(0.622146965196)
      else:
        arg[0,0,0]=(-0.564478749422)*x[0]**o+(-0.0357223504179)*x[0]+(0.396603298865)*x[1]**o+(-0.461941327703)*x[1]+(-0.667580657928)*x[2]**o+(-0.589957067263)*x[2]
        arg[0,0,1]=(0.930616008951)*x[0]**o+(-0.764067565635)*x[0]+(-0.71350617205)*x[1]**o+(-0.530942701898)*x[1]+(-0.103797356434)*x[2]**o+(0.410704036673)*x[2]
        arg[0,1,0]=(-0.59127392299)*x[0]**o+(0.452490122131)*x[0]+(0.714793625631)*x[1]**o+(-0.933114051133)*x[1]+(-0.693437840325)*x[2]**o+(0.979267231764)*x[2]
        arg[0,1,1]=(0.179682332267)*x[0]**o+(-0.419960476425)*x[0]+(-0.10986598341)*x[1]**o+(0.912986428497)*x[1]+(0.0886946782573)*x[2]**o+(0.516065958873)*x[2]
        arg[1,0,0]=(0.524446863481)*x[0]**o+(-0.439668193423)*x[0]+(0.681259811488)*x[1]**o+(0.439275262258)*x[1]+(0.84573881706)*x[2]**o+(0.377640553175)*x[2]
        arg[1,0,1]=(-0.007464905203)*x[0]**o+(0.234727832471)*x[0]+(0.0428956356618)*x[1]**o+(0.195772930172)*x[1]+(0.457790934614)*x[2]**o+(0.445309166495)*x[2]
        arg[1,1,0]=(0.821283375827)*x[0]**o+(0.212480005798)*x[0]+(0.148817990961)*x[1]**o+(0.119716150877)*x[1]+(0.658959126865)*x[2]**o+(-0.346216916131)*x[2]
        arg[1,1,1]=(-0.0587222422894)*x[0]**o+(-0.325321661973)*x[0]+(0.149293818689)*x[1]**o+(-0.632073115037)*x[1]+(0.018604271874)*x[2]**o+(-0.412288240166)*x[2]
        arg[2,0,0]=(-0.22588082169)*x[0]**o+(0.197780461217)*x[0]+(0.387336989213)*x[1]**o+(-0.680021661016)*x[1]+(-0.903782254346)*x[2]**o+(-0.273268109853)*x[2]
        arg[2,0,1]=(0.415876394297)*x[0]**o+(0.199596795904)*x[0]+(-0.445817134682)*x[1]**o+(0.672539590684)*x[1]+(0.907825473964)*x[2]**o+(0.459560469691)*x[2]
        arg[2,1,0]=(-0.052146887134)*x[0]**o+(-0.229450574738)*x[0]+(0.0929720151604)*x[1]**o+(-0.893861034491)*x[1]+(-0.294030328982)*x[2]**o+(-0.193256779109)*x[2]
        arg[2,1,1]=(0.361171422066)*x[0]**o+(-0.212022056495)*x[0]+(0.0672221746963)*x[1]**o+(0.57506871515)*x[1]+(0.654894469509)*x[2]**o+(0.0749777260533)*x[2]
        arg[3,0,0]=(0.505355253404)*x[0]**o+(-0.0753563586359)*x[0]+(0.544299279727)*x[1]**o+(-0.781954131328)*x[1]+(0.534390713891)*x[2]**o+(-0.719289344378)*x[2]
        arg[3,0,1]=(-0.589309652827)*x[0]**o+(-0.0557421151178)*x[0]+(-0.275311076837)*x[1]**o+(0.998180518149)*x[1]+(-0.844823856433)*x[2]**o+(-0.988843805162)*x[2]
        arg[3,1,0]=(-0.411075104301)*x[0]**o+(-0.405896828894)*x[0]+(0.242301856034)*x[1]**o+(-0.760648969984)*x[1]+(0.0940367373673)*x[2]**o+(-0.121420230853)*x[2]
        arg[3,1,1]=(-0.201426146967)*x[0]**o+(0.722847134095)*x[0]+(-0.00743580369433)*x[1]**o+(-0.593158423904)*x[1]+(0.630575332064)*x[2]**o+(-0.309839791343)*x[2]
        arg[4,0,0]=(0.370057366738)*x[0]**o+(0.35683022394)*x[0]+(0.886304520049)*x[1]**o+(0.834488899465)*x[1]+(0.27408661832)*x[2]**o+(0.325007409933)*x[2]
        arg[4,0,1]=(0.386153377977)*x[0]**o+(-0.889700309398)*x[0]+(-0.60695970137)*x[1]**o+(0.701208035732)*x[1]+(0.435164510485)*x[2]**o+(0.40223148148)*x[2]
        arg[4,1,0]=(-0.227256041906)*x[0]**o+(-0.0406115125763)*x[0]+(-0.0298974314183)*x[1]**o+(0.887167464033)*x[1]+(0.1004751563)*x[2]**o+(0.826924545719)*x[2]
        arg[4,1,1]=(0.733575640137)*x[0]**o+(-0.376269474248)*x[0]+(-0.841290711112)*x[1]**o+(-0.0633013917179)*x[1]+(0.335730636406)*x[2]**o+(0.913607658221)*x[2]
        arg[5,0,0]=(-0.896019254334)*x[0]**o+(0.465470987762)*x[0]+(0.338543664838)*x[1]**o+(0.182102591196)*x[1]+(0.476317455739)*x[2]**o+(0.191367623163)*x[2]
        arg[5,0,1]=(0.828458939676)*x[0]**o+(-0.37853033744)*x[0]+(-0.540809170906)*x[1]**o+(0.367869598038)*x[1]+(0.321196361443)*x[2]**o+(0.548981819769)*x[2]
        arg[5,1,0]=(-0.0805871335579)*x[0]**o+(-0.554679375424)*x[0]+(-0.259438414137)*x[1]**o+(0.72362534417)*x[1]+(-0.838855028889)*x[2]**o+(0.929802758377)*x[2]
        arg[5,1,1]=(-0.947910259125)*x[0]**o+(0.654012824998)*x[0]+(0.425232602078)*x[1]**o+(-0.649345176159)*x[1]+(-0.946280438348)*x[2]**o+(-0.429547284912)*x[2]
        ref[0,0,0]=(-0.835456108484)/(o+1.)+(-0.543810372692)
        ref[0,0,1]=(0.113312480467)/(o+1.)+(-0.44215311543)
        ref[0,1,0]=(-0.569918137684)/(o+1.)+(0.249321651381)
        ref[0,1,1]=(0.158511027114)/(o+1.)+(0.504545955472)
        ref[1,0,0]=(2.05144549203)/(o+1.)+(0.188623811006)
        ref[1,0,1]=(0.493221665073)/(o+1.)+(0.437904964569)
        ref[1,1,0]=(1.62906049365)/(o+1.)+(-0.00701037972825)
        ref[1,1,1]=(0.109175848273)/(o+1.)+(-0.684841508588)
        ref[2,0,0]=(-0.742326086822)/(o+1.)+(-0.377754654826)
        ref[2,0,1]=(0.877884733578)/(o+1.)+(0.665848428139)
        ref[2,1,0]=(-0.253205200956)/(o+1.)+(-0.658284194169)
        ref[2,1,1]=(1.08328806627)/(o+1.)+(0.219012192354)
        ref[3,0,0]=(1.58404524702)/(o+1.)+(-0.788299917171)
        ref[3,0,1]=(-1.7094445861)/(o+1.)+(-0.0232027010654)
        ref[3,1,0]=(-0.0747365109002)/(o+1.)+(-0.643983014866)
        ref[3,1,1]=(0.421713381403)/(o+1.)+(-0.0900755405762)
        ref[4,0,0]=(1.53044850511)/(o+1.)+(0.758163266669)
        ref[4,0,1]=(0.214358187092)/(o+1.)+(0.106869603907)
        ref[4,1,0]=(-0.156678317024)/(o+1.)+(0.836740248588)
        ref[4,1,1]=(0.228015565432)/(o+1.)+(0.237018396127)
        ref[5,0,0]=(-0.0811581337568)/(o+1.)+(0.419470601061)
        ref[5,0,1]=(0.608846130213)/(o+1.)+(0.269160540183)
        ref[5,1,0]=(-1.17888057658)/(o+1.)+(0.549374363561)
        ref[5,1,1]=(-1.4689580954)/(o+1.)+(-0.212439818036)
      res=integrate(arg)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunction_fromData_Function_rank4(self):
      """
      tests integral of rank 4 Data on the Function

      assumptions: Function(self.domain) exists
                   self.domain supports integral on Function
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=Function(self.domain)
      w=Function(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5, 3, 2),w)
      ref=numpy.zeros((4, 5, 3, 2),numpy.float_)
      if dim==2:
        arg[0,0,0,0]=(-0.216014106349)*x[0]**o+(0.98826230471)*x[0]+(0.522566177319)*x[1]**o+(-0.672520997452)*x[1]
        arg[0,0,0,1]=(0.368024408084)*x[0]**o+(0.047972971332)*x[0]+(-0.307374327321)*x[1]**o+(0.0745169735457)*x[1]
        arg[0,0,1,0]=(0.438735045452)*x[0]**o+(0.76355958752)*x[0]+(0.200829025437)*x[1]**o+(0.56912407759)*x[1]
        arg[0,0,1,1]=(-0.0328769393712)*x[0]**o+(-0.39701363611)*x[0]+(0.704125532519)*x[1]**o+(-0.6602435084)*x[1]
        arg[0,0,2,0]=(-0.547717992717)*x[0]**o+(-0.844256763776)*x[0]+(0.621300470147)*x[1]**o+(-0.105216181226)*x[1]
        arg[0,0,2,1]=(-0.723474063234)*x[0]**o+(-0.980738972901)*x[0]+(-0.183578659566)*x[1]**o+(0.586038450047)*x[1]
        arg[0,1,0,0]=(0.505379907001)*x[0]**o+(-0.470514711277)*x[0]+(0.799116811747)*x[1]**o+(0.682760846386)*x[1]
        arg[0,1,0,1]=(0.353992609198)*x[0]**o+(-0.848107655585)*x[0]+(-0.774065459855)*x[1]**o+(0.0323137462292)*x[1]
        arg[0,1,1,0]=(0.799068805694)*x[0]**o+(-0.979761821209)*x[0]+(0.642674987867)*x[1]**o+(-0.639029952044)*x[1]
        arg[0,1,1,1]=(0.542556683327)*x[0]**o+(0.876753086356)*x[0]+(-0.387121297643)*x[1]**o+(0.377543179628)*x[1]
        arg[0,1,2,0]=(0.424672899596)*x[0]**o+(-0.134184751569)*x[0]+(-0.561877304114)*x[1]**o+(0.549933699347)*x[1]
        arg[0,1,2,1]=(-0.126173294501)*x[0]**o+(-0.338964866844)*x[0]+(-0.163536115759)*x[1]**o+(0.747551558186)*x[1]
        arg[0,2,0,0]=(0.304334183452)*x[0]**o+(0.927809971187)*x[0]+(0.961504942988)*x[1]**o+(0.0698014252396)*x[1]
        arg[0,2,0,1]=(0.488820088455)*x[0]**o+(0.890735247208)*x[0]+(0.686782901398)*x[1]**o+(0.972954994564)*x[1]
        arg[0,2,1,0]=(-0.737577792766)*x[0]**o+(-0.351698913646)*x[0]+(0.392591489865)*x[1]**o+(0.79238240143)*x[1]
        arg[0,2,1,1]=(-0.997254027544)*x[0]**o+(0.579800166465)*x[0]+(-0.95000690643)*x[1]**o+(-0.166463083476)*x[1]
        arg[0,2,2,0]=(-0.999703749644)*x[0]**o+(0.0415211466468)*x[0]+(0.0816160245139)*x[1]**o+(-0.0921114587195)*x[1]
        arg[0,2,2,1]=(-0.172818721386)*x[0]**o+(0.10475692446)*x[0]+(-0.855815339933)*x[1]**o+(-0.50842580034)*x[1]
        arg[0,3,0,0]=(-0.154563121513)*x[0]**o+(-0.0956305513369)*x[0]+(-0.893783521038)*x[1]**o+(0.10772474638)*x[1]
        arg[0,3,0,1]=(0.365331678349)*x[0]**o+(0.403124820832)*x[0]+(-0.0750500521493)*x[1]**o+(0.62559762808)*x[1]
        arg[0,3,1,0]=(-0.42235875521)*x[0]**o+(0.831133238533)*x[0]+(-0.893322944982)*x[1]**o+(0.936074096319)*x[1]
        arg[0,3,1,1]=(0.587382334689)*x[0]**o+(0.996190710631)*x[0]+(-0.796032791654)*x[1]**o+(0.783491453801)*x[1]
        arg[0,3,2,0]=(-0.369527027508)*x[0]**o+(-0.501417552089)*x[0]+(-0.372105439045)*x[1]**o+(0.653283000152)*x[1]
        arg[0,3,2,1]=(0.914893750314)*x[0]**o+(-0.769183323793)*x[0]+(0.571638486717)*x[1]**o+(-0.939784652259)*x[1]
        arg[0,4,0,0]=(0.134859060251)*x[0]**o+(-0.297363165275)*x[0]+(0.198957028974)*x[1]**o+(-0.828531725127)*x[1]
        arg[0,4,0,1]=(-0.371375370217)*x[0]**o+(-0.0422058718402)*x[0]+(0.502010302278)*x[1]**o+(0.442051802597)*x[1]
        arg[0,4,1,0]=(0.513346571173)*x[0]**o+(0.182640867356)*x[0]+(0.76200283664)*x[1]**o+(0.134640432361)*x[1]
        arg[0,4,1,1]=(-0.815543712756)*x[0]**o+(0.0306226823724)*x[0]+(-0.0211231926242)*x[1]**o+(-0.718800228294)*x[1]
        arg[0,4,2,0]=(-0.980596107555)*x[0]**o+(-0.714237107759)*x[0]+(0.547986215851)*x[1]**o+(0.180131474534)*x[1]
        arg[0,4,2,1]=(-0.460780402239)*x[0]**o+(-0.483771113997)*x[0]+(-0.017140263255)*x[1]**o+(-0.127947184627)*x[1]
        arg[1,0,0,0]=(-0.0187443155707)*x[0]**o+(-0.490914415873)*x[0]+(-0.720022496093)*x[1]**o+(0.0265633770733)*x[1]
        arg[1,0,0,1]=(-0.634855765904)*x[0]**o+(0.345031954366)*x[0]+(-0.0532392363316)*x[1]**o+(0.983295524564)*x[1]
        arg[1,0,1,0]=(0.262182040476)*x[0]**o+(-0.883857758733)*x[0]+(0.880670543535)*x[1]**o+(0.193477695378)*x[1]
        arg[1,0,1,1]=(-0.0825976168371)*x[0]**o+(0.161838720337)*x[0]+(-0.0110587499089)*x[1]**o+(-0.634455402489)*x[1]
        arg[1,0,2,0]=(0.284221958342)*x[0]**o+(-0.948548729029)*x[0]+(0.430914344458)*x[1]**o+(-0.18563638241)*x[1]
        arg[1,0,2,1]=(-0.459558022898)*x[0]**o+(0.431907572866)*x[0]+(-0.0597669070218)*x[1]**o+(0.0694271372225)*x[1]
        arg[1,1,0,0]=(0.54071078447)*x[0]**o+(-0.244570994356)*x[0]+(0.00506219762373)*x[1]**o+(-0.0816378019186)*x[1]
        arg[1,1,0,1]=(-0.100752221379)*x[0]**o+(0.535738345566)*x[0]+(0.730325284553)*x[1]**o+(-0.885044748717)*x[1]
        arg[1,1,1,0]=(-0.0422460353138)*x[0]**o+(-0.986416505995)*x[0]+(-0.111766776997)*x[1]**o+(-0.842179226197)*x[1]
        arg[1,1,1,1]=(0.0690281593169)*x[0]**o+(-0.885054188663)*x[0]+(-0.680349314907)*x[1]**o+(0.00277190934345)*x[1]
        arg[1,1,2,0]=(0.502632146841)*x[0]**o+(-0.197332151163)*x[0]+(-0.0942257361149)*x[1]**o+(0.045171692749)*x[1]
        arg[1,1,2,1]=(0.427910466818)*x[0]**o+(-0.61355947216)*x[0]+(0.465128040517)*x[1]**o+(0.0874536555354)*x[1]
        arg[1,2,0,0]=(0.943819415417)*x[0]**o+(0.884020001869)*x[0]+(-0.769166027477)*x[1]**o+(-0.10032307068)*x[1]
        arg[1,2,0,1]=(0.0670224828564)*x[0]**o+(-0.645424481071)*x[0]+(-0.756361652678)*x[1]**o+(-0.274687916196)*x[1]
        arg[1,2,1,0]=(0.397836742229)*x[0]**o+(-0.708791058745)*x[0]+(0.920869587663)*x[1]**o+(-0.250037069918)*x[1]
        arg[1,2,1,1]=(0.722570486719)*x[0]**o+(-0.358082774936)*x[0]+(0.786344673969)*x[1]**o+(0.310591893637)*x[1]
        arg[1,2,2,0]=(-0.379404573553)*x[0]**o+(-0.969753501645)*x[0]+(-0.0745714915337)*x[1]**o+(-0.262943796911)*x[1]
        arg[1,2,2,1]=(0.817999459632)*x[0]**o+(0.731516351792)*x[0]+(-0.531411613325)*x[1]**o+(-0.824858061154)*x[1]
        arg[1,3,0,0]=(-0.206295444021)*x[0]**o+(-0.781973944918)*x[0]+(-0.173845545382)*x[1]**o+(-0.360122364573)*x[1]
        arg[1,3,0,1]=(-0.0604068005719)*x[0]**o+(0.238691416507)*x[0]+(-0.82262982031)*x[1]**o+(-0.893751755968)*x[1]
        arg[1,3,1,0]=(-0.686409669386)*x[0]**o+(0.711830599399)*x[0]+(0.0839688142166)*x[1]**o+(-0.99463449684)*x[1]
        arg[1,3,1,1]=(-0.0864745975077)*x[0]**o+(0.00068907895313)*x[0]+(-0.766810513534)*x[1]**o+(0.685236202954)*x[1]
        arg[1,3,2,0]=(0.0909338812695)*x[0]**o+(-0.709595654169)*x[0]+(-0.483428504499)*x[1]**o+(-0.0707341191006)*x[1]
        arg[1,3,2,1]=(-0.38342983855)*x[0]**o+(-0.0407824575865)*x[0]+(-0.201329040891)*x[1]**o+(0.0240007937141)*x[1]
        arg[1,4,0,0]=(0.512465803173)*x[0]**o+(-0.371898603608)*x[0]+(-0.79453595861)*x[1]**o+(0.340546986375)*x[1]
        arg[1,4,0,1]=(0.742383198442)*x[0]**o+(-0.603548189912)*x[0]+(-0.270277825546)*x[1]**o+(-0.525100913894)*x[1]
        arg[1,4,1,0]=(-0.693074384029)*x[0]**o+(-0.825037404844)*x[0]+(-0.626116177978)*x[1]**o+(0.628159342479)*x[1]
        arg[1,4,1,1]=(-0.595844943255)*x[0]**o+(0.250758194807)*x[0]+(0.504390052142)*x[1]**o+(-0.124588859702)*x[1]
        arg[1,4,2,0]=(-0.651727490997)*x[0]**o+(0.411138504151)*x[0]+(0.323769160331)*x[1]**o+(0.904379126671)*x[1]
        arg[1,4,2,1]=(-0.71307850178)*x[0]**o+(0.734964608119)*x[0]+(0.348693698506)*x[1]**o+(0.272270639772)*x[1]
        arg[2,0,0,0]=(0.558236567771)*x[0]**o+(0.516985445776)*x[0]+(0.0278915137861)*x[1]**o+(-0.268619977833)*x[1]
        arg[2,0,0,1]=(-0.871181709658)*x[0]**o+(-0.132251987744)*x[0]+(-0.215497073461)*x[1]**o+(0.508008002082)*x[1]
        arg[2,0,1,0]=(0.880795999572)*x[0]**o+(-0.231151166496)*x[0]+(-0.698846681617)*x[1]**o+(0.914483413825)*x[1]
        arg[2,0,1,1]=(0.684792911211)*x[0]**o+(-0.801408782989)*x[0]+(0.689713232168)*x[1]**o+(0.142847173797)*x[1]
        arg[2,0,2,0]=(-0.912566319945)*x[0]**o+(0.811138574657)*x[0]+(-0.884250879071)*x[1]**o+(-0.177206225477)*x[1]
        arg[2,0,2,1]=(-0.898648046678)*x[0]**o+(0.581554373688)*x[0]+(0.567642878443)*x[1]**o+(0.636312832751)*x[1]
        arg[2,1,0,0]=(-0.788853365648)*x[0]**o+(-0.135329327941)*x[0]+(0.180850147594)*x[1]**o+(0.742619029335)*x[1]
        arg[2,1,0,1]=(-0.824539980645)*x[0]**o+(0.857173393615)*x[0]+(0.850455264916)*x[1]**o+(0.975422322464)*x[1]
        arg[2,1,1,0]=(0.462293517317)*x[0]**o+(-0.173835273794)*x[0]+(-0.800072514276)*x[1]**o+(0.140214114627)*x[1]
        arg[2,1,1,1]=(-0.187890814293)*x[0]**o+(0.534472364495)*x[0]+(-0.799640665571)*x[1]**o+(0.301287955302)*x[1]
        arg[2,1,2,0]=(0.278901492946)*x[0]**o+(0.620544326886)*x[0]+(-0.705702443752)*x[1]**o+(-0.489690636328)*x[1]
        arg[2,1,2,1]=(-0.926772900235)*x[0]**o+(0.134654742845)*x[0]+(0.356561975337)*x[1]**o+(-0.919999359857)*x[1]
        arg[2,2,0,0]=(-0.675956301416)*x[0]**o+(0.149582589205)*x[0]+(0.186440827385)*x[1]**o+(0.88381610446)*x[1]
        arg[2,2,0,1]=(-0.697267962915)*x[0]**o+(0.991366596424)*x[0]+(0.310290094618)*x[1]**o+(-0.881771505066)*x[1]
        arg[2,2,1,0]=(0.506027358474)*x[0]**o+(0.294136137354)*x[0]+(0.732003937836)*x[1]**o+(0.633469131504)*x[1]
        arg[2,2,1,1]=(-0.427146039728)*x[0]**o+(-0.975737063903)*x[0]+(-0.934913653772)*x[1]**o+(-0.662568325005)*x[1]
        arg[2,2,2,0]=(0.477849916621)*x[0]**o+(-0.798883883367)*x[0]+(-0.154217684441)*x[1]**o+(-0.175817547912)*x[1]
        arg[2,2,2,1]=(0.935192620387)*x[0]**o+(-0.193802588249)*x[0]+(0.406181949213)*x[1]**o+(0.20422475396)*x[1]
        arg[2,3,0,0]=(0.965944278856)*x[0]**o+(-0.736175411725)*x[0]+(0.92404347386)*x[1]**o+(0.273152788392)*x[1]
        arg[2,3,0,1]=(-0.606944412413)*x[0]**o+(0.692555413476)*x[0]+(0.812085676275)*x[1]**o+(-0.0809920218886)*x[1]
        arg[2,3,1,0]=(-0.314634777406)*x[0]**o+(0.750181618592)*x[0]+(0.0554739779642)*x[1]**o+(-0.333788326726)*x[1]
        arg[2,3,1,1]=(0.85610815703)*x[0]**o+(0.910096005524)*x[0]+(-0.227092814843)*x[1]**o+(-0.0849532598914)*x[1]
        arg[2,3,2,0]=(0.735064128504)*x[0]**o+(-0.253919528943)*x[0]+(0.8457463307)*x[1]**o+(0.865727345925)*x[1]
        arg[2,3,2,1]=(0.520607989389)*x[0]**o+(0.569210317105)*x[0]+(0.500780081677)*x[1]**o+(-0.787630914351)*x[1]
        arg[2,4,0,0]=(0.179774694296)*x[0]**o+(-0.890872128607)*x[0]+(-0.185758177247)*x[1]**o+(0.888231734606)*x[1]
        arg[2,4,0,1]=(0.059863462227)*x[0]**o+(-0.0796463123948)*x[0]+(-0.104564520504)*x[1]**o+(-0.336182528617)*x[1]
        arg[2,4,1,0]=(-0.636828275967)*x[0]**o+(0.156894572218)*x[0]+(-0.884260096115)*x[1]**o+(-0.967431438795)*x[1]
        arg[2,4,1,1]=(-0.958168110389)*x[0]**o+(-0.0424772631477)*x[0]+(0.512363473316)*x[1]**o+(-0.716729419668)*x[1]
        arg[2,4,2,0]=(0.19736654467)*x[0]**o+(-0.288723518002)*x[0]+(0.243674007033)*x[1]**o+(0.911210601142)*x[1]
        arg[2,4,2,1]=(-0.847493343055)*x[0]**o+(0.115003265593)*x[0]+(0.360071936714)*x[1]**o+(-0.0265385061141)*x[1]
        arg[3,0,0,0]=(-0.510550590836)*x[0]**o+(-0.266174606631)*x[0]+(0.771580708159)*x[1]**o+(-0.770554374259)*x[1]
        arg[3,0,0,1]=(-0.65795024738)*x[0]**o+(-0.83394628525)*x[0]+(0.812461325452)*x[1]**o+(0.191593733482)*x[1]
        arg[3,0,1,0]=(0.136878718548)*x[0]**o+(-0.000345061692795)*x[0]+(-0.0780280205008)*x[1]**o+(0.173925537373)*x[1]
        arg[3,0,1,1]=(0.584079470149)*x[0]**o+(0.138674493948)*x[0]+(-0.563553909861)*x[1]**o+(-0.254304838008)*x[1]
        arg[3,0,2,0]=(0.326192887554)*x[0]**o+(0.993615980303)*x[0]+(-0.26995279693)*x[1]**o+(0.216112615287)*x[1]
        arg[3,0,2,1]=(0.684786021819)*x[0]**o+(0.158934017292)*x[0]+(-0.457448199191)*x[1]**o+(0.644722228163)*x[1]
        arg[3,1,0,0]=(-0.198252162262)*x[0]**o+(-0.821832378445)*x[0]+(0.831807441916)*x[1]**o+(-0.70643670471)*x[1]
        arg[3,1,0,1]=(0.361698759016)*x[0]**o+(0.556511107101)*x[0]+(0.892219700601)*x[1]**o+(-0.861598520361)*x[1]
        arg[3,1,1,0]=(-0.107946706634)*x[0]**o+(-0.910191934756)*x[0]+(0.502507607871)*x[1]**o+(0.649089810516)*x[1]
        arg[3,1,1,1]=(-0.92959760672)*x[0]**o+(0.950682520961)*x[0]+(0.103321004187)*x[1]**o+(-0.631562667815)*x[1]
        arg[3,1,2,0]=(0.858565242025)*x[0]**o+(-0.106810606075)*x[0]+(0.236377425048)*x[1]**o+(-0.296320858145)*x[1]
        arg[3,1,2,1]=(0.148558873625)*x[0]**o+(0.458488334082)*x[0]+(-0.0172824905523)*x[1]**o+(-0.321212766581)*x[1]
        arg[3,2,0,0]=(-0.239733890336)*x[0]**o+(0.558819103697)*x[0]+(-0.151683129885)*x[1]**o+(0.153995780645)*x[1]
        arg[3,2,0,1]=(-0.949100270937)*x[0]**o+(0.0687171499139)*x[0]+(0.968028021831)*x[1]**o+(-0.935600499345)*x[1]
        arg[3,2,1,0]=(-0.739823767237)*x[0]**o+(0.417825150389)*x[0]+(0.318612170707)*x[1]**o+(0.627557337421)*x[1]
        arg[3,2,1,1]=(-0.965142167324)*x[0]**o+(0.981446854383)*x[0]+(0.266191185951)*x[1]**o+(-0.581303142841)*x[1]
        arg[3,2,2,0]=(-0.00312606993817)*x[0]**o+(-0.191885030828)*x[0]+(-0.342525068561)*x[1]**o+(-0.953710538626)*x[1]
        arg[3,2,2,1]=(-0.689127187476)*x[0]**o+(-0.107190366063)*x[0]+(-0.0556716250594)*x[1]**o+(-0.609462166754)*x[1]
        arg[3,3,0,0]=(-0.765237197853)*x[0]**o+(-0.0707283944117)*x[0]+(0.891789872287)*x[1]**o+(-0.255939791858)*x[1]
        arg[3,3,0,1]=(-0.367658362103)*x[0]**o+(0.27406309422)*x[0]+(0.866288683947)*x[1]**o+(0.403194030287)*x[1]
        arg[3,3,1,0]=(-0.362795000993)*x[0]**o+(-0.498774415158)*x[0]+(0.0726808331583)*x[1]**o+(-0.871066791136)*x[1]
        arg[3,3,1,1]=(-0.177149059814)*x[0]**o+(0.287463753612)*x[0]+(0.101726559941)*x[1]**o+(0.561797204517)*x[1]
        arg[3,3,2,0]=(0.770615121039)*x[0]**o+(-0.503402148251)*x[0]+(0.319753007634)*x[1]**o+(0.586868894261)*x[1]
        arg[3,3,2,1]=(0.924125904724)*x[0]**o+(-0.508988920423)*x[0]+(-0.866752504824)*x[1]**o+(0.518544700652)*x[1]
        arg[3,4,0,0]=(-0.973660291124)*x[0]**o+(0.746869369283)*x[0]+(-0.634097171844)*x[1]**o+(0.723638944736)*x[1]
        arg[3,4,0,1]=(-0.599366869066)*x[0]**o+(-0.59893829688)*x[0]+(0.124483401866)*x[1]**o+(-0.924820315792)*x[1]
        arg[3,4,1,0]=(-0.707374787729)*x[0]**o+(-0.494748755022)*x[0]+(0.391752209698)*x[1]**o+(0.149710640116)*x[1]
        arg[3,4,1,1]=(0.329587748119)*x[0]**o+(-0.788289803764)*x[0]+(-0.145100163626)*x[1]**o+(0.141493810237)*x[1]
        arg[3,4,2,0]=(0.17496644337)*x[0]**o+(-0.270242104589)*x[0]+(0.996684379179)*x[1]**o+(-0.751697440046)*x[1]
        arg[3,4,2,1]=(0.79046837395)*x[0]**o+(0.568394473062)*x[0]+(0.359419854016)*x[1]**o+(0.630686948244)*x[1]
        ref[0,0,0,0]=(0.306552070969)/(o+1.)+(0.157870653629)
        ref[0,0,0,1]=(0.0606500807631)/(o+1.)+(0.0612449724389)
        ref[0,0,1,0]=(0.639564070889)/(o+1.)+(0.666341832555)
        ref[0,0,1,1]=(0.671248593147)/(o+1.)+(-0.528628572255)
        ref[0,0,2,0]=(0.0735824774293)/(o+1.)+(-0.474736472501)
        ref[0,0,2,1]=(-0.9070527228)/(o+1.)+(-0.197350261427)
        ref[0,1,0,0]=(1.30449671875)/(o+1.)+(0.106123067554)
        ref[0,1,0,1]=(-0.420072850657)/(o+1.)+(-0.407896954678)
        ref[0,1,1,0]=(1.44174379356)/(o+1.)+(-0.809395886627)
        ref[0,1,1,1]=(0.155435385684)/(o+1.)+(0.627148132992)
        ref[0,1,2,0]=(-0.137204404518)/(o+1.)+(0.207874473889)
        ref[0,1,2,1]=(-0.289709410261)/(o+1.)+(0.204293345671)
        ref[0,2,0,0]=(1.26583912644)/(o+1.)+(0.498805698213)
        ref[0,2,0,1]=(1.17560298985)/(o+1.)+(0.931845120886)
        ref[0,2,1,0]=(-0.344986302901)/(o+1.)+(0.220341743892)
        ref[0,2,1,1]=(-1.94726093397)/(o+1.)+(0.206668541494)
        ref[0,2,2,0]=(-0.91808772513)/(o+1.)+(-0.0252951560363)
        ref[0,2,2,1]=(-1.02863406132)/(o+1.)+(-0.20183443794)
        ref[0,3,0,0]=(-1.04834664255)/(o+1.)+(0.00604709752155)
        ref[0,3,0,1]=(0.2902816262)/(o+1.)+(0.514361224456)
        ref[0,3,1,0]=(-1.31568170019)/(o+1.)+(0.883603667426)
        ref[0,3,1,1]=(-0.208650456965)/(o+1.)+(0.889841082216)
        ref[0,3,2,0]=(-0.741632466554)/(o+1.)+(0.0759327240318)
        ref[0,3,2,1]=(1.48653223703)/(o+1.)+(-0.854483988026)
        ref[0,4,0,0]=(0.333816089225)/(o+1.)+(-0.562947445201)
        ref[0,4,0,1]=(0.130634932061)/(o+1.)+(0.199922965378)
        ref[0,4,1,0]=(1.27534940781)/(o+1.)+(0.158640649858)
        ref[0,4,1,1]=(-0.83666690538)/(o+1.)+(-0.344088772961)
        ref[0,4,2,0]=(-0.432609891704)/(o+1.)+(-0.267052816612)
        ref[0,4,2,1]=(-0.477920665494)/(o+1.)+(-0.305859149312)
        ref[1,0,0,0]=(-0.738766811663)/(o+1.)+(-0.2321755194)
        ref[1,0,0,1]=(-0.688095002235)/(o+1.)+(0.664163739465)
        ref[1,0,1,0]=(1.14285258401)/(o+1.)+(-0.345190031677)
        ref[1,0,1,1]=(-0.093656366746)/(o+1.)+(-0.236308341076)
        ref[1,0,2,0]=(0.7151363028)/(o+1.)+(-0.56709255572)
        ref[1,0,2,1]=(-0.51932492992)/(o+1.)+(0.250667355044)
        ref[1,1,0,0]=(0.545772982093)/(o+1.)+(-0.163104398138)
        ref[1,1,0,1]=(0.629573063174)/(o+1.)+(-0.174653201576)
        ref[1,1,1,0]=(-0.154012812311)/(o+1.)+(-0.914297866096)
        ref[1,1,1,1]=(-0.61132115559)/(o+1.)+(-0.44114113966)
        ref[1,1,2,0]=(0.408406410726)/(o+1.)+(-0.076080229207)
        ref[1,1,2,1]=(0.893038507335)/(o+1.)+(-0.263052908312)
        ref[1,2,0,0]=(0.17465338794)/(o+1.)+(0.391848465594)
        ref[1,2,0,1]=(-0.689339169821)/(o+1.)+(-0.460056198634)
        ref[1,2,1,0]=(1.31870632989)/(o+1.)+(-0.479414064332)
        ref[1,2,1,1]=(1.50891516069)/(o+1.)+(-0.0237454406495)
        ref[1,2,2,0]=(-0.453976065087)/(o+1.)+(-0.616348649278)
        ref[1,2,2,1]=(0.286587846307)/(o+1.)+(-0.0466708546809)
        ref[1,3,0,0]=(-0.380140989403)/(o+1.)+(-0.571048154745)
        ref[1,3,0,1]=(-0.883036620882)/(o+1.)+(-0.32753016973)
        ref[1,3,1,0]=(-0.60244085517)/(o+1.)+(-0.141401948721)
        ref[1,3,1,1]=(-0.853285111042)/(o+1.)+(0.342962640954)
        ref[1,3,2,0]=(-0.39249462323)/(o+1.)+(-0.390164886635)
        ref[1,3,2,1]=(-0.584758879441)/(o+1.)+(-0.00839083193621)
        ref[1,4,0,0]=(-0.282070155438)/(o+1.)+(-0.0156758086169)
        ref[1,4,0,1]=(0.472105372896)/(o+1.)+(-0.564324551903)
        ref[1,4,1,0]=(-1.31919056201)/(o+1.)+(-0.0984390311826)
        ref[1,4,1,1]=(-0.0914548911124)/(o+1.)+(0.0630846675525)
        ref[1,4,2,0]=(-0.327958330666)/(o+1.)+(0.657758815411)
        ref[1,4,2,1]=(-0.364384803274)/(o+1.)+(0.503617623946)
        ref[2,0,0,0]=(0.586128081557)/(o+1.)+(0.124182733972)
        ref[2,0,0,1]=(-1.08667878312)/(o+1.)+(0.187878007169)
        ref[2,0,1,0]=(0.181949317955)/(o+1.)+(0.341666123665)
        ref[2,0,1,1]=(1.37450614338)/(o+1.)+(-0.329280804596)
        ref[2,0,2,0]=(-1.79681719902)/(o+1.)+(0.31696617459)
        ref[2,0,2,1]=(-0.331005168235)/(o+1.)+(0.608933603219)
        ref[2,1,0,0]=(-0.608003218053)/(o+1.)+(0.303644850697)
        ref[2,1,0,1]=(0.0259152842714)/(o+1.)+(0.916297858039)
        ref[2,1,1,0]=(-0.33777899696)/(o+1.)+(-0.0168105795836)
        ref[2,1,1,1]=(-0.987531479864)/(o+1.)+(0.417880159899)
        ref[2,1,2,0]=(-0.426800950806)/(o+1.)+(0.0654268452787)
        ref[2,1,2,1]=(-0.570210924898)/(o+1.)+(-0.392672308506)
        ref[2,2,0,0]=(-0.489515474032)/(o+1.)+(0.516699346832)
        ref[2,2,0,1]=(-0.386977868298)/(o+1.)+(0.054797545679)
        ref[2,2,1,0]=(1.23803129631)/(o+1.)+(0.463802634429)
        ref[2,2,1,1]=(-1.3620596935)/(o+1.)+(-0.819152694454)
        ref[2,2,2,0]=(0.32363223218)/(o+1.)+(-0.48735071564)
        ref[2,2,2,1]=(1.3413745696)/(o+1.)+(0.00521108285542)
        ref[2,3,0,0]=(1.88998775272)/(o+1.)+(-0.231511311666)
        ref[2,3,0,1]=(0.205141263861)/(o+1.)+(0.305781695794)
        ref[2,3,1,0]=(-0.259160799442)/(o+1.)+(0.208196645933)
        ref[2,3,1,1]=(0.629015342187)/(o+1.)+(0.412571372816)
        ref[2,3,2,0]=(1.5808104592)/(o+1.)+(0.305903908491)
        ref[2,3,2,1]=(1.02138807107)/(o+1.)+(-0.109210298623)
        ref[2,4,0,0]=(-0.00598348295078)/(o+1.)+(-0.00132019700044)
        ref[2,4,0,1]=(-0.0447010582767)/(o+1.)+(-0.207914420506)
        ref[2,4,1,0]=(-1.52108837208)/(o+1.)+(-0.405268433289)
        ref[2,4,1,1]=(-0.445804637073)/(o+1.)+(-0.379603341408)
        ref[2,4,2,0]=(0.441040551703)/(o+1.)+(0.31124354157)
        ref[2,4,2,1]=(-0.487421406342)/(o+1.)+(0.0442323797396)
        ref[3,0,0,0]=(0.261030117323)/(o+1.)+(-0.518364490445)
        ref[3,0,0,1]=(0.154511078072)/(o+1.)+(-0.321176275884)
        ref[3,0,1,0]=(0.0588506980476)/(o+1.)+(0.0867902378403)
        ref[3,0,1,1]=(0.0205255602871)/(o+1.)+(-0.0578151720304)
        ref[3,0,2,0]=(0.0562400906237)/(o+1.)+(0.604864297795)
        ref[3,0,2,1]=(0.227337822629)/(o+1.)+(0.401828122727)
        ref[3,1,0,0]=(0.633555279654)/(o+1.)+(-0.764134541578)
        ref[3,1,0,1]=(1.25391845962)/(o+1.)+(-0.15254370663)
        ref[3,1,1,0]=(0.394560901238)/(o+1.)+(-0.13055106212)
        ref[3,1,1,1]=(-0.826276602534)/(o+1.)+(0.159559926573)
        ref[3,1,2,0]=(1.09494266707)/(o+1.)+(-0.20156573211)
        ref[3,1,2,1]=(0.131276383073)/(o+1.)+(0.0686377837506)
        ref[3,2,0,0]=(-0.391417020221)/(o+1.)+(0.356407442171)
        ref[3,2,0,1]=(0.0189277508933)/(o+1.)+(-0.433441674716)
        ref[3,2,1,0]=(-0.42121159653)/(o+1.)+(0.522691243905)
        ref[3,2,1,1]=(-0.698950981373)/(o+1.)+(0.200071855771)
        ref[3,2,2,0]=(-0.345651138499)/(o+1.)+(-0.572797784727)
        ref[3,2,2,1]=(-0.744798812536)/(o+1.)+(-0.358326266408)
        ref[3,3,0,0]=(0.126552674434)/(o+1.)+(-0.163334093135)
        ref[3,3,0,1]=(0.498630321844)/(o+1.)+(0.338628562253)
        ref[3,3,1,0]=(-0.290114167835)/(o+1.)+(-0.684920603147)
        ref[3,3,1,1]=(-0.0754224998727)/(o+1.)+(0.424630479064)
        ref[3,3,2,0]=(1.09036812867)/(o+1.)+(0.0417333730049)
        ref[3,3,2,1]=(0.0573733998999)/(o+1.)+(0.00477789011464)
        ref[3,4,0,0]=(-1.60775746297)/(o+1.)+(0.73525415701)
        ref[3,4,0,1]=(-0.4748834672)/(o+1.)+(-0.761879306336)
        ref[3,4,1,0]=(-0.31562257803)/(o+1.)+(-0.172519057453)
        ref[3,4,1,1]=(0.184487584493)/(o+1.)+(-0.323397996764)
        ref[3,4,2,0]=(1.17165082255)/(o+1.)+(-0.510969772317)
        ref[3,4,2,1]=(1.14988822797)/(o+1.)+(0.599540710653)
      else:
        arg[0,0,0,0]=(-0.517412938774)*x[0]**o+(0.824804313111)*x[0]+(-0.681706563374)*x[1]**o+(-0.259540440272)*x[1]+(0.876735283937)*x[2]**o+(0.307072859807)*x[2]
        arg[0,0,0,1]=(-0.545981202251)*x[0]**o+(-0.530080933212)*x[0]+(-0.339289111599)*x[1]**o+(-0.726311605088)*x[1]+(0.334837705444)*x[2]**o+(-0.422667311064)*x[2]
        arg[0,0,1,0]=(-0.94261387328)*x[0]**o+(-0.315825326734)*x[0]+(0.542502801277)*x[1]**o+(-0.927525943646)*x[1]+(-0.802118238213)*x[2]**o+(-0.106512314706)*x[2]
        arg[0,0,1,1]=(-0.877940099286)*x[0]**o+(-0.801256557869)*x[0]+(-0.155983792242)*x[1]**o+(0.200547191496)*x[1]+(-0.960553133432)*x[2]**o+(0.968862792367)*x[2]
        arg[0,0,2,0]=(-0.870342547946)*x[0]**o+(-0.956447402661)*x[0]+(-0.506797803004)*x[1]**o+(0.342841675107)*x[1]+(0.528354159344)*x[2]**o+(-0.824546729564)*x[2]
        arg[0,0,2,1]=(-0.411899137296)*x[0]**o+(0.133736458559)*x[0]+(0.852726131619)*x[1]**o+(0.692795229813)*x[1]+(0.047372576403)*x[2]**o+(0.810485765945)*x[2]
        arg[0,1,0,0]=(0.961758233254)*x[0]**o+(-0.606307924964)*x[0]+(0.0612696412665)*x[1]**o+(-0.947823541802)*x[1]+(-0.113021104365)*x[2]**o+(-0.00802440322155)*x[2]
        arg[0,1,0,1]=(-0.684097052467)*x[0]**o+(0.467140835537)*x[0]+(-0.0742537091205)*x[1]**o+(0.541641916143)*x[1]+(0.987491548505)*x[2]**o+(-0.558380262255)*x[2]
        arg[0,1,1,0]=(-0.901821988855)*x[0]**o+(0.34443714654)*x[0]+(-0.682552654793)*x[1]**o+(0.398276086461)*x[1]+(0.334644264965)*x[2]**o+(0.21454845251)*x[2]
        arg[0,1,1,1]=(0.929614060295)*x[0]**o+(-0.322609307947)*x[0]+(-0.426962200724)*x[1]**o+(0.464659992564)*x[1]+(-0.13983863674)*x[2]**o+(-0.681648817325)*x[2]
        arg[0,1,2,0]=(0.504166508914)*x[0]**o+(-0.978562859041)*x[0]+(-0.769650404806)*x[1]**o+(0.443500137153)*x[1]+(0.969669965936)*x[2]**o+(0.401742207403)*x[2]
        arg[0,1,2,1]=(-0.478561324822)*x[0]**o+(-0.772019838834)*x[0]+(-0.779194844829)*x[1]**o+(-0.747980400065)*x[1]+(0.827885608192)*x[2]**o+(0.135571151528)*x[2]
        arg[0,2,0,0]=(-0.210426221052)*x[0]**o+(-0.464938141903)*x[0]+(-0.776844066203)*x[1]**o+(0.267244555705)*x[1]+(-0.773135929429)*x[2]**o+(-0.00846060888954)*x[2]
        arg[0,2,0,1]=(0.0945858334531)*x[0]**o+(-0.492400895713)*x[0]+(-0.128564669727)*x[1]**o+(-0.0951458189983)*x[1]+(-0.54426715378)*x[2]**o+(0.106545346072)*x[2]
        arg[0,2,1,0]=(-0.482424571392)*x[0]**o+(0.0431208155652)*x[0]+(0.557138601636)*x[1]**o+(-0.725564282551)*x[1]+(0.9081107419)*x[2]**o+(0.0179201585352)*x[2]
        arg[0,2,1,1]=(-0.555731853052)*x[0]**o+(0.447137254671)*x[0]+(0.820958717703)*x[1]**o+(-0.580251434504)*x[1]+(0.227884622965)*x[2]**o+(0.34460189453)*x[2]
        arg[0,2,2,0]=(-0.25868983898)*x[0]**o+(0.759165881486)*x[0]+(-0.499903058341)*x[1]**o+(0.896471093639)*x[1]+(0.351955520633)*x[2]**o+(0.16566756566)*x[2]
        arg[0,2,2,1]=(-0.37859354794)*x[0]**o+(0.925387188779)*x[0]+(0.0111055293719)*x[1]**o+(-0.311998642154)*x[1]+(0.441009182046)*x[2]**o+(0.0968247918271)*x[2]
        arg[0,3,0,0]=(-0.461665900641)*x[0]**o+(0.812002051396)*x[0]+(-0.767456852027)*x[1]**o+(0.809069781575)*x[1]+(0.690416953885)*x[2]**o+(0.171662369216)*x[2]
        arg[0,3,0,1]=(-0.804811324639)*x[0]**o+(-0.0962711523954)*x[0]+(0.980130059825)*x[1]**o+(0.98279180603)*x[1]+(-0.0614353559054)*x[2]**o+(0.645637153538)*x[2]
        arg[0,3,1,0]=(0.97878415369)*x[0]**o+(-0.676321251698)*x[0]+(0.916075195717)*x[1]**o+(0.244509683371)*x[1]+(-0.939563193145)*x[2]**o+(-0.866506108913)*x[2]
        arg[0,3,1,1]=(-0.676965895162)*x[0]**o+(-0.124355338519)*x[0]+(0.680151101195)*x[1]**o+(-0.839412433156)*x[1]+(0.434088587223)*x[2]**o+(-0.275286048166)*x[2]
        arg[0,3,2,0]=(0.414822536626)*x[0]**o+(0.493836498194)*x[0]+(0.967077097745)*x[1]**o+(-0.0115301157869)*x[1]+(-0.307994657929)*x[2]**o+(-0.967766608312)*x[2]
        arg[0,3,2,1]=(-0.34064670348)*x[0]**o+(-0.975121872905)*x[0]+(0.0839457874766)*x[1]**o+(0.0995106978636)*x[1]+(-0.533837237516)*x[2]**o+(0.771268044302)*x[2]
        arg[0,4,0,0]=(0.223197544637)*x[0]**o+(0.712738871576)*x[0]+(-0.386972075747)*x[1]**o+(0.289829507744)*x[1]+(0.153418793839)*x[2]**o+(0.876087048492)*x[2]
        arg[0,4,0,1]=(0.696750948653)*x[0]**o+(0.466905335953)*x[0]+(-0.611476041907)*x[1]**o+(-0.939902457901)*x[1]+(-0.752675432265)*x[2]**o+(-0.869502717044)*x[2]
        arg[0,4,1,0]=(-0.465562201723)*x[0]**o+(-0.388716352051)*x[0]+(0.260044093503)*x[1]**o+(0.919379163294)*x[1]+(-0.902453394797)*x[2]**o+(0.378318744505)*x[2]
        arg[0,4,1,1]=(0.121990842946)*x[0]**o+(-0.691267894153)*x[0]+(-0.703960568755)*x[1]**o+(-0.582404896428)*x[1]+(0.57560180036)*x[2]**o+(0.537099448589)*x[2]
        arg[0,4,2,0]=(-0.495452196513)*x[0]**o+(-0.939389935669)*x[0]+(-0.237205717598)*x[1]**o+(0.61376516207)*x[1]+(-0.506082738728)*x[2]**o+(0.407704148542)*x[2]
        arg[0,4,2,1]=(0.450976180771)*x[0]**o+(-0.00739258898437)*x[0]+(0.210130848093)*x[1]**o+(0.624639007009)*x[1]+(0.173795671727)*x[2]**o+(0.508269170853)*x[2]
        arg[1,0,0,0]=(0.609685942177)*x[0]**o+(0.609299494609)*x[0]+(0.321046388121)*x[1]**o+(0.598197189498)*x[1]+(0.536092177618)*x[2]**o+(0.576427431157)*x[2]
        arg[1,0,0,1]=(0.412500706234)*x[0]**o+(0.905942239017)*x[0]+(-0.974980343169)*x[1]**o+(-0.131245997607)*x[1]+(0.745809068018)*x[2]**o+(0.543470258153)*x[2]
        arg[1,0,1,0]=(-0.903032545947)*x[0]**o+(-0.778693321051)*x[0]+(0.399342164105)*x[1]**o+(0.478212896767)*x[1]+(0.254596269802)*x[2]**o+(0.70308382001)*x[2]
        arg[1,0,1,1]=(-0.281529506793)*x[0]**o+(0.74166036027)*x[0]+(0.629713747725)*x[1]**o+(-0.275164746597)*x[1]+(0.37523909979)*x[2]**o+(-0.878026427978)*x[2]
        arg[1,0,2,0]=(0.236027764061)*x[0]**o+(-0.816910805083)*x[0]+(0.0782011624185)*x[1]**o+(-0.644004245666)*x[1]+(0.293586462602)*x[2]**o+(0.557045894695)*x[2]
        arg[1,0,2,1]=(0.868061727269)*x[0]**o+(0.53329318943)*x[0]+(0.367603915235)*x[1]**o+(0.89280277925)*x[1]+(-0.316595424948)*x[2]**o+(0.593830565147)*x[2]
        arg[1,1,0,0]=(0.886950595996)*x[0]**o+(-0.290323112286)*x[0]+(-0.863347030089)*x[1]**o+(0.545096994541)*x[1]+(0.338200944281)*x[2]**o+(-0.79898415048)*x[2]
        arg[1,1,0,1]=(0.103761462006)*x[0]**o+(-0.046916161864)*x[0]+(0.0312549239467)*x[1]**o+(0.214045118984)*x[1]+(-0.523161766402)*x[2]**o+(0.714207028317)*x[2]
        arg[1,1,1,0]=(0.494943183987)*x[0]**o+(-0.0543096386006)*x[0]+(-0.263875120588)*x[1]**o+(0.776031489925)*x[1]+(0.881224752145)*x[2]**o+(-0.650038283228)*x[2]
        arg[1,1,1,1]=(0.0871887540513)*x[0]**o+(-0.953848841349)*x[0]+(-0.223751459589)*x[1]**o+(0.227400823993)*x[1]+(-0.224200553009)*x[2]**o+(0.594012970899)*x[2]
        arg[1,1,2,0]=(-0.443116951346)*x[0]**o+(0.891806149626)*x[0]+(0.94731530917)*x[1]**o+(0.739499894809)*x[1]+(-0.92613053703)*x[2]**o+(0.464883087528)*x[2]
        arg[1,1,2,1]=(-0.149831145559)*x[0]**o+(0.913999704597)*x[0]+(0.76082978152)*x[1]**o+(-0.441535347622)*x[1]+(0.902841586936)*x[2]**o+(-0.509022004948)*x[2]
        arg[1,2,0,0]=(0.508230188489)*x[0]**o+(0.241669921231)*x[0]+(-0.802811365056)*x[1]**o+(0.298424305948)*x[1]+(-0.949407454532)*x[2]**o+(-0.131426880214)*x[2]
        arg[1,2,0,1]=(0.65372313274)*x[0]**o+(0.195891642771)*x[0]+(-0.160845429219)*x[1]**o+(0.500815823724)*x[1]+(0.875785249339)*x[2]**o+(0.325024339211)*x[2]
        arg[1,2,1,0]=(0.908405682536)*x[0]**o+(-0.146675166477)*x[0]+(0.24033815873)*x[1]**o+(-0.62814219771)*x[1]+(-0.512365788351)*x[2]**o+(-0.103817191415)*x[2]
        arg[1,2,1,1]=(0.824305237501)*x[0]**o+(0.175753992729)*x[0]+(0.992502114126)*x[1]**o+(0.973110627555)*x[1]+(0.355352002177)*x[2]**o+(0.112252263955)*x[2]
        arg[1,2,2,0]=(-0.485996285003)*x[0]**o+(-0.532551123887)*x[0]+(0.708205121405)*x[1]**o+(-0.803269038441)*x[1]+(-0.232049124534)*x[2]**o+(0.55437652866)*x[2]
        arg[1,2,2,1]=(0.551934776224)*x[0]**o+(0.727629664538)*x[0]+(0.197393511049)*x[1]**o+(-0.146619812174)*x[1]+(0.316592306936)*x[2]**o+(0.0263881493809)*x[2]
        arg[1,3,0,0]=(-0.918554166254)*x[0]**o+(-0.839150758884)*x[0]+(0.436713305329)*x[1]**o+(-0.410131731387)*x[1]+(0.321246028791)*x[2]**o+(0.706389384376)*x[2]
        arg[1,3,0,1]=(-0.870168753295)*x[0]**o+(-0.00473364206025)*x[0]+(0.594648540734)*x[1]**o+(-0.924887214906)*x[1]+(0.58858894515)*x[2]**o+(0.0152152159566)*x[2]
        arg[1,3,1,0]=(0.64005604234)*x[0]**o+(0.308265334836)*x[0]+(0.321645661286)*x[1]**o+(0.670017614875)*x[1]+(0.712764136703)*x[2]**o+(0.458986964342)*x[2]
        arg[1,3,1,1]=(-0.201376578944)*x[0]**o+(-0.271793865118)*x[0]+(-0.798181331582)*x[1]**o+(0.690100877945)*x[1]+(0.600350295622)*x[2]**o+(0.341066858436)*x[2]
        arg[1,3,2,0]=(0.459832113137)*x[0]**o+(0.0566780273877)*x[0]+(0.221723079112)*x[1]**o+(-0.681001728734)*x[1]+(0.265505135729)*x[2]**o+(-0.389365671367)*x[2]
        arg[1,3,2,1]=(0.715444384493)*x[0]**o+(-0.705697520786)*x[0]+(0.945208081248)*x[1]**o+(0.0233171867808)*x[1]+(0.740156166802)*x[2]**o+(-0.144314337012)*x[2]
        arg[1,4,0,0]=(-0.490162545806)*x[0]**o+(0.542981268396)*x[0]+(-0.841471465564)*x[1]**o+(0.79091798393)*x[1]+(0.245598142884)*x[2]**o+(0.4740296042)*x[2]
        arg[1,4,0,1]=(-0.504504615322)*x[0]**o+(0.45494067251)*x[0]+(-0.612427248728)*x[1]**o+(0.0880340051743)*x[1]+(0.645906227711)*x[2]**o+(0.772038416622)*x[2]
        arg[1,4,1,0]=(0.945538632492)*x[0]**o+(0.98519603491)*x[0]+(-0.245495451859)*x[1]**o+(-0.70732864026)*x[1]+(0.378733662069)*x[2]**o+(0.579554415018)*x[2]
        arg[1,4,1,1]=(0.886659064398)*x[0]**o+(-0.491870874438)*x[0]+(0.751814742427)*x[1]**o+(0.430589443767)*x[1]+(-0.729193959143)*x[2]**o+(-0.699344758538)*x[2]
        arg[1,4,2,0]=(-0.4190140481)*x[0]**o+(0.84509612452)*x[0]+(-0.263608911857)*x[1]**o+(0.693648282371)*x[1]+(-0.144463932166)*x[2]**o+(0.384293689266)*x[2]
        arg[1,4,2,1]=(-0.266973780787)*x[0]**o+(0.613988367707)*x[0]+(0.686955199789)*x[1]**o+(-0.301585818785)*x[1]+(-0.574601202558)*x[2]**o+(-0.430718616071)*x[2]
        arg[2,0,0,0]=(-0.117390727803)*x[0]**o+(-0.25850139145)*x[0]+(0.324909086031)*x[1]**o+(-0.593112241553)*x[1]+(-0.51016768177)*x[2]**o+(0.392299415334)*x[2]
        arg[2,0,0,1]=(0.290478531696)*x[0]**o+(0.758153023626)*x[0]+(0.531003993361)*x[1]**o+(-0.112481510374)*x[1]+(0.950222158203)*x[2]**o+(-0.742480526204)*x[2]
        arg[2,0,1,0]=(-0.754820921225)*x[0]**o+(-0.408482044077)*x[0]+(-0.229697426486)*x[1]**o+(-0.983010424713)*x[1]+(-0.286618604257)*x[2]**o+(-0.489034108451)*x[2]
        arg[2,0,1,1]=(0.96775757822)*x[0]**o+(0.797765725721)*x[0]+(-0.90936176648)*x[1]**o+(-0.372755885193)*x[1]+(-0.144727874143)*x[2]**o+(0.341624944551)*x[2]
        arg[2,0,2,0]=(0.672779422278)*x[0]**o+(-0.936102224964)*x[0]+(-0.763620352857)*x[1]**o+(0.265802509744)*x[1]+(0.275798875047)*x[2]**o+(0.235397520942)*x[2]
        arg[2,0,2,1]=(0.192866248041)*x[0]**o+(0.656863606616)*x[0]+(-0.902796948663)*x[1]**o+(0.692123089256)*x[1]+(0.311287987388)*x[2]**o+(0.555427507546)*x[2]
        arg[2,1,0,0]=(0.464585537593)*x[0]**o+(0.161744615208)*x[0]+(-0.325569920662)*x[1]**o+(-0.47142644441)*x[1]+(0.885151021784)*x[2]**o+(-0.840595219086)*x[2]
        arg[2,1,0,1]=(-0.659253397488)*x[0]**o+(0.54391314862)*x[0]+(-0.595612529952)*x[1]**o+(-0.0472758114379)*x[1]+(0.876391930188)*x[2]**o+(-0.440313199946)*x[2]
        arg[2,1,1,0]=(-0.445591423364)*x[0]**o+(0.179144841652)*x[0]+(-0.294166502086)*x[1]**o+(-0.373820778739)*x[1]+(0.464609314863)*x[2]**o+(0.407893781978)*x[2]
        arg[2,1,1,1]=(-0.767512700255)*x[0]**o+(0.63198312168)*x[0]+(0.652318598902)*x[1]**o+(0.817874971823)*x[1]+(0.397211574885)*x[2]**o+(0.33174354886)*x[2]
        arg[2,1,2,0]=(0.473801282131)*x[0]**o+(-0.778895701394)*x[0]+(-0.801488721847)*x[1]**o+(-0.585840161723)*x[1]+(-0.805496721223)*x[2]**o+(0.585971840421)*x[2]
        arg[2,1,2,1]=(0.472329620891)*x[0]**o+(-0.0473805663117)*x[0]+(0.0894117727522)*x[1]**o+(-0.308473625733)*x[1]+(-0.658376953956)*x[2]**o+(0.32027592346)*x[2]
        arg[2,2,0,0]=(0.843241796369)*x[0]**o+(0.346857959798)*x[0]+(0.0111223078269)*x[1]**o+(-0.949623227267)*x[1]+(-0.152362393376)*x[2]**o+(-0.886340001141)*x[2]
        arg[2,2,0,1]=(-0.555393018762)*x[0]**o+(0.666364344488)*x[0]+(0.181237902671)*x[1]**o+(0.670959040345)*x[1]+(0.57383285317)*x[2]**o+(-0.451027832775)*x[2]
        arg[2,2,1,0]=(0.249709446431)*x[0]**o+(0.568753063379)*x[0]+(-0.812386750097)*x[1]**o+(0.506805124672)*x[1]+(-0.454844217662)*x[2]**o+(0.843435269288)*x[2]
        arg[2,2,1,1]=(0.753144930025)*x[0]**o+(0.0181642998269)*x[0]+(0.622635241593)*x[1]**o+(-0.372883842874)*x[1]+(0.366920965577)*x[2]**o+(-0.092391524816)*x[2]
        arg[2,2,2,0]=(0.309601906611)*x[0]**o+(-0.502573192965)*x[0]+(-0.420795137333)*x[1]**o+(-0.807109356307)*x[1]+(-0.802887583918)*x[2]**o+(0.552609684817)*x[2]
        arg[2,2,2,1]=(0.320933223342)*x[0]**o+(0.541931544704)*x[0]+(0.988715316931)*x[1]**o+(-0.4197325237)*x[1]+(-0.438705512435)*x[2]**o+(-0.540175093599)*x[2]
        arg[2,3,0,0]=(0.018870015518)*x[0]**o+(-0.615120691773)*x[0]+(0.596295280542)*x[1]**o+(0.210874310884)*x[1]+(-0.652194502321)*x[2]**o+(0.74019575754)*x[2]
        arg[2,3,0,1]=(0.820285141622)*x[0]**o+(0.130985741886)*x[0]+(-0.397387767472)*x[1]**o+(0.975103051154)*x[1]+(0.0887369214611)*x[2]**o+(0.88498731668)*x[2]
        arg[2,3,1,0]=(0.160503336808)*x[0]**o+(0.641180113144)*x[0]+(-0.408109675737)*x[1]**o+(-0.355443951133)*x[1]+(-0.429622016495)*x[2]**o+(0.998729833779)*x[2]
        arg[2,3,1,1]=(-0.646524352112)*x[0]**o+(-0.289894880059)*x[0]+(0.909761879864)*x[1]**o+(0.30453651619)*x[1]+(0.884596709091)*x[2]**o+(0.329408737003)*x[2]
        arg[2,3,2,0]=(0.459385919432)*x[0]**o+(0.416447271916)*x[0]+(-0.400798659819)*x[1]**o+(-0.769043818187)*x[1]+(-0.631682400803)*x[2]**o+(-0.80150361084)*x[2]
        arg[2,3,2,1]=(-0.544824298969)*x[0]**o+(0.191301965562)*x[0]+(0.41061102918)*x[1]**o+(-0.999195759064)*x[1]+(-0.0226105975802)*x[2]**o+(-0.137078691725)*x[2]
        arg[2,4,0,0]=(-0.624033952375)*x[0]**o+(0.0792222332007)*x[0]+(0.766756718455)*x[1]**o+(0.581487379511)*x[1]+(-0.437940391245)*x[2]**o+(0.115617341492)*x[2]
        arg[2,4,0,1]=(-0.300526068838)*x[0]**o+(-0.822844066866)*x[0]+(0.660949310367)*x[1]**o+(-0.0993949179102)*x[1]+(0.599884615655)*x[2]**o+(-0.903177033523)*x[2]
        arg[2,4,1,0]=(-0.654642517091)*x[0]**o+(0.428391621765)*x[0]+(0.932377441464)*x[1]**o+(-0.459525189684)*x[1]+(0.667425199675)*x[2]**o+(-0.937213410686)*x[2]
        arg[2,4,1,1]=(0.258067525957)*x[0]**o+(0.619943768383)*x[0]+(0.458217567511)*x[1]**o+(0.0812191081554)*x[1]+(0.743260637332)*x[2]**o+(-0.796150472293)*x[2]
        arg[2,4,2,0]=(0.782473636556)*x[0]**o+(-0.441405613305)*x[0]+(0.3951595327)*x[1]**o+(-0.449855781187)*x[1]+(-0.964130179212)*x[2]**o+(-0.299199410525)*x[2]
        arg[2,4,2,1]=(-0.878340584397)*x[0]**o+(-0.674575785429)*x[0]+(0.838137461257)*x[1]**o+(-0.498710475715)*x[1]+(-0.117205127994)*x[2]**o+(0.0934669704542)*x[2]
        arg[3,0,0,0]=(0.139631366137)*x[0]**o+(-0.893001605238)*x[0]+(-0.420780773096)*x[1]**o+(0.439055863393)*x[1]+(0.173113363237)*x[2]**o+(0.893867443017)*x[2]
        arg[3,0,0,1]=(-0.804823042124)*x[0]**o+(0.250231181298)*x[0]+(0.358464048439)*x[1]**o+(-0.847391231172)*x[1]+(-0.515391975323)*x[2]**o+(-0.999640302937)*x[2]
        arg[3,0,1,0]=(0.328676968943)*x[0]**o+(-0.0521242139559)*x[0]+(0.861548411537)*x[1]**o+(-0.288979961428)*x[1]+(-0.927901849987)*x[2]**o+(0.857869958971)*x[2]
        arg[3,0,1,1]=(-0.654866876537)*x[0]**o+(-0.0515462321916)*x[0]+(-0.224164685097)*x[1]**o+(-0.84207399757)*x[1]+(-0.442392040219)*x[2]**o+(-0.915734076984)*x[2]
        arg[3,0,2,0]=(-0.322517406382)*x[0]**o+(-0.757146499161)*x[0]+(-0.541690197465)*x[1]**o+(0.931090695413)*x[1]+(-0.355505263744)*x[2]**o+(-0.773185404514)*x[2]
        arg[3,0,2,1]=(0.421030649833)*x[0]**o+(-0.83768939688)*x[0]+(-0.863791071447)*x[1]**o+(0.78393205545)*x[1]+(-0.489893911681)*x[2]**o+(-0.291989245709)*x[2]
        arg[3,1,0,0]=(0.550839051007)*x[0]**o+(-0.883853789938)*x[0]+(-0.339529971923)*x[1]**o+(-0.756936847344)*x[1]+(0.724888858656)*x[2]**o+(-0.981057762224)*x[2]
        arg[3,1,0,1]=(0.711792743156)*x[0]**o+(0.233248429378)*x[0]+(0.526076154408)*x[1]**o+(0.349364804166)*x[1]+(0.271475992486)*x[2]**o+(0.782429402713)*x[2]
        arg[3,1,1,0]=(-0.499072848146)*x[0]**o+(-0.0604888692308)*x[0]+(0.545442893272)*x[1]**o+(-0.688797522515)*x[1]+(0.305341295222)*x[2]**o+(0.130010601331)*x[2]
        arg[3,1,1,1]=(0.940695359452)*x[0]**o+(0.113015472294)*x[0]+(-0.506254395321)*x[1]**o+(-0.252044277039)*x[1]+(0.0172784972094)*x[2]**o+(-0.184256978193)*x[2]
        arg[3,1,2,0]=(0.550472143981)*x[0]**o+(-0.729809590776)*x[0]+(-0.293765999495)*x[1]**o+(-0.378042976721)*x[1]+(0.973242167993)*x[2]**o+(-0.75350391134)*x[2]
        arg[3,1,2,1]=(0.369266178598)*x[0]**o+(-0.764034492366)*x[0]+(-0.713325992328)*x[1]**o+(0.948932375717)*x[1]+(-0.335074977323)*x[2]**o+(0.117829591745)*x[2]
        arg[3,2,0,0]=(0.426458076699)*x[0]**o+(-0.520551329627)*x[0]+(-0.773464312884)*x[1]**o+(0.26332468989)*x[1]+(0.143497828773)*x[2]**o+(0.0676278275986)*x[2]
        arg[3,2,0,1]=(0.85806016531)*x[0]**o+(0.755361032731)*x[0]+(0.403942188112)*x[1]**o+(0.0113665816202)*x[1]+(-0.439835633198)*x[2]**o+(0.620930145183)*x[2]
        arg[3,2,1,0]=(-0.852838624821)*x[0]**o+(-0.918594624263)*x[0]+(-0.572898555139)*x[1]**o+(-0.14353390918)*x[1]+(-0.716010071307)*x[2]**o+(0.134687635305)*x[2]
        arg[3,2,1,1]=(-0.14282590746)*x[0]**o+(-0.169314199229)*x[0]+(0.880745511071)*x[1]**o+(0.540733710441)*x[1]+(-0.131424658303)*x[2]**o+(-0.93209431119)*x[2]
        arg[3,2,2,0]=(-0.864016075688)*x[0]**o+(-0.265794760131)*x[0]+(0.444625188371)*x[1]**o+(0.133693703972)*x[1]+(-0.749799597936)*x[2]**o+(0.886355820615)*x[2]
        arg[3,2,2,1]=(0.676281241167)*x[0]**o+(-0.351800298325)*x[0]+(-0.751498380099)*x[1]**o+(-0.561737786104)*x[1]+(-0.984872672853)*x[2]**o+(0.476840947707)*x[2]
        arg[3,3,0,0]=(0.457874366022)*x[0]**o+(-0.126929649966)*x[0]+(0.666077559139)*x[1]**o+(-0.966437669224)*x[1]+(-0.849000502816)*x[2]**o+(0.990790747926)*x[2]
        arg[3,3,0,1]=(0.45136859326)*x[0]**o+(0.771355456911)*x[0]+(-0.678558472676)*x[1]**o+(0.0366869703264)*x[1]+(-0.257247358952)*x[2]**o+(-0.290852285538)*x[2]
        arg[3,3,1,0]=(-0.623007069616)*x[0]**o+(-0.365255080044)*x[0]+(0.441927420573)*x[1]**o+(0.331640625664)*x[1]+(-0.21153951456)*x[2]**o+(0.297967669123)*x[2]
        arg[3,3,1,1]=(-0.836057742118)*x[0]**o+(0.959873293328)*x[0]+(-0.52034405502)*x[1]**o+(-0.178838686591)*x[1]+(0.317610563306)*x[2]**o+(-0.555134650833)*x[2]
        arg[3,3,2,0]=(0.0819108022024)*x[0]**o+(0.69432296821)*x[0]+(-0.23685434367)*x[1]**o+(-0.0442137744391)*x[1]+(-0.879175086057)*x[2]**o+(-0.28031950012)*x[2]
        arg[3,3,2,1]=(0.340083286763)*x[0]**o+(-0.381942838645)*x[0]+(0.570579443852)*x[1]**o+(0.772192708863)*x[1]+(0.42813556157)*x[2]**o+(0.735683065499)*x[2]
        arg[3,4,0,0]=(-0.866813692418)*x[0]**o+(-0.53715159706)*x[0]+(0.0725900211922)*x[1]**o+(0.0891167848614)*x[1]+(-0.381482922347)*x[2]**o+(-0.521504123034)*x[2]
        arg[3,4,0,1]=(0.487750922723)*x[0]**o+(0.967820473849)*x[0]+(0.305600886352)*x[1]**o+(0.350777657643)*x[1]+(-0.383731485788)*x[2]**o+(0.814016126476)*x[2]
        arg[3,4,1,0]=(-0.186663244344)*x[0]**o+(0.144872424379)*x[0]+(0.844454339304)*x[1]**o+(0.252133708483)*x[1]+(-0.748235508222)*x[2]**o+(0.702952185343)*x[2]
        arg[3,4,1,1]=(-0.962296047927)*x[0]**o+(0.874277032826)*x[0]+(-0.404689253219)*x[1]**o+(-0.174075799004)*x[1]+(0.65086350106)*x[2]**o+(-0.109850650677)*x[2]
        arg[3,4,2,0]=(-0.59515836027)*x[0]**o+(-0.0917296352911)*x[0]+(0.834779238353)*x[1]**o+(0.951252871777)*x[1]+(0.983008684497)*x[2]**o+(-0.784806954891)*x[2]
        arg[3,4,2,1]=(0.603110971708)*x[0]**o+(0.291507042984)*x[0]+(-0.401523286981)*x[1]**o+(-0.994760476862)*x[1]+(0.410516007475)*x[2]**o+(0.368982230001)*x[2]
        ref[0,0,0,0]=(-0.322384218211)/(o+1.)+(0.436168366323)
        ref[0,0,0,1]=(-0.550432608405)/(o+1.)+(-0.839529924682)
        ref[0,0,1,0]=(-1.20222931022)/(o+1.)+(-0.674931792543)
        ref[0,0,1,1]=(-1.99447702496)/(o+1.)+(0.184076712997)
        ref[0,0,2,0]=(-0.848786191606)/(o+1.)+(-0.719076228559)
        ref[0,0,2,1]=(0.488199570726)/(o+1.)+(0.818508727159)
        ref[0,1,0,0]=(0.910006770156)/(o+1.)+(-0.781077934994)
        ref[0,1,0,1]=(0.229140786917)/(o+1.)+(0.225201244712)
        ref[0,1,1,0]=(-1.24973037868)/(o+1.)+(0.478630842756)
        ref[0,1,1,1]=(0.362813222831)/(o+1.)+(-0.269799066354)
        ref[0,1,2,0]=(0.704186070044)/(o+1.)+(-0.0666602572428)
        ref[0,1,2,1]=(-0.429870561459)/(o+1.)+(-0.692214543685)
        ref[0,2,0,0]=(-1.76040621668)/(o+1.)+(-0.103077097544)
        ref[0,2,0,1]=(-0.578245990054)/(o+1.)+(-0.24050068432)
        ref[0,2,1,0]=(0.982824772144)/(o+1.)+(-0.332261654225)
        ref[0,2,1,1]=(0.493111487615)/(o+1.)+(0.105743857349)
        ref[0,2,2,0]=(-0.406637376689)/(o+1.)+(0.910652270393)
        ref[0,2,2,1]=(0.0735211634779)/(o+1.)+(0.355106669226)
        ref[0,3,0,0]=(-0.538705798782)/(o+1.)+(0.896367101094)
        ref[0,3,0,1]=(0.11388337928)/(o+1.)+(0.766078903586)
        ref[0,3,1,0]=(0.955296156262)/(o+1.)+(-0.64915883862)
        ref[0,3,1,1]=(0.437273793256)/(o+1.)+(-0.61952690992)
        ref[0,3,2,0]=(1.07390497644)/(o+1.)+(-0.242730112953)
        ref[0,3,2,1]=(-0.79053815352)/(o+1.)+(-0.0521715653697)
        ref[0,4,0,0]=(-0.0103557372704)/(o+1.)+(0.939327713906)
        ref[0,4,0,1]=(-0.66740052552)/(o+1.)+(-0.671249919496)
        ref[0,4,1,0]=(-1.10797150302)/(o+1.)+(0.454490777873)
        ref[0,4,1,1]=(-0.00636792544909)/(o+1.)+(-0.368286670996)
        ref[0,4,2,0]=(-1.23874065284)/(o+1.)+(0.0410396874718)
        ref[0,4,2,1]=(0.834902700591)/(o+1.)+(0.562757794439)
        ref[1,0,0,0]=(1.46682450791)/(o+1.)+(0.891962057632)
        ref[1,0,0,1]=(0.183329431084)/(o+1.)+(0.659083249781)
        ref[1,0,1,0]=(-0.24909411204)/(o+1.)+(0.201301697863)
        ref[1,0,1,1]=(0.723423340723)/(o+1.)+(-0.205765407152)
        ref[1,0,2,0]=(0.607815389082)/(o+1.)+(-0.451934578027)
        ref[1,0,2,1]=(0.919070217555)/(o+1.)+(1.00996326691)
        ref[1,1,0,0]=(0.361804510188)/(o+1.)+(-0.272105134113)
        ref[1,1,0,1]=(-0.388145380449)/(o+1.)+(0.440667992719)
        ref[1,1,1,0]=(1.11229281554)/(o+1.)+(0.0358417840478)
        ref[1,1,1,1]=(-0.360763258547)/(o+1.)+(-0.0662175232282)
        ref[1,1,2,0]=(-0.421932179206)/(o+1.)+(1.04809456598)
        ref[1,1,2,1]=(1.5138402229)/(o+1.)+(-0.0182788239862)
        ref[1,2,0,0]=(-1.2439886311)/(o+1.)+(0.204333673483)
        ref[1,2,0,1]=(1.36866295286)/(o+1.)+(0.510865902853)
        ref[1,2,1,0]=(0.636378052915)/(o+1.)+(-0.439317277801)
        ref[1,2,1,1]=(2.1721593538)/(o+1.)+(0.63055844212)
        ref[1,2,2,0]=(-0.00984028813171)/(o+1.)+(-0.390721816834)
        ref[1,2,2,1]=(1.06592059421)/(o+1.)+(0.303699000873)
        ref[1,3,0,0]=(-0.160594832134)/(o+1.)+(-0.271446552947)
        ref[1,3,0,1]=(0.313068732589)/(o+1.)+(-0.457202820505)
        ref[1,3,1,0]=(1.67446584033)/(o+1.)+(0.718634957026)
        ref[1,3,1,1]=(-0.399207614903)/(o+1.)+(0.379686935632)
        ref[1,3,2,0]=(0.947060327978)/(o+1.)+(-0.506844686357)
        ref[1,3,2,1]=(2.40080863254)/(o+1.)+(-0.413347335508)
        ref[1,4,0,0]=(-1.08603586849)/(o+1.)+(0.903964428263)
        ref[1,4,0,1]=(-0.471025636339)/(o+1.)+(0.657506547153)
        ref[1,4,1,0]=(1.0787768427)/(o+1.)+(0.428710904834)
        ref[1,4,1,1]=(0.909279847682)/(o+1.)+(-0.380313094604)
        ref[1,4,2,0]=(-0.827086892123)/(o+1.)+(0.961519048079)
        ref[1,4,2,1]=(-0.154619783556)/(o+1.)+(-0.0591580335741)
        ref[2,0,0,0]=(-0.302649323542)/(o+1.)+(-0.229657108834)
        ref[2,0,0,1]=(1.77170468326)/(o+1.)+(-0.0484045064764)
        ref[2,0,1,0]=(-1.27113695197)/(o+1.)+(-0.94026328862)
        ref[2,0,1,1]=(-0.0863320624027)/(o+1.)+(0.383317392539)
        ref[2,0,2,0]=(0.184957944467)/(o+1.)+(-0.217451097139)
        ref[2,0,2,1]=(-0.398642713235)/(o+1.)+(0.952207101709)
        ref[2,1,0,0]=(1.02416663872)/(o+1.)+(-0.575138524144)
        ref[2,1,0,1]=(-0.378473997252)/(o+1.)+(0.0281620686178)
        ref[2,1,1,0]=(-0.275148610588)/(o+1.)+(0.106608922446)
        ref[2,1,1,1]=(0.282017473532)/(o+1.)+(0.890800821181)
        ref[2,1,2,0]=(-1.13318416094)/(o+1.)+(-0.389382011348)
        ref[2,1,2,1]=(-0.0966355603128)/(o+1.)+(-0.0177891342927)
        ref[2,2,0,0]=(0.70200171082)/(o+1.)+(-0.744552634305)
        ref[2,2,0,1]=(0.199677737079)/(o+1.)+(0.443147776029)
        ref[2,2,1,0]=(-1.01752152133)/(o+1.)+(0.959496728669)
        ref[2,2,1,1]=(1.74270113719)/(o+1.)+(-0.223555533932)
        ref[2,2,2,0]=(-0.91408081464)/(o+1.)+(-0.378536432227)
        ref[2,2,2,1]=(0.870943027839)/(o+1.)+(-0.208988036298)
        ref[2,3,0,0]=(-0.0370292062615)/(o+1.)+(0.167974688326)
        ref[2,3,0,1]=(0.51163429561)/(o+1.)+(0.99553805486)
        ref[2,3,1,0]=(-0.677228355424)/(o+1.)+(0.642232997895)
        ref[2,3,1,1]=(1.14783423684)/(o+1.)+(0.172025186567)
        ref[2,3,2,0]=(-0.57309514119)/(o+1.)+(-0.577050078555)
        ref[2,3,2,1]=(-0.156823867369)/(o+1.)+(-0.472486242613)
        ref[2,4,0,0]=(-0.295217625165)/(o+1.)+(0.388163477102)
        ref[2,4,0,1]=(0.960307857184)/(o+1.)+(-0.91270800915)
        ref[2,4,1,0]=(0.945160124047)/(o+1.)+(-0.484173489303)
        ref[2,4,1,1]=(1.4595457308)/(o+1.)+(-0.0474937978771)
        ref[2,4,2,0]=(0.213502990044)/(o+1.)+(-0.595230402509)
        ref[2,4,2,1]=(-0.157408251134)/(o+1.)+(-0.539909645345)
        ref[3,0,0,0]=(-0.108036043722)/(o+1.)+(0.219960850586)
        ref[3,0,0,1]=(-0.961750969008)/(o+1.)+(-0.798400176406)
        ref[3,0,1,0]=(0.262323530493)/(o+1.)+(0.258382891794)
        ref[3,0,1,1]=(-1.32142360185)/(o+1.)+(-0.904677153373)
        ref[3,0,2,0]=(-1.21971286759)/(o+1.)+(-0.299620604131)
        ref[3,0,2,1]=(-0.932654333294)/(o+1.)+(-0.17287329357)
        ref[3,1,0,0]=(0.93619793774)/(o+1.)+(-1.31092419975)
        ref[3,1,0,1]=(1.50934489005)/(o+1.)+(0.682521318129)
        ref[3,1,1,0]=(0.351711340348)/(o+1.)+(-0.309637895207)
        ref[3,1,1,1]=(0.451719461341)/(o+1.)+(-0.161642891469)
        ref[3,1,2,0]=(1.22994831248)/(o+1.)+(-0.930678239419)
        ref[3,1,2,1]=(-0.679134791053)/(o+1.)+(0.151363737548)
        ref[3,2,0,0]=(-0.203508407413)/(o+1.)+(-0.0947994060691)
        ref[3,2,0,1]=(0.822166720224)/(o+1.)+(0.693828879767)
        ref[3,2,1,0]=(-2.14174725127)/(o+1.)+(-0.463720449069)
        ref[3,2,1,1]=(0.606494945308)/(o+1.)+(-0.280337399989)
        ref[3,2,2,0]=(-1.16919048525)/(o+1.)+(0.377127382228)
        ref[3,2,2,1]=(-1.06008981178)/(o+1.)+(-0.218348568361)
        ref[3,3,0,0]=(0.274951422345)/(o+1.)+(-0.0512882856318)
        ref[3,3,0,1]=(-0.484437238367)/(o+1.)+(0.25859507085)
        ref[3,3,1,0]=(-0.392619163603)/(o+1.)+(0.132176607371)
        ref[3,3,1,1]=(-1.03879123383)/(o+1.)+(0.112949977952)
        ref[3,3,2,0]=(-1.03411862752)/(o+1.)+(0.184894846825)
        ref[3,3,2,1]=(1.33879829218)/(o+1.)+(0.562966467859)
        ref[3,4,0,0]=(-1.17570659357)/(o+1.)+(-0.484769467616)
        ref[3,4,0,1]=(0.409620323287)/(o+1.)+(1.06630712898)
        ref[3,4,1,0]=(-0.0904444132627)/(o+1.)+(0.549979159103)
        ref[3,4,1,1]=(-0.716121800087)/(o+1.)+(0.295175291572)
        ref[3,4,2,0]=(1.22262956258)/(o+1.)+(0.0373581407971)
        ref[3,4,2,1]=(0.612103692201)/(o+1.)+(-0.167135601938)
      res=integrate(arg)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5, 3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunction_fromData_ContinuousFunction_rank0(self):
      """
      tests integral of rank 0 Data on the ReducedFunction

      assumptions: ContinuousFunction(self.domain) exists
                   self.domain supports integral on ReducedFunction
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=ReducedFunction(self.domain)
      w=ContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(),w)
      if dim==2:
        arg=(0.465976690369)*x[0]**o+(-0.25136070196)*x[0]+(0.409576841566)*x[1]**o+(0.726707772754)*x[1]
        ref=(0.875553531935)/(o+1.)+(0.237673535397)
      else:
        arg=(0.567785895928)*x[0]**o+(0.928477128396)*x[0]+(-0.625720943088)*x[1]**o+(-0.681460841298)*x[1]+(-0.866801010069)*x[2]**o+(0.159594300637)*x[2]
        ref=(-0.924736057229)/(o+1.)+(0.203305293867)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunction_fromData_ContinuousFunction_rank1(self):
      """
      tests integral of rank 1 Data on the ReducedFunction

      assumptions: ContinuousFunction(self.domain) exists
                   self.domain supports integral on ReducedFunction
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=ReducedFunction(self.domain)
      w=ContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(2,),w)
      ref=numpy.zeros((2,),numpy.float_)
      if dim==2:
        arg[0]=(-0.839020088668)*x[0]**o+(-0.880288035177)*x[0]+(0.260951734364)*x[1]**o+(-0.624208000982)*x[1]
        arg[1]=(0.985421831314)*x[0]**o+(0.660399229451)*x[0]+(-0.514003848134)*x[1]**o+(0.0312162082593)*x[1]
        ref[0]=(-0.578068354304)/(o+1.)+(-0.75224801808)
        ref[1]=(0.47141798318)/(o+1.)+(0.345807718855)
      else:
        arg[0]=(-0.627307316929)*x[0]**o+(0.500432411035)*x[0]+(0.233895908313)*x[1]**o+(-0.125048135432)*x[1]+(0.0695114009632)*x[2]**o+(0.995170977074)*x[2]
        arg[1]=(0.760033715785)*x[0]**o+(0.48895905139)*x[0]+(0.240372023176)*x[1]**o+(0.33946689896)*x[1]+(-0.774638489419)*x[2]**o+(-0.0777240871074)*x[2]
        ref[0]=(-0.323900007653)/(o+1.)+(0.685277626338)
        ref[1]=(0.225767249542)/(o+1.)+(0.375350931622)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunction_fromData_ContinuousFunction_rank2(self):
      """
      tests integral of rank 2 Data on the ReducedFunction

      assumptions: ContinuousFunction(self.domain) exists
                   self.domain supports integral on ReducedFunction
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=ReducedFunction(self.domain)
      w=ContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5),w)
      ref=numpy.zeros((4, 5),numpy.float_)
      if dim==2:
        arg[0,0]=(-0.844492048883)*x[0]**o+(0.727132679659)*x[0]+(-0.862576284722)*x[1]**o+(0.502937561417)*x[1]
        arg[0,1]=(-0.587157419309)*x[0]**o+(0.00327068097107)*x[0]+(0.422592349683)*x[1]**o+(0.167479615369)*x[1]
        arg[0,2]=(0.771156665672)*x[0]**o+(-0.371091138833)*x[0]+(-0.867750378658)*x[1]**o+(-0.744394290185)*x[1]
        arg[0,3]=(0.669718598855)*x[0]**o+(-0.0825909409121)*x[0]+(0.868872948791)*x[1]**o+(0.141891982204)*x[1]
        arg[0,4]=(-0.441072764653)*x[0]**o+(-0.929654649265)*x[0]+(-0.35883498816)*x[1]**o+(-0.84045480381)*x[1]
        arg[1,0]=(0.555668196622)*x[0]**o+(0.94306681043)*x[0]+(0.586512446245)*x[1]**o+(0.702007276327)*x[1]
        arg[1,1]=(0.792083771005)*x[0]**o+(-0.865830139617)*x[0]+(0.0929971631615)*x[1]**o+(0.620446289121)*x[1]
        arg[1,2]=(-0.201978375472)*x[0]**o+(-0.0777206885918)*x[0]+(0.798755459687)*x[1]**o+(-0.278086468846)*x[1]
        arg[1,3]=(0.063028419572)*x[0]**o+(-0.454786232294)*x[0]+(-0.0481393324293)*x[1]**o+(-0.612657132834)*x[1]
        arg[1,4]=(-0.743283787964)*x[0]**o+(0.854827303846)*x[0]+(0.94241635825)*x[1]**o+(0.436937526521)*x[1]
        arg[2,0]=(-0.814975695669)*x[0]**o+(0.579477783769)*x[0]+(0.00998264005965)*x[1]**o+(-0.913665878657)*x[1]
        arg[2,1]=(-0.526833976369)*x[0]**o+(-0.800386086232)*x[0]+(-0.487600053086)*x[1]**o+(0.00242562519003)*x[1]
        arg[2,2]=(-0.964119591874)*x[0]**o+(0.203212394214)*x[0]+(-0.96850837954)*x[1]**o+(-0.214886132512)*x[1]
        arg[2,3]=(-0.356249104564)*x[0]**o+(0.329769293221)*x[0]+(0.936664928754)*x[1]**o+(0.520603503283)*x[1]
        arg[2,4]=(0.255464219424)*x[0]**o+(-0.157783294516)*x[0]+(0.519451787865)*x[1]**o+(0.00960521397376)*x[1]
        arg[3,0]=(-0.117250486034)*x[0]**o+(0.192945431273)*x[0]+(-0.7580154297)*x[1]**o+(-0.308801650294)*x[1]
        arg[3,1]=(0.243931002845)*x[0]**o+(-0.626714191745)*x[0]+(-0.365870571265)*x[1]**o+(0.652257582021)*x[1]
        arg[3,2]=(-0.769756656122)*x[0]**o+(0.711106390017)*x[0]+(0.150749630653)*x[1]**o+(-0.0441309732448)*x[1]
        arg[3,3]=(0.92677837493)*x[0]**o+(0.507199859313)*x[0]+(-0.542275490828)*x[1]**o+(-0.0907270087228)*x[1]
        arg[3,4]=(-0.0306175885885)*x[0]**o+(-0.434666505018)*x[0]+(-0.815633427562)*x[1]**o+(-0.916424797903)*x[1]
        ref[0,0]=(-1.70706833361)/(o+1.)+(0.615035120538)
        ref[0,1]=(-0.164565069626)/(o+1.)+(0.08537514817)
        ref[0,2]=(-0.0965937129851)/(o+1.)+(-0.557742714509)
        ref[0,3]=(1.53859154765)/(o+1.)+(0.0296505206459)
        ref[0,4]=(-0.799907752814)/(o+1.)+(-0.885054726537)
        ref[1,0]=(1.14218064287)/(o+1.)+(0.822537043379)
        ref[1,1]=(0.885080934167)/(o+1.)+(-0.122691925248)
        ref[1,2]=(0.596777084215)/(o+1.)+(-0.177903578719)
        ref[1,3]=(0.0148890871427)/(o+1.)+(-0.533721682564)
        ref[1,4]=(0.199132570285)/(o+1.)+(0.645882415184)
        ref[2,0]=(-0.80499305561)/(o+1.)+(-0.167094047444)
        ref[2,1]=(-1.01443402945)/(o+1.)+(-0.398980230521)
        ref[2,2]=(-1.93262797141)/(o+1.)+(-0.00583686914904)
        ref[2,3]=(0.580415824189)/(o+1.)+(0.425186398252)
        ref[2,4]=(0.774916007289)/(o+1.)+(-0.074089040271)
        ref[3,0]=(-0.875265915734)/(o+1.)+(-0.0579281095106)
        ref[3,1]=(-0.12193956842)/(o+1.)+(0.0127716951382)
        ref[3,2]=(-0.619007025469)/(o+1.)+(0.333487708386)
        ref[3,3]=(0.384502884102)/(o+1.)+(0.208236425295)
        ref[3,4]=(-0.846251016151)/(o+1.)+(-0.675545651461)
      else:
        arg[0,0]=(0.596538443259)*x[0]**o+(0.479462980373)*x[0]+(-0.334469126403)*x[1]**o+(0.784201085556)*x[1]+(-0.231474140531)*x[2]**o+(-0.368096345589)*x[2]
        arg[0,1]=(0.188945383338)*x[0]**o+(-0.95033024122)*x[0]+(0.567904901047)*x[1]**o+(-0.76703296351)*x[1]+(-0.0116083690376)*x[2]**o+(-0.211176589764)*x[2]
        arg[0,2]=(0.225412552449)*x[0]**o+(-0.731615155924)*x[0]+(-0.362727501637)*x[1]**o+(-0.195442119344)*x[1]+(-0.215444456285)*x[2]**o+(0.412715796392)*x[2]
        arg[0,3]=(-0.00941287481408)*x[0]**o+(-0.207271678463)*x[0]+(0.66582433583)*x[1]**o+(-0.297871484308)*x[1]+(-0.819460465536)*x[2]**o+(-0.3316307033)*x[2]
        arg[0,4]=(-0.172265683581)*x[0]**o+(0.881658164973)*x[0]+(0.0267816673708)*x[1]**o+(-0.526757427521)*x[1]+(-0.666953449252)*x[2]**o+(0.366533690002)*x[2]
        arg[1,0]=(-0.575272180021)*x[0]**o+(-0.707236339163)*x[0]+(-0.734723292219)*x[1]**o+(-0.872019568084)*x[1]+(-0.513269190461)*x[2]**o+(-0.95077853535)*x[2]
        arg[1,1]=(0.687652562435)*x[0]**o+(0.500668143461)*x[0]+(0.127678854367)*x[1]**o+(0.135196530391)*x[1]+(0.219757431874)*x[2]**o+(0.43373842168)*x[2]
        arg[1,2]=(-0.674120744855)*x[0]**o+(-0.792769889708)*x[0]+(0.0414602844391)*x[1]**o+(-0.159887164563)*x[1]+(0.406441555189)*x[2]**o+(-0.662210077269)*x[2]
        arg[1,3]=(-0.623120612358)*x[0]**o+(0.895951757448)*x[0]+(0.957407224559)*x[1]**o+(-0.472225118848)*x[1]+(-0.294551998847)*x[2]**o+(0.503584320886)*x[2]
        arg[1,4]=(-0.304220645545)*x[0]**o+(0.812321371589)*x[0]+(0.932257196386)*x[1]**o+(0.736014913915)*x[1]+(0.119140391025)*x[2]**o+(0.910696940879)*x[2]
        arg[2,0]=(0.275470013797)*x[0]**o+(-0.903754203548)*x[0]+(-0.443211232771)*x[1]**o+(0.78815938507)*x[1]+(-0.0802934106722)*x[2]**o+(0.096532189886)*x[2]
        arg[2,1]=(-0.4505392488)*x[0]**o+(-0.668542586506)*x[0]+(0.487967999984)*x[1]**o+(-0.642310745404)*x[1]+(-0.956530771836)*x[2]**o+(0.517544404873)*x[2]
        arg[2,2]=(-0.984032885692)*x[0]**o+(-0.606688527142)*x[0]+(0.0565483776156)*x[1]**o+(-0.945228520749)*x[1]+(0.433070187921)*x[2]**o+(-0.599983370996)*x[2]
        arg[2,3]=(-0.673469008741)*x[0]**o+(0.612144012035)*x[0]+(-0.169497339443)*x[1]**o+(0.267990057152)*x[1]+(0.431716413548)*x[2]**o+(0.188904436012)*x[2]
        arg[2,4]=(0.320782309844)*x[0]**o+(-0.830068082984)*x[0]+(0.900183436056)*x[1]**o+(-0.343951186189)*x[1]+(-0.00137135137854)*x[2]**o+(-0.655764522249)*x[2]
        arg[3,0]=(-0.936193981767)*x[0]**o+(0.885260284217)*x[0]+(0.00397725971098)*x[1]**o+(-0.0864751234501)*x[1]+(0.25001166345)*x[2]**o+(-0.127859505667)*x[2]
        arg[3,1]=(-0.0841986843231)*x[0]**o+(-0.95678469673)*x[0]+(0.867049114883)*x[1]**o+(0.154499266839)*x[1]+(0.0217146474622)*x[2]**o+(-0.945774319527)*x[2]
        arg[3,2]=(-0.668161122306)*x[0]**o+(0.779140555525)*x[0]+(0.217613826157)*x[1]**o+(0.900077039607)*x[1]+(-0.251881965789)*x[2]**o+(-0.331281723354)*x[2]
        arg[3,3]=(-0.95624079226)*x[0]**o+(-0.403226291225)*x[0]+(0.303340462877)*x[1]**o+(-0.269819286355)*x[1]+(0.605804508829)*x[2]**o+(-0.461404715818)*x[2]
        arg[3,4]=(-0.933059329973)*x[0]**o+(-0.00851461368762)*x[0]+(0.0588544443918)*x[1]**o+(-0.682939702389)*x[1]+(-0.213965742261)*x[2]**o+(-0.102965977964)*x[2]
        ref[0,0]=(0.0305951763245)/(o+1.)+(0.44778386017)
        ref[0,1]=(0.745241915348)/(o+1.)+(-0.964269897247)
        ref[0,2]=(-0.352759405473)/(o+1.)+(-0.257170739438)
        ref[0,3]=(-0.163049004521)/(o+1.)+(-0.418386933036)
        ref[0,4]=(-0.812437465462)/(o+1.)+(0.360717213727)
        ref[1,0]=(-1.8232646627)/(o+1.)+(-1.2650172213)
        ref[1,1]=(1.03508884868)/(o+1.)+(0.534801547766)
        ref[1,2]=(-0.226218905227)/(o+1.)+(-0.80743356577)
        ref[1,3]=(0.0397346133545)/(o+1.)+(0.463655479743)
        ref[1,4]=(0.747176941866)/(o+1.)+(1.22951661319)
        ref[2,0]=(-0.248034629647)/(o+1.)+(-0.00953131429598)
        ref[2,1]=(-0.919102020652)/(o+1.)+(-0.396654463518)
        ref[2,2]=(-0.494414320156)/(o+1.)+(-1.07595020944)
        ref[2,3]=(-0.411249934637)/(o+1.)+(0.534519252599)
        ref[2,4]=(1.21959439452)/(o+1.)+(-0.914891895711)
        ref[3,0]=(-0.682205058606)/(o+1.)+(0.33546282755)
        ref[3,1]=(0.804565078022)/(o+1.)+(-0.874029874709)
        ref[3,2]=(-0.702429261938)/(o+1.)+(0.673967935889)
        ref[3,3]=(-0.0470958205546)/(o+1.)+(-0.567225146699)
        ref[3,4]=(-1.08817062784)/(o+1.)+(-0.39721014702)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunction_fromData_ContinuousFunction_rank3(self):
      """
      tests integral of rank 3 Data on the ReducedFunction

      assumptions: ContinuousFunction(self.domain) exists
                   self.domain supports integral on ReducedFunction
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=ReducedFunction(self.domain)
      w=ContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(6, 2, 2),w)
      ref=numpy.zeros((6, 2, 2),numpy.float_)
      if dim==2:
        arg[0,0,0]=(0.401537399178)*x[0]**o+(-0.868937065568)*x[0]+(0.356734884346)*x[1]**o+(0.0137485089422)*x[1]
        arg[0,0,1]=(0.705383698155)*x[0]**o+(-0.884988282352)*x[0]+(-0.979185994641)*x[1]**o+(0.929576942519)*x[1]
        arg[0,1,0]=(-0.292540086272)*x[0]**o+(-0.223230840915)*x[0]+(0.717291109855)*x[1]**o+(-0.826233591275)*x[1]
        arg[0,1,1]=(0.55739919643)*x[0]**o+(0.2571407205)*x[0]+(0.00434404142287)*x[1]**o+(0.200305144451)*x[1]
        arg[1,0,0]=(0.451841947443)*x[0]**o+(-0.576279050146)*x[0]+(-0.128142118055)*x[1]**o+(0.316406960557)*x[1]
        arg[1,0,1]=(-0.07174792925)*x[0]**o+(-0.392935435009)*x[0]+(-0.343535485172)*x[1]**o+(0.619131193274)*x[1]
        arg[1,1,0]=(-0.667031669611)*x[0]**o+(-0.184806094874)*x[0]+(-0.199375296384)*x[1]**o+(0.449351889815)*x[1]
        arg[1,1,1]=(-0.51924508404)*x[0]**o+(0.870508122509)*x[0]+(-0.274149433148)*x[1]**o+(-0.762367215962)*x[1]
        arg[2,0,0]=(-0.501546909709)*x[0]**o+(0.164122129343)*x[0]+(-0.601370828214)*x[1]**o+(0.888046948888)*x[1]
        arg[2,0,1]=(0.272472486899)*x[0]**o+(-0.958776296738)*x[0]+(-0.334676045163)*x[1]**o+(0.410943596779)*x[1]
        arg[2,1,0]=(0.363478201368)*x[0]**o+(0.753492598803)*x[0]+(-0.663726768274)*x[1]**o+(0.875443816963)*x[1]
        arg[2,1,1]=(0.995788536211)*x[0]**o+(-0.123932097555)*x[0]+(0.143025171293)*x[1]**o+(0.724918059214)*x[1]
        arg[3,0,0]=(-0.578717829755)*x[0]**o+(-0.899523186929)*x[0]+(0.58523203854)*x[1]**o+(0.751620714771)*x[1]
        arg[3,0,1]=(0.179841033744)*x[0]**o+(0.268015839688)*x[0]+(0.860292991989)*x[1]**o+(0.0117969420991)*x[1]
        arg[3,1,0]=(-0.575777000777)*x[0]**o+(0.170047012345)*x[0]+(-0.418743243278)*x[1]**o+(-0.278415849566)*x[1]
        arg[3,1,1]=(-0.913076795648)*x[0]**o+(0.486330450107)*x[0]+(-0.423570363453)*x[1]**o+(0.388575474946)*x[1]
        arg[4,0,0]=(0.16695368621)*x[0]**o+(0.133497460585)*x[0]+(-0.444434659151)*x[1]**o+(-0.895229636275)*x[1]
        arg[4,0,1]=(0.432018868106)*x[0]**o+(-0.281976075625)*x[0]+(-0.0455135209769)*x[1]**o+(0.76870037059)*x[1]
        arg[4,1,0]=(0.238542836112)*x[0]**o+(-0.806931884409)*x[0]+(-0.0365533871189)*x[1]**o+(0.731336634477)*x[1]
        arg[4,1,1]=(0.512035255059)*x[0]**o+(-0.683560978902)*x[0]+(-0.953782277737)*x[1]**o+(-0.311664210765)*x[1]
        arg[5,0,0]=(-0.842399299624)*x[0]**o+(0.460676816656)*x[0]+(-0.251781603841)*x[1]**o+(0.321792359074)*x[1]
        arg[5,0,1]=(-0.900576257062)*x[0]**o+(0.842393710528)*x[0]+(0.630445446843)*x[1]**o+(-0.647745144739)*x[1]
        arg[5,1,0]=(0.136687908652)*x[0]**o+(0.0252907890641)*x[0]+(0.813054611844)*x[1]**o+(-0.459930787056)*x[1]
        arg[5,1,1]=(-0.00242518550941)*x[0]**o+(-0.558785348137)*x[0]+(0.0568659980522)*x[1]**o+(0.689293614051)*x[1]
        ref[0,0,0]=(0.758272283524)/(o+1.)+(-0.427594278313)
        ref[0,0,1]=(-0.273802296487)/(o+1.)+(0.0222943300833)
        ref[0,1,0]=(0.424751023583)/(o+1.)+(-0.524732216095)
        ref[0,1,1]=(0.561743237852)/(o+1.)+(0.228722932476)
        ref[1,0,0]=(0.323699829388)/(o+1.)+(-0.129936044795)
        ref[1,0,1]=(-0.415283414421)/(o+1.)+(0.113097879133)
        ref[1,1,0]=(-0.866406965995)/(o+1.)+(0.132272897471)
        ref[1,1,1]=(-0.793394517189)/(o+1.)+(0.0540704532731)
        ref[2,0,0]=(-1.10291773792)/(o+1.)+(0.526084539115)
        ref[2,0,1]=(-0.0622035582637)/(o+1.)+(-0.273916349979)
        ref[2,1,0]=(-0.300248566906)/(o+1.)+(0.814468207883)
        ref[2,1,1]=(1.1388137075)/(o+1.)+(0.300492980829)
        ref[3,0,0]=(0.00651420878498)/(o+1.)+(-0.073951236079)
        ref[3,0,1]=(1.04013402573)/(o+1.)+(0.139906390894)
        ref[3,1,0]=(-0.994520244055)/(o+1.)+(-0.0541844186105)
        ref[3,1,1]=(-1.3366471591)/(o+1.)+(0.437452962527)
        ref[4,0,0]=(-0.27748097294)/(o+1.)+(-0.380866087845)
        ref[4,0,1]=(0.386505347129)/(o+1.)+(0.243362147483)
        ref[4,1,0]=(0.201989448993)/(o+1.)+(-0.0377976249661)
        ref[4,1,1]=(-0.441747022678)/(o+1.)+(-0.497612594833)
        ref[5,0,0]=(-1.09418090346)/(o+1.)+(0.391234587865)
        ref[5,0,1]=(-0.270130810219)/(o+1.)+(0.0973242828949)
        ref[5,1,0]=(0.949742520497)/(o+1.)+(-0.217319998996)
        ref[5,1,1]=(0.0544408125428)/(o+1.)+(0.0652541329568)
      else:
        arg[0,0,0]=(0.411788818613)*x[0]**o+(0.324151698669)*x[0]+(0.656551349616)*x[1]**o+(0.801577401197)*x[1]+(0.690440431022)*x[2]**o+(0.776409870151)*x[2]
        arg[0,0,1]=(-0.224985947659)*x[0]**o+(-0.564703086188)*x[0]+(-0.37196336116)*x[1]**o+(-0.114979853172)*x[1]+(0.274462080499)*x[2]**o+(-0.988489021983)*x[2]
        arg[0,1,0]=(0.512331530009)*x[0]**o+(0.519617843594)*x[0]+(-0.515174861158)*x[1]**o+(-0.675953902994)*x[1]+(0.796111490716)*x[2]**o+(0.229569489858)*x[2]
        arg[0,1,1]=(0.347765144376)*x[0]**o+(-0.0110194730031)*x[0]+(-0.593936645599)*x[1]**o+(-0.372786558089)*x[1]+(-0.604691321362)*x[2]**o+(-0.888002521077)*x[2]
        arg[1,0,0]=(-0.575954903433)*x[0]**o+(0.0160235655616)*x[0]+(0.305335632724)*x[1]**o+(-0.259402553796)*x[1]+(-0.221276854741)*x[2]**o+(-0.519932341069)*x[2]
        arg[1,0,1]=(-0.674987647238)*x[0]**o+(-0.45852667934)*x[0]+(-0.758937440978)*x[1]**o+(0.80566898197)*x[1]+(-0.961619154322)*x[2]**o+(0.861564430243)*x[2]
        arg[1,1,0]=(-0.297475412437)*x[0]**o+(0.881828705604)*x[0]+(-0.823267615874)*x[1]**o+(-0.0232728101313)*x[1]+(-0.676220001453)*x[2]**o+(-0.382414036981)*x[2]
        arg[1,1,1]=(-0.31008204616)*x[0]**o+(0.805195900401)*x[0]+(-0.758209772143)*x[1]**o+(0.861379709073)*x[1]+(0.0631063866518)*x[2]**o+(0.356607080553)*x[2]
        arg[2,0,0]=(0.305214964591)*x[0]**o+(0.815106526139)*x[0]+(0.138797156005)*x[1]**o+(-0.241921896332)*x[1]+(-0.103783035058)*x[2]**o+(-0.161952044984)*x[2]
        arg[2,0,1]=(-0.925375677635)*x[0]**o+(0.0341347568201)*x[0]+(-0.706905924964)*x[1]**o+(-0.193304368451)*x[1]+(-0.449866757104)*x[2]**o+(-0.894963680869)*x[2]
        arg[2,1,0]=(0.822676224279)*x[0]**o+(-0.28758006682)*x[0]+(-0.869025918917)*x[1]**o+(0.0107602044744)*x[1]+(-0.832322765163)*x[2]**o+(0.748440319389)*x[2]
        arg[2,1,1]=(-0.415326508997)*x[0]**o+(-0.879521424817)*x[0]+(-0.971781441062)*x[1]**o+(0.527586634108)*x[1]+(0.336682669856)*x[2]**o+(-0.948414041149)*x[2]
        arg[3,0,0]=(-0.345788531715)*x[0]**o+(0.626583498807)*x[0]+(-0.186418683649)*x[1]**o+(0.3141628424)*x[1]+(-0.781119787783)*x[2]**o+(0.972708430266)*x[2]
        arg[3,0,1]=(0.175536279095)*x[0]**o+(-0.150027774295)*x[0]+(-0.506653680683)*x[1]**o+(-0.482544397029)*x[1]+(-0.59090996152)*x[2]**o+(0.618123625026)*x[2]
        arg[3,1,0]=(0.737984493124)*x[0]**o+(0.650901680297)*x[0]+(-0.361389200009)*x[1]**o+(-0.784304665396)*x[1]+(-0.954132232142)*x[2]**o+(0.684341934348)*x[2]
        arg[3,1,1]=(-0.385757579645)*x[0]**o+(0.251136919845)*x[0]+(0.95143939933)*x[1]**o+(0.501860043225)*x[1]+(-0.675673011184)*x[2]**o+(0.337315450579)*x[2]
        arg[4,0,0]=(-0.740183402139)*x[0]**o+(-0.828278760743)*x[0]+(0.88000889117)*x[1]**o+(-0.0226801022184)*x[1]+(0.6811397662)*x[2]**o+(-0.254627446498)*x[2]
        arg[4,0,1]=(-0.194128952361)*x[0]**o+(-0.524854896388)*x[0]+(0.341100326577)*x[1]**o+(-0.26072443499)*x[1]+(-0.197396331776)*x[2]**o+(-0.589037632112)*x[2]
        arg[4,1,0]=(0.261995024865)*x[0]**o+(0.214193793699)*x[0]+(0.572024600244)*x[1]**o+(0.230659327198)*x[1]+(0.420971705453)*x[2]**o+(0.333445049827)*x[2]
        arg[4,1,1]=(-0.0679525054395)*x[0]**o+(0.0756646296473)*x[0]+(-0.299865836239)*x[1]**o+(-0.828294764914)*x[1]+(0.606170745631)*x[2]**o+(0.624267827393)*x[2]
        arg[5,0,0]=(0.312565542211)*x[0]**o+(-0.368815496171)*x[0]+(-0.0615973473105)*x[1]**o+(0.205012835971)*x[1]+(0.310818553076)*x[2]**o+(-0.287393908304)*x[2]
        arg[5,0,1]=(-0.409626309069)*x[0]**o+(0.977953573562)*x[0]+(0.583013544138)*x[1]**o+(0.542907128677)*x[1]+(0.488538336366)*x[2]**o+(-0.85823611123)*x[2]
        arg[5,1,0]=(-0.899607109395)*x[0]**o+(-0.0459053040522)*x[0]+(0.938423293041)*x[1]**o+(-0.112431665317)*x[1]+(0.620609328687)*x[2]**o+(-0.663082429319)*x[2]
        arg[5,1,1]=(0.00370253144626)*x[0]**o+(0.503733497956)*x[0]+(0.362457785043)*x[1]**o+(0.417891957303)*x[1]+(0.404543677829)*x[2]**o+(0.117828669122)*x[2]
        ref[0,0,0]=(1.75878059925)/(o+1.)+(0.951069485009)
        ref[0,0,1]=(-0.32248722832)/(o+1.)+(-0.834085980672)
        ref[0,1,0]=(0.793268159567)/(o+1.)+(0.036616715229)
        ref[0,1,1]=(-0.850862822585)/(o+1.)+(-0.635904276085)
        ref[1,0,0]=(-0.49189612545)/(o+1.)+(-0.381655664652)
        ref[1,0,1]=(-2.39554424254)/(o+1.)+(0.604353366437)
        ref[1,1,0]=(-1.79696302976)/(o+1.)+(0.238070929246)
        ref[1,1,1]=(-1.00518543165)/(o+1.)+(1.01159134501)
        ref[2,0,0]=(0.340229085537)/(o+1.)+(0.205616292412)
        ref[2,0,1]=(-2.0821483597)/(o+1.)+(-0.52706664625)
        ref[2,1,0]=(-0.878672459802)/(o+1.)+(0.235810228522)
        ref[2,1,1]=(-1.0504252802)/(o+1.)+(-0.650174415929)
        ref[3,0,0]=(-1.31332700315)/(o+1.)+(0.956727385736)
        ref[3,0,1]=(-0.922027363109)/(o+1.)+(-0.00722427314892)
        ref[3,1,0]=(-0.577536939027)/(o+1.)+(0.275469474624)
        ref[3,1,1]=(-0.109991191499)/(o+1.)+(0.545156206825)
        ref[4,0,0]=(0.820965255231)/(o+1.)+(-0.55279315473)
        ref[4,0,1]=(-0.0504249575602)/(o+1.)+(-0.687308481745)
        ref[4,1,0]=(1.25499133056)/(o+1.)+(0.389149085362)
        ref[4,1,1]=(0.238352403952)/(o+1.)+(-0.0641811539369)
        ref[5,0,0]=(0.561786747976)/(o+1.)+(-0.225598284252)
        ref[5,0,1]=(0.661925571434)/(o+1.)+(0.331312295504)
        ref[5,1,0]=(0.659425512333)/(o+1.)+(-0.410709699344)
        ref[5,1,1]=(0.770703994319)/(o+1.)+(0.519727062191)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunction_fromData_ContinuousFunction_rank4(self):
      """
      tests integral of rank 4 Data on the ReducedFunction

      assumptions: ContinuousFunction(self.domain) exists
                   self.domain supports integral on ReducedFunction
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=ReducedFunction(self.domain)
      w=ContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5, 3, 2),w)
      ref=numpy.zeros((4, 5, 3, 2),numpy.float_)
      if dim==2:
        arg[0,0,0,0]=(-0.164405890316)*x[0]**o+(-0.323303939764)*x[0]+(-0.0516829984856)*x[1]**o+(-0.220765052533)*x[1]
        arg[0,0,0,1]=(0.152172612108)*x[0]**o+(-0.166242164365)*x[0]+(0.807177808571)*x[1]**o+(0.0404817109573)*x[1]
        arg[0,0,1,0]=(0.016912808567)*x[0]**o+(0.623923316774)*x[0]+(-0.542886934905)*x[1]**o+(0.0110784560205)*x[1]
        arg[0,0,1,1]=(-0.663885786909)*x[0]**o+(-0.55563355485)*x[0]+(0.191993284382)*x[1]**o+(0.734973556043)*x[1]
        arg[0,0,2,0]=(-0.110316754183)*x[0]**o+(-0.584022783672)*x[0]+(0.56471376011)*x[1]**o+(0.846317747143)*x[1]
        arg[0,0,2,1]=(-0.767988009278)*x[0]**o+(0.440997912352)*x[0]+(0.45366348093)*x[1]**o+(0.376620990318)*x[1]
        arg[0,1,0,0]=(-0.138006508051)*x[0]**o+(0.8450564766)*x[0]+(-0.273559897633)*x[1]**o+(0.683678615645)*x[1]
        arg[0,1,0,1]=(-0.467521625474)*x[0]**o+(0.32289306962)*x[0]+(0.340974562152)*x[1]**o+(-0.58109391053)*x[1]
        arg[0,1,1,0]=(0.76867991208)*x[0]**o+(0.683527735131)*x[0]+(0.762986872389)*x[1]**o+(-0.189342734286)*x[1]
        arg[0,1,1,1]=(0.125150175351)*x[0]**o+(-0.175190872876)*x[0]+(-0.569766488076)*x[1]**o+(0.787503755206)*x[1]
        arg[0,1,2,0]=(-0.339695860559)*x[0]**o+(0.119452596206)*x[0]+(0.474378864192)*x[1]**o+(0.845982555265)*x[1]
        arg[0,1,2,1]=(0.820551334836)*x[0]**o+(-0.200024721285)*x[0]+(-0.525871855871)*x[1]**o+(-0.714639297791)*x[1]
        arg[0,2,0,0]=(-0.145788788262)*x[0]**o+(-0.695717636982)*x[0]+(0.736564332801)*x[1]**o+(0.634307016587)*x[1]
        arg[0,2,0,1]=(-0.25142006947)*x[0]**o+(0.027028767127)*x[0]+(-0.614777438028)*x[1]**o+(-0.691417785617)*x[1]
        arg[0,2,1,0]=(0.366081140839)*x[0]**o+(0.477490533426)*x[0]+(-0.116611335116)*x[1]**o+(-0.863552165712)*x[1]
        arg[0,2,1,1]=(-0.467659558308)*x[0]**o+(0.0836299545334)*x[0]+(0.903093252833)*x[1]**o+(-0.956858182533)*x[1]
        arg[0,2,2,0]=(0.870921031579)*x[0]**o+(-0.801681014465)*x[0]+(-0.951819608329)*x[1]**o+(-0.626222917084)*x[1]
        arg[0,2,2,1]=(0.414973183041)*x[0]**o+(-0.444580205703)*x[0]+(-0.346458764208)*x[1]**o+(-0.679140100745)*x[1]
        arg[0,3,0,0]=(-0.725326935158)*x[0]**o+(-0.914034759513)*x[0]+(0.702614377806)*x[1]**o+(0.819004276317)*x[1]
        arg[0,3,0,1]=(0.617099932352)*x[0]**o+(-0.155182850011)*x[0]+(-0.066428715129)*x[1]**o+(-0.286480585435)*x[1]
        arg[0,3,1,0]=(0.710942545488)*x[0]**o+(-0.734509277268)*x[0]+(-0.807474546329)*x[1]**o+(0.307631760449)*x[1]
        arg[0,3,1,1]=(-0.709966924732)*x[0]**o+(-0.462297022374)*x[0]+(0.773787774038)*x[1]**o+(0.762137020991)*x[1]
        arg[0,3,2,0]=(-0.425660932209)*x[0]**o+(-0.562130300181)*x[0]+(-0.516719014647)*x[1]**o+(-0.547832887369)*x[1]
        arg[0,3,2,1]=(0.149573176879)*x[0]**o+(-0.208999629436)*x[0]+(-0.191426033553)*x[1]**o+(-0.119046538873)*x[1]
        arg[0,4,0,0]=(-0.99932317151)*x[0]**o+(-0.832461558068)*x[0]+(-0.847073271676)*x[1]**o+(0.882046887729)*x[1]
        arg[0,4,0,1]=(0.14235557233)*x[0]**o+(-0.352572053276)*x[0]+(-0.231989519255)*x[1]**o+(0.146855135398)*x[1]
        arg[0,4,1,0]=(0.550561449012)*x[0]**o+(0.530745681425)*x[0]+(0.198889689472)*x[1]**o+(0.165503108658)*x[1]
        arg[0,4,1,1]=(0.059663401309)*x[0]**o+(0.168377411751)*x[0]+(-0.224682723925)*x[1]**o+(-0.255406907757)*x[1]
        arg[0,4,2,0]=(-0.747776894683)*x[0]**o+(-0.281763358533)*x[0]+(0.660369838923)*x[1]**o+(0.348541234687)*x[1]
        arg[0,4,2,1]=(-0.886248366716)*x[0]**o+(0.665256341312)*x[0]+(-0.118041122298)*x[1]**o+(0.192284814559)*x[1]
        arg[1,0,0,0]=(-0.212291422817)*x[0]**o+(-0.16123490382)*x[0]+(-0.884220059537)*x[1]**o+(-0.40104775897)*x[1]
        arg[1,0,0,1]=(0.389745305389)*x[0]**o+(0.544302066739)*x[0]+(-0.504770065056)*x[1]**o+(0.903371132654)*x[1]
        arg[1,0,1,0]=(0.983764499357)*x[0]**o+(0.0707766278425)*x[0]+(-0.119851167133)*x[1]**o+(0.668490934432)*x[1]
        arg[1,0,1,1]=(-0.61333738947)*x[0]**o+(-0.515837139916)*x[0]+(-0.825015725806)*x[1]**o+(-0.858420009384)*x[1]
        arg[1,0,2,0]=(0.838488008532)*x[0]**o+(0.320020900158)*x[0]+(0.265140913892)*x[1]**o+(0.493460973766)*x[1]
        arg[1,0,2,1]=(-0.957171379585)*x[0]**o+(0.911667647687)*x[0]+(0.0634495769152)*x[1]**o+(-0.244783653159)*x[1]
        arg[1,1,0,0]=(0.854312398231)*x[0]**o+(0.0177604750376)*x[0]+(-0.263135825468)*x[1]**o+(0.196452389095)*x[1]
        arg[1,1,0,1]=(0.496952473267)*x[0]**o+(-0.505581680291)*x[0]+(0.227653536528)*x[1]**o+(-0.138354150837)*x[1]
        arg[1,1,1,0]=(0.569682536816)*x[0]**o+(-0.523174667456)*x[0]+(-0.434073415962)*x[1]**o+(-0.853667257479)*x[1]
        arg[1,1,1,1]=(0.636882167235)*x[0]**o+(-0.122206427025)*x[0]+(0.934361025063)*x[1]**o+(-0.43037160177)*x[1]
        arg[1,1,2,0]=(0.0798318147504)*x[0]**o+(0.825231320185)*x[0]+(0.810701650808)*x[1]**o+(-0.544356371381)*x[1]
        arg[1,1,2,1]=(-0.426710170704)*x[0]**o+(0.200697079677)*x[0]+(0.242027752153)*x[1]**o+(0.746762119874)*x[1]
        arg[1,2,0,0]=(-0.497091856216)*x[0]**o+(0.0953816987137)*x[0]+(-0.388615292775)*x[1]**o+(0.794962452203)*x[1]
        arg[1,2,0,1]=(0.484769626964)*x[0]**o+(0.700484219762)*x[0]+(-0.230377697673)*x[1]**o+(-0.110327221848)*x[1]
        arg[1,2,1,0]=(-0.791806541129)*x[0]**o+(0.0199911517557)*x[0]+(-0.441657020836)*x[1]**o+(-0.972999930466)*x[1]
        arg[1,2,1,1]=(-0.566750983493)*x[0]**o+(0.193689535753)*x[0]+(0.659310468709)*x[1]**o+(0.209045229813)*x[1]
        arg[1,2,2,0]=(0.852553900341)*x[0]**o+(-0.869275488172)*x[0]+(-0.308429715369)*x[1]**o+(0.096335995297)*x[1]
        arg[1,2,2,1]=(0.169351018917)*x[0]**o+(0.269577471686)*x[0]+(-0.226840840616)*x[1]**o+(-0.94229540867)*x[1]
        arg[1,3,0,0]=(0.481711489234)*x[0]**o+(0.293576608201)*x[0]+(0.101998881767)*x[1]**o+(-0.10730442531)*x[1]
        arg[1,3,0,1]=(-0.824144820539)*x[0]**o+(0.174754759038)*x[0]+(-0.251803196848)*x[1]**o+(-0.389656255459)*x[1]
        arg[1,3,1,0]=(0.438310633462)*x[0]**o+(-0.268051020546)*x[0]+(0.670456978527)*x[1]**o+(-0.547393352784)*x[1]
        arg[1,3,1,1]=(-0.95082246973)*x[0]**o+(0.739061218443)*x[0]+(0.209495253219)*x[1]**o+(-0.604801908581)*x[1]
        arg[1,3,2,0]=(0.696900724687)*x[0]**o+(0.501191311274)*x[0]+(0.982952568975)*x[1]**o+(0.1345704134)*x[1]
        arg[1,3,2,1]=(0.823865642457)*x[0]**o+(0.344980106206)*x[0]+(0.0740071873515)*x[1]**o+(-0.950460516779)*x[1]
        arg[1,4,0,0]=(-0.428525532814)*x[0]**o+(0.496012936411)*x[0]+(-0.0756953876324)*x[1]**o+(0.8382068924)*x[1]
        arg[1,4,0,1]=(0.950388644852)*x[0]**o+(-0.225116488186)*x[0]+(0.515802878592)*x[1]**o+(-0.51004230039)*x[1]
        arg[1,4,1,0]=(0.868733254296)*x[0]**o+(-0.506544709529)*x[0]+(0.28935579533)*x[1]**o+(0.100277736612)*x[1]
        arg[1,4,1,1]=(-0.104533384722)*x[0]**o+(0.598995288066)*x[0]+(-0.738946772839)*x[1]**o+(-0.23245555216)*x[1]
        arg[1,4,2,0]=(-0.286232179213)*x[0]**o+(-0.31267596034)*x[0]+(0.588681250431)*x[1]**o+(-0.579857978487)*x[1]
        arg[1,4,2,1]=(0.405666376198)*x[0]**o+(0.380029842735)*x[0]+(-0.646672003588)*x[1]**o+(0.419606979519)*x[1]
        arg[2,0,0,0]=(0.183075434304)*x[0]**o+(0.0929532345426)*x[0]+(-0.742007638693)*x[1]**o+(-0.521914593386)*x[1]
        arg[2,0,0,1]=(0.292878118412)*x[0]**o+(-0.427165220789)*x[0]+(0.540033289295)*x[1]**o+(0.997945616043)*x[1]
        arg[2,0,1,0]=(0.872801856197)*x[0]**o+(0.0427577769706)*x[0]+(0.595354626662)*x[1]**o+(-0.878459740609)*x[1]
        arg[2,0,1,1]=(-0.259988866684)*x[0]**o+(0.154043636253)*x[0]+(0.0580047878991)*x[1]**o+(0.40696744056)*x[1]
        arg[2,0,2,0]=(-0.602648518466)*x[0]**o+(0.30877596654)*x[0]+(-0.268563459489)*x[1]**o+(-0.526379663585)*x[1]
        arg[2,0,2,1]=(-0.375777994931)*x[0]**o+(-0.175440011709)*x[0]+(-0.578650003477)*x[1]**o+(0.822157683321)*x[1]
        arg[2,1,0,0]=(-0.44389226502)*x[0]**o+(0.683580346916)*x[0]+(0.471857559282)*x[1]**o+(-0.869624665398)*x[1]
        arg[2,1,0,1]=(0.768272315036)*x[0]**o+(-0.709688321973)*x[0]+(0.200727881948)*x[1]**o+(0.617326752785)*x[1]
        arg[2,1,1,0]=(-0.371169228982)*x[0]**o+(-0.0248248405104)*x[0]+(-0.899904749204)*x[1]**o+(0.796779883083)*x[1]
        arg[2,1,1,1]=(0.220088670807)*x[0]**o+(0.691718163435)*x[0]+(-0.0834793048429)*x[1]**o+(-0.748110928988)*x[1]
        arg[2,1,2,0]=(0.246045293748)*x[0]**o+(-0.669660154233)*x[0]+(0.601040129345)*x[1]**o+(-0.0893792369983)*x[1]
        arg[2,1,2,1]=(0.318708374425)*x[0]**o+(-0.973048583674)*x[0]+(-0.575891882633)*x[1]**o+(0.229050058038)*x[1]
        arg[2,2,0,0]=(0.358008410645)*x[0]**o+(-0.834388697702)*x[0]+(0.0931465346237)*x[1]**o+(0.503934981947)*x[1]
        arg[2,2,0,1]=(-0.257597986495)*x[0]**o+(-0.58070568528)*x[0]+(0.735541935653)*x[1]**o+(-0.704666216263)*x[1]
        arg[2,2,1,0]=(0.0207421250729)*x[0]**o+(-0.724644198871)*x[0]+(0.637124335031)*x[1]**o+(-0.547300138808)*x[1]
        arg[2,2,1,1]=(0.280179217797)*x[0]**o+(-0.387279903944)*x[0]+(0.516275341085)*x[1]**o+(0.781250973385)*x[1]
        arg[2,2,2,0]=(-0.97771111517)*x[0]**o+(-0.140997969015)*x[0]+(-0.495112130955)*x[1]**o+(0.151523726543)*x[1]
        arg[2,2,2,1]=(-0.48638258963)*x[0]**o+(-0.251363723104)*x[0]+(-0.484911659879)*x[1]**o+(-0.841285872322)*x[1]
        arg[2,3,0,0]=(-0.70159761565)*x[0]**o+(-0.661658396735)*x[0]+(-0.380920250292)*x[1]**o+(0.682598882025)*x[1]
        arg[2,3,0,1]=(0.477056088757)*x[0]**o+(0.697397619448)*x[0]+(0.28617323153)*x[1]**o+(-0.0443690469658)*x[1]
        arg[2,3,1,0]=(-0.823038323615)*x[0]**o+(-0.742146866184)*x[0]+(0.258604766787)*x[1]**o+(-0.740368164141)*x[1]
        arg[2,3,1,1]=(0.958751077013)*x[0]**o+(0.353976218746)*x[0]+(-0.954344840728)*x[1]**o+(0.436915389196)*x[1]
        arg[2,3,2,0]=(0.432342075923)*x[0]**o+(-0.0699761039653)*x[0]+(0.701490046186)*x[1]**o+(0.0888793443125)*x[1]
        arg[2,3,2,1]=(0.612536667525)*x[0]**o+(0.107433723026)*x[0]+(-0.611874456965)*x[1]**o+(-0.253338899867)*x[1]
        arg[2,4,0,0]=(-0.0234819568317)*x[0]**o+(-0.0642964170457)*x[0]+(0.556679188908)*x[1]**o+(0.865922901373)*x[1]
        arg[2,4,0,1]=(0.669235019483)*x[0]**o+(0.744361910609)*x[0]+(0.383397302403)*x[1]**o+(0.669587709042)*x[1]
        arg[2,4,1,0]=(-0.143461051192)*x[0]**o+(-0.0272388772724)*x[0]+(0.723868767782)*x[1]**o+(0.303617835707)*x[1]
        arg[2,4,1,1]=(-0.829765013546)*x[0]**o+(-0.784193619495)*x[0]+(0.384376849236)*x[1]**o+(0.280904709362)*x[1]
        arg[2,4,2,0]=(0.38737050935)*x[0]**o+(0.749614613361)*x[0]+(-0.340848293488)*x[1]**o+(0.96291850214)*x[1]
        arg[2,4,2,1]=(0.619896872679)*x[0]**o+(0.970694589398)*x[0]+(-0.300698305892)*x[1]**o+(0.794408153506)*x[1]
        arg[3,0,0,0]=(-0.808442463698)*x[0]**o+(-0.527555255968)*x[0]+(0.246776596713)*x[1]**o+(-0.719809811547)*x[1]
        arg[3,0,0,1]=(-0.390016002326)*x[0]**o+(-0.864168521951)*x[0]+(-0.154649478359)*x[1]**o+(0.210766031283)*x[1]
        arg[3,0,1,0]=(-0.941916382966)*x[0]**o+(-0.0352551994705)*x[0]+(-0.660026082442)*x[1]**o+(-0.869241000235)*x[1]
        arg[3,0,1,1]=(-0.287992443961)*x[0]**o+(0.292551506636)*x[0]+(-0.619953648778)*x[1]**o+(0.746095811695)*x[1]
        arg[3,0,2,0]=(-0.233659077726)*x[0]**o+(-0.568647415596)*x[0]+(-0.22048065705)*x[1]**o+(0.658105381132)*x[1]
        arg[3,0,2,1]=(0.871784854897)*x[0]**o+(-0.118389479971)*x[0]+(0.143290959871)*x[1]**o+(0.753198625961)*x[1]
        arg[3,1,0,0]=(0.993988451174)*x[0]**o+(0.451849763406)*x[0]+(-0.972805640824)*x[1]**o+(0.796893857099)*x[1]
        arg[3,1,0,1]=(-0.424353439197)*x[0]**o+(-0.575799061446)*x[0]+(-0.233936106648)*x[1]**o+(-0.451864349448)*x[1]
        arg[3,1,1,0]=(0.639821887614)*x[0]**o+(0.771041944334)*x[0]+(0.420297869214)*x[1]**o+(0.671734395857)*x[1]
        arg[3,1,1,1]=(0.398121897075)*x[0]**o+(0.647679737081)*x[0]+(-0.544457308885)*x[1]**o+(0.58955287395)*x[1]
        arg[3,1,2,0]=(-0.118222351008)*x[0]**o+(-0.943807502172)*x[0]+(0.866866987138)*x[1]**o+(-0.0793340319339)*x[1]
        arg[3,1,2,1]=(-0.157286204244)*x[0]**o+(-0.789214982692)*x[0]+(0.646008960971)*x[1]**o+(0.866687256647)*x[1]
        arg[3,2,0,0]=(-0.620625598995)*x[0]**o+(-0.12874444072)*x[0]+(0.4402928966)*x[1]**o+(-0.0344340173898)*x[1]
        arg[3,2,0,1]=(-0.714597102332)*x[0]**o+(0.730206152989)*x[0]+(0.795717819655)*x[1]**o+(0.0543647593833)*x[1]
        arg[3,2,1,0]=(0.698721346048)*x[0]**o+(0.599204472096)*x[0]+(0.323529910402)*x[1]**o+(0.362678869119)*x[1]
        arg[3,2,1,1]=(0.976107824554)*x[0]**o+(0.559889392243)*x[0]+(0.240016231718)*x[1]**o+(0.343322878078)*x[1]
        arg[3,2,2,0]=(0.609695396838)*x[0]**o+(0.786331471916)*x[0]+(-0.965489152976)*x[1]**o+(0.553653504493)*x[1]
        arg[3,2,2,1]=(0.897952533418)*x[0]**o+(0.227242627929)*x[0]+(-0.121752169524)*x[1]**o+(0.844617689997)*x[1]
        arg[3,3,0,0]=(-0.249698211819)*x[0]**o+(-0.400095108874)*x[0]+(-0.44300562221)*x[1]**o+(0.387777895056)*x[1]
        arg[3,3,0,1]=(0.626469203641)*x[0]**o+(0.741893669108)*x[0]+(0.261882509899)*x[1]**o+(-0.726628233604)*x[1]
        arg[3,3,1,0]=(-0.525223500134)*x[0]**o+(0.918314547123)*x[0]+(0.426441176651)*x[1]**o+(0.338058005974)*x[1]
        arg[3,3,1,1]=(-0.31358142709)*x[0]**o+(-0.411070225036)*x[0]+(0.0902340021221)*x[1]**o+(0.404135075929)*x[1]
        arg[3,3,2,0]=(0.528842663849)*x[0]**o+(0.474768688828)*x[0]+(0.697643730083)*x[1]**o+(-0.170335125341)*x[1]
        arg[3,3,2,1]=(0.875221054674)*x[0]**o+(-0.441830492507)*x[0]+(-0.60312069438)*x[1]**o+(0.664895429753)*x[1]
        arg[3,4,0,0]=(0.563422972203)*x[0]**o+(-0.595004250424)*x[0]+(0.888927367905)*x[1]**o+(0.553640735136)*x[1]
        arg[3,4,0,1]=(0.79563891979)*x[0]**o+(-0.461626171695)*x[0]+(0.700194671524)*x[1]**o+(-0.687505010217)*x[1]
        arg[3,4,1,0]=(0.655248139354)*x[0]**o+(0.588175755435)*x[0]+(-0.29236593461)*x[1]**o+(0.575186295035)*x[1]
        arg[3,4,1,1]=(0.405730566113)*x[0]**o+(-0.161117488254)*x[0]+(0.912224633472)*x[1]**o+(0.0292423764587)*x[1]
        arg[3,4,2,0]=(-0.405819525278)*x[0]**o+(-0.362840757195)*x[0]+(0.961950188511)*x[1]**o+(0.205213542149)*x[1]
        arg[3,4,2,1]=(0.703066280254)*x[0]**o+(-0.936055321446)*x[0]+(0.861070629413)*x[1]**o+(0.0856428368256)*x[1]
        ref[0,0,0,0]=(-0.216088888802)/(o+1.)+(-0.272034496148)
        ref[0,0,0,1]=(0.959350420679)/(o+1.)+(-0.062880226704)
        ref[0,0,1,0]=(-0.525974126338)/(o+1.)+(0.317500886397)
        ref[0,0,1,1]=(-0.471892502528)/(o+1.)+(0.0896700005967)
        ref[0,0,2,0]=(0.454397005927)/(o+1.)+(0.131147481736)
        ref[0,0,2,1]=(-0.314324528348)/(o+1.)+(0.408809451335)
        ref[0,1,0,0]=(-0.411566405684)/(o+1.)+(0.764367546122)
        ref[0,1,0,1]=(-0.126547063321)/(o+1.)+(-0.129100420455)
        ref[0,1,1,0]=(1.53166678447)/(o+1.)+(0.247092500423)
        ref[0,1,1,1]=(-0.444616312725)/(o+1.)+(0.306156441165)
        ref[0,1,2,0]=(0.134683003633)/(o+1.)+(0.482717575736)
        ref[0,1,2,1]=(0.294679478965)/(o+1.)+(-0.457332009538)
        ref[0,2,0,0]=(0.590775544538)/(o+1.)+(-0.0307053101973)
        ref[0,2,0,1]=(-0.866197507498)/(o+1.)+(-0.332194509245)
        ref[0,2,1,0]=(0.249469805723)/(o+1.)+(-0.193030816143)
        ref[0,2,1,1]=(0.435433694525)/(o+1.)+(-0.436614114)
        ref[0,2,2,0]=(-0.0808985767498)/(o+1.)+(-0.713951965774)
        ref[0,2,2,1]=(0.0685144188336)/(o+1.)+(-0.561860153224)
        ref[0,3,0,0]=(-0.0227125573528)/(o+1.)+(-0.0475152415978)
        ref[0,3,0,1]=(0.550671217223)/(o+1.)+(-0.220831717723)
        ref[0,3,1,0]=(-0.0965320008409)/(o+1.)+(-0.213438758409)
        ref[0,3,1,1]=(0.0638208493064)/(o+1.)+(0.149919999309)
        ref[0,3,2,0]=(-0.942379946856)/(o+1.)+(-0.554981593775)
        ref[0,3,2,1]=(-0.0418528566735)/(o+1.)+(-0.164023084154)
        ref[0,4,0,0]=(-1.84639644319)/(o+1.)+(0.0247926648306)
        ref[0,4,0,1]=(-0.0896339469252)/(o+1.)+(-0.102858458939)
        ref[0,4,1,0]=(0.749451138484)/(o+1.)+(0.348124395042)
        ref[0,4,1,1]=(-0.165019322616)/(o+1.)+(-0.0435147480027)
        ref[0,4,2,0]=(-0.0874070557597)/(o+1.)+(0.033388938077)
        ref[0,4,2,1]=(-1.00428948901)/(o+1.)+(0.428770577935)
        ref[1,0,0,0]=(-1.09651148235)/(o+1.)+(-0.281141331395)
        ref[1,0,0,1]=(-0.115024759667)/(o+1.)+(0.723836599697)
        ref[1,0,1,0]=(0.863913332224)/(o+1.)+(0.369633781137)
        ref[1,0,1,1]=(-1.43835311528)/(o+1.)+(-0.68712857465)
        ref[1,0,2,0]=(1.10362892242)/(o+1.)+(0.406740936962)
        ref[1,0,2,1]=(-0.893721802669)/(o+1.)+(0.333441997264)
        ref[1,1,0,0]=(0.591176572763)/(o+1.)+(0.107106432066)
        ref[1,1,0,1]=(0.724606009795)/(o+1.)+(-0.321967915564)
        ref[1,1,1,0]=(0.135609120854)/(o+1.)+(-0.688420962467)
        ref[1,1,1,1]=(1.5712431923)/(o+1.)+(-0.276289014397)
        ref[1,1,2,0]=(0.890533465558)/(o+1.)+(0.140437474402)
        ref[1,1,2,1]=(-0.184682418551)/(o+1.)+(0.473729599776)
        ref[1,2,0,0]=(-0.88570714899)/(o+1.)+(0.445172075458)
        ref[1,2,0,1]=(0.254391929291)/(o+1.)+(0.295078498957)
        ref[1,2,1,0]=(-1.23346356196)/(o+1.)+(-0.476504389355)
        ref[1,2,1,1]=(0.0925594852161)/(o+1.)+(0.201367382783)
        ref[1,2,2,0]=(0.544124184972)/(o+1.)+(-0.386469746437)
        ref[1,2,2,1]=(-0.057489821699)/(o+1.)+(-0.336358968492)
        ref[1,3,0,0]=(0.583710371001)/(o+1.)+(0.0931360914458)
        ref[1,3,0,1]=(-1.07594801739)/(o+1.)+(-0.107450748211)
        ref[1,3,1,0]=(1.10876761199)/(o+1.)+(-0.407722186665)
        ref[1,3,1,1]=(-0.741327216511)/(o+1.)+(0.0671296549307)
        ref[1,3,2,0]=(1.67985329366)/(o+1.)+(0.317880862337)
        ref[1,3,2,1]=(0.897872829808)/(o+1.)+(-0.302740205287)
        ref[1,4,0,0]=(-0.504220920447)/(o+1.)+(0.667109914405)
        ref[1,4,0,1]=(1.46619152344)/(o+1.)+(-0.367579394288)
        ref[1,4,1,0]=(1.15808904963)/(o+1.)+(-0.203133486458)
        ref[1,4,1,1]=(-0.843480157561)/(o+1.)+(0.183269867953)
        ref[1,4,2,0]=(0.302449071217)/(o+1.)+(-0.446266969414)
        ref[1,4,2,1]=(-0.24100562739)/(o+1.)+(0.399818411127)
        ref[2,0,0,0]=(-0.558932204389)/(o+1.)+(-0.214480679421)
        ref[2,0,0,1]=(0.832911407707)/(o+1.)+(0.285390197627)
        ref[2,0,1,0]=(1.46815648286)/(o+1.)+(-0.417850981819)
        ref[2,0,1,1]=(-0.201984078785)/(o+1.)+(0.280505538406)
        ref[2,0,2,0]=(-0.871211977955)/(o+1.)+(-0.108801848522)
        ref[2,0,2,1]=(-0.954427998409)/(o+1.)+(0.323358835806)
        ref[2,1,0,0]=(0.0279652942615)/(o+1.)+(-0.0930221592407)
        ref[2,1,0,1]=(0.969000196983)/(o+1.)+(-0.0461807845941)
        ref[2,1,1,0]=(-1.27107397819)/(o+1.)+(0.385977521287)
        ref[2,1,1,1]=(0.136609365964)/(o+1.)+(-0.0281963827764)
        ref[2,1,2,0]=(0.847085423093)/(o+1.)+(-0.379519695616)
        ref[2,1,2,1]=(-0.257183508208)/(o+1.)+(-0.371999262818)
        ref[2,2,0,0]=(0.451154945268)/(o+1.)+(-0.165226857878)
        ref[2,2,0,1]=(0.477943949158)/(o+1.)+(-0.642685950772)
        ref[2,2,1,0]=(0.657866460103)/(o+1.)+(-0.635972168839)
        ref[2,2,1,1]=(0.796454558882)/(o+1.)+(0.19698553472)
        ref[2,2,2,0]=(-1.47282324613)/(o+1.)+(0.00526287876389)
        ref[2,2,2,1]=(-0.971294249509)/(o+1.)+(-0.546324797713)
        ref[2,3,0,0]=(-1.08251786594)/(o+1.)+(0.0104702426452)
        ref[2,3,0,1]=(0.763229320287)/(o+1.)+(0.326514286241)
        ref[2,3,1,0]=(-0.564433556829)/(o+1.)+(-0.741257515163)
        ref[2,3,1,1]=(0.00440623628526)/(o+1.)+(0.395445803971)
        ref[2,3,2,0]=(1.13383212211)/(o+1.)+(0.0094516201736)
        ref[2,3,2,1]=(0.000662210559809)/(o+1.)+(-0.0729525884203)
        ref[2,4,0,0]=(0.533197232076)/(o+1.)+(0.400813242164)
        ref[2,4,0,1]=(1.05263232189)/(o+1.)+(0.706974809826)
        ref[2,4,1,0]=(0.580407716589)/(o+1.)+(0.138189479217)
        ref[2,4,1,1]=(-0.44538816431)/(o+1.)+(-0.251644455066)
        ref[2,4,2,0]=(0.0465222158616)/(o+1.)+(0.85626655775)
        ref[2,4,2,1]=(0.319198566788)/(o+1.)+(0.882551371452)
        ref[3,0,0,0]=(-0.561665866984)/(o+1.)+(-0.623682533758)
        ref[3,0,0,1]=(-0.544665480685)/(o+1.)+(-0.326701245334)
        ref[3,0,1,0]=(-1.60194246541)/(o+1.)+(-0.452248099853)
        ref[3,0,1,1]=(-0.907946092739)/(o+1.)+(0.519323659165)
        ref[3,0,2,0]=(-0.454139734776)/(o+1.)+(0.0447289827682)
        ref[3,0,2,1]=(1.01507581477)/(o+1.)+(0.317404572995)
        ref[3,1,0,0]=(0.0211828103496)/(o+1.)+(0.624371810253)
        ref[3,1,0,1]=(-0.658289545845)/(o+1.)+(-0.513831705447)
        ref[3,1,1,0]=(1.06011975683)/(o+1.)+(0.721388170095)
        ref[3,1,1,1]=(-0.14633541181)/(o+1.)+(0.618616305516)
        ref[3,1,2,0]=(0.74864463613)/(o+1.)+(-0.511570767053)
        ref[3,1,2,1]=(0.488722756727)/(o+1.)+(0.0387361369774)
        ref[3,2,0,0]=(-0.180332702395)/(o+1.)+(-0.0815892290549)
        ref[3,2,0,1]=(0.0811207173227)/(o+1.)+(0.392285456186)
        ref[3,2,1,0]=(1.02225125645)/(o+1.)+(0.480941670608)
        ref[3,2,1,1]=(1.21612405627)/(o+1.)+(0.45160613516)
        ref[3,2,2,0]=(-0.355793756138)/(o+1.)+(0.669992488205)
        ref[3,2,2,1]=(0.776200363894)/(o+1.)+(0.535930158963)
        ref[3,3,0,0]=(-0.692703834029)/(o+1.)+(-0.00615860690903)
        ref[3,3,0,1]=(0.88835171354)/(o+1.)+(0.00763271775166)
        ref[3,3,1,0]=(-0.0987823234827)/(o+1.)+(0.628186276549)
        ref[3,3,1,1]=(-0.223347424968)/(o+1.)+(-0.00346757455392)
        ref[3,3,2,0]=(1.22648639393)/(o+1.)+(0.152216781743)
        ref[3,3,2,1]=(0.272100360294)/(o+1.)+(0.111532468623)
        ref[3,4,0,0]=(1.45235034011)/(o+1.)+(-0.0206817576441)
        ref[3,4,0,1]=(1.49583359131)/(o+1.)+(-0.574565590956)
        ref[3,4,1,0]=(0.362882204744)/(o+1.)+(0.581681025235)
        ref[3,4,1,1]=(1.31795519959)/(o+1.)+(-0.0659375558974)
        ref[3,4,2,0]=(0.556130663233)/(o+1.)+(-0.078813607523)
        ref[3,4,2,1]=(1.56413690967)/(o+1.)+(-0.42520624231)
      else:
        arg[0,0,0,0]=(0.956854727968)*x[0]**o+(-0.43263548815)*x[0]+(-0.1485543705)*x[1]**o+(-0.411060684134)*x[1]+(-0.306829513954)*x[2]**o+(0.89077924839)*x[2]
        arg[0,0,0,1]=(0.125905642347)*x[0]**o+(-0.645491335766)*x[0]+(-0.237571989123)*x[1]**o+(-0.239301942511)*x[1]+(-0.384215893518)*x[2]**o+(-0.371218179274)*x[2]
        arg[0,0,1,0]=(0.0204273419969)*x[0]**o+(0.354631489501)*x[0]+(0.235086936835)*x[1]**o+(0.647551574475)*x[1]+(-0.729066874572)*x[2]**o+(0.177733791046)*x[2]
        arg[0,0,1,1]=(-0.0938492166558)*x[0]**o+(-0.484339074939)*x[0]+(0.619333193491)*x[1]**o+(0.809157671781)*x[1]+(-0.28367956249)*x[2]**o+(-0.1758249267)*x[2]
        arg[0,0,2,0]=(-0.334944259519)*x[0]**o+(-0.126145567301)*x[0]+(-0.646972616146)*x[1]**o+(0.20287553181)*x[1]+(-0.383816067326)*x[2]**o+(-0.0265591031498)*x[2]
        arg[0,0,2,1]=(0.103308564583)*x[0]**o+(0.016108181951)*x[0]+(-0.367450237136)*x[1]**o+(0.657618788015)*x[1]+(-0.8327532511)*x[2]**o+(-0.267286876849)*x[2]
        arg[0,1,0,0]=(0.688995722152)*x[0]**o+(-0.982012541087)*x[0]+(-0.232339851099)*x[1]**o+(-0.097863461266)*x[1]+(0.25916639806)*x[2]**o+(0.0538637018789)*x[2]
        arg[0,1,0,1]=(0.446698337789)*x[0]**o+(-0.309434573629)*x[0]+(0.617004214016)*x[1]**o+(0.747666594435)*x[1]+(-0.869684246717)*x[2]**o+(0.75016186442)*x[2]
        arg[0,1,1,0]=(-0.559974353252)*x[0]**o+(0.966642794905)*x[0]+(-0.228473000256)*x[1]**o+(-0.435093965632)*x[1]+(0.426986383284)*x[2]**o+(0.131300817321)*x[2]
        arg[0,1,1,1]=(-0.696027598385)*x[0]**o+(-0.804778535506)*x[0]+(0.930171049019)*x[1]**o+(-0.900826020873)*x[1]+(0.844689166523)*x[2]**o+(0.0349718139676)*x[2]
        arg[0,1,2,0]=(0.359170897328)*x[0]**o+(0.992944582308)*x[0]+(0.579162388074)*x[1]**o+(0.743410851741)*x[1]+(-0.204325306226)*x[2]**o+(-0.992005484878)*x[2]
        arg[0,1,2,1]=(-0.883554988936)*x[0]**o+(0.486947073334)*x[0]+(-0.78078860326)*x[1]**o+(-0.134196299284)*x[1]+(-0.980852067031)*x[2]**o+(0.453023187355)*x[2]
        arg[0,2,0,0]=(0.804524524371)*x[0]**o+(0.397924403155)*x[0]+(0.712297039229)*x[1]**o+(0.713485622558)*x[1]+(-0.966783610543)*x[2]**o+(0.497869404576)*x[2]
        arg[0,2,0,1]=(-0.564706930001)*x[0]**o+(-0.482026983244)*x[0]+(0.284926154564)*x[1]**o+(0.844423322149)*x[1]+(0.444203847061)*x[2]**o+(0.333244188639)*x[2]
        arg[0,2,1,0]=(-0.630410083438)*x[0]**o+(-0.831218126629)*x[0]+(0.400237927956)*x[1]**o+(-0.70518378137)*x[1]+(-0.803754805917)*x[2]**o+(0.60059455602)*x[2]
        arg[0,2,1,1]=(0.198272547853)*x[0]**o+(0.0840356091496)*x[0]+(0.114385146155)*x[1]**o+(0.731057038288)*x[1]+(-0.976056408523)*x[2]**o+(-0.132565954304)*x[2]
        arg[0,2,2,0]=(0.983300141657)*x[0]**o+(-0.733168195162)*x[0]+(-0.948802450335)*x[1]**o+(-0.756616078178)*x[1]+(0.450302374372)*x[2]**o+(0.802115535717)*x[2]
        arg[0,2,2,1]=(-0.680853503774)*x[0]**o+(-0.910402862928)*x[0]+(0.583775613868)*x[1]**o+(0.741516645611)*x[1]+(-0.550498467533)*x[2]**o+(-0.610648481959)*x[2]
        arg[0,3,0,0]=(0.981909313325)*x[0]**o+(0.148145112359)*x[0]+(-0.989721117518)*x[1]**o+(0.0246558730659)*x[1]+(0.721439979982)*x[2]**o+(0.896052837889)*x[2]
        arg[0,3,0,1]=(0.109831691542)*x[0]**o+(-0.349555366958)*x[0]+(-0.54822642636)*x[1]**o+(0.121919022172)*x[1]+(-0.580228247481)*x[2]**o+(0.564276455713)*x[2]
        arg[0,3,1,0]=(0.106091093353)*x[0]**o+(0.410111742327)*x[0]+(-0.0669026478824)*x[1]**o+(-0.662684907574)*x[1]+(-0.351815520918)*x[2]**o+(-0.324802958038)*x[2]
        arg[0,3,1,1]=(0.857685803258)*x[0]**o+(-0.782627954043)*x[0]+(-0.555036389439)*x[1]**o+(-0.0126716685608)*x[1]+(0.260461379777)*x[2]**o+(0.117590260591)*x[2]
        arg[0,3,2,0]=(-0.987873674604)*x[0]**o+(0.0779662295512)*x[0]+(-0.928229202803)*x[1]**o+(0.508015712707)*x[1]+(0.248638760509)*x[2]**o+(-0.712158091156)*x[2]
        arg[0,3,2,1]=(-0.977973356138)*x[0]**o+(0.270015563339)*x[0]+(0.811157094127)*x[1]**o+(-0.756460384936)*x[1]+(-0.735557557504)*x[2]**o+(0.780333584929)*x[2]
        arg[0,4,0,0]=(-0.762444411646)*x[0]**o+(0.386894014691)*x[0]+(0.763898738855)*x[1]**o+(-0.422896581278)*x[1]+(0.428591445563)*x[2]**o+(-0.617626926396)*x[2]
        arg[0,4,0,1]=(-0.269524167286)*x[0]**o+(0.996895823788)*x[0]+(0.771396605337)*x[1]**o+(-0.993794940262)*x[1]+(-0.534803587481)*x[2]**o+(-0.928627946122)*x[2]
        arg[0,4,1,0]=(0.978497459584)*x[0]**o+(-0.343496507186)*x[0]+(-0.250581087884)*x[1]**o+(-0.855743445852)*x[1]+(-0.127512246124)*x[2]**o+(-0.133936486447)*x[2]
        arg[0,4,1,1]=(-0.844450818471)*x[0]**o+(0.263329730105)*x[0]+(0.411212783735)*x[1]**o+(-0.816379748952)*x[1]+(-0.443806901857)*x[2]**o+(0.1262969097)*x[2]
        arg[0,4,2,0]=(-0.326938381124)*x[0]**o+(-0.707054728551)*x[0]+(-0.725513874889)*x[1]**o+(0.655254977879)*x[1]+(0.772734170665)*x[2]**o+(-0.963856465024)*x[2]
        arg[0,4,2,1]=(0.896415185133)*x[0]**o+(-0.722431478737)*x[0]+(0.942058336843)*x[1]**o+(0.838230971246)*x[1]+(-0.279821311439)*x[2]**o+(-0.330130961206)*x[2]
        arg[1,0,0,0]=(-0.466537465363)*x[0]**o+(0.86493624537)*x[0]+(-0.943049537777)*x[1]**o+(0.658727518981)*x[1]+(-0.130355508544)*x[2]**o+(-0.158245981876)*x[2]
        arg[1,0,0,1]=(0.247046604605)*x[0]**o+(-0.287683750248)*x[0]+(0.266630203142)*x[1]**o+(0.372409648579)*x[1]+(-0.239411725166)*x[2]**o+(0.537339451105)*x[2]
        arg[1,0,1,0]=(-0.232312387253)*x[0]**o+(-0.0493954012747)*x[0]+(-0.461085479525)*x[1]**o+(-0.0744729096417)*x[1]+(-0.795163064273)*x[2]**o+(0.613075576806)*x[2]
        arg[1,0,1,1]=(-0.68570821748)*x[0]**o+(-0.492507082229)*x[0]+(-0.0673636097828)*x[1]**o+(0.569119992806)*x[1]+(-0.227653595613)*x[2]**o+(0.0337433636837)*x[2]
        arg[1,0,2,0]=(-0.124371489405)*x[0]**o+(-0.726635218442)*x[0]+(-0.835418167414)*x[1]**o+(-0.00414597707753)*x[1]+(0.8365917104)*x[2]**o+(-0.507314639615)*x[2]
        arg[1,0,2,1]=(-0.973228057052)*x[0]**o+(0.123228228923)*x[0]+(0.545712842666)*x[1]**o+(-0.451331566047)*x[1]+(0.299501907067)*x[2]**o+(-0.24898389883)*x[2]
        arg[1,1,0,0]=(-0.940690906836)*x[0]**o+(-0.357571197214)*x[0]+(0.98643983012)*x[1]**o+(-0.82113140872)*x[1]+(0.460009285424)*x[2]**o+(-0.0298714172583)*x[2]
        arg[1,1,0,1]=(-0.881543245921)*x[0]**o+(-0.640741814727)*x[0]+(-0.749244677096)*x[1]**o+(0.47161101909)*x[1]+(0.712893375828)*x[2]**o+(0.600151133959)*x[2]
        arg[1,1,1,0]=(-0.217214692405)*x[0]**o+(-0.504211076658)*x[0]+(0.857127736087)*x[1]**o+(0.0439761116172)*x[1]+(0.956380848189)*x[2]**o+(0.698482155013)*x[2]
        arg[1,1,1,1]=(-0.288969728734)*x[0]**o+(-0.63960886193)*x[0]+(-0.741293494258)*x[1]**o+(-0.125034323955)*x[1]+(-0.608855286755)*x[2]**o+(0.462427737944)*x[2]
        arg[1,1,2,0]=(0.126242043984)*x[0]**o+(0.811846484451)*x[0]+(-0.522920862127)*x[1]**o+(0.678142081935)*x[1]+(0.0392586436898)*x[2]**o+(0.403908304728)*x[2]
        arg[1,1,2,1]=(-0.290103861843)*x[0]**o+(0.336130241928)*x[0]+(-0.872641307678)*x[1]**o+(-0.372020796526)*x[1]+(0.172572077906)*x[2]**o+(-0.789431556966)*x[2]
        arg[1,2,0,0]=(-0.158114079913)*x[0]**o+(0.954919898537)*x[0]+(-0.711834968234)*x[1]**o+(-0.390487270434)*x[1]+(-0.086581514295)*x[2]**o+(0.442047084988)*x[2]
        arg[1,2,0,1]=(-0.914034594356)*x[0]**o+(0.521137372279)*x[0]+(0.12454416569)*x[1]**o+(0.627815188423)*x[1]+(0.770244577882)*x[2]**o+(-0.846691748091)*x[2]
        arg[1,2,1,0]=(0.769631824465)*x[0]**o+(0.988840796717)*x[0]+(-0.401836868027)*x[1]**o+(0.401846957832)*x[1]+(-0.877826997285)*x[2]**o+(0.857725824229)*x[2]
        arg[1,2,1,1]=(0.173539365276)*x[0]**o+(0.373206670706)*x[0]+(-0.597288730366)*x[1]**o+(-0.770220794555)*x[1]+(-0.586510454467)*x[2]**o+(-0.569370842441)*x[2]
        arg[1,2,2,0]=(0.673765155818)*x[0]**o+(-0.642914516709)*x[0]+(0.794438851788)*x[1]**o+(-0.17557754186)*x[1]+(0.997565143863)*x[2]**o+(0.722460144483)*x[2]
        arg[1,2,2,1]=(-0.878452482718)*x[0]**o+(-0.846862156166)*x[0]+(0.462351027504)*x[1]**o+(0.267435606154)*x[1]+(-0.452135086983)*x[2]**o+(0.701577407252)*x[2]
        arg[1,3,0,0]=(-0.0557389135964)*x[0]**o+(-0.781831460089)*x[0]+(0.214846092931)*x[1]**o+(0.597436260353)*x[1]+(0.713968428327)*x[2]**o+(-0.456335676572)*x[2]
        arg[1,3,0,1]=(0.288266531585)*x[0]**o+(-0.259529556521)*x[0]+(0.0286107975184)*x[1]**o+(0.465095455484)*x[1]+(-0.747648909025)*x[2]**o+(0.682866470382)*x[2]
        arg[1,3,1,0]=(-0.720339104668)*x[0]**o+(0.961100526257)*x[0]+(0.0814885381756)*x[1]**o+(-0.448775223649)*x[1]+(0.250972143198)*x[2]**o+(-0.240231439947)*x[2]
        arg[1,3,1,1]=(-0.459305786748)*x[0]**o+(0.272222143865)*x[0]+(-0.499164099401)*x[1]**o+(-0.225162547271)*x[1]+(-0.92261124167)*x[2]**o+(-0.436659790026)*x[2]
        arg[1,3,2,0]=(0.890752966329)*x[0]**o+(-0.775207922011)*x[0]+(0.715960284721)*x[1]**o+(0.843422841789)*x[1]+(0.513788850466)*x[2]**o+(-0.325733710281)*x[2]
        arg[1,3,2,1]=(0.958726172433)*x[0]**o+(-0.743672393249)*x[0]+(-0.523783133945)*x[1]**o+(0.133861976051)*x[1]+(-0.109230940116)*x[2]**o+(0.819716327751)*x[2]
        arg[1,4,0,0]=(0.54439729551)*x[0]**o+(0.0763503792087)*x[0]+(0.0634218657498)*x[1]**o+(-0.405471845044)*x[1]+(0.316327091756)*x[2]**o+(-0.539493863939)*x[2]
        arg[1,4,0,1]=(0.527330373182)*x[0]**o+(0.77531447123)*x[0]+(-0.481869315442)*x[1]**o+(0.556927692362)*x[1]+(-0.267215098309)*x[2]**o+(0.958919406465)*x[2]
        arg[1,4,1,0]=(0.540950372012)*x[0]**o+(-0.95002626611)*x[0]+(-0.523544509893)*x[1]**o+(-0.583056635057)*x[1]+(-0.00288409928193)*x[2]**o+(0.671385098427)*x[2]
        arg[1,4,1,1]=(-0.39802076092)*x[0]**o+(-0.583264022146)*x[0]+(-0.656261495961)*x[1]**o+(-0.527677078418)*x[1]+(-0.2632498932)*x[2]**o+(-0.537632951507)*x[2]
        arg[1,4,2,0]=(-0.77176416566)*x[0]**o+(0.812696440735)*x[0]+(0.0925789242005)*x[1]**o+(0.299698967601)*x[1]+(-0.0185377066587)*x[2]**o+(-0.883247318682)*x[2]
        arg[1,4,2,1]=(0.360949407452)*x[0]**o+(-0.463659754812)*x[0]+(0.799066387132)*x[1]**o+(0.0318151346916)*x[1]+(-0.380675976437)*x[2]**o+(-0.299341342112)*x[2]
        arg[2,0,0,0]=(-0.985332287141)*x[0]**o+(0.33398206098)*x[0]+(0.39204502373)*x[1]**o+(-0.586826360517)*x[1]+(0.868240161369)*x[2]**o+(-0.257897525907)*x[2]
        arg[2,0,0,1]=(-0.801369266967)*x[0]**o+(0.333496542809)*x[0]+(0.459444994717)*x[1]**o+(-0.950297568688)*x[1]+(-0.760722281765)*x[2]**o+(0.0933850221377)*x[2]
        arg[2,0,1,0]=(0.53372811812)*x[0]**o+(0.999551098511)*x[0]+(-0.761650890163)*x[1]**o+(-0.80456940786)*x[1]+(-0.452089806623)*x[2]**o+(-0.40841169158)*x[2]
        arg[2,0,1,1]=(-0.782294374468)*x[0]**o+(0.614736817827)*x[0]+(0.262995910218)*x[1]**o+(0.067349886804)*x[1]+(-0.167494656909)*x[2]**o+(-0.730462378769)*x[2]
        arg[2,0,2,0]=(0.261001857862)*x[0]**o+(0.959318765248)*x[0]+(-0.177224016298)*x[1]**o+(-0.979637664014)*x[1]+(0.858937930292)*x[2]**o+(-0.627116377798)*x[2]
        arg[2,0,2,1]=(-0.238083578675)*x[0]**o+(0.586367991705)*x[0]+(0.747378561289)*x[1]**o+(-0.540178926415)*x[1]+(-0.519781240185)*x[2]**o+(-0.42715014981)*x[2]
        arg[2,1,0,0]=(-0.21022574437)*x[0]**o+(-0.292359099348)*x[0]+(0.449614855785)*x[1]**o+(-0.416904619153)*x[1]+(0.507537288404)*x[2]**o+(0.943584716996)*x[2]
        arg[2,1,0,1]=(-0.410591525695)*x[0]**o+(-0.258796394127)*x[0]+(-0.474714731399)*x[1]**o+(-0.518288451462)*x[1]+(-0.15640903664)*x[2]**o+(0.97242574307)*x[2]
        arg[2,1,1,0]=(-0.930865834892)*x[0]**o+(0.961887630616)*x[0]+(0.688706862295)*x[1]**o+(-0.375163634404)*x[1]+(0.378954001649)*x[2]**o+(-0.747140890466)*x[2]
        arg[2,1,1,1]=(0.0454701384411)*x[0]**o+(-0.37651801332)*x[0]+(0.0808750051583)*x[1]**o+(-0.530355850075)*x[1]+(0.769675417234)*x[2]**o+(-0.767046792016)*x[2]
        arg[2,1,2,0]=(0.179573418696)*x[0]**o+(-0.0547918709025)*x[0]+(0.739792606043)*x[1]**o+(0.849619562235)*x[1]+(-0.92840796838)*x[2]**o+(-0.302941296337)*x[2]
        arg[2,1,2,1]=(0.160657807137)*x[0]**o+(0.551943403654)*x[0]+(0.649164334204)*x[1]**o+(0.298882266215)*x[1]+(0.33872360276)*x[2]**o+(-0.558910868754)*x[2]
        arg[2,2,0,0]=(-0.704330511528)*x[0]**o+(-0.326943167116)*x[0]+(0.626863819473)*x[1]**o+(-0.107147718189)*x[1]+(0.00668670393016)*x[2]**o+(0.572223662188)*x[2]
        arg[2,2,0,1]=(0.879695137799)*x[0]**o+(-0.313488700497)*x[0]+(-0.113134766565)*x[1]**o+(-0.131007759381)*x[1]+(0.55460969593)*x[2]**o+(0.898706374575)*x[2]
        arg[2,2,1,0]=(0.329932124981)*x[0]**o+(-0.624965470976)*x[0]+(0.82906774364)*x[1]**o+(-0.681403786977)*x[1]+(-0.447119491747)*x[2]**o+(0.143763601614)*x[2]
        arg[2,2,1,1]=(0.0367746207528)*x[0]**o+(0.47553259605)*x[0]+(-0.517723045973)*x[1]**o+(0.263932783195)*x[1]+(-0.131032589284)*x[2]**o+(-0.132729291841)*x[2]
        arg[2,2,2,0]=(-0.108349266564)*x[0]**o+(-0.684748849483)*x[0]+(0.203271582231)*x[1]**o+(-0.860842085896)*x[1]+(0.469684577785)*x[2]**o+(0.195736308182)*x[2]
        arg[2,2,2,1]=(0.915666799349)*x[0]**o+(0.383338953756)*x[0]+(-0.554011249579)*x[1]**o+(0.232285573367)*x[1]+(0.39288151674)*x[2]**o+(0.692592233016)*x[2]
        arg[2,3,0,0]=(0.81179207663)*x[0]**o+(0.786809236572)*x[0]+(-0.814758445611)*x[1]**o+(-0.78461653396)*x[1]+(0.171435802778)*x[2]**o+(0.660186909193)*x[2]
        arg[2,3,0,1]=(0.752694523669)*x[0]**o+(0.301473785296)*x[0]+(0.610379955695)*x[1]**o+(0.573729399495)*x[1]+(0.225983751859)*x[2]**o+(-0.349530143446)*x[2]
        arg[2,3,1,0]=(-0.647825490424)*x[0]**o+(-0.232573858642)*x[0]+(0.773519085771)*x[1]**o+(-0.224497979868)*x[1]+(-0.681726885952)*x[2]**o+(-0.217968759388)*x[2]
        arg[2,3,1,1]=(0.391888417642)*x[0]**o+(-0.240556139087)*x[0]+(-0.852397797723)*x[1]**o+(0.318342758913)*x[1]+(-0.773804983598)*x[2]**o+(-0.957590491908)*x[2]
        arg[2,3,2,0]=(0.7126582007)*x[0]**o+(0.806022164546)*x[0]+(0.232348951782)*x[1]**o+(-0.985774366344)*x[1]+(-0.365161907541)*x[2]**o+(0.0218977623299)*x[2]
        arg[2,3,2,1]=(-0.775068843872)*x[0]**o+(-0.595291330879)*x[0]+(0.681122481841)*x[1]**o+(0.152514252732)*x[1]+(0.510976191808)*x[2]**o+(-0.69539067488)*x[2]
        arg[2,4,0,0]=(-0.362761749405)*x[0]**o+(0.494347958562)*x[0]+(0.215244694522)*x[1]**o+(-0.527767969217)*x[1]+(-0.115976071059)*x[2]**o+(-0.998233415841)*x[2]
        arg[2,4,0,1]=(0.620291015098)*x[0]**o+(-0.222644358028)*x[0]+(0.0377714785847)*x[1]**o+(0.663978623913)*x[1]+(0.996481523389)*x[2]**o+(-0.689519418095)*x[2]
        arg[2,4,1,0]=(-0.0374746407394)*x[0]**o+(0.3186355021)*x[0]+(-0.704305640533)*x[1]**o+(-0.632264150766)*x[1]+(-0.166691199609)*x[2]**o+(-0.794128441334)*x[2]
        arg[2,4,1,1]=(0.117806334919)*x[0]**o+(-0.717334572919)*x[0]+(0.665341054571)*x[1]**o+(0.354208273372)*x[1]+(0.349883631586)*x[2]**o+(-0.260626035314)*x[2]
        arg[2,4,2,0]=(0.160337827685)*x[0]**o+(-0.92734430881)*x[0]+(0.872112181363)*x[1]**o+(-0.796372970466)*x[1]+(-0.776882702685)*x[2]**o+(0.246816968474)*x[2]
        arg[2,4,2,1]=(-0.317926812571)*x[0]**o+(-0.795779593996)*x[0]+(0.785287724793)*x[1]**o+(0.839033515634)*x[1]+(0.433187943714)*x[2]**o+(0.308175885606)*x[2]
        arg[3,0,0,0]=(-0.555193292291)*x[0]**o+(-0.0280452948283)*x[0]+(0.988455127227)*x[1]**o+(-0.637707060271)*x[1]+(-0.95062571916)*x[2]**o+(-0.563121211553)*x[2]
        arg[3,0,0,1]=(-0.169558534946)*x[0]**o+(0.886096637835)*x[0]+(-0.666307491933)*x[1]**o+(0.650321801985)*x[1]+(-0.268934932123)*x[2]**o+(-0.406617199906)*x[2]
        arg[3,0,1,0]=(0.497351781449)*x[0]**o+(0.768617651685)*x[0]+(0.200962956226)*x[1]**o+(0.694236078954)*x[1]+(0.0323702864321)*x[2]**o+(-0.666046265832)*x[2]
        arg[3,0,1,1]=(0.425151807387)*x[0]**o+(-0.886432438604)*x[0]+(-0.963257203569)*x[1]**o+(-0.698524427437)*x[1]+(0.0165810713355)*x[2]**o+(-0.142193988951)*x[2]
        arg[3,0,2,0]=(0.965746112788)*x[0]**o+(-0.333068915886)*x[0]+(0.331031372002)*x[1]**o+(-0.384004796757)*x[1]+(0.0144121092048)*x[2]**o+(-0.352607264387)*x[2]
        arg[3,0,2,1]=(-0.391557775789)*x[0]**o+(0.127020330586)*x[0]+(0.758832450081)*x[1]**o+(0.49669414469)*x[1]+(0.358403498634)*x[2]**o+(0.184963946351)*x[2]
        arg[3,1,0,0]=(0.978053897464)*x[0]**o+(-0.211787313494)*x[0]+(0.927706482976)*x[1]**o+(-0.451292956408)*x[1]+(0.592824645174)*x[2]**o+(0.896240907546)*x[2]
        arg[3,1,0,1]=(0.0780568875295)*x[0]**o+(0.130098698523)*x[0]+(-0.281871571918)*x[1]**o+(-0.596362849045)*x[1]+(-0.436455789199)*x[2]**o+(-0.657145053021)*x[2]
        arg[3,1,1,0]=(-0.278111817329)*x[0]**o+(-0.627384158565)*x[0]+(-0.995802044048)*x[1]**o+(0.42485189159)*x[1]+(0.483473275475)*x[2]**o+(-0.177928974458)*x[2]
        arg[3,1,1,1]=(0.127721047557)*x[0]**o+(0.93147711978)*x[0]+(-0.350992052283)*x[1]**o+(0.0993773637672)*x[1]+(0.29111744653)*x[2]**o+(-0.821710278883)*x[2]
        arg[3,1,2,0]=(-0.697941918276)*x[0]**o+(-0.585872232852)*x[0]+(0.697967779925)*x[1]**o+(-0.811186381728)*x[1]+(0.708726108998)*x[2]**o+(0.289602027081)*x[2]
        arg[3,1,2,1]=(-0.461424409147)*x[0]**o+(0.335089386385)*x[0]+(-0.964838280406)*x[1]**o+(0.828627561583)*x[1]+(-0.861416281013)*x[2]**o+(0.898354675269)*x[2]
        arg[3,2,0,0]=(0.752543705358)*x[0]**o+(0.164487088496)*x[0]+(0.756164002406)*x[1]**o+(-0.740019514032)*x[1]+(0.107245489678)*x[2]**o+(0.677707700079)*x[2]
        arg[3,2,0,1]=(0.763792557134)*x[0]**o+(0.725507550672)*x[0]+(0.171323949629)*x[1]**o+(-0.287172075258)*x[1]+(-0.94865226715)*x[2]**o+(-0.163093465336)*x[2]
        arg[3,2,1,0]=(-0.327892721048)*x[0]**o+(0.334145897353)*x[0]+(-0.936827293436)*x[1]**o+(0.0822687872593)*x[1]+(0.259794829604)*x[2]**o+(0.376908280759)*x[2]
        arg[3,2,1,1]=(0.595012545955)*x[0]**o+(0.617490014693)*x[0]+(-0.658036388462)*x[1]**o+(-0.921055393527)*x[1]+(-0.257878085344)*x[2]**o+(-0.623113089161)*x[2]
        arg[3,2,2,0]=(0.961175705937)*x[0]**o+(0.656031660842)*x[0]+(0.964101580317)*x[1]**o+(0.824033157042)*x[1]+(0.510244886658)*x[2]**o+(-0.236501162729)*x[2]
        arg[3,2,2,1]=(-0.128405588923)*x[0]**o+(-0.0327141502894)*x[0]+(0.909941485131)*x[1]**o+(0.79947258121)*x[1]+(0.635883050534)*x[2]**o+(0.841997575796)*x[2]
        arg[3,3,0,0]=(-0.906234907643)*x[0]**o+(-0.316076976009)*x[0]+(0.863208664533)*x[1]**o+(-0.968640170523)*x[1]+(0.791637381686)*x[2]**o+(-0.543353462098)*x[2]
        arg[3,3,0,1]=(-0.586709036832)*x[0]**o+(-0.299493216592)*x[0]+(-0.173966725899)*x[1]**o+(0.0971838711677)*x[1]+(0.385662922244)*x[2]**o+(0.288932625368)*x[2]
        arg[3,3,1,0]=(-0.33865322795)*x[0]**o+(0.442176445777)*x[0]+(0.706183231282)*x[1]**o+(0.62528018966)*x[1]+(-0.377746006979)*x[2]**o+(-0.261455124139)*x[2]
        arg[3,3,1,1]=(0.834855553776)*x[0]**o+(0.755005824764)*x[0]+(0.446589332183)*x[1]**o+(0.41883521757)*x[1]+(0.820177207209)*x[2]**o+(0.603811552083)*x[2]
        arg[3,3,2,0]=(-0.624282266193)*x[0]**o+(0.28109086861)*x[0]+(0.44286011655)*x[1]**o+(-0.117876187343)*x[1]+(-0.267660131821)*x[2]**o+(0.880359406159)*x[2]
        arg[3,3,2,1]=(-0.0722687033317)*x[0]**o+(-0.7013147214)*x[0]+(0.00614303249005)*x[1]**o+(-0.658138946692)*x[1]+(0.0557277088337)*x[2]**o+(0.614362766103)*x[2]
        arg[3,4,0,0]=(0.935244440586)*x[0]**o+(-0.0699305155896)*x[0]+(-0.849303362918)*x[1]**o+(0.728062537708)*x[1]+(0.404621498028)*x[2]**o+(0.611486631437)*x[2]
        arg[3,4,0,1]=(0.168117782017)*x[0]**o+(-0.0454396423181)*x[0]+(0.46021270423)*x[1]**o+(0.597343569492)*x[1]+(0.561215321486)*x[2]**o+(0.715771227777)*x[2]
        arg[3,4,1,0]=(-0.502675682461)*x[0]**o+(0.543910335185)*x[0]+(0.721732643536)*x[1]**o+(-0.862126899566)*x[1]+(0.794704104653)*x[2]**o+(-0.284248276797)*x[2]
        arg[3,4,1,1]=(-0.580751817984)*x[0]**o+(0.482838505281)*x[0]+(-0.426066109367)*x[1]**o+(-0.456724382007)*x[1]+(-0.770655426938)*x[2]**o+(-0.226232224569)*x[2]
        arg[3,4,2,0]=(0.199832861603)*x[0]**o+(-0.586239366436)*x[0]+(0.730019631541)*x[1]**o+(0.185265476259)*x[1]+(0.514243351617)*x[2]**o+(-0.757742857839)*x[2]
        arg[3,4,2,1]=(-0.941843068195)*x[0]**o+(0.211732407161)*x[0]+(-0.99705602485)*x[1]**o+(-0.198037742645)*x[1]+(0.154058794178)*x[2]**o+(0.0675962671791)*x[2]
        ref[0,0,0,0]=(0.501470843513)/(o+1.)+(0.0235415380528)
        ref[0,0,0,1]=(-0.495882240293)/(o+1.)+(-0.628005728776)
        ref[0,0,1,0]=(-0.47355259574)/(o+1.)+(0.589958427511)
        ref[0,0,1,1]=(0.241804414345)/(o+1.)+(0.0744968350706)
        ref[0,0,2,0]=(-1.36573294299)/(o+1.)+(0.0250854306796)
        ref[0,0,2,1]=(-1.09689492365)/(o+1.)+(0.203220046558)
        ref[0,1,0,0]=(0.715822269113)/(o+1.)+(-0.513006150237)
        ref[0,1,0,1]=(0.194018305088)/(o+1.)+(0.594196942613)
        ref[0,1,1,0]=(-0.361460970224)/(o+1.)+(0.331424823297)
        ref[0,1,1,1]=(1.07883261716)/(o+1.)+(-0.835316371206)
        ref[0,1,2,0]=(0.734007979175)/(o+1.)+(0.372174974585)
        ref[0,1,2,1]=(-2.64519565923)/(o+1.)+(0.402886980703)
        ref[0,2,0,0]=(0.550037953057)/(o+1.)+(0.804639715144)
        ref[0,2,0,1]=(0.164423071623)/(o+1.)+(0.347820263772)
        ref[0,2,1,0]=(-1.0339269614)/(o+1.)+(-0.46790367599)
        ref[0,2,1,1]=(-0.663398714515)/(o+1.)+(0.341263346566)
        ref[0,2,2,0]=(0.484800065694)/(o+1.)+(-0.343834368812)
        ref[0,2,2,1]=(-0.647576357439)/(o+1.)+(-0.389767349638)
        ref[0,3,0,0]=(0.713628175789)/(o+1.)+(0.534426911657)
        ref[0,3,0,1]=(-1.0186229823)/(o+1.)+(0.168320055464)
        ref[0,3,1,0]=(-0.312627075447)/(o+1.)+(-0.288688061643)
        ref[0,3,1,1]=(0.563110793595)/(o+1.)+(-0.338854681006)
        ref[0,3,2,0]=(-1.6674641169)/(o+1.)+(-0.0630880744491)
        ref[0,3,2,1]=(-0.902373819515)/(o+1.)+(0.146944381666)
        ref[0,4,0,0]=(0.430045772773)/(o+1.)+(-0.326814746492)
        ref[0,4,0,1]=(-0.0329311494305)/(o+1.)+(-0.462763531298)
        ref[0,4,1,0]=(0.600404125575)/(o+1.)+(-0.666588219743)
        ref[0,4,1,1]=(-0.877044936593)/(o+1.)+(-0.213376554573)
        ref[0,4,2,0]=(-0.279718085349)/(o+1.)+(-0.507828107848)
        ref[0,4,2,1]=(1.55865221054)/(o+1.)+(-0.107165734349)
        ref[1,0,0,0]=(-1.53994251168)/(o+1.)+(0.682708891237)
        ref[1,0,0,1]=(0.27426508258)/(o+1.)+(0.311032674718)
        ref[1,0,1,0]=(-1.48856093105)/(o+1.)+(0.244603632945)
        ref[1,0,1,1]=(-0.980725422876)/(o+1.)+(0.0551781371305)
        ref[1,0,2,0]=(-0.123197946419)/(o+1.)+(-0.619047917567)
        ref[1,0,2,1]=(-0.128013307318)/(o+1.)+(-0.288543617977)
        ref[1,1,0,0]=(0.505758208708)/(o+1.)+(-0.604287011596)
        ref[1,1,0,1]=(-0.917894547188)/(o+1.)+(0.215510169161)
        ref[1,1,1,0]=(1.59629389187)/(o+1.)+(0.119123594986)
        ref[1,1,1,1]=(-1.63911850975)/(o+1.)+(-0.15110772397)
        ref[1,1,2,0]=(-0.357420174452)/(o+1.)+(0.946948435557)
        ref[1,1,2,1]=(-0.990173091615)/(o+1.)+(-0.412661055782)
        ref[1,2,0,0]=(-0.956530562442)/(o+1.)+(0.503239856545)
        ref[1,2,0,1]=(-0.0192458507848)/(o+1.)+(0.151130406306)
        ref[1,2,1,0]=(-0.510032040848)/(o+1.)+(1.12420678939)
        ref[1,2,1,1]=(-1.01025981956)/(o+1.)+(-0.483192483145)
        ref[1,2,2,0]=(2.46576915147)/(o+1.)+(-0.0480159570431)
        ref[1,2,2,1]=(-0.868236542197)/(o+1.)+(0.0610754286198)
        ref[1,3,0,0]=(0.873075607662)/(o+1.)+(-0.320365438154)
        ref[1,3,0,1]=(-0.430771579922)/(o+1.)+(0.444216184672)
        ref[1,3,1,0]=(-0.387878423294)/(o+1.)+(0.13604693133)
        ref[1,3,1,1]=(-1.88108112782)/(o+1.)+(-0.194800096716)
        ref[1,3,2,0]=(2.12050210152)/(o+1.)+(-0.128759395251)
        ref[1,3,2,1]=(0.325712098372)/(o+1.)+(0.104952955276)
        ref[1,4,0,0]=(0.924146253016)/(o+1.)+(-0.434307664887)
        ref[1,4,0,1]=(-0.221754040568)/(o+1.)+(1.14558078503)
        ref[1,4,1,0]=(0.0145217628367)/(o+1.)+(-0.43084890137)
        ref[1,4,1,1]=(-1.31753215008)/(o+1.)+(-0.824287026036)
        ref[1,4,2,0]=(-0.697722948118)/(o+1.)+(0.114574044827)
        ref[1,4,2,1]=(0.779339818147)/(o+1.)+(-0.365592981116)
        ref[2,0,0,0]=(0.274952897958)/(o+1.)+(-0.255370912722)
        ref[2,0,0,1]=(-1.10264655401)/(o+1.)+(-0.261708001871)
        ref[2,0,1,0]=(-0.680012578665)/(o+1.)+(-0.106715000464)
        ref[2,0,1,1]=(-0.686793121159)/(o+1.)+(-0.0241878370688)
        ref[2,0,2,0]=(0.942715771856)/(o+1.)+(-0.323717638282)
        ref[2,0,2,1]=(-0.0104862575713)/(o+1.)+(-0.19048054226)
        ref[2,1,0,0]=(0.746926399818)/(o+1.)+(0.117160499248)
        ref[2,1,0,1]=(-1.04171529373)/(o+1.)+(0.0976704487401)
        ref[2,1,1,0]=(0.136795029052)/(o+1.)+(-0.0802084471266)
        ref[2,1,1,1]=(0.896020560833)/(o+1.)+(-0.836960327706)
        ref[2,1,2,0]=(-0.00904194364113)/(o+1.)+(0.245943197498)
        ref[2,1,2,1]=(1.1485457441)/(o+1.)+(0.145957400558)
        ref[2,2,0,0]=(-0.0707799881243)/(o+1.)+(0.0690663884416)
        ref[2,2,0,1]=(1.32117006716)/(o+1.)+(0.227104957348)
        ref[2,2,1,0]=(0.711880376874)/(o+1.)+(-0.58130282817)
        ref[2,2,1,1]=(-0.611981014504)/(o+1.)+(0.303368043702)
        ref[2,2,2,0]=(0.564606893452)/(o+1.)+(-0.674927313599)
        ref[2,2,2,1]=(0.754537066511)/(o+1.)+(0.654108380069)
        ref[2,3,0,0]=(0.168469433796)/(o+1.)+(0.331189805903)
        ref[2,3,0,1]=(1.58905823122)/(o+1.)+(0.262836520673)
        ref[2,3,1,0]=(-0.556033290604)/(o+1.)+(-0.337520298949)
        ref[2,3,1,1]=(-1.23431436368)/(o+1.)+(-0.439901936041)
        ref[2,3,2,0]=(0.579845244941)/(o+1.)+(-0.0789272197338)
        ref[2,3,2,1]=(0.417029829778)/(o+1.)+(-0.569083876513)
        ref[2,4,0,0]=(-0.263493125943)/(o+1.)+(-0.515826713248)
        ref[2,4,0,1]=(1.65454401707)/(o+1.)+(-0.124092576105)
        ref[2,4,1,0]=(-0.908471480882)/(o+1.)+(-0.553878545)
        ref[2,4,1,1]=(1.13303102108)/(o+1.)+(-0.31187616743)
        ref[2,4,2,0]=(0.255567306363)/(o+1.)+(-0.738450155401)
        ref[2,4,2,1]=(0.900548855936)/(o+1.)+(0.175714903622)
        ref[3,0,0,0]=(-0.517363884224)/(o+1.)+(-0.614436783326)
        ref[3,0,0,1]=(-1.104800959)/(o+1.)+(0.564900619957)
        ref[3,0,1,0]=(0.730685024108)/(o+1.)+(0.398403732404)
        ref[3,0,1,1]=(-0.521524324846)/(o+1.)+(-0.863575427496)
        ref[3,0,2,0]=(1.31118959399)/(o+1.)+(-0.534840488515)
        ref[3,0,2,1]=(0.725678172925)/(o+1.)+(0.404339210814)
        ref[3,1,0,0]=(2.49858502561)/(o+1.)+(0.116580318822)
        ref[3,1,0,1]=(-0.640270473588)/(o+1.)+(-0.561704601771)
        ref[3,1,1,0]=(-0.790440585902)/(o+1.)+(-0.190230620716)
        ref[3,1,1,1]=(0.0678464418036)/(o+1.)+(0.104572102332)
        ref[3,1,2,0]=(0.708751970646)/(o+1.)+(-0.553728293749)
        ref[3,1,2,1]=(-2.28767897057)/(o+1.)+(1.03103581162)
        ref[3,2,0,0]=(1.61595319744)/(o+1.)+(0.0510876372718)
        ref[3,2,0,1]=(-0.0135357603866)/(o+1.)+(0.137621005039)
        ref[3,2,1,0]=(-1.00492518488)/(o+1.)+(0.396661482686)
        ref[3,2,1,1]=(-0.320901927851)/(o+1.)+(-0.463339233997)
        ref[3,2,2,0]=(2.43552217291)/(o+1.)+(0.621781827577)
        ref[3,2,2,1]=(1.41741894674)/(o+1.)+(0.804378003359)
        ref[3,3,0,0]=(0.748611138576)/(o+1.)+(-0.914035304315)
        ref[3,3,0,1]=(-0.375012840487)/(o+1.)+(0.0433116399723)
        ref[3,3,1,0]=(-0.0102160036467)/(o+1.)+(0.403000755649)
        ref[3,3,1,1]=(2.10162209317)/(o+1.)+(0.888826297208)
        ref[3,3,2,0]=(-0.449082281464)/(o+1.)+(0.521787043713)
        ref[3,3,2,1]=(-0.0103979620079)/(o+1.)+(-0.372545450994)
        ref[3,4,0,0]=(0.490562575696)/(o+1.)+(0.634809326777)
        ref[3,4,0,1]=(1.18954580773)/(o+1.)+(0.633837577475)
        ref[3,4,1,0]=(1.01376106573)/(o+1.)+(-0.301232420589)
        ref[3,4,1,1]=(-1.77747335429)/(o+1.)+(-0.100059050648)
        ref[3,4,2,0]=(1.44409584476)/(o+1.)+(-0.579358374008)
        ref[3,4,2,1]=(-1.78484029887)/(o+1.)+(0.0406454658476)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5, 3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunction_fromData_Solution_rank0(self):
      """
      tests integral of rank 0 Data on the ReducedFunction

      assumptions: Solution(self.domain) exists
                   self.domain supports integral on ReducedFunction
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=ReducedFunction(self.domain)
      w=Solution(self.domain)
      x=w.getX()
      arg=Data(0,(),w)
      if dim==2:
        arg=(-0.788336609161)*x[0]**o+(-0.46539601764)*x[0]+(0.0970136798824)*x[1]**o+(-0.120665298671)*x[1]
        ref=(-0.691322929278)/(o+1.)+(-0.293030658156)
      else:
        arg=(0.922255261003)*x[0]**o+(0.0862232364707)*x[0]+(-0.94152635039)*x[1]**o+(0.771078486789)*x[1]+(-0.487642191181)*x[2]**o+(0.971754960362)*x[2]
        ref=(-0.506913280568)/(o+1.)+(0.914528341811)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunction_fromData_Solution_rank1(self):
      """
      tests integral of rank 1 Data on the ReducedFunction

      assumptions: Solution(self.domain) exists
                   self.domain supports integral on ReducedFunction
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=ReducedFunction(self.domain)
      w=Solution(self.domain)
      x=w.getX()
      arg=Data(0,(2,),w)
      ref=numpy.zeros((2,),numpy.float_)
      if dim==2:
        arg[0]=(-0.193071015033)*x[0]**o+(-0.281957941612)*x[0]+(0.269334552863)*x[1]**o+(0.0332262192804)*x[1]
        arg[1]=(-0.982634724425)*x[0]**o+(0.872638820737)*x[0]+(0.828028964603)*x[1]**o+(0.70464418136)*x[1]
        ref[0]=(0.0762635378296)/(o+1.)+(-0.124365861166)
        ref[1]=(-0.154605759822)/(o+1.)+(0.788641501048)
      else:
        arg[0]=(0.0746983082972)*x[0]**o+(0.281818436343)*x[0]+(-0.835321029206)*x[1]**o+(-0.349524270524)*x[1]+(-0.471905088601)*x[2]**o+(0.156327167402)*x[2]
        arg[1]=(-0.971935771255)*x[0]**o+(0.943151542057)*x[0]+(-0.60084258169)*x[1]**o+(0.458392855527)*x[1]+(-0.0173384187065)*x[2]**o+(0.538410338758)*x[2]
        ref[0]=(-1.23252780951)/(o+1.)+(0.0443106666103)
        ref[1]=(-1.59011677165)/(o+1.)+(0.969977368171)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunction_fromData_Solution_rank2(self):
      """
      tests integral of rank 2 Data on the ReducedFunction

      assumptions: Solution(self.domain) exists
                   self.domain supports integral on ReducedFunction
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=ReducedFunction(self.domain)
      w=Solution(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5),w)
      ref=numpy.zeros((4, 5),numpy.float_)
      if dim==2:
        arg[0,0]=(0.568783264759)*x[0]**o+(0.690377280868)*x[0]+(-0.689750922806)*x[1]**o+(-0.262828324126)*x[1]
        arg[0,1]=(-0.981657284673)*x[0]**o+(0.37118874852)*x[0]+(0.733992104146)*x[1]**o+(0.350268735699)*x[1]
        arg[0,2]=(0.713336100765)*x[0]**o+(0.737674447814)*x[0]+(0.239606043833)*x[1]**o+(0.814911366356)*x[1]
        arg[0,3]=(0.171706130967)*x[0]**o+(0.56324398491)*x[0]+(0.328884588679)*x[1]**o+(0.13307521808)*x[1]
        arg[0,4]=(-0.309130885049)*x[0]**o+(0.353596646674)*x[0]+(-0.282764686556)*x[1]**o+(-0.56977508596)*x[1]
        arg[1,0]=(0.974136161109)*x[0]**o+(-0.432477120938)*x[0]+(-0.841212164751)*x[1]**o+(-0.429211937256)*x[1]
        arg[1,1]=(0.0303890146585)*x[0]**o+(-0.712865964696)*x[0]+(0.572471431129)*x[1]**o+(-0.194324510922)*x[1]
        arg[1,2]=(-0.757134870981)*x[0]**o+(-0.018314520594)*x[0]+(0.250423290169)*x[1]**o+(-0.876678823729)*x[1]
        arg[1,3]=(-0.351885929068)*x[0]**o+(-0.561258605321)*x[0]+(0.208995575986)*x[1]**o+(0.428437856056)*x[1]
        arg[1,4]=(-0.313383912526)*x[0]**o+(-0.713484404328)*x[0]+(-0.279619274966)*x[1]**o+(0.72420661494)*x[1]
        arg[2,0]=(-0.686382641827)*x[0]**o+(0.678578408675)*x[0]+(-0.696532353786)*x[1]**o+(-0.154644995634)*x[1]
        arg[2,1]=(0.368548759951)*x[0]**o+(-0.674856247769)*x[0]+(0.908734263899)*x[1]**o+(0.331173572353)*x[1]
        arg[2,2]=(-0.1279136556)*x[0]**o+(-0.161474840876)*x[0]+(-0.775009015881)*x[1]**o+(-0.689552100815)*x[1]
        arg[2,3]=(-0.638450048485)*x[0]**o+(0.755485009392)*x[0]+(-0.793363159431)*x[1]**o+(0.255883452369)*x[1]
        arg[2,4]=(-0.831891512113)*x[0]**o+(-0.811678751836)*x[0]+(-0.487866621558)*x[1]**o+(0.240951890349)*x[1]
        arg[3,0]=(0.448249735424)*x[0]**o+(0.0809467993259)*x[0]+(0.232269153692)*x[1]**o+(0.477512628661)*x[1]
        arg[3,1]=(0.494093407104)*x[0]**o+(0.196761586998)*x[0]+(-0.702155296644)*x[1]**o+(-0.748641693731)*x[1]
        arg[3,2]=(0.432370067164)*x[0]**o+(-0.365671005874)*x[0]+(0.452009309356)*x[1]**o+(-0.951328172558)*x[1]
        arg[3,3]=(-0.993290240028)*x[0]**o+(0.172534714933)*x[0]+(0.00917693958413)*x[1]**o+(0.417515657393)*x[1]
        arg[3,4]=(0.541299019375)*x[0]**o+(0.479118172057)*x[0]+(-0.502475287154)*x[1]**o+(-0.04397899716)*x[1]
        ref[0,0]=(-0.120967658047)/(o+1.)+(0.213774478371)
        ref[0,1]=(-0.247665180527)/(o+1.)+(0.360728742109)
        ref[0,2]=(0.952942144598)/(o+1.)+(0.776292907085)
        ref[0,3]=(0.500590719646)/(o+1.)+(0.348159601495)
        ref[0,4]=(-0.591895571605)/(o+1.)+(-0.108089219643)
        ref[1,0]=(0.132923996358)/(o+1.)+(-0.430844529097)
        ref[1,1]=(0.602860445788)/(o+1.)+(-0.453595237809)
        ref[1,2]=(-0.506711580812)/(o+1.)+(-0.447496672161)
        ref[1,3]=(-0.142890353081)/(o+1.)+(-0.066410374632)
        ref[1,4]=(-0.593003187492)/(o+1.)+(0.00536110530621)
        ref[2,0]=(-1.38291499561)/(o+1.)+(0.261966706521)
        ref[2,1]=(1.27728302385)/(o+1.)+(-0.171841337708)
        ref[2,2]=(-0.902922671481)/(o+1.)+(-0.425513470846)
        ref[2,3]=(-1.43181320792)/(o+1.)+(0.505684230881)
        ref[2,4]=(-1.31975813367)/(o+1.)+(-0.285363430743)
        ref[3,0]=(0.680518889117)/(o+1.)+(0.279229713993)
        ref[3,1]=(-0.20806188954)/(o+1.)+(-0.275940053366)
        ref[3,2]=(0.88437937652)/(o+1.)+(-0.658499589216)
        ref[3,3]=(-0.984113300443)/(o+1.)+(0.295025186163)
        ref[3,4]=(0.0388237322207)/(o+1.)+(0.217569587448)
      else:
        arg[0,0]=(-0.866239193113)*x[0]**o+(0.479559390215)*x[0]+(0.512920648125)*x[1]**o+(-0.504775214511)*x[1]+(0.462210441279)*x[2]**o+(0.000249131516168)*x[2]
        arg[0,1]=(-0.11397206495)*x[0]**o+(0.0762933475692)*x[0]+(0.347716642793)*x[1]**o+(0.373262578958)*x[1]+(0.965468442129)*x[2]**o+(-0.743205128452)*x[2]
        arg[0,2]=(-0.214519408107)*x[0]**o+(-0.663675446512)*x[0]+(-0.424102625804)*x[1]**o+(-0.426644021856)*x[1]+(-0.0731662128704)*x[2]**o+(-0.375185307096)*x[2]
        arg[0,3]=(0.599595484989)*x[0]**o+(-0.303187098474)*x[0]+(0.278524134467)*x[1]**o+(-0.579432425528)*x[1]+(0.006698672684)*x[2]**o+(-0.820721120517)*x[2]
        arg[0,4]=(0.538847126305)*x[0]**o+(0.58093792853)*x[0]+(-0.21189097773)*x[1]**o+(0.637101424777)*x[1]+(0.0155994636183)*x[2]**o+(0.488675370945)*x[2]
        arg[1,0]=(-0.775581481064)*x[0]**o+(-0.968381856562)*x[0]+(-0.660755920498)*x[1]**o+(0.746481830633)*x[1]+(-0.300061615221)*x[2]**o+(-0.720031101537)*x[2]
        arg[1,1]=(0.990001351802)*x[0]**o+(-0.168997300944)*x[0]+(0.0134555819367)*x[1]**o+(0.969355591009)*x[1]+(-0.347553438064)*x[2]**o+(0.648537935984)*x[2]
        arg[1,2]=(-0.327760535516)*x[0]**o+(0.517489199537)*x[0]+(-0.0589046196438)*x[1]**o+(-0.893401439507)*x[1]+(-0.0821634552571)*x[2]**o+(0.861266440343)*x[2]
        arg[1,3]=(-0.808782893446)*x[0]**o+(-0.196363777506)*x[0]+(0.623837845847)*x[1]**o+(-0.05392424847)*x[1]+(0.616011657554)*x[2]**o+(0.800682023434)*x[2]
        arg[1,4]=(-0.425073222968)*x[0]**o+(-0.635248695204)*x[0]+(0.947222621311)*x[1]**o+(0.253198813179)*x[1]+(-0.907568459712)*x[2]**o+(-0.513049744062)*x[2]
        arg[2,0]=(-0.103867107528)*x[0]**o+(0.893031214224)*x[0]+(0.151292631908)*x[1]**o+(-0.407189878657)*x[1]+(0.618120612304)*x[2]**o+(0.983992964931)*x[2]
        arg[2,1]=(0.310432646627)*x[0]**o+(0.235143842057)*x[0]+(0.729683627093)*x[1]**o+(0.891778928523)*x[1]+(-0.148927367367)*x[2]**o+(0.913403960844)*x[2]
        arg[2,2]=(0.610293692912)*x[0]**o+(-0.28497731539)*x[0]+(-0.926462472232)*x[1]**o+(-0.427657211191)*x[1]+(0.467905160751)*x[2]**o+(0.195578740767)*x[2]
        arg[2,3]=(0.854869452114)*x[0]**o+(-0.752530632831)*x[0]+(0.0128220565485)*x[1]**o+(-0.726013651836)*x[1]+(-0.0979888078519)*x[2]**o+(-0.658955703807)*x[2]
        arg[2,4]=(0.536447848289)*x[0]**o+(-0.658205406579)*x[0]+(0.66427484293)*x[1]**o+(-0.250014589496)*x[1]+(-0.0654257159073)*x[2]**o+(0.538196496913)*x[2]
        arg[3,0]=(-0.514665034974)*x[0]**o+(-0.278189747423)*x[0]+(0.402996329453)*x[1]**o+(-0.0362969716175)*x[1]+(-0.707890815004)*x[2]**o+(-0.298762865096)*x[2]
        arg[3,1]=(0.7342439327)*x[0]**o+(-0.387336714116)*x[0]+(-0.67888634151)*x[1]**o+(0.647322640846)*x[1]+(-0.0346274738413)*x[2]**o+(0.985950301913)*x[2]
        arg[3,2]=(0.687555064124)*x[0]**o+(0.345506248144)*x[0]+(0.658692258064)*x[1]**o+(0.810796031447)*x[1]+(0.102608006398)*x[2]**o+(0.349877767777)*x[2]
        arg[3,3]=(-0.456241806837)*x[0]**o+(-0.181224535865)*x[0]+(0.401842456169)*x[1]**o+(-0.413936620519)*x[1]+(0.436794604974)*x[2]**o+(0.95051042082)*x[2]
        arg[3,4]=(-0.330739393408)*x[0]**o+(-0.261876658969)*x[0]+(0.287474017623)*x[1]**o+(0.463691723118)*x[1]+(0.757597823206)*x[2]**o+(0.49336246207)*x[2]
        ref[0,0]=(0.108891896292)/(o+1.)+(-0.01248334639)
        ref[0,1]=(1.19921301997)/(o+1.)+(-0.146824600962)
        ref[0,2]=(-0.711788246781)/(o+1.)+(-0.732752387732)
        ref[0,3]=(0.88481829214)/(o+1.)+(-0.85167032226)
        ref[0,4]=(0.342555612194)/(o+1.)+(0.853357362126)
        ref[1,0]=(-1.73639901678)/(o+1.)+(-0.470965563733)
        ref[1,1]=(0.655903495675)/(o+1.)+(0.724448113024)
        ref[1,2]=(-0.468828610417)/(o+1.)+(0.242677100186)
        ref[1,3]=(0.431066609955)/(o+1.)+(0.275196998729)
        ref[1,4]=(-0.385419061368)/(o+1.)+(-0.447549813044)
        ref[2,0]=(0.665546136685)/(o+1.)+(0.734917150249)
        ref[2,1]=(0.891188906354)/(o+1.)+(1.02016336571)
        ref[2,2]=(0.151736381432)/(o+1.)+(-0.258527892907)
        ref[2,3]=(0.76970270081)/(o+1.)+(-1.06874999424)
        ref[2,4]=(1.13529697531)/(o+1.)+(-0.185011749581)
        ref[3,0]=(-0.819559520525)/(o+1.)+(-0.306624792068)
        ref[3,1]=(0.020730117348)/(o+1.)+(0.622968114321)
        ref[3,2]=(1.44885532859)/(o+1.)+(0.753090023684)
        ref[3,3]=(0.382395254306)/(o+1.)+(0.177674632218)
        ref[3,4]=(0.714332447421)/(o+1.)+(0.34758876311)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunction_fromData_Solution_rank3(self):
      """
      tests integral of rank 3 Data on the ReducedFunction

      assumptions: Solution(self.domain) exists
                   self.domain supports integral on ReducedFunction
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=ReducedFunction(self.domain)
      w=Solution(self.domain)
      x=w.getX()
      arg=Data(0,(6, 2, 2),w)
      ref=numpy.zeros((6, 2, 2),numpy.float_)
      if dim==2:
        arg[0,0,0]=(0.779261125448)*x[0]**o+(-0.640503838419)*x[0]+(0.589085035572)*x[1]**o+(-0.0426687544445)*x[1]
        arg[0,0,1]=(-0.419361292473)*x[0]**o+(0.0850684591011)*x[0]+(-0.440758336688)*x[1]**o+(0.151015611594)*x[1]
        arg[0,1,0]=(-0.727043966709)*x[0]**o+(0.711730063365)*x[0]+(0.657759819512)*x[1]**o+(-0.210981156609)*x[1]
        arg[0,1,1]=(-0.276491768644)*x[0]**o+(-0.324865684188)*x[0]+(-0.921856877627)*x[1]**o+(-0.246833902926)*x[1]
        arg[1,0,0]=(0.720541615152)*x[0]**o+(-0.0406851820986)*x[0]+(0.33976001267)*x[1]**o+(0.937989079378)*x[1]
        arg[1,0,1]=(0.239272962875)*x[0]**o+(0.482026352856)*x[0]+(0.159148474482)*x[1]**o+(0.680831564145)*x[1]
        arg[1,1,0]=(0.55170300456)*x[0]**o+(-0.261357073238)*x[0]+(0.298185649596)*x[1]**o+(-0.540598468483)*x[1]
        arg[1,1,1]=(0.508262701362)*x[0]**o+(-0.531692643817)*x[0]+(0.151424571021)*x[1]**o+(-0.77488755481)*x[1]
        arg[2,0,0]=(0.265201389846)*x[0]**o+(0.377300848247)*x[0]+(0.0402148688087)*x[1]**o+(0.0263806695524)*x[1]
        arg[2,0,1]=(0.321397565411)*x[0]**o+(-0.277650913698)*x[0]+(0.909430556637)*x[1]**o+(-0.43244155765)*x[1]
        arg[2,1,0]=(-0.671417887851)*x[0]**o+(-0.772328065901)*x[0]+(0.0896878386442)*x[1]**o+(0.584773963567)*x[1]
        arg[2,1,1]=(0.318995094412)*x[0]**o+(-0.396578444213)*x[0]+(-0.627767362562)*x[1]**o+(-0.588717342984)*x[1]
        arg[3,0,0]=(-0.154999830633)*x[0]**o+(-0.263434767944)*x[0]+(0.0704862479101)*x[1]**o+(0.216237505495)*x[1]
        arg[3,0,1]=(0.956868643095)*x[0]**o+(-0.0331644426873)*x[0]+(-0.236340025853)*x[1]**o+(-0.814705425837)*x[1]
        arg[3,1,0]=(-0.477985064084)*x[0]**o+(-0.368896833289)*x[0]+(0.208122873207)*x[1]**o+(-0.566525593432)*x[1]
        arg[3,1,1]=(-0.299257275248)*x[0]**o+(0.937022356746)*x[0]+(0.671724793932)*x[1]**o+(0.80705820341)*x[1]
        arg[4,0,0]=(0.857855249735)*x[0]**o+(0.537756541143)*x[0]+(0.611784225987)*x[1]**o+(0.944061991186)*x[1]
        arg[4,0,1]=(0.825779290308)*x[0]**o+(-0.636461636739)*x[0]+(-0.382007450888)*x[1]**o+(0.170178262303)*x[1]
        arg[4,1,0]=(0.275666857657)*x[0]**o+(-0.404983003194)*x[0]+(0.476932065223)*x[1]**o+(0.931645726493)*x[1]
        arg[4,1,1]=(0.956182179652)*x[0]**o+(0.543482655321)*x[0]+(0.60492494312)*x[1]**o+(0.384497838341)*x[1]
        arg[5,0,0]=(0.476937368598)*x[0]**o+(-0.588434801718)*x[0]+(0.865080125217)*x[1]**o+(-0.954493375041)*x[1]
        arg[5,0,1]=(0.151850214007)*x[0]**o+(0.252651124764)*x[0]+(-0.570132867968)*x[1]**o+(-0.0885902498736)*x[1]
        arg[5,1,0]=(-0.69330471578)*x[0]**o+(0.165874518143)*x[0]+(-0.610503496045)*x[1]**o+(-0.871951964138)*x[1]
        arg[5,1,1]=(-0.539191264945)*x[0]**o+(0.558474736927)*x[0]+(0.752045569996)*x[1]**o+(-0.5305308467)*x[1]
        ref[0,0,0]=(1.36834616102)/(o+1.)+(-0.341586296432)
        ref[0,0,1]=(-0.860119629161)/(o+1.)+(0.118042035348)
        ref[0,1,0]=(-0.0692841471964)/(o+1.)+(0.250374453378)
        ref[0,1,1]=(-1.19834864627)/(o+1.)+(-0.285849793557)
        ref[1,0,0]=(1.06030162782)/(o+1.)+(0.448651948639)
        ref[1,0,1]=(0.398421437357)/(o+1.)+(0.5814289585)
        ref[1,1,0]=(0.849888654155)/(o+1.)+(-0.400977770861)
        ref[1,1,1]=(0.659687272383)/(o+1.)+(-0.653290099314)
        ref[2,0,0]=(0.305416258655)/(o+1.)+(0.2018407589)
        ref[2,0,1]=(1.23082812205)/(o+1.)+(-0.355046235674)
        ref[2,1,0]=(-0.581730049206)/(o+1.)+(-0.0937770511668)
        ref[2,1,1]=(-0.30877226815)/(o+1.)+(-0.492647893598)
        ref[3,0,0]=(-0.0845135827233)/(o+1.)+(-0.0235986312242)
        ref[3,0,1]=(0.720528617241)/(o+1.)+(-0.423934934262)
        ref[3,1,0]=(-0.269862190877)/(o+1.)+(-0.46771121336)
        ref[3,1,1]=(0.372467518684)/(o+1.)+(0.872040280078)
        ref[4,0,0]=(1.46963947572)/(o+1.)+(0.740909266165)
        ref[4,0,1]=(0.44377183942)/(o+1.)+(-0.233141687218)
        ref[4,1,0]=(0.75259892288)/(o+1.)+(0.26333136165)
        ref[4,1,1]=(1.56110712277)/(o+1.)+(0.463990246831)
        ref[5,0,0]=(1.34201749382)/(o+1.)+(-0.771464088379)
        ref[5,0,1]=(-0.418282653961)/(o+1.)+(0.0820304374451)
        ref[5,1,0]=(-1.30380821182)/(o+1.)+(-0.353038722997)
        ref[5,1,1]=(0.212854305051)/(o+1.)+(0.0139719451134)
      else:
        arg[0,0,0]=(-0.825926001774)*x[0]**o+(0.418049316138)*x[0]+(0.668695279849)*x[1]**o+(-0.7323605258)*x[1]+(-0.637717742224)*x[2]**o+(0.257065974143)*x[2]
        arg[0,0,1]=(-0.169779448443)*x[0]**o+(-0.648612931029)*x[0]+(-0.643814352995)*x[1]**o+(0.396704595574)*x[1]+(-0.248766540261)*x[2]**o+(-0.736410673282)*x[2]
        arg[0,1,0]=(-0.371929441192)*x[0]**o+(0.438661796459)*x[0]+(-0.893450426264)*x[1]**o+(0.643494077636)*x[1]+(0.252099102122)*x[2]**o+(-0.0857798075076)*x[2]
        arg[0,1,1]=(0.336251711034)*x[0]**o+(-0.317426111079)*x[0]+(-0.801221986791)*x[1]**o+(0.281123033042)*x[1]+(0.921153543163)*x[2]**o+(0.497003835923)*x[2]
        arg[1,0,0]=(0.40897311872)*x[0]**o+(0.326894801733)*x[0]+(0.216268807497)*x[1]**o+(0.946235364974)*x[1]+(-0.480995485096)*x[2]**o+(0.869772937982)*x[2]
        arg[1,0,1]=(0.108715554449)*x[0]**o+(0.0847529231253)*x[0]+(0.866120965844)*x[1]**o+(0.43989772616)*x[1]+(0.67962061792)*x[2]**o+(0.90311988632)*x[2]
        arg[1,1,0]=(0.28571484173)*x[0]**o+(0.593050819853)*x[0]+(-0.384864010146)*x[1]**o+(-0.57920566259)*x[1]+(-0.304024223769)*x[2]**o+(-0.0279518764989)*x[2]
        arg[1,1,1]=(0.249126658892)*x[0]**o+(0.243355410215)*x[0]+(0.154816978135)*x[1]**o+(-0.260343533153)*x[1]+(-0.973133787852)*x[2]**o+(0.752545508531)*x[2]
        arg[2,0,0]=(0.103612497259)*x[0]**o+(0.0919234144108)*x[0]+(0.416394965635)*x[1]**o+(-0.0132359939268)*x[1]+(-0.522775855178)*x[2]**o+(-0.985811066906)*x[2]
        arg[2,0,1]=(-0.592641488144)*x[0]**o+(0.703703144485)*x[0]+(-0.0149046721584)*x[1]**o+(0.220368283581)*x[1]+(-0.656858164281)*x[2]**o+(0.165951102113)*x[2]
        arg[2,1,0]=(0.309492848012)*x[0]**o+(-0.301037623847)*x[0]+(-0.224814259669)*x[1]**o+(-0.950794406688)*x[1]+(-0.296088293363)*x[2]**o+(-0.712930293168)*x[2]
        arg[2,1,1]=(0.0592633594166)*x[0]**o+(-0.179007619512)*x[0]+(-0.922169690831)*x[1]**o+(-0.778228075089)*x[1]+(-0.135749453777)*x[2]**o+(-0.146265051618)*x[2]
        arg[3,0,0]=(-0.71773633584)*x[0]**o+(0.471551385738)*x[0]+(-0.041415831943)*x[1]**o+(0.343380640343)*x[1]+(-0.211821446187)*x[2]**o+(-0.661201893183)*x[2]
        arg[3,0,1]=(-0.159071045836)*x[0]**o+(0.216715127143)*x[0]+(-0.846847485213)*x[1]**o+(0.662226888693)*x[1]+(0.050806337433)*x[2]**o+(0.744071347051)*x[2]
        arg[3,1,0]=(-0.363431877105)*x[0]**o+(-0.461513470653)*x[0]+(-0.130947882739)*x[1]**o+(-0.624729599981)*x[1]+(0.778572567699)*x[2]**o+(0.324584316744)*x[2]
        arg[3,1,1]=(0.395432262346)*x[0]**o+(0.826450301981)*x[0]+(0.176721657975)*x[1]**o+(-0.370954952405)*x[1]+(-0.708437370609)*x[2]**o+(-0.150075114354)*x[2]
        arg[4,0,0]=(-0.18662583335)*x[0]**o+(-0.0120831234457)*x[0]+(0.805686139762)*x[1]**o+(0.30195413136)*x[1]+(0.648067854031)*x[2]**o+(0.28073240632)*x[2]
        arg[4,0,1]=(-0.913625644854)*x[0]**o+(-0.264637827098)*x[0]+(-0.6531590441)*x[1]**o+(0.396422722623)*x[1]+(0.280591271146)*x[2]**o+(0.498346191707)*x[2]
        arg[4,1,0]=(0.208029597907)*x[0]**o+(-0.295747909396)*x[0]+(-0.541509895186)*x[1]**o+(-0.511394562575)*x[1]+(-0.459197562982)*x[2]**o+(-0.749433614666)*x[2]
        arg[4,1,1]=(-0.29700028905)*x[0]**o+(-0.590932598278)*x[0]+(0.36219700213)*x[1]**o+(0.482759242709)*x[1]+(-0.135696240971)*x[2]**o+(0.49658807849)*x[2]
        arg[5,0,0]=(-0.800345624219)*x[0]**o+(0.259709746207)*x[0]+(0.480073493816)*x[1]**o+(0.499332550127)*x[1]+(0.691794186846)*x[2]**o+(-0.319369042249)*x[2]
        arg[5,0,1]=(0.300879052118)*x[0]**o+(-0.985396817441)*x[0]+(0.971136879002)*x[1]**o+(0.806560091745)*x[1]+(0.361308859099)*x[2]**o+(0.264277089732)*x[2]
        arg[5,1,0]=(0.271857906548)*x[0]**o+(0.338260411675)*x[0]+(-0.170292665803)*x[1]**o+(0.675491566142)*x[1]+(0.897921755421)*x[2]**o+(-0.829233864443)*x[2]
        arg[5,1,1]=(0.673366845241)*x[0]**o+(0.121940550422)*x[0]+(0.116054155545)*x[1]**o+(0.0456556938689)*x[1]+(-0.978127590673)*x[2]**o+(0.180985351995)*x[2]
        ref[0,0,0]=(-0.794948464148)/(o+1.)+(-0.0286226177598)
        ref[0,0,1]=(-1.0623603417)/(o+1.)+(-0.494159504368)
        ref[0,1,0]=(-1.01328076533)/(o+1.)+(0.498188033294)
        ref[0,1,1]=(0.456183267406)/(o+1.)+(0.230350378943)
        ref[1,0,0]=(0.144246441122)/(o+1.)+(1.07145155234)
        ref[1,0,1]=(1.65445713821)/(o+1.)+(0.713885267803)
        ref[1,1,0]=(-0.403173392185)/(o+1.)+(-0.0070533596178)
        ref[1,1,1]=(-0.569190150825)/(o+1.)+(0.367778692797)
        ref[2,0,0]=(-0.00276839228366)/(o+1.)+(-0.453561823211)
        ref[2,0,1]=(-1.26440432458)/(o+1.)+(0.545011265089)
        ref[2,1,0]=(-0.21140970502)/(o+1.)+(-0.982381161851)
        ref[2,1,1]=(-0.998655785191)/(o+1.)+(-0.55175037311)
        ref[3,0,0]=(-0.97097361397)/(o+1.)+(0.0768650664488)
        ref[3,0,1]=(-0.955112193616)/(o+1.)+(0.811506681444)
        ref[3,1,0]=(0.284192807855)/(o+1.)+(-0.380829376946)
        ref[3,1,1]=(-0.136283450288)/(o+1.)+(0.152710117611)
        ref[4,0,0]=(1.26712816044)/(o+1.)+(0.285301707117)
        ref[4,0,1]=(-1.28619341781)/(o+1.)+(0.315065543616)
        ref[4,1,0]=(-0.792677860261)/(o+1.)+(-0.778288043318)
        ref[4,1,1]=(-0.0704995278906)/(o+1.)+(0.194207361461)
        ref[5,0,0]=(0.371522056443)/(o+1.)+(0.219836627043)
        ref[5,0,1]=(1.63332479022)/(o+1.)+(0.0427201820182)
        ref[5,1,0]=(0.999486996166)/(o+1.)+(0.0922590566873)
        ref[5,1,1]=(-0.188706589887)/(o+1.)+(0.174290798143)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunction_fromData_Solution_rank4(self):
      """
      tests integral of rank 4 Data on the ReducedFunction

      assumptions: Solution(self.domain) exists
                   self.domain supports integral on ReducedFunction
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=ReducedFunction(self.domain)
      w=Solution(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5, 3, 2),w)
      ref=numpy.zeros((4, 5, 3, 2),numpy.float_)
      if dim==2:
        arg[0,0,0,0]=(0.647932589381)*x[0]**o+(-0.092686685918)*x[0]+(0.167888464541)*x[1]**o+(0.698445840627)*x[1]
        arg[0,0,0,1]=(0.413204959658)*x[0]**o+(0.772926029512)*x[0]+(-0.63702948026)*x[1]**o+(0.608392823861)*x[1]
        arg[0,0,1,0]=(0.303506932052)*x[0]**o+(0.650309664063)*x[0]+(-0.805238344485)*x[1]**o+(-0.62971680995)*x[1]
        arg[0,0,1,1]=(-0.205701190736)*x[0]**o+(-0.0834042840814)*x[0]+(-0.525839398431)*x[1]**o+(0.779973223005)*x[1]
        arg[0,0,2,0]=(-0.97954633631)*x[0]**o+(-0.298973871114)*x[0]+(0.858022652298)*x[1]**o+(0.667671796632)*x[1]
        arg[0,0,2,1]=(-0.790133712345)*x[0]**o+(-0.733743414808)*x[0]+(0.337185148145)*x[1]**o+(-0.927853051244)*x[1]
        arg[0,1,0,0]=(0.617198112458)*x[0]**o+(0.70438849416)*x[0]+(-0.861661596364)*x[1]**o+(-0.251796116219)*x[1]
        arg[0,1,0,1]=(0.37107453636)*x[0]**o+(-0.490440661153)*x[0]+(0.849252675242)*x[1]**o+(-0.896547775077)*x[1]
        arg[0,1,1,0]=(0.375080744007)*x[0]**o+(-0.609557663503)*x[0]+(-0.930846980977)*x[1]**o+(0.0623096843381)*x[1]
        arg[0,1,1,1]=(-0.809863615805)*x[0]**o+(-0.999724521803)*x[0]+(0.493027030633)*x[1]**o+(-0.672546253752)*x[1]
        arg[0,1,2,0]=(0.219294077412)*x[0]**o+(-0.7854819847)*x[0]+(-0.353026321798)*x[1]**o+(-0.965348418509)*x[1]
        arg[0,1,2,1]=(0.177236321722)*x[0]**o+(-0.860491592895)*x[0]+(-0.458724268899)*x[1]**o+(-0.476508229653)*x[1]
        arg[0,2,0,0]=(0.0760611767074)*x[0]**o+(-0.00296398964042)*x[0]+(-0.972843039436)*x[1]**o+(-0.0422057743881)*x[1]
        arg[0,2,0,1]=(-0.926737777701)*x[0]**o+(-0.895791593378)*x[0]+(-0.776529150964)*x[1]**o+(0.557115706978)*x[1]
        arg[0,2,1,0]=(0.353264702858)*x[0]**o+(0.632583933138)*x[0]+(-0.354112684469)*x[1]**o+(0.0541346174117)*x[1]
        arg[0,2,1,1]=(-0.124181989217)*x[0]**o+(-0.803262099858)*x[0]+(0.733246361984)*x[1]**o+(0.0639821744038)*x[1]
        arg[0,2,2,0]=(-0.446733845136)*x[0]**o+(-0.187373171381)*x[0]+(-0.312309043572)*x[1]**o+(0.830789380477)*x[1]
        arg[0,2,2,1]=(0.403889932995)*x[0]**o+(-0.243474625233)*x[0]+(0.35707779102)*x[1]**o+(0.29725421988)*x[1]
        arg[0,3,0,0]=(-0.856376546696)*x[0]**o+(0.448431504377)*x[0]+(-0.535187100108)*x[1]**o+(-0.90994242894)*x[1]
        arg[0,3,0,1]=(-0.321475754775)*x[0]**o+(0.0389358177737)*x[0]+(-0.0949733279735)*x[1]**o+(0.921937240884)*x[1]
        arg[0,3,1,0]=(0.591276336387)*x[0]**o+(-0.598236438303)*x[0]+(-0.829829995185)*x[1]**o+(0.539669346291)*x[1]
        arg[0,3,1,1]=(-0.944726366074)*x[0]**o+(-0.796928273271)*x[0]+(-0.424606656681)*x[1]**o+(0.35341306063)*x[1]
        arg[0,3,2,0]=(-0.850914599406)*x[0]**o+(0.791179202404)*x[0]+(-0.0220845565851)*x[1]**o+(0.594756136957)*x[1]
        arg[0,3,2,1]=(0.355862680641)*x[0]**o+(0.134261680432)*x[0]+(0.477798827075)*x[1]**o+(0.508452338499)*x[1]
        arg[0,4,0,0]=(0.481974944781)*x[0]**o+(0.259271175551)*x[0]+(0.953827692515)*x[1]**o+(0.364160709544)*x[1]
        arg[0,4,0,1]=(-0.316627315143)*x[0]**o+(-0.856616735533)*x[0]+(0.308508314695)*x[1]**o+(-0.677602758367)*x[1]
        arg[0,4,1,0]=(-0.76069824745)*x[0]**o+(0.282351651183)*x[0]+(-0.15964276552)*x[1]**o+(-0.344276428838)*x[1]
        arg[0,4,1,1]=(9.00669303738e-05)*x[0]**o+(0.948498888115)*x[0]+(0.690370267333)*x[1]**o+(0.619119188327)*x[1]
        arg[0,4,2,0]=(0.957692561039)*x[0]**o+(-0.147005436631)*x[0]+(-0.177840574215)*x[1]**o+(-0.28259452614)*x[1]
        arg[0,4,2,1]=(-0.431771943451)*x[0]**o+(-0.158132166651)*x[0]+(-0.134385316147)*x[1]**o+(0.751243590835)*x[1]
        arg[1,0,0,0]=(-0.816841420645)*x[0]**o+(0.55608373851)*x[0]+(-0.885838014201)*x[1]**o+(-0.467159163867)*x[1]
        arg[1,0,0,1]=(-0.543361117401)*x[0]**o+(0.329490019675)*x[0]+(0.942703992489)*x[1]**o+(-0.261273838803)*x[1]
        arg[1,0,1,0]=(0.865108106801)*x[0]**o+(-0.376972427024)*x[0]+(-0.0257304693919)*x[1]**o+(0.694210909726)*x[1]
        arg[1,0,1,1]=(0.428354498121)*x[0]**o+(0.962704601892)*x[0]+(-0.442636806139)*x[1]**o+(0.424724924149)*x[1]
        arg[1,0,2,0]=(-0.157277242299)*x[0]**o+(0.487588443251)*x[0]+(-0.813089096341)*x[1]**o+(0.0406316718829)*x[1]
        arg[1,0,2,1]=(0.345322702588)*x[0]**o+(0.224429042659)*x[0]+(-0.24109926586)*x[1]**o+(0.433141241931)*x[1]
        arg[1,1,0,0]=(0.956901263519)*x[0]**o+(0.48418062914)*x[0]+(0.14089030013)*x[1]**o+(-0.492352201717)*x[1]
        arg[1,1,0,1]=(-0.679781695411)*x[0]**o+(0.29857196636)*x[0]+(-0.234968984185)*x[1]**o+(0.114575919411)*x[1]
        arg[1,1,1,0]=(0.488378272934)*x[0]**o+(-0.358508327343)*x[0]+(-0.337958846849)*x[1]**o+(0.189303452912)*x[1]
        arg[1,1,1,1]=(-0.506993859574)*x[0]**o+(-0.682478021258)*x[0]+(-0.540344338899)*x[1]**o+(0.366965535976)*x[1]
        arg[1,1,2,0]=(-0.681151638572)*x[0]**o+(0.790837550009)*x[0]+(-0.810785716023)*x[1]**o+(-0.00739097962416)*x[1]
        arg[1,1,2,1]=(-0.355290573677)*x[0]**o+(-0.263294580108)*x[0]+(0.10469018402)*x[1]**o+(-0.650388283116)*x[1]
        arg[1,2,0,0]=(0.906212689905)*x[0]**o+(-0.673956103538)*x[0]+(0.305335278934)*x[1]**o+(-0.0805949628719)*x[1]
        arg[1,2,0,1]=(-0.98311373852)*x[0]**o+(-0.165316704542)*x[0]+(0.197637770603)*x[1]**o+(0.647989184342)*x[1]
        arg[1,2,1,0]=(0.415990736251)*x[0]**o+(-0.710476520822)*x[0]+(0.791512786296)*x[1]**o+(-0.10853188742)*x[1]
        arg[1,2,1,1]=(0.04337234954)*x[0]**o+(0.595650094449)*x[0]+(0.228098665648)*x[1]**o+(-0.868930899722)*x[1]
        arg[1,2,2,0]=(-0.791842482688)*x[0]**o+(-0.611246387621)*x[0]+(0.0504978789412)*x[1]**o+(0.624626300694)*x[1]
        arg[1,2,2,1]=(0.0207375595263)*x[0]**o+(0.560638072838)*x[0]+(0.44126312508)*x[1]**o+(0.169151353084)*x[1]
        arg[1,3,0,0]=(-0.0512069089279)*x[0]**o+(0.51110271913)*x[0]+(0.0785633458626)*x[1]**o+(0.836821453174)*x[1]
        arg[1,3,0,1]=(0.185737468851)*x[0]**o+(0.516071031227)*x[0]+(0.0995388802342)*x[1]**o+(0.25527909092)*x[1]
        arg[1,3,1,0]=(0.318240681435)*x[0]**o+(-0.442725217398)*x[0]+(0.533066616909)*x[1]**o+(0.679570707339)*x[1]
        arg[1,3,1,1]=(0.353367397644)*x[0]**o+(-0.396487189861)*x[0]+(-0.117773711001)*x[1]**o+(0.885897675293)*x[1]
        arg[1,3,2,0]=(-0.647645988694)*x[0]**o+(-0.755258662164)*x[0]+(0.288300257472)*x[1]**o+(0.505342385389)*x[1]
        arg[1,3,2,1]=(0.000781386924262)*x[0]**o+(0.190244787656)*x[0]+(0.589388834071)*x[1]**o+(0.669705413859)*x[1]
        arg[1,4,0,0]=(0.320107896841)*x[0]**o+(0.673508197912)*x[0]+(0.667026895874)*x[1]**o+(-0.818578971562)*x[1]
        arg[1,4,0,1]=(-0.226755915796)*x[0]**o+(0.71800207991)*x[0]+(0.850490212043)*x[1]**o+(0.261708964138)*x[1]
        arg[1,4,1,0]=(-0.0147350282361)*x[0]**o+(0.743889536542)*x[0]+(-0.898802067607)*x[1]**o+(0.711589170571)*x[1]
        arg[1,4,1,1]=(-0.816145493818)*x[0]**o+(0.405931568806)*x[0]+(0.159156576579)*x[1]**o+(0.143576116984)*x[1]
        arg[1,4,2,0]=(0.994791165013)*x[0]**o+(0.65054901122)*x[0]+(0.039218059733)*x[1]**o+(-0.662978657934)*x[1]
        arg[1,4,2,1]=(-0.792277825616)*x[0]**o+(-0.257224872111)*x[0]+(-0.244502392787)*x[1]**o+(-0.276042189495)*x[1]
        arg[2,0,0,0]=(0.133211578016)*x[0]**o+(0.10029141733)*x[0]+(-0.52480547182)*x[1]**o+(0.814652707336)*x[1]
        arg[2,0,0,1]=(0.836429340685)*x[0]**o+(-0.837709322969)*x[0]+(-0.442953015462)*x[1]**o+(-0.984757344044)*x[1]
        arg[2,0,1,0]=(0.893738853578)*x[0]**o+(-0.238050980408)*x[0]+(-0.913668527302)*x[1]**o+(0.895136777323)*x[1]
        arg[2,0,1,1]=(-0.0457361766611)*x[0]**o+(0.414814798208)*x[0]+(-0.57261759439)*x[1]**o+(-0.358969048076)*x[1]
        arg[2,0,2,0]=(-0.320394419991)*x[0]**o+(-0.990129952721)*x[0]+(0.890888895016)*x[1]**o+(0.241063301452)*x[1]
        arg[2,0,2,1]=(-0.604164679932)*x[0]**o+(-0.598939404114)*x[0]+(-0.606633388454)*x[1]**o+(-0.666879580873)*x[1]
        arg[2,1,0,0]=(0.68515547034)*x[0]**o+(0.892247437859)*x[0]+(0.642003383991)*x[1]**o+(0.569240970392)*x[1]
        arg[2,1,0,1]=(0.918713108587)*x[0]**o+(0.865547962284)*x[0]+(0.517952260633)*x[1]**o+(-0.307334010857)*x[1]
        arg[2,1,1,0]=(0.353382468404)*x[0]**o+(-0.542718692644)*x[0]+(-0.957182863451)*x[1]**o+(-0.889352632225)*x[1]
        arg[2,1,1,1]=(0.505185847817)*x[0]**o+(-0.71137143775)*x[0]+(0.853785404375)*x[1]**o+(-0.731028307335)*x[1]
        arg[2,1,2,0]=(-0.0805774093876)*x[0]**o+(0.949831911897)*x[0]+(0.511009248375)*x[1]**o+(-0.0192710206538)*x[1]
        arg[2,1,2,1]=(-0.987709862593)*x[0]**o+(-0.536077508064)*x[0]+(-0.0726882784026)*x[1]**o+(-0.27275457372)*x[1]
        arg[2,2,0,0]=(0.271001953238)*x[0]**o+(0.0446246860776)*x[0]+(0.103657462724)*x[1]**o+(0.359900072694)*x[1]
        arg[2,2,0,1]=(0.613460277623)*x[0]**o+(-0.23324730519)*x[0]+(0.188458167579)*x[1]**o+(-0.574087648909)*x[1]
        arg[2,2,1,0]=(-0.787444207567)*x[0]**o+(0.143772612473)*x[0]+(-0.720451072166)*x[1]**o+(0.936819171338)*x[1]
        arg[2,2,1,1]=(-0.219701379296)*x[0]**o+(-0.343100221149)*x[0]+(-0.972737482064)*x[1]**o+(-0.614208415137)*x[1]
        arg[2,2,2,0]=(0.0292070295438)*x[0]**o+(-0.694263454424)*x[0]+(0.188064440036)*x[1]**o+(0.0901763471697)*x[1]
        arg[2,2,2,1]=(0.407629585245)*x[0]**o+(-0.745619322197)*x[0]+(0.949044983116)*x[1]**o+(-0.316057288128)*x[1]
        arg[2,3,0,0]=(0.124212182643)*x[0]**o+(0.621703244759)*x[0]+(-0.0592136772296)*x[1]**o+(0.571981450786)*x[1]
        arg[2,3,0,1]=(-0.683989210442)*x[0]**o+(-0.318774594157)*x[0]+(0.850235115277)*x[1]**o+(-0.871075179838)*x[1]
        arg[2,3,1,0]=(0.478657634044)*x[0]**o+(-0.43702074673)*x[0]+(0.535572237843)*x[1]**o+(0.22354698199)*x[1]
        arg[2,3,1,1]=(0.0391087869133)*x[0]**o+(0.765140505702)*x[0]+(-0.66279875624)*x[1]**o+(-0.41964006149)*x[1]
        arg[2,3,2,0]=(-0.11123799186)*x[0]**o+(0.533398540082)*x[0]+(-0.931488199925)*x[1]**o+(-0.83080969281)*x[1]
        arg[2,3,2,1]=(-0.138259313941)*x[0]**o+(-0.508937684332)*x[0]+(0.583603445711)*x[1]**o+(-0.0782724872225)*x[1]
        arg[2,4,0,0]=(-0.316223936951)*x[0]**o+(0.329776289164)*x[0]+(-0.692491559115)*x[1]**o+(0.814572730294)*x[1]
        arg[2,4,0,1]=(0.74890960681)*x[0]**o+(-0.79167945549)*x[0]+(0.175328119487)*x[1]**o+(-0.253003240064)*x[1]
        arg[2,4,1,0]=(0.657827625141)*x[0]**o+(-0.268878597055)*x[0]+(0.247374286921)*x[1]**o+(0.414500041415)*x[1]
        arg[2,4,1,1]=(-0.187587046202)*x[0]**o+(-0.594983066039)*x[0]+(-0.203784406879)*x[1]**o+(0.122237566739)*x[1]
        arg[2,4,2,0]=(0.671210794607)*x[0]**o+(0.886706498415)*x[0]+(-0.368997114212)*x[1]**o+(0.0418311781187)*x[1]
        arg[2,4,2,1]=(-0.272424416171)*x[0]**o+(-0.537477247783)*x[0]+(0.0405655834868)*x[1]**o+(-0.179684833703)*x[1]
        arg[3,0,0,0]=(0.457602964193)*x[0]**o+(-0.186625111516)*x[0]+(-0.75244105343)*x[1]**o+(-0.278893445377)*x[1]
        arg[3,0,0,1]=(-0.690378721784)*x[0]**o+(0.0157608350188)*x[0]+(-0.203765434813)*x[1]**o+(0.69771096957)*x[1]
        arg[3,0,1,0]=(-0.884824688493)*x[0]**o+(0.26613451569)*x[0]+(0.226081491639)*x[1]**o+(0.597724298389)*x[1]
        arg[3,0,1,1]=(0.0372124540969)*x[0]**o+(-0.420242551812)*x[0]+(-0.625683613012)*x[1]**o+(0.279682305563)*x[1]
        arg[3,0,2,0]=(-0.570873872458)*x[0]**o+(-0.82361574923)*x[0]+(0.970204188337)*x[1]**o+(-0.242725003569)*x[1]
        arg[3,0,2,1]=(-0.714826095124)*x[0]**o+(-0.334939427182)*x[0]+(-0.730058530908)*x[1]**o+(0.287148154312)*x[1]
        arg[3,1,0,0]=(0.164234307274)*x[0]**o+(-0.388849394356)*x[0]+(0.391518896617)*x[1]**o+(0.00142750126236)*x[1]
        arg[3,1,0,1]=(-0.327700781196)*x[0]**o+(0.0442604682565)*x[0]+(-0.683834466536)*x[1]**o+(-0.665567774985)*x[1]
        arg[3,1,1,0]=(0.135979934041)*x[0]**o+(-0.822614011173)*x[0]+(-0.946061768113)*x[1]**o+(-0.585442102671)*x[1]
        arg[3,1,1,1]=(0.235568769871)*x[0]**o+(0.738233870703)*x[0]+(0.437750339594)*x[1]**o+(-0.44521724321)*x[1]
        arg[3,1,2,0]=(0.449256898185)*x[0]**o+(-0.464211755196)*x[0]+(0.616886411002)*x[1]**o+(0.885500269963)*x[1]
        arg[3,1,2,1]=(0.0851105760559)*x[0]**o+(0.888976499957)*x[0]+(0.0842441111314)*x[1]**o+(-0.366994083394)*x[1]
        arg[3,2,0,0]=(0.77120050858)*x[0]**o+(0.354835584949)*x[0]+(0.406813878866)*x[1]**o+(-0.600042478179)*x[1]
        arg[3,2,0,1]=(-0.747876879184)*x[0]**o+(0.527204926634)*x[0]+(0.134965492713)*x[1]**o+(0.497662739459)*x[1]
        arg[3,2,1,0]=(0.367550591163)*x[0]**o+(-0.700608509353)*x[0]+(0.0737696587931)*x[1]**o+(-0.164466899816)*x[1]
        arg[3,2,1,1]=(0.0908791429433)*x[0]**o+(-0.168468767791)*x[0]+(0.85978873292)*x[1]**o+(-0.785241412884)*x[1]
        arg[3,2,2,0]=(-0.296455818292)*x[0]**o+(-0.0277004420512)*x[0]+(0.409183414933)*x[1]**o+(-0.879831945543)*x[1]
        arg[3,2,2,1]=(0.971512010223)*x[0]**o+(-0.811249982262)*x[0]+(-0.846549204737)*x[1]**o+(0.689986186691)*x[1]
        arg[3,3,0,0]=(-0.800235545702)*x[0]**o+(-0.796021196346)*x[0]+(-0.0660744375539)*x[1]**o+(-0.826632192842)*x[1]
        arg[3,3,0,1]=(0.305678237361)*x[0]**o+(0.203936000462)*x[0]+(-0.583171879217)*x[1]**o+(0.924112859119)*x[1]
        arg[3,3,1,0]=(-0.937148913762)*x[0]**o+(-0.172547991203)*x[0]+(0.928491154238)*x[1]**o+(-0.502876301351)*x[1]
        arg[3,3,1,1]=(0.731413856616)*x[0]**o+(0.837213097155)*x[0]+(-0.650032764358)*x[1]**o+(0.96973961589)*x[1]
        arg[3,3,2,0]=(-0.501994458598)*x[0]**o+(-0.705938508685)*x[0]+(0.664394117366)*x[1]**o+(0.692950029135)*x[1]
        arg[3,3,2,1]=(-0.0948399323409)*x[0]**o+(-0.686744554674)*x[0]+(0.577765681583)*x[1]**o+(0.969310750204)*x[1]
        arg[3,4,0,0]=(-0.484011788432)*x[0]**o+(-0.560744139846)*x[0]+(0.072097608773)*x[1]**o+(-0.103052228904)*x[1]
        arg[3,4,0,1]=(-0.201158934695)*x[0]**o+(0.463105335811)*x[0]+(0.733365706682)*x[1]**o+(-0.330035241794)*x[1]
        arg[3,4,1,0]=(0.252379277102)*x[0]**o+(-0.110596657969)*x[0]+(0.763687257878)*x[1]**o+(-0.345598371263)*x[1]
        arg[3,4,1,1]=(0.0868412855617)*x[0]**o+(0.385098679406)*x[0]+(0.291487288978)*x[1]**o+(-0.112145846209)*x[1]
        arg[3,4,2,0]=(-0.913724685255)*x[0]**o+(-0.0348735884313)*x[0]+(0.550688528861)*x[1]**o+(0.798394152329)*x[1]
        arg[3,4,2,1]=(-0.92870523219)*x[0]**o+(0.177354002469)*x[0]+(0.350393765313)*x[1]**o+(-0.974997198741)*x[1]
        ref[0,0,0,0]=(0.815821053922)/(o+1.)+(0.302879577355)
        ref[0,0,0,1]=(-0.223824520603)/(o+1.)+(0.690659426687)
        ref[0,0,1,0]=(-0.501731412433)/(o+1.)+(0.0102964270565)
        ref[0,0,1,1]=(-0.731540589167)/(o+1.)+(0.348284469462)
        ref[0,0,2,0]=(-0.121523684012)/(o+1.)+(0.184348962759)
        ref[0,0,2,1]=(-0.4529485642)/(o+1.)+(-0.830798233026)
        ref[0,1,0,0]=(-0.244463483906)/(o+1.)+(0.226296188971)
        ref[0,1,0,1]=(1.2203272116)/(o+1.)+(-0.693494218115)
        ref[0,1,1,0]=(-0.55576623697)/(o+1.)+(-0.273623989583)
        ref[0,1,1,1]=(-0.316836585171)/(o+1.)+(-0.836135387777)
        ref[0,1,2,0]=(-0.133732244387)/(o+1.)+(-0.875415201605)
        ref[0,1,2,1]=(-0.281487947177)/(o+1.)+(-0.668499911274)
        ref[0,2,0,0]=(-0.896781862729)/(o+1.)+(-0.0225848820142)
        ref[0,2,0,1]=(-1.70326692867)/(o+1.)+(-0.1693379432)
        ref[0,2,1,0]=(-0.00084798161079)/(o+1.)+(0.343359275275)
        ref[0,2,1,1]=(0.609064372767)/(o+1.)+(-0.369639962727)
        ref[0,2,2,0]=(-0.759042888708)/(o+1.)+(0.321708104548)
        ref[0,2,2,1]=(0.760967724015)/(o+1.)+(0.0268897973234)
        ref[0,3,0,0]=(-1.3915636468)/(o+1.)+(-0.230755462281)
        ref[0,3,0,1]=(-0.416449082748)/(o+1.)+(0.480436529329)
        ref[0,3,1,0]=(-0.238553658798)/(o+1.)+(-0.029283546006)
        ref[0,3,1,1]=(-1.36933302275)/(o+1.)+(-0.221757606321)
        ref[0,3,2,0]=(-0.872999155991)/(o+1.)+(0.69296766968)
        ref[0,3,2,1]=(0.833661507716)/(o+1.)+(0.321357009466)
        ref[0,4,0,0]=(1.4358026373)/(o+1.)+(0.311715942547)
        ref[0,4,0,1]=(-0.00811900044778)/(o+1.)+(-0.76710974695)
        ref[0,4,1,0]=(-0.92034101297)/(o+1.)+(-0.0309623888275)
        ref[0,4,1,1]=(0.690460334263)/(o+1.)+(0.783809038221)
        ref[0,4,2,0]=(0.779851986824)/(o+1.)+(-0.214799981386)
        ref[0,4,2,1]=(-0.566157259598)/(o+1.)+(0.296555712092)
        ref[1,0,0,0]=(-1.70267943485)/(o+1.)+(0.0444622873215)
        ref[1,0,0,1]=(0.399342875087)/(o+1.)+(0.0341080904363)
        ref[1,0,1,0]=(0.839377637409)/(o+1.)+(0.158619241351)
        ref[1,0,1,1]=(-0.0142823080176)/(o+1.)+(0.693714763021)
        ref[1,0,2,0]=(-0.97036633864)/(o+1.)+(0.264110057567)
        ref[1,0,2,1]=(0.104223436728)/(o+1.)+(0.328785142295)
        ref[1,1,0,0]=(1.09779156365)/(o+1.)+(-0.00408578628855)
        ref[1,1,0,1]=(-0.914750679597)/(o+1.)+(0.206573942886)
        ref[1,1,1,0]=(0.150419426085)/(o+1.)+(-0.0846024372156)
        ref[1,1,1,1]=(-1.04733819847)/(o+1.)+(-0.157756242641)
        ref[1,1,2,0]=(-1.49193735459)/(o+1.)+(0.391723285192)
        ref[1,1,2,1]=(-0.250600389657)/(o+1.)+(-0.456841431612)
        ref[1,2,0,0]=(1.21154796884)/(o+1.)+(-0.377275533205)
        ref[1,2,0,1]=(-0.785475967917)/(o+1.)+(0.2413362399)
        ref[1,2,1,0]=(1.20750352255)/(o+1.)+(-0.409504204121)
        ref[1,2,1,1]=(0.271471015188)/(o+1.)+(-0.136640402636)
        ref[1,2,2,0]=(-0.741344603747)/(o+1.)+(0.00668995653618)
        ref[1,2,2,1]=(0.462000684606)/(o+1.)+(0.364894712961)
        ref[1,3,0,0]=(0.0273564369347)/(o+1.)+(0.673962086152)
        ref[1,3,0,1]=(0.285276349085)/(o+1.)+(0.385675061073)
        ref[1,3,1,0]=(0.851307298344)/(o+1.)+(0.118422744971)
        ref[1,3,1,1]=(0.235593686643)/(o+1.)+(0.244705242716)
        ref[1,3,2,0]=(-0.359345731222)/(o+1.)+(-0.124958138388)
        ref[1,3,2,1]=(0.590170220995)/(o+1.)+(0.429975100757)
        ref[1,4,0,0]=(0.987134792715)/(o+1.)+(-0.0725353868253)
        ref[1,4,0,1]=(0.623734296247)/(o+1.)+(0.489855522024)
        ref[1,4,1,0]=(-0.913537095843)/(o+1.)+(0.727739353556)
        ref[1,4,1,1]=(-0.656988917238)/(o+1.)+(0.274753842895)
        ref[1,4,2,0]=(1.03400922475)/(o+1.)+(-0.00621482335664)
        ref[1,4,2,1]=(-1.0367802184)/(o+1.)+(-0.266633530803)
        ref[2,0,0,0]=(-0.391593893803)/(o+1.)+(0.457472062333)
        ref[2,0,0,1]=(0.393476325223)/(o+1.)+(-0.911233333506)
        ref[2,0,1,0]=(-0.0199296737245)/(o+1.)+(0.328542898457)
        ref[2,0,1,1]=(-0.618353771051)/(o+1.)+(0.0279228750664)
        ref[2,0,2,0]=(0.570494475025)/(o+1.)+(-0.374533325635)
        ref[2,0,2,1]=(-1.21079806839)/(o+1.)+(-0.632909492494)
        ref[2,1,0,0]=(1.32715885433)/(o+1.)+(0.730744204126)
        ref[2,1,0,1]=(1.43666536922)/(o+1.)+(0.279106975714)
        ref[2,1,1,0]=(-0.603800395047)/(o+1.)+(-0.716035662434)
        ref[2,1,1,1]=(1.35897125219)/(o+1.)+(-0.721199872542)
        ref[2,1,2,0]=(0.430431838988)/(o+1.)+(0.465280445622)
        ref[2,1,2,1]=(-1.060398141)/(o+1.)+(-0.404416040892)
        ref[2,2,0,0]=(0.374659415961)/(o+1.)+(0.202262379386)
        ref[2,2,0,1]=(0.801918445202)/(o+1.)+(-0.403667477049)
        ref[2,2,1,0]=(-1.50789527973)/(o+1.)+(0.540295891905)
        ref[2,2,1,1]=(-1.19243886136)/(o+1.)+(-0.478654318143)
        ref[2,2,2,0]=(0.21727146958)/(o+1.)+(-0.302043553627)
        ref[2,2,2,1]=(1.35667456836)/(o+1.)+(-0.530838305162)
        ref[2,3,0,0]=(0.0649985054133)/(o+1.)+(0.596842347772)
        ref[2,3,0,1]=(0.166245904835)/(o+1.)+(-0.594924886997)
        ref[2,3,1,0]=(1.01422987189)/(o+1.)+(-0.10673688237)
        ref[2,3,1,1]=(-0.623689969327)/(o+1.)+(0.172750222106)
        ref[2,3,2,0]=(-1.04272619178)/(o+1.)+(-0.148705576364)
        ref[2,3,2,1]=(0.445344131771)/(o+1.)+(-0.293605085777)
        ref[2,4,0,0]=(-1.00871549607)/(o+1.)+(0.572174509729)
        ref[2,4,0,1]=(0.924237726297)/(o+1.)+(-0.522341347777)
        ref[2,4,1,0]=(0.905201912063)/(o+1.)+(0.0728107221801)
        ref[2,4,1,1]=(-0.391371453081)/(o+1.)+(-0.23637274965)
        ref[2,4,2,0]=(0.302213680395)/(o+1.)+(0.464268838267)
        ref[2,4,2,1]=(-0.231858832685)/(o+1.)+(-0.358581040743)
        ref[3,0,0,0]=(-0.294838089237)/(o+1.)+(-0.232759278447)
        ref[3,0,0,1]=(-0.894144156597)/(o+1.)+(0.356735902294)
        ref[3,0,1,0]=(-0.658743196854)/(o+1.)+(0.43192940704)
        ref[3,0,1,1]=(-0.588471158915)/(o+1.)+(-0.0702801231243)
        ref[3,0,2,0]=(0.399330315879)/(o+1.)+(-0.5331703764)
        ref[3,0,2,1]=(-1.44488462603)/(o+1.)+(-0.0238956364351)
        ref[3,1,0,0]=(0.555753203891)/(o+1.)+(-0.193710946547)
        ref[3,1,0,1]=(-1.01153524773)/(o+1.)+(-0.310653653364)
        ref[3,1,1,0]=(-0.810081834072)/(o+1.)+(-0.704028056922)
        ref[3,1,1,1]=(0.673319109465)/(o+1.)+(0.146508313746)
        ref[3,1,2,0]=(1.06614330919)/(o+1.)+(0.210644257384)
        ref[3,1,2,1]=(0.169354687187)/(o+1.)+(0.260991208281)
        ref[3,2,0,0]=(1.17801438745)/(o+1.)+(-0.122603446615)
        ref[3,2,0,1]=(-0.612911386471)/(o+1.)+(0.512433833047)
        ref[3,2,1,0]=(0.441320249956)/(o+1.)+(-0.432537704584)
        ref[3,2,1,1]=(0.950667875863)/(o+1.)+(-0.476855090338)
        ref[3,2,2,0]=(0.112727596641)/(o+1.)+(-0.453766193797)
        ref[3,2,2,1]=(0.124962805486)/(o+1.)+(-0.0606318977858)
        ref[3,3,0,0]=(-0.866309983256)/(o+1.)+(-0.811326694594)
        ref[3,3,0,1]=(-0.277493641857)/(o+1.)+(0.564024429791)
        ref[3,3,1,0]=(-0.00865775952435)/(o+1.)+(-0.337712146277)
        ref[3,3,1,1]=(0.0813810922576)/(o+1.)+(0.903476356523)
        ref[3,3,2,0]=(0.162399658768)/(o+1.)+(-0.00649423977499)
        ref[3,3,2,1]=(0.482925749243)/(o+1.)+(0.141283097765)
        ref[3,4,0,0]=(-0.411914179659)/(o+1.)+(-0.331898184375)
        ref[3,4,0,1]=(0.532206771987)/(o+1.)+(0.0665350470083)
        ref[3,4,1,0]=(1.01606653498)/(o+1.)+(-0.228097514616)
        ref[3,4,1,1]=(0.37832857454)/(o+1.)+(0.136476416599)
        ref[3,4,2,0]=(-0.363036156393)/(o+1.)+(0.381760281949)
        ref[3,4,2,1]=(-0.578311466877)/(o+1.)+(-0.398821598136)
      else:
        arg[0,0,0,0]=(0.860676142569)*x[0]**o+(-0.156046991162)*x[0]+(-0.0462355579656)*x[1]**o+(0.70842844961)*x[1]+(0.617299123548)*x[2]**o+(-0.865174329565)*x[2]
        arg[0,0,0,1]=(-0.284308060865)*x[0]**o+(0.788848579389)*x[0]+(0.629321669042)*x[1]**o+(0.066637849809)*x[1]+(0.114152381009)*x[2]**o+(-0.466687745846)*x[2]
        arg[0,0,1,0]=(-0.690823198517)*x[0]**o+(-0.141510804573)*x[0]+(0.969978831967)*x[1]**o+(0.307926382658)*x[1]+(0.710373009094)*x[2]**o+(-0.494828315907)*x[2]
        arg[0,0,1,1]=(0.953361413913)*x[0]**o+(-0.103742611936)*x[0]+(0.63934955554)*x[1]**o+(-0.0778114476564)*x[1]+(-0.221185122698)*x[2]**o+(-0.287432047404)*x[2]
        arg[0,0,2,0]=(0.378306380558)*x[0]**o+(-0.117743922672)*x[0]+(-0.0606376630208)*x[1]**o+(-0.59439189505)*x[1]+(0.10507320657)*x[2]**o+(0.856411423067)*x[2]
        arg[0,0,2,1]=(0.777734672779)*x[0]**o+(0.286956002007)*x[0]+(-0.931811626583)*x[1]**o+(-0.806626659661)*x[1]+(-0.420390949282)*x[2]**o+(0.854080006578)*x[2]
        arg[0,1,0,0]=(0.403426948964)*x[0]**o+(0.707099419296)*x[0]+(0.593137130456)*x[1]**o+(-0.757560771598)*x[1]+(0.369632217934)*x[2]**o+(-0.348480285397)*x[2]
        arg[0,1,0,1]=(-0.491454189782)*x[0]**o+(0.547276640565)*x[0]+(-0.955045542708)*x[1]**o+(0.529989872531)*x[1]+(-0.161086874799)*x[2]**o+(-0.181810040724)*x[2]
        arg[0,1,1,0]=(0.46282177104)*x[0]**o+(0.254346937251)*x[0]+(0.824797105546)*x[1]**o+(-0.449979162246)*x[1]+(0.22307703071)*x[2]**o+(-0.592072135668)*x[2]
        arg[0,1,1,1]=(0.329230675141)*x[0]**o+(-0.733034697145)*x[0]+(0.762195609347)*x[1]**o+(-0.639097591362)*x[1]+(0.765354927905)*x[2]**o+(0.313571493405)*x[2]
        arg[0,1,2,0]=(0.3874127072)*x[0]**o+(-0.0569429727569)*x[0]+(-0.203121198708)*x[1]**o+(-0.736379499164)*x[1]+(-0.980629375436)*x[2]**o+(0.955793660048)*x[2]
        arg[0,1,2,1]=(-0.656255469455)*x[0]**o+(0.711904808699)*x[0]+(-0.363253678399)*x[1]**o+(0.921875000807)*x[1]+(0.686394816431)*x[2]**o+(0.818500716704)*x[2]
        arg[0,2,0,0]=(-0.931555347846)*x[0]**o+(-0.249573400742)*x[0]+(0.617166716404)*x[1]**o+(-0.929587051108)*x[1]+(0.830928715064)*x[2]**o+(0.461637431613)*x[2]
        arg[0,2,0,1]=(-0.8677719945)*x[0]**o+(-0.176892180445)*x[0]+(-0.881887456464)*x[1]**o+(0.923355677513)*x[1]+(-0.978544668453)*x[2]**o+(-0.427009384287)*x[2]
        arg[0,2,1,0]=(-0.0707937433084)*x[0]**o+(-0.872892141393)*x[0]+(0.606234798381)*x[1]**o+(0.935845076847)*x[1]+(-0.97559052201)*x[2]**o+(-0.31138518209)*x[2]
        arg[0,2,1,1]=(0.450210597865)*x[0]**o+(0.842939440378)*x[0]+(-0.333422856577)*x[1]**o+(-0.492809679443)*x[1]+(-0.254728278043)*x[2]**o+(-0.0702767077952)*x[2]
        arg[0,2,2,0]=(-0.250463216903)*x[0]**o+(0.756576293763)*x[0]+(0.596902629953)*x[1]**o+(-0.887080794947)*x[1]+(0.027319948008)*x[2]**o+(0.552496782776)*x[2]
        arg[0,2,2,1]=(-0.0883102140591)*x[0]**o+(0.988759610101)*x[0]+(-0.208152392508)*x[1]**o+(0.938999108564)*x[1]+(-0.773883611417)*x[2]**o+(0.653413306338)*x[2]
        arg[0,3,0,0]=(-0.399644211606)*x[0]**o+(-0.978595776407)*x[0]+(0.99454650986)*x[1]**o+(0.143313668715)*x[1]+(0.328592750842)*x[2]**o+(-0.958804581101)*x[2]
        arg[0,3,0,1]=(-0.0667966885959)*x[0]**o+(0.332653574704)*x[0]+(-0.762025150916)*x[1]**o+(0.895044013374)*x[1]+(0.627013843362)*x[2]**o+(-0.682392672232)*x[2]
        arg[0,3,1,0]=(-0.834479453773)*x[0]**o+(-0.375930859033)*x[0]+(0.129507966861)*x[1]**o+(0.995364280408)*x[1]+(-0.126081191572)*x[2]**o+(0.0344262779491)*x[2]
        arg[0,3,1,1]=(0.796952126596)*x[0]**o+(0.310562243091)*x[0]+(0.854548623788)*x[1]**o+(-0.719100537284)*x[1]+(-0.236266783438)*x[2]**o+(-0.646647129209)*x[2]
        arg[0,3,2,0]=(0.844073289115)*x[0]**o+(0.423300345347)*x[0]+(0.126354796852)*x[1]**o+(0.180372257606)*x[1]+(-0.564712722146)*x[2]**o+(0.466988430183)*x[2]
        arg[0,3,2,1]=(0.228775824367)*x[0]**o+(0.40701094831)*x[0]+(0.223101054593)*x[1]**o+(-0.894937905383)*x[1]+(0.425392942773)*x[2]**o+(-0.844238667169)*x[2]
        arg[0,4,0,0]=(0.3363848738)*x[0]**o+(-0.985327066062)*x[0]+(-0.107847971678)*x[1]**o+(0.596121801488)*x[1]+(0.655589801663)*x[2]**o+(-0.729650731949)*x[2]
        arg[0,4,0,1]=(-0.176435679219)*x[0]**o+(0.382558711725)*x[0]+(-0.658212492337)*x[1]**o+(0.584904267612)*x[1]+(-0.534684876705)*x[2]**o+(-0.441262614915)*x[2]
        arg[0,4,1,0]=(0.281301737167)*x[0]**o+(-0.642486274791)*x[0]+(0.420753884869)*x[1]**o+(0.482835955591)*x[1]+(0.169431333788)*x[2]**o+(0.922841193825)*x[2]
        arg[0,4,1,1]=(0.235812590604)*x[0]**o+(-0.446233855396)*x[0]+(0.465015868566)*x[1]**o+(-0.780441276888)*x[1]+(0.513632761351)*x[2]**o+(-0.0576482811678)*x[2]
        arg[0,4,2,0]=(-0.617245874462)*x[0]**o+(0.183661473466)*x[0]+(-0.711148911239)*x[1]**o+(0.125228991306)*x[1]+(-0.500218603959)*x[2]**o+(0.70597577229)*x[2]
        arg[0,4,2,1]=(0.56330304117)*x[0]**o+(0.631284986628)*x[0]+(-0.222485175776)*x[1]**o+(0.0639235737228)*x[1]+(-0.276613868747)*x[2]**o+(0.407488564857)*x[2]
        arg[1,0,0,0]=(0.159032146915)*x[0]**o+(-0.716040989693)*x[0]+(0.410709172613)*x[1]**o+(-0.527667963652)*x[1]+(-0.160515156497)*x[2]**o+(0.294448742382)*x[2]
        arg[1,0,0,1]=(0.941139035835)*x[0]**o+(0.707410046094)*x[0]+(-0.172095410803)*x[1]**o+(-0.967035204304)*x[1]+(0.870395272646)*x[2]**o+(0.830870986802)*x[2]
        arg[1,0,1,0]=(0.731015285309)*x[0]**o+(0.450519048697)*x[0]+(-0.484568714675)*x[1]**o+(0.454201220108)*x[1]+(-0.161850459099)*x[2]**o+(-0.617463263396)*x[2]
        arg[1,0,1,1]=(-0.958841628847)*x[0]**o+(-0.0667813400021)*x[0]+(0.659551780082)*x[1]**o+(0.514424015304)*x[1]+(0.920583608125)*x[2]**o+(0.89564899572)*x[2]
        arg[1,0,2,0]=(0.0482630479817)*x[0]**o+(0.911326064322)*x[0]+(-0.275134422092)*x[1]**o+(0.216495232459)*x[1]+(-0.973586171195)*x[2]**o+(0.275694688611)*x[2]
        arg[1,0,2,1]=(0.769733688293)*x[0]**o+(0.771829018421)*x[0]+(-0.22434549079)*x[1]**o+(0.131966885817)*x[1]+(-0.109424705679)*x[2]**o+(-0.369585136379)*x[2]
        arg[1,1,0,0]=(-0.0160180905877)*x[0]**o+(-0.268260755267)*x[0]+(-0.213940412022)*x[1]**o+(0.665987805279)*x[1]+(0.968879828017)*x[2]**o+(-0.449435538913)*x[2]
        arg[1,1,0,1]=(0.361055351825)*x[0]**o+(0.463294478488)*x[0]+(0.692655668486)*x[1]**o+(0.151415573206)*x[1]+(0.902280461515)*x[2]**o+(0.690926934073)*x[2]
        arg[1,1,1,0]=(0.591998602299)*x[0]**o+(0.516482177709)*x[0]+(-0.298511909182)*x[1]**o+(0.281818071894)*x[1]+(0.207089953979)*x[2]**o+(0.917004722388)*x[2]
        arg[1,1,1,1]=(0.479832408236)*x[0]**o+(-0.643374914165)*x[0]+(0.370844835249)*x[1]**o+(0.560458499239)*x[1]+(-0.865852319767)*x[2]**o+(0.560572130123)*x[2]
        arg[1,1,2,0]=(0.709293526635)*x[0]**o+(-0.388347223155)*x[0]+(0.60797330273)*x[1]**o+(-0.847624783099)*x[1]+(-0.0476367756382)*x[2]**o+(0.637841988478)*x[2]
        arg[1,1,2,1]=(0.343458045422)*x[0]**o+(-0.764451449587)*x[0]+(-0.444256801761)*x[1]**o+(0.385385997921)*x[1]+(-0.722095136671)*x[2]**o+(0.677852473207)*x[2]
        arg[1,2,0,0]=(0.710372887392)*x[0]**o+(-0.628494679119)*x[0]+(-0.59513507905)*x[1]**o+(-0.527824874295)*x[1]+(-0.611189170789)*x[2]**o+(0.090651976806)*x[2]
        arg[1,2,0,1]=(-0.980160150289)*x[0]**o+(-0.0286097581598)*x[0]+(-0.407906975953)*x[1]**o+(-0.319422835951)*x[1]+(-0.489787310147)*x[2]**o+(-0.694386341354)*x[2]
        arg[1,2,1,0]=(-0.416867224376)*x[0]**o+(0.631406877489)*x[0]+(-0.93939195942)*x[1]**o+(0.266858600313)*x[1]+(0.982173221612)*x[2]**o+(0.112901364617)*x[2]
        arg[1,2,1,1]=(0.412001480348)*x[0]**o+(0.624281489887)*x[0]+(0.306558457751)*x[1]**o+(0.95864454419)*x[1]+(0.299175627954)*x[2]**o+(-0.0531991017347)*x[2]
        arg[1,2,2,0]=(-0.0342261621741)*x[0]**o+(-0.587245704486)*x[0]+(-0.976407464107)*x[1]**o+(-0.0157215030322)*x[1]+(-0.529857472095)*x[2]**o+(-0.419626697673)*x[2]
        arg[1,2,2,1]=(-0.681737301037)*x[0]**o+(-0.980349460702)*x[0]+(-0.839469631393)*x[1]**o+(-0.412417814907)*x[1]+(0.451717398947)*x[2]**o+(0.962167685987)*x[2]
        arg[1,3,0,0]=(0.283453965913)*x[0]**o+(0.204445403787)*x[0]+(-0.806150376236)*x[1]**o+(0.122187094069)*x[1]+(-0.679291367964)*x[2]**o+(-0.875353966489)*x[2]
        arg[1,3,0,1]=(0.115974475175)*x[0]**o+(0.701473497654)*x[0]+(0.298012120304)*x[1]**o+(0.567597901709)*x[1]+(0.721211831476)*x[2]**o+(-0.255389572775)*x[2]
        arg[1,3,1,0]=(0.393891431092)*x[0]**o+(-0.563382537925)*x[0]+(0.974194829106)*x[1]**o+(0.659688559688)*x[1]+(-0.394266143828)*x[2]**o+(0.875266200191)*x[2]
        arg[1,3,1,1]=(-0.456930975697)*x[0]**o+(0.334590626231)*x[0]+(-0.227176883362)*x[1]**o+(0.76775699173)*x[1]+(0.839371167887)*x[2]**o+(-0.155473916326)*x[2]
        arg[1,3,2,0]=(0.346761981787)*x[0]**o+(-0.625121706322)*x[0]+(0.340655154638)*x[1]**o+(-0.0646323546999)*x[1]+(-0.211347402677)*x[2]**o+(0.233661266919)*x[2]
        arg[1,3,2,1]=(-0.086096644956)*x[0]**o+(-0.845623653209)*x[0]+(-0.0866093549957)*x[1]**o+(-0.467286349011)*x[1]+(-0.0621455515357)*x[2]**o+(0.164779979085)*x[2]
        arg[1,4,0,0]=(0.42022138208)*x[0]**o+(0.136984451355)*x[0]+(0.664770979086)*x[1]**o+(-0.537368896193)*x[1]+(0.874256726691)*x[2]**o+(-0.58614100824)*x[2]
        arg[1,4,0,1]=(-0.805860964792)*x[0]**o+(0.758511304044)*x[0]+(-0.574686863541)*x[1]**o+(-0.410873101247)*x[1]+(0.932015171551)*x[2]**o+(0.439987195351)*x[2]
        arg[1,4,1,0]=(-0.515187049123)*x[0]**o+(0.308078404334)*x[0]+(0.940959771774)*x[1]**o+(0.466269579691)*x[1]+(-0.190617144843)*x[2]**o+(0.738172533582)*x[2]
        arg[1,4,1,1]=(0.283326036586)*x[0]**o+(-0.693631747857)*x[0]+(0.478965080319)*x[1]**o+(-0.0465389396775)*x[1]+(-0.72886286959)*x[2]**o+(-0.9346424149)*x[2]
        arg[1,4,2,0]=(-0.71539766285)*x[0]**o+(0.361875987071)*x[0]+(0.905270678876)*x[1]**o+(-0.675024725702)*x[1]+(0.378259473658)*x[2]**o+(0.0520394226711)*x[2]
        arg[1,4,2,1]=(-0.319600525923)*x[0]**o+(0.0449956327134)*x[0]+(0.707418658521)*x[1]**o+(0.210137972776)*x[1]+(-0.767860728688)*x[2]**o+(0.232340825758)*x[2]
        arg[2,0,0,0]=(0.92844243651)*x[0]**o+(0.956810319504)*x[0]+(0.762198571194)*x[1]**o+(-0.363107634629)*x[1]+(0.0750393505254)*x[2]**o+(-0.43573581041)*x[2]
        arg[2,0,0,1]=(-0.193613121742)*x[0]**o+(0.440027871839)*x[0]+(-0.607905506346)*x[1]**o+(0.894840354958)*x[1]+(0.601279436178)*x[2]**o+(0.46806560762)*x[2]
        arg[2,0,1,0]=(0.370901517567)*x[0]**o+(-0.738769905647)*x[0]+(-0.698692979664)*x[1]**o+(-0.286318840335)*x[1]+(-0.879367315688)*x[2]**o+(-0.959912132991)*x[2]
        arg[2,0,1,1]=(-0.278939533774)*x[0]**o+(0.183795552426)*x[0]+(-0.341943551466)*x[1]**o+(0.491847260291)*x[1]+(-0.141030429384)*x[2]**o+(0.256287855539)*x[2]
        arg[2,0,2,0]=(-0.135930357616)*x[0]**o+(-0.382515749949)*x[0]+(-0.43030873581)*x[1]**o+(-0.0211891159981)*x[1]+(-0.377865120002)*x[2]**o+(0.109942325029)*x[2]
        arg[2,0,2,1]=(0.288827267201)*x[0]**o+(-0.907970544468)*x[0]+(-0.622258309213)*x[1]**o+(-0.763273879894)*x[1]+(0.438715013671)*x[2]**o+(-0.93873140988)*x[2]
        arg[2,1,0,0]=(-0.963487369906)*x[0]**o+(0.899051290288)*x[0]+(-0.268257956548)*x[1]**o+(-0.229092726584)*x[1]+(0.427024112314)*x[2]**o+(0.0435138095138)*x[2]
        arg[2,1,0,1]=(0.89996926597)*x[0]**o+(-0.206500706229)*x[0]+(-0.191769123095)*x[1]**o+(-0.348924155298)*x[1]+(-0.74405653776)*x[2]**o+(0.0724946691864)*x[2]
        arg[2,1,1,0]=(0.338170397901)*x[0]**o+(-0.725761081242)*x[0]+(-0.45928272676)*x[1]**o+(0.795535326783)*x[1]+(-0.316443844644)*x[2]**o+(-0.844205401892)*x[2]
        arg[2,1,1,1]=(0.509131757681)*x[0]**o+(0.319175801535)*x[0]+(-0.807058926279)*x[1]**o+(0.63928269996)*x[1]+(0.219324073432)*x[2]**o+(0.351217517434)*x[2]
        arg[2,1,2,0]=(-0.534090721611)*x[0]**o+(0.27146424294)*x[0]+(0.250778065716)*x[1]**o+(-0.901322994167)*x[1]+(-0.57588768699)*x[2]**o+(0.405043003456)*x[2]
        arg[2,1,2,1]=(0.101699906269)*x[0]**o+(0.361960318738)*x[0]+(0.278974923593)*x[1]**o+(-0.336289352037)*x[1]+(-0.677353936215)*x[2]**o+(0.0726793347753)*x[2]
        arg[2,2,0,0]=(-0.734961283823)*x[0]**o+(-0.593268557578)*x[0]+(0.310896858565)*x[1]**o+(-0.482326137951)*x[1]+(0.886472957099)*x[2]**o+(-0.0403563476581)*x[2]
        arg[2,2,0,1]=(-0.166071725106)*x[0]**o+(-0.454245686818)*x[0]+(-0.892674508991)*x[1]**o+(0.581554190251)*x[1]+(0.995323654908)*x[2]**o+(-0.0336714433349)*x[2]
        arg[2,2,1,0]=(-0.437491848099)*x[0]**o+(-0.195039508213)*x[0]+(0.822011086215)*x[1]**o+(0.771044412863)*x[1]+(0.697534261252)*x[2]**o+(0.666510217725)*x[2]
        arg[2,2,1,1]=(-0.529148276275)*x[0]**o+(-0.093016460024)*x[0]+(0.663641010649)*x[1]**o+(0.940439359843)*x[1]+(0.754151159716)*x[2]**o+(0.852349310591)*x[2]
        arg[2,2,2,0]=(0.859523244682)*x[0]**o+(0.704825125128)*x[0]+(0.221832129598)*x[1]**o+(0.573940020456)*x[1]+(0.935673751591)*x[2]**o+(0.23235170475)*x[2]
        arg[2,2,2,1]=(0.0359047473512)*x[0]**o+(0.836816935443)*x[0]+(0.6411608273)*x[1]**o+(0.291526953908)*x[1]+(0.735564515362)*x[2]**o+(-0.465781621769)*x[2]
        arg[2,3,0,0]=(0.352740040216)*x[0]**o+(0.54224020581)*x[0]+(0.681356174115)*x[1]**o+(-0.97589117869)*x[1]+(0.880755719884)*x[2]**o+(0.562259403362)*x[2]
        arg[2,3,0,1]=(0.563708471054)*x[0]**o+(0.525523219723)*x[0]+(0.423068360374)*x[1]**o+(0.29239160821)*x[1]+(0.874360368885)*x[2]**o+(0.922265851359)*x[2]
        arg[2,3,1,0]=(0.694309936305)*x[0]**o+(-0.409224769672)*x[0]+(-0.748835864803)*x[1]**o+(-0.21841534641)*x[1]+(0.497059735624)*x[2]**o+(-0.753234694148)*x[2]
        arg[2,3,1,1]=(0.109706510352)*x[0]**o+(-0.0160720403678)*x[0]+(0.888778080744)*x[1]**o+(-0.938692716268)*x[1]+(0.529262951478)*x[2]**o+(0.481430680214)*x[2]
        arg[2,3,2,0]=(0.65368659132)*x[0]**o+(-0.988176396392)*x[0]+(0.989546739333)*x[1]**o+(0.23178715846)*x[1]+(-0.557645438328)*x[2]**o+(0.063228513147)*x[2]
        arg[2,3,2,1]=(0.286316141831)*x[0]**o+(0.668138463424)*x[0]+(-0.950430671533)*x[1]**o+(-0.978012107187)*x[1]+(-0.107618382464)*x[2]**o+(-0.684084421276)*x[2]
        arg[2,4,0,0]=(-0.63510659432)*x[0]**o+(0.264242935995)*x[0]+(0.960647241624)*x[1]**o+(0.58483379289)*x[1]+(0.7255270271)*x[2]**o+(-0.476151653985)*x[2]
        arg[2,4,0,1]=(-0.221953908705)*x[0]**o+(-0.994143786509)*x[0]+(-0.359525421944)*x[1]**o+(0.533966197668)*x[1]+(-0.29739666047)*x[2]**o+(-0.0824204124466)*x[2]
        arg[2,4,1,0]=(-0.447043625871)*x[0]**o+(-0.904485300564)*x[0]+(0.26179071803)*x[1]**o+(0.437320719437)*x[1]+(-0.56501466909)*x[2]**o+(0.129662379166)*x[2]
        arg[2,4,1,1]=(0.259809756366)*x[0]**o+(0.446788808701)*x[0]+(-0.143702396023)*x[1]**o+(0.879531090832)*x[1]+(0.410047703156)*x[2]**o+(0.755750006468)*x[2]
        arg[2,4,2,0]=(-0.730881615951)*x[0]**o+(-0.357110792705)*x[0]+(0.552722654621)*x[1]**o+(0.965623848242)*x[1]+(0.555951256049)*x[2]**o+(0.259021370764)*x[2]
        arg[2,4,2,1]=(-0.465880467377)*x[0]**o+(-0.840984822217)*x[0]+(0.665817382039)*x[1]**o+(-0.390932961529)*x[1]+(-0.0857939888315)*x[2]**o+(-0.712185917672)*x[2]
        arg[3,0,0,0]=(0.710281429223)*x[0]**o+(-0.723903639057)*x[0]+(-0.267455509065)*x[1]**o+(-0.731684188612)*x[1]+(-0.287442751768)*x[2]**o+(-0.266871749947)*x[2]
        arg[3,0,0,1]=(0.460723870663)*x[0]**o+(0.0502154310124)*x[0]+(-0.260241363193)*x[1]**o+(-0.516473088648)*x[1]+(-0.925984113039)*x[2]**o+(-0.0297616649018)*x[2]
        arg[3,0,1,0]=(0.523186648278)*x[0]**o+(0.937042293367)*x[0]+(0.811740916657)*x[1]**o+(0.0544805175015)*x[1]+(-0.543484279955)*x[2]**o+(0.244714198306)*x[2]
        arg[3,0,1,1]=(-0.69212630167)*x[0]**o+(0.868708276385)*x[0]+(0.473290025811)*x[1]**o+(0.137043602294)*x[1]+(-0.19291314113)*x[2]**o+(-0.755147036188)*x[2]
        arg[3,0,2,0]=(0.0472638421122)*x[0]**o+(0.403135551064)*x[0]+(0.290083188289)*x[1]**o+(0.96419060449)*x[1]+(-0.240544184077)*x[2]**o+(-0.183028108529)*x[2]
        arg[3,0,2,1]=(0.354253383908)*x[0]**o+(0.628816528774)*x[0]+(0.588194429756)*x[1]**o+(-0.852128734457)*x[1]+(-0.84774778925)*x[2]**o+(0.796310123562)*x[2]
        arg[3,1,0,0]=(0.528880495806)*x[0]**o+(-0.74176589205)*x[0]+(-0.447816938371)*x[1]**o+(0.611297564802)*x[1]+(0.742134198482)*x[2]**o+(-0.625840664202)*x[2]
        arg[3,1,0,1]=(0.419633746259)*x[0]**o+(-0.247742274578)*x[0]+(0.302328885303)*x[1]**o+(-0.629854480502)*x[1]+(0.493919301288)*x[2]**o+(0.902910119017)*x[2]
        arg[3,1,1,0]=(0.509160481656)*x[0]**o+(-0.388129364133)*x[0]+(-0.595185230598)*x[1]**o+(-0.289592836865)*x[1]+(0.615663908134)*x[2]**o+(-0.560687711511)*x[2]
        arg[3,1,1,1]=(0.647782473568)*x[0]**o+(0.935840881725)*x[0]+(0.440348212287)*x[1]**o+(-0.489797988572)*x[1]+(0.766592631787)*x[2]**o+(0.217399649246)*x[2]
        arg[3,1,2,0]=(0.64944026906)*x[0]**o+(0.487409918475)*x[0]+(-0.0322210593784)*x[1]**o+(0.281311337861)*x[1]+(-0.694252407571)*x[2]**o+(-0.659678386121)*x[2]
        arg[3,1,2,1]=(0.806357278285)*x[0]**o+(0.419430374567)*x[0]+(-0.0525796600116)*x[1]**o+(-0.556886341914)*x[1]+(-0.288209814683)*x[2]**o+(-0.309696771495)*x[2]
        arg[3,2,0,0]=(-0.300806388527)*x[0]**o+(-0.39334416908)*x[0]+(0.000711617540648)*x[1]**o+(-0.618539958736)*x[1]+(0.0574098223152)*x[2]**o+(-0.351595384168)*x[2]
        arg[3,2,0,1]=(-0.898675793892)*x[0]**o+(0.904479958277)*x[0]+(0.0781945046018)*x[1]**o+(-0.0461053392586)*x[1]+(0.844798127031)*x[2]**o+(-0.82306838709)*x[2]
        arg[3,2,1,0]=(-0.268742335093)*x[0]**o+(-0.532639944208)*x[0]+(0.0101227109551)*x[1]**o+(-0.438312865518)*x[1]+(0.104288648246)*x[2]**o+(-0.115669027311)*x[2]
        arg[3,2,1,1]=(0.639871308792)*x[0]**o+(0.260847628514)*x[0]+(-0.455670719167)*x[1]**o+(0.210619035618)*x[1]+(-0.401809068119)*x[2]**o+(-0.0625194517026)*x[2]
        arg[3,2,2,0]=(-0.713684223829)*x[0]**o+(0.214281604766)*x[0]+(-0.60152592324)*x[1]**o+(0.249807228316)*x[1]+(-0.939395078261)*x[2]**o+(-0.519744396388)*x[2]
        arg[3,2,2,1]=(-0.340588993443)*x[0]**o+(-0.0165543854997)*x[0]+(-0.366479789338)*x[1]**o+(0.316347630987)*x[1]+(-0.457055798679)*x[2]**o+(-0.132368055553)*x[2]
        arg[3,3,0,0]=(-0.745211736715)*x[0]**o+(0.0106787556884)*x[0]+(-0.448180943446)*x[1]**o+(0.694982281891)*x[1]+(0.924431258221)*x[2]**o+(-0.841633806168)*x[2]
        arg[3,3,0,1]=(0.368054364362)*x[0]**o+(0.19288251992)*x[0]+(0.565854068035)*x[1]**o+(0.747534544754)*x[1]+(0.473165789569)*x[2]**o+(-0.59399986706)*x[2]
        arg[3,3,1,0]=(-0.142707671693)*x[0]**o+(0.177095043535)*x[0]+(-0.348946846847)*x[1]**o+(-0.545430013244)*x[1]+(-0.671405653136)*x[2]**o+(-0.531699969702)*x[2]
        arg[3,3,1,1]=(0.783201055631)*x[0]**o+(0.899495042323)*x[0]+(0.117802924059)*x[1]**o+(-0.277400900172)*x[1]+(-0.351268362305)*x[2]**o+(0.52756480897)*x[2]
        arg[3,3,2,0]=(-0.866758006698)*x[0]**o+(0.293139875238)*x[0]+(-0.318394231317)*x[1]**o+(-0.783701622124)*x[1]+(-0.95137399655)*x[2]**o+(-0.474198736639)*x[2]
        arg[3,3,2,1]=(0.0428398560619)*x[0]**o+(-0.970180601102)*x[0]+(-0.753897573393)*x[1]**o+(-0.0873352699251)*x[1]+(-0.131440532574)*x[2]**o+(0.875241605077)*x[2]
        arg[3,4,0,0]=(0.32399493643)*x[0]**o+(0.67295092524)*x[0]+(-0.29221110671)*x[1]**o+(0.118066079866)*x[1]+(-0.566541154667)*x[2]**o+(-0.806367499079)*x[2]
        arg[3,4,0,1]=(0.278776399755)*x[0]**o+(0.472327826919)*x[0]+(-0.956871861949)*x[1]**o+(0.109236874792)*x[1]+(-0.494095116883)*x[2]**o+(-0.194313775518)*x[2]
        arg[3,4,1,0]=(0.650920037415)*x[0]**o+(0.977698516929)*x[0]+(-0.0730777837998)*x[1]**o+(-0.576543152032)*x[1]+(0.781481934657)*x[2]**o+(-0.389328840554)*x[2]
        arg[3,4,1,1]=(-0.798341529522)*x[0]**o+(-0.630245234712)*x[0]+(-0.444550323489)*x[1]**o+(-0.539189248295)*x[1]+(-0.904964423673)*x[2]**o+(-0.611560462796)*x[2]
        arg[3,4,2,0]=(-0.392991417685)*x[0]**o+(0.313599061567)*x[0]+(-0.215747337398)*x[1]**o+(-0.310316872476)*x[1]+(-0.269452617785)*x[2]**o+(0.684353794645)*x[2]
        arg[3,4,2,1]=(-0.586668781403)*x[0]**o+(-0.0711284937177)*x[0]+(0.408273850302)*x[1]**o+(0.761860299288)*x[1]+(0.902079242449)*x[2]**o+(0.70161017722)*x[2]
        ref[0,0,0,0]=(1.43173970815)/(o+1.)+(-0.156396435558)
        ref[0,0,0,1]=(0.459165989186)/(o+1.)+(0.194399341676)
        ref[0,0,1,0]=(0.989528642545)/(o+1.)+(-0.164206368911)
        ref[0,0,1,1]=(1.37152584676)/(o+1.)+(-0.234493053498)
        ref[0,0,2,0]=(0.422741924108)/(o+1.)+(0.0721378026728)
        ref[0,0,2,1]=(-0.574467903086)/(o+1.)+(0.167204674462)
        ref[0,1,0,0]=(1.36619629735)/(o+1.)+(-0.19947081885)
        ref[0,1,0,1]=(-1.60758660729)/(o+1.)+(0.447728236186)
        ref[0,1,1,0]=(1.5106959073)/(o+1.)+(-0.393852180331)
        ref[0,1,1,1]=(1.85678121239)/(o+1.)+(-0.529280397551)
        ref[0,1,2,0]=(-0.796337866944)/(o+1.)+(0.0812355940636)
        ref[0,1,2,1]=(-0.333114331422)/(o+1.)+(1.22614026311)
        ref[0,2,0,0]=(0.516540083621)/(o+1.)+(-0.358761510118)
        ref[0,2,0,1]=(-2.72820411942)/(o+1.)+(0.15972705639)
        ref[0,2,1,0]=(-0.440149466937)/(o+1.)+(-0.124216123318)
        ref[0,2,1,1]=(-0.137940536755)/(o+1.)+(0.13992652657)
        ref[0,2,2,0]=(0.373759361058)/(o+1.)+(0.210996140796)
        ref[0,2,2,1]=(-1.07034621798)/(o+1.)+(1.2905860125)
        ref[0,3,0,0]=(0.923495049095)/(o+1.)+(-0.897043344396)
        ref[0,3,0,1]=(-0.20180799615)/(o+1.)+(0.272652457923)
        ref[0,3,1,0]=(-0.831052678484)/(o+1.)+(0.326929849662)
        ref[0,3,1,1]=(1.41523396695)/(o+1.)+(-0.5275927117)
        ref[0,3,2,0]=(0.405715363821)/(o+1.)+(0.535330516568)
        ref[0,3,2,1]=(0.877269821733)/(o+1.)+(-0.666082812121)
        ref[0,4,0,0]=(0.884126703786)/(o+1.)+(-0.559427998262)
        ref[0,4,0,1]=(-1.36933304826)/(o+1.)+(0.263100182211)
        ref[0,4,1,0]=(0.871486955825)/(o+1.)+(0.381595437312)
        ref[0,4,1,1]=(1.21446122052)/(o+1.)+(-0.642161706726)
        ref[0,4,2,0]=(-1.82861338966)/(o+1.)+(0.507433118531)
        ref[0,4,2,1]=(0.0642039966469)/(o+1.)+(0.551348562604)
        ref[1,0,0,0]=(0.40922616303)/(o+1.)+(-0.474630105482)
        ref[1,0,0,1]=(1.63943889768)/(o+1.)+(0.285622914296)
        ref[1,0,1,0]=(0.0845961115353)/(o+1.)+(0.143628502705)
        ref[1,0,1,1]=(0.62129375936)/(o+1.)+(0.671645835511)
        ref[1,0,2,0]=(-1.2004575453)/(o+1.)+(0.701757992696)
        ref[1,0,2,1]=(0.435963491825)/(o+1.)+(0.26710538393)
        ref[1,1,0,0]=(0.738921325407)/(o+1.)+(-0.0258542444502)
        ref[1,1,0,1]=(1.95599148183)/(o+1.)+(0.652818492883)
        ref[1,1,1,0]=(0.500576647095)/(o+1.)+(0.857652485995)
        ref[1,1,1,1]=(-0.0151750762821)/(o+1.)+(0.238827857598)
        ref[1,1,2,0]=(1.26963005373)/(o+1.)+(-0.299065008888)
        ref[1,1,2,1]=(-0.82289389301)/(o+1.)+(0.14939351077)
        ref[1,2,0,0]=(-0.495951362447)/(o+1.)+(-0.532833788304)
        ref[1,2,0,1]=(-1.87785443639)/(o+1.)+(-0.521209467732)
        ref[1,2,1,0]=(-0.374085962184)/(o+1.)+(0.505583421209)
        ref[1,2,1,1]=(1.01773556605)/(o+1.)+(0.764863466171)
        ref[1,2,2,0]=(-1.54049109838)/(o+1.)+(-0.511296952595)
        ref[1,2,2,1]=(-1.06948953348)/(o+1.)+(-0.215299794811)
        ref[1,3,0,0]=(-1.20198777829)/(o+1.)+(-0.274360734317)
        ref[1,3,0,1]=(1.13519842695)/(o+1.)+(0.506840913294)
        ref[1,3,1,0]=(0.973820116369)/(o+1.)+(0.485786110977)
        ref[1,3,1,1]=(0.155263308828)/(o+1.)+(0.473436850817)
        ref[1,3,2,0]=(0.476069733748)/(o+1.)+(-0.228046397052)
        ref[1,3,2,1]=(-0.234851551487)/(o+1.)+(-0.574065011567)
        ref[1,4,0,0]=(1.95924908786)/(o+1.)+(-0.493262726539)
        ref[1,4,0,1]=(-0.448532656782)/(o+1.)+(0.393812699074)
        ref[1,4,1,0]=(0.235155577808)/(o+1.)+(0.756260258803)
        ref[1,4,1,1]=(0.0334282473144)/(o+1.)+(-0.837406551217)
        ref[1,4,2,0]=(0.568132489684)/(o+1.)+(-0.13055465798)
        ref[1,4,2,1]=(-0.380042596089)/(o+1.)+(0.243737215624)
        ref[2,0,0,0]=(1.76568035823)/(o+1.)+(0.0789834372326)
        ref[2,0,0,1]=(-0.20023919191)/(o+1.)+(0.901466917208)
        ref[2,0,1,0]=(-1.20715877778)/(o+1.)+(-0.992500439486)
        ref[2,0,1,1]=(-0.761913514624)/(o+1.)+(0.465965334128)
        ref[2,0,2,0]=(-0.944104213428)/(o+1.)+(-0.146881270459)
        ref[2,0,2,1]=(0.105283971659)/(o+1.)+(-1.30498791712)
        ref[2,1,0,0]=(-0.804721214141)/(o+1.)+(0.356736186609)
        ref[2,1,0,1]=(-0.0358563948854)/(o+1.)+(-0.24146509617)
        ref[2,1,1,0]=(-0.437556173502)/(o+1.)+(-0.387215578175)
        ref[2,1,1,1]=(-0.0786030951662)/(o+1.)+(0.654838009465)
        ref[2,1,2,0]=(-0.859200342885)/(o+1.)+(-0.112407873885)
        ref[2,1,2,1]=(-0.296679106354)/(o+1.)+(0.0491751507381)
        ref[2,2,0,0]=(0.462408531841)/(o+1.)+(-0.557975521593)
        ref[2,2,0,1]=(-0.0634225791893)/(o+1.)+(0.0468185300489)
        ref[2,2,1,0]=(1.08205349937)/(o+1.)+(0.621257561187)
        ref[2,2,1,1]=(0.88864389409)/(o+1.)+(0.849886105205)
        ref[2,2,2,0]=(2.01702912587)/(o+1.)+(0.755558425167)
        ref[2,2,2,1]=(1.41263009001)/(o+1.)+(0.331281133791)
        ref[2,3,0,0]=(1.91485193421)/(o+1.)+(0.0643042152414)
        ref[2,3,0,1]=(1.86113720031)/(o+1.)+(0.870090339646)
        ref[2,3,1,0]=(0.442533807126)/(o+1.)+(-0.690437405116)
        ref[2,3,1,1]=(1.52774754257)/(o+1.)+(-0.236667038211)
        ref[2,3,2,0]=(1.08558789233)/(o+1.)+(-0.346580362392)
        ref[2,3,2,1]=(-0.771732912166)/(o+1.)+(-0.49697903252)
        ref[2,4,0,0]=(1.0510676744)/(o+1.)+(0.18646253745)
        ref[2,4,0,1]=(-0.878875991119)/(o+1.)+(-0.271299000644)
        ref[2,4,1,0]=(-0.75026757693)/(o+1.)+(-0.168751100981)
        ref[2,4,1,1]=(0.526155063499)/(o+1.)+(1.041034953)
        ref[2,4,2,0]=(0.377792294719)/(o+1.)+(0.433767213151)
        ref[2,4,2,1]=(0.114142925831)/(o+1.)+(-0.972051850709)
        ref[3,0,0,0]=(0.15538316839)/(o+1.)+(-0.861229788808)
        ref[3,0,0,1]=(-0.725501605569)/(o+1.)+(-0.248009661269)
        ref[3,0,1,0]=(0.791443284981)/(o+1.)+(0.618118504587)
        ref[3,0,1,1]=(-0.411749416988)/(o+1.)+(0.125302421245)
        ref[3,0,2,0]=(0.0968028463245)/(o+1.)+(0.592149023513)
        ref[3,0,2,1]=(0.0947000244134)/(o+1.)+(0.286498958939)
        ref[3,1,0,0]=(0.823197755916)/(o+1.)+(-0.378154495725)
        ref[3,1,0,1]=(1.21588193285)/(o+1.)+(0.0126566819682)
        ref[3,1,1,0]=(0.529639159192)/(o+1.)+(-0.619204956255)
        ref[3,1,1,1]=(1.85472331764)/(o+1.)+(0.3317212712)
        ref[3,1,2,0]=(-0.0770331978898)/(o+1.)+(0.0545214351076)
        ref[3,1,2,1]=(0.46556780359)/(o+1.)+(-0.223576369421)
        ref[3,2,0,0]=(-0.242684948671)/(o+1.)+(-0.681739755992)
        ref[3,2,0,1]=(0.0243168377401)/(o+1.)+(0.0176531159642)
        ref[3,2,1,0]=(-0.154330975892)/(o+1.)+(-0.543310918519)
        ref[3,2,1,1]=(-0.217608478494)/(o+1.)+(0.204473606215)
        ref[3,2,2,0]=(-2.25460522533)/(o+1.)+(-0.0278277816531)
        ref[3,2,2,1]=(-1.16412458146)/(o+1.)+(0.0837125949668)
        ref[3,3,0,0]=(-0.26896142194)/(o+1.)+(-0.0679863842941)
        ref[3,3,0,1]=(1.40707422197)/(o+1.)+(0.173208598807)
        ref[3,3,1,0]=(-1.16306017168)/(o+1.)+(-0.450017469706)
        ref[3,3,1,1]=(0.549735617385)/(o+1.)+(0.57482947556)
        ref[3,3,2,0]=(-2.13652623456)/(o+1.)+(-0.482380241762)
        ref[3,3,2,1]=(-0.842498249905)/(o+1.)+(-0.0911371329748)
        ref[3,4,0,0]=(-0.534757324947)/(o+1.)+(-0.00767524698684)
        ref[3,4,0,1]=(-1.17219057908)/(o+1.)+(0.193625463097)
        ref[3,4,1,0]=(1.35932418827)/(o+1.)+(0.00591326217154)
        ref[3,4,1,1]=(-2.14785627668)/(o+1.)+(-0.890497472901)
        ref[3,4,2,0]=(-0.878191372868)/(o+1.)+(0.343817991868)
        ref[3,4,2,1]=(0.723684311348)/(o+1.)+(0.696170991395)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5, 3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunction_fromData_ReducedSolution_rank0(self):
      """
      tests integral of rank 0 Data on the ReducedFunction

      assumptions: ReducedSolution(self.domain) exists
                   self.domain supports integral on ReducedFunction
      """
      o=1
      dim=self.domain.getDim()
      w_ref=ReducedFunction(self.domain)
      w=ReducedSolution(self.domain)
      x=w.getX()
      arg=Data(0,(),w)
      if dim==2:
        arg=(0.729700679932)*x[0]+(-1.38329769179)*x[1]
        ref=(0.329728333821)/(o+1.)+(-0.491662672839)
      else:
        arg=(-1.75076012766)*x[0]+(-0.0862990889349)*x[1]+(0.637429232833)*x[2]
        ref=(-0.577517867896)/(o+1.)+(-0.311056057931)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunction_fromData_ReducedSolution_rank1(self):
      """
      tests integral of rank 1 Data on the ReducedFunction

      assumptions: ReducedSolution(self.domain) exists
                   self.domain supports integral on ReducedFunction
      """
      o=1
      dim=self.domain.getDim()
      w_ref=ReducedFunction(self.domain)
      w=ReducedSolution(self.domain)
      x=w.getX()
      arg=Data(0,(2,),w)
      ref=numpy.zeros((2,),numpy.float_)
      if dim==2:
        arg[0]=(1.0296233191)*x[0]+(-0.347985432388)*x[1]
        arg[1]=(-0.182082548052)*x[0]+(0.114822539064)*x[1]
        ref[0]=(0.876900301573)/(o+1.)+(-0.0976312074308)
        ref[1]=(-1.09276229827)/(o+1.)+(0.512751144643)
      else:
        arg[0]=(0.390938934986)*x[0]+(-0.17813109236)*x[1]+(0.92611177061)*x[2]
        arg[1]=(0.169562781554)*x[0]+(-0.149625135256)*x[1]+(0.357445984273)*x[2]
        ref[0]=(1.07308269316)/(o+1.)+(0.0329184600363)
        ref[1]=(-1.07527269336)/(o+1.)+(0.726328161966)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunction_fromData_ReducedSolution_rank2(self):
      """
      tests integral of rank 2 Data on the ReducedFunction

      assumptions: ReducedSolution(self.domain) exists
                   self.domain supports integral on ReducedFunction
      """
      o=1
      dim=self.domain.getDim()
      w_ref=ReducedFunction(self.domain)
      w=ReducedSolution(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5),w)
      ref=numpy.zeros((4, 5),numpy.float_)
      if dim==2:
        arg[0,0]=(-0.798668686601)*x[0]+(-0.502017718729)*x[1]
        arg[0,1]=(-0.682368031728)*x[0]+(-0.915048032379)*x[1]
        arg[0,2]=(-0.0237771508029)*x[0]+(0.177441262563)*x[1]
        arg[0,3]=(0.135600942216)*x[0]+(0.496697614771)*x[1]
        arg[0,4]=(0.495272681631)*x[0]+(-0.457599050745)*x[1]
        arg[1,0]=(-0.252415627225)*x[0]+(-0.820872546406)*x[1]
        arg[1,1]=(-0.183362696688)*x[0]+(-0.721277471492)*x[1]
        arg[1,2]=(0.401101418876)*x[0]+(0.0483185737314)*x[1]
        arg[1,3]=(0.251246583055)*x[0]+(-0.837901564153)*x[1]
        arg[1,4]=(0.2984760032)*x[0]+(0.0504569221307)*x[1]
        arg[2,0]=(1.18468310151)*x[0]+(-0.545612792222)*x[1]
        arg[2,1]=(-1.08451540544)*x[0]+(0.820927137753)*x[1]
        arg[2,2]=(0.85435600905)*x[0]+(-0.309819803422)*x[1]
        arg[2,3]=(-0.184148030844)*x[0]+(0.978315441698)*x[1]
        arg[2,4]=(1.36943188922)*x[0]+(-1.3176984733)*x[1]
        arg[3,0]=(0.42189213134)*x[0]+(1.23097633986)*x[1]
        arg[3,1]=(0.0409353425792)*x[0]+(-1.15555830367)*x[1]
        arg[3,2]=(0.557942070395)*x[0]+(-0.591659744114)*x[1]
        arg[3,3]=(-1.07432449167)*x[0]+(-0.665475086161)*x[1]
        arg[3,4]=(-1.00421972544)*x[0]+(1.16737316771)*x[1]
        ref[0,0]=(-0.400434079596)/(o+1.)+(-0.450126162867)
        ref[0,1]=(-1.35836659237)/(o+1.)+(-0.11952473587)
        ref[0,2]=(0.154775934077)/(o+1.)+(-0.000555911158727)
        ref[0,3]=(-0.177525502969)/(o+1.)+(0.404912029978)
        ref[0,4]=(1.00122501632)/(o+1.)+(-0.481775692719)
        ref[1,0]=(-1.37507083806)/(o+1.)+(0.150891332214)
        ref[1,1]=(-1.37429063854)/(o+1.)+(0.234825235182)
        ref[1,2]=(1.30069236547)/(o+1.)+(-0.425636186429)
        ref[1,3]=(-0.414658324515)/(o+1.)+(-0.0859983282917)
        ref[1,4]=(0.757097138678)/(o+1.)+(-0.204082106674)
        ref[2,0]=(1.09244584366)/(o+1.)+(-0.226687767185)
        ref[2,1]=(0.109481169277)/(o+1.)+(-0.186534718481)
        ref[2,2]=(0.587125473152)/(o+1.)+(-0.0212946337622)
        ref[2,3]=(0.00878695699842)/(o+1.)+(0.392690226928)
        ref[2,4]=(0.306810290785)/(o+1.)+(-0.127538437433)
        ref[3,0]=(0.317826050636)/(o+1.)+(0.667521210281)
        ref[3,1]=(-0.276254182573)/(o+1.)+(-0.419184389261)
        ref[3,2]=(-0.753831729065)/(o+1.)+(0.360057027673)
        ref[3,3]=(-1.40505001195)/(o+1.)+(-0.167374782943)
        ref[3,4]=(0.53112066724)/(o+1.)+(-0.183983612488)
      else:
        arg[0,0]=(1.54447818842)*x[0]+(-0.560568590939)*x[1]+(-1.33749549019)*x[2]
        arg[0,1]=(-0.535544800192)*x[0]+(-0.732513980801)*x[1]+(0.309855030569)*x[2]
        arg[0,2]=(-0.45251604551)*x[0]+(-0.0933932995351)*x[1]+(-0.465724167794)*x[2]
        arg[0,3]=(0.10344455234)*x[0]+(-0.0451254971766)*x[1]+(-1.39912042607)*x[2]
        arg[0,4]=(0.617585449939)*x[0]+(1.27357810776)*x[1]+(-0.840161322302)*x[2]
        arg[1,0]=(0.650274355535)*x[0]+(0.547805408891)*x[1]+(0.314350326062)*x[2]
        arg[1,1]=(0.697200474806)*x[0]+(0.558681734309)*x[1]+(-0.0277936766536)*x[2]
        arg[1,2]=(-0.508578328229)*x[0]+(-0.513894226666)*x[1]+(-0.528417418023)*x[2]
        arg[1,3]=(-1.50391414876)*x[0]+(0.174080260823)*x[1]+(-1.48528641514)*x[2]
        arg[1,4]=(0.859470776434)*x[0]+(-0.121348493404)*x[1]+(1.69796346547)*x[2]
        arg[2,0]=(1.00593912049)*x[0]+(0.794254020467)*x[1]+(-1.02770358461)*x[2]
        arg[2,1]=(-1.64586076188)*x[0]+(-0.83927167526)*x[1]+(-1.13088449805)*x[2]
        arg[2,2]=(0.0628777004357)*x[0]+(-1.27316091149)*x[1]+(0.417655047748)*x[2]
        arg[2,3]=(-1.43661526118)*x[0]+(0.0676042964527)*x[1]+(0.174947572442)*x[2]
        arg[2,4]=(0.504347051098)*x[0]+(-1.62976096707)*x[1]+(0.61579932957)*x[2]
        arg[3,0]=(1.05212645875)*x[0]+(-1.10517003049)*x[1]+(0.902242097595)*x[2]
        arg[3,1]=(-0.997611854104)*x[0]+(-1.53611378293)*x[1]+(0.322835276897)*x[2]
        arg[3,2]=(-0.815135194015)*x[0]+(0.357734619962)*x[1]+(0.684966069635)*x[2]
        arg[3,3]=(0.4545417685)*x[0]+(0.386262464083)*x[1]+(0.131414279708)*x[2]
        arg[3,4]=(-0.222694646001)*x[0]+(-0.470499833621)*x[1]+(-0.593290862391)*x[2]
        ref[0,0]=(-0.593803934901)/(o+1.)+(0.120109021094)
        ref[0,1]=(0.0569115377867)/(o+1.)+(-0.507557644105)
        ref[0,2]=(-0.202741590274)/(o+1.)+(-0.404445961283)
        ref[0,3]=(-0.516888777015)/(o+1.)+(-0.411956296948)
        ref[0,4]=(0.52292787417)/(o+1.)+(0.264037180616)
        ref[1,0]=(-0.102685279742)/(o+1.)+(0.807557685115)
        ref[1,1]=(1.44527264276)/(o+1.)+(-0.108592055148)
        ref[1,2]=(-0.448767718864)/(o+1.)+(-0.551061127027)
        ref[1,3]=(-1.19936004685)/(o+1.)+(-0.807880128113)
        ref[1,4]=(1.09918755605)/(o+1.)+(0.668449096223)
        ref[2,0]=(-0.873708007453)/(o+1.)+(0.823098781902)
        ref[2,1]=(-1.9697776018)/(o+1.)+(-0.823119666698)
        ref[2,2]=(0.598023292545)/(o+1.)+(-0.695325727928)
        ref[2,3]=(-0.486077421455)/(o+1.)+(-0.353992985415)
        ref[2,4]=(0.65037669171)/(o+1.)+(-0.579995639058)
        ref[3,0]=(0.115401170876)/(o+1.)+(0.366898677493)
        ref[3,1]=(-0.836209920174)/(o+1.)+(-0.68734021998)
        ref[3,2]=(0.753819223079)/(o+1.)+(-0.263126863749)
        ref[3,3]=(-0.75294945942)/(o+1.)+(0.862583985856)
        ref[3,4]=(-1.44356156537)/(o+1.)+(0.0785381116785)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunction_fromData_ReducedSolution_rank3(self):
      """
      tests integral of rank 3 Data on the ReducedFunction

      assumptions: ReducedSolution(self.domain) exists
                   self.domain supports integral on ReducedFunction
      """
      o=1
      dim=self.domain.getDim()
      w_ref=ReducedFunction(self.domain)
      w=ReducedSolution(self.domain)
      x=w.getX()
      arg=Data(0,(6, 2, 2),w)
      ref=numpy.zeros((6, 2, 2),numpy.float_)
      if dim==2:
        arg[0,0,0]=(-0.409425595197)*x[0]+(-0.992427840513)*x[1]
        arg[0,0,1]=(0.645693185569)*x[0]+(-1.38338827813)*x[1]
        arg[0,1,0]=(0.838847764041)*x[0]+(-1.01358491173)*x[1]
        arg[0,1,1]=(0.381938057912)*x[0]+(0.61979806953)*x[1]
        arg[1,0,0]=(-1.36783302742)*x[0]+(-0.132422986617)*x[1]
        arg[1,0,1]=(-0.137283745381)*x[0]+(0.633534018338)*x[1]
        arg[1,1,0]=(0.0698189930006)*x[0]+(-1.45980415637)*x[1]
        arg[1,1,1]=(-0.391725001578)*x[0]+(0.355258883756)*x[1]
        arg[2,0,0]=(-0.596091494216)*x[0]+(-0.911870026464)*x[1]
        arg[2,0,1]=(0.138007894626)*x[0]+(1.76859278751)*x[1]
        arg[2,1,0]=(-0.791803431843)*x[0]+(-0.813170959331)*x[1]
        arg[2,1,1]=(-1.05466714677)*x[0]+(-1.12822729059)*x[1]
        arg[3,0,0]=(-0.417845531677)*x[0]+(1.13689234148)*x[1]
        arg[3,0,1]=(-0.0323347813877)*x[0]+(1.14381520037)*x[1]
        arg[3,1,0]=(0.639524393892)*x[0]+(0.864696525716)*x[1]
        arg[3,1,1]=(0.593104139176)*x[0]+(0.812642260561)*x[1]
        arg[4,0,0]=(-0.0580034066457)*x[0]+(0.802938738147)*x[1]
        arg[4,0,1]=(0.722143376522)*x[0]+(0.843814429252)*x[1]
        arg[4,1,0]=(-0.574057925551)*x[0]+(-0.298232158953)*x[1]
        arg[4,1,1]=(0.422529038525)*x[0]+(0.841942812244)*x[1]
        arg[5,0,0]=(1.90272013575)*x[0]+(-1.84811498267)*x[1]
        arg[5,0,1]=(-1.52156314991)*x[0]+(-0.797833417606)*x[1]
        arg[5,1,0]=(0.723748480291)*x[0]+(0.8667659268)*x[1]
        arg[5,1,1]=(-0.70289499551)*x[0]+(-0.726595802952)*x[1]
        ref[0,0,0]=(-1.1719312491)/(o+1.)+(-0.114961093303)
        ref[0,0,1]=(0.229993657297)/(o+1.)+(-0.483844374927)
        ref[0,1,0]=(-0.630442924796)/(o+1.)+(0.227852888553)
        ref[0,1,1]=(-0.806392286115)/(o+1.)+(0.904064206778)
        ref[1,0,0]=(-1.38312106397)/(o+1.)+(-0.0585674750338)
        ref[1,0,1]=(0.595552955095)/(o+1.)+(-0.049651341069)
        ref[1,1,0]=(-1.63606241234)/(o+1.)+(0.123038624483)
        ref[1,1,1]=(-0.340463713286)/(o+1.)+(0.151998797732)
        ref[2,0,0]=(-1.31925158304)/(o+1.)+(-0.094354968822)
        ref[2,0,1]=(0.699277735512)/(o+1.)+(0.60366147331)
        ref[2,1,0]=(-0.219878898763)/(o+1.)+(-0.692547746205)
        ref[2,1,1]=(-1.46750249607)/(o+1.)+(-0.357695970647)
        ref[3,0,0]=(0.71941698906)/(o+1.)+(-0.000185089626866)
        ref[3,0,1]=(-0.526993313502)/(o+1.)+(0.819236866243)
        ref[3,1,0]=(1.51671320209)/(o+1.)+(-0.00624614123899)
        ref[3,1,1]=(1.05185683613)/(o+1.)+(0.176944781803)
        ref[4,0,0]=(1.20432711487)/(o+1.)+(-0.229695891683)
        ref[4,0,1]=(1.26863112516)/(o+1.)+(0.14866334031)
        ref[4,1,0]=(0.34377834331)/(o+1.)+(-0.608034213907)
        ref[4,1,1]=(-0.189568721058)/(o+1.)+(0.727020285914)
        ref[5,0,0]=(0.116981035747)/(o+1.)+(-0.0311879413343)
        ref[5,0,1]=(-0.638728138162)/(o+1.)+(-0.840334214679)
        ref[5,1,0]=(0.891634589919)/(o+1.)+(0.349439908586)
        ref[5,1,1]=(-0.552480077628)/(o+1.)+(-0.438505360418)
      else:
        arg[0,0,0]=(-0.662295604848)*x[0]+(1.04606412328)*x[1]+(-0.376008345205)*x[2]
        arg[0,0,1]=(1.25030158303)*x[0]+(-0.0735167236581)*x[1]+(1.31623868156)*x[2]
        arg[0,1,0]=(0.717953620717)*x[0]+(-0.700464709991)*x[1]+(0.0345381771501)*x[2]
        arg[0,1,1]=(0.225944321569)*x[0]+(-1.09510173632)*x[1]+(-0.371235420166)*x[2]
        arg[1,0,0]=(-0.0755865552996)*x[0]+(-0.168214695387)*x[1]+(-0.437427629408)*x[2]
        arg[1,0,1]=(-1.52674685128)*x[0]+(-0.408158359188)*x[1]+(0.750091649824)*x[2]
        arg[1,1,0]=(-0.572400948498)*x[0]+(-0.802826119419)*x[1]+(-1.0233358643)*x[2]
        arg[1,1,1]=(0.623156259061)*x[0]+(0.378774331134)*x[1]+(0.0419766309042)*x[2]
        arg[2,0,0]=(0.443271002813)*x[0]+(-0.930092172267)*x[1]+(-0.20199150178)*x[2]
        arg[2,0,1]=(0.408614848881)*x[0]+(-1.3007040171)*x[1]+(-0.872615408174)*x[2]
        arg[2,1,0]=(-0.174910742406)*x[0]+(1.64872582781)*x[1]+(0.701172036891)*x[2]
        arg[2,1,1]=(0.417145974895)*x[0]+(0.428983721984)*x[1]+(-0.000787616769604)*x[2]
        arg[3,0,0]=(0.655644872855)*x[0]+(-0.268612479186)*x[1]+(0.11779790898)*x[2]
        arg[3,0,1]=(-0.244670332245)*x[0]+(-0.857502890419)*x[1]+(0.111792341131)*x[2]
        arg[3,1,0]=(0.131303542183)*x[0]+(0.231230724482)*x[1]+(0.116997126513)*x[2]
        arg[3,1,1]=(0.725434968911)*x[0]+(1.00206695553)*x[1]+(-0.319715936784)*x[2]
        arg[4,0,0]=(1.30949067662)*x[0]+(-1.14809917605)*x[1]+(0.00529778192088)*x[2]
        arg[4,0,1]=(0.969626585498)*x[0]+(-0.517908035476)*x[1]+(1.0467168328)*x[2]
        arg[4,1,0]=(0.362824899886)*x[0]+(0.890586602899)*x[1]+(-0.135163953653)*x[2]
        arg[4,1,1]=(-0.10353577028)*x[0]+(-0.0997924233294)*x[1]+(0.308973949876)*x[2]
        arg[5,0,0]=(-1.37242244095)*x[0]+(-0.746928231432)*x[1]+(-1.37385481504)*x[2]
        arg[5,0,1]=(0.231730881472)*x[0]+(-0.250265629505)*x[1]+(-0.903599685792)*x[2]
        arg[5,1,0]=(-0.813783753893)*x[0]+(0.41081474894)*x[1]+(-0.541868665288)*x[2]
        arg[5,1,1]=(0.19296641571)*x[0]+(1.8056250646)*x[1]+(0.283675443031)*x[2]
        ref[0,0,0]=(0.859538687531)/(o+1.)+(-0.425889257151)
        ref[0,0,1]=(-0.0674840316504)/(o+1.)+(1.28025378629)
        ref[0,1,0]=(-0.76922088579)/(o+1.)+(0.410623986833)
        ref[0,1,1]=(-0.582981712778)/(o+1.)+(-0.328705561068)
        ref[1,0,0]=(1.25605113432)/(o+1.)+(-0.968640007206)
        ref[1,0,1]=(-1.59448994908)/(o+1.)+(0.204838194218)
        ref[1,1,0]=(-2.07497671334)/(o+1.)+(-0.161793109438)
        ref[1,1,1]=(0.199659680372)/(o+1.)+(0.422123770364)
        ref[2,0,0]=(-1.86970457224)/(o+1.)+(0.590445950504)
        ref[2,0,1]=(-1.09505237157)/(o+1.)+(-0.334826102411)
        ref[2,1,0]=(0.395146564722)/(o+1.)+(0.889920278785)
        ref[2,1,1]=(1.13315474193)/(o+1.)+(-0.143906330909)
        ref[3,0,0]=(-0.804398097072)/(o+1.)+(0.65461419986)
        ref[3,0,1]=(-0.829074327243)/(o+1.)+(-0.0806532771448)
        ref[3,1,0]=(0.126278821088)/(o+1.)+(0.176626286045)
        ref[3,1,1]=(-0.498753315826)/(o+1.)+(0.953269651743)
        ref[4,0,0]=(0.33849802718)/(o+1.)+(-0.0859043723417)
        ref[4,0,1]=(0.153172891455)/(o+1.)+(0.672631245684)
        ref[4,1,0]=(1.42803542207)/(o+1.)+(-0.154893936468)
        ref[4,1,1]=(0.605555698814)/(o+1.)+(-0.249954971274)
        ref[5,0,0]=(-1.21823919194)/(o+1.)+(-1.13748314774)
        ref[5,0,1]=(-1.13684484437)/(o+1.)+(0.107355205274)
        ref[5,1,0]=(-1.21991007491)/(o+1.)+(0.137536202333)
        ref[5,1,1]=(2.05825898341)/(o+1.)+(0.112003969962)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunction_fromData_ReducedSolution_rank4(self):
      """
      tests integral of rank 4 Data on the ReducedFunction

      assumptions: ReducedSolution(self.domain) exists
                   self.domain supports integral on ReducedFunction
      """
      o=1
      dim=self.domain.getDim()
      w_ref=ReducedFunction(self.domain)
      w=ReducedSolution(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5, 3, 2),w)
      ref=numpy.zeros((4, 5, 3, 2),numpy.float_)
      if dim==2:
        arg[0,0,0,0]=(-0.2631296794)*x[0]+(-0.130117190301)*x[1]
        arg[0,0,0,1]=(0.635732952218)*x[0]+(-0.647555274619)*x[1]
        arg[0,0,1,0]=(1.06301179339)*x[0]+(-0.227813591181)*x[1]
        arg[0,0,1,1]=(-0.519979711376)*x[0]+(1.04747480856)*x[1]
        arg[0,0,2,0]=(0.0232818794686)*x[0]+(-1.14799687633)*x[1]
        arg[0,0,2,1]=(0.359605391616)*x[0]+(-1.32942998109)*x[1]
        arg[0,1,0,0]=(-0.555284020205)*x[0]+(0.11676202509)*x[1]
        arg[0,1,0,1]=(-1.83479976366)*x[0]+(0.413168061643)*x[1]
        arg[0,1,1,0]=(0.239736306037)*x[0]+(0.680002261596)*x[1]
        arg[0,1,1,1]=(0.0555978436642)*x[0]+(1.28910533087)*x[1]
        arg[0,1,2,0]=(0.732696470486)*x[0]+(0.538837927425)*x[1]
        arg[0,1,2,1]=(0.528018376979)*x[0]+(1.20428218411)*x[1]
        arg[0,2,0,0]=(0.863803666388)*x[0]+(0.187082373754)*x[1]
        arg[0,2,0,1]=(-1.41490739709)*x[0]+(-0.741935114995)*x[1]
        arg[0,2,1,0]=(0.0501211708116)*x[0]+(-0.370371645307)*x[1]
        arg[0,2,1,1]=(-0.0353423795999)*x[0]+(1.00911245551)*x[1]
        arg[0,2,2,0]=(1.79997397117)*x[0]+(0.240103369892)*x[1]
        arg[0,2,2,1]=(0.926052414873)*x[0]+(0.00914357056458)*x[1]
        arg[0,3,0,0]=(-1.38911305622)*x[0]+(-1.29285413766)*x[1]
        arg[0,3,0,1]=(0.464908270265)*x[0]+(-0.354824878801)*x[1]
        arg[0,3,1,0]=(0.17477491298)*x[0]+(0.882807721541)*x[1]
        arg[0,3,1,1]=(0.633177386897)*x[0]+(-0.312628707154)*x[1]
        arg[0,3,2,0]=(0.0841289229919)*x[0]+(-0.244085514309)*x[1]
        arg[0,3,2,1]=(0.198631529554)*x[0]+(-0.324387433152)*x[1]
        arg[0,4,0,0]=(-1.41503119851)*x[0]+(-0.760818812719)*x[1]
        arg[0,4,0,1]=(-0.0326236565753)*x[0]+(0.835716629913)*x[1]
        arg[0,4,1,0]=(1.84482629546)*x[0]+(-0.341557269651)*x[1]
        arg[0,4,1,1]=(-0.380600663728)*x[0]+(-0.295663906776)*x[1]
        arg[0,4,2,0]=(0.411308057157)*x[0]+(1.0109767907)*x[1]
        arg[0,4,2,1]=(-1.10494637617)*x[0]+(0.401374247774)*x[1]
        arg[1,0,0,0]=(0.437019391027)*x[0]+(0.0507256248679)*x[1]
        arg[1,0,0,1]=(-0.399457976892)*x[0]+(1.60118981524)*x[1]
        arg[1,0,1,0]=(-1.51289007675)*x[0]+(0.821406477166)*x[1]
        arg[1,0,1,1]=(0.405770274494)*x[0]+(0.637591733337)*x[1]
        arg[1,0,2,0]=(-1.4945593715)*x[0]+(1.0919176668)*x[1]
        arg[1,0,2,1]=(-0.482868528044)*x[0]+(0.578004764959)*x[1]
        arg[1,1,0,0]=(-1.44298476434)*x[0]+(0.197151370973)*x[1]
        arg[1,1,0,1]=(1.38517677391)*x[0]+(0.915352009234)*x[1]
        arg[1,1,1,0]=(1.69025511761)*x[0]+(0.0203268727501)*x[1]
        arg[1,1,1,1]=(0.615392847773)*x[0]+(-0.301873832451)*x[1]
        arg[1,1,2,0]=(0.3081860115)*x[0]+(-0.495230532585)*x[1]
        arg[1,1,2,1]=(-0.0258752746808)*x[0]+(0.441629273958)*x[1]
        arg[1,2,0,0]=(-0.947577742489)*x[0]+(0.218632343222)*x[1]
        arg[1,2,0,1]=(0.520029690788)*x[0]+(0.403806873912)*x[1]
        arg[1,2,1,0]=(1.32820278922)*x[0]+(-1.68314905373)*x[1]
        arg[1,2,1,1]=(1.30409876916)*x[0]+(-0.721819171819)*x[1]
        arg[1,2,2,0]=(1.37872038109)*x[0]+(0.0725139769178)*x[1]
        arg[1,2,2,1]=(0.35593830057)*x[0]+(0.28751298465)*x[1]
        arg[1,3,0,0]=(-1.17950311406)*x[0]+(-0.232819397686)*x[1]
        arg[1,3,0,1]=(0.17340184781)*x[0]+(-0.43329852484)*x[1]
        arg[1,3,1,0]=(-1.71125828064)*x[0]+(0.957284609667)*x[1]
        arg[1,3,1,1]=(-0.429043922461)*x[0]+(0.15749154348)*x[1]
        arg[1,3,2,0]=(-1.42626208855)*x[0]+(-1.12812746347)*x[1]
        arg[1,3,2,1]=(0.490516645689)*x[0]+(0.559039786338)*x[1]
        arg[1,4,0,0]=(0.436395831147)*x[0]+(0.382691724897)*x[1]
        arg[1,4,0,1]=(1.27903773839)*x[0]+(-0.0642376846324)*x[1]
        arg[1,4,1,0]=(0.503159792753)*x[0]+(0.392973700403)*x[1]
        arg[1,4,1,1]=(0.0881500114326)*x[0]+(0.834121935361)*x[1]
        arg[1,4,2,0]=(-0.620745962116)*x[0]+(-0.0770814791601)*x[1]
        arg[1,4,2,1]=(0.0231684675117)*x[0]+(-1.33994827808)*x[1]
        arg[2,0,0,0]=(-0.972311570714)*x[0]+(-0.235293375636)*x[1]
        arg[2,0,0,1]=(0.21633355466)*x[0]+(0.11325401613)*x[1]
        arg[2,0,1,0]=(0.922986727965)*x[0]+(-1.00516246282)*x[1]
        arg[2,0,1,1]=(-0.740778981838)*x[0]+(0.616728971608)*x[1]
        arg[2,0,2,0]=(1.73201175195)*x[0]+(-1.1634325209)*x[1]
        arg[2,0,2,1]=(1.15829744421)*x[0]+(-1.87219685708)*x[1]
        arg[2,1,0,0]=(-0.458901790255)*x[0]+(1.13812715713)*x[1]
        arg[2,1,0,1]=(1.03315725208)*x[0]+(-0.112487270341)*x[1]
        arg[2,1,1,0]=(-1.7070220527)*x[0]+(0.126058081856)*x[1]
        arg[2,1,1,1]=(0.863301459388)*x[0]+(0.226670498194)*x[1]
        arg[2,1,2,0]=(-1.0012992544)*x[0]+(-0.195604893926)*x[1]
        arg[2,1,2,1]=(-0.128048233768)*x[0]+(0.827383267773)*x[1]
        arg[2,2,0,0]=(-0.0876478859699)*x[0]+(1.69600082596)*x[1]
        arg[2,2,0,1]=(-0.616278041482)*x[0]+(-0.995548774612)*x[1]
        arg[2,2,1,0]=(0.780699602657)*x[0]+(-0.721498796198)*x[1]
        arg[2,2,1,1]=(-1.17591990575)*x[0]+(-0.66080847955)*x[1]
        arg[2,2,2,0]=(1.71259481962)*x[0]+(-1.12456334234)*x[1]
        arg[2,2,2,1]=(-0.721345747555)*x[0]+(-1.28222132706)*x[1]
        arg[2,3,0,0]=(-1.25971324146)*x[0]+(-0.00559590465243)*x[1]
        arg[2,3,0,1]=(-0.49234245798)*x[0]+(0.30900436328)*x[1]
        arg[2,3,1,0]=(1.3068919966)*x[0]+(-0.891484126193)*x[1]
        arg[2,3,1,1]=(0.319277203381)*x[0]+(-0.535113401495)*x[1]
        arg[2,3,2,0]=(0.529480518887)*x[0]+(0.00452686996008)*x[1]
        arg[2,3,2,1]=(-1.0978905592)*x[0]+(-0.421513714335)*x[1]
        arg[2,4,0,0]=(-1.27711509307)*x[0]+(-1.42208296178)*x[1]
        arg[2,4,0,1]=(-0.621292314868)*x[0]+(-1.10962518414)*x[1]
        arg[2,4,1,0]=(-0.183478331213)*x[0]+(1.0775900659)*x[1]
        arg[2,4,1,1]=(1.07354953663)*x[0]+(0.869765287054)*x[1]
        arg[2,4,2,0]=(-0.696050139647)*x[0]+(-0.120769338549)*x[1]
        arg[2,4,2,1]=(-0.895597167617)*x[0]+(1.41025466606)*x[1]
        arg[3,0,0,0]=(1.15262876611)*x[0]+(-0.505428168285)*x[1]
        arg[3,0,0,1]=(-0.277585299138)*x[0]+(0.352683243558)*x[1]
        arg[3,0,1,0]=(0.218694567952)*x[0]+(-0.170325565499)*x[1]
        arg[3,0,1,1]=(0.297325594036)*x[0]+(0.459308129811)*x[1]
        arg[3,0,2,0]=(-0.572509624935)*x[0]+(0.756188981433)*x[1]
        arg[3,0,2,1]=(0.226462444009)*x[0]+(-1.42100784266)*x[1]
        arg[3,1,0,0]=(-0.28388824767)*x[0]+(-0.306064847514)*x[1]
        arg[3,1,0,1]=(-0.805831316478)*x[0]+(0.756933495912)*x[1]
        arg[3,1,1,0]=(1.37344361233)*x[0]+(-0.377600924857)*x[1]
        arg[3,1,1,1]=(-0.0725302533884)*x[0]+(-0.425074450201)*x[1]
        arg[3,1,2,0]=(-0.066659431931)*x[0]+(1.39252394891)*x[1]
        arg[3,1,2,1]=(0.593440019224)*x[0]+(-0.945629564385)*x[1]
        arg[3,2,0,0]=(0.101850471598)*x[0]+(0.0552356415787)*x[1]
        arg[3,2,0,1]=(-0.407603005372)*x[0]+(-0.849371670514)*x[1]
        arg[3,2,1,0]=(0.435403192155)*x[0]+(0.213673097377)*x[1]
        arg[3,2,1,1]=(0.424081872219)*x[0]+(-0.0800529063085)*x[1]
        arg[3,2,2,0]=(0.189428134309)*x[0]+(-0.205813522374)*x[1]
        arg[3,2,2,1]=(-0.51449771505)*x[0]+(0.0570738169919)*x[1]
        arg[3,3,0,0]=(1.05431625119)*x[0]+(-0.0583139615539)*x[1]
        arg[3,3,0,1]=(0.0292261008496)*x[0]+(-0.569482715214)*x[1]
        arg[3,3,1,0]=(-0.155861158232)*x[0]+(0.628422800294)*x[1]
        arg[3,3,1,1]=(-1.24050157391)*x[0]+(-1.83949885089)*x[1]
        arg[3,3,2,0]=(-0.140709404689)*x[0]+(0.975529832713)*x[1]
        arg[3,3,2,1]=(-0.290824210426)*x[0]+(-0.198373932879)*x[1]
        arg[3,4,0,0]=(0.627236720719)*x[0]+(-0.306676521516)*x[1]
        arg[3,4,0,1]=(-0.964303098626)*x[0]+(-0.701698363587)*x[1]
        arg[3,4,1,0]=(-0.0447860906704)*x[0]+(0.324659175162)*x[1]
        arg[3,4,1,1]=(1.28273443124)*x[0]+(-0.764329261143)*x[1]
        arg[3,4,2,0]=(-0.023232709502)*x[0]+(-0.124231212653)*x[1]
        arg[3,4,2,1]=(0.510754136991)*x[0]+(0.570492207442)*x[1]
        ref[0,0,0,0]=(0.648764087616)/(o+1.)+(-0.521005478658)
        ref[0,0,0,1]=(-0.0127476670646)/(o+1.)+(0.000462672331599)
        ref[0,0,1,0]=(0.258601710827)/(o+1.)+(0.288298245692)
        ref[0,0,1,1]=(0.0577415806545)/(o+1.)+(0.234876758266)
        ref[0,0,2,0]=(-0.91129780757)/(o+1.)+(-0.106708594644)
        ref[0,0,2,1]=(0.0841946322525)/(o+1.)+(-0.527009610865)
        ref[0,1,0,0]=(0.111101563297)/(o+1.)+(-0.274811779206)
        ref[0,1,0,1]=(-0.504062576651)/(o+1.)+(-0.458784562683)
        ref[0,1,1,0]=(0.65560969277)/(o+1.)+(0.132064437431)
        ref[0,1,1,1]=(0.0866382135955)/(o+1.)+(0.629032480469)
        ref[0,1,2,0]=(1.36074036394)/(o+1.)+(-0.0446029830125)
        ref[0,1,2,1]=(1.16071753175)/(o+1.)+(0.285791514672)
        ref[0,2,0,0]=(0.693065154772)/(o+1.)+(0.178910442685)
        ref[0,2,0,1]=(-1.41209835364)/(o+1.)+(-0.372372079222)
        ref[0,2,1,0]=(-0.542625642859)/(o+1.)+(0.111187584182)
        ref[0,2,1,1]=(0.628714449405)/(o+1.)+(0.172527813253)
        ref[0,2,2,0]=(1.62887921583)/(o+1.)+(0.205599062617)
        ref[0,2,2,1]=(1.0640301541)/(o+1.)+(-0.0644170843301)
        ref[0,3,0,0]=(-1.09870552313)/(o+1.)+(-0.791630835375)
        ref[0,3,0,1]=(0.260706656546)/(o+1.)+(-0.0753116325413)
        ref[0,3,1,0]=(-0.0543842358149)/(o+1.)+(0.555983435168)
        ref[0,3,1,1]=(0.0369075211917)/(o+1.)+(0.141820579276)
        ref[0,3,2,0]=(-0.784250881508)/(o+1.)+(0.312147145095)
        ref[0,3,2,1]=(-0.178450092422)/(o+1.)+(0.0263470944113)
        ref[0,4,0,0]=(-0.993852999635)/(o+1.)+(-0.590998505799)
        ref[0,4,0,1]=(-0.0449039600312)/(o+1.)+(0.423998466684)
        ref[0,4,1,0]=(0.583535069477)/(o+1.)+(0.459866978166)
        ref[0,4,1,1]=(-0.251020886878)/(o+1.)+(-0.212621841813)
        ref[0,4,2,0]=(0.408688561613)/(o+1.)+(0.506798143121)
        ref[0,4,2,1]=(0.537090447747)/(o+1.)+(-0.620331288072)
        ref[1,0,0,0]=(0.37428496839)/(o+1.)+(0.0567300237522)
        ref[1,0,0,1]=(0.777672833448)/(o+1.)+(0.212029502451)
        ref[1,0,1,0]=(-0.745422608709)/(o+1.)+(0.026969504563)
        ref[1,0,1,1]=(0.466434983875)/(o+1.)+(0.288463511978)
        ref[1,0,2,0]=(0.237457788829)/(o+1.)+(-0.320049746767)
        ref[1,0,2,1]=(0.346974181328)/(o+1.)+(-0.125918972207)
        ref[1,1,0,0]=(-0.823737519034)/(o+1.)+(-0.211047937165)
        ref[1,1,0,1]=(1.00324369541)/(o+1.)+(0.648642543871)
        ref[1,1,1,0]=(0.674013158596)/(o+1.)+(0.518284415882)
        ref[1,1,1,1]=(0.403142463029)/(o+1.)+(-0.0448117238535)
        ref[1,1,2,0]=(0.0399328921703)/(o+1.)+(-0.113488706628)
        ref[1,1,2,1]=(0.111446633645)/(o+1.)+(0.152153682816)
        ref[1,2,0,0]=(-0.481646721692)/(o+1.)+(-0.123649338787)
        ref[1,2,0,1]=(0.857170000026)/(o+1.)+(0.033333282337)
        ref[1,2,1,0]=(-0.445362398126)/(o+1.)+(0.0452080668082)
        ref[1,2,1,1]=(-0.256201827386)/(o+1.)+(0.419240712362)
        ref[1,2,2,0]=(1.02991996311)/(o+1.)+(0.210657197448)
        ref[1,2,2,1]=(0.355299316895)/(o+1.)+(0.144075984162)
        ref[1,3,0,0]=(-0.679726322154)/(o+1.)+(-0.366298094796)
        ref[1,3,0,1]=(-0.67885027329)/(o+1.)+(0.20947679813)
        ref[1,3,1,0]=(-0.163081485233)/(o+1.)+(-0.295446092868)
        ref[1,3,1,1]=(-0.035768623673)/(o+1.)+(-0.117891877654)
        ref[1,3,2,0]=(-0.639368943038)/(o+1.)+(-0.957510304493)
        ref[1,3,2,1]=(0.160615066001)/(o+1.)+(0.444470683013)
        ref[1,4,0,0]=(1.0649487216)/(o+1.)+(-0.122930582777)
        ref[1,4,0,1]=(0.22766574122)/(o+1.)+(0.493567156271)
        ref[1,4,1,0]=(-0.481296537412)/(o+1.)+(0.688715015285)
        ref[1,4,1,1]=(-0.195268237132)/(o+1.)+(0.558770091963)
        ref[1,4,2,0]=(0.180618258117)/(o+1.)+(-0.439222849696)
        ref[1,4,2,1]=(0.0261012984826)/(o+1.)+(-0.671440554524)
        ref[2,0,0,0]=(-0.301101739758)/(o+1.)+(-0.453251603296)
        ref[2,0,0,1]=(0.00878006761908)/(o+1.)+(0.160403751586)
        ref[2,0,1,0]=(-0.80011300584)/(o+1.)+(0.35896863549)
        ref[2,0,1,1]=(0.00248831604776)/(o+1.)+(-0.0632691631389)
        ref[2,0,2,0]=(-0.0569220251136)/(o+1.)+(0.312750628078)
        ref[2,0,2,1]=(-0.440109525096)/(o+1.)+(-0.136894943888)
        ref[2,1,0,0]=(-0.465652661854)/(o+1.)+(0.572439014362)
        ref[2,1,0,1]=(0.0522958647723)/(o+1.)+(0.434187058485)
        ref[2,1,1,0]=(0.00194260180185)/(o+1.)+(-0.791453286324)
        ref[2,1,1,1]=(0.981996325082)/(o+1.)+(0.0539878162501)
        ref[2,1,2,0]=(-0.243334369882)/(o+1.)+(-0.476784889223)
        ref[2,1,2,1]=(1.53208640175)/(o+1.)+(-0.416375683873)
        ref[2,2,0,0]=(1.27849139532)/(o+1.)+(0.164930772337)
        ref[2,2,0,1]=(-1.4813889023)/(o+1.)+(-0.0652189568992)
        ref[2,2,1,0]=(-0.728282325831)/(o+1.)+(0.393741566145)
        ref[2,2,1,1]=(-0.537765426582)/(o+1.)+(-0.649481479359)
        ref[2,2,2,0]=(-0.266224153785)/(o+1.)+(0.427127815533)
        ref[2,2,2,1]=(-0.862349098277)/(o+1.)+(-0.570608988167)
        ref[2,3,0,0]=(-1.09340368619)/(o+1.)+(-0.0859527299593)
        ref[2,3,0,1]=(1.02842400947)/(o+1.)+(-0.605881052086)
        ref[2,3,1,0]=(0.0131810342609)/(o+1.)+(0.20111341807)
        ref[2,3,1,1]=(-1.11244447248)/(o+1.)+(0.448304137186)
        ref[2,3,2,0]=(-0.978348257673)/(o+1.)+(0.75617782326)
        ref[2,3,2,1]=(-0.140801959181)/(o+1.)+(-0.689301157175)
        ref[2,4,0,0]=(-0.733659962936)/(o+1.)+(-0.982769045955)
        ref[2,4,0,1]=(-1.6278906672)/(o+1.)+(-0.0515134159034)
        ref[2,4,1,0]=(1.52117858199)/(o+1.)+(-0.313533423654)
        ref[2,4,1,1]=(0.621221455007)/(o+1.)+(0.661046684337)
        ref[2,4,2,0]=(-0.157580028393)/(o+1.)+(-0.329619724901)
        ref[2,4,2,1]=(0.52303412622)/(o+1.)+(-0.00418831388809)
        ref[3,0,0,0]=(0.337890126112)/(o+1.)+(0.154655235856)
        ref[3,0,0,1]=(-0.100490423466)/(o+1.)+(0.0877941839427)
        ref[3,0,1,0]=(0.37549944857)/(o+1.)+(-0.163565223059)
        ref[3,0,1,1]=(-0.3212149071)/(o+1.)+(0.538924315474)
        ref[3,0,2,0]=(-0.359815373231)/(o+1.)+(0.271747364864)
        ref[3,0,2,1]=(-0.890575856923)/(o+1.)+(-0.151984770865)
        ref[3,1,0,0]=(-1.10384271628)/(o+1.)+(0.256944810546)
        ref[3,1,0,1]=(0.647425296535)/(o+1.)+(-0.348161558551)
        ref[3,1,1,0]=(1.41806770884)/(o+1.)+(-0.211112510685)
        ref[3,1,1,1]=(0.106528838317)/(o+1.)+(-0.302066770953)
        ref[3,1,2,0]=(0.917213295746)/(o+1.)+(0.204325610614)
        ref[3,1,2,1]=(-0.24738513406)/(o+1.)+(-0.0524022055505)
        ref[3,2,0,0]=(-0.0537683888373)/(o+1.)+(0.105427251007)
        ref[3,2,0,1]=(-1.24647336894)/(o+1.)+(-0.00525065347416)
        ref[3,2,1,0]=(0.187132797735)/(o+1.)+(0.230971745898)
        ref[3,2,1,1]=(0.492750417331)/(o+1.)+(-0.0743607257106)
        ref[3,2,2,0]=(0.062835017839)/(o+1.)+(-0.0396102029518)
        ref[3,2,2,1]=(-0.967272854749)/(o+1.)+(0.254924478345)
        ref[3,3,0,0]=(1.41291903854)/(o+1.)+(-0.208458374448)
        ref[3,3,0,1]=(-0.740922121718)/(o+1.)+(0.100332753677)
        ref[3,3,1,0]=(1.2839228132)/(o+1.)+(-0.405680585568)
        ref[3,3,1,1]=(-1.69516258968)/(o+1.)+(-0.69241891756)
        ref[3,3,2,0]=(0.216594952745)/(o+1.)+(0.309112737639)
        ref[3,3,2,1]=(-0.634806027047)/(o+1.)+(0.0728039418712)
        ref[3,4,0,0]=(-0.310491357088)/(o+1.)+(0.315525778146)
        ref[3,4,0,1]=(-0.751915044154)/(o+1.)+(-0.457043209029)
        ref[3,4,1,0]=(-0.339963591836)/(o+1.)+(0.309918338163)
        ref[3,4,1,1]=(0.0720250143477)/(o+1.)+(0.223190077872)
        ref[3,4,2,0]=(-0.0508554115866)/(o+1.)+(-0.0483042552841)
        ref[3,4,2,1]=(0.754037720807)/(o+1.)+(0.163604311813)
      else:
        arg[0,0,0,0]=(-1.24350959032)*x[0]+(-0.580181370761)*x[1]+(-0.174543112173)*x[2]
        arg[0,0,0,1]=(1.43450421585)*x[0]+(0.717062915298)*x[1]+(-0.91866562143)*x[2]
        arg[0,0,1,0]=(0.0188823859409)*x[0]+(1.24568141807)*x[1]+(0.0842604494481)*x[2]
        arg[0,0,1,1]=(0.871591276766)*x[0]+(0.653031319605)*x[1]+(0.037714556611)*x[2]
        arg[0,0,2,0]=(1.60473786555)*x[0]+(1.47509195705)*x[1]+(-0.103485610136)*x[2]
        arg[0,0,2,1]=(1.68030531336)*x[0]+(0.551201281257)*x[1]+(1.26951824468)*x[2]
        arg[0,1,0,0]=(0.491638014782)*x[0]+(1.26264219762)*x[1]+(-0.921297094204)*x[2]
        arg[0,1,0,1]=(0.20108233628)*x[0]+(-1.36096031252)*x[1]+(-0.878077046379)*x[2]
        arg[0,1,1,0]=(0.900775475455)*x[0]+(0.193423726606)*x[1]+(-0.139955126775)*x[2]
        arg[0,1,1,1]=(-0.0354326925578)*x[0]+(-0.464646718718)*x[1]+(1.49240476583)*x[2]
        arg[0,1,2,0]=(-1.40991971664)*x[0]+(-1.7747715584)*x[1]+(-0.96387650203)*x[2]
        arg[0,1,2,1]=(-0.476443682267)*x[0]+(0.55076135194)*x[1]+(0.190289559512)*x[2]
        arg[0,2,0,0]=(0.535490315669)*x[0]+(0.566915366497)*x[1]+(-0.41513643529)*x[2]
        arg[0,2,0,1]=(-0.18986562478)*x[0]+(-0.62347050233)*x[1]+(1.18004907973)*x[2]
        arg[0,2,1,0]=(-0.288440432828)*x[0]+(0.0375642125411)*x[1]+(0.574653643212)*x[2]
        arg[0,2,1,1]=(1.03327739612)*x[0]+(-0.0443874572943)*x[1]+(0.972142765323)*x[2]
        arg[0,2,2,0]=(-1.27131901306)*x[0]+(0.619179457229)*x[1]+(-0.436029340417)*x[2]
        arg[0,2,2,1]=(0.0573099137019)*x[0]+(0.211373076629)*x[1]+(0.859991510967)*x[2]
        arg[0,3,0,0]=(0.650974277613)*x[0]+(-0.926352792387)*x[1]+(-1.84479828618)*x[2]
        arg[0,3,0,1]=(-0.293748564815)*x[0]+(0.53683587198)*x[1]+(1.51147453744)*x[2]
        arg[0,3,1,0]=(0.837889627485)*x[0]+(0.62568225238)*x[1]+(0.365321068424)*x[2]
        arg[0,3,1,1]=(-0.671841278343)*x[0]+(-0.00822894090357)*x[1]+(-1.35180265459)*x[2]
        arg[0,3,2,0]=(0.823443588183)*x[0]+(-0.366903894987)*x[1]+(-0.878402795844)*x[2]
        arg[0,3,2,1]=(-0.856292830088)*x[0]+(-0.134584368493)*x[1]+(-0.299126849425)*x[2]
        arg[0,4,0,0]=(0.62792423481)*x[0]+(0.781414088068)*x[1]+(-1.12299220031)*x[2]
        arg[0,4,0,1]=(-0.00743646774917)*x[0]+(0.52784004028)*x[1]+(-1.44284109205)*x[2]
        arg[0,4,1,0]=(-0.855519018884)*x[0]+(-0.222306698876)*x[1]+(0.0502517275952)*x[2]
        arg[0,4,1,1]=(-0.311060622512)*x[0]+(0.321283094479)*x[1]+(-0.484147152851)*x[2]
        arg[0,4,2,0]=(-1.43220161062)*x[0]+(-0.27822988291)*x[1]+(-0.692367106541)*x[2]
        arg[0,4,2,1]=(1.3956686934)*x[0]+(-1.26426333906)*x[1]+(-1.09343602989)*x[2]
        arg[1,0,0,0]=(0.122525927098)*x[0]+(-0.484009414389)*x[1]+(-0.822036495419)*x[2]
        arg[1,0,0,1]=(-0.276748173334)*x[0]+(0.35307031133)*x[1]+(-1.23466402387)*x[2]
        arg[1,0,1,0]=(-0.479355493223)*x[0]+(-0.56368304348)*x[1]+(0.54679107219)*x[2]
        arg[1,0,1,1]=(-0.483618214285)*x[0]+(1.60438722757)*x[1]+(0.542389963596)*x[2]
        arg[1,0,2,0]=(0.130988624202)*x[0]+(-0.449468495509)*x[1]+(0.254640520023)*x[2]
        arg[1,0,2,1]=(1.13790232796)*x[0]+(-0.716218913004)*x[1]+(-0.193031264955)*x[2]
        arg[1,1,0,0]=(-0.0653152151059)*x[0]+(1.07890922508)*x[1]+(0.324995864884)*x[2]
        arg[1,1,0,1]=(0.769151345051)*x[0]+(-1.35911748672)*x[1]+(0.299326803785)*x[2]
        arg[1,1,1,0]=(-0.431128032697)*x[0]+(1.09261915758)*x[1]+(-0.029871998484)*x[2]
        arg[1,1,1,1]=(-0.922618915502)*x[0]+(1.30318141394)*x[1]+(-0.324039981399)*x[2]
        arg[1,1,2,0]=(-0.668178375077)*x[0]+(0.132203602988)*x[1]+(0.236035070297)*x[2]
        arg[1,1,2,1]=(-0.518594419598)*x[0]+(-0.0429695477263)*x[1]+(1.11877941395)*x[2]
        arg[1,2,0,0]=(0.610776644247)*x[0]+(-0.023900598958)*x[1]+(0.0480141870788)*x[2]
        arg[1,2,0,1]=(0.562736426016)*x[0]+(-0.294969624732)*x[1]+(-0.23898469143)*x[2]
        arg[1,2,1,0]=(0.645173726197)*x[0]+(-1.18365074993)*x[1]+(0.102581636468)*x[2]
        arg[1,2,1,1]=(-0.518976597355)*x[0]+(0.372817596528)*x[1]+(0.49035249397)*x[2]
        arg[1,2,2,0]=(0.0192623731103)*x[0]+(-0.241279406644)*x[1]+(-0.420015571981)*x[2]
        arg[1,2,2,1]=(0.318641827599)*x[0]+(-1.5357263392)*x[1]+(0.350150621892)*x[2]
        arg[1,3,0,0]=(-0.719401628191)*x[0]+(-1.77806447133)*x[1]+(1.19750443209)*x[2]
        arg[1,3,0,1]=(-1.00122204304)*x[0]+(0.675162216383)*x[1]+(1.1180105058)*x[2]
        arg[1,3,1,0]=(0.331149947519)*x[0]+(0.310832741708)*x[1]+(0.087866556795)*x[2]
        arg[1,3,1,1]=(-0.156343054316)*x[0]+(1.00796776488)*x[1]+(-1.53460740228)*x[2]
        arg[1,3,2,0]=(0.714240747972)*x[0]+(0.968170992322)*x[1]+(-0.962819191598)*x[2]
        arg[1,3,2,1]=(-0.200412114379)*x[0]+(-0.173781234544)*x[1]+(0.899584749309)*x[2]
        arg[1,4,0,0]=(-1.38011177642)*x[0]+(-0.316506470647)*x[1]+(0.266952180469)*x[2]
        arg[1,4,0,1]=(0.343359684584)*x[0]+(-1.13244466712)*x[1]+(-0.81840155125)*x[2]
        arg[1,4,1,0]=(0.295299129957)*x[0]+(0.199810907018)*x[1]+(0.631717407771)*x[2]
        arg[1,4,1,1]=(-0.46840784478)*x[0]+(-0.402683932206)*x[1]+(-0.782209293177)*x[2]
        arg[1,4,2,0]=(-1.8006389736)*x[0]+(0.119727360087)*x[1]+(0.482076626168)*x[2]
        arg[1,4,2,1]=(0.96553885281)*x[0]+(0.425097333238)*x[1]+(0.54533551085)*x[2]
        arg[2,0,0,0]=(0.932725602425)*x[0]+(-0.513780652569)*x[1]+(0.275742915648)*x[2]
        arg[2,0,0,1]=(1.10053578072)*x[0]+(0.787895323287)*x[1]+(-1.46544221328)*x[2]
        arg[2,0,1,0]=(0.83657269666)*x[0]+(-1.15717482853)*x[1]+(1.1090271297)*x[2]
        arg[2,0,1,1]=(1.59370135994)*x[0]+(0.452015708845)*x[1]+(0.0862230059082)*x[2]
        arg[2,0,2,0]=(-0.954832509866)*x[0]+(0.51192257473)*x[1]+(-1.48499771643)*x[2]
        arg[2,0,2,1]=(-0.556993361475)*x[0]+(0.438110051421)*x[1]+(0.237254018143)*x[2]
        arg[2,1,0,0]=(-0.318980966383)*x[0]+(-0.263071719097)*x[1]+(-0.348122442404)*x[2]
        arg[2,1,0,1]=(0.100191287326)*x[0]+(-0.422996999974)*x[1]+(-0.850702087112)*x[2]
        arg[2,1,1,0]=(-0.198942237141)*x[0]+(0.277459568173)*x[1]+(-1.66195774421)*x[2]
        arg[2,1,1,1]=(-1.62211381863)*x[0]+(-0.13417844308)*x[1]+(1.07441481356)*x[2]
        arg[2,1,2,0]=(-0.183120762554)*x[0]+(-0.158546687568)*x[1]+(1.05938565108)*x[2]
        arg[2,1,2,1]=(-0.530596736933)*x[0]+(-0.507872303289)*x[1]+(0.0437745116021)*x[2]
        arg[2,2,0,0]=(-0.430338834548)*x[0]+(-1.01989755268)*x[1]+(0.0364152729771)*x[2]
        arg[2,2,0,1]=(-0.55129175645)*x[0]+(0.283907226465)*x[1]+(0.281403899376)*x[2]
        arg[2,2,1,0]=(0.119036949181)*x[0]+(0.187265158741)*x[1]+(-0.0845814658663)*x[2]
        arg[2,2,1,1]=(1.39900884968)*x[0]+(1.42806588781)*x[1]+(0.211444691268)*x[2]
        arg[2,2,2,0]=(-0.609664666581)*x[0]+(-0.29988332429)*x[1]+(-0.28125800646)*x[2]
        arg[2,2,2,1]=(-1.77545247157)*x[0]+(0.337828315779)*x[1]+(-0.239141292616)*x[2]
        arg[2,3,0,0]=(-1.23141604656)*x[0]+(0.322998749903)*x[1]+(-1.70685281816)*x[2]
        arg[2,3,0,1]=(-1.4416737434)*x[0]+(1.49740114642)*x[1]+(0.732057753882)*x[2]
        arg[2,3,1,0]=(-1.59112622077)*x[0]+(0.665317566947)*x[1]+(0.826417253499)*x[2]
        arg[2,3,1,1]=(-1.22000830454)*x[0]+(0.618713896065)*x[1]+(1.27354984534)*x[2]
        arg[2,3,2,0]=(0.125639723221)*x[0]+(0.663241510656)*x[1]+(-0.426459288281)*x[2]
        arg[2,3,2,1]=(1.39120212155)*x[0]+(0.9115748844)*x[1]+(1.77889030223)*x[2]
        arg[2,4,0,0]=(-0.217927656387)*x[0]+(1.20391185439)*x[1]+(-0.188197734908)*x[2]
        arg[2,4,0,1]=(-0.442009082947)*x[0]+(0.399477101959)*x[1]+(-1.33974368956)*x[2]
        arg[2,4,1,0]=(0.636534809382)*x[0]+(0.872923470503)*x[1]+(-0.922286702006)*x[2]
        arg[2,4,1,1]=(-0.831840624431)*x[0]+(0.946223248622)*x[1]+(0.380300039133)*x[2]
        arg[2,4,2,0]=(0.0803088903901)*x[0]+(-0.857685204574)*x[1]+(-0.479475079431)*x[2]
        arg[2,4,2,1]=(-1.7233942516)*x[0]+(-1.38771345941)*x[1]+(0.761958201633)*x[2]
        arg[3,0,0,0]=(-0.738032075796)*x[0]+(1.33264583681)*x[1]+(-1.21101988385)*x[2]
        arg[3,0,0,1]=(1.40336327022)*x[0]+(-0.77078885213)*x[1]+(1.70960627204)*x[2]
        arg[3,0,1,0]=(0.452477629041)*x[0]+(0.235432009434)*x[1]+(1.69314009822)*x[2]
        arg[3,0,1,1]=(-0.816213143646)*x[0]+(-1.42050350309)*x[1]+(-0.0806105456758)*x[2]
        arg[3,0,2,0]=(0.0400708086023)*x[0]+(-0.336948609628)*x[1]+(-0.113687293995)*x[2]
        arg[3,0,2,1]=(0.414077492095)*x[0]+(-1.06937716207)*x[1]+(1.25972041173)*x[2]
        arg[3,1,0,0]=(-1.02975555201)*x[0]+(-0.123306171085)*x[1]+(-1.74750909033)*x[2]
        arg[3,1,0,1]=(-1.01988698009)*x[0]+(-0.142188487205)*x[1]+(0.274514515492)*x[2]
        arg[3,1,1,0]=(0.440626586141)*x[0]+(-0.0569580581917)*x[1]+(-0.77065631367)*x[2]
        arg[3,1,1,1]=(0.987903067972)*x[0]+(-1.47543187274)*x[1]+(-0.0888962732893)*x[2]
        arg[3,1,2,0]=(0.150506204416)*x[0]+(0.035476396146)*x[1]+(-0.852061690418)*x[2]
        arg[3,1,2,1]=(-1.15179800132)*x[0]+(-0.425570974979)*x[1]+(1.10092214699)*x[2]
        arg[3,2,0,0]=(1.01255188057)*x[0]+(1.24279465527)*x[1]+(1.25190246102)*x[2]
        arg[3,2,0,1]=(-1.16275538385)*x[0]+(1.13672085108)*x[1]+(0.131953731231)*x[2]
        arg[3,2,1,0]=(0.774290747173)*x[0]+(0.275410883089)*x[1]+(-0.825692266439)*x[2]
        arg[3,2,1,1]=(-1.07241838074)*x[0]+(-0.0974675737726)*x[1]+(1.1856849697)*x[2]
        arg[3,2,2,0]=(0.234209248938)*x[0]+(0.756893234577)*x[1]+(-0.365141055079)*x[2]
        arg[3,2,2,1]=(-1.6668858309)*x[0]+(-0.916769120288)*x[1]+(-0.716365681217)*x[2]
        arg[3,3,0,0]=(-0.36890215249)*x[0]+(-0.581062815398)*x[1]+(1.6280164507)*x[2]
        arg[3,3,0,1]=(-0.933634743559)*x[0]+(-0.0657657779145)*x[1]+(1.35011362804)*x[2]
        arg[3,3,1,0]=(1.13739289322)*x[0]+(0.0103208127738)*x[1]+(1.01628193014)*x[2]
        arg[3,3,1,1]=(0.037979696459)*x[0]+(-1.76926650495)*x[1]+(-1.28602572757)*x[2]
        arg[3,3,2,0]=(0.797830431467)*x[0]+(0.732108360923)*x[1]+(-0.436390867333)*x[2]
        arg[3,3,2,1]=(-0.571865447888)*x[0]+(0.895109047357)*x[1]+(-0.673234648776)*x[2]
        arg[3,4,0,0]=(-1.59667195158)*x[0]+(-1.06146105331)*x[1]+(-1.81255923783)*x[2]
        arg[3,4,0,1]=(-0.0721327214647)*x[0]+(1.23982376787)*x[1]+(1.43101011)*x[2]
        arg[3,4,1,0]=(0.117288395573)*x[0]+(-0.37678966212)*x[1]+(-1.14107285089)*x[2]
        arg[3,4,1,1]=(-0.303270532321)*x[0]+(1.33784402324)*x[1]+(-0.651624796413)*x[2]
        arg[3,4,2,0]=(0.215849013256)*x[0]+(0.247315697644)*x[1]+(0.404644392153)*x[2]
        arg[3,4,2,1]=(0.975781190478)*x[0]+(-0.252285063767)*x[1]+(-0.145694910854)*x[2]
        ref[0,0,0,0]=(-0.718934738033)/(o+1.)+(-0.639649667608)
        ref[0,0,0,1]=(1.01007662915)/(o+1.)+(0.111412440284)
        ref[0,0,1,0]=(0.827868953278)/(o+1.)+(0.26047765009)
        ref[0,0,1,1]=(1.48122352358)/(o+1.)+(0.0405568147034)
        ref[0,0,2,0]=(1.33282099436)/(o+1.)+(0.821761609054)
        ref[0,0,2,1]=(1.65298089862)/(o+1.)+(0.924021970342)
        ref[0,1,0,0]=(0.930162571453)/(o+1.)+(-0.0485897266263)
        ref[0,1,0,1]=(-0.347599158269)/(o+1.)+(-0.845177932176)
        ref[0,1,1,0]=(0.623152121663)/(o+1.)+(0.165545976811)
        ref[0,1,1,1]=(0.0506217690841)/(o+1.)+(0.470851792737)
        ref[0,1,2,0]=(-1.99181491207)/(o+1.)+(-1.0783764325)
        ref[0,1,2,1]=(-0.731065007801)/(o+1.)+(0.497836118493)
        ref[0,2,0,0]=(-0.0812249009151)/(o+1.)+(0.384247073896)
        ref[0,2,0,1]=(-0.115768505253)/(o+1.)+(0.241240728938)
        ref[0,2,1,0]=(0.152728009607)/(o+1.)+(0.0855247066589)
        ref[0,2,1,1]=(1.43086526503)/(o+1.)+(0.265083719563)
        ref[0,2,2,0]=(-0.967906369418)/(o+1.)+(-0.060131263415)
        ref[0,2,2,1]=(-0.269269238419)/(o+1.)+(0.698971869859)
        ref[0,3,0,0]=(-1.71659801402)/(o+1.)+(-0.201789393467)
        ref[0,3,0,1]=(0.265343635273)/(o+1.)+(0.744609104665)
        ref[0,3,1,0]=(1.83021875237)/(o+1.)+(-0.000662902039879)
        ref[0,3,1,1]=(-0.317529833375)/(o+1.)+(-0.857171520229)
        ref[0,3,2,0]=(-0.705757693524)/(o+1.)+(0.141947295438)
        ref[0,3,2,1]=(-1.27000495392)/(o+1.)+(-0.00999954704202)
        ref[0,4,0,0]=(0.061892522425)/(o+1.)+(0.112226800069)
        ref[0,4,0,1]=(-0.747062267148)/(o+1.)+(-0.0876876261864)
        ref[0,4,1,0]=(-1.19488448842)/(o+1.)+(0.0836552491261)
        ref[0,4,1,1]=(1.36573976292)/(o+1.)+(-0.9198322219)
        ref[0,4,2,0]=(-1.67982141697)/(o+1.)+(-0.361488591554)
        ref[0,4,2,1]=(-0.598050084547)/(o+1.)+(-0.181990295501)
        ref[1,0,0,0]=(-2.45927674426)/(o+1.)+(0.637878380777)
        ref[1,0,0,1]=(-0.575223026885)/(o+1.)+(-0.291559429493)
        ref[1,0,1,0]=(0.0871765959469)/(o+1.)+(-0.29171203023)
        ref[1,0,1,1]=(0.94050575309)/(o+1.)+(0.361326611893)
        ref[1,0,2,0]=(-0.306500880914)/(o+1.)+(0.121330764815)
        ref[1,0,2,1]=(1.60748296414)/(o+1.)+(-0.689415407072)
        ref[1,1,0,0]=(0.640401899755)/(o+1.)+(0.34909398755)
        ref[1,1,0,1]=(0.134022909144)/(o+1.)+(-0.212331123516)
        ref[1,1,1,0]=(0.248008519593)/(o+1.)+(0.191805303402)
        ref[1,1,1,1]=(0.638911732328)/(o+1.)+(-0.291194607647)
        ref[1,1,2,0]=(0.109264462855)/(o+1.)+(-0.204602082323)
        ref[1,1,2,1]=(-0.185963112656)/(o+1.)+(0.371589279643)
        ref[1,2,0,0]=(-0.590669733193)/(o+1.)+(0.61277998278)
        ref[1,2,0,1]=(-1.15856908266)/(o+1.)+(0.593675596257)
        ref[1,2,1,0]=(-0.451642187813)/(o+1.)+(0.00787340027166)
        ref[1,2,1,1]=(-0.428757501259)/(o+1.)+(0.386475497201)
        ref[1,2,2,0]=(-0.414532914271)/(o+1.)+(-0.113749845622)
        ref[1,2,2,1]=(-0.513613710163)/(o+1.)+(-0.176660089771)
        ref[1,3,0,0]=(-0.890487934256)/(o+1.)+(-0.20473686659)
        ref[1,3,0,1]=(1.64512779208)/(o+1.)+(-0.426588556466)
        ref[1,3,1,0]=(0.831527595966)/(o+1.)+(-0.0508391749725)
        ref[1,3,1,1]=(-0.718921003194)/(o+1.)+(0.0179691557385)
        ref[1,3,2,0]=(0.405959213301)/(o+1.)+(0.156816667697)
        ref[1,3,2,1]=(0.426319486322)/(o+1.)+(0.0495359570318)
        ref[1,4,0,0]=(-1.11364330874)/(o+1.)+(-0.158011378929)
        ref[1,4,0,1]=(-0.0993967101457)/(o+1.)+(-0.754044911817)
        ref[1,4,1,0]=(0.979493241598)/(o+1.)+(0.0736671015741)
        ref[1,4,1,1]=(-1.78692478393)/(o+1.)+(0.0668118568857)
        ref[1,4,2,0]=(-0.691473647622)/(o+1.)+(-0.253680669863)
        ref[1,4,2,1]=(1.16914997228)/(o+1.)+(0.383410862311)
        ref[2,0,0,0]=(0.790187638399)/(o+1.)+(-0.047749886448)
        ref[2,0,0,1]=(0.429625240932)/(o+1.)+(-0.00331817510394)
        ref[2,0,1,0]=(-0.101344997877)/(o+1.)+(0.444884997853)
        ref[2,0,1,1]=(2.15519953039)/(o+1.)+(-0.0116297278494)
        ref[2,0,2,0]=(-1.40159594866)/(o+1.)+(-0.263155851453)
        ref[2,0,2,1]=(-0.658391287862)/(o+1.)+(0.388380997976)
        ref[2,1,0,0]=(-0.773933630328)/(o+1.)+(-0.0781207487777)
        ref[2,1,0,1]=(0.0015551432788)/(o+1.)+(-0.587531471519)
        ref[2,1,1,0]=(0.122693324758)/(o+1.)+(-0.853066868967)
        ref[2,1,1,1]=(-0.219255630453)/(o+1.)+(-0.231310908847)
        ref[2,1,2,0]=(1.33176471286)/(o+1.)+(-0.307023255951)
        ref[2,1,2,1]=(-1.67471265259)/(o+1.)+(0.340009061986)
        ref[2,2,0,0]=(-0.727222063897)/(o+1.)+(-0.343299525178)
        ref[2,2,0,1]=(-1.87667590405)/(o+1.)+(0.945347636719)
        ref[2,2,1,0]=(0.776003927859)/(o+1.)+(-0.277141642902)
        ref[2,2,1,1]=(1.62850322537)/(o+1.)+(0.705008101698)
        ref[2,2,2,0]=(1.31869548331)/(o+1.)+(-1.25475074032)
        ref[2,2,2,1]=(-1.71691241115)/(o+1.)+(0.0200734813728)
        ref[2,3,0,0]=(-0.37741220744)/(o+1.)+(-1.11892895368)
        ref[2,3,0,1]=(0.684763737686)/(o+1.)+(0.0515107096073)
        ref[2,3,1,0]=(-0.0572023243047)/(o+1.)+(-0.0210945380092)
        ref[2,3,1,1]=(0.031457290932)/(o+1.)+(0.320399072966)
        ref[2,3,2,0]=(-0.71613334286)/(o+1.)+(0.539277644228)
        ref[2,3,2,1]=(2.10068005021)/(o+1.)+(0.990493628981)
        ref[2,4,0,0]=(0.830379294407)/(o+1.)+(-0.016296415655)
        ref[2,4,0,1]=(-1.05197706105)/(o+1.)+(-0.165149304749)
        ref[2,4,1,0]=(-0.0854959336547)/(o+1.)+(0.336333755767)
        ref[2,4,1,1]=(1.15845594915)/(o+1.)+(-0.331886642912)
        ref[2,4,2,0]=(-1.25248721318)/(o+1.)+(-0.00218209021539)
        ref[2,4,2,1]=(-0.991752052473)/(o+1.)+(-0.678698728452)
        ref[3,0,0,0]=(-0.262071815816)/(o+1.)+(-0.177167153508)
        ref[3,0,0,1]=(1.20598261029)/(o+1.)+(0.568099039922)
        ref[3,0,1,0]=(1.21237099375)/(o+1.)+(0.584339371474)
        ref[3,0,1,1]=(-1.04628498282)/(o+1.)+(-0.635521104799)
        ref[3,0,2,0]=(0.630843868383)/(o+1.)+(-0.520704481702)
        ref[3,0,2,1]=(1.56136208039)/(o+1.)+(-0.478470669318)
        ref[3,1,0,0]=(-0.372752943047)/(o+1.)+(-1.26390893519)
        ref[3,1,0,1]=(0.192139112377)/(o+1.)+(-0.53985003209)
        ref[3,1,1,0]=(-1.18480584023)/(o+1.)+(0.398909027255)
        ref[3,1,1,1]=(0.402069277892)/(o+1.)+(-0.489247177975)
        ref[3,1,2,0]=(0.27608763117)/(o+1.)+(-0.471083360513)
        ref[3,1,2,1]=(-0.273636427696)/(o+1.)+(-0.101405200809)
        ref[3,2,0,0]=(1.92572771773)/(o+1.)+(0.790760639563)
        ref[3,2,0,1]=(0.16264122626)/(o+1.)+(-0.0283610138982)
        ref[3,2,1,0]=(0.377561018039)/(o+1.)+(-0.076775827108)
        ref[3,2,1,1]=(-0.652466873272)/(o+1.)+(0.334132944227)
        ref[3,2,2,0]=(-0.0304721826842)/(o+1.)+(0.32821680556)
        ref[3,2,2,1]=(-1.52932569088)/(o+1.)+(-0.885347470761)
        ref[3,3,0,0]=(-0.451938162279)/(o+1.)+(0.564994822546)
        ref[3,3,0,1]=(-0.477583137662)/(o+1.)+(0.414148122113)
        ref[3,3,1,0]=(1.38037325689)/(o+1.)+(0.391811189625)
        ref[3,3,1,1]=(-1.01333159097)/(o+1.)+(-1.00199047255)
        ref[3,3,2,0]=(1.3222644334)/(o+1.)+(-0.114358254173)
        ref[3,3,2,1]=(-0.248989333346)/(o+1.)+(-0.0505008579799)
        ref[3,4,0,0]=(-1.92196761736)/(o+1.)+(-1.27436231268)
        ref[3,4,0,1]=(0.894404962457)/(o+1.)+(0.852148096971)
        ref[3,4,1,0]=(0.868763436571)/(o+1.)+(-1.134668777)
        ref[3,4,1,1]=(0.438747123022)/(o+1.)+(-0.0278992142568)
        ref[3,4,2,0]=(0.532575431402)/(o+1.)+(0.167616835826)
        ref[3,4,2,1]=(0.796744496799)/(o+1.)+(-0.109471640471)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5, 3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunction_fromData_ReducedContinuousFunction_rank0(self):
      """
      tests integral of rank 0 Data on the ReducedFunction

      assumptions: ReducedContinuousFunction(self.domain) exists
                   self.domain supports integral on ReducedFunction
      """
      o=1
      dim=self.domain.getDim()
      w_ref=ReducedFunction(self.domain)
      w=ReducedContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(),w)
      if dim==2:
        arg=(0.729700679932)*x[0]+(-1.38329769179)*x[1]
        ref=(0.329728333821)/(o+1.)+(-0.491662672839)
      else:
        arg=(-1.75076012766)*x[0]+(-0.0862990889349)*x[1]+(0.637429232833)*x[2]
        ref=(-0.577517867896)/(o+1.)+(-0.311056057931)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunction_fromData_ReducedContinuousFunction_rank1(self):
      """
      tests integral of rank 1 Data on the ReducedFunction

      assumptions: ReducedContinuousFunction(self.domain) exists
                   self.domain supports integral on ReducedFunction
      """
      o=1
      dim=self.domain.getDim()
      w_ref=ReducedFunction(self.domain)
      w=ReducedContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(2,),w)
      ref=numpy.zeros((2,),numpy.float_)
      if dim==2:
        arg[0]=(1.0296233191)*x[0]+(-0.347985432388)*x[1]
        arg[1]=(-0.182082548052)*x[0]+(0.114822539064)*x[1]
        ref[0]=(0.876900301573)/(o+1.)+(-0.0976312074308)
        ref[1]=(-1.09276229827)/(o+1.)+(0.512751144643)
      else:
        arg[0]=(0.390938934986)*x[0]+(-0.17813109236)*x[1]+(0.92611177061)*x[2]
        arg[1]=(0.169562781554)*x[0]+(-0.149625135256)*x[1]+(0.357445984273)*x[2]
        ref[0]=(1.07308269316)/(o+1.)+(0.0329184600363)
        ref[1]=(-1.07527269336)/(o+1.)+(0.726328161966)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunction_fromData_ReducedContinuousFunction_rank2(self):
      """
      tests integral of rank 2 Data on the ReducedFunction

      assumptions: ReducedContinuousFunction(self.domain) exists
                   self.domain supports integral on ReducedFunction
      """
      o=1
      dim=self.domain.getDim()
      w_ref=ReducedFunction(self.domain)
      w=ReducedContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5),w)
      ref=numpy.zeros((4, 5),numpy.float_)
      if dim==2:
        arg[0,0]=(-0.798668686601)*x[0]+(-0.502017718729)*x[1]
        arg[0,1]=(-0.682368031728)*x[0]+(-0.915048032379)*x[1]
        arg[0,2]=(-0.0237771508029)*x[0]+(0.177441262563)*x[1]
        arg[0,3]=(0.135600942216)*x[0]+(0.496697614771)*x[1]
        arg[0,4]=(0.495272681631)*x[0]+(-0.457599050745)*x[1]
        arg[1,0]=(-0.252415627225)*x[0]+(-0.820872546406)*x[1]
        arg[1,1]=(-0.183362696688)*x[0]+(-0.721277471492)*x[1]
        arg[1,2]=(0.401101418876)*x[0]+(0.0483185737314)*x[1]
        arg[1,3]=(0.251246583055)*x[0]+(-0.837901564153)*x[1]
        arg[1,4]=(0.2984760032)*x[0]+(0.0504569221307)*x[1]
        arg[2,0]=(1.18468310151)*x[0]+(-0.545612792222)*x[1]
        arg[2,1]=(-1.08451540544)*x[0]+(0.820927137753)*x[1]
        arg[2,2]=(0.85435600905)*x[0]+(-0.309819803422)*x[1]
        arg[2,3]=(-0.184148030844)*x[0]+(0.978315441698)*x[1]
        arg[2,4]=(1.36943188922)*x[0]+(-1.3176984733)*x[1]
        arg[3,0]=(0.42189213134)*x[0]+(1.23097633986)*x[1]
        arg[3,1]=(0.0409353425792)*x[0]+(-1.15555830367)*x[1]
        arg[3,2]=(0.557942070395)*x[0]+(-0.591659744114)*x[1]
        arg[3,3]=(-1.07432449167)*x[0]+(-0.665475086161)*x[1]
        arg[3,4]=(-1.00421972544)*x[0]+(1.16737316771)*x[1]
        ref[0,0]=(-0.400434079596)/(o+1.)+(-0.450126162867)
        ref[0,1]=(-1.35836659237)/(o+1.)+(-0.11952473587)
        ref[0,2]=(0.154775934077)/(o+1.)+(-0.000555911158727)
        ref[0,3]=(-0.177525502969)/(o+1.)+(0.404912029978)
        ref[0,4]=(1.00122501632)/(o+1.)+(-0.481775692719)
        ref[1,0]=(-1.37507083806)/(o+1.)+(0.150891332214)
        ref[1,1]=(-1.37429063854)/(o+1.)+(0.234825235182)
        ref[1,2]=(1.30069236547)/(o+1.)+(-0.425636186429)
        ref[1,3]=(-0.414658324515)/(o+1.)+(-0.0859983282917)
        ref[1,4]=(0.757097138678)/(o+1.)+(-0.204082106674)
        ref[2,0]=(1.09244584366)/(o+1.)+(-0.226687767185)
        ref[2,1]=(0.109481169277)/(o+1.)+(-0.186534718481)
        ref[2,2]=(0.587125473152)/(o+1.)+(-0.0212946337622)
        ref[2,3]=(0.00878695699842)/(o+1.)+(0.392690226928)
        ref[2,4]=(0.306810290785)/(o+1.)+(-0.127538437433)
        ref[3,0]=(0.317826050636)/(o+1.)+(0.667521210281)
        ref[3,1]=(-0.276254182573)/(o+1.)+(-0.419184389261)
        ref[3,2]=(-0.753831729065)/(o+1.)+(0.360057027673)
        ref[3,3]=(-1.40505001195)/(o+1.)+(-0.167374782943)
        ref[3,4]=(0.53112066724)/(o+1.)+(-0.183983612488)
      else:
        arg[0,0]=(1.54447818842)*x[0]+(-0.560568590939)*x[1]+(-1.33749549019)*x[2]
        arg[0,1]=(-0.535544800192)*x[0]+(-0.732513980801)*x[1]+(0.309855030569)*x[2]
        arg[0,2]=(-0.45251604551)*x[0]+(-0.0933932995351)*x[1]+(-0.465724167794)*x[2]
        arg[0,3]=(0.10344455234)*x[0]+(-0.0451254971766)*x[1]+(-1.39912042607)*x[2]
        arg[0,4]=(0.617585449939)*x[0]+(1.27357810776)*x[1]+(-0.840161322302)*x[2]
        arg[1,0]=(0.650274355535)*x[0]+(0.547805408891)*x[1]+(0.314350326062)*x[2]
        arg[1,1]=(0.697200474806)*x[0]+(0.558681734309)*x[1]+(-0.0277936766536)*x[2]
        arg[1,2]=(-0.508578328229)*x[0]+(-0.513894226666)*x[1]+(-0.528417418023)*x[2]
        arg[1,3]=(-1.50391414876)*x[0]+(0.174080260823)*x[1]+(-1.48528641514)*x[2]
        arg[1,4]=(0.859470776434)*x[0]+(-0.121348493404)*x[1]+(1.69796346547)*x[2]
        arg[2,0]=(1.00593912049)*x[0]+(0.794254020467)*x[1]+(-1.02770358461)*x[2]
        arg[2,1]=(-1.64586076188)*x[0]+(-0.83927167526)*x[1]+(-1.13088449805)*x[2]
        arg[2,2]=(0.0628777004357)*x[0]+(-1.27316091149)*x[1]+(0.417655047748)*x[2]
        arg[2,3]=(-1.43661526118)*x[0]+(0.0676042964527)*x[1]+(0.174947572442)*x[2]
        arg[2,4]=(0.504347051098)*x[0]+(-1.62976096707)*x[1]+(0.61579932957)*x[2]
        arg[3,0]=(1.05212645875)*x[0]+(-1.10517003049)*x[1]+(0.902242097595)*x[2]
        arg[3,1]=(-0.997611854104)*x[0]+(-1.53611378293)*x[1]+(0.322835276897)*x[2]
        arg[3,2]=(-0.815135194015)*x[0]+(0.357734619962)*x[1]+(0.684966069635)*x[2]
        arg[3,3]=(0.4545417685)*x[0]+(0.386262464083)*x[1]+(0.131414279708)*x[2]
        arg[3,4]=(-0.222694646001)*x[0]+(-0.470499833621)*x[1]+(-0.593290862391)*x[2]
        ref[0,0]=(-0.593803934901)/(o+1.)+(0.120109021094)
        ref[0,1]=(0.0569115377867)/(o+1.)+(-0.507557644105)
        ref[0,2]=(-0.202741590274)/(o+1.)+(-0.404445961283)
        ref[0,3]=(-0.516888777015)/(o+1.)+(-0.411956296948)
        ref[0,4]=(0.52292787417)/(o+1.)+(0.264037180616)
        ref[1,0]=(-0.102685279742)/(o+1.)+(0.807557685115)
        ref[1,1]=(1.44527264276)/(o+1.)+(-0.108592055148)
        ref[1,2]=(-0.448767718864)/(o+1.)+(-0.551061127027)
        ref[1,3]=(-1.19936004685)/(o+1.)+(-0.807880128113)
        ref[1,4]=(1.09918755605)/(o+1.)+(0.668449096223)
        ref[2,0]=(-0.873708007453)/(o+1.)+(0.823098781902)
        ref[2,1]=(-1.9697776018)/(o+1.)+(-0.823119666698)
        ref[2,2]=(0.598023292545)/(o+1.)+(-0.695325727928)
        ref[2,3]=(-0.486077421455)/(o+1.)+(-0.353992985415)
        ref[2,4]=(0.65037669171)/(o+1.)+(-0.579995639058)
        ref[3,0]=(0.115401170876)/(o+1.)+(0.366898677493)
        ref[3,1]=(-0.836209920174)/(o+1.)+(-0.68734021998)
        ref[3,2]=(0.753819223079)/(o+1.)+(-0.263126863749)
        ref[3,3]=(-0.75294945942)/(o+1.)+(0.862583985856)
        ref[3,4]=(-1.44356156537)/(o+1.)+(0.0785381116785)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunction_fromData_ReducedContinuousFunction_rank3(self):
      """
      tests integral of rank 3 Data on the ReducedFunction

      assumptions: ReducedContinuousFunction(self.domain) exists
                   self.domain supports integral on ReducedFunction
      """
      o=1
      dim=self.domain.getDim()
      w_ref=ReducedFunction(self.domain)
      w=ReducedContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(6, 2, 2),w)
      ref=numpy.zeros((6, 2, 2),numpy.float_)
      if dim==2:
        arg[0,0,0]=(-0.409425595197)*x[0]+(-0.992427840513)*x[1]
        arg[0,0,1]=(0.645693185569)*x[0]+(-1.38338827813)*x[1]
        arg[0,1,0]=(0.838847764041)*x[0]+(-1.01358491173)*x[1]
        arg[0,1,1]=(0.381938057912)*x[0]+(0.61979806953)*x[1]
        arg[1,0,0]=(-1.36783302742)*x[0]+(-0.132422986617)*x[1]
        arg[1,0,1]=(-0.137283745381)*x[0]+(0.633534018338)*x[1]
        arg[1,1,0]=(0.0698189930006)*x[0]+(-1.45980415637)*x[1]
        arg[1,1,1]=(-0.391725001578)*x[0]+(0.355258883756)*x[1]
        arg[2,0,0]=(-0.596091494216)*x[0]+(-0.911870026464)*x[1]
        arg[2,0,1]=(0.138007894626)*x[0]+(1.76859278751)*x[1]
        arg[2,1,0]=(-0.791803431843)*x[0]+(-0.813170959331)*x[1]
        arg[2,1,1]=(-1.05466714677)*x[0]+(-1.12822729059)*x[1]
        arg[3,0,0]=(-0.417845531677)*x[0]+(1.13689234148)*x[1]
        arg[3,0,1]=(-0.0323347813877)*x[0]+(1.14381520037)*x[1]
        arg[3,1,0]=(0.639524393892)*x[0]+(0.864696525716)*x[1]
        arg[3,1,1]=(0.593104139176)*x[0]+(0.812642260561)*x[1]
        arg[4,0,0]=(-0.0580034066457)*x[0]+(0.802938738147)*x[1]
        arg[4,0,1]=(0.722143376522)*x[0]+(0.843814429252)*x[1]
        arg[4,1,0]=(-0.574057925551)*x[0]+(-0.298232158953)*x[1]
        arg[4,1,1]=(0.422529038525)*x[0]+(0.841942812244)*x[1]
        arg[5,0,0]=(1.90272013575)*x[0]+(-1.84811498267)*x[1]
        arg[5,0,1]=(-1.52156314991)*x[0]+(-0.797833417606)*x[1]
        arg[5,1,0]=(0.723748480291)*x[0]+(0.8667659268)*x[1]
        arg[5,1,1]=(-0.70289499551)*x[0]+(-0.726595802952)*x[1]
        ref[0,0,0]=(-1.1719312491)/(o+1.)+(-0.114961093303)
        ref[0,0,1]=(0.229993657297)/(o+1.)+(-0.483844374927)
        ref[0,1,0]=(-0.630442924796)/(o+1.)+(0.227852888553)
        ref[0,1,1]=(-0.806392286115)/(o+1.)+(0.904064206778)
        ref[1,0,0]=(-1.38312106397)/(o+1.)+(-0.0585674750338)
        ref[1,0,1]=(0.595552955095)/(o+1.)+(-0.049651341069)
        ref[1,1,0]=(-1.63606241234)/(o+1.)+(0.123038624483)
        ref[1,1,1]=(-0.340463713286)/(o+1.)+(0.151998797732)
        ref[2,0,0]=(-1.31925158304)/(o+1.)+(-0.094354968822)
        ref[2,0,1]=(0.699277735512)/(o+1.)+(0.60366147331)
        ref[2,1,0]=(-0.219878898763)/(o+1.)+(-0.692547746205)
        ref[2,1,1]=(-1.46750249607)/(o+1.)+(-0.357695970647)
        ref[3,0,0]=(0.71941698906)/(o+1.)+(-0.000185089626866)
        ref[3,0,1]=(-0.526993313502)/(o+1.)+(0.819236866243)
        ref[3,1,0]=(1.51671320209)/(o+1.)+(-0.00624614123899)
        ref[3,1,1]=(1.05185683613)/(o+1.)+(0.176944781803)
        ref[4,0,0]=(1.20432711487)/(o+1.)+(-0.229695891683)
        ref[4,0,1]=(1.26863112516)/(o+1.)+(0.14866334031)
        ref[4,1,0]=(0.34377834331)/(o+1.)+(-0.608034213907)
        ref[4,1,1]=(-0.189568721058)/(o+1.)+(0.727020285914)
        ref[5,0,0]=(0.116981035747)/(o+1.)+(-0.0311879413343)
        ref[5,0,1]=(-0.638728138162)/(o+1.)+(-0.840334214679)
        ref[5,1,0]=(0.891634589919)/(o+1.)+(0.349439908586)
        ref[5,1,1]=(-0.552480077628)/(o+1.)+(-0.438505360418)
      else:
        arg[0,0,0]=(-0.662295604848)*x[0]+(1.04606412328)*x[1]+(-0.376008345205)*x[2]
        arg[0,0,1]=(1.25030158303)*x[0]+(-0.0735167236581)*x[1]+(1.31623868156)*x[2]
        arg[0,1,0]=(0.717953620717)*x[0]+(-0.700464709991)*x[1]+(0.0345381771501)*x[2]
        arg[0,1,1]=(0.225944321569)*x[0]+(-1.09510173632)*x[1]+(-0.371235420166)*x[2]
        arg[1,0,0]=(-0.0755865552996)*x[0]+(-0.168214695387)*x[1]+(-0.437427629408)*x[2]
        arg[1,0,1]=(-1.52674685128)*x[0]+(-0.408158359188)*x[1]+(0.750091649824)*x[2]
        arg[1,1,0]=(-0.572400948498)*x[0]+(-0.802826119419)*x[1]+(-1.0233358643)*x[2]
        arg[1,1,1]=(0.623156259061)*x[0]+(0.378774331134)*x[1]+(0.0419766309042)*x[2]
        arg[2,0,0]=(0.443271002813)*x[0]+(-0.930092172267)*x[1]+(-0.20199150178)*x[2]
        arg[2,0,1]=(0.408614848881)*x[0]+(-1.3007040171)*x[1]+(-0.872615408174)*x[2]
        arg[2,1,0]=(-0.174910742406)*x[0]+(1.64872582781)*x[1]+(0.701172036891)*x[2]
        arg[2,1,1]=(0.417145974895)*x[0]+(0.428983721984)*x[1]+(-0.000787616769604)*x[2]
        arg[3,0,0]=(0.655644872855)*x[0]+(-0.268612479186)*x[1]+(0.11779790898)*x[2]
        arg[3,0,1]=(-0.244670332245)*x[0]+(-0.857502890419)*x[1]+(0.111792341131)*x[2]
        arg[3,1,0]=(0.131303542183)*x[0]+(0.231230724482)*x[1]+(0.116997126513)*x[2]
        arg[3,1,1]=(0.725434968911)*x[0]+(1.00206695553)*x[1]+(-0.319715936784)*x[2]
        arg[4,0,0]=(1.30949067662)*x[0]+(-1.14809917605)*x[1]+(0.00529778192088)*x[2]
        arg[4,0,1]=(0.969626585498)*x[0]+(-0.517908035476)*x[1]+(1.0467168328)*x[2]
        arg[4,1,0]=(0.362824899886)*x[0]+(0.890586602899)*x[1]+(-0.135163953653)*x[2]
        arg[4,1,1]=(-0.10353577028)*x[0]+(-0.0997924233294)*x[1]+(0.308973949876)*x[2]
        arg[5,0,0]=(-1.37242244095)*x[0]+(-0.746928231432)*x[1]+(-1.37385481504)*x[2]
        arg[5,0,1]=(0.231730881472)*x[0]+(-0.250265629505)*x[1]+(-0.903599685792)*x[2]
        arg[5,1,0]=(-0.813783753893)*x[0]+(0.41081474894)*x[1]+(-0.541868665288)*x[2]
        arg[5,1,1]=(0.19296641571)*x[0]+(1.8056250646)*x[1]+(0.283675443031)*x[2]
        ref[0,0,0]=(0.859538687531)/(o+1.)+(-0.425889257151)
        ref[0,0,1]=(-0.0674840316504)/(o+1.)+(1.28025378629)
        ref[0,1,0]=(-0.76922088579)/(o+1.)+(0.410623986833)
        ref[0,1,1]=(-0.582981712778)/(o+1.)+(-0.328705561068)
        ref[1,0,0]=(1.25605113432)/(o+1.)+(-0.968640007206)
        ref[1,0,1]=(-1.59448994908)/(o+1.)+(0.204838194218)
        ref[1,1,0]=(-2.07497671334)/(o+1.)+(-0.161793109438)
        ref[1,1,1]=(0.199659680372)/(o+1.)+(0.422123770364)
        ref[2,0,0]=(-1.86970457224)/(o+1.)+(0.590445950504)
        ref[2,0,1]=(-1.09505237157)/(o+1.)+(-0.334826102411)
        ref[2,1,0]=(0.395146564722)/(o+1.)+(0.889920278785)
        ref[2,1,1]=(1.13315474193)/(o+1.)+(-0.143906330909)
        ref[3,0,0]=(-0.804398097072)/(o+1.)+(0.65461419986)
        ref[3,0,1]=(-0.829074327243)/(o+1.)+(-0.0806532771448)
        ref[3,1,0]=(0.126278821088)/(o+1.)+(0.176626286045)
        ref[3,1,1]=(-0.498753315826)/(o+1.)+(0.953269651743)
        ref[4,0,0]=(0.33849802718)/(o+1.)+(-0.0859043723417)
        ref[4,0,1]=(0.153172891455)/(o+1.)+(0.672631245684)
        ref[4,1,0]=(1.42803542207)/(o+1.)+(-0.154893936468)
        ref[4,1,1]=(0.605555698814)/(o+1.)+(-0.249954971274)
        ref[5,0,0]=(-1.21823919194)/(o+1.)+(-1.13748314774)
        ref[5,0,1]=(-1.13684484437)/(o+1.)+(0.107355205274)
        ref[5,1,0]=(-1.21991007491)/(o+1.)+(0.137536202333)
        ref[5,1,1]=(2.05825898341)/(o+1.)+(0.112003969962)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunction_fromData_ReducedContinuousFunction_rank4(self):
      """
      tests integral of rank 4 Data on the ReducedFunction

      assumptions: ReducedContinuousFunction(self.domain) exists
                   self.domain supports integral on ReducedFunction
      """
      o=1
      dim=self.domain.getDim()
      w_ref=ReducedFunction(self.domain)
      w=ReducedContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5, 3, 2),w)
      ref=numpy.zeros((4, 5, 3, 2),numpy.float_)
      if dim==2:
        arg[0,0,0,0]=(-0.2631296794)*x[0]+(-0.130117190301)*x[1]
        arg[0,0,0,1]=(0.635732952218)*x[0]+(-0.647555274619)*x[1]
        arg[0,0,1,0]=(1.06301179339)*x[0]+(-0.227813591181)*x[1]
        arg[0,0,1,1]=(-0.519979711376)*x[0]+(1.04747480856)*x[1]
        arg[0,0,2,0]=(0.0232818794686)*x[0]+(-1.14799687633)*x[1]
        arg[0,0,2,1]=(0.359605391616)*x[0]+(-1.32942998109)*x[1]
        arg[0,1,0,0]=(-0.555284020205)*x[0]+(0.11676202509)*x[1]
        arg[0,1,0,1]=(-1.83479976366)*x[0]+(0.413168061643)*x[1]
        arg[0,1,1,0]=(0.239736306037)*x[0]+(0.680002261596)*x[1]
        arg[0,1,1,1]=(0.0555978436642)*x[0]+(1.28910533087)*x[1]
        arg[0,1,2,0]=(0.732696470486)*x[0]+(0.538837927425)*x[1]
        arg[0,1,2,1]=(0.528018376979)*x[0]+(1.20428218411)*x[1]
        arg[0,2,0,0]=(0.863803666388)*x[0]+(0.187082373754)*x[1]
        arg[0,2,0,1]=(-1.41490739709)*x[0]+(-0.741935114995)*x[1]
        arg[0,2,1,0]=(0.0501211708116)*x[0]+(-0.370371645307)*x[1]
        arg[0,2,1,1]=(-0.0353423795999)*x[0]+(1.00911245551)*x[1]
        arg[0,2,2,0]=(1.79997397117)*x[0]+(0.240103369892)*x[1]
        arg[0,2,2,1]=(0.926052414873)*x[0]+(0.00914357056458)*x[1]
        arg[0,3,0,0]=(-1.38911305622)*x[0]+(-1.29285413766)*x[1]
        arg[0,3,0,1]=(0.464908270265)*x[0]+(-0.354824878801)*x[1]
        arg[0,3,1,0]=(0.17477491298)*x[0]+(0.882807721541)*x[1]
        arg[0,3,1,1]=(0.633177386897)*x[0]+(-0.312628707154)*x[1]
        arg[0,3,2,0]=(0.0841289229919)*x[0]+(-0.244085514309)*x[1]
        arg[0,3,2,1]=(0.198631529554)*x[0]+(-0.324387433152)*x[1]
        arg[0,4,0,0]=(-1.41503119851)*x[0]+(-0.760818812719)*x[1]
        arg[0,4,0,1]=(-0.0326236565753)*x[0]+(0.835716629913)*x[1]
        arg[0,4,1,0]=(1.84482629546)*x[0]+(-0.341557269651)*x[1]
        arg[0,4,1,1]=(-0.380600663728)*x[0]+(-0.295663906776)*x[1]
        arg[0,4,2,0]=(0.411308057157)*x[0]+(1.0109767907)*x[1]
        arg[0,4,2,1]=(-1.10494637617)*x[0]+(0.401374247774)*x[1]
        arg[1,0,0,0]=(0.437019391027)*x[0]+(0.0507256248679)*x[1]
        arg[1,0,0,1]=(-0.399457976892)*x[0]+(1.60118981524)*x[1]
        arg[1,0,1,0]=(-1.51289007675)*x[0]+(0.821406477166)*x[1]
        arg[1,0,1,1]=(0.405770274494)*x[0]+(0.637591733337)*x[1]
        arg[1,0,2,0]=(-1.4945593715)*x[0]+(1.0919176668)*x[1]
        arg[1,0,2,1]=(-0.482868528044)*x[0]+(0.578004764959)*x[1]
        arg[1,1,0,0]=(-1.44298476434)*x[0]+(0.197151370973)*x[1]
        arg[1,1,0,1]=(1.38517677391)*x[0]+(0.915352009234)*x[1]
        arg[1,1,1,0]=(1.69025511761)*x[0]+(0.0203268727501)*x[1]
        arg[1,1,1,1]=(0.615392847773)*x[0]+(-0.301873832451)*x[1]
        arg[1,1,2,0]=(0.3081860115)*x[0]+(-0.495230532585)*x[1]
        arg[1,1,2,1]=(-0.0258752746808)*x[0]+(0.441629273958)*x[1]
        arg[1,2,0,0]=(-0.947577742489)*x[0]+(0.218632343222)*x[1]
        arg[1,2,0,1]=(0.520029690788)*x[0]+(0.403806873912)*x[1]
        arg[1,2,1,0]=(1.32820278922)*x[0]+(-1.68314905373)*x[1]
        arg[1,2,1,1]=(1.30409876916)*x[0]+(-0.721819171819)*x[1]
        arg[1,2,2,0]=(1.37872038109)*x[0]+(0.0725139769178)*x[1]
        arg[1,2,2,1]=(0.35593830057)*x[0]+(0.28751298465)*x[1]
        arg[1,3,0,0]=(-1.17950311406)*x[0]+(-0.232819397686)*x[1]
        arg[1,3,0,1]=(0.17340184781)*x[0]+(-0.43329852484)*x[1]
        arg[1,3,1,0]=(-1.71125828064)*x[0]+(0.957284609667)*x[1]
        arg[1,3,1,1]=(-0.429043922461)*x[0]+(0.15749154348)*x[1]
        arg[1,3,2,0]=(-1.42626208855)*x[0]+(-1.12812746347)*x[1]
        arg[1,3,2,1]=(0.490516645689)*x[0]+(0.559039786338)*x[1]
        arg[1,4,0,0]=(0.436395831147)*x[0]+(0.382691724897)*x[1]
        arg[1,4,0,1]=(1.27903773839)*x[0]+(-0.0642376846324)*x[1]
        arg[1,4,1,0]=(0.503159792753)*x[0]+(0.392973700403)*x[1]
        arg[1,4,1,1]=(0.0881500114326)*x[0]+(0.834121935361)*x[1]
        arg[1,4,2,0]=(-0.620745962116)*x[0]+(-0.0770814791601)*x[1]
        arg[1,4,2,1]=(0.0231684675117)*x[0]+(-1.33994827808)*x[1]
        arg[2,0,0,0]=(-0.972311570714)*x[0]+(-0.235293375636)*x[1]
        arg[2,0,0,1]=(0.21633355466)*x[0]+(0.11325401613)*x[1]
        arg[2,0,1,0]=(0.922986727965)*x[0]+(-1.00516246282)*x[1]
        arg[2,0,1,1]=(-0.740778981838)*x[0]+(0.616728971608)*x[1]
        arg[2,0,2,0]=(1.73201175195)*x[0]+(-1.1634325209)*x[1]
        arg[2,0,2,1]=(1.15829744421)*x[0]+(-1.87219685708)*x[1]
        arg[2,1,0,0]=(-0.458901790255)*x[0]+(1.13812715713)*x[1]
        arg[2,1,0,1]=(1.03315725208)*x[0]+(-0.112487270341)*x[1]
        arg[2,1,1,0]=(-1.7070220527)*x[0]+(0.126058081856)*x[1]
        arg[2,1,1,1]=(0.863301459388)*x[0]+(0.226670498194)*x[1]
        arg[2,1,2,0]=(-1.0012992544)*x[0]+(-0.195604893926)*x[1]
        arg[2,1,2,1]=(-0.128048233768)*x[0]+(0.827383267773)*x[1]
        arg[2,2,0,0]=(-0.0876478859699)*x[0]+(1.69600082596)*x[1]
        arg[2,2,0,1]=(-0.616278041482)*x[0]+(-0.995548774612)*x[1]
        arg[2,2,1,0]=(0.780699602657)*x[0]+(-0.721498796198)*x[1]
        arg[2,2,1,1]=(-1.17591990575)*x[0]+(-0.66080847955)*x[1]
        arg[2,2,2,0]=(1.71259481962)*x[0]+(-1.12456334234)*x[1]
        arg[2,2,2,1]=(-0.721345747555)*x[0]+(-1.28222132706)*x[1]
        arg[2,3,0,0]=(-1.25971324146)*x[0]+(-0.00559590465243)*x[1]
        arg[2,3,0,1]=(-0.49234245798)*x[0]+(0.30900436328)*x[1]
        arg[2,3,1,0]=(1.3068919966)*x[0]+(-0.891484126193)*x[1]
        arg[2,3,1,1]=(0.319277203381)*x[0]+(-0.535113401495)*x[1]
        arg[2,3,2,0]=(0.529480518887)*x[0]+(0.00452686996008)*x[1]
        arg[2,3,2,1]=(-1.0978905592)*x[0]+(-0.421513714335)*x[1]
        arg[2,4,0,0]=(-1.27711509307)*x[0]+(-1.42208296178)*x[1]
        arg[2,4,0,1]=(-0.621292314868)*x[0]+(-1.10962518414)*x[1]
        arg[2,4,1,0]=(-0.183478331213)*x[0]+(1.0775900659)*x[1]
        arg[2,4,1,1]=(1.07354953663)*x[0]+(0.869765287054)*x[1]
        arg[2,4,2,0]=(-0.696050139647)*x[0]+(-0.120769338549)*x[1]
        arg[2,4,2,1]=(-0.895597167617)*x[0]+(1.41025466606)*x[1]
        arg[3,0,0,0]=(1.15262876611)*x[0]+(-0.505428168285)*x[1]
        arg[3,0,0,1]=(-0.277585299138)*x[0]+(0.352683243558)*x[1]
        arg[3,0,1,0]=(0.218694567952)*x[0]+(-0.170325565499)*x[1]
        arg[3,0,1,1]=(0.297325594036)*x[0]+(0.459308129811)*x[1]
        arg[3,0,2,0]=(-0.572509624935)*x[0]+(0.756188981433)*x[1]
        arg[3,0,2,1]=(0.226462444009)*x[0]+(-1.42100784266)*x[1]
        arg[3,1,0,0]=(-0.28388824767)*x[0]+(-0.306064847514)*x[1]
        arg[3,1,0,1]=(-0.805831316478)*x[0]+(0.756933495912)*x[1]
        arg[3,1,1,0]=(1.37344361233)*x[0]+(-0.377600924857)*x[1]
        arg[3,1,1,1]=(-0.0725302533884)*x[0]+(-0.425074450201)*x[1]
        arg[3,1,2,0]=(-0.066659431931)*x[0]+(1.39252394891)*x[1]
        arg[3,1,2,1]=(0.593440019224)*x[0]+(-0.945629564385)*x[1]
        arg[3,2,0,0]=(0.101850471598)*x[0]+(0.0552356415787)*x[1]
        arg[3,2,0,1]=(-0.407603005372)*x[0]+(-0.849371670514)*x[1]
        arg[3,2,1,0]=(0.435403192155)*x[0]+(0.213673097377)*x[1]
        arg[3,2,1,1]=(0.424081872219)*x[0]+(-0.0800529063085)*x[1]
        arg[3,2,2,0]=(0.189428134309)*x[0]+(-0.205813522374)*x[1]
        arg[3,2,2,1]=(-0.51449771505)*x[0]+(0.0570738169919)*x[1]
        arg[3,3,0,0]=(1.05431625119)*x[0]+(-0.0583139615539)*x[1]
        arg[3,3,0,1]=(0.0292261008496)*x[0]+(-0.569482715214)*x[1]
        arg[3,3,1,0]=(-0.155861158232)*x[0]+(0.628422800294)*x[1]
        arg[3,3,1,1]=(-1.24050157391)*x[0]+(-1.83949885089)*x[1]
        arg[3,3,2,0]=(-0.140709404689)*x[0]+(0.975529832713)*x[1]
        arg[3,3,2,1]=(-0.290824210426)*x[0]+(-0.198373932879)*x[1]
        arg[3,4,0,0]=(0.627236720719)*x[0]+(-0.306676521516)*x[1]
        arg[3,4,0,1]=(-0.964303098626)*x[0]+(-0.701698363587)*x[1]
        arg[3,4,1,0]=(-0.0447860906704)*x[0]+(0.324659175162)*x[1]
        arg[3,4,1,1]=(1.28273443124)*x[0]+(-0.764329261143)*x[1]
        arg[3,4,2,0]=(-0.023232709502)*x[0]+(-0.124231212653)*x[1]
        arg[3,4,2,1]=(0.510754136991)*x[0]+(0.570492207442)*x[1]
        ref[0,0,0,0]=(0.648764087616)/(o+1.)+(-0.521005478658)
        ref[0,0,0,1]=(-0.0127476670646)/(o+1.)+(0.000462672331599)
        ref[0,0,1,0]=(0.258601710827)/(o+1.)+(0.288298245692)
        ref[0,0,1,1]=(0.0577415806545)/(o+1.)+(0.234876758266)
        ref[0,0,2,0]=(-0.91129780757)/(o+1.)+(-0.106708594644)
        ref[0,0,2,1]=(0.0841946322525)/(o+1.)+(-0.527009610865)
        ref[0,1,0,0]=(0.111101563297)/(o+1.)+(-0.274811779206)
        ref[0,1,0,1]=(-0.504062576651)/(o+1.)+(-0.458784562683)
        ref[0,1,1,0]=(0.65560969277)/(o+1.)+(0.132064437431)
        ref[0,1,1,1]=(0.0866382135955)/(o+1.)+(0.629032480469)
        ref[0,1,2,0]=(1.36074036394)/(o+1.)+(-0.0446029830125)
        ref[0,1,2,1]=(1.16071753175)/(o+1.)+(0.285791514672)
        ref[0,2,0,0]=(0.693065154772)/(o+1.)+(0.178910442685)
        ref[0,2,0,1]=(-1.41209835364)/(o+1.)+(-0.372372079222)
        ref[0,2,1,0]=(-0.542625642859)/(o+1.)+(0.111187584182)
        ref[0,2,1,1]=(0.628714449405)/(o+1.)+(0.172527813253)
        ref[0,2,2,0]=(1.62887921583)/(o+1.)+(0.205599062617)
        ref[0,2,2,1]=(1.0640301541)/(o+1.)+(-0.0644170843301)
        ref[0,3,0,0]=(-1.09870552313)/(o+1.)+(-0.791630835375)
        ref[0,3,0,1]=(0.260706656546)/(o+1.)+(-0.0753116325413)
        ref[0,3,1,0]=(-0.0543842358149)/(o+1.)+(0.555983435168)
        ref[0,3,1,1]=(0.0369075211917)/(o+1.)+(0.141820579276)
        ref[0,3,2,0]=(-0.784250881508)/(o+1.)+(0.312147145095)
        ref[0,3,2,1]=(-0.178450092422)/(o+1.)+(0.0263470944113)
        ref[0,4,0,0]=(-0.993852999635)/(o+1.)+(-0.590998505799)
        ref[0,4,0,1]=(-0.0449039600312)/(o+1.)+(0.423998466684)
        ref[0,4,1,0]=(0.583535069477)/(o+1.)+(0.459866978166)
        ref[0,4,1,1]=(-0.251020886878)/(o+1.)+(-0.212621841813)
        ref[0,4,2,0]=(0.408688561613)/(o+1.)+(0.506798143121)
        ref[0,4,2,1]=(0.537090447747)/(o+1.)+(-0.620331288072)
        ref[1,0,0,0]=(0.37428496839)/(o+1.)+(0.0567300237522)
        ref[1,0,0,1]=(0.777672833448)/(o+1.)+(0.212029502451)
        ref[1,0,1,0]=(-0.745422608709)/(o+1.)+(0.026969504563)
        ref[1,0,1,1]=(0.466434983875)/(o+1.)+(0.288463511978)
        ref[1,0,2,0]=(0.237457788829)/(o+1.)+(-0.320049746767)
        ref[1,0,2,1]=(0.346974181328)/(o+1.)+(-0.125918972207)
        ref[1,1,0,0]=(-0.823737519034)/(o+1.)+(-0.211047937165)
        ref[1,1,0,1]=(1.00324369541)/(o+1.)+(0.648642543871)
        ref[1,1,1,0]=(0.674013158596)/(o+1.)+(0.518284415882)
        ref[1,1,1,1]=(0.403142463029)/(o+1.)+(-0.0448117238535)
        ref[1,1,2,0]=(0.0399328921703)/(o+1.)+(-0.113488706628)
        ref[1,1,2,1]=(0.111446633645)/(o+1.)+(0.152153682816)
        ref[1,2,0,0]=(-0.481646721692)/(o+1.)+(-0.123649338787)
        ref[1,2,0,1]=(0.857170000026)/(o+1.)+(0.033333282337)
        ref[1,2,1,0]=(-0.445362398126)/(o+1.)+(0.0452080668082)
        ref[1,2,1,1]=(-0.256201827386)/(o+1.)+(0.419240712362)
        ref[1,2,2,0]=(1.02991996311)/(o+1.)+(0.210657197448)
        ref[1,2,2,1]=(0.355299316895)/(o+1.)+(0.144075984162)
        ref[1,3,0,0]=(-0.679726322154)/(o+1.)+(-0.366298094796)
        ref[1,3,0,1]=(-0.67885027329)/(o+1.)+(0.20947679813)
        ref[1,3,1,0]=(-0.163081485233)/(o+1.)+(-0.295446092868)
        ref[1,3,1,1]=(-0.035768623673)/(o+1.)+(-0.117891877654)
        ref[1,3,2,0]=(-0.639368943038)/(o+1.)+(-0.957510304493)
        ref[1,3,2,1]=(0.160615066001)/(o+1.)+(0.444470683013)
        ref[1,4,0,0]=(1.0649487216)/(o+1.)+(-0.122930582777)
        ref[1,4,0,1]=(0.22766574122)/(o+1.)+(0.493567156271)
        ref[1,4,1,0]=(-0.481296537412)/(o+1.)+(0.688715015285)
        ref[1,4,1,1]=(-0.195268237132)/(o+1.)+(0.558770091963)
        ref[1,4,2,0]=(0.180618258117)/(o+1.)+(-0.439222849696)
        ref[1,4,2,1]=(0.0261012984826)/(o+1.)+(-0.671440554524)
        ref[2,0,0,0]=(-0.301101739758)/(o+1.)+(-0.453251603296)
        ref[2,0,0,1]=(0.00878006761908)/(o+1.)+(0.160403751586)
        ref[2,0,1,0]=(-0.80011300584)/(o+1.)+(0.35896863549)
        ref[2,0,1,1]=(0.00248831604776)/(o+1.)+(-0.0632691631389)
        ref[2,0,2,0]=(-0.0569220251136)/(o+1.)+(0.312750628078)
        ref[2,0,2,1]=(-0.440109525096)/(o+1.)+(-0.136894943888)
        ref[2,1,0,0]=(-0.465652661854)/(o+1.)+(0.572439014362)
        ref[2,1,0,1]=(0.0522958647723)/(o+1.)+(0.434187058485)
        ref[2,1,1,0]=(0.00194260180185)/(o+1.)+(-0.791453286324)
        ref[2,1,1,1]=(0.981996325082)/(o+1.)+(0.0539878162501)
        ref[2,1,2,0]=(-0.243334369882)/(o+1.)+(-0.476784889223)
        ref[2,1,2,1]=(1.53208640175)/(o+1.)+(-0.416375683873)
        ref[2,2,0,0]=(1.27849139532)/(o+1.)+(0.164930772337)
        ref[2,2,0,1]=(-1.4813889023)/(o+1.)+(-0.0652189568992)
        ref[2,2,1,0]=(-0.728282325831)/(o+1.)+(0.393741566145)
        ref[2,2,1,1]=(-0.537765426582)/(o+1.)+(-0.649481479359)
        ref[2,2,2,0]=(-0.266224153785)/(o+1.)+(0.427127815533)
        ref[2,2,2,1]=(-0.862349098277)/(o+1.)+(-0.570608988167)
        ref[2,3,0,0]=(-1.09340368619)/(o+1.)+(-0.0859527299593)
        ref[2,3,0,1]=(1.02842400947)/(o+1.)+(-0.605881052086)
        ref[2,3,1,0]=(0.0131810342609)/(o+1.)+(0.20111341807)
        ref[2,3,1,1]=(-1.11244447248)/(o+1.)+(0.448304137186)
        ref[2,3,2,0]=(-0.978348257673)/(o+1.)+(0.75617782326)
        ref[2,3,2,1]=(-0.140801959181)/(o+1.)+(-0.689301157175)
        ref[2,4,0,0]=(-0.733659962936)/(o+1.)+(-0.982769045955)
        ref[2,4,0,1]=(-1.6278906672)/(o+1.)+(-0.0515134159034)
        ref[2,4,1,0]=(1.52117858199)/(o+1.)+(-0.313533423654)
        ref[2,4,1,1]=(0.621221455007)/(o+1.)+(0.661046684337)
        ref[2,4,2,0]=(-0.157580028393)/(o+1.)+(-0.329619724901)
        ref[2,4,2,1]=(0.52303412622)/(o+1.)+(-0.00418831388809)
        ref[3,0,0,0]=(0.337890126112)/(o+1.)+(0.154655235856)
        ref[3,0,0,1]=(-0.100490423466)/(o+1.)+(0.0877941839427)
        ref[3,0,1,0]=(0.37549944857)/(o+1.)+(-0.163565223059)
        ref[3,0,1,1]=(-0.3212149071)/(o+1.)+(0.538924315474)
        ref[3,0,2,0]=(-0.359815373231)/(o+1.)+(0.271747364864)
        ref[3,0,2,1]=(-0.890575856923)/(o+1.)+(-0.151984770865)
        ref[3,1,0,0]=(-1.10384271628)/(o+1.)+(0.256944810546)
        ref[3,1,0,1]=(0.647425296535)/(o+1.)+(-0.348161558551)
        ref[3,1,1,0]=(1.41806770884)/(o+1.)+(-0.211112510685)
        ref[3,1,1,1]=(0.106528838317)/(o+1.)+(-0.302066770953)
        ref[3,1,2,0]=(0.917213295746)/(o+1.)+(0.204325610614)
        ref[3,1,2,1]=(-0.24738513406)/(o+1.)+(-0.0524022055505)
        ref[3,2,0,0]=(-0.0537683888373)/(o+1.)+(0.105427251007)
        ref[3,2,0,1]=(-1.24647336894)/(o+1.)+(-0.00525065347416)
        ref[3,2,1,0]=(0.187132797735)/(o+1.)+(0.230971745898)
        ref[3,2,1,1]=(0.492750417331)/(o+1.)+(-0.0743607257106)
        ref[3,2,2,0]=(0.062835017839)/(o+1.)+(-0.0396102029518)
        ref[3,2,2,1]=(-0.967272854749)/(o+1.)+(0.254924478345)
        ref[3,3,0,0]=(1.41291903854)/(o+1.)+(-0.208458374448)
        ref[3,3,0,1]=(-0.740922121718)/(o+1.)+(0.100332753677)
        ref[3,3,1,0]=(1.2839228132)/(o+1.)+(-0.405680585568)
        ref[3,3,1,1]=(-1.69516258968)/(o+1.)+(-0.69241891756)
        ref[3,3,2,0]=(0.216594952745)/(o+1.)+(0.309112737639)
        ref[3,3,2,1]=(-0.634806027047)/(o+1.)+(0.0728039418712)
        ref[3,4,0,0]=(-0.310491357088)/(o+1.)+(0.315525778146)
        ref[3,4,0,1]=(-0.751915044154)/(o+1.)+(-0.457043209029)
        ref[3,4,1,0]=(-0.339963591836)/(o+1.)+(0.309918338163)
        ref[3,4,1,1]=(0.0720250143477)/(o+1.)+(0.223190077872)
        ref[3,4,2,0]=(-0.0508554115866)/(o+1.)+(-0.0483042552841)
        ref[3,4,2,1]=(0.754037720807)/(o+1.)+(0.163604311813)
      else:
        arg[0,0,0,0]=(-1.24350959032)*x[0]+(-0.580181370761)*x[1]+(-0.174543112173)*x[2]
        arg[0,0,0,1]=(1.43450421585)*x[0]+(0.717062915298)*x[1]+(-0.91866562143)*x[2]
        arg[0,0,1,0]=(0.0188823859409)*x[0]+(1.24568141807)*x[1]+(0.0842604494481)*x[2]
        arg[0,0,1,1]=(0.871591276766)*x[0]+(0.653031319605)*x[1]+(0.037714556611)*x[2]
        arg[0,0,2,0]=(1.60473786555)*x[0]+(1.47509195705)*x[1]+(-0.103485610136)*x[2]
        arg[0,0,2,1]=(1.68030531336)*x[0]+(0.551201281257)*x[1]+(1.26951824468)*x[2]
        arg[0,1,0,0]=(0.491638014782)*x[0]+(1.26264219762)*x[1]+(-0.921297094204)*x[2]
        arg[0,1,0,1]=(0.20108233628)*x[0]+(-1.36096031252)*x[1]+(-0.878077046379)*x[2]
        arg[0,1,1,0]=(0.900775475455)*x[0]+(0.193423726606)*x[1]+(-0.139955126775)*x[2]
        arg[0,1,1,1]=(-0.0354326925578)*x[0]+(-0.464646718718)*x[1]+(1.49240476583)*x[2]
        arg[0,1,2,0]=(-1.40991971664)*x[0]+(-1.7747715584)*x[1]+(-0.96387650203)*x[2]
        arg[0,1,2,1]=(-0.476443682267)*x[0]+(0.55076135194)*x[1]+(0.190289559512)*x[2]
        arg[0,2,0,0]=(0.535490315669)*x[0]+(0.566915366497)*x[1]+(-0.41513643529)*x[2]
        arg[0,2,0,1]=(-0.18986562478)*x[0]+(-0.62347050233)*x[1]+(1.18004907973)*x[2]
        arg[0,2,1,0]=(-0.288440432828)*x[0]+(0.0375642125411)*x[1]+(0.574653643212)*x[2]
        arg[0,2,1,1]=(1.03327739612)*x[0]+(-0.0443874572943)*x[1]+(0.972142765323)*x[2]
        arg[0,2,2,0]=(-1.27131901306)*x[0]+(0.619179457229)*x[1]+(-0.436029340417)*x[2]
        arg[0,2,2,1]=(0.0573099137019)*x[0]+(0.211373076629)*x[1]+(0.859991510967)*x[2]
        arg[0,3,0,0]=(0.650974277613)*x[0]+(-0.926352792387)*x[1]+(-1.84479828618)*x[2]
        arg[0,3,0,1]=(-0.293748564815)*x[0]+(0.53683587198)*x[1]+(1.51147453744)*x[2]
        arg[0,3,1,0]=(0.837889627485)*x[0]+(0.62568225238)*x[1]+(0.365321068424)*x[2]
        arg[0,3,1,1]=(-0.671841278343)*x[0]+(-0.00822894090357)*x[1]+(-1.35180265459)*x[2]
        arg[0,3,2,0]=(0.823443588183)*x[0]+(-0.366903894987)*x[1]+(-0.878402795844)*x[2]
        arg[0,3,2,1]=(-0.856292830088)*x[0]+(-0.134584368493)*x[1]+(-0.299126849425)*x[2]
        arg[0,4,0,0]=(0.62792423481)*x[0]+(0.781414088068)*x[1]+(-1.12299220031)*x[2]
        arg[0,4,0,1]=(-0.00743646774917)*x[0]+(0.52784004028)*x[1]+(-1.44284109205)*x[2]
        arg[0,4,1,0]=(-0.855519018884)*x[0]+(-0.222306698876)*x[1]+(0.0502517275952)*x[2]
        arg[0,4,1,1]=(-0.311060622512)*x[0]+(0.321283094479)*x[1]+(-0.484147152851)*x[2]
        arg[0,4,2,0]=(-1.43220161062)*x[0]+(-0.27822988291)*x[1]+(-0.692367106541)*x[2]
        arg[0,4,2,1]=(1.3956686934)*x[0]+(-1.26426333906)*x[1]+(-1.09343602989)*x[2]
        arg[1,0,0,0]=(0.122525927098)*x[0]+(-0.484009414389)*x[1]+(-0.822036495419)*x[2]
        arg[1,0,0,1]=(-0.276748173334)*x[0]+(0.35307031133)*x[1]+(-1.23466402387)*x[2]
        arg[1,0,1,0]=(-0.479355493223)*x[0]+(-0.56368304348)*x[1]+(0.54679107219)*x[2]
        arg[1,0,1,1]=(-0.483618214285)*x[0]+(1.60438722757)*x[1]+(0.542389963596)*x[2]
        arg[1,0,2,0]=(0.130988624202)*x[0]+(-0.449468495509)*x[1]+(0.254640520023)*x[2]
        arg[1,0,2,1]=(1.13790232796)*x[0]+(-0.716218913004)*x[1]+(-0.193031264955)*x[2]
        arg[1,1,0,0]=(-0.0653152151059)*x[0]+(1.07890922508)*x[1]+(0.324995864884)*x[2]
        arg[1,1,0,1]=(0.769151345051)*x[0]+(-1.35911748672)*x[1]+(0.299326803785)*x[2]
        arg[1,1,1,0]=(-0.431128032697)*x[0]+(1.09261915758)*x[1]+(-0.029871998484)*x[2]
        arg[1,1,1,1]=(-0.922618915502)*x[0]+(1.30318141394)*x[1]+(-0.324039981399)*x[2]
        arg[1,1,2,0]=(-0.668178375077)*x[0]+(0.132203602988)*x[1]+(0.236035070297)*x[2]
        arg[1,1,2,1]=(-0.518594419598)*x[0]+(-0.0429695477263)*x[1]+(1.11877941395)*x[2]
        arg[1,2,0,0]=(0.610776644247)*x[0]+(-0.023900598958)*x[1]+(0.0480141870788)*x[2]
        arg[1,2,0,1]=(0.562736426016)*x[0]+(-0.294969624732)*x[1]+(-0.23898469143)*x[2]
        arg[1,2,1,0]=(0.645173726197)*x[0]+(-1.18365074993)*x[1]+(0.102581636468)*x[2]
        arg[1,2,1,1]=(-0.518976597355)*x[0]+(0.372817596528)*x[1]+(0.49035249397)*x[2]
        arg[1,2,2,0]=(0.0192623731103)*x[0]+(-0.241279406644)*x[1]+(-0.420015571981)*x[2]
        arg[1,2,2,1]=(0.318641827599)*x[0]+(-1.5357263392)*x[1]+(0.350150621892)*x[2]
        arg[1,3,0,0]=(-0.719401628191)*x[0]+(-1.77806447133)*x[1]+(1.19750443209)*x[2]
        arg[1,3,0,1]=(-1.00122204304)*x[0]+(0.675162216383)*x[1]+(1.1180105058)*x[2]
        arg[1,3,1,0]=(0.331149947519)*x[0]+(0.310832741708)*x[1]+(0.087866556795)*x[2]
        arg[1,3,1,1]=(-0.156343054316)*x[0]+(1.00796776488)*x[1]+(-1.53460740228)*x[2]
        arg[1,3,2,0]=(0.714240747972)*x[0]+(0.968170992322)*x[1]+(-0.962819191598)*x[2]
        arg[1,3,2,1]=(-0.200412114379)*x[0]+(-0.173781234544)*x[1]+(0.899584749309)*x[2]
        arg[1,4,0,0]=(-1.38011177642)*x[0]+(-0.316506470647)*x[1]+(0.266952180469)*x[2]
        arg[1,4,0,1]=(0.343359684584)*x[0]+(-1.13244466712)*x[1]+(-0.81840155125)*x[2]
        arg[1,4,1,0]=(0.295299129957)*x[0]+(0.199810907018)*x[1]+(0.631717407771)*x[2]
        arg[1,4,1,1]=(-0.46840784478)*x[0]+(-0.402683932206)*x[1]+(-0.782209293177)*x[2]
        arg[1,4,2,0]=(-1.8006389736)*x[0]+(0.119727360087)*x[1]+(0.482076626168)*x[2]
        arg[1,4,2,1]=(0.96553885281)*x[0]+(0.425097333238)*x[1]+(0.54533551085)*x[2]
        arg[2,0,0,0]=(0.932725602425)*x[0]+(-0.513780652569)*x[1]+(0.275742915648)*x[2]
        arg[2,0,0,1]=(1.10053578072)*x[0]+(0.787895323287)*x[1]+(-1.46544221328)*x[2]
        arg[2,0,1,0]=(0.83657269666)*x[0]+(-1.15717482853)*x[1]+(1.1090271297)*x[2]
        arg[2,0,1,1]=(1.59370135994)*x[0]+(0.452015708845)*x[1]+(0.0862230059082)*x[2]
        arg[2,0,2,0]=(-0.954832509866)*x[0]+(0.51192257473)*x[1]+(-1.48499771643)*x[2]
        arg[2,0,2,1]=(-0.556993361475)*x[0]+(0.438110051421)*x[1]+(0.237254018143)*x[2]
        arg[2,1,0,0]=(-0.318980966383)*x[0]+(-0.263071719097)*x[1]+(-0.348122442404)*x[2]
        arg[2,1,0,1]=(0.100191287326)*x[0]+(-0.422996999974)*x[1]+(-0.850702087112)*x[2]
        arg[2,1,1,0]=(-0.198942237141)*x[0]+(0.277459568173)*x[1]+(-1.66195774421)*x[2]
        arg[2,1,1,1]=(-1.62211381863)*x[0]+(-0.13417844308)*x[1]+(1.07441481356)*x[2]
        arg[2,1,2,0]=(-0.183120762554)*x[0]+(-0.158546687568)*x[1]+(1.05938565108)*x[2]
        arg[2,1,2,1]=(-0.530596736933)*x[0]+(-0.507872303289)*x[1]+(0.0437745116021)*x[2]
        arg[2,2,0,0]=(-0.430338834548)*x[0]+(-1.01989755268)*x[1]+(0.0364152729771)*x[2]
        arg[2,2,0,1]=(-0.55129175645)*x[0]+(0.283907226465)*x[1]+(0.281403899376)*x[2]
        arg[2,2,1,0]=(0.119036949181)*x[0]+(0.187265158741)*x[1]+(-0.0845814658663)*x[2]
        arg[2,2,1,1]=(1.39900884968)*x[0]+(1.42806588781)*x[1]+(0.211444691268)*x[2]
        arg[2,2,2,0]=(-0.609664666581)*x[0]+(-0.29988332429)*x[1]+(-0.28125800646)*x[2]
        arg[2,2,2,1]=(-1.77545247157)*x[0]+(0.337828315779)*x[1]+(-0.239141292616)*x[2]
        arg[2,3,0,0]=(-1.23141604656)*x[0]+(0.322998749903)*x[1]+(-1.70685281816)*x[2]
        arg[2,3,0,1]=(-1.4416737434)*x[0]+(1.49740114642)*x[1]+(0.732057753882)*x[2]
        arg[2,3,1,0]=(-1.59112622077)*x[0]+(0.665317566947)*x[1]+(0.826417253499)*x[2]
        arg[2,3,1,1]=(-1.22000830454)*x[0]+(0.618713896065)*x[1]+(1.27354984534)*x[2]
        arg[2,3,2,0]=(0.125639723221)*x[0]+(0.663241510656)*x[1]+(-0.426459288281)*x[2]
        arg[2,3,2,1]=(1.39120212155)*x[0]+(0.9115748844)*x[1]+(1.77889030223)*x[2]
        arg[2,4,0,0]=(-0.217927656387)*x[0]+(1.20391185439)*x[1]+(-0.188197734908)*x[2]
        arg[2,4,0,1]=(-0.442009082947)*x[0]+(0.399477101959)*x[1]+(-1.33974368956)*x[2]
        arg[2,4,1,0]=(0.636534809382)*x[0]+(0.872923470503)*x[1]+(-0.922286702006)*x[2]
        arg[2,4,1,1]=(-0.831840624431)*x[0]+(0.946223248622)*x[1]+(0.380300039133)*x[2]
        arg[2,4,2,0]=(0.0803088903901)*x[0]+(-0.857685204574)*x[1]+(-0.479475079431)*x[2]
        arg[2,4,2,1]=(-1.7233942516)*x[0]+(-1.38771345941)*x[1]+(0.761958201633)*x[2]
        arg[3,0,0,0]=(-0.738032075796)*x[0]+(1.33264583681)*x[1]+(-1.21101988385)*x[2]
        arg[3,0,0,1]=(1.40336327022)*x[0]+(-0.77078885213)*x[1]+(1.70960627204)*x[2]
        arg[3,0,1,0]=(0.452477629041)*x[0]+(0.235432009434)*x[1]+(1.69314009822)*x[2]
        arg[3,0,1,1]=(-0.816213143646)*x[0]+(-1.42050350309)*x[1]+(-0.0806105456758)*x[2]
        arg[3,0,2,0]=(0.0400708086023)*x[0]+(-0.336948609628)*x[1]+(-0.113687293995)*x[2]
        arg[3,0,2,1]=(0.414077492095)*x[0]+(-1.06937716207)*x[1]+(1.25972041173)*x[2]
        arg[3,1,0,0]=(-1.02975555201)*x[0]+(-0.123306171085)*x[1]+(-1.74750909033)*x[2]
        arg[3,1,0,1]=(-1.01988698009)*x[0]+(-0.142188487205)*x[1]+(0.274514515492)*x[2]
        arg[3,1,1,0]=(0.440626586141)*x[0]+(-0.0569580581917)*x[1]+(-0.77065631367)*x[2]
        arg[3,1,1,1]=(0.987903067972)*x[0]+(-1.47543187274)*x[1]+(-0.0888962732893)*x[2]
        arg[3,1,2,0]=(0.150506204416)*x[0]+(0.035476396146)*x[1]+(-0.852061690418)*x[2]
        arg[3,1,2,1]=(-1.15179800132)*x[0]+(-0.425570974979)*x[1]+(1.10092214699)*x[2]
        arg[3,2,0,0]=(1.01255188057)*x[0]+(1.24279465527)*x[1]+(1.25190246102)*x[2]
        arg[3,2,0,1]=(-1.16275538385)*x[0]+(1.13672085108)*x[1]+(0.131953731231)*x[2]
        arg[3,2,1,0]=(0.774290747173)*x[0]+(0.275410883089)*x[1]+(-0.825692266439)*x[2]
        arg[3,2,1,1]=(-1.07241838074)*x[0]+(-0.0974675737726)*x[1]+(1.1856849697)*x[2]
        arg[3,2,2,0]=(0.234209248938)*x[0]+(0.756893234577)*x[1]+(-0.365141055079)*x[2]
        arg[3,2,2,1]=(-1.6668858309)*x[0]+(-0.916769120288)*x[1]+(-0.716365681217)*x[2]
        arg[3,3,0,0]=(-0.36890215249)*x[0]+(-0.581062815398)*x[1]+(1.6280164507)*x[2]
        arg[3,3,0,1]=(-0.933634743559)*x[0]+(-0.0657657779145)*x[1]+(1.35011362804)*x[2]
        arg[3,3,1,0]=(1.13739289322)*x[0]+(0.0103208127738)*x[1]+(1.01628193014)*x[2]
        arg[3,3,1,1]=(0.037979696459)*x[0]+(-1.76926650495)*x[1]+(-1.28602572757)*x[2]
        arg[3,3,2,0]=(0.797830431467)*x[0]+(0.732108360923)*x[1]+(-0.436390867333)*x[2]
        arg[3,3,2,1]=(-0.571865447888)*x[0]+(0.895109047357)*x[1]+(-0.673234648776)*x[2]
        arg[3,4,0,0]=(-1.59667195158)*x[0]+(-1.06146105331)*x[1]+(-1.81255923783)*x[2]
        arg[3,4,0,1]=(-0.0721327214647)*x[0]+(1.23982376787)*x[1]+(1.43101011)*x[2]
        arg[3,4,1,0]=(0.117288395573)*x[0]+(-0.37678966212)*x[1]+(-1.14107285089)*x[2]
        arg[3,4,1,1]=(-0.303270532321)*x[0]+(1.33784402324)*x[1]+(-0.651624796413)*x[2]
        arg[3,4,2,0]=(0.215849013256)*x[0]+(0.247315697644)*x[1]+(0.404644392153)*x[2]
        arg[3,4,2,1]=(0.975781190478)*x[0]+(-0.252285063767)*x[1]+(-0.145694910854)*x[2]
        ref[0,0,0,0]=(-0.718934738033)/(o+1.)+(-0.639649667608)
        ref[0,0,0,1]=(1.01007662915)/(o+1.)+(0.111412440284)
        ref[0,0,1,0]=(0.827868953278)/(o+1.)+(0.26047765009)
        ref[0,0,1,1]=(1.48122352358)/(o+1.)+(0.0405568147034)
        ref[0,0,2,0]=(1.33282099436)/(o+1.)+(0.821761609054)
        ref[0,0,2,1]=(1.65298089862)/(o+1.)+(0.924021970342)
        ref[0,1,0,0]=(0.930162571453)/(o+1.)+(-0.0485897266263)
        ref[0,1,0,1]=(-0.347599158269)/(o+1.)+(-0.845177932176)
        ref[0,1,1,0]=(0.623152121663)/(o+1.)+(0.165545976811)
        ref[0,1,1,1]=(0.0506217690841)/(o+1.)+(0.470851792737)
        ref[0,1,2,0]=(-1.99181491207)/(o+1.)+(-1.0783764325)
        ref[0,1,2,1]=(-0.731065007801)/(o+1.)+(0.497836118493)
        ref[0,2,0,0]=(-0.0812249009151)/(o+1.)+(0.384247073896)
        ref[0,2,0,1]=(-0.115768505253)/(o+1.)+(0.241240728938)
        ref[0,2,1,0]=(0.152728009607)/(o+1.)+(0.0855247066589)
        ref[0,2,1,1]=(1.43086526503)/(o+1.)+(0.265083719563)
        ref[0,2,2,0]=(-0.967906369418)/(o+1.)+(-0.060131263415)
        ref[0,2,2,1]=(-0.269269238419)/(o+1.)+(0.698971869859)
        ref[0,3,0,0]=(-1.71659801402)/(o+1.)+(-0.201789393467)
        ref[0,3,0,1]=(0.265343635273)/(o+1.)+(0.744609104665)
        ref[0,3,1,0]=(1.83021875237)/(o+1.)+(-0.000662902039879)
        ref[0,3,1,1]=(-0.317529833375)/(o+1.)+(-0.857171520229)
        ref[0,3,2,0]=(-0.705757693524)/(o+1.)+(0.141947295438)
        ref[0,3,2,1]=(-1.27000495392)/(o+1.)+(-0.00999954704202)
        ref[0,4,0,0]=(0.061892522425)/(o+1.)+(0.112226800069)
        ref[0,4,0,1]=(-0.747062267148)/(o+1.)+(-0.0876876261864)
        ref[0,4,1,0]=(-1.19488448842)/(o+1.)+(0.0836552491261)
        ref[0,4,1,1]=(1.36573976292)/(o+1.)+(-0.9198322219)
        ref[0,4,2,0]=(-1.67982141697)/(o+1.)+(-0.361488591554)
        ref[0,4,2,1]=(-0.598050084547)/(o+1.)+(-0.181990295501)
        ref[1,0,0,0]=(-2.45927674426)/(o+1.)+(0.637878380777)
        ref[1,0,0,1]=(-0.575223026885)/(o+1.)+(-0.291559429493)
        ref[1,0,1,0]=(0.0871765959469)/(o+1.)+(-0.29171203023)
        ref[1,0,1,1]=(0.94050575309)/(o+1.)+(0.361326611893)
        ref[1,0,2,0]=(-0.306500880914)/(o+1.)+(0.121330764815)
        ref[1,0,2,1]=(1.60748296414)/(o+1.)+(-0.689415407072)
        ref[1,1,0,0]=(0.640401899755)/(o+1.)+(0.34909398755)
        ref[1,1,0,1]=(0.134022909144)/(o+1.)+(-0.212331123516)
        ref[1,1,1,0]=(0.248008519593)/(o+1.)+(0.191805303402)
        ref[1,1,1,1]=(0.638911732328)/(o+1.)+(-0.291194607647)
        ref[1,1,2,0]=(0.109264462855)/(o+1.)+(-0.204602082323)
        ref[1,1,2,1]=(-0.185963112656)/(o+1.)+(0.371589279643)
        ref[1,2,0,0]=(-0.590669733193)/(o+1.)+(0.61277998278)
        ref[1,2,0,1]=(-1.15856908266)/(o+1.)+(0.593675596257)
        ref[1,2,1,0]=(-0.451642187813)/(o+1.)+(0.00787340027166)
        ref[1,2,1,1]=(-0.428757501259)/(o+1.)+(0.386475497201)
        ref[1,2,2,0]=(-0.414532914271)/(o+1.)+(-0.113749845622)
        ref[1,2,2,1]=(-0.513613710163)/(o+1.)+(-0.176660089771)
        ref[1,3,0,0]=(-0.890487934256)/(o+1.)+(-0.20473686659)
        ref[1,3,0,1]=(1.64512779208)/(o+1.)+(-0.426588556466)
        ref[1,3,1,0]=(0.831527595966)/(o+1.)+(-0.0508391749725)
        ref[1,3,1,1]=(-0.718921003194)/(o+1.)+(0.0179691557385)
        ref[1,3,2,0]=(0.405959213301)/(o+1.)+(0.156816667697)
        ref[1,3,2,1]=(0.426319486322)/(o+1.)+(0.0495359570318)
        ref[1,4,0,0]=(-1.11364330874)/(o+1.)+(-0.158011378929)
        ref[1,4,0,1]=(-0.0993967101457)/(o+1.)+(-0.754044911817)
        ref[1,4,1,0]=(0.979493241598)/(o+1.)+(0.0736671015741)
        ref[1,4,1,1]=(-1.78692478393)/(o+1.)+(0.0668118568857)
        ref[1,4,2,0]=(-0.691473647622)/(o+1.)+(-0.253680669863)
        ref[1,4,2,1]=(1.16914997228)/(o+1.)+(0.383410862311)
        ref[2,0,0,0]=(0.790187638399)/(o+1.)+(-0.047749886448)
        ref[2,0,0,1]=(0.429625240932)/(o+1.)+(-0.00331817510394)
        ref[2,0,1,0]=(-0.101344997877)/(o+1.)+(0.444884997853)
        ref[2,0,1,1]=(2.15519953039)/(o+1.)+(-0.0116297278494)
        ref[2,0,2,0]=(-1.40159594866)/(o+1.)+(-0.263155851453)
        ref[2,0,2,1]=(-0.658391287862)/(o+1.)+(0.388380997976)
        ref[2,1,0,0]=(-0.773933630328)/(o+1.)+(-0.0781207487777)
        ref[2,1,0,1]=(0.0015551432788)/(o+1.)+(-0.587531471519)
        ref[2,1,1,0]=(0.122693324758)/(o+1.)+(-0.853066868967)
        ref[2,1,1,1]=(-0.219255630453)/(o+1.)+(-0.231310908847)
        ref[2,1,2,0]=(1.33176471286)/(o+1.)+(-0.307023255951)
        ref[2,1,2,1]=(-1.67471265259)/(o+1.)+(0.340009061986)
        ref[2,2,0,0]=(-0.727222063897)/(o+1.)+(-0.343299525178)
        ref[2,2,0,1]=(-1.87667590405)/(o+1.)+(0.945347636719)
        ref[2,2,1,0]=(0.776003927859)/(o+1.)+(-0.277141642902)
        ref[2,2,1,1]=(1.62850322537)/(o+1.)+(0.705008101698)
        ref[2,2,2,0]=(1.31869548331)/(o+1.)+(-1.25475074032)
        ref[2,2,2,1]=(-1.71691241115)/(o+1.)+(0.0200734813728)
        ref[2,3,0,0]=(-0.37741220744)/(o+1.)+(-1.11892895368)
        ref[2,3,0,1]=(0.684763737686)/(o+1.)+(0.0515107096073)
        ref[2,3,1,0]=(-0.0572023243047)/(o+1.)+(-0.0210945380092)
        ref[2,3,1,1]=(0.031457290932)/(o+1.)+(0.320399072966)
        ref[2,3,2,0]=(-0.71613334286)/(o+1.)+(0.539277644228)
        ref[2,3,2,1]=(2.10068005021)/(o+1.)+(0.990493628981)
        ref[2,4,0,0]=(0.830379294407)/(o+1.)+(-0.016296415655)
        ref[2,4,0,1]=(-1.05197706105)/(o+1.)+(-0.165149304749)
        ref[2,4,1,0]=(-0.0854959336547)/(o+1.)+(0.336333755767)
        ref[2,4,1,1]=(1.15845594915)/(o+1.)+(-0.331886642912)
        ref[2,4,2,0]=(-1.25248721318)/(o+1.)+(-0.00218209021539)
        ref[2,4,2,1]=(-0.991752052473)/(o+1.)+(-0.678698728452)
        ref[3,0,0,0]=(-0.262071815816)/(o+1.)+(-0.177167153508)
        ref[3,0,0,1]=(1.20598261029)/(o+1.)+(0.568099039922)
        ref[3,0,1,0]=(1.21237099375)/(o+1.)+(0.584339371474)
        ref[3,0,1,1]=(-1.04628498282)/(o+1.)+(-0.635521104799)
        ref[3,0,2,0]=(0.630843868383)/(o+1.)+(-0.520704481702)
        ref[3,0,2,1]=(1.56136208039)/(o+1.)+(-0.478470669318)
        ref[3,1,0,0]=(-0.372752943047)/(o+1.)+(-1.26390893519)
        ref[3,1,0,1]=(0.192139112377)/(o+1.)+(-0.53985003209)
        ref[3,1,1,0]=(-1.18480584023)/(o+1.)+(0.398909027255)
        ref[3,1,1,1]=(0.402069277892)/(o+1.)+(-0.489247177975)
        ref[3,1,2,0]=(0.27608763117)/(o+1.)+(-0.471083360513)
        ref[3,1,2,1]=(-0.273636427696)/(o+1.)+(-0.101405200809)
        ref[3,2,0,0]=(1.92572771773)/(o+1.)+(0.790760639563)
        ref[3,2,0,1]=(0.16264122626)/(o+1.)+(-0.0283610138982)
        ref[3,2,1,0]=(0.377561018039)/(o+1.)+(-0.076775827108)
        ref[3,2,1,1]=(-0.652466873272)/(o+1.)+(0.334132944227)
        ref[3,2,2,0]=(-0.0304721826842)/(o+1.)+(0.32821680556)
        ref[3,2,2,1]=(-1.52932569088)/(o+1.)+(-0.885347470761)
        ref[3,3,0,0]=(-0.451938162279)/(o+1.)+(0.564994822546)
        ref[3,3,0,1]=(-0.477583137662)/(o+1.)+(0.414148122113)
        ref[3,3,1,0]=(1.38037325689)/(o+1.)+(0.391811189625)
        ref[3,3,1,1]=(-1.01333159097)/(o+1.)+(-1.00199047255)
        ref[3,3,2,0]=(1.3222644334)/(o+1.)+(-0.114358254173)
        ref[3,3,2,1]=(-0.248989333346)/(o+1.)+(-0.0505008579799)
        ref[3,4,0,0]=(-1.92196761736)/(o+1.)+(-1.27436231268)
        ref[3,4,0,1]=(0.894404962457)/(o+1.)+(0.852148096971)
        ref[3,4,1,0]=(0.868763436571)/(o+1.)+(-1.134668777)
        ref[3,4,1,1]=(0.438747123022)/(o+1.)+(-0.0278992142568)
        ref[3,4,2,0]=(0.532575431402)/(o+1.)+(0.167616835826)
        ref[3,4,2,1]=(0.796744496799)/(o+1.)+(-0.109471640471)
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5, 3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunction_fromData_ReducedFunction_rank0(self):
      """
      tests integral of rank 0 Data on the ReducedFunction

      assumptions: ReducedFunction(self.domain) exists
                   self.domain supports integral on ReducedFunction
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=ReducedFunction(self.domain)
      w=ReducedFunction(self.domain)
      x=w.getX()
      arg=Data(0,(),w)
      if dim==2:
        arg=(-0.217470460227)*x[0]**o+(0.664132905811)*x[0]+(0.493878617743)*x[1]**o+(0.111327715714)*x[1]
        ref=(0.276408157516)/(o+1.)+(0.387730310763)
      else:
        arg=(0.371937838886)*x[0]**o+(0.653846807778)*x[0]+(0.172358058188)*x[1]**o+(-0.928017916614)*x[1]+(0.116156616174)*x[2]**o+(0.371332501889)*x[2]
        ref=(0.660452513249)/(o+1.)+(0.0485806965265)
      res=integrate(arg)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunction_fromData_ReducedFunction_rank1(self):
      """
      tests integral of rank 1 Data on the ReducedFunction

      assumptions: ReducedFunction(self.domain) exists
                   self.domain supports integral on ReducedFunction
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=ReducedFunction(self.domain)
      w=ReducedFunction(self.domain)
      x=w.getX()
      arg=Data(0,(2,),w)
      ref=numpy.zeros((2,),numpy.float_)
      if dim==2:
        arg[0]=(-0.329952046701)*x[0]**o+(0.12757022314)*x[0]+(0.198280362833)*x[1]**o+(0.129179854334)*x[1]
        arg[1]=(0.185895161332)*x[0]**o+(0.270740935936)*x[0]+(0.630122833594)*x[1]**o+(-0.0696403698787)*x[1]
        ref[0]=(-0.131671683869)/(o+1.)+(0.128375038737)
        ref[1]=(0.816017994925)/(o+1.)+(0.100550283029)
      else:
        arg[0]=(0.526257699936)*x[0]**o+(0.0731442796396)*x[0]+(-0.266121642119)*x[1]**o+(-0.929478242136)*x[1]+(0.743092861949)*x[2]**o+(0.22690419858)*x[2]
        arg[1]=(-0.121387039147)*x[0]**o+(-0.0580608263784)*x[0]+(0.761164059949)*x[1]**o+(0.821938972431)*x[1]+(-0.756423175962)*x[2]**o+(0.213207132353)*x[2]
        ref[0]=(1.00322891977)/(o+1.)+(-0.314714881958)
        ref[1]=(-0.11664615516)/(o+1.)+(0.488542639203)
      res=integrate(arg)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunction_fromData_ReducedFunction_rank2(self):
      """
      tests integral of rank 2 Data on the ReducedFunction

      assumptions: ReducedFunction(self.domain) exists
                   self.domain supports integral on ReducedFunction
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=ReducedFunction(self.domain)
      w=ReducedFunction(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5),w)
      ref=numpy.zeros((4, 5),numpy.float_)
      if dim==2:
        arg[0,0]=(-0.963318847499)*x[0]**o+(-0.184977238308)*x[0]+(0.153633231879)*x[1]**o+(-0.640028315216)*x[1]
        arg[0,1]=(0.0879103656102)*x[0]**o+(-0.928854425726)*x[0]+(-0.138566176095)*x[1]**o+(-0.670118398936)*x[1]
        arg[0,2]=(0.920856326204)*x[0]**o+(0.461047594595)*x[0]+(-0.914778320363)*x[1]**o+(-0.561644217568)*x[1]
        arg[0,3]=(0.506148289443)*x[0]**o+(-0.474044343536)*x[0]+(0.742127575836)*x[1]**o+(-0.52349314159)*x[1]
        arg[0,4]=(0.0810224543145)*x[0]**o+(-0.307992483498)*x[0]+(0.749673245291)*x[1]**o+(-0.554099873044)*x[1]
        arg[1,0]=(-0.270897099553)*x[0]**o+(-0.222178461886)*x[0]+(0.755729049844)*x[1]**o+(0.740743777894)*x[1]
        arg[1,1]=(0.0220934176819)*x[0]**o+(-0.309641658323)*x[0]+(-0.951833592348)*x[1]**o+(0.430554526918)*x[1]
        arg[1,2]=(0.176033311218)*x[0]**o+(0.728188001962)*x[0]+(-0.428357954006)*x[1]**o+(0.955059500699)*x[1]
        arg[1,3]=(0.946680422474)*x[0]**o+(-0.251968148743)*x[0]+(-0.3704600461)*x[1]**o+(0.692080922297)*x[1]
        arg[1,4]=(0.122655375591)*x[0]**o+(0.617285886533)*x[0]+(0.729184247986)*x[1]**o+(0.30369620264)*x[1]
        arg[2,0]=(-0.957780501271)*x[0]**o+(0.438290429468)*x[0]+(-0.504841431279)*x[1]**o+(0.935350834608)*x[1]
        arg[2,1]=(-0.49410765407)*x[0]**o+(-0.635142349002)*x[0]+(0.700131616572)*x[1]**o+(0.99523209861)*x[1]
        arg[2,2]=(0.0967686936303)*x[0]**o+(0.987754073381)*x[0]+(-0.161220173386)*x[1]**o+(-0.343926097784)*x[1]
        arg[2,3]=(0.560040886628)*x[0]**o+(-0.843778720323)*x[0]+(0.345354034369)*x[1]**o+(0.440584543858)*x[1]
        arg[2,4]=(0.988977815245)*x[0]**o+(0.661922598357)*x[0]+(-0.425421002549)*x[1]**o+(0.624279872655)*x[1]
        arg[3,0]=(-0.997217068098)*x[0]**o+(-0.973497091595)*x[0]+(-0.0980683851598)*x[1]**o+(0.408081557041)*x[1]
        arg[3,1]=(0.0827616522414)*x[0]**o+(0.931022772362)*x[0]+(-0.588688206358)*x[1]**o+(0.900467348992)*x[1]
        arg[3,2]=(0.0933642874383)*x[0]**o+(0.398732664158)*x[0]+(-0.437102212783)*x[1]**o+(-0.126538302466)*x[1]
        arg[3,3]=(0.251989644486)*x[0]**o+(0.765809815972)*x[0]+(-0.325455283605)*x[1]**o+(0.171819372538)*x[1]
        arg[3,4]=(0.649407682145)*x[0]**o+(0.963776897105)*x[0]+(0.671915752131)*x[1]**o+(-0.468729629153)*x[1]
        ref[0,0]=(-0.80968561562)/(o+1.)+(-0.412502776762)
        ref[0,1]=(-0.0506558104848)/(o+1.)+(-0.799486412331)
        ref[0,2]=(0.00607800584081)/(o+1.)+(-0.0502983114867)
        ref[0,3]=(1.24827586528)/(o+1.)+(-0.498768742563)
        ref[0,4]=(0.830695699605)/(o+1.)+(-0.431046178271)
        ref[1,0]=(0.484831950292)/(o+1.)+(0.259282658004)
        ref[1,1]=(-0.929740174666)/(o+1.)+(0.0604564342974)
        ref[1,2]=(-0.252324642788)/(o+1.)+(0.84162375133)
        ref[1,3]=(0.576220376373)/(o+1.)+(0.220056386777)
        ref[1,4]=(0.851839623577)/(o+1.)+(0.460491044586)
        ref[2,0]=(-1.46262193255)/(o+1.)+(0.686820632038)
        ref[2,1]=(0.206023962501)/(o+1.)+(0.180044874804)
        ref[2,2]=(-0.0644514797559)/(o+1.)+(0.321913987798)
        ref[2,3]=(0.905394920997)/(o+1.)+(-0.201597088233)
        ref[2,4]=(0.563556812695)/(o+1.)+(0.643101235506)
        ref[3,0]=(-1.09528545326)/(o+1.)+(-0.282707767277)
        ref[3,1]=(-0.505926554117)/(o+1.)+(0.915745060677)
        ref[3,2]=(-0.343737925345)/(o+1.)+(0.136097180846)
        ref[3,3]=(-0.0734656391194)/(o+1.)+(0.468814594255)
        ref[3,4]=(1.32132343428)/(o+1.)+(0.247523633976)
      else:
        arg[0,0]=(-0.512698606017)*x[0]**o+(-0.156853163498)*x[0]+(0.247997355633)*x[1]**o+(-0.363003068904)*x[1]+(-0.940097989108)*x[2]**o+(-0.661063850288)*x[2]
        arg[0,1]=(-0.886390181577)*x[0]**o+(0.724049511142)*x[0]+(0.260684592163)*x[1]**o+(0.265648885034)*x[1]+(0.123084281693)*x[2]**o+(0.181492979151)*x[2]
        arg[0,2]=(-0.570543998851)*x[0]**o+(0.0541041165388)*x[0]+(0.630994990544)*x[1]**o+(-0.89311079559)*x[1]+(0.750714073617)*x[2]**o+(-0.0684796380527)*x[2]
        arg[0,3]=(0.605486031268)*x[0]**o+(0.0387617487672)*x[0]+(-0.365027101406)*x[1]**o+(0.323275180514)*x[1]+(-0.763142660003)*x[2]**o+(-0.331069480921)*x[2]
        arg[0,4]=(-0.0779035848472)*x[0]**o+(-0.0234904624931)*x[0]+(0.591635786678)*x[1]**o+(-0.793680126332)*x[1]+(-0.874742296757)*x[2]**o+(-0.184528885157)*x[2]
        arg[1,0]=(-0.62480972544)*x[0]**o+(-0.99242215547)*x[0]+(0.634250679117)*x[1]**o+(-0.785448642692)*x[1]+(-0.013335654402)*x[2]**o+(0.52112750417)*x[2]
        arg[1,1]=(0.611541580179)*x[0]**o+(-0.203144044802)*x[0]+(-0.411277184995)*x[1]**o+(0.373487145939)*x[1]+(-0.679339345026)*x[2]**o+(-0.632413536218)*x[2]
        arg[1,2]=(-0.258927016205)*x[0]**o+(0.0242628959926)*x[0]+(-0.596909760383)*x[1]**o+(-0.295787354787)*x[1]+(-0.337930022823)*x[2]**o+(-0.825085100446)*x[2]
        arg[1,3]=(-0.0976663781507)*x[0]**o+(0.716600132838)*x[0]+(-0.682155341898)*x[1]**o+(-0.154484864167)*x[1]+(0.754686874908)*x[2]**o+(-0.102453470268)*x[2]
        arg[1,4]=(0.268714339964)*x[0]**o+(0.974234398844)*x[0]+(-0.370113158384)*x[1]**o+(0.0745228956861)*x[1]+(-0.0674370676047)*x[2]**o+(0.132605448278)*x[2]
        arg[2,0]=(-0.00750893262562)*x[0]**o+(0.507641887134)*x[0]+(0.638483932117)*x[1]**o+(-0.940295540912)*x[1]+(-0.201492519698)*x[2]**o+(0.504878595818)*x[2]
        arg[2,1]=(-0.0283522980593)*x[0]**o+(-0.76577346605)*x[0]+(0.0863472830598)*x[1]**o+(-0.904447859187)*x[1]+(-0.203390005492)*x[2]**o+(-0.741629136645)*x[2]
        arg[2,2]=(-0.0196123831667)*x[0]**o+(-0.320721372799)*x[0]+(0.0670421287749)*x[1]**o+(-0.415029682416)*x[1]+(0.450269143475)*x[2]**o+(-0.735201210502)*x[2]
        arg[2,3]=(-0.621537330758)*x[0]**o+(0.21573881897)*x[0]+(0.0364275955835)*x[1]**o+(0.873908952751)*x[1]+(-0.197046790303)*x[2]**o+(0.655654722165)*x[2]
        arg[2,4]=(0.432531007791)*x[0]**o+(-0.65605251166)*x[0]+(0.728245204833)*x[1]**o+(-0.769998936989)*x[1]+(-0.797879623133)*x[2]**o+(0.36575554332)*x[2]
        arg[3,0]=(0.512219734043)*x[0]**o+(-0.177942735603)*x[0]+(-0.307293350237)*x[1]**o+(-0.780181541911)*x[1]+(-0.510201727643)*x[2]**o+(-0.567980715525)*x[2]
        arg[3,1]=(-0.405123989366)*x[0]**o+(-0.133316566583)*x[0]+(0.552868689995)*x[1]**o+(-0.921149434884)*x[1]+(0.510851078054)*x[2]**o+(0.7207533866)*x[2]
        arg[3,2]=(0.150011788901)*x[0]**o+(-0.55947555552)*x[0]+(-0.353366762442)*x[1]**o+(0.204954197301)*x[1]+(-0.48254632377)*x[2]**o+(-0.272738900127)*x[2]
        arg[3,3]=(-0.690981207461)*x[0]**o+(0.38683432128)*x[0]+(0.655731808738)*x[1]**o+(0.162269870301)*x[1]+(-0.373468659959)*x[2]**o+(-0.841431521009)*x[2]
        arg[3,4]=(0.156455675319)*x[0]**o+(0.501690084272)*x[0]+(-0.339255791314)*x[1]**o+(0.0832260784264)*x[1]+(0.394137401365)*x[2]**o+(-0.92403906326)*x[2]
        ref[0,0]=(-1.20479923949)/(o+1.)+(-0.590460041345)
        ref[0,1]=(-0.502621307721)/(o+1.)+(0.585595687663)
        ref[0,2]=(0.811165065311)/(o+1.)+(-0.453743158552)
        ref[0,3]=(-0.522683730141)/(o+1.)+(0.0154837241797)
        ref[0,4]=(-0.361010094926)/(o+1.)+(-0.500849736991)
        ref[1,0]=(-0.00389470072479)/(o+1.)+(-0.628371646996)
        ref[1,1]=(-0.479074949842)/(o+1.)+(-0.23103521754)
        ref[1,2]=(-1.19376679941)/(o+1.)+(-0.54830477962)
        ref[1,3]=(-0.0251348451402)/(o+1.)+(0.229830899202)
        ref[1,4]=(-0.168835886025)/(o+1.)+(0.590681371404)
        ref[2,0]=(0.429482479793)/(o+1.)+(0.0361124710203)
        ref[2,1]=(-0.145395020492)/(o+1.)+(-1.20592523094)
        ref[2,2]=(0.497698889084)/(o+1.)+(-0.735476132859)
        ref[2,3]=(-0.782156525477)/(o+1.)+(0.872651246943)
        ref[2,4]=(0.362896589491)/(o+1.)+(-0.530147952664)
        ref[3,0]=(-0.305275343837)/(o+1.)+(-0.763052496519)
        ref[3,1]=(0.658595778683)/(o+1.)+(-0.166856307434)
        ref[3,2]=(-0.685901297311)/(o+1.)+(-0.313630129173)
        ref[3,3]=(-0.408718058683)/(o+1.)+(-0.146163664714)
        ref[3,4]=(0.21133728537)/(o+1.)+(-0.169561450281)
      res=integrate(arg)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunction_fromData_ReducedFunction_rank3(self):
      """
      tests integral of rank 3 Data on the ReducedFunction

      assumptions: ReducedFunction(self.domain) exists
                   self.domain supports integral on ReducedFunction
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=ReducedFunction(self.domain)
      w=ReducedFunction(self.domain)
      x=w.getX()
      arg=Data(0,(6, 2, 2),w)
      ref=numpy.zeros((6, 2, 2),numpy.float_)
      if dim==2:
        arg[0,0,0]=(-0.165423911829)*x[0]**o+(0.912542062064)*x[0]+(0.515840721225)*x[1]**o+(-0.855922021017)*x[1]
        arg[0,0,1]=(0.591873802546)*x[0]**o+(-0.039385167187)*x[0]+(0.0261484884274)*x[1]**o+(0.227950240434)*x[1]
        arg[0,1,0]=(-0.825833712046)*x[0]**o+(0.989606797765)*x[0]+(0.975260558791)*x[1]**o+(0.297963603984)*x[1]
        arg[0,1,1]=(-0.658054502482)*x[0]**o+(-0.865505090963)*x[0]+(0.817184084453)*x[1]**o+(-0.409588098899)*x[1]
        arg[1,0,0]=(0.538456987198)*x[0]**o+(-0.0768787509391)*x[0]+(0.424620431835)*x[1]**o+(0.799917948831)*x[1]
        arg[1,0,1]=(0.125945558199)*x[0]**o+(-0.48320049996)*x[0]+(0.105474962327)*x[1]**o+(-0.090713555582)*x[1]
        arg[1,1,0]=(0.835213375418)*x[0]**o+(-0.513407038769)*x[0]+(-0.342890811735)*x[1]**o+(-0.285057736244)*x[1]
        arg[1,1,1]=(0.762782085749)*x[0]**o+(-0.308457087352)*x[0]+(0.23756726753)*x[1]**o+(-0.909365956504)*x[1]
        arg[2,0,0]=(-0.76467980678)*x[0]**o+(-0.265552922006)*x[0]+(-0.261862355539)*x[1]**o+(-0.399407404618)*x[1]
        arg[2,0,1]=(0.79401193617)*x[0]**o+(-0.0300461495216)*x[0]+(-0.978834133508)*x[1]**o+(-0.984845382599)*x[1]
        arg[2,1,0]=(0.898033018534)*x[0]**o+(-0.184513430159)*x[0]+(0.0430634098526)*x[1]**o+(0.62213313954)*x[1]
        arg[2,1,1]=(0.126926439246)*x[0]**o+(0.671259428917)*x[0]+(-0.396062855417)*x[1]**o+(0.0461825691916)*x[1]
        arg[3,0,0]=(0.185756594279)*x[0]**o+(0.570946291973)*x[0]+(0.0673731944956)*x[1]**o+(0.367766610292)*x[1]
        arg[3,0,1]=(0.021269224946)*x[0]**o+(-0.463663279063)*x[0]+(-0.687930988068)*x[1]**o+(0.184714204849)*x[1]
        arg[3,1,0]=(0.754288639303)*x[0]**o+(-0.94612827517)*x[0]+(0.802463196709)*x[1]**o+(0.202355582077)*x[1]
        arg[3,1,1]=(0.840002857786)*x[0]**o+(-0.692207739447)*x[0]+(0.486569391181)*x[1]**o+(-0.713960793192)*x[1]
        arg[4,0,0]=(-0.205794539655)*x[0]**o+(-0.930458992533)*x[0]+(-0.62580205172)*x[1]**o+(-0.884738110673)*x[1]
        arg[4,0,1]=(-0.91433612349)*x[0]**o+(0.694961527804)*x[0]+(-0.510941334746)*x[1]**o+(-0.510725271706)*x[1]
        arg[4,1,0]=(0.852785499366)*x[0]**o+(0.870826628851)*x[0]+(-0.577529754198)*x[1]**o+(-0.583793684857)*x[1]
        arg[4,1,1]=(-0.604869226881)*x[0]**o+(0.482463572402)*x[0]+(0.829654316885)*x[1]**o+(0.597543822455)*x[1]
        arg[5,0,0]=(0.413459469749)*x[0]**o+(0.865143264083)*x[0]+(0.986197644249)*x[1]**o+(-0.2511021346)*x[1]
        arg[5,0,1]=(0.811572168213)*x[0]**o+(-0.110121334437)*x[0]+(-0.985492232242)*x[1]**o+(-0.913436084156)*x[1]
        arg[5,1,0]=(-0.660869051422)*x[0]**o+(-0.743235620427)*x[0]+(0.205393778238)*x[1]**o+(0.296069404687)*x[1]
        arg[5,1,1]=(-0.13285011267)*x[0]**o+(0.659400366834)*x[0]+(-0.889984368942)*x[1]**o+(0.584893563559)*x[1]
        ref[0,0,0]=(0.350416809396)/(o+1.)+(0.0283100205234)
        ref[0,0,1]=(0.618022290973)/(o+1.)+(0.0942825366236)
        ref[0,1,0]=(0.149426846745)/(o+1.)+(0.643785200874)
        ref[0,1,1]=(0.159129581971)/(o+1.)+(-0.637546594931)
        ref[1,0,0]=(0.963077419032)/(o+1.)+(0.361519598946)
        ref[1,0,1]=(0.231420520526)/(o+1.)+(-0.286957027771)
        ref[1,1,0]=(0.492322563683)/(o+1.)+(-0.399232387506)
        ref[1,1,1]=(1.00034935328)/(o+1.)+(-0.608911521928)
        ref[2,0,0]=(-1.02654216232)/(o+1.)+(-0.332480163312)
        ref[2,0,1]=(-0.184822197338)/(o+1.)+(-0.507445766061)
        ref[2,1,0]=(0.941096428387)/(o+1.)+(0.21880985469)
        ref[2,1,1]=(-0.269136416172)/(o+1.)+(0.358720999054)
        ref[3,0,0]=(0.253129788774)/(o+1.)+(0.469356451133)
        ref[3,0,1]=(-0.666661763122)/(o+1.)+(-0.139474537107)
        ref[3,1,0]=(1.55675183601)/(o+1.)+(-0.371886346546)
        ref[3,1,1]=(1.32657224897)/(o+1.)+(-0.703084266319)
        ref[4,0,0]=(-0.831596591375)/(o+1.)+(-0.907598551603)
        ref[4,0,1]=(-1.42527745824)/(o+1.)+(0.0921181280487)
        ref[4,1,0]=(0.275255745168)/(o+1.)+(0.143516471997)
        ref[4,1,1]=(0.224785090004)/(o+1.)+(0.540003697429)
        ref[5,0,0]=(1.399657114)/(o+1.)+(0.307020564742)
        ref[5,0,1]=(-0.173920064028)/(o+1.)+(-0.511778709296)
        ref[5,1,0]=(-0.455475273184)/(o+1.)+(-0.22358310787)
        ref[5,1,1]=(-1.02283448161)/(o+1.)+(0.622146965196)
      else:
        arg[0,0,0]=(-0.564478749422)*x[0]**o+(-0.0357223504179)*x[0]+(0.396603298865)*x[1]**o+(-0.461941327703)*x[1]+(-0.667580657928)*x[2]**o+(-0.589957067263)*x[2]
        arg[0,0,1]=(0.930616008951)*x[0]**o+(-0.764067565635)*x[0]+(-0.71350617205)*x[1]**o+(-0.530942701898)*x[1]+(-0.103797356434)*x[2]**o+(0.410704036673)*x[2]
        arg[0,1,0]=(-0.59127392299)*x[0]**o+(0.452490122131)*x[0]+(0.714793625631)*x[1]**o+(-0.933114051133)*x[1]+(-0.693437840325)*x[2]**o+(0.979267231764)*x[2]
        arg[0,1,1]=(0.179682332267)*x[0]**o+(-0.419960476425)*x[0]+(-0.10986598341)*x[1]**o+(0.912986428497)*x[1]+(0.0886946782573)*x[2]**o+(0.516065958873)*x[2]
        arg[1,0,0]=(0.524446863481)*x[0]**o+(-0.439668193423)*x[0]+(0.681259811488)*x[1]**o+(0.439275262258)*x[1]+(0.84573881706)*x[2]**o+(0.377640553175)*x[2]
        arg[1,0,1]=(-0.007464905203)*x[0]**o+(0.234727832471)*x[0]+(0.0428956356618)*x[1]**o+(0.195772930172)*x[1]+(0.457790934614)*x[2]**o+(0.445309166495)*x[2]
        arg[1,1,0]=(0.821283375827)*x[0]**o+(0.212480005798)*x[0]+(0.148817990961)*x[1]**o+(0.119716150877)*x[1]+(0.658959126865)*x[2]**o+(-0.346216916131)*x[2]
        arg[1,1,1]=(-0.0587222422894)*x[0]**o+(-0.325321661973)*x[0]+(0.149293818689)*x[1]**o+(-0.632073115037)*x[1]+(0.018604271874)*x[2]**o+(-0.412288240166)*x[2]
        arg[2,0,0]=(-0.22588082169)*x[0]**o+(0.197780461217)*x[0]+(0.387336989213)*x[1]**o+(-0.680021661016)*x[1]+(-0.903782254346)*x[2]**o+(-0.273268109853)*x[2]
        arg[2,0,1]=(0.415876394297)*x[0]**o+(0.199596795904)*x[0]+(-0.445817134682)*x[1]**o+(0.672539590684)*x[1]+(0.907825473964)*x[2]**o+(0.459560469691)*x[2]
        arg[2,1,0]=(-0.052146887134)*x[0]**o+(-0.229450574738)*x[0]+(0.0929720151604)*x[1]**o+(-0.893861034491)*x[1]+(-0.294030328982)*x[2]**o+(-0.193256779109)*x[2]
        arg[2,1,1]=(0.361171422066)*x[0]**o+(-0.212022056495)*x[0]+(0.0672221746963)*x[1]**o+(0.57506871515)*x[1]+(0.654894469509)*x[2]**o+(0.0749777260533)*x[2]
        arg[3,0,0]=(0.505355253404)*x[0]**o+(-0.0753563586359)*x[0]+(0.544299279727)*x[1]**o+(-0.781954131328)*x[1]+(0.534390713891)*x[2]**o+(-0.719289344378)*x[2]
        arg[3,0,1]=(-0.589309652827)*x[0]**o+(-0.0557421151178)*x[0]+(-0.275311076837)*x[1]**o+(0.998180518149)*x[1]+(-0.844823856433)*x[2]**o+(-0.988843805162)*x[2]
        arg[3,1,0]=(-0.411075104301)*x[0]**o+(-0.405896828894)*x[0]+(0.242301856034)*x[1]**o+(-0.760648969984)*x[1]+(0.0940367373673)*x[2]**o+(-0.121420230853)*x[2]
        arg[3,1,1]=(-0.201426146967)*x[0]**o+(0.722847134095)*x[0]+(-0.00743580369433)*x[1]**o+(-0.593158423904)*x[1]+(0.630575332064)*x[2]**o+(-0.309839791343)*x[2]
        arg[4,0,0]=(0.370057366738)*x[0]**o+(0.35683022394)*x[0]+(0.886304520049)*x[1]**o+(0.834488899465)*x[1]+(0.27408661832)*x[2]**o+(0.325007409933)*x[2]
        arg[4,0,1]=(0.386153377977)*x[0]**o+(-0.889700309398)*x[0]+(-0.60695970137)*x[1]**o+(0.701208035732)*x[1]+(0.435164510485)*x[2]**o+(0.40223148148)*x[2]
        arg[4,1,0]=(-0.227256041906)*x[0]**o+(-0.0406115125763)*x[0]+(-0.0298974314183)*x[1]**o+(0.887167464033)*x[1]+(0.1004751563)*x[2]**o+(0.826924545719)*x[2]
        arg[4,1,1]=(0.733575640137)*x[0]**o+(-0.376269474248)*x[0]+(-0.841290711112)*x[1]**o+(-0.0633013917179)*x[1]+(0.335730636406)*x[2]**o+(0.913607658221)*x[2]
        arg[5,0,0]=(-0.896019254334)*x[0]**o+(0.465470987762)*x[0]+(0.338543664838)*x[1]**o+(0.182102591196)*x[1]+(0.476317455739)*x[2]**o+(0.191367623163)*x[2]
        arg[5,0,1]=(0.828458939676)*x[0]**o+(-0.37853033744)*x[0]+(-0.540809170906)*x[1]**o+(0.367869598038)*x[1]+(0.321196361443)*x[2]**o+(0.548981819769)*x[2]
        arg[5,1,0]=(-0.0805871335579)*x[0]**o+(-0.554679375424)*x[0]+(-0.259438414137)*x[1]**o+(0.72362534417)*x[1]+(-0.838855028889)*x[2]**o+(0.929802758377)*x[2]
        arg[5,1,1]=(-0.947910259125)*x[0]**o+(0.654012824998)*x[0]+(0.425232602078)*x[1]**o+(-0.649345176159)*x[1]+(-0.946280438348)*x[2]**o+(-0.429547284912)*x[2]
        ref[0,0,0]=(-0.835456108484)/(o+1.)+(-0.543810372692)
        ref[0,0,1]=(0.113312480467)/(o+1.)+(-0.44215311543)
        ref[0,1,0]=(-0.569918137684)/(o+1.)+(0.249321651381)
        ref[0,1,1]=(0.158511027114)/(o+1.)+(0.504545955472)
        ref[1,0,0]=(2.05144549203)/(o+1.)+(0.188623811006)
        ref[1,0,1]=(0.493221665073)/(o+1.)+(0.437904964569)
        ref[1,1,0]=(1.62906049365)/(o+1.)+(-0.00701037972825)
        ref[1,1,1]=(0.109175848273)/(o+1.)+(-0.684841508588)
        ref[2,0,0]=(-0.742326086822)/(o+1.)+(-0.377754654826)
        ref[2,0,1]=(0.877884733578)/(o+1.)+(0.665848428139)
        ref[2,1,0]=(-0.253205200956)/(o+1.)+(-0.658284194169)
        ref[2,1,1]=(1.08328806627)/(o+1.)+(0.219012192354)
        ref[3,0,0]=(1.58404524702)/(o+1.)+(-0.788299917171)
        ref[3,0,1]=(-1.7094445861)/(o+1.)+(-0.0232027010654)
        ref[3,1,0]=(-0.0747365109002)/(o+1.)+(-0.643983014866)
        ref[3,1,1]=(0.421713381403)/(o+1.)+(-0.0900755405762)
        ref[4,0,0]=(1.53044850511)/(o+1.)+(0.758163266669)
        ref[4,0,1]=(0.214358187092)/(o+1.)+(0.106869603907)
        ref[4,1,0]=(-0.156678317024)/(o+1.)+(0.836740248588)
        ref[4,1,1]=(0.228015565432)/(o+1.)+(0.237018396127)
        ref[5,0,0]=(-0.0811581337568)/(o+1.)+(0.419470601061)
        ref[5,0,1]=(0.608846130213)/(o+1.)+(0.269160540183)
        ref[5,1,0]=(-1.17888057658)/(o+1.)+(0.549374363561)
        ref[5,1,1]=(-1.4689580954)/(o+1.)+(-0.212439818036)
      res=integrate(arg)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunction_fromData_ReducedFunction_rank4(self):
      """
      tests integral of rank 4 Data on the ReducedFunction

      assumptions: ReducedFunction(self.domain) exists
                   self.domain supports integral on ReducedFunction
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=ReducedFunction(self.domain)
      w=ReducedFunction(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5, 3, 2),w)
      ref=numpy.zeros((4, 5, 3, 2),numpy.float_)
      if dim==2:
        arg[0,0,0,0]=(-0.216014106349)*x[0]**o+(0.98826230471)*x[0]+(0.522566177319)*x[1]**o+(-0.672520997452)*x[1]
        arg[0,0,0,1]=(0.368024408084)*x[0]**o+(0.047972971332)*x[0]+(-0.307374327321)*x[1]**o+(0.0745169735457)*x[1]
        arg[0,0,1,0]=(0.438735045452)*x[0]**o+(0.76355958752)*x[0]+(0.200829025437)*x[1]**o+(0.56912407759)*x[1]
        arg[0,0,1,1]=(-0.0328769393712)*x[0]**o+(-0.39701363611)*x[0]+(0.704125532519)*x[1]**o+(-0.6602435084)*x[1]
        arg[0,0,2,0]=(-0.547717992717)*x[0]**o+(-0.844256763776)*x[0]+(0.621300470147)*x[1]**o+(-0.105216181226)*x[1]
        arg[0,0,2,1]=(-0.723474063234)*x[0]**o+(-0.980738972901)*x[0]+(-0.183578659566)*x[1]**o+(0.586038450047)*x[1]
        arg[0,1,0,0]=(0.505379907001)*x[0]**o+(-0.470514711277)*x[0]+(0.799116811747)*x[1]**o+(0.682760846386)*x[1]
        arg[0,1,0,1]=(0.353992609198)*x[0]**o+(-0.848107655585)*x[0]+(-0.774065459855)*x[1]**o+(0.0323137462292)*x[1]
        arg[0,1,1,0]=(0.799068805694)*x[0]**o+(-0.979761821209)*x[0]+(0.642674987867)*x[1]**o+(-0.639029952044)*x[1]
        arg[0,1,1,1]=(0.542556683327)*x[0]**o+(0.876753086356)*x[0]+(-0.387121297643)*x[1]**o+(0.377543179628)*x[1]
        arg[0,1,2,0]=(0.424672899596)*x[0]**o+(-0.134184751569)*x[0]+(-0.561877304114)*x[1]**o+(0.549933699347)*x[1]
        arg[0,1,2,1]=(-0.126173294501)*x[0]**o+(-0.338964866844)*x[0]+(-0.163536115759)*x[1]**o+(0.747551558186)*x[1]
        arg[0,2,0,0]=(0.304334183452)*x[0]**o+(0.927809971187)*x[0]+(0.961504942988)*x[1]**o+(0.0698014252396)*x[1]
        arg[0,2,0,1]=(0.488820088455)*x[0]**o+(0.890735247208)*x[0]+(0.686782901398)*x[1]**o+(0.972954994564)*x[1]
        arg[0,2,1,0]=(-0.737577792766)*x[0]**o+(-0.351698913646)*x[0]+(0.392591489865)*x[1]**o+(0.79238240143)*x[1]
        arg[0,2,1,1]=(-0.997254027544)*x[0]**o+(0.579800166465)*x[0]+(-0.95000690643)*x[1]**o+(-0.166463083476)*x[1]
        arg[0,2,2,0]=(-0.999703749644)*x[0]**o+(0.0415211466468)*x[0]+(0.0816160245139)*x[1]**o+(-0.0921114587195)*x[1]
        arg[0,2,2,1]=(-0.172818721386)*x[0]**o+(0.10475692446)*x[0]+(-0.855815339933)*x[1]**o+(-0.50842580034)*x[1]
        arg[0,3,0,0]=(-0.154563121513)*x[0]**o+(-0.0956305513369)*x[0]+(-0.893783521038)*x[1]**o+(0.10772474638)*x[1]
        arg[0,3,0,1]=(0.365331678349)*x[0]**o+(0.403124820832)*x[0]+(-0.0750500521493)*x[1]**o+(0.62559762808)*x[1]
        arg[0,3,1,0]=(-0.42235875521)*x[0]**o+(0.831133238533)*x[0]+(-0.893322944982)*x[1]**o+(0.936074096319)*x[1]
        arg[0,3,1,1]=(0.587382334689)*x[0]**o+(0.996190710631)*x[0]+(-0.796032791654)*x[1]**o+(0.783491453801)*x[1]
        arg[0,3,2,0]=(-0.369527027508)*x[0]**o+(-0.501417552089)*x[0]+(-0.372105439045)*x[1]**o+(0.653283000152)*x[1]
        arg[0,3,2,1]=(0.914893750314)*x[0]**o+(-0.769183323793)*x[0]+(0.571638486717)*x[1]**o+(-0.939784652259)*x[1]
        arg[0,4,0,0]=(0.134859060251)*x[0]**o+(-0.297363165275)*x[0]+(0.198957028974)*x[1]**o+(-0.828531725127)*x[1]
        arg[0,4,0,1]=(-0.371375370217)*x[0]**o+(-0.0422058718402)*x[0]+(0.502010302278)*x[1]**o+(0.442051802597)*x[1]
        arg[0,4,1,0]=(0.513346571173)*x[0]**o+(0.182640867356)*x[0]+(0.76200283664)*x[1]**o+(0.134640432361)*x[1]
        arg[0,4,1,1]=(-0.815543712756)*x[0]**o+(0.0306226823724)*x[0]+(-0.0211231926242)*x[1]**o+(-0.718800228294)*x[1]
        arg[0,4,2,0]=(-0.980596107555)*x[0]**o+(-0.714237107759)*x[0]+(0.547986215851)*x[1]**o+(0.180131474534)*x[1]
        arg[0,4,2,1]=(-0.460780402239)*x[0]**o+(-0.483771113997)*x[0]+(-0.017140263255)*x[1]**o+(-0.127947184627)*x[1]
        arg[1,0,0,0]=(-0.0187443155707)*x[0]**o+(-0.490914415873)*x[0]+(-0.720022496093)*x[1]**o+(0.0265633770733)*x[1]
        arg[1,0,0,1]=(-0.634855765904)*x[0]**o+(0.345031954366)*x[0]+(-0.0532392363316)*x[1]**o+(0.983295524564)*x[1]
        arg[1,0,1,0]=(0.262182040476)*x[0]**o+(-0.883857758733)*x[0]+(0.880670543535)*x[1]**o+(0.193477695378)*x[1]
        arg[1,0,1,1]=(-0.0825976168371)*x[0]**o+(0.161838720337)*x[0]+(-0.0110587499089)*x[1]**o+(-0.634455402489)*x[1]
        arg[1,0,2,0]=(0.284221958342)*x[0]**o+(-0.948548729029)*x[0]+(0.430914344458)*x[1]**o+(-0.18563638241)*x[1]
        arg[1,0,2,1]=(-0.459558022898)*x[0]**o+(0.431907572866)*x[0]+(-0.0597669070218)*x[1]**o+(0.0694271372225)*x[1]
        arg[1,1,0,0]=(0.54071078447)*x[0]**o+(-0.244570994356)*x[0]+(0.00506219762373)*x[1]**o+(-0.0816378019186)*x[1]
        arg[1,1,0,1]=(-0.100752221379)*x[0]**o+(0.535738345566)*x[0]+(0.730325284553)*x[1]**o+(-0.885044748717)*x[1]
        arg[1,1,1,0]=(-0.0422460353138)*x[0]**o+(-0.986416505995)*x[0]+(-0.111766776997)*x[1]**o+(-0.842179226197)*x[1]
        arg[1,1,1,1]=(0.0690281593169)*x[0]**o+(-0.885054188663)*x[0]+(-0.680349314907)*x[1]**o+(0.00277190934345)*x[1]
        arg[1,1,2,0]=(0.502632146841)*x[0]**o+(-0.197332151163)*x[0]+(-0.0942257361149)*x[1]**o+(0.045171692749)*x[1]
        arg[1,1,2,1]=(0.427910466818)*x[0]**o+(-0.61355947216)*x[0]+(0.465128040517)*x[1]**o+(0.0874536555354)*x[1]
        arg[1,2,0,0]=(0.943819415417)*x[0]**o+(0.884020001869)*x[0]+(-0.769166027477)*x[1]**o+(-0.10032307068)*x[1]
        arg[1,2,0,1]=(0.0670224828564)*x[0]**o+(-0.645424481071)*x[0]+(-0.756361652678)*x[1]**o+(-0.274687916196)*x[1]
        arg[1,2,1,0]=(0.397836742229)*x[0]**o+(-0.708791058745)*x[0]+(0.920869587663)*x[1]**o+(-0.250037069918)*x[1]
        arg[1,2,1,1]=(0.722570486719)*x[0]**o+(-0.358082774936)*x[0]+(0.786344673969)*x[1]**o+(0.310591893637)*x[1]
        arg[1,2,2,0]=(-0.379404573553)*x[0]**o+(-0.969753501645)*x[0]+(-0.0745714915337)*x[1]**o+(-0.262943796911)*x[1]
        arg[1,2,2,1]=(0.817999459632)*x[0]**o+(0.731516351792)*x[0]+(-0.531411613325)*x[1]**o+(-0.824858061154)*x[1]
        arg[1,3,0,0]=(-0.206295444021)*x[0]**o+(-0.781973944918)*x[0]+(-0.173845545382)*x[1]**o+(-0.360122364573)*x[1]
        arg[1,3,0,1]=(-0.0604068005719)*x[0]**o+(0.238691416507)*x[0]+(-0.82262982031)*x[1]**o+(-0.893751755968)*x[1]
        arg[1,3,1,0]=(-0.686409669386)*x[0]**o+(0.711830599399)*x[0]+(0.0839688142166)*x[1]**o+(-0.99463449684)*x[1]
        arg[1,3,1,1]=(-0.0864745975077)*x[0]**o+(0.00068907895313)*x[0]+(-0.766810513534)*x[1]**o+(0.685236202954)*x[1]
        arg[1,3,2,0]=(0.0909338812695)*x[0]**o+(-0.709595654169)*x[0]+(-0.483428504499)*x[1]**o+(-0.0707341191006)*x[1]
        arg[1,3,2,1]=(-0.38342983855)*x[0]**o+(-0.0407824575865)*x[0]+(-0.201329040891)*x[1]**o+(0.0240007937141)*x[1]
        arg[1,4,0,0]=(0.512465803173)*x[0]**o+(-0.371898603608)*x[0]+(-0.79453595861)*x[1]**o+(0.340546986375)*x[1]
        arg[1,4,0,1]=(0.742383198442)*x[0]**o+(-0.603548189912)*x[0]+(-0.270277825546)*x[1]**o+(-0.525100913894)*x[1]
        arg[1,4,1,0]=(-0.693074384029)*x[0]**o+(-0.825037404844)*x[0]+(-0.626116177978)*x[1]**o+(0.628159342479)*x[1]
        arg[1,4,1,1]=(-0.595844943255)*x[0]**o+(0.250758194807)*x[0]+(0.504390052142)*x[1]**o+(-0.124588859702)*x[1]
        arg[1,4,2,0]=(-0.651727490997)*x[0]**o+(0.411138504151)*x[0]+(0.323769160331)*x[1]**o+(0.904379126671)*x[1]
        arg[1,4,2,1]=(-0.71307850178)*x[0]**o+(0.734964608119)*x[0]+(0.348693698506)*x[1]**o+(0.272270639772)*x[1]
        arg[2,0,0,0]=(0.558236567771)*x[0]**o+(0.516985445776)*x[0]+(0.0278915137861)*x[1]**o+(-0.268619977833)*x[1]
        arg[2,0,0,1]=(-0.871181709658)*x[0]**o+(-0.132251987744)*x[0]+(-0.215497073461)*x[1]**o+(0.508008002082)*x[1]
        arg[2,0,1,0]=(0.880795999572)*x[0]**o+(-0.231151166496)*x[0]+(-0.698846681617)*x[1]**o+(0.914483413825)*x[1]
        arg[2,0,1,1]=(0.684792911211)*x[0]**o+(-0.801408782989)*x[0]+(0.689713232168)*x[1]**o+(0.142847173797)*x[1]
        arg[2,0,2,0]=(-0.912566319945)*x[0]**o+(0.811138574657)*x[0]+(-0.884250879071)*x[1]**o+(-0.177206225477)*x[1]
        arg[2,0,2,1]=(-0.898648046678)*x[0]**o+(0.581554373688)*x[0]+(0.567642878443)*x[1]**o+(0.636312832751)*x[1]
        arg[2,1,0,0]=(-0.788853365648)*x[0]**o+(-0.135329327941)*x[0]+(0.180850147594)*x[1]**o+(0.742619029335)*x[1]
        arg[2,1,0,1]=(-0.824539980645)*x[0]**o+(0.857173393615)*x[0]+(0.850455264916)*x[1]**o+(0.975422322464)*x[1]
        arg[2,1,1,0]=(0.462293517317)*x[0]**o+(-0.173835273794)*x[0]+(-0.800072514276)*x[1]**o+(0.140214114627)*x[1]
        arg[2,1,1,1]=(-0.187890814293)*x[0]**o+(0.534472364495)*x[0]+(-0.799640665571)*x[1]**o+(0.301287955302)*x[1]
        arg[2,1,2,0]=(0.278901492946)*x[0]**o+(0.620544326886)*x[0]+(-0.705702443752)*x[1]**o+(-0.489690636328)*x[1]
        arg[2,1,2,1]=(-0.926772900235)*x[0]**o+(0.134654742845)*x[0]+(0.356561975337)*x[1]**o+(-0.919999359857)*x[1]
        arg[2,2,0,0]=(-0.675956301416)*x[0]**o+(0.149582589205)*x[0]+(0.186440827385)*x[1]**o+(0.88381610446)*x[1]
        arg[2,2,0,1]=(-0.697267962915)*x[0]**o+(0.991366596424)*x[0]+(0.310290094618)*x[1]**o+(-0.881771505066)*x[1]
        arg[2,2,1,0]=(0.506027358474)*x[0]**o+(0.294136137354)*x[0]+(0.732003937836)*x[1]**o+(0.633469131504)*x[1]
        arg[2,2,1,1]=(-0.427146039728)*x[0]**o+(-0.975737063903)*x[0]+(-0.934913653772)*x[1]**o+(-0.662568325005)*x[1]
        arg[2,2,2,0]=(0.477849916621)*x[0]**o+(-0.798883883367)*x[0]+(-0.154217684441)*x[1]**o+(-0.175817547912)*x[1]
        arg[2,2,2,1]=(0.935192620387)*x[0]**o+(-0.193802588249)*x[0]+(0.406181949213)*x[1]**o+(0.20422475396)*x[1]
        arg[2,3,0,0]=(0.965944278856)*x[0]**o+(-0.736175411725)*x[0]+(0.92404347386)*x[1]**o+(0.273152788392)*x[1]
        arg[2,3,0,1]=(-0.606944412413)*x[0]**o+(0.692555413476)*x[0]+(0.812085676275)*x[1]**o+(-0.0809920218886)*x[1]
        arg[2,3,1,0]=(-0.314634777406)*x[0]**o+(0.750181618592)*x[0]+(0.0554739779642)*x[1]**o+(-0.333788326726)*x[1]
        arg[2,3,1,1]=(0.85610815703)*x[0]**o+(0.910096005524)*x[0]+(-0.227092814843)*x[1]**o+(-0.0849532598914)*x[1]
        arg[2,3,2,0]=(0.735064128504)*x[0]**o+(-0.253919528943)*x[0]+(0.8457463307)*x[1]**o+(0.865727345925)*x[1]
        arg[2,3,2,1]=(0.520607989389)*x[0]**o+(0.569210317105)*x[0]+(0.500780081677)*x[1]**o+(-0.787630914351)*x[1]
        arg[2,4,0,0]=(0.179774694296)*x[0]**o+(-0.890872128607)*x[0]+(-0.185758177247)*x[1]**o+(0.888231734606)*x[1]
        arg[2,4,0,1]=(0.059863462227)*x[0]**o+(-0.0796463123948)*x[0]+(-0.104564520504)*x[1]**o+(-0.336182528617)*x[1]
        arg[2,4,1,0]=(-0.636828275967)*x[0]**o+(0.156894572218)*x[0]+(-0.884260096115)*x[1]**o+(-0.967431438795)*x[1]
        arg[2,4,1,1]=(-0.958168110389)*x[0]**o+(-0.0424772631477)*x[0]+(0.512363473316)*x[1]**o+(-0.716729419668)*x[1]
        arg[2,4,2,0]=(0.19736654467)*x[0]**o+(-0.288723518002)*x[0]+(0.243674007033)*x[1]**o+(0.911210601142)*x[1]
        arg[2,4,2,1]=(-0.847493343055)*x[0]**o+(0.115003265593)*x[0]+(0.360071936714)*x[1]**o+(-0.0265385061141)*x[1]
        arg[3,0,0,0]=(-0.510550590836)*x[0]**o+(-0.266174606631)*x[0]+(0.771580708159)*x[1]**o+(-0.770554374259)*x[1]
        arg[3,0,0,1]=(-0.65795024738)*x[0]**o+(-0.83394628525)*x[0]+(0.812461325452)*x[1]**o+(0.191593733482)*x[1]
        arg[3,0,1,0]=(0.136878718548)*x[0]**o+(-0.000345061692795)*x[0]+(-0.0780280205008)*x[1]**o+(0.173925537373)*x[1]
        arg[3,0,1,1]=(0.584079470149)*x[0]**o+(0.138674493948)*x[0]+(-0.563553909861)*x[1]**o+(-0.254304838008)*x[1]
        arg[3,0,2,0]=(0.326192887554)*x[0]**o+(0.993615980303)*x[0]+(-0.26995279693)*x[1]**o+(0.216112615287)*x[1]
        arg[3,0,2,1]=(0.684786021819)*x[0]**o+(0.158934017292)*x[0]+(-0.457448199191)*x[1]**o+(0.644722228163)*x[1]
        arg[3,1,0,0]=(-0.198252162262)*x[0]**o+(-0.821832378445)*x[0]+(0.831807441916)*x[1]**o+(-0.70643670471)*x[1]
        arg[3,1,0,1]=(0.361698759016)*x[0]**o+(0.556511107101)*x[0]+(0.892219700601)*x[1]**o+(-0.861598520361)*x[1]
        arg[3,1,1,0]=(-0.107946706634)*x[0]**o+(-0.910191934756)*x[0]+(0.502507607871)*x[1]**o+(0.649089810516)*x[1]
        arg[3,1,1,1]=(-0.92959760672)*x[0]**o+(0.950682520961)*x[0]+(0.103321004187)*x[1]**o+(-0.631562667815)*x[1]
        arg[3,1,2,0]=(0.858565242025)*x[0]**o+(-0.106810606075)*x[0]+(0.236377425048)*x[1]**o+(-0.296320858145)*x[1]
        arg[3,1,2,1]=(0.148558873625)*x[0]**o+(0.458488334082)*x[0]+(-0.0172824905523)*x[1]**o+(-0.321212766581)*x[1]
        arg[3,2,0,0]=(-0.239733890336)*x[0]**o+(0.558819103697)*x[0]+(-0.151683129885)*x[1]**o+(0.153995780645)*x[1]
        arg[3,2,0,1]=(-0.949100270937)*x[0]**o+(0.0687171499139)*x[0]+(0.968028021831)*x[1]**o+(-0.935600499345)*x[1]
        arg[3,2,1,0]=(-0.739823767237)*x[0]**o+(0.417825150389)*x[0]+(0.318612170707)*x[1]**o+(0.627557337421)*x[1]
        arg[3,2,1,1]=(-0.965142167324)*x[0]**o+(0.981446854383)*x[0]+(0.266191185951)*x[1]**o+(-0.581303142841)*x[1]
        arg[3,2,2,0]=(-0.00312606993817)*x[0]**o+(-0.191885030828)*x[0]+(-0.342525068561)*x[1]**o+(-0.953710538626)*x[1]
        arg[3,2,2,1]=(-0.689127187476)*x[0]**o+(-0.107190366063)*x[0]+(-0.0556716250594)*x[1]**o+(-0.609462166754)*x[1]
        arg[3,3,0,0]=(-0.765237197853)*x[0]**o+(-0.0707283944117)*x[0]+(0.891789872287)*x[1]**o+(-0.255939791858)*x[1]
        arg[3,3,0,1]=(-0.367658362103)*x[0]**o+(0.27406309422)*x[0]+(0.866288683947)*x[1]**o+(0.403194030287)*x[1]
        arg[3,3,1,0]=(-0.362795000993)*x[0]**o+(-0.498774415158)*x[0]+(0.0726808331583)*x[1]**o+(-0.871066791136)*x[1]
        arg[3,3,1,1]=(-0.177149059814)*x[0]**o+(0.287463753612)*x[0]+(0.101726559941)*x[1]**o+(0.561797204517)*x[1]
        arg[3,3,2,0]=(0.770615121039)*x[0]**o+(-0.503402148251)*x[0]+(0.319753007634)*x[1]**o+(0.586868894261)*x[1]
        arg[3,3,2,1]=(0.924125904724)*x[0]**o+(-0.508988920423)*x[0]+(-0.866752504824)*x[1]**o+(0.518544700652)*x[1]
        arg[3,4,0,0]=(-0.973660291124)*x[0]**o+(0.746869369283)*x[0]+(-0.634097171844)*x[1]**o+(0.723638944736)*x[1]
        arg[3,4,0,1]=(-0.599366869066)*x[0]**o+(-0.59893829688)*x[0]+(0.124483401866)*x[1]**o+(-0.924820315792)*x[1]
        arg[3,4,1,0]=(-0.707374787729)*x[0]**o+(-0.494748755022)*x[0]+(0.391752209698)*x[1]**o+(0.149710640116)*x[1]
        arg[3,4,1,1]=(0.329587748119)*x[0]**o+(-0.788289803764)*x[0]+(-0.145100163626)*x[1]**o+(0.141493810237)*x[1]
        arg[3,4,2,0]=(0.17496644337)*x[0]**o+(-0.270242104589)*x[0]+(0.996684379179)*x[1]**o+(-0.751697440046)*x[1]
        arg[3,4,2,1]=(0.79046837395)*x[0]**o+(0.568394473062)*x[0]+(0.359419854016)*x[1]**o+(0.630686948244)*x[1]
        ref[0,0,0,0]=(0.306552070969)/(o+1.)+(0.157870653629)
        ref[0,0,0,1]=(0.0606500807631)/(o+1.)+(0.0612449724389)
        ref[0,0,1,0]=(0.639564070889)/(o+1.)+(0.666341832555)
        ref[0,0,1,1]=(0.671248593147)/(o+1.)+(-0.528628572255)
        ref[0,0,2,0]=(0.0735824774293)/(o+1.)+(-0.474736472501)
        ref[0,0,2,1]=(-0.9070527228)/(o+1.)+(-0.197350261427)
        ref[0,1,0,0]=(1.30449671875)/(o+1.)+(0.106123067554)
        ref[0,1,0,1]=(-0.420072850657)/(o+1.)+(-0.407896954678)
        ref[0,1,1,0]=(1.44174379356)/(o+1.)+(-0.809395886627)
        ref[0,1,1,1]=(0.155435385684)/(o+1.)+(0.627148132992)
        ref[0,1,2,0]=(-0.137204404518)/(o+1.)+(0.207874473889)
        ref[0,1,2,1]=(-0.289709410261)/(o+1.)+(0.204293345671)
        ref[0,2,0,0]=(1.26583912644)/(o+1.)+(0.498805698213)
        ref[0,2,0,1]=(1.17560298985)/(o+1.)+(0.931845120886)
        ref[0,2,1,0]=(-0.344986302901)/(o+1.)+(0.220341743892)
        ref[0,2,1,1]=(-1.94726093397)/(o+1.)+(0.206668541494)
        ref[0,2,2,0]=(-0.91808772513)/(o+1.)+(-0.0252951560363)
        ref[0,2,2,1]=(-1.02863406132)/(o+1.)+(-0.20183443794)
        ref[0,3,0,0]=(-1.04834664255)/(o+1.)+(0.00604709752155)
        ref[0,3,0,1]=(0.2902816262)/(o+1.)+(0.514361224456)
        ref[0,3,1,0]=(-1.31568170019)/(o+1.)+(0.883603667426)
        ref[0,3,1,1]=(-0.208650456965)/(o+1.)+(0.889841082216)
        ref[0,3,2,0]=(-0.741632466554)/(o+1.)+(0.0759327240318)
        ref[0,3,2,1]=(1.48653223703)/(o+1.)+(-0.854483988026)
        ref[0,4,0,0]=(0.333816089225)/(o+1.)+(-0.562947445201)
        ref[0,4,0,1]=(0.130634932061)/(o+1.)+(0.199922965378)
        ref[0,4,1,0]=(1.27534940781)/(o+1.)+(0.158640649858)
        ref[0,4,1,1]=(-0.83666690538)/(o+1.)+(-0.344088772961)
        ref[0,4,2,0]=(-0.432609891704)/(o+1.)+(-0.267052816612)
        ref[0,4,2,1]=(-0.477920665494)/(o+1.)+(-0.305859149312)
        ref[1,0,0,0]=(-0.738766811663)/(o+1.)+(-0.2321755194)
        ref[1,0,0,1]=(-0.688095002235)/(o+1.)+(0.664163739465)
        ref[1,0,1,0]=(1.14285258401)/(o+1.)+(-0.345190031677)
        ref[1,0,1,1]=(-0.093656366746)/(o+1.)+(-0.236308341076)
        ref[1,0,2,0]=(0.7151363028)/(o+1.)+(-0.56709255572)
        ref[1,0,2,1]=(-0.51932492992)/(o+1.)+(0.250667355044)
        ref[1,1,0,0]=(0.545772982093)/(o+1.)+(-0.163104398138)
        ref[1,1,0,1]=(0.629573063174)/(o+1.)+(-0.174653201576)
        ref[1,1,1,0]=(-0.154012812311)/(o+1.)+(-0.914297866096)
        ref[1,1,1,1]=(-0.61132115559)/(o+1.)+(-0.44114113966)
        ref[1,1,2,0]=(0.408406410726)/(o+1.)+(-0.076080229207)
        ref[1,1,2,1]=(0.893038507335)/(o+1.)+(-0.263052908312)
        ref[1,2,0,0]=(0.17465338794)/(o+1.)+(0.391848465594)
        ref[1,2,0,1]=(-0.689339169821)/(o+1.)+(-0.460056198634)
        ref[1,2,1,0]=(1.31870632989)/(o+1.)+(-0.479414064332)
        ref[1,2,1,1]=(1.50891516069)/(o+1.)+(-0.0237454406495)
        ref[1,2,2,0]=(-0.453976065087)/(o+1.)+(-0.616348649278)
        ref[1,2,2,1]=(0.286587846307)/(o+1.)+(-0.0466708546809)
        ref[1,3,0,0]=(-0.380140989403)/(o+1.)+(-0.571048154745)
        ref[1,3,0,1]=(-0.883036620882)/(o+1.)+(-0.32753016973)
        ref[1,3,1,0]=(-0.60244085517)/(o+1.)+(-0.141401948721)
        ref[1,3,1,1]=(-0.853285111042)/(o+1.)+(0.342962640954)
        ref[1,3,2,0]=(-0.39249462323)/(o+1.)+(-0.390164886635)
        ref[1,3,2,1]=(-0.584758879441)/(o+1.)+(-0.00839083193621)
        ref[1,4,0,0]=(-0.282070155438)/(o+1.)+(-0.0156758086169)
        ref[1,4,0,1]=(0.472105372896)/(o+1.)+(-0.564324551903)
        ref[1,4,1,0]=(-1.31919056201)/(o+1.)+(-0.0984390311826)
        ref[1,4,1,1]=(-0.0914548911124)/(o+1.)+(0.0630846675525)
        ref[1,4,2,0]=(-0.327958330666)/(o+1.)+(0.657758815411)
        ref[1,4,2,1]=(-0.364384803274)/(o+1.)+(0.503617623946)
        ref[2,0,0,0]=(0.586128081557)/(o+1.)+(0.124182733972)
        ref[2,0,0,1]=(-1.08667878312)/(o+1.)+(0.187878007169)
        ref[2,0,1,0]=(0.181949317955)/(o+1.)+(0.341666123665)
        ref[2,0,1,1]=(1.37450614338)/(o+1.)+(-0.329280804596)
        ref[2,0,2,0]=(-1.79681719902)/(o+1.)+(0.31696617459)
        ref[2,0,2,1]=(-0.331005168235)/(o+1.)+(0.608933603219)
        ref[2,1,0,0]=(-0.608003218053)/(o+1.)+(0.303644850697)
        ref[2,1,0,1]=(0.0259152842714)/(o+1.)+(0.916297858039)
        ref[2,1,1,0]=(-0.33777899696)/(o+1.)+(-0.0168105795836)
        ref[2,1,1,1]=(-0.987531479864)/(o+1.)+(0.417880159899)
        ref[2,1,2,0]=(-0.426800950806)/(o+1.)+(0.0654268452787)
        ref[2,1,2,1]=(-0.570210924898)/(o+1.)+(-0.392672308506)
        ref[2,2,0,0]=(-0.489515474032)/(o+1.)+(0.516699346832)
        ref[2,2,0,1]=(-0.386977868298)/(o+1.)+(0.054797545679)
        ref[2,2,1,0]=(1.23803129631)/(o+1.)+(0.463802634429)
        ref[2,2,1,1]=(-1.3620596935)/(o+1.)+(-0.819152694454)
        ref[2,2,2,0]=(0.32363223218)/(o+1.)+(-0.48735071564)
        ref[2,2,2,1]=(1.3413745696)/(o+1.)+(0.00521108285542)
        ref[2,3,0,0]=(1.88998775272)/(o+1.)+(-0.231511311666)
        ref[2,3,0,1]=(0.205141263861)/(o+1.)+(0.305781695794)
        ref[2,3,1,0]=(-0.259160799442)/(o+1.)+(0.208196645933)
        ref[2,3,1,1]=(0.629015342187)/(o+1.)+(0.412571372816)
        ref[2,3,2,0]=(1.5808104592)/(o+1.)+(0.305903908491)
        ref[2,3,2,1]=(1.02138807107)/(o+1.)+(-0.109210298623)
        ref[2,4,0,0]=(-0.00598348295078)/(o+1.)+(-0.00132019700044)
        ref[2,4,0,1]=(-0.0447010582767)/(o+1.)+(-0.207914420506)
        ref[2,4,1,0]=(-1.52108837208)/(o+1.)+(-0.405268433289)
        ref[2,4,1,1]=(-0.445804637073)/(o+1.)+(-0.379603341408)
        ref[2,4,2,0]=(0.441040551703)/(o+1.)+(0.31124354157)
        ref[2,4,2,1]=(-0.487421406342)/(o+1.)+(0.0442323797396)
        ref[3,0,0,0]=(0.261030117323)/(o+1.)+(-0.518364490445)
        ref[3,0,0,1]=(0.154511078072)/(o+1.)+(-0.321176275884)
        ref[3,0,1,0]=(0.0588506980476)/(o+1.)+(0.0867902378403)
        ref[3,0,1,1]=(0.0205255602871)/(o+1.)+(-0.0578151720304)
        ref[3,0,2,0]=(0.0562400906237)/(o+1.)+(0.604864297795)
        ref[3,0,2,1]=(0.227337822629)/(o+1.)+(0.401828122727)
        ref[3,1,0,0]=(0.633555279654)/(o+1.)+(-0.764134541578)
        ref[3,1,0,1]=(1.25391845962)/(o+1.)+(-0.15254370663)
        ref[3,1,1,0]=(0.394560901238)/(o+1.)+(-0.13055106212)
        ref[3,1,1,1]=(-0.826276602534)/(o+1.)+(0.159559926573)
        ref[3,1,2,0]=(1.09494266707)/(o+1.)+(-0.20156573211)
        ref[3,1,2,1]=(0.131276383073)/(o+1.)+(0.0686377837506)
        ref[3,2,0,0]=(-0.391417020221)/(o+1.)+(0.356407442171)
        ref[3,2,0,1]=(0.0189277508933)/(o+1.)+(-0.433441674716)
        ref[3,2,1,0]=(-0.42121159653)/(o+1.)+(0.522691243905)
        ref[3,2,1,1]=(-0.698950981373)/(o+1.)+(0.200071855771)
        ref[3,2,2,0]=(-0.345651138499)/(o+1.)+(-0.572797784727)
        ref[3,2,2,1]=(-0.744798812536)/(o+1.)+(-0.358326266408)
        ref[3,3,0,0]=(0.126552674434)/(o+1.)+(-0.163334093135)
        ref[3,3,0,1]=(0.498630321844)/(o+1.)+(0.338628562253)
        ref[3,3,1,0]=(-0.290114167835)/(o+1.)+(-0.684920603147)
        ref[3,3,1,1]=(-0.0754224998727)/(o+1.)+(0.424630479064)
        ref[3,3,2,0]=(1.09036812867)/(o+1.)+(0.0417333730049)
        ref[3,3,2,1]=(0.0573733998999)/(o+1.)+(0.00477789011464)
        ref[3,4,0,0]=(-1.60775746297)/(o+1.)+(0.73525415701)
        ref[3,4,0,1]=(-0.4748834672)/(o+1.)+(-0.761879306336)
        ref[3,4,1,0]=(-0.31562257803)/(o+1.)+(-0.172519057453)
        ref[3,4,1,1]=(0.184487584493)/(o+1.)+(-0.323397996764)
        ref[3,4,2,0]=(1.17165082255)/(o+1.)+(-0.510969772317)
        ref[3,4,2,1]=(1.14988822797)/(o+1.)+(0.599540710653)
      else:
        arg[0,0,0,0]=(-0.517412938774)*x[0]**o+(0.824804313111)*x[0]+(-0.681706563374)*x[1]**o+(-0.259540440272)*x[1]+(0.876735283937)*x[2]**o+(0.307072859807)*x[2]
        arg[0,0,0,1]=(-0.545981202251)*x[0]**o+(-0.530080933212)*x[0]+(-0.339289111599)*x[1]**o+(-0.726311605088)*x[1]+(0.334837705444)*x[2]**o+(-0.422667311064)*x[2]
        arg[0,0,1,0]=(-0.94261387328)*x[0]**o+(-0.315825326734)*x[0]+(0.542502801277)*x[1]**o+(-0.927525943646)*x[1]+(-0.802118238213)*x[2]**o+(-0.106512314706)*x[2]
        arg[0,0,1,1]=(-0.877940099286)*x[0]**o+(-0.801256557869)*x[0]+(-0.155983792242)*x[1]**o+(0.200547191496)*x[1]+(-0.960553133432)*x[2]**o+(0.968862792367)*x[2]
        arg[0,0,2,0]=(-0.870342547946)*x[0]**o+(-0.956447402661)*x[0]+(-0.506797803004)*x[1]**o+(0.342841675107)*x[1]+(0.528354159344)*x[2]**o+(-0.824546729564)*x[2]
        arg[0,0,2,1]=(-0.411899137296)*x[0]**o+(0.133736458559)*x[0]+(0.852726131619)*x[1]**o+(0.692795229813)*x[1]+(0.047372576403)*x[2]**o+(0.810485765945)*x[2]
        arg[0,1,0,0]=(0.961758233254)*x[0]**o+(-0.606307924964)*x[0]+(0.0612696412665)*x[1]**o+(-0.947823541802)*x[1]+(-0.113021104365)*x[2]**o+(-0.00802440322155)*x[2]
        arg[0,1,0,1]=(-0.684097052467)*x[0]**o+(0.467140835537)*x[0]+(-0.0742537091205)*x[1]**o+(0.541641916143)*x[1]+(0.987491548505)*x[2]**o+(-0.558380262255)*x[2]
        arg[0,1,1,0]=(-0.901821988855)*x[0]**o+(0.34443714654)*x[0]+(-0.682552654793)*x[1]**o+(0.398276086461)*x[1]+(0.334644264965)*x[2]**o+(0.21454845251)*x[2]
        arg[0,1,1,1]=(0.929614060295)*x[0]**o+(-0.322609307947)*x[0]+(-0.426962200724)*x[1]**o+(0.464659992564)*x[1]+(-0.13983863674)*x[2]**o+(-0.681648817325)*x[2]
        arg[0,1,2,0]=(0.504166508914)*x[0]**o+(-0.978562859041)*x[0]+(-0.769650404806)*x[1]**o+(0.443500137153)*x[1]+(0.969669965936)*x[2]**o+(0.401742207403)*x[2]
        arg[0,1,2,1]=(-0.478561324822)*x[0]**o+(-0.772019838834)*x[0]+(-0.779194844829)*x[1]**o+(-0.747980400065)*x[1]+(0.827885608192)*x[2]**o+(0.135571151528)*x[2]
        arg[0,2,0,0]=(-0.210426221052)*x[0]**o+(-0.464938141903)*x[0]+(-0.776844066203)*x[1]**o+(0.267244555705)*x[1]+(-0.773135929429)*x[2]**o+(-0.00846060888954)*x[2]
        arg[0,2,0,1]=(0.0945858334531)*x[0]**o+(-0.492400895713)*x[0]+(-0.128564669727)*x[1]**o+(-0.0951458189983)*x[1]+(-0.54426715378)*x[2]**o+(0.106545346072)*x[2]
        arg[0,2,1,0]=(-0.482424571392)*x[0]**o+(0.0431208155652)*x[0]+(0.557138601636)*x[1]**o+(-0.725564282551)*x[1]+(0.9081107419)*x[2]**o+(0.0179201585352)*x[2]
        arg[0,2,1,1]=(-0.555731853052)*x[0]**o+(0.447137254671)*x[0]+(0.820958717703)*x[1]**o+(-0.580251434504)*x[1]+(0.227884622965)*x[2]**o+(0.34460189453)*x[2]
        arg[0,2,2,0]=(-0.25868983898)*x[0]**o+(0.759165881486)*x[0]+(-0.499903058341)*x[1]**o+(0.896471093639)*x[1]+(0.351955520633)*x[2]**o+(0.16566756566)*x[2]
        arg[0,2,2,1]=(-0.37859354794)*x[0]**o+(0.925387188779)*x[0]+(0.0111055293719)*x[1]**o+(-0.311998642154)*x[1]+(0.441009182046)*x[2]**o+(0.0968247918271)*x[2]
        arg[0,3,0,0]=(-0.461665900641)*x[0]**o+(0.812002051396)*x[0]+(-0.767456852027)*x[1]**o+(0.809069781575)*x[1]+(0.690416953885)*x[2]**o+(0.171662369216)*x[2]
        arg[0,3,0,1]=(-0.804811324639)*x[0]**o+(-0.0962711523954)*x[0]+(0.980130059825)*x[1]**o+(0.98279180603)*x[1]+(-0.0614353559054)*x[2]**o+(0.645637153538)*x[2]
        arg[0,3,1,0]=(0.97878415369)*x[0]**o+(-0.676321251698)*x[0]+(0.916075195717)*x[1]**o+(0.244509683371)*x[1]+(-0.939563193145)*x[2]**o+(-0.866506108913)*x[2]
        arg[0,3,1,1]=(-0.676965895162)*x[0]**o+(-0.124355338519)*x[0]+(0.680151101195)*x[1]**o+(-0.839412433156)*x[1]+(0.434088587223)*x[2]**o+(-0.275286048166)*x[2]
        arg[0,3,2,0]=(0.414822536626)*x[0]**o+(0.493836498194)*x[0]+(0.967077097745)*x[1]**o+(-0.0115301157869)*x[1]+(-0.307994657929)*x[2]**o+(-0.967766608312)*x[2]
        arg[0,3,2,1]=(-0.34064670348)*x[0]**o+(-0.975121872905)*x[0]+(0.0839457874766)*x[1]**o+(0.0995106978636)*x[1]+(-0.533837237516)*x[2]**o+(0.771268044302)*x[2]
        arg[0,4,0,0]=(0.223197544637)*x[0]**o+(0.712738871576)*x[0]+(-0.386972075747)*x[1]**o+(0.289829507744)*x[1]+(0.153418793839)*x[2]**o+(0.876087048492)*x[2]
        arg[0,4,0,1]=(0.696750948653)*x[0]**o+(0.466905335953)*x[0]+(-0.611476041907)*x[1]**o+(-0.939902457901)*x[1]+(-0.752675432265)*x[2]**o+(-0.869502717044)*x[2]
        arg[0,4,1,0]=(-0.465562201723)*x[0]**o+(-0.388716352051)*x[0]+(0.260044093503)*x[1]**o+(0.919379163294)*x[1]+(-0.902453394797)*x[2]**o+(0.378318744505)*x[2]
        arg[0,4,1,1]=(0.121990842946)*x[0]**o+(-0.691267894153)*x[0]+(-0.703960568755)*x[1]**o+(-0.582404896428)*x[1]+(0.57560180036)*x[2]**o+(0.537099448589)*x[2]
        arg[0,4,2,0]=(-0.495452196513)*x[0]**o+(-0.939389935669)*x[0]+(-0.237205717598)*x[1]**o+(0.61376516207)*x[1]+(-0.506082738728)*x[2]**o+(0.407704148542)*x[2]
        arg[0,4,2,1]=(0.450976180771)*x[0]**o+(-0.00739258898437)*x[0]+(0.210130848093)*x[1]**o+(0.624639007009)*x[1]+(0.173795671727)*x[2]**o+(0.508269170853)*x[2]
        arg[1,0,0,0]=(0.609685942177)*x[0]**o+(0.609299494609)*x[0]+(0.321046388121)*x[1]**o+(0.598197189498)*x[1]+(0.536092177618)*x[2]**o+(0.576427431157)*x[2]
        arg[1,0,0,1]=(0.412500706234)*x[0]**o+(0.905942239017)*x[0]+(-0.974980343169)*x[1]**o+(-0.131245997607)*x[1]+(0.745809068018)*x[2]**o+(0.543470258153)*x[2]
        arg[1,0,1,0]=(-0.903032545947)*x[0]**o+(-0.778693321051)*x[0]+(0.399342164105)*x[1]**o+(0.478212896767)*x[1]+(0.254596269802)*x[2]**o+(0.70308382001)*x[2]
        arg[1,0,1,1]=(-0.281529506793)*x[0]**o+(0.74166036027)*x[0]+(0.629713747725)*x[1]**o+(-0.275164746597)*x[1]+(0.37523909979)*x[2]**o+(-0.878026427978)*x[2]
        arg[1,0,2,0]=(0.236027764061)*x[0]**o+(-0.816910805083)*x[0]+(0.0782011624185)*x[1]**o+(-0.644004245666)*x[1]+(0.293586462602)*x[2]**o+(0.557045894695)*x[2]
        arg[1,0,2,1]=(0.868061727269)*x[0]**o+(0.53329318943)*x[0]+(0.367603915235)*x[1]**o+(0.89280277925)*x[1]+(-0.316595424948)*x[2]**o+(0.593830565147)*x[2]
        arg[1,1,0,0]=(0.886950595996)*x[0]**o+(-0.290323112286)*x[0]+(-0.863347030089)*x[1]**o+(0.545096994541)*x[1]+(0.338200944281)*x[2]**o+(-0.79898415048)*x[2]
        arg[1,1,0,1]=(0.103761462006)*x[0]**o+(-0.046916161864)*x[0]+(0.0312549239467)*x[1]**o+(0.214045118984)*x[1]+(-0.523161766402)*x[2]**o+(0.714207028317)*x[2]
        arg[1,1,1,0]=(0.494943183987)*x[0]**o+(-0.0543096386006)*x[0]+(-0.263875120588)*x[1]**o+(0.776031489925)*x[1]+(0.881224752145)*x[2]**o+(-0.650038283228)*x[2]
        arg[1,1,1,1]=(0.0871887540513)*x[0]**o+(-0.953848841349)*x[0]+(-0.223751459589)*x[1]**o+(0.227400823993)*x[1]+(-0.224200553009)*x[2]**o+(0.594012970899)*x[2]
        arg[1,1,2,0]=(-0.443116951346)*x[0]**o+(0.891806149626)*x[0]+(0.94731530917)*x[1]**o+(0.739499894809)*x[1]+(-0.92613053703)*x[2]**o+(0.464883087528)*x[2]
        arg[1,1,2,1]=(-0.149831145559)*x[0]**o+(0.913999704597)*x[0]+(0.76082978152)*x[1]**o+(-0.441535347622)*x[1]+(0.902841586936)*x[2]**o+(-0.509022004948)*x[2]
        arg[1,2,0,0]=(0.508230188489)*x[0]**o+(0.241669921231)*x[0]+(-0.802811365056)*x[1]**o+(0.298424305948)*x[1]+(-0.949407454532)*x[2]**o+(-0.131426880214)*x[2]
        arg[1,2,0,1]=(0.65372313274)*x[0]**o+(0.195891642771)*x[0]+(-0.160845429219)*x[1]**o+(0.500815823724)*x[1]+(0.875785249339)*x[2]**o+(0.325024339211)*x[2]
        arg[1,2,1,0]=(0.908405682536)*x[0]**o+(-0.146675166477)*x[0]+(0.24033815873)*x[1]**o+(-0.62814219771)*x[1]+(-0.512365788351)*x[2]**o+(-0.103817191415)*x[2]
        arg[1,2,1,1]=(0.824305237501)*x[0]**o+(0.175753992729)*x[0]+(0.992502114126)*x[1]**o+(0.973110627555)*x[1]+(0.355352002177)*x[2]**o+(0.112252263955)*x[2]
        arg[1,2,2,0]=(-0.485996285003)*x[0]**o+(-0.532551123887)*x[0]+(0.708205121405)*x[1]**o+(-0.803269038441)*x[1]+(-0.232049124534)*x[2]**o+(0.55437652866)*x[2]
        arg[1,2,2,1]=(0.551934776224)*x[0]**o+(0.727629664538)*x[0]+(0.197393511049)*x[1]**o+(-0.146619812174)*x[1]+(0.316592306936)*x[2]**o+(0.0263881493809)*x[2]
        arg[1,3,0,0]=(-0.918554166254)*x[0]**o+(-0.839150758884)*x[0]+(0.436713305329)*x[1]**o+(-0.410131731387)*x[1]+(0.321246028791)*x[2]**o+(0.706389384376)*x[2]
        arg[1,3,0,1]=(-0.870168753295)*x[0]**o+(-0.00473364206025)*x[0]+(0.594648540734)*x[1]**o+(-0.924887214906)*x[1]+(0.58858894515)*x[2]**o+(0.0152152159566)*x[2]
        arg[1,3,1,0]=(0.64005604234)*x[0]**o+(0.308265334836)*x[0]+(0.321645661286)*x[1]**o+(0.670017614875)*x[1]+(0.712764136703)*x[2]**o+(0.458986964342)*x[2]
        arg[1,3,1,1]=(-0.201376578944)*x[0]**o+(-0.271793865118)*x[0]+(-0.798181331582)*x[1]**o+(0.690100877945)*x[1]+(0.600350295622)*x[2]**o+(0.341066858436)*x[2]
        arg[1,3,2,0]=(0.459832113137)*x[0]**o+(0.0566780273877)*x[0]+(0.221723079112)*x[1]**o+(-0.681001728734)*x[1]+(0.265505135729)*x[2]**o+(-0.389365671367)*x[2]
        arg[1,3,2,1]=(0.715444384493)*x[0]**o+(-0.705697520786)*x[0]+(0.945208081248)*x[1]**o+(0.0233171867808)*x[1]+(0.740156166802)*x[2]**o+(-0.144314337012)*x[2]
        arg[1,4,0,0]=(-0.490162545806)*x[0]**o+(0.542981268396)*x[0]+(-0.841471465564)*x[1]**o+(0.79091798393)*x[1]+(0.245598142884)*x[2]**o+(0.4740296042)*x[2]
        arg[1,4,0,1]=(-0.504504615322)*x[0]**o+(0.45494067251)*x[0]+(-0.612427248728)*x[1]**o+(0.0880340051743)*x[1]+(0.645906227711)*x[2]**o+(0.772038416622)*x[2]
        arg[1,4,1,0]=(0.945538632492)*x[0]**o+(0.98519603491)*x[0]+(-0.245495451859)*x[1]**o+(-0.70732864026)*x[1]+(0.378733662069)*x[2]**o+(0.579554415018)*x[2]
        arg[1,4,1,1]=(0.886659064398)*x[0]**o+(-0.491870874438)*x[0]+(0.751814742427)*x[1]**o+(0.430589443767)*x[1]+(-0.729193959143)*x[2]**o+(-0.699344758538)*x[2]
        arg[1,4,2,0]=(-0.4190140481)*x[0]**o+(0.84509612452)*x[0]+(-0.263608911857)*x[1]**o+(0.693648282371)*x[1]+(-0.144463932166)*x[2]**o+(0.384293689266)*x[2]
        arg[1,4,2,1]=(-0.266973780787)*x[0]**o+(0.613988367707)*x[0]+(0.686955199789)*x[1]**o+(-0.301585818785)*x[1]+(-0.574601202558)*x[2]**o+(-0.430718616071)*x[2]
        arg[2,0,0,0]=(-0.117390727803)*x[0]**o+(-0.25850139145)*x[0]+(0.324909086031)*x[1]**o+(-0.593112241553)*x[1]+(-0.51016768177)*x[2]**o+(0.392299415334)*x[2]
        arg[2,0,0,1]=(0.290478531696)*x[0]**o+(0.758153023626)*x[0]+(0.531003993361)*x[1]**o+(-0.112481510374)*x[1]+(0.950222158203)*x[2]**o+(-0.742480526204)*x[2]
        arg[2,0,1,0]=(-0.754820921225)*x[0]**o+(-0.408482044077)*x[0]+(-0.229697426486)*x[1]**o+(-0.983010424713)*x[1]+(-0.286618604257)*x[2]**o+(-0.489034108451)*x[2]
        arg[2,0,1,1]=(0.96775757822)*x[0]**o+(0.797765725721)*x[0]+(-0.90936176648)*x[1]**o+(-0.372755885193)*x[1]+(-0.144727874143)*x[2]**o+(0.341624944551)*x[2]
        arg[2,0,2,0]=(0.672779422278)*x[0]**o+(-0.936102224964)*x[0]+(-0.763620352857)*x[1]**o+(0.265802509744)*x[1]+(0.275798875047)*x[2]**o+(0.235397520942)*x[2]
        arg[2,0,2,1]=(0.192866248041)*x[0]**o+(0.656863606616)*x[0]+(-0.902796948663)*x[1]**o+(0.692123089256)*x[1]+(0.311287987388)*x[2]**o+(0.555427507546)*x[2]
        arg[2,1,0,0]=(0.464585537593)*x[0]**o+(0.161744615208)*x[0]+(-0.325569920662)*x[1]**o+(-0.47142644441)*x[1]+(0.885151021784)*x[2]**o+(-0.840595219086)*x[2]
        arg[2,1,0,1]=(-0.659253397488)*x[0]**o+(0.54391314862)*x[0]+(-0.595612529952)*x[1]**o+(-0.0472758114379)*x[1]+(0.876391930188)*x[2]**o+(-0.440313199946)*x[2]
        arg[2,1,1,0]=(-0.445591423364)*x[0]**o+(0.179144841652)*x[0]+(-0.294166502086)*x[1]**o+(-0.373820778739)*x[1]+(0.464609314863)*x[2]**o+(0.407893781978)*x[2]
        arg[2,1,1,1]=(-0.767512700255)*x[0]**o+(0.63198312168)*x[0]+(0.652318598902)*x[1]**o+(0.817874971823)*x[1]+(0.397211574885)*x[2]**o+(0.33174354886)*x[2]
        arg[2,1,2,0]=(0.473801282131)*x[0]**o+(-0.778895701394)*x[0]+(-0.801488721847)*x[1]**o+(-0.585840161723)*x[1]+(-0.805496721223)*x[2]**o+(0.585971840421)*x[2]
        arg[2,1,2,1]=(0.472329620891)*x[0]**o+(-0.0473805663117)*x[0]+(0.0894117727522)*x[1]**o+(-0.308473625733)*x[1]+(-0.658376953956)*x[2]**o+(0.32027592346)*x[2]
        arg[2,2,0,0]=(0.843241796369)*x[0]**o+(0.346857959798)*x[0]+(0.0111223078269)*x[1]**o+(-0.949623227267)*x[1]+(-0.152362393376)*x[2]**o+(-0.886340001141)*x[2]
        arg[2,2,0,1]=(-0.555393018762)*x[0]**o+(0.666364344488)*x[0]+(0.181237902671)*x[1]**o+(0.670959040345)*x[1]+(0.57383285317)*x[2]**o+(-0.451027832775)*x[2]
        arg[2,2,1,0]=(0.249709446431)*x[0]**o+(0.568753063379)*x[0]+(-0.812386750097)*x[1]**o+(0.506805124672)*x[1]+(-0.454844217662)*x[2]**o+(0.843435269288)*x[2]
        arg[2,2,1,1]=(0.753144930025)*x[0]**o+(0.0181642998269)*x[0]+(0.622635241593)*x[1]**o+(-0.372883842874)*x[1]+(0.366920965577)*x[2]**o+(-0.092391524816)*x[2]
        arg[2,2,2,0]=(0.309601906611)*x[0]**o+(-0.502573192965)*x[0]+(-0.420795137333)*x[1]**o+(-0.807109356307)*x[1]+(-0.802887583918)*x[2]**o+(0.552609684817)*x[2]
        arg[2,2,2,1]=(0.320933223342)*x[0]**o+(0.541931544704)*x[0]+(0.988715316931)*x[1]**o+(-0.4197325237)*x[1]+(-0.438705512435)*x[2]**o+(-0.540175093599)*x[2]
        arg[2,3,0,0]=(0.018870015518)*x[0]**o+(-0.615120691773)*x[0]+(0.596295280542)*x[1]**o+(0.210874310884)*x[1]+(-0.652194502321)*x[2]**o+(0.74019575754)*x[2]
        arg[2,3,0,1]=(0.820285141622)*x[0]**o+(0.130985741886)*x[0]+(-0.397387767472)*x[1]**o+(0.975103051154)*x[1]+(0.0887369214611)*x[2]**o+(0.88498731668)*x[2]
        arg[2,3,1,0]=(0.160503336808)*x[0]**o+(0.641180113144)*x[0]+(-0.408109675737)*x[1]**o+(-0.355443951133)*x[1]+(-0.429622016495)*x[2]**o+(0.998729833779)*x[2]
        arg[2,3,1,1]=(-0.646524352112)*x[0]**o+(-0.289894880059)*x[0]+(0.909761879864)*x[1]**o+(0.30453651619)*x[1]+(0.884596709091)*x[2]**o+(0.329408737003)*x[2]
        arg[2,3,2,0]=(0.459385919432)*x[0]**o+(0.416447271916)*x[0]+(-0.400798659819)*x[1]**o+(-0.769043818187)*x[1]+(-0.631682400803)*x[2]**o+(-0.80150361084)*x[2]
        arg[2,3,2,1]=(-0.544824298969)*x[0]**o+(0.191301965562)*x[0]+(0.41061102918)*x[1]**o+(-0.999195759064)*x[1]+(-0.0226105975802)*x[2]**o+(-0.137078691725)*x[2]
        arg[2,4,0,0]=(-0.624033952375)*x[0]**o+(0.0792222332007)*x[0]+(0.766756718455)*x[1]**o+(0.581487379511)*x[1]+(-0.437940391245)*x[2]**o+(0.115617341492)*x[2]
        arg[2,4,0,1]=(-0.300526068838)*x[0]**o+(-0.822844066866)*x[0]+(0.660949310367)*x[1]**o+(-0.0993949179102)*x[1]+(0.599884615655)*x[2]**o+(-0.903177033523)*x[2]
        arg[2,4,1,0]=(-0.654642517091)*x[0]**o+(0.428391621765)*x[0]+(0.932377441464)*x[1]**o+(-0.459525189684)*x[1]+(0.667425199675)*x[2]**o+(-0.937213410686)*x[2]
        arg[2,4,1,1]=(0.258067525957)*x[0]**o+(0.619943768383)*x[0]+(0.458217567511)*x[1]**o+(0.0812191081554)*x[1]+(0.743260637332)*x[2]**o+(-0.796150472293)*x[2]
        arg[2,4,2,0]=(0.782473636556)*x[0]**o+(-0.441405613305)*x[0]+(0.3951595327)*x[1]**o+(-0.449855781187)*x[1]+(-0.964130179212)*x[2]**o+(-0.299199410525)*x[2]
        arg[2,4,2,1]=(-0.878340584397)*x[0]**o+(-0.674575785429)*x[0]+(0.838137461257)*x[1]**o+(-0.498710475715)*x[1]+(-0.117205127994)*x[2]**o+(0.0934669704542)*x[2]
        arg[3,0,0,0]=(0.139631366137)*x[0]**o+(-0.893001605238)*x[0]+(-0.420780773096)*x[1]**o+(0.439055863393)*x[1]+(0.173113363237)*x[2]**o+(0.893867443017)*x[2]
        arg[3,0,0,1]=(-0.804823042124)*x[0]**o+(0.250231181298)*x[0]+(0.358464048439)*x[1]**o+(-0.847391231172)*x[1]+(-0.515391975323)*x[2]**o+(-0.999640302937)*x[2]
        arg[3,0,1,0]=(0.328676968943)*x[0]**o+(-0.0521242139559)*x[0]+(0.861548411537)*x[1]**o+(-0.288979961428)*x[1]+(-0.927901849987)*x[2]**o+(0.857869958971)*x[2]
        arg[3,0,1,1]=(-0.654866876537)*x[0]**o+(-0.0515462321916)*x[0]+(-0.224164685097)*x[1]**o+(-0.84207399757)*x[1]+(-0.442392040219)*x[2]**o+(-0.915734076984)*x[2]
        arg[3,0,2,0]=(-0.322517406382)*x[0]**o+(-0.757146499161)*x[0]+(-0.541690197465)*x[1]**o+(0.931090695413)*x[1]+(-0.355505263744)*x[2]**o+(-0.773185404514)*x[2]
        arg[3,0,2,1]=(0.421030649833)*x[0]**o+(-0.83768939688)*x[0]+(-0.863791071447)*x[1]**o+(0.78393205545)*x[1]+(-0.489893911681)*x[2]**o+(-0.291989245709)*x[2]
        arg[3,1,0,0]=(0.550839051007)*x[0]**o+(-0.883853789938)*x[0]+(-0.339529971923)*x[1]**o+(-0.756936847344)*x[1]+(0.724888858656)*x[2]**o+(-0.981057762224)*x[2]
        arg[3,1,0,1]=(0.711792743156)*x[0]**o+(0.233248429378)*x[0]+(0.526076154408)*x[1]**o+(0.349364804166)*x[1]+(0.271475992486)*x[2]**o+(0.782429402713)*x[2]
        arg[3,1,1,0]=(-0.499072848146)*x[0]**o+(-0.0604888692308)*x[0]+(0.545442893272)*x[1]**o+(-0.688797522515)*x[1]+(0.305341295222)*x[2]**o+(0.130010601331)*x[2]
        arg[3,1,1,1]=(0.940695359452)*x[0]**o+(0.113015472294)*x[0]+(-0.506254395321)*x[1]**o+(-0.252044277039)*x[1]+(0.0172784972094)*x[2]**o+(-0.184256978193)*x[2]
        arg[3,1,2,0]=(0.550472143981)*x[0]**o+(-0.729809590776)*x[0]+(-0.293765999495)*x[1]**o+(-0.378042976721)*x[1]+(0.973242167993)*x[2]**o+(-0.75350391134)*x[2]
        arg[3,1,2,1]=(0.369266178598)*x[0]**o+(-0.764034492366)*x[0]+(-0.713325992328)*x[1]**o+(0.948932375717)*x[1]+(-0.335074977323)*x[2]**o+(0.117829591745)*x[2]
        arg[3,2,0,0]=(0.426458076699)*x[0]**o+(-0.520551329627)*x[0]+(-0.773464312884)*x[1]**o+(0.26332468989)*x[1]+(0.143497828773)*x[2]**o+(0.0676278275986)*x[2]
        arg[3,2,0,1]=(0.85806016531)*x[0]**o+(0.755361032731)*x[0]+(0.403942188112)*x[1]**o+(0.0113665816202)*x[1]+(-0.439835633198)*x[2]**o+(0.620930145183)*x[2]
        arg[3,2,1,0]=(-0.852838624821)*x[0]**o+(-0.918594624263)*x[0]+(-0.572898555139)*x[1]**o+(-0.14353390918)*x[1]+(-0.716010071307)*x[2]**o+(0.134687635305)*x[2]
        arg[3,2,1,1]=(-0.14282590746)*x[0]**o+(-0.169314199229)*x[0]+(0.880745511071)*x[1]**o+(0.540733710441)*x[1]+(-0.131424658303)*x[2]**o+(-0.93209431119)*x[2]
        arg[3,2,2,0]=(-0.864016075688)*x[0]**o+(-0.265794760131)*x[0]+(0.444625188371)*x[1]**o+(0.133693703972)*x[1]+(-0.749799597936)*x[2]**o+(0.886355820615)*x[2]
        arg[3,2,2,1]=(0.676281241167)*x[0]**o+(-0.351800298325)*x[0]+(-0.751498380099)*x[1]**o+(-0.561737786104)*x[1]+(-0.984872672853)*x[2]**o+(0.476840947707)*x[2]
        arg[3,3,0,0]=(0.457874366022)*x[0]**o+(-0.126929649966)*x[0]+(0.666077559139)*x[1]**o+(-0.966437669224)*x[1]+(-0.849000502816)*x[2]**o+(0.990790747926)*x[2]
        arg[3,3,0,1]=(0.45136859326)*x[0]**o+(0.771355456911)*x[0]+(-0.678558472676)*x[1]**o+(0.0366869703264)*x[1]+(-0.257247358952)*x[2]**o+(-0.290852285538)*x[2]
        arg[3,3,1,0]=(-0.623007069616)*x[0]**o+(-0.365255080044)*x[0]+(0.441927420573)*x[1]**o+(0.331640625664)*x[1]+(-0.21153951456)*x[2]**o+(0.297967669123)*x[2]
        arg[3,3,1,1]=(-0.836057742118)*x[0]**o+(0.959873293328)*x[0]+(-0.52034405502)*x[1]**o+(-0.178838686591)*x[1]+(0.317610563306)*x[2]**o+(-0.555134650833)*x[2]
        arg[3,3,2,0]=(0.0819108022024)*x[0]**o+(0.69432296821)*x[0]+(-0.23685434367)*x[1]**o+(-0.0442137744391)*x[1]+(-0.879175086057)*x[2]**o+(-0.28031950012)*x[2]
        arg[3,3,2,1]=(0.340083286763)*x[0]**o+(-0.381942838645)*x[0]+(0.570579443852)*x[1]**o+(0.772192708863)*x[1]+(0.42813556157)*x[2]**o+(0.735683065499)*x[2]
        arg[3,4,0,0]=(-0.866813692418)*x[0]**o+(-0.53715159706)*x[0]+(0.0725900211922)*x[1]**o+(0.0891167848614)*x[1]+(-0.381482922347)*x[2]**o+(-0.521504123034)*x[2]
        arg[3,4,0,1]=(0.487750922723)*x[0]**o+(0.967820473849)*x[0]+(0.305600886352)*x[1]**o+(0.350777657643)*x[1]+(-0.383731485788)*x[2]**o+(0.814016126476)*x[2]
        arg[3,4,1,0]=(-0.186663244344)*x[0]**o+(0.144872424379)*x[0]+(0.844454339304)*x[1]**o+(0.252133708483)*x[1]+(-0.748235508222)*x[2]**o+(0.702952185343)*x[2]
        arg[3,4,1,1]=(-0.962296047927)*x[0]**o+(0.874277032826)*x[0]+(-0.404689253219)*x[1]**o+(-0.174075799004)*x[1]+(0.65086350106)*x[2]**o+(-0.109850650677)*x[2]
        arg[3,4,2,0]=(-0.59515836027)*x[0]**o+(-0.0917296352911)*x[0]+(0.834779238353)*x[1]**o+(0.951252871777)*x[1]+(0.983008684497)*x[2]**o+(-0.784806954891)*x[2]
        arg[3,4,2,1]=(0.603110971708)*x[0]**o+(0.291507042984)*x[0]+(-0.401523286981)*x[1]**o+(-0.994760476862)*x[1]+(0.410516007475)*x[2]**o+(0.368982230001)*x[2]
        ref[0,0,0,0]=(-0.322384218211)/(o+1.)+(0.436168366323)
        ref[0,0,0,1]=(-0.550432608405)/(o+1.)+(-0.839529924682)
        ref[0,0,1,0]=(-1.20222931022)/(o+1.)+(-0.674931792543)
        ref[0,0,1,1]=(-1.99447702496)/(o+1.)+(0.184076712997)
        ref[0,0,2,0]=(-0.848786191606)/(o+1.)+(-0.719076228559)
        ref[0,0,2,1]=(0.488199570726)/(o+1.)+(0.818508727159)
        ref[0,1,0,0]=(0.910006770156)/(o+1.)+(-0.781077934994)
        ref[0,1,0,1]=(0.229140786917)/(o+1.)+(0.225201244712)
        ref[0,1,1,0]=(-1.24973037868)/(o+1.)+(0.478630842756)
        ref[0,1,1,1]=(0.362813222831)/(o+1.)+(-0.269799066354)
        ref[0,1,2,0]=(0.704186070044)/(o+1.)+(-0.0666602572428)
        ref[0,1,2,1]=(-0.429870561459)/(o+1.)+(-0.692214543685)
        ref[0,2,0,0]=(-1.76040621668)/(o+1.)+(-0.103077097544)
        ref[0,2,0,1]=(-0.578245990054)/(o+1.)+(-0.24050068432)
        ref[0,2,1,0]=(0.982824772144)/(o+1.)+(-0.332261654225)
        ref[0,2,1,1]=(0.493111487615)/(o+1.)+(0.105743857349)
        ref[0,2,2,0]=(-0.406637376689)/(o+1.)+(0.910652270393)
        ref[0,2,2,1]=(0.0735211634779)/(o+1.)+(0.355106669226)
        ref[0,3,0,0]=(-0.538705798782)/(o+1.)+(0.896367101094)
        ref[0,3,0,1]=(0.11388337928)/(o+1.)+(0.766078903586)
        ref[0,3,1,0]=(0.955296156262)/(o+1.)+(-0.64915883862)
        ref[0,3,1,1]=(0.437273793256)/(o+1.)+(-0.61952690992)
        ref[0,3,2,0]=(1.07390497644)/(o+1.)+(-0.242730112953)
        ref[0,3,2,1]=(-0.79053815352)/(o+1.)+(-0.0521715653697)
        ref[0,4,0,0]=(-0.0103557372704)/(o+1.)+(0.939327713906)
        ref[0,4,0,1]=(-0.66740052552)/(o+1.)+(-0.671249919496)
        ref[0,4,1,0]=(-1.10797150302)/(o+1.)+(0.454490777873)
        ref[0,4,1,1]=(-0.00636792544909)/(o+1.)+(-0.368286670996)
        ref[0,4,2,0]=(-1.23874065284)/(o+1.)+(0.0410396874718)
        ref[0,4,2,1]=(0.834902700591)/(o+1.)+(0.562757794439)
        ref[1,0,0,0]=(1.46682450791)/(o+1.)+(0.891962057632)
        ref[1,0,0,1]=(0.183329431084)/(o+1.)+(0.659083249781)
        ref[1,0,1,0]=(-0.24909411204)/(o+1.)+(0.201301697863)
        ref[1,0,1,1]=(0.723423340723)/(o+1.)+(-0.205765407152)
        ref[1,0,2,0]=(0.607815389082)/(o+1.)+(-0.451934578027)
        ref[1,0,2,1]=(0.919070217555)/(o+1.)+(1.00996326691)
        ref[1,1,0,0]=(0.361804510188)/(o+1.)+(-0.272105134113)
        ref[1,1,0,1]=(-0.388145380449)/(o+1.)+(0.440667992719)
        ref[1,1,1,0]=(1.11229281554)/(o+1.)+(0.0358417840478)
        ref[1,1,1,1]=(-0.360763258547)/(o+1.)+(-0.0662175232282)
        ref[1,1,2,0]=(-0.421932179206)/(o+1.)+(1.04809456598)
        ref[1,1,2,1]=(1.5138402229)/(o+1.)+(-0.0182788239862)
        ref[1,2,0,0]=(-1.2439886311)/(o+1.)+(0.204333673483)
        ref[1,2,0,1]=(1.36866295286)/(o+1.)+(0.510865902853)
        ref[1,2,1,0]=(0.636378052915)/(o+1.)+(-0.439317277801)
        ref[1,2,1,1]=(2.1721593538)/(o+1.)+(0.63055844212)
        ref[1,2,2,0]=(-0.00984028813171)/(o+1.)+(-0.390721816834)
        ref[1,2,2,1]=(1.06592059421)/(o+1.)+(0.303699000873)
        ref[1,3,0,0]=(-0.160594832134)/(o+1.)+(-0.271446552947)
        ref[1,3,0,1]=(0.313068732589)/(o+1.)+(-0.457202820505)
        ref[1,3,1,0]=(1.67446584033)/(o+1.)+(0.718634957026)
        ref[1,3,1,1]=(-0.399207614903)/(o+1.)+(0.379686935632)
        ref[1,3,2,0]=(0.947060327978)/(o+1.)+(-0.506844686357)
        ref[1,3,2,1]=(2.40080863254)/(o+1.)+(-0.413347335508)
        ref[1,4,0,0]=(-1.08603586849)/(o+1.)+(0.903964428263)
        ref[1,4,0,1]=(-0.471025636339)/(o+1.)+(0.657506547153)
        ref[1,4,1,0]=(1.0787768427)/(o+1.)+(0.428710904834)
        ref[1,4,1,1]=(0.909279847682)/(o+1.)+(-0.380313094604)
        ref[1,4,2,0]=(-0.827086892123)/(o+1.)+(0.961519048079)
        ref[1,4,2,1]=(-0.154619783556)/(o+1.)+(-0.0591580335741)
        ref[2,0,0,0]=(-0.302649323542)/(o+1.)+(-0.229657108834)
        ref[2,0,0,1]=(1.77170468326)/(o+1.)+(-0.0484045064764)
        ref[2,0,1,0]=(-1.27113695197)/(o+1.)+(-0.94026328862)
        ref[2,0,1,1]=(-0.0863320624027)/(o+1.)+(0.383317392539)
        ref[2,0,2,0]=(0.184957944467)/(o+1.)+(-0.217451097139)
        ref[2,0,2,1]=(-0.398642713235)/(o+1.)+(0.952207101709)
        ref[2,1,0,0]=(1.02416663872)/(o+1.)+(-0.575138524144)
        ref[2,1,0,1]=(-0.378473997252)/(o+1.)+(0.0281620686178)
        ref[2,1,1,0]=(-0.275148610588)/(o+1.)+(0.106608922446)
        ref[2,1,1,1]=(0.282017473532)/(o+1.)+(0.890800821181)
        ref[2,1,2,0]=(-1.13318416094)/(o+1.)+(-0.389382011348)
        ref[2,1,2,1]=(-0.0966355603128)/(o+1.)+(-0.0177891342927)
        ref[2,2,0,0]=(0.70200171082)/(o+1.)+(-0.744552634305)
        ref[2,2,0,1]=(0.199677737079)/(o+1.)+(0.443147776029)
        ref[2,2,1,0]=(-1.01752152133)/(o+1.)+(0.959496728669)
        ref[2,2,1,1]=(1.74270113719)/(o+1.)+(-0.223555533932)
        ref[2,2,2,0]=(-0.91408081464)/(o+1.)+(-0.378536432227)
        ref[2,2,2,1]=(0.870943027839)/(o+1.)+(-0.208988036298)
        ref[2,3,0,0]=(-0.0370292062615)/(o+1.)+(0.167974688326)
        ref[2,3,0,1]=(0.51163429561)/(o+1.)+(0.99553805486)
        ref[2,3,1,0]=(-0.677228355424)/(o+1.)+(0.642232997895)
        ref[2,3,1,1]=(1.14783423684)/(o+1.)+(0.172025186567)
        ref[2,3,2,0]=(-0.57309514119)/(o+1.)+(-0.577050078555)
        ref[2,3,2,1]=(-0.156823867369)/(o+1.)+(-0.472486242613)
        ref[2,4,0,0]=(-0.295217625165)/(o+1.)+(0.388163477102)
        ref[2,4,0,1]=(0.960307857184)/(o+1.)+(-0.91270800915)
        ref[2,4,1,0]=(0.945160124047)/(o+1.)+(-0.484173489303)
        ref[2,4,1,1]=(1.4595457308)/(o+1.)+(-0.0474937978771)
        ref[2,4,2,0]=(0.213502990044)/(o+1.)+(-0.595230402509)
        ref[2,4,2,1]=(-0.157408251134)/(o+1.)+(-0.539909645345)
        ref[3,0,0,0]=(-0.108036043722)/(o+1.)+(0.219960850586)
        ref[3,0,0,1]=(-0.961750969008)/(o+1.)+(-0.798400176406)
        ref[3,0,1,0]=(0.262323530493)/(o+1.)+(0.258382891794)
        ref[3,0,1,1]=(-1.32142360185)/(o+1.)+(-0.904677153373)
        ref[3,0,2,0]=(-1.21971286759)/(o+1.)+(-0.299620604131)
        ref[3,0,2,1]=(-0.932654333294)/(o+1.)+(-0.17287329357)
        ref[3,1,0,0]=(0.93619793774)/(o+1.)+(-1.31092419975)
        ref[3,1,0,1]=(1.50934489005)/(o+1.)+(0.682521318129)
        ref[3,1,1,0]=(0.351711340348)/(o+1.)+(-0.309637895207)
        ref[3,1,1,1]=(0.451719461341)/(o+1.)+(-0.161642891469)
        ref[3,1,2,0]=(1.22994831248)/(o+1.)+(-0.930678239419)
        ref[3,1,2,1]=(-0.679134791053)/(o+1.)+(0.151363737548)
        ref[3,2,0,0]=(-0.203508407413)/(o+1.)+(-0.0947994060691)
        ref[3,2,0,1]=(0.822166720224)/(o+1.)+(0.693828879767)
        ref[3,2,1,0]=(-2.14174725127)/(o+1.)+(-0.463720449069)
        ref[3,2,1,1]=(0.606494945308)/(o+1.)+(-0.280337399989)
        ref[3,2,2,0]=(-1.16919048525)/(o+1.)+(0.377127382228)
        ref[3,2,2,1]=(-1.06008981178)/(o+1.)+(-0.218348568361)
        ref[3,3,0,0]=(0.274951422345)/(o+1.)+(-0.0512882856318)
        ref[3,3,0,1]=(-0.484437238367)/(o+1.)+(0.25859507085)
        ref[3,3,1,0]=(-0.392619163603)/(o+1.)+(0.132176607371)
        ref[3,3,1,1]=(-1.03879123383)/(o+1.)+(0.112949977952)
        ref[3,3,2,0]=(-1.03411862752)/(o+1.)+(0.184894846825)
        ref[3,3,2,1]=(1.33879829218)/(o+1.)+(0.562966467859)
        ref[3,4,0,0]=(-1.17570659357)/(o+1.)+(-0.484769467616)
        ref[3,4,0,1]=(0.409620323287)/(o+1.)+(1.06630712898)
        ref[3,4,1,0]=(-0.0904444132627)/(o+1.)+(0.549979159103)
        ref[3,4,1,1]=(-0.716121800087)/(o+1.)+(0.295175291572)
        ref[3,4,2,0]=(1.22262956258)/(o+1.)+(0.0373581407971)
        ref[3,4,2,1]=(0.612103692201)/(o+1.)+(-0.167135601938)
      res=integrate(arg)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5, 3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnBoundary_fromData_ContinuousFunction_rank0(self):
      """
      tests integral of rank 0 Data on the FunctionOnBoundary

      assumptions: ContinuousFunction(self.domain) exists
                   self.domain supports integral on FunctionOnBoundary
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnBoundary(self.domain)
      w=ContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(),w)
      if dim==2:
        arg=(0.506621877215)*x[0]**o+(-0.45560383685)*x[0]+(-0.598137301806)*x[1]**o+(0.784241595615)*x[1]
        ref=(-0.0915154245913)*(1+2.*(dim-1.)/(o+1.))+(0.328637758765)*dim
      else:
        arg=(-0.505489131484)*x[0]**o+(0.58139159259)*x[0]+(0.458086828888)*x[1]**o+(0.549090746672)*x[1]+(-0.0882325936167)*x[2]**o+(0.0170295954808)*x[2]
        ref=(-0.135634896213)*(1+2.*(dim-1.)/(o+1.))+(1.14751193474)*dim
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnBoundary_fromData_ContinuousFunction_rank1(self):
      """
      tests integral of rank 1 Data on the FunctionOnBoundary

      assumptions: ContinuousFunction(self.domain) exists
                   self.domain supports integral on FunctionOnBoundary
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnBoundary(self.domain)
      w=ContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(2,),w)
      ref=numpy.zeros((2,),numpy.float_)
      if dim==2:
        arg[0]=(-0.88779433873)*x[0]**o+(0.507955012301)*x[0]+(0.652739274924)*x[1]**o+(-0.971975755989)*x[1]
        arg[1]=(-0.0812690931139)*x[0]**o+(-0.283880962354)*x[0]+(0.094904517439)*x[1]**o+(-0.940528164353)*x[1]
        ref[0]=(-0.235055063806)*(1+2.*(dim-1.)/(o+1.))+(-0.464020743688)*dim
        ref[1]=(0.0136354243251)*(1+2.*(dim-1.)/(o+1.))+(-1.22440912671)*dim
      else:
        arg[0]=(-0.506412384389)*x[0]**o+(-0.465794405394)*x[0]+(0.346977499982)*x[1]**o+(0.386181176987)*x[1]+(-0.476729572397)*x[2]**o+(-0.956945933508)*x[2]
        arg[1]=(0.267092807416)*x[0]**o+(-0.54211155802)*x[0]+(-0.27815411727)*x[1]**o+(-0.984542450102)*x[1]+(0.656389296076)*x[2]**o+(0.0395863971401)*x[2]
        ref[0]=(-0.636164456804)*(1+2.*(dim-1.)/(o+1.))+(-1.03655916191)*dim
        ref[1]=(0.645327986222)*(1+2.*(dim-1.)/(o+1.))+(-1.48706761098)*dim
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnBoundary_fromData_ContinuousFunction_rank2(self):
      """
      tests integral of rank 2 Data on the FunctionOnBoundary

      assumptions: ContinuousFunction(self.domain) exists
                   self.domain supports integral on FunctionOnBoundary
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnBoundary(self.domain)
      w=ContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5),w)
      ref=numpy.zeros((4, 5),numpy.float_)
      if dim==2:
        arg[0,0]=(0.0607636892262)*x[0]**o+(0.740866012668)*x[0]+(0.145223136928)*x[1]**o+(-0.738387773526)*x[1]
        arg[0,1]=(0.0292598484511)*x[0]**o+(-0.189208290889)*x[0]+(0.902021405165)*x[1]**o+(0.423106545559)*x[1]
        arg[0,2]=(0.519658217226)*x[0]**o+(-0.768406018136)*x[0]+(-0.189532527097)*x[1]**o+(0.256417429833)*x[1]
        arg[0,3]=(-0.565327663481)*x[0]**o+(0.876762003673)*x[0]+(-0.243804758063)*x[1]**o+(0.399476036454)*x[1]
        arg[0,4]=(0.0632252409818)*x[0]**o+(0.0951130662538)*x[0]+(0.185662425763)*x[1]**o+(0.907446482506)*x[1]
        arg[1,0]=(-0.430504584794)*x[0]**o+(0.864814314023)*x[0]+(0.47460095501)*x[1]**o+(-0.599679142188)*x[1]
        arg[1,1]=(0.699618175298)*x[0]**o+(-0.561782605517)*x[0]+(-0.304250865015)*x[1]**o+(-0.341836776795)*x[1]
        arg[1,2]=(0.856956767316)*x[0]**o+(0.610026903213)*x[0]+(-0.666749954599)*x[1]**o+(0.885388885234)*x[1]
        arg[1,3]=(0.699377007631)*x[0]**o+(0.699906750287)*x[0]+(0.486690669414)*x[1]**o+(0.167425901549)*x[1]
        arg[1,4]=(0.466701415896)*x[0]**o+(-0.0719752759126)*x[0]+(0.795662047062)*x[1]**o+(0.979862503926)*x[1]
        arg[2,0]=(-0.912218732612)*x[0]**o+(0.950990472894)*x[0]+(-0.527999708902)*x[1]**o+(0.695735593043)*x[1]
        arg[2,1]=(-0.223468270048)*x[0]**o+(-0.882565147673)*x[0]+(0.750516681645)*x[1]**o+(-0.652794590093)*x[1]
        arg[2,2]=(0.897646682412)*x[0]**o+(-0.815401214058)*x[0]+(0.629680741327)*x[1]**o+(-0.9716988156)*x[1]
        arg[2,3]=(0.279739517524)*x[0]**o+(-0.0859797206016)*x[0]+(-0.0454325622757)*x[1]**o+(0.301214753868)*x[1]
        arg[2,4]=(-0.358056827092)*x[0]**o+(0.254207762658)*x[0]+(0.831499219864)*x[1]**o+(0.0930388028811)*x[1]
        arg[3,0]=(0.0761520020047)*x[0]**o+(0.650696394486)*x[0]+(-0.132369660678)*x[1]**o+(-0.321220385971)*x[1]
        arg[3,1]=(0.238715231585)*x[0]**o+(-0.297219391083)*x[0]+(-0.15446552614)*x[1]**o+(0.860560615056)*x[1]
        arg[3,2]=(0.708864754538)*x[0]**o+(-0.221186702958)*x[0]+(-0.478127729891)*x[1]**o+(0.0309604412212)*x[1]
        arg[3,3]=(0.362909366479)*x[0]**o+(0.382279106813)*x[0]+(0.827905062086)*x[1]**o+(-0.303048355949)*x[1]
        arg[3,4]=(0.573736946844)*x[0]**o+(0.645059764766)*x[0]+(0.301945383678)*x[1]**o+(-0.103083649994)*x[1]
        ref[0,0]=(0.205986826154)*(1+2.*(dim-1.)/(o+1.))+(0.0024782391428)*dim
        ref[0,1]=(0.931281253617)*(1+2.*(dim-1.)/(o+1.))+(0.23389825467)*dim
        ref[0,2]=(0.330125690129)*(1+2.*(dim-1.)/(o+1.))+(-0.511988588303)*dim
        ref[0,3]=(-0.809132421544)*(1+2.*(dim-1.)/(o+1.))+(1.27623804013)*dim
        ref[0,4]=(0.248887666745)*(1+2.*(dim-1.)/(o+1.))+(1.00255954876)*dim
        ref[1,0]=(0.044096370216)*(1+2.*(dim-1.)/(o+1.))+(0.265135171835)*dim
        ref[1,1]=(0.395367310283)*(1+2.*(dim-1.)/(o+1.))+(-0.903619382312)*dim
        ref[1,2]=(0.190206812717)*(1+2.*(dim-1.)/(o+1.))+(1.49541578845)*dim
        ref[1,3]=(1.18606767705)*(1+2.*(dim-1.)/(o+1.))+(0.867332651836)*dim
        ref[1,4]=(1.26236346296)*(1+2.*(dim-1.)/(o+1.))+(0.907887228013)*dim
        ref[2,0]=(-1.44021844151)*(1+2.*(dim-1.)/(o+1.))+(1.64672606594)*dim
        ref[2,1]=(0.527048411596)*(1+2.*(dim-1.)/(o+1.))+(-1.53535973777)*dim
        ref[2,2]=(1.52732742374)*(1+2.*(dim-1.)/(o+1.))+(-1.78710002966)*dim
        ref[2,3]=(0.234306955249)*(1+2.*(dim-1.)/(o+1.))+(0.215235033266)*dim
        ref[2,4]=(0.473442392772)*(1+2.*(dim-1.)/(o+1.))+(0.34724656554)*dim
        ref[3,0]=(-0.0562176586729)*(1+2.*(dim-1.)/(o+1.))+(0.329476008516)*dim
        ref[3,1]=(0.0842497054449)*(1+2.*(dim-1.)/(o+1.))+(0.563341223973)*dim
        ref[3,2]=(0.230737024647)*(1+2.*(dim-1.)/(o+1.))+(-0.190226261737)*dim
        ref[3,3]=(1.19081442856)*(1+2.*(dim-1.)/(o+1.))+(0.079230750864)*dim
        ref[3,4]=(0.875682330522)*(1+2.*(dim-1.)/(o+1.))+(0.541976114772)*dim
      else:
        arg[0,0]=(0.0640108068236)*x[0]**o+(0.0322830166043)*x[0]+(-0.703691335255)*x[1]**o+(-0.185228770711)*x[1]+(-0.738714166313)*x[2]**o+(-0.423576647077)*x[2]
        arg[0,1]=(-0.738341351945)*x[0]**o+(0.396306395436)*x[0]+(-0.369974848568)*x[1]**o+(0.750933208783)*x[1]+(0.554069869127)*x[2]**o+(0.634275493051)*x[2]
        arg[0,2]=(-0.259193248708)*x[0]**o+(-0.900007449583)*x[0]+(0.894379436915)*x[1]**o+(0.997814320752)*x[1]+(0.233192116622)*x[2]**o+(0.986616259649)*x[2]
        arg[0,3]=(0.381480250594)*x[0]**o+(0.734823219057)*x[0]+(0.569912440169)*x[1]**o+(-0.441084504695)*x[1]+(0.312539701584)*x[2]**o+(-0.3733004974)*x[2]
        arg[0,4]=(-0.779059807622)*x[0]**o+(0.538777922322)*x[0]+(-0.0695879211612)*x[1]**o+(-0.204862216499)*x[1]+(-0.0109698525672)*x[2]**o+(0.981875117617)*x[2]
        arg[1,0]=(-0.415857630044)*x[0]**o+(-0.356607866913)*x[0]+(-0.628854326903)*x[1]**o+(0.0401233333505)*x[1]+(0.687470169579)*x[2]**o+(-0.565349753114)*x[2]
        arg[1,1]=(-0.438364910402)*x[0]**o+(0.498078194367)*x[0]+(-0.120729597948)*x[1]**o+(-0.622076117713)*x[1]+(0.995130029172)*x[2]**o+(-0.7234601064)*x[2]
        arg[1,2]=(-0.957869585558)*x[0]**o+(0.374693219824)*x[0]+(0.0355192120972)*x[1]**o+(-0.516770140651)*x[1]+(0.521887953791)*x[2]**o+(0.00771564236502)*x[2]
        arg[1,3]=(-0.0495580194062)*x[0]**o+(-0.183895326318)*x[0]+(-0.55304086303)*x[1]**o+(0.920204449879)*x[1]+(0.143165855157)*x[2]**o+(-0.773684620462)*x[2]
        arg[1,4]=(0.795913131087)*x[0]**o+(0.653826129199)*x[0]+(-0.585562760813)*x[1]**o+(0.274083108918)*x[1]+(-0.165275654116)*x[2]**o+(0.358678758212)*x[2]
        arg[2,0]=(-0.0699685857843)*x[0]**o+(-0.42922435931)*x[0]+(-0.0494308981938)*x[1]**o+(-0.956597829176)*x[1]+(-0.912941305307)*x[2]**o+(0.214777332562)*x[2]
        arg[2,1]=(-0.900983338173)*x[0]**o+(-0.779729884378)*x[0]+(-0.860100306508)*x[1]**o+(0.14211537281)*x[1]+(-0.984439938285)*x[2]**o+(-0.549987779842)*x[2]
        arg[2,2]=(0.801489512389)*x[0]**o+(0.896468923826)*x[0]+(-0.155737972887)*x[1]**o+(0.639864330477)*x[1]+(0.798598705038)*x[2]**o+(-0.863019041525)*x[2]
        arg[2,3]=(-0.453612136504)*x[0]**o+(0.522662491688)*x[0]+(-0.463058076486)*x[1]**o+(0.398018809499)*x[1]+(-0.612044961611)*x[2]**o+(-0.481951883217)*x[2]
        arg[2,4]=(-0.0739336519425)*x[0]**o+(-0.241753740154)*x[0]+(0.0176292058898)*x[1]**o+(0.435641132809)*x[1]+(-0.0592828584976)*x[2]**o+(0.430115620801)*x[2]
        arg[3,0]=(0.604784358152)*x[0]**o+(-0.817214179503)*x[0]+(-0.423104272008)*x[1]**o+(0.287787881773)*x[1]+(0.678036469241)*x[2]**o+(0.578347129563)*x[2]
        arg[3,1]=(0.362390778598)*x[0]**o+(-0.389066580119)*x[0]+(-0.934087580396)*x[1]**o+(0.907587485226)*x[1]+(-0.902083043406)*x[2]**o+(0.904707538314)*x[2]
        arg[3,2]=(0.862463207465)*x[0]**o+(0.904923219336)*x[0]+(0.421361696042)*x[1]**o+(-0.779329196171)*x[1]+(0.059521522394)*x[2]**o+(-0.744510067387)*x[2]
        arg[3,3]=(0.412029226851)*x[0]**o+(-0.649730143511)*x[0]+(0.579628017001)*x[1]**o+(-0.949180147129)*x[1]+(-0.073016492309)*x[2]**o+(0.666554170757)*x[2]
        arg[3,4]=(-0.700401396504)*x[0]**o+(-0.89194120812)*x[0]+(0.592387042213)*x[1]**o+(-0.906632787891)*x[1]+(-0.486072860812)*x[2]**o+(-0.505386176523)*x[2]
        ref[0,0]=(-1.37839469474)*(1+2.*(dim-1.)/(o+1.))+(-0.576522401184)*dim
        ref[0,1]=(-0.554246331386)*(1+2.*(dim-1.)/(o+1.))+(1.78151509727)*dim
        ref[0,2]=(0.868378304829)*(1+2.*(dim-1.)/(o+1.))+(1.08442313082)*dim
        ref[0,3]=(1.26393239235)*(1+2.*(dim-1.)/(o+1.))+(-0.0795617830385)*dim
        ref[0,4]=(-0.85961758135)*(1+2.*(dim-1.)/(o+1.))+(1.31579082344)*dim
        ref[1,0]=(-0.357241787368)*(1+2.*(dim-1.)/(o+1.))+(-0.881834286676)*dim
        ref[1,1]=(0.436035520821)*(1+2.*(dim-1.)/(o+1.))+(-0.847458029746)*dim
        ref[1,2]=(-0.40046241967)*(1+2.*(dim-1.)/(o+1.))+(-0.134361278462)*dim
        ref[1,3]=(-0.459433027279)*(1+2.*(dim-1.)/(o+1.))+(-0.0373754969011)*dim
        ref[1,4]=(0.0450747161574)*(1+2.*(dim-1.)/(o+1.))+(1.28658799633)*dim
        ref[2,0]=(-1.03234078929)*(1+2.*(dim-1.)/(o+1.))+(-1.17104485592)*dim
        ref[2,1]=(-2.74552358297)*(1+2.*(dim-1.)/(o+1.))+(-1.18760229141)*dim
        ref[2,2]=(1.44435024454)*(1+2.*(dim-1.)/(o+1.))+(0.673314212777)*dim
        ref[2,3]=(-1.5287151746)*(1+2.*(dim-1.)/(o+1.))+(0.43872941797)*dim
        ref[2,4]=(-0.11558730455)*(1+2.*(dim-1.)/(o+1.))+(0.624003013456)*dim
        ref[3,0]=(0.859716555385)*(1+2.*(dim-1.)/(o+1.))+(0.0489208318321)*dim
        ref[3,1]=(-1.4737798452)*(1+2.*(dim-1.)/(o+1.))+(1.42322844342)*dim
        ref[3,2]=(1.3433464259)*(1+2.*(dim-1.)/(o+1.))+(-0.618916044222)*dim
        ref[3,3]=(0.918640751542)*(1+2.*(dim-1.)/(o+1.))+(-0.932356119883)*dim
        ref[3,4]=(-0.594087215103)*(1+2.*(dim-1.)/(o+1.))+(-2.30396017253)*dim
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnBoundary_fromData_ContinuousFunction_rank3(self):
      """
      tests integral of rank 3 Data on the FunctionOnBoundary

      assumptions: ContinuousFunction(self.domain) exists
                   self.domain supports integral on FunctionOnBoundary
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnBoundary(self.domain)
      w=ContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(6, 2, 2),w)
      ref=numpy.zeros((6, 2, 2),numpy.float_)
      if dim==2:
        arg[0,0,0]=(0.662707930283)*x[0]**o+(0.625512077969)*x[0]+(-0.222208840922)*x[1]**o+(0.374614317804)*x[1]
        arg[0,0,1]=(0.685952252808)*x[0]**o+(-0.88219802797)*x[0]+(-0.948918831634)*x[1]**o+(-0.00615202896776)*x[1]
        arg[0,1,0]=(-0.555231206657)*x[0]**o+(0.997112425959)*x[0]+(-0.186247870511)*x[1]**o+(-0.0658892046229)*x[1]
        arg[0,1,1]=(-0.735288285178)*x[0]**o+(0.941234848614)*x[0]+(-0.307761737256)*x[1]**o+(0.933293069433)*x[1]
        arg[1,0,0]=(-0.618191337825)*x[0]**o+(-0.12930895166)*x[0]+(0.76213325083)*x[1]**o+(-0.744049403156)*x[1]
        arg[1,0,1]=(-0.793412570981)*x[0]**o+(0.533175323401)*x[0]+(-0.87398581932)*x[1]**o+(-0.279651754725)*x[1]
        arg[1,1,0]=(-0.0654077907694)*x[0]**o+(0.239040458488)*x[0]+(0.730639927837)*x[1]**o+(0.805020533329)*x[1]
        arg[1,1,1]=(-0.560195748542)*x[0]**o+(-0.265389430135)*x[0]+(-0.295142324741)*x[1]**o+(-0.929699771024)*x[1]
        arg[2,0,0]=(0.935760865577)*x[0]**o+(-0.332458843131)*x[0]+(0.257302695187)*x[1]**o+(-0.347547927033)*x[1]
        arg[2,0,1]=(-0.245770653276)*x[0]**o+(0.126945988409)*x[0]+(0.302793434995)*x[1]**o+(-0.997483200172)*x[1]
        arg[2,1,0]=(0.523494364951)*x[0]**o+(-0.323782740449)*x[0]+(-0.563752593454)*x[1]**o+(-0.358325752838)*x[1]
        arg[2,1,1]=(-0.506330114515)*x[0]**o+(-0.733234050567)*x[0]+(0.734774667778)*x[1]**o+(0.795264566776)*x[1]
        arg[3,0,0]=(-0.21258295235)*x[0]**o+(-0.929331330718)*x[0]+(-0.546061847742)*x[1]**o+(0.0613931491065)*x[1]
        arg[3,0,1]=(-0.357847946656)*x[0]**o+(-0.133795684508)*x[0]+(-0.796646028688)*x[1]**o+(0.875795801429)*x[1]
        arg[3,1,0]=(0.923717804039)*x[0]**o+(-0.176266633621)*x[0]+(-0.605892867534)*x[1]**o+(0.633377612365)*x[1]
        arg[3,1,1]=(0.729025508129)*x[0]**o+(0.412034955257)*x[0]+(0.0933054864485)*x[1]**o+(0.386019085532)*x[1]
        arg[4,0,0]=(-0.0705524918008)*x[0]**o+(-0.869605321304)*x[0]+(0.244351267755)*x[1]**o+(0.898883388102)*x[1]
        arg[4,0,1]=(-0.453297333327)*x[0]**o+(-0.67824185123)*x[0]+(0.0497322840245)*x[1]**o+(0.581677879069)*x[1]
        arg[4,1,0]=(0.343975006358)*x[0]**o+(0.724084357103)*x[0]+(-0.186292335362)*x[1]**o+(-0.257997075018)*x[1]
        arg[4,1,1]=(-0.389731923798)*x[0]**o+(0.0949155578616)*x[0]+(-0.298428138752)*x[1]**o+(0.819510069666)*x[1]
        arg[5,0,0]=(0.827670169513)*x[0]**o+(-0.584607978668)*x[0]+(0.669056948658)*x[1]**o+(-0.393158541985)*x[1]
        arg[5,0,1]=(-0.471551968206)*x[0]**o+(-0.339964831396)*x[0]+(0.354554393554)*x[1]**o+(-0.334773591957)*x[1]
        arg[5,1,0]=(0.113562904864)*x[0]**o+(-0.775045505123)*x[0]+(0.783044874931)*x[1]**o+(0.52265843873)*x[1]
        arg[5,1,1]=(-0.115476036205)*x[0]**o+(-0.290822012728)*x[0]+(-0.398510477854)*x[1]**o+(0.398737205542)*x[1]
        ref[0,0,0]=(0.440499089362)*(1+2.*(dim-1.)/(o+1.))+(1.00012639577)*dim
        ref[0,0,1]=(-0.262966578826)*(1+2.*(dim-1.)/(o+1.))+(-0.888350056938)*dim
        ref[0,1,0]=(-0.741479077167)*(1+2.*(dim-1.)/(o+1.))+(0.931223221336)*dim
        ref[0,1,1]=(-1.04305002243)*(1+2.*(dim-1.)/(o+1.))+(1.87452791805)*dim
        ref[1,0,0]=(0.143941913005)*(1+2.*(dim-1.)/(o+1.))+(-0.873358354816)*dim
        ref[1,0,1]=(-1.6673983903)*(1+2.*(dim-1.)/(o+1.))+(0.253523568676)*dim
        ref[1,1,0]=(0.665232137068)*(1+2.*(dim-1.)/(o+1.))+(1.04406099182)*dim
        ref[1,1,1]=(-0.855338073283)*(1+2.*(dim-1.)/(o+1.))+(-1.19508920116)*dim
        ref[2,0,0]=(1.19306356076)*(1+2.*(dim-1.)/(o+1.))+(-0.680006770164)*dim
        ref[2,0,1]=(0.0570227817187)*(1+2.*(dim-1.)/(o+1.))+(-0.870537211764)*dim
        ref[2,1,0]=(-0.0402582285028)*(1+2.*(dim-1.)/(o+1.))+(-0.682108493287)*dim
        ref[2,1,1]=(0.228444553263)*(1+2.*(dim-1.)/(o+1.))+(0.0620305162089)*dim
        ref[3,0,0]=(-0.758644800093)*(1+2.*(dim-1.)/(o+1.))+(-0.867938181611)*dim
        ref[3,0,1]=(-1.15449397534)*(1+2.*(dim-1.)/(o+1.))+(0.742000116921)*dim
        ref[3,1,0]=(0.317824936505)*(1+2.*(dim-1.)/(o+1.))+(0.457110978744)*dim
        ref[3,1,1]=(0.822330994578)*(1+2.*(dim-1.)/(o+1.))+(0.798054040789)*dim
        ref[4,0,0]=(0.173798775954)*(1+2.*(dim-1.)/(o+1.))+(0.0292780667983)*dim
        ref[4,0,1]=(-0.403565049303)*(1+2.*(dim-1.)/(o+1.))+(-0.0965639721601)*dim
        ref[4,1,0]=(0.157682670995)*(1+2.*(dim-1.)/(o+1.))+(0.466087282085)*dim
        ref[4,1,1]=(-0.68816006255)*(1+2.*(dim-1.)/(o+1.))+(0.914425627528)*dim
        ref[5,0,0]=(1.49672711817)*(1+2.*(dim-1.)/(o+1.))+(-0.977766520654)*dim
        ref[5,0,1]=(-0.116997574651)*(1+2.*(dim-1.)/(o+1.))+(-0.674738423353)*dim
        ref[5,1,0]=(0.896607779795)*(1+2.*(dim-1.)/(o+1.))+(-0.252387066394)*dim
        ref[5,1,1]=(-0.513986514058)*(1+2.*(dim-1.)/(o+1.))+(0.107915192814)*dim
      else:
        arg[0,0,0]=(-0.418671467289)*x[0]**o+(0.498387164484)*x[0]+(0.208896299314)*x[1]**o+(-0.643038678771)*x[1]+(-0.85475466439)*x[2]**o+(0.59586368009)*x[2]
        arg[0,0,1]=(0.149229891253)*x[0]**o+(0.229836799955)*x[0]+(0.557946253001)*x[1]**o+(-0.434223003739)*x[1]+(-0.250979831671)*x[2]**o+(0.762582320269)*x[2]
        arg[0,1,0]=(-0.950902166094)*x[0]**o+(-0.312913424444)*x[0]+(-0.573887973428)*x[1]**o+(-0.372485255069)*x[1]+(-0.18228472562)*x[2]**o+(-0.566679093436)*x[2]
        arg[0,1,1]=(-0.191769651257)*x[0]**o+(-0.344504306057)*x[0]+(0.355926399715)*x[1]**o+(-0.721341598375)*x[1]+(0.362401271289)*x[2]**o+(0.601351093654)*x[2]
        arg[1,0,0]=(-0.375572370375)*x[0]**o+(-0.532723139333)*x[0]+(-0.738166755424)*x[1]**o+(0.785506757134)*x[1]+(0.192667976423)*x[2]**o+(-0.06534893872)*x[2]
        arg[1,0,1]=(-0.353750712453)*x[0]**o+(0.401872866285)*x[0]+(-0.747783222746)*x[1]**o+(0.663116752574)*x[1]+(-0.162954382466)*x[2]**o+(0.318112656687)*x[2]
        arg[1,1,0]=(-0.711345828912)*x[0]**o+(0.764701438614)*x[0]+(-0.232576947703)*x[1]**o+(-0.50541791533)*x[1]+(0.604603682696)*x[2]**o+(0.498828792648)*x[2]
        arg[1,1,1]=(-0.72474879447)*x[0]**o+(0.888266999605)*x[0]+(0.639903282531)*x[1]**o+(-0.777435844156)*x[1]+(-0.0814738680964)*x[2]**o+(-0.0135286853237)*x[2]
        arg[2,0,0]=(0.839793308998)*x[0]**o+(0.374051014226)*x[0]+(0.594653957466)*x[1]**o+(-0.962352782345)*x[1]+(0.888632514336)*x[2]**o+(0.491195067464)*x[2]
        arg[2,0,1]=(-0.827615661598)*x[0]**o+(0.675227575488)*x[0]+(-0.93068580658)*x[1]**o+(0.694434337954)*x[1]+(0.3583179263)*x[2]**o+(0.775147111187)*x[2]
        arg[2,1,0]=(-0.772656999491)*x[0]**o+(-0.464476706435)*x[0]+(0.569832694977)*x[1]**o+(0.500517676542)*x[1]+(-0.272153062279)*x[2]**o+(-0.398436498506)*x[2]
        arg[2,1,1]=(-0.3065198839)*x[0]**o+(-0.454353346315)*x[0]+(-0.539448433869)*x[1]**o+(0.776950547078)*x[1]+(-0.987178147402)*x[2]**o+(0.368935549126)*x[2]
        arg[3,0,0]=(-0.516131622594)*x[0]**o+(-0.680922252714)*x[0]+(0.304951059501)*x[1]**o+(-0.236361364379)*x[1]+(-0.37983306233)*x[2]**o+(0.693426329316)*x[2]
        arg[3,0,1]=(0.85094983056)*x[0]**o+(-0.356782375867)*x[0]+(0.362116478047)*x[1]**o+(0.269126381543)*x[1]+(0.950254707197)*x[2]**o+(0.592028802301)*x[2]
        arg[3,1,0]=(-0.17721437307)*x[0]**o+(-0.127432965766)*x[0]+(-0.754975923735)*x[1]**o+(0.755938625221)*x[1]+(0.776393089777)*x[2]**o+(0.913028629874)*x[2]
        arg[3,1,1]=(-0.779840479711)*x[0]**o+(0.432392796542)*x[0]+(0.0457670482312)*x[1]**o+(0.493363120993)*x[1]+(-0.290308698356)*x[2]**o+(0.263565329052)*x[2]
        arg[4,0,0]=(-0.594371032991)*x[0]**o+(0.673573089348)*x[0]+(0.0301238273919)*x[1]**o+(-0.900766544194)*x[1]+(0.937125579182)*x[2]**o+(-0.360512287398)*x[2]
        arg[4,0,1]=(-0.187561420369)*x[0]**o+(-0.727058667406)*x[0]+(0.643904893485)*x[1]**o+(0.547358609515)*x[1]+(-0.810276689361)*x[2]**o+(0.347967967049)*x[2]
        arg[4,1,0]=(0.0261452776327)*x[0]**o+(0.630125453041)*x[0]+(0.884509651558)*x[1]**o+(-0.580251885597)*x[1]+(0.141608918047)*x[2]**o+(0.417140237181)*x[2]
        arg[4,1,1]=(0.943149763428)*x[0]**o+(0.0414421588123)*x[0]+(-0.453944232191)*x[1]**o+(0.817963360891)*x[1]+(-0.0132464046434)*x[2]**o+(-0.247024507218)*x[2]
        arg[5,0,0]=(0.993356103007)*x[0]**o+(-0.604991443085)*x[0]+(-0.0141942193536)*x[1]**o+(-0.467982818195)*x[1]+(0.907608332852)*x[2]**o+(-0.918494340721)*x[2]
        arg[5,0,1]=(0.782079295393)*x[0]**o+(0.0125642236587)*x[0]+(-0.512764292821)*x[1]**o+(-0.0699705327379)*x[1]+(0.141341885735)*x[2]**o+(0.420508183565)*x[2]
        arg[5,1,0]=(-0.835665486427)*x[0]**o+(-0.536850925449)*x[0]+(-0.0958499285885)*x[1]**o+(-0.541119926637)*x[1]+(0.341886033336)*x[2]**o+(-0.772823771771)*x[2]
        arg[5,1,1]=(-0.391737530391)*x[0]**o+(-0.160045087084)*x[0]+(-0.984350699381)*x[1]**o+(0.303782477449)*x[1]+(-0.279906615513)*x[2]**o+(-0.690872289955)*x[2]
        ref[0,0,0]=(-1.06452983237)*(1+2.*(dim-1.)/(o+1.))+(0.451212165803)*dim
        ref[0,0,1]=(0.456196312583)*(1+2.*(dim-1.)/(o+1.))+(0.558196116486)*dim
        ref[0,1,0]=(-1.70707486514)*(1+2.*(dim-1.)/(o+1.))+(-1.25207777295)*dim
        ref[0,1,1]=(0.526558019747)*(1+2.*(dim-1.)/(o+1.))+(-0.464494810779)*dim
        ref[1,0,0]=(-0.921071149376)*(1+2.*(dim-1.)/(o+1.))+(0.187434679081)*dim
        ref[1,0,1]=(-1.26448831766)*(1+2.*(dim-1.)/(o+1.))+(1.38310227555)*dim
        ref[1,1,0]=(-0.339319093919)*(1+2.*(dim-1.)/(o+1.))+(0.758112315932)*dim
        ref[1,1,1]=(-0.166319380035)*(1+2.*(dim-1.)/(o+1.))+(0.0973024701251)*dim
        ref[2,0,0]=(2.3230797808)*(1+2.*(dim-1.)/(o+1.))+(-0.0971067006546)*dim
        ref[2,0,1]=(-1.39998354188)*(1+2.*(dim-1.)/(o+1.))+(2.14480902463)*dim
        ref[2,1,0]=(-0.474977366793)*(1+2.*(dim-1.)/(o+1.))+(-0.362395528399)*dim
        ref[2,1,1]=(-1.83314646517)*(1+2.*(dim-1.)/(o+1.))+(0.69153274989)*dim
        ref[3,0,0]=(-0.591013625423)*(1+2.*(dim-1.)/(o+1.))+(-0.223857287777)*dim
        ref[3,0,1]=(2.1633210158)*(1+2.*(dim-1.)/(o+1.))+(0.504372807977)*dim
        ref[3,1,0]=(-0.155797207029)*(1+2.*(dim-1.)/(o+1.))+(1.54153428933)*dim
        ref[3,1,1]=(-1.02438212984)*(1+2.*(dim-1.)/(o+1.))+(1.18932124659)*dim
        ref[4,0,0]=(0.372878373583)*(1+2.*(dim-1.)/(o+1.))+(-0.587705742243)*dim
        ref[4,0,1]=(-0.353933216245)*(1+2.*(dim-1.)/(o+1.))+(0.168267909158)*dim
        ref[4,1,0]=(1.05226384724)*(1+2.*(dim-1.)/(o+1.))+(0.467013804625)*dim
        ref[4,1,1]=(0.475959126593)*(1+2.*(dim-1.)/(o+1.))+(0.612381012486)*dim
        ref[5,0,0]=(1.88677021651)*(1+2.*(dim-1.)/(o+1.))+(-1.991468602)*dim
        ref[5,0,1]=(0.410656888307)*(1+2.*(dim-1.)/(o+1.))+(0.363101874486)*dim
        ref[5,1,0]=(-0.58962938168)*(1+2.*(dim-1.)/(o+1.))+(-1.85079462386)*dim
        ref[5,1,1]=(-1.65599484529)*(1+2.*(dim-1.)/(o+1.))+(-0.54713489959)*dim
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnBoundary_fromData_ContinuousFunction_rank4(self):
      """
      tests integral of rank 4 Data on the FunctionOnBoundary

      assumptions: ContinuousFunction(self.domain) exists
                   self.domain supports integral on FunctionOnBoundary
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnBoundary(self.domain)
      w=ContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5, 3, 2),w)
      ref=numpy.zeros((4, 5, 3, 2),numpy.float_)
      if dim==2:
        arg[0,0,0,0]=(0.606358724075)*x[0]**o+(-0.6326664687)*x[0]+(-0.521101256804)*x[1]**o+(0.702113497865)*x[1]
        arg[0,0,0,1]=(0.528315511137)*x[0]**o+(-0.116566989978)*x[0]+(-0.350847384045)*x[1]**o+(-0.581988878345)*x[1]
        arg[0,0,1,0]=(0.565076347032)*x[0]**o+(0.752532782601)*x[0]+(0.00706163321089)*x[1]**o+(-0.89265244135)*x[1]
        arg[0,0,1,1]=(-0.342136497951)*x[0]**o+(0.753153993866)*x[0]+(-0.199643944066)*x[1]**o+(0.995411752312)*x[1]
        arg[0,0,2,0]=(0.621283193015)*x[0]**o+(0.591186318333)*x[0]+(0.880169584588)*x[1]**o+(-0.11839698224)*x[1]
        arg[0,0,2,1]=(-0.831745199475)*x[0]**o+(0.293656166596)*x[0]+(0.0717745661823)*x[1]**o+(0.827680695965)*x[1]
        arg[0,1,0,0]=(0.686864676707)*x[0]**o+(-0.6552976723)*x[0]+(-0.65589888943)*x[1]**o+(-0.878054891784)*x[1]
        arg[0,1,0,1]=(-0.43359086203)*x[0]**o+(0.723819586663)*x[0]+(0.0741799035834)*x[1]**o+(0.13005305188)*x[1]
        arg[0,1,1,0]=(-0.0605346127445)*x[0]**o+(-0.717729061173)*x[0]+(0.825539660518)*x[1]**o+(0.873033938391)*x[1]
        arg[0,1,1,1]=(-0.286483607332)*x[0]**o+(-0.577255191498)*x[0]+(-0.328362280683)*x[1]**o+(0.499078653745)*x[1]
        arg[0,1,2,0]=(-0.334132932545)*x[0]**o+(0.462461110225)*x[0]+(-0.932311565873)*x[1]**o+(0.934683608666)*x[1]
        arg[0,1,2,1]=(-0.383203086272)*x[0]**o+(0.261035781624)*x[0]+(-0.203936125231)*x[1]**o+(0.657584310755)*x[1]
        arg[0,2,0,0]=(-0.637642146695)*x[0]**o+(0.80524493224)*x[0]+(-0.139137135459)*x[1]**o+(0.947434162736)*x[1]
        arg[0,2,0,1]=(-0.0738996477961)*x[0]**o+(0.0663360260869)*x[0]+(-0.599113718918)*x[1]**o+(-0.349514990363)*x[1]
        arg[0,2,1,0]=(-0.0539211250603)*x[0]**o+(0.262567947534)*x[0]+(0.115386681137)*x[1]**o+(-0.68179682822)*x[1]
        arg[0,2,1,1]=(-0.770114869627)*x[0]**o+(-0.388068404619)*x[0]+(-0.712914716647)*x[1]**o+(-0.810536809871)*x[1]
        arg[0,2,2,0]=(0.274425895455)*x[0]**o+(0.338467327995)*x[0]+(-0.78223636363)*x[1]**o+(0.406149090351)*x[1]
        arg[0,2,2,1]=(-0.748672361572)*x[0]**o+(0.508311939723)*x[0]+(0.875797107098)*x[1]**o+(0.0992795348726)*x[1]
        arg[0,3,0,0]=(-0.547350688209)*x[0]**o+(0.671683747412)*x[0]+(-0.0954212965969)*x[1]**o+(0.983487887304)*x[1]
        arg[0,3,0,1]=(0.626104338883)*x[0]**o+(-0.825311449834)*x[0]+(-0.422521614246)*x[1]**o+(0.304208397592)*x[1]
        arg[0,3,1,0]=(0.698777992012)*x[0]**o+(0.00291670095874)*x[0]+(0.327527809646)*x[1]**o+(0.283289333886)*x[1]
        arg[0,3,1,1]=(-0.0197648150597)*x[0]**o+(-0.902860814776)*x[0]+(0.663961228673)*x[1]**o+(0.675952386365)*x[1]
        arg[0,3,2,0]=(0.286747814345)*x[0]**o+(-0.0641946364041)*x[0]+(0.812019228115)*x[1]**o+(-0.225698609001)*x[1]
        arg[0,3,2,1]=(0.879776322226)*x[0]**o+(-0.251856143047)*x[0]+(-0.252403834757)*x[1]**o+(0.650790406323)*x[1]
        arg[0,4,0,0]=(-0.58782572388)*x[0]**o+(-0.144122228965)*x[0]+(0.521707348991)*x[1]**o+(-0.0460840614459)*x[1]
        arg[0,4,0,1]=(-0.777497837497)*x[0]**o+(-0.229011563089)*x[0]+(0.754657080443)*x[1]**o+(-0.690675521089)*x[1]
        arg[0,4,1,0]=(0.351528557097)*x[0]**o+(0.00192986530026)*x[0]+(-0.414849484269)*x[1]**o+(-0.250205659505)*x[1]
        arg[0,4,1,1]=(-0.814037796843)*x[0]**o+(0.00188512986627)*x[0]+(-0.810799710998)*x[1]**o+(-0.114406174711)*x[1]
        arg[0,4,2,0]=(-0.755511212878)*x[0]**o+(-0.502206951212)*x[0]+(0.555738333365)*x[1]**o+(-0.220359547926)*x[1]
        arg[0,4,2,1]=(0.247108894345)*x[0]**o+(0.479239235148)*x[0]+(0.663958827816)*x[1]**o+(0.220938477625)*x[1]
        arg[1,0,0,0]=(0.818774052595)*x[0]**o+(-0.651584728881)*x[0]+(-0.780674092348)*x[1]**o+(-0.397277502644)*x[1]
        arg[1,0,0,1]=(0.482101508249)*x[0]**o+(-0.523618202277)*x[0]+(0.806406836488)*x[1]**o+(-0.883738297664)*x[1]
        arg[1,0,1,0]=(0.272517667346)*x[0]**o+(-0.700461889638)*x[0]+(0.280736354342)*x[1]**o+(0.182983247089)*x[1]
        arg[1,0,1,1]=(0.566589111451)*x[0]**o+(-0.876744547538)*x[0]+(-0.934741703566)*x[1]**o+(-0.562770667492)*x[1]
        arg[1,0,2,0]=(-0.954128714623)*x[0]**o+(0.115699770127)*x[0]+(-0.580663262426)*x[1]**o+(-0.451619694161)*x[1]
        arg[1,0,2,1]=(0.268605150185)*x[0]**o+(-0.233415174179)*x[0]+(0.477143801066)*x[1]**o+(0.443696994244)*x[1]
        arg[1,1,0,0]=(-0.00240846686485)*x[0]**o+(0.776336851643)*x[0]+(0.488314964094)*x[1]**o+(-0.840196476788)*x[1]
        arg[1,1,0,1]=(-0.511732155638)*x[0]**o+(0.528782039485)*x[0]+(0.51274502262)*x[1]**o+(-0.700213610705)*x[1]
        arg[1,1,1,0]=(0.300359000499)*x[0]**o+(0.405101876401)*x[0]+(-0.47022583299)*x[1]**o+(-0.977634916267)*x[1]
        arg[1,1,1,1]=(-0.494482318948)*x[0]**o+(0.254112728167)*x[0]+(-0.541672595343)*x[1]**o+(-0.270970064522)*x[1]
        arg[1,1,2,0]=(-0.586624294233)*x[0]**o+(0.629601920904)*x[0]+(-0.525092252936)*x[1]**o+(-0.44611635978)*x[1]
        arg[1,1,2,1]=(0.716927085139)*x[0]**o+(0.700193925267)*x[0]+(0.0624921446647)*x[1]**o+(0.109485129877)*x[1]
        arg[1,2,0,0]=(0.325729307898)*x[0]**o+(0.773432634097)*x[0]+(-0.675612597356)*x[1]**o+(-0.0554913733692)*x[1]
        arg[1,2,0,1]=(-0.381606382262)*x[0]**o+(-0.689040382854)*x[0]+(-0.580191973398)*x[1]**o+(0.951689147387)*x[1]
        arg[1,2,1,0]=(-0.527642403744)*x[0]**o+(-0.936915203927)*x[0]+(0.631733997289)*x[1]**o+(-0.715975680015)*x[1]
        arg[1,2,1,1]=(0.0886898373323)*x[0]**o+(-0.0943320873915)*x[0]+(0.470668750443)*x[1]**o+(0.773832960093)*x[1]
        arg[1,2,2,0]=(-0.727720290197)*x[0]**o+(0.422119803173)*x[0]+(0.705882951779)*x[1]**o+(-0.350801865244)*x[1]
        arg[1,2,2,1]=(0.879335475407)*x[0]**o+(-0.0307168471577)*x[0]+(0.458578151544)*x[1]**o+(0.788796576339)*x[1]
        arg[1,3,0,0]=(0.182143508146)*x[0]**o+(-0.680787176288)*x[0]+(0.626099489145)*x[1]**o+(-0.771540378408)*x[1]
        arg[1,3,0,1]=(-0.578341681282)*x[0]**o+(0.856407179963)*x[0]+(0.525319425356)*x[1]**o+(0.345651851179)*x[1]
        arg[1,3,1,0]=(-0.316652838058)*x[0]**o+(0.652409898929)*x[0]+(-0.82123442746)*x[1]**o+(-0.136868558081)*x[1]
        arg[1,3,1,1]=(-0.755530700082)*x[0]**o+(0.354429731374)*x[0]+(-0.14330437792)*x[1]**o+(-0.221454495186)*x[1]
        arg[1,3,2,0]=(-0.520261483769)*x[0]**o+(0.0873707179205)*x[0]+(-0.882995356682)*x[1]**o+(-0.773921687804)*x[1]
        arg[1,3,2,1]=(0.404278952718)*x[0]**o+(0.651166017764)*x[0]+(-0.496695253747)*x[1]**o+(0.625671754182)*x[1]
        arg[1,4,0,0]=(-0.730831539043)*x[0]**o+(0.219467989041)*x[0]+(-0.853784008431)*x[1]**o+(0.274799624897)*x[1]
        arg[1,4,0,1]=(0.893313801855)*x[0]**o+(-0.539687651773)*x[0]+(-0.411582029239)*x[1]**o+(-0.293521483805)*x[1]
        arg[1,4,1,0]=(0.484878049594)*x[0]**o+(0.218930293134)*x[0]+(-0.138362733141)*x[1]**o+(-0.663241354177)*x[1]
        arg[1,4,1,1]=(-0.608410207857)*x[0]**o+(-0.0921006917331)*x[0]+(-0.885705130565)*x[1]**o+(0.103724764404)*x[1]
        arg[1,4,2,0]=(0.150946260009)*x[0]**o+(-0.601834829203)*x[0]+(0.0141091402169)*x[1]**o+(-0.98316187826)*x[1]
        arg[1,4,2,1]=(0.98764918249)*x[0]**o+(-0.841155365687)*x[0]+(0.185525084432)*x[1]**o+(-0.613421841538)*x[1]
        arg[2,0,0,0]=(0.485337970037)*x[0]**o+(-0.898602108837)*x[0]+(0.847812629133)*x[1]**o+(-0.23522051021)*x[1]
        arg[2,0,0,1]=(0.835641029366)*x[0]**o+(-0.0524661277338)*x[0]+(0.41697147238)*x[1]**o+(-0.622813671618)*x[1]
        arg[2,0,1,0]=(0.505181088966)*x[0]**o+(-0.870141755006)*x[0]+(0.451139050304)*x[1]**o+(0.957790881903)*x[1]
        arg[2,0,1,1]=(0.822955028051)*x[0]**o+(-0.103028201548)*x[0]+(-0.604614127349)*x[1]**o+(-0.655334670762)*x[1]
        arg[2,0,2,0]=(-0.166647802545)*x[0]**o+(-0.466885186725)*x[0]+(0.0782333081578)*x[1]**o+(0.814414769115)*x[1]
        arg[2,0,2,1]=(-0.980310280968)*x[0]**o+(-0.884306329761)*x[0]+(-0.895550319566)*x[1]**o+(0.160657671233)*x[1]
        arg[2,1,0,0]=(-0.846405435483)*x[0]**o+(-0.625465093616)*x[0]+(-0.0789995975268)*x[1]**o+(-0.440180761086)*x[1]
        arg[2,1,0,1]=(-0.949855662419)*x[0]**o+(-0.013707742372)*x[0]+(0.0397729185783)*x[1]**o+(-0.836054083952)*x[1]
        arg[2,1,1,0]=(0.427246928002)*x[0]**o+(0.835369889909)*x[0]+(0.925300799558)*x[1]**o+(-0.328266770631)*x[1]
        arg[2,1,1,1]=(-0.36143365614)*x[0]**o+(-0.744107540819)*x[0]+(-0.358554131639)*x[1]**o+(-0.562674543946)*x[1]
        arg[2,1,2,0]=(0.60580594275)*x[0]**o+(0.644796993923)*x[0]+(-0.682407021927)*x[1]**o+(0.938016908061)*x[1]
        arg[2,1,2,1]=(-0.934393274317)*x[0]**o+(-0.718420272664)*x[0]+(0.0357039032065)*x[1]**o+(0.446387943344)*x[1]
        arg[2,2,0,0]=(-0.329105411737)*x[0]**o+(0.588935500072)*x[0]+(-0.833362483432)*x[1]**o+(0.629996961074)*x[1]
        arg[2,2,0,1]=(0.723977530226)*x[0]**o+(0.0698430385616)*x[0]+(-0.503761809156)*x[1]**o+(0.907472279839)*x[1]
        arg[2,2,1,0]=(-0.767545510793)*x[0]**o+(0.615094783682)*x[0]+(0.894594933703)*x[1]**o+(0.922357219049)*x[1]
        arg[2,2,1,1]=(-0.155739055576)*x[0]**o+(0.646142299486)*x[0]+(0.865932681534)*x[1]**o+(-0.126887380871)*x[1]
        arg[2,2,2,0]=(-0.629534320112)*x[0]**o+(-0.0904366301836)*x[0]+(0.193019806731)*x[1]**o+(-0.866080160466)*x[1]
        arg[2,2,2,1]=(-0.714527761053)*x[0]**o+(-0.813712879405)*x[0]+(-0.291734238438)*x[1]**o+(0.232675536415)*x[1]
        arg[2,3,0,0]=(-0.447474292879)*x[0]**o+(-0.639046887788)*x[0]+(0.159616683647)*x[1]**o+(0.564127141623)*x[1]
        arg[2,3,0,1]=(-0.608953936619)*x[0]**o+(0.273179484103)*x[0]+(0.42693156927)*x[1]**o+(-0.983331075647)*x[1]
        arg[2,3,1,0]=(0.855859483301)*x[0]**o+(-0.432495503925)*x[0]+(-0.67928203769)*x[1]**o+(-0.856541006702)*x[1]
        arg[2,3,1,1]=(-0.581185594204)*x[0]**o+(0.947720323693)*x[0]+(-0.520212852628)*x[1]**o+(-0.37563221594)*x[1]
        arg[2,3,2,0]=(0.351358492535)*x[0]**o+(0.367210525846)*x[0]+(0.97870179383)*x[1]**o+(-0.199559052968)*x[1]
        arg[2,3,2,1]=(0.797294518541)*x[0]**o+(0.546397721552)*x[0]+(0.374121851646)*x[1]**o+(0.704436689798)*x[1]
        arg[2,4,0,0]=(-0.756047775149)*x[0]**o+(-0.705403684825)*x[0]+(-0.662421213376)*x[1]**o+(0.390069314234)*x[1]
        arg[2,4,0,1]=(-0.658842889869)*x[0]**o+(0.766273769822)*x[0]+(-0.779201775962)*x[1]**o+(-0.968670270742)*x[1]
        arg[2,4,1,0]=(-0.336220564628)*x[0]**o+(-0.134745604319)*x[0]+(0.391939888428)*x[1]**o+(-0.113113999792)*x[1]
        arg[2,4,1,1]=(-0.8345363851)*x[0]**o+(0.339223929857)*x[0]+(0.628403353717)*x[1]**o+(0.799364167182)*x[1]
        arg[2,4,2,0]=(-0.898050446701)*x[0]**o+(-0.666373288962)*x[0]+(-0.620778021659)*x[1]**o+(-0.990359850049)*x[1]
        arg[2,4,2,1]=(-0.933931027675)*x[0]**o+(-0.119077281033)*x[0]+(0.557313533556)*x[1]**o+(0.160336158304)*x[1]
        arg[3,0,0,0]=(0.972462636314)*x[0]**o+(0.607913740671)*x[0]+(0.490168539257)*x[1]**o+(-0.12515921391)*x[1]
        arg[3,0,0,1]=(0.856370886333)*x[0]**o+(-0.143225009869)*x[0]+(-0.330381845603)*x[1]**o+(-0.271958059018)*x[1]
        arg[3,0,1,0]=(0.86742129112)*x[0]**o+(-0.0259756341823)*x[0]+(-0.0137047849942)*x[1]**o+(-0.0165444161044)*x[1]
        arg[3,0,1,1]=(-0.445996036941)*x[0]**o+(0.247936136111)*x[0]+(-0.381484740124)*x[1]**o+(0.836678417006)*x[1]
        arg[3,0,2,0]=(0.161387456771)*x[0]**o+(0.687533753311)*x[0]+(-0.463845417603)*x[1]**o+(0.665023823118)*x[1]
        arg[3,0,2,1]=(0.243410612742)*x[0]**o+(-0.52431675734)*x[0]+(0.75630192488)*x[1]**o+(-0.703601557317)*x[1]
        arg[3,1,0,0]=(0.793530689056)*x[0]**o+(0.814806726176)*x[0]+(0.582051068232)*x[1]**o+(0.692432923027)*x[1]
        arg[3,1,0,1]=(-0.283506909465)*x[0]**o+(0.21438128538)*x[0]+(-0.327476122861)*x[1]**o+(0.990178870269)*x[1]
        arg[3,1,1,0]=(-0.117386491827)*x[0]**o+(0.747771272481)*x[0]+(-0.439767754004)*x[1]**o+(-0.905218338639)*x[1]
        arg[3,1,1,1]=(-0.544805647706)*x[0]**o+(-0.0484033735345)*x[0]+(0.861443439156)*x[1]**o+(-0.992086899129)*x[1]
        arg[3,1,2,0]=(0.596208599112)*x[0]**o+(-0.276136075021)*x[0]+(0.079604120535)*x[1]**o+(0.124232758054)*x[1]
        arg[3,1,2,1]=(0.809847641841)*x[0]**o+(0.956041370087)*x[0]+(0.65224420189)*x[1]**o+(0.565327743771)*x[1]
        arg[3,2,0,0]=(-0.264599479107)*x[0]**o+(-0.276726556903)*x[0]+(0.577911819561)*x[1]**o+(-0.656157381756)*x[1]
        arg[3,2,0,1]=(-0.0636193103458)*x[0]**o+(0.941482878452)*x[0]+(0.877581030494)*x[1]**o+(-0.254911385455)*x[1]
        arg[3,2,1,0]=(-0.28083673633)*x[0]**o+(0.849095032935)*x[0]+(0.6825284068)*x[1]**o+(0.445605171749)*x[1]
        arg[3,2,1,1]=(-0.729331034619)*x[0]**o+(0.036400895394)*x[0]+(0.365733359214)*x[1]**o+(0.0305324156609)*x[1]
        arg[3,2,2,0]=(-0.70932226428)*x[0]**o+(0.310455571562)*x[0]+(-0.101305643894)*x[1]**o+(0.346873071627)*x[1]
        arg[3,2,2,1]=(0.712053329488)*x[0]**o+(-0.845042122758)*x[0]+(0.909517101069)*x[1]**o+(-0.163109058477)*x[1]
        arg[3,3,0,0]=(0.451744730688)*x[0]**o+(0.308756227526)*x[0]+(0.13963307493)*x[1]**o+(-0.945789640023)*x[1]
        arg[3,3,0,1]=(-0.852609785834)*x[0]**o+(-0.821513824231)*x[0]+(-0.544690113726)*x[1]**o+(0.335547843106)*x[1]
        arg[3,3,1,0]=(0.337600834432)*x[0]**o+(-0.170181468776)*x[0]+(-0.302415145845)*x[1]**o+(0.985425450212)*x[1]
        arg[3,3,1,1]=(-0.638242591048)*x[0]**o+(-0.0221442303187)*x[0]+(0.421311946694)*x[1]**o+(-0.43495081316)*x[1]
        arg[3,3,2,0]=(-0.734168622382)*x[0]**o+(-0.889300572069)*x[0]+(0.808161580285)*x[1]**o+(-0.526348332439)*x[1]
        arg[3,3,2,1]=(0.998918324168)*x[0]**o+(-0.953729666152)*x[0]+(0.665101882861)*x[1]**o+(-0.581118145725)*x[1]
        arg[3,4,0,0]=(-0.251226888098)*x[0]**o+(-0.711615999514)*x[0]+(0.291022154479)*x[1]**o+(0.0772744442604)*x[1]
        arg[3,4,0,1]=(0.162379537091)*x[0]**o+(0.363750962136)*x[0]+(0.635943839083)*x[1]**o+(0.513856953389)*x[1]
        arg[3,4,1,0]=(-0.99052354199)*x[0]**o+(0.289202668813)*x[0]+(-0.759078503012)*x[1]**o+(-0.588801682903)*x[1]
        arg[3,4,1,1]=(-0.942278699306)*x[0]**o+(0.476003512559)*x[0]+(0.873920293292)*x[1]**o+(0.867413991424)*x[1]
        arg[3,4,2,0]=(-0.602386702281)*x[0]**o+(0.694432164577)*x[0]+(0.367008950869)*x[1]**o+(-0.803331549449)*x[1]
        arg[3,4,2,1]=(-0.00204465287104)*x[0]**o+(-0.697572466685)*x[0]+(-0.84755978799)*x[1]**o+(-0.758946055726)*x[1]
        ref[0,0,0,0]=(0.0852574672711)*(1+2.*(dim-1.)/(o+1.))+(0.069447029165)*dim
        ref[0,0,0,1]=(0.177468127092)*(1+2.*(dim-1.)/(o+1.))+(-0.698555868323)*dim
        ref[0,0,1,0]=(0.572137980243)*(1+2.*(dim-1.)/(o+1.))+(-0.140119658749)*dim
        ref[0,0,1,1]=(-0.541780442017)*(1+2.*(dim-1.)/(o+1.))+(1.74856574618)*dim
        ref[0,0,2,0]=(1.5014527776)*(1+2.*(dim-1.)/(o+1.))+(0.472789336092)*dim
        ref[0,0,2,1]=(-0.759970633293)*(1+2.*(dim-1.)/(o+1.))+(1.12133686256)*dim
        ref[0,1,0,0]=(0.030965787278)*(1+2.*(dim-1.)/(o+1.))+(-1.53335256408)*dim
        ref[0,1,0,1]=(-0.359410958446)*(1+2.*(dim-1.)/(o+1.))+(0.853872638544)*dim
        ref[0,1,1,0]=(0.765005047774)*(1+2.*(dim-1.)/(o+1.))+(0.155304877218)*dim
        ref[0,1,1,1]=(-0.614845888015)*(1+2.*(dim-1.)/(o+1.))+(-0.0781765377528)*dim
        ref[0,1,2,0]=(-1.26644449842)*(1+2.*(dim-1.)/(o+1.))+(1.39714471889)*dim
        ref[0,1,2,1]=(-0.587139211502)*(1+2.*(dim-1.)/(o+1.))+(0.918620092379)*dim
        ref[0,2,0,0]=(-0.776779282155)*(1+2.*(dim-1.)/(o+1.))+(1.75267909498)*dim
        ref[0,2,0,1]=(-0.673013366714)*(1+2.*(dim-1.)/(o+1.))+(-0.283178964276)*dim
        ref[0,2,1,0]=(0.0614655560763)*(1+2.*(dim-1.)/(o+1.))+(-0.419228880686)*dim
        ref[0,2,1,1]=(-1.48302958627)*(1+2.*(dim-1.)/(o+1.))+(-1.19860521449)*dim
        ref[0,2,2,0]=(-0.507810468175)*(1+2.*(dim-1.)/(o+1.))+(0.744616418346)*dim
        ref[0,2,2,1]=(0.127124745526)*(1+2.*(dim-1.)/(o+1.))+(0.607591474596)*dim
        ref[0,3,0,0]=(-0.642771984806)*(1+2.*(dim-1.)/(o+1.))+(1.65517163472)*dim
        ref[0,3,0,1]=(0.203582724637)*(1+2.*(dim-1.)/(o+1.))+(-0.521103052242)*dim
        ref[0,3,1,0]=(1.02630580166)*(1+2.*(dim-1.)/(o+1.))+(0.286206034845)*dim
        ref[0,3,1,1]=(0.644196413613)*(1+2.*(dim-1.)/(o+1.))+(-0.226908428411)*dim
        ref[0,3,2,0]=(1.09876704246)*(1+2.*(dim-1.)/(o+1.))+(-0.289893245405)*dim
        ref[0,3,2,1]=(0.627372487469)*(1+2.*(dim-1.)/(o+1.))+(0.398934263276)*dim
        ref[0,4,0,0]=(-0.0661183748894)*(1+2.*(dim-1.)/(o+1.))+(-0.190206290411)*dim
        ref[0,4,0,1]=(-0.022840757054)*(1+2.*(dim-1.)/(o+1.))+(-0.919687084178)*dim
        ref[0,4,1,0]=(-0.063320927172)*(1+2.*(dim-1.)/(o+1.))+(-0.248275794205)*dim
        ref[0,4,1,1]=(-1.62483750784)*(1+2.*(dim-1.)/(o+1.))+(-0.112521044845)*dim
        ref[0,4,2,0]=(-0.199772879513)*(1+2.*(dim-1.)/(o+1.))+(-0.722566499138)*dim
        ref[0,4,2,1]=(0.911067722161)*(1+2.*(dim-1.)/(o+1.))+(0.700177712774)*dim
        ref[1,0,0,0]=(0.0380999602468)*(1+2.*(dim-1.)/(o+1.))+(-1.04886223152)*dim
        ref[1,0,0,1]=(1.28850834474)*(1+2.*(dim-1.)/(o+1.))+(-1.40735649994)*dim
        ref[1,0,1,0]=(0.553254021688)*(1+2.*(dim-1.)/(o+1.))+(-0.517478642549)*dim
        ref[1,0,1,1]=(-0.368152592115)*(1+2.*(dim-1.)/(o+1.))+(-1.43951521503)*dim
        ref[1,0,2,0]=(-1.53479197705)*(1+2.*(dim-1.)/(o+1.))+(-0.335919924034)*dim
        ref[1,0,2,1]=(0.745748951251)*(1+2.*(dim-1.)/(o+1.))+(0.210281820065)*dim
        ref[1,1,0,0]=(0.485906497229)*(1+2.*(dim-1.)/(o+1.))+(-0.0638596251453)*dim
        ref[1,1,0,1]=(0.00101286698233)*(1+2.*(dim-1.)/(o+1.))+(-0.17143157122)*dim
        ref[1,1,1,0]=(-0.169866832491)*(1+2.*(dim-1.)/(o+1.))+(-0.572533039866)*dim
        ref[1,1,1,1]=(-1.03615491429)*(1+2.*(dim-1.)/(o+1.))+(-0.016857336355)*dim
        ref[1,1,2,0]=(-1.11171654717)*(1+2.*(dim-1.)/(o+1.))+(0.183485561125)*dim
        ref[1,1,2,1]=(0.779419229804)*(1+2.*(dim-1.)/(o+1.))+(0.809679055143)*dim
        ref[1,2,0,0]=(-0.349883289458)*(1+2.*(dim-1.)/(o+1.))+(0.717941260728)*dim
        ref[1,2,0,1]=(-0.96179835566)*(1+2.*(dim-1.)/(o+1.))+(0.262648764533)*dim
        ref[1,2,1,0]=(0.104091593546)*(1+2.*(dim-1.)/(o+1.))+(-1.65289088394)*dim
        ref[1,2,1,1]=(0.559358587775)*(1+2.*(dim-1.)/(o+1.))+(0.679500872701)*dim
        ref[1,2,2,0]=(-0.0218373384179)*(1+2.*(dim-1.)/(o+1.))+(0.0713179379289)*dim
        ref[1,2,2,1]=(1.33791362695)*(1+2.*(dim-1.)/(o+1.))+(0.758079729181)*dim
        ref[1,3,0,0]=(0.808242997292)*(1+2.*(dim-1.)/(o+1.))+(-1.4523275547)*dim
        ref[1,3,0,1]=(-0.0530222559262)*(1+2.*(dim-1.)/(o+1.))+(1.20205903114)*dim
        ref[1,3,1,0]=(-1.13788726552)*(1+2.*(dim-1.)/(o+1.))+(0.515541340848)*dim
        ref[1,3,1,1]=(-0.898835078002)*(1+2.*(dim-1.)/(o+1.))+(0.132975236188)*dim
        ref[1,3,2,0]=(-1.40325684045)*(1+2.*(dim-1.)/(o+1.))+(-0.686550969884)*dim
        ref[1,3,2,1]=(-0.0924163010291)*(1+2.*(dim-1.)/(o+1.))+(1.27683777195)*dim
        ref[1,4,0,0]=(-1.58461554747)*(1+2.*(dim-1.)/(o+1.))+(0.494267613937)*dim
        ref[1,4,0,1]=(0.481731772616)*(1+2.*(dim-1.)/(o+1.))+(-0.833209135578)*dim
        ref[1,4,1,0]=(0.346515316453)*(1+2.*(dim-1.)/(o+1.))+(-0.444311061043)*dim
        ref[1,4,1,1]=(-1.49411533842)*(1+2.*(dim-1.)/(o+1.))+(0.0116240726713)*dim
        ref[1,4,2,0]=(0.165055400226)*(1+2.*(dim-1.)/(o+1.))+(-1.58499670746)*dim
        ref[1,4,2,1]=(1.17317426692)*(1+2.*(dim-1.)/(o+1.))+(-1.45457720722)*dim
        ref[2,0,0,0]=(1.33315059917)*(1+2.*(dim-1.)/(o+1.))+(-1.13382261905)*dim
        ref[2,0,0,1]=(1.25261250175)*(1+2.*(dim-1.)/(o+1.))+(-0.675279799352)*dim
        ref[2,0,1,0]=(0.95632013927)*(1+2.*(dim-1.)/(o+1.))+(0.0876491268976)*dim
        ref[2,0,1,1]=(0.218340900702)*(1+2.*(dim-1.)/(o+1.))+(-0.75836287231)*dim
        ref[2,0,2,0]=(-0.0884144943875)*(1+2.*(dim-1.)/(o+1.))+(0.34752958239)*dim
        ref[2,0,2,1]=(-1.87586060053)*(1+2.*(dim-1.)/(o+1.))+(-0.723648658528)*dim
        ref[2,1,0,0]=(-0.92540503301)*(1+2.*(dim-1.)/(o+1.))+(-1.0656458547)*dim
        ref[2,1,0,1]=(-0.910082743841)*(1+2.*(dim-1.)/(o+1.))+(-0.849761826324)*dim
        ref[2,1,1,0]=(1.35254772756)*(1+2.*(dim-1.)/(o+1.))+(0.507103119278)*dim
        ref[2,1,1,1]=(-0.719987787779)*(1+2.*(dim-1.)/(o+1.))+(-1.30678208477)*dim
        ref[2,1,2,0]=(-0.0766010791773)*(1+2.*(dim-1.)/(o+1.))+(1.58281390198)*dim
        ref[2,1,2,1]=(-0.89868937111)*(1+2.*(dim-1.)/(o+1.))+(-0.27203232932)*dim
        ref[2,2,0,0]=(-1.16246789517)*(1+2.*(dim-1.)/(o+1.))+(1.21893246115)*dim
        ref[2,2,0,1]=(0.22021572107)*(1+2.*(dim-1.)/(o+1.))+(0.977315318401)*dim
        ref[2,2,1,0]=(0.12704942291)*(1+2.*(dim-1.)/(o+1.))+(1.53745200273)*dim
        ref[2,2,1,1]=(0.710193625958)*(1+2.*(dim-1.)/(o+1.))+(0.519254918615)*dim
        ref[2,2,2,0]=(-0.43651451338)*(1+2.*(dim-1.)/(o+1.))+(-0.956516790649)*dim
        ref[2,2,2,1]=(-1.00626199949)*(1+2.*(dim-1.)/(o+1.))+(-0.581037342991)*dim
        ref[2,3,0,0]=(-0.287857609232)*(1+2.*(dim-1.)/(o+1.))+(-0.0749197461649)*dim
        ref[2,3,0,1]=(-0.182022367349)*(1+2.*(dim-1.)/(o+1.))+(-0.710151591543)*dim
        ref[2,3,1,0]=(0.176577445611)*(1+2.*(dim-1.)/(o+1.))+(-1.28903651063)*dim
        ref[2,3,1,1]=(-1.10139844683)*(1+2.*(dim-1.)/(o+1.))+(0.572088107754)*dim
        ref[2,3,2,0]=(1.33006028637)*(1+2.*(dim-1.)/(o+1.))+(0.167651472878)*dim
        ref[2,3,2,1]=(1.17141637019)*(1+2.*(dim-1.)/(o+1.))+(1.25083441135)*dim
        ref[2,4,0,0]=(-1.41846898852)*(1+2.*(dim-1.)/(o+1.))+(-0.315334370591)*dim
        ref[2,4,0,1]=(-1.43804466583)*(1+2.*(dim-1.)/(o+1.))+(-0.20239650092)*dim
        ref[2,4,1,0]=(0.0557193237997)*(1+2.*(dim-1.)/(o+1.))+(-0.247859604111)*dim
        ref[2,4,1,1]=(-0.206133031383)*(1+2.*(dim-1.)/(o+1.))+(1.13858809704)*dim
        ref[2,4,2,0]=(-1.51882846836)*(1+2.*(dim-1.)/(o+1.))+(-1.65673313901)*dim
        ref[2,4,2,1]=(-0.376617494119)*(1+2.*(dim-1.)/(o+1.))+(0.041258877271)*dim
        ref[3,0,0,0]=(1.46263117557)*(1+2.*(dim-1.)/(o+1.))+(0.482754526761)*dim
        ref[3,0,0,1]=(0.52598904073)*(1+2.*(dim-1.)/(o+1.))+(-0.415183068887)*dim
        ref[3,0,1,0]=(0.853716506126)*(1+2.*(dim-1.)/(o+1.))+(-0.0425200502867)*dim
        ref[3,0,1,1]=(-0.827480777065)*(1+2.*(dim-1.)/(o+1.))+(1.08461455312)*dim
        ref[3,0,2,0]=(-0.302457960832)*(1+2.*(dim-1.)/(o+1.))+(1.35255757643)*dim
        ref[3,0,2,1]=(0.999712537623)*(1+2.*(dim-1.)/(o+1.))+(-1.22791831466)*dim
        ref[3,1,0,0]=(1.37558175729)*(1+2.*(dim-1.)/(o+1.))+(1.5072396492)*dim
        ref[3,1,0,1]=(-0.610983032326)*(1+2.*(dim-1.)/(o+1.))+(1.20456015565)*dim
        ref[3,1,1,0]=(-0.557154245831)*(1+2.*(dim-1.)/(o+1.))+(-0.157447066158)*dim
        ref[3,1,1,1]=(0.316637791451)*(1+2.*(dim-1.)/(o+1.))+(-1.04049027266)*dim
        ref[3,1,2,0]=(0.675812719647)*(1+2.*(dim-1.)/(o+1.))+(-0.151903316967)*dim
        ref[3,1,2,1]=(1.46209184373)*(1+2.*(dim-1.)/(o+1.))+(1.52136911386)*dim
        ref[3,2,0,0]=(0.313312340454)*(1+2.*(dim-1.)/(o+1.))+(-0.93288393866)*dim
        ref[3,2,0,1]=(0.813961720148)*(1+2.*(dim-1.)/(o+1.))+(0.686571492997)*dim
        ref[3,2,1,0]=(0.40169167047)*(1+2.*(dim-1.)/(o+1.))+(1.29470020468)*dim
        ref[3,2,1,1]=(-0.363597675405)*(1+2.*(dim-1.)/(o+1.))+(0.066933311055)*dim
        ref[3,2,2,0]=(-0.810627908174)*(1+2.*(dim-1.)/(o+1.))+(0.657328643189)*dim
        ref[3,2,2,1]=(1.62157043056)*(1+2.*(dim-1.)/(o+1.))+(-1.00815118124)*dim
        ref[3,3,0,0]=(0.591377805617)*(1+2.*(dim-1.)/(o+1.))+(-0.637033412497)*dim
        ref[3,3,0,1]=(-1.39729989956)*(1+2.*(dim-1.)/(o+1.))+(-0.485965981125)*dim
        ref[3,3,1,0]=(0.0351856885862)*(1+2.*(dim-1.)/(o+1.))+(0.815243981437)*dim
        ref[3,3,1,1]=(-0.216930644354)*(1+2.*(dim-1.)/(o+1.))+(-0.457095043478)*dim
        ref[3,3,2,0]=(0.0739929579029)*(1+2.*(dim-1.)/(o+1.))+(-1.41564890451)*dim
        ref[3,3,2,1]=(1.66402020703)*(1+2.*(dim-1.)/(o+1.))+(-1.53484781188)*dim
        ref[3,4,0,0]=(0.0397952663815)*(1+2.*(dim-1.)/(o+1.))+(-0.634341555254)*dim
        ref[3,4,0,1]=(0.798323376174)*(1+2.*(dim-1.)/(o+1.))+(0.877607915525)*dim
        ref[3,4,1,0]=(-1.749602045)*(1+2.*(dim-1.)/(o+1.))+(-0.29959901409)*dim
        ref[3,4,1,1]=(-0.0683584060144)*(1+2.*(dim-1.)/(o+1.))+(1.34341750398)*dim
        ref[3,4,2,0]=(-0.235377751413)*(1+2.*(dim-1.)/(o+1.))+(-0.108899384872)*dim
        ref[3,4,2,1]=(-0.849604440861)*(1+2.*(dim-1.)/(o+1.))+(-1.45651852241)*dim
      else:
        arg[0,0,0,0]=(-0.229219068739)*x[0]**o+(0.293241016924)*x[0]+(-0.2730739542)*x[1]**o+(0.980438707356)*x[1]+(-0.712463216466)*x[2]**o+(0.201036317883)*x[2]
        arg[0,0,0,1]=(0.626271947422)*x[0]**o+(0.379293769575)*x[0]+(-0.476378592342)*x[1]**o+(-0.0089906260934)*x[1]+(-0.433154385811)*x[2]**o+(0.126766298748)*x[2]
        arg[0,0,1,0]=(-0.517950283035)*x[0]**o+(-0.219780986066)*x[0]+(0.345048580545)*x[1]**o+(0.116833851128)*x[1]+(-0.790641523314)*x[2]**o+(0.40506306417)*x[2]
        arg[0,0,1,1]=(0.910924420538)*x[0]**o+(0.281102231596)*x[0]+(0.115454208619)*x[1]**o+(0.164254056171)*x[1]+(0.44960060103)*x[2]**o+(-0.14238069165)*x[2]
        arg[0,0,2,0]=(0.86195808217)*x[0]**o+(0.775425956811)*x[0]+(0.214559021523)*x[1]**o+(0.400663063038)*x[1]+(0.995663608954)*x[2]**o+(0.228658543268)*x[2]
        arg[0,0,2,1]=(0.9025917492)*x[0]**o+(0.643205662573)*x[0]+(-0.304382459664)*x[1]**o+(-0.653256441532)*x[1]+(0.591142459807)*x[2]**o+(-0.888899030043)*x[2]
        arg[0,1,0,0]=(-0.874720784838)*x[0]**o+(-0.961456282274)*x[0]+(-0.954972993457)*x[1]**o+(0.00963939708746)*x[1]+(-0.884848381277)*x[2]**o+(-0.859924591622)*x[2]
        arg[0,1,0,1]=(0.133592941787)*x[0]**o+(0.12505415793)*x[0]+(-0.872197297559)*x[1]**o+(-0.666936976734)*x[1]+(-0.540166108036)*x[2]**o+(0.739145002101)*x[2]
        arg[0,1,1,0]=(0.543825266564)*x[0]**o+(0.167854064952)*x[0]+(-0.876426395953)*x[1]**o+(0.317538118917)*x[1]+(0.0454531201303)*x[2]**o+(0.822731343286)*x[2]
        arg[0,1,1,1]=(0.294557050513)*x[0]**o+(0.811698886899)*x[0]+(0.906213580147)*x[1]**o+(0.47895749797)*x[1]+(-0.937617554883)*x[2]**o+(0.171907458148)*x[2]
        arg[0,1,2,0]=(-0.843310949654)*x[0]**o+(0.766452648675)*x[0]+(-0.801151193917)*x[1]**o+(-0.373068887299)*x[1]+(-0.908043738284)*x[2]**o+(0.742278324563)*x[2]
        arg[0,1,2,1]=(-0.83259110525)*x[0]**o+(-0.303796015231)*x[0]+(0.153727560059)*x[1]**o+(-0.848393854019)*x[1]+(0.887795931916)*x[2]**o+(0.713486413074)*x[2]
        arg[0,2,0,0]=(-0.928603918182)*x[0]**o+(0.631213545526)*x[0]+(0.0149216054247)*x[1]**o+(0.136454313226)*x[1]+(-0.182080277504)*x[2]**o+(-0.898437757742)*x[2]
        arg[0,2,0,1]=(0.553844129645)*x[0]**o+(0.91439079798)*x[0]+(0.260424446678)*x[1]**o+(0.786629777779)*x[1]+(-0.412353876109)*x[2]**o+(0.496953000487)*x[2]
        arg[0,2,1,0]=(-0.48365422571)*x[0]**o+(-0.772252454664)*x[0]+(0.692592845143)*x[1]**o+(0.861992232175)*x[1]+(0.65169271692)*x[2]**o+(0.889175716524)*x[2]
        arg[0,2,1,1]=(-0.761086015475)*x[0]**o+(-0.616528617148)*x[0]+(-0.949399556926)*x[1]**o+(-0.783355074724)*x[1]+(-0.260470660608)*x[2]**o+(-0.854537542803)*x[2]
        arg[0,2,2,0]=(-0.812649439778)*x[0]**o+(-0.844862957855)*x[0]+(0.719928253134)*x[1]**o+(-0.892803903061)*x[1]+(-0.434753054503)*x[2]**o+(0.814188548451)*x[2]
        arg[0,2,2,1]=(-0.831106881173)*x[0]**o+(0.893305297647)*x[0]+(-0.351357795644)*x[1]**o+(0.0318574503478)*x[1]+(0.721896657873)*x[2]**o+(-0.75215838035)*x[2]
        arg[0,3,0,0]=(-0.868084492214)*x[0]**o+(-0.108398365838)*x[0]+(0.994384983885)*x[1]**o+(0.199445529335)*x[1]+(-0.570120483013)*x[2]**o+(0.992752456282)*x[2]
        arg[0,3,0,1]=(-0.304553428923)*x[0]**o+(-0.589461304318)*x[0]+(-0.570142565817)*x[1]**o+(-0.858530013475)*x[1]+(0.0690574539796)*x[2]**o+(-0.0321587525658)*x[2]
        arg[0,3,1,0]=(-0.729413026954)*x[0]**o+(-0.728248951404)*x[0]+(-0.568598599095)*x[1]**o+(0.0823012989572)*x[1]+(0.248614424573)*x[2]**o+(-0.116584878805)*x[2]
        arg[0,3,1,1]=(0.586846538084)*x[0]**o+(-0.406933529762)*x[0]+(0.839125682244)*x[1]**o+(0.237338453386)*x[1]+(-0.953101889056)*x[2]**o+(0.164147861616)*x[2]
        arg[0,3,2,0]=(-0.603895026116)*x[0]**o+(-0.401045478885)*x[0]+(-0.524739550832)*x[1]**o+(-0.077907095917)*x[1]+(0.535404409567)*x[2]**o+(0.745721528391)*x[2]
        arg[0,3,2,1]=(-0.189151084147)*x[0]**o+(-0.865927541126)*x[0]+(0.55265091657)*x[1]**o+(-0.592847311104)*x[1]+(-0.467509383222)*x[2]**o+(-0.779649276678)*x[2]
        arg[0,4,0,0]=(-0.564283606569)*x[0]**o+(-0.588029571592)*x[0]+(0.544948665159)*x[1]**o+(0.140885576219)*x[1]+(-0.333531842397)*x[2]**o+(-0.579095398292)*x[2]
        arg[0,4,0,1]=(-0.189127238455)*x[0]**o+(-0.693907127162)*x[0]+(0.40947793861)*x[1]**o+(-0.371098494308)*x[1]+(-0.189273002147)*x[2]**o+(-0.863169616331)*x[2]
        arg[0,4,1,0]=(-0.929838141145)*x[0]**o+(-0.0237583231579)*x[0]+(0.721069982341)*x[1]**o+(0.0378559913864)*x[1]+(-0.354729434476)*x[2]**o+(0.486256042969)*x[2]
        arg[0,4,1,1]=(0.349085173473)*x[0]**o+(0.398939355666)*x[0]+(-0.165772805366)*x[1]**o+(-0.532771334584)*x[1]+(-0.971868054413)*x[2]**o+(-0.38430185761)*x[2]
        arg[0,4,2,0]=(0.0330282399287)*x[0]**o+(0.157336131691)*x[0]+(-0.232739262685)*x[1]**o+(-0.280149712596)*x[1]+(0.374058430291)*x[2]**o+(0.594113015973)*x[2]
        arg[0,4,2,1]=(0.841740974112)*x[0]**o+(-0.454359680369)*x[0]+(-0.29970574726)*x[1]**o+(-0.234747232007)*x[1]+(0.590942390095)*x[2]**o+(-0.241807214234)*x[2]
        arg[1,0,0,0]=(0.815585848775)*x[0]**o+(-0.915020392674)*x[0]+(-0.0246530790781)*x[1]**o+(-0.783552074981)*x[1]+(0.539847778607)*x[2]**o+(-0.239345161181)*x[2]
        arg[1,0,0,1]=(-0.34212551354)*x[0]**o+(-0.231769108271)*x[0]+(-0.495709071836)*x[1]**o+(-0.301124823696)*x[1]+(-0.567450536966)*x[2]**o+(0.553640896859)*x[2]
        arg[1,0,1,0]=(0.507091065517)*x[0]**o+(-0.844317985101)*x[0]+(0.488126307787)*x[1]**o+(-0.624165922596)*x[1]+(0.00960189690918)*x[2]**o+(-0.35617062688)*x[2]
        arg[1,0,1,1]=(-0.219877421252)*x[0]**o+(-0.574843930095)*x[0]+(-0.846995760292)*x[1]**o+(-0.778496435041)*x[1]+(-0.817593097977)*x[2]**o+(0.387406829939)*x[2]
        arg[1,0,2,0]=(-0.774922277048)*x[0]**o+(-0.238152671014)*x[0]+(-0.913770877808)*x[1]**o+(-0.737305937434)*x[1]+(-0.13977056904)*x[2]**o+(-0.0156668131016)*x[2]
        arg[1,0,2,1]=(0.565993265911)*x[0]**o+(0.0690768496992)*x[0]+(-0.019845214223)*x[1]**o+(0.16253166701)*x[1]+(0.848117215885)*x[2]**o+(0.838075165243)*x[2]
        arg[1,1,0,0]=(0.871499694148)*x[0]**o+(-0.640851595157)*x[0]+(-0.170827284611)*x[1]**o+(-0.855550549988)*x[1]+(0.796086141249)*x[2]**o+(-0.214410672377)*x[2]
        arg[1,1,0,1]=(-0.903644604832)*x[0]**o+(0.0604780584681)*x[0]+(0.12800839523)*x[1]**o+(-0.594186752891)*x[1]+(0.415537334962)*x[2]**o+(0.122944611099)*x[2]
        arg[1,1,1,0]=(-0.611432775957)*x[0]**o+(0.391078622669)*x[0]+(-0.943091949273)*x[1]**o+(0.952739057317)*x[1]+(0.205104161722)*x[2]**o+(-0.946040001739)*x[2]
        arg[1,1,1,1]=(0.122941344579)*x[0]**o+(0.836981501711)*x[0]+(0.488970739079)*x[1]**o+(0.210062341907)*x[1]+(0.0264620393683)*x[2]**o+(-0.396386059797)*x[2]
        arg[1,1,2,0]=(-0.782658774038)*x[0]**o+(0.00611865020069)*x[0]+(0.806802875462)*x[1]**o+(-0.504323859583)*x[1]+(0.796276800331)*x[2]**o+(-0.792865116401)*x[2]
        arg[1,1,2,1]=(-0.249170081188)*x[0]**o+(-0.0234780571796)*x[0]+(-0.314017467872)*x[1]**o+(-0.0819148249592)*x[1]+(-0.71553368486)*x[2]**o+(0.763463207899)*x[2]
        arg[1,2,0,0]=(-0.722263632571)*x[0]**o+(-0.922870211147)*x[0]+(0.380794419995)*x[1]**o+(-0.76088532378)*x[1]+(0.417253071012)*x[2]**o+(0.152980049966)*x[2]
        arg[1,2,0,1]=(-0.667661997967)*x[0]**o+(-0.520570751147)*x[0]+(0.526368189149)*x[1]**o+(-0.0729376045288)*x[1]+(-0.434764684251)*x[2]**o+(0.124284939267)*x[2]
        arg[1,2,1,0]=(0.610868035553)*x[0]**o+(0.958644088455)*x[0]+(0.876029854699)*x[1]**o+(-0.278348787699)*x[1]+(-0.576695356283)*x[2]**o+(-0.302696556744)*x[2]
        arg[1,2,1,1]=(-0.224897853162)*x[0]**o+(0.89405397732)*x[0]+(-0.865281799513)*x[1]**o+(0.778866760053)*x[1]+(-0.939295291326)*x[2]**o+(0.256725768393)*x[2]
        arg[1,2,2,0]=(0.763785330202)*x[0]**o+(-0.92861265304)*x[0]+(-0.73488309202)*x[1]**o+(0.912711310857)*x[1]+(-0.359622626692)*x[2]**o+(-0.901261216611)*x[2]
        arg[1,2,2,1]=(-0.70687794303)*x[0]**o+(0.186497640523)*x[0]+(-0.587602932017)*x[1]**o+(0.439204174904)*x[1]+(0.0527573129765)*x[2]**o+(-0.121921819047)*x[2]
        arg[1,3,0,0]=(-0.576513727689)*x[0]**o+(0.228971758182)*x[0]+(0.282735129059)*x[1]**o+(-0.375270097915)*x[1]+(0.360087142563)*x[2]**o+(-0.0629946372113)*x[2]
        arg[1,3,0,1]=(-0.934375471561)*x[0]**o+(-0.710481522462)*x[0]+(-0.506438555546)*x[1]**o+(0.515278994703)*x[1]+(-0.789552111338)*x[2]**o+(0.556694887963)*x[2]
        arg[1,3,1,0]=(0.418081142778)*x[0]**o+(-0.990334755424)*x[0]+(0.465998147831)*x[1]**o+(0.85211792131)*x[1]+(0.64293315455)*x[2]**o+(0.418636667223)*x[2]
        arg[1,3,1,1]=(-0.192682640907)*x[0]**o+(-0.367176463246)*x[0]+(-0.512565915338)*x[1]**o+(0.431102518715)*x[1]+(-0.0213613616827)*x[2]**o+(-0.329868944656)*x[2]
        arg[1,3,2,0]=(-0.63277285658)*x[0]**o+(0.679819079734)*x[0]+(0.986520110022)*x[1]**o+(0.988046257154)*x[1]+(0.973419900492)*x[2]**o+(-0.177540534942)*x[2]
        arg[1,3,2,1]=(-0.660456888389)*x[0]**o+(0.546277948387)*x[0]+(-0.437065207396)*x[1]**o+(-0.310754727958)*x[1]+(-0.786950057282)*x[2]**o+(0.959526365177)*x[2]
        arg[1,4,0,0]=(0.764190976082)*x[0]**o+(0.970535029727)*x[0]+(0.389051006993)*x[1]**o+(-0.289346925704)*x[1]+(-0.386263606576)*x[2]**o+(0.624234063589)*x[2]
        arg[1,4,0,1]=(-0.982030683687)*x[0]**o+(-0.310500292581)*x[0]+(-0.559096285575)*x[1]**o+(-0.622541272039)*x[1]+(0.49924233081)*x[2]**o+(-0.219258681197)*x[2]
        arg[1,4,1,0]=(-0.124268932891)*x[0]**o+(0.102696501388)*x[0]+(-0.135123912862)*x[1]**o+(-0.960215132796)*x[1]+(-0.63171550556)*x[2]**o+(-0.493725285999)*x[2]
        arg[1,4,1,1]=(-0.397915436399)*x[0]**o+(-0.629822230732)*x[0]+(-0.65097233947)*x[1]**o+(0.915396047247)*x[1]+(0.835684562564)*x[2]**o+(-0.538096550078)*x[2]
        arg[1,4,2,0]=(-0.946429448216)*x[0]**o+(-0.310412085571)*x[0]+(-0.0984587257278)*x[1]**o+(-0.662090989911)*x[1]+(-0.424711481307)*x[2]**o+(0.00916879493127)*x[2]
        arg[1,4,2,1]=(0.919399549311)*x[0]**o+(-0.890110018315)*x[0]+(-0.685928475328)*x[1]**o+(-0.4731186395)*x[1]+(0.961887038718)*x[2]**o+(0.850957316131)*x[2]
        arg[2,0,0,0]=(0.834851801283)*x[0]**o+(-0.401572429371)*x[0]+(0.313309302528)*x[1]**o+(0.0368590314435)*x[1]+(-0.299807168778)*x[2]**o+(0.250599904818)*x[2]
        arg[2,0,0,1]=(-0.717562866174)*x[0]**o+(0.374797888506)*x[0]+(0.254329847579)*x[1]**o+(-0.987429692718)*x[1]+(-0.886758521324)*x[2]**o+(0.309486900444)*x[2]
        arg[2,0,1,0]=(0.139962143021)*x[0]**o+(-0.134735774213)*x[0]+(-0.231254041503)*x[1]**o+(0.0694785371931)*x[1]+(0.769929630132)*x[2]**o+(0.486646094292)*x[2]
        arg[2,0,1,1]=(0.349691701871)*x[0]**o+(-0.0308285150496)*x[0]+(0.00977143515921)*x[1]**o+(0.0970015800122)*x[1]+(-0.725991176667)*x[2]**o+(0.433663449433)*x[2]
        arg[2,0,2,0]=(0.564474039592)*x[0]**o+(0.950620065312)*x[0]+(0.256658075134)*x[1]**o+(0.774458453189)*x[1]+(0.00181734756534)*x[2]**o+(-0.270638852561)*x[2]
        arg[2,0,2,1]=(-0.175813811707)*x[0]**o+(0.821341326493)*x[0]+(-0.464966990893)*x[1]**o+(-0.310747066257)*x[1]+(-0.413557861251)*x[2]**o+(-0.799160067701)*x[2]
        arg[2,1,0,0]=(-0.449817381621)*x[0]**o+(-0.0573780378383)*x[0]+(-0.22675349133)*x[1]**o+(-0.150314340624)*x[1]+(-0.116693721705)*x[2]**o+(-0.20577625675)*x[2]
        arg[2,1,0,1]=(0.955318430674)*x[0]**o+(-0.148100168239)*x[0]+(0.741705440574)*x[1]**o+(-0.0768022054634)*x[1]+(-0.721807872287)*x[2]**o+(-0.898235041116)*x[2]
        arg[2,1,1,0]=(-0.183120810472)*x[0]**o+(0.307305712226)*x[0]+(0.699835684443)*x[1]**o+(0.964082351856)*x[1]+(-0.0348051023458)*x[2]**o+(0.544435382916)*x[2]
        arg[2,1,1,1]=(0.786862912642)*x[0]**o+(0.37388939814)*x[0]+(0.608937302857)*x[1]**o+(0.122305286217)*x[1]+(-0.892536849678)*x[2]**o+(0.119257478145)*x[2]
        arg[2,1,2,0]=(0.758154908623)*x[0]**o+(0.695188856067)*x[0]+(0.458313343957)*x[1]**o+(0.979157471455)*x[1]+(0.194237063725)*x[2]**o+(0.773001101454)*x[2]
        arg[2,1,2,1]=(0.0754678814151)*x[0]**o+(-0.639556181007)*x[0]+(0.934627102078)*x[1]**o+(0.0832082572611)*x[1]+(0.668822756858)*x[2]**o+(0.627532210297)*x[2]
        arg[2,2,0,0]=(-0.192921889174)*x[0]**o+(0.989703287553)*x[0]+(0.00517568784693)*x[1]**o+(-0.360076937731)*x[1]+(0.947056110784)*x[2]**o+(0.469841550381)*x[2]
        arg[2,2,0,1]=(0.201501669595)*x[0]**o+(-0.959957961834)*x[0]+(0.0229159090152)*x[1]**o+(0.396188263084)*x[1]+(0.186211667989)*x[2]**o+(0.304503474778)*x[2]
        arg[2,2,1,0]=(0.333974995608)*x[0]**o+(0.210202867626)*x[0]+(-0.410565897643)*x[1]**o+(0.52659985644)*x[1]+(0.983960564934)*x[2]**o+(-0.439323691866)*x[2]
        arg[2,2,1,1]=(0.706119074329)*x[0]**o+(-0.00220875175563)*x[0]+(0.143693169123)*x[1]**o+(0.217254593235)*x[1]+(-0.412057975252)*x[2]**o+(-0.967360361422)*x[2]
        arg[2,2,2,0]=(0.217307356597)*x[0]**o+(0.861728452163)*x[0]+(-0.193239462241)*x[1]**o+(0.872273176526)*x[1]+(-0.436414721507)*x[2]**o+(-0.62398425635)*x[2]
        arg[2,2,2,1]=(0.467099668794)*x[0]**o+(0.109742400918)*x[0]+(-0.987522191801)*x[1]**o+(-0.250251964275)*x[1]+(0.442152166523)*x[2]**o+(-0.709157679576)*x[2]
        arg[2,3,0,0]=(0.769666242031)*x[0]**o+(0.2352745864)*x[0]+(-0.302893533611)*x[1]**o+(0.0806677655098)*x[1]+(0.886434893279)*x[2]**o+(0.366016417047)*x[2]
        arg[2,3,0,1]=(-0.612204706187)*x[0]**o+(-0.242961993133)*x[0]+(0.946225816441)*x[1]**o+(0.528642650581)*x[1]+(-0.0166837450337)*x[2]**o+(-0.0889397945206)*x[2]
        arg[2,3,1,0]=(0.218589745288)*x[0]**o+(-0.296125811204)*x[0]+(0.358743045538)*x[1]**o+(-0.945778818656)*x[1]+(-0.33155158646)*x[2]**o+(-0.293038455557)*x[2]
        arg[2,3,1,1]=(0.753737723433)*x[0]**o+(-0.772926141856)*x[0]+(-0.0107183417902)*x[1]**o+(0.125538612735)*x[1]+(0.225277420156)*x[2]**o+(-0.54876651451)*x[2]
        arg[2,3,2,0]=(0.947419453319)*x[0]**o+(-0.101935160067)*x[0]+(0.481841505589)*x[1]**o+(-0.873842146032)*x[1]+(-0.377764402402)*x[2]**o+(-0.361560947647)*x[2]
        arg[2,3,2,1]=(0.745703609187)*x[0]**o+(-0.488401078192)*x[0]+(0.190261500318)*x[1]**o+(-0.22176180212)*x[1]+(0.80182740764)*x[2]**o+(-0.504953098351)*x[2]
        arg[2,4,0,0]=(0.476355625206)*x[0]**o+(0.452225231178)*x[0]+(0.669315429911)*x[1]**o+(0.159511010158)*x[1]+(0.652056592943)*x[2]**o+(0.44515033609)*x[2]
        arg[2,4,0,1]=(0.512194809916)*x[0]**o+(0.654452558318)*x[0]+(-0.526832882528)*x[1]**o+(0.590859897981)*x[1]+(-0.155989159043)*x[2]**o+(-0.689178072559)*x[2]
        arg[2,4,1,0]=(0.0450244624553)*x[0]**o+(0.0790299000335)*x[0]+(-0.690643908994)*x[1]**o+(-0.262426911912)*x[1]+(-0.426935471303)*x[2]**o+(-0.434570506153)*x[2]
        arg[2,4,1,1]=(-0.04445890079)*x[0]**o+(-0.541574770252)*x[0]+(-0.879290270122)*x[1]**o+(0.545302743626)*x[1]+(-0.836472565261)*x[2]**o+(0.66750272736)*x[2]
        arg[2,4,2,0]=(-0.477788509677)*x[0]**o+(0.706195950613)*x[0]+(0.628430029293)*x[1]**o+(-0.297608932813)*x[1]+(0.38364501438)*x[2]**o+(0.855444938831)*x[2]
        arg[2,4,2,1]=(0.123183000451)*x[0]**o+(0.41176132265)*x[0]+(-0.379498883356)*x[1]**o+(-0.53442961658)*x[1]+(-0.101215056317)*x[2]**o+(0.247971839461)*x[2]
        arg[3,0,0,0]=(-0.239170701904)*x[0]**o+(0.369420936359)*x[0]+(0.972913513036)*x[1]**o+(0.775248415403)*x[1]+(-0.850310640223)*x[2]**o+(-0.815345534491)*x[2]
        arg[3,0,0,1]=(0.591124498004)*x[0]**o+(0.529522171858)*x[0]+(-0.974005221641)*x[1]**o+(0.430102201571)*x[1]+(0.570519406976)*x[2]**o+(-0.0915083364671)*x[2]
        arg[3,0,1,0]=(-0.27678065557)*x[0]**o+(-0.53225308602)*x[0]+(0.902356190541)*x[1]**o+(-0.564808732046)*x[1]+(0.0981283775668)*x[2]**o+(-0.64957631107)*x[2]
        arg[3,0,1,1]=(0.230523873408)*x[0]**o+(-0.382745197262)*x[0]+(0.41507101812)*x[1]**o+(0.48492708335)*x[1]+(-0.11672353881)*x[2]**o+(-0.56902283397)*x[2]
        arg[3,0,2,0]=(-0.400023140104)*x[0]**o+(-0.406234012546)*x[0]+(-0.17810623045)*x[1]**o+(-0.833748300281)*x[1]+(-0.463689953863)*x[2]**o+(-0.11095482465)*x[2]
        arg[3,0,2,1]=(0.448885918108)*x[0]**o+(0.0447046792888)*x[0]+(-0.781652457281)*x[1]**o+(-0.893726375109)*x[1]+(0.918099233406)*x[2]**o+(0.560717614604)*x[2]
        arg[3,1,0,0]=(0.046700262853)*x[0]**o+(0.729867044213)*x[0]+(-0.870868205823)*x[1]**o+(-0.711267787629)*x[1]+(0.760550112312)*x[2]**o+(-0.0188334901698)*x[2]
        arg[3,1,0,1]=(0.304305551549)*x[0]**o+(-0.73527291962)*x[0]+(0.722134673974)*x[1]**o+(-0.40822784698)*x[1]+(0.912464122456)*x[2]**o+(-0.857656020349)*x[2]
        arg[3,1,1,0]=(-0.857864508282)*x[0]**o+(-0.22160002809)*x[0]+(-0.0892530574233)*x[1]**o+(-0.547259900671)*x[1]+(0.641858628113)*x[2]**o+(-0.203700292205)*x[2]
        arg[3,1,1,1]=(-0.47064403668)*x[0]**o+(-0.105885697358)*x[0]+(-0.899570240486)*x[1]**o+(0.660942689756)*x[1]+(-0.365836826289)*x[2]**o+(0.9593955249)*x[2]
        arg[3,1,2,0]=(-0.51407433496)*x[0]**o+(0.369092182634)*x[0]+(-0.740281092517)*x[1]**o+(0.383443939366)*x[1]+(-0.140318759419)*x[2]**o+(-0.481274651104)*x[2]
        arg[3,1,2,1]=(-0.938924125636)*x[0]**o+(0.41994781084)*x[0]+(-0.946320511542)*x[1]**o+(-0.86702606561)*x[1]+(0.63547569926)*x[2]**o+(0.923988068696)*x[2]
        arg[3,2,0,0]=(0.212833889165)*x[0]**o+(-0.0823077975238)*x[0]+(0.038483237489)*x[1]**o+(0.393784376623)*x[1]+(0.573190557103)*x[2]**o+(0.403015743632)*x[2]
        arg[3,2,0,1]=(-0.169294531943)*x[0]**o+(-0.769228919251)*x[0]+(-0.907462814057)*x[1]**o+(-0.188574782291)*x[1]+(-0.859793027531)*x[2]**o+(-0.673334626964)*x[2]
        arg[3,2,1,0]=(0.646744388517)*x[0]**o+(0.346749855021)*x[0]+(-0.314382791156)*x[1]**o+(-0.0396545336605)*x[1]+(0.844678566939)*x[2]**o+(0.724721336727)*x[2]
        arg[3,2,1,1]=(-0.471761417345)*x[0]**o+(-0.801631597222)*x[0]+(-0.376444012893)*x[1]**o+(0.920939382926)*x[1]+(-0.728019331398)*x[2]**o+(0.905808341144)*x[2]
        arg[3,2,2,0]=(-0.713759272803)*x[0]**o+(-0.930638013429)*x[0]+(0.864451879958)*x[1]**o+(0.0566794358824)*x[1]+(-0.292092226905)*x[2]**o+(-0.195695530268)*x[2]
        arg[3,2,2,1]=(0.843437618194)*x[0]**o+(0.214911962718)*x[0]+(0.475690162826)*x[1]**o+(0.0744570063618)*x[1]+(0.351711310616)*x[2]**o+(-0.187223450548)*x[2]
        arg[3,3,0,0]=(0.108651500878)*x[0]**o+(-0.402624590847)*x[0]+(0.120700159277)*x[1]**o+(-0.363139807635)*x[1]+(-0.62413441951)*x[2]**o+(-0.474236131596)*x[2]
        arg[3,3,0,1]=(-0.0836636547536)*x[0]**o+(-0.187959781777)*x[0]+(0.756983584308)*x[1]**o+(0.756573839599)*x[1]+(0.195640878564)*x[2]**o+(-0.566596474032)*x[2]
        arg[3,3,1,0]=(-0.171427844449)*x[0]**o+(-0.266807792484)*x[0]+(0.380046037965)*x[1]**o+(-0.129297371793)*x[1]+(0.79139100315)*x[2]**o+(0.223804751067)*x[2]
        arg[3,3,1,1]=(-0.054757303264)*x[0]**o+(-0.823878124196)*x[0]+(0.80695256107)*x[1]**o+(0.583250192211)*x[1]+(0.894937939133)*x[2]**o+(0.878757173892)*x[2]
        arg[3,3,2,0]=(0.13923802434)*x[0]**o+(-0.600732296834)*x[0]+(0.111226716676)*x[1]**o+(-0.474524749496)*x[1]+(0.674290476359)*x[2]**o+(-0.06714767759)*x[2]
        arg[3,3,2,1]=(0.246187769682)*x[0]**o+(-0.871969439162)*x[0]+(0.56748693564)*x[1]**o+(-0.0831774328079)*x[1]+(-0.27380467532)*x[2]**o+(-0.623711953244)*x[2]
        arg[3,4,0,0]=(-0.345947366312)*x[0]**o+(-0.304842609526)*x[0]+(-0.720722450633)*x[1]**o+(0.889712922756)*x[1]+(-0.0451362945237)*x[2]**o+(-0.52344522887)*x[2]
        arg[3,4,0,1]=(-0.830119647904)*x[0]**o+(0.577888363835)*x[0]+(0.180691945154)*x[1]**o+(-0.489554894073)*x[1]+(0.848489044545)*x[2]**o+(-0.243139985297)*x[2]
        arg[3,4,1,0]=(0.846597693999)*x[0]**o+(0.757477455396)*x[0]+(-0.827604908657)*x[1]**o+(-0.108392312717)*x[1]+(0.597454707613)*x[2]**o+(0.35846243312)*x[2]
        arg[3,4,1,1]=(0.107973753109)*x[0]**o+(0.578649460223)*x[0]+(0.609012033524)*x[1]**o+(-0.8428861922)*x[1]+(0.42144759968)*x[2]**o+(-0.938952208084)*x[2]
        arg[3,4,2,0]=(0.286586330618)*x[0]**o+(0.7324646289)*x[0]+(-0.448461377611)*x[1]**o+(0.62562408085)*x[1]+(0.352429283116)*x[2]**o+(0.112047034296)*x[2]
        arg[3,4,2,1]=(-0.431663387082)*x[0]**o+(-0.612934322552)*x[0]+(-0.426644797074)*x[1]**o+(0.768645027094)*x[1]+(-0.453242885757)*x[2]**o+(0.893748345809)*x[2]
        ref[0,0,0,0]=(-1.2147562394)*(1+2.*(dim-1.)/(o+1.))+(1.47471604216)*dim
        ref[0,0,0,1]=(-0.283261030731)*(1+2.*(dim-1.)/(o+1.))+(0.49706944223)*dim
        ref[0,0,1,0]=(-0.963543225803)*(1+2.*(dim-1.)/(o+1.))+(0.302115929232)*dim
        ref[0,0,1,1]=(1.47597923019)*(1+2.*(dim-1.)/(o+1.))+(0.302975596116)*dim
        ref[0,0,2,0]=(2.07218071265)*(1+2.*(dim-1.)/(o+1.))+(1.40474756312)*dim
        ref[0,0,2,1]=(1.18935174934)*(1+2.*(dim-1.)/(o+1.))+(-0.898949809002)*dim
        ref[0,1,0,0]=(-2.71454215957)*(1+2.*(dim-1.)/(o+1.))+(-1.81174147681)*dim
        ref[0,1,0,1]=(-1.27877046381)*(1+2.*(dim-1.)/(o+1.))+(0.197262183297)*dim
        ref[0,1,1,0]=(-0.287148009258)*(1+2.*(dim-1.)/(o+1.))+(1.30812352716)*dim
        ref[0,1,1,1]=(0.263153075777)*(1+2.*(dim-1.)/(o+1.))+(1.46256384302)*dim
        ref[0,1,2,0]=(-2.55250588186)*(1+2.*(dim-1.)/(o+1.))+(1.13566208594)*dim
        ref[0,1,2,1]=(0.208932386724)*(1+2.*(dim-1.)/(o+1.))+(-0.438703456176)*dim
        ref[0,2,0,0]=(-1.09576259026)*(1+2.*(dim-1.)/(o+1.))+(-0.13076989899)*dim
        ref[0,2,0,1]=(0.401914700214)*(1+2.*(dim-1.)/(o+1.))+(2.19797357625)*dim
        ref[0,2,1,0]=(0.860631336354)*(1+2.*(dim-1.)/(o+1.))+(0.978915494035)*dim
        ref[0,2,1,1]=(-1.97095623301)*(1+2.*(dim-1.)/(o+1.))+(-2.25442123467)*dim
        ref[0,2,2,0]=(-0.527474241147)*(1+2.*(dim-1.)/(o+1.))+(-0.923478312465)*dim
        ref[0,2,2,1]=(-0.460568018944)*(1+2.*(dim-1.)/(o+1.))+(0.173004367644)*dim
        ref[0,3,0,0]=(-0.443819991343)*(1+2.*(dim-1.)/(o+1.))+(1.08379961978)*dim
        ref[0,3,0,1]=(-0.80563854076)*(1+2.*(dim-1.)/(o+1.))+(-1.48015007036)*dim
        ref[0,3,1,0]=(-1.04939720148)*(1+2.*(dim-1.)/(o+1.))+(-0.762532531252)*dim
        ref[0,3,1,1]=(0.472870331273)*(1+2.*(dim-1.)/(o+1.))+(-0.00544721475928)*dim
        ref[0,3,2,0]=(-0.593230167381)*(1+2.*(dim-1.)/(o+1.))+(0.26676895359)*dim
        ref[0,3,2,1]=(-0.104009550799)*(1+2.*(dim-1.)/(o+1.))+(-2.23842412891)*dim
        ref[0,4,0,0]=(-0.352866783806)*(1+2.*(dim-1.)/(o+1.))+(-1.02623939367)*dim
        ref[0,4,0,1]=(0.0310776980081)*(1+2.*(dim-1.)/(o+1.))+(-1.9281752378)*dim
        ref[0,4,1,0]=(-0.56349759328)*(1+2.*(dim-1.)/(o+1.))+(0.500353711197)*dim
        ref[0,4,1,1]=(-0.788555686306)*(1+2.*(dim-1.)/(o+1.))+(-0.518133836527)*dim
        ref[0,4,2,0]=(0.174347407535)*(1+2.*(dim-1.)/(o+1.))+(0.471299435068)*dim
        ref[0,4,2,1]=(1.13297761695)*(1+2.*(dim-1.)/(o+1.))+(-0.93091412661)*dim
        ref[1,0,0,0]=(1.3307805483)*(1+2.*(dim-1.)/(o+1.))+(-1.93791762884)*dim
        ref[1,0,0,1]=(-1.40528512234)*(1+2.*(dim-1.)/(o+1.))+(0.0207469648915)*dim
        ref[1,0,1,0]=(1.00481927021)*(1+2.*(dim-1.)/(o+1.))+(-1.82465453458)*dim
        ref[1,0,1,1]=(-1.88446627952)*(1+2.*(dim-1.)/(o+1.))+(-0.965933535196)*dim
        ref[1,0,2,0]=(-1.8284637239)*(1+2.*(dim-1.)/(o+1.))+(-0.991125421549)*dim
        ref[1,0,2,1]=(1.39426526757)*(1+2.*(dim-1.)/(o+1.))+(1.06968368195)*dim
        ref[1,1,0,0]=(1.49675855079)*(1+2.*(dim-1.)/(o+1.))+(-1.71081281752)*dim
        ref[1,1,0,1]=(-0.36009887464)*(1+2.*(dim-1.)/(o+1.))+(-0.410764083324)*dim
        ref[1,1,1,0]=(-1.34942056351)*(1+2.*(dim-1.)/(o+1.))+(0.397777678247)*dim
        ref[1,1,1,1]=(0.638374123027)*(1+2.*(dim-1.)/(o+1.))+(0.650657783821)*dim
        ref[1,1,2,0]=(0.820420901755)*(1+2.*(dim-1.)/(o+1.))+(-1.29107032578)*dim
        ref[1,1,2,1]=(-1.27872123392)*(1+2.*(dim-1.)/(o+1.))+(0.65807032576)*dim
        ref[1,2,0,0]=(0.075783858436)*(1+2.*(dim-1.)/(o+1.))+(-1.53077548496)*dim
        ref[1,2,0,1]=(-0.576058493068)*(1+2.*(dim-1.)/(o+1.))+(-0.469223416409)*dim
        ref[1,2,1,0]=(0.91020253397)*(1+2.*(dim-1.)/(o+1.))+(0.377598744012)*dim
        ref[1,2,1,1]=(-2.029474944)*(1+2.*(dim-1.)/(o+1.))+(1.92964650577)*dim
        ref[1,2,2,0]=(-0.330720388509)*(1+2.*(dim-1.)/(o+1.))+(-0.917162558794)*dim
        ref[1,2,2,1]=(-1.24172356207)*(1+2.*(dim-1.)/(o+1.))+(0.503779996381)*dim
        ref[1,3,0,0]=(0.0663085439332)*(1+2.*(dim-1.)/(o+1.))+(-0.209292976945)*dim
        ref[1,3,0,1]=(-2.23036613844)*(1+2.*(dim-1.)/(o+1.))+(0.361492360204)*dim
        ref[1,3,1,0]=(1.52701244516)*(1+2.*(dim-1.)/(o+1.))+(0.280419833109)*dim
        ref[1,3,1,1]=(-0.726609917928)*(1+2.*(dim-1.)/(o+1.))+(-0.265942889187)*dim
        ref[1,3,2,0]=(1.32716715393)*(1+2.*(dim-1.)/(o+1.))+(1.49032480195)*dim
        ref[1,3,2,1]=(-1.88447215307)*(1+2.*(dim-1.)/(o+1.))+(1.19504958561)*dim
        ref[1,4,0,0]=(0.766978376499)*(1+2.*(dim-1.)/(o+1.))+(1.30542216761)*dim
        ref[1,4,0,1]=(-1.04188463845)*(1+2.*(dim-1.)/(o+1.))+(-1.15230024582)*dim
        ref[1,4,1,0]=(-0.891108351313)*(1+2.*(dim-1.)/(o+1.))+(-1.35124391741)*dim
        ref[1,4,1,1]=(-0.213203213304)*(1+2.*(dim-1.)/(o+1.))+(-0.252522733564)*dim
        ref[1,4,2,0]=(-1.46959965525)*(1+2.*(dim-1.)/(o+1.))+(-0.963334280551)*dim
        ref[1,4,2,1]=(1.1953581127)*(1+2.*(dim-1.)/(o+1.))+(-0.512271341684)*dim
        ref[2,0,0,0]=(0.848353935033)*(1+2.*(dim-1.)/(o+1.))+(-0.11411349311)*dim
        ref[2,0,0,1]=(-1.34999153992)*(1+2.*(dim-1.)/(o+1.))+(-0.303144903768)*dim
        ref[2,0,1,0]=(0.67863773165)*(1+2.*(dim-1.)/(o+1.))+(0.421388857272)*dim
        ref[2,0,1,1]=(-0.366528039637)*(1+2.*(dim-1.)/(o+1.))+(0.499836514396)*dim
        ref[2,0,2,0]=(0.822949462292)*(1+2.*(dim-1.)/(o+1.))+(1.45443966594)*dim
        ref[2,0,2,1]=(-1.05433866385)*(1+2.*(dim-1.)/(o+1.))+(-0.288565807465)*dim
        ref[2,1,0,0]=(-0.793264594657)*(1+2.*(dim-1.)/(o+1.))+(-0.413468635212)*dim
        ref[2,1,0,1]=(0.975215998961)*(1+2.*(dim-1.)/(o+1.))+(-1.12313741482)*dim
        ref[2,1,1,0]=(0.481909771625)*(1+2.*(dim-1.)/(o+1.))+(1.815823447)*dim
        ref[2,1,1,1]=(0.503263365821)*(1+2.*(dim-1.)/(o+1.))+(0.615452162503)*dim
        ref[2,1,2,0]=(1.41070531631)*(1+2.*(dim-1.)/(o+1.))+(2.44734742898)*dim
        ref[2,1,2,1]=(1.67891774035)*(1+2.*(dim-1.)/(o+1.))+(0.0711842865505)*dim
        ref[2,2,0,0]=(0.759309909457)*(1+2.*(dim-1.)/(o+1.))+(1.0994679002)*dim
        ref[2,2,0,1]=(0.410629246599)*(1+2.*(dim-1.)/(o+1.))+(-0.259266223972)*dim
        ref[2,2,1,0]=(0.907369662899)*(1+2.*(dim-1.)/(o+1.))+(0.297479032201)*dim
        ref[2,2,1,1]=(0.437754268199)*(1+2.*(dim-1.)/(o+1.))+(-0.752314519943)*dim
        ref[2,2,2,0]=(-0.412346827151)*(1+2.*(dim-1.)/(o+1.))+(1.11001737234)*dim
        ref[2,2,2,1]=(-0.0782703564833)*(1+2.*(dim-1.)/(o+1.))+(-0.849667242932)*dim
        ref[2,3,0,0]=(1.3532076017)*(1+2.*(dim-1.)/(o+1.))+(0.681958768956)*dim
        ref[2,3,0,1]=(0.317337365221)*(1+2.*(dim-1.)/(o+1.))+(0.196740862928)*dim
        ref[2,3,1,0]=(0.245781204366)*(1+2.*(dim-1.)/(o+1.))+(-1.53494308542)*dim
        ref[2,3,1,1]=(0.968296801799)*(1+2.*(dim-1.)/(o+1.))+(-1.19615404363)*dim
        ref[2,3,2,0]=(1.05149655651)*(1+2.*(dim-1.)/(o+1.))+(-1.33733825375)*dim
        ref[2,3,2,1]=(1.73779251715)*(1+2.*(dim-1.)/(o+1.))+(-1.21511597866)*dim
        ref[2,4,0,0]=(1.79772764806)*(1+2.*(dim-1.)/(o+1.))+(1.05688657743)*dim
        ref[2,4,0,1]=(-0.170627231654)*(1+2.*(dim-1.)/(o+1.))+(0.55613438374)*dim
        ref[2,4,1,0]=(-1.07255491784)*(1+2.*(dim-1.)/(o+1.))+(-0.617967518031)*dim
        ref[2,4,1,1]=(-1.76022173617)*(1+2.*(dim-1.)/(o+1.))+(0.671230700733)*dim
        ref[2,4,2,0]=(0.534286533995)*(1+2.*(dim-1.)/(o+1.))+(1.26403195663)*dim
        ref[2,4,2,1]=(-0.357530939222)*(1+2.*(dim-1.)/(o+1.))+(0.12530354553)*dim
        ref[3,0,0,0]=(-0.11656782909)*(1+2.*(dim-1.)/(o+1.))+(0.32932381727)*dim
        ref[3,0,0,1]=(0.187638683339)*(1+2.*(dim-1.)/(o+1.))+(0.868116036962)*dim
        ref[3,0,1,0]=(0.723703912538)*(1+2.*(dim-1.)/(o+1.))+(-1.74663812914)*dim
        ref[3,0,1,1]=(0.528871352717)*(1+2.*(dim-1.)/(o+1.))+(-0.466840947882)*dim
        ref[3,0,2,0]=(-1.04181932442)*(1+2.*(dim-1.)/(o+1.))+(-1.35093713748)*dim
        ref[3,0,2,1]=(0.585332694234)*(1+2.*(dim-1.)/(o+1.))+(-0.288304081216)*dim
        ref[3,1,0,0]=(-0.0636178306576)*(1+2.*(dim-1.)/(o+1.))+(-0.000234233586678)*dim
        ref[3,1,0,1]=(1.93890434798)*(1+2.*(dim-1.)/(o+1.))+(-2.00115678695)*dim
        ref[3,1,1,0]=(-0.305258937592)*(1+2.*(dim-1.)/(o+1.))+(-0.972560220966)*dim
        ref[3,1,1,1]=(-1.73605110346)*(1+2.*(dim-1.)/(o+1.))+(1.5144525173)*dim
        ref[3,1,2,0]=(-1.3946741869)*(1+2.*(dim-1.)/(o+1.))+(0.271261470896)*dim
        ref[3,1,2,1]=(-1.24976893792)*(1+2.*(dim-1.)/(o+1.))+(0.476909813927)*dim
        ref[3,2,0,0]=(0.824507683756)*(1+2.*(dim-1.)/(o+1.))+(0.71449232273)*dim
        ref[3,2,0,1]=(-1.93655037353)*(1+2.*(dim-1.)/(o+1.))+(-1.63113832851)*dim
        ref[3,2,1,0]=(1.1770401643)*(1+2.*(dim-1.)/(o+1.))+(1.03181665809)*dim
        ref[3,2,1,1]=(-1.57622476164)*(1+2.*(dim-1.)/(o+1.))+(1.02511612685)*dim
        ref[3,2,2,0]=(-0.141399619749)*(1+2.*(dim-1.)/(o+1.))+(-1.06965410781)*dim
        ref[3,2,2,1]=(1.67083909164)*(1+2.*(dim-1.)/(o+1.))+(0.102145518531)*dim
        ref[3,3,0,0]=(-0.394782759355)*(1+2.*(dim-1.)/(o+1.))+(-1.24000053008)*dim
        ref[3,3,0,1]=(0.868960808118)*(1+2.*(dim-1.)/(o+1.))+(0.00201758379073)*dim
        ref[3,3,1,0]=(1.00000919667)*(1+2.*(dim-1.)/(o+1.))+(-0.17230041321)*dim
        ref[3,3,1,1]=(1.64713319694)*(1+2.*(dim-1.)/(o+1.))+(0.638129241907)*dim
        ref[3,3,2,0]=(0.924755217374)*(1+2.*(dim-1.)/(o+1.))+(-1.14240472392)*dim
        ref[3,3,2,1]=(0.539870030002)*(1+2.*(dim-1.)/(o+1.))+(-1.57885882521)*dim
        ref[3,4,0,0]=(-1.11180611147)*(1+2.*(dim-1.)/(o+1.))+(0.0614250843596)*dim
        ref[3,4,0,1]=(0.199061341795)*(1+2.*(dim-1.)/(o+1.))+(-0.154806515534)*dim
        ref[3,4,1,0]=(0.616447492955)*(1+2.*(dim-1.)/(o+1.))+(1.0075475758)*dim
        ref[3,4,1,1]=(1.13843338631)*(1+2.*(dim-1.)/(o+1.))+(-1.20318894006)*dim
        ref[3,4,2,0]=(0.190554236123)*(1+2.*(dim-1.)/(o+1.))+(1.47013574405)*dim
        ref[3,4,2,1]=(-1.31155106991)*(1+2.*(dim-1.)/(o+1.))+(1.04945905035)*dim
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5, 3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnBoundary_fromData_Solution_rank0(self):
      """
      tests integral of rank 0 Data on the FunctionOnBoundary

      assumptions: Solution(self.domain) exists
                   self.domain supports integral on FunctionOnBoundary
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnBoundary(self.domain)
      w=Solution(self.domain)
      x=w.getX()
      arg=Data(0,(),w)
      if dim==2:
        arg=(0.416878137535)*x[0]**o+(0.0809721296208)*x[0]+(-0.275084069478)*x[1]**o+(-0.704226656249)*x[1]
        ref=(0.141794068057)*(1+2.*(dim-1.)/(o+1.))+(-0.623254526628)*dim
      else:
        arg=(0.587142789605)*x[0]**o+(0.894018391929)*x[0]+(-0.417709558814)*x[1]**o+(0.0970658484968)*x[1]+(-0.840982437679)*x[2]**o+(-0.400242562091)*x[2]
        ref=(-0.671549206889)*(1+2.*(dim-1.)/(o+1.))+(0.590841678335)*dim
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnBoundary_fromData_Solution_rank1(self):
      """
      tests integral of rank 1 Data on the FunctionOnBoundary

      assumptions: Solution(self.domain) exists
                   self.domain supports integral on FunctionOnBoundary
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnBoundary(self.domain)
      w=Solution(self.domain)
      x=w.getX()
      arg=Data(0,(2,),w)
      ref=numpy.zeros((2,),numpy.float_)
      if dim==2:
        arg[0]=(0.051335729494)*x[0]**o+(0.257057962305)*x[0]+(0.00794214446212)*x[1]**o+(0.52307365061)*x[1]
        arg[1]=(0.419915493803)*x[0]**o+(-0.0105826268931)*x[0]+(0.369274332127)*x[1]**o+(0.340796701053)*x[1]
        ref[0]=(0.0592778739561)*(1+2.*(dim-1.)/(o+1.))+(0.780131612915)*dim
        ref[1]=(0.789189825931)*(1+2.*(dim-1.)/(o+1.))+(0.33021407416)*dim
      else:
        arg[0]=(-0.483302491019)*x[0]**o+(0.223892398776)*x[0]+(-0.555612541317)*x[1]**o+(0.75535374219)*x[1]+(-0.208573228788)*x[2]**o+(0.479879212564)*x[2]
        arg[1]=(-0.012341185885)*x[0]**o+(-0.959474230066)*x[0]+(0.397671123434)*x[1]**o+(-0.120340696903)*x[1]+(-0.645757653205)*x[2]**o+(-0.216057959957)*x[2]
        ref[0]=(-1.24748826112)*(1+2.*(dim-1.)/(o+1.))+(1.45912535353)*dim
        ref[1]=(-0.260427715657)*(1+2.*(dim-1.)/(o+1.))+(-1.29587288693)*dim
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnBoundary_fromData_Solution_rank2(self):
      """
      tests integral of rank 2 Data on the FunctionOnBoundary

      assumptions: Solution(self.domain) exists
                   self.domain supports integral on FunctionOnBoundary
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnBoundary(self.domain)
      w=Solution(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5),w)
      ref=numpy.zeros((4, 5),numpy.float_)
      if dim==2:
        arg[0,0]=(-0.493237934301)*x[0]**o+(-0.977176505302)*x[0]+(-0.837740403623)*x[1]**o+(0.204709065496)*x[1]
        arg[0,1]=(0.794064508577)*x[0]**o+(-0.917060945623)*x[0]+(0.0114185601474)*x[1]**o+(0.547108694374)*x[1]
        arg[0,2]=(-0.989778984923)*x[0]**o+(-0.499651095311)*x[0]+(-0.126817505183)*x[1]**o+(0.889077385524)*x[1]
        arg[0,3]=(-0.223453594493)*x[0]**o+(-0.17983229649)*x[0]+(0.352602269148)*x[1]**o+(0.00617724258273)*x[1]
        arg[0,4]=(0.401784356952)*x[0]**o+(0.839308128877)*x[0]+(0.322943402455)*x[1]**o+(0.770206521944)*x[1]
        arg[1,0]=(0.491015078355)*x[0]**o+(0.194909202321)*x[0]+(0.38986016491)*x[1]**o+(0.671106836372)*x[1]
        arg[1,1]=(0.200697856784)*x[0]**o+(0.389746053244)*x[0]+(0.850237988574)*x[1]**o+(-0.677750684271)*x[1]
        arg[1,2]=(0.300444129016)*x[0]**o+(-0.502811932395)*x[0]+(0.390751425646)*x[1]**o+(-0.0234058348564)*x[1]
        arg[1,3]=(-0.451410318068)*x[0]**o+(0.0959092789621)*x[0]+(0.364602756931)*x[1]**o+(-0.637883149677)*x[1]
        arg[1,4]=(0.84148898142)*x[0]**o+(0.447701396385)*x[0]+(-0.925960934959)*x[1]**o+(-0.216915201171)*x[1]
        arg[2,0]=(0.497685540053)*x[0]**o+(0.706445401439)*x[0]+(0.278688226804)*x[1]**o+(0.263560807934)*x[1]
        arg[2,1]=(0.846700886302)*x[0]**o+(0.794459987626)*x[0]+(0.206707548962)*x[1]**o+(0.536938178952)*x[1]
        arg[2,2]=(-0.825060225713)*x[0]**o+(0.00650901742391)*x[0]+(0.500591036904)*x[1]**o+(0.86251606869)*x[1]
        arg[2,3]=(-0.490203731083)*x[0]**o+(-0.758885551614)*x[0]+(0.44212368096)*x[1]**o+(0.422183463853)*x[1]
        arg[2,4]=(0.326180837967)*x[0]**o+(0.0511693925274)*x[0]+(0.450713817213)*x[1]**o+(-0.788372992203)*x[1]
        arg[3,0]=(0.427328335694)*x[0]**o+(-0.549570782632)*x[0]+(0.0484849428866)*x[1]**o+(-0.488921618371)*x[1]
        arg[3,1]=(-0.000395237589224)*x[0]**o+(-0.902889768871)*x[0]+(-0.991821787559)*x[1]**o+(-0.907842202962)*x[1]
        arg[3,2]=(0.737156280391)*x[0]**o+(0.104714673958)*x[0]+(0.102528971835)*x[1]**o+(-0.0982626427735)*x[1]
        arg[3,3]=(0.971460723075)*x[0]**o+(-0.759821212391)*x[0]+(0.385963754923)*x[1]**o+(-0.533924725996)*x[1]
        arg[3,4]=(0.25440424638)*x[0]**o+(-0.82432588605)*x[0]+(0.759285911224)*x[1]**o+(0.328467600746)*x[1]
        ref[0,0]=(-1.33097833792)*(1+2.*(dim-1.)/(o+1.))+(-0.772467439806)*dim
        ref[0,1]=(0.805483068724)*(1+2.*(dim-1.)/(o+1.))+(-0.36995225125)*dim
        ref[0,2]=(-1.11659649011)*(1+2.*(dim-1.)/(o+1.))+(0.389426290213)*dim
        ref[0,3]=(0.129148674655)*(1+2.*(dim-1.)/(o+1.))+(-0.173655053907)*dim
        ref[0,4]=(0.724727759407)*(1+2.*(dim-1.)/(o+1.))+(1.60951465082)*dim
        ref[1,0]=(0.880875243265)*(1+2.*(dim-1.)/(o+1.))+(0.866016038693)*dim
        ref[1,1]=(1.05093584536)*(1+2.*(dim-1.)/(o+1.))+(-0.288004631027)*dim
        ref[1,2]=(0.691195554662)*(1+2.*(dim-1.)/(o+1.))+(-0.526217767251)*dim
        ref[1,3]=(-0.0868075611369)*(1+2.*(dim-1.)/(o+1.))+(-0.541973870715)*dim
        ref[1,4]=(-0.0844719535392)*(1+2.*(dim-1.)/(o+1.))+(0.230786195214)*dim
        ref[2,0]=(0.776373766857)*(1+2.*(dim-1.)/(o+1.))+(0.970006209374)*dim
        ref[2,1]=(1.05340843526)*(1+2.*(dim-1.)/(o+1.))+(1.33139816658)*dim
        ref[2,2]=(-0.324469188808)*(1+2.*(dim-1.)/(o+1.))+(0.869025086114)*dim
        ref[2,3]=(-0.0480800501232)*(1+2.*(dim-1.)/(o+1.))+(-0.336702087761)*dim
        ref[2,4]=(0.776894655179)*(1+2.*(dim-1.)/(o+1.))+(-0.737203599675)*dim
        ref[3,0]=(0.47581327858)*(1+2.*(dim-1.)/(o+1.))+(-1.038492401)*dim
        ref[3,1]=(-0.992217025148)*(1+2.*(dim-1.)/(o+1.))+(-1.81073197183)*dim
        ref[3,2]=(0.839685252226)*(1+2.*(dim-1.)/(o+1.))+(0.00645203118434)*dim
        ref[3,3]=(1.357424478)*(1+2.*(dim-1.)/(o+1.))+(-1.29374593839)*dim
        ref[3,4]=(1.0136901576)*(1+2.*(dim-1.)/(o+1.))+(-0.495858285305)*dim
      else:
        arg[0,0]=(-0.543785449843)*x[0]**o+(-0.593471310422)*x[0]+(0.223732084513)*x[1]**o+(-0.126390494354)*x[1]+(0.602561108649)*x[2]**o+(-0.879650196727)*x[2]
        arg[0,1]=(-0.373708985284)*x[0]**o+(0.274723212925)*x[0]+(0.0901068561926)*x[1]**o+(0.0705878267418)*x[1]+(-0.87124855534)*x[2]**o+(-0.342156987834)*x[2]
        arg[0,2]=(0.197662792758)*x[0]**o+(0.993389348528)*x[0]+(0.300080063659)*x[1]**o+(-0.87329423474)*x[1]+(-0.708861668092)*x[2]**o+(-0.145567510817)*x[2]
        arg[0,3]=(-0.641680604707)*x[0]**o+(0.502870155472)*x[0]+(0.726460257571)*x[1]**o+(-0.796246941548)*x[1]+(-0.034138694789)*x[2]**o+(-0.756773445728)*x[2]
        arg[0,4]=(-0.910301138852)*x[0]**o+(0.370603440687)*x[0]+(0.30908632679)*x[1]**o+(0.399307258883)*x[1]+(0.734329887794)*x[2]**o+(-0.105355323212)*x[2]
        arg[1,0]=(-0.998162269861)*x[0]**o+(0.744436497607)*x[0]+(0.474991680035)*x[1]**o+(-0.523297766774)*x[1]+(-0.147411103004)*x[2]**o+(-0.366260797475)*x[2]
        arg[1,1]=(-0.895068058812)*x[0]**o+(-0.543026567202)*x[0]+(0.681910579939)*x[1]**o+(0.839238385122)*x[1]+(-0.943454697555)*x[2]**o+(-0.942882687085)*x[2]
        arg[1,2]=(-0.0327934757673)*x[0]**o+(-0.551060110327)*x[0]+(0.0246419122114)*x[1]**o+(0.290492943846)*x[1]+(-0.81629260256)*x[2]**o+(-0.0703654573398)*x[2]
        arg[1,3]=(0.0101368428324)*x[0]**o+(0.169228100407)*x[0]+(-0.0720675504637)*x[1]**o+(0.854569517064)*x[1]+(-0.185495614033)*x[2]**o+(0.159402892807)*x[2]
        arg[1,4]=(-0.879056526056)*x[0]**o+(-0.499865495878)*x[0]+(0.389281340553)*x[1]**o+(0.208323326648)*x[1]+(0.201495185869)*x[2]**o+(0.287693342543)*x[2]
        arg[2,0]=(-0.0980367959573)*x[0]**o+(-0.743908923973)*x[0]+(0.612332291133)*x[1]**o+(-0.833162883242)*x[1]+(-0.149234146487)*x[2]**o+(-0.265352382043)*x[2]
        arg[2,1]=(-0.731053054407)*x[0]**o+(-0.328824749845)*x[0]+(-0.270771833154)*x[1]**o+(-0.924718425874)*x[1]+(-0.333737248999)*x[2]**o+(-0.437707530432)*x[2]
        arg[2,2]=(0.0568240255423)*x[0]**o+(0.889929679474)*x[0]+(-0.550346007257)*x[1]**o+(-0.276114383355)*x[1]+(0.818856335741)*x[2]**o+(0.949830759195)*x[2]
        arg[2,3]=(-0.487278099196)*x[0]**o+(0.621833922878)*x[0]+(-0.641187646288)*x[1]**o+(-0.62187991518)*x[1]+(0.193101177186)*x[2]**o+(0.574039256278)*x[2]
        arg[2,4]=(0.411046922073)*x[0]**o+(-0.10196277248)*x[0]+(-0.880065306034)*x[1]**o+(0.629582927604)*x[1]+(-0.364618567951)*x[2]**o+(0.130293365676)*x[2]
        arg[3,0]=(-0.625244224555)*x[0]**o+(0.50345243521)*x[0]+(0.701616120451)*x[1]**o+(0.44401843794)*x[1]+(-0.881068054195)*x[2]**o+(0.337405641389)*x[2]
        arg[3,1]=(0.244113319139)*x[0]**o+(0.739760310044)*x[0]+(0.439528183979)*x[1]**o+(0.808496622595)*x[1]+(0.432069187682)*x[2]**o+(-0.106394550529)*x[2]
        arg[3,2]=(-0.200973139796)*x[0]**o+(-0.373932231384)*x[0]+(-0.438946407503)*x[1]**o+(0.543079830762)*x[1]+(-0.499599010695)*x[2]**o+(0.501967674296)*x[2]
        arg[3,3]=(-0.128263000609)*x[0]**o+(-0.408181851435)*x[0]+(0.403315054177)*x[1]**o+(0.75067593555)*x[1]+(0.361607987217)*x[2]**o+(0.269483801463)*x[2]
        arg[3,4]=(0.821969994275)*x[0]**o+(0.350771424951)*x[0]+(-0.761235507511)*x[1]**o+(0.148329827876)*x[1]+(-0.918493845231)*x[2]**o+(0.592094636272)*x[2]
        ref[0,0]=(0.282507743319)*(1+2.*(dim-1.)/(o+1.))+(-1.5995120015)*dim
        ref[0,1]=(-1.15485068443)*(1+2.*(dim-1.)/(o+1.))+(0.00315405183214)*dim
        ref[0,2]=(-0.211118811675)*(1+2.*(dim-1.)/(o+1.))+(-0.0254723970291)*dim
        ref[0,3]=(0.0506409580755)*(1+2.*(dim-1.)/(o+1.))+(-1.0501502318)*dim
        ref[0,4]=(0.133115075732)*(1+2.*(dim-1.)/(o+1.))+(0.664555376358)*dim
        ref[1,0]=(-0.67058169283)*(1+2.*(dim-1.)/(o+1.))+(-0.145122066642)*dim
        ref[1,1]=(-1.15661217643)*(1+2.*(dim-1.)/(o+1.))+(-0.646670869164)*dim
        ref[1,2]=(-0.824444166116)*(1+2.*(dim-1.)/(o+1.))+(-0.330932623821)*dim
        ref[1,3]=(-0.247426321664)*(1+2.*(dim-1.)/(o+1.))+(1.18320051028)*dim
        ref[1,4]=(-0.288279999634)*(1+2.*(dim-1.)/(o+1.))+(-0.00384882668765)*dim
        ref[2,0]=(0.365061348688)*(1+2.*(dim-1.)/(o+1.))+(-1.84242418926)*dim
        ref[2,1]=(-1.33556213656)*(1+2.*(dim-1.)/(o+1.))+(-1.69125070615)*dim
        ref[2,2]=(0.325334354026)*(1+2.*(dim-1.)/(o+1.))+(1.56364605531)*dim
        ref[2,3]=(-0.935364568298)*(1+2.*(dim-1.)/(o+1.))+(0.573993263976)*dim
        ref[2,4]=(-0.833636951913)*(1+2.*(dim-1.)/(o+1.))+(0.6579135208)*dim
        ref[3,0]=(-0.804696158299)*(1+2.*(dim-1.)/(o+1.))+(1.28487651454)*dim
        ref[3,1]=(1.1157106908)*(1+2.*(dim-1.)/(o+1.))+(1.44186238211)*dim
        ref[3,2]=(-1.13951855799)*(1+2.*(dim-1.)/(o+1.))+(0.671115273674)*dim
        ref[3,3]=(0.636660040784)*(1+2.*(dim-1.)/(o+1.))+(0.611977885578)*dim
        ref[3,4]=(-0.857759358467)*(1+2.*(dim-1.)/(o+1.))+(1.0911958891)*dim
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnBoundary_fromData_Solution_rank3(self):
      """
      tests integral of rank 3 Data on the FunctionOnBoundary

      assumptions: Solution(self.domain) exists
                   self.domain supports integral on FunctionOnBoundary
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnBoundary(self.domain)
      w=Solution(self.domain)
      x=w.getX()
      arg=Data(0,(6, 2, 2),w)
      ref=numpy.zeros((6, 2, 2),numpy.float_)
      if dim==2:
        arg[0,0,0]=(-0.331954160625)*x[0]**o+(0.713423346586)*x[0]+(0.922333771405)*x[1]**o+(-0.306799323245)*x[1]
        arg[0,0,1]=(-0.837278273696)*x[0]**o+(-0.322068356249)*x[0]+(-0.952056283446)*x[1]**o+(0.78147215355)*x[1]
        arg[0,1,0]=(-0.71217197633)*x[0]**o+(-0.222121067189)*x[0]+(-0.740851323426)*x[1]**o+(-0.664415003166)*x[1]
        arg[0,1,1]=(0.304797844524)*x[0]**o+(-0.644642249615)*x[0]+(-0.942312793933)*x[1]**o+(0.758857258357)*x[1]
        arg[1,0,0]=(-0.978734634026)*x[0]**o+(0.697829375729)*x[0]+(-0.161023313194)*x[1]**o+(0.61960280308)*x[1]
        arg[1,0,1]=(0.65091302745)*x[0]**o+(-0.411592147008)*x[0]+(0.748017319704)*x[1]**o+(0.751661257345)*x[1]
        arg[1,1,0]=(-0.752749816617)*x[0]**o+(-0.889417028079)*x[0]+(0.187075769961)*x[1]**o+(0.96605423194)*x[1]
        arg[1,1,1]=(-0.791276410919)*x[0]**o+(-0.66910236639)*x[0]+(-0.107283654349)*x[1]**o+(0.556315348977)*x[1]
        arg[2,0,0]=(0.0717781211643)*x[0]**o+(-0.895974517834)*x[0]+(-0.388350799289)*x[1]**o+(0.55541171641)*x[1]
        arg[2,0,1]=(0.886475701949)*x[0]**o+(0.0794480579219)*x[0]+(-0.77671944551)*x[1]**o+(-0.29611930088)*x[1]
        arg[2,1,0]=(0.891530897911)*x[0]**o+(-0.767892439984)*x[0]+(-0.692871902002)*x[1]**o+(0.296346783431)*x[1]
        arg[2,1,1]=(0.904417221616)*x[0]**o+(-0.379208335246)*x[0]+(0.491675297059)*x[1]**o+(0.994131453216)*x[1]
        arg[3,0,0]=(-0.278750282129)*x[0]**o+(-0.889235749208)*x[0]+(-0.970025256676)*x[1]**o+(0.459265023633)*x[1]
        arg[3,0,1]=(-0.718548177793)*x[0]**o+(-0.801618785436)*x[0]+(-0.0489351007933)*x[1]**o+(-0.0331281021404)*x[1]
        arg[3,1,0]=(0.838645233036)*x[0]**o+(0.174830460073)*x[0]+(0.343887140274)*x[1]**o+(0.475643625956)*x[1]
        arg[3,1,1]=(-0.376629360434)*x[0]**o+(0.965850026388)*x[0]+(-0.62284011199)*x[1]**o+(-0.548261522438)*x[1]
        arg[4,0,0]=(-0.747109622101)*x[0]**o+(-0.793286186527)*x[0]+(0.700502366132)*x[1]**o+(0.708459489695)*x[1]
        arg[4,0,1]=(-0.372920704339)*x[0]**o+(-0.856662712404)*x[0]+(0.735191558102)*x[1]**o+(0.451959287084)*x[1]
        arg[4,1,0]=(-0.825227447648)*x[0]**o+(-0.978701927456)*x[0]+(0.856562109765)*x[1]**o+(0.226820057686)*x[1]
        arg[4,1,1]=(-0.883951567752)*x[0]**o+(0.346114896878)*x[0]+(0.22103721575)*x[1]**o+(0.766907412924)*x[1]
        arg[5,0,0]=(0.440140408264)*x[0]**o+(0.576442002932)*x[0]+(0.181136518153)*x[1]**o+(0.467290325887)*x[1]
        arg[5,0,1]=(0.323653297993)*x[0]**o+(-0.0169080219964)*x[0]+(-0.790889359324)*x[1]**o+(-0.752609210154)*x[1]
        arg[5,1,0]=(-0.771859095701)*x[0]**o+(0.392074466446)*x[0]+(0.672832544535)*x[1]**o+(-0.518196376899)*x[1]
        arg[5,1,1]=(-0.714106835785)*x[0]**o+(-0.392974811152)*x[0]+(0.477349412757)*x[1]**o+(-0.978738343032)*x[1]
        ref[0,0,0]=(0.59037961078)*(1+2.*(dim-1.)/(o+1.))+(0.406624023341)*dim
        ref[0,0,1]=(-1.78933455714)*(1+2.*(dim-1.)/(o+1.))+(0.459403797301)*dim
        ref[0,1,0]=(-1.45302329976)*(1+2.*(dim-1.)/(o+1.))+(-0.886536070355)*dim
        ref[0,1,1]=(-0.637514949409)*(1+2.*(dim-1.)/(o+1.))+(0.114215008741)*dim
        ref[1,0,0]=(-1.13975794722)*(1+2.*(dim-1.)/(o+1.))+(1.31743217881)*dim
        ref[1,0,1]=(1.39893034715)*(1+2.*(dim-1.)/(o+1.))+(0.340069110337)*dim
        ref[1,1,0]=(-0.565674046656)*(1+2.*(dim-1.)/(o+1.))+(0.076637203861)*dim
        ref[1,1,1]=(-0.898560065268)*(1+2.*(dim-1.)/(o+1.))+(-0.112787017413)*dim
        ref[2,0,0]=(-0.316572678125)*(1+2.*(dim-1.)/(o+1.))+(-0.340562801424)*dim
        ref[2,0,1]=(0.10975625644)*(1+2.*(dim-1.)/(o+1.))+(-0.216671242958)*dim
        ref[2,1,0]=(0.198658995909)*(1+2.*(dim-1.)/(o+1.))+(-0.471545656553)*dim
        ref[2,1,1]=(1.39609251868)*(1+2.*(dim-1.)/(o+1.))+(0.614923117971)*dim
        ref[3,0,0]=(-1.2487755388)*(1+2.*(dim-1.)/(o+1.))+(-0.429970725575)*dim
        ref[3,0,1]=(-0.767483278586)*(1+2.*(dim-1.)/(o+1.))+(-0.834746887577)*dim
        ref[3,1,0]=(1.18253237331)*(1+2.*(dim-1.)/(o+1.))+(0.650474086029)*dim
        ref[3,1,1]=(-0.999469472424)*(1+2.*(dim-1.)/(o+1.))+(0.41758850395)*dim
        ref[4,0,0]=(-0.0466072559697)*(1+2.*(dim-1.)/(o+1.))+(-0.0848266968319)*dim
        ref[4,0,1]=(0.362270853763)*(1+2.*(dim-1.)/(o+1.))+(-0.404703425319)*dim
        ref[4,1,0]=(0.0313346621166)*(1+2.*(dim-1.)/(o+1.))+(-0.75188186977)*dim
        ref[4,1,1]=(-0.662914352001)*(1+2.*(dim-1.)/(o+1.))+(1.1130223098)*dim
        ref[5,0,0]=(0.621276926417)*(1+2.*(dim-1.)/(o+1.))+(1.04373232882)*dim
        ref[5,0,1]=(-0.467236061331)*(1+2.*(dim-1.)/(o+1.))+(-0.76951723215)*dim
        ref[5,1,0]=(-0.0990265511653)*(1+2.*(dim-1.)/(o+1.))+(-0.126121910453)*dim
        ref[5,1,1]=(-0.236757423028)*(1+2.*(dim-1.)/(o+1.))+(-1.37171315418)*dim
      else:
        arg[0,0,0]=(-0.676850371385)*x[0]**o+(0.343194249293)*x[0]+(-0.289569238124)*x[1]**o+(0.546143025789)*x[1]+(-0.292429216951)*x[2]**o+(0.100638805907)*x[2]
        arg[0,0,1]=(-0.558403199885)*x[0]**o+(0.369265961569)*x[0]+(0.098539502761)*x[1]**o+(-0.555747072012)*x[1]+(0.0142505043744)*x[2]**o+(0.110677590143)*x[2]
        arg[0,1,0]=(0.480265324249)*x[0]**o+(-0.0801799981418)*x[0]+(-0.659040110213)*x[1]**o+(0.315685251032)*x[1]+(0.146222780181)*x[2]**o+(-0.307845817615)*x[2]
        arg[0,1,1]=(-0.469643195547)*x[0]**o+(0.631180947887)*x[0]+(0.0873749097691)*x[1]**o+(-0.907805635326)*x[1]+(-0.445238378297)*x[2]**o+(-0.583750385876)*x[2]
        arg[1,0,0]=(-0.519648347947)*x[0]**o+(-0.580046561107)*x[0]+(-0.474702629402)*x[1]**o+(-0.76698196707)*x[1]+(0.631290211156)*x[2]**o+(0.283150656338)*x[2]
        arg[1,0,1]=(0.828049802595)*x[0]**o+(-0.0996271417739)*x[0]+(0.13247948046)*x[1]**o+(0.64657332105)*x[1]+(0.901448888477)*x[2]**o+(0.895379597076)*x[2]
        arg[1,1,0]=(-0.197367099356)*x[0]**o+(-0.933210661759)*x[0]+(-0.0134349749548)*x[1]**o+(-0.651111981269)*x[1]+(0.256945439519)*x[2]**o+(0.528412689094)*x[2]
        arg[1,1,1]=(-0.922943172982)*x[0]**o+(-0.883102479542)*x[0]+(-0.0037345936911)*x[1]**o+(-0.598256159261)*x[1]+(-0.8562089409)*x[2]**o+(-0.298502617143)*x[2]
        arg[2,0,0]=(0.499285053237)*x[0]**o+(-0.347877602065)*x[0]+(0.783117300662)*x[1]**o+(0.0413016895448)*x[1]+(0.22555777414)*x[2]**o+(0.089246166469)*x[2]
        arg[2,0,1]=(-0.180602931202)*x[0]**o+(0.883958054663)*x[0]+(0.904974519387)*x[1]**o+(-0.169618307434)*x[1]+(0.361963890467)*x[2]**o+(-0.364236450353)*x[2]
        arg[2,1,0]=(0.332239090882)*x[0]**o+(-0.392239851855)*x[0]+(0.905142609029)*x[1]**o+(-0.471493540964)*x[1]+(-0.305327788141)*x[2]**o+(0.581213078834)*x[2]
        arg[2,1,1]=(0.481276189253)*x[0]**o+(0.645400809127)*x[0]+(-0.840844470697)*x[1]**o+(-0.47587396865)*x[1]+(0.134145050668)*x[2]**o+(-0.315081369951)*x[2]
        arg[3,0,0]=(-0.480497217552)*x[0]**o+(0.595345393671)*x[0]+(0.776315451392)*x[1]**o+(-0.396110510671)*x[1]+(0.0437422791996)*x[2]**o+(0.348675326203)*x[2]
        arg[3,0,1]=(0.867117243092)*x[0]**o+(-0.222483151231)*x[0]+(0.0856439550139)*x[1]**o+(0.355855801395)*x[1]+(-0.779380984653)*x[2]**o+(0.799440639884)*x[2]
        arg[3,1,0]=(-0.869234399353)*x[0]**o+(-0.747500724089)*x[0]+(-0.83154206968)*x[1]**o+(-0.771906212766)*x[1]+(-0.188734869619)*x[2]**o+(0.7312382704)*x[2]
        arg[3,1,1]=(-0.729677636148)*x[0]**o+(-0.358066533942)*x[0]+(-0.342651052704)*x[1]**o+(-0.531453801913)*x[1]+(-0.898300547827)*x[2]**o+(0.0387220517653)*x[2]
        arg[4,0,0]=(-0.380489041999)*x[0]**o+(0.517823660835)*x[0]+(0.722149782685)*x[1]**o+(0.510169845258)*x[1]+(-0.979340093133)*x[2]**o+(0.846050053391)*x[2]
        arg[4,0,1]=(0.390895580804)*x[0]**o+(0.122461788897)*x[0]+(-0.809243912452)*x[1]**o+(0.871689879803)*x[1]+(-0.991689466458)*x[2]**o+(-0.0373953689911)*x[2]
        arg[4,1,0]=(0.00670611033858)*x[0]**o+(-0.700084486793)*x[0]+(0.365358002826)*x[1]**o+(-0.928347441251)*x[1]+(-0.675081280461)*x[2]**o+(-0.262871553902)*x[2]
        arg[4,1,1]=(-0.566521655794)*x[0]**o+(-0.23325381374)*x[0]+(0.744538505444)*x[1]**o+(0.263970013855)*x[1]+(0.360984907343)*x[2]**o+(0.704696800059)*x[2]
        arg[5,0,0]=(0.754229123633)*x[0]**o+(-0.523353302691)*x[0]+(0.404181763418)*x[1]**o+(0.175414644029)*x[1]+(0.627455124887)*x[2]**o+(-0.450163514418)*x[2]
        arg[5,0,1]=(0.804547370453)*x[0]**o+(-0.748034210243)*x[0]+(-0.943261499953)*x[1]**o+(0.360715335834)*x[1]+(-0.952785406204)*x[2]**o+(0.188838771539)*x[2]
        arg[5,1,0]=(0.653730390007)*x[0]**o+(0.663746229033)*x[0]+(0.454911944273)*x[1]**o+(-0.548110740107)*x[1]+(0.799171265206)*x[2]**o+(-0.646356900812)*x[2]
        arg[5,1,1]=(0.533464266683)*x[0]**o+(-0.0650985333228)*x[0]+(-0.599880569232)*x[1]**o+(0.645120695827)*x[1]+(0.985316456657)*x[2]**o+(0.527848503917)*x[2]
        ref[0,0,0]=(-1.25884882646)*(1+2.*(dim-1.)/(o+1.))+(0.989976080988)*dim
        ref[0,0,1]=(-0.445613192749)*(1+2.*(dim-1.)/(o+1.))+(-0.0758035202994)*dim
        ref[0,1,0]=(-0.0325520057832)*(1+2.*(dim-1.)/(o+1.))+(-0.0723405647253)*dim
        ref[0,1,1]=(-0.827506664075)*(1+2.*(dim-1.)/(o+1.))+(-0.860375073315)*dim
        ref[1,0,0]=(-0.363060766193)*(1+2.*(dim-1.)/(o+1.))+(-1.06387787184)*dim
        ref[1,0,1]=(1.86197817153)*(1+2.*(dim-1.)/(o+1.))+(1.44232577635)*dim
        ref[1,1,0]=(0.0461433652081)*(1+2.*(dim-1.)/(o+1.))+(-1.05590995393)*dim
        ref[1,1,1]=(-1.78288670757)*(1+2.*(dim-1.)/(o+1.))+(-1.77986125595)*dim
        ref[2,0,0]=(1.50796012804)*(1+2.*(dim-1.)/(o+1.))+(-0.217329746051)*dim
        ref[2,0,1]=(1.08633547865)*(1+2.*(dim-1.)/(o+1.))+(0.350103296876)*dim
        ref[2,1,0]=(0.932053911771)*(1+2.*(dim-1.)/(o+1.))+(-0.282520313985)*dim
        ref[2,1,1]=(-0.225423230776)*(1+2.*(dim-1.)/(o+1.))+(-0.145554529474)*dim
        ref[3,0,0]=(0.33956051304)*(1+2.*(dim-1.)/(o+1.))+(0.547910209204)*dim
        ref[3,0,1]=(0.173380213454)*(1+2.*(dim-1.)/(o+1.))+(0.932813290049)*dim
        ref[3,1,0]=(-1.88951133865)*(1+2.*(dim-1.)/(o+1.))+(-0.788168666455)*dim
        ref[3,1,1]=(-1.97062923668)*(1+2.*(dim-1.)/(o+1.))+(-0.850798284089)*dim
        ref[4,0,0]=(-0.637679352447)*(1+2.*(dim-1.)/(o+1.))+(1.87404355948)*dim
        ref[4,0,1]=(-1.41003779811)*(1+2.*(dim-1.)/(o+1.))+(0.956756299709)*dim
        ref[4,1,0]=(-0.303017167297)*(1+2.*(dim-1.)/(o+1.))+(-1.89130348195)*dim
        ref[4,1,1]=(0.539001756993)*(1+2.*(dim-1.)/(o+1.))+(0.735413000173)*dim
        ref[5,0,0]=(1.78586601194)*(1+2.*(dim-1.)/(o+1.))+(-0.79810217308)*dim
        ref[5,0,1]=(-1.0914995357)*(1+2.*(dim-1.)/(o+1.))+(-0.198480102871)*dim
        ref[5,1,0]=(1.90781359949)*(1+2.*(dim-1.)/(o+1.))+(-0.530721411887)*dim
        ref[5,1,1]=(0.918900154108)*(1+2.*(dim-1.)/(o+1.))+(1.10787066642)*dim
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnBoundary_fromData_Solution_rank4(self):
      """
      tests integral of rank 4 Data on the FunctionOnBoundary

      assumptions: Solution(self.domain) exists
                   self.domain supports integral on FunctionOnBoundary
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnBoundary(self.domain)
      w=Solution(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5, 3, 2),w)
      ref=numpy.zeros((4, 5, 3, 2),numpy.float_)
      if dim==2:
        arg[0,0,0,0]=(0.181966612335)*x[0]**o+(-0.461883712153)*x[0]+(-0.420217253531)*x[1]**o+(0.52197109374)*x[1]
        arg[0,0,0,1]=(-0.122704378373)*x[0]**o+(0.549708706786)*x[0]+(-0.728108908074)*x[1]**o+(-0.839262748545)*x[1]
        arg[0,0,1,0]=(0.58557741427)*x[0]**o+(-0.970633305521)*x[0]+(0.215513381389)*x[1]**o+(-0.178877868196)*x[1]
        arg[0,0,1,1]=(0.646118894595)*x[0]**o+(0.839917605008)*x[0]+(-0.608843960266)*x[1]**o+(-0.840842208189)*x[1]
        arg[0,0,2,0]=(-0.126004594384)*x[0]**o+(0.663487355648)*x[0]+(0.801751176464)*x[1]**o+(-0.76620842256)*x[1]
        arg[0,0,2,1]=(-0.244425023768)*x[0]**o+(0.736809513316)*x[0]+(0.736776514098)*x[1]**o+(0.382648973627)*x[1]
        arg[0,1,0,0]=(-0.264460763052)*x[0]**o+(-0.325868872518)*x[0]+(-0.0636203008538)*x[1]**o+(-0.398059740187)*x[1]
        arg[0,1,0,1]=(0.443639751435)*x[0]**o+(0.523903519493)*x[0]+(-0.222575248757)*x[1]**o+(-0.168203019401)*x[1]
        arg[0,1,1,0]=(-0.329523829258)*x[0]**o+(-0.908408437436)*x[0]+(0.695441587034)*x[1]**o+(-0.236957609006)*x[1]
        arg[0,1,1,1]=(-0.57363301023)*x[0]**o+(-0.168677045141)*x[0]+(0.455235475098)*x[1]**o+(-0.498803217861)*x[1]
        arg[0,1,2,0]=(0.09906404243)*x[0]**o+(0.48682122702)*x[0]+(-0.250940180366)*x[1]**o+(-0.641013418008)*x[1]
        arg[0,1,2,1]=(-0.963193555161)*x[0]**o+(0.129868691037)*x[0]+(0.821192174481)*x[1]**o+(-0.245576703223)*x[1]
        arg[0,2,0,0]=(-0.867410251576)*x[0]**o+(0.170194043006)*x[0]+(0.837669366536)*x[1]**o+(0.285772890787)*x[1]
        arg[0,2,0,1]=(0.767089114154)*x[0]**o+(-0.425174482112)*x[0]+(-0.460840645159)*x[1]**o+(-0.314295694392)*x[1]
        arg[0,2,1,0]=(-0.932934920269)*x[0]**o+(0.399552332498)*x[0]+(-0.137764102275)*x[1]**o+(-0.0375128952657)*x[1]
        arg[0,2,1,1]=(0.507999826854)*x[0]**o+(-0.60826518686)*x[0]+(0.766838061511)*x[1]**o+(0.800723296773)*x[1]
        arg[0,2,2,0]=(0.372064864992)*x[0]**o+(0.29029476011)*x[0]+(0.203299629846)*x[1]**o+(-0.386357855884)*x[1]
        arg[0,2,2,1]=(0.661268311956)*x[0]**o+(0.60257115577)*x[0]+(0.88881711536)*x[1]**o+(0.335431084294)*x[1]
        arg[0,3,0,0]=(-0.756295024079)*x[0]**o+(-0.982472476783)*x[0]+(-0.264060825724)*x[1]**o+(0.082684535426)*x[1]
        arg[0,3,0,1]=(0.514580120205)*x[0]**o+(0.524816925461)*x[0]+(-0.745096223094)*x[1]**o+(-0.0796881815577)*x[1]
        arg[0,3,1,0]=(0.597556449974)*x[0]**o+(0.527680495979)*x[0]+(0.517638167762)*x[1]**o+(-0.357836534374)*x[1]
        arg[0,3,1,1]=(0.354366560101)*x[0]**o+(-0.617697325436)*x[0]+(0.339071431286)*x[1]**o+(0.299934132319)*x[1]
        arg[0,3,2,0]=(0.26563455123)*x[0]**o+(0.821495614519)*x[0]+(0.177496963332)*x[1]**o+(-0.154521746257)*x[1]
        arg[0,3,2,1]=(0.983477388632)*x[0]**o+(0.524806318175)*x[0]+(-0.604096007297)*x[1]**o+(0.350360164328)*x[1]
        arg[0,4,0,0]=(0.311369778768)*x[0]**o+(-0.648416457513)*x[0]+(0.463469290711)*x[1]**o+(0.0719628828709)*x[1]
        arg[0,4,0,1]=(-0.743081056195)*x[0]**o+(-0.803573973461)*x[0]+(-0.766982068691)*x[1]**o+(0.387240020398)*x[1]
        arg[0,4,1,0]=(-0.0715117411345)*x[0]**o+(-0.341871715212)*x[0]+(0.0593378446291)*x[1]**o+(-0.10183864979)*x[1]
        arg[0,4,1,1]=(0.538850329978)*x[0]**o+(0.0280201497914)*x[0]+(0.350206859589)*x[1]**o+(0.316831165676)*x[1]
        arg[0,4,2,0]=(-0.327852385434)*x[0]**o+(0.408866306659)*x[0]+(0.940378517318)*x[1]**o+(0.578309623908)*x[1]
        arg[0,4,2,1]=(0.378380112483)*x[0]**o+(0.696828716067)*x[0]+(-0.908233000582)*x[1]**o+(-0.328611483675)*x[1]
        arg[1,0,0,0]=(0.753991558436)*x[0]**o+(0.0542891140783)*x[0]+(-0.428934638249)*x[1]**o+(0.318155063446)*x[1]
        arg[1,0,0,1]=(-0.387609085343)*x[0]**o+(0.711023478158)*x[0]+(-0.969026331169)*x[1]**o+(0.456175434422)*x[1]
        arg[1,0,1,0]=(-0.468504469917)*x[0]**o+(-0.314756281356)*x[0]+(0.0145647044588)*x[1]**o+(0.074035761055)*x[1]
        arg[1,0,1,1]=(0.240749620446)*x[0]**o+(0.906736060537)*x[0]+(0.391320123368)*x[1]**o+(-0.261639387239)*x[1]
        arg[1,0,2,0]=(-0.461436333094)*x[0]**o+(-0.809258428817)*x[0]+(0.0670777162266)*x[1]**o+(-0.445304287721)*x[1]
        arg[1,0,2,1]=(-0.569794448172)*x[0]**o+(-0.0624052432724)*x[0]+(-0.759840810858)*x[1]**o+(-0.609106375016)*x[1]
        arg[1,1,0,0]=(0.751570143287)*x[0]**o+(0.78189919599)*x[0]+(-0.103246458909)*x[1]**o+(0.732156635312)*x[1]
        arg[1,1,0,1]=(-0.0126214545951)*x[0]**o+(-0.612853966263)*x[0]+(-0.238555698245)*x[1]**o+(-0.668009933698)*x[1]
        arg[1,1,1,0]=(0.816325838743)*x[0]**o+(0.545518357738)*x[0]+(0.893296287925)*x[1]**o+(0.694579919864)*x[1]
        arg[1,1,1,1]=(0.927638834795)*x[0]**o+(0.750347688466)*x[0]+(-0.15924462002)*x[1]**o+(-0.952995938892)*x[1]
        arg[1,1,2,0]=(0.970761253678)*x[0]**o+(0.205308852313)*x[0]+(0.909209906751)*x[1]**o+(-0.178635180174)*x[1]
        arg[1,1,2,1]=(-0.632065128479)*x[0]**o+(-0.300905398347)*x[0]+(0.865317638002)*x[1]**o+(0.886398250669)*x[1]
        arg[1,2,0,0]=(-0.0529293620539)*x[0]**o+(-0.909022003179)*x[0]+(0.235231916345)*x[1]**o+(-0.623380752415)*x[1]
        arg[1,2,0,1]=(0.873344987616)*x[0]**o+(0.144231546529)*x[0]+(0.307375566381)*x[1]**o+(-0.201617507084)*x[1]
        arg[1,2,1,0]=(-0.957945047658)*x[0]**o+(-0.432955474381)*x[0]+(0.0742288164869)*x[1]**o+(-0.106080430366)*x[1]
        arg[1,2,1,1]=(-0.664227561512)*x[0]**o+(0.0821007781774)*x[0]+(-0.541506076231)*x[1]**o+(-0.88307957937)*x[1]
        arg[1,2,2,0]=(0.754702675232)*x[0]**o+(0.175176916913)*x[0]+(0.519607893979)*x[1]**o+(-0.24129526204)*x[1]
        arg[1,2,2,1]=(-0.415660998528)*x[0]**o+(-0.964071393069)*x[0]+(0.458656303526)*x[1]**o+(0.752153826846)*x[1]
        arg[1,3,0,0]=(-0.15638393187)*x[0]**o+(0.0828824719)*x[0]+(-0.0846205011509)*x[1]**o+(-0.438939463736)*x[1]
        arg[1,3,0,1]=(0.222994482074)*x[0]**o+(-0.488426101986)*x[0]+(-0.799262651825)*x[1]**o+(0.0638880385288)*x[1]
        arg[1,3,1,0]=(0.690529298963)*x[0]**o+(-0.845248154386)*x[0]+(0.748635110096)*x[1]**o+(0.820882250869)*x[1]
        arg[1,3,1,1]=(-0.0989162396328)*x[0]**o+(-0.171252734498)*x[0]+(-0.363447812361)*x[1]**o+(-0.06937701403)*x[1]
        arg[1,3,2,0]=(-0.739647298124)*x[0]**o+(-0.83384135572)*x[0]+(0.546124768813)*x[1]**o+(0.415761104064)*x[1]
        arg[1,3,2,1]=(0.859487140106)*x[0]**o+(0.481214854724)*x[0]+(0.788685241166)*x[1]**o+(-0.776410333263)*x[1]
        arg[1,4,0,0]=(-0.312428038681)*x[0]**o+(-0.178596293803)*x[0]+(-0.229770546807)*x[1]**o+(-0.43765633326)*x[1]
        arg[1,4,0,1]=(-0.974305286188)*x[0]**o+(-0.978154435248)*x[0]+(-0.671799350916)*x[1]**o+(0.645287291694)*x[1]
        arg[1,4,1,0]=(-0.219711371151)*x[0]**o+(-0.432784911532)*x[0]+(0.552940388883)*x[1]**o+(-0.622742205614)*x[1]
        arg[1,4,1,1]=(0.677289749287)*x[0]**o+(-0.92825132284)*x[0]+(0.818767358259)*x[1]**o+(-0.426000012305)*x[1]
        arg[1,4,2,0]=(-0.507474696573)*x[0]**o+(0.0882151625837)*x[0]+(0.826611187078)*x[1]**o+(0.157884229299)*x[1]
        arg[1,4,2,1]=(0.902398378671)*x[0]**o+(-0.372679984773)*x[0]+(-0.998272135687)*x[1]**o+(0.292294917465)*x[1]
        arg[2,0,0,0]=(0.699035433674)*x[0]**o+(0.380640017665)*x[0]+(-0.650735771214)*x[1]**o+(-0.55187467604)*x[1]
        arg[2,0,0,1]=(0.703438074924)*x[0]**o+(0.795508285134)*x[0]+(-0.0650847718675)*x[1]**o+(-0.0250766383001)*x[1]
        arg[2,0,1,0]=(0.493174434488)*x[0]**o+(-0.775523541313)*x[0]+(0.23184619496)*x[1]**o+(0.343368328234)*x[1]
        arg[2,0,1,1]=(0.404974280918)*x[0]**o+(0.0673627086244)*x[0]+(-0.843957699955)*x[1]**o+(0.645895713789)*x[1]
        arg[2,0,2,0]=(0.985047143685)*x[0]**o+(0.969582300743)*x[0]+(0.929427858098)*x[1]**o+(-0.951787413112)*x[1]
        arg[2,0,2,1]=(0.618505056215)*x[0]**o+(0.685760720523)*x[0]+(0.14594727158)*x[1]**o+(0.816579780815)*x[1]
        arg[2,1,0,0]=(0.0824953418089)*x[0]**o+(-0.896753351387)*x[0]+(0.415841596602)*x[1]**o+(0.349436600222)*x[1]
        arg[2,1,0,1]=(-0.904246301273)*x[0]**o+(0.51403543319)*x[0]+(0.0950147832662)*x[1]**o+(0.0482220256459)*x[1]
        arg[2,1,1,0]=(0.653418547332)*x[0]**o+(0.703094950575)*x[0]+(-0.294924381355)*x[1]**o+(0.889408306649)*x[1]
        arg[2,1,1,1]=(-0.329361050221)*x[0]**o+(-0.93072021487)*x[0]+(0.0845194032793)*x[1]**o+(-0.310617126183)*x[1]
        arg[2,1,2,0]=(-0.282371473471)*x[0]**o+(0.387458491178)*x[0]+(0.643908016757)*x[1]**o+(0.303741991451)*x[1]
        arg[2,1,2,1]=(-0.416872502153)*x[0]**o+(0.373465764073)*x[0]+(-0.90689140487)*x[1]**o+(-0.351052228378)*x[1]
        arg[2,2,0,0]=(0.494198770026)*x[0]**o+(0.744917026846)*x[0]+(-0.806591346974)*x[1]**o+(-0.812396479226)*x[1]
        arg[2,2,0,1]=(0.230573331783)*x[0]**o+(-0.449815323831)*x[0]+(-0.662178492084)*x[1]**o+(0.0281786827584)*x[1]
        arg[2,2,1,0]=(-0.546296599764)*x[0]**o+(-0.457939754236)*x[0]+(-0.0432225723667)*x[1]**o+(0.0431706529852)*x[1]
        arg[2,2,1,1]=(-0.808174460895)*x[0]**o+(-0.473584203382)*x[0]+(0.567514752294)*x[1]**o+(-0.115780229631)*x[1]
        arg[2,2,2,0]=(-0.134425008633)*x[0]**o+(-0.103550646345)*x[0]+(-0.588484951305)*x[1]**o+(0.955148281194)*x[1]
        arg[2,2,2,1]=(0.550095417437)*x[0]**o+(0.943564321046)*x[0]+(-0.990366145789)*x[1]**o+(0.872652302953)*x[1]
        arg[2,3,0,0]=(0.209341033615)*x[0]**o+(-0.285197252362)*x[0]+(0.961111991429)*x[1]**o+(0.810022268034)*x[1]
        arg[2,3,0,1]=(0.238319839582)*x[0]**o+(-0.714990515235)*x[0]+(0.0925371590198)*x[1]**o+(0.938145796375)*x[1]
        arg[2,3,1,0]=(0.889068347007)*x[0]**o+(-0.301904392078)*x[0]+(0.41386941601)*x[1]**o+(-0.973115118701)*x[1]
        arg[2,3,1,1]=(-0.522083695846)*x[0]**o+(-0.582455208417)*x[0]+(0.725431346859)*x[1]**o+(-0.950157996033)*x[1]
        arg[2,3,2,0]=(-0.897838142917)*x[0]**o+(0.745488698358)*x[0]+(-0.38910727894)*x[1]**o+(0.614539961316)*x[1]
        arg[2,3,2,1]=(0.924500645496)*x[0]**o+(-0.68988446876)*x[0]+(-0.466238481805)*x[1]**o+(-0.957703006413)*x[1]
        arg[2,4,0,0]=(0.280692018949)*x[0]**o+(0.600320489083)*x[0]+(-0.402390231845)*x[1]**o+(0.906721159456)*x[1]
        arg[2,4,0,1]=(0.50594574787)*x[0]**o+(0.130986910768)*x[0]+(0.272643351394)*x[1]**o+(0.53790545383)*x[1]
        arg[2,4,1,0]=(-0.627308461808)*x[0]**o+(0.690560028115)*x[0]+(-0.455768647207)*x[1]**o+(0.134926199015)*x[1]
        arg[2,4,1,1]=(-0.501163279392)*x[0]**o+(0.980144824993)*x[0]+(-0.0358930723173)*x[1]**o+(0.720157452357)*x[1]
        arg[2,4,2,0]=(-0.609861117539)*x[0]**o+(0.965699461849)*x[0]+(0.305674136837)*x[1]**o+(-0.578333683729)*x[1]
        arg[2,4,2,1]=(-0.921051416627)*x[0]**o+(-0.703479746028)*x[0]+(-0.0189027931731)*x[1]**o+(0.403415497212)*x[1]
        arg[3,0,0,0]=(-0.846266714886)*x[0]**o+(-0.103621657972)*x[0]+(0.217340457606)*x[1]**o+(-0.537654290693)*x[1]
        arg[3,0,0,1]=(0.186251745403)*x[0]**o+(0.694458841658)*x[0]+(0.126837362883)*x[1]**o+(0.0997502401792)*x[1]
        arg[3,0,1,0]=(0.153737843143)*x[0]**o+(0.811590040899)*x[0]+(0.189363446699)*x[1]**o+(0.00688399857093)*x[1]
        arg[3,0,1,1]=(-0.912444421339)*x[0]**o+(-0.68504171198)*x[0]+(-0.820717153919)*x[1]**o+(-0.00475951426309)*x[1]
        arg[3,0,2,0]=(0.908522439455)*x[0]**o+(-0.911126936858)*x[0]+(0.0424198145962)*x[1]**o+(-0.352097158128)*x[1]
        arg[3,0,2,1]=(0.935127271967)*x[0]**o+(0.154700792222)*x[0]+(-0.283148208173)*x[1]**o+(-0.607541482501)*x[1]
        arg[3,1,0,0]=(-0.131329160543)*x[0]**o+(0.24170180662)*x[0]+(-0.00893591013587)*x[1]**o+(0.262635048958)*x[1]
        arg[3,1,0,1]=(-0.559200924561)*x[0]**o+(0.509310116425)*x[0]+(0.484486720282)*x[1]**o+(-0.89314080386)*x[1]
        arg[3,1,1,0]=(0.355094442114)*x[0]**o+(-0.748398915688)*x[0]+(-0.379187476624)*x[1]**o+(0.447409858108)*x[1]
        arg[3,1,1,1]=(0.170855109103)*x[0]**o+(-0.48925054508)*x[0]+(0.840559727559)*x[1]**o+(0.0359562471889)*x[1]
        arg[3,1,2,0]=(0.827586094521)*x[0]**o+(0.0231820503547)*x[0]+(0.457192172855)*x[1]**o+(0.826960230377)*x[1]
        arg[3,1,2,1]=(-0.962890676345)*x[0]**o+(-0.112439391098)*x[0]+(-0.865510686025)*x[1]**o+(-0.539059426684)*x[1]
        arg[3,2,0,0]=(-0.165030011664)*x[0]**o+(-0.285694090419)*x[0]+(-0.522579583443)*x[1]**o+(0.73946640768)*x[1]
        arg[3,2,0,1]=(0.809351962503)*x[0]**o+(-0.872047628102)*x[0]+(0.61415433255)*x[1]**o+(0.132268088986)*x[1]
        arg[3,2,1,0]=(-0.587823544835)*x[0]**o+(-0.416092975475)*x[0]+(0.829027070431)*x[1]**o+(0.805271620727)*x[1]
        arg[3,2,1,1]=(-0.0107935093527)*x[0]**o+(0.988434287979)*x[0]+(0.941743055822)*x[1]**o+(0.137807879334)*x[1]
        arg[3,2,2,0]=(-0.855931527408)*x[0]**o+(0.575075608849)*x[0]+(0.11916967111)*x[1]**o+(-0.472844336533)*x[1]
        arg[3,2,2,1]=(-0.542112514875)*x[0]**o+(0.081308081083)*x[0]+(-0.934575648869)*x[1]**o+(-0.218208369535)*x[1]
        arg[3,3,0,0]=(0.486635391195)*x[0]**o+(0.995728500002)*x[0]+(-0.686767180456)*x[1]**o+(0.246849483855)*x[1]
        arg[3,3,0,1]=(-0.331571848212)*x[0]**o+(0.801870431206)*x[0]+(-0.91142654055)*x[1]**o+(0.089854528457)*x[1]
        arg[3,3,1,0]=(-0.657455652936)*x[0]**o+(-0.721466128651)*x[0]+(0.0420989631458)*x[1]**o+(-0.982316453325)*x[1]
        arg[3,3,1,1]=(-0.709257565086)*x[0]**o+(0.689028730506)*x[0]+(-0.196516960475)*x[1]**o+(-0.216704841188)*x[1]
        arg[3,3,2,0]=(-0.910547650394)*x[0]**o+(-0.977471949987)*x[0]+(0.37319115353)*x[1]**o+(-0.248978955796)*x[1]
        arg[3,3,2,1]=(-0.71025804576)*x[0]**o+(0.373559860529)*x[0]+(-0.9064074733)*x[1]**o+(-0.542012058073)*x[1]
        arg[3,4,0,0]=(-0.649273580176)*x[0]**o+(0.525784948768)*x[0]+(-0.213929862516)*x[1]**o+(-0.162295172275)*x[1]
        arg[3,4,0,1]=(0.209784328978)*x[0]**o+(-0.972601183002)*x[0]+(-0.730886568265)*x[1]**o+(0.966415518759)*x[1]
        arg[3,4,1,0]=(-0.334060094293)*x[0]**o+(0.29371522739)*x[0]+(0.936156150108)*x[1]**o+(-0.257026769476)*x[1]
        arg[3,4,1,1]=(-0.0465451767064)*x[0]**o+(0.115525862669)*x[0]+(0.891280073228)*x[1]**o+(-0.0436233989979)*x[1]
        arg[3,4,2,0]=(-0.748697989123)*x[0]**o+(0.246672415445)*x[0]+(0.62642302161)*x[1]**o+(-0.411644161038)*x[1]
        arg[3,4,2,1]=(0.850132316298)*x[0]**o+(0.939463534845)*x[0]+(0.0437756618735)*x[1]**o+(-0.955198146074)*x[1]
        ref[0,0,0,0]=(-0.238250641196)*(1+2.*(dim-1.)/(o+1.))+(0.0600873815876)*dim
        ref[0,0,0,1]=(-0.850813286447)*(1+2.*(dim-1.)/(o+1.))+(-0.28955404176)*dim
        ref[0,0,1,0]=(0.801090795659)*(1+2.*(dim-1.)/(o+1.))+(-1.14951117372)*dim
        ref[0,0,1,1]=(0.0372749343294)*(1+2.*(dim-1.)/(o+1.))+(-0.000924603180129)*dim
        ref[0,0,2,0]=(0.675746582079)*(1+2.*(dim-1.)/(o+1.))+(-0.102721066912)*dim
        ref[0,0,2,1]=(0.49235149033)*(1+2.*(dim-1.)/(o+1.))+(1.11945848694)*dim
        ref[0,1,0,0]=(-0.328081063906)*(1+2.*(dim-1.)/(o+1.))+(-0.723928612706)*dim
        ref[0,1,0,1]=(0.221064502678)*(1+2.*(dim-1.)/(o+1.))+(0.355700500092)*dim
        ref[0,1,1,0]=(0.365917757777)*(1+2.*(dim-1.)/(o+1.))+(-1.14536604644)*dim
        ref[0,1,1,1]=(-0.118397535132)*(1+2.*(dim-1.)/(o+1.))+(-0.667480263002)*dim
        ref[0,1,2,0]=(-0.151876137936)*(1+2.*(dim-1.)/(o+1.))+(-0.154192190988)*dim
        ref[0,1,2,1]=(-0.14200138068)*(1+2.*(dim-1.)/(o+1.))+(-0.115708012186)*dim
        ref[0,2,0,0]=(-0.0297408850402)*(1+2.*(dim-1.)/(o+1.))+(0.455966933793)*dim
        ref[0,2,0,1]=(0.306248468995)*(1+2.*(dim-1.)/(o+1.))+(-0.739470176504)*dim
        ref[0,2,1,0]=(-1.07069902254)*(1+2.*(dim-1.)/(o+1.))+(0.362039437233)*dim
        ref[0,2,1,1]=(1.27483788836)*(1+2.*(dim-1.)/(o+1.))+(0.192458109913)*dim
        ref[0,2,2,0]=(0.575364494839)*(1+2.*(dim-1.)/(o+1.))+(-0.0960630957749)*dim
        ref[0,2,2,1]=(1.55008542732)*(1+2.*(dim-1.)/(o+1.))+(0.938002240064)*dim
        ref[0,3,0,0]=(-1.0203558498)*(1+2.*(dim-1.)/(o+1.))+(-0.899787941357)*dim
        ref[0,3,0,1]=(-0.230516102889)*(1+2.*(dim-1.)/(o+1.))+(0.445128743903)*dim
        ref[0,3,1,0]=(1.11519461774)*(1+2.*(dim-1.)/(o+1.))+(0.169843961604)*dim
        ref[0,3,1,1]=(0.693437991388)*(1+2.*(dim-1.)/(o+1.))+(-0.317763193117)*dim
        ref[0,3,2,0]=(0.443131514562)*(1+2.*(dim-1.)/(o+1.))+(0.666973868262)*dim
        ref[0,3,2,1]=(0.379381381335)*(1+2.*(dim-1.)/(o+1.))+(0.875166482503)*dim
        ref[0,4,0,0]=(0.774839069479)*(1+2.*(dim-1.)/(o+1.))+(-0.576453574642)*dim
        ref[0,4,0,1]=(-1.51006312489)*(1+2.*(dim-1.)/(o+1.))+(-0.416333953063)*dim
        ref[0,4,1,0]=(-0.0121738965054)*(1+2.*(dim-1.)/(o+1.))+(-0.443710365002)*dim
        ref[0,4,1,1]=(0.889057189568)*(1+2.*(dim-1.)/(o+1.))+(0.344851315468)*dim
        ref[0,4,2,0]=(0.612526131884)*(1+2.*(dim-1.)/(o+1.))+(0.987175930567)*dim
        ref[0,4,2,1]=(-0.529852888099)*(1+2.*(dim-1.)/(o+1.))+(0.368217232392)*dim
        ref[1,0,0,0]=(0.325056920187)*(1+2.*(dim-1.)/(o+1.))+(0.372444177524)*dim
        ref[1,0,0,1]=(-1.35663541651)*(1+2.*(dim-1.)/(o+1.))+(1.16719891258)*dim
        ref[1,0,1,0]=(-0.453939765458)*(1+2.*(dim-1.)/(o+1.))+(-0.240720520301)*dim
        ref[1,0,1,1]=(0.632069743814)*(1+2.*(dim-1.)/(o+1.))+(0.645096673298)*dim
        ref[1,0,2,0]=(-0.394358616867)*(1+2.*(dim-1.)/(o+1.))+(-1.25456271654)*dim
        ref[1,0,2,1]=(-1.32963525903)*(1+2.*(dim-1.)/(o+1.))+(-0.671511618288)*dim
        ref[1,1,0,0]=(0.648323684378)*(1+2.*(dim-1.)/(o+1.))+(1.5140558313)*dim
        ref[1,1,0,1]=(-0.251177152841)*(1+2.*(dim-1.)/(o+1.))+(-1.28086389996)*dim
        ref[1,1,1,0]=(1.70962212667)*(1+2.*(dim-1.)/(o+1.))+(1.2400982776)*dim
        ref[1,1,1,1]=(0.768394214775)*(1+2.*(dim-1.)/(o+1.))+(-0.202648250425)*dim
        ref[1,1,2,0]=(1.87997116043)*(1+2.*(dim-1.)/(o+1.))+(0.0266736721392)*dim
        ref[1,1,2,1]=(0.233252509524)*(1+2.*(dim-1.)/(o+1.))+(0.585492852322)*dim
        ref[1,2,0,0]=(0.182302554291)*(1+2.*(dim-1.)/(o+1.))+(-1.53240275559)*dim
        ref[1,2,0,1]=(1.180720554)*(1+2.*(dim-1.)/(o+1.))+(-0.0573859605549)*dim
        ref[1,2,1,0]=(-0.883716231171)*(1+2.*(dim-1.)/(o+1.))+(-0.539035904746)*dim
        ref[1,2,1,1]=(-1.20573363774)*(1+2.*(dim-1.)/(o+1.))+(-0.800978801193)*dim
        ref[1,2,2,0]=(1.27431056921)*(1+2.*(dim-1.)/(o+1.))+(-0.066118345127)*dim
        ref[1,2,2,1]=(0.0429953049979)*(1+2.*(dim-1.)/(o+1.))+(-0.211917566223)*dim
        ref[1,3,0,0]=(-0.241004433021)*(1+2.*(dim-1.)/(o+1.))+(-0.356056991836)*dim
        ref[1,3,0,1]=(-0.576268169751)*(1+2.*(dim-1.)/(o+1.))+(-0.424538063457)*dim
        ref[1,3,1,0]=(1.43916440906)*(1+2.*(dim-1.)/(o+1.))+(-0.0243659035174)*dim
        ref[1,3,1,1]=(-0.462364051993)*(1+2.*(dim-1.)/(o+1.))+(-0.240629748528)*dim
        ref[1,3,2,0]=(-0.193522529311)*(1+2.*(dim-1.)/(o+1.))+(-0.418080251656)*dim
        ref[1,3,2,1]=(1.64817238127)*(1+2.*(dim-1.)/(o+1.))+(-0.295195478539)*dim
        ref[1,4,0,0]=(-0.542198585489)*(1+2.*(dim-1.)/(o+1.))+(-0.616252627062)*dim
        ref[1,4,0,1]=(-1.6461046371)*(1+2.*(dim-1.)/(o+1.))+(-0.332867143554)*dim
        ref[1,4,1,0]=(0.333229017732)*(1+2.*(dim-1.)/(o+1.))+(-1.05552711715)*dim
        ref[1,4,1,1]=(1.49605710755)*(1+2.*(dim-1.)/(o+1.))+(-1.35425133515)*dim
        ref[1,4,2,0]=(0.319136490505)*(1+2.*(dim-1.)/(o+1.))+(0.246099391883)*dim
        ref[1,4,2,1]=(-0.0958737570155)*(1+2.*(dim-1.)/(o+1.))+(-0.0803850673082)*dim
        ref[2,0,0,0]=(0.0482996624604)*(1+2.*(dim-1.)/(o+1.))+(-0.171234658375)*dim
        ref[2,0,0,1]=(0.638353303057)*(1+2.*(dim-1.)/(o+1.))+(0.770431646834)*dim
        ref[2,0,1,0]=(0.725020629449)*(1+2.*(dim-1.)/(o+1.))+(-0.432155213079)*dim
        ref[2,0,1,1]=(-0.438983419037)*(1+2.*(dim-1.)/(o+1.))+(0.713258422414)*dim
        ref[2,0,2,0]=(1.91447500178)*(1+2.*(dim-1.)/(o+1.))+(0.0177948876308)*dim
        ref[2,0,2,1]=(0.764452327795)*(1+2.*(dim-1.)/(o+1.))+(1.50234050134)*dim
        ref[2,1,0,0]=(0.498336938411)*(1+2.*(dim-1.)/(o+1.))+(-0.547316751164)*dim
        ref[2,1,0,1]=(-0.809231518007)*(1+2.*(dim-1.)/(o+1.))+(0.562257458836)*dim
        ref[2,1,1,0]=(0.358494165977)*(1+2.*(dim-1.)/(o+1.))+(1.59250325722)*dim
        ref[2,1,1,1]=(-0.244841646941)*(1+2.*(dim-1.)/(o+1.))+(-1.24133734105)*dim
        ref[2,1,2,0]=(0.361536543287)*(1+2.*(dim-1.)/(o+1.))+(0.691200482629)*dim
        ref[2,1,2,1]=(-1.32376390702)*(1+2.*(dim-1.)/(o+1.))+(0.0224135356952)*dim
        ref[2,2,0,0]=(-0.312392576948)*(1+2.*(dim-1.)/(o+1.))+(-0.06747945238)*dim
        ref[2,2,0,1]=(-0.431605160301)*(1+2.*(dim-1.)/(o+1.))+(-0.421636641073)*dim
        ref[2,2,1,0]=(-0.589519172131)*(1+2.*(dim-1.)/(o+1.))+(-0.414769101251)*dim
        ref[2,2,1,1]=(-0.240659708601)*(1+2.*(dim-1.)/(o+1.))+(-0.589364433013)*dim
        ref[2,2,2,0]=(-0.722909959939)*(1+2.*(dim-1.)/(o+1.))+(0.851597634849)*dim
        ref[2,2,2,1]=(-0.440270728352)*(1+2.*(dim-1.)/(o+1.))+(1.816216624)*dim
        ref[2,3,0,0]=(1.17045302504)*(1+2.*(dim-1.)/(o+1.))+(0.524825015672)*dim
        ref[2,3,0,1]=(0.330856998602)*(1+2.*(dim-1.)/(o+1.))+(0.22315528114)*dim
        ref[2,3,1,0]=(1.30293776302)*(1+2.*(dim-1.)/(o+1.))+(-1.27501951078)*dim
        ref[2,3,1,1]=(0.203347651013)*(1+2.*(dim-1.)/(o+1.))+(-1.53261320445)*dim
        ref[2,3,2,0]=(-1.28694542186)*(1+2.*(dim-1.)/(o+1.))+(1.36002865967)*dim
        ref[2,3,2,1]=(0.458262163691)*(1+2.*(dim-1.)/(o+1.))+(-1.64758747517)*dim
        ref[2,4,0,0]=(-0.121698212896)*(1+2.*(dim-1.)/(o+1.))+(1.50704164854)*dim
        ref[2,4,0,1]=(0.778589099264)*(1+2.*(dim-1.)/(o+1.))+(0.668892364598)*dim
        ref[2,4,1,0]=(-1.08307710902)*(1+2.*(dim-1.)/(o+1.))+(0.82548622713)*dim
        ref[2,4,1,1]=(-0.537056351709)*(1+2.*(dim-1.)/(o+1.))+(1.70030227735)*dim
        ref[2,4,2,0]=(-0.304186980702)*(1+2.*(dim-1.)/(o+1.))+(0.38736577812)*dim
        ref[2,4,2,1]=(-0.9399542098)*(1+2.*(dim-1.)/(o+1.))+(-0.300064248815)*dim
        ref[3,0,0,0]=(-0.62892625728)*(1+2.*(dim-1.)/(o+1.))+(-0.641275948665)*dim
        ref[3,0,0,1]=(0.313089108286)*(1+2.*(dim-1.)/(o+1.))+(0.794209081837)*dim
        ref[3,0,1,0]=(0.343101289842)*(1+2.*(dim-1.)/(o+1.))+(0.81847403947)*dim
        ref[3,0,1,1]=(-1.73316157526)*(1+2.*(dim-1.)/(o+1.))+(-0.689801226243)*dim
        ref[3,0,2,0]=(0.950942254051)*(1+2.*(dim-1.)/(o+1.))+(-1.26322409499)*dim
        ref[3,0,2,1]=(0.651979063795)*(1+2.*(dim-1.)/(o+1.))+(-0.452840690279)*dim
        ref[3,1,0,0]=(-0.140265070679)*(1+2.*(dim-1.)/(o+1.))+(0.504336855579)*dim
        ref[3,1,0,1]=(-0.074714204279)*(1+2.*(dim-1.)/(o+1.))+(-0.383830687435)*dim
        ref[3,1,1,0]=(-0.0240930345095)*(1+2.*(dim-1.)/(o+1.))+(-0.30098905758)*dim
        ref[3,1,1,1]=(1.01141483666)*(1+2.*(dim-1.)/(o+1.))+(-0.453294297891)*dim
        ref[3,1,2,0]=(1.28477826738)*(1+2.*(dim-1.)/(o+1.))+(0.850142280731)*dim
        ref[3,1,2,1]=(-1.82840136237)*(1+2.*(dim-1.)/(o+1.))+(-0.651498817783)*dim
        ref[3,2,0,0]=(-0.687609595107)*(1+2.*(dim-1.)/(o+1.))+(0.453772317262)*dim
        ref[3,2,0,1]=(1.42350629505)*(1+2.*(dim-1.)/(o+1.))+(-0.739779539116)*dim
        ref[3,2,1,0]=(0.241203525596)*(1+2.*(dim-1.)/(o+1.))+(0.389178645253)*dim
        ref[3,2,1,1]=(0.930949546469)*(1+2.*(dim-1.)/(o+1.))+(1.12624216731)*dim
        ref[3,2,2,0]=(-0.736761856298)*(1+2.*(dim-1.)/(o+1.))+(0.102231272315)*dim
        ref[3,2,2,1]=(-1.47668816374)*(1+2.*(dim-1.)/(o+1.))+(-0.136900288452)*dim
        ref[3,3,0,0]=(-0.200131789262)*(1+2.*(dim-1.)/(o+1.))+(1.24257798386)*dim
        ref[3,3,0,1]=(-1.24299838876)*(1+2.*(dim-1.)/(o+1.))+(0.891724959663)*dim
        ref[3,3,1,0]=(-0.61535668979)*(1+2.*(dim-1.)/(o+1.))+(-1.70378258198)*dim
        ref[3,3,1,1]=(-0.905774525561)*(1+2.*(dim-1.)/(o+1.))+(0.472323889318)*dim
        ref[3,3,2,0]=(-0.537356496864)*(1+2.*(dim-1.)/(o+1.))+(-1.22645090578)*dim
        ref[3,3,2,1]=(-1.61666551906)*(1+2.*(dim-1.)/(o+1.))+(-0.168452197544)*dim
        ref[3,4,0,0]=(-0.863203442693)*(1+2.*(dim-1.)/(o+1.))+(0.363489776493)*dim
        ref[3,4,0,1]=(-0.521102239287)*(1+2.*(dim-1.)/(o+1.))+(-0.00618566424301)*dim
        ref[3,4,1,0]=(0.602096055815)*(1+2.*(dim-1.)/(o+1.))+(0.0366884579134)*dim
        ref[3,4,1,1]=(0.844734896522)*(1+2.*(dim-1.)/(o+1.))+(0.0719024636709)*dim
        ref[3,4,2,0]=(-0.122274967513)*(1+2.*(dim-1.)/(o+1.))+(-0.164971745593)*dim
        ref[3,4,2,1]=(0.893907978172)*(1+2.*(dim-1.)/(o+1.))+(-0.0157346112292)*dim
      else:
        arg[0,0,0,0]=(-0.0445963085788)*x[0]**o+(0.649401127647)*x[0]+(-0.949035518207)*x[1]**o+(0.620675629012)*x[1]+(0.379727527503)*x[2]**o+(0.0321033403935)*x[2]
        arg[0,0,0,1]=(-0.334483142061)*x[0]**o+(-0.408749824143)*x[0]+(-0.427042441291)*x[1]**o+(-0.263818177378)*x[1]+(-0.0490001897328)*x[2]**o+(0.925792628022)*x[2]
        arg[0,0,1,0]=(0.632498844224)*x[0]**o+(-0.756487058254)*x[0]+(-0.0277391041902)*x[1]**o+(0.615017171892)*x[1]+(-0.0956053270216)*x[2]**o+(-0.0383148218356)*x[2]
        arg[0,0,1,1]=(0.560189295143)*x[0]**o+(-0.565632451768)*x[0]+(0.131609132282)*x[1]**o+(0.689012872903)*x[1]+(-0.0542418297927)*x[2]**o+(-0.816694105058)*x[2]
        arg[0,0,2,0]=(-0.606523401606)*x[0]**o+(-0.129782638438)*x[0]+(0.286554604329)*x[1]**o+(-0.264164004519)*x[1]+(-0.145168356054)*x[2]**o+(0.346147680755)*x[2]
        arg[0,0,2,1]=(-0.847528471559)*x[0]**o+(-0.50641170288)*x[0]+(-0.773325001399)*x[1]**o+(0.517649361249)*x[1]+(-0.531000881434)*x[2]**o+(0.220222077563)*x[2]
        arg[0,1,0,0]=(-0.0946495378384)*x[0]**o+(-0.312650449416)*x[0]+(0.183321918505)*x[1]**o+(-0.359448777446)*x[1]+(0.713219915733)*x[2]**o+(-0.436742264028)*x[2]
        arg[0,1,0,1]=(0.856947337921)*x[0]**o+(0.497228192699)*x[0]+(0.351718755501)*x[1]**o+(-0.928210477058)*x[1]+(0.586853747677)*x[2]**o+(-0.915642724205)*x[2]
        arg[0,1,1,0]=(-0.745330469339)*x[0]**o+(-0.724610807566)*x[0]+(0.821804346007)*x[1]**o+(0.241991559492)*x[1]+(0.508380905493)*x[2]**o+(0.756706999359)*x[2]
        arg[0,1,1,1]=(-0.647952004012)*x[0]**o+(0.940964624139)*x[0]+(-0.620895246845)*x[1]**o+(0.577736885116)*x[1]+(0.228716730257)*x[2]**o+(0.336914427419)*x[2]
        arg[0,1,2,0]=(-0.671541115529)*x[0]**o+(0.153595444626)*x[0]+(0.420649971716)*x[1]**o+(-0.459078302559)*x[1]+(-0.583263239511)*x[2]**o+(-0.927250932344)*x[2]
        arg[0,1,2,1]=(-0.893954406097)*x[0]**o+(0.266297374506)*x[0]+(-0.777463374391)*x[1]**o+(0.0905062188555)*x[1]+(-0.668756964585)*x[2]**o+(-0.580980380472)*x[2]
        arg[0,2,0,0]=(0.424449538697)*x[0]**o+(0.0787502268502)*x[0]+(-0.645996834851)*x[1]**o+(-0.0409730346273)*x[1]+(0.835463212243)*x[2]**o+(0.716052012249)*x[2]
        arg[0,2,0,1]=(0.0241441154932)*x[0]**o+(-0.580242261617)*x[0]+(0.693484316968)*x[1]**o+(0.198598746328)*x[1]+(-0.586283866696)*x[2]**o+(-0.595008846645)*x[2]
        arg[0,2,1,0]=(0.074643270363)*x[0]**o+(0.454053362138)*x[0]+(0.861855819289)*x[1]**o+(-0.0310194739742)*x[1]+(-0.35738290172)*x[2]**o+(-0.400226698584)*x[2]
        arg[0,2,1,1]=(0.61780464269)*x[0]**o+(-0.702818770068)*x[0]+(0.0840653474306)*x[1]**o+(0.786842849442)*x[1]+(0.0704342755849)*x[2]**o+(0.386459053939)*x[2]
        arg[0,2,2,0]=(-0.815363379053)*x[0]**o+(-0.621823615812)*x[0]+(-0.13535005543)*x[1]**o+(-0.127080917834)*x[1]+(-0.178866000441)*x[2]**o+(-0.0753933476244)*x[2]
        arg[0,2,2,1]=(0.400194366678)*x[0]**o+(0.075003395019)*x[0]+(-0.35708995046)*x[1]**o+(-0.370028795924)*x[1]+(-0.674944523932)*x[2]**o+(-0.437373855225)*x[2]
        arg[0,3,0,0]=(-0.736317751792)*x[0]**o+(0.671627391036)*x[0]+(0.0768737429532)*x[1]**o+(-0.90036869218)*x[1]+(0.796767364925)*x[2]**o+(-0.185034722102)*x[2]
        arg[0,3,0,1]=(0.530921403378)*x[0]**o+(0.970385957718)*x[0]+(-0.75659968724)*x[1]**o+(-0.437200812366)*x[1]+(0.0519374612092)*x[2]**o+(0.657323406602)*x[2]
        arg[0,3,1,0]=(-0.109751102222)*x[0]**o+(-0.154990305064)*x[0]+(-0.502004367351)*x[1]**o+(0.535266424739)*x[1]+(-0.968926773395)*x[2]**o+(-0.138032126456)*x[2]
        arg[0,3,1,1]=(0.509506900641)*x[0]**o+(-0.872357145765)*x[0]+(-0.154129018225)*x[1]**o+(0.0190914956768)*x[1]+(0.119094726078)*x[2]**o+(0.29593674915)*x[2]
        arg[0,3,2,0]=(-0.133398979819)*x[0]**o+(-0.780852347888)*x[0]+(-0.346576066687)*x[1]**o+(0.187923380934)*x[1]+(0.939817544138)*x[2]**o+(-0.497081081638)*x[2]
        arg[0,3,2,1]=(-0.908783140711)*x[0]**o+(0.541754659297)*x[0]+(-0.297350066105)*x[1]**o+(0.118676031731)*x[1]+(0.985854358996)*x[2]**o+(-0.424715359194)*x[2]
        arg[0,4,0,0]=(0.341191520307)*x[0]**o+(-0.252381555804)*x[0]+(-0.307289805415)*x[1]**o+(0.762228901652)*x[1]+(0.390901852611)*x[2]**o+(0.147349139229)*x[2]
        arg[0,4,0,1]=(0.236625833503)*x[0]**o+(0.856322905978)*x[0]+(-0.160244298088)*x[1]**o+(-0.971940590481)*x[1]+(-0.0829366716357)*x[2]**o+(-0.687610358703)*x[2]
        arg[0,4,1,0]=(0.619591455739)*x[0]**o+(-0.62589432832)*x[0]+(-0.0275687591477)*x[1]**o+(0.367590198294)*x[1]+(0.745367285277)*x[2]**o+(0.232571513762)*x[2]
        arg[0,4,1,1]=(0.317669904199)*x[0]**o+(0.741734360559)*x[0]+(-0.923644886773)*x[1]**o+(-0.725994528938)*x[1]+(-0.949385908603)*x[2]**o+(-0.0515088731457)*x[2]
        arg[0,4,2,0]=(0.82547889841)*x[0]**o+(-0.926970312436)*x[0]+(0.946941744691)*x[1]**o+(-0.171582097545)*x[1]+(-0.185628570945)*x[2]**o+(0.40495393086)*x[2]
        arg[0,4,2,1]=(0.51879613184)*x[0]**o+(-0.517447781109)*x[0]+(-0.288826585096)*x[1]**o+(-0.736890998485)*x[1]+(-0.167119326999)*x[2]**o+(-0.653082372893)*x[2]
        arg[1,0,0,0]=(-0.879008868811)*x[0]**o+(-0.0613987509179)*x[0]+(-0.671288168832)*x[1]**o+(-0.959517297906)*x[1]+(0.292554039162)*x[2]**o+(0.942246368678)*x[2]
        arg[1,0,0,1]=(0.599956815218)*x[0]**o+(0.731601305574)*x[0]+(-0.61593947163)*x[1]**o+(-0.597924796707)*x[1]+(-0.791620695721)*x[2]**o+(-0.562978718552)*x[2]
        arg[1,0,1,0]=(0.448381207242)*x[0]**o+(-0.178004176549)*x[0]+(0.631841924913)*x[1]**o+(0.311836637044)*x[1]+(0.0619475841251)*x[2]**o+(0.606564490624)*x[2]
        arg[1,0,1,1]=(0.159907398582)*x[0]**o+(-0.278605461684)*x[0]+(0.822914081538)*x[1]**o+(0.75112795022)*x[1]+(0.91389895436)*x[2]**o+(0.929177149059)*x[2]
        arg[1,0,2,0]=(-0.472286972241)*x[0]**o+(0.00950883684535)*x[0]+(0.41110884679)*x[1]**o+(0.109422159345)*x[1]+(-0.707436024011)*x[2]**o+(0.795057014561)*x[2]
        arg[1,0,2,1]=(-0.0156784802605)*x[0]**o+(-0.657283760492)*x[0]+(-0.0422880440635)*x[1]**o+(-0.326436491982)*x[1]+(-0.543917764251)*x[2]**o+(-0.691698129266)*x[2]
        arg[1,1,0,0]=(-0.814424723604)*x[0]**o+(0.746801606056)*x[0]+(0.22025107532)*x[1]**o+(-0.785254149057)*x[1]+(-0.845637273475)*x[2]**o+(-0.468272754144)*x[2]
        arg[1,1,0,1]=(-0.0815177460988)*x[0]**o+(-0.47210691717)*x[0]+(-0.285964111717)*x[1]**o+(0.527387496664)*x[1]+(0.87059583429)*x[2]**o+(-0.543899395468)*x[2]
        arg[1,1,1,0]=(0.424163426295)*x[0]**o+(0.649692204305)*x[0]+(0.0570010318344)*x[1]**o+(-0.426047169951)*x[1]+(0.334750596342)*x[2]**o+(-0.746549570927)*x[2]
        arg[1,1,1,1]=(-0.120493849358)*x[0]**o+(0.567677110538)*x[0]+(-0.355891513526)*x[1]**o+(0.68069794787)*x[1]+(0.786560156246)*x[2]**o+(-0.422804642829)*x[2]
        arg[1,1,2,0]=(-0.80298202932)*x[0]**o+(-0.674766348751)*x[0]+(-0.29945072468)*x[1]**o+(0.385941328636)*x[1]+(-0.756272624651)*x[2]**o+(-0.607541099269)*x[2]
        arg[1,1,2,1]=(0.915020713521)*x[0]**o+(-0.875957199283)*x[0]+(0.0217597029367)*x[1]**o+(-0.0649461013997)*x[1]+(-0.302450259541)*x[2]**o+(0.738498325526)*x[2]
        arg[1,2,0,0]=(0.239905462913)*x[0]**o+(0.197348893262)*x[0]+(-0.151173414577)*x[1]**o+(0.368405558905)*x[1]+(0.891448032058)*x[2]**o+(0.874975769684)*x[2]
        arg[1,2,0,1]=(-0.304971614823)*x[0]**o+(-0.0166065674003)*x[0]+(0.883397157318)*x[1]**o+(-0.918803270424)*x[1]+(0.590875856583)*x[2]**o+(-0.360659400522)*x[2]
        arg[1,2,1,0]=(0.185368540041)*x[0]**o+(0.498866146124)*x[0]+(0.591601152982)*x[1]**o+(-0.986932631453)*x[1]+(-0.525706872664)*x[2]**o+(-0.0462993899877)*x[2]
        arg[1,2,1,1]=(0.260677894495)*x[0]**o+(-0.216847258131)*x[0]+(0.932876363707)*x[1]**o+(0.761630676808)*x[1]+(0.260889221066)*x[2]**o+(0.546259908582)*x[2]
        arg[1,2,2,0]=(-0.316803600202)*x[0]**o+(-0.898044300642)*x[0]+(-0.0865001657852)*x[1]**o+(0.760424275199)*x[1]+(0.771564532427)*x[2]**o+(-0.0852237819025)*x[2]
        arg[1,2,2,1]=(-0.54274371504)*x[0]**o+(0.0822524750896)*x[0]+(0.872655885939)*x[1]**o+(0.594054041006)*x[1]+(0.322150325252)*x[2]**o+(0.774020612668)*x[2]
        arg[1,3,0,0]=(0.034593806358)*x[0]**o+(0.541638857337)*x[0]+(0.291359214696)*x[1]**o+(-0.203872730757)*x[1]+(0.0312260615928)*x[2]**o+(0.62568815669)*x[2]
        arg[1,3,0,1]=(-0.31929050432)*x[0]**o+(-0.931756481553)*x[0]+(0.755537511071)*x[1]**o+(-0.182578278538)*x[1]+(0.00958902784528)*x[2]**o+(0.345520390047)*x[2]
        arg[1,3,1,0]=(-0.435083251076)*x[0]**o+(0.248332963857)*x[0]+(-0.122701495966)*x[1]**o+(0.669350594899)*x[1]+(-0.0758596665333)*x[2]**o+(0.0784997907526)*x[2]
        arg[1,3,1,1]=(-0.725479421981)*x[0]**o+(0.657242746138)*x[0]+(-0.199958263735)*x[1]**o+(-0.784165440236)*x[1]+(-0.550936676647)*x[2]**o+(-0.31907813038)*x[2]
        arg[1,3,2,0]=(0.657177346512)*x[0]**o+(-0.168053866293)*x[0]+(0.782142168258)*x[1]**o+(0.660514922866)*x[1]+(0.788106295165)*x[2]**o+(-0.254290645125)*x[2]
        arg[1,3,2,1]=(-0.215166688682)*x[0]**o+(0.743577328376)*x[0]+(-0.374827404054)*x[1]**o+(0.267885661143)*x[1]+(0.728818301014)*x[2]**o+(0.6828999278)*x[2]
        arg[1,4,0,0]=(0.371646403413)*x[0]**o+(0.00998777673275)*x[0]+(-0.313378345871)*x[1]**o+(-0.952580545501)*x[1]+(-0.680596974717)*x[2]**o+(-0.181426263871)*x[2]
        arg[1,4,0,1]=(-0.828437965469)*x[0]**o+(0.302443878606)*x[0]+(0.581310358652)*x[1]**o+(0.714092786618)*x[1]+(0.650179420347)*x[2]**o+(0.663482256421)*x[2]
        arg[1,4,1,0]=(0.220318191456)*x[0]**o+(0.443108440795)*x[0]+(-0.538413779146)*x[1]**o+(0.660052783252)*x[1]+(-0.846698606343)*x[2]**o+(-0.0447806256913)*x[2]
        arg[1,4,1,1]=(0.866921895372)*x[0]**o+(0.776291161128)*x[0]+(-0.385318933957)*x[1]**o+(0.841514060614)*x[1]+(-0.401465329431)*x[2]**o+(0.474475442638)*x[2]
        arg[1,4,2,0]=(-0.222535436657)*x[0]**o+(0.921341450452)*x[0]+(-0.765043798058)*x[1]**o+(-0.492946667792)*x[1]+(0.758817008648)*x[2]**o+(0.0950595321909)*x[2]
        arg[1,4,2,1]=(0.333884542089)*x[0]**o+(0.671265678974)*x[0]+(-0.719262407869)*x[1]**o+(0.498176251457)*x[1]+(-0.867726438404)*x[2]**o+(-0.872614195608)*x[2]
        arg[2,0,0,0]=(-0.303331399994)*x[0]**o+(-0.563887060818)*x[0]+(0.64537626236)*x[1]**o+(-0.657611314342)*x[1]+(0.800882746828)*x[2]**o+(0.700671892157)*x[2]
        arg[2,0,0,1]=(0.417691239931)*x[0]**o+(-0.822789801933)*x[0]+(0.712214951539)*x[1]**o+(0.160960575173)*x[1]+(0.0400205244239)*x[2]**o+(0.49892939112)*x[2]
        arg[2,0,1,0]=(-0.828677767124)*x[0]**o+(-0.347411290435)*x[0]+(-0.700177402223)*x[1]**o+(-0.42498634463)*x[1]+(0.984172081328)*x[2]**o+(-0.677168643292)*x[2]
        arg[2,0,1,1]=(-0.423091398412)*x[0]**o+(-0.786913239913)*x[0]+(-0.888102258019)*x[1]**o+(-0.119525894432)*x[1]+(-0.158513498081)*x[2]**o+(0.503403674219)*x[2]
        arg[2,0,2,0]=(0.547217868614)*x[0]**o+(-0.737401808325)*x[0]+(-0.865530737659)*x[1]**o+(-0.750671412749)*x[1]+(-0.06731420082)*x[2]**o+(0.658572191108)*x[2]
        arg[2,0,2,1]=(0.501785022047)*x[0]**o+(-0.0542441989122)*x[0]+(-0.379614025724)*x[1]**o+(-0.785618793316)*x[1]+(0.994476365391)*x[2]**o+(-0.436996817494)*x[2]
        arg[2,1,0,0]=(0.0329439903445)*x[0]**o+(0.688097592632)*x[0]+(0.554030860489)*x[1]**o+(0.60949736297)*x[1]+(0.499716171437)*x[2]**o+(0.501464282299)*x[2]
        arg[2,1,0,1]=(0.459629414019)*x[0]**o+(0.810658146454)*x[0]+(-0.902023596001)*x[1]**o+(-0.982998009297)*x[1]+(0.487736529196)*x[2]**o+(-0.509164692477)*x[2]
        arg[2,1,1,0]=(0.0818151896334)*x[0]**o+(-0.931023788966)*x[0]+(-0.217349745187)*x[1]**o+(-0.785586110912)*x[1]+(0.16783052001)*x[2]**o+(-0.515867681053)*x[2]
        arg[2,1,1,1]=(-0.758874975703)*x[0]**o+(0.219408816573)*x[0]+(0.508187203896)*x[1]**o+(0.306700464008)*x[1]+(-0.641891226105)*x[2]**o+(-0.137752888266)*x[2]
        arg[2,1,2,0]=(-0.952430879482)*x[0]**o+(0.179895758684)*x[0]+(0.502285174801)*x[1]**o+(-0.82300826605)*x[1]+(-0.267848001419)*x[2]**o+(-0.882529775244)*x[2]
        arg[2,1,2,1]=(0.371372868601)*x[0]**o+(-0.521801538326)*x[0]+(0.894906517432)*x[1]**o+(0.205799057307)*x[1]+(-0.315840560279)*x[2]**o+(0.65823096555)*x[2]
        arg[2,2,0,0]=(0.585088762354)*x[0]**o+(0.527662713737)*x[0]+(0.83511043292)*x[1]**o+(-0.263597549489)*x[1]+(-0.342210974079)*x[2]**o+(0.532149528802)*x[2]
        arg[2,2,0,1]=(0.985868978475)*x[0]**o+(0.162798954854)*x[0]+(-0.887525429296)*x[1]**o+(0.618905848201)*x[1]+(-0.213643353241)*x[2]**o+(0.0170513098359)*x[2]
        arg[2,2,1,0]=(-0.216402064475)*x[0]**o+(0.350853951704)*x[0]+(0.592219675443)*x[1]**o+(-0.915293916561)*x[1]+(0.160588856435)*x[2]**o+(-0.574686420146)*x[2]
        arg[2,2,1,1]=(0.233118497878)*x[0]**o+(0.482035958353)*x[0]+(-0.581971661478)*x[1]**o+(0.832450520208)*x[1]+(-0.430914444794)*x[2]**o+(-0.791658025494)*x[2]
        arg[2,2,2,0]=(-0.710516844)*x[0]**o+(0.0266391730256)*x[0]+(-0.154541997151)*x[1]**o+(-0.232843778379)*x[1]+(0.453522207498)*x[2]**o+(-0.688939516711)*x[2]
        arg[2,2,2,1]=(0.26860058468)*x[0]**o+(-0.969638661751)*x[0]+(0.623784614857)*x[1]**o+(0.686359051774)*x[1]+(0.677056704385)*x[2]**o+(0.156688176847)*x[2]
        arg[2,3,0,0]=(-0.0620909841534)*x[0]**o+(-0.364946660535)*x[0]+(-0.388275430806)*x[1]**o+(-0.947908160286)*x[1]+(0.0774302410344)*x[2]**o+(0.374469033649)*x[2]
        arg[2,3,0,1]=(0.164845147121)*x[0]**o+(0.480213031852)*x[0]+(0.58157030274)*x[1]**o+(0.882820521347)*x[1]+(0.584714825748)*x[2]**o+(0.0285154925376)*x[2]
        arg[2,3,1,0]=(0.664711234489)*x[0]**o+(0.247415435607)*x[0]+(0.741592287061)*x[1]**o+(0.963505986041)*x[1]+(-0.987844687003)*x[2]**o+(0.998294459659)*x[2]
        arg[2,3,1,1]=(-0.734529316764)*x[0]**o+(-0.177480004713)*x[0]+(0.923555424779)*x[1]**o+(0.727331498256)*x[1]+(0.452312367674)*x[2]**o+(0.00652243396085)*x[2]
        arg[2,3,2,0]=(-0.367377897953)*x[0]**o+(-0.157529442804)*x[0]+(-0.121881528621)*x[1]**o+(0.865116726884)*x[1]+(-0.200087079925)*x[2]**o+(0.930889629788)*x[2]
        arg[2,3,2,1]=(-0.900394562978)*x[0]**o+(-0.762334071565)*x[0]+(-0.723776245294)*x[1]**o+(0.332728994729)*x[1]+(0.390453779523)*x[2]**o+(0.0447350797932)*x[2]
        arg[2,4,0,0]=(0.178230443819)*x[0]**o+(0.0898732089623)*x[0]+(0.808448952186)*x[1]**o+(0.164579824494)*x[1]+(-0.553032263217)*x[2]**o+(-0.782141879825)*x[2]
        arg[2,4,0,1]=(-0.105582488753)*x[0]**o+(0.995415696949)*x[0]+(0.978971670365)*x[1]**o+(-0.977866735017)*x[1]+(0.156938966924)*x[2]**o+(-0.120871488936)*x[2]
        arg[2,4,1,0]=(0.652354601889)*x[0]**o+(0.381186764672)*x[0]+(-0.783479778972)*x[1]**o+(-0.158898829615)*x[1]+(0.794693790129)*x[2]**o+(0.914720093741)*x[2]
        arg[2,4,1,1]=(0.571506419376)*x[0]**o+(0.906086581152)*x[0]+(-0.798519792125)*x[1]**o+(-0.22840897962)*x[1]+(-0.750891721145)*x[2]**o+(-0.591014038221)*x[2]
        arg[2,4,2,0]=(0.515982716306)*x[0]**o+(0.299569326845)*x[0]+(-0.407423180268)*x[1]**o+(-0.252103363573)*x[1]+(-0.0228238994531)*x[2]**o+(-0.282150542844)*x[2]
        arg[2,4,2,1]=(0.5335415942)*x[0]**o+(-0.885728570489)*x[0]+(-0.750810131327)*x[1]**o+(0.604053940679)*x[1]+(-0.0946405866686)*x[2]**o+(-0.423845465234)*x[2]
        arg[3,0,0,0]=(-0.498025341136)*x[0]**o+(-0.296354416633)*x[0]+(-0.600874889655)*x[1]**o+(0.544042300882)*x[1]+(0.318784851038)*x[2]**o+(-0.801109111437)*x[2]
        arg[3,0,0,1]=(-0.492361949745)*x[0]**o+(-0.708188092599)*x[0]+(-0.863737808435)*x[1]**o+(0.674889485553)*x[1]+(-0.982254652233)*x[2]**o+(0.682172862189)*x[2]
        arg[3,0,1,0]=(-0.872692397532)*x[0]**o+(0.563852836655)*x[0]+(-0.0219331486599)*x[1]**o+(-0.858175679559)*x[1]+(0.802432631538)*x[2]**o+(-0.655545287387)*x[2]
        arg[3,0,1,1]=(-0.538117166854)*x[0]**o+(-0.62507518352)*x[0]+(0.101524298816)*x[1]**o+(0.590432448393)*x[1]+(0.668057302006)*x[2]**o+(0.405382330056)*x[2]
        arg[3,0,2,0]=(-0.322916455064)*x[0]**o+(0.950624857669)*x[0]+(0.485623846818)*x[1]**o+(-0.12611522301)*x[1]+(0.503586393973)*x[2]**o+(0.109451719709)*x[2]
        arg[3,0,2,1]=(0.0289965995645)*x[0]**o+(0.0101440661359)*x[0]+(-0.303827119958)*x[1]**o+(0.355269935141)*x[1]+(0.29889427526)*x[2]**o+(-0.707999988781)*x[2]
        arg[3,1,0,0]=(-0.541382521848)*x[0]**o+(0.534108535206)*x[0]+(0.875610773119)*x[1]**o+(-0.575849439451)*x[1]+(-0.0836939606164)*x[2]**o+(0.433678856889)*x[2]
        arg[3,1,0,1]=(0.0322576144027)*x[0]**o+(0.794219359514)*x[0]+(0.629442621265)*x[1]**o+(0.0517726053499)*x[1]+(-0.776050101686)*x[2]**o+(0.697910196708)*x[2]
        arg[3,1,1,0]=(-0.198211682395)*x[0]**o+(-0.926268101941)*x[0]+(-0.904760383993)*x[1]**o+(-0.820075667018)*x[1]+(-0.101262044428)*x[2]**o+(-0.0621221772941)*x[2]
        arg[3,1,1,1]=(-0.531324749521)*x[0]**o+(0.736017207718)*x[0]+(0.487299696259)*x[1]**o+(-0.00604423021253)*x[1]+(-0.635722590088)*x[2]**o+(-0.143501730927)*x[2]
        arg[3,1,2,0]=(-0.731421637848)*x[0]**o+(-0.0371648783851)*x[0]+(-0.290362466891)*x[1]**o+(-0.375048584106)*x[1]+(-0.854622532092)*x[2]**o+(-0.794295905388)*x[2]
        arg[3,1,2,1]=(-0.766383422105)*x[0]**o+(0.149248159984)*x[0]+(0.970103674649)*x[1]**o+(-0.117794368107)*x[1]+(0.0178775633047)*x[2]**o+(-0.221126829278)*x[2]
        arg[3,2,0,0]=(0.706586887601)*x[0]**o+(0.162520023447)*x[0]+(-0.731539880336)*x[1]**o+(-0.992558519713)*x[1]+(0.426569303014)*x[2]**o+(0.0451783819487)*x[2]
        arg[3,2,0,1]=(0.0392295508272)*x[0]**o+(-0.157929005382)*x[0]+(0.0636418899878)*x[1]**o+(0.892501759578)*x[1]+(-0.368161001401)*x[2]**o+(-0.715520794543)*x[2]
        arg[3,2,1,0]=(0.889667707133)*x[0]**o+(0.308604168817)*x[0]+(-0.275558663983)*x[1]**o+(-0.408210095215)*x[1]+(-0.0918000869138)*x[2]**o+(0.142754064645)*x[2]
        arg[3,2,1,1]=(0.585831954407)*x[0]**o+(-0.705004375516)*x[0]+(0.657277099061)*x[1]**o+(-0.972401794257)*x[1]+(-0.658049310396)*x[2]**o+(-0.553961555706)*x[2]
        arg[3,2,2,0]=(0.88916769079)*x[0]**o+(-0.1401162684)*x[0]+(-0.396496633317)*x[1]**o+(0.215029522669)*x[1]+(-0.619023294208)*x[2]**o+(-0.187670706968)*x[2]
        arg[3,2,2,1]=(-0.262983340384)*x[0]**o+(-0.763236528477)*x[0]+(-0.339720289405)*x[1]**o+(-0.283077066751)*x[1]+(0.625600522278)*x[2]**o+(-0.132697432298)*x[2]
        arg[3,3,0,0]=(-0.346408105223)*x[0]**o+(0.299760534041)*x[0]+(0.567951405594)*x[1]**o+(-0.880683463459)*x[1]+(0.784127947661)*x[2]**o+(0.268895744072)*x[2]
        arg[3,3,0,1]=(0.064159345936)*x[0]**o+(-0.685098798117)*x[0]+(-0.182310885124)*x[1]**o+(-0.700081075257)*x[1]+(0.543959312437)*x[2]**o+(0.125104170434)*x[2]
        arg[3,3,1,0]=(0.460375837083)*x[0]**o+(0.82355800923)*x[0]+(-0.186000244328)*x[1]**o+(0.8418355439)*x[1]+(-0.375269292115)*x[2]**o+(0.712431840155)*x[2]
        arg[3,3,1,1]=(0.57260867805)*x[0]**o+(-0.146405851776)*x[0]+(-0.50640779246)*x[1]**o+(0.741760309299)*x[1]+(0.069752390409)*x[2]**o+(-0.376724191471)*x[2]
        arg[3,3,2,0]=(0.795493408403)*x[0]**o+(0.396861648344)*x[0]+(-0.728716068325)*x[1]**o+(0.739330072416)*x[1]+(0.202019645585)*x[2]**o+(0.841240509566)*x[2]
        arg[3,3,2,1]=(-0.967372744621)*x[0]**o+(-0.334044549662)*x[0]+(-0.0686982670779)*x[1]**o+(0.180083293629)*x[1]+(-0.208697793793)*x[2]**o+(0.0305446253751)*x[2]
        arg[3,4,0,0]=(0.94168350996)*x[0]**o+(0.0545532586137)*x[0]+(-0.300430532207)*x[1]**o+(-0.960362884765)*x[1]+(0.738261342527)*x[2]**o+(-0.351477574571)*x[2]
        arg[3,4,0,1]=(-0.525893857873)*x[0]**o+(0.227252001588)*x[0]+(0.179831911446)*x[1]**o+(0.350952712138)*x[1]+(-0.671550635312)*x[2]**o+(-0.338834217909)*x[2]
        arg[3,4,1,0]=(-0.60610056656)*x[0]**o+(0.219192673102)*x[0]+(-0.801595717702)*x[1]**o+(-0.696853420225)*x[1]+(-0.495201990047)*x[2]**o+(-0.500683649414)*x[2]
        arg[3,4,1,1]=(-0.37343020332)*x[0]**o+(-0.585536591527)*x[0]+(0.81794997494)*x[1]**o+(-0.240232527585)*x[1]+(-0.538970681636)*x[2]**o+(-0.146105210473)*x[2]
        arg[3,4,2,0]=(0.857322149358)*x[0]**o+(-0.552847286493)*x[0]+(0.530970378294)*x[1]**o+(0.0746811386922)*x[1]+(0.946170976628)*x[2]**o+(-0.329916962713)*x[2]
        arg[3,4,2,1]=(-0.353323100685)*x[0]**o+(0.808796594618)*x[0]+(0.481450588496)*x[1]**o+(0.1202874944)*x[1]+(0.671293736572)*x[2]**o+(-0.905074286629)*x[2]
        ref[0,0,0,0]=(-0.613904299283)*(1+2.*(dim-1.)/(o+1.))+(1.30218009705)*dim
        ref[0,0,0,1]=(-0.810525773084)*(1+2.*(dim-1.)/(o+1.))+(0.253224626501)*dim
        ref[0,0,1,0]=(0.509154413012)*(1+2.*(dim-1.)/(o+1.))+(-0.179784708198)*dim
        ref[0,0,1,1]=(0.637556597632)*(1+2.*(dim-1.)/(o+1.))+(-0.693313683923)*dim
        ref[0,0,2,0]=(-0.465137153331)*(1+2.*(dim-1.)/(o+1.))+(-0.0477989622018)*dim
        ref[0,0,2,1]=(-2.15185435439)*(1+2.*(dim-1.)/(o+1.))+(0.231459735932)*dim
        ref[0,1,0,0]=(0.8018922964)*(1+2.*(dim-1.)/(o+1.))+(-1.10884149089)*dim
        ref[0,1,0,1]=(1.7955198411)*(1+2.*(dim-1.)/(o+1.))+(-1.34662500856)*dim
        ref[0,1,1,0]=(0.584854782162)*(1+2.*(dim-1.)/(o+1.))+(0.274087751285)*dim
        ref[0,1,1,1]=(-1.0401305206)*(1+2.*(dim-1.)/(o+1.))+(1.85561593667)*dim
        ref[0,1,2,0]=(-0.834154383324)*(1+2.*(dim-1.)/(o+1.))+(-1.23273379028)*dim
        ref[0,1,2,1]=(-2.34017474507)*(1+2.*(dim-1.)/(o+1.))+(-0.224176787111)*dim
        ref[0,2,0,0]=(0.613915916089)*(1+2.*(dim-1.)/(o+1.))+(0.753829204471)*dim
        ref[0,2,0,1]=(0.131344565765)*(1+2.*(dim-1.)/(o+1.))+(-0.976652361934)*dim
        ref[0,2,1,0]=(0.579116187931)*(1+2.*(dim-1.)/(o+1.))+(0.0228071895801)*dim
        ref[0,2,1,1]=(0.772304265705)*(1+2.*(dim-1.)/(o+1.))+(0.470483133313)*dim
        ref[0,2,2,0]=(-1.12957943492)*(1+2.*(dim-1.)/(o+1.))+(-0.82429788127)*dim
        ref[0,2,2,1]=(-0.631840107715)*(1+2.*(dim-1.)/(o+1.))+(-0.732399256131)*dim
        ref[0,3,0,0]=(0.137323356087)*(1+2.*(dim-1.)/(o+1.))+(-0.413776023245)*dim
        ref[0,3,0,1]=(-0.173740822652)*(1+2.*(dim-1.)/(o+1.))+(1.19050855195)*dim
        ref[0,3,1,0]=(-1.58068224297)*(1+2.*(dim-1.)/(o+1.))+(0.24224399322)*dim
        ref[0,3,1,1]=(0.474472608494)*(1+2.*(dim-1.)/(o+1.))+(-0.557328900938)*dim
        ref[0,3,2,0]=(0.459842497633)*(1+2.*(dim-1.)/(o+1.))+(-1.09001004859)*dim
        ref[0,3,2,1]=(-0.22027884782)*(1+2.*(dim-1.)/(o+1.))+(0.235715331835)*dim
        ref[0,4,0,0]=(0.424803567504)*(1+2.*(dim-1.)/(o+1.))+(0.657196485077)*dim
        ref[0,4,0,1]=(-0.00655513622033)*(1+2.*(dim-1.)/(o+1.))+(-0.803228043207)*dim
        ref[0,4,1,0]=(1.33738998187)*(1+2.*(dim-1.)/(o+1.))+(-0.0257326162636)*dim
        ref[0,4,1,1]=(-1.55536089118)*(1+2.*(dim-1.)/(o+1.))+(-0.0357690415245)*dim
        ref[0,4,2,0]=(1.58679207216)*(1+2.*(dim-1.)/(o+1.))+(-0.693598479122)*dim
        ref[0,4,2,1]=(0.0628502197447)*(1+2.*(dim-1.)/(o+1.))+(-1.90742115249)*dim
        ref[1,0,0,0]=(-1.25774299848)*(1+2.*(dim-1.)/(o+1.))+(-0.0786696801461)*dim
        ref[1,0,0,1]=(-0.807603352133)*(1+2.*(dim-1.)/(o+1.))+(-0.429302209684)*dim
        ref[1,0,1,0]=(1.14217071628)*(1+2.*(dim-1.)/(o+1.))+(0.740396951119)*dim
        ref[1,0,1,1]=(1.89672043448)*(1+2.*(dim-1.)/(o+1.))+(1.4016996376)*dim
        ref[1,0,2,0]=(-0.768614149462)*(1+2.*(dim-1.)/(o+1.))+(0.913988010751)*dim
        ref[1,0,2,1]=(-0.601884288575)*(1+2.*(dim-1.)/(o+1.))+(-1.67541838174)*dim
        ref[1,1,0,0]=(-1.43981092176)*(1+2.*(dim-1.)/(o+1.))+(-0.506725297145)*dim
        ref[1,1,0,1]=(0.503113976474)*(1+2.*(dim-1.)/(o+1.))+(-0.488618815974)*dim
        ref[1,1,1,0]=(0.815915054472)*(1+2.*(dim-1.)/(o+1.))+(-0.522904536573)*dim
        ref[1,1,1,1]=(0.310174793362)*(1+2.*(dim-1.)/(o+1.))+(0.825570415579)*dim
        ref[1,1,2,0]=(-1.85870537865)*(1+2.*(dim-1.)/(o+1.))+(-0.896366119383)*dim
        ref[1,1,2,1]=(0.634330156916)*(1+2.*(dim-1.)/(o+1.))+(-0.202404975156)*dim
        ref[1,2,0,0]=(0.980180080393)*(1+2.*(dim-1.)/(o+1.))+(1.44073022185)*dim
        ref[1,2,0,1]=(1.16930139908)*(1+2.*(dim-1.)/(o+1.))+(-1.29606923835)*dim
        ref[1,2,1,0]=(0.251262820359)*(1+2.*(dim-1.)/(o+1.))+(-0.534365875316)*dim
        ref[1,2,1,1]=(1.45444347927)*(1+2.*(dim-1.)/(o+1.))+(1.09104332726)*dim
        ref[1,2,2,0]=(0.36826076644)*(1+2.*(dim-1.)/(o+1.))+(-0.222843807345)*dim
        ref[1,2,2,1]=(0.652062496151)*(1+2.*(dim-1.)/(o+1.))+(1.45032712876)*dim
        ref[1,3,0,0]=(0.357179082647)*(1+2.*(dim-1.)/(o+1.))+(0.96345428327)*dim
        ref[1,3,0,1]=(0.445836034597)*(1+2.*(dim-1.)/(o+1.))+(-0.768814370045)*dim
        ref[1,3,1,0]=(-0.633644413575)*(1+2.*(dim-1.)/(o+1.))+(0.996183349509)*dim
        ref[1,3,1,1]=(-1.47637436236)*(1+2.*(dim-1.)/(o+1.))+(-0.446000824477)*dim
        ref[1,3,2,0]=(2.22742580994)*(1+2.*(dim-1.)/(o+1.))+(0.238170411449)*dim
        ref[1,3,2,1]=(0.138824208277)*(1+2.*(dim-1.)/(o+1.))+(1.69436291732)*dim
        ref[1,4,0,0]=(-0.622328917175)*(1+2.*(dim-1.)/(o+1.))+(-1.12401903264)*dim
        ref[1,4,0,1]=(0.40305181353)*(1+2.*(dim-1.)/(o+1.))+(1.68001892165)*dim
        ref[1,4,1,0]=(-1.16479419403)*(1+2.*(dim-1.)/(o+1.))+(1.05838059836)*dim
        ref[1,4,1,1]=(0.0801376319834)*(1+2.*(dim-1.)/(o+1.))+(2.09228066438)*dim
        ref[1,4,2,0]=(-0.228762226067)*(1+2.*(dim-1.)/(o+1.))+(0.523454314851)*dim
        ref[1,4,2,1]=(-1.25310430418)*(1+2.*(dim-1.)/(o+1.))+(0.296827734822)*dim
        ref[2,0,0,0]=(1.14292760919)*(1+2.*(dim-1.)/(o+1.))+(-0.520826483003)*dim
        ref[2,0,0,1]=(1.16992671589)*(1+2.*(dim-1.)/(o+1.))+(-0.162899835641)*dim
        ref[2,0,1,0]=(-0.544683088018)*(1+2.*(dim-1.)/(o+1.))+(-1.44956627836)*dim
        ref[2,0,1,1]=(-1.46970715451)*(1+2.*(dim-1.)/(o+1.))+(-0.403035460127)*dim
        ref[2,0,2,0]=(-0.385627069865)*(1+2.*(dim-1.)/(o+1.))+(-0.829501029966)*dim
        ref[2,0,2,1]=(1.11664736171)*(1+2.*(dim-1.)/(o+1.))+(-1.27685980972)*dim
        ref[2,1,0,0]=(1.08669102227)*(1+2.*(dim-1.)/(o+1.))+(1.7990592379)*dim
        ref[2,1,0,1]=(0.0453423472143)*(1+2.*(dim-1.)/(o+1.))+(-0.68150455532)*dim
        ref[2,1,1,0]=(0.0322959644566)*(1+2.*(dim-1.)/(o+1.))+(-2.23247758093)*dim
        ref[2,1,1,1]=(-0.892578997912)*(1+2.*(dim-1.)/(o+1.))+(0.388356392315)*dim
        ref[2,1,2,0]=(-0.7179937061)*(1+2.*(dim-1.)/(o+1.))+(-1.52564228261)*dim
        ref[2,1,2,1]=(0.950438825754)*(1+2.*(dim-1.)/(o+1.))+(0.342228484532)*dim
        ref[2,2,0,0]=(1.0779882212)*(1+2.*(dim-1.)/(o+1.))+(0.796214693049)*dim
        ref[2,2,0,1]=(-0.115299804062)*(1+2.*(dim-1.)/(o+1.))+(0.798756112891)*dim
        ref[2,2,1,0]=(0.536406467403)*(1+2.*(dim-1.)/(o+1.))+(-1.139126385)*dim
        ref[2,2,1,1]=(-0.779767608394)*(1+2.*(dim-1.)/(o+1.))+(0.522828453067)*dim
        ref[2,2,2,0]=(-0.411536633652)*(1+2.*(dim-1.)/(o+1.))+(-0.895144122064)*dim
        ref[2,2,2,1]=(1.56944190392)*(1+2.*(dim-1.)/(o+1.))+(-0.12659143313)*dim
        ref[2,3,0,0]=(-0.372936173925)*(1+2.*(dim-1.)/(o+1.))+(-0.938385787171)*dim
        ref[2,3,0,1]=(1.33113027561)*(1+2.*(dim-1.)/(o+1.))+(1.39154904574)*dim
        ref[2,3,1,0]=(0.418458834548)*(1+2.*(dim-1.)/(o+1.))+(2.20921588131)*dim
        ref[2,3,1,1]=(0.64133847569)*(1+2.*(dim-1.)/(o+1.))+(0.556373927504)*dim
        ref[2,3,2,0]=(-0.689346506499)*(1+2.*(dim-1.)/(o+1.))+(1.63847691387)*dim
        ref[2,3,2,1]=(-1.23371702875)*(1+2.*(dim-1.)/(o+1.))+(-0.384869997042)*dim
        ref[2,4,0,0]=(0.433647132788)*(1+2.*(dim-1.)/(o+1.))+(-0.527688846368)*dim
        ref[2,4,0,1]=(1.03032814854)*(1+2.*(dim-1.)/(o+1.))+(-0.103322527004)*dim
        ref[2,4,1,0]=(0.663568613046)*(1+2.*(dim-1.)/(o+1.))+(1.1370080288)*dim
        ref[2,4,1,1]=(-0.977905093893)*(1+2.*(dim-1.)/(o+1.))+(0.0866635633103)*dim
        ref[2,4,2,0]=(0.085735636585)*(1+2.*(dim-1.)/(o+1.))+(-0.234684579572)*dim
        ref[2,4,2,1]=(-0.311909123795)*(1+2.*(dim-1.)/(o+1.))+(-0.705520095044)*dim
        ref[3,0,0,0]=(-0.780115379753)*(1+2.*(dim-1.)/(o+1.))+(-0.553421227188)*dim
        ref[3,0,0,1]=(-2.33835441041)*(1+2.*(dim-1.)/(o+1.))+(0.648874255143)*dim
        ref[3,0,1,0]=(-0.0921929146543)*(1+2.*(dim-1.)/(o+1.))+(-0.949868130291)*dim
        ref[3,0,1,1]=(0.231464433969)*(1+2.*(dim-1.)/(o+1.))+(0.370739594929)*dim
        ref[3,0,2,0]=(0.666293785727)*(1+2.*(dim-1.)/(o+1.))+(0.933961354368)*dim
        ref[3,0,2,1]=(0.0240637548661)*(1+2.*(dim-1.)/(o+1.))+(-0.342585987504)*dim
        ref[3,1,0,0]=(0.250534290654)*(1+2.*(dim-1.)/(o+1.))+(0.391937952644)*dim
        ref[3,1,0,1]=(-0.114349866018)*(1+2.*(dim-1.)/(o+1.))+(1.54390216157)*dim
        ref[3,1,1,0]=(-1.20423411082)*(1+2.*(dim-1.)/(o+1.))+(-1.80846594625)*dim
        ref[3,1,1,1]=(-0.679747643349)*(1+2.*(dim-1.)/(o+1.))+(0.586471246578)*dim
        ref[3,1,2,0]=(-1.87640663683)*(1+2.*(dim-1.)/(o+1.))+(-1.20650936788)*dim
        ref[3,1,2,1]=(0.221597815849)*(1+2.*(dim-1.)/(o+1.))+(-0.189673037401)*dim
        ref[3,2,0,0]=(0.40161631028)*(1+2.*(dim-1.)/(o+1.))+(-0.784860114318)*dim
        ref[3,2,0,1]=(-0.265289560586)*(1+2.*(dim-1.)/(o+1.))+(0.0190519596518)*dim
        ref[3,2,1,0]=(0.522308956237)*(1+2.*(dim-1.)/(o+1.))+(0.0431481382473)*dim
        ref[3,2,1,1]=(0.585059743072)*(1+2.*(dim-1.)/(o+1.))+(-2.23136772548)*dim
        ref[3,2,2,0]=(-0.126352236735)*(1+2.*(dim-1.)/(o+1.))+(-0.112757452698)*dim
        ref[3,2,2,1]=(0.0228968924883)*(1+2.*(dim-1.)/(o+1.))+(-1.17901102753)*dim
        ref[3,3,0,0]=(1.00567124803)*(1+2.*(dim-1.)/(o+1.))+(-0.312027185345)*dim
        ref[3,3,0,1]=(0.425807773249)*(1+2.*(dim-1.)/(o+1.))+(-1.26007570294)*dim
        ref[3,3,1,0]=(-0.10089369936)*(1+2.*(dim-1.)/(o+1.))+(2.37782539328)*dim
        ref[3,3,1,1]=(0.135953275999)*(1+2.*(dim-1.)/(o+1.))+(0.218630266051)*dim
        ref[3,3,2,0]=(0.268796985663)*(1+2.*(dim-1.)/(o+1.))+(1.97743223033)*dim
        ref[3,3,2,1]=(-1.24476880549)*(1+2.*(dim-1.)/(o+1.))+(-0.123416630658)*dim
        ref[3,4,0,0]=(1.37951432028)*(1+2.*(dim-1.)/(o+1.))+(-1.25728720072)*dim
        ref[3,4,0,1]=(-1.01761258174)*(1+2.*(dim-1.)/(o+1.))+(0.239370495817)*dim
        ref[3,4,1,0]=(-1.90289827431)*(1+2.*(dim-1.)/(o+1.))+(-0.978344396537)*dim
        ref[3,4,1,1]=(-0.0944509100161)*(1+2.*(dim-1.)/(o+1.))+(-0.971874329585)*dim
        ref[3,4,2,0]=(2.33446350428)*(1+2.*(dim-1.)/(o+1.))+(-0.808083110513)*dim
        ref[3,4,2,1]=(0.799421224383)*(1+2.*(dim-1.)/(o+1.))+(0.0240098023897)*dim
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5, 3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnBoundary_fromData_ReducedSolution_rank0(self):
      """
      tests integral of rank 0 Data on the FunctionOnBoundary

      assumptions: ReducedSolution(self.domain) exists
                   self.domain supports integral on FunctionOnBoundary
      """
      o=1
      dim=self.domain.getDim()
      w_ref=FunctionOnBoundary(self.domain)
      w=ReducedSolution(self.domain)
      x=w.getX()
      arg=Data(0,(),w)
      if dim==2:
        arg=(-0.0177156089276)*x[0]+(-1.07750293477)*x[1]
        ref=(0.674554765151)*(1+2.*(dim-1.)/(o+1.))+(-1.76977330884)*dim
      else:
        arg=(0.304688056778)*x[0]+(0.548485298428)*x[1]+(0.672370309114)*x[2]
        ref=(0.0121419382123)*(1+2.*(dim-1.)/(o+1.))+(1.51340172611)*dim
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnBoundary_fromData_ReducedSolution_rank1(self):
      """
      tests integral of rank 1 Data on the FunctionOnBoundary

      assumptions: ReducedSolution(self.domain) exists
                   self.domain supports integral on FunctionOnBoundary
      """
      o=1
      dim=self.domain.getDim()
      w_ref=FunctionOnBoundary(self.domain)
      w=ReducedSolution(self.domain)
      x=w.getX()
      arg=Data(0,(2,),w)
      ref=numpy.zeros((2,),numpy.float_)
      if dim==2:
        arg[0]=(-1.81332918209)*x[0]+(0.825631690982)*x[1]
        arg[1]=(0.568335098118)*x[0]+(-0.504485431124)*x[1]
        ref[0]=(0.0338855214148)*(1+2.*(dim-1.)/(o+1.))+(-1.02158301252)*dim
        ref[1]=(0.348045520014)*(1+2.*(dim-1.)/(o+1.))+(-0.284195853019)*dim
      else:
        arg[0]=(-0.34247651789)*x[0]+(0.952434108232)*x[1]+(0.956872478767)*x[2]
        arg[1]=(-0.73212870207)*x[0]+(1.25659546958)*x[1]+(-0.690842998605)*x[2]
        ref[0]=(1.22184577749)*(1+2.*(dim-1.)/(o+1.))+(0.344984291617)*dim
        ref[1]=(-0.600249739257)*(1+2.*(dim-1.)/(o+1.))+(0.433873508162)*dim
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnBoundary_fromData_ReducedSolution_rank2(self):
      """
      tests integral of rank 2 Data on the FunctionOnBoundary

      assumptions: ReducedSolution(self.domain) exists
                   self.domain supports integral on FunctionOnBoundary
      """
      o=1
      dim=self.domain.getDim()
      w_ref=FunctionOnBoundary(self.domain)
      w=ReducedSolution(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5),w)
      ref=numpy.zeros((4, 5),numpy.float_)
      if dim==2:
        arg[0,0]=(-1.78262197231)*x[0]+(1.45174573417)*x[1]
        arg[0,1]=(0.213586223199)*x[0]+(-0.917876059198)*x[1]
        arg[0,2]=(-1.12034819339)*x[0]+(0.59993130233)*x[1]
        arg[0,3]=(-0.156732957083)*x[0]+(0.744660873695)*x[1]
        arg[0,4]=(0.217327014577)*x[0]+(1.40157420042)*x[1]
        arg[1,0]=(0.0588042984116)*x[0]+(1.40376148561)*x[1]
        arg[1,1]=(-0.264973303233)*x[0]+(1.77692483003)*x[1]
        arg[1,2]=(-1.17910097102)*x[0]+(-1.20301960774)*x[1]
        arg[1,3]=(-0.511409983468)*x[0]+(-1.02065230228)*x[1]
        arg[1,4]=(1.1254497221)*x[0]+(-1.23163629376)*x[1]
        arg[2,0]=(-0.780207200735)*x[0]+(-1.49983791458)*x[1]
        arg[2,1]=(0.20679132359)*x[0]+(0.640203014983)*x[1]
        arg[2,2]=(0.198135205267)*x[0]+(0.30781411018)*x[1]
        arg[2,3]=(0.465906968942)*x[0]+(-0.812198212)*x[1]
        arg[2,4]=(0.190357350559)*x[0]+(-1.5894498455)*x[1]
        arg[3,0]=(-1.1362863086)*x[0]+(-0.0654142353086)*x[1]
        arg[3,1]=(0.0545924852084)*x[0]+(0.00262152155818)*x[1]
        arg[3,2]=(-0.0497600832907)*x[0]+(-0.609396009303)*x[1]
        arg[3,3]=(0.347834191962)*x[0]+(-0.768159599963)*x[1]
        arg[3,4]=(-0.178383166761)*x[0]+(-0.0690942212822)*x[1]
        ref[0,0]=(0.0436748214249)*(1+2.*(dim-1.)/(o+1.))+(-0.374551059559)*dim
        ref[0,1]=(-1.05854206043)*(1+2.*(dim-1.)/(o+1.))+(0.354252224435)*dim
        ref[0,2]=(-0.71782952995)*(1+2.*(dim-1.)/(o+1.))+(0.197412638885)*dim
        ref[0,3]=(-0.255252590976)*(1+2.*(dim-1.)/(o+1.))+(0.843180507587)*dim
        ref[0,4]=(0.177908820578)*(1+2.*(dim-1.)/(o+1.))+(1.44099239442)*dim
        ref[1,0]=(1.56040501086)*(1+2.*(dim-1.)/(o+1.))+(-0.0978392268304)*dim
        ref[1,1]=(1.33164571828)*(1+2.*(dim-1.)/(o+1.))+(0.180305808513)*dim
        ref[1,2]=(-0.885397553926)*(1+2.*(dim-1.)/(o+1.))+(-1.49672302484)*dim
        ref[1,3]=(0.105451964854)*(1+2.*(dim-1.)/(o+1.))+(-1.63751425061)*dim
        ref[1,4]=(-0.774237262923)*(1+2.*(dim-1.)/(o+1.))+(0.668050691269)*dim
        ref[2,0]=(-0.626493344552)*(1+2.*(dim-1.)/(o+1.))+(-1.65355177076)*dim
        ref[2,1]=(0.383826450148)*(1+2.*(dim-1.)/(o+1.))+(0.463167888425)*dim
        ref[2,2]=(-0.530241676713)*(1+2.*(dim-1.)/(o+1.))+(1.03619099216)*dim
        ref[2,3]=(-1.18245389658)*(1+2.*(dim-1.)/(o+1.))+(0.836162653525)*dim
        ref[2,4]=(-1.32396763993)*(1+2.*(dim-1.)/(o+1.))+(-0.0751248550082)*dim
        ref[3,0]=(0.15610314985)*(1+2.*(dim-1.)/(o+1.))+(-1.35780369376)*dim
        ref[3,1]=(-0.376931239611)*(1+2.*(dim-1.)/(o+1.))+(0.434145246378)*dim
        ref[3,2]=(0.0188741177717)*(1+2.*(dim-1.)/(o+1.))+(-0.678030210365)*dim
        ref[3,3]=(-0.264501990209)*(1+2.*(dim-1.)/(o+1.))+(-0.155823417792)*dim
        ref[3,4]=(-0.394609208758)*(1+2.*(dim-1.)/(o+1.))+(0.147131820715)*dim
      else:
        arg[0,0]=(-1.55600852596)*x[0]+(-0.624393737912)*x[1]+(-0.747032596279)*x[2]
        arg[0,1]=(0.158636782846)*x[0]+(-0.751760245066)*x[1]+(-1.62081735864)*x[2]
        arg[0,2]=(1.18083264152)*x[0]+(-1.16696814878)*x[1]+(-0.567354369023)*x[2]
        arg[0,3]=(0.176344525636)*x[0]+(-0.825319538089)*x[1]+(0.32760822446)*x[2]
        arg[0,4]=(0.570849136808)*x[0]+(-0.86277959325)*x[1]+(-0.253106033513)*x[2]
        arg[1,0]=(0.325475826081)*x[0]+(0.338988467638)*x[1]+(-0.596090654044)*x[2]
        arg[1,1]=(-1.29950586693)*x[0]+(-0.980591079821)*x[1]+(-0.7995507704)*x[2]
        arg[1,2]=(0.108948096835)*x[0]+(0.323001832704)*x[1]+(1.09678982473)*x[2]
        arg[1,3]=(-0.0307191696266)*x[0]+(0.134014984197)*x[1]+(-0.271363693134)*x[2]
        arg[1,4]=(0.0166805959434)*x[0]+(-0.323260440667)*x[1]+(-0.0168023639197)*x[2]
        arg[2,0]=(1.66243206086)*x[0]+(-0.478403313987)*x[1]+(1.45469230407)*x[2]
        arg[2,1]=(0.635927142861)*x[0]+(0.00628277747568)*x[1]+(-0.335502970321)*x[2]
        arg[2,2]=(-0.168440428449)*x[0]+(-0.302470381257)*x[1]+(1.26059253147)*x[2]
        arg[2,3]=(-0.0183286590129)*x[0]+(-0.0950506276918)*x[1]+(-0.721364110928)*x[2]
        arg[2,4]=(-1.04551179644)*x[0]+(0.20728433783)*x[1]+(-0.631521487025)*x[2]
        arg[3,0]=(1.0112442447)*x[0]+(0.0279472213885)*x[1]+(-1.24445628735)*x[2]
        arg[3,1]=(-1.36197290176)*x[0]+(0.145431495907)*x[1]+(0.518562214899)*x[2]
        arg[3,2]=(-0.249567074212)*x[0]+(-1.02991047199)*x[1]+(1.34627546588)*x[2]
        arg[3,3]=(1.72315708213)*x[0]+(0.00515532937983)*x[1]+(-0.479917704073)*x[2]
        arg[3,4]=(1.16297504595)*x[0]+(-0.176518851423)*x[1]+(0.184727222764)*x[2]
        ref[0,0]=(-2.09480681002)*(1+2.*(dim-1.)/(o+1.))+(-0.832628050126)*dim
        ref[0,1]=(-0.759565702286)*(1+2.*(dim-1.)/(o+1.))+(-1.45437511857)*dim
        ref[0,2]=(-1.08318812741)*(1+2.*(dim-1.)/(o+1.))+(0.529698251127)*dim
        ref[0,3]=(1.48988138495)*(1+2.*(dim-1.)/(o+1.))+(-1.81124817294)*dim
        ref[0,4]=(0.116706866039)*(1+2.*(dim-1.)/(o+1.))+(-0.661743355994)*dim
        ref[1,0]=(0.584176790043)*(1+2.*(dim-1.)/(o+1.))+(-0.515803150369)*dim
        ref[1,1]=(-1.25638916576)*(1+2.*(dim-1.)/(o+1.))+(-1.82325855139)*dim
        ref[1,2]=(-0.276318333951)*(1+2.*(dim-1.)/(o+1.))+(1.80505808822)*dim
        ref[1,3]=(-0.732277841606)*(1+2.*(dim-1.)/(o+1.))+(0.564209963043)*dim
        ref[1,4]=(-0.540285295646)*(1+2.*(dim-1.)/(o+1.))+(0.216903087004)*dim
        ref[2,0]=(2.23339929969)*(1+2.*(dim-1.)/(o+1.))+(0.405321751257)*dim
        ref[2,1]=(0.706137929075)*(1+2.*(dim-1.)/(o+1.))+(-0.39943097906)*dim
        ref[2,2]=(-0.474421381251)*(1+2.*(dim-1.)/(o+1.))+(1.26410310301)*dim
        ref[2,3]=(1.43870260444)*(1+2.*(dim-1.)/(o+1.))+(-2.27344600207)*dim
        ref[2,4]=(-1.00154864599)*(1+2.*(dim-1.)/(o+1.))+(-0.468200299646)*dim
        ref[3,0]=(-1.42181650401)*(1+2.*(dim-1.)/(o+1.))+(1.21655168275)*dim
        ref[3,1]=(-0.853191175508)*(1+2.*(dim-1.)/(o+1.))+(0.155211984554)*dim
        ref[3,2]=(-0.823496284217)*(1+2.*(dim-1.)/(o+1.))+(0.890294203901)*dim
        ref[3,3]=(1.54168027082)*(1+2.*(dim-1.)/(o+1.))+(-0.293285563378)*dim
        ref[3,4]=(1.66366656195)*(1+2.*(dim-1.)/(o+1.))+(-0.492483144667)*dim
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnBoundary_fromData_ReducedSolution_rank3(self):
      """
      tests integral of rank 3 Data on the FunctionOnBoundary

      assumptions: ReducedSolution(self.domain) exists
                   self.domain supports integral on FunctionOnBoundary
      """
      o=1
      dim=self.domain.getDim()
      w_ref=FunctionOnBoundary(self.domain)
      w=ReducedSolution(self.domain)
      x=w.getX()
      arg=Data(0,(6, 2, 2),w)
      ref=numpy.zeros((6, 2, 2),numpy.float_)
      if dim==2:
        arg[0,0,0]=(-0.775903850503)*x[0]+(0.20657985447)*x[1]
        arg[0,0,1]=(1.42793412139)*x[0]+(1.11703130088)*x[1]
        arg[0,1,0]=(1.89750028353)*x[0]+(1.77903425203)*x[1]
        arg[0,1,1]=(-0.67280619036)*x[0]+(-0.283600511942)*x[1]
        arg[1,0,0]=(0.12590667282)*x[0]+(-0.109778270548)*x[1]
        arg[1,0,1]=(-1.06454156291)*x[0]+(-0.952754071628)*x[1]
        arg[1,1,0]=(1.27939438243)*x[0]+(1.41358757535)*x[1]
        arg[1,1,1]=(-1.68847666929)*x[0]+(-0.13467497005)*x[1]
        arg[2,0,0]=(0.116194220064)*x[0]+(0.457728827844)*x[1]
        arg[2,0,1]=(0.205487821412)*x[0]+(-0.657189976052)*x[1]
        arg[2,1,0]=(0.276192450801)*x[0]+(0.998191496643)*x[1]
        arg[2,1,1]=(1.63929218914)*x[0]+(0.268648283241)*x[1]
        arg[3,0,0]=(-0.419114071163)*x[0]+(-0.365697048391)*x[1]
        arg[3,0,1]=(0.802747458947)*x[0]+(0.137023645119)*x[1]
        arg[3,1,0]=(-0.0712635889142)*x[0]+(0.495489688269)*x[1]
        arg[3,1,1]=(-0.559073395964)*x[0]+(0.248364992354)*x[1]
        arg[4,0,0]=(0.372130696656)*x[0]+(1.22892579337)*x[1]
        arg[4,0,1]=(-1.47944537272)*x[0]+(1.46750125425)*x[1]
        arg[4,1,0]=(-0.0683410762004)*x[0]+(0.0982446782745)*x[1]
        arg[4,1,1]=(0.0698227670903)*x[0]+(0.227875508249)*x[1]
        arg[5,0,0]=(0.237110604157)*x[0]+(0.814767844631)*x[1]
        arg[5,0,1]=(0.172109909542)*x[0]+(-1.56055537357)*x[1]
        arg[5,1,0]=(0.92885449265)*x[0]+(0.820977987322)*x[1]
        arg[5,1,1]=(-1.76483093472)*x[0]+(0.526316351009)*x[1]
        ref[0,0,0]=(0.574633189339)*(1+2.*(dim-1.)/(o+1.))+(-1.14395718537)*dim
        ref[0,0,1]=(1.25832290545)*(1+2.*(dim-1.)/(o+1.))+(1.28664251682)*dim
        ref[0,1,0]=(1.75085914739)*(1+2.*(dim-1.)/(o+1.))+(1.92567538817)*dim
        ref[0,1,1]=(-1.57747201281)*(1+2.*(dim-1.)/(o+1.))+(0.621065310511)*dim
        ref[1,0,0]=(0.710883587246)*(1+2.*(dim-1.)/(o+1.))+(-0.694755184974)*dim
        ref[1,0,1]=(-0.870861363869)*(1+2.*(dim-1.)/(o+1.))+(-1.14643427067)*dim
        ref[1,1,0]=(1.21302125269)*(1+2.*(dim-1.)/(o+1.))+(1.4799607051)*dim
        ref[1,1,1]=(-0.163344093083)*(1+2.*(dim-1.)/(o+1.))+(-1.65980754626)*dim
        ref[2,0,0]=(1.18070929814)*(1+2.*(dim-1.)/(o+1.))+(-0.606786250235)*dim
        ref[2,0,1]=(-1.26011538751)*(1+2.*(dim-1.)/(o+1.))+(0.808413232875)*dim
        ref[2,1,0]=(-0.128253427149)*(1+2.*(dim-1.)/(o+1.))+(1.40263737459)*dim
        ref[2,1,1]=(1.22388877116)*(1+2.*(dim-1.)/(o+1.))+(0.68405170122)*dim
        ref[3,0,0]=(-1.51984063257)*(1+2.*(dim-1.)/(o+1.))+(0.735029513018)*dim
        ref[3,0,1]=(1.63702893126)*(1+2.*(dim-1.)/(o+1.))+(-0.697257827196)*dim
        ref[3,1,0]=(-0.274208683077)*(1+2.*(dim-1.)/(o+1.))+(0.698434782432)*dim
        ref[3,1,1]=(0.649113059071)*(1+2.*(dim-1.)/(o+1.))+(-0.95982146268)*dim
        ref[4,0,0]=(1.11723294318)*(1+2.*(dim-1.)/(o+1.))+(0.483823546848)*dim
        ref[4,0,1]=(-0.0369311601619)*(1+2.*(dim-1.)/(o+1.))+(0.0249870416905)*dim
        ref[4,1,0]=(-0.0969818481854)*(1+2.*(dim-1.)/(o+1.))+(0.126885450259)*dim
        ref[4,1,1]=(0.607834304349)*(1+2.*(dim-1.)/(o+1.))+(-0.31013602901)*dim
        ref[5,0,0]=(0.612356854718)*(1+2.*(dim-1.)/(o+1.))+(0.43952159407)*dim
        ref[5,0,1]=(-0.245849747535)*(1+2.*(dim-1.)/(o+1.))+(-1.1425957165)*dim
        ref[5,1,0]=(0.218025498167)*(1+2.*(dim-1.)/(o+1.))+(1.5318069818)*dim
        ref[5,1,1]=(-0.39419356418)*(1+2.*(dim-1.)/(o+1.))+(-0.844321019532)*dim
      else:
        arg[0,0,0]=(1.52622777835)*x[0]+(-0.627217363559)*x[1]+(-0.221339024656)*x[2]
        arg[0,0,1]=(-0.0735955925339)*x[0]+(-1.44492195287)*x[1]+(-0.0704081180966)*x[2]
        arg[0,1,0]=(-0.330700790074)*x[0]+(-1.04769579979)*x[1]+(0.508799976666)*x[2]
        arg[0,1,1]=(0.349349373439)*x[0]+(0.174054703574)*x[1]+(-0.129296220879)*x[2]
        arg[1,0,0]=(0.0683757878129)*x[0]+(1.33394387082)*x[1]+(0.892302980732)*x[2]
        arg[1,0,1]=(0.61014567223)*x[0]+(1.23846801628)*x[1]+(-0.990674941877)*x[2]
        arg[1,1,0]=(0.964575308424)*x[0]+(0.354822939168)*x[1]+(-0.466180761866)*x[2]
        arg[1,1,1]=(-1.79700594499)*x[0]+(-1.41821199521)*x[1]+(0.366764057361)*x[2]
        arg[2,0,0]=(0.670092254407)*x[0]+(-0.188523225253)*x[1]+(0.363046866484)*x[2]
        arg[2,0,1]=(0.420367065732)*x[0]+(0.798321474665)*x[1]+(-1.11347160928)*x[2]
        arg[2,1,0]=(0.54732858526)*x[0]+(0.249248458735)*x[1]+(1.33905485076)*x[2]
        arg[2,1,1]=(-1.45292600653)*x[0]+(-0.0838212918177)*x[1]+(-0.210961166185)*x[2]
        arg[3,0,0]=(-0.406792355772)*x[0]+(-0.230709788944)*x[1]+(1.12300383125)*x[2]
        arg[3,0,1]=(-1.02748413184)*x[0]+(-1.3840845484)*x[1]+(-0.181262479318)*x[2]
        arg[3,1,0]=(-1.39003872882)*x[0]+(-0.302577309791)*x[1]+(-0.627070288119)*x[2]
        arg[3,1,1]=(0.258496709277)*x[0]+(-1.32012543621)*x[1]+(0.921022534515)*x[2]
        arg[4,0,0]=(-0.639952053735)*x[0]+(0.172303718581)*x[1]+(-1.55894055416)*x[2]
        arg[4,0,1]=(-1.27660129414)*x[0]+(0.142177942574)*x[1]+(0.84156755899)*x[2]
        arg[4,1,0]=(0.49812465572)*x[0]+(1.53755810837)*x[1]+(0.10504146131)*x[2]
        arg[4,1,1]=(0.114534785643)*x[0]+(0.616103931999)*x[1]+(0.515764206037)*x[2]
        arg[5,0,0]=(0.22562941228)*x[0]+(0.70743104453)*x[1]+(0.882965136406)*x[2]
        arg[5,0,1]=(0.674003358504)*x[0]+(0.753008881236)*x[1]+(0.033083269963)*x[2]
        arg[5,1,0]=(-0.495809169314)*x[0]+(-1.09613938465)*x[1]+(-1.03297996938)*x[2]
        arg[5,1,1]=(-0.5222251005)*x[0]+(-0.56129023192)*x[1]+(0.529084741893)*x[2]
        ref[0,0,0]=(0.164333186185)*(1+2.*(dim-1.)/(o+1.))+(0.513338203948)*dim
        ref[0,0,1]=(-0.322072217228)*(1+2.*(dim-1.)/(o+1.))+(-1.26685344627)*dim
        ref[0,1,0]=(1.37061498302)*(1+2.*(dim-1.)/(o+1.))+(-2.24021159622)*dim
        ref[0,1,1]=(0.65996582358)*(1+2.*(dim-1.)/(o+1.))+(-0.265857967445)*dim
        ref[1,0,0]=(1.83616555678)*(1+2.*(dim-1.)/(o+1.))+(0.458457082585)*dim
        ref[1,0,1]=(1.01886896436)*(1+2.*(dim-1.)/(o+1.))+(-0.160930217723)*dim
        ref[1,1,0]=(0.95375222959)*(1+2.*(dim-1.)/(o+1.))+(-0.100534743865)*dim
        ref[1,1,1]=(-1.48059917075)*(1+2.*(dim-1.)/(o+1.))+(-1.3678547121)*dim
        ref[2,0,0]=(-0.0805152837328)*(1+2.*(dim-1.)/(o+1.))+(0.925131179371)*dim
        ref[2,0,1]=(0.155604473013)*(1+2.*(dim-1.)/(o+1.))+(-0.0503875419008)*dim
        ref[2,1,0]=(0.117237902139)*(1+2.*(dim-1.)/(o+1.))+(2.01839399262)*dim
        ref[2,1,1]=(0.196870433027)*(1+2.*(dim-1.)/(o+1.))+(-1.94457889755)*dim
        ref[3,0,0]=(-1.16272511547)*(1+2.*(dim-1.)/(o+1.))+(1.648226802)*dim
        ref[3,0,1]=(-1.54275766077)*(1+2.*(dim-1.)/(o+1.))+(-1.05007349878)*dim
        ref[3,1,0]=(-2.21939430114)*(1+2.*(dim-1.)/(o+1.))+(-0.100292025596)*dim
        ref[3,1,1]=(-0.974866109202)*(1+2.*(dim-1.)/(o+1.))+(0.834259916779)*dim
        ref[4,0,0]=(-1.81015652027)*(1+2.*(dim-1.)/(o+1.))+(-0.216432369042)*dim
        ref[4,0,1]=(-0.0692225921476)*(1+2.*(dim-1.)/(o+1.))+(-0.223633200428)*dim
        ref[4,1,0]=(0.907628019251)*(1+2.*(dim-1.)/(o+1.))+(1.23309620615)*dim
        ref[4,1,1]=(-0.77625708658)*(1+2.*(dim-1.)/(o+1.))+(2.02266001026)*dim
        ref[5,0,0]=(0.040336826806)*(1+2.*(dim-1.)/(o+1.))+(1.77568876641)*dim
        ref[5,0,1]=(-0.741065482842)*(1+2.*(dim-1.)/(o+1.))+(2.20116099255)*dim
        ref[5,1,0]=(-1.93385845474)*(1+2.*(dim-1.)/(o+1.))+(-0.691070068594)*dim
        ref[5,1,1]=(-0.539652361393)*(1+2.*(dim-1.)/(o+1.))+(-0.014778229134)*dim
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnBoundary_fromData_ReducedSolution_rank4(self):
      """
      tests integral of rank 4 Data on the FunctionOnBoundary

      assumptions: ReducedSolution(self.domain) exists
                   self.domain supports integral on FunctionOnBoundary
      """
      o=1
      dim=self.domain.getDim()
      w_ref=FunctionOnBoundary(self.domain)
      w=ReducedSolution(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5, 3, 2),w)
      ref=numpy.zeros((4, 5, 3, 2),numpy.float_)
      if dim==2:
        arg[0,0,0,0]=(0.139765044323)*x[0]+(0.763652540371)*x[1]
        arg[0,0,0,1]=(1.22422998499)*x[0]+(-0.704059494925)*x[1]
        arg[0,0,1,0]=(-0.0738568759745)*x[0]+(0.227297731345)*x[1]
        arg[0,0,1,1]=(-0.438016525512)*x[0]+(-0.00290173552923)*x[1]
        arg[0,0,2,0]=(-0.142466265385)*x[0]+(0.159086109387)*x[1]
        arg[0,0,2,1]=(-1.29207470861)*x[0]+(0.171024552932)*x[1]
        arg[0,1,0,0]=(0.228469040214)*x[0]+(-1.11207386711)*x[1]
        arg[0,1,0,1]=(-0.0716121324998)*x[0]+(-0.293214318059)*x[1]
        arg[0,1,1,0]=(-0.552331240231)*x[0]+(0.483311325934)*x[1]
        arg[0,1,1,1]=(0.759050036013)*x[0]+(-0.192218075397)*x[1]
        arg[0,1,2,0]=(0.736408127399)*x[0]+(0.61962966835)*x[1]
        arg[0,1,2,1]=(-0.170110592043)*x[0]+(1.12024324836)*x[1]
        arg[0,2,0,0]=(1.15358129137)*x[0]+(0.00288801991025)*x[1]
        arg[0,2,0,1]=(0.34101930461)*x[0]+(1.44561406668)*x[1]
        arg[0,2,1,0]=(1.20180064935)*x[0]+(1.53696717349)*x[1]
        arg[0,2,1,1]=(-0.955282659666)*x[0]+(-0.232022391367)*x[1]
        arg[0,2,2,0]=(0.0907361523468)*x[0]+(0.0735957910178)*x[1]
        arg[0,2,2,1]=(0.377295180443)*x[0]+(-0.694318356175)*x[1]
        arg[0,3,0,0]=(1.38936937129)*x[0]+(-1.11460156576)*x[1]
        arg[0,3,0,1]=(-0.477623155591)*x[0]+(-1.38984205639)*x[1]
        arg[0,3,1,0]=(0.0643534425446)*x[0]+(1.38744715898)*x[1]
        arg[0,3,1,1]=(0.0144448337619)*x[0]+(-0.107478295727)*x[1]
        arg[0,3,2,0]=(0.494522779873)*x[0]+(-0.623994602046)*x[1]
        arg[0,3,2,1]=(0.865352526765)*x[0]+(-0.0330471549519)*x[1]
        arg[0,4,0,0]=(0.876802965861)*x[0]+(-0.72991482831)*x[1]
        arg[0,4,0,1]=(-1.59217671369)*x[0]+(-1.17443275047)*x[1]
        arg[0,4,1,0]=(0.980556838858)*x[0]+(-0.134374268551)*x[1]
        arg[0,4,1,1]=(-0.799387621998)*x[0]+(0.290707689722)*x[1]
        arg[0,4,2,0]=(-0.152070506377)*x[0]+(-0.720051351365)*x[1]
        arg[0,4,2,1]=(-0.726386342076)*x[0]+(-0.821126213298)*x[1]
        arg[1,0,0,0]=(0.0108792321621)*x[0]+(1.08482229519)*x[1]
        arg[1,0,0,1]=(0.395066734758)*x[0]+(-0.147576996716)*x[1]
        arg[1,0,1,0]=(1.34674167646)*x[0]+(-1.7762597766)*x[1]
        arg[1,0,1,1]=(0.145955749421)*x[0]+(0.0758020791017)*x[1]
        arg[1,0,2,0]=(-1.3779896638)*x[0]+(0.879047617812)*x[1]
        arg[1,0,2,1]=(-0.034533186491)*x[0]+(-0.961616456719)*x[1]
        arg[1,1,0,0]=(1.93531133223)*x[0]+(0.085806770105)*x[1]
        arg[1,1,0,1]=(-1.40810156335)*x[0]+(0.647954800386)*x[1]
        arg[1,1,1,0]=(-0.898682130534)*x[0]+(0.0293908578939)*x[1]
        arg[1,1,1,1]=(0.906365050374)*x[0]+(0.493604087615)*x[1]
        arg[1,1,2,0]=(0.200697611638)*x[0]+(0.595460978794)*x[1]
        arg[1,1,2,1]=(-0.137201357175)*x[0]+(-0.456327010456)*x[1]
        arg[1,2,0,0]=(-0.435882659968)*x[0]+(-0.0312083604574)*x[1]
        arg[1,2,0,1]=(-0.678923893852)*x[0]+(-0.559827568731)*x[1]
        arg[1,2,1,0]=(-0.721204171338)*x[0]+(-0.054379212442)*x[1]
        arg[1,2,1,1]=(-0.0624007046506)*x[0]+(0.168316149134)*x[1]
        arg[1,2,2,0]=(-1.58839995565)*x[0]+(-0.212071943846)*x[1]
        arg[1,2,2,1]=(-1.12936118963)*x[0]+(1.37132899453)*x[1]
        arg[1,3,0,0]=(-0.382791970195)*x[0]+(-1.08331380956)*x[1]
        arg[1,3,0,1]=(-0.83075188923)*x[0]+(-0.771795988467)*x[1]
        arg[1,3,1,0]=(-0.491393878842)*x[0]+(0.256082770048)*x[1]
        arg[1,3,1,1]=(1.16535762668)*x[0]+(1.15868916612)*x[1]
        arg[1,3,2,0]=(0.930916100823)*x[0]+(0.447810335514)*x[1]
        arg[1,3,2,1]=(0.542008480891)*x[0]+(0.428660301354)*x[1]
        arg[1,4,0,0]=(0.820283847448)*x[0]+(-0.861049765761)*x[1]
        arg[1,4,0,1]=(0.794597952616)*x[0]+(0.484963133102)*x[1]
        arg[1,4,1,0]=(-0.0864183500936)*x[0]+(-0.48652009912)*x[1]
        arg[1,4,1,1]=(-0.729261074705)*x[0]+(0.572520019661)*x[1]
        arg[1,4,2,0]=(0.874563670592)*x[0]+(-1.03978413308)*x[1]
        arg[1,4,2,1]=(-0.119980177921)*x[0]+(-0.416981275935)*x[1]
        arg[2,0,0,0]=(0.00851163981688)*x[0]+(0.0537719250367)*x[1]
        arg[2,0,0,1]=(0.325152071307)*x[0]+(0.852063357275)*x[1]
        arg[2,0,1,0]=(0.3798978673)*x[0]+(0.103107604149)*x[1]
        arg[2,0,1,1]=(0.241778324259)*x[0]+(0.840767359849)*x[1]
        arg[2,0,2,0]=(0.504220642882)*x[0]+(-0.296421218478)*x[1]
        arg[2,0,2,1]=(1.21850888213)*x[0]+(0.361422173343)*x[1]
        arg[2,1,0,0]=(1.23948863042)*x[0]+(-0.454591957846)*x[1]
        arg[2,1,0,1]=(-0.834341417745)*x[0]+(0.836809823188)*x[1]
        arg[2,1,1,0]=(0.305283593888)*x[0]+(-0.443709359754)*x[1]
        arg[2,1,1,1]=(-1.73751494542)*x[0]+(0.28249619071)*x[1]
        arg[2,1,2,0]=(1.64628846378)*x[0]+(0.929422692022)*x[1]
        arg[2,1,2,1]=(1.55964953056)*x[0]+(0.294507226966)*x[1]
        arg[2,2,0,0]=(-0.122178600734)*x[0]+(-0.836106215351)*x[1]
        arg[2,2,0,1]=(0.0560920262009)*x[0]+(-0.959524453552)*x[1]
        arg[2,2,1,0]=(-0.0235093950509)*x[0]+(1.00069942275)*x[1]
        arg[2,2,1,1]=(-1.6685983361)*x[0]+(-0.283415608089)*x[1]
        arg[2,2,2,0]=(1.88757268407)*x[0]+(-0.0970308753337)*x[1]
        arg[2,2,2,1]=(-0.620474974074)*x[0]+(0.24769276713)*x[1]
        arg[2,3,0,0]=(1.01637595389)*x[0]+(0.705160942822)*x[1]
        arg[2,3,0,1]=(1.14707395208)*x[0]+(0.705275592532)*x[1]
        arg[2,3,1,0]=(-0.758160952494)*x[0]+(0.305309048122)*x[1]
        arg[2,3,1,1]=(0.772402918026)*x[0]+(-0.052854508055)*x[1]
        arg[2,3,2,0]=(-0.154978687298)*x[0]+(0.418368826949)*x[1]
        arg[2,3,2,1]=(-0.845976442257)*x[0]+(0.20623978887)*x[1]
        arg[2,4,0,0]=(-0.466185710519)*x[0]+(-0.233766119139)*x[1]
        arg[2,4,0,1]=(-0.922877220469)*x[0]+(-0.688487528304)*x[1]
        arg[2,4,1,0]=(0.95836287786)*x[0]+(0.939609093639)*x[1]
        arg[2,4,1,1]=(-0.11982304155)*x[0]+(-0.257073186684)*x[1]
        arg[2,4,2,0]=(0.345817495959)*x[0]+(0.151979044778)*x[1]
        arg[2,4,2,1]=(-1.05379548205)*x[0]+(-0.436065928744)*x[1]
        arg[3,0,0,0]=(0.744945706398)*x[0]+(-0.649458468468)*x[1]
        arg[3,0,0,1]=(-1.04958965249)*x[0]+(0.639942303499)*x[1]
        arg[3,0,1,0]=(0.337716118074)*x[0]+(1.93458666396)*x[1]
        arg[3,0,1,1]=(0.303244531116)*x[0]+(0.436648998724)*x[1]
        arg[3,0,2,0]=(-1.55484904142)*x[0]+(0.0691158242075)*x[1]
        arg[3,0,2,1]=(-0.184571563941)*x[0]+(-0.617496186026)*x[1]
        arg[3,1,0,0]=(0.54665472418)*x[0]+(-0.685197219861)*x[1]
        arg[3,1,0,1]=(0.270341054659)*x[0]+(0.0123904138595)*x[1]
        arg[3,1,1,0]=(-0.551927232745)*x[0]+(-0.0835992799883)*x[1]
        arg[3,1,1,1]=(-0.731741032186)*x[0]+(0.351597478907)*x[1]
        arg[3,1,2,0]=(-0.678408784965)*x[0]+(-0.0779391332051)*x[1]
        arg[3,1,2,1]=(-0.873182924195)*x[0]+(-0.641772673128)*x[1]
        arg[3,2,0,0]=(-0.826223714873)*x[0]+(0.631844849849)*x[1]
        arg[3,2,0,1]=(-0.899714515249)*x[0]+(-0.010350665009)*x[1]
        arg[3,2,1,0]=(0.290678744119)*x[0]+(-0.5871053663)*x[1]
        arg[3,2,1,1]=(0.208288673139)*x[0]+(0.631917010925)*x[1]
        arg[3,2,2,0]=(-0.262706136542)*x[0]+(0.0853212850096)*x[1]
        arg[3,2,2,1]=(-1.28934754486)*x[0]+(0.686855142119)*x[1]
        arg[3,3,0,0]=(-1.36704032596)*x[0]+(-1.55543480824)*x[1]
        arg[3,3,0,1]=(-1.17014698336)*x[0]+(-0.652446157654)*x[1]
        arg[3,3,1,0]=(-0.547499757476)*x[0]+(-0.156519515585)*x[1]
        arg[3,3,1,1]=(0.228140026291)*x[0]+(-1.41014817371)*x[1]
        arg[3,3,2,0]=(-0.055056583329)*x[0]+(-0.412525603966)*x[1]
        arg[3,3,2,1]=(-0.524790489086)*x[0]+(-0.774411367008)*x[1]
        arg[3,4,0,0]=(-1.33491047191)*x[0]+(1.47855022808)*x[1]
        arg[3,4,0,1]=(-0.301405501225)*x[0]+(0.687178741851)*x[1]
        arg[3,4,1,0]=(0.213050803289)*x[0]+(0.428086322291)*x[1]
        arg[3,4,1,1]=(-1.67469009469)*x[0]+(0.857275314994)*x[1]
        arg[3,4,2,0]=(0.748812148687)*x[0]+(-0.0339722859922)*x[1]
        arg[3,4,2,1]=(-0.829454912563)*x[0]+(-0.637161668052)*x[1]
        ref[0,0,0,0]=(0.538138880568)*(1+2.*(dim-1.)/(o+1.))+(0.365278704126)*dim
        ref[0,0,0,1]=(0.532354921417)*(1+2.*(dim-1.)/(o+1.))+(-0.0121844313566)*dim
        ref[0,0,1,0]=(1.27037934234)*(1+2.*(dim-1.)/(o+1.))+(-1.11693848697)*dim
        ref[0,0,1,1]=(-0.317494874699)*(1+2.*(dim-1.)/(o+1.))+(-0.123423386343)*dim
        ref[0,0,2,0]=(-0.274542501137)*(1+2.*(dim-1.)/(o+1.))+(0.291162345138)*dim
        ref[0,0,2,1]=(-0.669231324784)*(1+2.*(dim-1.)/(o+1.))+(-0.451818830894)*dim
        ref[0,1,0,0]=(-1.55791021679)*(1+2.*(dim-1.)/(o+1.))+(0.674305389899)*dim
        ref[0,1,0,1]=(-1.04225461385)*(1+2.*(dim-1.)/(o+1.))+(0.67742816329)*dim
        ref[0,1,1,0]=(-0.553787932867)*(1+2.*(dim-1.)/(o+1.))+(0.48476801857)*dim
        ref[0,1,1,1]=(0.774284436582)*(1+2.*(dim-1.)/(o+1.))+(-0.207452475966)*dim
        ref[0,1,2,0]=(0.754663298028)*(1+2.*(dim-1.)/(o+1.))+(0.601374497721)*dim
        ref[0,1,2,1]=(0.378489135155)*(1+2.*(dim-1.)/(o+1.))+(0.571643521166)*dim
        ref[0,2,0,0]=(0.84270626402)*(1+2.*(dim-1.)/(o+1.))+(0.313763047258)*dim
        ref[0,2,0,1]=(0.741388749098)*(1+2.*(dim-1.)/(o+1.))+(1.0452446222)*dim
        ref[0,2,1,0]=(1.31801677967)*(1+2.*(dim-1.)/(o+1.))+(1.42075104317)*dim
        ref[0,2,1,1]=(0.342128499103)*(1+2.*(dim-1.)/(o+1.))+(-1.52943355014)*dim
        ref[0,2,2,0]=(-0.2097885639)*(1+2.*(dim-1.)/(o+1.))+(0.374120507264)*dim
        ref[0,2,2,1]=(0.517181369396)*(1+2.*(dim-1.)/(o+1.))+(-0.834204545127)*dim
        ref[0,3,0,0]=(0.546538945104)*(1+2.*(dim-1.)/(o+1.))+(-0.271771139576)*dim
        ref[0,3,0,1]=(-1.11041658618)*(1+2.*(dim-1.)/(o+1.))+(-0.757048625806)*dim
        ref[0,3,1,0]=(1.71248931933)*(1+2.*(dim-1.)/(o+1.))+(-0.260688717804)*dim
        ref[0,3,1,1]=(0.426783834682)*(1+2.*(dim-1.)/(o+1.))+(-0.519817296646)*dim
        ref[0,3,2,0]=(-0.159038051631)*(1+2.*(dim-1.)/(o+1.))+(0.0295662294573)*dim
        ref[0,3,2,1]=(-0.0935533877409)*(1+2.*(dim-1.)/(o+1.))+(0.925858759554)*dim
        ref[0,4,0,0]=(0.254413611162)*(1+2.*(dim-1.)/(o+1.))+(-0.107525473611)*dim
        ref[0,4,0,1]=(-1.80344635667)*(1+2.*(dim-1.)/(o+1.))+(-0.963163107492)*dim
        ref[0,4,1,0]=(0.77936654687)*(1+2.*(dim-1.)/(o+1.))+(0.0668160234379)*dim
        ref[0,4,1,1]=(-0.854656189741)*(1+2.*(dim-1.)/(o+1.))+(0.345976257465)*dim
        ref[0,4,2,0]=(0.236075361826)*(1+2.*(dim-1.)/(o+1.))+(-1.10819721957)*dim
        ref[0,4,2,1]=(-1.41443045549)*(1+2.*(dim-1.)/(o+1.))+(-0.133082099885)*dim
        ref[1,0,0,0]=(0.254601110632)*(1+2.*(dim-1.)/(o+1.))+(0.841100416717)*dim
        ref[1,0,0,1]=(-1.05142234949)*(1+2.*(dim-1.)/(o+1.))+(1.29891208753)*dim
        ref[1,0,1,0]=(-0.476182365596)*(1+2.*(dim-1.)/(o+1.))+(0.0466642654563)*dim
        ref[1,0,1,1]=(-1.2117362156)*(1+2.*(dim-1.)/(o+1.))+(1.43349404413)*dim
        ref[1,0,2,0]=(-0.82694411065)*(1+2.*(dim-1.)/(o+1.))+(0.328002064662)*dim
        ref[1,0,2,1]=(-0.784489634199)*(1+2.*(dim-1.)/(o+1.))+(-0.211660009011)*dim
        ref[1,1,0,0]=(1.67906802629)*(1+2.*(dim-1.)/(o+1.))+(0.342050076043)*dim
        ref[1,1,0,1]=(-0.798764638314)*(1+2.*(dim-1.)/(o+1.))+(0.03861787535)*dim
        ref[1,1,1,0]=(0.348279136631)*(1+2.*(dim-1.)/(o+1.))+(-1.21757040927)*dim
        ref[1,1,1,1]=(0.452104104709)*(1+2.*(dim-1.)/(o+1.))+(0.94786503328)*dim
        ref[1,1,2,0]=(0.855057493837)*(1+2.*(dim-1.)/(o+1.))+(-0.0588989034046)*dim
        ref[1,1,2,1]=(-0.666495642664)*(1+2.*(dim-1.)/(o+1.))+(0.0729672750328)*dim
        ref[1,2,0,0]=(-0.566691483673)*(1+2.*(dim-1.)/(o+1.))+(0.0996004632473)*dim
        ref[1,2,0,1]=(-0.127941886194)*(1+2.*(dim-1.)/(o+1.))+(-1.11080957639)*dim
        ref[1,2,1,0]=(-0.249856548379)*(1+2.*(dim-1.)/(o+1.))+(-0.525726835401)*dim
        ref[1,2,1,1]=(1.00195986166)*(1+2.*(dim-1.)/(o+1.))+(-0.896044417179)*dim
        ref[1,2,2,0]=(-0.96849834562)*(1+2.*(dim-1.)/(o+1.))+(-0.831973553872)*dim
        ref[1,2,2,1]=(0.204504064885)*(1+2.*(dim-1.)/(o+1.))+(0.0374637400154)*dim
        ref[1,3,0,0]=(-0.587139938504)*(1+2.*(dim-1.)/(o+1.))+(-0.878965841253)*dim
        ref[1,3,0,1]=(-0.013960550118)*(1+2.*(dim-1.)/(o+1.))+(-1.58858732758)*dim
        ref[1,3,1,0]=(-0.626445648631)*(1+2.*(dim-1.)/(o+1.))+(0.391134539838)*dim
        ref[1,3,1,1]=(0.793357894316)*(1+2.*(dim-1.)/(o+1.))+(1.53068889848)*dim
        ref[1,3,2,0]=(-0.30795291971)*(1+2.*(dim-1.)/(o+1.))+(1.68667935605)*dim
        ref[1,3,2,1]=(0.865506632636)*(1+2.*(dim-1.)/(o+1.))+(0.105162149608)*dim
        ref[1,4,0,0]=(0.0142921032519)*(1+2.*(dim-1.)/(o+1.))+(-0.0550580215643)*dim
        ref[1,4,0,1]=(1.74635688975)*(1+2.*(dim-1.)/(o+1.))+(-0.466795804028)*dim
        ref[1,4,1,0]=(0.202247079284)*(1+2.*(dim-1.)/(o+1.))+(-0.775185528498)*dim
        ref[1,4,1,1]=(0.518302547238)*(1+2.*(dim-1.)/(o+1.))+(-0.675043602282)*dim
        ref[1,4,2,0]=(-0.0110996487494)*(1+2.*(dim-1.)/(o+1.))+(-0.154120813742)*dim
        ref[1,4,2,1]=(0.445368223328)*(1+2.*(dim-1.)/(o+1.))+(-0.982329677184)*dim
        ref[2,0,0,0]=(1.11858205689)*(1+2.*(dim-1.)/(o+1.))+(-1.05629849203)*dim
        ref[2,0,0,1]=(0.326749663749)*(1+2.*(dim-1.)/(o+1.))+(0.850465764833)*dim
        ref[2,0,1,0]=(1.17260350442)*(1+2.*(dim-1.)/(o+1.))+(-0.689598032967)*dim
        ref[2,0,1,1]=(1.26453211291)*(1+2.*(dim-1.)/(o+1.))+(-0.181986428807)*dim
        ref[2,0,2,0]=(0.28543047736)*(1+2.*(dim-1.)/(o+1.))+(-0.0776310529561)*dim
        ref[2,0,2,1]=(1.67283723918)*(1+2.*(dim-1.)/(o+1.))+(-0.092906183711)*dim
        ref[2,1,0,0]=(0.375606493975)*(1+2.*(dim-1.)/(o+1.))+(0.409290178597)*dim
        ref[2,1,0,1]=(0.391717224425)*(1+2.*(dim-1.)/(o+1.))+(-0.389248818981)*dim
        ref[2,1,1,0]=(0.540376031372)*(1+2.*(dim-1.)/(o+1.))+(-0.678801797238)*dim
        ref[2,1,1,1]=(-0.781880488997)*(1+2.*(dim-1.)/(o+1.))+(-0.673138265714)*dim
        ref[2,1,2,0]=(1.19474704445)*(1+2.*(dim-1.)/(o+1.))+(1.38096411135)*dim
        ref[2,1,2,1]=(0.3333062674)*(1+2.*(dim-1.)/(o+1.))+(1.52085049012)*dim
        ref[2,2,0,0]=(-0.570564746042)*(1+2.*(dim-1.)/(o+1.))+(-0.387720070044)*dim
        ref[2,2,0,1]=(-0.906439974412)*(1+2.*(dim-1.)/(o+1.))+(0.00300754706095)*dim
        ref[2,2,1,0]=(1.35643048678)*(1+2.*(dim-1.)/(o+1.))+(-0.379240459088)*dim
        ref[2,2,1,1]=(-1.68522696847)*(1+2.*(dim-1.)/(o+1.))+(-0.266786975721)*dim
        ref[2,2,2,0]=(1.64773077393)*(1+2.*(dim-1.)/(o+1.))+(0.14281103481)*dim
        ref[2,2,2,1]=(0.161577377439)*(1+2.*(dim-1.)/(o+1.))+(-0.534359584383)*dim
        ref[2,3,0,0]=(0.117577368411)*(1+2.*(dim-1.)/(o+1.))+(1.6039595283)*dim
        ref[2,3,0,1]=(1.07270794583)*(1+2.*(dim-1.)/(o+1.))+(0.779641598778)*dim
        ref[2,3,1,0]=(0.940083930247)*(1+2.*(dim-1.)/(o+1.))+(-1.39293583462)*dim
        ref[2,3,1,1]=(-0.322170646263)*(1+2.*(dim-1.)/(o+1.))+(1.04171905623)*dim
        ref[2,3,2,0]=(-1.00498391043)*(1+2.*(dim-1.)/(o+1.))+(1.26837405008)*dim
        ref[2,3,2,1]=(0.148227404921)*(1+2.*(dim-1.)/(o+1.))+(-0.787964058307)*dim
        ref[2,4,0,0]=(-0.567878073929)*(1+2.*(dim-1.)/(o+1.))+(-0.132073755729)*dim
        ref[2,4,0,1]=(-0.265369521409)*(1+2.*(dim-1.)/(o+1.))+(-1.34599522736)*dim
        ref[2,4,1,0]=(0.87362235129)*(1+2.*(dim-1.)/(o+1.))+(1.02434962021)*dim
        ref[2,4,1,1]=(-1.05435356817)*(1+2.*(dim-1.)/(o+1.))+(0.67745733994)*dim
        ref[2,4,2,0]=(-0.0889174414067)*(1+2.*(dim-1.)/(o+1.))+(0.586713982143)*dim
        ref[2,4,2,1]=(-0.950446212097)*(1+2.*(dim-1.)/(o+1.))+(-0.539415198695)*dim
        ref[3,0,0,0]=(0.585351095733)*(1+2.*(dim-1.)/(o+1.))+(-0.489863857802)*dim
        ref[3,0,0,1]=(0.0818488289135)*(1+2.*(dim-1.)/(o+1.))+(-0.491496177908)*dim
        ref[3,0,1,0]=(1.83790469827)*(1+2.*(dim-1.)/(o+1.))+(0.43439808376)*dim
        ref[3,0,1,1]=(-0.812249042145)*(1+2.*(dim-1.)/(o+1.))+(1.55214257198)*dim
        ref[3,0,2,0]=(-1.62498605972)*(1+2.*(dim-1.)/(o+1.))+(0.139252842511)*dim
        ref[3,0,2,1]=(0.248660359392)*(1+2.*(dim-1.)/(o+1.))+(-1.05072810936)*dim
        ref[3,1,0,0]=(0.701570774477)*(1+2.*(dim-1.)/(o+1.))+(-0.840113270158)*dim
        ref[3,1,0,1]=(-1.01112314775)*(1+2.*(dim-1.)/(o+1.))+(1.29385461627)*dim
        ref[3,1,1,0]=(-0.38604780773)*(1+2.*(dim-1.)/(o+1.))+(-0.249478705003)*dim
        ref[3,1,1,1]=(-0.505728989541)*(1+2.*(dim-1.)/(o+1.))+(0.125585436262)*dim
        ref[3,1,2,0]=(0.170090820399)*(1+2.*(dim-1.)/(o+1.))+(-0.926438738569)*dim
        ref[3,1,2,1]=(-0.929125306455)*(1+2.*(dim-1.)/(o+1.))+(-0.585830290868)*dim
        ref[3,2,0,0]=(-0.155713736332)*(1+2.*(dim-1.)/(o+1.))+(-0.038665128691)*dim
        ref[3,2,0,1]=(-1.52172024486)*(1+2.*(dim-1.)/(o+1.))+(0.6116550646)*dim
        ref[3,2,1,0]=(0.320960641588)*(1+2.*(dim-1.)/(o+1.))+(-0.617387263769)*dim
        ref[3,2,1,1]=(1.35000898226)*(1+2.*(dim-1.)/(o+1.))+(-0.509803298192)*dim
        ref[3,2,2,0]=(-1.08694938974)*(1+2.*(dim-1.)/(o+1.))+(0.909564538204)*dim
        ref[3,2,2,1]=(-0.484173893901)*(1+2.*(dim-1.)/(o+1.))+(-0.11831850884)*dim
        ref[3,3,0,0]=(-1.30137401361)*(1+2.*(dim-1.)/(o+1.))+(-1.62110112059)*dim
        ref[3,3,0,1]=(-1.01955795295)*(1+2.*(dim-1.)/(o+1.))+(-0.803035188058)*dim
        ref[3,3,1,0]=(-0.86522946065)*(1+2.*(dim-1.)/(o+1.))+(0.161210187589)*dim
        ref[3,3,1,1]=(-1.45434545656)*(1+2.*(dim-1.)/(o+1.))+(0.272337309144)*dim
        ref[3,3,2,0]=(-1.09044516001)*(1+2.*(dim-1.)/(o+1.))+(0.622862972717)*dim
        ref[3,3,2,1]=(-1.56164744337)*(1+2.*(dim-1.)/(o+1.))+(0.262445587275)*dim
        ref[3,4,0,0]=(-0.0717688395634)*(1+2.*(dim-1.)/(o+1.))+(0.215408595737)*dim
        ref[3,4,0,1]=(-0.361057910751)*(1+2.*(dim-1.)/(o+1.))+(0.746831151378)*dim
        ref[3,4,1,0]=(1.3065764036)*(1+2.*(dim-1.)/(o+1.))+(-0.665439278018)*dim
        ref[3,4,1,1]=(0.00897444539195)*(1+2.*(dim-1.)/(o+1.))+(-0.826389225091)*dim
        ref[3,4,2,0]=(0.0968985993748)*(1+2.*(dim-1.)/(o+1.))+(0.61794126332)*dim
        ref[3,4,2,1]=(-0.247002319776)*(1+2.*(dim-1.)/(o+1.))+(-1.21961426084)*dim
      else:
        arg[0,0,0,0]=(0.249559417601)*x[0]+(-1.12919415836)*x[1]+(0.535083953587)*x[2]
        arg[0,0,0,1]=(-1.61154525877)*x[0]+(-0.339535503099)*x[1]+(-0.3734414422)*x[2]
        arg[0,0,1,0]=(-1.33426413331)*x[0]+(0.503654661149)*x[1]+(0.900718166029)*x[2]
        arg[0,0,1,1]=(-1.47348135079)*x[0]+(0.294426872223)*x[1]+(-0.092932038156)*x[2]
        arg[0,0,2,0]=(-0.659338105033)*x[0]+(0.276825152205)*x[1]+(-0.825904690993)*x[2]
        arg[0,0,2,1]=(-0.871512897535)*x[0]+(0.16237107696)*x[1]+(-1.37935248231)*x[2]
        arg[0,1,0,0]=(-0.83315792018)*x[0]+(-0.00675809409276)*x[1]+(0.464774846632)*x[2]
        arg[0,1,0,1]=(0.527155362932)*x[0]+(0.856707619808)*x[1]+(-1.68267649456)*x[2]
        arg[0,1,1,0]=(-0.524091384241)*x[0]+(-0.540620074365)*x[1]+(0.992745803077)*x[2]
        arg[0,1,1,1]=(1.61356254869)*x[0]+(1.50422956091)*x[1]+(0.522074610423)*x[2]
        arg[0,1,2,0]=(-0.786383950132)*x[0]+(0.240998133454)*x[1]+(-0.609039834527)*x[2]
        arg[0,1,2,1]=(-0.549608404174)*x[0]+(-0.850827315095)*x[1]+(0.267519458624)*x[2]
        arg[0,2,0,0]=(-0.171255468598)*x[0]+(-0.315808418391)*x[1]+(-1.31408126593)*x[2]
        arg[0,2,0,1]=(-0.0537048497397)*x[0]+(-1.05283989597)*x[1]+(-0.230505226375)*x[2]
        arg[0,2,1,0]=(0.175422426556)*x[0]+(-1.37203245283)*x[1]+(-1.31579047196)*x[2]
        arg[0,2,1,1]=(-0.678719594441)*x[0]+(0.325375648943)*x[1]+(-1.3715383088)*x[2]
        arg[0,2,2,0]=(1.34295310191)*x[0]+(-1.29856762722)*x[1]+(-0.604804069008)*x[2]
        arg[0,2,2,1]=(-0.38925176646)*x[0]+(-0.391078913511)*x[1]+(-0.147119806342)*x[2]
        arg[0,3,0,0]=(0.55897935716)*x[0]+(-0.938161813577)*x[1]+(-1.01170144437)*x[2]
        arg[0,3,0,1]=(-0.214720994691)*x[0]+(-0.936636326408)*x[1]+(1.23434495942)*x[2]
        arg[0,3,1,0]=(0.665136478224)*x[0]+(-0.223132283946)*x[1]+(0.513937660496)*x[2]
        arg[0,3,1,1]=(0.99663942166)*x[0]+(-0.870793744717)*x[1]+(-0.44642261239)*x[2]
        arg[0,3,2,0]=(0.426708530385)*x[0]+(-0.399913590631)*x[1]+(0.621569032554)*x[2]
        arg[0,3,2,1]=(-0.385564378699)*x[0]+(-1.17310910463)*x[1]+(-0.386725711607)*x[2]
        arg[0,4,0,0]=(-0.907605366918)*x[0]+(-1.15620807313)*x[1]+(0.690709991448)*x[2]
        arg[0,4,0,1]=(0.68925487184)*x[0]+(-0.731402302062)*x[1]+(-0.278506722879)*x[2]
        arg[0,4,1,0]=(0.607545316973)*x[0]+(-0.480514687124)*x[1]+(-0.261850296368)*x[2]
        arg[0,4,1,1]=(1.39016358303)*x[0]+(0.571194445557)*x[1]+(1.02359000173)*x[2]
        arg[0,4,2,0]=(1.19042892582)*x[0]+(-0.265599567429)*x[1]+(-0.412933583721)*x[2]
        arg[0,4,2,1]=(0.105768402681)*x[0]+(0.529442827604)*x[1]+(1.70332495695)*x[2]
        arg[1,0,0,0]=(-0.466512450689)*x[0]+(-0.821724116037)*x[1]+(0.315646947734)*x[2]
        arg[1,0,0,1]=(-0.343965024161)*x[0]+(-0.718276908655)*x[1]+(-0.89648206258)*x[2]
        arg[1,0,1,0]=(0.533001165605)*x[0]+(0.0431632707839)*x[1]+(-0.34028160642)*x[2]
        arg[1,0,1,1]=(-0.0905898060243)*x[0]+(-0.634642149101)*x[1]+(-1.13501642684)*x[2]
        arg[1,0,2,0]=(0.660943497868)*x[0]+(-0.354792953445)*x[1]+(1.28780893672)*x[2]
        arg[1,0,2,1]=(-1.2338460337)*x[0]+(-1.07489268654)*x[1]+(-1.32063885434)*x[2]
        arg[1,1,0,0]=(-0.564276365398)*x[0]+(1.03120151704)*x[1]+(-1.15799041025)*x[2]
        arg[1,1,0,1]=(0.715308242213)*x[0]+(1.38372056508)*x[1]+(0.143855804663)*x[2]
        arg[1,1,1,0]=(-0.359352464902)*x[0]+(1.12698364061)*x[1]+(0.592356171876)*x[2]
        arg[1,1,1,1]=(-0.450554686066)*x[0]+(-0.546866217679)*x[1]+(1.0842968414)*x[2]
        arg[1,1,2,0]=(-0.477267172046)*x[0]+(-0.863491255396)*x[1]+(-0.991333852913)*x[2]
        arg[1,1,2,1]=(0.402599196931)*x[0]+(-0.635287203482)*x[1]+(-0.277210261227)*x[2]
        arg[1,2,0,0]=(-1.56932441015)*x[0]+(-1.56442238007)*x[1]+(-0.375354109955)*x[2]
        arg[1,2,0,1]=(0.193969262086)*x[0]+(-0.0913153952113)*x[1]+(-0.764434182025)*x[2]
        arg[1,2,1,0]=(-0.836765580561)*x[0]+(0.84683301026)*x[1]+(-1.05033967172)*x[2]
        arg[1,2,1,1]=(1.68348957076)*x[0]+(-1.14474223991)*x[1]+(-0.0028142026612)*x[2]
        arg[1,2,2,0]=(-0.0921618478149)*x[0]+(-0.555660652821)*x[1]+(-0.398214997034)*x[2]
        arg[1,2,2,1]=(1.03261791276)*x[0]+(-0.650225649301)*x[1]+(1.63375423051)*x[2]
        arg[1,3,0,0]=(-0.348599695393)*x[0]+(-1.20215183267)*x[1]+(-0.1129459677)*x[2]
        arg[1,3,0,1]=(0.877317784105)*x[0]+(-0.587978566735)*x[1]+(1.10042804546)*x[2]
        arg[1,3,1,0]=(-0.547033849227)*x[0]+(-1.28198893788)*x[1]+(-0.424854587074)*x[2]
        arg[1,3,1,1]=(0.739360669292)*x[0]+(-0.66359460621)*x[1]+(1.08588749669)*x[2]
        arg[1,3,2,0]=(0.137130432185)*x[0]+(-0.123436803848)*x[1]+(-1.31161936702)*x[2]
        arg[1,3,2,1]=(-0.0171220220237)*x[0]+(1.44556512776)*x[1]+(0.665008556384)*x[2]
        arg[1,4,0,0]=(-0.304568160155)*x[0]+(-0.453489411096)*x[1]+(0.769736915169)*x[2]
        arg[1,4,0,1]=(-0.389567079496)*x[0]+(1.01465816808)*x[1]+(1.05389792181)*x[2]
        arg[1,4,1,0]=(-0.293019902659)*x[0]+(0.935020375302)*x[1]+(0.465003429909)*x[2]
        arg[1,4,1,1]=(1.26426164192)*x[0]+(0.53660188438)*x[1]+(-1.26997852414)*x[2]
        arg[1,4,2,0]=(0.447013260347)*x[0]+(-0.292646095073)*x[1]+(-0.141236248636)*x[2]
        arg[1,4,2,1]=(0.627246227193)*x[0]+(-1.07310030602)*x[1]+(0.414244174014)*x[2]
        arg[2,0,0,0]=(0.0450404092088)*x[0]+(-0.092484507099)*x[1]+(0.416562031216)*x[2]
        arg[2,0,0,1]=(0.437290559939)*x[0]+(-1.00144259207)*x[1]+(1.26666155425)*x[2]
        arg[2,0,1,0]=(0.0919297766737)*x[0]+(0.925519273623)*x[1]+(-0.582669368425)*x[2]
        arg[2,0,1,1]=(1.40247347323)*x[0]+(-0.896360612277)*x[1]+(-0.235401929947)*x[2]
        arg[2,0,2,0]=(-0.0760543780641)*x[0]+(-1.12543060216)*x[1]+(-1.73005861985)*x[2]
        arg[2,0,2,1]=(-0.0540914682488)*x[0]+(0.686173107995)*x[1]+(-0.499264219762)*x[2]
        arg[2,1,0,0]=(1.29412006754)*x[0]+(1.17564161143)*x[1]+(0.30140132879)*x[2]
        arg[2,1,0,1]=(-0.987572082341)*x[0]+(-1.73401714184)*x[1]+(0.542753625023)*x[2]
        arg[2,1,1,0]=(-0.162426291727)*x[0]+(0.508287986142)*x[1]+(0.42345261935)*x[2]
        arg[2,1,1,1]=(-0.436007145923)*x[0]+(0.344485478615)*x[1]+(0.867081685885)*x[2]
        arg[2,1,2,0]=(1.23340638096)*x[0]+(1.16627983483)*x[1]+(0.291792397233)*x[2]
        arg[2,1,2,1]=(1.81405521045)*x[0]+(-0.358711641776)*x[1]+(1.67495221975)*x[2]
        arg[2,2,0,0]=(-0.998776647505)*x[0]+(1.38487801142)*x[1]+(-0.199886556937)*x[2]
        arg[2,2,0,1]=(-0.151564577745)*x[0]+(0.100245326484)*x[1]+(-0.98457000197)*x[2]
        arg[2,2,1,0]=(-0.767959771993)*x[0]+(-0.641538616024)*x[1]+(0.0206020120898)*x[2]
        arg[2,2,1,1]=(-0.116988718822)*x[0]+(-1.15499563681)*x[1]+(-0.187589599016)*x[2]
        arg[2,2,2,0]=(0.682354528276)*x[0]+(0.606005834005)*x[1]+(0.11468472132)*x[2]
        arg[2,2,2,1]=(-0.547781355521)*x[0]+(1.23047427005)*x[1]+(0.854018236174)*x[2]
        arg[2,3,0,0]=(0.368349078554)*x[0]+(-1.12153473446)*x[1]+(0.114116055257)*x[2]
        arg[2,3,0,1]=(0.838089123891)*x[0]+(0.283469213399)*x[1]+(0.249874271688)*x[2]
        arg[2,3,1,0]=(-1.03882746612)*x[0]+(0.159711106612)*x[1]+(-1.57272656988)*x[2]
        arg[2,3,1,1]=(-1.25439210606)*x[0]+(-0.801588333704)*x[1]+(-0.166928557775)*x[2]
        arg[2,3,2,0]=(0.0650011438717)*x[0]+(0.879974749381)*x[1]+(-0.276534309239)*x[2]
        arg[2,3,2,1]=(-0.774913691581)*x[0]+(-0.360358063631)*x[1]+(-0.368288145309)*x[2]
        arg[2,4,0,0]=(-0.855093207242)*x[0]+(0.82049336972)*x[1]+(-0.496007273696)*x[2]
        arg[2,4,0,1]=(-0.113408635338)*x[0]+(0.116297312211)*x[1]+(-1.05279504872)*x[2]
        arg[2,4,1,0]=(1.59034968962)*x[0]+(-0.993664270946)*x[1]+(-1.44097244774)*x[2]
        arg[2,4,1,1]=(-0.554191144546)*x[0]+(-1.30747375036)*x[1]+(-1.17946836248)*x[2]
        arg[2,4,2,0]=(0.626012137217)*x[0]+(0.518243725142)*x[1]+(0.0933303150622)*x[2]
        arg[2,4,2,1]=(-0.10169564723)*x[0]+(1.45820149276)*x[1]+(-0.0393269028179)*x[2]
        arg[3,0,0,0]=(-0.0404687905735)*x[0]+(0.0715700915639)*x[1]+(1.45121483418)*x[2]
        arg[3,0,0,1]=(1.25470808156)*x[0]+(-0.119317783496)*x[1]+(0.28341020961)*x[2]
        arg[3,0,1,0]=(0.352807293701)*x[0]+(0.230591898707)*x[1]+(-0.580416381826)*x[2]
        arg[3,0,1,1]=(1.6169875055)*x[0]+(0.515147269207)*x[1]+(-0.461088044559)*x[2]
        arg[3,0,2,0]=(0.345374169211)*x[0]+(0.165179946106)*x[1]+(0.0761591935161)*x[2]
        arg[3,0,2,1]=(-0.367306414632)*x[0]+(-0.236156646602)*x[1]+(-0.388955870267)*x[2]
        arg[3,1,0,0]=(-0.27984693051)*x[0]+(-0.0431489812153)*x[1]+(-0.315624571395)*x[2]
        arg[3,1,0,1]=(-0.923109329983)*x[0]+(0.71133451696)*x[1]+(-0.793551470792)*x[2]
        arg[3,1,1,0]=(-0.434858211337)*x[0]+(0.943115368224)*x[1]+(0.195339633021)*x[2]
        arg[3,1,1,1]=(0.517533657447)*x[0]+(-0.975487574798)*x[1]+(0.00389777788827)*x[2]
        arg[3,1,2,0]=(-1.29728995298)*x[0]+(-0.519355341481)*x[1]+(-0.986985327874)*x[2]
        arg[3,1,2,1]=(-0.104824066468)*x[0]+(0.155059832097)*x[1]+(-0.513194354212)*x[2]
        arg[3,2,0,0]=(-0.053676798897)*x[0]+(-0.957854779002)*x[1]+(0.575673251241)*x[2]
        arg[3,2,0,1]=(0.828930053602)*x[0]+(0.122518313839)*x[1]+(-0.711848816688)*x[2]
        arg[3,2,1,0]=(0.468063581449)*x[0]+(0.233244036405)*x[1]+(-1.06333921952)*x[2]
        arg[3,2,1,1]=(-0.178080946586)*x[0]+(0.481804828166)*x[1]+(-0.25332042181)*x[2]
        arg[3,2,2,0]=(-0.2520881536)*x[0]+(-1.03459378989)*x[1]+(-0.284550808049)*x[2]
        arg[3,2,2,1]=(-0.778922518429)*x[0]+(1.0037625091)*x[1]+(-0.252088333444)*x[2]
        arg[3,3,0,0]=(0.467759844055)*x[0]+(0.598630111441)*x[1]+(-0.180631591223)*x[2]
        arg[3,3,0,1]=(-0.704465112342)*x[0]+(0.577992884135)*x[1]+(0.614639798328)*x[2]
        arg[3,3,1,0]=(0.58772281958)*x[0]+(0.267751378071)*x[1]+(0.11546949909)*x[2]
        arg[3,3,1,1]=(-0.220490484007)*x[0]+(-0.346075755455)*x[1]+(0.709249188793)*x[2]
        arg[3,3,2,0]=(-0.03823885045)*x[0]+(1.0070390689)*x[1]+(0.786191739951)*x[2]
        arg[3,3,2,1]=(0.468772686804)*x[0]+(-0.416774817651)*x[1]+(1.65342478719)*x[2]
        arg[3,4,0,0]=(0.69436779527)*x[0]+(-0.644342658009)*x[1]+(-0.6765845785)*x[2]
        arg[3,4,0,1]=(0.210253965392)*x[0]+(0.208735045316)*x[1]+(-0.553442080096)*x[2]
        arg[3,4,1,0]=(-0.0680478107141)*x[0]+(0.770215954791)*x[1]+(-0.876270039559)*x[2]
        arg[3,4,1,1]=(-0.384812708168)*x[0]+(-0.482707860969)*x[1]+(1.00274437781)*x[2]
        arg[3,4,2,0]=(-0.144567417774)*x[0]+(0.277773632892)*x[1]+(1.3123295509)*x[2]
        arg[3,4,2,1]=(0.00870646304914)*x[0]+(-0.254229329332)*x[1]+(-1.66538680273)*x[2]
        ref[0,0,0,0]=(0.841760617241)*(1+2.*(dim-1.)/(o+1.))+(-1.18631140441)*dim
        ref[0,0,0,1]=(-0.610007337864)*(1+2.*(dim-1.)/(o+1.))+(-1.71451486621)*dim
        ref[0,0,1,0]=(-0.765605897612)*(1+2.*(dim-1.)/(o+1.))+(0.83571459148)*dim
        ref[0,0,1,1]=(-0.128014940891)*(1+2.*(dim-1.)/(o+1.))+(-1.14397157584)*dim
        ref[0,0,2,0]=(-0.969876086955)*(1+2.*(dim-1.)/(o+1.))+(-0.238541556867)*dim
        ref[0,0,2,1]=(-0.434074835341)*(1+2.*(dim-1.)/(o+1.))+(-1.65441946755)*dim
        ref[0,1,0,0]=(0.346878720962)*(1+2.*(dim-1.)/(o+1.))+(-0.722019888602)*dim
        ref[0,1,0,1]=(-0.163332213104)*(1+2.*(dim-1.)/(o+1.))+(-0.135481298717)*dim
        ref[0,1,1,0]=(0.64836517897)*(1+2.*(dim-1.)/(o+1.))+(-0.720330834499)*dim
        ref[0,1,1,1]=(1.50578185724)*(1+2.*(dim-1.)/(o+1.))+(2.13408486278)*dim
        ref[0,1,2,0]=(-1.24719412419)*(1+2.*(dim-1.)/(o+1.))+(0.0927684729819)*dim
        ref[0,1,2,1]=(0.798388744364)*(1+2.*(dim-1.)/(o+1.))+(-1.93130500501)*dim
        ref[0,2,0,0]=(-1.9926882083)*(1+2.*(dim-1.)/(o+1.))+(0.191543055387)*dim
        ref[0,2,0,1]=(-2.16203894433)*(1+2.*(dim-1.)/(o+1.))+(0.824988972246)*dim
        ref[0,2,1,0]=(-2.24934415172)*(1+2.*(dim-1.)/(o+1.))+(-0.263056346511)*dim
        ref[0,2,1,1]=(-0.719152132879)*(1+2.*(dim-1.)/(o+1.))+(-1.00573012142)*dim
        ref[0,2,2,0]=(-0.83751635258)*(1+2.*(dim-1.)/(o+1.))+(0.277097758257)*dim
        ref[0,2,2,1]=(-0.979077481565)*(1+2.*(dim-1.)/(o+1.))+(0.0516269952525)*dim
        ref[0,3,0,0]=(-0.623873484234)*(1+2.*(dim-1.)/(o+1.))+(-0.767010416553)*dim
        ref[0,3,0,1]=(-1.0425822452)*(1+2.*(dim-1.)/(o+1.))+(1.12556988353)*dim
        ref[0,3,1,0]=(0.548600253744)*(1+2.*(dim-1.)/(o+1.))+(0.407341601031)*dim
        ref[0,3,1,1]=(-0.531963828356)*(1+2.*(dim-1.)/(o+1.))+(0.211386892908)*dim
        ref[0,3,2,0]=(1.33006849004)*(1+2.*(dim-1.)/(o+1.))+(-0.681704517735)*dim
        ref[0,3,2,1]=(-1.98393315604)*(1+2.*(dim-1.)/(o+1.))+(0.0385339611081)*dim
        ref[0,4,0,0]=(-0.381511396261)*(1+2.*(dim-1.)/(o+1.))+(-0.991592052336)*dim
        ref[0,4,0,1]=(1.00472425157)*(1+2.*(dim-1.)/(o+1.))+(-1.32537840467)*dim
        ref[0,4,1,0]=(1.04630307783)*(1+2.*(dim-1.)/(o+1.))+(-1.18112274434)*dim
        ref[0,4,1,1]=(1.61731240292)*(1+2.*(dim-1.)/(o+1.))+(1.3676356274)*dim
        ref[0,4,2,0]=(1.41653781097)*(1+2.*(dim-1.)/(o+1.))+(-0.904642036296)*dim
        ref[0,4,2,1]=(1.61515833451)*(1+2.*(dim-1.)/(o+1.))+(0.723377852725)*dim
        ref[1,0,0,0]=(-0.879218635)*(1+2.*(dim-1.)/(o+1.))+(-0.0933709839921)*dim
        ref[1,0,0,1]=(-1.26687353853)*(1+2.*(dim-1.)/(o+1.))+(-0.691850456865)*dim
        ref[1,0,1,0]=(1.79928802103)*(1+2.*(dim-1.)/(o+1.))+(-1.56340519106)*dim
        ref[1,0,1,1]=(-1.05495675007)*(1+2.*(dim-1.)/(o+1.))+(-0.805291631901)*dim
        ref[1,0,2,0]=(-0.339423840638)*(1+2.*(dim-1.)/(o+1.))+(1.93338332178)*dim
        ref[1,0,2,1]=(-1.15347644771)*(1+2.*(dim-1.)/(o+1.))+(-2.47590112686)*dim
        ref[1,1,0,0]=(-1.22232845026)*(1+2.*(dim-1.)/(o+1.))+(0.531263191656)*dim
        ref[1,1,0,1]=(0.318005373727)*(1+2.*(dim-1.)/(o+1.))+(1.92487923823)*dim
        ref[1,1,1,0]=(0.157193055461)*(1+2.*(dim-1.)/(o+1.))+(1.20279429213)*dim
        ref[1,1,1,1]=(0.0525914033484)*(1+2.*(dim-1.)/(o+1.))+(0.0342845343071)*dim
        ref[1,1,2,0]=(-0.319573431275)*(1+2.*(dim-1.)/(o+1.))+(-2.01251884908)*dim
        ref[1,1,2,1]=(-1.67524932091)*(1+2.*(dim-1.)/(o+1.))+(1.16535105313)*dim
        ref[1,2,0,0]=(-2.25088459423)*(1+2.*(dim-1.)/(o+1.))+(-1.25821630594)*dim
        ref[1,2,0,1]=(-0.447759463011)*(1+2.*(dim-1.)/(o+1.))+(-0.214020852139)*dim
        ref[1,2,1,0]=(-1.26590071291)*(1+2.*(dim-1.)/(o+1.))+(0.225628470886)*dim
        ref[1,2,1,1]=(-0.372878924951)*(1+2.*(dim-1.)/(o+1.))+(0.90881205314)*dim
        ref[1,2,2,0]=(1.36047043606)*(1+2.*(dim-1.)/(o+1.))+(-2.40650793373)*dim
        ref[1,2,2,1]=(1.13871462186)*(1+2.*(dim-1.)/(o+1.))+(0.877431872103)*dim
        ref[1,3,0,0]=(-1.17967356605)*(1+2.*(dim-1.)/(o+1.))+(-0.48402392971)*dim
        ref[1,3,0,1]=(-0.177974933247)*(1+2.*(dim-1.)/(o+1.))+(1.56774219607)*dim
        ref[1,3,1,0]=(-1.40936193115)*(1+2.*(dim-1.)/(o+1.))+(-0.844515443034)*dim
        ref[1,3,1,1]=(0.790537554795)*(1+2.*(dim-1.)/(o+1.))+(0.371116004973)*dim
        ref[1,3,2,0]=(0.242057644413)*(1+2.*(dim-1.)/(o+1.))+(-1.5399833831)*dim
        ref[1,3,2,1]=(1.70371239948)*(1+2.*(dim-1.)/(o+1.))+(0.389739262639)*dim
        ref[1,4,0,0]=(-0.375645743677)*(1+2.*(dim-1.)/(o+1.))+(0.387325087595)*dim
        ref[1,4,0,1]=(0.0489286471987)*(1+2.*(dim-1.)/(o+1.))+(1.63006036319)*dim
        ref[1,4,1,0]=(1.33203691786)*(1+2.*(dim-1.)/(o+1.))+(-0.225033015313)*dim
        ref[1,4,1,1]=(0.332297845981)*(1+2.*(dim-1.)/(o+1.))+(0.198587156175)*dim
        ref[1,4,2,0]=(-1.22288078692)*(1+2.*(dim-1.)/(o+1.))+(1.23601170355)*dim
        ref[1,4,2,1]=(0.0299493142028)*(1+2.*(dim-1.)/(o+1.))+(-0.0615592190171)*dim
        ref[2,0,0,0]=(0.0963155205346)*(1+2.*(dim-1.)/(o+1.))+(0.272802412791)*dim
        ref[2,0,0,1]=(1.12280541023)*(1+2.*(dim-1.)/(o+1.))+(-0.420295888105)*dim
        ref[2,0,1,0]=(-0.296878488858)*(1+2.*(dim-1.)/(o+1.))+(0.73165817073)*dim
        ref[2,0,1,1]=(-0.407704337172)*(1+2.*(dim-1.)/(o+1.))+(0.678415268175)*dim
        ref[2,0,2,0]=(-2.37402978977)*(1+2.*(dim-1.)/(o+1.))+(-0.557513810308)*dim
        ref[2,0,2,1]=(1.2564150767)*(1+2.*(dim-1.)/(o+1.))+(-1.12359765672)*dim
        ref[2,1,0,0]=(0.824412811392)*(1+2.*(dim-1.)/(o+1.))+(1.94675019637)*dim
        ref[2,1,0,1]=(-1.06505178789)*(1+2.*(dim-1.)/(o+1.))+(-1.11378381126)*dim
        ref[2,1,1,0]=(0.200096305752)*(1+2.*(dim-1.)/(o+1.))+(0.569218008012)*dim
        ref[2,1,1,1]=(1.08680720067)*(1+2.*(dim-1.)/(o+1.))+(-0.311247182096)*dim
        ref[2,1,2,0]=(1.3764914842)*(1+2.*(dim-1.)/(o+1.))+(1.31498712883)*dim
        ref[2,1,2,1]=(2.26537199157)*(1+2.*(dim-1.)/(o+1.))+(0.864923796858)*dim
        ref[2,2,0,0]=(0.750025182887)*(1+2.*(dim-1.)/(o+1.))+(-0.563810375913)*dim
        ref[2,2,0,1]=(-0.806933575047)*(1+2.*(dim-1.)/(o+1.))+(-0.228955678184)*dim
        ref[2,2,1,0]=(-1.53132620021)*(1+2.*(dim-1.)/(o+1.))+(0.14242982428)*dim
        ref[2,2,1,1]=(-0.889411443106)*(1+2.*(dim-1.)/(o+1.))+(-0.570162511538)*dim
        ref[2,2,2,0]=(-0.0309717316948)*(1+2.*(dim-1.)/(o+1.))+(1.4340168153)*dim
        ref[2,2,2,1]=(1.48963237375)*(1+2.*(dim-1.)/(o+1.))+(0.0470787769518)*dim
        ref[2,3,0,0]=(-0.0184174113004)*(1+2.*(dim-1.)/(o+1.))+(-0.620652189349)*dim
        ref[2,3,0,1]=(0.597433991421)*(1+2.*(dim-1.)/(o+1.))+(0.773998617557)*dim
        ref[2,3,1,0]=(-1.10704622688)*(1+2.*(dim-1.)/(o+1.))+(-1.34479670251)*dim
        ref[2,3,1,1]=(-0.369209798891)*(1+2.*(dim-1.)/(o+1.))+(-1.85369919864)*dim
        ref[2,3,2,0]=(-0.286165609161)*(1+2.*(dim-1.)/(o+1.))+(0.954607193175)*dim
        ref[2,3,2,1]=(-0.0948450773617)*(1+2.*(dim-1.)/(o+1.))+(-1.40871482316)*dim
        ref[2,4,0,0]=(-0.350521344827)*(1+2.*(dim-1.)/(o+1.))+(-0.180085766391)*dim
        ref[2,4,0,1]=(-1.10948105015)*(1+2.*(dim-1.)/(o+1.))+(0.0595746783013)*dim
        ref[2,4,1,0]=(-0.251521967553)*(1+2.*(dim-1.)/(o+1.))+(-0.592765061514)*dim
        ref[2,4,1,1]=(-2.22732408506)*(1+2.*(dim-1.)/(o+1.))+(-0.813809172329)*dim
        ref[2,4,2,0]=(-1.4091057051)*(1+2.*(dim-1.)/(o+1.))+(2.64669188252)*dim
        ref[2,4,2,1]=(-0.0262518153279)*(1+2.*(dim-1.)/(o+1.))+(1.34343075804)*dim
        ref[3,0,0,0]=(2.08524442815)*(1+2.*(dim-1.)/(o+1.))+(-0.602928292982)*dim
        ref[3,0,0,1]=(0.227442743255)*(1+2.*(dim-1.)/(o+1.))+(1.19135776442)*dim
        ref[3,0,1,0]=(1.03371796145)*(1+2.*(dim-1.)/(o+1.))+(-1.03073515087)*dim
        ref[3,0,1,1]=(0.205903763449)*(1+2.*(dim-1.)/(o+1.))+(1.4651429667)*dim
        ref[3,0,2,0]=(1.54387435524)*(1+2.*(dim-1.)/(o+1.))+(-0.957161046404)*dim
        ref[3,0,2,1]=(-0.487693556268)*(1+2.*(dim-1.)/(o+1.))+(-0.504725375232)*dim
        ref[3,1,0,0]=(0.319450917291)*(1+2.*(dim-1.)/(o+1.))+(-0.958071400411)*dim
        ref[3,1,0,1]=(-0.0756676477654)*(1+2.*(dim-1.)/(o+1.))+(-0.929658636049)*dim
        ref[3,1,1,0]=(0.0109030665653)*(1+2.*(dim-1.)/(o+1.))+(0.692693723343)*dim
        ref[3,1,1,1]=(0.540904091106)*(1+2.*(dim-1.)/(o+1.))+(-0.994960230569)*dim
        ref[3,1,2,0]=(-1.39229945953)*(1+2.*(dim-1.)/(o+1.))+(-1.4113311628)*dim
        ref[3,1,2,1]=(-0.120906779216)*(1+2.*(dim-1.)/(o+1.))+(-0.342051809366)*dim
        ref[3,2,0,0]=(-0.61477618653)*(1+2.*(dim-1.)/(o+1.))+(0.178917859873)*dim
        ref[3,2,0,1]=(0.286238606318)*(1+2.*(dim-1.)/(o+1.))+(-0.0466390555654)*dim
        ref[3,2,1,0]=(-0.0560560726323)*(1+2.*(dim-1.)/(o+1.))+(-0.305975529033)*dim
        ref[3,2,1,1]=(0.935166077059)*(1+2.*(dim-1.)/(o+1.))+(-0.884762617288)*dim
        ref[3,2,2,0]=(0.57015079501)*(1+2.*(dim-1.)/(o+1.))+(-2.14138354655)*dim
        ref[3,2,2,1]=(-0.699470490706)*(1+2.*(dim-1.)/(o+1.))+(0.672222147932)*dim
        ref[3,3,0,0]=(1.33431543118)*(1+2.*(dim-1.)/(o+1.))+(-0.448557066912)*dim
        ref[3,3,0,1]=(-1.03724498716)*(1+2.*(dim-1.)/(o+1.))+(1.52541255729)*dim
        ref[3,3,1,0]=(-0.00371131395733)*(1+2.*(dim-1.)/(o+1.))+(0.974655010698)*dim
        ref[3,3,1,1]=(0.0918410742793)*(1+2.*(dim-1.)/(o+1.))+(0.0508418750519)*dim
        ref[3,3,2,0]=(1.63372857769)*(1+2.*(dim-1.)/(o+1.))+(0.121263380706)*dim
        ref[3,3,2,1]=(0.761569446324)*(1+2.*(dim-1.)/(o+1.))+(0.943853210019)*dim
        ref[3,4,0,0]=(-0.625751279162)*(1+2.*(dim-1.)/(o+1.))+(-0.000808162076656)*dim
        ref[3,4,0,1]=(-0.113324890356)*(1+2.*(dim-1.)/(o+1.))+(-0.0211281790319)*dim
        ref[3,4,1,0]=(0.296144237506)*(1+2.*(dim-1.)/(o+1.))+(-0.470246132989)*dim
        ref[3,4,1,1]=(0.194264324372)*(1+2.*(dim-1.)/(o+1.))+(-0.0590405157015)*dim
        ref[3,4,2,0]=(1.63851379122)*(1+2.*(dim-1.)/(o+1.))+(-0.192978025203)*dim
        ref[3,4,2,1]=(-0.757374882295)*(1+2.*(dim-1.)/(o+1.))+(-1.15353478672)*dim
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5, 3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnBoundary_fromData_ReducedContinuousFunction_rank0(self):
      """
      tests integral of rank 0 Data on the FunctionOnBoundary

      assumptions: ReducedContinuousFunction(self.domain) exists
                   self.domain supports integral on FunctionOnBoundary
      """
      o=1
      dim=self.domain.getDim()
      w_ref=FunctionOnBoundary(self.domain)
      w=ReducedContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(),w)
      if dim==2:
        arg=(-0.0177156089276)*x[0]+(-1.07750293477)*x[1]
        ref=(0.674554765151)*(1+2.*(dim-1.)/(o+1.))+(-1.76977330884)*dim
      else:
        arg=(0.304688056778)*x[0]+(0.548485298428)*x[1]+(0.672370309114)*x[2]
        ref=(0.0121419382123)*(1+2.*(dim-1.)/(o+1.))+(1.51340172611)*dim
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnBoundary_fromData_ReducedContinuousFunction_rank1(self):
      """
      tests integral of rank 1 Data on the FunctionOnBoundary

      assumptions: ReducedContinuousFunction(self.domain) exists
                   self.domain supports integral on FunctionOnBoundary
      """
      o=1
      dim=self.domain.getDim()
      w_ref=FunctionOnBoundary(self.domain)
      w=ReducedContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(2,),w)
      ref=numpy.zeros((2,),numpy.float_)
      if dim==2:
        arg[0]=(-1.81332918209)*x[0]+(0.825631690982)*x[1]
        arg[1]=(0.568335098118)*x[0]+(-0.504485431124)*x[1]
        ref[0]=(0.0338855214148)*(1+2.*(dim-1.)/(o+1.))+(-1.02158301252)*dim
        ref[1]=(0.348045520014)*(1+2.*(dim-1.)/(o+1.))+(-0.284195853019)*dim
      else:
        arg[0]=(-0.34247651789)*x[0]+(0.952434108232)*x[1]+(0.956872478767)*x[2]
        arg[1]=(-0.73212870207)*x[0]+(1.25659546958)*x[1]+(-0.690842998605)*x[2]
        ref[0]=(1.22184577749)*(1+2.*(dim-1.)/(o+1.))+(0.344984291617)*dim
        ref[1]=(-0.600249739257)*(1+2.*(dim-1.)/(o+1.))+(0.433873508162)*dim
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnBoundary_fromData_ReducedContinuousFunction_rank2(self):
      """
      tests integral of rank 2 Data on the FunctionOnBoundary

      assumptions: ReducedContinuousFunction(self.domain) exists
                   self.domain supports integral on FunctionOnBoundary
      """
      o=1
      dim=self.domain.getDim()
      w_ref=FunctionOnBoundary(self.domain)
      w=ReducedContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5),w)
      ref=numpy.zeros((4, 5),numpy.float_)
      if dim==2:
        arg[0,0]=(-1.78262197231)*x[0]+(1.45174573417)*x[1]
        arg[0,1]=(0.213586223199)*x[0]+(-0.917876059198)*x[1]
        arg[0,2]=(-1.12034819339)*x[0]+(0.59993130233)*x[1]
        arg[0,3]=(-0.156732957083)*x[0]+(0.744660873695)*x[1]
        arg[0,4]=(0.217327014577)*x[0]+(1.40157420042)*x[1]
        arg[1,0]=(0.0588042984116)*x[0]+(1.40376148561)*x[1]
        arg[1,1]=(-0.264973303233)*x[0]+(1.77692483003)*x[1]
        arg[1,2]=(-1.17910097102)*x[0]+(-1.20301960774)*x[1]
        arg[1,3]=(-0.511409983468)*x[0]+(-1.02065230228)*x[1]
        arg[1,4]=(1.1254497221)*x[0]+(-1.23163629376)*x[1]
        arg[2,0]=(-0.780207200735)*x[0]+(-1.49983791458)*x[1]
        arg[2,1]=(0.20679132359)*x[0]+(0.640203014983)*x[1]
        arg[2,2]=(0.198135205267)*x[0]+(0.30781411018)*x[1]
        arg[2,3]=(0.465906968942)*x[0]+(-0.812198212)*x[1]
        arg[2,4]=(0.190357350559)*x[0]+(-1.5894498455)*x[1]
        arg[3,0]=(-1.1362863086)*x[0]+(-0.0654142353086)*x[1]
        arg[3,1]=(0.0545924852084)*x[0]+(0.00262152155818)*x[1]
        arg[3,2]=(-0.0497600832907)*x[0]+(-0.609396009303)*x[1]
        arg[3,3]=(0.347834191962)*x[0]+(-0.768159599963)*x[1]
        arg[3,4]=(-0.178383166761)*x[0]+(-0.0690942212822)*x[1]
        ref[0,0]=(0.0436748214249)*(1+2.*(dim-1.)/(o+1.))+(-0.374551059559)*dim
        ref[0,1]=(-1.05854206043)*(1+2.*(dim-1.)/(o+1.))+(0.354252224435)*dim
        ref[0,2]=(-0.71782952995)*(1+2.*(dim-1.)/(o+1.))+(0.197412638885)*dim
        ref[0,3]=(-0.255252590976)*(1+2.*(dim-1.)/(o+1.))+(0.843180507587)*dim
        ref[0,4]=(0.177908820578)*(1+2.*(dim-1.)/(o+1.))+(1.44099239442)*dim
        ref[1,0]=(1.56040501086)*(1+2.*(dim-1.)/(o+1.))+(-0.0978392268304)*dim
        ref[1,1]=(1.33164571828)*(1+2.*(dim-1.)/(o+1.))+(0.180305808513)*dim
        ref[1,2]=(-0.885397553926)*(1+2.*(dim-1.)/(o+1.))+(-1.49672302484)*dim
        ref[1,3]=(0.105451964854)*(1+2.*(dim-1.)/(o+1.))+(-1.63751425061)*dim
        ref[1,4]=(-0.774237262923)*(1+2.*(dim-1.)/(o+1.))+(0.668050691269)*dim
        ref[2,0]=(-0.626493344552)*(1+2.*(dim-1.)/(o+1.))+(-1.65355177076)*dim
        ref[2,1]=(0.383826450148)*(1+2.*(dim-1.)/(o+1.))+(0.463167888425)*dim
        ref[2,2]=(-0.530241676713)*(1+2.*(dim-1.)/(o+1.))+(1.03619099216)*dim
        ref[2,3]=(-1.18245389658)*(1+2.*(dim-1.)/(o+1.))+(0.836162653525)*dim
        ref[2,4]=(-1.32396763993)*(1+2.*(dim-1.)/(o+1.))+(-0.0751248550082)*dim
        ref[3,0]=(0.15610314985)*(1+2.*(dim-1.)/(o+1.))+(-1.35780369376)*dim
        ref[3,1]=(-0.376931239611)*(1+2.*(dim-1.)/(o+1.))+(0.434145246378)*dim
        ref[3,2]=(0.0188741177717)*(1+2.*(dim-1.)/(o+1.))+(-0.678030210365)*dim
        ref[3,3]=(-0.264501990209)*(1+2.*(dim-1.)/(o+1.))+(-0.155823417792)*dim
        ref[3,4]=(-0.394609208758)*(1+2.*(dim-1.)/(o+1.))+(0.147131820715)*dim
      else:
        arg[0,0]=(-1.55600852596)*x[0]+(-0.624393737912)*x[1]+(-0.747032596279)*x[2]
        arg[0,1]=(0.158636782846)*x[0]+(-0.751760245066)*x[1]+(-1.62081735864)*x[2]
        arg[0,2]=(1.18083264152)*x[0]+(-1.16696814878)*x[1]+(-0.567354369023)*x[2]
        arg[0,3]=(0.176344525636)*x[0]+(-0.825319538089)*x[1]+(0.32760822446)*x[2]
        arg[0,4]=(0.570849136808)*x[0]+(-0.86277959325)*x[1]+(-0.253106033513)*x[2]
        arg[1,0]=(0.325475826081)*x[0]+(0.338988467638)*x[1]+(-0.596090654044)*x[2]
        arg[1,1]=(-1.29950586693)*x[0]+(-0.980591079821)*x[1]+(-0.7995507704)*x[2]
        arg[1,2]=(0.108948096835)*x[0]+(0.323001832704)*x[1]+(1.09678982473)*x[2]
        arg[1,3]=(-0.0307191696266)*x[0]+(0.134014984197)*x[1]+(-0.271363693134)*x[2]
        arg[1,4]=(0.0166805959434)*x[0]+(-0.323260440667)*x[1]+(-0.0168023639197)*x[2]
        arg[2,0]=(1.66243206086)*x[0]+(-0.478403313987)*x[1]+(1.45469230407)*x[2]
        arg[2,1]=(0.635927142861)*x[0]+(0.00628277747568)*x[1]+(-0.335502970321)*x[2]
        arg[2,2]=(-0.168440428449)*x[0]+(-0.302470381257)*x[1]+(1.26059253147)*x[2]
        arg[2,3]=(-0.0183286590129)*x[0]+(-0.0950506276918)*x[1]+(-0.721364110928)*x[2]
        arg[2,4]=(-1.04551179644)*x[0]+(0.20728433783)*x[1]+(-0.631521487025)*x[2]
        arg[3,0]=(1.0112442447)*x[0]+(0.0279472213885)*x[1]+(-1.24445628735)*x[2]
        arg[3,1]=(-1.36197290176)*x[0]+(0.145431495907)*x[1]+(0.518562214899)*x[2]
        arg[3,2]=(-0.249567074212)*x[0]+(-1.02991047199)*x[1]+(1.34627546588)*x[2]
        arg[3,3]=(1.72315708213)*x[0]+(0.00515532937983)*x[1]+(-0.479917704073)*x[2]
        arg[3,4]=(1.16297504595)*x[0]+(-0.176518851423)*x[1]+(0.184727222764)*x[2]
        ref[0,0]=(-2.09480681002)*(1+2.*(dim-1.)/(o+1.))+(-0.832628050126)*dim
        ref[0,1]=(-0.759565702286)*(1+2.*(dim-1.)/(o+1.))+(-1.45437511857)*dim
        ref[0,2]=(-1.08318812741)*(1+2.*(dim-1.)/(o+1.))+(0.529698251127)*dim
        ref[0,3]=(1.48988138495)*(1+2.*(dim-1.)/(o+1.))+(-1.81124817294)*dim
        ref[0,4]=(0.116706866039)*(1+2.*(dim-1.)/(o+1.))+(-0.661743355994)*dim
        ref[1,0]=(0.584176790043)*(1+2.*(dim-1.)/(o+1.))+(-0.515803150369)*dim
        ref[1,1]=(-1.25638916576)*(1+2.*(dim-1.)/(o+1.))+(-1.82325855139)*dim
        ref[1,2]=(-0.276318333951)*(1+2.*(dim-1.)/(o+1.))+(1.80505808822)*dim
        ref[1,3]=(-0.732277841606)*(1+2.*(dim-1.)/(o+1.))+(0.564209963043)*dim
        ref[1,4]=(-0.540285295646)*(1+2.*(dim-1.)/(o+1.))+(0.216903087004)*dim
        ref[2,0]=(2.23339929969)*(1+2.*(dim-1.)/(o+1.))+(0.405321751257)*dim
        ref[2,1]=(0.706137929075)*(1+2.*(dim-1.)/(o+1.))+(-0.39943097906)*dim
        ref[2,2]=(-0.474421381251)*(1+2.*(dim-1.)/(o+1.))+(1.26410310301)*dim
        ref[2,3]=(1.43870260444)*(1+2.*(dim-1.)/(o+1.))+(-2.27344600207)*dim
        ref[2,4]=(-1.00154864599)*(1+2.*(dim-1.)/(o+1.))+(-0.468200299646)*dim
        ref[3,0]=(-1.42181650401)*(1+2.*(dim-1.)/(o+1.))+(1.21655168275)*dim
        ref[3,1]=(-0.853191175508)*(1+2.*(dim-1.)/(o+1.))+(0.155211984554)*dim
        ref[3,2]=(-0.823496284217)*(1+2.*(dim-1.)/(o+1.))+(0.890294203901)*dim
        ref[3,3]=(1.54168027082)*(1+2.*(dim-1.)/(o+1.))+(-0.293285563378)*dim
        ref[3,4]=(1.66366656195)*(1+2.*(dim-1.)/(o+1.))+(-0.492483144667)*dim
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnBoundary_fromData_ReducedContinuousFunction_rank3(self):
      """
      tests integral of rank 3 Data on the FunctionOnBoundary

      assumptions: ReducedContinuousFunction(self.domain) exists
                   self.domain supports integral on FunctionOnBoundary
      """
      o=1
      dim=self.domain.getDim()
      w_ref=FunctionOnBoundary(self.domain)
      w=ReducedContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(6, 2, 2),w)
      ref=numpy.zeros((6, 2, 2),numpy.float_)
      if dim==2:
        arg[0,0,0]=(-0.775903850503)*x[0]+(0.20657985447)*x[1]
        arg[0,0,1]=(1.42793412139)*x[0]+(1.11703130088)*x[1]
        arg[0,1,0]=(1.89750028353)*x[0]+(1.77903425203)*x[1]
        arg[0,1,1]=(-0.67280619036)*x[0]+(-0.283600511942)*x[1]
        arg[1,0,0]=(0.12590667282)*x[0]+(-0.109778270548)*x[1]
        arg[1,0,1]=(-1.06454156291)*x[0]+(-0.952754071628)*x[1]
        arg[1,1,0]=(1.27939438243)*x[0]+(1.41358757535)*x[1]
        arg[1,1,1]=(-1.68847666929)*x[0]+(-0.13467497005)*x[1]
        arg[2,0,0]=(0.116194220064)*x[0]+(0.457728827844)*x[1]
        arg[2,0,1]=(0.205487821412)*x[0]+(-0.657189976052)*x[1]
        arg[2,1,0]=(0.276192450801)*x[0]+(0.998191496643)*x[1]
        arg[2,1,1]=(1.63929218914)*x[0]+(0.268648283241)*x[1]
        arg[3,0,0]=(-0.419114071163)*x[0]+(-0.365697048391)*x[1]
        arg[3,0,1]=(0.802747458947)*x[0]+(0.137023645119)*x[1]
        arg[3,1,0]=(-0.0712635889142)*x[0]+(0.495489688269)*x[1]
        arg[3,1,1]=(-0.559073395964)*x[0]+(0.248364992354)*x[1]
        arg[4,0,0]=(0.372130696656)*x[0]+(1.22892579337)*x[1]
        arg[4,0,1]=(-1.47944537272)*x[0]+(1.46750125425)*x[1]
        arg[4,1,0]=(-0.0683410762004)*x[0]+(0.0982446782745)*x[1]
        arg[4,1,1]=(0.0698227670903)*x[0]+(0.227875508249)*x[1]
        arg[5,0,0]=(0.237110604157)*x[0]+(0.814767844631)*x[1]
        arg[5,0,1]=(0.172109909542)*x[0]+(-1.56055537357)*x[1]
        arg[5,1,0]=(0.92885449265)*x[0]+(0.820977987322)*x[1]
        arg[5,1,1]=(-1.76483093472)*x[0]+(0.526316351009)*x[1]
        ref[0,0,0]=(0.574633189339)*(1+2.*(dim-1.)/(o+1.))+(-1.14395718537)*dim
        ref[0,0,1]=(1.25832290545)*(1+2.*(dim-1.)/(o+1.))+(1.28664251682)*dim
        ref[0,1,0]=(1.75085914739)*(1+2.*(dim-1.)/(o+1.))+(1.92567538817)*dim
        ref[0,1,1]=(-1.57747201281)*(1+2.*(dim-1.)/(o+1.))+(0.621065310511)*dim
        ref[1,0,0]=(0.710883587246)*(1+2.*(dim-1.)/(o+1.))+(-0.694755184974)*dim
        ref[1,0,1]=(-0.870861363869)*(1+2.*(dim-1.)/(o+1.))+(-1.14643427067)*dim
        ref[1,1,0]=(1.21302125269)*(1+2.*(dim-1.)/(o+1.))+(1.4799607051)*dim
        ref[1,1,1]=(-0.163344093083)*(1+2.*(dim-1.)/(o+1.))+(-1.65980754626)*dim
        ref[2,0,0]=(1.18070929814)*(1+2.*(dim-1.)/(o+1.))+(-0.606786250235)*dim
        ref[2,0,1]=(-1.26011538751)*(1+2.*(dim-1.)/(o+1.))+(0.808413232875)*dim
        ref[2,1,0]=(-0.128253427149)*(1+2.*(dim-1.)/(o+1.))+(1.40263737459)*dim
        ref[2,1,1]=(1.22388877116)*(1+2.*(dim-1.)/(o+1.))+(0.68405170122)*dim
        ref[3,0,0]=(-1.51984063257)*(1+2.*(dim-1.)/(o+1.))+(0.735029513018)*dim
        ref[3,0,1]=(1.63702893126)*(1+2.*(dim-1.)/(o+1.))+(-0.697257827196)*dim
        ref[3,1,0]=(-0.274208683077)*(1+2.*(dim-1.)/(o+1.))+(0.698434782432)*dim
        ref[3,1,1]=(0.649113059071)*(1+2.*(dim-1.)/(o+1.))+(-0.95982146268)*dim
        ref[4,0,0]=(1.11723294318)*(1+2.*(dim-1.)/(o+1.))+(0.483823546848)*dim
        ref[4,0,1]=(-0.0369311601619)*(1+2.*(dim-1.)/(o+1.))+(0.0249870416905)*dim
        ref[4,1,0]=(-0.0969818481854)*(1+2.*(dim-1.)/(o+1.))+(0.126885450259)*dim
        ref[4,1,1]=(0.607834304349)*(1+2.*(dim-1.)/(o+1.))+(-0.31013602901)*dim
        ref[5,0,0]=(0.612356854718)*(1+2.*(dim-1.)/(o+1.))+(0.43952159407)*dim
        ref[5,0,1]=(-0.245849747535)*(1+2.*(dim-1.)/(o+1.))+(-1.1425957165)*dim
        ref[5,1,0]=(0.218025498167)*(1+2.*(dim-1.)/(o+1.))+(1.5318069818)*dim
        ref[5,1,1]=(-0.39419356418)*(1+2.*(dim-1.)/(o+1.))+(-0.844321019532)*dim
      else:
        arg[0,0,0]=(1.52622777835)*x[0]+(-0.627217363559)*x[1]+(-0.221339024656)*x[2]
        arg[0,0,1]=(-0.0735955925339)*x[0]+(-1.44492195287)*x[1]+(-0.0704081180966)*x[2]
        arg[0,1,0]=(-0.330700790074)*x[0]+(-1.04769579979)*x[1]+(0.508799976666)*x[2]
        arg[0,1,1]=(0.349349373439)*x[0]+(0.174054703574)*x[1]+(-0.129296220879)*x[2]
        arg[1,0,0]=(0.0683757878129)*x[0]+(1.33394387082)*x[1]+(0.892302980732)*x[2]
        arg[1,0,1]=(0.61014567223)*x[0]+(1.23846801628)*x[1]+(-0.990674941877)*x[2]
        arg[1,1,0]=(0.964575308424)*x[0]+(0.354822939168)*x[1]+(-0.466180761866)*x[2]
        arg[1,1,1]=(-1.79700594499)*x[0]+(-1.41821199521)*x[1]+(0.366764057361)*x[2]
        arg[2,0,0]=(0.670092254407)*x[0]+(-0.188523225253)*x[1]+(0.363046866484)*x[2]
        arg[2,0,1]=(0.420367065732)*x[0]+(0.798321474665)*x[1]+(-1.11347160928)*x[2]
        arg[2,1,0]=(0.54732858526)*x[0]+(0.249248458735)*x[1]+(1.33905485076)*x[2]
        arg[2,1,1]=(-1.45292600653)*x[0]+(-0.0838212918177)*x[1]+(-0.210961166185)*x[2]
        arg[3,0,0]=(-0.406792355772)*x[0]+(-0.230709788944)*x[1]+(1.12300383125)*x[2]
        arg[3,0,1]=(-1.02748413184)*x[0]+(-1.3840845484)*x[1]+(-0.181262479318)*x[2]
        arg[3,1,0]=(-1.39003872882)*x[0]+(-0.302577309791)*x[1]+(-0.627070288119)*x[2]
        arg[3,1,1]=(0.258496709277)*x[0]+(-1.32012543621)*x[1]+(0.921022534515)*x[2]
        arg[4,0,0]=(-0.639952053735)*x[0]+(0.172303718581)*x[1]+(-1.55894055416)*x[2]
        arg[4,0,1]=(-1.27660129414)*x[0]+(0.142177942574)*x[1]+(0.84156755899)*x[2]
        arg[4,1,0]=(0.49812465572)*x[0]+(1.53755810837)*x[1]+(0.10504146131)*x[2]
        arg[4,1,1]=(0.114534785643)*x[0]+(0.616103931999)*x[1]+(0.515764206037)*x[2]
        arg[5,0,0]=(0.22562941228)*x[0]+(0.70743104453)*x[1]+(0.882965136406)*x[2]
        arg[5,0,1]=(0.674003358504)*x[0]+(0.753008881236)*x[1]+(0.033083269963)*x[2]
        arg[5,1,0]=(-0.495809169314)*x[0]+(-1.09613938465)*x[1]+(-1.03297996938)*x[2]
        arg[5,1,1]=(-0.5222251005)*x[0]+(-0.56129023192)*x[1]+(0.529084741893)*x[2]
        ref[0,0,0]=(0.164333186185)*(1+2.*(dim-1.)/(o+1.))+(0.513338203948)*dim
        ref[0,0,1]=(-0.322072217228)*(1+2.*(dim-1.)/(o+1.))+(-1.26685344627)*dim
        ref[0,1,0]=(1.37061498302)*(1+2.*(dim-1.)/(o+1.))+(-2.24021159622)*dim
        ref[0,1,1]=(0.65996582358)*(1+2.*(dim-1.)/(o+1.))+(-0.265857967445)*dim
        ref[1,0,0]=(1.83616555678)*(1+2.*(dim-1.)/(o+1.))+(0.458457082585)*dim
        ref[1,0,1]=(1.01886896436)*(1+2.*(dim-1.)/(o+1.))+(-0.160930217723)*dim
        ref[1,1,0]=(0.95375222959)*(1+2.*(dim-1.)/(o+1.))+(-0.100534743865)*dim
        ref[1,1,1]=(-1.48059917075)*(1+2.*(dim-1.)/(o+1.))+(-1.3678547121)*dim
        ref[2,0,0]=(-0.0805152837328)*(1+2.*(dim-1.)/(o+1.))+(0.925131179371)*dim
        ref[2,0,1]=(0.155604473013)*(1+2.*(dim-1.)/(o+1.))+(-0.0503875419008)*dim
        ref[2,1,0]=(0.117237902139)*(1+2.*(dim-1.)/(o+1.))+(2.01839399262)*dim
        ref[2,1,1]=(0.196870433027)*(1+2.*(dim-1.)/(o+1.))+(-1.94457889755)*dim
        ref[3,0,0]=(-1.16272511547)*(1+2.*(dim-1.)/(o+1.))+(1.648226802)*dim
        ref[3,0,1]=(-1.54275766077)*(1+2.*(dim-1.)/(o+1.))+(-1.05007349878)*dim
        ref[3,1,0]=(-2.21939430114)*(1+2.*(dim-1.)/(o+1.))+(-0.100292025596)*dim
        ref[3,1,1]=(-0.974866109202)*(1+2.*(dim-1.)/(o+1.))+(0.834259916779)*dim
        ref[4,0,0]=(-1.81015652027)*(1+2.*(dim-1.)/(o+1.))+(-0.216432369042)*dim
        ref[4,0,1]=(-0.0692225921476)*(1+2.*(dim-1.)/(o+1.))+(-0.223633200428)*dim
        ref[4,1,0]=(0.907628019251)*(1+2.*(dim-1.)/(o+1.))+(1.23309620615)*dim
        ref[4,1,1]=(-0.77625708658)*(1+2.*(dim-1.)/(o+1.))+(2.02266001026)*dim
        ref[5,0,0]=(0.040336826806)*(1+2.*(dim-1.)/(o+1.))+(1.77568876641)*dim
        ref[5,0,1]=(-0.741065482842)*(1+2.*(dim-1.)/(o+1.))+(2.20116099255)*dim
        ref[5,1,0]=(-1.93385845474)*(1+2.*(dim-1.)/(o+1.))+(-0.691070068594)*dim
        ref[5,1,1]=(-0.539652361393)*(1+2.*(dim-1.)/(o+1.))+(-0.014778229134)*dim
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnBoundary_fromData_ReducedContinuousFunction_rank4(self):
      """
      tests integral of rank 4 Data on the FunctionOnBoundary

      assumptions: ReducedContinuousFunction(self.domain) exists
                   self.domain supports integral on FunctionOnBoundary
      """
      o=1
      dim=self.domain.getDim()
      w_ref=FunctionOnBoundary(self.domain)
      w=ReducedContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5, 3, 2),w)
      ref=numpy.zeros((4, 5, 3, 2),numpy.float_)
      if dim==2:
        arg[0,0,0,0]=(0.139765044323)*x[0]+(0.763652540371)*x[1]
        arg[0,0,0,1]=(1.22422998499)*x[0]+(-0.704059494925)*x[1]
        arg[0,0,1,0]=(-0.0738568759745)*x[0]+(0.227297731345)*x[1]
        arg[0,0,1,1]=(-0.438016525512)*x[0]+(-0.00290173552923)*x[1]
        arg[0,0,2,0]=(-0.142466265385)*x[0]+(0.159086109387)*x[1]
        arg[0,0,2,1]=(-1.29207470861)*x[0]+(0.171024552932)*x[1]
        arg[0,1,0,0]=(0.228469040214)*x[0]+(-1.11207386711)*x[1]
        arg[0,1,0,1]=(-0.0716121324998)*x[0]+(-0.293214318059)*x[1]
        arg[0,1,1,0]=(-0.552331240231)*x[0]+(0.483311325934)*x[1]
        arg[0,1,1,1]=(0.759050036013)*x[0]+(-0.192218075397)*x[1]
        arg[0,1,2,0]=(0.736408127399)*x[0]+(0.61962966835)*x[1]
        arg[0,1,2,1]=(-0.170110592043)*x[0]+(1.12024324836)*x[1]
        arg[0,2,0,0]=(1.15358129137)*x[0]+(0.00288801991025)*x[1]
        arg[0,2,0,1]=(0.34101930461)*x[0]+(1.44561406668)*x[1]
        arg[0,2,1,0]=(1.20180064935)*x[0]+(1.53696717349)*x[1]
        arg[0,2,1,1]=(-0.955282659666)*x[0]+(-0.232022391367)*x[1]
        arg[0,2,2,0]=(0.0907361523468)*x[0]+(0.0735957910178)*x[1]
        arg[0,2,2,1]=(0.377295180443)*x[0]+(-0.694318356175)*x[1]
        arg[0,3,0,0]=(1.38936937129)*x[0]+(-1.11460156576)*x[1]
        arg[0,3,0,1]=(-0.477623155591)*x[0]+(-1.38984205639)*x[1]
        arg[0,3,1,0]=(0.0643534425446)*x[0]+(1.38744715898)*x[1]
        arg[0,3,1,1]=(0.0144448337619)*x[0]+(-0.107478295727)*x[1]
        arg[0,3,2,0]=(0.494522779873)*x[0]+(-0.623994602046)*x[1]
        arg[0,3,2,1]=(0.865352526765)*x[0]+(-0.0330471549519)*x[1]
        arg[0,4,0,0]=(0.876802965861)*x[0]+(-0.72991482831)*x[1]
        arg[0,4,0,1]=(-1.59217671369)*x[0]+(-1.17443275047)*x[1]
        arg[0,4,1,0]=(0.980556838858)*x[0]+(-0.134374268551)*x[1]
        arg[0,4,1,1]=(-0.799387621998)*x[0]+(0.290707689722)*x[1]
        arg[0,4,2,0]=(-0.152070506377)*x[0]+(-0.720051351365)*x[1]
        arg[0,4,2,1]=(-0.726386342076)*x[0]+(-0.821126213298)*x[1]
        arg[1,0,0,0]=(0.0108792321621)*x[0]+(1.08482229519)*x[1]
        arg[1,0,0,1]=(0.395066734758)*x[0]+(-0.147576996716)*x[1]
        arg[1,0,1,0]=(1.34674167646)*x[0]+(-1.7762597766)*x[1]
        arg[1,0,1,1]=(0.145955749421)*x[0]+(0.0758020791017)*x[1]
        arg[1,0,2,0]=(-1.3779896638)*x[0]+(0.879047617812)*x[1]
        arg[1,0,2,1]=(-0.034533186491)*x[0]+(-0.961616456719)*x[1]
        arg[1,1,0,0]=(1.93531133223)*x[0]+(0.085806770105)*x[1]
        arg[1,1,0,1]=(-1.40810156335)*x[0]+(0.647954800386)*x[1]
        arg[1,1,1,0]=(-0.898682130534)*x[0]+(0.0293908578939)*x[1]
        arg[1,1,1,1]=(0.906365050374)*x[0]+(0.493604087615)*x[1]
        arg[1,1,2,0]=(0.200697611638)*x[0]+(0.595460978794)*x[1]
        arg[1,1,2,1]=(-0.137201357175)*x[0]+(-0.456327010456)*x[1]
        arg[1,2,0,0]=(-0.435882659968)*x[0]+(-0.0312083604574)*x[1]
        arg[1,2,0,1]=(-0.678923893852)*x[0]+(-0.559827568731)*x[1]
        arg[1,2,1,0]=(-0.721204171338)*x[0]+(-0.054379212442)*x[1]
        arg[1,2,1,1]=(-0.0624007046506)*x[0]+(0.168316149134)*x[1]
        arg[1,2,2,0]=(-1.58839995565)*x[0]+(-0.212071943846)*x[1]
        arg[1,2,2,1]=(-1.12936118963)*x[0]+(1.37132899453)*x[1]
        arg[1,3,0,0]=(-0.382791970195)*x[0]+(-1.08331380956)*x[1]
        arg[1,3,0,1]=(-0.83075188923)*x[0]+(-0.771795988467)*x[1]
        arg[1,3,1,0]=(-0.491393878842)*x[0]+(0.256082770048)*x[1]
        arg[1,3,1,1]=(1.16535762668)*x[0]+(1.15868916612)*x[1]
        arg[1,3,2,0]=(0.930916100823)*x[0]+(0.447810335514)*x[1]
        arg[1,3,2,1]=(0.542008480891)*x[0]+(0.428660301354)*x[1]
        arg[1,4,0,0]=(0.820283847448)*x[0]+(-0.861049765761)*x[1]
        arg[1,4,0,1]=(0.794597952616)*x[0]+(0.484963133102)*x[1]
        arg[1,4,1,0]=(-0.0864183500936)*x[0]+(-0.48652009912)*x[1]
        arg[1,4,1,1]=(-0.729261074705)*x[0]+(0.572520019661)*x[1]
        arg[1,4,2,0]=(0.874563670592)*x[0]+(-1.03978413308)*x[1]
        arg[1,4,2,1]=(-0.119980177921)*x[0]+(-0.416981275935)*x[1]
        arg[2,0,0,0]=(0.00851163981688)*x[0]+(0.0537719250367)*x[1]
        arg[2,0,0,1]=(0.325152071307)*x[0]+(0.852063357275)*x[1]
        arg[2,0,1,0]=(0.3798978673)*x[0]+(0.103107604149)*x[1]
        arg[2,0,1,1]=(0.241778324259)*x[0]+(0.840767359849)*x[1]
        arg[2,0,2,0]=(0.504220642882)*x[0]+(-0.296421218478)*x[1]
        arg[2,0,2,1]=(1.21850888213)*x[0]+(0.361422173343)*x[1]
        arg[2,1,0,0]=(1.23948863042)*x[0]+(-0.454591957846)*x[1]
        arg[2,1,0,1]=(-0.834341417745)*x[0]+(0.836809823188)*x[1]
        arg[2,1,1,0]=(0.305283593888)*x[0]+(-0.443709359754)*x[1]
        arg[2,1,1,1]=(-1.73751494542)*x[0]+(0.28249619071)*x[1]
        arg[2,1,2,0]=(1.64628846378)*x[0]+(0.929422692022)*x[1]
        arg[2,1,2,1]=(1.55964953056)*x[0]+(0.294507226966)*x[1]
        arg[2,2,0,0]=(-0.122178600734)*x[0]+(-0.836106215351)*x[1]
        arg[2,2,0,1]=(0.0560920262009)*x[0]+(-0.959524453552)*x[1]
        arg[2,2,1,0]=(-0.0235093950509)*x[0]+(1.00069942275)*x[1]
        arg[2,2,1,1]=(-1.6685983361)*x[0]+(-0.283415608089)*x[1]
        arg[2,2,2,0]=(1.88757268407)*x[0]+(-0.0970308753337)*x[1]
        arg[2,2,2,1]=(-0.620474974074)*x[0]+(0.24769276713)*x[1]
        arg[2,3,0,0]=(1.01637595389)*x[0]+(0.705160942822)*x[1]
        arg[2,3,0,1]=(1.14707395208)*x[0]+(0.705275592532)*x[1]
        arg[2,3,1,0]=(-0.758160952494)*x[0]+(0.305309048122)*x[1]
        arg[2,3,1,1]=(0.772402918026)*x[0]+(-0.052854508055)*x[1]
        arg[2,3,2,0]=(-0.154978687298)*x[0]+(0.418368826949)*x[1]
        arg[2,3,2,1]=(-0.845976442257)*x[0]+(0.20623978887)*x[1]
        arg[2,4,0,0]=(-0.466185710519)*x[0]+(-0.233766119139)*x[1]
        arg[2,4,0,1]=(-0.922877220469)*x[0]+(-0.688487528304)*x[1]
        arg[2,4,1,0]=(0.95836287786)*x[0]+(0.939609093639)*x[1]
        arg[2,4,1,1]=(-0.11982304155)*x[0]+(-0.257073186684)*x[1]
        arg[2,4,2,0]=(0.345817495959)*x[0]+(0.151979044778)*x[1]
        arg[2,4,2,1]=(-1.05379548205)*x[0]+(-0.436065928744)*x[1]
        arg[3,0,0,0]=(0.744945706398)*x[0]+(-0.649458468468)*x[1]
        arg[3,0,0,1]=(-1.04958965249)*x[0]+(0.639942303499)*x[1]
        arg[3,0,1,0]=(0.337716118074)*x[0]+(1.93458666396)*x[1]
        arg[3,0,1,1]=(0.303244531116)*x[0]+(0.436648998724)*x[1]
        arg[3,0,2,0]=(-1.55484904142)*x[0]+(0.0691158242075)*x[1]
        arg[3,0,2,1]=(-0.184571563941)*x[0]+(-0.617496186026)*x[1]
        arg[3,1,0,0]=(0.54665472418)*x[0]+(-0.685197219861)*x[1]
        arg[3,1,0,1]=(0.270341054659)*x[0]+(0.0123904138595)*x[1]
        arg[3,1,1,0]=(-0.551927232745)*x[0]+(-0.0835992799883)*x[1]
        arg[3,1,1,1]=(-0.731741032186)*x[0]+(0.351597478907)*x[1]
        arg[3,1,2,0]=(-0.678408784965)*x[0]+(-0.0779391332051)*x[1]
        arg[3,1,2,1]=(-0.873182924195)*x[0]+(-0.641772673128)*x[1]
        arg[3,2,0,0]=(-0.826223714873)*x[0]+(0.631844849849)*x[1]
        arg[3,2,0,1]=(-0.899714515249)*x[0]+(-0.010350665009)*x[1]
        arg[3,2,1,0]=(0.290678744119)*x[0]+(-0.5871053663)*x[1]
        arg[3,2,1,1]=(0.208288673139)*x[0]+(0.631917010925)*x[1]
        arg[3,2,2,0]=(-0.262706136542)*x[0]+(0.0853212850096)*x[1]
        arg[3,2,2,1]=(-1.28934754486)*x[0]+(0.686855142119)*x[1]
        arg[3,3,0,0]=(-1.36704032596)*x[0]+(-1.55543480824)*x[1]
        arg[3,3,0,1]=(-1.17014698336)*x[0]+(-0.652446157654)*x[1]
        arg[3,3,1,0]=(-0.547499757476)*x[0]+(-0.156519515585)*x[1]
        arg[3,3,1,1]=(0.228140026291)*x[0]+(-1.41014817371)*x[1]
        arg[3,3,2,0]=(-0.055056583329)*x[0]+(-0.412525603966)*x[1]
        arg[3,3,2,1]=(-0.524790489086)*x[0]+(-0.774411367008)*x[1]
        arg[3,4,0,0]=(-1.33491047191)*x[0]+(1.47855022808)*x[1]
        arg[3,4,0,1]=(-0.301405501225)*x[0]+(0.687178741851)*x[1]
        arg[3,4,1,0]=(0.213050803289)*x[0]+(0.428086322291)*x[1]
        arg[3,4,1,1]=(-1.67469009469)*x[0]+(0.857275314994)*x[1]
        arg[3,4,2,0]=(0.748812148687)*x[0]+(-0.0339722859922)*x[1]
        arg[3,4,2,1]=(-0.829454912563)*x[0]+(-0.637161668052)*x[1]
        ref[0,0,0,0]=(0.538138880568)*(1+2.*(dim-1.)/(o+1.))+(0.365278704126)*dim
        ref[0,0,0,1]=(0.532354921417)*(1+2.*(dim-1.)/(o+1.))+(-0.0121844313566)*dim
        ref[0,0,1,0]=(1.27037934234)*(1+2.*(dim-1.)/(o+1.))+(-1.11693848697)*dim
        ref[0,0,1,1]=(-0.317494874699)*(1+2.*(dim-1.)/(o+1.))+(-0.123423386343)*dim
        ref[0,0,2,0]=(-0.274542501137)*(1+2.*(dim-1.)/(o+1.))+(0.291162345138)*dim
        ref[0,0,2,1]=(-0.669231324784)*(1+2.*(dim-1.)/(o+1.))+(-0.451818830894)*dim
        ref[0,1,0,0]=(-1.55791021679)*(1+2.*(dim-1.)/(o+1.))+(0.674305389899)*dim
        ref[0,1,0,1]=(-1.04225461385)*(1+2.*(dim-1.)/(o+1.))+(0.67742816329)*dim
        ref[0,1,1,0]=(-0.553787932867)*(1+2.*(dim-1.)/(o+1.))+(0.48476801857)*dim
        ref[0,1,1,1]=(0.774284436582)*(1+2.*(dim-1.)/(o+1.))+(-0.207452475966)*dim
        ref[0,1,2,0]=(0.754663298028)*(1+2.*(dim-1.)/(o+1.))+(0.601374497721)*dim
        ref[0,1,2,1]=(0.378489135155)*(1+2.*(dim-1.)/(o+1.))+(0.571643521166)*dim
        ref[0,2,0,0]=(0.84270626402)*(1+2.*(dim-1.)/(o+1.))+(0.313763047258)*dim
        ref[0,2,0,1]=(0.741388749098)*(1+2.*(dim-1.)/(o+1.))+(1.0452446222)*dim
        ref[0,2,1,0]=(1.31801677967)*(1+2.*(dim-1.)/(o+1.))+(1.42075104317)*dim
        ref[0,2,1,1]=(0.342128499103)*(1+2.*(dim-1.)/(o+1.))+(-1.52943355014)*dim
        ref[0,2,2,0]=(-0.2097885639)*(1+2.*(dim-1.)/(o+1.))+(0.374120507264)*dim
        ref[0,2,2,1]=(0.517181369396)*(1+2.*(dim-1.)/(o+1.))+(-0.834204545127)*dim
        ref[0,3,0,0]=(0.546538945104)*(1+2.*(dim-1.)/(o+1.))+(-0.271771139576)*dim
        ref[0,3,0,1]=(-1.11041658618)*(1+2.*(dim-1.)/(o+1.))+(-0.757048625806)*dim
        ref[0,3,1,0]=(1.71248931933)*(1+2.*(dim-1.)/(o+1.))+(-0.260688717804)*dim
        ref[0,3,1,1]=(0.426783834682)*(1+2.*(dim-1.)/(o+1.))+(-0.519817296646)*dim
        ref[0,3,2,0]=(-0.159038051631)*(1+2.*(dim-1.)/(o+1.))+(0.0295662294573)*dim
        ref[0,3,2,1]=(-0.0935533877409)*(1+2.*(dim-1.)/(o+1.))+(0.925858759554)*dim
        ref[0,4,0,0]=(0.254413611162)*(1+2.*(dim-1.)/(o+1.))+(-0.107525473611)*dim
        ref[0,4,0,1]=(-1.80344635667)*(1+2.*(dim-1.)/(o+1.))+(-0.963163107492)*dim
        ref[0,4,1,0]=(0.77936654687)*(1+2.*(dim-1.)/(o+1.))+(0.0668160234379)*dim
        ref[0,4,1,1]=(-0.854656189741)*(1+2.*(dim-1.)/(o+1.))+(0.345976257465)*dim
        ref[0,4,2,0]=(0.236075361826)*(1+2.*(dim-1.)/(o+1.))+(-1.10819721957)*dim
        ref[0,4,2,1]=(-1.41443045549)*(1+2.*(dim-1.)/(o+1.))+(-0.133082099885)*dim
        ref[1,0,0,0]=(0.254601110632)*(1+2.*(dim-1.)/(o+1.))+(0.841100416717)*dim
        ref[1,0,0,1]=(-1.05142234949)*(1+2.*(dim-1.)/(o+1.))+(1.29891208753)*dim
        ref[1,0,1,0]=(-0.476182365596)*(1+2.*(dim-1.)/(o+1.))+(0.0466642654563)*dim
        ref[1,0,1,1]=(-1.2117362156)*(1+2.*(dim-1.)/(o+1.))+(1.43349404413)*dim
        ref[1,0,2,0]=(-0.82694411065)*(1+2.*(dim-1.)/(o+1.))+(0.328002064662)*dim
        ref[1,0,2,1]=(-0.784489634199)*(1+2.*(dim-1.)/(o+1.))+(-0.211660009011)*dim
        ref[1,1,0,0]=(1.67906802629)*(1+2.*(dim-1.)/(o+1.))+(0.342050076043)*dim
        ref[1,1,0,1]=(-0.798764638314)*(1+2.*(dim-1.)/(o+1.))+(0.03861787535)*dim
        ref[1,1,1,0]=(0.348279136631)*(1+2.*(dim-1.)/(o+1.))+(-1.21757040927)*dim
        ref[1,1,1,1]=(0.452104104709)*(1+2.*(dim-1.)/(o+1.))+(0.94786503328)*dim
        ref[1,1,2,0]=(0.855057493837)*(1+2.*(dim-1.)/(o+1.))+(-0.0588989034046)*dim
        ref[1,1,2,1]=(-0.666495642664)*(1+2.*(dim-1.)/(o+1.))+(0.0729672750328)*dim
        ref[1,2,0,0]=(-0.566691483673)*(1+2.*(dim-1.)/(o+1.))+(0.0996004632473)*dim
        ref[1,2,0,1]=(-0.127941886194)*(1+2.*(dim-1.)/(o+1.))+(-1.11080957639)*dim
        ref[1,2,1,0]=(-0.249856548379)*(1+2.*(dim-1.)/(o+1.))+(-0.525726835401)*dim
        ref[1,2,1,1]=(1.00195986166)*(1+2.*(dim-1.)/(o+1.))+(-0.896044417179)*dim
        ref[1,2,2,0]=(-0.96849834562)*(1+2.*(dim-1.)/(o+1.))+(-0.831973553872)*dim
        ref[1,2,2,1]=(0.204504064885)*(1+2.*(dim-1.)/(o+1.))+(0.0374637400154)*dim
        ref[1,3,0,0]=(-0.587139938504)*(1+2.*(dim-1.)/(o+1.))+(-0.878965841253)*dim
        ref[1,3,0,1]=(-0.013960550118)*(1+2.*(dim-1.)/(o+1.))+(-1.58858732758)*dim
        ref[1,3,1,0]=(-0.626445648631)*(1+2.*(dim-1.)/(o+1.))+(0.391134539838)*dim
        ref[1,3,1,1]=(0.793357894316)*(1+2.*(dim-1.)/(o+1.))+(1.53068889848)*dim
        ref[1,3,2,0]=(-0.30795291971)*(1+2.*(dim-1.)/(o+1.))+(1.68667935605)*dim
        ref[1,3,2,1]=(0.865506632636)*(1+2.*(dim-1.)/(o+1.))+(0.105162149608)*dim
        ref[1,4,0,0]=(0.0142921032519)*(1+2.*(dim-1.)/(o+1.))+(-0.0550580215643)*dim
        ref[1,4,0,1]=(1.74635688975)*(1+2.*(dim-1.)/(o+1.))+(-0.466795804028)*dim
        ref[1,4,1,0]=(0.202247079284)*(1+2.*(dim-1.)/(o+1.))+(-0.775185528498)*dim
        ref[1,4,1,1]=(0.518302547238)*(1+2.*(dim-1.)/(o+1.))+(-0.675043602282)*dim
        ref[1,4,2,0]=(-0.0110996487494)*(1+2.*(dim-1.)/(o+1.))+(-0.154120813742)*dim
        ref[1,4,2,1]=(0.445368223328)*(1+2.*(dim-1.)/(o+1.))+(-0.982329677184)*dim
        ref[2,0,0,0]=(1.11858205689)*(1+2.*(dim-1.)/(o+1.))+(-1.05629849203)*dim
        ref[2,0,0,1]=(0.326749663749)*(1+2.*(dim-1.)/(o+1.))+(0.850465764833)*dim
        ref[2,0,1,0]=(1.17260350442)*(1+2.*(dim-1.)/(o+1.))+(-0.689598032967)*dim
        ref[2,0,1,1]=(1.26453211291)*(1+2.*(dim-1.)/(o+1.))+(-0.181986428807)*dim
        ref[2,0,2,0]=(0.28543047736)*(1+2.*(dim-1.)/(o+1.))+(-0.0776310529561)*dim
        ref[2,0,2,1]=(1.67283723918)*(1+2.*(dim-1.)/(o+1.))+(-0.092906183711)*dim
        ref[2,1,0,0]=(0.375606493975)*(1+2.*(dim-1.)/(o+1.))+(0.409290178597)*dim
        ref[2,1,0,1]=(0.391717224425)*(1+2.*(dim-1.)/(o+1.))+(-0.389248818981)*dim
        ref[2,1,1,0]=(0.540376031372)*(1+2.*(dim-1.)/(o+1.))+(-0.678801797238)*dim
        ref[2,1,1,1]=(-0.781880488997)*(1+2.*(dim-1.)/(o+1.))+(-0.673138265714)*dim
        ref[2,1,2,0]=(1.19474704445)*(1+2.*(dim-1.)/(o+1.))+(1.38096411135)*dim
        ref[2,1,2,1]=(0.3333062674)*(1+2.*(dim-1.)/(o+1.))+(1.52085049012)*dim
        ref[2,2,0,0]=(-0.570564746042)*(1+2.*(dim-1.)/(o+1.))+(-0.387720070044)*dim
        ref[2,2,0,1]=(-0.906439974412)*(1+2.*(dim-1.)/(o+1.))+(0.00300754706095)*dim
        ref[2,2,1,0]=(1.35643048678)*(1+2.*(dim-1.)/(o+1.))+(-0.379240459088)*dim
        ref[2,2,1,1]=(-1.68522696847)*(1+2.*(dim-1.)/(o+1.))+(-0.266786975721)*dim
        ref[2,2,2,0]=(1.64773077393)*(1+2.*(dim-1.)/(o+1.))+(0.14281103481)*dim
        ref[2,2,2,1]=(0.161577377439)*(1+2.*(dim-1.)/(o+1.))+(-0.534359584383)*dim
        ref[2,3,0,0]=(0.117577368411)*(1+2.*(dim-1.)/(o+1.))+(1.6039595283)*dim
        ref[2,3,0,1]=(1.07270794583)*(1+2.*(dim-1.)/(o+1.))+(0.779641598778)*dim
        ref[2,3,1,0]=(0.940083930247)*(1+2.*(dim-1.)/(o+1.))+(-1.39293583462)*dim
        ref[2,3,1,1]=(-0.322170646263)*(1+2.*(dim-1.)/(o+1.))+(1.04171905623)*dim
        ref[2,3,2,0]=(-1.00498391043)*(1+2.*(dim-1.)/(o+1.))+(1.26837405008)*dim
        ref[2,3,2,1]=(0.148227404921)*(1+2.*(dim-1.)/(o+1.))+(-0.787964058307)*dim
        ref[2,4,0,0]=(-0.567878073929)*(1+2.*(dim-1.)/(o+1.))+(-0.132073755729)*dim
        ref[2,4,0,1]=(-0.265369521409)*(1+2.*(dim-1.)/(o+1.))+(-1.34599522736)*dim
        ref[2,4,1,0]=(0.87362235129)*(1+2.*(dim-1.)/(o+1.))+(1.02434962021)*dim
        ref[2,4,1,1]=(-1.05435356817)*(1+2.*(dim-1.)/(o+1.))+(0.67745733994)*dim
        ref[2,4,2,0]=(-0.0889174414067)*(1+2.*(dim-1.)/(o+1.))+(0.586713982143)*dim
        ref[2,4,2,1]=(-0.950446212097)*(1+2.*(dim-1.)/(o+1.))+(-0.539415198695)*dim
        ref[3,0,0,0]=(0.585351095733)*(1+2.*(dim-1.)/(o+1.))+(-0.489863857802)*dim
        ref[3,0,0,1]=(0.0818488289135)*(1+2.*(dim-1.)/(o+1.))+(-0.491496177908)*dim
        ref[3,0,1,0]=(1.83790469827)*(1+2.*(dim-1.)/(o+1.))+(0.43439808376)*dim
        ref[3,0,1,1]=(-0.812249042145)*(1+2.*(dim-1.)/(o+1.))+(1.55214257198)*dim
        ref[3,0,2,0]=(-1.62498605972)*(1+2.*(dim-1.)/(o+1.))+(0.139252842511)*dim
        ref[3,0,2,1]=(0.248660359392)*(1+2.*(dim-1.)/(o+1.))+(-1.05072810936)*dim
        ref[3,1,0,0]=(0.701570774477)*(1+2.*(dim-1.)/(o+1.))+(-0.840113270158)*dim
        ref[3,1,0,1]=(-1.01112314775)*(1+2.*(dim-1.)/(o+1.))+(1.29385461627)*dim
        ref[3,1,1,0]=(-0.38604780773)*(1+2.*(dim-1.)/(o+1.))+(-0.249478705003)*dim
        ref[3,1,1,1]=(-0.505728989541)*(1+2.*(dim-1.)/(o+1.))+(0.125585436262)*dim
        ref[3,1,2,0]=(0.170090820399)*(1+2.*(dim-1.)/(o+1.))+(-0.926438738569)*dim
        ref[3,1,2,1]=(-0.929125306455)*(1+2.*(dim-1.)/(o+1.))+(-0.585830290868)*dim
        ref[3,2,0,0]=(-0.155713736332)*(1+2.*(dim-1.)/(o+1.))+(-0.038665128691)*dim
        ref[3,2,0,1]=(-1.52172024486)*(1+2.*(dim-1.)/(o+1.))+(0.6116550646)*dim
        ref[3,2,1,0]=(0.320960641588)*(1+2.*(dim-1.)/(o+1.))+(-0.617387263769)*dim
        ref[3,2,1,1]=(1.35000898226)*(1+2.*(dim-1.)/(o+1.))+(-0.509803298192)*dim
        ref[3,2,2,0]=(-1.08694938974)*(1+2.*(dim-1.)/(o+1.))+(0.909564538204)*dim
        ref[3,2,2,1]=(-0.484173893901)*(1+2.*(dim-1.)/(o+1.))+(-0.11831850884)*dim
        ref[3,3,0,0]=(-1.30137401361)*(1+2.*(dim-1.)/(o+1.))+(-1.62110112059)*dim
        ref[3,3,0,1]=(-1.01955795295)*(1+2.*(dim-1.)/(o+1.))+(-0.803035188058)*dim
        ref[3,3,1,0]=(-0.86522946065)*(1+2.*(dim-1.)/(o+1.))+(0.161210187589)*dim
        ref[3,3,1,1]=(-1.45434545656)*(1+2.*(dim-1.)/(o+1.))+(0.272337309144)*dim
        ref[3,3,2,0]=(-1.09044516001)*(1+2.*(dim-1.)/(o+1.))+(0.622862972717)*dim
        ref[3,3,2,1]=(-1.56164744337)*(1+2.*(dim-1.)/(o+1.))+(0.262445587275)*dim
        ref[3,4,0,0]=(-0.0717688395634)*(1+2.*(dim-1.)/(o+1.))+(0.215408595737)*dim
        ref[3,4,0,1]=(-0.361057910751)*(1+2.*(dim-1.)/(o+1.))+(0.746831151378)*dim
        ref[3,4,1,0]=(1.3065764036)*(1+2.*(dim-1.)/(o+1.))+(-0.665439278018)*dim
        ref[3,4,1,1]=(0.00897444539195)*(1+2.*(dim-1.)/(o+1.))+(-0.826389225091)*dim
        ref[3,4,2,0]=(0.0968985993748)*(1+2.*(dim-1.)/(o+1.))+(0.61794126332)*dim
        ref[3,4,2,1]=(-0.247002319776)*(1+2.*(dim-1.)/(o+1.))+(-1.21961426084)*dim
      else:
        arg[0,0,0,0]=(0.249559417601)*x[0]+(-1.12919415836)*x[1]+(0.535083953587)*x[2]
        arg[0,0,0,1]=(-1.61154525877)*x[0]+(-0.339535503099)*x[1]+(-0.3734414422)*x[2]
        arg[0,0,1,0]=(-1.33426413331)*x[0]+(0.503654661149)*x[1]+(0.900718166029)*x[2]
        arg[0,0,1,1]=(-1.47348135079)*x[0]+(0.294426872223)*x[1]+(-0.092932038156)*x[2]
        arg[0,0,2,0]=(-0.659338105033)*x[0]+(0.276825152205)*x[1]+(-0.825904690993)*x[2]
        arg[0,0,2,1]=(-0.871512897535)*x[0]+(0.16237107696)*x[1]+(-1.37935248231)*x[2]
        arg[0,1,0,0]=(-0.83315792018)*x[0]+(-0.00675809409276)*x[1]+(0.464774846632)*x[2]
        arg[0,1,0,1]=(0.527155362932)*x[0]+(0.856707619808)*x[1]+(-1.68267649456)*x[2]
        arg[0,1,1,0]=(-0.524091384241)*x[0]+(-0.540620074365)*x[1]+(0.992745803077)*x[2]
        arg[0,1,1,1]=(1.61356254869)*x[0]+(1.50422956091)*x[1]+(0.522074610423)*x[2]
        arg[0,1,2,0]=(-0.786383950132)*x[0]+(0.240998133454)*x[1]+(-0.609039834527)*x[2]
        arg[0,1,2,1]=(-0.549608404174)*x[0]+(-0.850827315095)*x[1]+(0.267519458624)*x[2]
        arg[0,2,0,0]=(-0.171255468598)*x[0]+(-0.315808418391)*x[1]+(-1.31408126593)*x[2]
        arg[0,2,0,1]=(-0.0537048497397)*x[0]+(-1.05283989597)*x[1]+(-0.230505226375)*x[2]
        arg[0,2,1,0]=(0.175422426556)*x[0]+(-1.37203245283)*x[1]+(-1.31579047196)*x[2]
        arg[0,2,1,1]=(-0.678719594441)*x[0]+(0.325375648943)*x[1]+(-1.3715383088)*x[2]
        arg[0,2,2,0]=(1.34295310191)*x[0]+(-1.29856762722)*x[1]+(-0.604804069008)*x[2]
        arg[0,2,2,1]=(-0.38925176646)*x[0]+(-0.391078913511)*x[1]+(-0.147119806342)*x[2]
        arg[0,3,0,0]=(0.55897935716)*x[0]+(-0.938161813577)*x[1]+(-1.01170144437)*x[2]
        arg[0,3,0,1]=(-0.214720994691)*x[0]+(-0.936636326408)*x[1]+(1.23434495942)*x[2]
        arg[0,3,1,0]=(0.665136478224)*x[0]+(-0.223132283946)*x[1]+(0.513937660496)*x[2]
        arg[0,3,1,1]=(0.99663942166)*x[0]+(-0.870793744717)*x[1]+(-0.44642261239)*x[2]
        arg[0,3,2,0]=(0.426708530385)*x[0]+(-0.399913590631)*x[1]+(0.621569032554)*x[2]
        arg[0,3,2,1]=(-0.385564378699)*x[0]+(-1.17310910463)*x[1]+(-0.386725711607)*x[2]
        arg[0,4,0,0]=(-0.907605366918)*x[0]+(-1.15620807313)*x[1]+(0.690709991448)*x[2]
        arg[0,4,0,1]=(0.68925487184)*x[0]+(-0.731402302062)*x[1]+(-0.278506722879)*x[2]
        arg[0,4,1,0]=(0.607545316973)*x[0]+(-0.480514687124)*x[1]+(-0.261850296368)*x[2]
        arg[0,4,1,1]=(1.39016358303)*x[0]+(0.571194445557)*x[1]+(1.02359000173)*x[2]
        arg[0,4,2,0]=(1.19042892582)*x[0]+(-0.265599567429)*x[1]+(-0.412933583721)*x[2]
        arg[0,4,2,1]=(0.105768402681)*x[0]+(0.529442827604)*x[1]+(1.70332495695)*x[2]
        arg[1,0,0,0]=(-0.466512450689)*x[0]+(-0.821724116037)*x[1]+(0.315646947734)*x[2]
        arg[1,0,0,1]=(-0.343965024161)*x[0]+(-0.718276908655)*x[1]+(-0.89648206258)*x[2]
        arg[1,0,1,0]=(0.533001165605)*x[0]+(0.0431632707839)*x[1]+(-0.34028160642)*x[2]
        arg[1,0,1,1]=(-0.0905898060243)*x[0]+(-0.634642149101)*x[1]+(-1.13501642684)*x[2]
        arg[1,0,2,0]=(0.660943497868)*x[0]+(-0.354792953445)*x[1]+(1.28780893672)*x[2]
        arg[1,0,2,1]=(-1.2338460337)*x[0]+(-1.07489268654)*x[1]+(-1.32063885434)*x[2]
        arg[1,1,0,0]=(-0.564276365398)*x[0]+(1.03120151704)*x[1]+(-1.15799041025)*x[2]
        arg[1,1,0,1]=(0.715308242213)*x[0]+(1.38372056508)*x[1]+(0.143855804663)*x[2]
        arg[1,1,1,0]=(-0.359352464902)*x[0]+(1.12698364061)*x[1]+(0.592356171876)*x[2]
        arg[1,1,1,1]=(-0.450554686066)*x[0]+(-0.546866217679)*x[1]+(1.0842968414)*x[2]
        arg[1,1,2,0]=(-0.477267172046)*x[0]+(-0.863491255396)*x[1]+(-0.991333852913)*x[2]
        arg[1,1,2,1]=(0.402599196931)*x[0]+(-0.635287203482)*x[1]+(-0.277210261227)*x[2]
        arg[1,2,0,0]=(-1.56932441015)*x[0]+(-1.56442238007)*x[1]+(-0.375354109955)*x[2]
        arg[1,2,0,1]=(0.193969262086)*x[0]+(-0.0913153952113)*x[1]+(-0.764434182025)*x[2]
        arg[1,2,1,0]=(-0.836765580561)*x[0]+(0.84683301026)*x[1]+(-1.05033967172)*x[2]
        arg[1,2,1,1]=(1.68348957076)*x[0]+(-1.14474223991)*x[1]+(-0.0028142026612)*x[2]
        arg[1,2,2,0]=(-0.0921618478149)*x[0]+(-0.555660652821)*x[1]+(-0.398214997034)*x[2]
        arg[1,2,2,1]=(1.03261791276)*x[0]+(-0.650225649301)*x[1]+(1.63375423051)*x[2]
        arg[1,3,0,0]=(-0.348599695393)*x[0]+(-1.20215183267)*x[1]+(-0.1129459677)*x[2]
        arg[1,3,0,1]=(0.877317784105)*x[0]+(-0.587978566735)*x[1]+(1.10042804546)*x[2]
        arg[1,3,1,0]=(-0.547033849227)*x[0]+(-1.28198893788)*x[1]+(-0.424854587074)*x[2]
        arg[1,3,1,1]=(0.739360669292)*x[0]+(-0.66359460621)*x[1]+(1.08588749669)*x[2]
        arg[1,3,2,0]=(0.137130432185)*x[0]+(-0.123436803848)*x[1]+(-1.31161936702)*x[2]
        arg[1,3,2,1]=(-0.0171220220237)*x[0]+(1.44556512776)*x[1]+(0.665008556384)*x[2]
        arg[1,4,0,0]=(-0.304568160155)*x[0]+(-0.453489411096)*x[1]+(0.769736915169)*x[2]
        arg[1,4,0,1]=(-0.389567079496)*x[0]+(1.01465816808)*x[1]+(1.05389792181)*x[2]
        arg[1,4,1,0]=(-0.293019902659)*x[0]+(0.935020375302)*x[1]+(0.465003429909)*x[2]
        arg[1,4,1,1]=(1.26426164192)*x[0]+(0.53660188438)*x[1]+(-1.26997852414)*x[2]
        arg[1,4,2,0]=(0.447013260347)*x[0]+(-0.292646095073)*x[1]+(-0.141236248636)*x[2]
        arg[1,4,2,1]=(0.627246227193)*x[0]+(-1.07310030602)*x[1]+(0.414244174014)*x[2]
        arg[2,0,0,0]=(0.0450404092088)*x[0]+(-0.092484507099)*x[1]+(0.416562031216)*x[2]
        arg[2,0,0,1]=(0.437290559939)*x[0]+(-1.00144259207)*x[1]+(1.26666155425)*x[2]
        arg[2,0,1,0]=(0.0919297766737)*x[0]+(0.925519273623)*x[1]+(-0.582669368425)*x[2]
        arg[2,0,1,1]=(1.40247347323)*x[0]+(-0.896360612277)*x[1]+(-0.235401929947)*x[2]
        arg[2,0,2,0]=(-0.0760543780641)*x[0]+(-1.12543060216)*x[1]+(-1.73005861985)*x[2]
        arg[2,0,2,1]=(-0.0540914682488)*x[0]+(0.686173107995)*x[1]+(-0.499264219762)*x[2]
        arg[2,1,0,0]=(1.29412006754)*x[0]+(1.17564161143)*x[1]+(0.30140132879)*x[2]
        arg[2,1,0,1]=(-0.987572082341)*x[0]+(-1.73401714184)*x[1]+(0.542753625023)*x[2]
        arg[2,1,1,0]=(-0.162426291727)*x[0]+(0.508287986142)*x[1]+(0.42345261935)*x[2]
        arg[2,1,1,1]=(-0.436007145923)*x[0]+(0.344485478615)*x[1]+(0.867081685885)*x[2]
        arg[2,1,2,0]=(1.23340638096)*x[0]+(1.16627983483)*x[1]+(0.291792397233)*x[2]
        arg[2,1,2,1]=(1.81405521045)*x[0]+(-0.358711641776)*x[1]+(1.67495221975)*x[2]
        arg[2,2,0,0]=(-0.998776647505)*x[0]+(1.38487801142)*x[1]+(-0.199886556937)*x[2]
        arg[2,2,0,1]=(-0.151564577745)*x[0]+(0.100245326484)*x[1]+(-0.98457000197)*x[2]
        arg[2,2,1,0]=(-0.767959771993)*x[0]+(-0.641538616024)*x[1]+(0.0206020120898)*x[2]
        arg[2,2,1,1]=(-0.116988718822)*x[0]+(-1.15499563681)*x[1]+(-0.187589599016)*x[2]
        arg[2,2,2,0]=(0.682354528276)*x[0]+(0.606005834005)*x[1]+(0.11468472132)*x[2]
        arg[2,2,2,1]=(-0.547781355521)*x[0]+(1.23047427005)*x[1]+(0.854018236174)*x[2]
        arg[2,3,0,0]=(0.368349078554)*x[0]+(-1.12153473446)*x[1]+(0.114116055257)*x[2]
        arg[2,3,0,1]=(0.838089123891)*x[0]+(0.283469213399)*x[1]+(0.249874271688)*x[2]
        arg[2,3,1,0]=(-1.03882746612)*x[0]+(0.159711106612)*x[1]+(-1.57272656988)*x[2]
        arg[2,3,1,1]=(-1.25439210606)*x[0]+(-0.801588333704)*x[1]+(-0.166928557775)*x[2]
        arg[2,3,2,0]=(0.0650011438717)*x[0]+(0.879974749381)*x[1]+(-0.276534309239)*x[2]
        arg[2,3,2,1]=(-0.774913691581)*x[0]+(-0.360358063631)*x[1]+(-0.368288145309)*x[2]
        arg[2,4,0,0]=(-0.855093207242)*x[0]+(0.82049336972)*x[1]+(-0.496007273696)*x[2]
        arg[2,4,0,1]=(-0.113408635338)*x[0]+(0.116297312211)*x[1]+(-1.05279504872)*x[2]
        arg[2,4,1,0]=(1.59034968962)*x[0]+(-0.993664270946)*x[1]+(-1.44097244774)*x[2]
        arg[2,4,1,1]=(-0.554191144546)*x[0]+(-1.30747375036)*x[1]+(-1.17946836248)*x[2]
        arg[2,4,2,0]=(0.626012137217)*x[0]+(0.518243725142)*x[1]+(0.0933303150622)*x[2]
        arg[2,4,2,1]=(-0.10169564723)*x[0]+(1.45820149276)*x[1]+(-0.0393269028179)*x[2]
        arg[3,0,0,0]=(-0.0404687905735)*x[0]+(0.0715700915639)*x[1]+(1.45121483418)*x[2]
        arg[3,0,0,1]=(1.25470808156)*x[0]+(-0.119317783496)*x[1]+(0.28341020961)*x[2]
        arg[3,0,1,0]=(0.352807293701)*x[0]+(0.230591898707)*x[1]+(-0.580416381826)*x[2]
        arg[3,0,1,1]=(1.6169875055)*x[0]+(0.515147269207)*x[1]+(-0.461088044559)*x[2]
        arg[3,0,2,0]=(0.345374169211)*x[0]+(0.165179946106)*x[1]+(0.0761591935161)*x[2]
        arg[3,0,2,1]=(-0.367306414632)*x[0]+(-0.236156646602)*x[1]+(-0.388955870267)*x[2]
        arg[3,1,0,0]=(-0.27984693051)*x[0]+(-0.0431489812153)*x[1]+(-0.315624571395)*x[2]
        arg[3,1,0,1]=(-0.923109329983)*x[0]+(0.71133451696)*x[1]+(-0.793551470792)*x[2]
        arg[3,1,1,0]=(-0.434858211337)*x[0]+(0.943115368224)*x[1]+(0.195339633021)*x[2]
        arg[3,1,1,1]=(0.517533657447)*x[0]+(-0.975487574798)*x[1]+(0.00389777788827)*x[2]
        arg[3,1,2,0]=(-1.29728995298)*x[0]+(-0.519355341481)*x[1]+(-0.986985327874)*x[2]
        arg[3,1,2,1]=(-0.104824066468)*x[0]+(0.155059832097)*x[1]+(-0.513194354212)*x[2]
        arg[3,2,0,0]=(-0.053676798897)*x[0]+(-0.957854779002)*x[1]+(0.575673251241)*x[2]
        arg[3,2,0,1]=(0.828930053602)*x[0]+(0.122518313839)*x[1]+(-0.711848816688)*x[2]
        arg[3,2,1,0]=(0.468063581449)*x[0]+(0.233244036405)*x[1]+(-1.06333921952)*x[2]
        arg[3,2,1,1]=(-0.178080946586)*x[0]+(0.481804828166)*x[1]+(-0.25332042181)*x[2]
        arg[3,2,2,0]=(-0.2520881536)*x[0]+(-1.03459378989)*x[1]+(-0.284550808049)*x[2]
        arg[3,2,2,1]=(-0.778922518429)*x[0]+(1.0037625091)*x[1]+(-0.252088333444)*x[2]
        arg[3,3,0,0]=(0.467759844055)*x[0]+(0.598630111441)*x[1]+(-0.180631591223)*x[2]
        arg[3,3,0,1]=(-0.704465112342)*x[0]+(0.577992884135)*x[1]+(0.614639798328)*x[2]
        arg[3,3,1,0]=(0.58772281958)*x[0]+(0.267751378071)*x[1]+(0.11546949909)*x[2]
        arg[3,3,1,1]=(-0.220490484007)*x[0]+(-0.346075755455)*x[1]+(0.709249188793)*x[2]
        arg[3,3,2,0]=(-0.03823885045)*x[0]+(1.0070390689)*x[1]+(0.786191739951)*x[2]
        arg[3,3,2,1]=(0.468772686804)*x[0]+(-0.416774817651)*x[1]+(1.65342478719)*x[2]
        arg[3,4,0,0]=(0.69436779527)*x[0]+(-0.644342658009)*x[1]+(-0.6765845785)*x[2]
        arg[3,4,0,1]=(0.210253965392)*x[0]+(0.208735045316)*x[1]+(-0.553442080096)*x[2]
        arg[3,4,1,0]=(-0.0680478107141)*x[0]+(0.770215954791)*x[1]+(-0.876270039559)*x[2]
        arg[3,4,1,1]=(-0.384812708168)*x[0]+(-0.482707860969)*x[1]+(1.00274437781)*x[2]
        arg[3,4,2,0]=(-0.144567417774)*x[0]+(0.277773632892)*x[1]+(1.3123295509)*x[2]
        arg[3,4,2,1]=(0.00870646304914)*x[0]+(-0.254229329332)*x[1]+(-1.66538680273)*x[2]
        ref[0,0,0,0]=(0.841760617241)*(1+2.*(dim-1.)/(o+1.))+(-1.18631140441)*dim
        ref[0,0,0,1]=(-0.610007337864)*(1+2.*(dim-1.)/(o+1.))+(-1.71451486621)*dim
        ref[0,0,1,0]=(-0.765605897612)*(1+2.*(dim-1.)/(o+1.))+(0.83571459148)*dim
        ref[0,0,1,1]=(-0.128014940891)*(1+2.*(dim-1.)/(o+1.))+(-1.14397157584)*dim
        ref[0,0,2,0]=(-0.969876086955)*(1+2.*(dim-1.)/(o+1.))+(-0.238541556867)*dim
        ref[0,0,2,1]=(-0.434074835341)*(1+2.*(dim-1.)/(o+1.))+(-1.65441946755)*dim
        ref[0,1,0,0]=(0.346878720962)*(1+2.*(dim-1.)/(o+1.))+(-0.722019888602)*dim
        ref[0,1,0,1]=(-0.163332213104)*(1+2.*(dim-1.)/(o+1.))+(-0.135481298717)*dim
        ref[0,1,1,0]=(0.64836517897)*(1+2.*(dim-1.)/(o+1.))+(-0.720330834499)*dim
        ref[0,1,1,1]=(1.50578185724)*(1+2.*(dim-1.)/(o+1.))+(2.13408486278)*dim
        ref[0,1,2,0]=(-1.24719412419)*(1+2.*(dim-1.)/(o+1.))+(0.0927684729819)*dim
        ref[0,1,2,1]=(0.798388744364)*(1+2.*(dim-1.)/(o+1.))+(-1.93130500501)*dim
        ref[0,2,0,0]=(-1.9926882083)*(1+2.*(dim-1.)/(o+1.))+(0.191543055387)*dim
        ref[0,2,0,1]=(-2.16203894433)*(1+2.*(dim-1.)/(o+1.))+(0.824988972246)*dim
        ref[0,2,1,0]=(-2.24934415172)*(1+2.*(dim-1.)/(o+1.))+(-0.263056346511)*dim
        ref[0,2,1,1]=(-0.719152132879)*(1+2.*(dim-1.)/(o+1.))+(-1.00573012142)*dim
        ref[0,2,2,0]=(-0.83751635258)*(1+2.*(dim-1.)/(o+1.))+(0.277097758257)*dim
        ref[0,2,2,1]=(-0.979077481565)*(1+2.*(dim-1.)/(o+1.))+(0.0516269952525)*dim
        ref[0,3,0,0]=(-0.623873484234)*(1+2.*(dim-1.)/(o+1.))+(-0.767010416553)*dim
        ref[0,3,0,1]=(-1.0425822452)*(1+2.*(dim-1.)/(o+1.))+(1.12556988353)*dim
        ref[0,3,1,0]=(0.548600253744)*(1+2.*(dim-1.)/(o+1.))+(0.407341601031)*dim
        ref[0,3,1,1]=(-0.531963828356)*(1+2.*(dim-1.)/(o+1.))+(0.211386892908)*dim
        ref[0,3,2,0]=(1.33006849004)*(1+2.*(dim-1.)/(o+1.))+(-0.681704517735)*dim
        ref[0,3,2,1]=(-1.98393315604)*(1+2.*(dim-1.)/(o+1.))+(0.0385339611081)*dim
        ref[0,4,0,0]=(-0.381511396261)*(1+2.*(dim-1.)/(o+1.))+(-0.991592052336)*dim
        ref[0,4,0,1]=(1.00472425157)*(1+2.*(dim-1.)/(o+1.))+(-1.32537840467)*dim
        ref[0,4,1,0]=(1.04630307783)*(1+2.*(dim-1.)/(o+1.))+(-1.18112274434)*dim
        ref[0,4,1,1]=(1.61731240292)*(1+2.*(dim-1.)/(o+1.))+(1.3676356274)*dim
        ref[0,4,2,0]=(1.41653781097)*(1+2.*(dim-1.)/(o+1.))+(-0.904642036296)*dim
        ref[0,4,2,1]=(1.61515833451)*(1+2.*(dim-1.)/(o+1.))+(0.723377852725)*dim
        ref[1,0,0,0]=(-0.879218635)*(1+2.*(dim-1.)/(o+1.))+(-0.0933709839921)*dim
        ref[1,0,0,1]=(-1.26687353853)*(1+2.*(dim-1.)/(o+1.))+(-0.691850456865)*dim
        ref[1,0,1,0]=(1.79928802103)*(1+2.*(dim-1.)/(o+1.))+(-1.56340519106)*dim
        ref[1,0,1,1]=(-1.05495675007)*(1+2.*(dim-1.)/(o+1.))+(-0.805291631901)*dim
        ref[1,0,2,0]=(-0.339423840638)*(1+2.*(dim-1.)/(o+1.))+(1.93338332178)*dim
        ref[1,0,2,1]=(-1.15347644771)*(1+2.*(dim-1.)/(o+1.))+(-2.47590112686)*dim
        ref[1,1,0,0]=(-1.22232845026)*(1+2.*(dim-1.)/(o+1.))+(0.531263191656)*dim
        ref[1,1,0,1]=(0.318005373727)*(1+2.*(dim-1.)/(o+1.))+(1.92487923823)*dim
        ref[1,1,1,0]=(0.157193055461)*(1+2.*(dim-1.)/(o+1.))+(1.20279429213)*dim
        ref[1,1,1,1]=(0.0525914033484)*(1+2.*(dim-1.)/(o+1.))+(0.0342845343071)*dim
        ref[1,1,2,0]=(-0.319573431275)*(1+2.*(dim-1.)/(o+1.))+(-2.01251884908)*dim
        ref[1,1,2,1]=(-1.67524932091)*(1+2.*(dim-1.)/(o+1.))+(1.16535105313)*dim
        ref[1,2,0,0]=(-2.25088459423)*(1+2.*(dim-1.)/(o+1.))+(-1.25821630594)*dim
        ref[1,2,0,1]=(-0.447759463011)*(1+2.*(dim-1.)/(o+1.))+(-0.214020852139)*dim
        ref[1,2,1,0]=(-1.26590071291)*(1+2.*(dim-1.)/(o+1.))+(0.225628470886)*dim
        ref[1,2,1,1]=(-0.372878924951)*(1+2.*(dim-1.)/(o+1.))+(0.90881205314)*dim
        ref[1,2,2,0]=(1.36047043606)*(1+2.*(dim-1.)/(o+1.))+(-2.40650793373)*dim
        ref[1,2,2,1]=(1.13871462186)*(1+2.*(dim-1.)/(o+1.))+(0.877431872103)*dim
        ref[1,3,0,0]=(-1.17967356605)*(1+2.*(dim-1.)/(o+1.))+(-0.48402392971)*dim
        ref[1,3,0,1]=(-0.177974933247)*(1+2.*(dim-1.)/(o+1.))+(1.56774219607)*dim
        ref[1,3,1,0]=(-1.40936193115)*(1+2.*(dim-1.)/(o+1.))+(-0.844515443034)*dim
        ref[1,3,1,1]=(0.790537554795)*(1+2.*(dim-1.)/(o+1.))+(0.371116004973)*dim
        ref[1,3,2,0]=(0.242057644413)*(1+2.*(dim-1.)/(o+1.))+(-1.5399833831)*dim
        ref[1,3,2,1]=(1.70371239948)*(1+2.*(dim-1.)/(o+1.))+(0.389739262639)*dim
        ref[1,4,0,0]=(-0.375645743677)*(1+2.*(dim-1.)/(o+1.))+(0.387325087595)*dim
        ref[1,4,0,1]=(0.0489286471987)*(1+2.*(dim-1.)/(o+1.))+(1.63006036319)*dim
        ref[1,4,1,0]=(1.33203691786)*(1+2.*(dim-1.)/(o+1.))+(-0.225033015313)*dim
        ref[1,4,1,1]=(0.332297845981)*(1+2.*(dim-1.)/(o+1.))+(0.198587156175)*dim
        ref[1,4,2,0]=(-1.22288078692)*(1+2.*(dim-1.)/(o+1.))+(1.23601170355)*dim
        ref[1,4,2,1]=(0.0299493142028)*(1+2.*(dim-1.)/(o+1.))+(-0.0615592190171)*dim
        ref[2,0,0,0]=(0.0963155205346)*(1+2.*(dim-1.)/(o+1.))+(0.272802412791)*dim
        ref[2,0,0,1]=(1.12280541023)*(1+2.*(dim-1.)/(o+1.))+(-0.420295888105)*dim
        ref[2,0,1,0]=(-0.296878488858)*(1+2.*(dim-1.)/(o+1.))+(0.73165817073)*dim
        ref[2,0,1,1]=(-0.407704337172)*(1+2.*(dim-1.)/(o+1.))+(0.678415268175)*dim
        ref[2,0,2,0]=(-2.37402978977)*(1+2.*(dim-1.)/(o+1.))+(-0.557513810308)*dim
        ref[2,0,2,1]=(1.2564150767)*(1+2.*(dim-1.)/(o+1.))+(-1.12359765672)*dim
        ref[2,1,0,0]=(0.824412811392)*(1+2.*(dim-1.)/(o+1.))+(1.94675019637)*dim
        ref[2,1,0,1]=(-1.06505178789)*(1+2.*(dim-1.)/(o+1.))+(-1.11378381126)*dim
        ref[2,1,1,0]=(0.200096305752)*(1+2.*(dim-1.)/(o+1.))+(0.569218008012)*dim
        ref[2,1,1,1]=(1.08680720067)*(1+2.*(dim-1.)/(o+1.))+(-0.311247182096)*dim
        ref[2,1,2,0]=(1.3764914842)*(1+2.*(dim-1.)/(o+1.))+(1.31498712883)*dim
        ref[2,1,2,1]=(2.26537199157)*(1+2.*(dim-1.)/(o+1.))+(0.864923796858)*dim
        ref[2,2,0,0]=(0.750025182887)*(1+2.*(dim-1.)/(o+1.))+(-0.563810375913)*dim
        ref[2,2,0,1]=(-0.806933575047)*(1+2.*(dim-1.)/(o+1.))+(-0.228955678184)*dim
        ref[2,2,1,0]=(-1.53132620021)*(1+2.*(dim-1.)/(o+1.))+(0.14242982428)*dim
        ref[2,2,1,1]=(-0.889411443106)*(1+2.*(dim-1.)/(o+1.))+(-0.570162511538)*dim
        ref[2,2,2,0]=(-0.0309717316948)*(1+2.*(dim-1.)/(o+1.))+(1.4340168153)*dim
        ref[2,2,2,1]=(1.48963237375)*(1+2.*(dim-1.)/(o+1.))+(0.0470787769518)*dim
        ref[2,3,0,0]=(-0.0184174113004)*(1+2.*(dim-1.)/(o+1.))+(-0.620652189349)*dim
        ref[2,3,0,1]=(0.597433991421)*(1+2.*(dim-1.)/(o+1.))+(0.773998617557)*dim
        ref[2,3,1,0]=(-1.10704622688)*(1+2.*(dim-1.)/(o+1.))+(-1.34479670251)*dim
        ref[2,3,1,1]=(-0.369209798891)*(1+2.*(dim-1.)/(o+1.))+(-1.85369919864)*dim
        ref[2,3,2,0]=(-0.286165609161)*(1+2.*(dim-1.)/(o+1.))+(0.954607193175)*dim
        ref[2,3,2,1]=(-0.0948450773617)*(1+2.*(dim-1.)/(o+1.))+(-1.40871482316)*dim
        ref[2,4,0,0]=(-0.350521344827)*(1+2.*(dim-1.)/(o+1.))+(-0.180085766391)*dim
        ref[2,4,0,1]=(-1.10948105015)*(1+2.*(dim-1.)/(o+1.))+(0.0595746783013)*dim
        ref[2,4,1,0]=(-0.251521967553)*(1+2.*(dim-1.)/(o+1.))+(-0.592765061514)*dim
        ref[2,4,1,1]=(-2.22732408506)*(1+2.*(dim-1.)/(o+1.))+(-0.813809172329)*dim
        ref[2,4,2,0]=(-1.4091057051)*(1+2.*(dim-1.)/(o+1.))+(2.64669188252)*dim
        ref[2,4,2,1]=(-0.0262518153279)*(1+2.*(dim-1.)/(o+1.))+(1.34343075804)*dim
        ref[3,0,0,0]=(2.08524442815)*(1+2.*(dim-1.)/(o+1.))+(-0.602928292982)*dim
        ref[3,0,0,1]=(0.227442743255)*(1+2.*(dim-1.)/(o+1.))+(1.19135776442)*dim
        ref[3,0,1,0]=(1.03371796145)*(1+2.*(dim-1.)/(o+1.))+(-1.03073515087)*dim
        ref[3,0,1,1]=(0.205903763449)*(1+2.*(dim-1.)/(o+1.))+(1.4651429667)*dim
        ref[3,0,2,0]=(1.54387435524)*(1+2.*(dim-1.)/(o+1.))+(-0.957161046404)*dim
        ref[3,0,2,1]=(-0.487693556268)*(1+2.*(dim-1.)/(o+1.))+(-0.504725375232)*dim
        ref[3,1,0,0]=(0.319450917291)*(1+2.*(dim-1.)/(o+1.))+(-0.958071400411)*dim
        ref[3,1,0,1]=(-0.0756676477654)*(1+2.*(dim-1.)/(o+1.))+(-0.929658636049)*dim
        ref[3,1,1,0]=(0.0109030665653)*(1+2.*(dim-1.)/(o+1.))+(0.692693723343)*dim
        ref[3,1,1,1]=(0.540904091106)*(1+2.*(dim-1.)/(o+1.))+(-0.994960230569)*dim
        ref[3,1,2,0]=(-1.39229945953)*(1+2.*(dim-1.)/(o+1.))+(-1.4113311628)*dim
        ref[3,1,2,1]=(-0.120906779216)*(1+2.*(dim-1.)/(o+1.))+(-0.342051809366)*dim
        ref[3,2,0,0]=(-0.61477618653)*(1+2.*(dim-1.)/(o+1.))+(0.178917859873)*dim
        ref[3,2,0,1]=(0.286238606318)*(1+2.*(dim-1.)/(o+1.))+(-0.0466390555654)*dim
        ref[3,2,1,0]=(-0.0560560726323)*(1+2.*(dim-1.)/(o+1.))+(-0.305975529033)*dim
        ref[3,2,1,1]=(0.935166077059)*(1+2.*(dim-1.)/(o+1.))+(-0.884762617288)*dim
        ref[3,2,2,0]=(0.57015079501)*(1+2.*(dim-1.)/(o+1.))+(-2.14138354655)*dim
        ref[3,2,2,1]=(-0.699470490706)*(1+2.*(dim-1.)/(o+1.))+(0.672222147932)*dim
        ref[3,3,0,0]=(1.33431543118)*(1+2.*(dim-1.)/(o+1.))+(-0.448557066912)*dim
        ref[3,3,0,1]=(-1.03724498716)*(1+2.*(dim-1.)/(o+1.))+(1.52541255729)*dim
        ref[3,3,1,0]=(-0.00371131395733)*(1+2.*(dim-1.)/(o+1.))+(0.974655010698)*dim
        ref[3,3,1,1]=(0.0918410742793)*(1+2.*(dim-1.)/(o+1.))+(0.0508418750519)*dim
        ref[3,3,2,0]=(1.63372857769)*(1+2.*(dim-1.)/(o+1.))+(0.121263380706)*dim
        ref[3,3,2,1]=(0.761569446324)*(1+2.*(dim-1.)/(o+1.))+(0.943853210019)*dim
        ref[3,4,0,0]=(-0.625751279162)*(1+2.*(dim-1.)/(o+1.))+(-0.000808162076656)*dim
        ref[3,4,0,1]=(-0.113324890356)*(1+2.*(dim-1.)/(o+1.))+(-0.0211281790319)*dim
        ref[3,4,1,0]=(0.296144237506)*(1+2.*(dim-1.)/(o+1.))+(-0.470246132989)*dim
        ref[3,4,1,1]=(0.194264324372)*(1+2.*(dim-1.)/(o+1.))+(-0.0590405157015)*dim
        ref[3,4,2,0]=(1.63851379122)*(1+2.*(dim-1.)/(o+1.))+(-0.192978025203)*dim
        ref[3,4,2,1]=(-0.757374882295)*(1+2.*(dim-1.)/(o+1.))+(-1.15353478672)*dim
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5, 3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnBoundary_fromData_FunctionOnBoundary_rank0(self):
      """
      tests integral of rank 0 Data on the FunctionOnBoundary

      assumptions: FunctionOnBoundary(self.domain) exists
                   self.domain supports integral on FunctionOnBoundary
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnBoundary(self.domain)
      w=FunctionOnBoundary(self.domain)
      x=w.getX()
      arg=Data(0,(),w)
      if dim==2:
        arg=(0.626062912046)*x[0]**o+(0.600712856149)*x[0]+(0.0478861882442)*x[1]**o+(0.77008552734)*x[1]
        ref=(0.67394910029)*(1+2.*(dim-1.)/(o+1.))+(1.37079838349)*dim
      else:
        arg=(0.85588880126)*x[0]**o+(-0.898804533462)*x[0]+(-0.270816921088)*x[1]**o+(-0.842823912373)*x[1]+(-0.47559690762)*x[2]**o+(0.751770125129)*x[2]
        ref=(0.109474972552)*(1+2.*(dim-1.)/(o+1.))+(-0.989858320706)*dim
      res=integrate(arg)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnBoundary_fromData_FunctionOnBoundary_rank1(self):
      """
      tests integral of rank 1 Data on the FunctionOnBoundary

      assumptions: FunctionOnBoundary(self.domain) exists
                   self.domain supports integral on FunctionOnBoundary
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnBoundary(self.domain)
      w=FunctionOnBoundary(self.domain)
      x=w.getX()
      arg=Data(0,(2,),w)
      ref=numpy.zeros((2,),numpy.float_)
      if dim==2:
        arg[0]=(0.298913063662)*x[0]**o+(-0.250822875451)*x[0]+(0.707311253225)*x[1]**o+(0.0408091445557)*x[1]
        arg[1]=(-0.70143850042)*x[0]**o+(-0.80121227022)*x[0]+(0.0731514590408)*x[1]**o+(0.621969876113)*x[1]
        ref[0]=(1.00622431689)*(1+2.*(dim-1.)/(o+1.))+(-0.210013730895)*dim
        ref[1]=(-0.628287041379)*(1+2.*(dim-1.)/(o+1.))+(-0.179242394108)*dim
      else:
        arg[0]=(0.676976938336)*x[0]**o+(-0.928207054947)*x[0]+(0.864749399199)*x[1]**o+(0.864358390848)*x[1]+(-0.0255569869984)*x[2]**o+(0.0268536872793)*x[2]
        arg[1]=(0.270542056965)*x[0]**o+(0.0696194435081)*x[0]+(0.401360001043)*x[1]**o+(0.917064528874)*x[1]+(-0.755510144559)*x[2]**o+(0.881244515908)*x[2]
        ref[0]=(1.51616935054)*(1+2.*(dim-1.)/(o+1.))+(-0.0369949768193)*dim
        ref[1]=(-0.083608086551)*(1+2.*(dim-1.)/(o+1.))+(1.86792848829)*dim
      res=integrate(arg)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnBoundary_fromData_FunctionOnBoundary_rank2(self):
      """
      tests integral of rank 2 Data on the FunctionOnBoundary

      assumptions: FunctionOnBoundary(self.domain) exists
                   self.domain supports integral on FunctionOnBoundary
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnBoundary(self.domain)
      w=FunctionOnBoundary(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5),w)
      ref=numpy.zeros((4, 5),numpy.float_)
      if dim==2:
        arg[0,0]=(-0.381843423534)*x[0]**o+(-0.563210012057)*x[0]+(0.730631055205)*x[1]**o+(0.13636238698)*x[1]
        arg[0,1]=(0.306357653338)*x[0]**o+(-0.797201232101)*x[0]+(-0.66333793372)*x[1]**o+(-0.943650360464)*x[1]
        arg[0,2]=(-0.34058649734)*x[0]**o+(-0.27047792232)*x[0]+(0.57156687267)*x[1]**o+(-0.659114355468)*x[1]
        arg[0,3]=(0.73319004289)*x[0]**o+(0.229139540294)*x[0]+(-0.51558765225)*x[1]**o+(0.45052521066)*x[1]
        arg[0,4]=(-0.264117198359)*x[0]**o+(0.807950109095)*x[0]+(-0.290558748289)*x[1]**o+(0.161864708045)*x[1]
        arg[1,0]=(0.988732415297)*x[0]**o+(-0.725047414136)*x[0]+(-0.722337599885)*x[1]**o+(0.739130135256)*x[1]
        arg[1,1]=(-0.315086233906)*x[0]**o+(-0.783177846939)*x[0]+(-0.271931454399)*x[1]**o+(-0.935744613016)*x[1]
        arg[1,2]=(-0.787047622811)*x[0]**o+(0.00746242211892)*x[0]+(-0.539716273578)*x[1]**o+(-0.340449115622)*x[1]
        arg[1,3]=(-0.782830405823)*x[0]**o+(0.842318742811)*x[0]+(-0.118894493096)*x[1]**o+(0.645943277412)*x[1]
        arg[1,4]=(0.0594068367647)*x[0]**o+(-0.184711682975)*x[0]+(-0.0643050731295)*x[1]**o+(-0.981593287261)*x[1]
        arg[2,0]=(0.17170754931)*x[0]**o+(0.582651363494)*x[0]+(0.334422749985)*x[1]**o+(0.852747429144)*x[1]
        arg[2,1]=(0.683967516208)*x[0]**o+(-0.612989314204)*x[0]+(0.845175135723)*x[1]**o+(0.374223395314)*x[1]
        arg[2,2]=(0.588917536035)*x[0]**o+(0.0130713873939)*x[0]+(0.881942872071)*x[1]**o+(0.683941046151)*x[1]
        arg[2,3]=(-0.63861264887)*x[0]**o+(0.453919752415)*x[0]+(0.448731342326)*x[1]**o+(-0.259183783496)*x[1]
        arg[2,4]=(-0.132175638549)*x[0]**o+(0.710071566415)*x[0]+(-0.870119332524)*x[1]**o+(0.523394538403)*x[1]
        arg[3,0]=(0.768036036634)*x[0]**o+(-0.218321673662)*x[0]+(0.50224512768)*x[1]**o+(0.694191572559)*x[1]
        arg[3,1]=(-0.298369075336)*x[0]**o+(-0.276776010079)*x[0]+(-0.659309162004)*x[1]**o+(0.436841182692)*x[1]
        arg[3,2]=(-0.00370049871499)*x[0]**o+(-0.41700259949)*x[0]+(0.726806931339)*x[1]**o+(-0.113165181924)*x[1]
        arg[3,3]=(0.288175924931)*x[0]**o+(-0.489667003521)*x[0]+(-0.531842953651)*x[1]**o+(-0.443517285929)*x[1]
        arg[3,4]=(-0.645574931589)*x[0]**o+(0.464738749544)*x[0]+(-0.538213524368)*x[1]**o+(-0.94443292146)*x[1]
        ref[0,0]=(0.348787631671)*(1+2.*(dim-1.)/(o+1.))+(-0.426847625077)*dim
        ref[0,1]=(-0.356980280382)*(1+2.*(dim-1.)/(o+1.))+(-1.74085159257)*dim
        ref[0,2]=(0.23098037533)*(1+2.*(dim-1.)/(o+1.))+(-0.929592277787)*dim
        ref[0,3]=(0.21760239064)*(1+2.*(dim-1.)/(o+1.))+(0.679664750954)*dim
        ref[0,4]=(-0.554675946648)*(1+2.*(dim-1.)/(o+1.))+(0.96981481714)*dim
        ref[1,0]=(0.266394815412)*(1+2.*(dim-1.)/(o+1.))+(0.0140827211195)*dim
        ref[1,1]=(-0.587017688304)*(1+2.*(dim-1.)/(o+1.))+(-1.71892245996)*dim
        ref[1,2]=(-1.32676389639)*(1+2.*(dim-1.)/(o+1.))+(-0.332986693503)*dim
        ref[1,3]=(-0.901724898919)*(1+2.*(dim-1.)/(o+1.))+(1.48826202022)*dim
        ref[1,4]=(-0.00489823636486)*(1+2.*(dim-1.)/(o+1.))+(-1.16630497024)*dim
        ref[2,0]=(0.506130299295)*(1+2.*(dim-1.)/(o+1.))+(1.43539879264)*dim
        ref[2,1]=(1.52914265193)*(1+2.*(dim-1.)/(o+1.))+(-0.238765918889)*dim
        ref[2,2]=(1.47086040811)*(1+2.*(dim-1.)/(o+1.))+(0.697012433545)*dim
        ref[2,3]=(-0.189881306544)*(1+2.*(dim-1.)/(o+1.))+(0.194735968919)*dim
        ref[2,4]=(-1.00229497107)*(1+2.*(dim-1.)/(o+1.))+(1.23346610482)*dim
        ref[3,0]=(1.27028116431)*(1+2.*(dim-1.)/(o+1.))+(0.475869898897)*dim
        ref[3,1]=(-0.957678237339)*(1+2.*(dim-1.)/(o+1.))+(0.160065172613)*dim
        ref[3,2]=(0.723106432624)*(1+2.*(dim-1.)/(o+1.))+(-0.530167781414)*dim
        ref[3,3]=(-0.243667028719)*(1+2.*(dim-1.)/(o+1.))+(-0.93318428945)*dim
        ref[3,4]=(-1.18378845596)*(1+2.*(dim-1.)/(o+1.))+(-0.479694171917)*dim
      else:
        arg[0,0]=(0.751222317885)*x[0]**o+(0.953666267123)*x[0]+(-0.707417048158)*x[1]**o+(0.310686543483)*x[1]+(0.326267797641)*x[2]**o+(-0.722772500437)*x[2]
        arg[0,1]=(-0.038709086264)*x[0]**o+(-0.483307819428)*x[0]+(0.611120835303)*x[1]**o+(0.885935954604)*x[1]+(0.630199910046)*x[2]**o+(-0.101764126615)*x[2]
        arg[0,2]=(0.890606731439)*x[0]**o+(-0.0555491679283)*x[0]+(0.334835177686)*x[1]**o+(0.999030888839)*x[1]+(-0.891653525154)*x[2]**o+(-0.14056280616)*x[2]
        arg[0,3]=(-0.458305223634)*x[0]**o+(0.679075077126)*x[0]+(0.1176620176)*x[1]**o+(0.685223087684)*x[1]+(-0.952262623464)*x[2]**o+(-0.905568306628)*x[2]
        arg[0,4]=(0.623179050934)*x[0]**o+(-0.0150847487218)*x[0]+(0.0823495403769)*x[1]**o+(0.48303169009)*x[1]+(0.610276253905)*x[2]**o+(0.353104308072)*x[2]
        arg[1,0]=(-0.709185342895)*x[0]**o+(0.789980598884)*x[0]+(0.655538102878)*x[1]**o+(0.713913568814)*x[1]+(0.222131732773)*x[2]**o+(0.712644729381)*x[2]
        arg[1,1]=(-0.683646421822)*x[0]**o+(-0.943691516528)*x[0]+(-0.134261002557)*x[1]**o+(0.28225941018)*x[1]+(0.906235993779)*x[2]**o+(-0.819214603609)*x[2]
        arg[1,2]=(0.713781993938)*x[0]**o+(-0.189551367619)*x[0]+(-0.021310841804)*x[1]**o+(-0.873466428012)*x[1]+(0.248337664546)*x[2]**o+(0.911824874935)*x[2]
        arg[1,3]=(-0.393142772637)*x[0]**o+(0.773501237865)*x[0]+(0.759039429922)*x[1]**o+(0.67036876499)*x[1]+(-0.921411979023)*x[2]**o+(-0.649371534276)*x[2]
        arg[1,4]=(0.526708082723)*x[0]**o+(-0.522449586718)*x[0]+(0.951176548558)*x[1]**o+(0.604602039898)*x[1]+(0.345888192325)*x[2]**o+(-0.973729954334)*x[2]
        arg[2,0]=(-0.139931215293)*x[0]**o+(-0.996072915638)*x[0]+(-0.0451731431264)*x[1]**o+(0.433694224874)*x[1]+(-0.955189760302)*x[2]**o+(-0.68361667242)*x[2]
        arg[2,1]=(0.139626552661)*x[0]**o+(-0.91217468751)*x[0]+(-0.854252204911)*x[1]**o+(0.417539913464)*x[1]+(0.557096050124)*x[2]**o+(-0.195272359868)*x[2]
        arg[2,2]=(0.960352725049)*x[0]**o+(0.98274500341)*x[0]+(0.967782565025)*x[1]**o+(0.566011353381)*x[1]+(0.583885094583)*x[2]**o+(-0.943173847936)*x[2]
        arg[2,3]=(0.959388333494)*x[0]**o+(0.531908797155)*x[0]+(-0.62531126203)*x[1]**o+(-0.288763293646)*x[1]+(-0.823398780526)*x[2]**o+(-0.164348871248)*x[2]
        arg[2,4]=(0.73397702179)*x[0]**o+(0.407844592399)*x[0]+(-0.835769355727)*x[1]**o+(0.742360626394)*x[1]+(0.31130186529)*x[2]**o+(0.114221168872)*x[2]
        arg[3,0]=(-0.42278940444)*x[0]**o+(-0.513158064442)*x[0]+(-0.133233894233)*x[1]**o+(0.701381913887)*x[1]+(0.283776333553)*x[2]**o+(-0.805096661642)*x[2]
        arg[3,1]=(-0.417338508269)*x[0]**o+(0.535961900783)*x[0]+(0.0836386907734)*x[1]**o+(0.449680928704)*x[1]+(0.998998322868)*x[2]**o+(0.983057825698)*x[2]
        arg[3,2]=(0.24102382276)*x[0]**o+(0.0291470157398)*x[0]+(-0.539542451698)*x[1]**o+(0.281017043631)*x[1]+(-0.464652133499)*x[2]**o+(0.652828720185)*x[2]
        arg[3,3]=(-0.482577920013)*x[0]**o+(0.931756756394)*x[0]+(0.96648206338)*x[1]**o+(0.73941459549)*x[1]+(0.136743015429)*x[2]**o+(0.41477914722)*x[2]
        arg[3,4]=(0.879881885972)*x[0]**o+(0.208970079585)*x[0]+(-0.200079958414)*x[1]**o+(-0.0291238772241)*x[1]+(-0.333569505857)*x[2]**o+(0.846835729875)*x[2]
        ref[0,0]=(0.370073067368)*(1+2.*(dim-1.)/(o+1.))+(0.541580310169)*dim
        ref[0,1]=(1.20261165909)*(1+2.*(dim-1.)/(o+1.))+(0.300864008561)*dim
        ref[0,2]=(0.333788383971)*(1+2.*(dim-1.)/(o+1.))+(0.80291891475)*dim
        ref[0,3]=(-1.2929058295)*(1+2.*(dim-1.)/(o+1.))+(0.458729858183)*dim
        ref[0,4]=(1.31580484522)*(1+2.*(dim-1.)/(o+1.))+(0.82105124944)*dim
        ref[1,0]=(0.168484492756)*(1+2.*(dim-1.)/(o+1.))+(2.21653889708)*dim
        ref[1,1]=(0.0883285694004)*(1+2.*(dim-1.)/(o+1.))+(-1.48064670996)*dim
        ref[1,2]=(0.940808816681)*(1+2.*(dim-1.)/(o+1.))+(-0.151192920696)*dim
        ref[1,3]=(-0.555515321739)*(1+2.*(dim-1.)/(o+1.))+(0.79449846858)*dim
        ref[1,4]=(1.82377282361)*(1+2.*(dim-1.)/(o+1.))+(-0.891577501154)*dim
        ref[2,0]=(-1.14029411872)*(1+2.*(dim-1.)/(o+1.))+(-1.24599536318)*dim
        ref[2,1]=(-0.157529602127)*(1+2.*(dim-1.)/(o+1.))+(-0.689907133913)*dim
        ref[2,2]=(2.51202038466)*(1+2.*(dim-1.)/(o+1.))+(0.605582508855)*dim
        ref[2,3]=(-0.489321709063)*(1+2.*(dim-1.)/(o+1.))+(0.0787966322607)*dim
        ref[2,4]=(0.209509531353)*(1+2.*(dim-1.)/(o+1.))+(1.26442638766)*dim
        ref[3,0]=(-0.27224696512)*(1+2.*(dim-1.)/(o+1.))+(-0.616872812198)*dim
        ref[3,1]=(0.665298505372)*(1+2.*(dim-1.)/(o+1.))+(1.96870065519)*dim
        ref[3,2]=(-0.763170762437)*(1+2.*(dim-1.)/(o+1.))+(0.962992779555)*dim
        ref[3,3]=(0.620647158797)*(1+2.*(dim-1.)/(o+1.))+(2.0859504991)*dim
        ref[3,4]=(0.346232421701)*(1+2.*(dim-1.)/(o+1.))+(1.02668193224)*dim
      res=integrate(arg)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnBoundary_fromData_FunctionOnBoundary_rank3(self):
      """
      tests integral of rank 3 Data on the FunctionOnBoundary

      assumptions: FunctionOnBoundary(self.domain) exists
                   self.domain supports integral on FunctionOnBoundary
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnBoundary(self.domain)
      w=FunctionOnBoundary(self.domain)
      x=w.getX()
      arg=Data(0,(6, 2, 2),w)
      ref=numpy.zeros((6, 2, 2),numpy.float_)
      if dim==2:
        arg[0,0,0]=(0.201115186772)*x[0]**o+(0.324758103748)*x[0]+(0.285065583242)*x[1]**o+(0.0264985832782)*x[1]
        arg[0,0,1]=(-0.443023003973)*x[0]**o+(-0.194732927607)*x[0]+(0.64414908519)*x[1]**o+(-0.248261857779)*x[1]
        arg[0,1,0]=(0.587534172195)*x[0]**o+(0.021088951369)*x[0]+(-0.986195557459)*x[1]**o+(0.595659005312)*x[1]
        arg[0,1,1]=(-0.916961145134)*x[0]**o+(-0.766484469605)*x[0]+(-0.281345964683)*x[1]**o+(-0.399113856826)*x[1]
        arg[1,0,0]=(0.175943424949)*x[0]**o+(0.344853386001)*x[0]+(0.175474681848)*x[1]**o+(-0.0775102874697)*x[1]
        arg[1,0,1]=(-0.874074467007)*x[0]**o+(0.903506914387)*x[0]+(0.177699372538)*x[1]**o+(0.969818154968)*x[1]
        arg[1,1,0]=(0.765751472387)*x[0]**o+(-0.0288724203687)*x[0]+(0.655676828008)*x[1]**o+(0.22281307247)*x[1]
        arg[1,1,1]=(0.871009077152)*x[0]**o+(-0.20052353101)*x[0]+(-0.889496767621)*x[1]**o+(0.8458762887)*x[1]
        arg[2,0,0]=(-0.15377513542)*x[0]**o+(0.963714892768)*x[0]+(0.942514815983)*x[1]**o+(0.846111018838)*x[1]
        arg[2,0,1]=(0.000727086938663)*x[0]**o+(0.243136987492)*x[0]+(-0.308674899886)*x[1]**o+(-0.963435664436)*x[1]
        arg[2,1,0]=(0.780912245051)*x[0]**o+(0.227139041778)*x[0]+(-0.336203258384)*x[1]**o+(0.092568284679)*x[1]
        arg[2,1,1]=(0.0464862681168)*x[0]**o+(0.284493375176)*x[0]+(-0.811073509077)*x[1]**o+(0.54280235419)*x[1]
        arg[3,0,0]=(0.605642304612)*x[0]**o+(-0.103442921567)*x[0]+(0.106253324829)*x[1]**o+(0.562480007904)*x[1]
        arg[3,0,1]=(0.497668538008)*x[0]**o+(-0.859679982844)*x[0]+(-0.255499355194)*x[1]**o+(0.0725919239243)*x[1]
        arg[3,1,0]=(0.641826893201)*x[0]**o+(0.889514581887)*x[0]+(-0.211781857836)*x[1]**o+(0.45900873193)*x[1]
        arg[3,1,1]=(-0.29349600565)*x[0]**o+(-0.627542961817)*x[0]+(0.313009642686)*x[1]**o+(0.338028134809)*x[1]
        arg[4,0,0]=(-0.641915241397)*x[0]**o+(-0.682547311604)*x[0]+(-0.106472077451)*x[1]**o+(0.103402239052)*x[1]
        arg[4,0,1]=(-0.822562520978)*x[0]**o+(-0.430539530156)*x[0]+(-0.0225528898923)*x[1]**o+(-0.0908986665674)*x[1]
        arg[4,1,0]=(-0.456160645978)*x[0]**o+(-0.264592336242)*x[0]+(-0.672702978868)*x[1]**o+(-0.893197955639)*x[1]
        arg[4,1,1]=(-0.714689169676)*x[0]**o+(0.513612105215)*x[0]+(-0.3093340377)*x[1]**o+(-0.34467035156)*x[1]
        arg[5,0,0]=(-0.993761664154)*x[0]**o+(0.375953817044)*x[0]+(0.0463240503072)*x[1]**o+(-0.595646829466)*x[1]
        arg[5,0,1]=(0.453145186545)*x[0]**o+(0.331353840672)*x[0]+(-0.908533078632)*x[1]**o+(0.755175839674)*x[1]
        arg[5,1,0]=(-0.736874540101)*x[0]**o+(-0.867437170846)*x[0]+(-0.472395301622)*x[1]**o+(0.78289019661)*x[1]
        arg[5,1,1]=(-0.900966156016)*x[0]**o+(-0.160363876891)*x[0]+(-0.0969222200475)*x[1]**o+(0.0527983470624)*x[1]
        ref[0,0,0]=(0.486180770014)*(1+2.*(dim-1.)/(o+1.))+(0.351256687026)*dim
        ref[0,0,1]=(0.201126081217)*(1+2.*(dim-1.)/(o+1.))+(-0.442994785386)*dim
        ref[0,1,0]=(-0.398661385264)*(1+2.*(dim-1.)/(o+1.))+(0.616747956681)*dim
        ref[0,1,1]=(-1.19830710982)*(1+2.*(dim-1.)/(o+1.))+(-1.16559832643)*dim
        ref[1,0,0]=(0.351418106797)*(1+2.*(dim-1.)/(o+1.))+(0.267343098531)*dim
        ref[1,0,1]=(-0.696375094469)*(1+2.*(dim-1.)/(o+1.))+(1.87332506936)*dim
        ref[1,1,0]=(1.42142830039)*(1+2.*(dim-1.)/(o+1.))+(0.193940652101)*dim
        ref[1,1,1]=(-0.0184876904693)*(1+2.*(dim-1.)/(o+1.))+(0.64535275769)*dim
        ref[2,0,0]=(0.788739680562)*(1+2.*(dim-1.)/(o+1.))+(1.80982591161)*dim
        ref[2,0,1]=(-0.307947812947)*(1+2.*(dim-1.)/(o+1.))+(-0.720298676944)*dim
        ref[2,1,0]=(0.444708986667)*(1+2.*(dim-1.)/(o+1.))+(0.319707326457)*dim
        ref[2,1,1]=(-0.76458724096)*(1+2.*(dim-1.)/(o+1.))+(0.827295729366)*dim
        ref[3,0,0]=(0.711895629441)*(1+2.*(dim-1.)/(o+1.))+(0.459037086337)*dim
        ref[3,0,1]=(0.242169182814)*(1+2.*(dim-1.)/(o+1.))+(-0.78708805892)*dim
        ref[3,1,0]=(0.430045035365)*(1+2.*(dim-1.)/(o+1.))+(1.34852331382)*dim
        ref[3,1,1]=(0.019513637036)*(1+2.*(dim-1.)/(o+1.))+(-0.289514827008)*dim
        ref[4,0,0]=(-0.748387318848)*(1+2.*(dim-1.)/(o+1.))+(-0.579145072551)*dim
        ref[4,0,1]=(-0.845115410871)*(1+2.*(dim-1.)/(o+1.))+(-0.521438196723)*dim
        ref[4,1,0]=(-1.12886362485)*(1+2.*(dim-1.)/(o+1.))+(-1.15779029188)*dim
        ref[4,1,1]=(-1.02402320738)*(1+2.*(dim-1.)/(o+1.))+(0.168941753655)*dim
        ref[5,0,0]=(-0.947437613847)*(1+2.*(dim-1.)/(o+1.))+(-0.219693012422)*dim
        ref[5,0,1]=(-0.455387892087)*(1+2.*(dim-1.)/(o+1.))+(1.08652968035)*dim
        ref[5,1,0]=(-1.20926984172)*(1+2.*(dim-1.)/(o+1.))+(-0.0845469742352)*dim
        ref[5,1,1]=(-0.997888376064)*(1+2.*(dim-1.)/(o+1.))+(-0.107565529828)*dim
      else:
        arg[0,0,0]=(-0.760764230601)*x[0]**o+(-0.955215307904)*x[0]+(0.376251560231)*x[1]**o+(0.147849416477)*x[1]+(0.627037741745)*x[2]**o+(-0.93996244415)*x[2]
        arg[0,0,1]=(0.904449671196)*x[0]**o+(-0.297804849063)*x[0]+(0.773059038176)*x[1]**o+(-0.836104534497)*x[1]+(-0.150040994258)*x[2]**o+(0.837988726131)*x[2]
        arg[0,1,0]=(0.354255190058)*x[0]**o+(0.463977999833)*x[0]+(0.218855388406)*x[1]**o+(-0.163086423536)*x[1]+(0.653790071045)*x[2]**o+(-0.457721486689)*x[2]
        arg[0,1,1]=(0.357317871424)*x[0]**o+(-0.752862019242)*x[0]+(0.674187130836)*x[1]**o+(-0.983503125785)*x[1]+(-0.776153051078)*x[2]**o+(0.390700377058)*x[2]
        arg[1,0,0]=(-0.465423805042)*x[0]**o+(0.629855541232)*x[0]+(-0.405300083887)*x[1]**o+(-0.211254598072)*x[1]+(0.502581881406)*x[2]**o+(-0.853779249469)*x[2]
        arg[1,0,1]=(0.716025450344)*x[0]**o+(0.876021396077)*x[0]+(-0.530324190586)*x[1]**o+(0.700207449985)*x[1]+(-0.788565669024)*x[2]**o+(0.963118484198)*x[2]
        arg[1,1,0]=(0.813943574582)*x[0]**o+(-0.938778607938)*x[0]+(-0.451619613547)*x[1]**o+(-0.916990884105)*x[1]+(-0.671730738367)*x[2]**o+(0.109664560689)*x[2]
        arg[1,1,1]=(-0.557598711518)*x[0]**o+(-0.541691706474)*x[0]+(0.500874328683)*x[1]**o+(-0.959653364589)*x[1]+(0.572057415183)*x[2]**o+(-0.842306660222)*x[2]
        arg[2,0,0]=(0.0470537168687)*x[0]**o+(-0.151521222488)*x[0]+(0.168279496456)*x[1]**o+(0.869948894317)*x[1]+(-0.554196665086)*x[2]**o+(0.444189090676)*x[2]
        arg[2,0,1]=(-0.662480394662)*x[0]**o+(0.506778773041)*x[0]+(0.664368833417)*x[1]**o+(-0.76728005759)*x[1]+(0.729668063743)*x[2]**o+(-0.476499535761)*x[2]
        arg[2,1,0]=(0.40405640406)*x[0]**o+(0.537691275434)*x[0]+(0.675845932799)*x[1]**o+(-0.663037107454)*x[1]+(-0.947246204559)*x[2]**o+(-0.210977364198)*x[2]
        arg[2,1,1]=(-0.460999541334)*x[0]**o+(-0.403551870206)*x[0]+(-0.59751233801)*x[1]**o+(0.389018763902)*x[1]+(-0.916883519483)*x[2]**o+(0.712714107738)*x[2]
        arg[3,0,0]=(-0.547139864054)*x[0]**o+(0.91774354334)*x[0]+(-0.953248927323)*x[1]**o+(0.693673610785)*x[1]+(-0.474082903895)*x[2]**o+(0.187174904886)*x[2]
        arg[3,0,1]=(0.538597533339)*x[0]**o+(0.903368885629)*x[0]+(-0.548058780044)*x[1]**o+(-0.380406303455)*x[1]+(-0.389244091021)*x[2]**o+(0.137402766164)*x[2]
        arg[3,1,0]=(-0.11691167078)*x[0]**o+(-0.812294829383)*x[0]+(0.592192015269)*x[1]**o+(-0.3278056345)*x[1]+(-0.60462640896)*x[2]**o+(-0.701541632465)*x[2]
        arg[3,1,1]=(-0.196382907034)*x[0]**o+(-0.496234735923)*x[0]+(-0.646502033751)*x[1]**o+(-0.18981379172)*x[1]+(0.627608199527)*x[2]**o+(-0.468659552408)*x[2]
        arg[4,0,0]=(0.56789650708)*x[0]**o+(0.854984079971)*x[0]+(-0.641473205573)*x[1]**o+(-0.791240666887)*x[1]+(0.453562162212)*x[2]**o+(0.849111318036)*x[2]
        arg[4,0,1]=(0.90836228457)*x[0]**o+(-0.666684673376)*x[0]+(0.245928632693)*x[1]**o+(0.313580017685)*x[1]+(0.353126203728)*x[2]**o+(0.467530918033)*x[2]
        arg[4,1,0]=(0.670926585272)*x[0]**o+(0.768555336283)*x[0]+(-0.170656824353)*x[1]**o+(-0.346425800834)*x[1]+(-0.969641886683)*x[2]**o+(0.775335418066)*x[2]
        arg[4,1,1]=(-0.152500967035)*x[0]**o+(0.575400101502)*x[0]+(0.901331752685)*x[1]**o+(-0.672576642993)*x[1]+(0.803176375377)*x[2]**o+(0.202785337179)*x[2]
        arg[5,0,0]=(-0.798851736802)*x[0]**o+(-0.221276048656)*x[0]+(0.773295190339)*x[1]**o+(0.0903112854292)*x[1]+(0.884961364642)*x[2]**o+(0.326429585537)*x[2]
        arg[5,0,1]=(0.482124059007)*x[0]**o+(-0.39227677557)*x[0]+(-0.23074922677)*x[1]**o+(0.986783507045)*x[1]+(0.591759402442)*x[2]**o+(-0.680278617308)*x[2]
        arg[5,1,0]=(-0.857456972644)*x[0]**o+(-0.200661025977)*x[0]+(-0.850031395341)*x[1]**o+(0.830863037447)*x[1]+(-0.323765625026)*x[2]**o+(-0.969463605839)*x[2]
        arg[5,1,1]=(-0.582538216697)*x[0]**o+(-0.495303456972)*x[0]+(0.0134139524699)*x[1]**o+(-0.750144603651)*x[1]+(-0.632380446311)*x[2]**o+(0.38782755017)*x[2]
        ref[0,0,0]=(0.242525071375)*(1+2.*(dim-1.)/(o+1.))+(-1.74732833558)*dim
        ref[0,0,1]=(1.52746771511)*(1+2.*(dim-1.)/(o+1.))+(-0.295920657428)*dim
        ref[0,1,0]=(1.22690064951)*(1+2.*(dim-1.)/(o+1.))+(-0.156829910392)*dim
        ref[0,1,1]=(0.255351951182)*(1+2.*(dim-1.)/(o+1.))+(-1.34566476797)*dim
        ref[1,0,0]=(-0.368142007523)*(1+2.*(dim-1.)/(o+1.))+(-0.43517830631)*dim
        ref[1,0,1]=(-0.602864409266)*(1+2.*(dim-1.)/(o+1.))+(2.53934733026)*dim
        ref[1,1,0]=(-0.309406777332)*(1+2.*(dim-1.)/(o+1.))+(-1.74610493135)*dim
        ref[1,1,1]=(0.515333032348)*(1+2.*(dim-1.)/(o+1.))+(-2.34365173129)*dim
        ref[2,0,0]=(-0.338863451761)*(1+2.*(dim-1.)/(o+1.))+(1.16261676251)*dim
        ref[2,0,1]=(0.731556502498)*(1+2.*(dim-1.)/(o+1.))+(-0.73700082031)*dim
        ref[2,1,0]=(0.132656132299)*(1+2.*(dim-1.)/(o+1.))+(-0.336323196217)*dim
        ref[2,1,1]=(-1.97539539883)*(1+2.*(dim-1.)/(o+1.))+(0.698181001434)*dim
        ref[3,0,0]=(-1.97447169527)*(1+2.*(dim-1.)/(o+1.))+(1.79859205901)*dim
        ref[3,0,1]=(-0.398705337727)*(1+2.*(dim-1.)/(o+1.))+(0.660365348338)*dim
        ref[3,1,0]=(-0.129346064471)*(1+2.*(dim-1.)/(o+1.))+(-1.84164209635)*dim
        ref[3,1,1]=(-0.215276741258)*(1+2.*(dim-1.)/(o+1.))+(-1.15470808005)*dim
        ref[4,0,0]=(0.379985463719)*(1+2.*(dim-1.)/(o+1.))+(0.91285473112)*dim
        ref[4,0,1]=(1.50741712099)*(1+2.*(dim-1.)/(o+1.))+(0.114426262342)*dim
        ref[4,1,0]=(-0.469372125764)*(1+2.*(dim-1.)/(o+1.))+(1.19746495352)*dim
        ref[4,1,1]=(1.55200716103)*(1+2.*(dim-1.)/(o+1.))+(0.105608795688)*dim
        ref[5,0,0]=(0.859404818179)*(1+2.*(dim-1.)/(o+1.))+(0.19546482231)*dim
        ref[5,0,1]=(0.843134234679)*(1+2.*(dim-1.)/(o+1.))+(-0.085771885833)*dim
        ref[5,1,0]=(-2.03125399301)*(1+2.*(dim-1.)/(o+1.))+(-0.339261594369)*dim
        ref[5,1,1]=(-1.20150471054)*(1+2.*(dim-1.)/(o+1.))+(-0.857620510454)*dim
      res=integrate(arg)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnBoundary_fromData_FunctionOnBoundary_rank4(self):
      """
      tests integral of rank 4 Data on the FunctionOnBoundary

      assumptions: FunctionOnBoundary(self.domain) exists
                   self.domain supports integral on FunctionOnBoundary
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnBoundary(self.domain)
      w=FunctionOnBoundary(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5, 3, 2),w)
      ref=numpy.zeros((4, 5, 3, 2),numpy.float_)
      if dim==2:
        arg[0,0,0,0]=(-0.161068279134)*x[0]**o+(-0.20016693624)*x[0]+(0.360967926263)*x[1]**o+(0.519774730324)*x[1]
        arg[0,0,0,1]=(-0.470257945181)*x[0]**o+(0.0652334206708)*x[0]+(0.652139991357)*x[1]**o+(-0.592482756578)*x[1]
        arg[0,0,1,0]=(0.143126570256)*x[0]**o+(0.683474249212)*x[0]+(0.839611756722)*x[1]**o+(-0.436612603526)*x[1]
        arg[0,0,1,1]=(-0.82328367976)*x[0]**o+(-0.848145278113)*x[0]+(-0.649046694126)*x[1]**o+(0.7686615034)*x[1]
        arg[0,0,2,0]=(-0.89676527056)*x[0]**o+(-0.466316806084)*x[0]+(-0.454908744166)*x[1]**o+(0.425524454897)*x[1]
        arg[0,0,2,1]=(-0.720978656424)*x[0]**o+(-0.655449460498)*x[0]+(0.466383340682)*x[1]**o+(-0.609615749634)*x[1]
        arg[0,1,0,0]=(-0.358670380794)*x[0]**o+(-0.764939003508)*x[0]+(-0.550898156906)*x[1]**o+(0.803150103728)*x[1]
        arg[0,1,0,1]=(-0.652241076457)*x[0]**o+(-0.614033274928)*x[0]+(-0.308770011731)*x[1]**o+(0.925847136965)*x[1]
        arg[0,1,1,0]=(0.710539026622)*x[0]**o+(-0.345587804078)*x[0]+(-0.170026226995)*x[1]**o+(0.340557574804)*x[1]
        arg[0,1,1,1]=(0.348069554895)*x[0]**o+(-0.633629515001)*x[0]+(-0.199233786202)*x[1]**o+(0.500291488845)*x[1]
        arg[0,1,2,0]=(0.262179751197)*x[0]**o+(-0.55460029497)*x[0]+(-0.354335283745)*x[1]**o+(0.301130103531)*x[1]
        arg[0,1,2,1]=(0.212709964525)*x[0]**o+(0.205907518129)*x[0]+(0.167040570615)*x[1]**o+(-0.0684155270612)*x[1]
        arg[0,2,0,0]=(-0.272141616042)*x[0]**o+(0.187045318075)*x[0]+(-0.364974531949)*x[1]**o+(-0.969702579161)*x[1]
        arg[0,2,0,1]=(-0.997155483171)*x[0]**o+(0.864439328971)*x[0]+(0.887907754657)*x[1]**o+(-0.323021835147)*x[1]
        arg[0,2,1,0]=(-0.544228085563)*x[0]**o+(-0.56752289714)*x[0]+(-0.389234395916)*x[1]**o+(-0.969287386144)*x[1]
        arg[0,2,1,1]=(-0.786136450866)*x[0]**o+(0.589465339989)*x[0]+(-0.411595778776)*x[1]**o+(0.726975706536)*x[1]
        arg[0,2,2,0]=(-0.788299350517)*x[0]**o+(0.697037609691)*x[0]+(-0.391342918209)*x[1]**o+(0.432368662365)*x[1]
        arg[0,2,2,1]=(-0.390536554506)*x[0]**o+(0.547528359725)*x[0]+(0.58957966188)*x[1]**o+(0.146918361066)*x[1]
        arg[0,3,0,0]=(0.176059579799)*x[0]**o+(-0.774041865433)*x[0]+(0.52369869825)*x[1]**o+(-0.812474996262)*x[1]
        arg[0,3,0,1]=(-0.938727595513)*x[0]**o+(0.734277246158)*x[0]+(-0.500452658276)*x[1]**o+(0.591562924701)*x[1]
        arg[0,3,1,0]=(-0.109970490351)*x[0]**o+(-0.463382333915)*x[0]+(0.430588389365)*x[1]**o+(-0.174978181104)*x[1]
        arg[0,3,1,1]=(-0.564486046162)*x[0]**o+(0.325599920584)*x[0]+(-0.177334934858)*x[1]**o+(-0.789210879368)*x[1]
        arg[0,3,2,0]=(-0.409769653157)*x[0]**o+(-0.674668709075)*x[0]+(0.977465383205)*x[1]**o+(-0.178249094926)*x[1]
        arg[0,3,2,1]=(0.112394584736)*x[0]**o+(-0.229767891006)*x[0]+(0.952422132907)*x[1]**o+(-0.133952947363)*x[1]
        arg[0,4,0,0]=(0.927747625444)*x[0]**o+(-0.147390883234)*x[0]+(-0.0523168758823)*x[1]**o+(0.698643952882)*x[1]
        arg[0,4,0,1]=(0.986627759385)*x[0]**o+(-0.0486375625834)*x[0]+(0.27598075052)*x[1]**o+(0.970272004318)*x[1]
        arg[0,4,1,0]=(-0.461643441984)*x[0]**o+(-0.815261492927)*x[0]+(-0.0806771439228)*x[1]**o+(0.877302861263)*x[1]
        arg[0,4,1,1]=(-0.108403920958)*x[0]**o+(0.651566419087)*x[0]+(-0.629972296369)*x[1]**o+(-0.235413582155)*x[1]
        arg[0,4,2,0]=(-0.177527213798)*x[0]**o+(0.987202381273)*x[0]+(-0.15522298966)*x[1]**o+(0.234268561378)*x[1]
        arg[0,4,2,1]=(0.521954293269)*x[0]**o+(-0.776898814469)*x[0]+(0.783085515988)*x[1]**o+(-0.469921197982)*x[1]
        arg[1,0,0,0]=(0.858157489751)*x[0]**o+(0.444181035326)*x[0]+(0.3426806552)*x[1]**o+(-0.219249318925)*x[1]
        arg[1,0,0,1]=(0.405857394564)*x[0]**o+(0.178361294933)*x[0]+(0.0879917927852)*x[1]**o+(-0.0689581267347)*x[1]
        arg[1,0,1,0]=(0.63175565694)*x[0]**o+(0.0810462363705)*x[0]+(0.0350399072162)*x[1]**o+(-0.269906961338)*x[1]
        arg[1,0,1,1]=(-0.273973794035)*x[0]**o+(0.824519533391)*x[0]+(0.596785621044)*x[1]**o+(-0.243043370261)*x[1]
        arg[1,0,2,0]=(-0.959246118913)*x[0]**o+(0.3570024829)*x[0]+(-0.260098162669)*x[1]**o+(0.341044029457)*x[1]
        arg[1,0,2,1]=(0.966671029776)*x[0]**o+(0.598898116395)*x[0]+(0.392360541053)*x[1]**o+(-0.322723646751)*x[1]
        arg[1,1,0,0]=(-0.658365963559)*x[0]**o+(-0.674107595463)*x[0]+(-0.0103911551403)*x[1]**o+(0.858879803218)*x[1]
        arg[1,1,0,1]=(0.340459206102)*x[0]**o+(-0.681434505015)*x[0]+(0.917412924129)*x[1]**o+(0.703699154381)*x[1]
        arg[1,1,1,0]=(0.0473840209756)*x[0]**o+(-0.787070458106)*x[0]+(0.11979244907)*x[1]**o+(0.541238063489)*x[1]
        arg[1,1,1,1]=(0.184943768213)*x[0]**o+(-0.942961758488)*x[0]+(-0.844785429901)*x[1]**o+(-0.0109779269179)*x[1]
        arg[1,1,2,0]=(0.921469748172)*x[0]**o+(-0.122684636542)*x[0]+(0.149552532716)*x[1]**o+(-0.0102623511735)*x[1]
        arg[1,1,2,1]=(-0.720466710917)*x[0]**o+(-0.755841224189)*x[0]+(0.372635950892)*x[1]**o+(0.271604227274)*x[1]
        arg[1,2,0,0]=(-0.548557725319)*x[0]**o+(0.745340396995)*x[0]+(0.461556064961)*x[1]**o+(0.696460891668)*x[1]
        arg[1,2,0,1]=(-0.738939238644)*x[0]**o+(0.344829756443)*x[0]+(0.272341172248)*x[1]**o+(0.768063359753)*x[1]
        arg[1,2,1,0]=(-0.213161389496)*x[0]**o+(0.458730188555)*x[0]+(-0.783049463147)*x[1]**o+(0.785273989074)*x[1]
        arg[1,2,1,1]=(0.189092239774)*x[0]**o+(-0.0649917239139)*x[0]+(0.0171281685468)*x[1]**o+(0.397065812267)*x[1]
        arg[1,2,2,0]=(-0.809663598067)*x[0]**o+(0.587214021528)*x[0]+(0.912814601228)*x[1]**o+(0.860792008172)*x[1]
        arg[1,2,2,1]=(0.487927123118)*x[0]**o+(-0.760831928598)*x[0]+(-0.770486346056)*x[1]**o+(-0.715077841171)*x[1]
        arg[1,3,0,0]=(0.0202747853647)*x[0]**o+(0.0666642857036)*x[0]+(-0.717756666598)*x[1]**o+(-0.481009806467)*x[1]
        arg[1,3,0,1]=(-0.251164550146)*x[0]**o+(0.0519935051437)*x[0]+(-0.703400138413)*x[1]**o+(-0.990329903234)*x[1]
        arg[1,3,1,0]=(0.970793375169)*x[0]**o+(-0.365675269998)*x[0]+(0.77706784549)*x[1]**o+(-0.237785293481)*x[1]
        arg[1,3,1,1]=(-0.162786953006)*x[0]**o+(0.985402077919)*x[0]+(0.739483663725)*x[1]**o+(0.559554321882)*x[1]
        arg[1,3,2,0]=(0.184701463663)*x[0]**o+(0.946173117295)*x[0]+(-0.451420616712)*x[1]**o+(0.244506578089)*x[1]
        arg[1,3,2,1]=(0.894307549357)*x[0]**o+(0.474983970689)*x[0]+(0.626273940678)*x[1]**o+(0.107985157225)*x[1]
        arg[1,4,0,0]=(-0.180994231135)*x[0]**o+(-0.899808624585)*x[0]+(0.395935138342)*x[1]**o+(-0.387788497342)*x[1]
        arg[1,4,0,1]=(0.0713358452056)*x[0]**o+(0.505276043139)*x[0]+(-0.635565006093)*x[1]**o+(0.956951408169)*x[1]
        arg[1,4,1,0]=(0.355608577722)*x[0]**o+(0.58482326687)*x[0]+(0.654137690443)*x[1]**o+(-0.849016362888)*x[1]
        arg[1,4,1,1]=(-0.275966957458)*x[0]**o+(0.856095068377)*x[0]+(-0.760077050664)*x[1]**o+(-0.904159432699)*x[1]
        arg[1,4,2,0]=(-0.0637454805096)*x[0]**o+(-0.833057387633)*x[0]+(0.279008913569)*x[1]**o+(-0.093356790193)*x[1]
        arg[1,4,2,1]=(0.165869666472)*x[0]**o+(-0.4554345089)*x[0]+(0.697149094381)*x[1]**o+(0.244460087482)*x[1]
        arg[2,0,0,0]=(0.388188511348)*x[0]**o+(-0.00552398040882)*x[0]+(0.0679336322536)*x[1]**o+(-0.281132460216)*x[1]
        arg[2,0,0,1]=(0.968019690109)*x[0]**o+(0.525335591339)*x[0]+(0.660069087546)*x[1]**o+(-0.795388487336)*x[1]
        arg[2,0,1,0]=(-0.444010450901)*x[0]**o+(-0.498771184141)*x[0]+(-0.394999176962)*x[1]**o+(-0.579230388911)*x[1]
        arg[2,0,1,1]=(-0.96211072138)*x[0]**o+(-0.0977519108256)*x[0]+(0.206179333903)*x[1]**o+(-0.206155312014)*x[1]
        arg[2,0,2,0]=(0.131307090845)*x[0]**o+(0.179396626286)*x[0]+(-0.436118623166)*x[1]**o+(0.829727875456)*x[1]
        arg[2,0,2,1]=(-0.206336435516)*x[0]**o+(0.175744210813)*x[0]+(0.818204199205)*x[1]**o+(0.433292546177)*x[1]
        arg[2,1,0,0]=(0.920189410059)*x[0]**o+(-0.629286762249)*x[0]+(-0.14444305096)*x[1]**o+(0.564022686795)*x[1]
        arg[2,1,0,1]=(0.43550707444)*x[0]**o+(-0.0203691253164)*x[0]+(0.632472807002)*x[1]**o+(0.722390213081)*x[1]
        arg[2,1,1,0]=(0.827084071535)*x[0]**o+(0.864139638834)*x[0]+(0.795893591186)*x[1]**o+(0.0492667773115)*x[1]
        arg[2,1,1,1]=(0.863813787478)*x[0]**o+(0.268800194137)*x[0]+(-0.567637801205)*x[1]**o+(0.931221493563)*x[1]
        arg[2,1,2,0]=(-0.613463525024)*x[0]**o+(-0.0688030533166)*x[0]+(-0.786395692457)*x[1]**o+(0.862189706883)*x[1]
        arg[2,1,2,1]=(0.315306814851)*x[0]**o+(0.76879509223)*x[0]+(0.451451342779)*x[1]**o+(-0.478873421268)*x[1]
        arg[2,2,0,0]=(-0.785116047841)*x[0]**o+(0.347614087057)*x[0]+(-0.0106716042015)*x[1]**o+(-0.350898768187)*x[1]
        arg[2,2,0,1]=(-0.987778931187)*x[0]**o+(-0.380379654757)*x[0]+(-0.383604999655)*x[1]**o+(0.022203803886)*x[1]
        arg[2,2,1,0]=(0.508327115069)*x[0]**o+(0.443678958632)*x[0]+(-0.320069211602)*x[1]**o+(-0.778140184171)*x[1]
        arg[2,2,1,1]=(-0.532025577107)*x[0]**o+(-0.0475799386434)*x[0]+(0.650417175266)*x[1]**o+(0.33739661369)*x[1]
        arg[2,2,2,0]=(-0.406412097333)*x[0]**o+(0.0510500610605)*x[0]+(-0.25755890225)*x[1]**o+(-0.350209779392)*x[1]
        arg[2,2,2,1]=(0.401361048956)*x[0]**o+(-0.594793585281)*x[0]+(0.155257431052)*x[1]**o+(-0.674885216524)*x[1]
        arg[2,3,0,0]=(0.352606222236)*x[0]**o+(0.64529574473)*x[0]+(-0.838258091233)*x[1]**o+(0.527911574536)*x[1]
        arg[2,3,0,1]=(-0.62652610251)*x[0]**o+(0.879664434218)*x[0]+(0.730845316192)*x[1]**o+(0.0318659384791)*x[1]
        arg[2,3,1,0]=(-0.311152313179)*x[0]**o+(-0.333154399786)*x[0]+(0.743640958069)*x[1]**o+(0.313394944493)*x[1]
        arg[2,3,1,1]=(-0.728968478378)*x[0]**o+(0.500229380574)*x[0]+(0.756628133572)*x[1]**o+(0.270533146994)*x[1]
        arg[2,3,2,0]=(-0.219775843272)*x[0]**o+(0.882943220401)*x[0]+(0.79341547188)*x[1]**o+(0.503486054551)*x[1]
        arg[2,3,2,1]=(-0.850285149353)*x[0]**o+(0.78820884302)*x[0]+(0.348024963962)*x[1]**o+(0.83743616537)*x[1]
        arg[2,4,0,0]=(0.868649421127)*x[0]**o+(-0.622361223999)*x[0]+(0.836321867516)*x[1]**o+(0.423582994131)*x[1]
        arg[2,4,0,1]=(0.140963355402)*x[0]**o+(0.190496469385)*x[0]+(0.765311231744)*x[1]**o+(0.368696267095)*x[1]
        arg[2,4,1,0]=(-0.680854080076)*x[0]**o+(-0.729641193897)*x[0]+(-0.843347646585)*x[1]**o+(-0.0993036180842)*x[1]
        arg[2,4,1,1]=(-0.130039648356)*x[0]**o+(0.534197300899)*x[0]+(-0.142588861186)*x[1]**o+(-0.308376644249)*x[1]
        arg[2,4,2,0]=(0.331212825202)*x[0]**o+(0.466247840288)*x[0]+(-0.201714210846)*x[1]**o+(0.80521345364)*x[1]
        arg[2,4,2,1]=(0.454299813247)*x[0]**o+(0.646694941459)*x[0]+(-0.929854585458)*x[1]**o+(-0.365252296746)*x[1]
        arg[3,0,0,0]=(-0.623915395721)*x[0]**o+(-0.789827882448)*x[0]+(-0.862568457189)*x[1]**o+(-0.141799958679)*x[1]
        arg[3,0,0,1]=(0.197040281353)*x[0]**o+(0.155785418081)*x[0]+(-0.587898788817)*x[1]**o+(0.700612189317)*x[1]
        arg[3,0,1,0]=(-0.899654954269)*x[0]**o+(-0.801426785588)*x[0]+(-0.528678401571)*x[1]**o+(-0.224185097206)*x[1]
        arg[3,0,1,1]=(0.291298188198)*x[0]**o+(0.786698690825)*x[0]+(0.925634940697)*x[1]**o+(0.617661715197)*x[1]
        arg[3,0,2,0]=(0.788272473274)*x[0]**o+(-0.0400044705274)*x[0]+(-0.669092974469)*x[1]**o+(-0.063697728491)*x[1]
        arg[3,0,2,1]=(-0.0612398565521)*x[0]**o+(-0.884311783009)*x[0]+(0.444689078628)*x[1]**o+(0.302741981611)*x[1]
        arg[3,1,0,0]=(0.483756249722)*x[0]**o+(-0.214185139167)*x[0]+(0.526387014144)*x[1]**o+(-0.172619975799)*x[1]
        arg[3,1,0,1]=(0.0685836489777)*x[0]**o+(-0.425861440836)*x[0]+(0.94669899123)*x[1]**o+(-0.649068988929)*x[1]
        arg[3,1,1,0]=(0.935225203148)*x[0]**o+(-0.649378812349)*x[0]+(-0.791553458732)*x[1]**o+(0.776307740809)*x[1]
        arg[3,1,1,1]=(-0.45319945403)*x[0]**o+(-0.473310763902)*x[0]+(-0.268480872039)*x[1]**o+(0.847382076174)*x[1]
        arg[3,1,2,0]=(-0.573022434421)*x[0]**o+(0.584143976397)*x[0]+(0.131688457808)*x[1]**o+(-0.628894863435)*x[1]
        arg[3,1,2,1]=(0.628566928363)*x[0]**o+(-0.868564569338)*x[0]+(-0.482273553287)*x[1]**o+(0.679652796519)*x[1]
        arg[3,2,0,0]=(-0.64046509323)*x[0]**o+(-0.108928537952)*x[0]+(-0.512288515479)*x[1]**o+(0.683072290041)*x[1]
        arg[3,2,0,1]=(0.472589328326)*x[0]**o+(0.219741638671)*x[0]+(0.136368535311)*x[1]**o+(-0.525113003647)*x[1]
        arg[3,2,1,0]=(-0.45731256203)*x[0]**o+(0.0854410475382)*x[0]+(0.257725845542)*x[1]**o+(0.430430185351)*x[1]
        arg[3,2,1,1]=(0.948699678725)*x[0]**o+(-0.00538658505415)*x[0]+(0.921898600708)*x[1]**o+(-0.448522286404)*x[1]
        arg[3,2,2,0]=(-0.0863353766204)*x[0]**o+(-0.652417730681)*x[0]+(0.755871413533)*x[1]**o+(0.206065623655)*x[1]
        arg[3,2,2,1]=(0.263075920527)*x[0]**o+(0.94162407402)*x[0]+(0.750015836722)*x[1]**o+(-0.815154267718)*x[1]
        arg[3,3,0,0]=(-0.606984299829)*x[0]**o+(0.120704522147)*x[0]+(-0.324211323778)*x[1]**o+(-0.123647081263)*x[1]
        arg[3,3,0,1]=(0.838995936232)*x[0]**o+(0.854147779134)*x[0]+(-0.192271552952)*x[1]**o+(0.0281108740635)*x[1]
        arg[3,3,1,0]=(-0.0343159887291)*x[0]**o+(0.242432659911)*x[0]+(-0.37299830191)*x[1]**o+(-0.456792158625)*x[1]
        arg[3,3,1,1]=(-0.305277044867)*x[0]**o+(-0.930386582491)*x[0]+(-0.979921197617)*x[1]**o+(-0.507736755766)*x[1]
        arg[3,3,2,0]=(0.548675744195)*x[0]**o+(0.755596482286)*x[0]+(0.0722267054576)*x[1]**o+(-0.393768047035)*x[1]
        arg[3,3,2,1]=(0.74781993508)*x[0]**o+(-0.655033306209)*x[0]+(0.480653847364)*x[1]**o+(-0.169400714085)*x[1]
        arg[3,4,0,0]=(0.269186769422)*x[0]**o+(-0.0786214595922)*x[0]+(-0.975508552024)*x[1]**o+(-0.798249509899)*x[1]
        arg[3,4,0,1]=(-0.599201275441)*x[0]**o+(0.473490319504)*x[0]+(0.0277044125859)*x[1]**o+(0.818107843614)*x[1]
        arg[3,4,1,0]=(-0.46272642862)*x[0]**o+(0.740282148493)*x[0]+(0.35998379499)*x[1]**o+(-0.938681489717)*x[1]
        arg[3,4,1,1]=(0.187067286301)*x[0]**o+(-0.284171344521)*x[0]+(0.612229918327)*x[1]**o+(0.3917984881)*x[1]
        arg[3,4,2,0]=(-0.535270910341)*x[0]**o+(-0.00723888213927)*x[0]+(-0.776984534618)*x[1]**o+(-0.0899536033273)*x[1]
        arg[3,4,2,1]=(-0.382290376155)*x[0]**o+(-0.0593490177599)*x[0]+(-0.19716515678)*x[1]**o+(-0.268991693613)*x[1]
        ref[0,0,0,0]=(0.19989964713)*(1+2.*(dim-1.)/(o+1.))+(0.319607794084)*dim
        ref[0,0,0,1]=(0.181882046176)*(1+2.*(dim-1.)/(o+1.))+(-0.527249335907)*dim
        ref[0,0,1,0]=(0.982738326978)*(1+2.*(dim-1.)/(o+1.))+(0.246861645686)*dim
        ref[0,0,1,1]=(-1.47233037389)*(1+2.*(dim-1.)/(o+1.))+(-0.0794837747136)*dim
        ref[0,0,2,0]=(-1.35167401473)*(1+2.*(dim-1.)/(o+1.))+(-0.0407923511861)*dim
        ref[0,0,2,1]=(-0.254595315742)*(1+2.*(dim-1.)/(o+1.))+(-1.26506521013)*dim
        ref[0,1,0,0]=(-0.909568537701)*(1+2.*(dim-1.)/(o+1.))+(0.0382111002203)*dim
        ref[0,1,0,1]=(-0.961011088187)*(1+2.*(dim-1.)/(o+1.))+(0.311813862038)*dim
        ref[0,1,1,0]=(0.540512799627)*(1+2.*(dim-1.)/(o+1.))+(-0.00503022927381)*dim
        ref[0,1,1,1]=(0.148835768693)*(1+2.*(dim-1.)/(o+1.))+(-0.133338026157)*dim
        ref[0,1,2,0]=(-0.0921555325482)*(1+2.*(dim-1.)/(o+1.))+(-0.253470191439)*dim
        ref[0,1,2,1]=(0.37975053514)*(1+2.*(dim-1.)/(o+1.))+(0.137491991068)*dim
        ref[0,2,0,0]=(-0.637116147991)*(1+2.*(dim-1.)/(o+1.))+(-0.782657261086)*dim
        ref[0,2,0,1]=(-0.109247728514)*(1+2.*(dim-1.)/(o+1.))+(0.541417493824)*dim
        ref[0,2,1,0]=(-0.933462481479)*(1+2.*(dim-1.)/(o+1.))+(-1.53681028328)*dim
        ref[0,2,1,1]=(-1.19773222964)*(1+2.*(dim-1.)/(o+1.))+(1.31644104653)*dim
        ref[0,2,2,0]=(-1.17964226873)*(1+2.*(dim-1.)/(o+1.))+(1.12940627206)*dim
        ref[0,2,2,1]=(0.199043107375)*(1+2.*(dim-1.)/(o+1.))+(0.694446720792)*dim
        ref[0,3,0,0]=(0.699758278049)*(1+2.*(dim-1.)/(o+1.))+(-1.5865168617)*dim
        ref[0,3,0,1]=(-1.43918025379)*(1+2.*(dim-1.)/(o+1.))+(1.32584017086)*dim
        ref[0,3,1,0]=(0.320617899014)*(1+2.*(dim-1.)/(o+1.))+(-0.63836051502)*dim
        ref[0,3,1,1]=(-0.741820981021)*(1+2.*(dim-1.)/(o+1.))+(-0.463610958784)*dim
        ref[0,3,2,0]=(0.567695730048)*(1+2.*(dim-1.)/(o+1.))+(-0.852917804)*dim
        ref[0,3,2,1]=(1.06481671764)*(1+2.*(dim-1.)/(o+1.))+(-0.363720838369)*dim
        ref[0,4,0,0]=(0.875430749562)*(1+2.*(dim-1.)/(o+1.))+(0.551253069648)*dim
        ref[0,4,0,1]=(1.26260850991)*(1+2.*(dim-1.)/(o+1.))+(0.921634441735)*dim
        ref[0,4,1,0]=(-0.542320585907)*(1+2.*(dim-1.)/(o+1.))+(0.0620413683361)*dim
        ref[0,4,1,1]=(-0.738376217327)*(1+2.*(dim-1.)/(o+1.))+(0.416152836932)*dim
        ref[0,4,2,0]=(-0.332750203458)*(1+2.*(dim-1.)/(o+1.))+(1.22147094265)*dim
        ref[0,4,2,1]=(1.30503980926)*(1+2.*(dim-1.)/(o+1.))+(-1.24682001245)*dim
        ref[1,0,0,0]=(1.20083814495)*(1+2.*(dim-1.)/(o+1.))+(0.224931716401)*dim
        ref[1,0,0,1]=(0.493849187349)*(1+2.*(dim-1.)/(o+1.))+(0.109403168198)*dim
        ref[1,0,1,0]=(0.666795564157)*(1+2.*(dim-1.)/(o+1.))+(-0.188860724968)*dim
        ref[1,0,1,1]=(0.322811827009)*(1+2.*(dim-1.)/(o+1.))+(0.58147616313)*dim
        ref[1,0,2,0]=(-1.21934428158)*(1+2.*(dim-1.)/(o+1.))+(0.698046512357)*dim
        ref[1,0,2,1]=(1.35903157083)*(1+2.*(dim-1.)/(o+1.))+(0.276174469644)*dim
        ref[1,1,0,0]=(-0.668757118699)*(1+2.*(dim-1.)/(o+1.))+(0.184772207756)*dim
        ref[1,1,0,1]=(1.25787213023)*(1+2.*(dim-1.)/(o+1.))+(0.0222646493659)*dim
        ref[1,1,1,0]=(0.167176470046)*(1+2.*(dim-1.)/(o+1.))+(-0.245832394617)*dim
        ref[1,1,1,1]=(-0.659841661688)*(1+2.*(dim-1.)/(o+1.))+(-0.953939685406)*dim
        ref[1,1,2,0]=(1.07102228089)*(1+2.*(dim-1.)/(o+1.))+(-0.132946987715)*dim
        ref[1,1,2,1]=(-0.347830760025)*(1+2.*(dim-1.)/(o+1.))+(-0.484236996915)*dim
        ref[1,2,0,0]=(-0.087001660358)*(1+2.*(dim-1.)/(o+1.))+(1.44180128866)*dim
        ref[1,2,0,1]=(-0.466598066397)*(1+2.*(dim-1.)/(o+1.))+(1.1128931162)*dim
        ref[1,2,1,0]=(-0.996210852643)*(1+2.*(dim-1.)/(o+1.))+(1.24400417763)*dim
        ref[1,2,1,1]=(0.206220408321)*(1+2.*(dim-1.)/(o+1.))+(0.332074088353)*dim
        ref[1,2,2,0]=(0.103151003161)*(1+2.*(dim-1.)/(o+1.))+(1.4480060297)*dim
        ref[1,2,2,1]=(-0.282559222939)*(1+2.*(dim-1.)/(o+1.))+(-1.47590976977)*dim
        ref[1,3,0,0]=(-0.697481881234)*(1+2.*(dim-1.)/(o+1.))+(-0.414345520764)*dim
        ref[1,3,0,1]=(-0.954564688559)*(1+2.*(dim-1.)/(o+1.))+(-0.93833639809)*dim
        ref[1,3,1,0]=(1.74786122066)*(1+2.*(dim-1.)/(o+1.))+(-0.603460563479)*dim
        ref[1,3,1,1]=(0.576696710719)*(1+2.*(dim-1.)/(o+1.))+(1.5449563998)*dim
        ref[1,3,2,0]=(-0.26671915305)*(1+2.*(dim-1.)/(o+1.))+(1.19067969538)*dim
        ref[1,3,2,1]=(1.52058149003)*(1+2.*(dim-1.)/(o+1.))+(0.582969127914)*dim
        ref[1,4,0,0]=(0.214940907208)*(1+2.*(dim-1.)/(o+1.))+(-1.28759712193)*dim
        ref[1,4,0,1]=(-0.564229160887)*(1+2.*(dim-1.)/(o+1.))+(1.46222745131)*dim
        ref[1,4,1,0]=(1.00974626816)*(1+2.*(dim-1.)/(o+1.))+(-0.264193096019)*dim
        ref[1,4,1,1]=(-1.03604400812)*(1+2.*(dim-1.)/(o+1.))+(-0.048064364322)*dim
        ref[1,4,2,0]=(0.215263433059)*(1+2.*(dim-1.)/(o+1.))+(-0.926414177826)*dim
        ref[1,4,2,1]=(0.863018760853)*(1+2.*(dim-1.)/(o+1.))+(-0.210974421418)*dim
        ref[2,0,0,0]=(0.456122143601)*(1+2.*(dim-1.)/(o+1.))+(-0.286656440625)*dim
        ref[2,0,0,1]=(1.62808877765)*(1+2.*(dim-1.)/(o+1.))+(-0.270052895997)*dim
        ref[2,0,1,0]=(-0.839009627862)*(1+2.*(dim-1.)/(o+1.))+(-1.07800157305)*dim
        ref[2,0,1,1]=(-0.755931387477)*(1+2.*(dim-1.)/(o+1.))+(-0.30390722284)*dim
        ref[2,0,2,0]=(-0.304811532322)*(1+2.*(dim-1.)/(o+1.))+(1.00912450174)*dim
        ref[2,0,2,1]=(0.61186776369)*(1+2.*(dim-1.)/(o+1.))+(0.60903675699)*dim
        ref[2,1,0,0]=(0.775746359099)*(1+2.*(dim-1.)/(o+1.))+(-0.0652640754538)*dim
        ref[2,1,0,1]=(1.06797988144)*(1+2.*(dim-1.)/(o+1.))+(0.702021087765)*dim
        ref[2,1,1,0]=(1.62297766272)*(1+2.*(dim-1.)/(o+1.))+(0.913406416145)*dim
        ref[2,1,1,1]=(0.296175986273)*(1+2.*(dim-1.)/(o+1.))+(1.2000216877)*dim
        ref[2,1,2,0]=(-1.39985921748)*(1+2.*(dim-1.)/(o+1.))+(0.793386653567)*dim
        ref[2,1,2,1]=(0.76675815763)*(1+2.*(dim-1.)/(o+1.))+(0.289921670962)*dim
        ref[2,2,0,0]=(-0.795787652043)*(1+2.*(dim-1.)/(o+1.))+(-0.0032846811301)*dim
        ref[2,2,0,1]=(-1.37138393084)*(1+2.*(dim-1.)/(o+1.))+(-0.358175850871)*dim
        ref[2,2,1,0]=(0.188257903467)*(1+2.*(dim-1.)/(o+1.))+(-0.334461225539)*dim
        ref[2,2,1,1]=(0.118391598158)*(1+2.*(dim-1.)/(o+1.))+(0.289816675047)*dim
        ref[2,2,2,0]=(-0.663970999582)*(1+2.*(dim-1.)/(o+1.))+(-0.299159718332)*dim
        ref[2,2,2,1]=(0.556618480008)*(1+2.*(dim-1.)/(o+1.))+(-1.2696788018)*dim
        ref[2,3,0,0]=(-0.485651868997)*(1+2.*(dim-1.)/(o+1.))+(1.17320731927)*dim
        ref[2,3,0,1]=(0.104319213682)*(1+2.*(dim-1.)/(o+1.))+(0.911530372697)*dim
        ref[2,3,1,0]=(0.43248864489)*(1+2.*(dim-1.)/(o+1.))+(-0.0197594552935)*dim
        ref[2,3,1,1]=(0.0276596551936)*(1+2.*(dim-1.)/(o+1.))+(0.770762527568)*dim
        ref[2,3,2,0]=(0.573639628608)*(1+2.*(dim-1.)/(o+1.))+(1.38642927495)*dim
        ref[2,3,2,1]=(-0.502260185391)*(1+2.*(dim-1.)/(o+1.))+(1.62564500839)*dim
        ref[2,4,0,0]=(1.70497128864)*(1+2.*(dim-1.)/(o+1.))+(-0.198778229868)*dim
        ref[2,4,0,1]=(0.906274587146)*(1+2.*(dim-1.)/(o+1.))+(0.55919273648)*dim
        ref[2,4,1,0]=(-1.52420172666)*(1+2.*(dim-1.)/(o+1.))+(-0.828944811982)*dim
        ref[2,4,1,1]=(-0.272628509541)*(1+2.*(dim-1.)/(o+1.))+(0.225820656649)*dim
        ref[2,4,2,0]=(0.129498614356)*(1+2.*(dim-1.)/(o+1.))+(1.27146129393)*dim
        ref[2,4,2,1]=(-0.475554772211)*(1+2.*(dim-1.)/(o+1.))+(0.281442644713)*dim
        ref[3,0,0,0]=(-1.48648385291)*(1+2.*(dim-1.)/(o+1.))+(-0.931627841127)*dim
        ref[3,0,0,1]=(-0.390858507465)*(1+2.*(dim-1.)/(o+1.))+(0.856397607398)*dim
        ref[3,0,1,0]=(-1.42833335584)*(1+2.*(dim-1.)/(o+1.))+(-1.02561188279)*dim
        ref[3,0,1,1]=(1.2169331289)*(1+2.*(dim-1.)/(o+1.))+(1.40436040602)*dim
        ref[3,0,2,0]=(0.119179498805)*(1+2.*(dim-1.)/(o+1.))+(-0.103702199018)*dim
        ref[3,0,2,1]=(0.383449222076)*(1+2.*(dim-1.)/(o+1.))+(-0.581569801398)*dim
        ref[3,1,0,0]=(1.01014326387)*(1+2.*(dim-1.)/(o+1.))+(-0.386805114966)*dim
        ref[3,1,0,1]=(1.01528264021)*(1+2.*(dim-1.)/(o+1.))+(-1.07493042977)*dim
        ref[3,1,1,0]=(0.143671744416)*(1+2.*(dim-1.)/(o+1.))+(0.12692892846)*dim
        ref[3,1,1,1]=(-0.721680326068)*(1+2.*(dim-1.)/(o+1.))+(0.374071312272)*dim
        ref[3,1,2,0]=(-0.441333976613)*(1+2.*(dim-1.)/(o+1.))+(-0.0447508870373)*dim
        ref[3,1,2,1]=(0.146293375075)*(1+2.*(dim-1.)/(o+1.))+(-0.188911772819)*dim
        ref[3,2,0,0]=(-1.15275360871)*(1+2.*(dim-1.)/(o+1.))+(0.57414375209)*dim
        ref[3,2,0,1]=(0.608957863637)*(1+2.*(dim-1.)/(o+1.))+(-0.305371364976)*dim
        ref[3,2,1,0]=(-0.199586716488)*(1+2.*(dim-1.)/(o+1.))+(0.515871232889)*dim
        ref[3,2,1,1]=(1.87059827943)*(1+2.*(dim-1.)/(o+1.))+(-0.453908871458)*dim
        ref[3,2,2,0]=(0.669536036913)*(1+2.*(dim-1.)/(o+1.))+(-0.446352107026)*dim
        ref[3,2,2,1]=(1.01309175725)*(1+2.*(dim-1.)/(o+1.))+(0.126469806303)*dim
        ref[3,3,0,0]=(-0.931195623607)*(1+2.*(dim-1.)/(o+1.))+(-0.00294255911617)*dim
        ref[3,3,0,1]=(0.64672438328)*(1+2.*(dim-1.)/(o+1.))+(0.882258653197)*dim
        ref[3,3,1,0]=(-0.407314290639)*(1+2.*(dim-1.)/(o+1.))+(-0.214359498715)*dim
        ref[3,3,1,1]=(-1.28519824248)*(1+2.*(dim-1.)/(o+1.))+(-1.43812333826)*dim
        ref[3,3,2,0]=(0.620902449652)*(1+2.*(dim-1.)/(o+1.))+(0.361828435252)*dim
        ref[3,3,2,1]=(1.22847378244)*(1+2.*(dim-1.)/(o+1.))+(-0.824434020294)*dim
        ref[3,4,0,0]=(-0.706321782603)*(1+2.*(dim-1.)/(o+1.))+(-0.876870969492)*dim
        ref[3,4,0,1]=(-0.571496862855)*(1+2.*(dim-1.)/(o+1.))+(1.29159816312)*dim
        ref[3,4,1,0]=(-0.102742633631)*(1+2.*(dim-1.)/(o+1.))+(-0.198399341224)*dim
        ref[3,4,1,1]=(0.799297204627)*(1+2.*(dim-1.)/(o+1.))+(0.107627143578)*dim
        ref[3,4,2,0]=(-1.31225544496)*(1+2.*(dim-1.)/(o+1.))+(-0.0971924854666)*dim
        ref[3,4,2,1]=(-0.579455532935)*(1+2.*(dim-1.)/(o+1.))+(-0.328340711373)*dim
      else:
        arg[0,0,0,0]=(-0.727570048979)*x[0]**o+(-0.769770353069)*x[0]+(-0.532381054852)*x[1]**o+(0.590589994369)*x[1]+(0.775348726284)*x[2]**o+(-0.840239939796)*x[2]
        arg[0,0,0,1]=(-0.441856577751)*x[0]**o+(-0.39234308235)*x[0]+(-0.834378647798)*x[1]**o+(0.84269918161)*x[1]+(0.441102591728)*x[2]**o+(0.166331058665)*x[2]
        arg[0,0,1,0]=(0.391773069521)*x[0]**o+(0.496759920942)*x[0]+(0.555386226902)*x[1]**o+(0.980197783628)*x[1]+(0.549622021652)*x[2]**o+(-0.596179893504)*x[2]
        arg[0,0,1,1]=(-0.104197387892)*x[0]**o+(-0.64636505375)*x[0]+(0.377612339028)*x[1]**o+(-0.215111619231)*x[1]+(0.637249692625)*x[2]**o+(-0.0998227822351)*x[2]
        arg[0,0,2,0]=(0.780693043561)*x[0]**o+(0.571405344342)*x[0]+(-0.793680154472)*x[1]**o+(-0.0481469136436)*x[1]+(0.959561051293)*x[2]**o+(0.0756152497582)*x[2]
        arg[0,0,2,1]=(0.88678762282)*x[0]**o+(0.254753272214)*x[0]+(0.344376798516)*x[1]**o+(-0.908135902735)*x[1]+(-0.3584387726)*x[2]**o+(-0.890225037617)*x[2]
        arg[0,1,0,0]=(0.680262917473)*x[0]**o+(0.65860357305)*x[0]+(0.859832322473)*x[1]**o+(0.326105580592)*x[1]+(0.132170297876)*x[2]**o+(-0.928606849775)*x[2]
        arg[0,1,0,1]=(-0.500503822293)*x[0]**o+(-0.456331012174)*x[0]+(0.900478846187)*x[1]**o+(0.350542503874)*x[1]+(0.755849038617)*x[2]**o+(-0.514240007142)*x[2]
        arg[0,1,1,0]=(0.124404217291)*x[0]**o+(-0.499651000805)*x[0]+(-0.968178694708)*x[1]**o+(0.306780500818)*x[1]+(-0.946470794819)*x[2]**o+(0.919424575523)*x[2]
        arg[0,1,1,1]=(0.722520017474)*x[0]**o+(-0.718216593718)*x[0]+(0.801196051455)*x[1]**o+(-0.791951278546)*x[1]+(-0.675975417529)*x[2]**o+(0.355069676047)*x[2]
        arg[0,1,2,0]=(-0.137017428857)*x[0]**o+(-0.760267879487)*x[0]+(0.625869259413)*x[1]**o+(-0.697626713352)*x[1]+(-0.467876751423)*x[2]**o+(0.456475454123)*x[2]
        arg[0,1,2,1]=(0.646536442797)*x[0]**o+(0.0981671744166)*x[0]+(0.477147756414)*x[1]**o+(-0.23211823966)*x[1]+(-0.553808167147)*x[2]**o+(0.405402676302)*x[2]
        arg[0,2,0,0]=(-0.0604502966371)*x[0]**o+(0.297995946813)*x[0]+(0.837045433858)*x[1]**o+(-0.270340963575)*x[1]+(-0.58462190484)*x[2]**o+(0.292953993287)*x[2]
        arg[0,2,0,1]=(-0.463109088268)*x[0]**o+(0.346260157368)*x[0]+(0.0228615151352)*x[1]**o+(0.905507471559)*x[1]+(-0.423777553665)*x[2]**o+(-0.45964535654)*x[2]
        arg[0,2,1,0]=(-0.50761294542)*x[0]**o+(-0.465260578419)*x[0]+(-0.729953382373)*x[1]**o+(0.903666219263)*x[1]+(-0.756794344711)*x[2]**o+(0.817263008431)*x[2]
        arg[0,2,1,1]=(-0.551253781393)*x[0]**o+(-0.911481117802)*x[0]+(0.180349330394)*x[1]**o+(0.419118491074)*x[1]+(-0.691917875703)*x[2]**o+(-0.835365504347)*x[2]
        arg[0,2,2,0]=(0.32567932803)*x[0]**o+(0.413167839883)*x[0]+(0.781320691089)*x[1]**o+(-0.777223093057)*x[1]+(0.297556831029)*x[2]**o+(-0.952676550814)*x[2]
        arg[0,2,2,1]=(0.226250474873)*x[0]**o+(0.279287242249)*x[0]+(0.421824124143)*x[1]**o+(-0.97664927412)*x[1]+(0.703245736838)*x[2]**o+(-0.629660622476)*x[2]
        arg[0,3,0,0]=(-0.138403445102)*x[0]**o+(0.0979152923187)*x[0]+(-0.177116218507)*x[1]**o+(0.0100523961708)*x[1]+(-0.634159492232)*x[2]**o+(-0.847618220907)*x[2]
        arg[0,3,0,1]=(0.612268738957)*x[0]**o+(-0.462682392204)*x[0]+(-0.146623835506)*x[1]**o+(0.699258469487)*x[1]+(0.0139983038691)*x[2]**o+(0.236461011719)*x[2]
        arg[0,3,1,0]=(-0.0518123102809)*x[0]**o+(0.386822059696)*x[0]+(-0.208858744655)*x[1]**o+(0.971393362011)*x[1]+(-0.171649562077)*x[2]**o+(-0.507479926411)*x[2]
        arg[0,3,1,1]=(0.0140396270391)*x[0]**o+(-0.837711757872)*x[0]+(-0.00108924459289)*x[1]**o+(-0.724087080496)*x[1]+(-0.793071741694)*x[2]**o+(0.38654936043)*x[2]
        arg[0,3,2,0]=(-0.897711437057)*x[0]**o+(0.562533782571)*x[0]+(-0.354013039442)*x[1]**o+(-0.473034213198)*x[1]+(0.4598333464)*x[2]**o+(0.467503992754)*x[2]
        arg[0,3,2,1]=(-0.198441232877)*x[0]**o+(-0.411568849672)*x[0]+(0.0850211273328)*x[1]**o+(-0.333702582633)*x[1]+(0.727888461746)*x[2]**o+(-0.174835541819)*x[2]
        arg[0,4,0,0]=(-0.114679585752)*x[0]**o+(0.0446596462171)*x[0]+(-0.950604674991)*x[1]**o+(-0.307622186566)*x[1]+(0.956712754366)*x[2]**o+(0.334630480874)*x[2]
        arg[0,4,0,1]=(0.345475911118)*x[0]**o+(0.834620576123)*x[0]+(0.903176273684)*x[1]**o+(-0.364211856466)*x[1]+(0.910903954488)*x[2]**o+(0.392248545744)*x[2]
        arg[0,4,1,0]=(-0.593360797686)*x[0]**o+(-0.0199892857836)*x[0]+(0.293839091284)*x[1]**o+(0.580208854742)*x[1]+(0.060499817751)*x[2]**o+(0.907498841261)*x[2]
        arg[0,4,1,1]=(-0.446192713548)*x[0]**o+(0.29870251614)*x[0]+(0.754231657585)*x[1]**o+(0.321710920549)*x[1]+(0.287978396665)*x[2]**o+(-0.963473165492)*x[2]
        arg[0,4,2,0]=(0.780174386478)*x[0]**o+(0.603377914483)*x[0]+(-0.841703120145)*x[1]**o+(0.316522795462)*x[1]+(-0.192916583378)*x[2]**o+(-0.734091913995)*x[2]
        arg[0,4,2,1]=(-0.130567656486)*x[0]**o+(-0.342928842892)*x[0]+(-0.0827376128775)*x[1]**o+(-0.771104057041)*x[1]+(0.733571283734)*x[2]**o+(-0.880531838119)*x[2]
        arg[1,0,0,0]=(-0.634428121043)*x[0]**o+(0.294135783753)*x[0]+(-0.566305561339)*x[1]**o+(0.0332700472211)*x[1]+(0.128130119463)*x[2]**o+(0.0468678470468)*x[2]
        arg[1,0,0,1]=(0.82168858107)*x[0]**o+(-0.75544177649)*x[0]+(0.260115562827)*x[1]**o+(-0.149575433725)*x[1]+(-0.780945910032)*x[2]**o+(-0.0539291857071)*x[2]
        arg[1,0,1,0]=(-0.97885291605)*x[0]**o+(0.777955971883)*x[0]+(0.115355866293)*x[1]**o+(-0.456449594589)*x[1]+(-0.832816352387)*x[2]**o+(-0.418070257158)*x[2]
        arg[1,0,1,1]=(-0.908396506245)*x[0]**o+(-0.982147653872)*x[0]+(0.138657052347)*x[1]**o+(0.633837624386)*x[1]+(-0.529748303776)*x[2]**o+(-0.950225664343)*x[2]
        arg[1,0,2,0]=(0.733974435377)*x[0]**o+(-0.758228777635)*x[0]+(0.841368499977)*x[1]**o+(0.871715889444)*x[1]+(0.541055759071)*x[2]**o+(0.669798500731)*x[2]
        arg[1,0,2,1]=(-0.572994548563)*x[0]**o+(0.806241853906)*x[0]+(0.141217952929)*x[1]**o+(0.832291361477)*x[1]+(0.207737014762)*x[2]**o+(-0.279769695917)*x[2]
        arg[1,1,0,0]=(-0.394247339923)*x[0]**o+(0.548338039327)*x[0]+(-0.845822483858)*x[1]**o+(-0.941161341738)*x[1]+(-0.244513027249)*x[2]**o+(-0.524790843131)*x[2]
        arg[1,1,0,1]=(0.509994957918)*x[0]**o+(0.691470630092)*x[0]+(0.249241759795)*x[1]**o+(-0.995293157716)*x[1]+(0.162652088127)*x[2]**o+(0.873578806104)*x[2]
        arg[1,1,1,0]=(-0.22367137056)*x[0]**o+(-0.896306406631)*x[0]+(0.613214186844)*x[1]**o+(-0.460055562163)*x[1]+(-0.704736634406)*x[2]**o+(-0.517741014186)*x[2]
        arg[1,1,1,1]=(-0.426072795689)*x[0]**o+(-0.13452184053)*x[0]+(0.908778671123)*x[1]**o+(0.194472828672)*x[1]+(-0.658909495204)*x[2]**o+(-0.234260758459)*x[2]
        arg[1,1,2,0]=(0.236086562474)*x[0]**o+(0.33312461561)*x[0]+(0.867089643801)*x[1]**o+(-0.720157305369)*x[1]+(0.812845198077)*x[2]**o+(-0.375694713863)*x[2]
        arg[1,1,2,1]=(-0.561145817489)*x[0]**o+(0.746630137163)*x[0]+(-0.809280840008)*x[1]**o+(-0.690912076356)*x[1]+(0.88933641231)*x[2]**o+(0.246647817564)*x[2]
        arg[1,2,0,0]=(0.701319152729)*x[0]**o+(-0.309074024603)*x[0]+(0.567988628717)*x[1]**o+(-0.0596939135728)*x[1]+(0.242135294918)*x[2]**o+(-0.101136005445)*x[2]
        arg[1,2,0,1]=(0.880665674235)*x[0]**o+(-0.833066777852)*x[0]+(-0.0556340786728)*x[1]**o+(0.620095583249)*x[1]+(-0.390556829051)*x[2]**o+(0.793481461864)*x[2]
        arg[1,2,1,0]=(0.832046852431)*x[0]**o+(0.4276990324)*x[0]+(-0.486292331356)*x[1]**o+(0.990296730852)*x[1]+(-0.381732943927)*x[2]**o+(0.225838720655)*x[2]
        arg[1,2,1,1]=(-0.971477034333)*x[0]**o+(0.819409035213)*x[0]+(-0.610341797903)*x[1]**o+(0.842186877805)*x[1]+(0.571187503044)*x[2]**o+(-0.150565006374)*x[2]
        arg[1,2,2,0]=(0.266080624706)*x[0]**o+(0.893377400845)*x[0]+(0.718525426441)*x[1]**o+(-0.847615622286)*x[1]+(0.614914866)*x[2]**o+(0.826503774974)*x[2]
        arg[1,2,2,1]=(0.998060871997)*x[0]**o+(-0.572523112154)*x[0]+(-0.786785614822)*x[1]**o+(0.556952019953)*x[1]+(0.804849394796)*x[2]**o+(-0.201157281118)*x[2]
        arg[1,3,0,0]=(-0.322048105751)*x[0]**o+(0.215441634353)*x[0]+(0.203298400258)*x[1]**o+(0.452071048161)*x[1]+(0.0134315902731)*x[2]**o+(0.240098266028)*x[2]
        arg[1,3,0,1]=(-0.109845136841)*x[0]**o+(0.0479663457074)*x[0]+(0.899394604931)*x[1]**o+(0.853608214151)*x[1]+(-0.27487885579)*x[2]**o+(-0.0236702866567)*x[2]
        arg[1,3,1,0]=(0.948550474293)*x[0]**o+(0.359077475995)*x[0]+(0.655387467125)*x[1]**o+(-0.0501114084308)*x[1]+(-0.244470769181)*x[2]**o+(0.670276065978)*x[2]
        arg[1,3,1,1]=(-0.860769147822)*x[0]**o+(-0.738449680804)*x[0]+(0.840959189643)*x[1]**o+(0.229930283198)*x[1]+(-0.521034843253)*x[2]**o+(-0.50213233896)*x[2]
        arg[1,3,2,0]=(0.29910216992)*x[0]**o+(0.411953562132)*x[0]+(0.777584507308)*x[1]**o+(0.41482951697)*x[1]+(-0.948751119313)*x[2]**o+(-0.526573791564)*x[2]
        arg[1,3,2,1]=(-0.768167653244)*x[0]**o+(-0.318762199437)*x[0]+(0.7566706966)*x[1]**o+(-0.151347269999)*x[1]+(-0.626690637485)*x[2]**o+(-0.731518248609)*x[2]
        arg[1,4,0,0]=(0.0147733854143)*x[0]**o+(-0.812988077571)*x[0]+(0.815385483312)*x[1]**o+(0.285326283687)*x[1]+(0.872502797762)*x[2]**o+(-0.748392032843)*x[2]
        arg[1,4,0,1]=(0.172931930795)*x[0]**o+(0.425088340505)*x[0]+(0.0122969950852)*x[1]**o+(0.556865378995)*x[1]+(0.614686173604)*x[2]**o+(0.966581266651)*x[2]
        arg[1,4,1,0]=(0.725270959871)*x[0]**o+(-0.672683754253)*x[0]+(0.486696038678)*x[1]**o+(-0.53856418144)*x[1]+(-0.295197744155)*x[2]**o+(0.0610876962829)*x[2]
        arg[1,4,1,1]=(0.942300770794)*x[0]**o+(0.876440453364)*x[0]+(0.432652207093)*x[1]**o+(-0.562629961162)*x[1]+(0.497526473424)*x[2]**o+(-0.165824324383)*x[2]
        arg[1,4,2,0]=(-0.668927175078)*x[0]**o+(0.587283304457)*x[0]+(0.588305038304)*x[1]**o+(0.087906366831)*x[1]+(-0.907721198172)*x[2]**o+(-0.506067369924)*x[2]
        arg[1,4,2,1]=(0.899071894553)*x[0]**o+(0.920678485704)*x[0]+(-0.244107214117)*x[1]**o+(0.196327093214)*x[1]+(-0.538075261013)*x[2]**o+(-0.925646646321)*x[2]
        arg[2,0,0,0]=(-0.113452114076)*x[0]**o+(0.541178389983)*x[0]+(0.501083424074)*x[1]**o+(-0.234654083639)*x[1]+(-0.435569149263)*x[2]**o+(0.671517947867)*x[2]
        arg[2,0,0,1]=(-0.00139653535956)*x[0]**o+(-0.93731587459)*x[0]+(-0.387786776826)*x[1]**o+(0.361445822788)*x[1]+(-0.0958490804452)*x[2]**o+(0.22475163005)*x[2]
        arg[2,0,1,0]=(0.100832420019)*x[0]**o+(-0.957577198313)*x[0]+(0.244479124631)*x[1]**o+(-0.589295650233)*x[1]+(-0.11020792463)*x[2]**o+(0.969023424875)*x[2]
        arg[2,0,1,1]=(-0.82757398282)*x[0]**o+(0.665274113482)*x[0]+(0.219624146627)*x[1]**o+(0.604687589384)*x[1]+(0.539774513376)*x[2]**o+(0.676136495778)*x[2]
        arg[2,0,2,0]=(-0.869539421193)*x[0]**o+(0.449645390105)*x[0]+(-0.216356671763)*x[1]**o+(-0.191616800811)*x[1]+(-0.324126591392)*x[2]**o+(0.726527544226)*x[2]
        arg[2,0,2,1]=(-0.522340761977)*x[0]**o+(-0.253136332372)*x[0]+(-0.993693358789)*x[1]**o+(0.689338979208)*x[1]+(-0.73030598892)*x[2]**o+(0.604868622965)*x[2]
        arg[2,1,0,0]=(-0.87285436276)*x[0]**o+(-0.665917385941)*x[0]+(0.105570880359)*x[1]**o+(0.773780432278)*x[1]+(-0.0932677950401)*x[2]**o+(0.55563257373)*x[2]
        arg[2,1,0,1]=(-0.717823895492)*x[0]**o+(-0.843992380543)*x[0]+(0.348730031558)*x[1]**o+(-0.0641727895105)*x[1]+(-0.575332519937)*x[2]**o+(0.566731137573)*x[2]
        arg[2,1,1,0]=(-0.749216309273)*x[0]**o+(0.945232679736)*x[0]+(-0.984694983452)*x[1]**o+(-0.697422850579)*x[1]+(0.573407778839)*x[2]**o+(0.654548962131)*x[2]
        arg[2,1,1,1]=(-0.951637762959)*x[0]**o+(-0.0299724748286)*x[0]+(-0.784527116834)*x[1]**o+(0.490138786409)*x[1]+(0.945743565011)*x[2]**o+(0.158024494836)*x[2]
        arg[2,1,2,0]=(0.80439224967)*x[0]**o+(0.889509225531)*x[0]+(-0.178335319647)*x[1]**o+(-0.365669987413)*x[1]+(-0.170055755127)*x[2]**o+(0.258843711555)*x[2]
        arg[2,1,2,1]=(0.520706901588)*x[0]**o+(0.897178241272)*x[0]+(-0.128495163664)*x[1]**o+(-0.404742110163)*x[1]+(0.140333496135)*x[2]**o+(0.31429703244)*x[2]
        arg[2,2,0,0]=(-0.933201442136)*x[0]**o+(0.0083650170184)*x[0]+(-0.281807642508)*x[1]**o+(-0.09193002721)*x[1]+(-0.422582781227)*x[2]**o+(0.358915935551)*x[2]
        arg[2,2,0,1]=(-0.09712053324)*x[0]**o+(-0.0752448353447)*x[0]+(0.41291697894)*x[1]**o+(0.593823615119)*x[1]+(0.954183095989)*x[2]**o+(0.479576723309)*x[2]
        arg[2,2,1,0]=(-0.48592273499)*x[0]**o+(-0.555807193072)*x[0]+(0.842926147107)*x[1]**o+(-0.0760474608316)*x[1]+(0.864590424311)*x[2]**o+(-0.854721691042)*x[2]
        arg[2,2,1,1]=(0.238261310086)*x[0]**o+(-0.506114025767)*x[0]+(-0.534154656267)*x[1]**o+(-0.282262066272)*x[1]+(-0.241950678835)*x[2]**o+(-0.596913007205)*x[2]
        arg[2,2,2,0]=(0.35002425052)*x[0]**o+(-0.724235915518)*x[0]+(-0.443311896206)*x[1]**o+(0.0183169190559)*x[1]+(-0.180995873436)*x[2]**o+(0.949105760518)*x[2]
        arg[2,2,2,1]=(-0.524596480672)*x[0]**o+(-0.72511087285)*x[0]+(-0.793159588655)*x[1]**o+(0.197579071177)*x[1]+(0.214510478855)*x[2]**o+(0.592011219443)*x[2]
        arg[2,3,0,0]=(-0.777144207151)*x[0]**o+(0.821815068413)*x[0]+(0.828183715461)*x[1]**o+(-0.77742744879)*x[1]+(0.520088528639)*x[2]**o+(0.564796500907)*x[2]
        arg[2,3,0,1]=(-0.444252706506)*x[0]**o+(-0.999658246414)*x[0]+(0.458490846376)*x[1]**o+(0.206371738627)*x[1]+(-0.5943830165)*x[2]**o+(0.125954475804)*x[2]
        arg[2,3,1,0]=(-0.499555379078)*x[0]**o+(0.667767779032)*x[0]+(-0.169053599596)*x[1]**o+(0.587608309525)*x[1]+(-0.99317016838)*x[2]**o+(-0.580232194372)*x[2]
        arg[2,3,1,1]=(0.881736170108)*x[0]**o+(-0.0221128885544)*x[0]+(-0.239202356219)*x[1]**o+(-0.825250840397)*x[1]+(-0.34138233902)*x[2]**o+(-0.900437209752)*x[2]
        arg[2,3,2,0]=(-0.810415345283)*x[0]**o+(0.671639285402)*x[0]+(0.0577696593549)*x[1]**o+(0.12466039802)*x[1]+(-0.164952472389)*x[2]**o+(-0.881356759483)*x[2]
        arg[2,3,2,1]=(0.641411687742)*x[0]**o+(0.939465110042)*x[0]+(0.182541049996)*x[1]**o+(-0.502132165421)*x[1]+(-0.914212670088)*x[2]**o+(0.361356243622)*x[2]
        arg[2,4,0,0]=(0.696033328872)*x[0]**o+(0.854329989895)*x[0]+(0.66092830442)*x[1]**o+(-0.165623528752)*x[1]+(0.587322879532)*x[2]**o+(-0.441145527759)*x[2]
        arg[2,4,0,1]=(0.885589185979)*x[0]**o+(0.170856006811)*x[0]+(-0.500114387063)*x[1]**o+(-0.772574748562)*x[1]+(0.599300325869)*x[2]**o+(0.0950082175327)*x[2]
        arg[2,4,1,0]=(0.877739402691)*x[0]**o+(0.494715865857)*x[0]+(-0.4548916845)*x[1]**o+(-0.108381552398)*x[1]+(-0.620522177203)*x[2]**o+(-0.375546939131)*x[2]
        arg[2,4,1,1]=(-0.108044365878)*x[0]**o+(-0.592058059304)*x[0]+(-0.64481990676)*x[1]**o+(-0.903674017194)*x[1]+(0.0964052945913)*x[2]**o+(0.357819426568)*x[2]
        arg[2,4,2,0]=(-0.686157836559)*x[0]**o+(0.585514351597)*x[0]+(-0.5623443108)*x[1]**o+(-0.636059380122)*x[1]+(0.469863235081)*x[2]**o+(0.374567303622)*x[2]
        arg[2,4,2,1]=(0.214800008136)*x[0]**o+(0.0680448664819)*x[0]+(0.722930302776)*x[1]**o+(0.107573165145)*x[1]+(0.0027804595356)*x[2]**o+(0.725566605544)*x[2]
        arg[3,0,0,0]=(-0.273256791086)*x[0]**o+(0.801129507153)*x[0]+(-0.840227644476)*x[1]**o+(-0.31594258736)*x[1]+(-0.140806177065)*x[2]**o+(-0.0958103115672)*x[2]
        arg[3,0,0,1]=(0.588534583503)*x[0]**o+(0.242235627811)*x[0]+(-0.541539488722)*x[1]**o+(0.624385574868)*x[1]+(-0.0641371754789)*x[2]**o+(-0.635676751281)*x[2]
        arg[3,0,1,0]=(0.81967100218)*x[0]**o+(0.671597165436)*x[0]+(0.160179001916)*x[1]**o+(-0.226714136475)*x[1]+(-0.523709112534)*x[2]**o+(0.932797007296)*x[2]
        arg[3,0,1,1]=(0.707793115256)*x[0]**o+(-0.0995173271712)*x[0]+(-0.594597614398)*x[1]**o+(-0.72096550461)*x[1]+(-0.373631371949)*x[2]**o+(-0.282151921803)*x[2]
        arg[3,0,2,0]=(0.873307297971)*x[0]**o+(0.0950091191819)*x[0]+(-0.534108539131)*x[1]**o+(0.567381193719)*x[1]+(-0.276691140405)*x[2]**o+(0.84882104254)*x[2]
        arg[3,0,2,1]=(-0.895148878411)*x[0]**o+(-0.880973451462)*x[0]+(0.105602819158)*x[1]**o+(-0.836475687487)*x[1]+(-0.18380164164)*x[2]**o+(-0.391643939001)*x[2]
        arg[3,1,0,0]=(-0.144070349397)*x[0]**o+(-0.247578520564)*x[0]+(0.512469248899)*x[1]**o+(0.0369808259097)*x[1]+(0.67517925106)*x[2]**o+(-0.306122403522)*x[2]
        arg[3,1,0,1]=(0.389023591419)*x[0]**o+(0.0335324297663)*x[0]+(0.166380432563)*x[1]**o+(0.415133512101)*x[1]+(-0.956055398028)*x[2]**o+(-0.069254560597)*x[2]
        arg[3,1,1,0]=(-0.253194755795)*x[0]**o+(-0.432240637655)*x[0]+(0.886730830871)*x[1]**o+(0.94685399231)*x[1]+(-0.0193581886643)*x[2]**o+(-0.828124732153)*x[2]
        arg[3,1,1,1]=(0.948739050057)*x[0]**o+(-0.200402018913)*x[0]+(0.062573402994)*x[1]**o+(-0.408727698463)*x[1]+(-0.0334411087309)*x[2]**o+(-0.676147119957)*x[2]
        arg[3,1,2,0]=(0.249237167322)*x[0]**o+(-0.621387879739)*x[0]+(-0.745335388751)*x[1]**o+(-0.183173004074)*x[1]+(0.56632553768)*x[2]**o+(-0.519909176525)*x[2]
        arg[3,1,2,1]=(-0.660822495815)*x[0]**o+(0.0459275869317)*x[0]+(-0.0853936912232)*x[1]**o+(-0.895502076207)*x[1]+(0.150013238845)*x[2]**o+(0.201527905115)*x[2]
        arg[3,2,0,0]=(0.0846880899978)*x[0]**o+(-0.450721252185)*x[0]+(0.0100224498309)*x[1]**o+(-0.729399043167)*x[1]+(0.249073157161)*x[2]**o+(-0.313056982897)*x[2]
        arg[3,2,0,1]=(0.349123663394)*x[0]**o+(-0.0439266646004)*x[0]+(0.201749457716)*x[1]**o+(-0.29961557135)*x[1]+(0.247657591065)*x[2]**o+(0.702448030656)*x[2]
        arg[3,2,1,0]=(-0.67551758529)*x[0]**o+(-0.856271486186)*x[0]+(-0.337698270178)*x[1]**o+(-0.855526501932)*x[1]+(0.204628923051)*x[2]**o+(0.0425229547861)*x[2]
        arg[3,2,1,1]=(0.913895958603)*x[0]**o+(0.33488814727)*x[0]+(0.725768429772)*x[1]**o+(0.31487179035)*x[1]+(-0.926895768372)*x[2]**o+(-0.303623128744)*x[2]
        arg[3,2,2,0]=(0.695497510732)*x[0]**o+(-0.71819800795)*x[0]+(0.0949892877746)*x[1]**o+(-0.371952684199)*x[1]+(-0.712679288901)*x[2]**o+(-0.839032166941)*x[2]
        arg[3,2,2,1]=(0.847915632309)*x[0]**o+(0.489320767225)*x[0]+(0.884694888545)*x[1]**o+(-0.836734066966)*x[1]+(-0.782643394677)*x[2]**o+(0.998012000062)*x[2]
        arg[3,3,0,0]=(0.400343531732)*x[0]**o+(-0.977399566427)*x[0]+(-0.0491815761683)*x[1]**o+(0.820343030844)*x[1]+(0.145276965371)*x[2]**o+(0.106181410329)*x[2]
        arg[3,3,0,1]=(0.39509119571)*x[0]**o+(-0.532152199068)*x[0]+(0.0961163031465)*x[1]**o+(-0.845734786516)*x[1]+(0.0685207262665)*x[2]**o+(0.10725771579)*x[2]
        arg[3,3,1,0]=(-0.326475146957)*x[0]**o+(0.0491193078417)*x[0]+(-0.193405294452)*x[1]**o+(0.283579426128)*x[1]+(0.885442239373)*x[2]**o+(-0.838544460442)*x[2]
        arg[3,3,1,1]=(-0.502234549601)*x[0]**o+(-0.764617462736)*x[0]+(0.52701849705)*x[1]**o+(-0.423245302614)*x[1]+(0.728096320995)*x[2]**o+(0.143356081954)*x[2]
        arg[3,3,2,0]=(0.924562662058)*x[0]**o+(-0.0120799887424)*x[0]+(-0.342886608544)*x[1]**o+(0.609435784717)*x[1]+(0.392296911228)*x[2]**o+(0.572866548262)*x[2]
        arg[3,3,2,1]=(0.661267105524)*x[0]**o+(-0.71847134372)*x[0]+(-0.913866381825)*x[1]**o+(-0.502556458508)*x[1]+(0.201214639904)*x[2]**o+(-0.431939125167)*x[2]
        arg[3,4,0,0]=(0.391785002874)*x[0]**o+(0.424885787931)*x[0]+(-0.377558346944)*x[1]**o+(-0.556924666311)*x[1]+(0.511852194353)*x[2]**o+(0.514261734074)*x[2]
        arg[3,4,0,1]=(-0.269148156354)*x[0]**o+(0.692033978804)*x[0]+(-0.390604590454)*x[1]**o+(-0.551681907742)*x[1]+(0.0613720420346)*x[2]**o+(-0.508841982991)*x[2]
        arg[3,4,1,0]=(0.593170387694)*x[0]**o+(-0.00676980821753)*x[0]+(0.788564472913)*x[1]**o+(0.823094059865)*x[1]+(0.52895485971)*x[2]**o+(0.88849618397)*x[2]
        arg[3,4,1,1]=(0.772185283881)*x[0]**o+(0.759014894154)*x[0]+(-0.142739578724)*x[1]**o+(-0.141899255898)*x[1]+(0.479086662114)*x[2]**o+(0.68993908918)*x[2]
        arg[3,4,2,0]=(0.998097253401)*x[0]**o+(0.621254339576)*x[0]+(-0.692540865624)*x[1]**o+(0.716545091382)*x[1]+(0.478209703594)*x[2]**o+(0.259063088226)*x[2]
        arg[3,4,2,1]=(-0.321962318028)*x[0]**o+(0.0377531939042)*x[0]+(-0.362041758315)*x[1]**o+(0.632474956936)*x[1]+(0.133972776903)*x[2]**o+(-0.341803258375)*x[2]
        ref[0,0,0,0]=(-0.484602377547)*(1+2.*(dim-1.)/(o+1.))+(-1.0194202985)*dim
        ref[0,0,0,1]=(-0.835132633821)*(1+2.*(dim-1.)/(o+1.))+(0.616687157924)*dim
        ref[0,0,1,0]=(1.49678131808)*(1+2.*(dim-1.)/(o+1.))+(0.880777811066)*dim
        ref[0,0,1,1]=(0.910664643761)*(1+2.*(dim-1.)/(o+1.))+(-0.961299455217)*dim
        ref[0,0,2,0]=(0.946573940383)*(1+2.*(dim-1.)/(o+1.))+(0.598873680457)*dim
        ref[0,0,2,1]=(0.872725648736)*(1+2.*(dim-1.)/(o+1.))+(-1.54360766814)*dim
        ref[0,1,0,0]=(1.67226553782)*(1+2.*(dim-1.)/(o+1.))+(0.0561023038674)*dim
        ref[0,1,0,1]=(1.15582406251)*(1+2.*(dim-1.)/(o+1.))+(-0.620028515442)*dim
        ref[0,1,1,0]=(-1.79024527224)*(1+2.*(dim-1.)/(o+1.))+(0.726554075535)*dim
        ref[0,1,1,1]=(0.8477406514)*(1+2.*(dim-1.)/(o+1.))+(-1.15509819622)*dim
        ref[0,1,2,0]=(0.0209750791326)*(1+2.*(dim-1.)/(o+1.))+(-1.00141913872)*dim
        ref[0,1,2,1]=(0.569876032064)*(1+2.*(dim-1.)/(o+1.))+(0.271451611059)*dim
        ref[0,2,0,0]=(0.19197323238)*(1+2.*(dim-1.)/(o+1.))+(0.320608976525)*dim
        ref[0,2,0,1]=(-0.864025126798)*(1+2.*(dim-1.)/(o+1.))+(0.792122272387)*dim
        ref[0,2,1,0]=(-1.9943606725)*(1+2.*(dim-1.)/(o+1.))+(1.25566864927)*dim
        ref[0,2,1,1]=(-1.0628223267)*(1+2.*(dim-1.)/(o+1.))+(-1.32772813107)*dim
        ref[0,2,2,0]=(1.40455685015)*(1+2.*(dim-1.)/(o+1.))+(-1.31673180399)*dim
        ref[0,2,2,1]=(1.35132033585)*(1+2.*(dim-1.)/(o+1.))+(-1.32702265435)*dim
        ref[0,3,0,0]=(-0.949679155841)*(1+2.*(dim-1.)/(o+1.))+(-0.739650532418)*dim
        ref[0,3,0,1]=(0.47964320732)*(1+2.*(dim-1.)/(o+1.))+(0.473037089003)*dim
        ref[0,3,1,0]=(-0.432320617013)*(1+2.*(dim-1.)/(o+1.))+(0.850735495297)*dim
        ref[0,3,1,1]=(-0.780121359247)*(1+2.*(dim-1.)/(o+1.))+(-1.17524947794)*dim
        ref[0,3,2,0]=(-0.791891130098)*(1+2.*(dim-1.)/(o+1.))+(0.557003562128)*dim
        ref[0,3,2,1]=(0.614468356202)*(1+2.*(dim-1.)/(o+1.))+(-0.920106974125)*dim
        ref[0,4,0,0]=(-0.108571506377)*(1+2.*(dim-1.)/(o+1.))+(0.0716679405256)*dim
        ref[0,4,0,1]=(2.15955613929)*(1+2.*(dim-1.)/(o+1.))+(0.862657265402)*dim
        ref[0,4,1,0]=(-0.239021888652)*(1+2.*(dim-1.)/(o+1.))+(1.46771841022)*dim
        ref[0,4,1,1]=(0.596017340702)*(1+2.*(dim-1.)/(o+1.))+(-0.343059728803)*dim
        ref[0,4,2,0]=(-0.254445317045)*(1+2.*(dim-1.)/(o+1.))+(0.185808795951)*dim
        ref[0,4,2,1]=(0.520266014371)*(1+2.*(dim-1.)/(o+1.))+(-1.99456473805)*dim
        ref[1,0,0,0]=(-1.07260356292)*(1+2.*(dim-1.)/(o+1.))+(0.374273678021)*dim
        ref[1,0,0,1]=(0.300858233864)*(1+2.*(dim-1.)/(o+1.))+(-0.958946395922)*dim
        ref[1,0,1,0]=(-1.69631340214)*(1+2.*(dim-1.)/(o+1.))+(-0.096563879864)*dim
        ref[1,0,1,1]=(-1.29948775767)*(1+2.*(dim-1.)/(o+1.))+(-1.29853569383)*dim
        ref[1,0,2,0]=(2.11639869443)*(1+2.*(dim-1.)/(o+1.))+(0.783285612541)*dim
        ref[1,0,2,1]=(-0.224039580872)*(1+2.*(dim-1.)/(o+1.))+(1.35876351947)*dim
        ref[1,1,0,0]=(-1.48458285103)*(1+2.*(dim-1.)/(o+1.))+(-0.917614145542)*dim
        ref[1,1,0,1]=(0.92188880584)*(1+2.*(dim-1.)/(o+1.))+(0.569756278479)*dim
        ref[1,1,1,0]=(-0.315193818122)*(1+2.*(dim-1.)/(o+1.))+(-1.87410298298)*dim
        ref[1,1,1,1]=(-0.17620361977)*(1+2.*(dim-1.)/(o+1.))+(-0.174309770317)*dim
        ref[1,1,2,0]=(1.91602140435)*(1+2.*(dim-1.)/(o+1.))+(-0.762727403622)*dim
        ref[1,1,2,1]=(-0.481090245186)*(1+2.*(dim-1.)/(o+1.))+(0.302365878371)*dim
        ref[1,2,0,0]=(1.51144307636)*(1+2.*(dim-1.)/(o+1.))+(-0.469903943621)*dim
        ref[1,2,0,1]=(0.434474766511)*(1+2.*(dim-1.)/(o+1.))+(0.580510267262)*dim
        ref[1,2,1,0]=(-0.0359784228521)*(1+2.*(dim-1.)/(o+1.))+(1.64383448391)*dim
        ref[1,2,1,1]=(-1.01063132919)*(1+2.*(dim-1.)/(o+1.))+(1.51103090664)*dim
        ref[1,2,2,0]=(1.59952091715)*(1+2.*(dim-1.)/(o+1.))+(0.872265553532)*dim
        ref[1,2,2,1]=(1.01612465197)*(1+2.*(dim-1.)/(o+1.))+(-0.216728373319)*dim
        ref[1,3,0,0]=(-0.10531811522)*(1+2.*(dim-1.)/(o+1.))+(0.907610948542)*dim
        ref[1,3,0,1]=(0.5146706123)*(1+2.*(dim-1.)/(o+1.))+(0.877904273202)*dim
        ref[1,3,1,0]=(1.35946717224)*(1+2.*(dim-1.)/(o+1.))+(0.979242133542)*dim
        ref[1,3,1,1]=(-0.540844801432)*(1+2.*(dim-1.)/(o+1.))+(-1.01065173657)*dim
        ref[1,3,2,0]=(0.127935557915)*(1+2.*(dim-1.)/(o+1.))+(0.300209287538)*dim
        ref[1,3,2,1]=(-0.63818759413)*(1+2.*(dim-1.)/(o+1.))+(-1.20162771805)*dim
        ref[1,4,0,0]=(1.70266166649)*(1+2.*(dim-1.)/(o+1.))+(-1.27605382673)*dim
        ref[1,4,0,1]=(0.799915099484)*(1+2.*(dim-1.)/(o+1.))+(1.94853498615)*dim
        ref[1,4,1,0]=(0.916769254394)*(1+2.*(dim-1.)/(o+1.))+(-1.15016023941)*dim
        ref[1,4,1,1]=(1.87247945131)*(1+2.*(dim-1.)/(o+1.))+(0.147986167818)*dim
        ref[1,4,2,0]=(-0.988343334946)*(1+2.*(dim-1.)/(o+1.))+(0.169122301365)*dim
        ref[1,4,2,1]=(0.116889419423)*(1+2.*(dim-1.)/(o+1.))+(0.191358932597)*dim
        ref[2,0,0,0]=(-0.0479378392653)*(1+2.*(dim-1.)/(o+1.))+(0.978042254211)*dim
        ref[2,0,0,1]=(-0.485032392631)*(1+2.*(dim-1.)/(o+1.))+(-0.351118421752)*dim
        ref[2,0,1,0]=(0.23510362002)*(1+2.*(dim-1.)/(o+1.))+(-0.577849423671)*dim
        ref[2,0,1,1]=(-0.068175322817)*(1+2.*(dim-1.)/(o+1.))+(1.94609819864)*dim
        ref[2,0,2,0]=(-1.41002268435)*(1+2.*(dim-1.)/(o+1.))+(0.98455613352)*dim
        ref[2,0,2,1]=(-2.24634010969)*(1+2.*(dim-1.)/(o+1.))+(1.0410712698)*dim
        ref[2,1,0,0]=(-0.860551277441)*(1+2.*(dim-1.)/(o+1.))+(0.663495620068)*dim
        ref[2,1,0,1]=(-0.944426383871)*(1+2.*(dim-1.)/(o+1.))+(-0.341434032481)*dim
        ref[2,1,1,0]=(-1.16050351389)*(1+2.*(dim-1.)/(o+1.))+(0.902358791288)*dim
        ref[2,1,1,1]=(-0.790421314781)*(1+2.*(dim-1.)/(o+1.))+(0.618190806416)*dim
        ref[2,1,2,0]=(0.456001174897)*(1+2.*(dim-1.)/(o+1.))+(0.782682949672)*dim
        ref[2,1,2,1]=(0.532545234059)*(1+2.*(dim-1.)/(o+1.))+(0.806733163549)*dim
        ref[2,2,0,0]=(-1.63759186587)*(1+2.*(dim-1.)/(o+1.))+(0.275350925359)*dim
        ref[2,2,0,1]=(1.26997954169)*(1+2.*(dim-1.)/(o+1.))+(0.998155503083)*dim
        ref[2,2,1,0]=(1.22159383643)*(1+2.*(dim-1.)/(o+1.))+(-1.48657634495)*dim
        ref[2,2,1,1]=(-0.537844025016)*(1+2.*(dim-1.)/(o+1.))+(-1.38528909924)*dim
        ref[2,2,2,0]=(-0.274283519122)*(1+2.*(dim-1.)/(o+1.))+(0.243186764056)*dim
        ref[2,2,2,1]=(-1.10324559047)*(1+2.*(dim-1.)/(o+1.))+(0.0644794177698)*dim
        ref[2,3,0,0]=(0.571128036949)*(1+2.*(dim-1.)/(o+1.))+(0.60918412053)*dim
        ref[2,3,0,1]=(-0.580144876629)*(1+2.*(dim-1.)/(o+1.))+(-0.667332031982)*dim
        ref[2,3,1,0]=(-1.66177914705)*(1+2.*(dim-1.)/(o+1.))+(0.675143894185)*dim
        ref[2,3,1,1]=(0.301151474869)*(1+2.*(dim-1.)/(o+1.))+(-1.7478009387)*dim
        ref[2,3,2,0]=(-0.917598158317)*(1+2.*(dim-1.)/(o+1.))+(-0.0850570760611)*dim
        ref[2,3,2,1]=(-0.09025993235)*(1+2.*(dim-1.)/(o+1.))+(0.798689188243)*dim
        ref[2,4,0,0]=(1.94428451282)*(1+2.*(dim-1.)/(o+1.))+(0.247560933384)*dim
        ref[2,4,0,1]=(0.984775124785)*(1+2.*(dim-1.)/(o+1.))+(-0.506710524219)*dim
        ref[2,4,1,0]=(-0.197674459012)*(1+2.*(dim-1.)/(o+1.))+(0.0107873743289)*dim
        ref[2,4,1,1]=(-0.656458978046)*(1+2.*(dim-1.)/(o+1.))+(-1.13791264993)*dim
        ref[2,4,2,0]=(-0.778638912279)*(1+2.*(dim-1.)/(o+1.))+(0.324022275097)*dim
        ref[2,4,2,1]=(0.940510770447)*(1+2.*(dim-1.)/(o+1.))+(0.901184637171)*dim
        ref[3,0,0,0]=(-1.25429061263)*(1+2.*(dim-1.)/(o+1.))+(0.389376608226)*dim
        ref[3,0,0,1]=(-0.0171420806973)*(1+2.*(dim-1.)/(o+1.))+(0.230944451397)*dim
        ref[3,0,1,0]=(0.456140891562)*(1+2.*(dim-1.)/(o+1.))+(1.37768003626)*dim
        ref[3,0,1,1]=(-0.260435871091)*(1+2.*(dim-1.)/(o+1.))+(-1.10263475358)*dim
        ref[3,0,2,0]=(0.0625076184349)*(1+2.*(dim-1.)/(o+1.))+(1.51121135544)*dim
        ref[3,0,2,1]=(-0.973347700893)*(1+2.*(dim-1.)/(o+1.))+(-2.10909307795)*dim
        ref[3,1,0,0]=(1.04357815056)*(1+2.*(dim-1.)/(o+1.))+(-0.516720098176)*dim
        ref[3,1,0,1]=(-0.400651374046)*(1+2.*(dim-1.)/(o+1.))+(0.37941138127)*dim
        ref[3,1,1,0]=(0.614177886412)*(1+2.*(dim-1.)/(o+1.))+(-0.313511377498)*dim
        ref[3,1,1,1]=(0.97787134432)*(1+2.*(dim-1.)/(o+1.))+(-1.28527683733)*dim
        ref[3,1,2,0]=(0.0702273162506)*(1+2.*(dim-1.)/(o+1.))+(-1.32447006034)*dim
        ref[3,1,2,1]=(-0.596202948193)*(1+2.*(dim-1.)/(o+1.))+(-0.648046584161)*dim
        ref[3,2,0,0]=(0.34378369699)*(1+2.*(dim-1.)/(o+1.))+(-1.49317727825)*dim
        ref[3,2,0,1]=(0.798530712175)*(1+2.*(dim-1.)/(o+1.))+(0.358905794706)*dim
        ref[3,2,1,0]=(-0.808586932417)*(1+2.*(dim-1.)/(o+1.))+(-1.66927503333)*dim
        ref[3,2,1,1]=(0.712768620002)*(1+2.*(dim-1.)/(o+1.))+(0.346136808876)*dim
        ref[3,2,2,0]=(0.0778075096053)*(1+2.*(dim-1.)/(o+1.))+(-1.92918285909)*dim
        ref[3,2,2,1]=(0.949967126178)*(1+2.*(dim-1.)/(o+1.))+(0.650598700321)*dim
        ref[3,3,0,0]=(0.496438920934)*(1+2.*(dim-1.)/(o+1.))+(-0.0508751252546)*dim
        ref[3,3,0,1]=(0.559728225123)*(1+2.*(dim-1.)/(o+1.))+(-1.27062926979)*dim
        ref[3,3,1,0]=(0.365561797964)*(1+2.*(dim-1.)/(o+1.))+(-0.505845726473)*dim
        ref[3,3,1,1]=(0.752880268444)*(1+2.*(dim-1.)/(o+1.))+(-1.0445066834)*dim
        ref[3,3,2,0]=(0.973972964742)*(1+2.*(dim-1.)/(o+1.))+(1.17022234424)*dim
        ref[3,3,2,1]=(-0.0513846363967)*(1+2.*(dim-1.)/(o+1.))+(-1.65296692739)*dim
        ref[3,4,0,0]=(0.526078850284)*(1+2.*(dim-1.)/(o+1.))+(0.382222855695)*dim
        ref[3,4,0,1]=(-0.598380704773)*(1+2.*(dim-1.)/(o+1.))+(-0.368489911929)*dim
        ref[3,4,1,0]=(1.91068972032)*(1+2.*(dim-1.)/(o+1.))+(1.70482043562)*dim
        ref[3,4,1,1]=(1.10853236727)*(1+2.*(dim-1.)/(o+1.))+(1.30705472744)*dim
        ref[3,4,2,0]=(0.78376609137)*(1+2.*(dim-1.)/(o+1.))+(1.59686251918)*dim
        ref[3,4,2,1]=(-0.550031299439)*(1+2.*(dim-1.)/(o+1.))+(0.328424892465)*dim
      res=integrate(arg)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5, 3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunctionOnBoundary_fromData_ContinuousFunction_rank0(self):
      """
      tests integral of rank 0 Data on the ReducedFunctionOnBoundary

      assumptions: ContinuousFunction(self.domain) exists
                   self.domain supports integral on ReducedFunctionOnBoundary
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=ReducedFunctionOnBoundary(self.domain)
      w=ContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(),w)
      if dim==2:
        arg=(0.506621877215)*x[0]**o+(-0.45560383685)*x[0]+(-0.598137301806)*x[1]**o+(0.784241595615)*x[1]
        ref=(-0.0915154245913)*(1+2.*(dim-1.)/(o+1.))+(0.328637758765)*dim
      else:
        arg=(-0.505489131484)*x[0]**o+(0.58139159259)*x[0]+(0.458086828888)*x[1]**o+(0.549090746672)*x[1]+(-0.0882325936167)*x[2]**o+(0.0170295954808)*x[2]
        ref=(-0.135634896213)*(1+2.*(dim-1.)/(o+1.))+(1.14751193474)*dim
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunctionOnBoundary_fromData_ContinuousFunction_rank1(self):
      """
      tests integral of rank 1 Data on the ReducedFunctionOnBoundary

      assumptions: ContinuousFunction(self.domain) exists
                   self.domain supports integral on ReducedFunctionOnBoundary
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=ReducedFunctionOnBoundary(self.domain)
      w=ContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(2,),w)
      ref=numpy.zeros((2,),numpy.float_)
      if dim==2:
        arg[0]=(-0.88779433873)*x[0]**o+(0.507955012301)*x[0]+(0.652739274924)*x[1]**o+(-0.971975755989)*x[1]
        arg[1]=(-0.0812690931139)*x[0]**o+(-0.283880962354)*x[0]+(0.094904517439)*x[1]**o+(-0.940528164353)*x[1]
        ref[0]=(-0.235055063806)*(1+2.*(dim-1.)/(o+1.))+(-0.464020743688)*dim
        ref[1]=(0.0136354243251)*(1+2.*(dim-1.)/(o+1.))+(-1.22440912671)*dim
      else:
        arg[0]=(-0.506412384389)*x[0]**o+(-0.465794405394)*x[0]+(0.346977499982)*x[1]**o+(0.386181176987)*x[1]+(-0.476729572397)*x[2]**o+(-0.956945933508)*x[2]
        arg[1]=(0.267092807416)*x[0]**o+(-0.54211155802)*x[0]+(-0.27815411727)*x[1]**o+(-0.984542450102)*x[1]+(0.656389296076)*x[2]**o+(0.0395863971401)*x[2]
        ref[0]=(-0.636164456804)*(1+2.*(dim-1.)/(o+1.))+(-1.03655916191)*dim
        ref[1]=(0.645327986222)*(1+2.*(dim-1.)/(o+1.))+(-1.48706761098)*dim
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunctionOnBoundary_fromData_ContinuousFunction_rank2(self):
      """
      tests integral of rank 2 Data on the ReducedFunctionOnBoundary

      assumptions: ContinuousFunction(self.domain) exists
                   self.domain supports integral on ReducedFunctionOnBoundary
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=ReducedFunctionOnBoundary(self.domain)
      w=ContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5),w)
      ref=numpy.zeros((4, 5),numpy.float_)
      if dim==2:
        arg[0,0]=(0.0607636892262)*x[0]**o+(0.740866012668)*x[0]+(0.145223136928)*x[1]**o+(-0.738387773526)*x[1]
        arg[0,1]=(0.0292598484511)*x[0]**o+(-0.189208290889)*x[0]+(0.902021405165)*x[1]**o+(0.423106545559)*x[1]
        arg[0,2]=(0.519658217226)*x[0]**o+(-0.768406018136)*x[0]+(-0.189532527097)*x[1]**o+(0.256417429833)*x[1]
        arg[0,3]=(-0.565327663481)*x[0]**o+(0.876762003673)*x[0]+(-0.243804758063)*x[1]**o+(0.399476036454)*x[1]
        arg[0,4]=(0.0632252409818)*x[0]**o+(0.0951130662538)*x[0]+(0.185662425763)*x[1]**o+(0.907446482506)*x[1]
        arg[1,0]=(-0.430504584794)*x[0]**o+(0.864814314023)*x[0]+(0.47460095501)*x[1]**o+(-0.599679142188)*x[1]
        arg[1,1]=(0.699618175298)*x[0]**o+(-0.561782605517)*x[0]+(-0.304250865015)*x[1]**o+(-0.341836776795)*x[1]
        arg[1,2]=(0.856956767316)*x[0]**o+(0.610026903213)*x[0]+(-0.666749954599)*x[1]**o+(0.885388885234)*x[1]
        arg[1,3]=(0.699377007631)*x[0]**o+(0.699906750287)*x[0]+(0.486690669414)*x[1]**o+(0.167425901549)*x[1]
        arg[1,4]=(0.466701415896)*x[0]**o+(-0.0719752759126)*x[0]+(0.795662047062)*x[1]**o+(0.979862503926)*x[1]
        arg[2,0]=(-0.912218732612)*x[0]**o+(0.950990472894)*x[0]+(-0.527999708902)*x[1]**o+(0.695735593043)*x[1]
        arg[2,1]=(-0.223468270048)*x[0]**o+(-0.882565147673)*x[0]+(0.750516681645)*x[1]**o+(-0.652794590093)*x[1]
        arg[2,2]=(0.897646682412)*x[0]**o+(-0.815401214058)*x[0]+(0.629680741327)*x[1]**o+(-0.9716988156)*x[1]
        arg[2,3]=(0.279739517524)*x[0]**o+(-0.0859797206016)*x[0]+(-0.0454325622757)*x[1]**o+(0.301214753868)*x[1]
        arg[2,4]=(-0.358056827092)*x[0]**o+(0.254207762658)*x[0]+(0.831499219864)*x[1]**o+(0.0930388028811)*x[1]
        arg[3,0]=(0.0761520020047)*x[0]**o+(0.650696394486)*x[0]+(-0.132369660678)*x[1]**o+(-0.321220385971)*x[1]
        arg[3,1]=(0.238715231585)*x[0]**o+(-0.297219391083)*x[0]+(-0.15446552614)*x[1]**o+(0.860560615056)*x[1]
        arg[3,2]=(0.708864754538)*x[0]**o+(-0.221186702958)*x[0]+(-0.478127729891)*x[1]**o+(0.0309604412212)*x[1]
        arg[3,3]=(0.362909366479)*x[0]**o+(0.382279106813)*x[0]+(0.827905062086)*x[1]**o+(-0.303048355949)*x[1]
        arg[3,4]=(0.573736946844)*x[0]**o+(0.645059764766)*x[0]+(0.301945383678)*x[1]**o+(-0.103083649994)*x[1]
        ref[0,0]=(0.205986826154)*(1+2.*(dim-1.)/(o+1.))+(0.0024782391428)*dim
        ref[0,1]=(0.931281253617)*(1+2.*(dim-1.)/(o+1.))+(0.23389825467)*dim
        ref[0,2]=(0.330125690129)*(1+2.*(dim-1.)/(o+1.))+(-0.511988588303)*dim
        ref[0,3]=(-0.809132421544)*(1+2.*(dim-1.)/(o+1.))+(1.27623804013)*dim
        ref[0,4]=(0.248887666745)*(1+2.*(dim-1.)/(o+1.))+(1.00255954876)*dim
        ref[1,0]=(0.044096370216)*(1+2.*(dim-1.)/(o+1.))+(0.265135171835)*dim
        ref[1,1]=(0.395367310283)*(1+2.*(dim-1.)/(o+1.))+(-0.903619382312)*dim
        ref[1,2]=(0.190206812717)*(1+2.*(dim-1.)/(o+1.))+(1.49541578845)*dim
        ref[1,3]=(1.18606767705)*(1+2.*(dim-1.)/(o+1.))+(0.867332651836)*dim
        ref[1,4]=(1.26236346296)*(1+2.*(dim-1.)/(o+1.))+(0.907887228013)*dim
        ref[2,0]=(-1.44021844151)*(1+2.*(dim-1.)/(o+1.))+(1.64672606594)*dim
        ref[2,1]=(0.527048411596)*(1+2.*(dim-1.)/(o+1.))+(-1.53535973777)*dim
        ref[2,2]=(1.52732742374)*(1+2.*(dim-1.)/(o+1.))+(-1.78710002966)*dim
        ref[2,3]=(0.234306955249)*(1+2.*(dim-1.)/(o+1.))+(0.215235033266)*dim
        ref[2,4]=(0.473442392772)*(1+2.*(dim-1.)/(o+1.))+(0.34724656554)*dim
        ref[3,0]=(-0.0562176586729)*(1+2.*(dim-1.)/(o+1.))+(0.329476008516)*dim
        ref[3,1]=(0.0842497054449)*(1+2.*(dim-1.)/(o+1.))+(0.563341223973)*dim
        ref[3,2]=(0.230737024647)*(1+2.*(dim-1.)/(o+1.))+(-0.190226261737)*dim
        ref[3,3]=(1.19081442856)*(1+2.*(dim-1.)/(o+1.))+(0.079230750864)*dim
        ref[3,4]=(0.875682330522)*(1+2.*(dim-1.)/(o+1.))+(0.541976114772)*dim
      else:
        arg[0,0]=(0.0640108068236)*x[0]**o+(0.0322830166043)*x[0]+(-0.703691335255)*x[1]**o+(-0.185228770711)*x[1]+(-0.738714166313)*x[2]**o+(-0.423576647077)*x[2]
        arg[0,1]=(-0.738341351945)*x[0]**o+(0.396306395436)*x[0]+(-0.369974848568)*x[1]**o+(0.750933208783)*x[1]+(0.554069869127)*x[2]**o+(0.634275493051)*x[2]
        arg[0,2]=(-0.259193248708)*x[0]**o+(-0.900007449583)*x[0]+(0.894379436915)*x[1]**o+(0.997814320752)*x[1]+(0.233192116622)*x[2]**o+(0.986616259649)*x[2]
        arg[0,3]=(0.381480250594)*x[0]**o+(0.734823219057)*x[0]+(0.569912440169)*x[1]**o+(-0.441084504695)*x[1]+(0.312539701584)*x[2]**o+(-0.3733004974)*x[2]
        arg[0,4]=(-0.779059807622)*x[0]**o+(0.538777922322)*x[0]+(-0.0695879211612)*x[1]**o+(-0.204862216499)*x[1]+(-0.0109698525672)*x[2]**o+(0.981875117617)*x[2]
        arg[1,0]=(-0.415857630044)*x[0]**o+(-0.356607866913)*x[0]+(-0.628854326903)*x[1]**o+(0.0401233333505)*x[1]+(0.687470169579)*x[2]**o+(-0.565349753114)*x[2]
        arg[1,1]=(-0.438364910402)*x[0]**o+(0.498078194367)*x[0]+(-0.120729597948)*x[1]**o+(-0.622076117713)*x[1]+(0.995130029172)*x[2]**o+(-0.7234601064)*x[2]
        arg[1,2]=(-0.957869585558)*x[0]**o+(0.374693219824)*x[0]+(0.0355192120972)*x[1]**o+(-0.516770140651)*x[1]+(0.521887953791)*x[2]**o+(0.00771564236502)*x[2]
        arg[1,3]=(-0.0495580194062)*x[0]**o+(-0.183895326318)*x[0]+(-0.55304086303)*x[1]**o+(0.920204449879)*x[1]+(0.143165855157)*x[2]**o+(-0.773684620462)*x[2]
        arg[1,4]=(0.795913131087)*x[0]**o+(0.653826129199)*x[0]+(-0.585562760813)*x[1]**o+(0.274083108918)*x[1]+(-0.165275654116)*x[2]**o+(0.358678758212)*x[2]
        arg[2,0]=(-0.0699685857843)*x[0]**o+(-0.42922435931)*x[0]+(-0.0494308981938)*x[1]**o+(-0.956597829176)*x[1]+(-0.912941305307)*x[2]**o+(0.214777332562)*x[2]
        arg[2,1]=(-0.900983338173)*x[0]**o+(-0.779729884378)*x[0]+(-0.860100306508)*x[1]**o+(0.14211537281)*x[1]+(-0.984439938285)*x[2]**o+(-0.549987779842)*x[2]
        arg[2,2]=(0.801489512389)*x[0]**o+(0.896468923826)*x[0]+(-0.155737972887)*x[1]**o+(0.639864330477)*x[1]+(0.798598705038)*x[2]**o+(-0.863019041525)*x[2]
        arg[2,3]=(-0.453612136504)*x[0]**o+(0.522662491688)*x[0]+(-0.463058076486)*x[1]**o+(0.398018809499)*x[1]+(-0.612044961611)*x[2]**o+(-0.481951883217)*x[2]
        arg[2,4]=(-0.0739336519425)*x[0]**o+(-0.241753740154)*x[0]+(0.0176292058898)*x[1]**o+(0.435641132809)*x[1]+(-0.0592828584976)*x[2]**o+(0.430115620801)*x[2]
        arg[3,0]=(0.604784358152)*x[0]**o+(-0.817214179503)*x[0]+(-0.423104272008)*x[1]**o+(0.287787881773)*x[1]+(0.678036469241)*x[2]**o+(0.578347129563)*x[2]
        arg[3,1]=(0.362390778598)*x[0]**o+(-0.389066580119)*x[0]+(-0.934087580396)*x[1]**o+(0.907587485226)*x[1]+(-0.902083043406)*x[2]**o+(0.904707538314)*x[2]
        arg[3,2]=(0.862463207465)*x[0]**o+(0.904923219336)*x[0]+(0.421361696042)*x[1]**o+(-0.779329196171)*x[1]+(0.059521522394)*x[2]**o+(-0.744510067387)*x[2]
        arg[3,3]=(0.412029226851)*x[0]**o+(-0.649730143511)*x[0]+(0.579628017001)*x[1]**o+(-0.949180147129)*x[1]+(-0.073016492309)*x[2]**o+(0.666554170757)*x[2]
        arg[3,4]=(-0.700401396504)*x[0]**o+(-0.89194120812)*x[0]+(0.592387042213)*x[1]**o+(-0.906632787891)*x[1]+(-0.486072860812)*x[2]**o+(-0.505386176523)*x[2]
        ref[0,0]=(-1.37839469474)*(1+2.*(dim-1.)/(o+1.))+(-0.576522401184)*dim
        ref[0,1]=(-0.554246331386)*(1+2.*(dim-1.)/(o+1.))+(1.78151509727)*dim
        ref[0,2]=(0.868378304829)*(1+2.*(dim-1.)/(o+1.))+(1.08442313082)*dim
        ref[0,3]=(1.26393239235)*(1+2.*(dim-1.)/(o+1.))+(-0.0795617830385)*dim
        ref[0,4]=(-0.85961758135)*(1+2.*(dim-1.)/(o+1.))+(1.31579082344)*dim
        ref[1,0]=(-0.357241787368)*(1+2.*(dim-1.)/(o+1.))+(-0.881834286676)*dim
        ref[1,1]=(0.436035520821)*(1+2.*(dim-1.)/(o+1.))+(-0.847458029746)*dim
        ref[1,2]=(-0.40046241967)*(1+2.*(dim-1.)/(o+1.))+(-0.134361278462)*dim
        ref[1,3]=(-0.459433027279)*(1+2.*(dim-1.)/(o+1.))+(-0.0373754969011)*dim
        ref[1,4]=(0.0450747161574)*(1+2.*(dim-1.)/(o+1.))+(1.28658799633)*dim
        ref[2,0]=(-1.03234078929)*(1+2.*(dim-1.)/(o+1.))+(-1.17104485592)*dim
        ref[2,1]=(-2.74552358297)*(1+2.*(dim-1.)/(o+1.))+(-1.18760229141)*dim
        ref[2,2]=(1.44435024454)*(1+2.*(dim-1.)/(o+1.))+(0.673314212777)*dim
        ref[2,3]=(-1.5287151746)*(1+2.*(dim-1.)/(o+1.))+(0.43872941797)*dim
        ref[2,4]=(-0.11558730455)*(1+2.*(dim-1.)/(o+1.))+(0.624003013456)*dim
        ref[3,0]=(0.859716555385)*(1+2.*(dim-1.)/(o+1.))+(0.0489208318321)*dim
        ref[3,1]=(-1.4737798452)*(1+2.*(dim-1.)/(o+1.))+(1.42322844342)*dim
        ref[3,2]=(1.3433464259)*(1+2.*(dim-1.)/(o+1.))+(-0.618916044222)*dim
        ref[3,3]=(0.918640751542)*(1+2.*(dim-1.)/(o+1.))+(-0.932356119883)*dim
        ref[3,4]=(-0.594087215103)*(1+2.*(dim-1.)/(o+1.))+(-2.30396017253)*dim
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunctionOnBoundary_fromData_ContinuousFunction_rank3(self):
      """
      tests integral of rank 3 Data on the ReducedFunctionOnBoundary

      assumptions: ContinuousFunction(self.domain) exists
                   self.domain supports integral on ReducedFunctionOnBoundary
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=ReducedFunctionOnBoundary(self.domain)
      w=ContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(6, 2, 2),w)
      ref=numpy.zeros((6, 2, 2),numpy.float_)
      if dim==2:
        arg[0,0,0]=(0.662707930283)*x[0]**o+(0.625512077969)*x[0]+(-0.222208840922)*x[1]**o+(0.374614317804)*x[1]
        arg[0,0,1]=(0.685952252808)*x[0]**o+(-0.88219802797)*x[0]+(-0.948918831634)*x[1]**o+(-0.00615202896776)*x[1]
        arg[0,1,0]=(-0.555231206657)*x[0]**o+(0.997112425959)*x[0]+(-0.186247870511)*x[1]**o+(-0.0658892046229)*x[1]
        arg[0,1,1]=(-0.735288285178)*x[0]**o+(0.941234848614)*x[0]+(-0.307761737256)*x[1]**o+(0.933293069433)*x[1]
        arg[1,0,0]=(-0.618191337825)*x[0]**o+(-0.12930895166)*x[0]+(0.76213325083)*x[1]**o+(-0.744049403156)*x[1]
        arg[1,0,1]=(-0.793412570981)*x[0]**o+(0.533175323401)*x[0]+(-0.87398581932)*x[1]**o+(-0.279651754725)*x[1]
        arg[1,1,0]=(-0.0654077907694)*x[0]**o+(0.239040458488)*x[0]+(0.730639927837)*x[1]**o+(0.805020533329)*x[1]
        arg[1,1,1]=(-0.560195748542)*x[0]**o+(-0.265389430135)*x[0]+(-0.295142324741)*x[1]**o+(-0.929699771024)*x[1]
        arg[2,0,0]=(0.935760865577)*x[0]**o+(-0.332458843131)*x[0]+(0.257302695187)*x[1]**o+(-0.347547927033)*x[1]
        arg[2,0,1]=(-0.245770653276)*x[0]**o+(0.126945988409)*x[0]+(0.302793434995)*x[1]**o+(-0.997483200172)*x[1]
        arg[2,1,0]=(0.523494364951)*x[0]**o+(-0.323782740449)*x[0]+(-0.563752593454)*x[1]**o+(-0.358325752838)*x[1]
        arg[2,1,1]=(-0.506330114515)*x[0]**o+(-0.733234050567)*x[0]+(0.734774667778)*x[1]**o+(0.795264566776)*x[1]
        arg[3,0,0]=(-0.21258295235)*x[0]**o+(-0.929331330718)*x[0]+(-0.546061847742)*x[1]**o+(0.0613931491065)*x[1]
        arg[3,0,1]=(-0.357847946656)*x[0]**o+(-0.133795684508)*x[0]+(-0.796646028688)*x[1]**o+(0.875795801429)*x[1]
        arg[3,1,0]=(0.923717804039)*x[0]**o+(-0.176266633621)*x[0]+(-0.605892867534)*x[1]**o+(0.633377612365)*x[1]
        arg[3,1,1]=(0.729025508129)*x[0]**o+(0.412034955257)*x[0]+(0.0933054864485)*x[1]**o+(0.386019085532)*x[1]
        arg[4,0,0]=(-0.0705524918008)*x[0]**o+(-0.869605321304)*x[0]+(0.244351267755)*x[1]**o+(0.898883388102)*x[1]
        arg[4,0,1]=(-0.453297333327)*x[0]**o+(-0.67824185123)*x[0]+(0.0497322840245)*x[1]**o+(0.581677879069)*x[1]
        arg[4,1,0]=(0.343975006358)*x[0]**o+(0.724084357103)*x[0]+(-0.186292335362)*x[1]**o+(-0.257997075018)*x[1]
        arg[4,1,1]=(-0.389731923798)*x[0]**o+(0.0949155578616)*x[0]+(-0.298428138752)*x[1]**o+(0.819510069666)*x[1]
        arg[5,0,0]=(0.827670169513)*x[0]**o+(-0.584607978668)*x[0]+(0.669056948658)*x[1]**o+(-0.393158541985)*x[1]
        arg[5,0,1]=(-0.471551968206)*x[0]**o+(-0.339964831396)*x[0]+(0.354554393554)*x[1]**o+(-0.334773591957)*x[1]
        arg[5,1,0]=(0.113562904864)*x[0]**o+(-0.775045505123)*x[0]+(0.783044874931)*x[1]**o+(0.52265843873)*x[1]
        arg[5,1,1]=(-0.115476036205)*x[0]**o+(-0.290822012728)*x[0]+(-0.398510477854)*x[1]**o+(0.398737205542)*x[1]
        ref[0,0,0]=(0.440499089362)*(1+2.*(dim-1.)/(o+1.))+(1.00012639577)*dim
        ref[0,0,1]=(-0.262966578826)*(1+2.*(dim-1.)/(o+1.))+(-0.888350056938)*dim
        ref[0,1,0]=(-0.741479077167)*(1+2.*(dim-1.)/(o+1.))+(0.931223221336)*dim
        ref[0,1,1]=(-1.04305002243)*(1+2.*(dim-1.)/(o+1.))+(1.87452791805)*dim
        ref[1,0,0]=(0.143941913005)*(1+2.*(dim-1.)/(o+1.))+(-0.873358354816)*dim
        ref[1,0,1]=(-1.6673983903)*(1+2.*(dim-1.)/(o+1.))+(0.253523568676)*dim
        ref[1,1,0]=(0.665232137068)*(1+2.*(dim-1.)/(o+1.))+(1.04406099182)*dim
        ref[1,1,1]=(-0.855338073283)*(1+2.*(dim-1.)/(o+1.))+(-1.19508920116)*dim
        ref[2,0,0]=(1.19306356076)*(1+2.*(dim-1.)/(o+1.))+(-0.680006770164)*dim
        ref[2,0,1]=(0.0570227817187)*(1+2.*(dim-1.)/(o+1.))+(-0.870537211764)*dim
        ref[2,1,0]=(-0.0402582285028)*(1+2.*(dim-1.)/(o+1.))+(-0.682108493287)*dim
        ref[2,1,1]=(0.228444553263)*(1+2.*(dim-1.)/(o+1.))+(0.0620305162089)*dim
        ref[3,0,0]=(-0.758644800093)*(1+2.*(dim-1.)/(o+1.))+(-0.867938181611)*dim
        ref[3,0,1]=(-1.15449397534)*(1+2.*(dim-1.)/(o+1.))+(0.742000116921)*dim
        ref[3,1,0]=(0.317824936505)*(1+2.*(dim-1.)/(o+1.))+(0.457110978744)*dim
        ref[3,1,1]=(0.822330994578)*(1+2.*(dim-1.)/(o+1.))+(0.798054040789)*dim
        ref[4,0,0]=(0.173798775954)*(1+2.*(dim-1.)/(o+1.))+(0.0292780667983)*dim
        ref[4,0,1]=(-0.403565049303)*(1+2.*(dim-1.)/(o+1.))+(-0.0965639721601)*dim
        ref[4,1,0]=(0.157682670995)*(1+2.*(dim-1.)/(o+