/* KInterbasDB Python Module
**
** Version 3.0.1
**
** The following contributors hold Copyright (C) over their respective
** portions of code (see license.txt for details):
**
** [Original Author (maintained through version 2.0-0.3.1):]
**   1998-2001 [alex]  Alexander Kuznetsov   <alexan@users.sourceforge.net>
** [Maintainers (after version 2.0-0.3.1):]
**   2001-2002 [maz]   Marek Isalski         <kinterbasdb@maz.nu>
**   2002      [dsr]   David Rushby          <woodsplitter@rocketmail.com>
** [Contributors:]
**   2001      [eac]   Evgeny A. Cherkashin  <eugeneai@icc.ru>
**   2001-2002 [janez] Janez Jere            <janez.jere@void.si>
*/

/* NOTE TO THE ADVENTUROUS:
** This C code is the underlying infrastructure of the Python module named
** kinterbasdb.  Structure, function, and variable declarations made herein are
** not to be mistaken for an interface on which third parties can rely.
**
** Instead, rely on the public interface of the Python package named
** kinterbasdb (see __init__.py), which strives to be compliant with the Python
** DB API Specification 2.0.  See the kinterbasdb documentation (docs dir) for
** more information about the public interface.
**
** Crack monkeys should consider themselves warned.
*/

#define KIDB_HOME_PAGE "http://sourceforge.net/projects/kinterbasdb"
#define KIDB_REPORT " -- please report this to the developers at "

/* DSR: begin block 2002.02.22 */
/* The standard guide to Embedding and Extending says: "Since Python may
** define some pre-processor definitions which affect the standard headers
** on some systems, you must include Python.h before any standard headers
** are included."
*/
#include "Python.h"
/* DSR: end block 2002.02.22 */

#include <time.h>
#include <math.h>

#include "ibase.h"
#include "mxDateTime.h"

/* DSR:2002.07.10:begin block */
#ifndef FOR_ZOPE
  #define DETERMINE_FIELD_PRECISION
#endif
/* DSR:2002.07.10:end block */

/* DSR:2002.06.21:572326:begin block */
#define PYTHON_2_2_OR_LATER (PY_VERSION_HEX >= 0x020200F0)
/* DSR:2002.06.21:572326:end block */

/* DSR:2002.04.13:begin block */
#ifndef LONG_LONG_MIN
  #ifdef LLONG_MIN
    #define LONG_LONG_MIN LLONG_MIN
    #define LONG_LONG_MAX LLONG_MAX
  #else
    #ifdef _I64_MIN
      #define LONG_LONG_MIN _I64_MIN
      #define LONG_LONG_MAX _I64_MAX
    #endif
  #endif
#endif
/* DSR:2002.04.13:end block */

/* DSR: begin block 2002.02.17 */
/* Macro redefinitions for Python 1.5.2 compatibility: */
#if (PY_VERSION_HEX <= 0x010502F0)
  #define PyObject_New PyObject_NEW
  #define PyObject_Del(x) PyMem_Free( (char*) x)
  #define PySequence_Size PySequence_Length
#endif /* PYTHON VERSION DETECTION */
/* DSR: end block 2002.02.17 */

/* DSR:2002.02.15: Detect Interbase 6 or later (including Firebird). */
#if (defined(SQL_DIALECT_CURRENT) && defined(SQL_DIALECT_V6))
  #define INTERBASE6_OR_LATER
  #define SQL_DIALECT_DEFAULT SQL_DIALECT_CURRENT
  #define SQL_DIALECT_OLDEST SQL_DIALECT_V5
#else
  #define SQL_DIALECT_DEFAULT 1
  #define SQL_DIALECT_OLDEST 1
  /* DSR:2002.07.18:begin block */
  /* IB < 6.0 had less differentiated datetime types; SQL_TIMESTAMP is more
  ** representative of what columns called SQL_DATE actually contain. */
  #define SQL_TIMESTAMP SQL_DATE
  /* DSR:2002.07.18:end block */
#endif


/******************** COMPILATION OPTIONS:BEGIN ********************/

/*
** Define KIDB_DEBUGGERING to get lots of debugging output to stderr.
** (Maz renamed this #define because of the possibility of lots of things
** clashing with "DEBUG".)
*/
/* #define KIDB_DEBUGGERING */

/******************** COMPILATION OPTIONS:END ********************/

/******************** HARD-CODED LIMITS:BEGIN ********************/

/* INITIAL_SQLVAR_CAPACITY is the maximum number of XSQLVARs that a freshly
** allocated XSQLDA is configured to hold.  If the initial capacity proves
** insufficient, more space is allocated automatically.  Therefore,
** INITIAL_SQLVAR_CAPACITY is really an "optimization hint" rather than a
** "hard-coded limit". */
#define INITIAL_SQLVAR_CAPACITY        100

/* MAX_BLOB_SEGMENT_SIZE is an IB/Firebird engine constraint, not something
** we could overcome here in kinterbasdb.  For that matter, it's not a
** "constraint" in any meaningful sense, since a single blob can have many
** segments, and kinterbasdb manages the segmentation transparently. */
#define MAX_BLOB_SEGMENT_SIZE 65535

/* DSR:2002.02.15:
** I went through and tried to extract all hard-coded buffer size
** specifications to the HARD-CODED LIMITS section.
** Generally, the respective sizes I chose are those used by the
** Interbase 6 API Guide, though it does not officially define the sizes,
** which leaves the looming specter of buffer overflows, as maz has noted
** with "X_X:maz: possible buffer overflow..." or something of the sort.
*/

#define MAX_ISC_ERROR_MESSAGE_BUFFER_SIZE 512

#define STATUS_VECTOR_SIZE 20

#define DPB_BUFFER_SIZE 256

#define ISC_INFO_BUFFER_SIZE 20

/* DSR:2002.06.17:begin block */
#define ISC_INFO_RESULT_BUFFER_SIZE 2048

#define GENERAL_UTILITY_BUFFER_SIZE 1024
/* DSR:2002.06.17:end block */

/******************** HARD-CODED LIMITS:END ********************/

/******************** HANDY ABBREVIATIONS:BEGIN ********************/

/* DSR:2002.03.14:
** I replaced all multiline macros with their single-line equivalents
** for compatibility with some older GCC versions that folks were trying
** to use "in the wild".
** I *hated* to sacrifice the readability, but...
*/

#define SIGN_POSITIVE 1
#define SIGN_NEGATIVE 0

/* DSR:2002.04.13:begin block */
#define SUBTYPE_NONE    0
#define SUBTYPE_NUMERIC 1
#define SUBTYPE_DECIMAL 2
/* DSR:2002.04.13:end block */

/* DSR:2002.07.25:begin block */
#define SQLIND_NOT_NULL 0
#define SQLIND_NULL -1

/* See IB6 API Guide page 92. */
#define XSQLVAR_IS_ALLOWED_TO_BE_NULL(sqlvar) ((sqlvar->sqltype & 1) != 0)
#define XSQLVAR_IS_NULL(sqlvar) (*(sqlvar->sqlind) == SQLIND_NULL)

#define XSQLVAR_SET_NULL(sqlvar)     (*(sqlvar->sqlind) = SQLIND_NULL)
#define XSQLVAR_SET_NOT_NULL(sqlvar) (*(sqlvar->sqlind) = SQLIND_NOT_NULL)
/* DSR:2002.07.25:end block */

/* Macros for printing debugging information to stderr. */
#ifdef KIDB_DEBUGGERING
  #define debug_printf( x ) fprintf( stderr, ( x ) )
  #define debug_printf1( x, y ) fprintf( stderr, ( x ), ( y ) )
  #define debug_printf2( x, y, z ) fprintf( stderr, ( x ), ( y ), ( z ) )

  #define debug_dump_status_vector(x) dumpStatusVector(x)
#else
  #define debug_printf( x )
  #define debug_printf1( x, y )
  #define debug_printf2( x, y, z )

  #define debug_dump_status_vector(x)
#endif /* KIDB_DEBUGGERING */

/* DB_API_ERROR acts like a boolean function, returning true if
** status_vector indicates an error.
*/
#define DB_API_ERROR(status_vector) (status_vector[0] == 1) && status_vector[1] > 0


#define CONVERT_DB_NUMERIC_TO_PYTHON_FLOAT( db_numeric_data, db_numeric_type, db_numeric_decimal_places, py_float_var ) py_float_var = PyFloat_FromDouble( (*(db_numeric_type)( db_numeric_data )) /* data cast as type */ / pow(10, db_numeric_decimal_places) /* 10 raised to the number of decimal places */ );

#define MIN( a, b ) ( ( a < b ) ? a : b )
#define MAX( a, b ) ( ( a > b ) ? a : b )

/******************** HANDY ABBREVIATIONS:END ********************/

/* DSR:2002.04.13:begin block */
/******************** MISC. GLOBAL VARIABLES *********************/
/* These are initialized in init_kinterbasdb. */
PyObject* SHRT_MIN_As_PyObject;
PyObject* SHRT_MAX_As_PyObject;

PyObject* INT_MIN_As_PyObject;
PyObject* INT_MAX_As_PyObject;

PyObject* LONG_MIN_As_PyObject;
PyObject* LONG_MAX_As_PyObject;

PyObject* LONG_LONG_MIN_As_PyObject;
PyObject* LONG_LONG_MAX_As_PyObject;
/******************** MISC. GLOBAL VARIABLES:END *****************/
/* DSR:2002.04.13:end block */

/******************** MODULE TYPE DEFINITIONS:BEGIN ********************/
/* See Appendix B of Interbase 6 API Guide for explanation of Interbase API
** structures.
*/

/* Global references to this module's exception type objects, as documented
** in the Python DB API (values supplied by function init_kidb_exceptions).
*/
static PyObject* Warning                   = NULL;
static PyObject* Error                     = NULL;
static PyObject* InterfaceError            = NULL;
static PyObject* DatabaseError             = NULL;
static PyObject* DataError                 = NULL;
static PyObject* OperationalError          = NULL;
static PyObject* IntegrityError            = NULL;
static PyObject* InternalError             = NULL;
static PyObject* ProgrammingError          = NULL;
static PyObject* NotSupportedError         = NULL;

staticforward PyTypeObject ConnectionType;
staticforward PyTypeObject CursorType;

/* DSR:2002.06.09:begin block */
/* This structure supports a moderately ugly approach to determining the
** precision of a given field (querying the system tables). */
typedef struct {
  isc_stmt_handle stmt_handle_table;
  isc_stmt_handle stmt_handle_stored_procedure;

  XSQLDA *in_da;
  XSQLDA *out_da;

  XSQLVAR *out_var;

  /* DSR:2002.06.13:begin block */
  PyObject *result_cache;
  /* DSR:2002.06.13:end block */
} CursorDescriptionCache;
/* DSR:2002.06.09:end block */

typedef struct { /* definition of type ConnectionObject */
  PyObject_HEAD /* Python API - infrastructural macro. */

  unsigned short   dialect;
  isc_db_handle    db_handle;
  isc_tr_handle    trans_handle;

  /* Buffer used by Interbase API to store error status of calls. */
  ISC_STATUS       status_vector[ STATUS_VECTOR_SIZE ];

  /* Connection state flag. */
  char             _state;
  #define CONNECTION_STATE_CLOSED         0
  #define CONNECTION_STATE_OPEN           1

  /* DSR:2002.02.28:bugfix #517093:begin block */
  char             precision_mode;
  #define PRECISION_MODE_IMPRECISE        0
  #define PRECISION_MODE_PRECISE          1

  #define DEFAULT_PRECISION_MODE          PRECISION_MODE_IMPRECISE
  /* DSR:2002.02.28:bugfix #517093:end block */

  /* DSR:2002.06.09:begin block */
  /* DSR:2002.07.10:begin block */
  #ifdef DETERMINE_FIELD_PRECISION
  CursorDescriptionCache *desc_cache;
  #endif
  /* DSR:2002.07.10:end block */
  /* DSR:2002.06.09:end block */
} ConnectionObject;

/* #define CONN_REQUIRE_OPEN (connection) if ( _conn_require_open(connection, NULL) != 0 ) return NULL; */
#define CONN_REQUIRE_OPEN(connection) if ( _conn_require_open(connection, NULL) != 0 ) return NULL;

#define CONN_REQUIRE_OPEN2(connection, failure_message) if ( _conn_require_open(connection, failure_message) != 0 ) return NULL;

typedef struct { /* definition of type CursorObject */
  PyObject_HEAD /* Python API - infrastructural macro. */

  /* Connection associated with this cursor. */
  ConnectionObject*           connection;

  /* Interbase API - statement handle. */
  isc_stmt_handle             stmt_handle;
  /* Interbase API - containers for input and result parameter structures. */
  XSQLDA*                     in_sqlda;
  XSQLDA*                     out_sqlda;

  /* Utility buffers for input and result parameters. */
  char*                       in_buffer;
  char*                       out_buffer;

  /* Variables to support caching the last SQL string executed (in order to
  ** detect whether a new statement is identical and thereby avoid having to
  ** reparse, etc.).
  */
  char*                       lastsql;
  int                         lastsql_len;

  /* DSR:2002.02.21:
  ** Variable to support caching a single row of results from a stored
  ** procedure invoked using EXECUTE PROCEDURE syntax.  This approach is
  ** exceptionaly clumsy, but necessary in order to deal with of Interbase
  ** API quirks and still comply with the Python DB API 2.0.
  ** (See bug #520793.)
  ** (See special cases involving isc_info_sql_stmt_exec_procedure in
  ** functions pyob_execute and pyob_fetch.)
  */
  PyObject*                   exec_proc_results;

  /* DSR:2002.04.27:
  ** last_fetch_status caches the return code of the last isc_dsql_fetch call
  ** made with this cursor, in order to assist in bridging a gap between the
  ** IB API and the Python DB API.
  **  The Python DB API requires that extraneous fetches after a result set
  ** has been exhausted must be tolerated; the IB API raises an error in such
  ** cases.  This flag allows kinterbasdb to detect this situation and handle
  ** it in the Pythonic way, instead of calling isc_dsql_fetch and
  ** encountering an error.
  */
  int last_fetch_status;

  /* Buffer used by Interbase API to store error status of calls. */
  ISC_STATUS                  status_vector[ STATUS_VECTOR_SIZE ];

  /* Cursor state flag. */
  char             _state;
  #define CURSOR_STATE_CLOSED         0
  #define CURSOR_STATE_OPEN           1

} CursorObject;

#define CUR_REQUIRE_OPEN(cursor) if ( _cur_require_open(cursor, NULL) != 0 ) return NULL;

#define CUR_REQUIRE_OPEN2(cursor, failure_message) if ( _cur_require_open(cursor, failure_message) != 0 ) return NULL;

/******************** MODULE TYPE DEFINITIONS:END ********************/

/******************** FUNCTION PROTOTYPES:BEGIN ********************/

/* YYY:DSR:2002.02.15: List all function prototypes here for documentation purposes. */
static ConnectionObject* new_connection( void );
static PyObject* XSQLVAR2PyObject( CursorObject* cursor, XSQLVAR* sqlvar );
static PyObject* XSQLDA2Tuple( CursorObject* cursor, XSQLDA* sqlda );
static int PyBufferObject2Blob( CursorObject*, PyObject*, ISC_QUAD*);

#ifndef _MSC_VER
void init_kinterbasdb( void );
#endif /* _MSC_VER */

/* DSR: begin DSR-defined function prototypes */
#ifdef INTERBASE6_OR_LATER
static PyObject* pyob_get_precision_mode( PyObject* self, PyObject* args);
static PyObject* pyob_set_precision_mode( PyObject* self, PyObject* args);
#endif /* INTERBASE6_OR_LATER */

static int _conn_require_open(ConnectionObject* self, char* failure_message);

int init_kidb_ibase_header_constants( PyObject* );
void _init_kidb_ibase_header_constants_transaction_parameters( PyObject* );
void _init_kidb_ibase_header_constants_database_info( PyObject* );
static void free_cursor_exec_proc_results_cache( CursorObject* );

int determineStatementType( isc_stmt_handle* statementHandle,
                            ISC_STATUS statusVector[] );
short couldStatementTypePossiblyReturnResultSet(int statementType);
static int _blob_info_total_size_and_max_segment_size(
  ISC_STATUS* status_vector,
  isc_blob_handle* blob_handle,

  ISC_LONG* total_size,
  unsigned short* max_segment_size
);

#ifdef KIDB_DEBUGGERING
void dumpXSQLDA( XSQLDA* sqlda );
void dumpXSQLVAR( XSQLVAR* sqlvar );
void dumpStatusVector( ISC_STATUS* sv );
#endif /* KIDB_DEBUGGERING */

/* DSR:2002.04.13:begin block */
int PyObject2XSQLVAR_check_range_SQL_INTEGER(
    int data_type, short data_subtype, short num_decimal_places,
    PyObject* n, PyObject* min, PyObject* max
  );

int PyObject2XSQLVAR_check_range_SQL_CHARACTER(PyObject* o,
    int actualLength, int maxLength
  );

const char* get_external_data_type_name(int data_type, int data_subtype);
const char* get_internal_data_type_name(int data_type);
/* DSR:2002.04.13:end block */

/* DSR: end DSR-defined function prototypes */

/******************** FUNCTION PROTOTYPES:END ********************/


/******************** EXCEPTION FUNCTIONS:BEGIN ********************/
static void raise_sql_exception( PyObject* exc, const char *where,
                                 ISC_STATUS pvector[] ) {
  char buf[ MAX_ISC_ERROR_MESSAGE_BUFFER_SIZE ];
    /* DSR:2002.06.17: Buffer overflow potential here is *extremely* low, since
    ** only values supplied by the database engine are placed in the fixed-size
    ** buffer.  The variable-length parameter -where is handled dynamically
    ** (with PyString_FromString), so it poses no risk. */

  long SQLCODE;
  PyObject* o;
  PyObject* message;

  if ( where )
    message = PyString_FromString( where );
  else
    message = PyString_FromString( "" );

  SQLCODE = isc_sqlcode( pvector );

  while ( isc_interprete( buf, &pvector ) ) {
    strcat( buf, ". " );
    o = PyString_FromString( buf );
    PyString_ConcatAndDel( &message, o );
  }

  PyErr_SetObject( exc, Py_BuildValue( "(iO)", SQLCODE, message ) );
  Py_DECREF( message );
} /* raise_sql_exception */

/* DSR:2002.04.13:begin block */

/* raise_exception_with_numeric_error_code allows the SQL error code to be
** set directly, without involving an ISC_STATUS vector.
** Thus, raise_exception_with_numeric_error_code might be said to "fall midway
** between raise_sql_exception and raise_exception".
*/
static void raise_exception_with_numeric_error_code(
  PyObject* exc,
  int code,
  const char* description
) {
  PyErr_SetObject( exc, Py_BuildValue("(is)", code, description) );
} /* raise_exception_with_numeric_error_code */

static void raise_exception( PyObject* exc, const char* description ) {
  raise_exception_with_numeric_error_code( exc, 0, description);
} /* raise_exception */
/* DSR:2002.04.13:end block */

/******************** EXCEPTION FUNCTIONS:END ********************/


/******************** TRANSACTION FUNCTIONS:BEGIN ********************/

static int start_transaction( ConnectionObject* connection,
                              char* userparam, int plen ) {
  static char tpb[] = { isc_tpb_version3, isc_tpb_concurrency, isc_tpb_shared,
                        isc_tpb_wait, isc_tpb_write, isc_tpb_read_committed,
                        isc_tpb_rec_version };
  char* tpbuf;

  /* DSR:2002.07.20:begin block */
  /* Raise a more informative error message if the previous transaction is
  ** still active when the client attempts to start another.  The old approach
  ** was to go ahead and try to start the new transaction regardless.  If there
  ** was already an active transaction, the resulting exception made no mention
  ** of it, which was very confusing. */
  if (connection->trans_handle != NULL) {
    raise_exception_with_numeric_error_code(ProgrammingError, -901,
        "Previous transaction still active; cannot start new transaction."
        "  Use commit() or rollback() to resolve the old transaction first."
      );
    return -1;
  }
  /* DSR:2002.07.20:end block */

  if ( userparam ) {
    tpbuf = (char*)( malloc( plen + 1 ) );

    if ( !tpbuf ) {
      PyErr_NoMemory();
      return -1;
    }

    tpbuf[ 0 ] = isc_tpb_version3;
    memcpy( tpbuf + 1, userparam, plen );
    isc_start_transaction( connection->status_vector,
                                 &(connection->trans_handle),
                                 1,
                                 &(connection->db_handle),
                                 plen + 1,
                                 tpbuf );
    free( tpbuf );
  } else {
    /* DSR:2002.03.12:
    ** This code no longer attempts to support Interbase 3 or earlier.
    */
    isc_start_transaction( connection->status_vector,
                                 &(connection->trans_handle),
                                 1,
                                 &(connection->db_handle),
                                 sizeof( tpb ),
                                 tpb );
  }

  if ( DB_API_ERROR(connection->status_vector) ) {
    connection->trans_handle = NULL;
    raise_sql_exception( OperationalError,
                         "start_transaction.isc_start_transaction: ",
                         connection->status_vector );
    return -1;
  }

  return 0;
} /* start_transaction */


static int commit_transaction( ConnectionObject* connection ) {
  /* DSR:2002.07.20:begin block */
  if (connection->trans_handle == NULL) {
    raise_exception_with_numeric_error_code(ProgrammingError, -901,
        "There is no active transaction to commit. Consider using begin()"
        " to explicitly start a transaction in advance of this commit() call."
      );
    return -1;
  }
  /* DSR:2002.07.20:end block */
  isc_commit_transaction( connection->status_vector,
                          &(connection->trans_handle) );

  if ( DB_API_ERROR(connection->status_vector) ) {
    connection->trans_handle = NULL;
    raise_sql_exception( OperationalError,
                         "commit_transaction.isc_commit_transaction: ",
                         connection->status_vector );
    return -1;
  }

  connection->trans_handle = NULL;
  return 0;
} /* commit_transaction */


/* DSR:2002.07.18:begin block */
/* Fixed bug that caused exception when code such as the following was executed:
**
** con = kinterbasdb.connect(...)
** cur = con.cursor()
** con.rollback()
**
** The problem manifested itself in the above snippet because a transaction
** is only started if:
** a) explicit: con.begin() is called
** b) implicit: cur.execute() is called and a there is not already an active
**    transaction
**
** Normal code wouldn't invoke this bug, but the above snippet falls within
** the realm of technically DB API-compliant code, and thus ought not raise an
** exception.
*/
static int rollback_transaction( ConnectionObject* connection ) {
  isc_tr_handle trans_handle = connection->trans_handle;

  /* If there is not an active transaction, rolling back is meaningless. */
  if (trans_handle == NULL) {
    return 0;
  }

  isc_rollback_transaction(connection->status_vector, &trans_handle);
  if ( DB_API_ERROR(connection->status_vector) ) {
    connection->trans_handle = NULL;
    raise_sql_exception( OperationalError,
         "rollback_transaction.isc_rollback_transaction: ",
         connection->status_vector
       );
    return -1;
  }

  connection->trans_handle = NULL;
  return 0;
} /* rollback_transaction */
/* DSR:2002.07.18:end block */

static PyObject* pyob_begin( PyObject* self, PyObject* args ) {
  ConnectionObject* connection;
  char* param = NULL;
  int plen = 0;

  if ( !PyArg_ParseTuple( args, "O!|z#",
                          &ConnectionType, &connection,
                          &param, &plen ) ) {
    return NULL;
  }

  CONN_REQUIRE_OPEN(connection);

  if ( !( param && plen ) ) {
    param = NULL;
    plen = 0;
  }

  if ( start_transaction( connection, param, plen ) != 0 ) {
    return NULL;
  }

  Py_INCREF( Py_None );
  return Py_None;
} /* pyob_begin */


static PyObject* pyob_commit( PyObject* self, PyObject* args ) {
  ConnectionObject* connection;

  if ( !PyArg_ParseTuple( args, "O!",
                          &ConnectionType, &connection ) ) {
    return NULL;
  }

  CONN_REQUIRE_OPEN(connection);

  if ( commit_transaction(connection) != 0 ) {
    return NULL;
  }

  Py_INCREF( Py_None );
  return Py_None;
} /* pyob_commit */


static PyObject* pyob_rollback( PyObject* self, PyObject* args ) {
  ConnectionObject* connection;

  if ( !PyArg_ParseTuple( args, "O!",
                          &ConnectionType, &connection ) ) {
    return NULL;
  }

  CONN_REQUIRE_OPEN(connection);

  if ( rollback_transaction( connection ) != 0 ) {
    return NULL;
  }

  Py_INCREF( Py_None );
  return Py_None;
} /* pyob_rollback */

/******************** TRANSACTION FUNCTIONS:END ********************/


/******************** CONNECTION FUNCTIONS:BEGIN ********************/

/* Creates and returns new connection object. */
static ConnectionObject* new_connection() {
  ConnectionObject* connection =
    PyObject_New( ConnectionObject, &ConnectionType );

  if ( connection != NULL ) {
    connection->dialect = SQL_DIALECT_DEFAULT;
    connection->db_handle = NULL;
    connection->trans_handle = NULL;
    connection->_state = CONNECTION_STATE_CLOSED;
    /* DSR:2002.02.28:bugfix #517093:begin block */
    connection->precision_mode = DEFAULT_PRECISION_MODE;
    /* DSR:2002.02.28:bugfix #517093:end block */
    /* DSR:2002.06.09:begin block */
    /* DSR:2002.07.10:begin block */
    #ifdef DETERMINE_FIELD_PRECISION
    connection->desc_cache = NULL;
    #endif
    /* DSR:2002.07.10:end block */
    /* DSR:2002.06.09:end block */
  }

  return connection;
} /* new_connection */

/* Cleans up the resources used by an existing connection.
** Also rolls back any associated transactions.
*/
static void delete_connection( ConnectionObject* connection ) {
  if ( connection->db_handle ) {
    if ( connection->trans_handle ) {
        /* DSR: begin block 2002.02.21 */
        /* Explation:  Spec says:  "Closing a connection without committing the
        ** changes first will cause an implicit rollback to be performed."
        */
        rollback_transaction( connection );
        /* DSR: end block 2002.02.21 */
    }
    isc_detach_database( connection->status_vector, &(connection->db_handle) );
  }

  connection->db_handle = NULL;
  connection->trans_handle = NULL;
  connection->_state = CONNECTION_STATE_CLOSED;
  /* DSR:2002.02.28:bugfix #517093:begin block */
  connection->precision_mode = DEFAULT_PRECISION_MODE;
  /* DSR:2002.02.28:bugfix #517093:end block */
  /* DSR:2002.06.09:begin block */
  /* DSR:2002.07.10:begin block */
  #ifdef DETERMINE_FIELD_PRECISION
  {
    CursorDescriptionCache* cache = connection->desc_cache;

    if (cache != NULL) {
      /* DSR:2002.06.13:begin block */
      Py_DECREF(cache->result_cache);
      /* DSR:2002.06.13:end block */

      isc_dsql_free_statement( connection->status_vector,
                               &(cache->stmt_handle_table),
                               DSQL_drop );
      isc_dsql_free_statement( connection->status_vector,
                               &(cache->stmt_handle_stored_procedure),
                               DSQL_drop );

      free(cache->out_var->sqldata);
      free(cache->out_var->sqlind);

      free(cache->in_da);
      free(cache->out_da);

      free(cache);
    }
  }
  #endif
  /* DSR:2002.07.10:end block */
  /* DSR:2002.06.09:end block */
} /* delete_connection */

/* DSR:2002.02.24:
** pyob_close_connection is necessary to comply with the Python DB API spec.
** Previously, the kinterbasdb.py Connection class simply deleted its
** reference to the ConnectionObject; there was no explicit call to a
** _kinterbasdb function to close the method.
** The problem with that behavior is revealed by a closer reading of the DB
** API spec's description of Connection.close:
** """
** Close the connection now (rather than whenever __del__ is called). The
** connection will be unusable from this point forward; an Error (or subclass)
** exception will be raised if any operation is attempted with the connection.
** The same applies to all cursor objects trying to use the connection.
** """
** The kinterbasdb.py Connection class previously just deleted its reference
** to the ConnectionObject, but any cursors open on the ConnectionObject
** retained references to it, preventing it from being garbage collected.
** Therefore, the cursors never realized they were working with an "officially
** closed" connection.
**
** pyob_close_connection was added so that CursorObjects will immediately
** know when they check their ->connection->_state that the connection has
** been explicitly closed.  The cursors will therefore refuse to work as soon
** as the close method has been called on the Connection, "rather than
** whenever __del__ is called" (to quote the DB API spec).
*/
static PyObject* pyob_close_connection( PyObject* self, PyObject* args ) {
  char* conn_open_failure_message = "Attempt to reclose a closed connection.";
  ConnectionObject* connection;

  if ( !PyArg_ParseTuple( args, "O!", &ConnectionType, &connection ) ) {
    return NULL;
  }

  if (connection->_state != CONNECTION_STATE_OPEN) {
    raise_exception(ProgrammingError, conn_open_failure_message);
    return NULL;
  }

  connection->_state = CONNECTION_STATE_CLOSED;

  Py_INCREF( Py_None );
  return Py_None;
} /* pyob_close_connection */

/* DSR:2002.02.24:_conn_require_open
** If self is not an open connection, raises the supplied error message
** (or a default if no error message is supplied).
** Returns 0 if the connection was open; -1 if it failed the test.
*/
static int _conn_require_open(
    ConnectionObject* self,
    char* failure_message
) {
  if ( self != NULL
     && self->_state == CONNECTION_STATE_OPEN
  ) {
    return 0;
  }

  if (failure_message == NULL) {
    failure_message = "Invalid connection state.  The connection must be"
      " open to perform this operation.";
  }

  raise_exception(ProgrammingError, failure_message);
  return -1;
} /* _conn_require_open */


static PyObject* pyob_attach_db(
    PyObject* self,
    PyObject* args,
    PyObject* keywds
 )
{
  char* dsn = NULL;
  char* host = NULL;
  char* database = NULL;
  char* user_name = NULL;
  char* user_passwd = NULL;
  /* DSR:2002.06.09:begin block */
  char* role = NULL;
  /* DSR:2002.06.09:end block */
  char* charset = NULL;
  short dialect;
  char* dpb;
  char dpb_buffer[ DPB_BUFFER_SIZE ]; /* X_X:maz buffer overflow possible? */
    /* DSR:2002.06.17:  No.  The dpb_buffer is resized dynamically by
    ** isc_expand_dpb if necessary.  IB6 API Guide page 351:
    ** "If the space allocated for the DPB is not large enough for the
    ** parameters passed to isc_expand_dpb(), then the function reallocates a
    ** larger DPB, preserving its current contents, and adds the new parameters."
    */
  short dpb_length = 0;
  /* DSR:2002.06.17:begin block */
  /* Eliminated this buffer overflow potential by using dynamic Python string
  ** rather than static C character buffer: */
  #if PYTHON_2_2_OR_LATER
    PyObject* buffer = NULL;
  #else
    char buffer[ GENERAL_UTILITY_BUFFER_SIZE ]; /* buffer overflow possible */
  #endif
  /* DSR:2002.06.17:end block */
  /* DSR:2002.06.09:begin block */
  static char* kwlist[] = { "dsn", "user", "password", "role",
                            "charset", "host", "database", "dialect", NULL };
  /* DSR:2002.06.09:end block */
  ConnectionObject* connection;

  connection = new_connection();
  if ( connection == NULL ) {
    raise_exception( InternalError,
                     "pyob_attach_db.new_connection: "
                     "Cannot allocate a new connection"
                     KIDB_REPORT KIDB_HOME_PAGE );
    return NULL;
  }

  dialect = connection->dialect;

  /* DSR:2002.06.09:begin block */
  if ( !PyArg_ParseTupleAndKeywords(
          args, keywds, "|zzzzzzzi", kwlist,
          &dsn, &user_name, &user_passwd, &role, &charset, &host, &database,
          &dialect
       )
     )
  {
  /* DSR:2002.06.09:end block */
    return NULL;
  }

  if ( connection->dialect != dialect ) {
    connection->dialect = dialect;
  }

  if ( ( !dsn && !host && !database ) ||
       ( dsn && host && database ) ||
       ( host && !database ) ||
       ( !host && database )
     )
  {
    delete_connection( connection );
    raise_exception( InterfaceError,
                    "Keyword argument dsn='host:/path/database' or "
                    "host='host' with database='database' "
                    "must be specified and not both" );
    return NULL;
  }

  if ( host && database ) {
    /* DSR:2002.06.17:begin block */
    #if PYTHON_2_2_OR_LATER
      buffer = PyString_FromFormat("%s:%s", host, database);
      dsn = PyString_AsString(buffer);
    #else
      sprintf( buffer, "%s:%s", host, database ); /* buffer overflow possible */
      dsn = buffer;
    #endif
    /* DSR:2002.06.17:end block */
  }

  dpb = dpb_buffer;
  *dpb = isc_dpb_version1;
  dpb_length = 1;

  if ( user_name ) {
      isc_expand_dpb( &dpb, (short*)&dpb_length,
                      isc_dpb_user_name, user_name, NULL );
  }

  if ( user_passwd ) {
    isc_expand_dpb( &dpb, (short*)&dpb_length,
                    isc_dpb_password, user_passwd, NULL );
  }

  /* DSR:2002.06.09:begin block */
  if ( role ) {
    isc_expand_dpb( &dpb, (short*)&dpb_length,
                    isc_dpb_sql_role_name, role, NULL );
  }
  /* DSR:2002.06.09:end block */

  if ( charset ) {
    isc_expand_dpb( &dpb, (short*)&dpb_length,
                    isc_dpb_lc_ctype, charset, NULL );
  }

  isc_attach_database( connection->status_vector,
                       (short)strlen( dsn ), dsn,
                       &(connection->db_handle), dpb_length, dpb );

  /* DSR:2002.06.17:begin block */
  #if PYTHON_2_2_OR_LATER
  /* Whether the isc_attach_database call succeeded or not, we're done with the
  ** char* -dsn and the (possibly NULL) PyObject* -buffer, whose internal
  ** buffer -dsn might point to. */
  Py_XDECREF(buffer);
  #endif
  /* DSR:2002.06.17:end block */

  if ( DB_API_ERROR(connection->status_vector) ) {
    delete_connection( connection );
    raise_sql_exception( OperationalError,
                         "attach_db.isc_attach_database: ",
                         connection->status_vector );
    return NULL;
  }

  connection->_state = CONNECTION_STATE_OPEN;

  return (PyObject*)( connection );
} /* pyob_attach_db */


static PyObject* pyob_detach_db( PyObject* self, PyObject* args ) {
  ConnectionObject* connection;

  if ( !PyArg_ParseTuple( args, "O!", &ConnectionType, &connection ) )
    return NULL;

  CONN_REQUIRE_OPEN(connection);

  delete_connection( connection );

  Py_INCREF( Py_None );
  return Py_None;
} /* detach_db */


#ifdef INTERBASE6_OR_LATER

static PyObject* pyob_get_dialect( PyObject* self, PyObject* args) {
  ConnectionObject* connection;

  if ( !PyArg_ParseTuple( args, "O!", &ConnectionType, &connection ) )
    return NULL;

  CONN_REQUIRE_OPEN(connection);

  return PyInt_FromLong( (long)( connection->dialect ) );
} /* get_dialect */


static PyObject* pyob_set_dialect( PyObject* self, PyObject* args ) {
  ConnectionObject* connection;
  int dialect;

  if ( !PyArg_ParseTuple( args, "O!i",
          &ConnectionType, &connection, &dialect )
  ) {
    return NULL;
  }

/*
  if ( !PyArg_ParseTuple( args, "Oi",
          &connection, &dialect )
  ) {
    return NULL;
  }
if (connection == NULL) {
  raise_exception(ProgrammingError, "Joe");
}
*/
  CONN_REQUIRE_OPEN(connection);

  connection->dialect = dialect;

  Py_INCREF( Py_None );
  return Py_None;
} /* set_dialect */

/* DSR:2002.02.28:bugfix #517093:begin block */

static PyObject* pyob_get_precision_mode( PyObject* self, PyObject* args) {
  ConnectionObject* connection;

  if ( !PyArg_ParseTuple( args, "O!", &ConnectionType, &connection ) )
    return NULL;

  return PyInt_FromLong( (long)( connection->precision_mode ) );
} /* get_precision_mode */

static PyObject* pyob_set_precision_mode( PyObject* self, PyObject* args ) {
  ConnectionObject* connection;
  int precision_mode;

  if ( !PyArg_ParseTuple( args, "O!i",
                          &ConnectionType, &connection, &precision_mode ) )
    return NULL;
  /* DSR:2002.07.10:begin block */
  connection->precision_mode = (char) precision_mode;
  /* DSR:2002.07.10:end block */

  Py_INCREF( Py_None );
  return Py_None;
} /* set_precision_mode */

/* DSR:2002.02.28:bugfix #517093:end block */

#endif /* INTERBASE6_OR_LATER */


static PyObject* pyob_database_info( PyObject* self, PyObject* args ) {
  ConnectionObject* connection;
  char requestbuffer[] = { isc_info_end, isc_info_end };
  char resulttype;    /* 'i'-short integer, 's'-string */
  char resultbuffer[ ISC_INFO_RESULT_BUFFER_SIZE ]; /* XXX:maz possible buffer overflow? */
    /* DSR: Yes, definitely.  One such case might be isc_info_user_names */
  long len;
  PyObject* res = NULL;

  if ( !PyArg_ParseTuple( args, "O!bc",
                          &ConnectionType, &connection,
                          &(requestbuffer[ 0 ]), &resulttype ) )
    return NULL;

  CONN_REQUIRE_OPEN(connection);

  isc_database_info( connection->status_vector,
                     &(connection->db_handle),
                     sizeof( requestbuffer ), requestbuffer,
                     sizeof( resultbuffer ), resultbuffer );

  if ( DB_API_ERROR(connection->status_vector) ) {
    raise_sql_exception( OperationalError,
                         "database_info.isc_database_info: ",
                         connection->status_vector );
    return NULL;
  }

  len = isc_vax_integer( resultbuffer + 1, 2 );

  switch ( resulttype ) {
  case 'i':
  case 'I':
    res = PyInt_FromLong( isc_vax_integer( resultbuffer + 3, (short)len ) );
    break;
  case 's':
  case 'S':
    res = PyString_FromStringAndSize( resultbuffer + 3, len );
    break;
  default:
    raise_exception( InterfaceError,
                     "Unknown result type in database_info" );
    res = NULL;
  }

  return res;
} /* database_info */

/* DSR:2002.03.15:
** Conversion function to support use of pyob_database_info by Python
** programmers.
** Casts the received Python buffer to a C int, then returns that int
** as a PyInt.
*/
static PyObject* pyob_raw_byte_to_int( PyObject* self, PyObject* args ) {
  char* buf = NULL;
  int bufSize;

  if ( !PyArg_ParseTuple( args, "s#", &buf, &bufSize ) ) {
    return NULL;
  }

  if ( bufSize != 1 ) {
    raise_exception(ProgrammingError,
      "raw_byte_to_int only accepts buffers of length 1");
    return NULL;
  }

  return PyInt_FromLong( (int) *buf );
}

/******************** CONNECTION FUNCTIONS:END ********************/


/******************* XSQLDA MANAGEMENT FUNCTIONS:BEGIN ********************/

/* DSR:2002.06.09:begin block */
/* Remodelled this function to make it fail cleanly in case of a memory
** allocation problem.  Previously, it was a) confusingly structured, and
** b) returned inconsistent error codes. */
static int reallocate_sqlda( XSQLDA** psqlda ) {
  XSQLDA* sqlda = *psqlda;

  /* If the XSQLDA has never been allocated (is NULL), allocate a new one and
  ** return.
  ** If the XSQLDA *was* previously allocated, resize it to accomodate as many
  ** XSQLVARs as will need, or do nothing if it already has enough space. */
  if ( sqlda == NULL ) {
    sqlda = (XSQLDA*)( malloc( XSQLDA_LENGTH( INITIAL_SQLVAR_CAPACITY ) ) );

    if ( sqlda == NULL ) {
      *psqlda = NULL;
      PyErr_NoMemory();
      return -1;
    }

    sqlda->sqln = INITIAL_SQLVAR_CAPACITY;
    sqlda->version = SQLDA_VERSION1;
    *psqlda = sqlda;

    return 0;
  } else if( sqlda->sqld > sqlda->sqln ) {
    int required_number_of_sqlvars = sqlda->sqld;
    XSQLDA* new_sqlda;

    new_sqlda = (XSQLDA*)(
        realloc(sqlda, XSQLDA_LENGTH(required_number_of_sqlvars))
      );

    if ( new_sqlda != NULL ) {
      /* The reallocation succeeded, meaning that realloc has already
      ** deallocated the old memory for us. */
      sqlda = new_sqlda;
    } else {
      /* As maz pointed out on 2002.01.11, "If realloc() fails, the original
      ** block is left untouched - it is not freed or moved."  Having failed
      ** to enlarge sqlda, we must manually free the space it originally
      ** occupied before raising an exception, since realloc did not free this
      ** memory for us. */
      free( sqlda );

      *psqlda = NULL;
      PyErr_NoMemory();
      return -1;
    }

    sqlda->sqln = required_number_of_sqlvars;
    sqlda->version = SQLDA_VERSION1;
    *psqlda = sqlda;

    return 0;
  }

  /* Nothing needed to be done. */
  return 0;
} /* reallocate_sqlda */
/* DSR:2002.06.09:end block */


/* YYY:DSR Maybe these #defines should be moved to the top of the file with
** their brethren?
*/
#define INITIAL_SIZE 256
#define INCREMENT    INITIAL_SIZE
#define ALIGN(n,b)          ((n + b - 1) & ~(b - 1))


static char* allocate_buffer( XSQLDA* sqlda ) {
  int i;
  char* buf;
  char* new_buf;
  unsigned int bufsize = INITIAL_SIZE;
  int type;
  XSQLVAR* var;
  int length;
  int offset = 0;
  int alignment;

  buf = (char*)( malloc( bufsize ) );

  if ( !buf ) {
    return (char*) PyErr_NoMemory();
  }

  for ( i = 0; i < sqlda->sqld; i++ ) {
    var = sqlda->sqlvar + i; /* address of XSQLVAR structure */
    length = alignment = var->sqllen;
    type = var->sqltype & ~1;

    if ( type == SQL_TEXT )
      alignment = 1;
    else if ( type == SQL_VARYING ) {
      length = alignment = var->sqllen;
      length += sizeof( short );
      alignment = sizeof( short );
    }

    offset = ALIGN( offset, alignment );
    var->sqldata = (char*)( offset );  /* simply store an offset for a while */
    offset += length;

    if ( (unsigned int)offset > bufsize ) { /* enlarge buffer and again */
      /* needed size + indicator + INCREMENT */
      bufsize = offset + sizeof( short ) + INCREMENT;
      new_buf = (char*)( realloc( buf, bufsize ) );

      /* maz:11.01.2002
      ** If realloc() fails the original block is left untouched - it is
      ** not freed or moved. */
      if ( !new_buf ) {
        free( buf );
      }

      buf = new_buf;

      if ( !buf ) {
        return (char*) PyErr_NoMemory();
      }
    }

    offset = ALIGN( offset, sizeof( short ) );
    /* simply store an offset */
    var->sqlind = (short*)( (char ISC_FAR*)( offset ) );
    offset += sizeof( short );
  }

  /* Now remember we have offsets instead of pointers */
  /* here we replacing it */
  for ( i = 0; i < sqlda->sqld; i++ ) {
    var = sqlda->sqlvar + i;
    var->sqldata = (char ISC_FAR*)( buf + (int)( var->sqldata ) );
    var->sqlind = (short ISC_FAR*)( buf + (int)( var->sqlind ) );
  }

  return buf;
} /* allocate_buffer */

/******************* XSQLDA MANAGEMENT FUNCTIONS:END ********************/

/******************* CURSOR MANAGEMENT FUNCTIONS:BEGIN ********************/

/* DSR:2002.06.09:begin block */
/* Remodelled this function to fail cleanly and consistently in case of a
** memory allocation error. */
static CursorObject* new_cursor( ConnectionObject* connection ) {
  CursorObject* cursor;

  Py_INCREF( connection );

  cursor = PyObject_New( CursorObject, &CursorType );
  if ( cursor == NULL ) {
    goto new_cursor_FAIL;
  }

  cursor->connection = connection; /* Already incremented ref count above. */

  cursor->in_sqlda = NULL;
  if ( reallocate_sqlda( &(cursor->in_sqlda) ) != 0 ) {
    goto new_cursor_FAIL;
  }

  cursor->out_sqlda = NULL;
  if ( reallocate_sqlda( &(cursor->out_sqlda) ) != 0 ) {
    goto new_cursor_FAIL;
  }

  cursor->in_buffer = NULL;
  cursor->out_buffer = NULL;
  cursor->lastsql = NULL;
  cursor->lastsql_len = 0;
  cursor->exec_proc_results = NULL;
  cursor->last_fetch_status = 0;
  cursor->_state = CURSOR_STATE_OPEN;

  memcpy( &(cursor->status_vector),
          &(connection->status_vector),
          sizeof( ISC_STATUS ) * STATUS_VECTOR_SIZE );

  cursor->stmt_handle = NULL;
  isc_dsql_allocate_statement( cursor->status_vector,
                               &(cursor->connection->db_handle),
                               &(cursor->stmt_handle) );

  if ( DB_API_ERROR(cursor->status_vector) ) {
    raise_sql_exception( OperationalError,
                         "new_cursor.isc_dsql_allocate_statement: ",
                         cursor->status_vector );
    goto new_cursor_FAIL;
  }

  return cursor;

new_cursor_FAIL:
    Py_DECREF( connection );
    if ( cursor != NULL ) {
      PyObject_Del( cursor );
    }

    return NULL;
} /* new_cursor */
/* DSR:2002.06.09:end block */


/* DSR:2002.02.24:_cur_require_open
** If self is not an open cursor, raises the supplied error message
** (or a default if no error message is supplied).
** Returns 0 if the cursor was open; -1 if it failed the test.
*/
static int _cur_require_open(
    CursorObject* self,
    char* failure_message
) {
  if ( self != NULL ) {
    char* conn_failure_message = "Invalid cursor state.  The connection"
      " associated with this cursor is not open, and therefore the cursor"
      " should not be open either.";
    if ( _conn_require_open(self->connection, conn_failure_message) != 0 ) {
      return -1;
    } else if ( self->_state == CURSOR_STATE_OPEN) {
      return 0;
    }
  }

  if (failure_message == NULL) {
      failure_message = "Invalid cursor state.  The cursor must be"
        " OPEN to perform this operation.";
  }
  raise_exception(ProgrammingError, failure_message);
  return -1;
} /* _cur_require_open */

static void free_cursor_cache( CursorObject* cursor ) {
  if ( cursor->lastsql ) {
    free( cursor->lastsql );
    cursor->lastsql = NULL;
  }

  cursor->lastsql_len = 0;

  /* DSR: begin block 2002.02.21 */
  free_cursor_exec_proc_results_cache( cursor );
  /* DSR: end block 2002.02.21 */

} /* free_cursor_cache */

/* DSR: begin block 2002.02.21 */
static void free_cursor_exec_proc_results_cache( CursorObject* cursor ) {
  /* Free ONLY those cursor variables that pertain to the cache of
  ** EXECUTE PROCEDURE results that was introduced in answer to bug #520793.
  ** This function is separate from free_cursor_cache so that it can be
  ** called separately, when it makes sense to free the cached EXEC PROC row,
  ** but not to free the cursor's memory as a whole.
  */

  PyObject_Del( cursor->exec_proc_results );
  cursor->exec_proc_results = NULL;
} /* free_cursor_exec_proc_results_cache */
/* DSR: end block 2002.02.21 */

static void shutdown_cursor( CursorObject* cursor ) {
  if ( cursor->stmt_handle ) {
    /* Close the cursor (as far as any outstanding statements are concerned)
    ** but don't actually delete it. */
    isc_dsql_free_statement( cursor->status_vector,
                             &(cursor->stmt_handle),
                             DSQL_close );

    /* XXX:DSR:STATE:  The isc_dsql_free_statement call above is
    ** often redundant, but the status_vector was not checked for the
    ** (-501, "Attempt to reclose a closed cursor") error that occurs. */
    #ifdef KIDB_DEBUGGERING
    if ( DB_API_ERROR(cursor->status_vector) ) {
      debug_printf("[in shutdown_cursor] ");
      isc_print_status(cursor->status_vector);
    }
    #endif /* KIDB_DEBUGGERING */
  }

  if ( cursor->in_buffer ) {
    free( cursor->in_buffer );
    cursor->in_buffer = NULL;
  }

  /* DSR: begin block 2002.02.21 */
  free_cursor_exec_proc_results_cache( cursor );
  /* DSR: end block 2002.02.21 */
  /* DSR:2002.04.27:begin block */
  cursor->last_fetch_status = 0;
  /* DSR:2002.04.27:end block */
  cursor->_state = CURSOR_STATE_CLOSED;
} /* shutdown_cursor */


static void close_cursor( CursorObject* cursor ) {
  shutdown_cursor( cursor );

  if ( cursor->out_buffer ) {
    free( cursor->out_buffer );
    cursor->out_buffer = NULL;
  }

  free_cursor_cache( cursor );
} /* close_cursor */


static void delete_cursor( CursorObject* cursor ) {
  close_cursor( cursor );

  isc_dsql_free_statement( cursor->status_vector,
                           &(cursor->stmt_handle),
                           DSQL_drop );

  Py_XDECREF( cursor->connection );
  cursor->connection = NULL;
  /* Not being deallocated in KInterbasDB-2.0 -- they are now -- maz */
  if ( cursor->in_sqlda )
    free( cursor->in_sqlda );
  if ( cursor->out_sqlda )
    free( cursor->out_sqlda );

  cursor->stmt_handle = NULL;
} /* delete_cursor */


static PyObject* pyob_cursor( PyObject* self, PyObject* args ) {
  ConnectionObject* connection;

  if ( !PyArg_ParseTuple( args, "O!", &ConnectionType, &connection ) ) {
    return NULL;
  }

  CONN_REQUIRE_OPEN(connection);

  return (PyObject*)( new_cursor( connection ) );
} /* pyob_cursor */


static PyObject* pyob_close_cursor( PyObject* self, PyObject* args ) {
  CursorObject* cursor;

  if ( !PyArg_ParseTuple( args, "O!", &CursorType, &cursor ) ) {
    return NULL;
  }

  CUR_REQUIRE_OPEN(cursor);
  close_cursor( cursor );

  Py_INCREF( Py_None );
  return Py_None;
} /* pyob_close_cusor */

/******************* CURSOR MANAGEMENT FUNCTIONS:END ********************/

/******************* CONVERSION FUNCTIONS:BEGIN ********************/

/* DSR created this version of PyString2Blob on 2002.02.23 to replace the
** previous implementation, which broke with strings of length >= 2^16.
** This function just creates a Python buffer object from the PyString
** pointer it receives.  This "conversion" is QUITE A CHEAP OPERATION.
** It involved no memory copying because it simply creates a "read-only
** reference" into the string's existing character buffer.
*/
static int PyString2Blob(
  CursorObject* cursor,
  PyObject* str,
  ISC_QUAD* blob_id
) {
  PyObject* pyBuffer = PyBuffer_FromObject( str, 0, PyString_Size(str) );
/* DSR:2002.07.13:begin block */
  int result;

  result = PyBufferObject2Blob( cursor, pyBuffer, blob_id );

  Py_DECREF(pyBuffer); /* Previous versions neglected this DECREF->big mem leak. */

  /* PyBufferObject2Blob will take care of raising an exception if it must; we'll
  ** just pass its return value upward. */
  return result;
/* DSR:2002.07.13:end block */
} /* PyString2Blob */


/* DSR:2002.02.23:
** Cleaner implementation of PyBufferObject2Blob that uses the Python raw
** buffer interface rather than slicing and converting each segment to a
** string before passing it to isc_put_segment.
*/
static int PyBufferObject2Blob(
    CursorObject* cursor,
    PyObject* py_buf,
    ISC_QUAD* blob_id
  )
{
  isc_blob_handle blob_handle = NULL;

  PyBufferProcs* bufferProcs;
  char* py_buf_start_ptr;

  int bytes_written_so_far;
  unsigned short bytes_to_write_this_time;

  int total_size = PySequence_Size( py_buf );

  debug_printf1("[in PyBufferObject2Blob] size of input buffer: %d\n",
    total_size);

  /* Get a pointer to the PyBufferObject's getreadbuffer method, then call
  ** that method, which will make py_buf_start_ptr point to the start of
  ** the PyBufferObject's raw data buffer.
  */
  bufferProcs = py_buf->ob_type->tp_as_buffer;
  (*bufferProcs->bf_getreadbuffer)(
    py_buf, 0, (void **)&py_buf_start_ptr );

  /* Create a blob and retrieve its handle into blob_handle. */
  isc_create_blob2( cursor->status_vector,
                    &(cursor->connection->db_handle),
                    &(cursor->connection->trans_handle),
                    &blob_handle,
                    blob_id,
                    0,
                    NULL );

  if ( DB_API_ERROR(cursor->status_vector) ) {
    raise_sql_exception( OperationalError,
                         "PyBufferObject2Blob.isc_create_blob2: ",
                         cursor->status_vector );
    return -1;
  }

  /* Copy the data from py_buf's buffer into the database in chunks of size
  ** MAX_BLOB_SEGMENT_SIZE (all but the last chunk, which may be smaller).
  */
  bytes_written_so_far = 0;
  bytes_to_write_this_time = MAX_BLOB_SEGMENT_SIZE;
  while (bytes_written_so_far < total_size) {
    if (total_size - bytes_written_so_far < MAX_BLOB_SEGMENT_SIZE) {
      bytes_to_write_this_time = total_size - bytes_written_so_far;
    }

    debug_printf1("[in PyBufferObject2Blob] writing segment of %d bytes\n",
      bytes_to_write_this_time);

    isc_put_segment(
        cursor->status_vector,
        &blob_handle,
        bytes_to_write_this_time,
        py_buf_start_ptr + bytes_written_so_far
      );

    if ( DB_API_ERROR(cursor->status_vector) ) {
      raise_sql_exception( OperationalError,
                         "PyBufferObject2Blob.isc_put_segment: ",
                         cursor->status_vector );
      isc_cancel_blob( cursor->status_vector, &blob_handle );
      return -1;
    }

    bytes_written_so_far += bytes_to_write_this_time;
  }

  isc_close_blob( cursor->status_vector, &blob_handle );
  return 0;
} /* PyBufferObject2Blob */


/* DSR: begin block 2002.02.15 - bugfix #517840 */
/* New normalize_double implementation on 2002.02.15 by DSR in response to
** (his own) SourceForge bug report #517840 - "C function normalize_double
** inf. loop".
**
** When reviewing the original normalize_double, maz wrote:
**   X_X:maz this is a nasty function, and I fear floating-point oddities
**   X_X:maz potential for accuracy loss or floating-point miscomputation?
** Maz is right to fear floating-point imprecision in kinterbasdb's handling
** of fixed-point database types; DSR submitted a bug report on the issue
** (SourceForge #517093 - "broken fixed-point handling in 3.0").
**
** However, the issues of floating-point imprecision and how kinterbasdb will
** handle it are outside the scope of the normalize_double function.
** normalize_double is imprecise because floating-point representation is
** imprecise--normalize_double itself makes no claims of perfect precision.
** This new normalize_double implementation is intended to address
** bug #517840 (that is, to fix the normalize_double function per se),
** not #517093 (which requests a fix of kinterbasdb's potentially imprecise
** handling of the database's fixed-point types).
**
** YYY:DSR This normalize_double implementation should be audited for overflow
** issues.
 */
static LONG_LONG normalize_double( double d, short* sqlscalePtr ) {
    /* DSR: Note that my implementation could just as well receive a
    ** 'short sqlscale' as the current 'pointer to short sqlscale'.
    ** I left it as a pointer in order to adhere to the function's original
    ** interface, but since the normalize_double function is currently only
    ** used in one place elsewhere in _kinterbasdb.c, perhaps the interface
    ** could be changed?
     */

    /* We iteratively build the normalized double in this variable: */
    LONG_LONG result;

    /* During this the bulk of this function's computations, we deal with the
    ** ABSOLUTE value of d, so we must remember the original sign: */
    short signOfD = (d >= 0.0 ? SIGN_POSITIVE : SIGN_NEGATIVE);

    short numDecPlaces = -(*sqlscalePtr); /* See Interbase 6 API Guide p88. */

    double wholePartRemaining;
    double fractionalPartRemaining;

    short i;
    short mostRecentlyExtractedDigit;

    /* Initially, extract the whole number portion of d; truncate the
    ** fractional portion.
     */
    if (signOfD == SIGN_POSITIVE) {
        result = (LONG_LONG) fabs(floor(d));
    } else {
        result = (LONG_LONG) fabs(ceil(d));
    }

    fractionalPartRemaining = fabs(modf(d, &wholePartRemaining));
    for (i = 0; i < numDecPlaces; i++) {
        /* Move into the ones place the next digit to be extracted. */
        fractionalPartRemaining *= 10;

        /* DSR:2002.03.13:begin block */
        /* Only round when processing the last decimal place we're saving. */
        if (i != numDecPlaces - 1) {
          mostRecentlyExtractedDigit = (short)floor(fractionalPartRemaining);
        } else {
          /* Round away from zero.
          ** Since we always process the absolute value of the number and
          ** restore its sign after the calculations are done, we need not
          ** differentiate between the meaning of "away from zero" for
          ** positive and negative numbers.
          */
          mostRecentlyExtractedDigit = (short)
            floor(fractionalPartRemaining
              + (0.5
                  /* Add "a tiny bit more" to "compensate" for floating point
                  ** imprecision in the way that most non-CS users would
                  ** expect.
                  ** Of course, this does not really offer a solution to
                  ** floating point imprecision (that's what precision_mode
                  ** 1 is for); it just makes the behavior of this rounding
                  ** code closer to that of the database engine's rounding
                  ** rules.
                  */
                  + (numDecPlaces == 0
                      ? 0.0
                      : pow(10, -MIN(numDecPlaces * 2, 19))
                    )
                )
            );
        }
        /* DSR:2002.03.13:end block */

        /* "Move the digits in result to the left" and place the most recently
        ** extracted digit in the ones place: */
        result = (result * 10) + mostRecentlyExtractedDigit;

        /* Prepare to extract the next digit. */
        fractionalPartRemaining = modf(fractionalPartRemaining,
            &wholePartRemaining);
    }

    /* DSR:2002.03.13:begin block */
    /* Switch back to the original sign of d. */
    if ( signOfD == SIGN_NEGATIVE ) {
      result = -result;
    }
    /* DSR:2002.03.13:end block */

    return result;
} /* normalize_double */
/* DSR: end block 2002.02.15 - bugfix #517840 */

static int PyObject2XSQLVAR(
    CursorObject *cursor,
    XSQLVAR *sqlvar,
    PyObject *inputPyObj
  )
{
  int n;
  short data_type;
  /* DSR:2002.04.13:begin block */
  short data_subtype;
  /* DSR:2002.04.13:end block */
  short num_decimal_places;
  /* DSR:bugfixes #517093 and #522774:2002.02.28:begin block */
  char precision_mode = cursor->connection->precision_mode;
  /* DSR:bugfixes #517093 and #522774:2002.02.28:end block */

  char* s;
  struct tm lt;

  data_type = sqlvar->sqltype & ~1;
  /* DSR:2002.04.13:begin block */
  data_subtype = sqlvar->sqlsubtype;
  /* DSR:2002.04.13:end block */
  num_decimal_places = -(sqlvar->sqlscale);

  #ifdef KIDB_DEBUGGERING
  dumpXSQLVAR( sqlvar );
  #endif /* KIDB_DEBUGGERING */

  /* DSR:2002.07.25:begin block */
  /* If the input value is None, ensure that the destination field allows NULL
  ** values.  If it does, set the XSQLVAR's NULL flag and return immediately;
  ** no further conversion is necessary. */
  if (inputPyObj == Py_None) {
    if ( !XSQLVAR_IS_ALLOWED_TO_BE_NULL(sqlvar) ) {
      raise_exception( DataError,
          "Database parameter or field cannot be NULL, so Python's None is"
          " not an acceptable input value."
        );
      return -1;
    }

    XSQLVAR_SET_NULL(sqlvar);
    return 0;
  }

  /* It is now certain the sqlvar will not represent a NULL value; make that
  ** understanding explicit. */
  XSQLVAR_SET_NOT_NULL(sqlvar);
  /* DSR:2002.07.25:end block */

  switch ( data_type ) {

  case SQL_TEXT:
    if ( !PyString_Check(inputPyObj) ) {
      raise_exception( InterfaceError,
                       "PyObject2XSQLVAR.type_mismatch: "
                       "PyString_Check and SQL_TEXT" );
      return -1;
    }

    s = PyString_AsString(inputPyObj);
    n = PyString_Size(inputPyObj);

    /* DSR:2002.04.13:begin block */
    /* OLD version could overflow and silently truncate:
    strncpy( sqlvar->sqldata, s, MIN( n, sqlvar->sqllen ) );
    */
    if ( !PyObject2XSQLVAR_check_range_SQL_CHARACTER( inputPyObj, n, sqlvar->sqllen ) ) {
      return -1;
    }

    strncpy( sqlvar->sqldata, s, n );
    /* DSR:2002.04.13:end block */

    if ( sqlvar->sqllen > n ) {
      sqlvar->sqldata[n] = '\0';
    }
    break;


  case SQL_VARYING:
    if ( !PyString_Check(inputPyObj) ) {
      raise_exception( InterfaceError,
                       "PyObject2XSQLVAR.type_mismatch: "
                       "PyString_Check and SQL_VARYING" );
      return -1;
    }

    s = PyString_AsString(inputPyObj);
    n = PyString_Size(inputPyObj);

    /* DSR:2002.04.13:begin block */
    /* OLD version could overflow and silently truncate:
    n = MIN( n, sqlvar->sqllen );
    */
    if ( !PyObject2XSQLVAR_check_range_SQL_CHARACTER( inputPyObj, n, sqlvar->sqllen ) ) {
      return -1;
    }
    /* DSR:2002.04.13:end block */

    /* X_X:maz gods, pointers! */
    /* The first sizeof(short) bytes of the input buffer must point to a short
    ** that designates the length of the incoming string value. */
    *( (short*)sqlvar->sqldata ) = (short) n;
    strncpy( sqlvar->sqldata + sizeof(short), s, n );

    break;

/* DSR:2002.04.13:begin block */
  case SQL_SHORT:
  case SQL_LONG:
#ifdef INTERBASE6_OR_LATER
  case SQL_INT64:
#endif /* INTERBASE6_OR_LATER */
    {
    /* First of all, establish a Python long object that represents the scaled
    ** version of the incoming Python float/int/long object inputPyObj. */
    PyObject* pyScaledN;
    PyObject* minN;
    PyObject* maxN;
    char isSQLShort = data_type == SQL_SHORT;
    char isSQLLong = data_type == SQL_LONG;
    char preciseModeIsOn = precision_mode == PRECISION_MODE_PRECISE;

    if ( PyFloat_Check(inputPyObj) ) {
      /* Refuse a float if
      ** (in precise mode) AND (dealing with a NUMERIC/DECIMAL field). */
      if ( preciseModeIsOn && data_subtype != SUBTYPE_NONE ) {
        raise_exception( InterfaceError,
            "PyObject2XSQLVAR.type_mismatch:"
            " Cannot use floating point value for NUMERIC/DECIMAL field when"
            " connection is in precise mode.  Use scaled integer instead."
          );
        return -1;
      }

      pyScaledN = PyLong_FromLongLong(
          normalize_double( PyFloat_AsDouble(inputPyObj), &(sqlvar->sqlscale) )
        );
    } else if ( PyInt_Check(inputPyObj) || PyLong_Check(inputPyObj) ) {
      /* Don't need to scale if there are no decimal places or the connection
      ** is in precise mode (precise mode requires the client programmer to
      ** do the scaling).
      */
      if ( num_decimal_places == 0 || preciseModeIsOn ) {
        Py_INCREF(inputPyObj);
        pyScaledN = inputPyObj;
      } else {
        /* Create a scaled representation of inputPyObj without resorting to potentially
        ** imprecise floating point computations.
        ** multiplier will not overflow because num_decimal_places will never
        ** be greater than 18 (at least, not with current versions of the DB
        ** engine), and 10 to the 18th is less than LONG_LONG_MAX.
        ** The actual scaling is done using Python longs, will never overflow.
        */
        LONG_LONG multiplier = 1;
        PyObject* pyMultiplier;
        short i;
        for (i = 0; i < num_decimal_places; i++)
          multiplier *= 10;
        pyMultiplier = PyLong_FromLongLong(multiplier);

        pyScaledN = PyNumber_Multiply(inputPyObj, pyMultiplier);

        Py_DECREF(pyMultiplier);
      }
    } else {
      /* DSR:2002.07.26:begin block */
      /* Raise a more informative error message.
      ** --But only on Python 2.2 or later.  Of course, this could be done with
      ** C string operations rather than PyString_FromFormat on earlier Python
      ** versions, but why spend the extra effort in support of Luddites? */
      #if PYTHON_2_2_OR_LATER
        PyObject *inputPyObj_type = PyObject_Type(inputPyObj);
        PyObject *inputPyObj_type_repr = PyObject_Repr(inputPyObj_type);
        PyObject *inputPyObj_repr = PyObject_Repr(inputPyObj);
        PyObject *buffer = PyString_FromFormat (
            "Type mismatch while attempting to convert object of type %s"
            " to database-internal numeric type for storage."
            "  The object in question is: %s",
            PyString_AsString(inputPyObj_type_repr),
            PyString_AsString(inputPyObj_repr)
          );

        raise_exception( InterfaceError, PyString_AsString(buffer) );

        Py_DECREF(inputPyObj_type);
        Py_DECREF(inputPyObj_type_repr);
        Py_DECREF(buffer);
      #else /* not PYTHON_2_2_OR_LATER */
        raise_exception( InterfaceError,
            "Type mismatch while attempting to convert object"
            " to database-internal integer type for storage."
          );
      #endif /* PYTHON_2_2_OR_LATER */
      /* DSR:2002.07.26:end block */

      return -1;
    }

    /* At this point, regardless of how the value arrived and what precision
    ** mode we're operating in, pyScaledN is a Python long object representing
    ** the appropriately scaled value.
    ** The next step is to ensure that the scaled value is not too large for
    ** storage in its internal format.
    ** If it is not too large, we will finally transfer the value from its
    ** Pythonic representation to the XSQLVAR.
    */
    if ( isSQLShort ) {
      minN = SHRT_MIN_As_PyObject;
      maxN = SHRT_MAX_As_PyObject;
    } else if ( isSQLLong ) {
      minN = LONG_MIN_As_PyObject;
      maxN = LONG_MAX_As_PyObject;
#ifdef INTERBASE6_OR_LATER
    } else { /* data_type must be SQL_INT64 */
      minN = LONG_LONG_MIN_As_PyObject;
      maxN = LONG_LONG_MAX_As_PyObject;
#endif /* INTERBASE6_OR_LATER */
    }

    /* XXX:DSR:
    ** This check does not take into account the defined number of digits in
    ** NUMERIC/DECIMAL fields.  For example, a NUMERIC(2,1) field, which is
    ** stored as a SQL_SHORT, will accept the value 123.4 because 1234 fits in
    ** a short.  According to the NUMERIC(2,1) definition, however, we should
    ** refuse any value with more than 1 digit to the left of the decimal
    ** point.
    ** The problem is that I haven't found a way to use the API to discover the
    ** maximum digit width of a NUMERIC/DECIMAL field.  Querying the system
    ** tables would work if sqlvar->relname and sqlvar->ownname were supplied
    ** for input XSQLVARs, but a) those fields are not present for input
    ** XSQLVARs, and b) querying the system tables in this situation would be
    ** prohibitively expensive.
    */
    if ( !PyObject2XSQLVAR_check_range_SQL_INTEGER(
              data_type, data_subtype, num_decimal_places,
              pyScaledN, minN, maxN
            )
    ) {
      Py_DECREF( pyScaledN );
      return -1;
    }

    /* DSR:2002.06.21:572326:begin block */
    /* pyScaledN might have been constructed in various ways depending on
    ** the originally received PyObject* named inputPyObj, and on the state of the
    ** connection.
    ** However, we KNOW at this point that pyScaledN is either a PyInt or
    ** a PyLong.  With Python 2.2 or later, PyLong_AsLong will work for both
    ** PyLongs and PyInts, since PyInts and PyLong have been "unified".  On
    ** earlier Python versions, we must use the _exact_ conversion function.
    ** So:  do not include the specific check for PyInt on Python 2.2 or later.
    */
    #if !PYTHON_2_2_OR_LATER
    {
      char isPyInt = PyInt_Check(pyScaledN);
    #endif
    if ( isSQLShort ) {
      #if !PYTHON_2_2_OR_LATER
      if ( isPyInt ) {
        *(short*)sqlvar->sqldata = (short) PyInt_AsLong(pyScaledN);
      } else { /* Must be PyLong */
      #endif
        *(short*)sqlvar->sqldata = (short) PyLong_AsLong(pyScaledN);
      #if !PYTHON_2_2_OR_LATER
      }
      #endif
    } else if ( isSQLLong ) {
      #if !PYTHON_2_2_OR_LATER
      if ( isPyInt ) {
        *(long*)sqlvar->sqldata = PyInt_AsLong(pyScaledN);
      } else { /* Must be PyLong */
      #endif
        *(long*)sqlvar->sqldata = PyLong_AsLong(pyScaledN);
      #if !PYTHON_2_2_OR_LATER
      }
      #endif
#ifdef INTERBASE6_OR_LATER
    } else { /* data_type must be SQL_INT64 */
      #if !PYTHON_2_2_OR_LATER
      if ( isPyInt ) {
        /* There is no PyInt_AsLongLong because a PyInt's value is stored
        ** internally as a C long. */
        *(LONG_LONG*)sqlvar->sqldata = (LONG_LONG) PyInt_AsLong(pyScaledN);
      } else { /* Must be PyLong */
      #endif
        *(LONG_LONG*)sqlvar->sqldata = PyLong_AsLongLong(pyScaledN);
      #if !PYTHON_2_2_OR_LATER
      }
      #endif
#endif /* INTERBASE6_OR_LATER */
    }
    #if !PYTHON_2_2_OR_LATER
    } /* Close the block in which the variable isPyInt is declared. */
    #endif
    /* DSR:2002.06.21:572326:end block */

    Py_DECREF( pyScaledN );
    }
    break;
/* DSR:2002.04.13:end block */

  case SQL_FLOAT:
    /* Extra type support from janez's first patch to maz (added on
    ** 19.jan/y2k2 by maz).
    ** Maz added the casts to float on the right-hand-side. */
    if ( PyFloat_Check(inputPyObj) ) {
      *( (float*)sqlvar->sqldata ) = (float) PyFloat_AsDouble(inputPyObj);
      sqlvar->sqlscale = 0;
    } else if ( PyInt_Check(inputPyObj) ) {
      *( (float*)sqlvar->sqldata ) = (float) PyInt_AsLong(inputPyObj);
      sqlvar->sqlscale = 0;
    } else if ( PyLong_Check(inputPyObj) ) {
      *( (float*)sqlvar->sqldata ) = (float) PyLong_AsLong(inputPyObj);
      sqlvar->sqlscale = 0;
    } else {
      raise_exception( InterfaceError,
          "PyObject2XSQLVAR.type_mismatch: "
          "PyFloat_Check/PyInt_Check/PyLong_Check and SQL_FLOAT"
        );
      return -1;
    }

    break;


  case SQL_DOUBLE:
  case SQL_D_FLOAT:
    if ( PyFloat_Check(inputPyObj) ) {
      *( (double*)sqlvar->sqldata ) = (double) PyFloat_AsDouble(inputPyObj);
    } else if ( PyInt_Check(inputPyObj) ) {
      *( (double*)sqlvar->sqldata ) = (double) PyInt_AsLong(inputPyObj);
    } else if ( PyLong_Check(inputPyObj) ) {
      *( (double*)sqlvar->sqldata ) = (double) PyLong_AsLong(inputPyObj);
    } else {
      raise_exception( InterfaceError,
           "PyObject2XSQLVAR.type_mismatch: "
           "PyFloat_Check/PyLong_Check/PyInt_Check and SQL_D_FLOAT/SQL_DOUBLE"
         );
      return -1;
    }

    break;


  case SQL_BLOB:
    /* DSR:2002.04.13:begin comment */
    /* It would be cute to check for overflow here, just as in many other CASEs
    ** in this function, but in reality it is not necessary.  Interbase blobs
    ** have a theoretical maximum size of 34359738368 bytes (32GB), which far
    ** exceeds the theoretical limit of a Python buffer or string (INT_MAX,
    ** 2147483647 bytes on a typical 32-bit platform).
    ** The Python programmer simply could not create a value large enough to
    ** overflow an Interbase blob (not even by implementing a custom sequence
    ** class that represents the composition of several buffers, since the
    ** Python C API function PySequence_Size returns an int).
    */
    /* DSR:2002.04.13:end comment */

    if ( PyBuffer_Check(inputPyObj) ) {
      /* PyBufferObject2Blob will raise an exception if necessary; we'll just
      ** pass its return value upward.
      */
      return
          PyBufferObject2Blob ( cursor, inputPyObj, (ISC_QUAD*)( sqlvar->sqldata ) );
    } else if ( PyString_Check(inputPyObj) ) {
      /* PyString2Blob will raise an exception if necessary; we'll just
      ** pass its return value upward.
      */
      return
          PyString2Blob ( cursor, inputPyObj, (ISC_QUAD*)( sqlvar->sqldata ) );
    } else {
      raise_exception( InterfaceError,
                       "PyObject2XSQLVAR.type_mismatch: "
                       "PyBuffer_Check/PyString_Check and SQL_BLOB" );
      return -1;
    }

    break;


  /* Handle TIMESTAMP, DATE, and TIME fields: */
  case SQL_TIMESTAMP: /* TIMESTAMP */
    if ( !mxDateTime_Check(inputPyObj) ) {
      /* DSR:2002.07.18:begin block */
      /* Raise a more specific error if a mxDateTimeDeltaObject is received. */
      /* XXX:factor out this duplicated code block: */
      if ( mxDateTimeDelta_Check(inputPyObj) ) {
        raise_exception( InterfaceError,
            "A Time object (represented internally as mx DateTimeDelta)"
            " does not contain enough information to populate a TIMESTAMP"
            " field without guessing (the year, month, and day are missing,"
            " and \"explicit is better than implicit\")."
            "  Instead of using a Time object, you should construct a DateTime"
            " object (represented internally as mx DateTime) via either"
            " kinterbasdb.TimeStamp or kinterbasdb.TimeStampFromTicks."
          );
        return -1;
      }

      /* Raise a more informative error message than did the old version: */
      { PyObject *theType = PyObject_Type(inputPyObj);
        PyObject *theTypeAsString = PyObject_Str(theType);
        size_t theTypeAsString_len = PyString_Size(theTypeAsString);

        char *baseMsg = "Type mismatch:  For a TIMESTAMP field, you must"
          " supply a DateTime object, not a ";
        size_t baseMsg_len = strlen(baseMsg);

        char *errMsgBuf = (char*) malloc( sizeof(char) *
            (baseMsg_len + theTypeAsString_len + 1)
          );
        strncpy(errMsgBuf, baseMsg, baseMsg_len);
        strncpy(errMsgBuf + baseMsg_len,
            PyString_AsString(theTypeAsString), theTypeAsString_len
          );
        errMsgBuf[baseMsg_len + theTypeAsString_len] = '\0';

        raise_exception( InterfaceError, errMsgBuf);

        Py_DECREF(theTypeAsString);
        /* raise_exception makes a *copy* of the error message buffer that it
        ** receives, so it's our responsibility to free the buffer we
        ** allocated locally. */
        free(errMsgBuf);
      }
      /* DSR:2002.07.18:end block */
      return -1;
    }

    /* Moved the PyObject2ib{Date,Time,DateTime} functions here to keep
    ** consistency.  This is from janez's first patch to maz (added on
    ** 19.jan/y2k2 by maz). */
    /* DSR:2002.07.18:begin block */
    mxDateTime.DateTime_AsTmStruct( (mxDateTimeObject*)(inputPyObj), &lt );
    /* OLD: isc_encode_date( &lt, (void*)( sqlvar->sqldata ) ); */
    isc_encode_timestamp( &lt, (ISC_TIMESTAMP *) sqlvar->sqldata );
    /* DSR:2002.07.18:end block */

    break;


#ifdef INTERBASE6_OR_LATER
  case SQL_TYPE_DATE: /* DATE */
    if ( !mxDateTime_Check(inputPyObj) ) {
      /* DSR:2002.07.18:begin block */
      /* Raise a more informative error message than did the old version: */
      /* XXX:factor out this duplicated code block: */
      { PyObject *theType = PyObject_Type(inputPyObj);
        PyObject *theTypeAsString = PyObject_Str(theType);
        size_t theTypeAsString_len = PyString_Size(theTypeAsString);

        char *baseMsg = "Type mismatch:  For a DATE field, you must"
          " supply a DateTime object, not a ";
        size_t baseMsg_len = strlen(baseMsg);

        char *errMsgBuf = (char*) malloc( sizeof(char) *
            (baseMsg_len + theTypeAsString_len + 1)
          );
        strncpy(errMsgBuf, baseMsg, baseMsg_len);
        strncpy(errMsgBuf + baseMsg_len,
            PyString_AsString(theTypeAsString), theTypeAsString_len
          );
        errMsgBuf[baseMsg_len + theTypeAsString_len] = '\0';

        raise_exception( InterfaceError, errMsgBuf);

        Py_DECREF(theTypeAsString);
        /* raise_exception makes a *copy* of the error message buffer that it
        ** receives, so it's our responsibility to free the buffer we
        ** allocated locally. */
        free(errMsgBuf);
      }
      /* DSR:2002.07.18:end block */
      return -1;
    }

    /* Moved the PyObject2ib{Date,Time,DateTime} functions here to keep
    ** consistency.  This is from janez's first patch to maz (added on
    ** 19.jan/y2k2 by maz). */
    mxDateTime.DateTime_AsTmStruct( (mxDateTimeObject*)(inputPyObj), &lt );
    isc_encode_sql_date( &lt, (void*)( sqlvar->sqldata ) );

    break;


  case SQL_TYPE_TIME: /* TIME */
  /* DSR:2002.07.18:begin block */
  /* Changed this clause so that it accepts either an mxDateTimeObject or an
  ** mxDateTimeDeltaObject, rather than only an mxDateTimeObject.  This brings
  ** kinterbasdb's database date and time type constructors (namely
  ** kinterbasdb.TimestampFromTicks) into compliance with the DB API.
  */
  { mxDateTimeObject *theInputValueAsmxDateTime;

    if ( mxDateTimeDelta_Check(inputPyObj) ) {
      theInputValueAsmxDateTime = (mxDateTimeObject*) mxDateTime.DateTime_FromAbsDays(
          mxDateTime.DateTimeDelta_AsDays( (mxDateTimeDeltaObject*) inputPyObj )
        );
      /* We just created a new object; no need to INCREF here. */
    } else if ( mxDateTime_Check(inputPyObj) ) {
      theInputValueAsmxDateTime = (mxDateTimeObject*) inputPyObj;
      Py_INCREF(theInputValueAsmxDateTime); /* For uniformity with the previous
        ** clause, which actually creates a new object. */
    } else {
      raise_exception( InterfaceError,
           "PyObject2XSQLVAR.type_mismatch: mxDateTime_Check and SQL_TYPE_TIME"
         );
      return -1;
    }

    /* Moved the PyObject2ib{Date,Time,DateTime} functions here to keep
    ** consistency.  This is from janez's first patch to maz (added on
    ** 19.jan/y2k2 by maz). */
    mxDateTime.DateTime_AsTmStruct( theInputValueAsmxDateTime, &lt );
    isc_encode_sql_time( &lt, (void*)(sqlvar->sqldata) );

    Py_DECREF(theInputValueAsmxDateTime);
  }
/* DSR:2002.07.18:end block */
    break;
#endif /* INTERBASE6_OR_LATER */


  default:
    raise_exception( NotSupportedError,
                     "SQL type is currently not supported"
                     KIDB_REPORT KIDB_HOME_PAGE );
    return -1;

  }

  return 0;
} /* PyObject2XSQLVAR */


/* DSR:2002.04.13:begin block */
int PyObject2XSQLVAR_check_range_SQL_CHARACTER(
    PyObject* s,
    int actualLength,
    int maxLength
  )
{
  if ( actualLength > maxLength ) {
    PyObject* messageFormat = PyString_FromString(
        "string overflow: value %d characters long cannot fit in character"
        " field of maximum length %d (value is '%s')." );
    PyObject* messageArgs = Py_BuildValue( "(iiO)", actualLength, maxLength, s );
    PyObject* errorMessage = PyString_Format( messageFormat, messageArgs );

    raise_exception_with_numeric_error_code( ProgrammingError,
      -802, /* -802 is the IB error code for an overflow */
      PyString_AsString(errorMessage) );

    Py_DECREF(messageFormat);
    Py_DECREF(messageArgs);
    Py_DECREF(errorMessage);

    return 0;
  }

  return 1;
}

int PyObject2XSQLVAR_check_range_SQL_INTEGER(
    int data_type,
    short data_subtype,
    short num_decimal_places,
    PyObject* n,
    PyObject* min,
    PyObject* max
  )
{
  if ( PyObject_Compare(n, min) < 0 || PyObject_Compare(n, max) > 0 ) {
    const char* externalDataTypeName = get_external_data_type_name( data_type,
        data_subtype );
    const char* internalDataTypeName = get_internal_data_type_name( data_type );

    PyObject* messageFormat = PyString_FromString(
        "numeric overflow: value %d (%s scaled for %d decimal places) is of"
        " too great a magnitude to fit into its internal storage type %s,"
        " which has range [%d, %d]."
      );
    PyObject* messageArgs = Py_BuildValue( "(OsisOO)", n, externalDataTypeName,
      num_decimal_places, internalDataTypeName, min, max );
    PyObject* errorMessage = PyString_Format( messageFormat, messageArgs );

    raise_exception_with_numeric_error_code( ProgrammingError,
      -802, /* -802 is the IB error code for an overflow */
      PyString_AsString(errorMessage) );

    Py_DECREF(messageFormat);
    Py_DECREF(messageArgs);
    Py_DECREF(errorMessage);

    return 0;
  }

  return 1;
}

const char* get_external_data_type_name(int data_type, int data_subtype) {
  switch ( data_type ) {
    case SQL_TEXT:
      return "CHAR";
    case SQL_VARYING:
      return "VARCHAR";
    case SQL_SHORT:
    case SQL_LONG:
#ifdef INTERBASE6_OR_LATER
    case SQL_INT64:
#endif /* INTERBASE6_OR_LATER */
      switch ( data_subtype ) {
        case SUBTYPE_NONE:
          return (data_type == SQL_SHORT ? "SMALLINT" : "INTEGER");
        case SUBTYPE_NUMERIC:
          return "NUMERIC";
        case SUBTYPE_DECIMAL:
          return "DECIMAL";
      }
    case SQL_FLOAT:
      return "FLOAT";
    case SQL_DOUBLE:
    case SQL_D_FLOAT:
      return "DOUBLE";
    case SQL_TIMESTAMP:
      return "TIMESTAMP";
#ifdef INTERBASE6_OR_LATER
    case SQL_TYPE_DATE:
      return "DATE";
    case SQL_TYPE_TIME:
      return "TIME";
#endif /* INTERBASE6_OR_LATER */
    case SQL_BLOB:
      return "BLOB";
    default:
      return "UNKNOWN";
  }
}

const char* get_internal_data_type_name(int data_type) {
  switch ( data_type ) {
    case SQL_TEXT:
      return "SQL_TEXT";
    case SQL_VARYING:
      return "SQL_VARYING";
    case SQL_SHORT:
      return "SQL_SHORT";
    case SQL_LONG:
      return "SQL_LONG";
#ifdef INTERBASE6_OR_LATER
    case SQL_INT64:
      return "SQL_INT64";
#endif /* INTERBASE6_OR_LATER */
    case SQL_FLOAT:
      return "SQL_FLOAT";
    case SQL_DOUBLE:
    case SQL_D_FLOAT:
      return "SQL_DOUBLE";
    case SQL_TIMESTAMP:
      return "SQL_TIMESTAMP";
#ifdef INTERBASE6_OR_LATER
    case SQL_TYPE_DATE:
      return "SQL_TYPE_DATE";
    case SQL_TYPE_TIME:
      return "SQL_TYPE_TIME";
#endif /* INTERBASE6_OR_LATER */
    case SQL_BLOB:
      return "SQL_BLOB";
    default:
      return "UNKNOWN";
  }
}
/* DSR:2002.04.13:end block */

/* DSR:2002.06.15:begin block */
static int PyObject2XSQLDA (
    CursorObject* cursor,
    XSQLDA* sqlda,
    PyObject* params
  )
{
  /* Assumption:  the type of argument params has already been screened by
  ** pyob_execute. */

  short num_required_statement_params = sqlda->sqld;
  short num_supplied_statement_params = PySequence_Size(params);

  int conversion_status;
  int i;
  PyObject *cur_param;
  XSQLVAR *cur_sqlvar;

  if ( num_supplied_statement_params != num_required_statement_params ) {
    char raw_error_msg[] = "PyObject2XSQLDA: Incorrect number of input parameters."
      "  Expected %d; received %d.";
    char* processed_error_msg = malloc( strlen(raw_error_msg) + 12 );
    sprintf( processed_error_msg, raw_error_msg, num_required_statement_params,
      num_supplied_statement_params );

    raise_exception( InterfaceError,
                     processed_error_msg );

    free( processed_error_msg );

    return -1;
  }

  for (
      i = 0, cur_sqlvar = sqlda->sqlvar;
      i < num_required_statement_params;
      i++, cur_sqlvar++
    )
  {
    /* On 2002.03.31, I switched from PyTuple_GetItem, which returns a BORROWED
    ** reference, to PySequence_GetItem, which returns a NEW reference.  This
    ** created a refcount-based memory leak, which I'm now fixing. */
    cur_param = PySequence_GetItem(params, i);
    if (cur_param == NULL) {
      return -1;
    }
    conversion_status = PyObject2XSQLVAR(cursor, cur_sqlvar, cur_param);
    Py_DECREF( cur_param );

    if (conversion_status != 0) {
      return -1;
    }
  }

  return 0;
} /* PyObject2XSQLVAR */
/* DSR:2002.06.15:end block */

/* DSR:2002.06.09:begin block */
/* DSR:2002.07.10:begin block */
#ifdef DETERMINE_FIELD_PRECISION

#define ENTITY_TYPE_UNKNOWN 0
#define ENTITY_TYPE_TABLE 1
#define ENTITY_TYPE_STORED_PROCEDURE 2

#define ENTITY_TYPE_LAST ENTITY_TYPE_STORED_PROCEDURE

static PyObject* determine_field_precision(
    short entity_type_code,
    char* entity_name, short entity_name_length,
    char* field_name, short field_name_length,
    CursorObject* cursor
  )
{
  /* Returns:
  ** - a new reference to a PyObject* containing the precision figure on
  **   success (may be the PyInt zero)
  ** - a new reference to a PyObject* containing the PyInt zero on routine
  **   inability to determine precision (e.g., dynamic field)
  ** - NULL on error (also sets up exception)
  */

  /* The relname and sqlname strings in an IB API XSQLVAR are not
  ** NULL-terminated (see IB 6 API Guide page 88), so the lengths are provided
  ** as parameters to this function rather than determined locally by strlen. */

  PyObject* precision;
  PyObject* result_cache;
  PyObject* result_cache_this_entity;

  const char *sql_statement_table =
       "SELECT FIELD_SPEC.RDB$FIELD_PRECISION"
      " FROM RDB$FIELDS FIELD_SPEC, RDB$RELATION_FIELDS REL_FIELDS"
      " WHERE"
            " FIELD_SPEC.RDB$FIELD_NAME = REL_FIELDS.RDB$FIELD_SOURCE"
        " AND REL_FIELDS.RDB$RELATION_NAME = ?"
        " AND REL_FIELDS.RDB$FIELD_NAME = ?"
    ;
  unsigned short sql_statement_table_length = strlen(sql_statement_table);

  const char *sql_statement_stored_procedure =
       "SELECT FIELD_SPEC.RDB$FIELD_PRECISION"
      " FROM RDB$FIELDS FIELD_SPEC, RDB$PROCEDURE_PARAMETERS REL_FIELDS"
      " WHERE"
            " FIELD_SPEC.RDB$FIELD_NAME = REL_FIELDS.RDB$FIELD_SOURCE"
        " AND RDB$PROCEDURE_NAME = ?"
        " AND RDB$PARAMETER_NAME = ?"
        " AND RDB$PARAMETER_TYPE = 1" /* 1 is the parameter type of output parameters */
    ;
  unsigned short sql_statement_stored_procedure_length =
    strlen(sql_statement_stored_procedure);

  /* The following variables are just local "excessive dereference reducers". */
  XSQLDA *in_da, *out_da;
  XSQLVAR *in_var;

  CursorDescriptionCache *cache = cursor->connection->desc_cache;

  /* Default to normal table. */
  if (entity_type_code == ENTITY_TYPE_UNKNOWN) {
    entity_type_code = ENTITY_TYPE_TABLE;
  }

  if ( entity_name_length == 0 || field_name_length == 0 ) {
    /* Either or both of the entity name and the field name are not supplied,
    ** so we cannot determine this output field's precision.  This is not
    ** an exceptional situation; it occurs routinely in queries with
    ** dynamically computed fields (e.g., select count(*) from some_table). */
    /* DSR:2002.06.13:begin block */
    return PyInt_FromLong(0);
    /* DSR:2002.06.13:end block */
  }

  /* If the cache has not yet been allocated and prepared, do so now.
  ** If it has already been allocated, just set some local "dereference cache
  ** variables" and proceed directly to the query execution. */

  if (cache != NULL) {
    /* DSR:2002.06.13:begin block */
    /* If the precison figure for this entity.field is already cached, just
    ** retrieve it from the cache dictionary and return. */
    result_cache = cache->result_cache;
    result_cache_this_entity =
      PyDict_GetItemString(result_cache, entity_name); /* borrowed ref */

    if (result_cache_this_entity != NULL) {
      precision = PyDict_GetItemString(result_cache_this_entity, field_name);

      if (precision != NULL) {
        /* PyDict_GetItemString borrows its reference, so we need to INCREF. */
        Py_INCREF(precision);
        return precision;
      }
    } else {
      /* There is not even a cache for this entity, so there cannot possibly be
      ** one for this entity+field.  Create a new dictionary to hold the cached
      ** precision figures for this entity. */
      result_cache_this_entity = PyDict_New();
      if (result_cache_this_entity == NULL) {
        return PyErr_NoMemory();
      }

      if (PyDict_SetItemString(result_cache, entity_name, result_cache_this_entity)
          == -1
         )
      {
        return NULL;
      }
    }
    /* The precision figure was not cached; fall through and query the system
    ** tables. */
    /* DSR:2002.06.13:end block */

    in_da = cache->in_da;
    out_da = cache->out_da;
  } else {
    /* Allocate the cache structure. */
    cache = cursor->connection->desc_cache = malloc(sizeof(CursorDescriptionCache));

    /* DSR:2002.06.13:begin block */
    /* This dictionary will cache the precision figures that have been
    ** determined via queries to system tables. */
    result_cache = cache->result_cache = PyDict_New();
    if ( result_cache == NULL ) {
      return PyErr_NoMemory();
    }
    /* There was no cache at all, so there could not have been a cache for this
    ** entity.  Create one. */
    result_cache_this_entity = PyDict_New();
    if ( result_cache_this_entity == NULL ) {
      return PyErr_NoMemory();
    }

    if (PyDict_SetItemString(result_cache, entity_name, result_cache_this_entity)
        == -1
       )
    {
      return NULL;
    }
    /* DSR:2002.06.13:end block */

    /* Set up the output structures.  We know at design time exactly how they
    ** should be configured; there's no convoluted dance of dynamism here, as
    ** there is in servicing a generic Python-level query. */
    cache->out_da = (XSQLDA *)malloc(XSQLDA_LENGTH(1));
    out_da = cache->out_da;
    out_da->version = SQLDA_VERSION1;
    out_da->sqln = 1;

    /* Set up the input structures (again, their configuration is mostly
    ** static). */
    cache->in_da = (XSQLDA *)malloc(XSQLDA_LENGTH(2));
    in_da = cache->in_da;
    in_da->version = SQLDA_VERSION1;
    in_da->sqln = 2;
    in_da->sqld = 2;

    in_da->sqlvar      ->sqltype = SQL_TEXT;
    (in_da->sqlvar + 1)->sqltype = SQL_TEXT;

    /* Allocate the statement structures. */
    /* MUST set statement handles to NULL before isc_dsql_allocate_statement
    ** calls. */
    cache->stmt_handle_table = NULL;
    isc_dsql_allocate_statement( cursor->status_vector,
        &(cursor->connection->db_handle),
        &(cache->stmt_handle_table)
      );

    cache->stmt_handle_stored_procedure = NULL;
    isc_dsql_allocate_statement( cursor->status_vector,
        &(cursor->connection->db_handle),
        &(cache->stmt_handle_stored_procedure)
      );

    if ( DB_API_ERROR(cursor->status_vector) ) {
      raise_sql_exception( OperationalError,
          "Unable to determine field precison from system tables: ",
          cursor->status_vector
        );
      close_cursor(cursor);
      return NULL;
    }

    /* Prepare the statements. */
    isc_dsql_prepare( cursor->status_vector,
        &(cursor->connection->trans_handle),
        &(cache->stmt_handle_table),
        sql_statement_table_length,
        (char*)sql_statement_table,
        cursor->connection->dialect,
        out_da
      );

    isc_dsql_prepare( cursor->status_vector,
        &(cursor->connection->trans_handle),
        &(cache->stmt_handle_stored_procedure),
        sql_statement_stored_procedure_length,
        (char*)sql_statement_stored_procedure,
        cursor->connection->dialect,
        out_da
      );

    if ( DB_API_ERROR(cursor->status_vector) ) {
      raise_sql_exception( OperationalError,
          "Unable to determine field precison from system tables: ",
          cursor->status_vector
        );
      close_cursor(cursor);
      return NULL;
    }

    /* cache->out_var points to the first (and only) element of
    ** out_da->sqlvar ; cache->out_var is nothing more than a convenient
    ** reference. */
    cache->out_var = out_da->sqlvar;
    cache->out_var->sqldata = (char*)malloc(sizeof(short));
    /* We won't actually use the null status of the output field (it will never
    ** be NULL), but the API requires that space be allocated anyway. */
    cache->out_var->sqlind = (short *) malloc(sizeof(short));
  } /* We are now done (allocating and preparing new)/(loading references to
    ** existing) description cache structure. */

  /* Set the names of the relation.field for which we're determining precision. */
  in_var = in_da->sqlvar; /* First input variable. */
  in_var->sqllen = entity_name_length;
  in_var->sqldata = entity_name;

  in_var++; /* Second input variable. */
  in_var->sqllen = field_name_length;
  in_var->sqldata = field_name;

  /* Execute the prepared statement. */
  switch (entity_type_code) {
    case ENTITY_TYPE_TABLE:
      isc_dsql_execute2( cursor->status_vector,
          &(cursor->connection->trans_handle),
          &(cache->stmt_handle_table),
          cursor->connection->dialect,
          in_da,
          out_da
        );
      break;
    case ENTITY_TYPE_STORED_PROCEDURE:
      isc_dsql_execute2( cursor->status_vector,
          &(cursor->connection->trans_handle),
          &(cache->stmt_handle_stored_procedure),
          cursor->connection->dialect,
          in_da,
          out_da
        );
      break;
    default:
      raise_exception(InternalError, "determine_field_precision called with"
        " invalid entity type directive.");
  }

  if ( DB_API_ERROR(cursor->status_vector) ) {
    /* If we've run out of entity types to try, we must give up and raise an
    ** error. */
    if (entity_type_code == ENTITY_TYPE_LAST) {
      raise_sql_exception( InternalError,
          "Unable to determine field precison from system tables: ",
          cursor->status_vector
        );
      close_cursor( cursor );
      return NULL;
    } else {
      /* Recursively try the next alternative entity type. */
      precision = determine_field_precision(
          (short) (entity_type_code + 1),
          entity_name, entity_name_length,
          field_name, field_name_length,
          cursor
        );
    }
  } else {
    /* DSR:2002.06.13:begin block */
    /* Both PyInt_FromLong and PyDict_SetItemString create new references, so
    ** there's no need to increment the reference count of precision here. */
    precision = PyInt_FromLong( (long) *( (short*)cache->out_var->sqldata ) );

    /* Cache the precision figure. */
    if (PyDict_SetItemString(result_cache_this_entity, field_name, precision)
        == -1
       )
    {
      return NULL;
    }
    /* DSR:2002.06.13:end block */
  }

  return precision;
}
#endif
/* DSR:2002.07.10:end block */
/* DSR:2002.06.09:end block */


static PyObject* XSQLDA2Description( XSQLDA* sqlda, CursorObject* cursor ) {
  int i;
  PyObject* tuple;
  PyObject* o;
  PyObject* type;
  XSQLVAR* sqlvar;
  short data_type;
  int display_size = -1;
  int internal_size;
  /* DSR:2002.06.13:begin block */
  PyObject* precision;
  /* DSR:2002.06.13:end block */
  int scale;
  int null_ok;

  tuple = PyList_New( sqlda->sqld );

  for ( i = 0; i < sqlda->sqld; i++ ) {
    sqlvar = sqlda->sqlvar + i;
    o = PyList_New( 7 );

    data_type = sqlvar->sqltype & ~1;
    internal_size = (int)( sqlvar->sqllen );
    scale = (int)( sqlvar->sqlscale );
    /* DSR:2002.06.09:begin block */
    /* The old, totally invalid approach was to set precision to zero in every
    ** case. */
    /* DSR:2002.07.10:begin block */
    #ifdef DETERMINE_FIELD_PRECISION
      precision = determine_field_precision(
          ENTITY_TYPE_UNKNOWN,
          sqlvar->relname, sqlvar->relname_length,
          sqlvar->sqlname, sqlvar->sqlname_length,
          cursor
        );

      if ( precision == NULL ) {
        /* Tried to determine the field's precision, but encountered an
        ** exception while doing so.
        ** determine_field_precision will have set up the exception, so we
        ** just need to return NULL. */
        return NULL;
      }
    #else
      precision = PyInt_FromLong(0);
    #endif
    /* DSR:2002.07.10:end block */
    /* DSR:2002.06.09:end block */

    null_ok = sqlvar->sqltype & 1;

    switch ( data_type ) {
    case SQL_TEXT:
    case SQL_VARYING:
      display_size = (int)( sqlvar->sqllen );
      Py_XINCREF( &PyString_Type );
      type = (PyObject*)( &PyString_Type );
      break;

    case SQL_SHORT:
      Py_XINCREF( &PyInt_Type );
      type = (PyObject*)( &PyInt_Type );
      display_size = 6;
      break;

    case SQL_LONG:
      Py_XINCREF( &PyInt_Type );
      type = (PyObject*)( &PyInt_Type );
      display_size = 12;
      break;

    case SQL_DOUBLE:
    case SQL_FLOAT:
    case SQL_D_FLOAT:
      Py_XINCREF( &PyFloat_Type );
      type=(PyObject*)( &PyFloat_Type );
      display_size = 17;
      break;

    case SQL_TIMESTAMP:
      display_size = 22;
      Py_XINCREF( mxDateTime.DateTime_Type );
      type = (PyObject*)( mxDateTime.DateTime_Type );
      break;

    case SQL_BLOB:
      display_size = 0;
      scale = sqlvar->sqlsubtype;
      Py_XINCREF( &PyBuffer_Type );
      type = (PyObject*)( &PyBuffer_Type );
      break;

#ifdef INTERBASE6_OR_LATER
    case SQL_TYPE_DATE:
      display_size = 10;
      Py_XINCREF( mxDateTime.DateTime_Type );
      type = (PyObject*)( mxDateTime.DateTime_Type );
      break;

    case SQL_TYPE_TIME:
      display_size = 11;
      Py_XINCREF( mxDateTime.DateTimeDelta_Type );
      type = (PyObject*)( mxDateTime.DateTimeDelta_Type );
      break;

    case SQL_INT64:
      display_size = 17;
      Py_XINCREF( &PyLong_Type );
      type = (PyObject*)( &PyLong_Type );
      break;
#endif

    default:
      type = Py_None;
    }

    /*
    ** DSR: begin block 2001.12.31
    **
    ** Fix for the alias ignore bug:
    **
    **   If there is an alias, place the alias, rather than the real
    **   column name, in the column name field of the descriptor tuple.
    **   Before this fix, the presence of an alias made no difference
    **   whatsoever in the descriptor setup, and was thus inaccessible
    **   to the client programmer.
    */
    if (    ( sqlvar->aliasname_length != sqlvar->sqlname_length )
         || ( strcmp( sqlvar->sqlname, sqlvar->aliasname ) != 0 )
       )
    {
      PyList_SetItem( o, 0,
                      PyString_FromStringAndSize( sqlvar->aliasname,
                                                  (int)( sqlvar->
                                                         aliasname_length ) ) );
    } else {
      PyList_SetItem(o, 0,
                     PyString_FromStringAndSize( sqlvar->sqlname,
                                                 (int)( sqlvar->
                                                        sqlname_length ) ) );
    }
    /* DSR: end block 2001.12.31 */

    Py_INCREF( type );
    PyList_SetItem( o, 1, type );
    PyList_SetItem( o, 2, PyInt_FromLong( (long)( display_size ) ) );
    PyList_SetItem( o, 3, PyInt_FromLong( (long)( internal_size ) ) );
    /* DSR:2002.06.13:begin block */
    PyList_SetItem( o, 4, precision );
    /* DSR:2002.06.13:end block */
    PyList_SetItem( o, 5, PyInt_FromLong( (long)( scale ) ) );
    PyList_SetItem( o, 6, PyInt_FromLong( (long)( null_ok ) ) );
    PyList_SetItem( tuple, i, o );
  }

  return tuple;
} /* XSQLDA2Description */

/* DSR:2002.07.18:begin block */
static PyObject* DB_TIMESTAMP_To_PyObject( ISC_TIMESTAMP *db_timestamp ) {
  PyObject* result;
  struct tm c_time_struct;

  /* OLD: isc_decode_date( quad, &t ); */
  isc_decode_timestamp( db_timestamp, &c_time_struct );
  result = mxDateTime.DateTime_FromTmStruct( &c_time_struct );

  return result;
} /* DB_TIMESTAMP_To_PyObject */

#ifdef INTERBASE6_OR_LATER
static PyObject* DB_DATE_To_PyObject( ISC_DATE *db_date ) {
  PyObject* result;
  struct tm c_time_struct;

  /* OLD (functionally equivalent): isc_decode_sql_date( quad, &t ); */
  isc_decode_sql_date( db_date, &c_time_struct );
  result = mxDateTime.DateTime_FromTmStruct( &c_time_struct );

  return result;
} /* ibDate2PyObject */

static PyObject* DB_TIME_To_PyObject( ISC_TIME *db_time ) {
  PyObject* result;
  struct tm c_time_struct;

  /* OLD (functionally equivalent): isc_decode_sql_time( quad, &t ); */
  isc_decode_sql_time( db_time, &c_time_struct );
  result = mxDateTime.DateTimeDelta_FromTime(
      c_time_struct.tm_hour, c_time_struct.tm_min, c_time_struct.tm_sec
    );

  return result;
} /* ibTime2PyObject */
#endif /* INTERBASE6_OR_LATER */
/* DSR:2002.07.18:end block */


/* DSR:2002.02.23:
** The old Blob2PyObject was sickeningly slow with large blobs because it
** a) ignored this particular blob's segment size and used 1024
** b) realloc'd the entire buffer after fetching each 1024 bytes
**
** I reimplemented the function this way:
** - determine the total size of this blob and its maximum segment size at
**   the outset
** - pre-allocate a Python buffer (PyBufferObject) large enough to hold the
**   entire blob
** - using the segment size appropriate for this blob, transfer each segment
**   directly into the Python buffer using the Python buffer interface
**
** The new implementation is exponentially faster, though the difference isn't
** necessarily noticeable with small blobs.
*/
static PyObject* Blob2PyObject( CursorObject* cursor, ISC_QUAD* blob_id ) {
  isc_blob_handle blob_handle = NULL;

  ISC_LONG total_size;
  unsigned short max_segment_size;

  ISC_LONG bytes_read_so_far;
  unsigned short bytes_actually_read;

  PyObject* py_buf;
  PyBufferProcs *bufferProcs;
  char* py_buf_start_ptr;

  int blob_stat;

  /* Get a handle to the blob. */
  isc_open_blob2( cursor->status_vector,
                  &(cursor->connection->db_handle),
                  &(cursor->connection->trans_handle),
                  &blob_handle,
                  blob_id,
                  0,
                  NULL );

  if ( DB_API_ERROR(cursor->status_vector) ) {
    raise_sql_exception( OperationalError,
                         "Blob2PyObject.isc_open_blob2: ",
                         cursor->status_vector );
    return NULL;
  }

  /* Determine the total size of the blob and the size of its largest segment. */
  if ( 0 !=
    _blob_info_total_size_and_max_segment_size(
        cursor->status_vector,
        blob_handle,
        &total_size,
        &max_segment_size
      )
  ) {
    /* _blob_info_total_size_and_max_segment_size has already raised
    ** an exception.
    */
    return NULL;
  }

  debug_printf2("[in Blob2PyObject] total_size is %d and max_segment_size is"
    " %d\n", total_size, max_segment_size);

  /* Create a PyBufferObject large enough to hold the entire blob. */

  /* Handle the very remote possibility that passing an ISC_LONG to
  ** PyBuffer_New would cause an overflow.
  */
  if ( total_size > INT_MAX ) {
      raise_exception(InternalError,
          "Blob2PyObject: blob too large; kinterbasdb only supports blob"
          " sizes up to system-defined INT_MAX."
        );
      return NULL;
  }
  py_buf = PyBuffer_New( (int) total_size );

  /* Get a pointer to the PyBufferObject's getwritebuffer method, then call
  ** that method, which will make py_buf_start_ptr point to the start of the
  ** PyBufferObject's raw data buffer.
  */
  bufferProcs = py_buf->ob_type->tp_as_buffer;
  (*bufferProcs->bf_getwritebuffer)(
    py_buf, 0, (void **)&py_buf_start_ptr );

  bytes_read_so_far = 0;
  while ( bytes_read_so_far < total_size ) {
    blob_stat = isc_get_segment(
        cursor->status_vector,
        &blob_handle,
        &bytes_actually_read,
        /* DSR:2002.03.24:begin block */
        /* Here's a fix from Scott Yang for a crash that apparently only occurs
        ** on Windows 2000 when this is the last iteration of the loop (the
        ** last in a series of isc_get_segment calls) and the seg_buffer_length
        ** argument to isc_get_segment does not match the actual remaining
        ** space in py_buf.
        */
        /* Was:
        **   max_segment_size,
        ** Changed to: */
        (unsigned short) MIN( (long)max_segment_size, total_size - bytes_read_so_far ),
        /* DSR:2002.03.24:end block */
        py_buf_start_ptr + bytes_read_so_far
      );

    /* isc_get_segment can return non-zero values under normal circumstances,
    ** but not under circumstances that would be normal RIGHT HERE, because
    ** we already knew the exact size of the blob and the length of its
    ** longest segment before we started this loop.
    ** Therefore, there is no need to check specifically for isc_segment or
    ** isc_segstr_eof.
    */
    if ( blob_stat != 0 ) {
      raise_sql_exception(OperationalError,
          "Blob2PyObject.isc_get_segment: ",
          cursor->status_vector
        );
      Py_DECREF( py_buf );
      return NULL;
    }

    bytes_read_so_far += bytes_actually_read;
  }

  isc_close_blob( cursor->status_vector, &blob_handle );

  return py_buf;
} /* Blob2PyObject */


static PyObject* XSQLVAR2PyObject( CursorObject* cursor, XSQLVAR* sqlvar ) {
  PyObject* result = NULL;

  /* Crucial information about the data we're trying to convert: */
  short data_type = sqlvar->sqltype & ~1;
  short num_decimal_places = -(sqlvar->sqlscale);
  /* DSR:bugfixes #517093 and #522774:2002.02.28:begin block */
  char precision_mode = cursor->connection->precision_mode;
  /* DSR:bugfixes #517093 and #522774:2002.02.28:end block */

  /* Utility variables for the conversion process: */
  long conv_long;
  double conv_double;

  /* DSR:2002.07.25:begin block */
  if (
       XSQLVAR_IS_ALLOWED_TO_BE_NULL(sqlvar)
    && XSQLVAR_IS_NULL(sqlvar)
  ) {
    /* SQL NULL becomes Python None regardless of field type. */
    Py_INCREF(Py_None);
    return Py_None;
  }
  /* DSR:2002.07.25:end block */

  /* For documentation of these data_type options, see the IB6 API Guide
  ** section entitled "SQL datatype macro constants".
  */
  switch ( data_type ) {
  /* Character data: */
  case SQL_TEXT:
    /* DSR:2002.06.17:begin block */
    /* The original approach (still shown below, but commented out) almost
    ** always misbehaved when the value was shorter than the field's declared
    ** length.  The 'empty area' of the returned string was filled with null
    ** characters, not spaces.  I initially tried to fill in the spaces myself,
    ** but ran into trouble with multibyte character sets.
    ** Of course, the new implementation disallows the retrieval of null-
    ** containing strings from CHAR fields, but one should use a BLOB to store
    ** null-containing strings anyway.
    result = PyString_FromStringAndSize( sqlvar->sqldata, sqlvar->sqllen );
    */
    { /* End the string at the maximum possible length of the field
      ** (sqlvar->sqllen) or the first null byte, whichever comes first. */
    unsigned short max_possible_bytes_that_comprise_value = sqlvar->sqllen;
    unsigned short bytes_before_first_null_byte = strlen(sqlvar->sqldata);
    result = PyString_FromStringAndSize( sqlvar->sqldata,
        MIN(bytes_before_first_null_byte, max_possible_bytes_that_comprise_value)
      );
    }
    /* DSR:2002.06.17:end block */
    break;

  case SQL_VARYING:
    /* DSR:2002.06.15:begin block */
    /* The first sizeof(short) bytes of sqlvar->sqldata contain the length of
    ** the string. */
    result = PyString_FromStringAndSize(
        sqlvar->sqldata + sizeof(short),
        (int) *( (short*)sqlvar->sqldata )
      );
    /* DSR:2002.06.15:end block */
    break;

  /* Numeric data: */

  case SQL_SHORT:
  case SQL_LONG:
    /* DSR:bugfixes #517093 and #522774:2002.02.28:begin block */
    conv_long = (data_type == SQL_SHORT
        ? *(short*)sqlvar->sqldata
        : *(long*)sqlvar->sqldata
      );

    if ( num_decimal_places == 0 || precision_mode != PRECISION_MODE_IMPRECISE ) {
      /* One of the following is true:
      ** - this is a NUMERIC/DECIMAL field being requested in precise mode as
      **   an uninterpreted integer
      ** - this is a NUMERIC/DECIMAL with no decimal places
      ** - this is not a NUMERIC/DECIMAL field at all (just a plain
      **   SMALLINT/INTEGER)
      */
      result = PyInt_FromLong( conv_long );
    } else {
      /* This is a NUMERIC/DECIMAL field being requested in imprecise mode as a
      ** float, so scale the result. */
      result = PyFloat_FromDouble( conv_long / pow(10, num_decimal_places) );
    }
    /* DSR:bugfixes #517093 and #522774:2002.02.28:end block */
    break;

#ifdef INTERBASE6_OR_LATER
  case SQL_INT64:
    {
    /* DSR:bugfixes #517093 and #522774:2002.02.28:begin block */
    /* This particular block of code is a slight modification of Janez's patch
    ** from circa 2002.01.19.  Like the original, it will not return huge
    ** values properly in imprecise mode (the 64-bit int will overflow the
    ** double).  This can be overcome by using precise mode.
    */
    LONG_LONG conv_long_long = *(LONG_LONG*)( sqlvar->sqldata );
    if ( num_decimal_places == 0 || precision_mode != PRECISION_MODE_IMPRECISE ) {
      /* This is either a NUMERIC/DECIMAL field being requested in precise
      ** mode as an uninterpreted integer, or a NUMERIC/DECIMAL field being
      ** requested in imprecise mode as an interpreted integer (but it
      ** requires not "interpretation" because there are no decimal places).
      */
      result = PyLong_FromLongLong( conv_long_long );
    } else {
      /* This is a NUMERIC/DECIMAL being requested in imprecise mode as a
      ** float.
      */
      result = PyFloat_FromDouble( conv_long_long
          / pow(10, num_decimal_places) );
    }
    /* DSR:bugfixes #517093 and #522774:2002.02.28:end block */
    }
    break;
#endif /* INTERBASE6_OR_LATER */

  case SQL_FLOAT:
    conv_double = *(float*)sqlvar->sqldata;
    result = PyFloat_FromDouble( conv_double );
    break;

  case SQL_DOUBLE:
  case SQL_D_FLOAT:
    conv_double = *(double*)sqlvar->sqldata;
    result = PyFloat_FromDouble( conv_double );
    break;

  /* Date and time data: */
  /* DSR:2002.07.18:begin block */
  case SQL_TIMESTAMP: /* TIMESTAMP */
    result = DB_TIMESTAMP_To_PyObject( (ISC_TIMESTAMP*) sqlvar->sqldata );
    break;

#ifdef INTERBASE6_OR_LATER
  case SQL_TYPE_DATE: /* DATE */
    result = DB_DATE_To_PyObject( (ISC_DATE*) sqlvar->sqldata );
    break;

  case SQL_TYPE_TIME: /* TIME */
    result = DB_TIME_To_PyObject( (ISC_TIME*) sqlvar->sqldata );
    break;
#endif /* INTERBASE6_OR_LATER */
  /* DSR:2002.07.18:end block */

  /* Blob data (including blobs of subtype TEXT): */
  case SQL_BLOB:
    /* DSR:2002.04.13:begin comment */
    /* In theory, I should test to ensure that the blob size does not exceed
    ** the maximum size of a Python buffer (2GB), but in practice, anyone who
    ** stores a single blob larger than 2GB is very probably an idiot; I have
    ** no interest in jumping through hoops to prevent him from shooting his
    ** last toe off.
    */
    /* DSR:2002.04.13:end comment */
    result = Blob2PyObject( cursor, (ISC_QUAD*)sqlvar->sqldata );
    break;

  /* DSR:2002.03.24:begin block */
  /* Fix failure to raise specific error message regarding lack of ARRAY
  ** support (reported by Phil Harris).
  */
  case SQL_ARRAY:
    raise_exception( NotSupportedError, "kinterbasdb does not support the"
      " database engine's ARRAY datatype.  Consult the kinterbasdb Usage"
      " Guide for details.");
    break;
  /* DSR:2002.03.24:end block */

  default:
    raise_exception( NotSupportedError,
                     "XSQLVAR2PyObject: type not supported"
                     KIDB_REPORT KIDB_HOME_PAGE );
    return NULL;
  }

  return result;
} /* XSQLVAR2PyObject */


static PyObject* XSQLDA2Tuple( CursorObject* cursor, XSQLDA* sqlda ) {
  int i;
  PyObject* record = PyTuple_New( sqlda->sqld );
  PyObject* o;

  for ( i = 0; i < sqlda->sqld; i++ ) {
    o = XSQLVAR2PyObject( cursor, sqlda->sqlvar + i );
    if ( !o ) {
      Py_DECREF( record );
      return NULL;
    }
    PyTuple_SetItem( record, i, o );
  }
  return record;
} /* XSQLDA2Tuple */

/******************* CONVERSION FUNCTIONS:END ********************/


/********************** EXECUTION FUNCTIONS:BEGIN ***********************/

/* DSR:2002.02.25:
** For an explanation of this function's purpose, see the documentation for
** Python function kinterbasb.create_database.
*/
static PyObject* pyob_create_database( PyObject* self, PyObject* args ) {
  ConnectionObject* connection;
  int sql_len;
  char* sql;

  connection = new_connection();
  if ( connection == NULL ) {
    raise_exception( InternalError,
                     "pyob_create_database.new_connection: "
                     "Cannot allocate a new connection."
                     KIDB_REPORT KIDB_HOME_PAGE );
    return NULL;
  }

  if ( !PyArg_ParseTuple(args, "s#|i",
          &sql, &sql_len,
          &(connection->dialect)
        )
  ) {
    return NULL;
  }

  isc_dsql_execute_immediate(
      connection->status_vector,
      &(connection->db_handle),
      &(connection->trans_handle),
      (unsigned short)sql_len,
      sql,
      connection->dialect,
      NULL
    );

  if ( DB_API_ERROR(connection->status_vector) ) {
    raise_sql_exception(ProgrammingError, "pyob_create_database: ",
      connection->status_vector);
    return NULL;
  }

  connection->_state = CONNECTION_STATE_OPEN;

  return (PyObject*) connection;
} /* pyob_create_database */


/* DSR:2002.03.27:
** For an explanation of this function's purpose, see the documentation for
** Python function kinterbasb.drop_database.
*/
static PyObject* pyob_drop_database( PyObject* self, PyObject* args ) {
  ConnectionObject* connection;

  if ( !PyArg_ParseTuple(args, "O!", &ConnectionType, &connection) ) {
    return NULL;
  }

  isc_drop_database( connection->status_vector, &(connection->db_handle) );

  if ( DB_API_ERROR(connection->status_vector) ) {
    raise_sql_exception(OperationalError, "pyob_drop_database: ",
      connection->status_vector);
    return NULL;
  }

  Py_INCREF( Py_None );
  return Py_None;
} /* pyob_drop_database */


/* DSR:2002.02.25:
** execute_immediate in the context of a connection.
** Like isc_dsql_execute_immediate (which it wraps), this function cannot
** execute SQL statements that return result sets.
**
** It would be easy to separately wrap isc_dsql_exec_immed2, which can return
** up to one row of output, but I don't see the point.  Note that
** isc_dsql_exec_immed2 would have to be SEPARATELY wrapped, because here
** (unlike in pyob_execute), we don't have a prepared representation of
** the query, and therefore CANNOT do a similar dynamic selection of execution
** function, for we don't know the statement type.
*/
static PyObject* pyob_execute_immediate_with_connection (
    PyObject* self,
    PyObject* args
  )
{
  ConnectionObject* connection;
  char* sql;
  int sql_len;

  if ( !PyArg_ParseTuple( args, "O!s#",
                          &ConnectionType, &connection,
                          &sql, &sql_len ) ) {
    return NULL;
  }


  CONN_REQUIRE_OPEN(connection);

  if (connection->trans_handle == NULL) {
    raise_exception(ProgrammingError,
        "Invalid connection state.  The connection must have an active"
        " transaction to perform this operation."
        "  Suggestion: call the connection's 'begin' method to explicitly"
        " start a transaction."
      );
    return NULL;
  }

  isc_dsql_execute_immediate( connection->status_vector,
                              &(connection->db_handle),
                              &(connection->trans_handle),
                              (unsigned short) sql_len,
                              sql,
                              connection->dialect,
                              NULL );

  if ( DB_API_ERROR(connection->status_vector) ) {
    raise_sql_exception( ProgrammingError,
                         "execute_immediate.isc_dsql_execute_immediate: ",
                         connection->status_vector );
    return NULL;
  }

  Py_INCREF( Py_None );
  return Py_None;
} /* pyob_execute_immediate_with_connection */

/* XXX:DSR:2002.06.09: The generic failure actions in this function ought to
** be factored into a generic error handler block that can be goto'ed, instead
** of repeating the same decrefs, etc. in all of the many failure cases. */
static PyObject* pyob_execute( PyObject* self, PyObject* args ) {
  CursorObject* cursor;
  /* DSR:2002.06.09:begin block */
  PyObject* cursor_description;
  /* DSR:2002.06.09:end block */
  char* sql;
  int sqllen;
  PyObject* params = NULL;
  PyObject* o;
  int need_prep;
  int statementType;

/* DSR:2002.03.31:begin block */
  if ( !PyArg_ParseTuple(
          args, "O!s#|O",
          &CursorType, &cursor, &sql, &sqllen, &params
        )
     )
  {
    return NULL;
  }

  /* Accept any sequence except a string. */
  if ( !PySequence_Check(params) ) {
    raise_exception( InterfaceError,
                     "input parameter container is not a sequence" );
    return NULL;
  }

  if ( PyString_Check(params) ) {
    raise_exception( InterfaceError,
                     "input parameter sequence cannot be a string" );
    return NULL;
  }
/* DSR:2002.03.31:end block */

  CUR_REQUIRE_OPEN(cursor);

  /* DSR:2002.04.27:begin block */
  cursor->last_fetch_status = 0;
  /* DSR:2002.04.27:end block */

  if ( !params ) {
    params = PyTuple_New( 0 );
  } else {
    Py_INCREF( params );
  }

  need_prep = 1;

  shutdown_cursor( cursor );  /* closing previous cursor if opened */

  if ( cursor->lastsql && ( sqllen == cursor->lastsql_len ) )
    if ( strncmp( sql, cursor->lastsql, sqllen ) == 0 )
      need_prep = 0;

  /* starting transaction if not started */
  if ( !cursor->connection->trans_handle ) {
    debug_printf( "[in pyob_execute] starting a transaction\n" );
    if ( start_transaction( cursor->connection, NULL, 0 ) != 0 ) {
      return NULL;
    }
  }
  if ( !need_prep ) {
    cursor->_state = CURSOR_STATE_OPEN;
  } else {
    close_cursor( cursor );
    isc_dsql_prepare( cursor->status_vector,
                      &(cursor->connection->trans_handle),
                      &(cursor->stmt_handle),
                      (unsigned short)sqllen,
                      sql,
                      cursor->connection->dialect,
                      cursor->out_sqlda );

    if ( DB_API_ERROR(cursor->status_vector) ) {
      /* DSR:2002.02.21:  The exception raised below was an OperationalError.
      ** The DB API spec plainly states that a ProgrammingError is the exception
      ** to be raised in case of "table not found... syntax error in the SQL
      ** statement... ,etc.".  Those are both errors raised by isc_dsql_prepare,
      ** so I changed the exception type raised below to ProgrammingError.
      */
      raise_sql_exception( ProgrammingError,
                           "execute.isc_dsql_prepare: ",
                           cursor->status_vector );
      Py_DECREF( params );
      return NULL;
    }

    /* The cursor was closed before the isc_dsql_prepare call, so we need to
    ** indicate that it is now open again.
    */
    cursor->_state = CURSOR_STATE_OPEN;

    /* DSR:2002.06.09:begin block */
    if ( reallocate_sqlda( &(cursor->out_sqlda) ) != 0 ) {
      free_cursor_cache( cursor );
      Py_DECREF( params );
      return NULL;
    } else { /* Reallocation succeeded. */
    /* DSR:2002.06.09:end block */
      isc_dsql_describe( cursor->status_vector,
                         &(cursor->stmt_handle),
                         cursor->connection->dialect,
                         cursor->out_sqlda );

      if ( DB_API_ERROR(cursor->status_vector) ) {
        raise_sql_exception( OperationalError,
                             "execute.isc_dsql_describe[2]: ",
                             cursor->status_vector );
        free_cursor_cache( cursor );
        Py_DECREF( params );
        return NULL;
      }
    }

    isc_dsql_describe_bind( cursor->status_vector,
                            &(cursor->stmt_handle),
                            cursor->connection->dialect,
                            cursor->in_sqlda );

    if ( DB_API_ERROR(cursor->status_vector) ) {
      raise_sql_exception( OperationalError,
                           "execute.isc_dsql_describe_bind: ",
                           cursor->status_vector );
      free_cursor_cache( cursor );
      Py_DECREF( params );
      return NULL;
    }

    /* DSR:2002.06.09:begin block */
    if ( reallocate_sqlda( &(cursor->in_sqlda) ) != 0 ) {
      free_cursor_cache( cursor );
      Py_DECREF( params );
      return NULL;
    } else { /* Reallocation succeeded. */
    /* DSR:2002.06.09:end block */
      isc_dsql_describe_bind( cursor->status_vector,
                              &(cursor->stmt_handle),
                              cursor->connection->dialect,
                              cursor->in_sqlda );

      if ( DB_API_ERROR(cursor->status_vector) ) {
        raise_sql_exception( OperationalError,
                             "execute.isc_dsql_describe_bind[2]: ",
                             cursor->status_vector );
        free_cursor_cache( cursor );
        Py_DECREF( params );
        return NULL;
      }
    }
  } /* if need prep */

  /* allocating buffer for input parameters */
  if ( cursor->in_buffer ) {
    free( cursor->in_buffer );
    cursor->in_buffer = NULL;
  }

  cursor->in_buffer = allocate_buffer( cursor->in_sqlda );

  /* Convert the Python input arguments to their XSQLVAR equivalents. */
  if ( PyObject2XSQLDA( cursor, cursor->in_sqlda, params ) < 0 ) {
    free_cursor_cache( cursor );
    Py_DECREF( params );
    return NULL;
  }

  if ( cursor->out_sqlda->sqld ) { /* query statements */
    if ( cursor->out_buffer ) {
      free( cursor->out_buffer );
    }
    cursor->out_buffer = allocate_buffer( cursor->out_sqlda );
  } else { /* non-query statements */
    debug_printf("[in pyob_execute] executing nonquery statement\n");
  }

  /*
  ** DSR: 2002.01.01
  **
  ** It looks like the original author was dealing with some similar
  ** EXECUTE PROCEDURE troubles, but they weren't the same as the one I'm
  ** here to fix.
  **
  ** The fact that there are output fields (that is,
  ** cursors[cur].in_sqlda->sqld is >= 1) does not guarantee that the
  ** statement is compatible with a standard execute-fetch approach; it
  ** could be an EXECUTE on a stored procedure with output params.
  **
  ** Special case:
  **   The cursor's statement type is isc_info_sql_stmt_exec_procedure
  **   and it has at least one output parameter.
  **
  **   Unlike a SELECT statement whose target is a stored procedure, an
  **   EXECUTE PROCEDURE statement must not return more than one row.
  **   This is because such a procedure must be executed with
  **   isc_dsql_execute2, which returns its results immediately and is
  **   therefore not compatible with a standard execute-fetch approach.
  **
  **   However, I later (2002.02.21) imposed the "appearance of execute-fetch"
  **   on even this special case by caching the results in the cursor object
  **   and returning the ONE *cached* row if/when fetch is called.
  **   This behavior was set up in order to satisfy the Python DB API spec).
  **
  **   kinterbasdb up to and including version 2.0-0.3.1 would choke with an
  **   exception instead of behaving in the standard way.  It did so because
  **   it tried to execute the procedure in this special case with
  **   isc_dsql_execute rather than isc_dsql_execute2.  That appeared to work
  **   fine during the execution step, but gagged if/when fetch was
  **   subsequently called.
  */
  /* YYY:DSR:2002.02.17:  The statement type could be cached for better
  ** performance, in a manner similar to that of cursor->lastsql and
  ** cursor->lastsql_len.  determineStatementType calls isc_dsql_sql_info,
  ** which ?may? be expensive.
  */
  statementType = determineStatementType( &(cursor->stmt_handle),
                                          cursor->status_vector );
  debug_printf1( "[in pyob_execute] STATEMENT TYPE: %d\n", statementType );

  if ( ( statementType == isc_info_sql_stmt_exec_procedure ) &&
       ( cursor->out_sqlda->sqld > 0 ) ) {
    /* DSR:2002.01.01:
    ** The crucial difference between isc_dsql_execute and
    ** isc_dsql_execute2 is that the latter loads information about the
    ** first output row into the output structures immediately, without
    ** waiting for a call to isc_dsql_fetch().  It is important to prevent
    ** isc_dsql_fetch from being called on a cursor that has been executed
    ** with isc_dsql_execute2.
    ** DSR:2002.02.21:
    ** Although it true that isc_dsql_fetch must not be called on a cursor
    ** that has been executed with isc_dsql_execute2, we must impose the
    ** appearance of that behavior in order to satisfy the Python DB API
    ** Specification 2.0, which does not allow a direct return of stored
    ** procedure results unless they are output parameters or
    ** input/output parameters (in the sense that a client variables is passed
    ** in and its memory space if filled with the output value).  IB and FB
    ** lack such "shared memory space" parameters, so even though we have
    ** the result row immediately after calling isc_dsql_execute2, we not
    ** return it directly but instead save it and wait for a possible fetch
    ** call at some later point.
    ** The current implementation caches the single result row from
    ** isc_dsql_execute2 in cursor->exec_proc_results.
    */
    isc_dsql_execute2( cursor->status_vector,
                       &(cursor->connection->trans_handle),
                       &(cursor->stmt_handle),
                       cursor->connection->dialect,
                       cursor->in_sqlda,
                       cursor->out_sqlda );
/* DSR: begin block 2002.02.21 */
    debug_dump_status_vector(cursor->status_vector);

    if ( DB_API_ERROR(cursor->status_vector) ) {
      raise_sql_exception( ProgrammingError,
                           "execute.isc_dsql_execute2: ",
                           cursor->status_vector );
      close_cursor( cursor );
      Py_DECREF( params );
      return NULL;
    }

/* NEW WAY: */
    /* First, cache the result of the procedure call so that it is available
    ** if and when fetch is called in the future.
    */
    cursor->exec_proc_results = XSQLDA2Tuple( cursor, cursor->out_sqlda );

    /* Next, return the description (but not the results, as the initial
    ** fix from circa 2002.01.01 was doing), just as would be done in the
    ** case of a normal result-returning query (which we are crudely
    ** simulating via the cursor->exec_proc_results cache variable).
    */
    Py_DECREF( params );

    o = PyTuple_New( 2 );
    Py_INCREF( Py_None );
    PyTuple_SetItem( o, 0, Py_None );
    /* DSR:2002.06.09:begin block */
    /* Added cursor as second parameter of XSQLDA2Description because
    ** determine_field_precision needs access to the cursor for low-level
    ** administrative reasons. */
    cursor_description = XSQLDA2Description( cursor->out_sqlda, cursor );

    if ( cursor_description == NULL ) {
      close_cursor( cursor );
      Py_DECREF( params );
      return NULL;
    }
    PyTuple_SetItem( o, 1, cursor_description );
    /* DSR:2002.06.09:end block */

    return o;

/* OLD WAY (direct return, circa 2002.01.01): */
    /* Return the output row directly from this function. */
/*
    Py_DECREF( params );

    o = PyTuple_New( 2 );
    PyTuple_SetItem( o, 0, XSQLDA2Tuple( cursor, cursor->out_sqlda ) );
    PyTuple_SetItem( o, 1, XSQLDA2Description( cursor->out_sqlda ) );

    return o;
*/
/* DSR: end block 2002.02.21 */
  } else { /* Everything except the special case(s) defined above. */
    isc_dsql_execute( cursor->status_vector,
                      &(cursor->connection->trans_handle),
                      &(cursor->stmt_handle),
                      cursor->connection->dialect,
                      cursor->in_sqlda );

    debug_printf1("isc_sqlcode(cursor->status_vector): %d\n",
      isc_sqlcode(cursor->status_vector));
  }
  /* DSR: end block 2002.01.01 */

  /* YYY:DSR:2002.02.21:
  ** (as part of fix for bug #520793)
  ** Conceptually, there should be an error in the status vector at this
  ** point in cases where
  **   a SELECT statement whose target is
  **     a stored procedure
  **       that raises an IB user-defined EXCEPTION
  ** has just been executed.  However, said exception is present ONLY
  ** conceptually, not really.
  **
  ** Apparently, the IB API only "realizes" that a user-defined EXCEPTION has
  ** been raised when it tries to fetch the first row of output from the
  ** stored procedure in question.  Because of the way IB stored procedures
  ** work (generate... suspend; generate... suspend;), I can understand why
  ** the IB API works the way it does.
  **
  ** This quirk explains why the branch of this function that uses
  ** isc_dsql_execute2 works as it conceptually should (i.e., it "realizes"
  ** that a user-defined exception has been raised as soon as it is
  ** conceptually raised), and why the branch that uses isc_dsql_execute--
  ** the variant that does not immediately fetch the first row of output--
  ** does not raise an error until fetch is called (and fetch may never be
  ** called!).
  **
  ** Here is the basic problem:  suppose a user executes a statement like:
  ** SELECT * FROM PROCEDURE_THAT_RAISES_EXCEPTION
  ** , but the user happens not to call fetch (i.e., decides to ignore the
  ** result set).  In such a case, the user would never know that an
  ** exception had been raised, because the IB API itself would never
  ** register the fact that an exception had taken place.
  **
  ** I've searched extensively for a way to fix this, but haven't found one.
  */

  if ( DB_API_ERROR(cursor->status_vector) ) {
    /* DSR:2002.02.21:  The exception raised below used to be an
    ** OperationalError. The DB API spec plainly states that a
    ** ProgrammingError is the exception to be raised when the programmer's
    ** own error is responsible for the problem (as seems most likely for a
    ** call that executes any old SQL string).
    ** Therefore, I changed the exception raised below to ProgrammingError
    ** rather than OperationalError.
    ** Depending on how systematically the IB error codes are assigned, it
    ** might be better to filter specific ranges of IB API error codes, and
    ** thereby decide whether to raise a ProgrammingError or an
    ** OperationalError.
    */
    raise_sql_exception( ProgrammingError,
                         "execute.isc_dsql_execute: ",
                         cursor->status_vector );
/* DSR:2002.02.21:  close_cursor calls free_cursor_cache itself; no need to call it explicitly.
    free_cursor_cache( cursor );
*/
    close_cursor( cursor );
    Py_DECREF( params );
    return NULL;
  }

  if ( need_prep ) {
    /* YYY:DSR:2002.02.21:  Why wasn't function free_cursor_cache
    ** (formerly named free_lastsql) used here?
    ** I needed to have the variables which were cached for the exec_proc fix
    ** freed too, so I replaced the previous free call (which freed only the
    ** char* cursor->lastsql and nothing else) with a call to
    ** free_cursor_cache.
    */
    /* NEW (does what old did, and more): */
    free_cursor_cache( cursor );
    /* OLD:
    if ( cursor->lastsql ) {
      free( cursor->lastsql );
    }
    */

    cursor->lastsql = (char*)( malloc( sqllen + 1 ) );
    if ( cursor->lastsql == NULL ) {
      return PyErr_NoMemory();
    }

    memcpy( cursor->lastsql, sql, sqllen );
    cursor->lastsql_len = sqllen;
  }

  Py_DECREF( params );
  o = PyTuple_New( 2 );

  if ( cursor->out_sqlda->sqld ) {
    Py_INCREF( Py_None );
    PyTuple_SetItem( o, 0, Py_None );

    /* DSR:2002.06.09:begin block */
    /* Added cursor as second parameter of XSQLDA2Description because
    ** determine_field_precision needs access to the cursor for low-level
    ** administrative reasons. */
    cursor_description = XSQLDA2Description( cursor->out_sqlda, cursor );

    if ( cursor_description == NULL ) {
      close_cursor( cursor );
      Py_DECREF( params );
      return NULL;
    }
    PyTuple_SetItem( o, 1, cursor_description );
    /* DSR:2002.06.09:end block */

    return o;
  }

/* XXX:DSR:STATE:  Why shut down the cursor here, when we've not even fetched
** any result rows yet?  If there are no result rows, then the cursor
** doesn't need to be shut down; if there are, it will be shut down when the
** next statement is prepared, or upon cursor garbase collection.
** Since I began tracking cursor state in order to comply with the DB API
** spec, these gratuitous shutdowns have suddenly come to light.  They should
** be dealt with (and disabled in the meantime) because the program is now
** aware of cursor state and throws an exception when an operation is executed
** on a cursor that is "logically closed but really open".
*/
/*  shutdown_cursor( cursor ); */

  PyTuple_SetItem( o, 0, PyInt_FromLong( 1l ) );
  /* DSR: begin block 2002.02.21 */
  /* Changed
  ** PyTuple_SetItem( o, 1, PyTuple_New( 0 ) );
  ** to insert None rather than an empty tuple; the new behavior makes more
  ** sense in light of the spec's comment about Cursor.description: "This
  ** attribute will be None for operations that do not return rows...".
   */
  Py_INCREF( Py_None );
  PyTuple_SetItem( o, 1, Py_None );
  /* DSR: end block 2002.02.21 */

  return o;
} /* pyob_execute */

/********************** EXECUTION FUNCTIONS:END ***********************/

/********************** FETCH FUNCTIONS:BEGIN ***********************/

static PyObject* pyob_fetch( PyObject* self, PyObject* args ) {
  CursorObject* cursor;
  int fetch_status;
  PyObject* row;
  int statementType;
#ifdef KIDB_DEBUGGERING
  int numberOfOutputParams;
#endif /* KIDB_DEBUGGERING */
  if ( !PyArg_ParseTuple(args, "O!", &CursorType, &cursor ) ) {
    return NULL;
  }

  CUR_REQUIRE_OPEN(cursor);

  /* DSR:2002.04.27:begin block */
  /* If the result set has been exhausted and the cursor hasn't executed a
  ** fresh statement since, return None rather than raising an error, as
  ** required by the Python DB API.
  */
  if ( cursor->last_fetch_status == 100 ) {
    Py_INCREF( Py_None );
    return Py_None;
  }
  /* DSR:2002.04.27:end block */

  /* DSR: begin block 2002.01.01 */
  /* YYY:DSR:2002.02.17:  The statement type could be cached for better
  ** performance, in a manner similar to that of cursor->lastsql and
  ** cursor->lastsql_len.  determineStatementType calls isc_dsql_sql_info,
  ** which ?may? be expensive.
  ** Because the fetch function is among the most performance-critical in
  ** the module, even a modest optimization such as this could save much
  ** time for large result sets.
  */
  statementType = determineStatementType( &(cursor->stmt_handle ),
                                            cursor->status_vector );

#ifdef KIDB_DEBUGGERING
  numberOfOutputParams = cursor->out_sqlda->sqld;
#endif /* KIDB_DEBUGGERING */
  debug_printf2( "\n[in pyob_fetch] STATEMENT TYPE: %d, "
                 "# of output params: %d\n",
                 statementType,
                 numberOfOutputParams );

  /* If the cursor's statement type is isc_info_sql_stmt_exec_procedure,
  ** then use pseudo-fetch (for reasons explained in the comments in
  ** function pyob_execute regarding the difference between isc_dsql_execute
  ** and isc_dsql_execute2).
  */
  if ( statementType == isc_info_sql_stmt_exec_procedure ) {
    if (cursor->exec_proc_results != NULL) {
      PyObject* cachedRow = cursor->exec_proc_results;
      debug_printf("[in pyob_execute] returning the single cached"
        " EXECUTE PROCEDURE result row\n");
      /* Don't need to change reference count of exec_proc_results because
      ** we're passing reference ownership to the caller of this function.
      */
      cursor->exec_proc_results = NULL;

      return cachedRow;
    } else {
      debug_printf("[in pyob_execute] returning None because cached"
        " EXECUTE PROCEDURE results already used\n");
      Py_INCREF( Py_None );
      return Py_None;
    }
  /* DSR: begin block 2002.02.15 - bugfix #517842 */
  } else if ( !couldStatementTypePossiblyReturnResultSet(statementType) ) {
    /* If the last executed statement couldn't return a result set, the client
    ** programmer should not be asking for one.
    ** It is imperative that this situation be detected before the
    ** isc_dsql_fetch call below.  Otherwise, isc_dsql_fetch will claim to
    ** have succeeded without really having done so, and will leave the
    ** cursor statement object in an invalid state that causes the program
    ** to freeze when isc_dsql_free_statement is called (usually by object
    ** destructors or other cleanup code).
     */

    /* DSR:2002.05.02:begin block */
    /* These 2002.05.02 changes fix two bugs: 551098 and 551207. */
    const char* baseErrMsg = "Attempt to fetch row of results from a statement"
      " that does not produce a result set.  That statement was:  ";
    int baseErrMsg_len = strlen(baseErrMsg);
    int entireErrorMessage_len = baseErrMsg_len + cursor->lastsql_len;
    char* entireErrorMessage = (char*) malloc( sizeof(char) * (entireErrorMessage_len + 1) );
    strncpy(entireErrorMessage, baseErrMsg, baseErrMsg_len);
    strncpy(entireErrorMessage + baseErrMsg_len, cursor->lastsql,
      cursor->lastsql_len);
    entireErrorMessage[entireErrorMessage_len] = '\0'; /* Add NULL terminator. */

    raise_exception( ProgrammingError, entireErrorMessage );
    free(entireErrorMessage);
    /* DSR:2002.05.02:end block */

    close_cursor( cursor );
    return NULL;
  }
  /* DSR: end block 2002.02.15 - bugfix #517842 */

  fetch_status = isc_dsql_fetch( cursor->status_vector,
                               &(cursor->stmt_handle),
                               cursor->connection->dialect,
                               cursor->out_sqlda );
  debug_printf1("[in pyob_fetch] fetch_status is %d\n", fetch_status);

  /* isc_dsql_fetch return value meanings:
  **     0         -> success
  **   100         -> result set exhausted
  ** anything else -> error
  */
  switch ( fetch_status ) {
  case 0:
    row = XSQLDA2Tuple( cursor, cursor->out_sqlda );
    if ( !row )
      free_cursor_cache( cursor );
    /* DSR:2002.04.27:begin block */
    cursor->last_fetch_status = fetch_status;
    /* DSR:2002.04.27:end block */
    return row;

  case 100:
/* XXX:DSR:STATE:  Logically, the cursor should be shut down here.
** However, shutdown_cursor is being called even in cases where it should not
** be at the beginning of pyob_execute, so it will definitely be shut down
** before the next statement is executed, or when the cursor is garbage
** collected.
** Since I began tracking cursor state in order to comply with the DB API
** spec, these gratuitous shutdowns have suddenly come to light.  They should
** be dealt with (and disabled in the meantime) because the program is now
** aware of cursor state and throws an exception when an operation is executed
** on a cursor that is "logically closed but really open".
*/
/*    shutdown_cursor( cursor );*/
    /* DSR:2002.04.27:begin block */
    cursor->last_fetch_status = fetch_status;
    /* DSR:2002.04.27:end block */
    Py_INCREF( Py_None );
    return Py_None;
  }
  /* default: */

  /* DSR:2002.02.21:  The exception raised below was an OperationalError.
  ** Typically, the exception handler below is only reached when the client
  ** programmer causes an EXCEPTION statement to be triggered in a stored
  ** procedure or something along those lines.  The DB API spec plainly
  ** states that a ProgrammingError is the exception to raise when the
  ** programmer caused the error through his own error.
  **
  ** Before the DB API compliance review and the fixes applied in answer to it,
  ** this exception handler was also reached if the client programmer tried to
  ** fetch past the end of the cursor.  Although that behavior is not compliant
  ** with the DB API spec (and has been changed), it was certainly more of a
  ** ProgrammingError than an OperationalError; so are the other errors
  ** encountered here, as far as I know.
  **
  ** YYY:Perhaps specific testing needs to be done in order to differentiate
  ** between ProgrammingError and OperationalError depending on the IB C
  ** API's error code.
  */
  raise_sql_exception( ProgrammingError,
                       "fetch.isc_dsql_fetch: ",
                       cursor->status_vector );
  /* DSR: end block 2002.02.21 */
  close_cursor( cursor );
  /* DSR:2002.04.27:begin block */
  cursor->last_fetch_status = fetch_status;
  /* DSR:2002.04.27:end block */
  return NULL;
} /* pyob_fetch */

/********************** FETCH FUNCTIONS:END ***********************/

/********************** MODULE INFRASTRUCTURE FUNCTIONS:BEGIN ***********************/

static PyMethodDef kinterbasdb_GlobalMethods[] = {
  { "create_database",   pyob_create_database,            METH_VARARGS      },
  { "drop_database",     pyob_drop_database,              METH_VARARGS      },

  /* maz: cast because attach_db has keywords */
  { "attach_db",         (PyCFunction)( pyob_attach_db ), ( METH_VARARGS |
                                                            METH_KEYWORDS ) },
  { "detach_db",         pyob_detach_db,                  METH_VARARGS      },

#ifdef INTERBASE6_OR_LATER
  { "get_dialect",       pyob_get_dialect,                METH_VARARGS      },
  { "set_dialect",       pyob_set_dialect,                METH_VARARGS      },

/* DSR:2002.02.28:bugfix #517093:begin block */
  { "get_precision_mode",pyob_get_precision_mode,         METH_VARARGS      },
  { "set_precision_mode",pyob_set_precision_mode,         METH_VARARGS      },
/* DSR:2002.02.28:bugfix #517093:end block */
#endif /* INTERBASE6_OR_LATER */

  { "begin",             pyob_begin,                      METH_VARARGS      },
  { "execute_immediate_with_connection",
      pyob_execute_immediate_with_connection,             METH_VARARGS      },
  { "commit",            pyob_commit,                     METH_VARARGS      },
  { "rollback",          pyob_rollback,                   METH_VARARGS      },
  { "close_connection",  pyob_close_connection,           METH_VARARGS      },


  { "cursor",            pyob_cursor,                     METH_VARARGS      },
  { "execute",           pyob_execute,                    METH_VARARGS      },
  { "fetch",             pyob_fetch,                      METH_VARARGS      },
  { "close_cursor",      pyob_close_cursor,               METH_VARARGS      },
  { "database_info",     pyob_database_info,              METH_VARARGS      },
  /* DSR:2002.03.15:begin block */
  { "raw_byte_to_int",   pyob_raw_byte_to_int,            METH_VARARGS      },
  /* DSR:2002.03.15:end block */
  { NULL,                NULL                             /* the end */     }
};


static int init_kidb_exceptions( PyObject *d ) {
  Warning =
    PyErr_NewException( "kinterbasdb.Warning", PyExc_StandardError, NULL );

  Error =
    PyErr_NewException( "kinterbasdb.Error", PyExc_StandardError, NULL );

  InterfaceError =
    PyErr_NewException( "kinterbasdb.InterfaceError", Error, NULL );

  DatabaseError =
    PyErr_NewException( "kinterbasdb.DatabaseError", Error, NULL );

  DataError =
    PyErr_NewException( "kinterbasdb.DataError", DatabaseError, NULL );

  OperationalError =
    PyErr_NewException( "kinterbasdb.OperationalError", DatabaseError, NULL );

  IntegrityError =
    PyErr_NewException( "kinterbasdb.IntegrityError", DatabaseError, NULL );

  InternalError =
    PyErr_NewException( "kinterbasdb.InternalError", DatabaseError, NULL );

  /* DSR:2002.02.15:  First param of the PyErr_NewException call below was
  ** erroneously "kinterbasdb.InterfaceError"; I changed it to
  ** "kinterbasdb.ProgrammingError".
   */
  ProgrammingError =
    PyErr_NewException( "kinterbasdb.ProgrammingError", DatabaseError, NULL );

  NotSupportedError =
    PyErr_NewException( "kinterbasdb.NotSupportedError", DatabaseError, NULL );

  if ( !Warning || !Error || !InterfaceError || !DatabaseError ||
       !DataError || !OperationalError || !IntegrityError || !InternalError ||
       !ProgrammingError || !NotSupportedError )
    return -1;

  PyDict_SetItemString(   d,   "Error",               Error               );
  PyDict_SetItemString(   d,   "Warning",             Warning             );
  PyDict_SetItemString(   d,   "InterfaceError",      InterfaceError      );
  PyDict_SetItemString(   d,   "DatabaseError",       DatabaseError       );
  PyDict_SetItemString(   d,   "DataError",           DataError           );
  PyDict_SetItemString(   d,   "OperationalError",    OperationalError    );
  PyDict_SetItemString(   d,   "IntegrityError",      IntegrityError      );
  PyDict_SetItemString(   d,   "InternalError",       InternalError       );
  PyDict_SetItemString(   d,   "ProgrammingError",    ProgrammingError    );
  PyDict_SetItemString(   d,   "NotSupportedError",   NotSupportedError   );

  return 0;
} /* init_kidb_exceptions */


static void pyob_cursor_del( PyObject* cursor ) {
  delete_cursor( (CursorObject*)( cursor ) );
  PyObject_Del( cursor );
} /* pyob_cursor_del */


static void pyob_connection_del( PyObject* connection ) {
  delete_connection( (ConnectionObject*)( connection ) );
  PyObject_Del( connection );
} /* pyob_connection_del */


static PyTypeObject ConnectionType = {
  PyObject_HEAD_INIT( NULL )

  0,
  "kinterbasdb.Connection",
  sizeof( ConnectionObject ),
  0,
  pyob_connection_del,
  0,
  0,
  0,
  0,
  0,
  0,
  0,
  0,
  0
};


static PyTypeObject CursorType = {
  PyObject_HEAD_INIT( NULL )

  0,
  "kinterbasdb.Cursor",
  sizeof( CursorObject ),
  0,
  pyob_cursor_del,
  0,
  0,
  0,
  0,
  0,
  0,
  0,
  0,
  0
};


/* DSR:2002.07.18:begin block */
/* A more standard approach to some infrastructural boilerplate: */
/* OLD:
#ifdef MS_WIN32
__declspec( dllexport )
#endif
*/
DL_EXPORT(void)
init_kinterbasdb(void) {
/* DSR:2002.07.18:end block */
  PyObject *m, *d;
  ConnectionObject *null_connection;
  CursorObject *null_cursor;

  m = Py_InitModule( "_kinterbasdb", kinterbasdb_GlobalMethods );
  d = PyModule_GetDict( m );

  ConnectionType.ob_type = &PyType_Type;
  CursorType.ob_type = &PyType_Type;

  /* DSR:2002.04.13:begin block */
  SHRT_MIN_As_PyObject = PyInt_FromLong(SHRT_MIN);
  SHRT_MAX_As_PyObject = PyInt_FromLong(SHRT_MAX);

  INT_MIN_As_PyObject = PyInt_FromLong(INT_MIN);
  INT_MAX_As_PyObject = PyInt_FromLong(INT_MAX);

  LONG_MIN_As_PyObject = PyInt_FromLong(LONG_MIN);
  LONG_MAX_As_PyObject = PyInt_FromLong(LONG_MAX);

  LONG_LONG_MIN_As_PyObject = PyLong_FromLongLong(LONG_LONG_MIN);
  LONG_LONG_MAX_As_PyObject = PyLong_FromLongLong(LONG_LONG_MAX);
  /* DSR:2002.04.13:end block */

  /* DSR: begin block 2002.02.24 */
  /* DSR added null_connection when moving connection state detection from the
  ** Python level to the C level.  From the perspective of kinterbasdb.py,
  ** _kinterbasdb.null_connection is essentially None, except that it is of
  ** type ConnectionType instead of NoneType.
  ** The kinterbasdb.py Connection class can set its reference to its
  ** equivalent C type (self._C_conn) to _kinterbasdb.null_connection to
  ** indicate that the connection is no longer valid.  Then the pyob_*
  ** functions in this C code that demand an argument of ConnectionType are
  ** satisfied by null_connection long enough to detect that it is not open
  ** (and that therefore the requested operation is not allowed).
  */
  null_connection = PyObject_New( ConnectionObject, &ConnectionType );
  null_connection->dialect = -1;
  null_connection->db_handle = NULL;
  null_connection->trans_handle = NULL;
  null_connection->_state = CONNECTION_STATE_CLOSED;
  /* DSR:2002.02.28:bugfix #517093:begin block */
  null_connection->precision_mode = DEFAULT_PRECISION_MODE;
  /* DSR:2002.02.28:bugfix #517093:end block */
  /* DSR:2002.06.09:begin block */
  /* DSR:2002.07.10:begin block */
  #ifdef DETERMINE_FIELD_PRECISION
  null_connection->desc_cache = NULL;
  #endif
  /* DSR:2002.07.10:end block */
  /* DSR:2002.06.09:end block */

  PyDict_SetItemString(d, "null_connection", (PyObject *) null_connection);

  /* null_cursor is the CursorType equivalent of null_connection. */
  null_cursor = PyObject_New( CursorObject, &CursorType );

  null_cursor->connection = NULL;
  null_cursor->stmt_handle = NULL;
  null_cursor->in_sqlda = NULL;
  null_cursor->out_sqlda = NULL;
  null_cursor->in_buffer = NULL;
  null_cursor->out_buffer = NULL;
  null_cursor->lastsql = NULL;
  null_cursor->lastsql_len = -1;
  null_cursor->exec_proc_results = NULL;
  /* DSR:2002.04.27:begin block */
  null_cursor->last_fetch_status = 0;
  /* DSR:2002.04.27:end block */
  null_cursor->_state = CURSOR_STATE_CLOSED;

  PyDict_SetItemString(d, "null_cursor", (PyObject *) null_cursor);

  /* DSR: end block 2002.02.24 */

/* DSR: begin block 2002.01.20 */
  if ( init_kidb_ibase_header_constants( d ) )
    fprintf( stderr, "Header constants initialization failed!\n" );
/* DSR: end block 2002.01.20 */

  if ( init_kidb_exceptions( d ) )
    fprintf( stderr, "Exceptions initialization failed!\n" );

  if ( mxDateTime_ImportModuleAndAPI() )
    fprintf( stderr, "Unable to initialize mxDateTime\n" );

} /* init_kinterbasdb */


/********************** MODULE INFRASTRUCTURE FUNCTIONS:END ***********************/

/********************** NEW DSR-DEFINED FUNCTIONS:BEGIN ***********************/
/* YYY:DSR:  I will eventually integrate these functions into their respective
** categories.
*/

/* DSR: begin block 2002.01.20 */
int init_kidb_ibase_header_constants(PyObject* d) {
  /* Define transaction parameter buffer constants. */
  _init_kidb_ibase_header_constants_transaction_parameters(d);

  /* Define database information constants. */
  _init_kidb_ibase_header_constants_database_info(d);

  return 0;
}

void _init_kidb_ibase_header_constants_transaction_parameters(PyObject* d) {
  char convArray[1];

/*
** SET_TPB_CONST is just a shortcut for defining transaction parameter buffer
** constants, which were previously defined in kinterbasdb.py as strings
** containing octal escape codes.
**
** For example, if the definition of isc_some_dumb_const is:
** #define isc_some_dumb_const 16
** then the brittle version of kinterbasdb.py would feature this line:
** isc_some_dumb_const = '\020'
**
** The point of SET_TPB_CONST is to enter into dict d the equivalent of the
** Python string '\020', when passed the name and value of isc_some_dumb_const.
 */
#define SET_TPB_CONST(name, value) convArray[0] = (char)value; PyDict_SetItemString(d, name, PyString_FromStringAndSize(convArray, 1));

  SET_TPB_CONST("isc_tpb_consistency",        isc_tpb_consistency);
  SET_TPB_CONST("isc_tpb_concurrency",        isc_tpb_concurrency);
  SET_TPB_CONST("isc_tpb_shared",             isc_tpb_shared);
  SET_TPB_CONST("isc_tpb_protected",          isc_tpb_protected);
  SET_TPB_CONST("isc_tpb_exclusive",          isc_tpb_exclusive);
  SET_TPB_CONST("isc_tpb_wait",               isc_tpb_wait);
  SET_TPB_CONST("isc_tpb_nowait",             isc_tpb_nowait);
  SET_TPB_CONST("isc_tpb_read",               isc_tpb_read);
  SET_TPB_CONST("isc_tpb_write",              isc_tpb_write);
  SET_TPB_CONST("isc_tpb_lock_read",          isc_tpb_lock_read);
  SET_TPB_CONST("isc_tpb_lock_write",         isc_tpb_lock_write);
  SET_TPB_CONST("isc_tpb_verb_time",          isc_tpb_verb_time);
  SET_TPB_CONST("isc_tpb_commit_time",        isc_tpb_commit_time);
  SET_TPB_CONST("isc_tpb_ignore_limbo",       isc_tpb_ignore_limbo);
  SET_TPB_CONST("isc_tpb_read_committed",     isc_tpb_read_committed);
  SET_TPB_CONST("isc_tpb_autocommit",         isc_tpb_autocommit);
  SET_TPB_CONST("isc_tpb_rec_version",        isc_tpb_rec_version);
  SET_TPB_CONST("isc_tpb_no_rec_version",     isc_tpb_no_rec_version);
  SET_TPB_CONST("isc_tpb_restart_requests",   isc_tpb_restart_requests);
  SET_TPB_CONST("isc_tpb_no_auto_undo",       isc_tpb_no_auto_undo);
}

void _init_kidb_ibase_header_constants_database_info(PyObject* d) {
/*
** SET_INT_CONST is just a shortcut for entering database info constants into
** dict d.
 */
#define SET_INT_CONST(name, value) PyDict_SetItemString(d, name, PyInt_FromLong(value));

  SET_INT_CONST("isc_info_db_id",                   isc_info_db_id);
  SET_INT_CONST("isc_info_reads",                   isc_info_reads);
  SET_INT_CONST("isc_info_writes",                  isc_info_writes);
  SET_INT_CONST("isc_info_fetches",                 isc_info_fetches);
  SET_INT_CONST("isc_info_marks",                   isc_info_marks);
  SET_INT_CONST("isc_info_implementation",          isc_info_implementation);
  SET_INT_CONST("isc_info_version",                 isc_info_version);
  SET_INT_CONST("isc_info_base_level",              isc_info_base_level);
  SET_INT_CONST("isc_info_page_size",               isc_info_page_size);
  SET_INT_CONST("isc_info_num_buffers",             isc_info_num_buffers);
  SET_INT_CONST("isc_info_limbo",                   isc_info_limbo);
  SET_INT_CONST("isc_info_current_memory",          isc_info_current_memory);
  SET_INT_CONST("isc_info_max_memory",              isc_info_max_memory);
  SET_INT_CONST("isc_info_window_turns",            isc_info_window_turns);
  SET_INT_CONST("isc_info_license",                 isc_info_license);
  SET_INT_CONST("isc_info_allocation",              isc_info_allocation);
  SET_INT_CONST("isc_info_attachment_id",           isc_info_attachment_id);
  SET_INT_CONST("isc_info_read_seq_count",          isc_info_read_seq_count);
  SET_INT_CONST("isc_info_read_idx_count",          isc_info_read_idx_count);
  SET_INT_CONST("isc_info_insert_count",            isc_info_insert_count);
  SET_INT_CONST("isc_info_update_count",            isc_info_update_count);
  SET_INT_CONST("isc_info_delete_count",            isc_info_delete_count);
  SET_INT_CONST("isc_info_backout_count",           isc_info_backout_count);
  SET_INT_CONST("isc_info_purge_count",             isc_info_purge_count);
  SET_INT_CONST("isc_info_expunge_count",           isc_info_expunge_count);
  SET_INT_CONST("isc_info_sweep_interval",          isc_info_sweep_interval);
  SET_INT_CONST("isc_info_ods_version",             isc_info_ods_version);
  SET_INT_CONST("isc_info_ods_minor_version",       isc_info_ods_minor_version);
  SET_INT_CONST("isc_info_no_reserve",              isc_info_no_reserve);
  SET_INT_CONST("isc_info_logfile",                 isc_info_logfile);
  SET_INT_CONST("isc_info_cur_logfile_name",        isc_info_cur_logfile_name);
  SET_INT_CONST("isc_info_cur_log_part_offset",     isc_info_cur_log_part_offset);
  SET_INT_CONST("isc_info_num_wal_buffers",         isc_info_num_wal_buffers);
  SET_INT_CONST("isc_info_wal_buffer_size",         isc_info_wal_buffer_size);
  SET_INT_CONST("isc_info_wal_ckpt_length",         isc_info_wal_ckpt_length);
  SET_INT_CONST("isc_info_wal_cur_ckpt_interval",   isc_info_wal_cur_ckpt_interval);
  SET_INT_CONST("isc_info_wal_prv_ckpt_fname",      isc_info_wal_prv_ckpt_fname);
  SET_INT_CONST("isc_info_wal_prv_ckpt_poffset",    isc_info_wal_prv_ckpt_poffset);
  SET_INT_CONST("isc_info_wal_recv_ckpt_fname",     isc_info_wal_recv_ckpt_fname);
  SET_INT_CONST("isc_info_wal_recv_ckpt_poffset",   isc_info_wal_recv_ckpt_poffset);
  SET_INT_CONST("isc_info_wal_grpc_wait_usecs",     isc_info_wal_grpc_wait_usecs);
  SET_INT_CONST("isc_info_wal_num_io",              isc_info_wal_num_io);
  SET_INT_CONST("isc_info_wal_avg_io_size",         isc_info_wal_avg_io_size);
  SET_INT_CONST("isc_info_wal_num_commits",         isc_info_wal_num_commits);
  SET_INT_CONST("isc_info_wal_avg_grpc_size",       isc_info_wal_avg_grpc_size);
  SET_INT_CONST("isc_info_forced_writes",           isc_info_forced_writes);
  SET_INT_CONST("isc_info_user_names",              isc_info_user_names);
  SET_INT_CONST("isc_info_page_errors",             isc_info_page_errors);
  SET_INT_CONST("isc_info_record_errors",           isc_info_record_errors);
  SET_INT_CONST("isc_info_bpage_errors",            isc_info_bpage_errors);
  SET_INT_CONST("isc_info_dpage_errors",            isc_info_dpage_errors);
  SET_INT_CONST("isc_info_ipage_errors",            isc_info_ipage_errors);
  SET_INT_CONST("isc_info_ppage_errors",            isc_info_ppage_errors);
  SET_INT_CONST("isc_info_tpage_errors",            isc_info_tpage_errors);
  SET_INT_CONST("isc_info_set_page_buffers",        isc_info_set_page_buffers);
#ifdef INTERBASE6_OR_LATER
  SET_INT_CONST("isc_info_db_SQL_dialect",          isc_info_db_SQL_dialect);
  SET_INT_CONST("isc_info_db_read_only",            isc_info_db_read_only);
  SET_INT_CONST("isc_info_db_size_in_pages",        isc_info_db_size_in_pages);
#endif /* INTERBASE6_OR_LATER */
}
/* DSR: end block 2002.01.20 */

int determineStatementType( isc_stmt_handle* statementHandle,
                            ISC_STATUS statusVector[] ) {
  /*
  ** Dynamically determine the statement type.  Follows p. 343 of IB Beta 6
  ** API Guide and IB example apifull.c.
  */
  int statementType;
  int statementTypeLength;
  static char sqlInfoStatementTypeRequest[] = { isc_info_sql_stmt_type };
  char sqlInfoResultBuffer[ ISC_INFO_BUFFER_SIZE ];

  isc_dsql_sql_info( statusVector,
                     statementHandle,

                     sizeof( sqlInfoStatementTypeRequest ),
                     sqlInfoStatementTypeRequest,

                     sizeof( sqlInfoResultBuffer ),
                     sqlInfoResultBuffer );

  /*
  ** Next two lines follow the Interbase example apifull.c rather than the
  ** API Guide.
  */
  statementTypeLength = (short)( isc_vax_integer( (char ISC_FAR*)
                                                  sqlInfoResultBuffer + 1,
                                                  2 ) );
  statementType = (int)isc_vax_integer( (char ISC_FAR*)sqlInfoResultBuffer + 3,
                                   (short)statementTypeLength );

  return statementType;
} /* determineStatementType */

/* DSR: begin block 2002.02.15 */
short couldStatementTypePossiblyReturnResultSet(int statementType) {
  /* Returns a boolean. */

  /* It would be more efficient to do a *positive* check for the statement
  ** types that *can* return a result set, but then the code would exclude
  ** any result set-returning statement types added in the future.
   */
  switch (statementType) {
  case isc_info_sql_stmt_exec_procedure:
  case isc_info_sql_stmt_insert:
  case isc_info_sql_stmt_update:
  case isc_info_sql_stmt_delete:
  case isc_info_sql_stmt_ddl:
/*? case isc_info_sql_stmt_get_segment: */
/*? case isc_info_sql_stmt_put_segment: */
  case isc_info_sql_stmt_start_trans:
  case isc_info_sql_stmt_commit:
  case isc_info_sql_stmt_rollback:
/*? case isc_info_sql_stmt_select_for_upd: */
  case isc_info_sql_stmt_set_generator:
    return 0;

  default:
    return 1;
  }
} /* couldStatementTypePossiblyReturnResultSet */
/* DSR: end block 2002.02.15 */

/* DSR:2002.02.23:
** _blob_info_total_size_and_max_segment_size inserts into its
** arguments -total_size and -max_segment_size the total size and maximum
** segment size (respectively) of the specified blob.
** Returns 0 if successful, otherwise -1.
**
** See IB6 API Guide chapter entitled "Working with Blob Data".
*/
static int _blob_info_total_size_and_max_segment_size(
  ISC_STATUS* status_vector,
  isc_blob_handle* blob_handle,

  ISC_LONG* total_size,
  unsigned short* max_segment_size
) {
  char blob_info_items[] = {
    isc_info_blob_total_length,
    isc_info_blob_max_segment
  };

  char result_buffer[ISC_INFO_BUFFER_SIZE];

  short length;
  char* ptr;
  char item;

  isc_blob_info(
    status_vector,
    (isc_blob_handle*) &blob_handle,
    sizeof(blob_info_items),
    blob_info_items,
    sizeof(result_buffer),
    result_buffer
  );

  if ( DB_API_ERROR(status_vector) ) {
    raise_sql_exception(InternalError,
        "_blob_info_total_size_and_max_segment_size.isc_blob_info: ",
        status_vector
      );
    return -1;
  };

  /* Extract the values returned in the result buffer. */
  for (ptr = result_buffer; *ptr != isc_info_end ;) {
    item = *ptr++;
    length = (short)isc_vax_integer(ptr, 2);
    ptr += 2;

    switch (item)
    {
    case isc_info_blob_total_length:
      *total_size = isc_vax_integer(ptr, length);
      break;
    case isc_info_blob_max_segment:
      *max_segment_size = (unsigned short)isc_vax_integer(ptr, length);
      break;
    case isc_info_truncated:
      raise_sql_exception(InternalError,
          "_blob_info_total_size_and_max_segment_size.isc_vax_integer: ",
          status_vector
        );
      return -1;
    }
    ptr += length;
  }

  return 0;
}

#ifdef KIDB_DEBUGGERING
void dumpXSQLDA( XSQLDA* sqlda ) {
  unsigned int i;

  debug_printf( "SQLDA DUMP\n" );
  debug_printf1( " version: %d\n", sqlda->version );
  /*
  ** Next two are documented for "future use" but not actually present yet.
  **
  ** debug_printf1( " sqlaid: %s\n", sqlda->sqlaid );
  ** debug_printf1( " sqlabc: %d\n", sqlda->sqlabc );
  */
  debug_printf1( " sqln: %d\n", sqlda->sqln );
  debug_printf1( " sqld: %d\n", sqlda->sqld );
  debug_printf( "END SQLDA DUMP\n" );

  debug_printf( "SQLDA->sqlvar DUMP\n" );
  debug_printf1( " sqlvar->sqltype: %d\n", sqlda->sqlvar->sqltype );
  debug_printf1( " sqlvar->sqlscale: %d\n", sqlda->sqlvar->sqlscale );
  debug_printf1( " sqlvar->sqlsubtype: %d\n", sqlda->sqlvar->sqlsubtype );
  debug_printf1( " sqlvar->sqllen %d\n", sqlda->sqlvar->sqllen );

  debug_printf1( "strlen(sqlda->sqlvar->sqldata): %d\n",
                 strlen( sqlda->sqlvar->sqldata ) );
  for ( i = 0; i < strlen( sqlda->sqlvar->sqldata ); i++ ) {
    debug_printf1( "%c", sqlda->sqlvar->sqldata[ i ] );
  }

  /* touchy! */
  debug_printf1( " sqlvar->sqldata == NULL: %d\n",
                 sqlda->sqlvar->sqldata == NULL );

  debug_printf1( " sqlvar->sqlname_length: %d\n",
                 sqlda->sqlvar->sqlname_length );
  debug_printf1( " sqlvar->sqlname %s\n",
                 sqlda->sqlvar->sqlname );

  debug_printf1( " sqlvar->relname_length: %d\n",
                 sqlda->sqlvar->relname_length );
  debug_printf1( " sqlvar->relname: %s\n",
                 sqlda->sqlvar->relname );
  debug_printf1( " sqlvar->ownname_length: %d\n",
                 sqlda->sqlvar->ownname_length );
  debug_printf1( " sqlvar->ownname: %s\n",
                 sqlda->sqlvar->ownname );

  debug_printf1( " sqlvar->aliasname_length: %d\n",
                 sqlda->sqlvar->aliasname_length );
  debug_printf1( " sqlvar->aliasname: %s\n",
                 sqlda->sqlvar->aliasname );
  debug_printf( "END SQLDA->sqlvar DUMP\n" );
} /* dumpXSQLDA */

void dumpStatusVector( ISC_STATUS* sv ) {
  int i;

  debug_printf("[dumping status vector:]\n");

  for (i = 0; i < STATUS_VECTOR_SIZE; i++) {
    debug_printf2("status_vector[%d] = %d\n", i, sv[i]);
  }
} /* dumpStatusVector */

void dumpXSQLVAR( XSQLVAR* sqlvar ) {
  /* This debugging code originally from janez's first patch to maz
  ** (added on 19.jan/y2k2 by maz).
  */
  debug_printf( "[dumping XSQLVAR:]\n" );
  debug_printf1( "  sqlvar->sqltype: %d\n",
                 sqlvar->sqltype );
  debug_printf1( "  sqlvar->sqlsubtype: %d\n",
                 sqlvar->sqlsubtype );
  debug_printf1( "  sqlvar->sqllen: %d\n",
                 sqlvar->sqllen );
  debug_printf1( "  sqlvar->sqlname_length: %d\n",
                 sqlvar->sqlname_length );
  debug_printf1( "  sqlvar->relname_length: %d\n",
                 sqlvar->relname_length );
  debug_printf1( "  sqlvar->ownname_length: %d\n",
                 sqlvar->ownname_length );
  debug_printf1( "  sqlvar->aliasname_length: %d\n",
                 sqlvar->aliasname_length );
}

#endif /* KIDB_DEBUGGERING */

/********************** NEW DSR-DEFINED FUNCTIONS:END ***********************/

/* There, now that wasn't so bad, was it, reader? */
