/*
 * Copyright (c) 2006 Bea Lam. All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

//
//  BBOBEXClientSession.m
//  BTUtil
//
//  Much thanks to the authors of Apple's OBEXSample example code (in
//  /Developer/Examples/Bluetooth) which shows how to use OBEXSession.
//
//  Like in BBOBEXServerSession, I really should fix all this to separate each
//  operation into a separate class. It's really messy and disjointed like 
//  this.
//
//  To-do: it would be good if the ABORT can time out and return 
//  kOBEXTimeoutError like OBEXFileTransferServices. 
//

#import <IOBluetooth/IOBluetoothUtilities.h>
#import <IOBluetooth/OBEXBluetooth.h>
#import <IOBluetooth/objc/IOBluetoothOBEXSession.h>
#import <IOBluetooth/objc/IOBluetoothDevice.h>

#import "BBOBEXUtil.h"
#import "BBOBEXClientSession.h"


#define DEBUG_NAME @"[BBOBEXClientSession] "


static BOOL debug;
static SEL callbackServerResponded;	
static SEL callbackConnectComplete, callbackDisconnectComplete, 
	callbackPutProgress, callbackPutComplete, 
	callbackGetProgress, callbackGetComplete, 
	callbackAbortComplete, callbackSetPathComplete;


@implementation BBOBEXClientSession

+ (void)initialize
{	
	debug = NO;
	
	callbackConnectComplete = @selector(obexClientSessionConnectComplete:error:fromWho:connectionID:);
	callbackDisconnectComplete = @selector(obexClientSessionDisconnectComplete:error:);
	callbackPutProgress = @selector(obexClientSessionPutProgress:byAmount:);	
	callbackPutComplete = @selector(obexClientSessionPutComplete:error:);
	callbackGetProgress = @selector(obexClientSessionGetProgress:byAmount:);
	callbackGetComplete = @selector(obexClientSessionGetComplete:error:);
	callbackAbortComplete = @selector(obexClientSessionAbortComplete:error:);
	callbackSetPathComplete = @selector(obexClientSessionSetPathComplete:error:);	
	
	callbackServerResponded = @selector(handleSessionEvent:);		
}


- (id)initWithDelegate:(id)delegate
{
	self = [super init];
	
	mDelegate = delegate;
	
	// essential states
	mOBEXSession = nil;
	mClosed = NO;
	mCurrentOp = 0;
	
	// other states
	mMaxPacketLength = 0x2000;		// set some starting value
	mLastErrorDesc = @"";
	mUserWantsAbort = NO;
	mClosePutFileWhenDone = NO;
	
	// header values
	mConnectionID = -1;
	mConnectionIDDataRef = NULL;
	mWhoHeaderValue = nil;
	
	return self;	
}

- (id)init
{
	return [self initWithDelegate:nil];
}

- (void)cleanUp
{
	[mOBEXSession release];
	[mLastErrorDesc release];
	
	if (mConnectionIDDataRef) CFRelease(mConnectionIDDataRef);
	
	//[mCurrentPutFile release];
	[mCurrentPutBodyHandler release];
	
	[mCurrentGetFile release];
	
	if (mConnectHeadersDataRef) CFRelease(mConnectHeadersDataRef);
	if (mPutHeadersDataRef) CFRelease(mPutHeadersDataRef);
	if (mGetHeadersDataRef) CFRelease(mGetHeadersDataRef);
	if (mSetPathHeadersDataRef) CFRelease(mSetPathHeadersDataRef);		
}


#pragma mark -
#pragma mark Error handling


- (NSString *)lastErrorDescription
{
	return mLastErrorDesc;
}

// Notes an error.
// For convenience returns the given error.
- (OBEXError)reportError:(OBEXError)error summary:(NSString *)summary
{
	NSString *errDesc = [DEBUG_NAME stringByAppendingString:summary];
	
	[errDesc retain];
	[mLastErrorDesc release];
	mLastErrorDesc = errDesc;
	
	if (debug) NSLog(errDesc);
	
	return error;
}

- (OBEXError)reportBusyError
{
	return [self reportError:kOBEXSessionBusyError 
					 summary:@"Already busy with another operation"];
}

- (OBEXError)reportCreateHeadersError
{
	return [self reportError:kOBEXInternalError 
					 summary:@"Error building OBEX headers for request"];
}

- (OBEXError)reportAddHeaderError:(NSString *)headerDesc error:(OBEXError)err
{
	NSString *desc = [NSString stringWithFormat:@"Error adding %@ header", headerDesc];
	return [self reportError:err summary:desc];
}

- (OBEXError)reportNotConnectedError
{
	return [self reportError:kOBEXSessionNotConnectedError 
					 summary:@"Not connected, cannot perform requested operation"];
}


#pragma mark -
#pragma mark Utility methods


// Checks the availability of the local bluetooth device.
- (BOOL)bluetoothDeviceAvailable
{
	// is there a bluetooth device?
	if (!IOBluetoothLocalDeviceAvailable()) 
		return NO;
	
	// is the bluetooth device switched on?
	BluetoothHCIPowerState powerState;
	OBEXError status = IOBluetoothLocalDeviceGetPowerState(&powerState);
	if (status != kIOReturnSuccess) 
		return NO;
	if (powerState != kBluetoothHCIPowerStateON) 
		return NO;
	
	return YES;
}

- (BOOL)isConnected
{
	return (!mClosed && [mOBEXSession hasOpenOBEXConnection]);
}

- (long)connectionIDDataRefToLong:(CFDataRef)dataRef
{
	long connID = -1;
	CFDataGetBytes(dataRef, CFRangeMake(0, 4), (uint8_t*)&connID);
	return connID;	
}

// returned data ref must be released
- (CFDataRef)connectionIDLongToDataRef:(long)connectionID
{
	CFDataRef dataRef = CFDataCreate(NULL, (uint8_t*)&connectionID, 4);
	return dataRef;
}

// If there's no Connection ID, returns kOBEXSuccess anyway, so you don't need
// to check if mConnectionID is valid before calling this.
- (OBEXError)addConnectionIDToHeaders:(CFMutableDictionaryRef)headersDictRef
{
	OBEXError status = kOBEXSuccess;
	if (mConnectionIDDataRef) {
		status = OBEXAddConnectionIDHeader((void *)CFDataGetBytePtr(mConnectionIDDataRef),
			CFDataGetLength(mConnectionIDDataRef), headersDictRef);			
	}
	return status;
}


#pragma mark -
#pragma mark marking operations

- (void)startedOp:(OBEXOpCode)opType
{
	mCurrentOp = opType;
}

- (void)completedCurrentOp
{
	mCurrentOp = 0;
}


#pragma mark -
#pragma mark Abort

/*
 //
 // Called when the timeout passes for waiting for the chance to send an ABORT command
 //
 - (void)awaitAbortChanceTimeout:(NSTimer *)timer
 {
	 mUserWantsAbort = NO;
	 [self completedCurrentOp:kOBEXOpCodeAbort error:kOBEXSessionTimeoutError];
 }
 */

- (void)completedAbort:(OBEXError)error
{
	[self completedCurrentOp];
	if ([mDelegate respondsToSelector:callbackAbortComplete]) {
		[mDelegate obexClientSessionAbortComplete:self 
											error:error];
	}
}


- (OBEXError)abort
{
	if (debug) NSLog(DEBUG_NAME @"[abort] entry.");		
	
	// check is connected
	if (![self isConnected]) 
		return [self reportNotConnectedError];

	// if not in the middle of a PUT or GET, can't abort
	if (mCurrentOp != kOBEXOpCodePut && mCurrentOp != kOBEXOpCodeGet) {
		return [self reportError:kOBEXSessionBadRequestError summary:@"Cannot abort, no PUT or GET in progress."];
	}
	
	// Just set an abort flag -- can't send OBEXAbort right away because we 
	// might be in the middle of a transaction, and we must wait our turn to 
	// send the abort (i.e. wait until after we've received a server response)
	mUserWantsAbort = YES;
	
	// commented out this timeout cos I haven't been able to get it to work -- need
	// to notify delegate of timeout from main thread, and not from a timer 
	// thread.
	/*
	 // if the server never responds to the current command in progress (e.g. PUT
	 // or whatever), then we'll never get a turn to send an ABORT, so start a 
	 // timer so that delegate is informed of the timeout if the abort is never 
	 // sent.
	 mAwaitAbortChanceTimer = [NSTimer scheduledTimerWithTimeInterval:mResponseTimeout
															   target:self
															 selector:@selector(awaitAbortChanceTimeout:)
															 userInfo:nil
															  repeats:NO];
	 */
	
	return kOBEXSuccess;
}

- (void)performAbort
{
	if (debug) NSLog(DEBUG_NAME @"[performAbort] entry.");	
	OBEXError status;
	
	// here's an opportunity to abort
	//[mAwaitAbortChanceTimer invalidate];
	//mAwaitAbortChanceTimer = nil;
	
	// set flag
	mUserWantsAbort = NO;
	
	if (![self isBusy]) {
		[self completedAbort:[self reportError:kOBEXGeneralError 
									   summary:@"Cannot abort, no operations in progress."]];
		return;
	}
	
	// create request headers (only need Connection ID in these headers)
	CFMutableDataRef headersDataRef = NULL;
	if (mConnectionIDDataRef) {		// don't make headers if no Connection ID
		CFMutableDictionaryRef headersDictRef = CFDictionaryCreateMutable(NULL,
			1, &kCFTypeDictionaryKeyCallBacks, &kCFTypeDictionaryValueCallBacks);
		if (!headersDictRef) {
			[self reportCreateHeadersError];
			return;
		}
		
		// add connection ID header
		status = [self addConnectionIDToHeaders:headersDictRef];
		if (status != kOBEXSuccess) {
			[self reportAddHeaderError:@"Connection-ID" error:status];
			CFRelease(headersDictRef);
			return;
		}
		
		headersDataRef = OBEXHeadersToBytes(headersDictRef);
		CFRelease(headersDictRef);
	}
	void *headers = (headersDataRef)? (void*)CFDataGetBytePtr(headersDataRef) : NULL;
	size_t headersLength = (headersDataRef)? CFDataGetLength(headersDataRef): 0;	
	
	// send abort command
	status = [mOBEXSession OBEXAbort:headers
			   optionalHeadersLength:headersLength
					   eventSelector:callbackServerResponded
					  selectorTarget:(void *)self
							  refCon:NULL];
	
	if (headersDataRef) 
		CFRelease(headersDataRef);	
	
	if (status == kOBEXSuccess) {
		// abort command sent successfully
		[self startedOp:kOBEXOpCodeAbort];
		
	} else {
		// abort failed
		[self reportError:status summary:@"OBEXAbort failed"];
		[self completedAbort:status];
	}
}


- (void)handleAbortResponse:(const OBEXAbortCommandResponseData*)eventData
{
	if (debug) NSLog(DEBUG_NAME @"[handleAbortResponse] Entry. responsecode = 0x%x\n", eventData->serverResponseOpCode);
	
	if (eventData->serverResponseOpCode != kOBEXResponseCodeSuccessWithFinalBit) {
		// abort failure
		[self completedAbort:[self reportError:kOBEXGeneralError 
									   summary:[NSString stringWithFormat:@"ABORT request failed. Server response code: 0x%x", eventData->serverResponseOpCode]]];		
	} else {
		// connection success
		if (debug) NSLog(DEBUG_NAME @"Abort successful.");
		[self completedAbort:kOBEXSuccess];
	}
}

//
// If user wants to abort, and the server response indicates we're not yet at the
// end of an operation (i.e there's an operation to abort), this sends an ABORT 
// and returns YES.
// Else returns NO.
//
- (BOOL)abortOnCurrentServerResponseCode:(OBEXOpCode)serverResponse
{
	// only allow abort if flag is set, and the operation is not complete
	// (as indicated by server response code)
	if (mUserWantsAbort && serverResponse != kOBEXResponseCodeSuccessWithFinalBit) {	
		[self performAbort];
		
		return YES;	
	}
	
	return NO;
}



#pragma mark -
#pragma mark Connect

- (void)completedConnect:(OBEXError)error
				 fromWho:(NSData *)who 
			connectionID:(long)connectionID
{
	[self completedCurrentOp];
	if ([mDelegate respondsToSelector:callbackConnectComplete]) {
		[mDelegate obexClientSessionConnectComplete:self
											  error:error 
											fromWho:who
									   connectionID:connectionID];
	}
}

- (void)completedConnect:(OBEXError)error
{
	[self completedConnect:error
				   fromWho:nil
			  connectionID:-1];
}

- (OBEXError)connectToDevice:(IOBluetoothDevice *)device
			   withChannelID:(BluetoothRFCOMMChannelID)channelID
					toTarget:(NSData *)target
{	
	if (debug) NSLog(DEBUG_NAME @"[connectToDevice] entry. device:%@ channel:%d", [device getAddressString], channelID);
	OBEXError status;
	
	if (mClosed) 
		return [self reportError:kOBEXInternalError summary:@"Session has been closed"];
	
	// check arguments
	if (device == nil) 
		return [self reportError:kOBEXBadArgumentError summary:@"Given device was nil"];
	
	// check not busy with other operation
	if ([self isBusy]) 
		return [self reportBusyError];
		
	// check local bluetooth device 
	if (![self bluetoothDeviceAvailable])
		return [self reportError:kOBEXGeneralError summary:@"No bluetooth device available"];
	
	/*
	// ensure a minimum packet length
	if (mMaxPacketLength > 0x2000) 
		mMaxPacketLength = 0x2000;
	 */	
	
	// Create OBEX session
 	if (!mOBEXSession) {	
		mOBEXSession = [[IOBluetoothOBEXSession alloc] initWithDevice:device channelID:channelID];
		if (!mOBEXSession) {
			return [self reportError:kOBEXNoResourcesError summary:@"Error creating IOBluetoothOBEXSession"];
		}
	}
	
	// already connected?
	if ([mOBEXSession hasOpenOBEXConnection]) 	
		return [self reportError:kOBEXSessionAlreadyConnectedError summary:@"OBEXSession already connected"];

	// release old headers if necessary
	if (mConnectHeadersDataRef) {
		CFRelease(mConnectHeadersDataRef);
		mConnectHeadersDataRef = NULL;
	}	

	// pack the request headers
	CFMutableDictionaryRef headersDictRef = CFDictionaryCreateMutable(NULL, 2, &kCFTypeDictionaryKeyCallBacks, &kCFTypeDictionaryValueCallBacks);
	if (!headersDictRef) return [self reportCreateHeadersError];

	// add target header
	if (target != nil) {
		status = OBEXAddTargetHeader([target bytes], [target length], headersDictRef);
		if (status != kOBEXSuccess) {
			CFRelease(headersDictRef);
			return [self reportAddHeaderError:@"Target" error:status];
		}
	}
	
	// add who header
	if (mWhoHeaderValue != nil) {
		status = OBEXAddWhoHeader([mWhoHeaderValue bytes], [mWhoHeaderValue length], headersDictRef);
		if (status != kOBEXSuccess) {
			CFRelease(headersDictRef);
			return [self reportAddHeaderError:@"Who" error:status];
		}		
	}

	/*
	if (debug) {
		NSLog(@"CONNECT request: >>>>>>");
		CFShow(headersDictRef);	
	}
	 */
	
	void *headersData = NULL;
	size_t headersLength = 0;	
	
	if (CFDictionaryGetCount(headersDictRef) > 0) {	
		// get headers as bytes
		mConnectHeadersDataRef = OBEXHeadersToBytes(headersDictRef);
		CFRelease(headersDictRef);

		headersData = (void*)CFDataGetBytePtr(mConnectHeadersDataRef);
		headersLength = CFDataGetLength(mConnectHeadersDataRef);
	}
	
	// Make the OBEX connection.
	status = [mOBEXSession OBEXConnect:(OBEXFlags)kOBEXConnectFlagNone
					   maxPacketLength:(OBEXMaxPacketLength)mMaxPacketLength
					   optionalHeaders:headersData
				 optionalHeadersLength:headersLength
						 eventSelector:callbackServerResponded
						selectorTarget:(void *)self
								refCon:NULL];
	
	if (status == kOBEXSuccess) {
		// connect command sent successfully
		[self startedOp:kOBEXOpCodeConnect];		
		
	} else {
		if (mConnectHeadersDataRef) {
			CFRelease(mConnectHeadersDataRef);
			mConnectHeadersDataRef = NULL;
		}
		
		[mOBEXSession release];
		mOBEXSession = NULL;
		
		[self reportError:status summary:@"OBEXConnect failed"];
	}
	
	return status;	
}


- (OBEXError)connectToDeviceWithAddress:(NSString *)address 
						  withChannelID:(BluetoothRFCOMMChannelID)channelID
							   toTarget:(NSData *)target
{
	if (debug) NSLog(DEBUG_NAME @"[connectToDeviceWithAddress] entry. address:%@ channelID:%d", address, channelID);	
	
	BluetoothDeviceAddress deviceAddress;
	OBEXError status = kOBEXSuccess;
	
	status = IOBluetoothNSStringToDeviceAddress(address, &deviceAddress);
	if (status != kOBEXSuccess) {
		return [self reportError:status summary:[NSString stringWithFormat:@"Cannot read given address '%@'", address]];
	}
	
	IOBluetoothDevice *device = [IOBluetoothDevice withAddress:&deviceAddress];
	return [self connectToDevice:device withChannelID:channelID toTarget:target];	
}

//
// Read server response to my 'Connect' request
//
- (void)handleConnectResponse:(const OBEXConnectCommandResponseData*)eventData
{
	if (debug) NSLog(DEBUG_NAME @"[handleConnectResponse] Entry. Server response = 0x%x", eventData->serverResponseOpCode);	
	
	// release the initial request headers
	if (mConnectHeadersDataRef) {
		CFRelease(mConnectHeadersDataRef);
		mConnectHeadersDataRef = NULL;
	}
	
	// process the server response
	mMaxPacketLength = eventData->maxPacketSize;
	if (debug) NSLog(DEBUG_NAME @"Negotiated max packet length: %d", mMaxPacketLength);
	
	if (eventData->serverResponseOpCode != kOBEXResponseCodeSuccessWithFinalBit) {
		[self completedConnect:[self reportError:kOBEXSessionBadRequestError 
										 summary:[NSString stringWithFormat:@"CONNECT request failed. Server response code: 0x%x", eventData->serverResponseOpCode]]];
		
	} else {
		if (debug) NSLog(DEBUG_NAME @"Connected over RFCOMM/OBEX.");
		
		// Read response headers...
		
		CFDictionaryRef headersDictRef = 
			OBEXGetHeaders(eventData->headerDataPtr, eventData->headerDataLength);
		
		CFDataRef whoDataRef = NULL;
		long connectionID = -1;
		
		if (headersDictRef) {
			if (debug) {
				NSLog(@"CONNECT response: <<<<<<");
				CFShow(headersDictRef);	
			}			
			if (CFDictionaryGetCountOfKey(headersDictRef, kOBEXHeaderIDKeyWho) > 0) {
				whoDataRef = (CFDataRef)(CFDictionaryGetValue(headersDictRef, kOBEXHeaderIDKeyWho));
			}			
			if (CFDictionaryGetCountOfKey(headersDictRef, kOBEXHeaderIDKeyConnectionID) > 0) {
				CFDataRef dataRef = (CFDataRef)CFDictionaryGetValue(headersDictRef, kOBEXHeaderIDKeyConnectionID);	
				if (dataRef) {
					connectionID = [self connectionIDDataRefToLong:dataRef];
				}
			}			
		}
		
		// if there was a connection ID header, and there's no connection ID
		// currently set for this session, set it to be the newly received one
		if (connectionID != -1 && [self connectionID] == -1) {
			[self setConnectionID:connectionID];
		}			
		
		[self completedConnect:kOBEXSuccess 
					   fromWho:(whoDataRef)? (NSData *)whoDataRef : nil
				  connectionID:connectionID];		
		
		if (headersDictRef)
			CFRelease(headersDictRef);
	}	
}


#pragma mark -
#pragma mark Disconnect

- (void)completedDisconnect:(OBEXError)error
{
	[self completedCurrentOp];
	if ([mDelegate respondsToSelector:callbackDisconnectComplete]) {
		[mDelegate obexClientSessionDisconnectComplete:self 
												 error:error];
	}
}

- (OBEXError)disconnect
{
	if (debug) NSLog(DEBUG_NAME @"[disconnect] entry.");	
	OBEXError status;
	
	// check is connected
	if (![self isConnected]) 
		return [self reportNotConnectedError];
	
	// check not busy with other operation
	if ([self isBusy]) 
		return [self reportBusyError];

	// send the request
	status = [mOBEXSession OBEXDisconnect:(void *)NULL
					optionalHeadersLength:(size_t)0
							eventSelector:callbackServerResponded
						   selectorTarget:(void *)self
								   refCon:NULL];

	if (status == kOBEXSuccess) {
		// disconnect command sent successfully
		[self startedOp:kOBEXOpCodeDisconnect];	
	} else {
		[self reportError:status summary:@"OBEXDisconnect failed"];
	}
	
	return status;
}
	
//
// Read server response to my 'Disconnect' request
//
- (void)handleDisconnectResponse:(const OBEXDisconnectCommandResponseData*)eventData
{
	if (debug) NSLog(DEBUG_NAME @"[handleDisconnectResponse] Entry. Server response = 0x%x\n", eventData->serverResponseOpCode);
	
	// process the server response
	if (eventData->serverResponseOpCode != kOBEXResponseCodeSuccessWithFinalBit) {
		// disconnection failure
		[self completedDisconnect:[self reportError:kOBEXGeneralError 
											summary:[NSString stringWithFormat:@"DISCONNECT request failed. Server response code: 0x%x", eventData->serverResponseOpCode]]];
	} else {
		// disconnection success
		if (debug) NSLog(DEBUG_NAME @"Disconnection successful.");
		[self completedDisconnect:kOBEXSuccess];
	}	
}


#pragma mark -
#pragma mark Put

- (void)cleanupPut
{
	if (mClosePutFileWhenDone && mCurrentPutBodyHandler) {
		[[mCurrentPutBodyHandler file] closeFile];
	}
	
	mClosePutFileWhenDone = NO;
	
	if (mCurrentPutBodyHandler != nil) {
		[mCurrentPutBodyHandler release];
		mCurrentPutBodyHandler = nil;
	}		
	
	if (mTempPutDataBuffer != nil) {
		[mTempPutDataBuffer release];
		mTempPutDataBuffer = nil;
	}		
}

- (void)completedPut:(OBEXError)error
{
	[self cleanupPut];
	[self completedCurrentOp];
	if ([mDelegate respondsToSelector:callbackPutComplete]) {
		[mDelegate obexClientSessionPutComplete:self 
										  error:error];
	}
}

- (void)notifyPutProgress:(int)amount
{
	if ([mDelegate respondsToSelector:callbackPutProgress]) {
		[mDelegate obexClientSessionPutProgress:self 
									   byAmount:amount];
	}
}

- (OBEXError)putFileAtPath:(NSString *)localFilePath
{
	NSFileHandle *fileHandle = [NSFileHandle fileHandleForReadingAtPath:localFilePath];
	if (!fileHandle) {
		NSString *s = [NSString stringWithFormat:@"Cannot read file '%@'", localFilePath];
		return [self reportError:kOBEXBadArgumentError summary:s];		
	}
	
	NSString *name;
	NSNumber *fileSize;
	int length = -1;
		
	NSDictionary *attrs;
	if (attrs = [[NSFileManager defaultManager] fileAttributesAtPath:localFilePath
														traverseLink:YES]) {
		if (fileSize = [attrs objectForKey:NSFileSize])
			length = [fileSize intValue];
	}
	
	// close file handle when PUT is finished
	mClosePutFileWhenDone = YES;
	
	return [self putFile:fileHandle 
				withName:[localFilePath lastPathComponent]
					type:nil 
				  length:length];
}


- (OBEXError)putFile:(NSFileHandle *)file
			withName:(NSString *)name
				type:(NSString *)type
			  length:(int)totalLength
{
	if (debug) NSLog(DEBUG_NAME @"[putFile] entry. Name: %@", name);
	
	OBEXError status = kOBEXSuccess;
	
	// check args
	if (name == nil) 
		return [self reportError:kOBEXBadArgumentError summary:@"Given name was nil"];
	
	// check is connected
	if (![self isConnected]) 
		return [self reportNotConnectedError];
	
	// check not busy with other operation
	if ([self isBusy]) 
		return [self reportBusyError];
	
	// release old headers if necessary
	if (mPutHeadersDataRef) {
		CFRelease(mPutHeadersDataRef);
		mPutHeadersDataRef = NULL;
	}
	
	// pack the headers. 
	CFMutableDictionaryRef headersDictRef = CFDictionaryCreateMutable(NULL, 4, &kCFTypeDictionaryKeyCallBacks, &kCFTypeDictionaryValueCallBacks );
	if (!headersDictRef) return [self reportCreateHeadersError];
	
	// add connection ID header
	status = [self addConnectionIDToHeaders:headersDictRef];
	if (status != kOBEXSuccess) {
		CFRelease(headersDictRef);
		return [self reportAddHeaderError:@"Connection-ID" error:status];
	}		
	
	// add name header
	status = OBEXAddNameHeader((CFStringRef)name, headersDictRef);
	if (status != kOBEXSuccess) {
		CFRelease(headersDictRef);
		return [self reportAddHeaderError:@"Name" error:status];
	}
	
	// add type header
	if (type != nil) {
		status = OBEXAddTypeHeader((CFStringRef)type, headersDictRef);
		if (status != kOBEXSuccess) {
			CFRelease(headersDictRef);
			return [self reportAddHeaderError:@"Type" error:status];
		}
	}
	
	// add length header
	if (totalLength != -1) {
		status = OBEXAddLengthHeader((uint32_t)totalLength, headersDictRef);
		if (status != kOBEXSuccess) {
			CFRelease(headersDictRef);
			return [self reportAddHeaderError:@"Length" error:status];
		}
	}
	
	/*
	if (debug) {
		NSLog(@"PUT request: >>>>>>");
		CFShow(headersDictRef);		
	}
	*/
	
	// get headers as bytes
	mPutHeadersDataRef = OBEXHeadersToBytes(headersDictRef);
	CFRelease(headersDictRef);	
	
	// now get body data
	BOOL isLastChunk = NO;
	void *bodyData = NULL;
	size_t bodyDataLength = 0;
	if (file != nil) {
		
		[mCurrentPutBodyHandler release];
		mCurrentPutBodyHandler = nil;
		mCurrentPutBodyHandler = [[BBOBEXFileReader alloc] initWithFile:file
															  forSession:mOBEXSession 
																isClient:YES];
		mTempPutDataBuffer = 
			[mCurrentPutBodyHandler readNextChunkForHeaderLength:CFDataGetLength(mPutHeadersDataRef)
															forOp:kOBEXOpCodePut 
													  isLastChunk:&isLastChunk];
		// retain data.
		[mTempPutDataBuffer retain];
		
		
		// set to use this body data for this packet
		bodyData = (void *)[mTempPutDataBuffer bytes];
		bodyDataLength = [mTempPutDataBuffer length];
	}
		
	// send the request
	status = [mOBEXSession OBEXPut:isLastChunk
					   headersData:(void*)CFDataGetBytePtr(mPutHeadersDataRef)
				 headersDataLength:CFDataGetLength(mPutHeadersDataRef)
						  bodyData:bodyData
					bodyDataLength:bodyDataLength
					 eventSelector:callbackServerResponded
					selectorTarget:(void *)self
							refCon:NULL];
	
	if (status == kOBEXSuccess) {
		// PUT command sent successfully
		[self startedOp:kOBEXOpCodePut];		
	} else {
		if (mPutHeadersDataRef) {
			CFRelease(mPutHeadersDataRef);
			mPutHeadersDataRef = NULL;
		}
		
		[self reportError:status summary:@"OBEXPut failed"];
	}	
	return status;	
}

//
// Read server response to my PUT request
//
- (void)handlePutResponse:(const OBEXPutCommandResponseData*)eventData
{
	if (debug) NSLog(DEBUG_NAME "[handlePutResponse] Entry. Server response = 0x%x", eventData->serverResponseOpCode);
	
	// release the initial request headers
	if (mPutHeadersDataRef) {
		CFRelease(mPutHeadersDataRef);
		mPutHeadersDataRef = NULL;
	}
	
	// Does the user want to abort the current command?
	if ([self abortOnCurrentServerResponseCode:eventData->serverResponseOpCode]) {
		return;
	}	
	
	switch (eventData->serverResponseOpCode) {
		
		case kOBEXResponseCodeContinueWithFinalBit:
			// Server is asking us to continue the PUT.
			if (debug) NSLog(DEBUG_NAME @"[handlePutResponse] got kOBEXResponseCodeContinueWithFinalBit.");

			// Error if we got "continue" response even though we have nothing
			// to send
			if (!mCurrentPutBodyHandler) {
				[self completedPut:[self reportError:kOBEXSessionBadResponseError
											 summary:@"Got 'continue' response but nothing left to PUT"]];
				return;
			}
			
			BOOL isLastChunk = NO;
			OBEXError status;		
			
			NSData *tempPutDataBuffer = 
				[mCurrentPutBodyHandler readNextChunkForHeaderLength:(size_t)0
																forOp:kOBEXOpCodePut 
														  isLastChunk:&isLastChunk];	
			
			[tempPutDataBuffer retain];
			[mTempPutDataBuffer release];	// release previous chunk
			mTempPutDataBuffer = tempPutDataBuffer;
			 
			if (!mTempPutDataBuffer) {
				[self completedPut:[self reportError:kOBEXInternalError
											 summary:@"Error reading file, cannot PUT next chunk"]];
				return;				
			}
			
			// send the request
			status = [mOBEXSession OBEXPut:isLastChunk
							   headersData:(void*)NULL
						 headersDataLength:(size_t)0
								  bodyData:(void*)[mTempPutDataBuffer bytes]
							bodyDataLength:[mTempPutDataBuffer length]
							 eventSelector:callbackServerResponded
							selectorTarget:(void *)self
									refCon:NULL];

			if (status == kOBEXSuccess) {
				if (debug) NSLog(DEBUG_NAME @"[handlePutResponse] Sent next OBEXPut request");
				
				// tell delegate more data has been sent
				[self notifyPutProgress:[mTempPutDataBuffer length]];
				
			} else {
				// quit out of the PUT now
				NSString *s = [NSString stringWithFormat:@"Sending of next OBEXPut failed. Error = 0x%x", status];
				[self completedPut:[self reportError:status summary:s]];
			}	
			break;

		case kOBEXResponseCodeSuccessWithFinalBit:
			// The PUT operation is complete.
			if (debug) NSLog(DEBUG_NAME @"[handlePutResponse] got kOBEXResponseCodeSuccessWithFinalBit.");
		
			// finish up and notify delegate
			[self completedPut:kOBEXSuccess];
			break;
			
		default:
		{
			// quit out of the PUT now
			NSString *msg = [NSString stringWithFormat:@"PUT failed, server responded 0x%02x ('%@')", 
				eventData->serverResponseOpCode,
				[BBOBEXUtil describeServerResponse:eventData->serverResponseOpCode]];
			[self completedPut:[self reportError:kOBEXGeneralError summary:msg]];
		}
	}
}


#pragma mark -
#pragma mark Get


- (void)completedGet:(OBEXError)error
{
	[self completedCurrentOp];
	if ([mDelegate respondsToSelector:callbackGetComplete]) {
		[mDelegate obexClientSessionGetComplete:self 
										  error:error];
	}
}

- (void)notifyGetProgress:(int)amount
{
	if ([mDelegate respondsToSelector:callbackGetProgress]) {
		[mDelegate obexClientSessionGetProgress:self 
									   byAmount:amount];
	}
}


- (OBEXError)getWithName:(NSString *)name
					type:(NSString *)type
			 writeToFile:(NSFileHandle *)file
{
	if (debug) NSLog(DEBUG_NAME @"[getWithName] entry. name:%@", name);	
	
	// check args
	if (name == nil && type == nil)
		return [self reportError:kOBEXBadArgumentError summary:@"Given name and type are both nil"];
	if (file == nil)
		return [self reportError:kOBEXBadArgumentError summary:@"Given file handle is nil"];		
	
	// check is connected
	if (![self isConnected]) 
		return [self reportNotConnectedError];
	
	// check not busy with other operation
	if ([self isBusy]) 
		return [self reportBusyError];
	
	// release old headers if necessary
	if (mGetHeadersDataRef) {
		CFRelease(mGetHeadersDataRef);
		mGetHeadersDataRef = NULL;
	}		
	
	// pack the new headers. 
	CFMutableDictionaryRef headersDictRef = CFDictionaryCreateMutable(NULL, 3, 
		&kCFTypeDictionaryKeyCallBacks, &kCFTypeDictionaryValueCallBacks);
	if (!headersDictRef) return [self reportCreateHeadersError];
	
	OBEXError status;
	
	// add connection ID header
	status = [self addConnectionIDToHeaders:headersDictRef];
	if (status != kOBEXSuccess) {
		CFRelease(headersDictRef);
		return [self reportAddHeaderError:@"Connection-ID" error:status];
	}		
	
	// add name header
	if (name != nil) {
		status = OBEXAddNameHeader((CFStringRef)name, headersDictRef);
		if (status != kOBEXSuccess) {
			CFRelease(headersDictRef);
			return [self reportAddHeaderError:@"Name" error:status];
		}
	}
	
	// add type header
	if (type != nil) {
		status = OBEXAddTypeHeader((CFStringRef)type, headersDictRef);
		if (status != kOBEXSuccess) {
			CFRelease(headersDictRef);
			return [self reportAddHeaderError:@"Type" error:status];
		}
	}
	
	// get headers as bytes 
	mGetHeadersDataRef = OBEXHeadersToBytes(headersDictRef);
	CFRelease(headersDictRef);		
	
	// Send the request
	status = [mOBEXSession OBEXGet:TRUE
						   headers:(void*)CFDataGetBytePtr(mGetHeadersDataRef)
					 headersLength:CFDataGetLength(mGetHeadersDataRef)
					 eventSelector:callbackServerResponded
					selectorTarget:(void *)self
							refCon:self];	
	
	if (status == kOBEXSuccess) {
		// GET command sent successfully
		
		// retain the given file
		[file retain];
		[mCurrentGetFile release];
		mCurrentGetFile = file;
		
		// notify delegate
		[self startedOp:kOBEXOpCodeGet];		
		
	} else {
		if (mGetHeadersDataRef) {
			CFRelease(mGetHeadersDataRef);
			mGetHeadersDataRef = NULL;
		}
		
		[self reportError:status summary:@"OBEXGet failed"];
	}
	
	return status;
}

- (void)handleGetResponse:(const OBEXGetCommandResponseData*)eventData
{
	if (debug) NSLog(DEBUG_NAME @"[handleGetResponse] Entry. Server response = 0x%x\n", eventData->serverResponseOpCode);		
	
	// release the initial request headers
	if (mGetHeadersDataRef) {
		CFRelease(mGetHeadersDataRef);
		mGetHeadersDataRef = NULL;
	}		
	
	// abort command?
	if ([self abortOnCurrentServerResponseCode:eventData->serverResponseOpCode]) {
		return;
	}
	
	// now parse the headers to get the received body data and write it to file.
	CFDictionaryRef headersDictRef = 
		OBEXGetHeaders(eventData->headerDataPtr, eventData->headerDataLength);	
	
	// get body data
	if (headersDictRef) {
		CFDataRef bodyDataRef = NULL;
		CFDataRef endOfBodyDataRef = NULL;			
		int bodyDataReceived = 0;
		
		/*
		if (debug) {
			NSLog(@"GET response: <<<<<<");
			CFShow(headersDictRef);		
		}
		*/
		
		if (CFDictionaryGetCountOfKey(headersDictRef, kOBEXHeaderIDKeyBody) > 0) {
			bodyDataRef = (CFDataRef)(CFDictionaryGetValue(headersDictRef, kOBEXHeaderIDKeyBody));
			if (bodyDataRef) {
				[mCurrentGetFile writeData:(NSData *)bodyDataRef];
				bodyDataReceived += CFDataGetLength(bodyDataRef);
			}
		}
	
		// get end-of-body-data
		if (CFDictionaryGetCountOfKey(headersDictRef, kOBEXHeaderIDKeyEndOfBody) > 0) {
			endOfBodyDataRef = (CFDataRef)(CFDictionaryGetValue(headersDictRef, kOBEXHeaderIDKeyEndOfBody));
			if (endOfBodyDataRef) {					
				[mCurrentGetFile writeData:(NSData *)endOfBodyDataRef];			
				bodyDataReceived += CFDataGetLength(endOfBodyDataRef);
			}
		}	
		
		// notify delegate we've gotten some data
		if (bodyDataReceived > 0) { 
			[self notifyGetProgress:bodyDataReceived];
		}
		
		// clean up
		CFRelease(headersDictRef);
	}
	
	
	// now process the server response
	
	switch (eventData->serverResponseOpCode) {
		
		case kOBEXResponseCodeContinueWithFinalBit:
			if (debug) NSLog(DEBUG_NAME @"[handleGetResponse] kOBEXResponseCodeContinueWithFinalBit");
			
			// previous GET was successful, and we need to do another GET
			// request to get more data.
			OBEXError status = [mOBEXSession OBEXGet:TRUE
											 headers:NULL
									   headersLength:0
									   eventSelector:callbackServerResponded
									  selectorTarget:(void *)self
											  refCon:self];			
			
			if (status == kOBEXSuccess) {
				if (debug) NSLog(DEBUG_NAME @"[handleGetResponse] Sent next OBEXGet request");
				
			} else {
				// quit out of the GET now				
				NSString *s = [NSString stringWithFormat:@"Sending of next OBEXGet failed. Error = 0x%x", status];
				[self completedGet:[self reportError:status summary:s]];
			}
			break;			
			
		case kOBEXResponseCodeSuccessWithFinalBit:
			if (debug) NSLog(DEBUG_NAME @"[handleGetResponse] kOBEXResponseCodeSuccessWithFinalBit");
			
			// GET is complete
			[self completedGet:kOBEXSuccess];
			break;			
			
		default:
		{
			// quit out of the GET now
			NSString *msg = [NSString stringWithFormat:@"GET failed, server responded 0x%02x ('%@')", 
				eventData->serverResponseOpCode,
				[BBOBEXUtil describeServerResponse:eventData->serverResponseOpCode]];
			[self completedGet:[self reportError:kOBEXGeneralError summary:msg]];
		}
	}
}
	

#pragma mark -
#pragma mark Set path

- (void)completedSetPath:(OBEXError)error
{
	[self completedCurrentOp];
	if ([mDelegate respondsToSelector:callbackSetPathComplete]) {
		[mDelegate obexClientSessionSetPathComplete:self 
											  error:error];
	}
}

- (OBEXError)setPath:(NSString *)pathName 
		   withFlags:(OBEXFlags)flags
{
	if (debug) NSLog(DEBUG_NAME @"[setPath] entry. pathName:%@", pathName);	
	OBEXError status = kOBEXSuccess;
	
	// check is connected
	if (![self isConnected]) 
		return [self reportNotConnectedError];
	
	// check not busy with other operation
	if ([self isBusy]) 
		return [self reportBusyError];	
	
	// release old headers if necessary
	if (mSetPathHeadersDataRef) {
		CFRelease(mSetPathHeadersDataRef);
		mSetPathHeadersDataRef = NULL;
	}		
	
	// pack the name header
	void *headersData = NULL;
	size_t headersLength = 0;	
	if (pathName != nil) {
		CFMutableDictionaryRef headersDictRef = CFDictionaryCreateMutable(NULL, 2, NULL, NULL);
		if (!headersDictRef) return [self reportCreateHeadersError];
		
		// add connection ID header
		status = [self addConnectionIDToHeaders:headersDictRef];
		if (status != kOBEXSuccess) {
			CFRelease(headersDictRef);
			return [self reportAddHeaderError:@"Connection-ID" error:status];
		}	
		
		status = OBEXAddNameHeader((CFStringRef)pathName, headersDictRef);
		if (status != kOBEXSuccess) {
			CFRelease(headersDictRef);
			return [self reportAddHeaderError:@"Name" error:status];
		}
		
		/*
		if (debug) {
			NSLog(@"SETPATH request: >>>>>>");
			CFShow(headersDictRef);	
		}
		 */
		
		// get headers as bytes
		mSetPathHeadersDataRef = OBEXHeadersToBytes(headersDictRef);
		CFRelease(headersDictRef);
		
		headersData = (void*)CFDataGetBytePtr(mSetPathHeadersDataRef);
		headersLength = CFDataGetLength(mSetPathHeadersDataRef);
	}
	
	// set path should always fit in one packet?
	status = [mOBEXSession OBEXSetPath:flags
							 constants:0
					   optionalHeaders:headersData
				 optionalHeadersLength:headersLength
						 eventSelector:callbackServerResponded
						selectorTarget:(void *)self
								refCon:NULL];
	
	if (status == kOBEXSuccess) {
		// set path command sent successfully
		[self startedOp:kOBEXOpCodeSetPath];		
	} else {
		if (mSetPathHeadersDataRef) {
			CFRelease(mSetPathHeadersDataRef);
			mSetPathHeadersDataRef = NULL;
		}
		
		[self reportError:status summary:@"OBEXSetPath failed"];
	}
	return status;		
}

- (void)handleSetPathResponse:(const OBEXSetPathCommandResponseData*)eventData
{
	if (debug) NSLog(DEBUG_NAME @"[handleSetPathResponse] Entry. Server response = 0x%x\n", eventData->serverResponseOpCode);	

	// release the initial request headers
	if (mSetPathHeadersDataRef) {
		CFRelease(mSetPathHeadersDataRef);
		mSetPathHeadersDataRef = NULL;
	}	
	
	// process the server response
	if (eventData->serverResponseOpCode != kOBEXResponseCodeSuccessWithFinalBit) {
		// set path failure
		[self completedSetPath:[self reportError:kOBEXGeneralError 
										 summary:[NSString stringWithFormat:@"SETPATH request failed. Server response code: 0x%x", eventData->serverResponseOpCode]]];
	} else {
		// connection success
		if (debug) NSLog(DEBUG_NAME @"Set Path successful.");
		[self completedSetPath:kOBEXSuccess];		
	}

}


#pragma mark -
#pragma mark Process server responses


- (void)forceOpComplete:(OBEXOpCode)opType error:(OBEXError)error
{	
	if (opType != 0) {
		switch (opType) {
			case kOBEXOpCodeConnect:
				[self completedConnect:error];
				break;
			case kOBEXOpCodeDisconnect:
				[self completedDisconnect:error];
				break;
			case kOBEXOpCodePut:
				[self completedPut:error];
				break;
			case kOBEXOpCodeGet:		
				[self completedGet:error];
				break;
			case kOBEXOpCodeSetPath:
				[self completedSetPath:error];
				break;			
			case kOBEXOpCodeAbort:
				[self completedAbort:error];
				break;	
		}
	}
}


//
// Called when an event occurs - e.g. we have received a response from the 
// server.
//
- (void)handleSessionEvent:(const OBEXSessionEvent *)event
{
	if (debug) NSLog(DEBUG_NAME @"[handleSessionEvent] entry.");
	
	switch (event->type)
	{
		case (kOBEXSessionEventTypeConnectCommandResponseReceived):
			[self handleConnectResponse:&event->u.connectCommandResponseData];
			break;
			
		case (kOBEXSessionEventTypeDisconnectCommandResponseReceived):
			[self handleDisconnectResponse:&event->u.disconnectCommandResponseData];
			break;
			
		case (kOBEXSessionEventTypeGetCommandResponseReceived):
			[self handleGetResponse:&event->u.getCommandResponseData];
			break;
			
		case (kOBEXSessionEventTypePutCommandResponseReceived):
			[self handlePutResponse:&event->u.putCommandResponseData];
			break;
			
		case (kOBEXSessionEventTypeAbortCommandResponseReceived):
			[self handleAbortResponse:&event->u.abortCommandResponseData];
			break;
			
		case (kOBEXSessionEventTypeSetPathCommandResponseReceived):
			[self handleSetPathResponse:&event->u.setPathCommandResponseData];
			break;
			
		case (kOBEXSessionEventTypeError):
			if (debug) NSLog(DEBUG_NAME @"[handleSessionEvent] kOBEXSessionEventTypeError");
			OBEXErrorData errorData = event->u.errorData;
			
			if (errorData.error == kOBEXSessionTransportDiedError) {
				// this is a common one so do a special error msg
				[self reportError:errorData.error summary:@"bluetooth transport connection broken"];
				
			} else {
				// just do a general error msg
				[self reportError:errorData.error summary:[NSString stringWithFormat:@"OBEXSession error (error=%d)", errorData.error]];
			}
			
			// don't know how to deal with it, try to finish current op
			[self forceOpComplete:mCurrentOp error:errorData.error];
			break;
			
		default:
			if (debug) NSLog(DEBUG_NAME @"[handleSessionEvent] unknown command type (%x)", event->type);
			
			// don't know how to deal with it, try to finish current op
			[self forceOpComplete:mCurrentOp error:kOBEXUnsupportedError];
			break;
	}
}

#pragma mark -
#pragma mark Other public methods

- (void)setDelegate:(id)delegate
{
	mDelegate = delegate;
}

- (id)delegate
{
	return mDelegate;
}

- (BOOL)isBusy
{
	return mCurrentOp != 0;
}

- (void)setMaxPacketLength:(OBEXMaxPacketLength)length
{
	mMaxPacketLength = length;
}

- (void)setWhoHeaderValue:(NSData *)whoValue
{
	[whoValue retain];
	[mWhoHeaderValue release];
	mWhoHeaderValue = whoValue;
}

- (NSData *)whoHeaderValue
{
	return mWhoHeaderValue;
}

- (void)setConnectionID:(long)connectionID
{
	CFDataRef connIDDataRef = [self connectionIDLongToDataRef:connectionID];
	
	if (connIDDataRef) {
		// set internal connection ID (CFDatRef format)
		if (mConnectionIDDataRef) 
			CFRelease(mConnectionIDDataRef);
		mConnectionIDDataRef = connIDDataRef;
		
		// set internal connection ID (long format)		
		mConnectionID = connectionID;
	} else {
		if (debug) NSLog(DEBUG_NAME @"[setConnectionID] error converting long to CFDataRef");
	}
}

- (long)connectionID
{
	return mConnectionID;
}

- (BOOL)hasOpenOBEXConnection
{
	if (mOBEXSession) {
		return [mOBEXSession hasOpenOBEXConnection];
	}
	return NO;
}

- (BOOL)hasOpenTransportConnection
{
	if (mOBEXSession) {
		return [mOBEXSession hasOpenTransportConnection];
	}
	return NO;
}

// If already closed, this does nothing and returns kOBEXSessionNotConnectedError
- (OBEXError)close
{
	if (debug) NSLog(DEBUG_NAME @"[close] entry.");
	if (mClosed)
		return kOBEXSessionNotConnectedError;
	
	OBEXError status = kOBEXSuccess;
	
	if (mOBEXSession == nil) {
		status = kOBEXSessionNotConnectedError;
		[self reportError:status summary:@"Session not connected"];
	} else {
		status = [mOBEXSession closeTransportConnection];
		if (status != kOBEXSuccess)
			[self reportError:status summary:@"Error closing transport connection"];
	}
	
	[self cleanUp];
	mClosed = YES;
	
	return status;
}


- (IOBluetoothRFCOMMChannel *)getRFCOMMChannel
{
	if (mOBEXSession) {
		return [mOBEXSession getRFCOMMChannel];
	} 
	
	return nil;
}

+ (void)setDebug:(BOOL)doDebug
{
	debug = doDebug;
}


#pragma mark -

- (void)dealloc
{
	if (!mClosed) [self close];
	
	[super dealloc];
}

@end