# $Id: panes.py,v 1.21 2006/09/18 21:43:28 mwm Exp $
#
# panes.py -- Handle panes (sometimes known as "frames")
#
#    Copyright (C) 2001  Mike Meyer <mwm@mired.org>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Panes - provide panes to put plwm windows in.

The idea is that each screen is completely covered by panes. Each
pixel on the screen must belong to one and only one pane. Focus
ignores the mouse, but is moved from pane to pane via the
keyboard. Windows open in the current pane, and are told to resize
themselves to fit that pane.

The rest of the UI - well, that's up to you."""

from Xlib import X, Xutil, Xatom
from plwm import wmanager, wmevents, modewindow, cfilter

WM_TRANSIENT_FOR = None
class panesManager:
    "panesManager - pane mixin for window manager."

    panes_window_gravity = X.CenterGravity
    panes_maxsize_gravity = X.CenterGravity
    panes_transient_gravity = X.CenterGravity

    def __wm_screen_init__(my):
        "Create the list of panes with no current pane."
        global WM_TRANSIENT_FOR

        wmanager.debug('panesManager', 'inited')

        # Warning - if we manage more than one display, this breaks!
        if not WM_TRANSIENT_FOR:
            WM_TRANSIENT_FOR = my.display.intern_atom("WM_TRANSIENT_FOR")

        my.panes_list = []
        my.panes_current = None

    def __wm_init__(my):
        "Enable activation, then activate the first pane."

        Pane.activate = Pane.do_activate
        my.panes_list[my.panes_current].activate()

    def panes_add(my, pane):
        "Add the given pane to the list of all panes."

        wmanager.debug('panesManager', 'added pane %s', `pane`)
        my.panes_list.append(pane)
        if my.panes_current is None: my.panes_current = 0

    def panes_remove(my, test):
        "Remove panes that match the filter."

        old = my.panes_list[my.panes_current]
        my.panes_list = filter(cfilter.Not(test), my.panes_list)
        try: my.panes_current = my.panes_list.index(old)
        except ValueError: my.panes_current = 0

    def panes_goto(my, new):
        "Go to the given pane."

        if new == my.panes_current: return
        if 0 <= new < len(my.panes_list):
            my.panes_list[my.panes_current].deactivate()
            my.panes_current = new
            my.panes_list[new].activate()

    def panes_activate(my, pane):
        "Make the given pane the current pane."

        my.panes_goto(my.panes_list.index(pane))

    def panes_next(my):
        "Move to the next pane."

        if len(my.panes_list) > 1:
            wmanager.debug('panesManager', 'next pane')
            my.panes_list[my.panes_current].deactivate()
            my.panes_current = my.panes_current + 1
            if my.panes_current == len(my.panes_list): my.panes_current = 0
            my.panes_list[my.panes_current].activate()

    def panes_prev(my):
        "Move to the previous pane."

        if len(my.panes_list) > 1:
            wmanager.debug('panesManager', 'previous pane')
            my.panes_list[my.panes_current].deactivate()
            my.panes_current = my.panes_current - 1
            if my.panes_current < 0: my.panes_current = len(my.panes_list) - 1
            my.panes_list[my.panes_current].activate()


    def panes_number(my, number):
        "Reorder the panes list so the current pane has the given number."

        my.panes_list[my.panes_current], my.panes_list[number] = \
                                         my.panes_list[number], \
                                         my.panes_list[my.panes_current]
        my.panes_current = number

    # The intent is that if the user has a favorite configuration of panes
    # and windows, that gets set up and then panes_save gets invoked. Later,
    # the windows can be put back in the same panes by invoking panes_restore.
    # I recommend a script that recreates the pane configuration then
    # calls panes_restore be in the config file.
    panes_saved = {}
    def panes_save(my):
        "Record which pane all the windows are in."
        
        my.panes_saved.clear()
        for client in my.query_clients():
            if client.panes_pane:
                my.panes_saved[client] = my.panes_list.index(client.panes_pane)

    def panes_restore(my):
        "Put the clients back in the saved panes."

        clients = my.query_clients()
        for client in clients:
            pane = my.panes_saved.get(client, None)
            if pane is not None and 0 <= pane < len(my.panes_list) and \
               my.panes_list[pane].screen == client.screen:
                my.panes_list[pane].add_window(client)

        for client in clients:
            if not client.panes_pane: client.iconify()
        

class panesScreen:
    "paneScreen - pane mixin for Screens."

    def  __screen_client_init__(my):
        "Create the initial pane object for this screen."

        wmanager.debug('panesScreen', 'Initializing screen %d', my.number)
        my.dispatch.add_handler(X.ConfigureRequest, my.panes_configure)
        pane = Pane(my, 0, 0, my.root_width, my.root_height)
        my.panes_fullscreen(pane)
        my.wm.panes_add(pane)

    def panes_fullscreen(my, pane):
        "Make the pane use the all the available screen."

        pane.width = my.root_width
        pane.x = 0
        if not hasattr(my, 'modewindow_pos'):
            pane.height = my.root_height
            pane.y = 0
        else:
            pane.height = my.root_height - my.modewindow_mw.height
            if my.modewindow_pos != modewindow.TOP: pane.y = 0
            else: pane.y = my.modewindow_mw.height
        
    def panes_configure(my, event):
        "A window changed, so pass it on to my pane."

        w = my.get_window(event.window)
        if w:
            if event.value_mask & (X.CWX | X.CWY | X.CWWidth | X.CWHeight):
                w.panes_pane.place_window(w)
            if event.value_mask & X.CWStackMode and event.stack_mode == X.Above \
               and my.allow_self_changes(w):
                w.panes_pane.add_window(w)


class panesClient:
    """panesClient - pane mixin for clients

    Note that this needs to be mixed in *after* any mixins that affect window
    geometry, such as border."""

    def __client_init__(my):
        "Arrange to open in the current pane."

        wmanager.debug('Pane', 'Initing client %s', my)
        # Set this clients gravity 
        if my.window.get_property(WM_TRANSIENT_FOR, Xatom.WINDOW, 0, 1) is not None:
            my.panes_gravity = my.wm.panes_transient_gravity
        elif my.sizehints and my.sizehints.flags & Xutil.PMaxSize:
            my.panes_gravity = my.wm.panes_maxsize_gravity 
        else:
            my.panes_gravity = my.wm.panes_window_gravity

        my.panes_pointer_pos = my.panes_pane = None
        pane = my.wm.panes_list[my.wm.panes_current]
        if pane.screen != my.screen:
            pane = filter(lambda p, m=my.screen: p.screen == m, my.wm.panes_list)[0]
        pane.add_window(my)
        my.dispatch.add_handler(X.UnmapNotify, my.panes_unmap)
        my.dispatch.add_handler(X.DestroyNotify, my.panes_unmap)

    def panes_unmap(my, event):
        "The window is going away or gone - make sure it's not taking up a pane"

        if my.panes_pane: my.panes_pane.remove_window(my)


class Pane:
    "Pane - the object(s) that manages windows on the screen."
    
    def __init__(my, screen, x, y, width, height):
        "Initialize a pane of the given size on the given screen."

        my.screen, my.x, my.y, my.width, my.height = screen, x, y, width, height
        my.wm = screen.wm
        my.window = None

    def add_window(my, window):
        "Add a window to this pane."

        wmanager.debug('Pane', 'Adding window %s to pane %s', window, my)
        if window == my.window: return
        old = window.panes_pane
        if old != my:
            if old: old.remove_window(window)
            my.place_window(window)
        window.panes_pane = my
        if my.window: my.deactivate()
        my.window = window
        if my.wm.panes_list[my.wm.panes_current] == my:
            my.activate()

    def iconify_window(my):
        "Iconify my window, if any."

        if my.window:
            my.window.iconify()
            my.remove_window(my.window)

    def remove_window(my, window):
        "Tag a window as not belonging to me."

        wmanager.debug('Pane', 'Removing window %s from pane %s' % (window, my))
        window.panes_pane = None
        if my.window == window:
            my.deactivate()
            clients = my.screen.query_clients(panefilter(my), 1)
            if not clients: my.window = None
            else:
                my.window = clients[len(clients) - 1]
                if my.wm.panes_list[my.wm.panes_current] == my:
                    my.activate()

    def place_window(my, window = None):
        "Figure out where the window should be put."

        if not window: window = my.window
        wmanager.debug('Pane', 'Placing window %s for pane %s' %
                       (window, my))
        width, height = window.follow_size_hints(my.width - 2 * window.border_width,
                                                 my.height - 2 * window.border_width)

        # If it doesn't fit, just force it.
        if width > my.width - 2 * window.border_width:
            width = my.width - 2 * window.border_width
        if height > my.height - 2 * window.border_width:
            height = my.height - 2 * window.border_width

        if window.panes_gravity in (X.NorthEastGravity, X.EastGravity,
                              X.SouthEastGravity):
            x = my.x
        elif window.panes_gravity in (X.NorthGravity, X.CenterGravity,
                                      X.SouthGravity):
            x = my.x + (my.width - width) / 2 - window.border_width
        else:
            x = my.x + my.width - width - (2 * window.border_width)

        if window.panes_gravity in (X.NorthEastGravity, X.NorthGravity,
                                    X.NorthWestGravity):
            y = my.y
        elif window.panes_gravity in (X.EastGravity, X.CenterGravity,
                                      X.WestGravity):
            y = my.y + (my.height - height) / 2 - window.border_width
        else:
            y = my.y + my.height - height - (2 * window.border_width)

        x, y, width, height = window.keep_on_screen(x, y, width, height)

        wmanager.debug('Pane-configure', 'Resizing window from %d, %d to %d, %d' %
                       (window.width, window.height, width, height))
        window.moveresize(x, y, width, height)

    def force_window(my):
        "Try and force an application to notice what size it's window is."

        if not my.window: return
        my.window.resize(my.width / 2, my.height / 2)
        my.wm.display.flush()
        my.place_window()

    def next_window(my):
        "Move to the next window in this pane."

        wmanager.debug('Pane', 'next window')
        clients = my.screen.query_clients(panefilter(my), 1)
        if len(clients) > 1:
            my.deactivate()
            my.window = clients[0]
            my.activate()

    def prev_window(my):
        "Move to the previous window in this pane."

        wmanager.debug('Pane', 'previous window')
        clients = my.screen.query_clients(panefilter(my), 1)
        if len(clients) > 1:
            my.deactivate()
            # Lower the old window to make it the "next" window.
            my.window.lowerwindow()
            my.window = clients[len(clients) - 2]
            my.activate()

    def deactivate(my):
        "A place to do anything appropriate for us when losing the focus."

        if my.window and not my.window.withdrawn:
            my.window.panes_pointer_pos = my.window.pointer_position()
            if my.wm.panes_list[my.wm.panes_current] == my:
                my.wm.set_current_client(None)

    def activate(my):
        "Dummy function, reset to do_activate after all windows are opened."

    def do_activate(my):
        "Activate whatever is currently my window."

        my.wm.current_screen = my.screen
        if my.window and not my.window.withdrawn:
            wmanager.debug('Pane', 'Activating window %s in pane %s' %
                           (my.window, my))

            # Will raise window and give focus
            my.window.activate()
            pos = my.window.panes_pointer_pos
            if pos:
                my.window.warppointer(pos[0], pos[1])

    def horizontal_split(my, frac = .5):
        "Split the pane horizontally, taking frac off the bottom."

        if frac <= 0 or 1 <= frac:
            raise ValueError, "Pane splits must be between 0 and 1."

        new_height = int(my.height * frac)
        my.height = my.height - new_height
        new_y = my.y + my.height
        map(my.place_window, my.screen.query_clients(panefilter(my)))
        new_pane = Pane(my.screen, my.x, new_y, my.width, new_height)
        my.wm.panes_add(new_pane)
        my.wm.panes_activate(new_pane)

    def vertical_split(my, frac = .5):
        "Split the pane vertically, taking frac off the right."

        if frac <= 0 or 1 <= frac:
            raise ValueError, "Pane splits must be between 0 and 1."

        new_width = int(my.width * frac)
        my.width = my.width - new_width
        new_x = my.x + my.width
        map(my.place_window, my.screen.query_clients(panefilter(my)))
        new_pane = Pane(my.screen, new_x, my.y, new_width, my.height)
        my.wm.panes_add(new_pane)
        my.wm.panes_activate(new_pane)

    def maximize(my):
        "Make me the only pane on my screen."

        my.wm.panes_remove(lambda x, s = my.screen, m = my:
                           x.screen == s and x != m)
        my.screen.panes_fullscreen(my)
        for window in my.screen.query_clients():
            window.panes_pane = my
            my.place_window(window)
        my.activate()

class panefilter:
    "Filter for windows mapped in the current pane."

    def __init__(my, pane):
        "Set the pane we're active in."

        my.pane = pane

    def __call__(my, window):
        "Check to see if this is our pane."

        return my.pane == window.panes_pane and not cfilter.iconified(window)
