
"""
__version__ = "$Revision: 1.28 $"
__date__ = "$Date: 2004/03/18 02:21:40 $"
"""

from wxPython import wx

import sys
from PythonCardPrototype import binding, event, font, graphic, registry, widget

try:
    import Image
    # necessary to avoid name collision with Image class
    from Image import fromstring
    PIL_FOUND = 1
except ImportError:
    PIL_FOUND = 0

try:
    from Numeric import ArrayType
    NUMERIC_FOUND = 1
except ImportError:
    NUMERIC_FOUND = 0


class BitmapCanvasSpec(widget.WidgetSpec):
    def __init__(self):
        widget.WidgetSpec.__init__(self)
        
        self.name = 'BitmapCanvas'
        self.parent = 'Widget'
        self.parentName = self.parent
        
        self.attributes = self.parseAttributes(self._attributes)
        self.requiredAttributes = self.parseRequiredAttributes()
        self.optionalAttributes = self.parseOptionalAttributes()


"""
wxAND                 src AND dst
wxAND_INVERT          (NOT src) AND dst
wxAND_REVERSE         src AND (NOT dst)
wxCLEAR               0
wxCOPY                src
wxEQUIV               (NOT src) XOR dst
wxINVERT              NOT dst
wxNAND                (NOT src) OR (NOT dst)
wxNOR                 (NOT src) AND (NOT dst)
wxNO_OP               dst
wxOR                  src OR dst
wxOR_INVERT           (NOT src) OR dst
wxOR_REVERSE          src OR (NOT dst)
wxSET                 1
wxSRC_INVERT          NOT src
wxXOR                 src XOR dst
"""

LogicalCopyModes = {
        "AND"    : wx.wxAND,
        "AND_INVERT"          : wx.wxAND_INVERT,
        "AND_REVERSE": wx.wxAND_REVERSE,
        "CLEAR" : wx.wxCLEAR,
        "COPY": wx.wxCOPY,
        "EQUIV"     : wx.wxEQUIV,
        "INVERT": wx.wxINVERT,
        "NAND"  : wx.wxNAND,
        "NOR"  : wx.wxNOR,
        "NO_OP"  : wx.wxNO_OP,
        "OR"  : wx.wxOR,
        "OR_INVERT"  : wx.wxOR_INVERT,
        "OR_REVERSE"  : wx.wxOR_REVERSE,
        "SET"  : wx.wxSET,
        "SRC_INVERT"  : wx.wxSRC_INVERT,
        "XOR"  : wx.wxXOR,
        }

BrushFillStyleList = {
        "Transparent"    : wx.wxTRANSPARENT,
        "Solid"          : wx.wxSOLID,
        "BiDiagonalHatch": wx.wxBDIAGONAL_HATCH,
        "CrossDiagHatch" : wx.wxCROSSDIAG_HATCH,
        "FDiagonal_Hatch": wx.wxFDIAGONAL_HATCH,
        "CrossHatch"     : wx.wxCROSS_HATCH,
        "HorizontalHatch": wx.wxHORIZONTAL_HATCH,
        "VerticalHatch"  : wx.wxVERTICAL_HATCH
        }

class BitmapCanvas(widget.Widget, wx.wxWindow):
    """
    A double-bufferd bitmap drawing area.
    """

    _spec = BitmapCanvasSpec()
    
    def __init__( self, aParent, aResource ) :
        attributes = ['autoRefresh', '_bufImage', '_drawingInProgress', '_pen', '_penColor', '_thickness', '_size']
        self._createAttributes(attributes)
        
        self._thickness = 1
        self.autoRefresh = 1
        self._drawingInProgress = 0

        widget.Widget.__init__( self, aParent, aResource )

        wx.wxWindow.__init__(
            self,
            aParent, 
            self.getId(),
            wx.wxPoint( aResource.position[ 0 ], aResource.position[ 1 ] ), 
            wx.wxSize( aResource.size[ 0 ], aResource.size[ 1 ] ),
            style = wx.wxNO_FULL_REPAINT_ON_RESIZE | wx.wxCLIP_SIBLINGS,
            name = aResource.name                       
        )
        wx.EVT_PAINT(self, self.OnPaint)
        wx.EVT_SIZE(self, self.OnSize)
        wx.EVT_WINDOW_DESTROY(self, self._OnDestroy)

        # then call Widget._postInit which were the initialization
        # steps done after the _delegate was created
        widget.Widget._postInit(self, aParent, aResource)

        self._size = self.GetClientSize()
        bitmap = wx.wxEmptyBitmap(self._size[0],self._size[1])
        self._bufImage = wx.wxMemoryDC()
        self._bufImage.SelectObject(bitmap)
        del bitmap
        self.backgroundColor = (255, 255, 255)  # 'white'
        self.clear()
        
        self._setForegroundColor((0,0,0))       # 'black'


    def _OnDestroy(self, event):
        # memory leak cleanup
        self._bufImage = None
        self._penColor = None
        self._pen = None

    def _bindEvents(self):
        adapter = binding.wxPython_EventBinding(self)
        adapter.bindEvents()

    def setCopyMode(self, copyMode):
        self._bufImage.SetLogicalFunction(LogicalCopyModes[copyMode.upper()])

    def _getBackgroundColor( self ) :
        if self._bufImage is not None:
            brush = self._bufImage.GetBackground()
            return brush.GetColour()

    def _setBackgroundColor( self, aColor ) :
        aColor = self._getDefaultColor( aColor )
        if self._bufImage is not None:
            brush = self._bufImage.GetBackground()
            brush.SetColour(aColor)
            self._bufImage.SetBackground(brush)

    def setFillColor(self, aColor):
        aColor = self._getDefaultColor( aColor )
        if self._bufImage is not None:
            # KEA 2004-03-01
            # updated to work with wxPython 2.4.x
            # and wxPython 2.5.x
            # need to double-check other places copies of pen and brush
            # are manipulated in the rest of the framework and samples!!!
            nb = wx.wxBrush(aColor)
            ob = self._bufImage.GetBrush()
            if ob.Ok():
                nb.SetStyle(ob.GetStyle())
                s = ob.GetStipple()
                if s.Ok():
                    nb.SetStipple(s)
            self._bufImage.SetBrush(nb)

    def setFillMode(self, fillStyle):
        brush = self._bufImage.GetBrush()
        brush.SetStyle(BrushFillStyleList[fillStyle])
        self._bufImage.SetBrush(brush)

    def _getForegroundColor( self ) :
        return self._penColor

    def _setForegroundColor( self, aColor ) :
        aColor = self._getDefaultColor( aColor )
        self._penColor = aColor
        self._pen = wx.wxPen(self._penColor, self._thickness, wx.wxSOLID)
        #self._pen.SetCap(wx.wxCAP_PROJECTING)
        if self._bufImage is not None:
            self._bufImage.SetPen(self._pen)
            self._bufImage.SetTextForeground(self._penColor)

    def _getThickness(self):
        return self._thickness

    def _setThickness(self, thickness):
        self._thickness = thickness
        self._pen = wx.wxPen(self._penColor, self._thickness, wx.wxSOLID)
        if self._bufImage is not None:
            self._bufImage.SetPen(self._pen)

    def _getFont(self):
        if self._font is None:
            desc = font.fontDescription(self.GetFont())
            self._font = font.Font(desc)
        return self._font

    def _setFont(self, aFont):
        if isinstance(aFont, dict):
            aFont = font.Font(aFont, aParent=self)
        else: # Bind the font to this widget.
            aFont._parent = self
        self._font = aFont
        aWxFont = aFont._getFont()
        self._bufImage.SetFont(aWxFont)


    # when we Blit to the "window", we're actually blitting to the
    # offscreen bitmap and then that is blitted onscreen in the same
    # operation
    def Blit(self, xdest, ydest, width, height, source, xsrc, ysrc,
             logicalFunc=wx.wxCOPY, useMask=wx.FALSE): #, xsrcMask=-1, ysrcMask=-1):
        self._bufImage.Blit(xdest, ydest, width, height, source, xsrc, ysrc,
                            logicalFunc, useMask) #, xsrcMask, ysrcMask)
        if self.autoRefresh:
            dc = wx.wxClientDC(self)
            dc.Blit(0, 0, self._size[0],self._size[1], self._bufImage, 0, 0)

    def Clear(self):
        self._bufImage.Clear()
        if self.autoRefresh:
            dc = wx.wxClientDC(self)
            dc.Blit(0, 0, self._size[0],self._size[1], self._bufImage, 0, 0)

    # this is poorly named, it should be DrawAxis
    def CrossHair(self, x, y):
        self._bufImage.CrossHair(x, y)
        if self.autoRefresh:
            dc = wx.wxClientDC(self)
            dc.Blit(0, 0, self._size[0],self._size[1], self._bufImage, 0, 0)

    def DrawArc(self, x1, y1, x2, y2, xc, yc):
        """
        Draws an arc of a circle, centred on (xc, yc), with starting
        point (x1, y1) and ending at (x2, y2). The current pen is used
        for the outline and the current brush for filling the shape.

        The arc is drawn in an anticlockwise direction from the start
        point to the end point.
        """

        self._bufImage.DrawArc(x1, y1, x2, y2, xc, yc)
        if self.autoRefresh:
            dc = wx.wxClientDC(self)
            dc.Blit(0, 0, self._size[0],self._size[1], self._bufImage, 0, 0)

    # doesn't exist in wxMemoryDC
    #def DrawCheckMark(self, x, y, width, height):
    #    self._bufImage.DrawCheckMark(x, y, width, height)
    #    if self.autoRefresh: self.refresh()

    def DrawBitmap(self, aBitmap, x=0, y=0, transparency=1):
        if isinstance(aBitmap, graphic.Bitmap):
            bmp = aBitmap.getBits()
        elif isinstance(aBitmap, (wx.wxBitmap, wx.wxBitmapPtr)):
            bmp = aBitmap            
        elif isinstance(aBitmap, (wx.wxImage, wx.wxImagePtr)):
            bmp = wx.wxBitmapFromImage(aBitmap)
        elif PIL_FOUND and isinstance(aBitmap, Image.Image):
            bmp = wx.wxBitmapFromImage(graphic.PILToImage(aBitmap))
        elif NUMERIC_FOUND and isinstance(aBitmap, ArrayType):
            bmp = wx.wxBitmapFromImage(graphic.numericArrayToImage(aBitmap))
        else:
            return

        self._bufImage.DrawBitmap(bmp, x, y, transparency)
        if self.autoRefresh:
            dc = wx.wxClientDC(self)
            dc.Blit(0, 0, self._size[0],self._size[1], self._bufImage, 0, 0)

    def drawBitmapScaled(self, aBitmap, x, y, size, transparency=1):
        if isinstance(aBitmap, graphic.Bitmap):
            img = wx.wxImageFromBitmap(aBitmap.getBits())
        elif isinstance(aBitmap, (wx.wxBitmap, wx.wxBitmapPtr)):
            img = wx.wxImageFromBitmap(aBitmap)
        elif isinstance(aBitmap, (wx.wxImage, wx.wxImagePtr)):
            img = aBitmap
        elif PIL_FOUND and isinstance(aBitmap, Image.Image):
            img = graphic.PILToImage(aBitmap)
        elif NUMERIC_FOUND and isinstance(aBitmap, ArrayType):
            img = graphic.numericArrayToImage(aBitmap)
        else:
            return

        bmp = wx.wxBitmapFromImage(img.Scale(size[0], size[1]))
        self._bufImage.DrawBitmap(bmp, x, y, transparency)
        if self.autoRefresh:
            dc = wx.wxClientDC(self)
            dc.Blit(0, 0, self._size[0],self._size[1], self._bufImage, 0, 0)

    def DrawCircle(self, x, y, radius):
        self._bufImage.DrawCircle(x, y, radius)
        if self.autoRefresh:
            dc = wx.wxClientDC(self)
            dc.Blit(0, 0, self._size[0],self._size[1], self._bufImage, 0, 0)

    def DrawEllipse(self, x, y, width, height):
        self._bufImage.DrawEllipse(x, y, width, height)
        if self.autoRefresh:
            dc = wx.wxClientDC(self)
            dc.Blit(0, 0, self._size[0],self._size[1], self._bufImage, 0, 0)

    def DrawEllipticArc(self, x, y, width, height, start, end):
        self._bufImage.DrawEllipticArc(x, y, width, height, start, end)
        if self.autoRefresh:
            dc = wx.wxClientDC(self)
            dc.Blit(0, 0, self._size[0],self._size[1], self._bufImage, 0, 0)

    def DrawIcon(self, aIcon, x, y):
        self._bufImage.DrawIcon(aIcon, x, y)
        if self.autoRefresh:
            dc = wx.wxClientDC(self)
            dc.Blit(0, 0, self._size[0],self._size[1], self._bufImage, 0, 0)

    def DrawLine(self, x1, y1, x2, y2):
        self._bufImage.DrawLine(x1, y1, x2, y2)
        if self.autoRefresh:
            dc = wx.wxClientDC(self)
            dc.Blit(0, 0, self._size[0],self._size[1], self._bufImage, 0, 0)

    def DrawLines(self, pointsList):
        self._bufImage.DrawLines(pointsList)
        if self.autoRefresh:
            dc = wx.wxClientDC(self)
            dc.Blit(0, 0, self._size[0],self._size[1], self._bufImage, 0, 0)

    def DrawPoint(self, x, y):
        self._bufImage.DrawPoint(x, y)
        if self.autoRefresh:
            dc = wx.wxClientDC(self)
            dc.Blit(0, 0, self._size[0],self._size[1], self._bufImage, 0, 0)

    def DrawPolygon(self, pointsList):
        self._bufImage.DrawPolygon(pointsList)
        if self.autoRefresh:
            dc = wx.wxClientDC(self)
            dc.Blit(0, 0, self._size[0],self._size[1], self._bufImage, 0, 0)

    def DrawRectangle(self, x, y, width, height):
        self._bufImage.DrawRectangle(x, y, width, height)
        if self.autoRefresh:
            dc = wx.wxClientDC(self)
            dc.Blit(0, 0, self._size[0],self._size[1], self._bufImage, 0, 0)

    def DrawPointList(self, points, pens=None):
        self._bufImage.DrawPointList(points, pens)
        if self.autoRefresh:
            dc = wx.wxClientDC(self)
            dc.Blit(0, 0, self._size[0],self._size[1], self._bufImage, 0, 0)

    # KEA 2003-03-14
    # need to add other DrawXXXList methods once we are
    # requiring wxPython 2.4.0.6 or higher
    def DrawRectangleList(self, rects, pens=None, brushes=None):
        self._bufImage.DrawRectangleList(rects, pens, brushes)
        if self.autoRefresh:
            dc = wx.wxClientDC(self)
            dc.Blit(0, 0, self._size[0],self._size[1], self._bufImage, 0, 0)

    def DrawRotatedText(self, aString, x, y, angle):
        self._bufImage.DrawRotatedText(aString, x, y, angle)
        if self.autoRefresh:
            dc = wx.wxClientDC(self)
            dc.Blit(0, 0, self._size[0],self._size[1], self._bufImage, 0, 0)

    def DrawRoundedRectangle(self, x, y, width, height, radius):
        self._bufImage.DrawRoundedRectangle(x, y, width, height, radius)
        if self.autoRefresh:
            dc = wx.wxClientDC(self)
            dc.Blit(0, 0, self._size[0],self._size[1], self._bufImage, 0, 0)

    def DrawSpline(self, pointsList):
        self._bufImage.DrawSpline(pointsList)
        if self.autoRefresh:
            dc = wx.wxClientDC(self)
            dc.Blit(0, 0, self._size[0],self._size[1], self._bufImage, 0, 0)

    def DrawLineList(self, lines, pens=None):
        self._bufImage.DrawLineList(lines, pens)
        if self.autoRefresh:
            dc = wx.wxClientDC(self)
            dc.Blit(0, 0, self._size[0],self._size[1], self._bufImage, 0, 0)

    def DrawText(self, aString, x, y):
        self._bufImage.DrawText(aString, x, y)
        if self.autoRefresh:
            dc = wx.wxClientDC(self)
            dc.Blit(0, 0, self._size[0],self._size[1], self._bufImage, 0, 0)

    # KEA 2002-03-31 this doesn't seem to work
    def FloodFill(self, x, y, colour, style=wx.wxFLOOD_SURFACE):
        self._bufImage.FloodFill(x, y, colour, style)
        if self.autoRefresh:
            dc = wx.wxClientDC(self)
            dc.Blit(0, 0, self._size[0],self._size[1], self._bufImage, 0, 0)

    def GetPixel(self, x, y):
        return self._bufImage.GetPixel(x, y)

    # these are aliases for the old BitmapCanvas methods that
    # started with a lowercase letter
    clear = Clear
    drawBitmap = DrawBitmap
    drawLineList = DrawLineList
    drawPointList = DrawPointList
    drawRectangleList = DrawRectangleList
    drawRect = drawRectangle = DrawRectangle
    drawText = DrawText
    drawRotatedText = DrawRotatedText
    drawEllipse = DrawEllipse
    line = drawLine = DrawLine
    plot = drawPoint = DrawPoint

    def refresh(self):
        dc = wx.wxClientDC(self)
        dc.Blit(0, 0, self._size[0],self._size[1], self._bufImage, 0, 0)
        #self._delegate.Refresh()


    def OnPaint(self, evt):
        #starttime = time.time()
        rect = self.GetUpdateRegion().GetBox()
        #print "OnPaint", rect
        dc = wx.wxPaintDC(self)
        #dc.Blit(0, 0, self._size[0],self._size[1], self._bufImage, 0, 0)
        dc.Blit(rect[0], rect[1], rect[2], rect[3], self._bufImage, rect[0], rect[1])
        #stoptime = time.time()
        #elapsed = stoptime - starttime
        #print "OnPaint time: %f seconds" % (elapsed)

    def resizeBuffer(self, size):
        brush = self._bufImage.GetBackground()
        bitmap = wx.wxEmptyBitmap(size[0], size[1])
        _bufImage = wx.wxMemoryDC()
        _bufImage.SelectObject(bitmap)
        del bitmap
        _bufImage.SetBackground(brush)
        _bufImage.SetPen(self._pen)
        aWxFont = self._getFont()._getFont()
        _bufImage.SetFont(aWxFont)
        _bufImage.SetTextForeground(self._penColor)
        _bufImage.Clear()
        _bufImage.Blit(0, 0, self._size[0],self._size[1], self._bufImage, 0, 0)
        
        self._size = size

        #self._bufImage.SelectObject(wx.wxNullBitmap)
        self._bufImage = _bufImage

    # need to copy the old bitmap
    # before resizing
    # plus the various pen attributes
    # there is probably a memory leak in the code below
    def OnSize(self, evt):
        size = self.GetClientSizeTuple()
        #print "OnSize", size, self._size
        if size == (0, 0) or not self._pen.GetColour().Ok():
            evt.Skip()
            return

        try:
            # in case self._bufImage hasn't been initialized
            brush = self._bufImage.GetBackground()
        except:
            evt.Skip()
            return

        # KEA 2002-03-30
        # only resize the offscreen bitmap when the onscreen window gets
        # larger than the offscreen bitmap
        #minX = min(size[0], self._size[0])
        #minY = min(size[1], self._size[1])
        maxX = max(size[0], self._size[0])
        maxY = max(size[1], self._size[1])
        #print minX, minY, maxX, maxY
        if (self._size[0] < maxX) or (self._size[1] < maxY):
            self.resizeBuffer((maxX, maxY))
            evt.Skip()

    def getBitmap(self):
        size = self.GetClientSizeTuple() 
        memory = wx.wxMemoryDC()
        bitmap = wx.wxEmptyBitmap(size[0], size[1])
        memory.SelectObject(bitmap)
        memory.Blit(0, 0, size[0], size[1], self._bufImage, 0, 0)
        memory.SelectObject(wx.wxNullBitmap)
        return bitmap


registry.getRegistry().register( sys.modules[__name__].BitmapCanvas )

