"""
__version__ = "$Revision: 1.36 $"
__date__ = "$Date: 2004/03/17 23:25:39 $"
"""

import error
import log
import sys, types

from wxPython import wx

class Handler:
    """
    Maps a function name to a function, parsing the
    PythonCard-specific name into meaningful parts.
    """

    def __init__(self, aFunctionObject):

        # Hold on to the full name.
        self._name = aFunctionObject.__name__

        # Parse the function/method name.
        parts = self._name.split('_')

        # The 'on' prefix, i.e. on_button1_click  
        self._prefix = parts[ 0 ] 

        if len(parts) == 3:
            # The name of the Widget instance that is the target.
            self._sourceName = parts[ 1 ]

            # The Event class identifier.
            self._eventName = parts[ 2 ]
        else:
            # KEA 2001-09-06
            # need to pass in the name of the Background subclass instance ?
            self._sourceName = ''
            self._eventName = parts[ 1 ]

        # Grab the function/method object.
        self._function = aFunctionObject
        
    def getName( self ) :
        return self._name 
        
    def getSourceName( self ) :
        return self._sourceName

    def getEventName( self ) :
        return self._eventName

    def getFunction( self ) :
        return self._function

    def __repr__( self ) :
        return str( self.__dict__ )

class ObjectLookup :
    """
    ObjectLookup - ABSTRACT CLASS

    Having a single event queue object centralizes the posting
    of events and makes it easier to abstract away wxWindows

    This would be even better if it followed the Singleton pattern
    where there could only be ONE EventQueue object in the system.

    Adding the Observer/Observable pattern makes it easier to write
    an event watcher that is decoupled from the EventQueue.

    See monitor.py and monitor_test.py

    Provides an interface for looking up objects by path.
    """

    # Brain dead way to implement the Singleton pattern
    # in Python.  This class CANNOT be subclassed, but then
    # again, we may not ever need to. If we do, we'll search
    # on Google for a better way to implement Singleton :).

    def __call__( self ) :
        return self

    def findByName( self, targetPath ) :
        # Must be overridden
        raise error.AbstractMethodException

    def findId( self, targetPath ) :
        # Must be overridden
        raise error.AbstractMethodException

class ChangeEvent :
    """
    A ChangeEvent indicates that some attribute of
    and object has changed.  The event carries a
    reference to the object, and the attribute
    value that changed.
    """

    def __init__( self, aObject, aOldValue ) :
        """
        Initialize this instance.
        """
        self._object = aObject
        self._oldValue = aOldValue

    def getChangedObject( self ) :
        """
        Get a reference to the object that was modified.
        """
        return self._object 

    def getOldValue( self ) :
        """
        Get the old attribute value that changed.
        NOTE: We'll probably need to provide a
        getAttributeName() method to identify the
        actual value that changed within the object.
        """
        return self._oldValue

class ChangeListener :
    """
    Defines an interface that must be implemented
    by classes that want to listen for ChangeEvents.
    """

    def changed( aChangeEvent ) :
        """
        Called by a Changeable object when it's
        contents are modified.
        """
        raise error.AbstractMethodException

class Changeable :
    """
    Defines an interface that must be implemented by
    classes that want to generate a notification when
    their contents change.
    """

    def __init__( self ) :
        """
        Initialize this instance.
        """
        self._listeners = []

    def addChangeEventListener( self, aChangeListener ) :
        """
        Add a ChangeListener to this Changeable object's
        list of interested listeners.
        """
        self._listeners.append( aChangeListener )

    def fireChanged( self, oldValue ) :
        """
        Broadcast a ChangeEvent to all registered ChangeListeners.
        """
        evt = ChangeEvent( self, oldValue )
        for listener in self._listeners :
            listener.changed( evt )

class Event :
    """
    Superclass of all event classes.
    """

    name = 'event'

    def __init__( self, aSource ) :
        self._source = aSource
        self._used = 0
        # KEA 2002-03-07
        # change to (self, event) event handler dispatch
        # add target manually since this is a custom event
        # and doesn't go through normal dispatch
        self.target = self._source
        self.eventObject = self.target

    def getUsed( self ) :
        return self._used

    def setUsed( self, used ) :
        self._used = used

    def getSource( self ) :
        """
        Get a reference to the object that
        generated this Event.
        """
        return self._source

    def getName( self ) :
        global eventMap
        return eventMap.classToName( self.__class__ )

    def sendTo( self, aOwner, aHandler ) :
        """
        Tell an Event to dispatch itself to aOwner by
        calling aHandler.getFunction().
        """
        aHandler.getFunction()( aOwner, self.getSource(), self )

    def accept( self, aVisitor, aParam ) :
        """
        A generic implementation of the 'visitee' part
        of the Visitor pattern.
        """
        getattr( aVisitor, 'visit' + self.__class__.__name__ )( aParam )

    def getHandlerName( self ) :
        return 'on_' + self._source._getName() + '_' + self.getName()

# NOTE: The following two methods are only provided to help
# determine what event info should be transferred from wxEvents 
# to PythonCard Events.  They will go away some time soon.

    def setNativeEvent( self, aNativeEvent ) :
        self._nativeEvent = aNativeEvent

    def getNativeEvent( self ) :
        return self._nativeEvent

    def skip( self ) :
        self.getNativeEvent().Skip()


        #RDS-NEW-EVENTS:

#myEVT_OPEN_BACKGROUND = wxNewEventType()

#def EVT_OPEN_BACKGROUND(win, id, func):
#    win.Connect(id, -1, myEVT_OPEN_BACKGROUND, func)

#class MyCustomEvent(wxPyCommandEvent):
#    def __init__(self, evtType, id):
#        wxPyCommandEvent.__init__(self, evtType, id)
#        self.myVal = None
#
#    #def __del__(self):
#    #    print '__del__'
#    #    wxPyCommandEvent.__del__(self)
#
#    def SetMyVal(self, val):
#        self.myVal = val
#
#    def GetMyVal(self):
#        return self.myVal

class OpenBackgroundEvent( Event ) :

    name = 'openBackground'

    def getHandlerName( self ) :
        return 'on_openBackground'


class WidgetEvent( Event ) :
    """
    A widget specific Event subclass.
    """

    name = 'widgetEvent'

    def __init__( self, aWidgetName, aSource ) :
        Event.__init__( self, aSource )
        self._widgetName = widgetName
    
    def getWidgetName( self ) :
        return self._widgetName
    
class MessageEvent( Event ) :
    """
    A generic messaging event that accepts a target path
    and the method to call on the object identified by
    the target path.
    """

    name = 'messageEvent'

    def __init__( self, aTargetPath, aMessage, aParam, aSource ) :
        Event.__init__( self, aSource )
        self._targetPath = aTargetPath
        self._message = aMessage
        self._param = aParam

    def getTargetPath( self ) :
        return self._targetPath.split( '.' )

    def getMessage( self ) :
        return self._message

    def sendTo( self, aOwner, aHandler ) :
        """
        Tell an Event to dispatch itself to aOwner by
        calling aHandler.getFunction().
        """
        name = self.getTargetPath()[ 0 ]
        target = aOwner.name

        getattr( target, self.getMessage() )()


class EventSource :
    """
    All Components are EventSources.
    """

    def __init__( self ) :
        self._listeners = []
        
    def _addEventListener(self, listener):
        """
        Add an EventListener as an observer of this object.
        """
        if listener.__class__.__name__ == 'MessageWatcher':
            self._listeners.insert(0, listener)
        else: 
            self._listeners.append(listener)
    
    def _notifyEventListeners( self, event ) :
        """
        Notify all of our EventListeners that an event has occured.
        """
        for listener in self._listeners :
            if listener.__class__.__name__ == 'ObjectMap':
                #print "listener", listener
                # KEA 2002-06-10
                # I have no idea where ObjectMap is getting added
                # to the listeners, but I think it is a bug
                pass
            else:
                listener.eventOccurred( event )

class EventListener :
    """
    ABSTRACT INTERFACE

    Define an interface that clients can implement in order
    to act as observers of the EventQueue.
    """

    # I don't know if it's possible to make a method
    # 'abstract' in python, i.e. specify that it MUST
    # be overridden?

    def eventOccurred( self, event ) :
        raise error.AbstractMethodException
               
class MenuEvent( Event ) :
    """
    The Event sent when an unbound Menu item is clicked.
    """

    name = 'menu'

    def __init__( self, aMenuItemId, aMenu ) :
        Event.__init__( self, aMenu )
        self._itemId = aMenuItemId

    def getMenuItemId( self ) :
        return self._itemId

class CommandEvent( Event ) :
    """
    The Event sent when by a command source object
    like Button, ImageButton, MenuItem, etc.
    """

    name = 'command'

    def __init__( self, aName, aSource ) :
        Event.__init__( self, aSource )
        self._name = aName

    def getCommand( self  ) :
        return self._name
 
    def getHandlerName( self ) :
        return 'on_' + self.getCommand() + '_' + self.getName()

class GainFocusEvent( Event ) :
    name = 'gainFocus'

class LoseFocusEvent( Event ) :
    name = 'loseFocus'

class KeyPressEvent( Event ) :    
    name = 'keyPress'

class KeyDownEvent( Event ) :
    name = 'keyDown'

class KeyUpEvent( Event ) :
    name = 'keyUp'

class TextUpdateEvent( Event ) :
    name = 'textUpdate'

class TextEnterEvent( Event ) :
    name = 'textEnter'

"""
class SpinDownEvent(Event):
    name = 'spinDown'

class SpinUpEvent(Event):
    name = 'spinUp'
"""

class SelectEvent( Event ) :
    name = 'select'

class MoveEvent(Event):
    name = 'move'

class SizeEvent(Event):
    name = 'size'

class CloseEvent(Event):
    name = 'close'

class MinimizeEvent(Event):
    name = 'minimize'

class MaximizeEvent(Event):
    name = 'maximize'

class ActivateEvent(Event):
    name = 'activate'

class IdleEvent(Event):
    name = 'idle'

class TimerEvent(Event):
    name = 'timer'

class MouseClickEvent( Event ) :
    name = 'mouseClick'

class MouseDoubleClickEvent( Event ) :
    name = 'mouseDoubleClick'

class MouseDownEvent( Event ) :
    name = 'mouseDown'

class MouseUpEvent( Event ) :
    name = 'mouseUp'

class MouseMiddleDownEvent( Event ) :
    name = 'mouseMiddleDown'

class MouseMiddleUpEvent( Event ) :
    name = 'mouseMiddleUp'

class MouseMiddleDoubleClickEvent( Event ) :
    name = 'mouseMiddleDoubleClick'

class MouseContextDownEvent( Event ) :
    name = 'mouseContextDown'

class MouseContextUpEvent( Event ) :
    name = 'mouseContextUp'

class MouseContextClickEvent( Event ) :
    name = 'mouseContextClick'

class MouseContextDoubleClickEvent( Event ) :
    name = 'mouseContextDoubleClick'

class MouseEnterEvent( Event ) :
    name = 'mouseEnter'

class MouseLeaveEvent( Event ) :
    name = 'mouseLeave'

class MouseMoveEvent( Event ) :
    name = 'mouseMove'

class MouseDragEvent( Event ) :
    name = 'mouseDrag'

class CalendarMouseDoubleClickEvent(Event):
    name = 'calendarMouseDoubleClick'

class CalendarChangedEvent(Event):
    name = 'calendarChanged'

class CalendarDayEvent(Event):
    name = 'calendarDay'

class CalendarMonthEvent(Event):
    name = 'calendarMonth'

class CalendarYearEvent(Event):
    name = 'calendarYear'

class CalendarWeekDayHeaderEvent(Event):
    name = 'calendarWeekDayHeader'

class ItemActivatedEvent(Event):
    name = 'itemActivated'

class ItemFocusedEvent(Event):
    name = 'itemFocused'

class ItemExpandedEvent(Event):
    name = 'itemExpanded'

class ItemExpandingEvent(Event):
    name = 'itemExpanding'

class SelectionChangedEvent(Event):
    name = 'selectionChanged'

class SelectionChangingEvent(Event):
    name = 'selectionChanging'

class ListColumnClickEvent(Event):
    name = 'columnClick'

class IEHtmlTitleChangeEvent(Event):
    name = 'titleChange'

class IEHtmlStatusTextChangeEvent(Event):
    name = 'statusTextChange'

class IEHtmlDocumentCompleteEvent(Event):
    name = 'documentComplete'


MOUSE_EVENTS = [
    TimerEvent,
    GainFocusEvent,
    LoseFocusEvent,
    MouseDoubleClickEvent,
    MouseDownEvent,
    MouseUpEvent,
    MouseMiddleDownEvent,
    MouseMiddleUpEvent,
    MouseMiddleDoubleClickEvent,
    MouseContextDownEvent,
    MouseContextUpEvent,
    MouseContextDoubleClickEvent,
    MouseEnterEvent,
    MouseLeaveEvent,
    MouseMoveEvent 
]

# System Events

# KEA 2002-04-04
# custom event posted after loseFocus when a field
# has been changed
wxEVT_CLOSE_FIELD = wx.wxNewEventType()

def EVT_CLOSE_FIELD(win, id, func):
    win.Connect(id, -1, wxEVT_CLOSE_FIELD, func)
        
class CloseFieldEvent(Event):
    name = 'closeField'


class EventMap :
    """
    Maps Event classes to handler names for use in user code, and vice-versa.
    """
    def __init__( self ) :
        
        self._table = [
            GainFocusEvent,
            LoseFocusEvent,
            MouseClickEvent,
            MouseDownEvent,
            MouseUpEvent,
            MouseDoubleClickEvent,
            MouseContextDownEvent,
            MouseContextUpEvent,
            MouseContextDoubleClickEvent,
            MouseMiddleDownEvent,
            MouseMiddleUpEvent,
            MouseMiddleDoubleClickEvent,
            MouseEnterEvent,
            MouseLeaveEvent,
            MouseMoveEvent,
            MouseDragEvent,
            CommandEvent,
            KeyPressEvent,
            KeyDownEvent,
            KeyUpEvent,
            TextUpdateEvent,
            TextEnterEvent,
            #SpinUpEvent,
            #SpinDownEvent,
            SelectEvent,
            MoveEvent,
            SizeEvent,
            CloseEvent,
            ActivateEvent,
            MinimizeEvent,
            MaximizeEvent,
            IdleEvent, 
            TimerEvent,
                   
            #System Events
            CloseFieldEvent,

            #RDS-NEW-EVENTS:
            OpenBackgroundEvent,

            # Calendar events
            CalendarMouseDoubleClickEvent,
            CalendarChangedEvent,
            CalendarDayEvent,
            CalendarMonthEvent,
            CalendarYearEvent,
            CalendarWeekDayHeaderEvent,

            # Tree and MultiColumnList events
            MouseContextClickEvent,
            ItemFocusedEvent,
            ItemActivatedEvent,
            ItemExpandedEvent,
            ItemExpandingEvent,
            SelectionChangedEvent,
            SelectionChangingEvent,
            ListColumnClickEvent,

            IEHtmlTitleChangeEvent,
            IEHtmlStatusTextChangeEvent,
            IEHtmlDocumentCompleteEvent,
        ]
        
        self._classToNameMap = {}
    
        self._nameToClassMap = {}
    
        # RDS: replace static list above with the following
        # code once i figure out how to get findEventClasses to
        # work.

        #self._table = []

        #self._findEventClasses()

        self._createClassToNameMap()
    
        self._createNameToClassMap()
    
    def _findEventClasses( self ) :
        """
        Find all of the subclasses of Event in 
        this module and put them in an array.
        """

        for item in __dict__.itervalues():
            if isinstance(item, types.ClassType):
                if issubclass(item, Event):
                    self._table.append(item)
            
    def _createClassToNameMap( self ) :
        for clazz in self._table :
            self._classToNameMap[ clazz ] = clazz.name
    
    def _createNameToClassMap( self ) :
        for clazz in self._table :
            self._nameToClassMap[ clazz.name ] = clazz
    
    def classToName( self, aEventClass ) :
        return self._classToNameMap[ aEventClass ]
    
    def nameToClass( self, aName ) :
        return self._nameToClassMap[ aName ]
    
    def __repr__( self ) :
        return str( self._classToNameMap ) + '\n\n' + str( self._nameToClassMap )
    
queue = None

class EventQueue :
    """
    Manages the dispatching of Events.

    The EventQueue class delegates to a single instance
    of EventQueueImpl - yansi ( yet another singleton implementation ;)
    """

    _impl = None

    class EventQueueImpl( EventSource, EventListener ) :

        def __init__( self ) :
            EventSource.__init__( self )

        def eventOccurred( self, aEvent ) :
            self._notifyEventListeners( aEvent )

    def __init__( self ) :
        if EventQueue._impl is None :
            EventQueue._impl = self.EventQueueImpl()

    def listenTo( self, aEventSource ) :
        aEventSource._addEventListener(self._impl)

    def addListener( self, aEventListener ) :
        self._impl._addEventListener(aEventListener)
    
# Unit Test

class Dummy :

    def visitIdleEvent( self, aParam ) :
        log.debug( 'visitIdleEvent(): ', aParam )

# Unit Test

if __name__ == '__main__' :

    print 'Unit Test - event'

    dummy = Dummy()

    evt = IdleEvent( dummy )

    print evt

    evt.accept( dummy, 'this is a parameter' )


# Module Initialization

# RDS: A nasty global - I'm a BAD MONKEY! 

eventMap = EventMap()
