# -*- coding: ISO-8859-1 -*-

# Copyright (C) 2002, 2003 Jrg Lehmann <joerg@luga.de>
#
# Ogg Vorbis interface by Byron Ellacott <bje@apnic.net>.
#
# This file is part of PyTone (http://www.luga.de/pytone/)
#
# PyTone is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2
# as published by the Free Software Foundation.
#
# PyTone is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with PyTone; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import os.path, locale, re, string, sys, time
import log

try:
    import eyeD3
    eyeD3present = 1
    log.info("using eyeD3 module for id3 tag parsing")
except:
    log.info("using integrated MP3Info module for id3 tag parsing")
    eyeD3present = 0
    
import MP3Info

fallbacklocalecharset = 'iso-8859-1'

# Try to determine "correct" character set for the reencoding of the
# unicode strings contained in Ogg Vorbis files 
try:
    # works only in python > 2.3
    localecharset = locale.getpreferredencoding()
except:
    try:
        localecharset = locale.getdefaultlocale()[1]
    except:
        try:
            localecharset = sys.getdefaultencoding()
        except:
            localecharset = fallbacklocalecharset
if localecharset in [None, 'ANSI_X3.4-1968']:
    localecharset = fallbacklocalecharset

try:
    import ogg.vorbis
    oggsupport = 1
except:
    oggsupport = 0
    log.info("Ogg Vorbis support disabled, since module is not present")
    pass

tracknrandtitlere = re.compile("^\[?(\d+)\]? ?[- ] ?(.*)\.(mp3|ogg)$")
UNKNOWN = "Unknown"


class dbitem:

    """ base class for various items stored in database:

    songs, albums, artists, genres, years, playlists"""

    def __cmp__(self, other):
        try:
            return cmp(self.__class__, other.__class__) or cmp(self.id, other.id)
        except:
            return 1

    def __repr__(self):
        return "%s(%s)" % (self.__class__, self.id)


class song(dbitem):

    def __init__(self, relpath, basedir, tracknrandtitlere, capitalize, stripleadingarticle, removeaccents):
        # use relative path of song as its id
        self.id = os.path.normpath(relpath)
        self.name = os.path.basename(self.id)

        # we set the path later in readid3info
        self.path = None 

        if self.id.lower().endswith(".mp3"):
            self.type = "mp3"
        elif self.id.lower().endswith(".ogg"):
            self.type = "ogg"
        else:
            raise RuntimeError("Fileformat of song '%s' not supported" % (self.id))

        # song metadata
        self.title = ""
        self.album = ""
        self.artist = ""
        self.year = ""
        self.genre = ""
        self.tracknr = ""
        self.length = 0
        
        # statistical information
        self.nrplayed = 0
        self.lastplayed = []
        self.added = time.time()
        self.rating = None
        # where does rating come from: 0=song itself, 1=album, 2=artist
        # This information is used when you rate an album or an artist to not
        # overwrite the rating already given to a specific song or all songs
        # on a given album of that artist, respectively.
        self.ratingsource = None

        self.scanfile(basedir, tracknrandtitlere, capitalize, stripleadingarticle, removeaccents)

    def __getattr__(self, name):
        if name=="albumid":
            return self.album
        elif name=="artistid":
            return artist
        elif name=="genreid":
            return genre
        else:
            raise AttributeError

    def readid3tag_eyeD3(self):
        mp3file = eyeD3.Mp3AudioFile(self.path)
        mp3info = mp3file.getTag()

        # we definitely want the length of the MP3 file, even if no ID3 tag is present,
        # so extract this info before anything goes wrong
        self.length = mp3file.getPlayTime()

        self.title = mp3info.getTitle()
        self.title = self.title.encode(localecharset, 'replace')
        self.title = MP3Info._strip_zero(self.title)

        self.album = mp3info.getAlbum()
        self.album = self.album.encode(localecharset, 'replace')
        self.album = MP3Info._strip_zero(self.album)

        self.artist = mp3info.getArtist()
        self.artist = self.artist.encode(localecharset, 'replace')
        self.artist = MP3Info._strip_zero(self.artist)

        self.year = mp3info.getYear()
        if self.year:
            self.year = self.year.encode(localecharset, 'replace')

        try:
            self.genre = mp3info.getGenre()
            if self.genre:
                self.genre = self.genre.getName()
        except eyeD3.tag.GenreException, e:
            self.genre = e.msg.split(':')[1].strip()

        self.tracknr = str(mp3info.getTrackNum()[0])

        # if the playtime is also in the ID3 tag information, we
        # try to read it from there
        if mp3info.frames["TLEN"]:
            length = None
            try:
                length = int(int(mp3info.frames["TLEN"])/1000)
            except:
                # time in seconds (?), possibly with bad decimal separator, e.g "186,333"
                try:
                    length = int(float(mp3info.frames["TLEN"].replace(",", ".")))
                except:
                    pass
            if length:
                self.length = length

    def readid3tag_MP3Info(self):
        mp3file = open(self.path, "rb")
        mp3info = MP3Info.MP3Info(mp3file)
        self.title = mp3info.title
        self.album = mp3info.album
        self.artist = mp3info.artist
        self.year = mp3info.year
        self.genre  = mp3info.genre
        self.tracknr = mp3info.track
        if mp3info.frames.tags.has_key("TLEN"):
            try:
                self.length = int(mp3info.id3.tags["TLEN"])/1000
            except:
                # time in seconds (?), possibly with bad decimal separator, e.g "186,333"
                t = mp3info.id3.tags["TLEN"].replace(",", ".")
                self.length = int(float(t))
        else:
            self.length = mp3file.getPlayTime()
        mp3file.close()

    def readid3tag_Vorbis(self):
        vf = ogg.vorbis.VorbisFile(self.path)
        id3get = vf.comment().as_dict().get
        self.title = id3get('TITLE', [""])[0]
        self.title = self.title.encode(localecharset, 'replace')
        self.album = id3get('ALBUM', [""])[0]
        self.album = self.album.encode(localecharset, 'replace')
        self.artist = id3get('ARTIST', [""])[0]
        self.artist = self.artist.encode(localecharset, 'replace')
        self.year = id3get('DATE', [""])[0]
        self.year = self.year.encode(localecharset, 'replace')
        self.genre  = id3get('GENRE', [""])[0]
        self.genre = self.genre.encode(localecharset, 'replace')
        self.tracknr = id3get('TRACKNUMBER', [""])[0]
        self.tracknr = self.tracknr.encode(localecharset, 'replace')
        self.length = vf.time_total(0)

    def scanfile(self, basedir, tracknrandtitlere, capitalize, stripleadingarticle, removeaccents):
        """ update path info for song and scan id3 information """
        self.path = os.path.normpath(os.path.join(basedir, self.id))
        if not os.access(self.path, os.R_OK):
            raise IOError("cannot read song")

        # guesses for title and tracknr using the filename
        match = re.match(tracknrandtitlere, self.name)
        if match:
            fntracknr = str(int(match.group(1)))
            fntitle = match.group(2)
        else:
            fntracknr = ""
            fntitle = self.name
            if fntitle.lower().endswith(".mp3") or fntitle.lower().endswith(".ogg"):
                fntitle = fntitle[:-4]

        first, second = os.path.split(os.path.dirname(self.id))
        if first and second and not os.path.split(first)[0]:
            fnartist = first
            fnalbum = second
        else:
            fnartist = fnalbum = ""

        fntitle = fntitle.replace("_", " ")
        fnalbum = fnalbum.replace("_", " ")
        fnartist = fnartist.replace("_", " ")
                
        try:
            if self.type=="mp3":
                if eyeD3present:
                    self.readid3tag_eyeD3()
                else:
                    self.readid3tag_MP3Info()
            else:
                self.readid3tag_Vorbis()
        except:
            pass

        # sanity check for tracknr
        try:
            self.tracknr= str(int(self.tracknr))
        except:
            # treat track number like "3/12"
            try:
                self.tracknr= str(int(self.tracknr[:self.tracknr.index('/')]))
            except:
                self.tracknr= ""

        # do some further treatment of the song info

        # use title from filename, if it is a longer version of
        # the id3 tag title
        if not self.title or fntitle.startswith(self.title):
            self.title = fntitle

        # also try to use tracknr from filename, if not present as id3 tag
        if not self.tracknr or self.tracknr=="0":
            self.tracknr = fntracknr

        # we don't want empty album names
        if not self.album:
            if fnalbum:
                self.album = fnalbum
            else:
                self.album = UNKNOWN

        # nor empty artist names
        if not self.artist:
            if fnartist:
                self.artist = fnartist
            else:
                self.artist = UNKNOWN

        # nor empty genres
        if not self.genre:
            self.genre = UNKNOWN

        if not self.year or self.year=="0":
            self.year = None
        else:
            try:
                self.year = int(self.year)
            except:
                self.year = None

        if capitalize:
            # normalize artist, album and title
            self.artist = string.capwords(self.artist)
            self.album = string.capwords(self.album)
            self.title = string.capwords(self.title)

        if stripleadingarticle:
            # strip leading "The " in artist names, often used inconsistently
            if self.artist.startswith("The ") and len(self.artist)>4:
                self.artist = self.artist[4:]

        if removeaccents:
            translationtable = string.maketrans('',
                                                'AAAAEEEEIIIIOOOOUUUUaaaaeeeeiiiioooouuuu')
            self.artist = string.translate(self.artist, translationtable)
            self.album = string.translate(self.album, translationtable)
            self.title = string.translate(self.title, translationtable)

    def play(self):
        self.nrplayed += 1
        self.lastplayed.append(time.time())
        # only store last 10 playing times
        self.lastplayed = self.lastplayed[-10:]

    def unplay(self):
        if self.nrplayed or 1:
            self.nrplayed -= 1
            self.lastplayed.pop()

    def update(self, newsong):
        """ update song metadata using the information in newsong"""
        self.title = newsong.title
        self.album = newsong.album
        self.artist = newsong.artist
        self.year = newsong.year
        self.genre = newsong.genre
        self.tracknr = newsong.tracknr
        self.length = newsong.length


class artist(dbitem):
    def __init__(self, name):
        self.id = name
        self.name = name
        self.albums = []
        self.songs = []


class album(dbitem):
    def __init__(self, name):
        self.id = name
        self.name = name
        self.artists = []
        self.songs = []


class playlist(dbitem):
    def __init__(self, path):
        assert os.path.isfile(path), "'%s' doesn't exist" % path
        self.path = self.id = os.path.normpath(path)
        self.name = os.path.basename(path)
        if self.name.endswith(".m3u"):
            self.name = self.name[:-4]
        self.songs = []

        file = open(self.path, "r")

        for line in file.xreadlines():
            # XXX: interpret extended m3u format (especially for streams)
            # see: http://forums.winamp.com/showthread.php?s=dbec47f3a05d10a3a77959f17926d39c&threadid=65772
            if not line.startswith("#") and not chr(0) in line:
                path = line.strip()
                if not path.startswith("/"):
                    path = os.path.join(self.path, path)
                if os.path.isfile(path):
                    self.songs.append(path)
        file.close()

class dbindex(dbitem):

    """ base class for indices (besides albums and artists) in database """
    
    def __init__(self, id):
        self.id = id
        self.artists = []
        self.albums = []
        self.songs = []

class genre(dbindex):
    def __init__(self, name):
        dbindex.__init__(self, name)
        self.name = name


class year(dbindex):
    def __init__(self, year):
        dbindex.__init__(self, str(year))
        self.year = year


class rating(dbindex):
    def __init__(self, rating):
        dbindex.__init__(self, str(rating))
        self.rating = rating
