# -*- coding: ISO-8859-1 -*-

# Copyright (C) 2002, 2003, 2004 Jrg Lehmann <joerg@luga.de>
#
# This file is part of PyTone (http://www.luga.de/pytone/)
#
# PyTone is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2
# as published by the Free Software Foundation.
#
# PyTone is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with PyTone; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


import os.path, math, random, string, time
import config, dbitem
import events, hub, requests
import helper

# conversion commodity function: songdbid + list of dbitem.song -> list of item.song

def _dbsongs_to_songs(songdbid, dbsongs):
    return map(lambda asong, songdbid=songdbid:
               song(songdbid, song=asong),
               dbsongs)

# 
# randomly select songs from a given list (adapted from services.songdbs.local.songdb)
#

# TODO: check if can use the version in services.songdbs.local.songdb.py now that
# the rating has been simplified

def _genrandomchoice(songs):
    h = 0
    result = []
    length = 0
    for i in range(len(songs)):
        # Simple heuristic algorithm to consider song ratings
        # for random selection. Certainly not optimal!
        while True:
            nr = random.randrange(0, len(songs))
            aitem = songs[nr]
            rating = aitem.rating or 3
            if aitem.lastplayed:
                last = max(0, int((time.time()-aitem.lastplayed[-1])/60))
                rating -= 2*math.exp(-last/(60.0*60*24))
            if random.random()>=2.0**(-rating) or len(songs)==1:
                break

        del songs[nr]
        length += aitem.length
        result.append(aitem)
        if length>config.general.randominsertlength:
            break
    return result


class item:
    """ base class for various items presentend in the database and
    playlist windows (as opposed to those stored in the database
    itself (cf. module dbitem)"""

    def __init__(self, songdbid):
        """ each item has to be bound to a specific database
        identified by songdbid """
        self.songdbid = songdbid

    def getname(self):
        """ short name used for item in lists """
        return ""


    def getinfo(self):
        """ 4x4 array containing rows and columns used for display of item
        in iteminfowin"""
        return [["", "", "", ""]]

    def getinfolong(self):
        """ nx4 array containing rows and columns used for display of item
        in iteminfowin2"""
        return self.getinfo()

class diritem(item):

    """ item containing other items """

    def getname(self):
        return "%s/" % self.name

    def cmpitems(self, x, y):
        """ compare the two items x, y contained in self """
        pass

    def getcontents(self):
        """ return items contained in self """
        pass

    def getcontentsrecursive(self):
        """ return items contained in self including subdirs (in arbitrary order)"""
        result = []
        for aitem in self.getcontents():
            if isinstance(aitem, diritem):
                result.extend(aitem.getcontentsrecursive())
            else:
                result.append(aitem)

        return result

    def getcontentsrecursivesorted(self):
        """ return items contained in self including subdirs (sorted)"""
        result = []
        for aitem in self.getcontents():
            if isinstance(aitem, diritem):
                result.extend(aitem.getcontentsrecursivesorted())
            else:
                result.append(aitem)

        return result

    def getcontentsrecursiverandom(self):
        """ return random list of items contained in self including subdirs """
        # this should be implemented by subclasses
        return []

    def getheader(self, item):
        """ return header (used for title bar in filelistwin) of item in self.

        Note that item can be None!
        """
        pass

    def isartistoralbum(self):
        """ does self represent an artist or an album? """
        return 0


class song(item):
    def __init__(self, songdbid, id=None, song=None):
        """ song either defined by id or as instance of dbitem.song

        In the former case, we lazily fetch dbitem.song when needed """

        self.songdbid = songdbid
        if id is None:
            assert isinstance(song, dbitem.song), "song has to be a dbitem.song instance, not a %s instance" % repr(song.__class__)
            self.id = song.id
            self.song = song
        elif song is None:
            self.id = id
            self.song = None
        else:
            raise ValueError

    def __repr__(self):
        return "song(%s) in %s database" % (self.id, self.songdbid)

    __str__ = __repr__

    def __getattr__(self, attr):
        # Python tries to call __setstate__ upon unpickling -- prevent this
        if attr=="__setstate__":
            raise AttributeError
        self._readsong()
        if attr != "song":
            return getattr(self.song, attr)
        else:
            return self.song

    def _readsong(self):
        if self.__dict__["song"] is None:
            self.song = hub.request(requests.getsong(self.songdbid, self.id))

    def _updatesong(self):
        """ notify database of song changes """
        self._readsong()
        hub.notify(events.updatesong(self.songdbid, self.song))

    def getname(self):
        self._readsong()
        return self.song.title

    def getinfo(self):
        self._readsong()
        l = [["", "", "", ""]]*4
        l[0] = [_("Title:"), self.song.title]
        if self.song.tracknr:
            l[0] += [_("Nr:"), str(self.song.tracknr)]
        else:
            l[0] += ["", ""]
        l[1] = [_("Album:"),  self.song.album]
        if self.song.year:
            l[1] += [_("Year:"), str(self.song.year)]
        else:
            l[1] += ["", ""]
        l[2] = [_("Artist:"), self.song.artist,
              _("Time:"), helper.formattime(self.song.length)]
        l[3] = [_("Genre:"), self.song.genre]
        if self.song.nrplayed>0:
            last = int((time.time()-self.song.lastplayed[-1])/60)
            days, rest = divmod(last, 24*60)
            hours, minutes = divmod(rest, 60)
            if days>=10:
                lastplayed = "%dd" % days
            elif days>0:
                lastplayed = "%dd %dh" % (days, hours)
            elif hours>0:
                lastplayed = "%dh %dm" % (hours, minutes)
            else:
                lastplayed = "%dm" % minutes
            if self.song.rating:
                lastplayed = lastplayed + " (%s)" % ("*"*self.song.rating)
            l[3] += [_("Played:"),
                   _("#%d, %s ago") % (self.song.nrplayed, lastplayed)]

        else:
            if self.song.rating:
                l[3] += [_("Rating:"), "*"*self.song.rating]
            else:
                l[3] += ["", ""]
        return l

    def getinfolong(self):
        self._readsong()
        l = []
        l.append([_("Title:"), self.song.title, "", ""])
        l.append([_("Album:"),  self.song.album, "", ""])
        l.append([_("Artist:"), self.song.artist, "", ""])
        if self.song.year:
            l.append([_("Year:"), str(self.song.year), "", ""])
        else:
            l.append([_("Year:"), "", "", ""])

        if self.song.tracknr:
            l.append([_("Track No:"), str(self.song.tracknr), "", ""])
        else:
            l.append([_("Track No:"), "", "", ""])

        l.append([_("Genre:"), self.song.genre, "", ""])
        l.append([_("Time:"), "%d:%02d" % divmod(self.song.length, 60), "", ""])

        if self.song.rating:
            l.append([_("Rating:"), "*"*self.song.rating, "", ""])
        
        l.append([_("Times played:"), str(self.song.nrplayed), "", ""])
        
        for played in self.song.lastplayed[-1:-6:-1]:
            last = int((time.time()-played)/60)
            days, rest = divmod(last, 24*60)
            hours, minutes = divmod(rest, 60)
            if days>0:
                lastplayed = "%dd %dh %dm" % (days, hours, minutes)
            elif hours>0:
                lastplayed = "%dh %dm" % (hours, minutes)
            else:
                lastplayed = "%dm" % minutes

            l.append([_("Played:"), "%s (%s)" % (time.ctime(played), _("%s ago") % lastplayed), "", ""])

        return l


    def isartistoralbum(self):
        """ does self represent an artist or an album? """
        return False

    def format(self, formatstring, adddict={}, safe=False):
        """format song info using formatstring. Further song information
        in adddict is added. If safe is True, all values are cleaned
        of characters which are neither letters, digits, a blank or a colon.
        """

        self._readsong()
        d = {}
        d.update(self.song.__dict__)
        d.update(adddict)
        d["minutes"], d["seconds"] = divmod(d["length"], 60)
        d["length"] = "%d:%2d" % (d["minutes"], d["seconds"])

        if safe:
            allowedchars = string.letters + string.digits + " :"
            for key, value in d.items():
                try:
                    l = []
                    for c in value:
                        if c in allowedchars:
                            l.append(c)
                    d[key] = "".join(l)
                except TypeError:
                    pass

        return formatstring % d

    def play(self):
        self.song.play()
        self._updatesong()

    def unplay(self):
        """ forget last time song has been played (e.g., because playback was not complete) """
        self.song.unplay()
        self._updatesong()

    def rate(self, rating):
        if rating:
            self.song.rating = rating
        else:
            self.song.rating = None
        self.song.ratingsource = 0
        self._updatesong()

    def rescan(self):
        """rescan id3 information for song, keeping playing statistic, rating, etc."""
        self._readsong()
        hub.notify(events.rescansong(self.songdbid, self.song))


class artist(diritem):

    """ artist bound to specific songdb """

    def __init__(self, songdbid, name):
        self.songdbid = songdbid
        self.name = name

    def __repr__(self):
        return "artist(%s) in %s" % (self.name, self.songdbid)

    def cmpitem(self, x, y):
        if isinstance(x, album) and isinstance(y, album):
            return cmp(x.name, y.name)
        elif isinstance(x, songs):
            return 1
        else:
            return -1

    def getcontents(self):
        albums = hub.request(requests.getalbums(self.songdbid, self.name))
        albums = map(lambda aalbum, songdbid=self.songdbid:
                     album(songdbid, aalbum.id, self.name, aalbum.name),
                     albums)
        albums.sort(self.cmpitem)
        albums.append(songs(self.songdbid, self.name))
        return albums

    def getcontentsrecursive(self):
        asongs = hub.request(requests.getsongs(self.songdbid, artist=self.name))
        return _dbsongs_to_songs(self.songdbid, asongs)

    def getcontentsrecursivesorted(self):
        albums = hub.request(requests.getalbums(self.songdbid, self.name))
        albums = map(lambda aalbum, songdbid=self.songdbid:
                     album(songdbid, aalbum.id, self.name, aalbum.name),
                     albums)
        albums.sort(self.cmpitem)
        result = []
        for aalbum in albums:
            result.extend(aalbum.getcontentsrecursivesorted())
        return result

    def getcontentsrecursiverandom(self):
        asongs = hub.request(requests.getsongs(self.songdbid, artist=self.name, random=True))
        return _dbsongs_to_songs(self.songdbid, asongs)

    def getheader(self, item):
        return self.name

    def getinfo(self):
        l = [[_("Artist:"), self.name, "", ""]]
        return l

    def isartistoralbum(self):
        return True

    def rate(self, rating):
        for song in self.getcontentsrecursive():
            if song.ratingsource is None or song.ratingsource==2:
                if rating:
                    song.song.rating = rating
                else:
                    song.song.rating = None
                song.song.ratingsource = 2
                song._updatesong()

class album(diritem):

    """ album bound to specific songdb """

    def __init__(self, songdbid, id, artist, name):
        self.songdbid = songdbid
        self.id = id
        self.artist = artist
        self.name = name

    def __repr__(self):
        return "album(%s) in %s" % (self.id, self.songdbid)

    def cmpitem(self, x, y):
        return ( x.tracknr!="" and y.tracknr!="" and
                 cmp(int(x.tracknr), int(y.tracknr)) or
                 cmp(x.name, y.name) or
                 cmp(x.path, y.path) )

    def getcontents(self):
        songs = hub.request(requests.getsongs(self.songdbid, artist=self.artist, album=self.name))
        songs.sort(self.cmpitem)
        return _dbsongs_to_songs(self.songdbid, songs)

    def getcontentsrecursive(self):
        songs = hub.request(requests.getsongs(self.songdbid, artist=self.artist, album=self.name))
        return _dbsongs_to_songs(self.songdbid, songs)

    def getcontentsrecursiverandom(self):
        songs =  hub.request(requests.getsongs(self.songdbid,
                                                   artist=self.artist,
                                                   album=self.name,
                                                   random=True))
        return _dbsongs_to_songs(self.songdbid, songs)

    def getheader(self, item):
        if self.artist:
            return self.artist + " - " + self.name
        else:
            return self.name

    def getinfo(self):
        # XXX we have delete self.artist in the following line. check consequences
        # album = hub.request(requests.getalbum(self.songdbid, self.name))
        # if len(album.artists) != 1:
        #    artist = _("various")
        # else:
        #     artist = album.artists[0]
        l = [[_("Artist:"), self.artist is None and _("various") or self.artist, "", ""],
             [_("Album:"), self.name, "", ""]]

        return l

    def isartistoralbum(self):
        return True

    def rate(self, rating):
        for song in self.getcontentsrecursive():
            if song.ratingsource is None or song.ratingsource >= 1:
                if rating:
                    song.song.rating = rating
                else:
                    song.song.rating = None
                song.song.ratingsource = 1
                song._updatesong()
        

class playlist(diritem):

    """ songs in a playlist in the corresponding database """

    def __init__(self, songdbid, path, name, songs):
        self.songdbid = songdbid
        self.path = path
        self.name = name
        self.songs = songs

    def cmpitem(self, x, y):
        # This method does not make much sense for a playlist, since
        # the order of items is defined by the playlist
        # itself. However, just for completeness we set:
        return cmp(x.name, y.name)

    def getcontents(self):
        songs =  hub.request(requests.getsongsinplaylist(self.songdbid, self.path))
        return _dbsongs_to_songs(self.songdbid, songs)

    def getcontentsrecursive(self):
        songs = hub.request(requests.getsongsinplaylist(self.songdbid, self.path))
        return _dbsongs_to_songs(self.songdbid, songs)

    def getcontentsrecursiverandom(self):
        songs = hub.request(requests.getsongsinplaylist(self.songdbid, self.path, random=True))
        return _dbsongs_to_songs(self.songdbid, songs)

    def getheader(self, item):
        if item:
            return item.artist + " - " + item.album
        else:
            return self.name

    def getinfo(self):
        return [["%s:" % _("Playlist"), self.name, "", ""]]

class filtereditem(diritem):

    """ super class for item, which filters only items of a given kind"""

    def __init__(self, songdbid, item, indexname, indexid):
        self.songdbid = songdbid
        self.item = item
        self.indexname = indexname
        self.indexid = indexid
        self.nritems = None

    def getname(self):
        if isinstance(self.item, albums) or isinstance(self.item, songs):
            if self.nritems is None:
                self.nritems = len(self.getcontents())
            return "%s (%d) <%s>/" % (self.item.name, self.nritems, self.fname)
        else:
            return "%s <%s>/" % (self.item.name, self.fname)

    def cmpitem(self, x, y):
        try:
            return self.item.cmpitem(x.item, y.item)
        except AttributeError:
            return self.item.cmpitem(x, y)

    def getheader(self, item):
        if item is None:
            return ""
        if isinstance(self.item, albums):
            return "%s <%s>" % (self.item.getheader(item.item),
                                self.fname)
        else:
            return "%s <%s>" % (self.item.getheader(item),
                                self.fname)

    def getcontents(self):
        if isinstance(self.item, artist):
            aalbums = hub.request(requests.getalbums(self.songdbid,
                                                         self.item.name,
                                                         indexname=self.indexname, indexid=self.indexid))
            contents = map(lambda aalbum, klass=self.__class__, indexid=self.indexid, songdbid=self.songdbid:
                           klass(songdbid,
                                 album(songdbid, aalbum.id, self.item.name, aalbum.name),
                                 indexid),
                           aalbums)
            contents.append(self.__class__(self.songdbid, songs(self.songdbid, artist=self.item.name), self.indexid))
        elif isinstance(self.item, album):
            asongs = hub.request(requests.getsongs(self.songdbid,
                                                       artist=self.item.artist,
                                                       album=self.item.name,
                                                       indexname=self.indexname, indexid=self.indexid))
            contents = _dbsongs_to_songs(self.songdbid, asongs)
        elif isinstance(self.item, songs):
            asongs = hub.request(requests.getsongs(self.songdbid,
                                                       artist=self.item.artist,
                                                       indexname=self.indexname, indexid=self.indexid))
            contents = _dbsongs_to_songs(self.songdbid, asongs)
        elif isinstance(self.item, albums):
            aalbums = hub.request(requests.getalbums(self.songdbid,
                                                         indexname=self.indexname, indexid=self.indexid))
            contents = map(lambda aalbum, klass=self.__class__, indexid=self.indexid, songdbid=self.songdbid:
                           klass(songdbid,
                                 album(songdbid, aalbum.id, None, aalbum.name),
                                 indexid),
                           aalbums)
        else:
            # should not happen
            contents = []
            
        contents.sort(self.cmpitem)
        return contents

    def getcontentsrecursive(self):
        if isinstance(self.item, artist):
            asongs =  hub.request(requests.getsongs(self.songdbid,
                                                        artist=self.item.name,
                                                        indexname=self.indexname, indexid=self.indexid))
            return _dbsongs_to_songs(self.songdbid, asongs)
        elif isinstance(self.item, album):
            asongs =  hub.request(requests.getsongs(self.songdbid,
                                                        artist=self.item.artist,
                                                        album=self.item.name,
                                                        indexname=self.indexname, indexid=self.indexid))
            return _dbsongs_to_songs(self.songdbid, asongs)
        elif isinstance(self.item, songs) or isinstance(self.item, albums):
            asongs = hub.request(requests.getsongs(self.songdbid,
                                                       indexname=self.indexname, indexid=self.indexid))
            return _dbsongs_to_songs(self.songdbid, asongs)

        # should not happen
        return []

    def getcontentsrecursiverandom(self):
        if isinstance(self.item, artist):
            asongs =  hub.request(requests.getsongs(self.songdbid,
                                                        artist=self.item.name,
                                                        indexname=self.indexname, indexid=self.indexid,
                                                        random=True))
            return _dbsongs_to_songs(self.songdbid, asongs)
        elif isinstance(self.item, album):
            asongs =  hub.request(requests.getsongs(self.songdbid,
                                                        artist=self.item.artist,
                                                        album=self.item.name,
                                                        indexname=self.indexname, indexid=self.indexid,
                                                        random=True))
            return _dbsongs_to_songs(self.songdbid, asongs)
        elif isinstance(self.item, songs) or isinstance(self.item, albums):
            asongs = hub.request(requests.getsongs(self.songdbid,
                                                       indexname=self.indexnam, indexid=self.indexid,
                                                       random=True))
            return _dbsongs_to_songs(self.songdbid, asongs)

        # should not happen
        return []


    def getinfo(self):
        l = self.item.getinfo()
        l.append([_("Filter:"), self.fname, "", ""])
        return l

    def isartistoralbum(self):
        return isinstance(self.item, artist) or isinstance(self.item, album)


class filtereddecade(filtereditem):

    """ item, which filters only items of agiven decade"""

    def __init__(self, songdbid, item, indexid):
        filtereditem.__init__(self, songdbid, item, indexname="decade", indexid=indexid)
        self.decade = indexid
        self.fname = "%s=%s" % (_("Decade"),
                                self.decade and "%ds" % self.decade or _("Unknown"))
        self.name = "%s <%s>" % (self.item.name, self.fname)


class filteredgenre(filtereditem):

    """ item, which filters only items of given genre"""

    def __init__(self, songdbid, item, indexid):
        filtereditem.__init__(self, songdbid, item, indexname="genre", indexid=indexid)
        self.genre = indexid
        self.fname = "%s=%s" % (_("Genre"), self.genre)
        self.name = "%s <%s>" % (self.item.getname(), self.fname)
        
class filteredrating(filtereditem):

    """ item, which filters only items of given rating"""

    def __init__(self, songdbid, item, indexid):
        filtereditem.__init__(self, songdbid, item, indexname="rating", indexid=indexid)
        self.rating = indexid
        if self.rating is not None:
            self.fname = "%s=%s" % (_("Rating"), "*" * self.rating)
        else:
            self.fname = "%s=%s" % (_("Rating"), _("Not rated"))
        self.name = "%s <%s>" % (self.item.getname(), self.fname)

class index(diritem):

    """ artists, albums + songs filtered by a given index e in the corresponding database """

    def __init__(self, songdbid, indexname, indexid, indexclass):
        self.songdbid = songdbid
        self.indexname = indexname
        self.indexid= indexid
        self.indexclass = indexclass

    def getcontents(self):
        artists = hub.request(requests.getartists(self.songdbid, indexname=self.indexname, indexid=self.indexid))
        contents = map(lambda aartist, indexid=self.indexid, indexclass=self.indexclass, songdbid=self.songdbid:
                       indexclass(songdbid, artist(songdbid, aartist.name), indexid),
                       artists)
        contents.sort(self.cmpitem)
        contents.append(self.indexclass(self.songdbid, albums(self.songdbid), self.indexid))
        contents.append(self.indexclass(self.songdbid, songs(self.songdbid), self.indexid))
        return contents

    def getcontentsrecursivesorted(self):
        # we cannot rely on the default implementation since we don't want
        # to have the albums and songs included trice
        artists = hub.request(requests.getartists(self.songdbid, indexname=self.indexname, indexid=self.indexid))
        artists = map(lambda aartist, indexid=self.indexid, indexclass=self.indexclass, songdbid=self.songdbid:
                       indexclass(songdbid, artist(songdbid, aartist.name), self.indexid),
                       artists)
        artists.sort(self.cmpitem)
        result = []
        for aartist in artists:
            result.extend(aartist.getcontentsrecursivesorted())
        return result

    def getcontentsrecursive(self):
        asongs = hub.request(requests.getsongs(self.songdbid,
                                                   indexname=self.indexname, indexid=self.indexid))
        return _dbsongs_to_songs(self.songdbid, asongs)

    def getcontentsrecursiverandom(self):
        asongs = hub.request(requests.getsongs(self.songdbid,
                                                   indexname=self.indexname, indexid=self.indexid,
                                                   random=True))
        return _dbsongs_to_songs(self.songdbid, asongs)


class genre(index):

    """ artists, albums + songs from a specific genre in the corresponding database """

    def __init__(self, songdbid, name):
        index.__init__(self, songdbid, indexname="genre", indexid=name, indexclass=filteredgenre)
        self.name = name

    def cmpitem(self, x, y):
        return cmp(x.name, y.name)

    def getheader(self, item):
        return self.name

    def getinfo(self):
        return [["%s:" % _("Genre"), self.name, "", ""]]


class decade(index):

    """ artists, albums + songs from a specific decade in the corresponding database """

    def __init__(self, songdbid, decade):
        # decade = None, ..., 1960, 1970, ...
        assert decade is None or decade%10 == 0, \
               "decade has to be an integer multiple of 10 or None"

        index.__init__(self, songdbid, indexname="decade", indexid=decade, indexclass=filtereddecade)
        self.decade = decade
        self.name = decade and "%ds" % decade or _("Unknown")

    def cmpitem(self, x, y):
        return cmp(x.name, y.name)
    def getheader(self, item):
        return self.name

    def getinfo(self):
        return [["%s:" % _("Decade"), self.name, "", ""]]

class rating(index):

    """ artists, albums + songs with a specific rating in the corresponding database """

    def __init__(self, songdbid, rating):
        index.__init__(self, songdbid, indexname="rating", indexid=rating, indexclass=filteredrating)
        self.rating = rating

    def cmpitem(self, x, y):
        return cmp(x.rating, y.rating)

    def getname(self):
        if self.rating is not None:
            return "%s/" % ("*" * self.rating)
        else:
            return ("%s/" % _("Not rated"))

    def getheader(self, item):
        return str(self.rating)

    def getinfo(self):
        if self.rating is not None:
            return [["%s:" % _("Rating"), "*" * self.rating, "", ""]]
        else:
            return [["%s:" % _("Rating"), _("Not rated"), "", ""]]


class randomsonglist(diritem):

    """ random list of songs out of  the corresponding database """

    def __init__(self, songdbid, maxnr):
        self.songdbid = songdbid
        self.name = "[%s]" % _("Random song list")
        self.maxnr = maxnr

    def cmpitem(self, x, y):
        # just for completeness, we set:
        return cmp(x.name, y.name)

    def getcontents(self):
        songs = []
        while len(songs)<self.maxnr:
            newsongs = hub.request(requests.getsongs(self.songdbid, random=True))
            if len(newsongs) > 0:
                songs.extend(newsongs)
            else:
                break
        songs = songs[:self.maxnr]
        return _dbsongs_to_songs(self.songdbid, songs)

    def getcontentsrecursive(self):
        songs = hub.request(requests.getsongs(self.songdbid))
        return _dbsongs_to_songs(self.songdbid, songs)
    
    def getcontentsrecursiverandom(self):
        songs = hub.request(requests.getsongs(self.songdbid, random=True))
        return _dbsongs_to_songs(self.songdbid, songs)

    def getheader(self, item):
        if item:
            return item.artist + " - " + item.album
        else:
            return _("Random song list")

    def getinfo(self):
        return [[_("Random song list"), "", "", ""]]


class lastplayedsongs(diritem):

    """ songs last played out of the corresponding databases """

    def __init__(self, songdbids):
        self.songdbids = songdbids
        self.name = "[%s]" % _("Last played songs")

    def cmpitem(self, x, y):
        return cmp(y.lastplayed, x.lastplayed)

    def getcontents(self):
        lastplayed = []
        for songdbid in self.songdbids:
            newlastplayed = hub.request(requests.getlastplayedsongs(songdbid))
            for asong, aplayingtime in newlastplayed:
                lastplayed.append((song(songdbid, song=asong), aplayingtime))
        # sort by playing time
        lastplayed.sort(lambda x,y: cmp(y[1], x[1]))
        return [asong for asong, playingtime in lastplayed]

    getcontentsrecursive = getcontentsrecursivesorted = getcontents

    def getcontentsrecursiverandom(self):
        songs = []
        for songdbid in self.songdbids:
            newlastplayed = hub.request(requests.getlastplayedsongs(songdbid))
            for asong, aplayingtime in newlastplayed:
                songs.append(song(songdbid, song=asong))
        return _genrandomchoice(songs)

    def getheader(self, item):
        if item:
            return item.artist + " - " + item.album
        else:
            return _("Last played songs")

    def getinfo(self):
        return [[_("Last played songs"), "", "", ""]]


class topplayedsongs(diritem):

    """ songs most often played of the corresponding databases """

    def __init__(self, songdbids):
        self.songdbids = songdbids
        self.name = "[%s]" % _("Top played songs")

    def cmpitem(self, x, y):
        return cmp(y.nrplayed, x.nrplayed) or cmp(y.lastplayed, x.lastplayed)

    def getcontents(self):
        songs = []
        for songdbid in self.songdbids:
            songs.extend(_dbsongs_to_songs(songdbid, hub.request(requests.gettopplayedsongs(songdbid))))
        songs.sort(self.cmpitem)
        return songs

    getcontentsrecursive = getcontentsrecursivesorted = getcontents

    def getcontentsrecursiverandom(self):
        songs = []
        for songdbid in self.songdbids:
            songs.extend(_dbsongs_to_songs(songdbid, hub.request(requests.gettopplayedsongs(songdbid))))
        return _genrandomchoice(songs)

    def getheader(self, item):
        if item:
            return item.artist + " - " + item.album
        else:
            return _("Top played songs")

    def getinfo(self):
        return [[_("Top played songs"), "", "", ""]]



class lastaddedsongs(diritem):

    """ songs last added to the corresponding database """

    def __init__(self, songdbid):
        self.songdbid = songdbid
        self.name = "[%s]" % _("Last added songs")

    def cmpitem(self, x, y):
        return cmp(y.added, x.added)

    def getcontents(self):
        songs = hub.request(requests.getlastaddedsongs(self.songdbid))
        return _dbsongs_to_songs(self.songdbid, songs)

    getcontentsrecursive = getcontentsrecursivesorted = getcontents

    def getcontentsrecursiverandom(self):
        return hub.request(requests.getlastaddedsongs(self.songdbid, random=True))

    def getheader(self, item):
        if item:
            return item.artist + " - " + item.album
        else:
            return _("Last added songs")

    def getinfo(self):
        return [[_("Last added songs"), "", "", ""]]


class albums(diritem):

    """ all albums in the corresponding database """

    def __init__(self, songdbid):
        self.songdbid = songdbid
        self.name = _("Albums")
        self.nralbums = None

    def getname(self):
        if self.nralbums is None:
            self.nralbums = hub.request(requests.getnumberofalbums(self.songdbid))
        return "[%s (%d)]/" % (self.name, self.nralbums)

    def cmpitem(self, x, y):
        return cmp(x.name, y.name)

    def getcontents(self):
        albums = hub.request(requests.getalbums(self.songdbid))
        albums = map(lambda aalbum, songdbid=self.songdbid:
                     album(songdbid, aalbum.id, None, aalbum.name),
                     albums)
        albums.sort(self.cmpitem)
        return albums

    def getcontentsrecursive(self):
        asongs = hub.request(requests.getsongs(self.songdbid))
        return _dbsongs_to_songs(self.songdbid, asongs)

    def getcontentsrecursiverandom(self):
        asongs = hub.request(requests.getsongs(self.songdbid, random=True))
        return _dbsongs_to_songs(self.songdbid, asongs)

    def getheader(self, item):
        #if item:
        #    return item.artist
        #else:
        #    return self.getname()[1:-2]
        return self.getname()[1:-2]

    def getinfo(self):
        return [[_("Albums"), "", "", ""]]


class genres(diritem):

    """ all genres in the corresponding database """

    def __init__(self, songdbid):
        self.songdbid = songdbid
        self.name = _("Genres")
        self.nrgenres = None

    def getname(self):
        if self.nrgenres is None:
            self.nrgenres = hub.request(requests.getnumberofgenres(self.songdbid))
        return "[%s (%d)]/" % (_("Genres"), self.nrgenres)

    def cmpitem(self, x, y):
        return cmp(x.name, y.name)

    def getcontents(self):
        genres = hub.request(requests.getgenres(self.songdbid))
        genres = map(lambda agenre, songdbid=self.songdbid:
                     genre(songdbid, agenre.name),
                     genres)
        genres.sort(self.cmpitem)
        return genres

    def getcontentsrecursive(self):
        asongs = hub.request(requests.getsongs(self.songdbid))
        return _dbsongs_to_songs(self.songdbid, asongs)

    def getcontentsrecursiverandom(self):
        asongs = hub.request(requests.getsongs(self.songdbid, random=True))
        return _dbsongs_to_songs(self.songdbid, asongs)

    def getheader(self, item):
        nrgenres = hub.request(requests.getnumberofgenres(self.songdbid))
        return "%s (%d)" % (_("Genres"), nrgenres)

    def getinfo(self):
        return [[_("Genres"), "", "", ""]]


class decades(diritem):

    """ all decades in the corresponding database """

    def __init__(self, songdbid):
        self.songdbid = songdbid
        self.name = _("Decades")
        self.nrdecades = None

    def getname(self):
        if self.nrdecades is None:
            self.nrdecades = hub.request(requests.getnumberofdecades(self.songdbid))
        return "[%s (%d)]/" % (_("Decades"), self.nrdecades)

    def cmpitem(self, x, y):
        return cmp(x.name, y.name)

    def getcontents(self):
        decades = map(lambda d, songdbid=self.songdbid:
                      decade(songdbid, d),
                      hub.request(requests.getdecades(self.songdbid)))
        decades.sort(self.cmpitem)
        return decades

    def getcontentsrecursive(self):
        asongs = hub.request(requests.getsongs(self.songdbid))
        return _dbsongs_to_songs(self.songdbid, asongs)

    def getcontentsrecursiverandom(self):
        asongs = hub.request(requests.getsongs(self.songdbid, random=True))
        return _dbsongs_to_songs(self.songdbid, asongs)

    def getheader(self, item):
        nrdecades = hub.request(requests.getnumberofdecades(self.songdbid))
        return "%s (%d)" % (_("Decades"), nrdecades)

    def getinfo(self):
        return [[_("Decades"), "", "", ""]]

class ratings(diritem):

    """ all ratings in the corresponding database """

    def __init__(self, songdbid):
        self.songdbid = songdbid
        self.name = _("Ratings")
        self.nrratings = None

    def getname(self):
        if self.nrratings is None:
            self.nrratings = hub.request(requests.getnumberofratings(self.songdbid))
        return "[%s (%d)]/" % (_("Ratings"), self.nrratings)

    def cmpitem(self, x, y):
        if x.rating is None:
            return 1
        elif y.rating is None:
            return -1
        else:
            return cmp(y.rating, x.rating)

    def getcontents(self):
        ratings = hub.request(requests.getratings(self.songdbid))
        ratings = map(lambda arating, songdbid=self.songdbid:
                     rating(songdbid, arating.rating),
                     ratings)
        ratings.sort(self.cmpitem)
        return ratings

    def getcontentsrecursive(self):
        asongs = hub.request(requests.getsongs(self.songdbid))
        return _dbsongs_to_songs(self.songdbid, asongs)

    def getcontentsrecursiverandom(self):
        asongs = hub.request(requests.getsongs(self.songdbid, random=True))
        return _dbsongs_to_songs(self.songdbid, asongs)

    def getheader(self, item):
        nrratings = hub.request(requests.getnumberofratings(self.songdbid))
        return "%s (%d)" % (_("Ratings"), nrratings)

    def getinfo(self):
        return [[_("Ratings"), "", "", ""]]



class songs(diritem):

    """ all songs in the corresponding database """

    def __init__(self, songdbid, artist=None):
        self.songdbid = songdbid
        self.name = _("Songs")
        self.artist = artist
        self.nrsongs = None

    def getname(self):
        if self.artist is None:
            if self.nrsongs is None:
                self.nrsongs = hub.request(requests.getnumberofsongs(self.songdbid))
            return "[%s (%d)]/" % (self.name, self.nrsongs)
        else:
            if self.nrsongs is None:
                self.nrsongs = len(self.getcontents())
            return "[%s (%d)]/" % (_("Songs"), self.nrsongs)

    def cmpitem(self, x, y):
        return ( cmp(x.title, y.title) or
                 cmp(x.album, y.album) or
                 cmp(x.path, y.path)
                 )

    def getcontents(self):
        songs = hub.request(requests.getsongs(self.songdbid, artist=self.artist))
        songs.sort(self.cmpitem)
        return _dbsongs_to_songs(self.songdbid, songs)

    getcontentsrecursivesorted = getcontents

    def getcontentsrecursive(self):
        songs = hub.request(requests.getsongs(self.songdbid, artist=self.artist))
        return _dbsongs_to_songs(self.songdbid, songs)

    def getcontentsrecursiverandom(self):
        songs = hub.request(requests.getsongs(self.songdbid, artist=self.artist, random=True))
        for song in songs:
            assert isinstance(song, dbitem.song), repr(song)
        return _dbsongs_to_songs(self.songdbid, songs)

    def getheader(self, item):
        if item:
            return item.artist + " - " + item.album
        else:
            return self.getname()[1:-2]

    def getinfo(self):
        if self.artist is not None:
            return [[_("Artist:"), self.artist, "", ""],
                    [_("Songs"), "", "", ""]]
        else:
            return [[_("Songs"), "", "", ""]]


class playlists(diritem):

    """ all playlists in the corresponding database """

    def __init__(self, songdbid):
        self.songdbid = songdbid
        self.name = _("Playlists")
        self.nrplaylists = None

    def getname(self):
        if self.nrplaylists is None:
            self.nrplaylists = len(self.getcontents())
        return "[%s (%d)]/" % (_("Playlists"), self.nrplaylists)

    def cmpitem(self, x, y):
        return cmp(x.name, y.name)

    def getcontents(self):
        playlists = hub.request(requests.getplaylists(self.songdbid))
        if playlists:
            playlists = map(lambda aplaylist, songdbid=self.songdbid:
                            playlist(songdbid, aplaylist.path, aplaylist.name, aplaylist.songs),
                            playlists)
        playlists.sort(self.cmpitem)
        return playlists

    def getcontentsrecursive(self):
        songs = hub.request(requests.getsongsinplaylists(self.songdbid))
        return _dbsongs_to_songs(self.songdbid, songs)

    def getcontentsrecursiverandom(self):
        songs = hub.request(requests.getsongsinplaylists(self.songdbid, random=True))
        return _dbsongs_to_songs(self.songdbid, songs)

    def getheader(self, item):
        return "%s (%d)" % (_("Playlists"), len(self.getcontents()))

    def getinfo(self):
        return [[_("Playlists"), "", "", ""]]


class filesystemdir(diritem):

    """ diritem corresponding to directory in filesystem """

    def __init__(self, songdbid, basedir, dir):
        self.songdbid = songdbid
        self.basedir = basedir
        self.dir = dir

        if self.dir==self.basedir:
            self.name = "[%s]" % _("Filesystem")
        else:
            self.name = self.dir[len(self.basedir):].split("/")[-1]

    def cmpitem(self, x, y):
        return cmp(x.name, y.name)

    def getname(self):
        return "%s/" % self.name

    def getcontents(self):
        items = []
        try:
            for name in os.listdir(self.dir):
                try:
                    path = os.path.join(self.dir, name)
                    if os.path.isdir(path) and os.access(path, os.R_OK|os.X_OK):
                        newitem = filesystemdir(self.songdbid, self.basedir, path)
                        items.append(newitem)
                    elif name.lower().endswith(".mp3") and os.access(path, os.R_OK):
                        newsong = song(self.songdbid,
                                       song=hub.request(requests.queryregistersong(self.songdbid, path)))
                        items.append(newsong)
                    elif name.lower().endswith(".ogg") and os.access(path, os.R_OK) and dbitem.oggsupport:
                        newsong = song(self.songdbid,
                                       song=hub.request(requests.queryregistersong(self.songdbid, path)))

                        items.append(newsong)
                except (IOError, OSError) : pass
        except OSError:
            return None
        items.sort(self.cmpitem)
        return items

    def getcontentsrecursiverandom(self):
        songs = self.getcontentsrecursive()
        return _genrandomchoice(songs)

    def getheader(self, item):
        if self.dir==self.basedir:
            return _("Filesystem")
        else:
            return self.name

    def getinfo(self):
        return [["%s:" % _("Filesystem"), self.dir, "", ""]]

    def isbasedir(self):
        """ return whether the filesystemdir is the basedir of a song database """
        return self.dir == self.basedir


class basedir(diritem):

    """ base dir of database view"""

    def __init__(self, songdbids, maxnr, virtualdirectoriesattop):
        self.name = _("Song Database")
        self.songdbids = songdbids
        self.songdbid = songdbids[0]
        self.type, self.basedir = hub.request(requests.getdatabaseinfo(self.songdbid))
        self.maxnr = maxnr
        self.virtualdirectoriesattop = virtualdirectoriesattop
        self.nrsongs = None

    def cmpitem(self, x, y):
        if isinstance(x, artist) and isinstance(y, artist):
            return cmp(x.name, y.name)
        elif isinstance(x, artist):
            return self.virtualdirectoriesattop and 1 or -1
        elif isinstance(y, artist):
            return self.virtualdirectoriesattop and -1 or 1
        else:
            return cmp(x.name, y.name)

    def getname(self):
        return "[%s]/" % self.getheader(None)

    def getcontents(self):
        artists = hub.request(requests.getartists(self.songdbid))
        contents = map(lambda aartist, songdbid=self.songdbid:
                       artist(songdbid, aartist.name),
                       artists)
        contents.sort(self.cmpitem)

        if self.type=="local":
            afilesystemdir = filesystemdir(self.songdbid, self.basedir, self.basedir)
        else:
            afilesystemdir = None
        asongs = songs(self.songdbid)
        aalbums = albums(self.songdbid)
        adecades = decades(self.songdbid)
        agenres = genres(self.songdbid)
        aratings = ratings(self.songdbid)
        atopplayedsongs = topplayedsongs(self.songdbids)
        alastplayedsongs = lastplayedsongs(self.songdbids)
        alastaddedsongs = lastaddedsongs(self.songdbid)
        arandomsonglist = randomsonglist(self.songdbid, self.maxnr)
        aplaylists = playlists(self.songdbid)
        if len(self.songdbids) > 1:
            asubbasedir = basedir(self.songdbids[1:], self.maxnr, self.virtualdirectoriesattop)
        else:
            asubbasedir = None

        virtdirs = [asongs,
                    aalbums,
                    adecades,
                    agenres,
                    aratings,
                    atopplayedsongs,
                    alastplayedsongs,
                    alastaddedsongs,
                    arandomsonglist,
                    aplaylists]

        if afilesystemdir is not None:
            virtdirs[:0] = [afilesystemdir]

        if asubbasedir is not None:
            virtdirs.append(asubbasedir)

        if self.virtualdirectoriesattop:
            contents[:0] = virtdirs
        else:
            contents[len(contents):] = virtdirs
        return contents

    def getcontentsrecursivesorted(self):
        # we cannot rely on the default implementation since we don't want
        # to have the albums and songs included trice
        artists = hub.request(requests.getartists(self.songdbid))
        artists = map(lambda aartist, songdbid=self.songdbid:
                      artist(songdbid, aartist.name),
                      artists)
        artists.sort(self.cmpitem)
        result = []
        for aartist in artists:
            result.extend(aartist.getcontentsrecursivesorted())
        return result

    def getcontentsrecursive(self):
        asongs = hub.request(requests.getsongs(self.songdbid))
        return _dbsongs_to_songs(self.songdbid, asongs)

    def getcontentsrecursiverandom(self):
        asongs = hub.request(requests.getsongs(self.songdbid, random=True))
        return _dbsongs_to_songs(self.songdbid, asongs)

    def getheader(self, item):
        if self.nrsongs is None:
            self.nrsongs = hub.request(requests.getnumberofsongs(self.songdbid))
        return _("Database (%s, %d songs)") % (self.basedir, self.nrsongs)

    def getinfo(self):
        return [["%s:" % _("Database"), self.basedir, "", ""]]
