/***************************************************************************
  controller.cpp
  -------------------
  Control class for QBrew
  -------------------
  Copyright (c) 2003 David Johnson
  Please see the header file for copyright and license information.
***************************************************************************/

#include <qaction.h>
#include <qapplication.h>
#include <qdir.h>
#include <qfiledialog.h>
#include <qlabel.h>
#include <qmenubar.h>
#include <qmessagebox.h>
#include <qpaintdevicemetrics.h>
#include <qpainter.h>
#include <qprinter.h>
#include <qregexp.h>
#include <qstatusbar.h>
#include <qtoolbar.h>
#include <qtoolbutton.h>
#include <qwhatsthis.h>

#if QT_VERSION > 300
#include <qstylefactory.h>
#endif

#include "helpwindow.h"
#include "hydrometertool.h"
#include "model.h"
#include "preferences.h"
#include "setupdialog.h"
#include "view.h"

#include "controller.h"

using namespace AppResource;

Controller *Controller::instance_ = 0;

//////////////////////////////////////////////////////////////////////////////
// Construction, Destruction, Initialization                                //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// Controller()
// ------------
// Private constructor

Controller::Controller()
    : model_(0), view_(0), preferences_(0), filename_(""),
      newflag_(false), filemenu_(0), optionsmenu_(0), toolsmenu_(0),
      helpmenu_(0), toolbar_(0), units_("")
{ ; }

void Controller::initialize(const QString &filename)
{
    // load preferences from rc file
    preferences_ = new Preferences(QDIR_HOME + "/" + ID_PREFERENCES_FILE,
                                   PACKAGE, VERSION);

    setIcon(QPixmap(dataBase() + "qbrew.png"));
#if QT_VERSION > 300
    QIconSet::setIconSize(QIconSet::Large, QSize(22, 22));
#endif
    setUsesBigPixmaps(preferences_->getBool(ID_PREF_LARGE_ICONS,
                                            ID_PREF_LARGE_ICONS_DEFAULT));
    
    // initialize frame elements
    initActions();
    initMenuBar();
    initToolBar();
    initStatusBar();

    // set up model
    filename_ = filename;
    model_ = Model::instance();
    // load or create a recipe
    if ((filename_ != ID_DEFAULT_FILE) && (filename_ != "")) {
        setCaption(ID_TITLE + " - " + filename_);
        model_->loadRecipe(filename);
    } else {
        setCaption(ID_TITLE + VERSION);
        model_->newRecipe();
        newflag_ = true;
    }

    // give Model what it needs
    units_ = preferences_->getString(ID_PREF_UNITS, ID_PREF_UNITS_DEFAULT);
    double batch = preferences_->getDouble(ID_PREF_BATCH, ID_PREF_BATCH_DEFAULT);
    QString style = preferences_->getString(ID_PREF_RECIPE_STYLE,
                                            ID_PREF_RECIPE_STYLE_DEFAULT);
    if (units_ == UNIT_METRIC) {
        model_->setDefaultSize(Volume(batch, Volume::liter));
        model_->setDefaultGrainUnit(Weight::kilogram);
        model_->setDefaultHopsUnit(Weight::gram);
    } else if (units_ == UNIT_US) {
        model_->setDefaultSize(Volume(batch, Volume::gallon));
        model_->setDefaultGrainUnit(Weight::pound);
        model_->setDefaultHopsUnit(Weight::ounce);
    }
    model_->setDefaultMiscUnit(Quantity::generic);
    model_->setDefaultStyle(Style(style));

    // new documents are not modified
    model_->setModified(false);

    // give Calc what it needs
    Calc::setEfficiency(preferences_->getDouble(ID_PREF_EFFICIENCY,
                                                ID_PREF_EFFICIENCY_DEFAULT));
    Calc::setTinseth(preferences_->getBool(ID_PREF_TINSETH,
                                           ID_PREF_TINSETH_DEFAULT));

    // set up view
    view_ = new View(this, model_);
    setCentralWidget(view_);

#if QT_VERSION > 300
    // setup theme
    QString theme = preferences_->getString(ID_PREF_WIDGET_STYLE,
                                            ID_PREF_WIDGET_STYLE_DEFAULT);

    // QStyleFactory::keys() will give "fcntl: Bad file descriptor" warning
    if (QStyleFactory::keys().contains(theme)) {
        QApplication::setStyle(QStyleFactory::create(theme));
    }
#endif
}

//////////////////////////////////////////////////////////////////////////////
// ~Controller()
// -------------
// Private destructor

Controller::~Controller()
{
    if (preferences_) delete preferences_;
    if (view_) delete view_;
    if (filemenu_) delete filemenu_;
    if (optionsmenu_) delete optionsmenu_;
    if (toolsmenu_) delete toolsmenu_;
    if (helpmenu_) delete helpmenu_;
    if (toolbar_) delete toolbar_;
    // TODO: Do I delete the actions?
}

//////////////////////////////////////////////////////////////////////////////
// instance()
// ----------
// Return pointer to the controller.

Controller *Controller::instance()
{
    if (!instance_)
        instance_ = new Controller();
    return instance_;
}

//////////////////////////////////////////////////////////////////////////////
// initActions()
// -------------
// Initialize the actions

void Controller::initActions()
{
    QString smallicon = dataBase() + "icons/16x16/actions/";
    QString largeicon = dataBase() + "icons/22x22/actions/";

    // file actions
    filenew_ = new QAction("New",
                           QIconSet(QPixmap(smallicon + "filenew.png"),
                                    QPixmap(largeicon + "filenew.png")),
                           "&New", CTRL+Key_N, this);
    fileopen_ = new QAction("Open",
                            QIconSet(QPixmap(smallicon + "fileopen.png"),
                                     QPixmap(largeicon + "fileopen.png")),
                            "&Open...", CTRL+Key_O, this);
    // TODO: Open Recent...
    filesave_ = new QAction("Save",
                            QIconSet(QPixmap(smallicon + "filesave.png"),
                                     QPixmap(largeicon + "filesave.png")),
                            "&Save",  CTRL+Key_S, this);
    filesaveas_ = new QAction("Save As",
                              QIconSet(QPixmap(smallicon + "filesaveas.png"),
                                       QPixmap(largeicon + "filesaveas.png")),
                              "Save &as...", 0, this);
    fileprint_ = new QAction("Print",
                             QIconSet(QPixmap(smallicon + "fileprint.png"),
                                      QPixmap(largeicon + "fileprint.png")),
                              "&Print...", CTRL+Key_P, this);
    filequit_ = new QAction("Quit",
                            QIconSet(QPixmap(smallicon + "exit.png"),
                                     QPixmap(largeicon + "exit.png")),
                            "&Quit", CTRL+Key_Q, this);

    // options actions
    optionstoolbar_ = new QAction("Toolbar",
                                  "&Toolbar", 0, this, 0, true);
    optionsstatusbar_ = new QAction("Statusbar",
                                    "&Statusbar", 0, this, 0, true);
    // TODO: Configure Key Bindings...
    optionssetup_ = new QAction("Configure",
                                    "&Configure...", 0, this);
    optionssavesetup_ = new QAction("Save Configuration",
                                        "Sa&ve Configuration", 0, this);

    // tools actions
    toolhydrometer_ = new QAction("Hydrometer Correction",
                                   "&Hydrometer Correction...", 0, this);

    // help actions
    helpcontents_ = new QAction("Help",
                                QIconSet(QPixmap(smallicon + "contents.png"),
                                         QPixmap(largeicon + "contents.png")),
                                "&Contents", Key_F1, this);
    helpprimer_ = new QAction("Primer",
                              QIconSet(QPixmap(smallicon + "contents.png"),
                                       QPixmap(largeicon + "contents.png")),
                                "&Primer", 0, this);
    helpabout_ = new QAction("About", "&About...", 0, this);
    helpcontext_ = new QAction("Context",
                               QIconSet(QPixmap(smallicon + "contexthelp.png"),
                                        QPixmap(largeicon + "contexthelp.png")),
                               "&What's This?", SHIFT+Key_F1, this);

    // create status tips
    filenew_->setStatusTip("Create a new recipe");
    fileopen_->setStatusTip("Open an existing recipe");
    filesave_->setStatusTip("Save the recipe");
    filesaveas_->setStatusTip("Save the recipe under a new name");
    fileprint_->setStatusTip("Print the recipe");
    filequit_->setStatusTip("Quit the application");
    optionstoolbar_->setStatusTip("Enable or disable the Toolbar");
    optionsstatusbar_->setStatusTip("Enable or disable the Statusbar");
    optionssetup_->setStatusTip("Display the configuration dialog");
    optionssavesetup_->setStatusTip("Save the configuration");
    toolhydrometer_->setStatusTip("Hydrometer correction utility");
    helpcontents_->setStatusTip("Display the application handbook");
    helpprimer_->setStatusTip("Display the brewing primer");
    helpabout_->setStatusTip("Application information");
    helpcontext_->setStatusTip("Context Sensitive Help");

    // create what's this help
    filenew_->setWhatsThis("Click this button to create a new recipe.\n\n"
        "You can also select the New command from the File menu.");
    fileopen_->setWhatsThis("Click this button to open a recipe.\n\n"
        "You can also select the Open command from the File menu.");
    filesave_->setWhatsThis("Click this button to save the recipe you "
        "are editing.  You will be prompted for a file name.\n\n"
        "You can also select the Save command from the File menu.");
    fileprint_->setWhatsThis("Click this button to print the recipe.\n\n"
        "You can also select the Print command from the File menu.");
    // TODO: whatsthis for other items?

    // create connections
    connect(filenew_, SIGNAL(activated()), this,
            SLOT(fileNew()));
    connect(fileopen_, SIGNAL(activated()), this,
            SLOT(fileOpen()));
    connect(filesave_, SIGNAL(activated()), this,
            SLOT(fileSave()));
    connect(filesaveas_, SIGNAL(activated()), this,
            SLOT(fileSaveAs()));
    connect(fileprint_, SIGNAL(activated()), this,
            SLOT(filePrint()));
    connect(filequit_, SIGNAL(activated()), qApp,
            SLOT(closeAllWindows()));
    connect(optionstoolbar_, SIGNAL(activated()), this,
            SLOT(optionsToolbar()));
    connect(optionsstatusbar_, SIGNAL(activated()), this,
            SLOT(optionsStatusbar()));
    connect(optionssetup_, SIGNAL(activated()), this,
            SLOT(optionsSetup()));
    connect(optionssavesetup_, SIGNAL(activated()), this,
            SLOT(optionsSaveSetup()));
    connect(toolhydrometer_, SIGNAL(activated()), this,
            SLOT(toolsHydrometer()));
    connect(helpcontents_, SIGNAL(activated()), this,
            SLOT(helpContents()));
    connect(helpprimer_, SIGNAL(activated()), this,
            SLOT(helpPrimer()));
    connect(helpabout_, SIGNAL(activated()), this,
            SLOT(helpAbout()));
    connect(helpcontext_, SIGNAL(activated()), this,
            SLOT(whatsThis()));

    // enable/disable appropriate items
    filesave_->setEnabled(false);
    optionstoolbar_->setOn(preferences_->getBool(ID_PREF_TOOLBAR, 
                           ID_PREF_TOOLBAR_DEFAULT));
    optionsstatusbar_->setOn(preferences_->getBool(ID_PREF_STATUSBAR,
                        ID_PREF_STATUSBAR_DEFAULT));
}
//////////////////////////////////////////////////////////////////////////////
// initMenuBar()
// -------------
// Initialize the menu bar

void Controller::initMenuBar()
{
    // file menu
    filemenu_ = new QPopupMenu();
    filenew_->addTo(filemenu_);
    fileopen_->addTo(filemenu_);
    filemenu_->insertSeparator();
    filesave_->addTo(filemenu_);
    filesaveas_->addTo(filemenu_);
    filemenu_->insertSeparator();
    fileprint_->addTo(filemenu_);
    filemenu_->insertSeparator();
    filequit_->addTo(filemenu_);

    // options menu
    optionsmenu_ = new QPopupMenu();
    optionsmenu_->setCheckable(true);
    optionstoolbar_->addTo(optionsmenu_);
    optionsstatusbar_->addTo(optionsmenu_);
    optionsmenu_->insertSeparator();
    optionssetup_->addTo(optionsmenu_);
    optionsmenu_->insertSeparator();
    optionssavesetup_->addTo(optionsmenu_);

    // options menu
    toolsmenu_ = new QPopupMenu();
    toolhydrometer_->addTo(toolsmenu_);

    // help menu
    helpmenu_ = new QPopupMenu();
    helpcontents_->addTo(helpmenu_);
    helpprimer_->addTo(helpmenu_);
    helpcontext_->addTo(helpmenu_);
    helpmenu_->insertSeparator();
    helpabout_->addTo(helpmenu_);

    // insert submenus into main menu
    menuBar()->insertItem("&File", filemenu_);
    menuBar()->insertItem("&Options", optionsmenu_);
    menuBar()->insertItem("&Tools", toolsmenu_);
    menuBar()->insertSeparator();
    menuBar()->insertItem("&Help", helpmenu_);
}

//////////////////////////////////////////////////////////////////////////////
// initToolBar()
// -------------
// Initialize the toolbar

void Controller::initToolBar()
{
    setRightJustification(false);
    toolbar_ = new QToolBar("Main Toolbar", this);

    filenew_->addTo(toolbar_);
    fileopen_->addTo(toolbar_);
    filesave_->addTo(toolbar_);
    fileprint_->addTo(toolbar_);
    toolbar_->addSeparator();
    helpcontext_->addTo(toolbar_);

    // now show or hide toolbar depending on initial setting
    if (preferences_->getBool(ID_PREF_TOOLBAR, true)) toolbar_->show();
    else toolbar_->hide();
}

//////////////////////////////////////////////////////////////////////////////
// initStatusBar()
// ---------------
// Initialize the status bar

void Controller::initStatusBar()
{
    statusBar()->message(ID_READY, 2000);
    // now show or hide statusbar depending on initial setting
    if (preferences_->getBool(ID_PREF_STATUSBAR, true)) statusBar()->show();
    else statusBar()->hide();
}

//////////////////////////////////////////////////////////////////////////////
// File Menu Implementation                                                 //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// fileNew()
// ---------
// Create a new recipe

void Controller::fileNew()
{
    if (model_->modified()) {
        // file needs to be saved, what do we do
        switch (querySave()) {
            case 0: // yes, save the file
                fileSave();
                break;
            case 1: // no, don't save the file
                break;
            case 2: // cancel creating new file
                statusBar()->message("Canceled...", 2000);
                // exit function
                return;
        }
    }
    // create a new file
    statusBar()->message("Creating new recipe...");
    model_->newRecipe();
    newflag_ = true;
    filesave_->setEnabled(false);
    view_->viewModified();
    // no file name yet, so set it as junk
    filename_ = ID_DEFAULT_FILE;
    setCaption(ID_TITLE + " - " + filename_);
    statusBar()->message(ID_READY, 2000);
}

//////////////////////////////////////////////////////////////////////////////
// fileOpen()
// ----------
// Open a recipe

void Controller::fileOpen()
{
    if (model_->modified()) {
        // file needs to be saved, what do we do
        switch (querySave()) {
            case 0: // yes, save the file
                fileSave();
                break;
            case 1: // no, don't save the file
                break;
            case 2: // cancel creating new file
                statusBar()->message("Canceled...", 2000);
                return;
        }
    }
    // open the file
    statusBar()->message("Opening recipe...");
    QString fname = QFileDialog::getOpenFileName(0, ID_FILE_FILTER, this);
    // keep information about the file
    QFileInfo *finfo = new QFileInfo(fname);
    if (!fname.isEmpty()) {
        // file dialog returned a file name
        if (model_->loadRecipe(fname)) {
            // load was successful
            newflag_ = false;
            view_->viewModified();
            filesave_->setEnabled(false);
            setCaption(ID_TITLE + " - " + finfo->fileName());
            statusBar()->message("Loaded recipe: " + finfo->fileName(), 2000);
            // save name of file
            filename_ = fname;
        } else {
            // load was unsuccessful
            QMessageBox::warning(this, ID_TITLE, ID_TITLE +
                                 " was unable to load the recipe " +
                                 finfo->fileName());
            statusBar()->message("Error in loading " + finfo->fileName() , 2000);
        }
    } else {
        // file dialog didn't return a file name
        statusBar()->message("Loading aborted", 2000);
    }
    delete finfo;
}

//////////////////////////////////////////////////////////////////////////////
// fileSave()
// ----------
// Save a recipe

void Controller::fileSave()
{
    if (newflag_)
    {
        fileSaveAs();
    } else {
        // file exists so save it
        statusBar()->message("Saving recipe...");
        if (model_->saveRecipe(filename_)) {
            // successful in saving file
            newflag_ = false;
            filesave_->setEnabled(false);
            statusBar()->message(ID_READY, 2000);
        } else {
            // error in saving file
            QMessageBox::warning(this, ID_TITLE, ID_TITLE +
                                 " was unable to save the recipe " +
                                 filename_);
            statusBar()->message("Error in saving recipe", 2000);
        }
    }
}

//////////////////////////////////////////////////////////////////////////////
// fileSaveAs()
// ------------
// Save a recipe under a new name

void Controller::fileSaveAs()
{
    statusBar()->message("Saving recipe under new filename...");
    QString fname = QFileDialog::getSaveFileName(0, ID_FILE_FILTER, this);
    if (!fname.isEmpty()) { // we got a valid filename
        // append .qbrew if there's no suffix
        if (fname.findRev('.') == -1) fname += "." + ID_FILE_EXT;
        QFileInfo *finfo = new QFileInfo(fname);
        if (model_->saveRecipe(fname)) {
            // successfully saved
            newflag_ = false;
            filesave_->setEnabled(false);
            setCaption(ID_TITLE + " - " + finfo->fileName());
            statusBar()->message(ID_READY, 2000);
            // save name of file
            filename_ = fname;
        } else {
            // error in saving
            QMessageBox::warning(this, ID_TITLE, ID_TITLE +
                                 " was unable to save the recipe " +
                                 finfo->fileName());
            statusBar()->message("Error in saving recipe", 2000);
        }
        delete finfo;
    } else {
        // no file name chosen
        statusBar()->message("Saving aborted", 2000);
    }
}

//////////////////////////////////////////////////////////////////////////////
// filePrint()
// -----------
// Print the recipe. Much of this method derived from Qt example programs,
// including "helpviewer" copyright 1992-2000 Troll Tech AS.

void Controller::filePrint()
{
    // TODO: eventually will have an export function to export to html. At that time
    // switch this to print html/richtext.
    statusBar()->message("Printing...");

    QPrinter* printer = new QPrinter();
    printer->setFullPage(true);

    if (printer->setup())
    {
        QPainter painter(printer);
        QPaintDeviceMetrics metrics(painter.device());

        QString line;
        int tpos;
        int ypos = 12;              // y position for each line
        const int margin = 18;      // eighteen point margin

        // painter.begin(printer);
        QFontMetrics fm = painter.fontMetrics();
        QFont font("times", 10);    // serif font best for printouts
        font.setPointSize(10);
        painter.setFont(font);      // use fixed width font
        QString text = model_->recipeText();
        int pageno = 1;             // keep track of pages
        // for each line of text...
        while (text.length() > 0) {
            // get line of text
            tpos = text.find('\n');
            if (tpos > 0) {
                line = text.left(tpos);
                text.remove(0, tpos+1);
            } else {
                // get last line if text doesn't end in newline
                line = text; text = "";
            }
            // is there space for this line on page?
            if ((margin + ypos) > (metrics.height() - margin)) {
                statusBar()->message("Printing (page " +
                                     QString::number(++pageno) + ")...");
                printer->newPage();         // no more room on this page
                ypos = 12;                  // back to top of page
            }
            // print the line
            painter.drawText(margin, margin + ypos, metrics.width(),
                             fm.lineSpacing(), ExpandTabs | DontClip, line );
            // update paper position
            ypos += fm.lineSpacing();
        }
        painter.end();        // send job to printer
        statusBar()->message(ID_READY, 2000);
    } else {
        // user chose not to print
        statusBar()->message("Printing cancelled", 2000);
    }
    delete printer;
}

//////////////////////////////////////////////////////////////////////////////
// fileQuit()
// ----------
// Quit the application

void Controller::fileQuit()
{
    if (model_->modified()) {
        // file needs to be saved, what do we do
        switch (querySave()) {
            case 0: // yes, save the file
                fileSave();
                break;
            case 1: // no, go ahead and exit
                break;
            case 2: // cancel exit
                statusBar()->message(ID_READY, 2000);
                return;
        }
    }
    qApp->quit();
}

//////////////////////////////////////////////////////////////////////////////
// Options Menu Implementation                                              //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// optionsToolbar()
// ----------------
// Toggle toolbar status

void Controller::optionsToolbar()
{
    if (toolbar_->isVisible()) {
        toolbar_->hide();
        optionstoolbar_->setOn(false);
        preferences_->setBool(ID_PREF_TOOLBAR, false);
    } else {
        toolbar_->show();
        optionstoolbar_->setOn(true);
        preferences_->setBool(ID_PREF_TOOLBAR, true);
    }
    statusBar()->message(ID_READY, 2000);
}

//////////////////////////////////////////////////////////////////////////////
// optionsSatusbar()
// -----------------
// Toggle statusbar status

void Controller::optionsStatusbar()
{
    if (statusBar()->isVisible()) {
        statusBar()->hide();
        optionsstatusbar_->setOn(false);
        preferences_->setBool(ID_PREF_STATUSBAR, false);
    } else {
        statusBar()->show();
        optionsstatusbar_->setOn(true);
        preferences_->setBool(ID_PREF_STATUSBAR, true);
    }
    statusBar()->message(ID_READY, 2000);
}

//////////////////////////////////////////////////////////////////////////////
// optionsSetup()
// --------------
// Display the setup dialog

void Controller::optionsSetup()
{
    statusBar()->message("Configuring " + ID_TITLE + "...");
    SetupDialog* dialog = new SetupDialog(this, "SetupDialog");
    dialog->setCaption(ID_TITLE + " - Configure");

    // save some stuff we are interested in...
    double oldefficiency = preferences_->getDouble(ID_PREF_EFFICIENCY,
                                                ID_PREF_EFFICIENCY_DEFAULT);
    bool oldtinseth = preferences_->getBool(ID_PREF_TINSETH,
                                         ID_PREF_TINSETH_DEFAULT);
    QString oldunits = preferences_->getString(ID_PREF_UNITS,
                                               ID_PREF_UNITS_DEFAULT);
    double oldbatch = preferences_->getDouble(ID_PREF_BATCH,
                                              ID_PREF_BATCH_DEFAULT);
    QString oldstyle = preferences_->getString(ID_PREF_RECIPE_STYLE,
                                               ID_PREF_RECIPE_STYLE_DEFAULT);
#if QT_VERSION > 300
    QString oldtheme = preferences_->getString(ID_PREF_WIDGET_STYLE,
                                               ID_PREF_WIDGET_STYLE_DEFAULT);
#endif

    // setup dialog
    dialog->setAppDir(preferences_->getString(ID_PREF_APPDIR,
                                              ID_PREF_APPDIR_DEFAULT));
    dialog->setBatch(preferences_->getDouble(ID_PREF_BATCH,
                                             ID_PREF_BATCH_DEFAULT));
    dialog->setEfficiency(preferences_->getDouble(ID_PREF_EFFICIENCY,
                                                  ID_PREF_EFFICIENCY_DEFAULT));
    dialog->setRecipeStyleBox(model_->stylesList());
    dialog->setRecipeStyle(preferences_->getString(ID_PREF_RECIPE_STYLE,
                                                   ID_PREF_RECIPE_STYLE_DEFAULT));
    dialog->setTinseth(preferences_->getBool(ID_PREF_TINSETH,
                                             ID_PREF_TINSETH_DEFAULT));
    dialog->addUnits(UNIT_METRIC);
    dialog->addUnits(UNIT_US);
    dialog->setUnits(preferences_->getString(ID_PREF_UNITS,
                                             ID_PREF_UNITS_DEFAULT));
    dialog->setLargeIcons(preferences_->getBool(ID_PREF_LARGE_ICONS,
                                                ID_PREF_LARGE_ICONS_DEFAULT));
#if QT_VERSION > 300
    dialog->setWidgetStyleBox(QStyleFactory::keys());
    dialog->setWidgetStyle(oldtheme);
#endif

    // execute the dialog
    int ret = dialog->exec();
    if (ret == QDialog::Accepted) {
        preferences_->setString(ID_PREF_APPDIR, dialog->appDir());
        preferences_->setDouble(ID_PREF_BATCH, dialog->batch());
        preferences_->setDouble(ID_PREF_EFFICIENCY, dialog->efficiency());
        preferences_->setString(ID_PREF_RECIPE_STYLE, dialog->recipeStyle());
        preferences_->setBool(ID_PREF_TINSETH, dialog->tinseth());
        preferences_->setString(ID_PREF_UNITS, dialog->units());
        preferences_->setBool(ID_PREF_LARGE_ICONS, dialog->largeIcons());
#if QT_VERSION > 300
        preferences_->setString(ID_PREF_WIDGET_STYLE, dialog->widgetStyle());
#endif
    }

    // activate new style
    QString theme = preferences_->getString(ID_PREF_WIDGET_STYLE,
                                            ID_PREF_WIDGET_STYLE_DEFAULT);
    setUsesBigPixmaps(preferences_->getBool(ID_PREF_LARGE_ICONS,
                                            ID_PREF_LARGE_ICONS_DEFAULT));    
    double batch = preferences_->getDouble(ID_PREF_BATCH, ID_PREF_BATCH_DEFAULT);
    QString style = preferences_->getString(ID_PREF_RECIPE_STYLE,
                                            ID_PREF_RECIPE_STYLE_DEFAULT);
    units_ = preferences_->getString(ID_PREF_UNITS, ID_PREF_UNITS_DEFAULT);
    if ((oldunits != units_) || (oldbatch != batch) || (oldstyle != style)) {
        if (units_ == UNIT_METRIC) {
            model_->setDefaultSize(Volume(batch, Volume::liter));
            model_->setDefaultGrainUnit(Weight::kilogram);
            model_->setDefaultHopsUnit(Weight::gram);
        } else if (units_ == UNIT_US) {
            model_->setDefaultSize(Volume(batch, Volume::gallon));
            model_->setDefaultGrainUnit(Weight::pound);
            model_->setDefaultHopsUnit(Weight::ounce);
        }
        model_->setDefaultMiscUnit(Quantity::generic);
        model_->setDefaultStyle(style);
        view_->viewModified();
    }
#if QT_VERSION > 300
    if (oldtheme != theme)  {
        QApplication::setStyle(QStyleFactory::create(theme));
    }
#endif

    // do we need to recalc?
    double efficiency = preferences_->getDouble(ID_PREF_EFFICIENCY,
                                                ID_PREF_EFFICIENCY_DEFAULT);
    bool tinseth = preferences_->getBool(ID_PREF_TINSETH,
                                         ID_PREF_TINSETH_DEFAULT);
    if ((oldefficiency != efficiency) || (oldtinseth != tinseth)) {
        Calc::setEfficiency(efficiency);
        Calc::setTinseth(tinseth);
    	model_->recalc();
        view_->viewModified();
    }

    statusBar()->message(ID_READY, 2000);
}

//////////////////////////////////////////////////////////////////////////////
// optionsSaveSetup()
// ------------------
// Received to save the application settings

void Controller::optionsSaveSetup()
{
    statusBar()->message("Saving configuration...");
    preferences_->flush();
    statusBar()->message(ID_READY, 2000);
}

//////////////////////////////////////////////////////////////////////////////
// Tools Menu Implementation                                                //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// toolsHydrometer()
// -----------------
// A utility dialog for hydrometer correction

void Controller::toolsHydrometer()
{
    (new HydrometerTool(this))->show();
    statusBar()->message(ID_READY, 2000);
}

//////////////////////////////////////////////////////////////////////////////
// Help Menu Implementation                                                 //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// helpContents()
// --------------
// Display the application manual

void Controller::helpContents()
{
    QString home = docBase() + ID_HELP_FILE;
    (new HelpWindow(home, dataBase(), 0, "HelpWindow"))->show();
    statusBar()->message(ID_READY, 2000);
}

//////////////////////////////////////////////////////////////////////////////
// helpPrimer()
// ------------
// Display the brewing primer

void Controller::helpPrimer()
{
    QString home = docBase() + ID_PRIMER_FILE;
    (new HelpWindow(home, dataBase(), 0, "HelpWindow"))->show();
    statusBar()->message(ID_READY, 2000);
}

//////////////////////////////////////////////////////////////////////////////
// helpAbout()
// -----------
// Display the About dialog

void Controller::helpAbout()
{
    QString message;

    message = "<nobr><big><b>" + ID_TITLE
        + " Version " + VERSION + "</b></big></nobr>";
    message += "<p><b>" + ID_DESCRIPTION + "</b>";
    message += "<nobr><p>" + ID_COPYRIGHT + ' ' + ID_AUTHOR + "</nobr>";
    message += "<br>Contributions by " + ID_CONTRIBUTORS;

    QMessageBox::about(this, "helpAbout", message);
}

//////////////////////////////////////////////////////////////////////////////
// Miscellaneous                                                            //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// documentModified()
// ----------------------
// Received when document is modified

void Controller::documentModified()
{
    filesave_->setEnabled(true);
}

//////////////////////////////////////////////////////////////////////////////
// querySave()
// -----------
// Ask the user if they want to save their work before going on

int Controller::querySave()
{
    return QMessageBox::information(this, ID_TITLE + " - Save?",
        "Do you wish to save your work first?", "Yes", "No", "Cancel", 0, 2);
}

//////////////////////////////////////////////////////////////////////////////
// dataBase()
// ---------
// Figure out the base directory for the data
// Under Unix, the default will be /usr/local/share/qbrew
// Under Windows, the default will be currentDirPath()

QString Controller::dataBase()
{
    static QString base;
    if (base.isNull()) {
        base = preferences_->getString(ID_PREF_APPDIR,
                                       ID_PREF_APPDIR_DEFAULT);
        if (base == "") base = QDir::currentDirPath();
        if ((QRegExp("$qbrew/?", false).match(base) != -1) || (OWNBASEDIR)) {
            // we have our own subdirectory
            base += "/";
        } else {
            // we are in a generic directory like '/usr/local'
            base += "/share/qbrew/";
        }
    }
    return base;
}

//////////////////////////////////////////////////////////////////////////////
// docBase()
// ---------
// Figure out the base directory for the documentation
// Under Unix the default will be /usr/local/share/doc/qbrew/en
// Under Windows the default will be currentDirPath()/doc/en/

QString Controller::docBase()
{
    static QString base;
    if (base.isNull()) {
        base = preferences_->getString(ID_PREF_APPDIR,
                                       ID_PREF_APPDIR_DEFAULT);
        if (base == "") base = QDir::currentDirPath();
        if ((QRegExp("$qbrew/?", false).match(base) != -1) || (OWNBASEDIR)) {
            // we have our own subdirectory
            base += "/doc/en/";
        } else {
            // we are in a generic directory like '/usr/local'
            base += "/share/doc/qbrew/en/";
        }
    }
    return base;
}
