/***************************************************************************
  grainpage.cpp
  -------------------
  A dialog page for grains
  -------------------
  begin         October 10th, 1999
  author        David Johnson <david@usermode.org>
  -------------------
  Copyright 1999-2002, David Johnson
  Please see the header file for copyright and license information
 ***************************************************************************/

#include <qcheckbox.h>
#include <qcombobox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qlistview.h>
#include <qpushbutton.h>
#include <qstringlist.h>

#include "fixedspinbox.h"

#include "resource.h"
#include "model.h"
#include "view.h"

#include "grainpage.h"

//////////////////////////////////////////////////////////////////////////////
// Construction, Destruction                                                //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// GrainPage()
// -----------
// Constructor

GrainPage::GrainPage(Model *model, View *view, Controller *controller,
                     const char *name)
    : IngredientPage(model, view, controller, name), extractlabel_(0),
      extractspin_(0), colorlabel_(0), colorspin_(0), uselabel_(0),
      usecombo_(0), currentitem_(0)
{
    // setup listview
    itemslistview_->addColumn("Extract");
    itemslistview_->addColumn("Color");
    itemslistview_->addColumn("Use");
    itemslistview_->setColumnAlignment(2, AlignRight);
    itemslistview_->setColumnAlignment(3, AlignRight);
    itemslistview_->setColumnAlignment(4, AlignRight);
    itemslistview_->setSorting(0);
    itemslistview_->setSorting(1);
    itemslistview_->setSorting(2);
    itemslistview_->setSorting(3);
    itemslistview_->setSorting(4);

    // setup widgets
    extractlabel_ = new QLabel("Extract", this, "extractlabel_");
    extractspin_ = new FixedSpinBox(1.000, 1.100, 0.001, 3, this, "extractspin_");
    colorlabel_ = new QLabel("Color", this, "colorlabel_");
    colorspin_ = new FixedSpinBox(0.0, 500.0, 1.0, 1, this, "colorspin_");
    colorspin_->setSuffix(AppResource::CHAR_LATIN_DEGREE);
    uselabel_ = new QLabel("Use", this, "uselabel_");
    usecombo_ = new QComboBox(false, this, "usecombo_");
    usecombo_->insertStringList(Grain::useStringList());

    // setup layout
    rowtwolayout_->addWidget(quantitylabel_, 0);
    rowtwolayout_->addWidget(quantityspin_, 0, AlignRight);
    rowtwolayout_->addWidget(extractlabel_, 0);
    rowtwolayout_->addWidget(extractspin_, 0);
    rowtwolayout_->addWidget(colorlabel_, 0);
    rowtwolayout_->addWidget(colorspin_, 0);
    rowtwolayout_->addWidget(uselabel_, 0);
    rowtwolayout_->addWidget(usecombo_, 0);

    rowtwolayout_->addWidget(spacer_, 1);
    rowtwolayout_->addWidget(addbutton_, 0);
    rowtwolayout_->addWidget(applybutton_, 0);
    rowtwolayout_->addWidget(removebutton_, 0);

    // set the leftmost labels to the same minimum size
    quantitylabel_->adjustSize();
    quantitylabel_->setMinimumWidth(quantitylabel_->width());
    itemlabel_->setMinimumWidth(quantitylabel_->width());

    // fixup tab orders
    QWidget::setTabOrder(itemcombo_, quantityspin_);
    QWidget::setTabOrder(quantityspin_, extractspin_);
    QWidget::setTabOrder(extractspin_, colorspin_);
    QWidget::setTabOrder(colorspin_, usecombo_);

    // setup connections
    connect(itemcombo_, SIGNAL(activated(const QString &)),
            this, SLOT(updateEntryItems(const QString &)));
    connect(addbutton_, SIGNAL(clicked()), SLOT(addGrain()));
    connect(applybutton_, SIGNAL(clicked()), SLOT(applyGrain()));
    connect(removebutton_, SIGNAL(clicked()), SLOT(removeGrain()));
    connect(itemslistview_, SIGNAL(selectionChanged(QListViewItem *)),
            SLOT(itemSelected(QListViewItem *)));

    documentChanged();
}

GrainPage::~GrainPage() {}

//////////////////////////////////////////////////////////////////////////////
// Slots                                                                    //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// updateEntryItems()
// ------------------
// Update the items in the entry area

void GrainPage::updateEntryItems(const QString &name)
{
    Grain *thisgrain = model_->grain(name);
    quantityspin_->setValue(thisgrain->weight().amount());
    quantityspin_->setSuffix(" " + thisgrain->weight().unit().symbol());
    extractspin_->setValue(thisgrain->extract());
    colorspin_->setValue(thisgrain->color());
    // vary default use according to recipe type
    int utemp = thisgrain->use();
    QString ustr = thisgrain->useString();
    if (model_->recipe()->mashed()) {
        if (utemp == GRAIN_STEEPED) ustr = "mashed";
    } else {
        if (utemp == GRAIN_MASHED) ustr = "steeped";
    }
    View::setComboItem(usecombo_, ustr);
}

//////////////////////////////////////////////////////////////////////////////
// documentChanged()
// -----------------
// Document has changed

void GrainPage::documentChanged()
{
    if (itemcombo_->count() == 0) {
        itemcombo_->clear();
        itemcombo_->insertStringList(model_->grainsList());
    }
    updateEntryItems(itemcombo_->currentText());
    updateListView();
}

//////////////////////////////////////////////////////////////////////////////
// addGrain()
// ----------
// Add a grain to the recipe

void GrainPage::addGrain()
{
    GrainIterator it;
    GrainListItem *item;

    Grain grain(itemcombo_->currentText(),
                Weight(quantityspin_->value(), model_->defaultGrainUnit()),
                extractspin_->value(), colorspin_->value(), 
                usecombo_->currentText());
    model_->checkGrain(grain);

    it = model_->recipe()->addGrain(grain);
    item = new GrainListItem(itemslistview_, (*it).name(),
        QString::number((*it).weight().amount(), 'f', 2)
                             + " " + (*it).weight().unit().symbol(),
        QString::number((*it).extract(), 'f', 3),
        QString::number((*it).color()) + AppResource::CHAR_LATIN_DEGREE,
        (*it).useString() );
    item->setGrainIterator(it);

    view_->viewModified();
}

//////////////////////////////////////////////////////////////////////////////
// applyGrain()
// ------------
// Apply a grain to the recipe

void GrainPage::applyGrain()
{
    if (currentitem_ != 0) {
        Grain grain(itemcombo_->currentText(),
                    Weight(quantityspin_->value(), model_->defaultGrainUnit()),
                    extractspin_->value(), colorspin_->value(), 
                    usecombo_->currentText());
        model_->checkGrain(grain);

        model_->recipe()->applyGrain(currentitem_->grainIterator(), grain);
        currentitem_->setToIterator();
        view_->viewModified();
    }
}

//////////////////////////////////////////////////////////////////////////////
// removeGrain()
// -------------
// Remove a grain from the recipe

void GrainPage::removeGrain()
{
    if (currentitem_ != 0) {
        model_->recipe()->removeGrain(currentitem_->grainIterator());
        delete currentitem_; currentitem_ = 0;
        view_->viewModified();
    }
}

//////////////////////////////////////////////////////////////////////////////
// itemSelected()
// --------------
// A grain is selected in the listview

void GrainPage::itemSelected(QListViewItem *item)
{
    if (item) {
        currentitem_ = dynamic_cast<GrainListItem*>(item);
        if (currentitem_ == 0) {
            qWarning("bad dynamic cast, line" + __LINE__);
            return;
        }
        itemcombo_->setEditText(item->text(0));
        quantityspin_->setValue(item->text(1).toDouble());
        extractspin_->setValue(item->text(2).toDouble());
        colorspin_->setValue(item->text(3).toDouble());
        View::setComboItem(usecombo_, item->text(4));
    }
}

//////////////////////////////////////////////////////////////////////////////
// Miscellaneous                                                            //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// updateListView()
// ----------------
// Update the list view

void GrainPage::updateListView()
{
    GrainList *list = model_->recipe()->grains();
    GrainIterator it;
    GrainListItem *item;
    itemslistview_->clear();
    for (it=list->begin(); it != list->end(); ++it) {
        item = new GrainListItem(itemslistview_, (*it).name(),
                                 QString::number((*it).weight().amount(), 'f', 2)
                                 + " " + (*it).weight().unit().symbol(),
                                 QString::number((*it).extract(), 'f', 3),
                                 QString::number((*it).color())
                                 + AppResource::CHAR_LATIN_DEGREE,
                                 (*it).useString() );
        item->setGrainIterator(it);
    }
}
