/***************************************************************************
  hopspage.cpp
  -------------------
  description   A dialog page for hops
  -------------------
  begin         October 20th, 1999
  author        David Johnson <david@usermode.org>
  -------------------
  Copyright 1999-2002, David Johnson
  Please see the header file for copyright and license information
 ***************************************************************************/

#include <qcheckbox.h>
#include <qcombobox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qlistview.h>
#include <qpushbutton.h>
#include <qstringlist.h>

#include "fixedspinbox.h"

#include "model.h"
#include "view.h"

#include "hopspage.h"

//////////////////////////////////////////////////////////////////////////////
// Construction, Destruction                                                //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// HopsPage()
// ----------
// Constructor

HopsPage::HopsPage(Model *model, View *view, Controller *controller,
                   const char *name)
    : IngredientPage(model, view, controller, name), formlabel_(0),
      formcombo_(0), alphalabel_(0), alphaspin_(0), timelabel_(0),
      timespin_(0), currentitem_(0)
{
    // setup listview
    itemslistview_->addColumn("Form");
    itemslistview_->addColumn("Alpha");
    itemslistview_->addColumn("Time");
    itemslistview_->setColumnAlignment(2, AlignRight);
    itemslistview_->setColumnAlignment(3, AlignRight);
    itemslistview_->setColumnAlignment(4, AlignRight);
    itemslistview_->setSorting(0);
    itemslistview_->setSorting(1);
    itemslistview_->setSorting(2);
    itemslistview_->setSorting(3);
    itemslistview_->setSorting(4);

    // setup widgets
    formlabel_ = new QLabel("Form", this, "formlabel_");
    formcombo_ = new QComboBox(true, this, "formcombo_");
    formcombo_->insertStringList(model_->formsList());
    alphalabel_ = new QLabel("Alpha", this, "alphalabel_");
    alphaspin_ = new FixedSpinBox(0.0, 50.0, 0.1, 1, this, "alphaspin_");
    alphaspin_->setSuffix("%");
    timelabel_ = new QLabel("Time", this, "timelabel_");
    timespin_ = new QSpinBox(0, 120, 5, this, "time");
    timespin_->setSuffix(" min");

    // setup layout
    rowtwolayout_->addWidget(quantitylabel_, 0);
    rowtwolayout_->addWidget(quantityspin_, 0, AlignRight);
    rowtwolayout_->addWidget(formlabel_, 0);
    rowtwolayout_->addWidget(formcombo_, 0);
    rowtwolayout_->addWidget(alphalabel_, 0);
    rowtwolayout_->addWidget(alphaspin_, 0);
    rowtwolayout_->addWidget(timelabel_, 0);
    rowtwolayout_->addWidget(timespin_, 0);

    rowtwolayout_->addWidget(spacer_, 1);
    rowtwolayout_->addWidget(addbutton_, 0);
    rowtwolayout_->addWidget(applybutton_, 0);
    rowtwolayout_->addWidget(removebutton_, 0);

    // set the leftmost labels to the same minimum size
    quantitylabel_->adjustSize();
    quantitylabel_->setMinimumWidth(quantitylabel_->width());
    itemlabel_->setMinimumWidth(quantitylabel_->width());

    // fixup tab orders
    QWidget::setTabOrder(itemcombo_, quantityspin_);
    QWidget::setTabOrder(quantityspin_, formcombo_);
    QWidget::setTabOrder(formcombo_, alphaspin_);
    QWidget::setTabOrder(alphaspin_, timespin_);

    // setup connections
    connect(itemcombo_, SIGNAL(activated(const QString &)),
            this, SLOT(updateEntryItems(const QString &)));
    connect(addbutton_, SIGNAL(clicked()), SLOT(addHops()));
    connect(applybutton_, SIGNAL(clicked()), SLOT(applyHops()));
    connect(removebutton_, SIGNAL(clicked()), SLOT(removeHops()));
    connect(itemslistview_, SIGNAL(selectionChanged(QListViewItem *)),
            SLOT(itemSelected(QListViewItem *)));
    // initialize widgets with values
    documentChanged();
}

HopsPage::~HopsPage() { ; }

//////////////////////////////////////////////////////////////////////////////
// Slots                                                                    //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// slotUpdateEntryItems()
// ----------------------
// update the hop entry widgets

void HopsPage::updateEntryItems(const QString &name)
{
    Hops *thishop = model_->hop(name);
    quantityspin_->setValue(thishop->weight().amount());
    quantityspin_->setSuffix(" " + thishop->weight().unit().symbol());
    formcombo_->setEditText(thishop->form());
    alphaspin_->setValue(thishop->alpha());
    timespin_->setValue(thishop->time());
}

//////////////////////////////////////////////////////////////////////////////
// documentChanged()
// -----------------
// the document has changed

void HopsPage::documentChanged()
{
    if (itemcombo_->count() == 0) {
        itemcombo_->clear();
        itemcombo_->insertStringList(model_->hopsList());
    }
    updateEntryItems(itemcombo_->currentText());
    updateListView();
}

//////////////////////////////////////////////////////////////////////////////
// addHops()
// ---------
// Add a hop to recipe

void HopsPage::addHops()
{
    HopIterator it;
    HopsListItem *item;
    Hops hop(itemcombo_->currentText(),
             Weight(quantityspin_->value(), model_->defaultHopsUnit()),
             formcombo_->currentText(), alphaspin_->value(),
             timespin_->value());
    model_->checkHops(hop);

    it = model_->recipe()->addHop(hop);
    item = new HopsListItem(itemslistview_, (*it).name(),
                            QString::number((*it).weight().amount(), 'f', 2)
                            + " " + (*it).weight().unit().symbol(),
                            (*it).form(),
                            QString::number((*it).alpha()) + '%',
                            QString::number((*it).time()) );
    item->setHopIterator(it);

    view_->viewModified();
}

//////////////////////////////////////////////////////////////////////////////
// applyHops()
// -----------
// Apply a hop to recipe

void HopsPage::applyHops()
{
    if (currentitem_ != 0) {
        Hops hop(itemcombo_->currentText(),
             Weight(quantityspin_->value(), model_->defaultHopsUnit()),
             formcombo_->currentText(), alphaspin_->value(),
             timespin_->value());
        model_->checkHops(hop);

        model_->recipe()->applyHop(currentitem_->hopIterator(), hop);
        currentitem_->setToIterator();
        view_->viewModified();
    }
}

//////////////////////////////////////////////////////////////////////////////
// removeHops()
// ------------
// Remove a hop from recipe

void HopsPage::removeHops()
{
    if (currentitem_ != 0) {
        model_->recipe()->removeHop(currentitem_->hopIterator());
        delete currentitem_; currentitem_ = 0;
        view_->viewModified();
    }
}

//////////////////////////////////////////////////////////////////////////////
// itemSelected()
// -----------------
// Received when user selects an item in listview

void HopsPage::itemSelected(QListViewItem *item)
{
    if (item) {
        currentitem_ = dynamic_cast<HopsListItem*>(item);
        if (currentitem_ == 0) {
            qWarning("bad dynamic cast, line" + __LINE__);
            return;
        }
        itemcombo_->setEditText(item->text(0));
        quantityspin_->setValue(item->text(1).toDouble());
        formcombo_->setEditText(item->text(2));
        alphaspin_->setValue(item->text(3).toDouble());
        timespin_->setValue(item->text(4).toUInt());
    }

}

//////////////////////////////////////////////////////////////////////////////
// Miscellaneous                                                            //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// updateListView()
// ----------------
// Update the listview

void HopsPage::updateListView()
{
    HopsList *list = model_->recipe()->hops();
    HopIterator it;
    HopsListItem *item;
    itemslistview_->clear();
    for (it=list->begin(); it != list->end(); ++it) {
        item = new HopsListItem(itemslistview_, (*it).name(),
                                QString::number((*it).weight().amount(), 'f', 2)
                                + " " + (*it).weight().unit().symbol(),
                                (*it).form(),
                                QString::number((*it).alpha()) + '%',
                                QString::number((*it).time()) );
        item->setHopIterator(it);
    }
}
