/***************************************************************************
  setupdialog.cpp
  -------------------
  A setup dialog for QBrew
  -------------------
  begin         September 22th, 1999
  author        David Johnson <david@usermode.org>
  -------------------
  Copyright 1999-2002, David Johnson
  Please see the header file for copyright and license information
 ***************************************************************************/

#include <qpushbutton.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <qframe.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qlineedit.h>

#include "model.h"
#include "resource.h"
#include "setupdialog.h"

#include "fixedspinbox.h"

using namespace AppResource;

//////////////////////////////////////////////////////////////////////////////
// Construction, Destruction                                                //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// SetupDialog()
// -------------
// Constructor

SetupDialog::SetupDialog(QWidget* parent, const char* name)
    : QDialog(parent, name, true)
{
    // create label widgets
    qbrewlabel_ = new QLabel(this, "qbrewlabel_");
    qbrewlabel_->setText("QBrew Directory");
    widgetstylelabel_ = new QLabel(this, "widgetstylelabel_");
    widgetstylelabel_->setText("Qt Widget Style");
    unitslabel_ = new QLabel(this, "unitslabel_");
    unitslabel_->setText("Units of Measurement");
    batchlabel_ = new QLabel(this, "batchlabel_");
    batchlabel_->setText("Default Batch Size");
    efficiencylabel_ = new QLabel(this, "efficiencylabel_");
    efficiencylabel_->setText("Mash Efficiency");
    recipestylelabel_ = new QLabel(this, "recipestylelabel_");
    recipestylelabel_->setText("Default Recipe Style");
    separatorone_ = new QFrame(this, "separatorone_");
    separatorone_->setFrameStyle(QFrame::HLine | QFrame::Sunken);
    separatortwo_ = new QFrame(this, "separatortwo_");
    separatortwo_->setFrameStyle(QFrame::HLine | QFrame::Sunken);

    // create other widgets
    qbrewedit_ = new QLineEdit(this, "qbrewedit_");
    widgetstylebox_ = new QComboBox(this, "widgetstylebox_");
    unitsbox_ = new QComboBox(this, "unitsbox_");
    batchspin_ = new FixedSpinBox(0.00, 100.00, 0.25, 2, this, "batchspin_");
    efficiencyspin_ = new FixedSpinBox(0.00, 1.0, 0.01, 2, this, "efficiencyspin_");
    recipestylebox_ = new QComboBox(this, "recipestylebox_");
    tinsethcheck_ = new QCheckBox(this, "tinsethcheck_");
    tinsethcheck_->setText("Use Tinseth Hop Calculation");
    iconcheck_ = new QCheckBox(this, "iconcheck_");
    iconcheck_->setText("Use large icons");
    cancelbutton_ = new QPushButton(this, "cancelbutton_");
    cancelbutton_->setText("Cancel");
    okbutton_ = new QPushButton(this, "okbutton_");
    okbutton_->setText("OK");
    okbutton_->setDefault(true);

    // create layout
    layout_ = new QGridLayout(this, 3, 11, 3, 3, "layout_");
    layout_->addColSpacing(1, 3);
    layout_->setColStretch(0, 1);
    layout_->setColStretch(2, 1);
    layout_->addWidget(qbrewlabel_, 0, 0, Qt::AlignRight);
    layout_->addWidget(qbrewedit_, 0, 2);
    layout_->addWidget(widgetstylelabel_, 1, 0, Qt::AlignRight);
    layout_->addWidget(widgetstylebox_, 1, 2);
    layout_->addMultiCellWidget(iconcheck_, 2, 2, 0, 2); 
    layout_->addMultiCellWidget(separatorone_, 3, 3, 0, 2);
    layout_->addWidget(unitslabel_, 4, 0, Qt::AlignRight);
    layout_->addWidget(unitsbox_, 4, 2);
    layout_->addWidget(batchlabel_, 5, 0, Qt::AlignRight);
    layout_->addWidget(batchspin_, 5, 2);
    layout_->addWidget(efficiencylabel_, 6, 0, Qt::AlignRight);
    layout_->addWidget(efficiencyspin_, 6, 2);
    layout_->addWidget(recipestylelabel_, 7, 0, Qt::AlignRight);
    layout_->addWidget(recipestylebox_, 7, 2);
    // TODO: use a label so it will look aligned
    layout_->addMultiCellWidget(tinsethcheck_, 8, 8, 0, 2); 
    layout_->addMultiCellWidget(separatortwo_, 9, 9, 0, 2);
    layout_->addWidget(cancelbutton_, 10, 0, QLabel::AlignRight);
    layout_->addWidget(okbutton_, 10, 2, QLabel::AlignRight);

#if QT_VERSION < 300
	widgetstylebox_->setEnabled(false);
#else
    widgetstylebox_->clear();
#endif
    recipestylebox_->clear();
    batchspin_->setSuffix(" " + Model::instance()->defaultSize().unit().symbol());
    setCaption(AppResource::ID_TITLE + " Setup");

    // set up connections
    connect(okbutton_, SIGNAL(clicked()), SLOT(accept()));
    connect(cancelbutton_, SIGNAL(clicked()), SLOT(reject()));
    connect(unitsbox_, SIGNAL(activated(const QString&)),
            this, SLOT(convertBatchSpin(const QString&)));
}

SetupDialog::~SetupDialog() { ; }

//////////////////////////////////////////////////////////////////////////////
// setStyle()
// ----------
// Set the Style combobox

void SetupDialog::setWidgetStyle(const QString &style)
{
#if QT_VERSION > 300
    int item;
    // search combo box for widget style string
    for (item=widgetstylebox_->count()-1; item!=0; --item) {
        if (widgetstylebox_->text(item) == style)
            break;
    }
    // if item not found, will set it to the first one
    widgetstylebox_->setCurrentItem(item);
#endif
}

void SetupDialog::setRecipeStyle(const QString &style)
{
    int item;
    // search combo box for recipe style string
    for (item=recipestylebox_->count()-1; item!=0; --item) {
        if (recipestylebox_->text(item) == style)
            break;
    }
    // if item not found, will set it to the first one
    recipestylebox_->setCurrentItem(item);
}

void SetupDialog::setUnits(const QString &units)
{
    int item;
    // search combo box for units string
    for (item=unitsbox_->count()-1; item!=0; --item) {
        if (unitsbox_->text(item) == units)
            break;
    }
    // if item not found, will set it to the first one
    unitsbox_->setCurrentItem(item);
}

void SetupDialog::setAppDir(const QString &qdir) 
    { qbrewedit_->setText(qdir); }

const QString SetupDialog::appDir() 
    { return qbrewedit_->text(); }

const QString SetupDialog::widgetStyle()
{
#if QT_VERSION > 300
    return widgetstylebox_->currentText();
#else
    return ID_PREF_WIDGET_STYLE_DEFAULT;
#endif
}

void SetupDialog::setBatch(const double batch)
    { batchspin_->setValue(batch); }

double SetupDialog::batch()
    { return batchspin_->value(); }

void SetupDialog::setEfficiency(const double eff)
    { efficiencyspin_->setValue(eff); }

double SetupDialog::efficiency()
    { return efficiencyspin_->value(); }

const QString SetupDialog::recipeStyle()
    { return recipestylebox_->currentText(); }

const QString SetupDialog::units()
    { return unitsbox_->currentText(); }

void SetupDialog::setTinseth(const bool tinseth)
    { tinsethcheck_->setChecked(tinseth); }

bool SetupDialog::tinseth()
    { return tinsethcheck_->isChecked(); }

void SetupDialog::addUnits(const QString &unit)
    { unitsbox_->insertItem(unit); }

void SetupDialog::setRecipeStyleBox(const QStringList &styles)
    { recipestylebox_->insertStringList(styles); }

void SetupDialog::setWidgetStyleBox(const QStringList &styles)
{
#if QT_VERSION > 300
widgetstylebox_->insertStringList(styles);
#endif
}

void SetupDialog::convertBatchSpin(const QString &selection)
{
    if (selection == AppResource::UNIT_US)
        batchspin_->setSuffix(" " + Volume::gallon.symbol());
    else if (selection == AppResource::UNIT_METRIC)
        batchspin_->setSuffix(" " + Volume::liter.symbol());
    else
        batchspin_->setSuffix("");
}

void SetupDialog::setLargeIcons(bool large)
    { iconcheck_->setChecked(large); }

bool SetupDialog::largeIcons()
    { return iconcheck_->isChecked(); }
