/***************************************************************************
  calc.h
  -------------------
  Calculations for brewing model
  -------------------
  Copyright (c) 2001-2005 David Johnson
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ***************************************************************************/

#ifndef CALC_H
#define CALC_H

#include <qstring.h>

#include "recipe.h"

namespace CalcResource {

// previous compatible qbrewdata version
const QString CALC_PREVIOUS        = "0.3.0";

//////////////////////////////////////////////////////////////////////////////
// File Names

const QString ID_CALC_FILE          = "qbrewdata";

//////////////////////////////////////////////////////////////////////////////
// XML Format Strings

const QString tagRecipe             = "recipe";
const QString attrGenerator         = "generator";
const QString tagTitle              = "title";
const QString tagBrewer             = "brewer";
const QString tagBatch              = "batch";
const QString attrMash              = "mash";
const QString tagNotes              = "notes";
const QString attrClass             = "class";
const QString classRecipe           = tagRecipe;
const QString classBatch            = tagBatch;

const QString tagDoc                = "qbrewdata";
const QString attrVersion           = "version";
const QString tagStyles             = "styles";
const QString tagStyle              = "style";
const QString attrOGLow             = "oglow";
const QString attrOGHigh            = "oghigh";
const QString attrFGLow             = "fglow";
const QString attrFGHigh            = "fghigh";
const QString attrIBULow            = "ibulow";
const QString attrIBUHigh           = "ibuhigh";
const QString attrSRMLow            = "srmlow";
const QString attrSRMHigh           = "srmhigh";
const QString tagGrains             = "grains";
const QString tagGrain              = "grain";
const QString attrQuantity          = "quantity";
const QString attrExtract           = "extract";
const QString attrColor             = "color";
const QString attrUse               = "use";
const QString tagHops               = "hops";
const QString tagHop                = "hop";
const QString attrForm              = "form";
const QString attrAlpha             = "alpha";
const QString attrTime              = "time";
const QString tagMiscIngredients    = "miscingredients";
const QString tagMiscIngredient     = "miscingredient";
const QString attrNotes             = "notes";
const QString tagUtilization        = "utilization";
const QString tagEntry              = "entry";
const QString attrUtil              = "util";

// BeerXML Format Strings

// TODO: see if I can use lowercase, converting on the fly during import/export
const QString beerXMLVersion        = "1";
const QString tagRECIPES            = "RECIPES";
const QString tagRECIPE             = "RECIPE";
const QString tagNAME               = "NAME";
const QString tagVERSION            = "VERSION";
const QString tagTYPE               = "TYPE";
const QString tagSTYLE              = "STYLE";
const QString tagBREWER             = "BREWER";
const QString tagBATCHSIZE          = "BATCH_SIZE";
const QString tagEFFICIENCY         = "EFFICIENCY";
const QString tagFERMENTABLES       = "FERMENTABLES";
const QString tagFERMENTABLE        = "FERMENTABLE";
const QString tagHOPS               = "HOPS";
const QString tagHOP                = "HOP";
const QString tagYEASTS             = "YEASTS";
const QString tagYEAST              = "YEAST";
const QString tagMISCS              = "MISCS";
const QString tagMISC               = "MISC";
const QString tagNOTES              = "NOTES";
const QString tagWATERS             = "WATERS";

const QString tagOGMIN              = "OG_MIN";
const QString tagOGMAX              = "OG_MAX";
const QString tagFGMIN              = "FG_MIN";
const QString tagFGMAX              = "FG_MAX";
const QString tagIBUMIN             = "IBU_MIN";
const QString tagIBUMAX             = "IBU_MAX";
const QString tagCOLORMIN           = "COLOR_MIN";
const QString tagCOLORMAX           = "COLOR_MAX";

const QString tagAMOUNT             = "AMOUNT";
const QString tagYIELD              = "YIELD";
const QString tagCOLOR              = "COLOR";

const QString tagALPHA              = "ALPHA";
const QString tagUSE                = "USE";
const QString tagTIME               = "TIME";
const QString tagFORM               = "FORM";

const QString TYPE_EXTRACT          = "Extract";
const QString TYPE_PARTIAL          = "Partial Mash";
const QString TYPE_ALLGRAIN         = "All Grain";

} // CalcResource
 
// utilization table entry structure
struct UEntry {
    unsigned time;
    unsigned utilization;
};

// Calculations
class Calc {
 public:
    // recalculate all values
    static void recalc(Recipe *r);

    // set efficiency
    static void setEfficiency(double e);
    static double efficiency();
    // set ibu calc method
    static void setTinseth(bool tinseth);
    // set srm calc method
    static void setMorey(bool morey);

    // return original gravity
    static double OG(Recipe *r);
    // return estimated final gravity
    static double FGEstimate(Recipe *r);
    // return alcohol %v
    static double ABV(Recipe *r);
    static double ABV(double og, double fg);
    // return alcohol %w
    static double ABW(Recipe *r);
    static double ABW(double og, double fg);
    // return bitterness
    static int IBU(Recipe *r);
    // return color
    static int SRM(Recipe *r);

    // convert Specific Gravity to Plato
    static double Calc::SgToP(double sg);
    // convert grain extract to yield
    static double Calc::extractToYield(double extract);
    static double Calc::yieldToExtract(double yield);

    // add an entry to the utilization table
    static void addUEntry(const UEntry &u);
    // get the table of utilizations
    static const QValueList<UEntry> &Calc::getUEntryList();

 private:
    static double calcOG(Recipe *r);
    static int calcSRM(Recipe *r);
    static int calcIBU(Recipe *r);
    static int calcRagerIBU(Recipe *r);
    static int calcTinsethIBU(Recipe *r);
    static double utilization(unsigned time);

 private:
    static QValueList<UEntry> utable_;
    static double efficiency_;
    static bool tinseth_;
    static bool morey_;
};

// Inlined Methods //////////////////////////////////////////////////////////

inline void Calc::setEfficiency(double e) { efficiency_ = e; }

inline double Calc::efficiency() { return efficiency_; }

inline void Calc::setTinseth(bool tinseth) { tinseth_ = tinseth; }

inline void Calc::setMorey(bool morey) { morey_ = morey; }

inline double Calc::OG(Recipe *r) { return r->og_; }

inline int Calc::IBU(Recipe *r) { return r->ibu_; }

inline int Calc::SRM(Recipe *r) { return r->srm_; }

inline const QValueList<UEntry> &Calc::getUEntryList() { return utable_; }

#endif // CALC_H
