/***************************************************************************
  databasetool.cpp
  -------------------
  Database editor for QBrew
  -------------------
  Copyright 2005, David Johnson
  Please see the header file for copyright and license information
 ***************************************************************************/

#include <qaction.h>
#include <qdir.h>
#include <qimage.h>
#include <qmenubar.h>
#include <qmessagebox.h>
#include <qpopupmenu.h>
#include <qtabwidget.h>
#include <qtoolbar.h>

#include "graindb.h"
#include "hopdb.h"
#include "miscdb.h"
#include "styledb.h"

#include "controller.h"
#include "model.h"

#include "databasetool.h"

using namespace AppResource;
using namespace CalcResource;

//////////////////////////////////////////////////////////////////////////////
// Construction, Destruction                                                //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// DatabaseTool()
// --------------
// Constructor

DatabaseTool::DatabaseTool(QWidget* parent, const char* name, Model *model)
    : QMainWindow(parent, name, WDestructiveClose),
      graindb_(0), hopdb_(0), miscdb_(0), styledb_(0), model_(model)
{
    setCaption(ID_TITLE + " - Database Tool");

    // setup menu
    QPopupMenu* filemenu = new QPopupMenu(this);
    menuBar()->insertItem("&File", filemenu);

    // setup actions
    QAction *action;
    action = new QAction("Save", 
                         QIconSet(Controller::findImage("sm_filesave"),
                                  Controller::findImage("filesave")),
                         "&Save",  CTRL+Key_S, this);
    action->setStatusTip("Save the database");
    // TODO: implement enable/disable of save...
    //action->setEnabled(false);
    connect(action, SIGNAL(activated()), this, SLOT(fileSave()));
    action->addTo(filemenu);

    action = new QAction("Close",
                         QIconSet(Controller::findImage("sm_exit"),
                                  Controller::findImage("exit")),
                         "&Close", CTRL+Key_Q, this);
    action->setStatusTip("Close the database tool");
    connect(action, SIGNAL(activated()), this, SLOT(close()));
    filemenu->insertSeparator();
    action->addTo(filemenu);

    // create pages
    graindb_ = new GrainDB(this, "graindb_", model_);
    hopdb_ = new HopDB(this, "hopdb_", model_);
    miscdb_ = new MiscDB(this, "miscdb_", model_);
    styledb_ = new StyleDB(this, "styledb_", model_);

    QTabWidget *view = new QTabWidget(this, "databaseview");

    view->addTab(graindb_, "&Grains");
    view->addTab(hopdb_, "&Hops");
    view->addTab(miscdb_, "&Misc");
    view->addTab(styledb_, "&Styles");

    setCentralWidget(view);

    // initialize pages
    graindb_->refresh();
    hopdb_->refresh();
    miscdb_->refresh();
    styledb_->refresh();
}

DatabaseTool::~DatabaseTool() { ; }

//////////////////////////////////////////////////////////////////////////////
// Slot Implementations                                                     //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// fileSave()
// ----------
// save the database to the local database file (~/.qbrewdata)

void DatabaseTool::fileSave()
{
    QString localbase = QDIR_HOME + "/." + ID_CALC_FILE;

    QFileInfo finfo(localbase);
    if (finfo.exists() && !finfo.isWritable()) {
        // no write permission
        QMessageBox::warning(this, ID_TITLE,
                             QString("<p>Unable to save the database."
                                     "You do not have permission "
                                     "to write to %1").arg(localbase));
    } else {
        if (!model_->saveData(localbase)) {
            // error in saving file
            QMessageBox::warning(this, ID_TITLE, ID_TITLE +
                                 QString("<p>Unable to save the database."
                                         "Error in saving %1").arg(localbase));
        }
    }
}
