/***************************************************************************
  grain.cpp
  -------------------
  A grain class
  -------------------
  Copyright 1999-2005, David Johnson
  Please see the header file for copyright and license information
 ***************************************************************************/

#include <qregexp.h>
#include <qstring.h>
#include <qstringlist.h>

#include "resource.h"
#include "grain.h"

//////////////////////////////////////////////////////////////////////////////
// Grain Class                                                              //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// Grain()
// -------
// Constructor

Grain::Grain()
    : name_("Generic"), weight_(), extract_(0.0), color_(0.0),
      use_(GRAIN_OTHER), yield_(0.0), hcu_(0.0)
{ ; }

Grain::Grain(const QString &name, const Weight &weight, const double &extract,
             const double &color, const QString &use)
    : name_(name), weight_(weight), extract_(extract), color_(color),
      use_(useStringToEnum(use))
{
    yield_ = weight_.amount(Weight::pound) * (extract_ - 1.000);
    hcu_ = weight_.amount(Weight::pound) * color_;
}

Grain::~Grain() { ; }

//////////////////////////////////////////////////////////////////////////////
// Grain(const Grain &)
// --------------------
// Copy Constructor

Grain::Grain(const Grain &g)
    : name_(g.name_), weight_(g.weight_), extract_(g.extract_),
      color_(g.color_), use_(g.use_), yield_(g.yield_), hcu_(g.hcu_)
{ ; }

//////////////////////////////////////////////////////////////////////////////
// operator=()
// -----------
// Assignment operator

Grain Grain::operator=(const Grain &g)
{
    if (this != &g) {
        name_ = g.name_;
        weight_ = g.weight_;
        extract_ = g.extract_;
        color_ = g.color_;
        use_ = g.use_;
        yield_ = g.yield_;
        hcu_ = g.hcu_;
    }
    return *this;
}

//////////////////////////////////////////////////////////////////////////////
// operator==()
// ------------
// Equivalence operator

bool Grain::operator==(const Grain &g)
{
    return (name_ == g.name_)
        && (weight_ == g.weight_)
        && (extract_ == g.extract_)
        && (color_ == g.color_)
        && (use_ == g.use_);
}

///////////////////////////////////////////////////////////////////////////////
// useString()
// -----------
// Return usage as a string

QString Grain::useString() const
{
    switch (use_) {
        case GRAIN_EXTRACT: return "extract";
        case GRAIN_MASHED: return "mashed";
        case GRAIN_STEEPED:return "steeped";
    }
    return "other";
}

///////////////////////////////////////////////////////////////////////////////
// useStringList()
// -----------
// Return string list of usage types

QStringList Grain::useStringList()
{
    QStringList stemp = "extract";
    stemp += "mashed";
    stemp += "steeped";
    stemp += "other";
    return stemp;
}

///////////////////////////////////////////////////////////////////////////////
// useStringToEnum()
// -----------------
// Convert use string to enumeration type

int Grain::useStringToEnum(const QString &name)
{
    if ((name.lower() == "mashed") || (name.lower() == "mash"))
        return GRAIN_MASHED;
    if ((name.lower() == "steeped") || (name.lower() == "steep"))
        return GRAIN_STEEPED;
    if (name.lower() == "extract") return GRAIN_EXTRACT;
    return GRAIN_OTHER;
}

//////////////////////////////////////////////////////////////////////////////
// GrainItem class                                                          //
//////////////////////////////////////////////////////////////////////////////

GrainItem::GrainItem(QListView * parent) : QListViewItem(parent), git_(0) { ; }

GrainItem::~GrainItem() { ; }

void GrainItem::refresh()
{
    setText(GITEM_NAME, (*git_).name());
    setText(GITEM_QUANTITY,
            QString::number((*git_).weight().amount(Weight::pound), 'f', 2)
            + " " + (*git_).weight().unit().symbol());
    setText(GITEM_EXTRACT, QString::number((*git_).extract(), 'f', 3));
    setText(GITEM_COLOR, QString::number((*git_).color(), 'f', 1)
            + AppResource::CHAR_LATIN_DEGREE);
    setText(GITEM_USE, (*git_).useString());
}

// reimplemented for numerical sorting
int GrainItem::compare(QListViewItem* item, int column, bool asc) const
{
    switch (column) {
      case GITEM_QUANTITY:
      case GITEM_EXTRACT:
      case GITEM_COLOR: { // remove suffixes and decimals in these fields
          QRegExp regex("[0-9.]+");
          regex.search(text(column));
          int first = (int) regex.cap(0).toDouble() * 1000;
          regex.search(item->text(column));
          int second = (int) regex.cap(0).toDouble() * 1000;
          return (first - second);
      }

      default:
          return QListViewItem::compare(item, column, asc);
    }
}
