/***************************************************************************
  grainpage.cpp
  -------------------
  A dialog page for grains
  -------------------
  Copyright 1999-2005, David Johnson
  Please see the header file for copyright and license information
 ***************************************************************************/

#include <qcombobox.h>
#include <qimage.h>
#include <qlabel.h>
#include <qlistview.h>
#include <qpushbutton.h>
#include <qstringlist.h>

#include "fixedspinbox.h"

#include "resource.h"
#include "controller.h"
#include "model.h"
#include "view.h"

#include "grainpage.h"

using namespace AppResource;

//////////////////////////////////////////////////////////////////////////////
// Construction, Destruction                                                //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// GrainPage()
// -----------
// Constructor

GrainPage::GrainPage(QWidget *parent, const char *name, Model *model)
    : GrainView(parent, name), model_(model), currentitem_(0)
{
    // additional widget setup
    grainslistview->setColumnAlignment(GITEM_NAME, AlignLeft);
    grainslistview->setColumnAlignment(GITEM_QUANTITY, AlignRight);
    grainslistview->setColumnAlignment(GITEM_EXTRACT, AlignRight);
    grainslistview->setColumnAlignment(GITEM_COLOR, AlignRight);
    grainslistview->setColumnAlignment(GITEM_USE, AlignRight);

    quantitylabel->setBuddy(quantityspin);
    quantityspin->setMinimum(0.00);
    quantityspin->setMaximum(100.00);
    quantityspin->setSingleStep(0.25);
    quantityspin->setPrecision(2);

    extractlabel->setBuddy(extractspin);
    extractspin->setMinimum(1.000);
    extractspin->setMaximum(1.100);
    extractspin->setSingleStep(0.001);
    extractspin->setPrecision(3);

    colorlabel->setBuddy(colorspin);
    colorspin->setMinimum(0.0);
    colorspin->setMaximum(500.0);
    colorspin->setSingleStep(1.0);
    colorspin->setPrecision(1);
    colorspin->setSuffix(AppResource::CHAR_LATIN_DEGREE);

    usecombo->insertStringList(Grain::useStringList());

    addbutton->setPixmap(Controller::findImage("sm_add"));
    applybutton->setPixmap(Controller::findImage("sm_apply"));
    removebutton->setPixmap(Controller::findImage("sm_remove"));

    // adjust size of leftmost labels so they line up
    namelabel->adjustSize();
    int leftmost = namelabel->width();
    quantitylabel->adjustSize();
    leftmost = QMAX(leftmost, quantitylabel->width());
    quantitylabel->setMinimumWidth(leftmost);
    namelabel->setMinimumWidth(leftmost);

    // fixup tab orders
    QWidget::setTabOrder(namecombo, quantityspin);
    QWidget::setTabOrder(quantityspin, extractspin);
    QWidget::setTabOrder(extractspin, colorspin);
    QWidget::setTabOrder(colorspin, usecombo);

    // setup connections
    connect(addbutton, SIGNAL(clicked()), SLOT(addGrain()));
    connect(applybutton, SIGNAL(clicked()), SLOT(applyGrain()));
    connect(removebutton, SIGNAL(clicked()), SLOT(removeGrain()));
    connect(namecombo, SIGNAL(activated(const QString &)),
            SLOT(updateFields(const QString &)));
    connect(grainslistview, SIGNAL(selectionChanged(QListViewItem *)),
            SLOT(itemSelected(QListViewItem *)));
}

GrainPage::~GrainPage() {}

//////////////////////////////////////////////////////////////////////////////
// Implementation                                                           //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// refresh()
// ---------
// initialize list and fields (new/opened document)

void GrainPage::refresh()
{
    // initialize list
    GrainIterator it;
    GrainItem *item;
    GrainList *list  = model_->recipe()->grains();
    grainslistview->clear();
    for (it=list->begin(); it != list->end(); ++it) {
        item = new GrainItem(grainslistview);
        item->setText(GITEM_NAME,
                      (*it).name());
        item->setText(GITEM_QUANTITY,
                      QString::number((*it).weight().amount(),'f',2)
                      + " " + (*it).weight().unit().symbol());
        item->setText(GITEM_EXTRACT,
                      QString::number((*it).extract(),'f',3));
        item->setText(GITEM_COLOR,
                      QString::number((*it).color(),'f',1)+CHAR_LATIN_DEGREE);
        item->setText(GITEM_USE,
                      (*it).useString());
        item->setIterator(it);
    }

    // initialize fields
    namecombo->clear();
    namecombo->insertStringList(model_->grainsList());
    updateFields(namecombo->currentText());
}

//////////////////////////////////////////////////////////////////////////////
// updateFields()
// --------------
// Update the fields in the entry area

void GrainPage::updateFields(const QString &name)
{
    Grain *thisgrain = model_->grain(name);
    if (thisgrain) { // sanity check
        quantityspin->setValue(thisgrain->weight().amount());
        quantityspin->setSuffix(" " + thisgrain->weight().unit().symbol());
        extractspin->setValue(thisgrain->extract());
        colorspin->setValue(thisgrain->color());

        // vary default use according to recipe type
        int utemp = thisgrain->use();
        QString ustr = thisgrain->useString();
        if (model_->recipe()->mashed()) {
            if (utemp == GRAIN_STEEPED) ustr = "mashed";
        } else {
            if (utemp == GRAIN_MASHED) ustr = "steeped";
        }
        View::setComboItem(usecombo, ustr);
    }
}

//////////////////////////////////////////////////////////////////////////////
// addGrain()
// ----------
// Add a grain to the recipe

// TODO: use selectedItem()/currentItem() instead of currentitem_?
void GrainPage::addGrain()
{
    GrainIterator it;
    GrainItem *item;

    Grain grain(namecombo->currentText(),
                Weight(quantityspin->value(), model_->defaultGrainUnit()),
                extractspin->value(), colorspin->value(),
                usecombo->currentText());

    // add grain to database if it doesn't exist
    model_->addGrain(grain, false);

    // create new listview item
    item = new GrainItem(grainslistview);
    item->setText(GITEM_NAME, grain.name());
    item->setText(GITEM_QUANTITY,
                  QString::number(grain.weight().amount(),'f',2)
                  + " " + grain.weight().unit().symbol());
    item->setText(GITEM_EXTRACT,
                  QString::number(grain.extract(),'f',3));
    item->setText(GITEM_COLOR,
                  QString::number(grain.color(),'f',1)+CHAR_LATIN_DEGREE);
    item->setText(GITEM_USE, grain.useString());

    // add grain to recipe
    it = model_->recipe()->addGrain(grain);
    item->setIterator(it);
    model_->setModified(true);
    emit modified();

    grainslistview->ensureItemVisible(item);
    grainslistview->setCurrentItem(item);
}

//////////////////////////////////////////////////////////////////////////////
// applyGrain()
// ------------
// Apply a grain to the recipe

void GrainPage::applyGrain()
{
    if (currentitem_ != 0) {
        Grain grain(namecombo->currentText(),
                    Weight(quantityspin->value(), model_->defaultGrainUnit()),
                    extractspin->value(), colorspin->value(), 
                    usecombo->currentText());
        // add grain to database if it doesn't exist
        model_->addGrain(grain, false);

        // change the grain in the recipe
        model_->recipe()->applyGrain(currentitem_->iterator(), grain);
        model_->setModified(true);
        currentitem_->refresh();

        emit modified();
    }
}

//////////////////////////////////////////////////////////////////////////////
// removeGrain()
// -------------
// Remove a grain from the recipe

void GrainPage::removeGrain()
{
    if (currentitem_ != 0) {
        // remove grain from recipe
        model_->recipe()->removeGrain(currentitem_->iterator());
        model_->setModified(true);
        delete currentitem_; currentitem_ = 0;

        emit modified();
    }
}

//////////////////////////////////////////////////////////////////////////////
// itemSelected()
// --------------
// A grain is selected in the listview

void GrainPage::itemSelected(QListViewItem *item)
{
    if (item) {
        currentitem_ = dynamic_cast<GrainItem*>(item);
        if (currentitem_ == 0) {
            qWarning("qbrew: bad dynamic cast, line" + __LINE__);
            return;
        }
        namecombo->setEditText(item->text(GITEM_NAME));
        quantityspin->setValue(item->text(GITEM_QUANTITY));
        extractspin->setValue(item->text(GITEM_EXTRACT));
        colorspin->setValue(item->text(GITEM_COLOR));
        View::setComboItem(usecombo, item->text(GITEM_USE));
    }
}
