/***************************************************************************
  helpwindow.cpp
  -------------------
  General purpose help file viewer
  -------------------
  Copyright (c) 1999-2005 David Johnson
  Please see the header file for copyright and license information.
***************************************************************************/

#include <qaction.h>
#include <qapplication.h>
#include <qdir.h>
#include <qiconset.h>
#include <qimage.h>
#include <qmenubar.h>
#include <qpaintdevicemetrics.h>
#include <qpopupmenu.h>
#include <qpainter.h>
#include <qprinter.h>
#include <qtextbrowser.h>
#include <qtoolbar.h>
#include <qtoolbutton.h>
#include <qsimplerichtext.h>
#include <qstatusbar.h>
#include <qstylesheet.h>

#include "resource.h"
#include "controller.h"
#include "helpwindow.h"

using namespace AppResource;

//////////////////////////////////////////////////////////////////////////////
// Construction                                                             //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// HelpWindow()
// ------------
// Constructor

HelpWindow::HelpWindow(const QString& home, QWidget* parent, const char *name)
    : QMainWindow(parent, name, WDestructiveClose), browser_(0), printer_(0)
{
    setUsesBigPixmaps(Controller::instance()->state().general.largeicons);
    QIconSet::setIconSize(QIconSet::Large, QSize(22, 22));

    browser_ = new QTextBrowser(this);
    browser_->mimeSourceFactory()->addFilePath(".");
    browser_->mimeSourceFactory()->addFilePath(QFileInfo(home).dirPath(true));
    browser_->mimeSourceFactory()->addFilePath(qApp->applicationDirPath());
    browser_->setFrameStyle(QFrame::Panel | QFrame::Sunken);
    if (!home.isEmpty()) browser_->setSource(home);
    textChanged();
    setCentralWidget(browser_);

    // set a reasonable width
    int width = 480;
    if ((width * 3) > (QApplication::desktop()->width() * 2))
        width = QApplication::desktop()->width() * 2 / 3;
    else if ((width * 2) < (QApplication::desktop()->width()))
        width = QApplication::desktop()->width() / 2;
    // set a reasonable height
    int height = QApplication::desktop()->height() * 3 / 4;
    resize(width, height);

    // setup menus and toolbar
    QPopupMenu* filemenu = new QPopupMenu(this);
    QPopupMenu* navmenu = new QPopupMenu(this);
    menuBar()->insertItem("&File", filemenu);
    menuBar()->insertItem("&Navigate", navmenu);
    QToolBar* toolbar = new QToolBar(this);

    // setup actions
    QAction *action;
    action = new QAction("Back",
                         QIconSet(Controller::findImage("sm_back"),
                                  Controller::findImage("back")),
                         "&Backward", ALT+Key_Left, this);
    action->setStatusTip("Go to the previous page");
    connect(action, SIGNAL(activated()), browser_, SLOT(backward()));
    connect(browser_, SIGNAL(backwardAvailable(bool)),
            action, SLOT(setEnabled(bool)));
    action->setEnabled(false);
    action->addTo(navmenu);
    action->addTo(toolbar);

    action = new QAction("Forward",
                         QIconSet(Controller::findImage("sm_forward"),
                                  Controller::findImage("forward")),
                         "&Forward", ALT+Key_Right, this);
    action->setStatusTip("Go to the next page");
    connect(action, SIGNAL(activated()), browser_, SLOT(forward()));
    connect(browser_, SIGNAL(forwardAvailable(bool)),
            action, SLOT(setEnabled(bool)));
    action->addTo(navmenu);
    action->setEnabled(false);
    action->addTo(toolbar);

    action = new QAction("Home",
                         QIconSet(Controller::findImage("sm_gohome"),
                                  Controller::findImage("gohome")),
                         "&Home", CTRL+Key_Home, this);
    action->setStatusTip("Go to the index page");
    connect(action, SIGNAL(activated()), browser_, SLOT(home()));
    action->addTo(navmenu);
    action->addTo(toolbar);

    action = new QAction("Print",
                         QIconSet(Controller::findImage("sm_fileprint"),
                                  Controller::findImage("fileprint")),
                         "&Print...", CTRL+Key_P, this);
    action->setStatusTip("Print the document");
    connect(action, SIGNAL(activated()), this, SLOT(print()));
    action->addTo(filemenu);
    toolbar->addSeparator();
    action->addTo(toolbar);

    action = new QAction("Close",
                         QIconSet(Controller::findImage("sm_exit"),
                                  Controller::findImage("exit")),
                         "&Close", CTRL+Key_Q, this);
    action->setStatusTip("Close the help viewer");
    connect(action, SIGNAL(activated()), this, SLOT(close()));
    filemenu->insertSeparator();
    action->addTo(filemenu);
    action->addTo(toolbar);

    // setup connections
    connect(browser_, SIGNAL(textChanged()),
            this, SLOT(textChanged()));
    connect(browser_, SIGNAL(highlighted(const QString&)), statusBar(),
            SLOT(message(const QString&)));

    browser_->setFocus();
}

//////////////////////////////////////////////////////////////////////////////
// ~HelpWindow()
// -------------
// Destructor

HelpWindow::~HelpWindow()
{
    if (browser_) delete browser_;
    if (printer_) delete printer_;
}

//////////////////////////////////////////////////////////////////////////////
// textChanged()
// -------------
// The displayed document has changed, so change caption

void HelpWindow::textChanged()
{
    if (browser_->documentTitle().isNull())
        setCaption(ID_TITLE +  " Help - " + browser_->context());
    else
        setCaption(ID_TITLE + " Help - " + browser_->documentTitle());
}

//////////////////////////////////////////////////////////////////////////////
// print()
// -------
// Print the contents of the help window

void HelpWindow::print()
{
    if (!printer_) printer_ = new QPrinter(QPrinter::HighResolution);
    printer_->setFullPage(true);
    if (printer_->setup(this)) {
        QPainter painter(printer_);
	if (!painter.isActive()) return;
        QPaintDeviceMetrics metrics(painter.device());

        int dpix = metrics.logicalDpiX();
        int dpiy = metrics.logicalDpiY();
        const int marginx = (int) (dpix * 2 / 2.54); // two cm margins
        const int marginy = (int) (dpiy * 2 / 2.54); 

        QFont font("sans", 10);
	QRect body(marginx, marginy,
                   metrics.width() - (2 * marginx),
                   metrics.height() - (2 * marginx));
	QSimpleRichText richtext(browser_->text(),
                                 font,
                                 browser_->context(),
                                 browser_->styleSheet(),
				 browser_->mimeSourceFactory(),
                                 body.height());
	richtext.setWidth(&painter, body.width() );

	int page = 1;
	while (true) {
	    richtext.draw(&painter, marginx, marginy, body, colorGroup());
	    body.moveBy(0, body.height());
	    painter.translate(0, -body.height());
	    painter.drawText(body.right()
                             - painter.fontMetrics().width(QString::number(page)),
                             body.bottom() + painter.fontMetrics().ascent() + 5, 
                             QString::number(page));
	    if (body.top()-marginy >= richtext.height())
		break;
	    printer_->newPage();
	    page++;
	}
    }
}
