/***************************************************************************
  hop.cpp
  -------------------
  A hop class
  -------------------
  Copyright 1999-2005, David Johnson
  Please see the header file for copyright and license information
 ***************************************************************************/

#include <qregexp.h>
#include <qstringlist.h>

#include "hops.h"

//////////////////////////////////////////////////////////////////////////////
// Hop                                                                     //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// Hop()
// ------
// constructor

Hop::Hop()
    : name_("Generic"), weight_(), form_( QString::null), alpha_(0.0), time_(0)
{ ; }

Hop::Hop(const QString &name, const Weight &weight, const QString &form,
         const double &alpha, const unsigned &time)
    : name_(name), weight_(weight), form_(form), alpha_(alpha), time_(time)
      
{
    hbu_ = 0.75 * alpha * weight_.amount(Weight::ounce);
}

Hop::~Hop() { ; }

//////////////////////////////////////////////////////////////////////////////
// Hop(const Hop &)
// ----------------
// Copy constructor

Hop::Hop(const Hop &h)
    : name_(h.name_), weight_(h.weight_), form_(h.form_), alpha_(h.alpha_),
      time_(h.time_), hbu_(h.hbu_)
{ ; }

//////////////////////////////////////////////////////////////////////////////
// operator=()
// -----------
// Assignment operator

Hop Hop::operator=(const Hop &h)
{
    if (this != &h) {
        name_ = h.name_;
        weight_ = h.weight_;
        form_ = h.form_;
        alpha_ = h.alpha_;
        time_ = h.time_;
        hbu_ = h.hbu_;
    }
    return *this;
}

//////////////////////////////////////////////////////////////////////////////
// operator==()
// ------------
// Equivalence operator

bool Hop::operator==(const Hop &h)
{
    return (name_ == h.name_)
        && (weight_ == h.weight_)
        && (form_ == h.form_)
        && (alpha_ == h.alpha_)
        && (time_ == h.time_);
}

//////////////////////////////////////////////////////////////////////////////
// HopItem Class                                                            //
//////////////////////////////////////////////////////////////////////////////

HopItem::HopItem(QListView * parent) : QListViewItem(parent), hit_(0) { ; }

HopItem::~HopItem() { ; }

void HopItem::refresh()
{
    setText(HITEM_NAME, (*hit_).name());
    setText(HITEM_QUANTITY,
            QString::number((*hit_).weight().amount(), 'f', 2)
            + " " + (*hit_).weight().unit().symbol());
    setText(HITEM_ALPHA, QString::number((*hit_).alpha(), 'f', 1) + '%');
    setText(HITEM_FORM, (*hit_).form());
    setText(HITEM_TIME, QString::number((*hit_).time()));
}

// reimplemented for numerical sorting
int HopItem::compare(QListViewItem* item, int column, bool asc) const
{
    switch (column) {
      case HITEM_QUANTITY:
      case HITEM_ALPHA: { // remove suffixes and decimals in these fields
          QRegExp regex("[0-9.]+");
          regex.search(text(column));
          int first = (int) regex.cap(0).toDouble() * 1000;
          regex.search(item->text(column));
          int second = (int) regex.cap(0).toDouble() * 1000;
          return (first - second);
      }

      case HITEM_TIME: // plain int with no suffix
          return (text(column).toInt() - item->text(column).toInt());

      default:
          return QListViewItem::compare(item, column, asc);
    }
}
