/***************************************************************************
  hydrometertool.h
  -------------------
  A hydrometer correction utility for QBrew
  -------------------
  Copyright 2004-2005, David Johnson <david@usermode.org>
  Based on code Copyright 2001, Abe Kabakoff <abe_kabakoff@gmx.de>
  Please see the header file for copyright and license information
 ***************************************************************************/

#include <qlabel.h>
#include <qpushbutton.h>
#include <qwhatsthis.h>

#include "fixedspinbox.h"
#include "resource.h"

#include "hydrometertool.h"

//////////////////////////////////////////////////////////////////////////////
// Constants

const double BOILING = 212.0;
const double FREEZING = 32.0;
const double TEMP_STEP = 0.2;
const double MINSG = 0.850;
const double MAXSG = 1.200;
const double SG_STEP = 0.001;

//////////////////////////////////////////////////////////////////////////////
// HydrometerTool()
// ----------------
// Constructor

HydrometerTool:: HydrometerTool(QWidget* parent, const char* name)
    : HydrometerDialog(parent, name, false, WDestructiveClose)
{
    // TODO: use appropriate units (F or C)
    QString SUFFIX(AppResource::CHAR_LATIN_DEGREE);
    SUFFIX += " F";

    setCaption(AppResource::ID_TITLE + " - Hydrometer Tool");

    // additional setup for the spinboxes
    calibratedlabel->setBuddy(calibrated);
    calibrated->setMinimum(FREEZING);
    calibrated->setMaximum(BOILING);
    calibrated->setSingleStep(TEMP_STEP);
    calibrated->setPrecision(1);
    calibrated->setValue(60.0);
    calibrated->setSuffix(SUFFIX);
    QWhatsThis::add(calibrated,
                    "Enter the temperature your hydrometer is calibrated for");

    samplelabel->setBuddy(sample);
    sample->setMinimum(FREEZING);
    sample->setMaximum(BOILING);
    sample->setSingleStep(TEMP_STEP);
    sample->setPrecision(1);
    sample->setValue(60.0);
    sample->setSuffix(SUFFIX);
    QWhatsThis::add(sample, "Enter the temperature of your sample");

    readinglabel->setBuddy(reading);
    reading->setMinimum(MINSG);
    reading->setMaximum(MAXSG);
    reading->setSingleStep(SG_STEP);
    reading->setPrecision(3);
    reading->setValue(1.000);
    QWhatsThis::add(reading, "Enter the reading obtained from your hydrometer");


    corrected->setText("<strong>1.000</strong>");

    // connections
    connect(sample, SIGNAL(valueChanged(double)), this, SLOT(recalc()));
    connect(calibrated, SIGNAL(valueChanged(double)), this, SLOT(recalc()));
    connect(reading, SIGNAL(valueChanged(double)), this, SLOT(recalc()));

    recalc();
}

//////////////////////////////////////////////////////////////////////////////
// recalc()
// --------
// the signal to calculate the actual SG

void HydrometerTool::recalc()
{
    // see http://www.hbd.org/brewery/library/HydromCorr0992.html

    const double COEFF1 = 1.313454;
    const double COEFF2 = 0.132674;
    const double COEFF3 = 2.057793E-3;
    const double COEFF4 = 2.627634E-6;

    double calterm;   // E is the correction factor for the calibration temp
    double corr;      // the corrected reading
    double temp = sample->value();
    double cal = calibrated->value();
    double read = reading->value();

    // Correction(@59F) = 1.313454 - 0.132674*T + 2.057793e-3*T**2 - 2.627634e-6*T**3
    // the equation is set up for 59F (15C)
    calterm = (COEFF1
               - (COEFF2 * cal)
               + (COEFF3 * (cal*cal))
               - (COEFF4 * (cal*cal*cal)));
    corr = (COEFF1
            - (COEFF2 * temp)
            + (COEFF3 * (temp*temp))
            - (COEFF4 * (temp*temp*temp))) - calterm;

    corr /= 1000.0;
    corr = 1.0 - corr;
    corr = read / corr;

    // now that we have the value change the reading
    corrected->setText("<strong>" + QString::number(corr, 'f', 3) + "<strong>");
}
