/***************************************************************************
  miscdb.cpp
  -------------------
  A dialog page for misc ingredients
  -------------------
  Copyright 2005, David Johnson
  Please see the header file for copyright and license information
 ***************************************************************************/

#include <qcombobox.h>
#include <qimage.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qlistview.h>
#include <qmessagebox.h>
#include <qpushbutton.h>
#include <qstringlist.h>

#include "fixedspinbox.h"
#include "controller.h"
#include "model.h"

#include "miscdb.h"

using AppResource::ID_TITLE;

// Note that this class is very similar to the GrainPage class. This is
// because they both implement very similar interfaces. They are separate
// for maintenance reasons.

//////////////////////////////////////////////////////////////////////////////
// Construction, Destruction                                                //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// MiscDB()
// --------
// Constructor

MiscDB::MiscDB(QWidget *parent, const char *name, Model *model)
    : MiscView(parent, name), model_(model), currentitem_(0)
{
    // additional widget setup
    misclistview->setColumnAlignment(MITEM_NAME, AlignLeft);
    misclistview->setColumnAlignment(MITEM_NOTES, AlignLeft);

    addbutton->setPixmap(Controller::findImage("sm_add"));
    applybutton->setPixmap(Controller::findImage("sm_apply"));
    removebutton->setPixmap(Controller::findImage("sm_remove"));

    // hide quantity in database editor
    misclistview->setColumnWidthMode(MITEM_QUANTITY, QListView::Manual);
    misclistview->hideColumn(MITEM_QUANTITY);
    quantitylabel->hide();
    quantityspin->hide();

    // adjust size of leftmost labels so they line up
    namelabel->adjustSize();
    int leftmost = namelabel->width();
    noteslabel->adjustSize();
    leftmost = QMAX(leftmost, noteslabel->width());
    noteslabel->setMinimumWidth(leftmost);
    namelabel->setMinimumWidth(leftmost);

    // fixup tab orders
    QWidget::setTabOrder(namecombo, quantityspin);
    QWidget::setTabOrder(quantityspin, notesedit);

    // setup connections
    connect(addbutton, SIGNAL(clicked()), SLOT(addMiscIngredient()));
    connect(applybutton, SIGNAL(clicked()), SLOT(applyMiscIngredient()));
    connect(removebutton, SIGNAL(clicked()), SLOT(removeMiscIngredient()));
    connect(namecombo, SIGNAL(activated(const QString &)),
            SLOT(updateFields(const QString &)));
    connect(misclistview, SIGNAL(selectionChanged(QListViewItem *)),
            SLOT(itemSelected(QListViewItem *)));
}

MiscDB::~MiscDB() { ; }

//////////////////////////////////////////////////////////////////////////////
// Implementation                                                           //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// refresh()
// ---------
// initialize list and fields

void MiscDB::refresh()
{
    // update list
    MiscIngredientIterator it;
    MiscItem *item;
    MiscIngredientList *list = model_->miscDB();
    misclistview->clear();
    for (it=list->begin(); it != list->end(); ++it) {
        item = new MiscItem(misclistview);
        item->setText(MITEM_NAME, (*it).name());
        item->setText(MITEM_NOTES, (*it).notes());
        item->setIterator(it);
    }

    // update fields
    namecombo->clear();
    namecombo->insertStringList(model_->miscList());
    updateFields(namecombo->currentText());
}

//////////////////////////////////////////////////////////////////////////////
// updateFields()
// ------------------
// Update the fields in the entry area

void MiscDB::updateFields(const QString &name)
{
    MiscIngredient *thismisc = model_->misc(name);
    if (thismisc) { // sanity check
        notesedit->setText(thismisc->notes());
    }
}

//////////////////////////////////////////////////////////////////////////////
// addMiscIngredient()
// -----------------------
// adds a misc ingredient to recipe

void MiscDB::addMiscIngredient()
{
    MiscIngredientIterator it;
    MiscItem *item;

    MiscIngredient misc(namecombo->currentText(),
                        Quantity(quantityspin->value(),
                                 model_->defaultMiscUnit()),
                        notesedit->text());

    if (model_->checkMisc(misc.name())) {
        // already exists, query user
        if (QMessageBox::information(this, ID_TITLE + " - Replace?",
                                     QString("<p>Are you sure you want to "
                                             "replace the entry \"%1\"")
                                     .arg(misc.name()),
                                     QMessageBox::No | QMessageBox::Escape,
                                     QMessageBox::Yes)
            == QMessageBox::No) return;

        // remove existing item from listview
        item = (MiscItem*)misclistview->findItem(misc.name(), MITEM_NAME);
        if (item) {
            misclistview->takeItem(item);
            delete item;
        }
    }

    // create new listview item
    item = new MiscItem(misclistview);
    item->setText(MITEM_NAME,  misc.name());
    item->setText(MITEM_NOTES, misc.notes());

    // add ingredient to recipe
    it = model_->addMisc(misc, true);
    item->setIterator(it);

    misclistview->ensureItemVisible(item);
    misclistview->setCurrentItem(item);
}

//////////////////////////////////////////////////////////////////////////////
// applyMiscIngredient()
// -------------------------
// applies a misc ingredient to recipe

void MiscDB::applyMiscIngredient()
{
    if (currentitem_ != 0) {
        MiscIngredient misc(namecombo->currentText(),
                            Quantity(quantityspin->value(),
                                     model_->defaultMiscUnit()),
                            notesedit->text());
        // apply ingredient (overwrite is simplest way)
        MiscIngredientIterator it = model_->addMisc(misc, true);
        currentitem_->setIterator(it);
        currentitem_->refresh();
     }
}

//////////////////////////////////////////////////////////////////////////////
// removeMiscIngredient()
// --------------------------
// rremoves a misc ingredient from recipe

void MiscDB::removeMiscIngredient()
{
    if (currentitem_ != 0) {
        // remove ingredient from recipe
        model_->removeMisc(currentitem_->iterator());
        delete currentitem_; currentitem_ = 0;
    }
}

//////////////////////////////////////////////////////////////////////////////
// itemSelected()
// ------------------
// Received when user selects an item in the list view

void MiscDB::itemSelected(QListViewItem *item)
{
    if (item) {
        currentitem_ = dynamic_cast<MiscItem*>(item);
        if (currentitem_ == 0) {
            qWarning("qbrew: bad dynamic cast, line" + __LINE__);
            return;
        }
        namecombo->setEditText(item->text(MITEM_NAME));
        notesedit->setText(item->text(MITEM_NOTES));
    }
}
