/***************************************************************************
  recipe.h
  -------------------
  A brew recipe class.
  -------------------
  Copyright 2001-2005, David Johnson
  Please see the header file for copyright and license information
 ***************************************************************************/

#include "calc.h"
#include "recipe.h"

//////////////////////////////////////////////////////////////////////////////
// Construction, destruction                                                //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// Recipe()
// --------
// Default constructor
Recipe::Recipe()
    : title_(""), brewer_(""), mash_(false), size_(5.0, Volume::gallon),
      style_(),  grains_(), hops_(), miscs_(), recipenotes_(""), batchnotes_(""),
      modified_(false), og_(0.0), ibu_(0), srm_(0)
{ ; }

//////////////////////////////////////////////////////////////////////////////
// Recipe()
// --------
// Full constructor
Recipe::Recipe(const QString &title, const QString &brewer, bool mash,
               const Volume &size, const Style &style, const GrainList &grains,
               const HopList &hops, const MiscIngredientList &miscs,
               const QString &recipenotes, const QString &batchnotes)
    : title_(title), brewer_(brewer), mash_(mash), size_(size), style_(style),
      grains_(grains), hops_(hops), miscs_(miscs), recipenotes_(recipenotes),
      batchnotes_(batchnotes), modified_(false), og_(0.0), ibu_(0), srm_(0)
{ ; }

//////////////////////////////////////////////////////////////////////////////
// Recipe()
// --------
// Copy constructor
Recipe::Recipe(const Recipe &r)
    : title_(r.title_), brewer_(r.brewer_), mash_(r.mash_), size_(r.size_),
      style_(r.style_), grains_(r.grains_), hops_(r.hops_), miscs_(r.miscs_),
      recipenotes_(r.recipenotes_), batchnotes_(r.batchnotes_), modified_(false),
      og_(0.0), ibu_(0), srm_(0)
{ ; }

Recipe::~Recipe() { ; }

//////////////////////////////////////////////////////////////////////////////
// operator=()
// -----------
// Assignment operator
Recipe Recipe::operator=(const Recipe &r)
{
    if (this != &r) {
        title_ = r.title_;
        brewer_ = r.brewer_;
        mash_ = r.mash_;
        size_ = r.size_;
        style_ = r.style_; 
        grains_ = r.grains_;
        hops_ = r.hops_;
        miscs_ = r.miscs_;
        recipenotes_ = r.recipenotes_;
        batchnotes_ = r.batchnotes_;
        modified_ = r.modified_;
        og_ = r.og_;
        ibu_ = r.ibu_;
        srm_ = r.srm_;
    }
    return *this;
}

//////////////////////////////////////////////////////////////////////////////
// setSize()
// ----------
// Set the recipe batch size

void Recipe::setSize(const Volume &s)
{
    size_ = s;
    Calc::recalc(this);
    modified_ = true;
}

//////////////////////////////////////////////////////////////////////////////
// addGrain()
// ----------
// Add a grain ingredient to the recipe

GrainIterator Recipe::addGrain(const Grain &g)
{
    // TODO: need to keep list sorted
    GrainList::Iterator it;
    it = grains_.append(g);
    Calc::recalc(this);
    modified_ = true;
    return it;
}

//////////////////////////////////////////////////////////////////////////////
// applyGrain()
// ------------
// Apply (change) a grain ingredient to the recipe

void Recipe::applyGrain(GrainIterator it, const Grain &g)
{
    (*it) = g;
    Calc::recalc(this);
    modified_ = true;
}

//////////////////////////////////////////////////////////////////////////////
// removeGrain()
// -------------
// Remove a grain ingredient from the recipe

void Recipe::removeGrain(GrainIterator it)
{
    grains_.remove(it);
    Calc::recalc(this);
    modified_ = true;
}

//////////////////////////////////////////////////////////////////////////////
// addHop()
// ----------
// Add a hop ingredient to the recipe

HopIterator Recipe::addHop(const Hop &h)
{
    HopList::Iterator it;
    it = hops_.append(h);
    Calc::recalc(this);
    modified_ = true;
    return it;
}

//////////////////////////////////////////////////////////////////////////////
// applyHop()
// ------------
// Apply (change) a hop ingredient to the recipe

void Recipe::applyHop(HopIterator it, const Hop &h)
{
    (*it) = h;
    Calc::recalc(this);
    modified_ = true;
}

//////////////////////////////////////////////////////////////////////////////
// removeHop()
// -------------
// Remove a hop ingredient from the recipe

void Recipe::removeHop(HopIterator it)
{
    hops_.remove(it);
    Calc::recalc(this);
    modified_ = true;
}

//////////////////////////////////////////////////////////////////////////////
// addMisc()
// ----------
// Add a misc ingredient to the recipe

MiscIngredientIterator Recipe::addMisc(const MiscIngredient &m)
{
    MiscIngredientList::Iterator it;
    it = miscs_.append(m);
    Calc::recalc(this);
    modified_ = true;
    return it;
}

//////////////////////////////////////////////////////////////////////////////
// applyMisc()
// ------------
// Apply (change) a misc ingredient to the recipe

void Recipe::applyMisc(MiscIngredientIterator it, const MiscIngredient &m)
{
    (*it) = m;
    Calc::recalc(this);
    modified_ = true;
}

//////////////////////////////////////////////////////////////////////////////
// removeMisc()
// -------------
// Remove a misc ingredient from the recipe

void Recipe::removeMisc(MiscIngredientIterator it)
{
    miscs_.remove(it);
    Calc::recalc(this);
    modified_ = true;
}

//////////////////////////////////////////////////////////////////////////////
// recipeType()
// ------------
// Return type of recipe

int Recipe::recipeType()
{
    int extract = 0;
    int mash = 0;
    GrainIterator it;
    for (it=grains_.begin(); it!=grains_.end(); ++it) {
        if ((*it).use() == GRAIN_MASHED) mash++;
        else if ((*it).use() == GRAIN_EXTRACT) extract++;
    }

    if (mash > 0) {
        if (extract > 0) return TypePartial;
        else             return TypeAllGrain;
    }
    return TypeExtract;
}
