/***************************************************************************
  view.cpp
  -------------------
  View class for QBrew
  -------------------
  Copyright 1999-2005, David Johnson
  Please see the header file for copyright and license information
 ***************************************************************************/

#include <qcheckbox.h>
#include <qcombobox.h>
#include <qgroupbox.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qlistbox.h>
#include <qstringlist.h>
#include <qtabwidget.h>

#include "controller.h"
#include "fixedspinbox.h"
#include "grainpage.h"
#include "hopspage.h"
#include "miscpage.h"
#include "notepage.h"
#include "model.h"
#include "resource.h"
#include "style.h"

#include "view.h"

using AppResource::CHAR_NBSP;
using AppResource::CHAR_LATIN_DEGREE;

//////////////////////////////////////////////////////////////////////////////
// Construction, Destruction                                                //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// View()
// ------
// Constructor

View::View(Controller *controller, Model *model)
    : ViewWidget(controller, "View"), controller_(controller), model_(model)
{
    grainpage_ = new GrainPage(this, "grainpage_", model_);
    hoppage_ = new HopPage(this, "hoppage_", model_);
    miscpage_ = new MiscPage(this, "miscpage_", model_);
    notepage_ = new NotePage(this, "notepage_", model_);

    // additional setup
    stylecombo->insertStringList(model_->stylesList());

    sizespin->setMinimum(0.00);
    sizespin->setMaximum(100.00);
    sizespin->setSingleStep(0.25);
    sizespin->setPrecision(2);

    ingredients->addTab(grainpage_, "&Grains");
    ingredients->addTab(hoppage_, "Ho&ps");
    ingredients->addTab(miscpage_, "&Miscellaneous");
    ingredients->addTab(notepage_, "&Notes");

    abvlabel->setAlignment(AlignAuto | AlignVCenter);
    abwlabel->setAlignment(AlignAuto | AlignVCenter);

    // widget connections
    connect(titleedit, SIGNAL(textChanged(const QString &)),
            this, SLOT(setTitle(const QString &)));
    connect(stylecombo, SIGNAL(activated(const QString &)),
            this, SLOT(setStyle(const QString &)));
    connect(breweredit, SIGNAL(textChanged(const QString &)),
            this, SLOT(setBrewer(const QString &)));
    connect(sizespin, SIGNAL(valueChanged(double)),
            this, SLOT(setSize(double)));
    connect(mashcheck, SIGNAL(toggled(bool)),
            this, SLOT(setMash(bool)));

    // start with new view
    flush();
}

View::~View() { ; }

//////////////////////////////////////////////////////////////////////////////
// Slot Implementations                                                     //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// setTitle()
// ---------------------
// set the recipe title

void View::setTitle(const QString &title)
{
    model_->recipe()->setTitle(title);
    model_->setModified(true);
}

//////////////////////////////////////////////////////////////////////////////
// setStyle()
// ---------------------
// set the recipe style

void View::setStyle(const QString &style)
{
    model_->recipe()->setStyle(*model_->style(style));
    model_->setModified(true);
}

//////////////////////////////////////////////////////////////////////////////
// setBrewer()
// ---------------------
// set the recipe brewer

void View::setBrewer(const QString &brewer)
{
    model_->recipe()->setBrewer(brewer);
    model_->setModified(true);
}

//////////////////////////////////////////////////////////////////////////////
// setSize()
// ---------------------
// set the recipe size

void View::setSize(double size)
{
    model_->recipe()->setSize(Volume(size, model_->recipe()->size().unit()));
    model_->setModified(true);
}

//////////////////////////////////////////////////////////////////////////////
// setMash()
// ---------------------
// set mashed / unmashed

void View::setMash(bool mash)
{
    model_->recipe()->setMashed(mash);
    model_->setModified(true);
    grainpage_->refresh(); // update use combo box
}

///////////////////////////////////////////////////////////////////////////////
// Miscellaneous                                                             //
///////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// flush()
// -------
// refresh the entire view

void View::flush()
{
    // save the modified flag to prevent side effects of setting widgets
    bool oldmod = model_->modified();
    // update recipe widgets
    titleedit->setText(model_->recipe()->title());
    breweredit->setText(model_->recipe()->brewer());
    stylecombo->clear();
    stylecombo->insertStringList(model_->stylesList());
    setComboItem(stylecombo, model_->recipe()->style().name());
    sizespin->setValue(model_->recipe()->size().amount());
    sizespin->setSuffix(" " + model_->recipe()->size().unit().name());
    mashcheck->setChecked(model_->recipe()->mashed());
    // update page widgets
    grainpage_->refresh();
    hoppage_->refresh();
    miscpage_->refresh();
    notepage_->refresh();
    // update style widgets
    refresh();
    // restore modified flag
    model_->setModified(oldmod);
}

//////////////////////////////////////////////////////////////////////////////
// refresh()
// ---------
// refresh just the display widgets

void View::refresh()
{
    const Style &style = model_->recipe()->style();

    // beer style has changed, so update labels in stylelayout_
    stylebox->setTitle(stylecombo->currentText() + " Characteristics");
    ogmin->setText(QString::number(style.OGLow(), 'f', 3));
    ogmax->setText(QString::number(style.OGHi(), 'f', 3));
    ogrec->setText("<b>" + QString::number(Calc::OG(model_->recipe()), 'f', 3) + "</b>");
    ibumin->setText(QString::number(style.IBULow()));
    ibumax->setText(QString::number(style.IBUHi()));
    iburec->setText("<b>" + QString::number(Calc::IBU(model_->recipe())) + "</b>");
    srmmin->setText(QString::number(style.SRMLow()) + CHAR_LATIN_DEGREE);
    srmmax->setText(QString::number(style.SRMHi()) + CHAR_LATIN_DEGREE);
    srmrec->setText("<b>" + QString::number(Calc::SRM(model_->recipe())) + CHAR_LATIN_DEGREE + "</b>");
    fg->setText(QString::number(Calc::FGEstimate(model_->recipe()), 'f', 3));
    abv->setText("<b>" + QString::number(Calc::ABV(model_->recipe()) * 100.0, 'f', 1) + "%</b>");
    abw->setText("<b>" + QString::number(Calc::ABW(model_->recipe()) * 100.0, 'f', 1) + "%</b>");
}

///////////////////////////////////////////////////////////////////////////////
// setComboItem()
// --------------
// Set a given combo box to the given entry.

void View::setComboItem(QComboBox *combo, const QString &txt)
{
    QListBox *box = combo->listBox();
    QListBoxItem *item = box->findItem(txt, Qt::ExactMatch);
    if (item) {
        combo->setCurrentItem(box->index(item));
    } else {
        combo->insertItem(txt);
    }
}
