/***************************************************************************
  grainmodel.cpp
  -------------------
  Grain model
  -------------------
  Copyright 2006, David Johnson
  Please see the header file for copyright and license information
 ***************************************************************************/

#include <QMessageBox>
#include <QPair>

#include "data.h"
#include "recipe.h"
#include "resource.h"
#include "grainmodel.h"

const int extra = 1; // always display one extra blank row

//////////////////////////////////////////////////////////////////////////////
// GrainModel()
// ------------
// Constructor

// TODO: add "unique" flag, disallowing  duplicates (for DatabaseTool)

GrainModel::GrainModel(QObject *parent, GrainList *list, Recipe *recipe)
    : QAbstractTableModel(parent), list_(list), recipe_(recipe)
{}

GrainModel::~GrainModel(){}

//////////////////////////////////////////////////////////////////////////////
// data()
// ------
// Return data at index

QVariant GrainModel::data(const QModelIndex &index, int role) const
{
    if (!index.isValid()) return QVariant();
    if (index.row() >= list_->count()) return QVariant();

    // row is the entry in the QList
    const Grain &grain = list_->at(index.row());

    // column is the grain "field"
    if (role == Qt::DisplayRole) {
        switch (index.column()) {
          case Grain::NAME:
              return grain.name();
          case Grain::QUANTITY:
              return grain.weight().toString(2);
          case Grain::EXTRACT:
              return QString::number(grain.extract(), 'f', 3);
          case Grain::COLOR:
              return QString::number(grain.color(), 'f', 1) + Resource::DEGREE;
          case Grain::USE:
              return grain.useString();
          default:
              return QVariant();
        }
    } else if (role == Qt::EditRole) {
        switch (index.column()) {
          case Grain::NAME:
              return grain.name();
          case Grain::QUANTITY: {
              // return converted quantity
              Weight weight = grain.weight();
              weight.convert(Data::instance()->defaultGrainUnit());
              return weight.amount();
          }
          case Grain::EXTRACT:
              return grain.extract();
          case Grain::COLOR:
              return grain.color();
          case Grain::USE:
              return grain.useString();
          default:
              return QVariant();
        }
    } else if (role == Qt::TextAlignmentRole) {
        switch (index.column()) {
          case Grain::NAME:
          case Grain::USE:
              return Qt::AlignLeft;
          case Grain::QUANTITY:
          case Grain::EXTRACT:
          case Grain::COLOR:
          default:
              return Qt::AlignRight;
        }
    } else {
        return QVariant();
    }
}

//////////////////////////////////////////////////////////////////////////////
// setData()
// ---------
// Set data at index

bool GrainModel::setData(const QModelIndex &index,
                       const QVariant &value, int role)
{
    static bool deleting = false;

    Grain grain;
    QString name;
    Data *data = Data::instance();
    int row = index.row();
    int column = index.column();

    if (!index.isValid()) return false;
    if (role != Qt::EditRole) return false;
    if ((row >= list_->count())
        && (column != Grain::NAME)) return false;

    // grab existing grain
    if (row < list_->count()) grain = list_->value(row);

    switch (column) {
      case Grain::NAME:
          // editing name as several special cases
          name = value.toString();

          // deleting name deletes ingredient
          if (name.isEmpty()) {
              if (row >= list_->count()) return false; // empty

              // TODO: for some reason this gets entered recursively...
              if (deleting) return false;
              deleting = true;

              // remove grain
              beginRemoveRows(index.parent(), row, row);

              list_->removeAt(row);
              emit modified();
              endRemoveRows();

              deleting = false;
              return true;
          }

          // setting name on blank row adds grain
          if (row >= list_->count()) {
              if (data->hasGrain(name)) {
                  grain = data->grain(name);
              } else {
                  grain.setName(name);
              }

              // vary default use according to recipe type
              if (recipe_) { // only if we're in a recipe
                  if (recipe_->mashed()) {
                      if (grain.use() == USE_STEEPED) grain.setUse(USE_MASHED);
                  } else {
                      if (grain.use() == USE_MASHED) grain.setUse(USE_STEEPED);
                  }
              }

              beginInsertRows(QModelIndex(), row, row);
              list_->append(grain);
              emit modified();
              endInsertRows();

              return true;
          }

          // changed name
          grain.setName(name);
          break;

      case Grain::QUANTITY:
          grain.setWeight(Weight(value.toDouble(),
                                 Data::instance()->defaultGrainUnit()));
          break;

      case Grain::EXTRACT:
          grain.setExtract(value.toDouble());
          break;

      case Grain::COLOR:
          grain.setColor(value.toDouble());
          break;

      case Grain::USE:
          grain.setUse(value.toString());
          break;

      default:
          return false;
    }

    list_->replace(row, grain);
    emit modified();

    // whole row may have changed
    emit dataChanged(index.sibling(row, Grain::NAME),
                     index.sibling(row, Grain::USE));

    return true;
}

//////////////////////////////////////////////////////////////////////////////
// headerData()
// ------------
// Return header information

QVariant GrainModel::headerData(int section, Qt::Orientation orientation,
                                int role) const
{
    if (orientation == Qt::Horizontal && role == Qt::DisplayRole) {
        switch (section) {
          case Grain::NAME:
              return "Grain";
          case Grain::QUANTITY:
              return "Quantity";
          case Grain::EXTRACT:
              return "Extract";
          case Grain::COLOR:
              return "Color";
          case Grain::USE:
              return "Use";
          default:
              return QVariant();
        }
    }

    return QVariant();
}

//////////////////////////////////////////////////////////////////////////////
// flags()
// ------
// Return flags at index

Qt::ItemFlags GrainModel::flags(const QModelIndex &index) const
{
    if (!index.isValid()) return Qt::ItemIsEnabled;

    return QAbstractTableModel::flags(index) | Qt::ItemIsEditable;
}

//////////////////////////////////////////////////////////////////////////////
// rowCount()
// ----------
// Return number of rows of data

int GrainModel::rowCount(const QModelIndex &) const
{
    return list_->count() + extra;
}

//////////////////////////////////////////////////////////////////////////////
// columnCount()
// -------------
// Return number of columns of data

int GrainModel::columnCount(const QModelIndex &) const
{
    return Grain::COUNT;
}

//////////////////////////////////////////////////////////////////////////////
// sort()
// ------
// Sort by column

void GrainModel::sort(int column, Qt::SortOrder order)
{
    QList<QPair<QString,Grain> > sortlist;

    foreach(Grain grain, *list_) {
        QString field;
        switch (column) {
          case Grain::NAME:
              field = grain.name();
              break;
          case Grain::QUANTITY:
              field = QString::number(grain.weight().amount()).rightJustified(6,'0');
              break;
          case Grain::EXTRACT:
              field = QString::number(grain.extract(),'f',3).rightJustified(6,'0');
              break;
          case Grain::COLOR:
              field = QString::number(grain.color(),'f',1).rightJustified(6,'0');
              break;
          case Grain::USE:
          default:
              field = grain.useString();
              break;
        }
        sortlist.append(QPair<QString,Grain>(field, grain));
    }

    // sort list
    qSort(sortlist.begin(), sortlist.end());

    // create new list
    list_->clear();
    QPair<QString,Grain> pair;
    foreach(pair, sortlist) {
        if (order == Qt::AscendingOrder)
            list_->append(pair.second);
        else
            list_->prepend(pair.second);
    }

    emit layoutChanged();
}
