/***************************************************************************
 *   Copyright (C) 2007 by Anistratov Oleg                                 *
 *   ower@users.sourceforge.net                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation;                         *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 ***************************************************************************/

#include "chatcore.h"
#include "globals.h"

#include <QMapIterator>
#include <QApplication>

#include "inputtextwgt.h"
#include "preferences.h"
#include "userinfo.h"
#include "userprofile.h"
#include "channelwgt.h"

CmdArgs* ChatCore::processCmdArgs(int argc, char* argv[])
{
  int i;
  CmdArgs* args = new CmdArgs;

  for(i = 1; i < argc; i++)
  {
    if(!strcmp("--settingsdir", argv[i]) && (++i < argc))
    {
      qDebug("[ChatCore::processCmdArgs]: settingsDirName = %s", argv[i]);
      m_settingsDirName = QString().fromLocal8Bit(argv[i]);
    }

    if(!strcmp("--iport", argv[i]) && (++i < argc))
    {
      args->port_in = atoi(argv[i]);
      qDebug("[ChatCore::processCmdArgs]: input port = %d", args->port_in);
    }

    if(!strcmp("--oport", argv[i]) && (++i < argc))
    {
      args->port_out = atoi(argv[i]);
      qDebug("[ChatCore::processCmdArgs]: output port = %d", args->port_out);
    }
  }

  return args;
}
//\*****************************************************************************
bool ChatCore::initSettings()
{
  QString home = homePath();
  QFile   file;
  QDir    settingsDir(home);
  QString path;


  if(m_settingsDirName != QString(""))
  {
     path            = m_settingsDirName + QString("/settings.ini"     );
     m_settings      = new QSettings(path, QSettings::IniFormat);
  }
  else if(home == "")
  {
    // TODO : Dialog vybora faila nastroek
    return false;
  }
  else
  {
     m_settingsDirName = QString("%1/%2").arg(home).arg(".qchat");

     if(!QDir(m_settingsDirName).exists())
     {
       printf("[ChatCore::initSettings]: settingsDir does not exists...\n");
       if(settingsDir.mkdir(".qchat"))
         printf(" created!\n");
       else
       {
         printf(" and cannot create :(\n");
         return false;
       }
     }
     // TODO sdelat' proverku na readOnly i v etom sluchae vyvodit' dialog vybora faila nastroek

     path            = m_settingsDirName + QString("/settings.ini"     );
     m_settings      = new QSettings(path, QSettings::IniFormat);
  }

  printf("[ChatCore::initSettings]: settingsDir = %s\n", m_settingsDirName.toLocal8Bit().data());

  Preferences::setSettingsDir(m_settingsDirName);

  if(QFile::exists(m_settingsDirName + "/Charmap"))
    InputTextWgt::initChars(m_settingsDirName + "/Charmap");
#ifdef Q_OS_LINUX
  else if(QFile::exists(QApplication::applicationDirPath() + "/../share/qchat/Charmap"))
    InputTextWgt::initChars(QCoreApplication::applicationDirPath() + "/../share/qchat/Charmap");
#else
  else if(QFile::exists(QApplication::applicationDirPath() + "/Charmap"))
    InputTextWgt::initChars(QApplication::applicationDirPath() + "/Charmap");
#endif

  return true;
}
//\*****************************************************************************
void ChatCore::slot_saveSettings()
{
  QByteArray data;
  QByteArray ba;
  QBuffer    buffer;

  qDebug("[ChatCore::slot_saveSettings]");

  m_settings->beginGroup("Network");

  m_settings->setValue  ("AllowDifferentPorts" , Globals::prefs()->allowDiffPorts());
  m_settings->setValue  ("InputPort" , (quint64)Globals::prefs()->portIn());
  if(Globals::prefs()->allowDiffPorts())
    m_settings->setValue  ("OutputPort", (quint64)Globals::prefs()->portOut());
  else
    m_settings->setValue  ("OutputPort", (quint64)Globals::prefs()->portIn());
  m_settings->setValue    ("IP"        , QHostAddress(Globals::prefs()->ip()).toString());
  m_settings->setValue    ("Broadcast" , Globals::prefs()->broadcast().toString());
  m_settings->setValue    ("CustomIP"        , QHostAddress(Globals::prefs()->customIp()).toString());
  m_settings->setValue    ("CustomBroadcast" , Globals::prefs()->customBroadcast().toString());

  m_settings->endGroup  ();

  //***************************
  QMapIterator<QString, UserProfile*> pr(m_profiles);
  QStringList profiles;
  while (pr.hasNext())
  {
    pr.next();
    writeProfile(pr.value());
    profiles << pr.key();
  }
  //**********************************
  m_settings->setValue("ProfileNames", profiles);
  m_settings->setValue("LastProfile" , m_currentProfile->name());
  m_settings->setValue("Language"    , m_lang);

  if(m_chatWgt)
  {
    m_settings->setValue("Geometry", m_chatWgt->saveGeometry());
    m_settings->setValue("MainWindowState", m_chatWgt->saveState());

    QList<ChannelWgt*> l = m_chatWgt->channels();

    foreach(ChannelWgt* c, l)
      setChannelState(c->name(), c->saveState());
  }

  m_settings->setValue("ChannelsStates", QStringList(m_channelsStates.keys()));

  QMapIterator<QString, QByteArray> ch(m_channelsStates);
  while (ch.hasNext())
  {
    ch.next();
    m_settings->setValue(ch.key() + "ChannelState", ch.value());
  }

  m_settings->sync();
}
//\*****************************************************************************
void ChatCore::slot_loadSettings()
{
  QNetworkAddressEntry addr;
  QString str;
  QStringList profiles;

  m_settings->beginGroup("Network");

  Preferences::setAllowDiffPorts(m_settings->value("AllowDifferentPorts" , false).toBool());
  Preferences::setPortIn        (m_settings->value("InputPort"           , 61108).toUInt());

  if(Preferences::allowDiffPorts())
    Preferences::setPortOut(m_settings->value("OutputPort", 61108).toUInt());
  else
    Preferences::setPortOut(Preferences::portIn());

  //*******************************
  addr.setIp(QHostAddress(QString(m_settings->value("IP").toByteArray().data ())));
  addr.setBroadcast(QHostAddress(QString(m_settings->value("Broadcast").toByteArray().data ())));

  QList<QNetworkInterface> all_i = QNetworkInterface::allInterfaces();
  QList<QNetworkAddressEntry> valid_a;

  // get all valid(that can broadcast) interfaces
  foreach(QNetworkInterface i, all_i)
    foreach(QNetworkAddressEntry a, i.addressEntries())
    {
      if(a.broadcast().isNull())
        continue;

       valid_a.append(a);
    }

  // this will be used to determine do we need to
  // warning user about problems with network settings
  m_needCheckIp = 1 + valid_a.isEmpty();

  // if network settings was not set yet
  // or they are is broken we are try to set correct ones
  if(addr.ip().isNull() && !valid_a.isEmpty())
    addr = valid_a[0];
  else if(addr.broadcast().isNull() && !valid_a.isEmpty())
    addr.setBroadcast(valid_a[0].broadcast());

  // is network settings are corresponds any of existing interface?
  foreach(QNetworkAddressEntry a, valid_a)
    if((a.ip() == addr.ip()) && (a.broadcast() == addr.broadcast()))
    {
      m_needCheckIp = 0;
      break;
    }

  Preferences::setIp(addr.ip().toIPv4Address());
  Preferences::setBroadcast(addr.broadcast());

  QHostAddress tmp;

  tmp = QHostAddress(m_settings->value("CustomIP").toString());
  if(!tmp.isNull())
    addr.setIp(tmp);

  tmp = QHostAddress(m_settings->value("CustomBroadcast").toString());
  if(!tmp.isNull())
    addr.setBroadcast(tmp);

  Preferences::setCustomIp(addr.ip().toIPv4Address());
  Preferences::setCustomBroadcast(addr.broadcast());

  //*******************************

  m_settings->endGroup();

  //***************************
  profiles   = m_settings->value("ProfileNames", QStringList()).toStringList();
  str        = m_settings->value("LastProfile" , "Default").toString();
  m_geometry = m_settings->value("Geometry").toByteArray();
  m_state    = m_settings->value("MainWindowState").toByteArray();
  m_lang     = m_settings->value("Language").toString();

  QStringList channels = m_settings->value("ChannelsStates", QStringList()).toStringList();

  foreach(QString s, channels)
    m_channelsStates.insert(s, m_settings->value(s + "ChannelState").toByteArray());

//   m_channelsStates.insert("Main", m_settings->value("MainChannelState").toByteArray());

  if(profiles.isEmpty())
    m_profiles.insert("Default", readProfile("Default"));
  else
    foreach(QString s, profiles)
      m_profiles.insert(s, readProfile(s));

  slot_loadProfile(str);
}
//\*****************************************************************************
UserProfile* ChatCore::readProfile(const QString & name)
{
  UserInfo*    info    = new UserInfo;
  Preferences* prefs   = new Preferences;
  UserProfile* profile = new UserProfile(name, prefs, info);
  QByteArray   ba;

  m_settings->beginGroup("Profiles");

  m_settings->beginGroup(name);

  m_settings->beginGroup("Warnings");

  prefs->setWarningAboutTray(m_settings->value("WarningAboutHidingInTray", true).toBool());

  m_settings->endGroup();//  m_settings->beginGroup("Warnings");

  m_settings->beginGroup("UserInfo");

  info->setGender     (m_settings->value("Gender"     ,  0).toInt());
  info->setNickname   (m_settings->value("NickName"   , QString(QHostInfo::localHostName())).toString());
  info->setLastName   (m_settings->value("LastName"   , "").toString());
  info->setFirstName  (m_settings->value("FirstName"  , "").toString());
  info->setSecondName (m_settings->value("SecondName" , "").toString());
  info->setDateOfBorn (m_settings->value("DateOfBorn" , "").toDate  ());
  info->setAddress    (m_settings->value("Address"    , "").toString());
  info->setHomePhone  (m_settings->value("HomePhone"  , "").toString());
  info->setWorkPhone  (m_settings->value("WorkPhone"  , "").toString());
  info->setMobilePhone(m_settings->value("MobilePhone", "").toString());
  info->setE_mail     (m_settings->value("e-mail"     , "").toString());
  info->setICQ        (m_settings->value("ICQ"        , "").toString());
  info->setHomepage   (m_settings->value("Homepage"   , "").toString());
  info->setAboutInfo  (m_settings->value("About"      , "").toString());

  m_settings->endGroup();//  m_settings->beginGroup("UserInfo");

  info->setPhoto  (m_settings->value("Photo"  , "/").toString());
  info->setPicture(m_settings->value("Picture", "/").toString());

  m_settings->beginGroup("Preferences");

  prefs->setActivateOnMsgIn(m_settings->value("ActivateOnMessageIn", false).toBool());
  prefs->setSoundOnMsgIn   (m_settings->value("SoundOnMessageIn"   , false).toBool());

  QColor c(Qt::red);
  ba = QByteArray();
  ba = m_settings->value("MyMessagesColor", QByteArray().append(c.red())
                                                        .append(c.green())
                                                        .append(c.blue())).toByteArray();
  prefs->setMyColor(QColor((quint8)ba[0], (quint8)ba[1], (quint8)ba[2]));
  c  = QColor(Qt::gray);
  ba = QByteArray();
  ba = m_settings->value("SystemMessagesColor", QByteArray().append(c.red())
                                                            .append(c.green())
                                                            .append(c.blue())).toByteArray();
  prefs->setSysColor(QColor((quint8)ba[0], (quint8)ba[1], (quint8)ba[2]));
  c  = QColor(Qt::black);
  ba = QByteArray();
  ba = m_settings->value("BaseColor"       , QByteArray().append(c.red())
                                                         .append(c.green())
                                                         .append(c.blue())).toByteArray();
  prefs->setBaseColor(QColor((quint8)ba[0], (quint8)ba[1], (quint8)ba[2]));

  prefs->setMsgColorMode   (m_settings->value("ColorWholeMessage"        , false).toBool());
  prefs->setSysMsgColorMode(m_settings->value("ColorWholeSystemMessage"  , true).toBool());

  prefs->setSmilesThemePath(m_settings->value("SmilesTheme", Preferences::defaultSmilesDir()).toString());

  prefs->setExecuteCommandOnIncomingMsg(m_settings->value("ExecuteCommandOnIncomingMessage", "").toString());
  prefs->setIsExecuteCommandOnIncomingMsg(m_settings->value("IsExecuteCommandOnIncomingMessage", true).toBool());

  prefs->setDisplayMsgFormat(m_settings->value("DisplayMessagesFormat", "%time%hh:mm:ss%time% | [[%user@%comp]] : ").toString());

prefs->setNHistoryMsgs(m_settings->value("MaximumMessagesHistoryNumber", 0).toInt());
  prefs->setHistoryReqTimeout(m_settings->value("MessagesHistoryRequestTimeout", 5000).toInt()); // msecs

  prefs->setUsersListRefreshInterval(m_settings->value("UsersListRefreshInterval", 60).toInt()); // secs
  prefs->setUsersListDeepRefreshInterval(m_settings->value("UsersListDeepRefreshInterval", 600).toInt()); // secs

  m_settings->endGroup();//  m_settings->beginGroup("Preferences");

  m_settings->beginGroup("Shortcuts");

//   prefs->setSendMessageSeq(QKeySequence::fromString(m_settings->value("SendMessage", "Ctrl+Key_Return").toString()));

  m_settings->endGroup();//  m_settings->beginGroup("Shortcuts");

  m_settings->beginGroup("StatusDescriptions");

  prefs->setStatusDescription(m_settings->value("Ready4Chat").toString(), Globals::READY4CHAT);
  prefs->setStatusDescription(m_settings->value("Free"      ).toString(), Globals::FREE);
  prefs->setStatusDescription(m_settings->value("Busy"      ).toString(), Globals::BUSY);
  prefs->setStatusDescription(m_settings->value("Dnd"       ).toString(), Globals::DND);
  prefs->setStatusDescription(m_settings->value("Inactive"  ).toString(), Globals::INACTIVE);
  prefs->setStatusDescription(m_settings->value("Away"      ).toString(), Globals::AWAY);

  prefs->setStatusDescriptions(m_settings->value("DescriptionsHistory").toStringList());

  m_settings->endGroup();//  m_settings->beginGroup("StatusDescriptions");

  m_settings->beginGroup("NowListening");

  prefs->setNlMode(m_settings->value("SendWithMessage", false).toBool() +
                   m_settings->value("SetInStatus"    , false).toBool() * 2);
  prefs->setNlFormat(m_settings->value("Format", "Now Listening to : %t by %a on %b").toString());

  m_settings->endGroup();//  m_settings->beginGroup("NowListening");

  m_settings->endGroup();//  m_settings->beginGroup(name);

  m_settings->endGroup();// m_settings->beginGroup("Profiles");

  return profile;
}
//\*****************************************************************************
void ChatCore::writeProfile(const UserProfile* profile)
{
  Preferences* prefs = profile->prefs();
  UserInfo*    info  = profile->info();
  QByteArray bpic;
  QByteArray ba;
  QByteArray data;
  QBuffer    buffer;

  m_settings->beginGroup("Profiles");

  m_settings->beginGroup(profile->name());

  m_settings->setValue("Photo"              , info->photoFilename  ());
  m_settings->setValue("Picture"            , info->pictureFilename());

  m_settings->beginGroup("Warnings");

  m_settings->setValue("WarningAboutHidingInTray", prefs->warningAboutTray());

  m_settings->endGroup();//  m_settings->beginGroup("Warnings");


  m_settings->beginGroup("UserInfo");

  m_settings->setValue("Gender"       , info->gender());
  m_settings->setValue("NickName"     , info->nickname());
  m_settings->setValue("LastName"     , info->lastName());
  m_settings->setValue("FirstName"    , info->firstName());
  m_settings->setValue("SecondName"   , info->secondName());
  m_settings->setValue("DateOfBorn"   , info->dateOfBorn());
  m_settings->setValue("Address"      , info->address());
  m_settings->setValue("HomePhone"    , info->homePhone());
  m_settings->setValue("WorkPhone"    , info->workPhone());
  m_settings->setValue("MobilePhone"  , info->mobilePhone());
  m_settings->setValue("e-mail"       , info->e_mail());
  m_settings->setValue("ICQ"          , info->icq());
  m_settings->setValue("Homepage"     , info->homepage());
  m_settings->setValue("About"        , info->aboutInfo());

  m_settings->endGroup();//  m_settings->beginGroup("UserInfo");

  m_settings->beginGroup("Preferences");

  ba = QByteArray();
  ba = QByteArray().append(prefs->myColor().red())
                   .append(prefs->myColor().green())
                   .append(prefs->myColor().blue());
  m_settings->setValue("MyMessagesColor"         , ba);

  ba = QByteArray();
  ba = QByteArray().append(prefs->sysColor().red())
                   .append(prefs->sysColor().green())
                   .append(prefs->sysColor().blue());
  m_settings->setValue("SystemMessagesColor"     , ba);

  ba = QByteArray();
  ba = QByteArray().append(prefs->baseColor().red())
                   .append(prefs->baseColor().green())
                   .append(prefs->baseColor().blue());
  m_settings->setValue("BaseColor"               , ba);

  m_settings->setValue("ColorWholeMessage"       , prefs->colorWholeMsg());
  m_settings->setValue("ColorWholeSystemMessage" , prefs->colorWholeSysMsg());
  m_settings->setValue("ActivateOnMessageIn"     , prefs->activateOnMsgIn());
  m_settings->setValue("SoundOnMessageIn"        , prefs->soundOnMsgIn());

  m_settings->setValue("SmilesTheme"                    , prefs->smilesThemePath());

  m_settings->setValue("ExecuteCommandOnIncomingMessage", prefs->executeCommandOnIncomingMsg());
  m_settings->setValue("IsExecuteCommandOnIncomingMessage", prefs->isExecuteCommandOnIncomingMsg());

  m_settings->setValue("DisplayMessagesFormat"          , prefs->displayMsgFormat());

  m_settings->setValue("MaximumMessagesHistoryNumber"   , prefs->nHistoryMsgs());
  m_settings->setValue("MessagesHistoryRequestTimeout"  , prefs->historyReqTimeout());

  m_settings->setValue("UsersListRefreshInterval"       , prefs->usersListRefreshInterval());
  m_settings->setValue("UsersListDeepRefreshInterval"   , prefs->usersListDeepRefreshInterval());

  m_settings->endGroup();//  m_settings->beginGroup("Preferences");

  m_settings->beginGroup("Shortcuts");

//   m_settings->value("SendMessage", prefs->sendMessageSeq().toString());

  m_settings->endGroup();//  m_settings->beginGroup("Shortcuts");

  m_settings->beginGroup("StatusDescriptions");

  m_settings->setValue("Ready4Chat", prefs->statusDescription(Globals::READY4CHAT));
  m_settings->setValue("Free"      , prefs->statusDescription(Globals::FREE));
  m_settings->setValue("Busy"      , prefs->statusDescription(Globals::BUSY));
  m_settings->setValue("Dnd"       , prefs->statusDescription(Globals::DND));
  m_settings->setValue("Inactive"  , prefs->statusDescription(Globals::INACTIVE));
  m_settings->setValue("Away"      , prefs->statusDescription(Globals::AWAY));

  m_settings->setValue("DescriptionsHistory", prefs->statusDescriptions());

  m_settings->endGroup();// m_settings->beginGroup("StatusDescriptions");

  m_settings->beginGroup("NowListening");

  m_settings->setValue("SendWithMessage", prefs->nlMode() & 1);
  m_settings->setValue("SetInStatus"    , prefs->nlMode() & 2);
  m_settings->setValue("Format"         , prefs->nlFormat());

  m_settings->endGroup();//  m_settings->beginGroup("NowListening");

  m_settings->endGroup();// m_settings->beginGroup(profile->name());

  m_settings->endGroup();// m_settings->beginGroup("Profiles");

}
//\*****************************************************************************
