/***************************************************************************
 *   Copyright (C) 2007 by Anistratov Oleg                                 *
 *   ower@users.sourceforge.net                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation;                         *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 ***************************************************************************/

#include <assert.h>

#include <QCoreApplication>

#include "largedatagram.h"
#include "receiverthread.h"

ReceiverThread::ReceiverThread(QObject *parent) :
  QThread          (parent),
  m_opened         (false),
  m_finished       (false),
  m_datagrams      (NULL),
  m_datagramsNum   (0),
  m_datagramsMaxNum(0),
  m_bufferSize     (65535),
  m_socket         (NULL),
  m_socketNotifier (NULL),
  m_port           (61108)
{
  moveToThread(this);

  m_buffer          = (char*)malloc(m_bufferSize);
  assert(NULL      != m_buffer);
}
//\*****************************************************************************
ReceiverThread::~ReceiverThread()
{
  delete m_socketNotifier;
  free(m_buffer);
}
//\*****************************************************************************
void ReceiverThread::run()
{
  m_socket = new QUdpSocket(this);
  exec();
}
//\*****************************************************************************
void ReceiverThread::changePort(quint16 port_)
{
  Q_ASSERT(currentThread() == this);

  qDebug("[ReceiverThread::setPort]: port = %d", port_);

  if(m_opened)
    m_socket->close();

  qDebug("[ReceiverThread::setPort]: closed");

  m_opened = false;
  m_port   = port_;
  m_opened = m_socket->bind(m_port);

  qDebug("[ReceiverThread::setPort]: opened = %d", m_opened);

  if(m_opened == 0)
  {
    emit openSocketError(port_);
    return;
  }

  if(m_socketNotifier)
    disconnect(m_socketNotifier, SIGNAL(activated(int)), this, SLOT(receiving()));

  delete m_socketNotifier;
  m_socketNotifier = NULL;

  m_socketNotifier = new QSocketNotifier(m_socket->socketDescriptor(), QSocketNotifier::Read, this);

  connect(m_socketNotifier, SIGNAL(activated(int)), this, SLOT(receiving()));
}
//\*****************************************************************************
LargeDatagram* ReceiverThread::findDatagram(quint64 IP, quint32 ID) const
{
  for(quint32 i = 0; i < m_datagramsNum; i++)
    if(m_datagrams[i]->cmp(IP, ID))
      return m_datagrams[i];

  return NULL;
}
//\*****************************************************************************
void ReceiverThread::removeDatagram(LargeDatagram* dtgrm)
{
  if(!dtgrm)
    return;

  for(quint32 i = 0; i < m_datagramsNum; i++)
    if(m_datagrams[i] == dtgrm)
    {
      --m_datagramsNum;
      m_datagrams[i] = m_datagrams[m_datagramsNum];
    }
}
//\*****************************************************************************
LargeDatagram* ReceiverThread::addDatagram(quint64 IP, quint32 ID)
{
  Q_ASSERT(currentThread() == this);

  LargeDatagram* dtgrm = new LargeDatagram(IP, ID, this);
  LargeDatagram** tmp;

  dtgrm->moveToThread(this);

  qDebug("[ReceiverThread::addDatagram]: adding(%lu, %lu)", (unsigned long)IP, (unsigned long)ID);

  if(dtgrm)
  {
    m_datagramsNum++;

    if(m_datagramsMaxNum < m_datagramsNum)
    {
      m_datagramsMaxNum++;
      tmp = (LargeDatagram**)realloc(m_datagrams, m_datagramsMaxNum * sizeof(LargeDatagram*));

      if(!tmp)
      {
        delete dtgrm;
        return NULL;
      }

      m_datagrams = tmp;
    }

    m_datagrams[m_datagramsNum - 1] = dtgrm;

    connect(dtgrm, SIGNAL(  wantDie(LargeDatagram*)), this, SLOT(deleteDatagram(LargeDatagram*)));
    connect(dtgrm, SIGNAL(completed(LargeDatagram*)), this, SLOT(deleteDatagram(LargeDatagram*)));
    connect(dtgrm, SIGNAL(wantFragments(char*, quint32, quint32, QHostAddress)),
             this, SIGNAL(wantFragments(char*, quint32, quint32, QHostAddress)));
    connect(dtgrm, SIGNAL(percentsRemain(quint8, quint16, QHostAddress)),
             this, SIGNAL(percentsRemain(quint8, quint16, QHostAddress)));
    connect(dtgrm, SIGNAL(readyReceive  (quint16, quint64)),
             this, SIGNAL(readyReceive  (quint16, quint64)));
  }

  return dtgrm;
}
//\*****************************************************************************
quint32 ReceiverThread::getValidID(quint64 IP) const
{
  for(quint32 i = 1; i > 0; i++)
    if(!findDatagram(IP, i))
      return i;

  return 0;
}
//\*****************************************************************************
void ReceiverThread::receiving()
{
  quint32 dtgrmNum;
  quint32 dtgrmID;
  quint64 senderIP;
  qint32  dataSize;

  LargeDatagram* dtgrm;

  quint8 type;

  if(m_opened)
  {
    if(m_socket->hasPendingDatagrams())
    {
#if defined (Q_OS_WIN)
      m_socketNotifier->setEnabled(false);
      dataSize = m_socket->readDatagram(m_buffer, m_bufferSize);
      m_socketNotifier->setEnabled(true);
#else
      dataSize = m_socket->readDatagram(m_buffer, m_bufferSize);
#endif
      if(dataSize < 0)
      {
        qWarning("[ReceiverThread::receiving]: datagram read error");
        return;
      }

      if(dataSize >= Globals::ProtocolLen && !strncmp(m_buffer, Globals::ProgramID, strlen(Globals::ProgramID)))
      {
        type     = (quint8)*(m_buffer + 38);
        senderIP =         str2ULL(m_buffer + 30);
        dtgrmID  = (quint32)str2US(m_buffer + 39);
        dtgrmNum = (quint32)str2UL(m_buffer + 43);

        if(type == Globals::SINGLE_MESSAGE)
           qDebug("[ReceiverThread::run]: SINGLE_MESSAGE");

        if(type == Globals::FRAGMENTS_REQUEST)
        {
           qDebug("[ReceiverThread::run]: FRAGMENTS_REQUEST");
           emit fragmentsRequest(m_buffer, dataSize);
        }
        else if(type == Globals::FINISHED)
        {
          qDebug("[ReceiverThread::run]: FINISHED!");
//           emit dtgrmFinished(dtgrmID);
        }
        else if(dtgrmID != 0)
        {
          switch(type)
          {
            case Globals::CONFIRM :
              qDebug("[ReceiverThread::run]: CONFIRM| dtgrmID = %d, dtgrmNum = %d", dtgrmID, dtgrmNum);
              emit percentsConfirmed(dtgrmNum, dtgrmID, QHostAddress(senderIP));
              break;

            case Globals::ACCEPT:
              qDebug("[ReceiverThread::run]: ACCEPT");
              emit receivingAccepted(dtgrmID);
              break;

            case Globals::REJECT:
              qDebug("[ReceiverThread::run]: REJECT");
              qDebug("[ReceiverThread::run]: reason = %d", dtgrmNum);

              if(dtgrmNum == 0)
                emit receivingRejected(dtgrmID);
              else if(dtgrmNum == 1)
                emit receivingCancelled(dtgrmID);
              else if(dtgrmNum == 2)
              {
                deleteDatagram(findDatagram(senderIP, dtgrmID));
                emit sendingCancelled(dtgrmID, senderIP);
              }
              break;

              default:
                if(!(dtgrm = findDatagram(senderIP, dtgrmID)))
                {
                  qDebug("[ReceiverThread::run]: ADDING DATAGRAM");
                  dtgrm = addDatagram(senderIP, dtgrmID);
                  if(NULL == dtgrm)
                    return;
                }

                if(dtgrmID != 0)
                {
                  if(!dtgrmNum)
                  {
                    dtgrm->initDatagram(m_buffer, dataSize);
                    if(type == Globals::FILE)
                      emit wantReceiveFile(dtgrm->filename(), dtgrmID, senderIP);
                  }
                  else if(type == Globals::FILE)
                  {
                    dtgrm->addFileFragment(m_buffer, dataSize);
                    if(dtgrm->complete())
                      qDebug("[ReceiverThread::run]: File COMPLETED!");
                  }
                  else
                  {
                    dtgrm->addFragment(m_buffer, dataSize);
                    if(dtgrm->complete())
                    {
                      qDebug("[ReceiverThread::run]: COMPLETED!");
                      emit largeDataReceived(dtgrm);
                    }
                  }
               }// default:
                //   if(dtgrmID != 0)
           } // switch(type)

        } // else if(dtgrmID != 0)

        else
        {
          qDebug("[ReceiverThread::run]: DATA_RECEIVED!");
          emit dataReceived(datadup(m_buffer, dataSize), dataSize);
        }
      }//if(data_len >= 69 && !strncmp(data, "QChat4Linux&Others", 18))

    }
  }
}
//\*****************************************************************************
void ReceiverThread::deleteDatagram(LargeDatagram* dtgrm)
{
  Q_ASSERT(currentThread() == this);
  if(!dtgrm)
    return;

  qDebug("[ReceiverThread::deleteDatagram]");

  if(!dtgrm->complete())
    emit receivingTimeout(dtgrm->id(), dtgrm->ip());

  removeDatagram(dtgrm);

  delete dtgrm;
}
//\*****************************************************************************
void ReceiverThread::slot_acceptDatagram(const QString & filename, quint16 ID, quint64 IP)
{
  qDebug("[ReceiverThread::slot_acceptDatagram]");

  LargeDatagram* dtgrm = findDatagram(IP, ID);

  if(dtgrm)
    dtgrm->slot_initFile(filename);
  else
    qWarning("[ReceiverThread::slot_acceptDatagram]: [error] dtgrm is NULL");
}
//\*****************************************************************************
void ReceiverThread::slot_rejectDatagram(quint16 ID, quint64 IP)
{
  deleteDatagram(findDatagram(IP, ID));
}
