/****************************************************************************
**
** Copyright (C) 2007-2007 Trolltech ASA. All rights reserved.
**
** This file is part of the accessibility project on Trolltech Labs.
**
** This file may be used under the terms of the GNU General Public
** License version 2.0 as published by the Free Software Foundation
** and appearing in the file LICENSE.GPL included in the packaging of
** this file.  Please review the following information to ensure GNU
** General Public Licensing requirements will be met:
** http://www.trolltech.com/products/qt/opensource.html
**
** If you are unsure which license is appropriate for your use, please
** review the following information:
** http://www.trolltech.com/products/qt/licensing.html or contact the
** sales department at sales@trolltech.com.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/

#include "accessibleobject.h"
#include "accessibleobjectadaptor.h"
#include "accessibletextadaptor.h"
#include "accessibleeditabletextadaptor.h"
#include "accessiblevalueadaptor.h"
#include "accessibleactionadaptor.h"
#include "accessiblecomponentadaptor.h"
#include "accessibletableadaptor.h"

#include "accessiblenullinterface.h"

#include <QtGui/QtGui>
#include <QtDBus/QtDBus>

enum { RoleNameCount = 64 };
static const char * const roleNames[RoleNameCount] = {
        QT_TRANSLATE_NOOP("IAccessible2", "NoRole"),
        QT_TRANSLATE_NOOP("IAccessible2", "TitleBar"),
        QT_TRANSLATE_NOOP("IAccessible2", "MenuBar"),
        QT_TRANSLATE_NOOP("IAccessible2", "ScrollBar"),
        QT_TRANSLATE_NOOP("IAccessible2", "Grip"),
        QT_TRANSLATE_NOOP("IAccessible2", "Sound"),
        QT_TRANSLATE_NOOP("IAccessible2", "Cursor"),
        QT_TRANSLATE_NOOP("IAccessible2", "Caret"),
        QT_TRANSLATE_NOOP("IAccessible2", "AlertMessage"),
        QT_TRANSLATE_NOOP("IAccessible2", "Window"),
        QT_TRANSLATE_NOOP("IAccessible2", "Client"),
        QT_TRANSLATE_NOOP("IAccessible2", "PopupMenu"),
        QT_TRANSLATE_NOOP("IAccessible2", "MenuItem"),
        QT_TRANSLATE_NOOP("IAccessible2", "ToolTip"),
        QT_TRANSLATE_NOOP("IAccessible2", "Application"),
        QT_TRANSLATE_NOOP("IAccessible2", "Document"),
        QT_TRANSLATE_NOOP("IAccessible2", "Pane"),
        QT_TRANSLATE_NOOP("IAccessible2", "Chart"),
        QT_TRANSLATE_NOOP("IAccessible2", "Dialog"),
        QT_TRANSLATE_NOOP("IAccessible2", "Border"),
        QT_TRANSLATE_NOOP("IAccessible2", "Grouping"),
        QT_TRANSLATE_NOOP("IAccessible2", "Separator"),
        QT_TRANSLATE_NOOP("IAccessible2", "ToolBar"),
        QT_TRANSLATE_NOOP("IAccessible2", "StatusBar"),
        QT_TRANSLATE_NOOP("IAccessible2", "Table"),
        QT_TRANSLATE_NOOP("IAccessible2", "ColumnHeader"),
        QT_TRANSLATE_NOOP("IAccessible2", "RowHeader"),
        QT_TRANSLATE_NOOP("IAccessible2", "Column"),
        QT_TRANSLATE_NOOP("IAccessible2", "Row"),
        QT_TRANSLATE_NOOP("IAccessible2", "Cell"),
        QT_TRANSLATE_NOOP("IAccessible2", "Link"),
        QT_TRANSLATE_NOOP("IAccessible2", "HelpBalloon"),
        QT_TRANSLATE_NOOP("IAccessible2", "Assistant"),
        QT_TRANSLATE_NOOP("IAccessible2", "List"),
        QT_TRANSLATE_NOOP("IAccessible2", "ListItem"),
        QT_TRANSLATE_NOOP("IAccessible2", "Tree"),
        QT_TRANSLATE_NOOP("IAccessible2", "TreeItem"),
        QT_TRANSLATE_NOOP("IAccessible2", "PageTab"),
        QT_TRANSLATE_NOOP("IAccessible2", "PropertyPage"),
        QT_TRANSLATE_NOOP("IAccessible2", "Indicator"),
        QT_TRANSLATE_NOOP("IAccessible2", "Graphic"),
        QT_TRANSLATE_NOOP("IAccessible2", "StaticText"),
        QT_TRANSLATE_NOOP("IAccessible2", "EditableText"),
        QT_TRANSLATE_NOOP("IAccessible2", "PushButton"),
        QT_TRANSLATE_NOOP("IAccessible2", "CheckBox"),
        QT_TRANSLATE_NOOP("IAccessible2", "RadioButton"),
        QT_TRANSLATE_NOOP("IAccessible2", "ComboBox"),
        QT_TRANSLATE_NOOP("IAccessible2", "DropList"),
        QT_TRANSLATE_NOOP("IAccessible2", "ProgressBar"),
        QT_TRANSLATE_NOOP("IAccessible2", "Dial"),
        QT_TRANSLATE_NOOP("IAccessible2", "HotkeyField"),
        QT_TRANSLATE_NOOP("IAccessible2", "Slider"),
        QT_TRANSLATE_NOOP("IAccessible2", "SpinBox"),
        QT_TRANSLATE_NOOP("IAccessible2", "Canvas"),
        QT_TRANSLATE_NOOP("IAccessible2", "Animation"),
        QT_TRANSLATE_NOOP("IAccessible2", "Equation"),
        QT_TRANSLATE_NOOP("IAccessible2", "ButtonDropDown"),
        QT_TRANSLATE_NOOP("IAccessible2", "ButtonMenu"),
        QT_TRANSLATE_NOOP("IAccessible2", "ButtonDropGrid"),
        QT_TRANSLATE_NOOP("IAccessible2", "Whitespace"),
        QT_TRANSLATE_NOOP("IAccessible2", "PageTabList"),
        QT_TRANSLATE_NOOP("IAccessible2", "Clock"),
        QT_TRANSLATE_NOOP("IAccessible2", "Splitter"),
        QT_TRANSLATE_NOOP("IAccessible2", "LayeredPane") };

enum { StateNameCount = 29 };
static const struct { QAccessible::State state; const char * name; } stateNames[StateNameCount] = {
    { QAccessible::Normal, "Normal" },
    { QAccessible::Unavailable, "Unavailable" },
    { QAccessible::Selected, "Selected" },
    { QAccessible::Focused, "Focused" },
    { QAccessible::Pressed, "Pressed" },
    { QAccessible::Checked, "Checked" },
    { QAccessible::Mixed, "Mixed" },
    { QAccessible::ReadOnly, "ReadOnly" },
    { QAccessible::HotTracked, "HotTracked" },
    { QAccessible::DefaultButton, "DefaultButton" },
    { QAccessible::Expanded, "Expanded" },
    { QAccessible::Collapsed, "Collapsed" },
    { QAccessible::Busy, "Busy" },
    { QAccessible::Marqueed, "Marqueed" },
    { QAccessible::Animated, "Animated" },
    { QAccessible::Invisible, "Invisible" },
    { QAccessible::Offscreen, "Offscreen" },
    { QAccessible::Sizeable, "Sizeable" },
    { QAccessible::Movable, "Movable" },
    { QAccessible::SelfVoicing, "SelfVoicing" },
    { QAccessible::Focusable, "Focusable" },
    { QAccessible::Selectable, "Selectable" },
    { QAccessible::Linked, "Linked" },
    { QAccessible::Traversed, "Traversed" },
    { QAccessible::MultiSelectable, "MultiSelectable" },
    { QAccessible::ExtSelectable, "ExtSelectable" },
    { QAccessible::Protected, "Protected" },
    { QAccessible::HasPopup, "HasPopup" },
    { QAccessible::Modal, "Modal" } };

typedef QHash<QObject *, AccessibleObject *> AccessibleCache;
Q_GLOBAL_STATIC(AccessibleCache, accessibleCache)

AccessibleObject::AccessibleObject(const QString &path, QObject *aclient, AccessibleObject *parent)
    : QObject(parent), iface(0), client(aclient), dPath(path), childrenDirty(true), uniqueChildId(0)
{
    if (client) {
        // get the accessible interface
        iface = QAccessible::queryAccessibleInterface(client);
        Q_ASSERT(iface);

        // register ourselves in the cache
        Q_ASSERT(!accessibleCache()->value(client));
        accessibleCache()->insert(client, this);

        // make sure we don't crash on dangling pointers
        connect(client, SIGNAL(destroyed()), this, SLOT(objectDestroyed()));

    } else {
        iface = new AccessibleNullInterface;
    }

    new AccessibleObjectAdaptor(this);

    if (iface->textInterface())
        new AccessibleTextAdaptor(this);
    if (iface->editableTextInterface())
        new AccessibleEditableTextAdaptor(this);
    if (iface->valueInterface())
        new AccessibleValueAdaptor(this);
    if (client->isWidgetType()) {
        new AccessibleComponentAdaptor(this);
        new AccessibleActionAdaptor(this);
    }
    if (iface->tableInterface())
        new AccessibleTableAdaptor(this);

    QDBusConnection::sessionBus().registerObject(dPath, this, QDBusConnection::ExportAdaptors);
}

AccessibleObject::~AccessibleObject()
{
    if (AccessibleObject *parentObject = qobject_cast<AccessibleObject *>(parent()))
        parentObject->childrenPaths.removeAll(dPath);

    accessibleCache()->remove(client);
    delete iface;
}

QString AccessibleObject::name() const
{
    return iface->text(QAccessible::Name, 0);
}

QString AccessibleObject::className() const
{
    return client ? QString::fromLatin1(client->metaObject()->className()) : QString();
}

QString AccessibleObject::description() const
{
    return iface->text(QAccessible::Description, 0);
}

QString AccessibleObject::role() const
{
    int role = iface->role(0);
    if (role < 0 || role > RoleNameCount) {
        qDebug("Unknown role: %d", role);
        return QString();
    }
    return QLatin1String(roleNames[role]);
}

QString AccessibleObject::localizedRoleName() const
{
    return QApplication::instance()->tr("IAccessible2", role().toLatin1());
}

int AccessibleObject::states() const
{
    enum AccessibleStates
    {
        IA2_STATE_ACTIVE = 0x1,
        IA2_STATE_ARMED = 0x2,
        IA2_STATE_DEFUNCT = 0x4,
        IA2_STATE_EDITABLE = 0x8,
        IA2_STATE_HORIZONTAL = 0x10,
        IA2_STATE_ICONIFIED = 0x20,
        IA2_STATE_INVALID = 0x40,
        IA2_STATE_INVALID_ENTRY = 0x80,
        IA2_STATE_MANAGES_DESCENDANTS = 0x100,
        IA2_STATE_MODAL = 0x200,
        IA2_STATE_MULTI_LINE = 0x400,
        IA2_STATE_OPAQUE = 0x800,
        IA2_STATE_REQUIRED = 0x1000,
        IA2_STATE_SELECTABLE_TEXT = 0x2000,
        IA2_STATE_SINGLE_LINE = 0x4000,
        IA2_STATE_STALE = 0x8000,
        IA2_STATE_SUPPORTS_AUTOCOMPLETION = 0x10000,
        IA2_STATE_TRANSIENT = 0x20000,
        IA2_STATE_VERTICAL = 0x40000
    };

    int result = 0;
    QAccessible::State state = iface->state(0);

    // TODO - add the other states
    if (state & QAccessible::Modal)
        result |= IA2_STATE_MODAL;

    return result;
}

QRect AccessibleObject::location() const
{
    return iface->rect(0);
}

int AccessibleObject::childCount()
{
    if (childrenDirty)
        updateChildren();
    return childrenPaths.count();
}

QList<QDBusObjectPath> AccessibleObject::children()
{
    if (childrenDirty)
        updateChildren();

    QList<QDBusObjectPath> result;
    foreach (QString childPath, childrenPaths)
        result << QDBusObjectPath(childPath);

    return result;
}

void AccessibleObject::updateChildren()
{
    if (!client)
        return;

    // we only have to check for new children - deleted
    // children are handled via objectDestroyed

    AccessibleCache *cache = accessibleCache();
    const QObjectList objects = clientChildren();
    for (int i = 0; i < objects.count(); ++i) {
        QObject *obj = objects.at(i);
        if (!obj->isWidgetType())
            continue;
        if (!cache->contains(obj))
            registerChild(obj);
    }

    childrenDirty = false;
}

QString AccessibleObject::path() const
{
    return dPath;
}

bool AccessibleObject::setFocus()
{
    QWidget *widget = client->isWidgetType()
                         ? static_cast<QWidget *>(client)
                         : static_cast<QWidget *>(0);
    if (!widget)
        return false;

    widget->setFocus();
    return true;
}

// declared in qdbusutil_p.h
namespace QDBusUtil {
    extern bool isValidObjectPath(const QString &path);
}

QString AccessibleObject::uniqueName(QObject *child)
{
    const QString rootPath = dPath + QLatin1Char('/');
    QString childPath = rootPath;

    QString childName = child->objectName();
    if (!childName.isEmpty()) {
        childName.replace(QLatin1Char('/'), QLatin1Char(' '));
        childPath += childName;
    }

    if (childName.isEmpty() || !QDBusUtil::isValidObjectPath(childPath)) {
        // the object name is not D-Bus conform, fall back to class name
        childPath = rootPath + QLatin1String(child->metaObject()->className());
    }

    // append numbers until we hit a unique name
    QString uniqueChildPath = childPath;
    while (childrenPaths.contains(uniqueChildPath))
        uniqueChildPath = childPath + QString::number(++uniqueChildId);

    return uniqueChildPath;
}

AccessibleObject *AccessibleObject::registerChild(QObject *child)
{
    Q_ASSERT(child);

    AccessibleObject *accObj = exportedObject(child);
    if (!accObj) {
        QString childPath = uniqueName(child);
        accObj = new AccessibleObject(childPath, child, this);
        childrenPaths << childPath;
    }

    return accObj;
}

void AccessibleObject::objectDestroyed()
{
    delete this;
}

AccessibleObject *AccessibleObject::exportedObject(QObject *obj)
{
    return accessibleCache()->value(obj, 0);
}

QObjectList AccessibleObject::clientChildren() const
{
    return client ? client->children() : QObjectList();
}

AccessibleObject *AccessibleObject::ensureExported(QObject *obj)
{
    if (!obj)
        return 0;
    if (AccessibleObject *accObj = exportedObject(obj))
        return accObj;

    // make sure all our parents are exported
    QObject *parentObject = obj->parent();
    AccessibleObject *accParent = ensureExported(parentObject
            ? parentObject
            : QApplication::instance());
    if (!accParent)
        return 0;

    return accParent->registerChild(obj);
}

QString AccessibleObject::hitTest(int x, int y)
{
    if (!client || !client->isWidgetType())
        return QString();

    AccessibleObject *accObj = ensureExported(static_cast<QWidget *>(client)->childAt(x, y));
    if (!accObj)
        return QString();

    return accObj->path();
}

int AccessibleObject::indexInParent() const
{
    // TODO - is -1 a reasonable "no found" return value?
    if (AccessibleObject *parentObject = qobject_cast<AccessibleObject *>(parent()))
        return parentObject->childrenPaths.indexOf(dPath);

    return -1;
}

QDBusObjectPath AccessibleObject::navigate(int direction)
{
    QAccessibleInterface *iface = accessibleInterface();
    if (!iface || !iface->isValid())
        return QDBusObjectPath();
    QAccessibleInterface *dest = 0;
    int child = -1;

    switch (direction) {
    case QAccessible::Up:
    case QAccessible::Down:
    case QAccessible::Left:
    case QAccessible::Right:
    case QAccessible::Ancestor:
    case QAccessible::Child:
        child = iface->navigate(QAccessible::RelationFlag(direction), 1, &dest);
        if (child == -1)
            return QDBusObjectPath();
        if (dest) {
            AccessibleObject *destObj = ensureExported(dest->object());
            delete dest;
            return QDBusObjectPath(destObj ? destObj->path() : QString());
        }
        break;
    }

    return QDBusObjectPath();
}

