#ifndef QGSSPIT_H
#define QGSSPIT_H
/***************************************************************************
                        qgsspit.h  -  description
                           -------------------
  begin                : Fri Dec 19 2003
  copyright            : (C) 2003 by Denis Antipov
                       : (C) 2004 by Gary Sherman
  email                :
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
/* $Id: qgsspit.h 5934 2006-10-09 06:38:04Z g_j_m $ */
#include <vector>
#include <algorithm>

#include <QStringList>
#include <QString>
#include <QItemDelegate>

#include "qgsshapefile.h"
#include "ui_qgsspitbase.h"

class QTableWidgetItem;

extern "C"
{
#include <libpq-fe.h>
}

class QgsSpit : public QDialog, private Ui::QgsSpitBase
{
  Q_OBJECT;
public:
  QgsSpit( QWidget *parent = 0, Qt::WFlags fl = 0 );
  ~QgsSpit();
  //! Populate the list of available database connections
  void populateConnectionList();
  //! Connect to the selected database
  void dbConnect() {};
  //! Return a list of selected tables
  QStringList selectedTables();
  //! Return the connection info
  QString connInfo();
  //! Create a new PostgreSQL connection
  void newConnection();
  //! Edit a PostgreSQL connection
  void editConnection();
  //! Remove a PostgreSQL connection
  void removeConnection();
  //! Add file to the queue
  void addFile();
  //! Remove selected file from the queue
  void removeFile();
  //! Remove all files from the queue
  void removeAllFiles();
  //! Use the default SRID (Spatial Reference ID)
  void useDefaultSrid();
  //! Use the default geometry field name (the_geom)
  void useDefaultGeom();
  //! Show brief help
  void helpInfo();
  //! Get schemas available in the database
  void getSchema();
  void updateSchema();
  //! Import shapefiles into PostgreSQL
  void import();
  //! Edit the column names for a shapefile in the queue
  void editColumns( int, int, int, const QPoint & );
  //! Edit import properties of a shapefile in the queue
  void editShapefile( int, int, int, const QPoint & );

public slots:

  // In porting from Qt3 to Qt4 it was easier to have these small
  // redirects for the widget signals rather than rename the existing
  // functions (which would be been connected to the widgets using the
  // Qt3 designer signal/slot connection mechanism), in case you were
  // wondering. 
  void on_btnConnect_clicked()    { dbConnect();        }
  void on_btnEdit_clicked()       { editConnection();   }
  void on_btnNew_clicked()        { newConnection();    }
  void on_btnRemove_clicked()     { removeConnection(); }
  void on_btnImport_clicked()     { import();           } 
  void on_btnHelp_clicked()       { helpInfo();         }
  void on_btnQuit_clicked()       { close();           }
  void on_btnAddFile_clicked()    { addFile();          }
  void on_btnRemoveAll_clicked()  { removeAllFiles();   }
  void on_btnRemoveFile_clicked() { removeFile();       }
  void on_tblShapefiles_itemClicked(QTableWidgetItem* item) 
    { tblShapefiles->editItem(item); }
  // When the user changes the selected connection, update the schema list
  void on_cmbConnections_activated(int) { getSchema(); }
  void on_chkUseDefaultSrid_toggled(bool) { useDefaultSrid(); }
  void on_chkUseDefaultGeom_toggled(bool) { useDefaultGeom(); }

private:

  // Enum of table columns indexes
  enum ShpTableColumns
  {
    ColFILENAME = 0,
    ColFEATURECLASS = 1, // is editable
    ColFEATURECOUNT = 2,
    ColDBRELATIONNAME = 3,  // is editable
    ColDBSCHEMA = 4  // is editable
  };
  
  PGconn* checkConnection();
  QStringList schema_list;
  QStringList geometry_list;
  int total_features;
  std::vector <QgsShapeFile *> fileList;
  int defSrid;
  QString defGeom;
  int defaultSridValue;
  QString defaultGeomValue;
  QString gl_key;
};

// We want to provide combo boxes in the table of shape files to
// load. Qt4 doesn't provide an 'out-of-the-box' way to do this
// (unlike Qt3), so we have to use the Qt4 delegate technique to
// provide combo boxes for the table, hence this class...

class ShapefileTableDelegate : public QItemDelegate
{
  Q_OBJECT;

 public:
  ShapefileTableDelegate(QObject *parent, QStringList& schema_list) :
    mSchemaList(schema_list) {}

  QWidget *createEditor(QWidget *parent, const QStyleOptionViewItem &option,
                        const QModelIndex &index) const;

  void setEditorData(QWidget *editor, const QModelIndex &index) const;
  void setModelData(QWidget *editor, QAbstractItemModel *model,
                    const QModelIndex &index) const;

  void updateEditorGeometry(QWidget *editor,
             const QStyleOptionViewItem &option, const QModelIndex &index) const;
  void updateSchemaList(QStringList& schema_list, QString currentSchema);

 private:
  QStringList mSchemaList;
  int mCurrentIndex;
};

#endif
