/*
 * STQGLExampleObjects.h
 * $Id:
 *
 * Copyright (C) 2001 Markus Janich
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * As a special exception to the GPL, the QGLViewer authors (Markus
 * Janich, Michael Meissner, Richard Guenther, Alexander Buck and Thomas
 * Woerner) give permission to link this program with Qt (non-)commercial
 * edition, and distribute the resulting executable, without including
 * the source code for the Qt (non-)commercial edition in the source
 * distribution.
 *
 */

//  Description : Definition of example nodes


    
#ifndef STQGLEXAMPLEOBECTS_H
#define STQGLEXAMPLEOBECTS_H


// Own
////////
#include "QSceneTreeNode.h"
#include "QViewingModeDialog.h"



/** Base node of a drawable sphere. */
class STQGLExampleSphere : public QSceneTreeDrawable
{
public:

  /** Default constructor. */
  STQGLExampleSphere(const CP3D &cCenter, float radius, 
		     int n1,int n2,
		     float red, float green, float blue,
		     float lw=2.5)
    : QSceneTreeDrawable(),
      m_radius(radius),
      m_lw(lw),
      m_red(red),
      m_green(green),
      m_blue(blue),
      m_n1(n1),
      m_n2(n2)
    { setCenter(cCenter); };

  /** Defines the global coordinates of the sphere. */
  virtual void setCenter(const CP3D &cCenter) {
    resetTransformation();
    applyTransformation(CMat4D::PTranslate(cCenter.getCV3D()));
  };

  /** Returns the center coordinates. */
  virtual const CP3D getCenter() {
    CV4D cCentrVec = (*getTransformation())(3);

    return CP3D(cCentrVec.getX(), cCentrVec.getY(), cCentrVec.getZ());
  };

  /** Returns the bounding box of the sphere. */
  virtual CBoundingBox3D getBoundingBox() const {
    return QSceneTreeDrawable::getBoundingBox() +
      CBoundingBox3D(CP3D(-m_radius,-m_radius,-m_radius),
		     CP3D(m_radius,m_radius,m_radius));
  };
    
  /** Reimplented method. Here it manages all double-clicks
    * over a scene object by popping up a dialog. */
  virtual bool event(const QEvent *pqEvent) {
    switch (pqEvent->type()) {
    case QEvent::MouseButtonDblClick: {
      const QMouseEvent *pqMouseEvent = (const QMouseEvent *)pqEvent;

      m_ViewingMode = normal;
      requestUpdate();

      m_ViewingMode = QViewingModeDialog::getViewingMode(pqMouseEvent->globalX(),
							 pqMouseEvent->globalY(),
							 normal);
      
      requestUpdate();
      break;
    }
    default:
      return false;
    }

    return true;
  };


protected:
  float m_radius, m_lw;
  float m_red, m_green, m_blue; 
  int m_n1, m_n2;
};



/** This node provides a drawable sphere in wire frame style. */
class STQGLExampleWiredSphere : public STQGLExampleSphere
{
public:

  /** Default constructor. */
  STQGLExampleWiredSphere(const CP3D &cCenter, float radius, 
			  int n1,int n2,
			  float red, float green, float blue,
			  float lw=2.5)
    : STQGLExampleSphere(cCenter,radius,n1,n2,red,green,blue,lw)
    {};

  /** Implementation of the inherited method. It does all
    * OpenGL drawing into the current context. */
  void draw();
};



/** This node provides a drawable sphere in solid style. */
class STQGLExampleSolidSphere : public STQGLExampleSphere
{
public:
  /** Default constructor. */
  STQGLExampleSolidSphere(const CP3D &cCenter, float radius, 
			  int n1,int n2,
			  float red, float green, float blue)
    : STQGLExampleSphere(cCenter,radius,n1,n2,red,green,blue)
    {
      int i;
      for (i=0; i<16; i++) {
	m_anMask[8*i] = m_anMask[8*i+1] = 
	  m_anMask[8*i+2] = m_anMask[8*i+3] = 0xAA;
	m_anMask[8*i+4] = m_anMask[8*i+5] = 
	  m_anMask[8*i+6] = m_anMask[8*i+7] = 0x55;
      }
    };

  /** Implementation of the inherited method. It does all
    * OpenGL drawing into the current context. */
  void draw();


protected:
  GLubyte m_anMask[128];
};


/** This node is for doing some initialisation stuff. */
class STQGLExampleInitNode : public QSceneTreeDrawable
{
public:
  /** Default constructor. */
  STQGLExampleInitNode() : QSceneTreeDrawable() {};

  /** Implementation of the inherited method. It does all
    * OpenGL drawing into the current context. */
  void draw();
};


/** This node switches the lights on for all nodes
  * that are traversed after this node while drawing
  * the scene tree. */
class STQGLExampleLightNode : public QSceneTreeDrawable
{
public:
  /** Default constructor. */
  STQGLExampleLightNode() : QSceneTreeDrawable() {};

  /** Implementation of the inherited method. It does all
    * OpenGL drawing into the current context. */
  void draw() {
    glEnable(GL_LIGHT0);
    glEnable(GL_LIGHTING);
  };
};


#endif   // STQGLEXAMPLEOBECTS_H
