/***************************************************************************
 *   Copyright (C) 2009 by The qGo Project                                 *
 *                                                                         *
 *   This file is part of qGo.   					   *
 *                                                                         *
 *   qGo is free software: you can redistribute it and/or modify           *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, see <http://www.gnu.org/licenses/>   *
 *   or write to the Free Software Foundation, Inc.,                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#include "tree.h"
#include "move.h"
#include "matrix.h"
#include "group.h"
#include "messages.h"

#include <vector>

#include <QtCore>

Tree::Tree(int board_size, float komi)
    : boardSize(board_size), root(NULL), komi(komi)
{
    insertStoneFlag = false;
	checkPositionTags = NULL;
    init();
    loadingSGF = false;
	koStoneX = 0;
    koStoneY = 0;
}

void Tree::init()
{
	if(root)
		clear();
    root = new Move(boardSize);
	// node index used for IGS review
	root->setNodeIndex(1);
	lastMoveInMainBranch = current = root;
	
	if(checkPositionTags)
		delete checkPositionTags;
    checkPositionTags = new Matrix(boardSize);
}

Tree::~Tree()
{
	delete checkPositionTags;
    clear();
}

/*
 *  Adds a brother at the end of the brother chain of current()
 */
// may be kind of silly if we move it to the Move class FIXME
bool Tree::addBrother(Move *node)
{
	if (root == NULL)
	{
		qFatal("Error: No root!");
	}
	else
	{
		if (current == NULL)
		{
			qFatal("Error: No current node!");
			return false;
		}
		
		current->addBrother(node);
	}
	
	current = node;
	
	return true;
}

/*
 * Returns: true  - A son found, added as brother
 *          false - No son found, added as first son
 */
bool Tree::addSon(Move *node)
{
	if (root == NULL)
	{
		qFatal("Error: No root!");
		return false;
	}
    if (current == NULL)
    {
        qFatal("Error: No current node!");
        return false;
    }

    // current node has no son?
    if (current->son == NULL)
    {
        if(current == node)
        {
            qDebug("Attempting to add move as its own son!");
            return false;
        }
        current->son = node;

        node->parent = current;
        node->setTimeinfo(false);
        current = node;
        if(isInMainBranch(current->parent))
            lastMoveInMainBranch = current;
#ifdef NOTWORKING
        // This is from an undo, but its awkward here
        // presumably son would not have been null with a marker
        // if this wasn't from undos or online review or the like
        // we'll understand this better if we look into online
        // review code FIXME
        if(current->parent->marker)
        {
            if(!addBrother(current->parent->marker))
            {
                qFatal("Failed to add undo brother.");
                return false;
            }
            else
            {
                Move * m = current->parent->marker;
                current->parent->marker = NULL;
                //check for multiple undos in a row
                while(m->marker && m->son == NULL)	//is son check right?
                {
                    m->son = m->marker;
                    m->marker = NULL;
                    m = m->son;
                }
            }
        }
#endif //NOTWORKING
        return false;
    }
    // A son found. Add the new node as farest right brother of that son
    else
    {
        current = current->son;
        if (!addBrother(node))
        {
            qFatal("Failed to add a brother.");
            return false;
        }
        if(!loadingSGF)		//since this would put every branch on the last brother
            current->parent->marker = node;

        current = node;
        return true;
    }

}

bool Tree::hasNextBrother()
{
	if (current == NULL || current->brother == NULL)
		return false;
	
	return current->brother != NULL;
}

bool Tree::hasPrevBrother(Move *m)
{
	if (m == NULL)
		m = current;
	
	if (root == NULL || m == NULL)
		return false;
	
	Move *tmp;
	
	if (m->parent == NULL)
	{
		if (m == root)
			return false;
		else
			tmp = root;
	}
	else
		tmp = m->parent->son;
	
	return tmp != m;
}

int Tree::getNumSons(Move *m)
{
	if (m == NULL)
	{
		if (current == NULL)
			return 0;
		
		m = current;
	}
	
	return m->getNumSons();
//	Move *tmp = m->son;
	
//	if (tmp == NULL)
//		return 0;
	
//	int counter = 1;
//	while ((tmp = tmp->brother) != NULL)
//		counter ++;
	
//	return counter;
}

int Tree::getBranchLength(Move *node)
{
	Move *tmp;
	
	if (node == NULL)
	{
		if (current == NULL)
			return -1;
		tmp = current;
	}
	else
		tmp = node;
	
	Q_CHECK_PTR(tmp);
	
	int counter=0;
	// Go down the current branch, use the marker if possible to remember a previously used path
	while ((tmp = (tmp->marker != NULL ? tmp->marker : tmp->son)) != NULL)
		counter ++;
	
	return counter;
}

Move* Tree::nextMove()
{
	if (root == NULL || current == NULL || current->son == NULL)
		return NULL;
	
	if (current->marker == NULL)  // No marker, simply take the main son
        current = current->son;
	else
        current = current->marker;  // Marker set, use this to go the remembered path in the tree
	
	current->parent->marker = current;  // Parents remembers this move we went to
    return current;
}

Move* Tree::previousMove()
{
	if (root == NULL || current == NULL || current->parent == NULL)
		return NULL;
	
	current->parent->marker = current;  // Remember the son we came from
    current = current->parent;
    return current;
}

Move* Tree::nextVariation()
{
	if (root == NULL || current == NULL || current->brother == NULL)
		return NULL;
	
    current = current->brother;
    return current;
}

Move* Tree::previousVariation()
{
	if (root == NULL || current == NULL)
		return NULL;
	
	Move *tmp, *old;
	
	if (current->parent == NULL)
	{
		if (current == root)
			return NULL;
		else
			tmp = root;
	}
	else
		tmp = current->parent->son;
	
	old = tmp;
	
	while ((tmp = tmp->brother) != NULL)
	{
		if (tmp == current)
		{
            current = old;
            return current;
		}
		old = tmp;
	}
	
	return NULL;
}

bool Tree::hasSon(Move *m)
{
	if (root == NULL || m == NULL || current == NULL || current->son == NULL)
		return false;
	
	Move *tmp = current->son;
	
	do {
		if (m->equals(tmp))
		{
            current = tmp;
			return true;
		}
	} while ((tmp = tmp->brother) != NULL);
	
	return false;
}

void Tree::clear()
{
#ifndef NO_DEBUG
	qDebug("Tree had %d nodes.", count());
#endif
	
	if (root == NULL)
		return;
	
	traverseClear(root);
	
	root = NULL;
	current = NULL;
}

/*
 * Traverse the tree and deletes all moves after the given move (including the move)
 */
void Tree::traverseClear(Move *m)
{
	Q_CHECK_PTR(m);
	QStack<Move*> stack;
	QStack<Move*> trash;

	Move *t = NULL;

	if(isInMainBranch(m))
		lastMoveInMainBranch = m->parent;
	//drop every node into stack trash	
	stack.push(m);
	
	while (!stack.isEmpty())
	{
		t = stack.pop();
		if (t != NULL)
		{
			trash.push(t);
			stack.push(t->brother);
			stack.push(t->son);
		}
	}
	
	// Clearing this stack deletes all moves. Smart, eh?
	//trash.clear();
	// QT doc advises this code instead of the above
	while (!trash.isEmpty())
		delete trash.pop();
}

/* This is slower than it could be, but traverseClear I think is seldom called
 * we also use this in addSon though... */
bool Tree::isInMainBranch(Move * m) const
{
	while(m->parent && m->parent->son == m)
		m = m->parent;
	if(m == root)
		return true;
	else
		return false;
}

/*
 * Find a move starting from the given in the argument in the 
 * all following branches.
 * Results are saved into the reference variable result.
 */
void Tree::traverseFind(Move *m, int x, int y, QStack<Move*> &result)
{
	Q_CHECK_PTR(m);
	QStack<Move*> stack;
	Move *t = NULL;
	
	// Traverse the tree and drop every node into stack result
	stack.push(m);
	
	while (!stack.isEmpty())
	{
		t = stack.pop();
		if (t != NULL)
		{
			if (t->getX() == x && t->getY() == y)
				result.push(t);
			stack.push(t->brother);
			stack.push(t->son);
		}
	}
}

/*
 * Find a move starting from the given in the argument in the main branch, or at marked move.
 */
Move* Tree::findMove(Move *start, int x, int y, bool checkmarker)
{
	if (start == NULL)
		return NULL;
	
	Move *t = start;
	
	do {
		if (t->getX() == x && t->getY() == y)
			return t;
		if (checkmarker && t->marker != NULL)
			t = t->marker;
		else
			t = t->son;
	} while (t != NULL);
	
	return NULL;
}

/*
 * Find a move starting from the given in the argument in the 
 * all following branches.
 */
Move* Tree::findNode(Move *m, int node)
{
	Q_CHECK_PTR(m);
	QStack<Move*> stack;
	Move *t = NULL;
	
	// Traverse the tree and drop every node into stack result
	stack.push(m);
	
	while (!stack.isEmpty())
	{
		t = stack.pop();
		if (t != NULL)
		{
			if (t->getNodeIndex() == node)
				return t;
			stack.push(t->brother);
			stack.push(t->son);
		}
	}

	// node index not found
	return NULL;
}

/*
 * count all moves in the tree.
 */
int Tree::count()
{
	if (root == NULL)
		return 0;
	
	QStack<Move*> stack;
	int counter = 0;
	Move *t = NULL;
	
	// Traverse the tree and count all moves
	stack.push(root);
	
	while (!stack.isEmpty())
	{
		t = stack.pop();
		if (t != NULL)
		{
			counter ++;
			stack.push(t->brother);
			stack.push(t->son);
		}
	}
	
	return counter;
}

void Tree::setCurrent(Move *m)
{
    current = m;
    emit currentMoveChanged(current);
}

void Tree::setToFirstMove()
{
	if (root == NULL)
		qFatal("Error: No root!");
    current = root;
}

int Tree::mainBranchSize()
{
	if (root == NULL || current == NULL )
		return 0;
	
	Move *tmp = root;
	
	int counter = 1;
	while ((tmp = tmp->son) != NULL)
		counter ++;
	
	return counter;    
}

Move *Tree::findLastMoveInMainBranch()
{
#ifdef OLD
	Move *m = root;
	Q_CHECK_PTR(m);

	if (m==NULL)
	return NULL;
  
	// Descend tree until root reached
	while (m->son != NULL)
		m = m->son;

	return m;
#endif //OLD
	return lastMoveInMainBranch;
}

Move *Tree::findLastMoveInCurrentBranch()
{
	Move *m = getCurrent();
	Q_CHECK_PTR(m);

	if (m==NULL)
	return NULL;
  
	while (m->son != NULL)
	{
		if(m->marker)
			m = m->marker;
		else
			m = m->son;
	}
	return m;
}

void Tree::addEmptyMove( bool /*brother*/)
{
	// qDebug("BoardHandler::createMoveSGF() - %d", mode);
	
	Move *m;
	
	Matrix *mat = current->getMatrix();
	m = new Move(stoneBlack, -1, -1, current->getMoveNumber()+1, phaseOngoing, *mat, true);
	/*else	//fastload
	{
		m = new Move(stoneBlack, -1, -1, current->getMoveNumber()+1, phaseOngoing);
	}*/
#ifdef FIXME
	if (!brother && hasSon(m))
	{
		/* FIXME in loading that Kogo's joseki dictionary we get
		 * about 20 of these, probably sources to a sgfparser issue */
		/* Okay, as far as I know, this function is never called with
		 * "brother", so it just always does addSon.
		 * Then addSon will add a brother if it should be a brother.
		 * Obviously this needs clarification */
		qDebug("*** HAVE THIS SON ALREADY! ***");
		delete m;
		return;
	}
#endif //FIXME	
	
	/* Below removed since brother never used */
	//if (!brother)
		addSon(m);
	//else
	//	addBrother(m);

}

/* These two functions are a little awkward here, just done to minimize
 * issues with qgoboard edit FIXME */
void Tree::updateCurrentMatrix(StoneColor c, int x, int y)
{
	// Passing?
	if (x == 20 && y == 20)
		return;
	
    if ((x < 1) || (x > boardSize) || (y < 1) || (y > boardSize))
	{
		qWarning("   *** Tree::updateCurrentMatrix() - Invalid move given: %d/%d at move %d ***",
			x, y, current->getMoveNumber());
		return;
	}
	
	Q_CHECK_PTR(current->getMatrix());
	if(current->getGamePhase() == phaseEdit)
		current->getMatrix()->insertStone(x, y, c, true);
	else
		current->getMatrix()->insertStone(x, y, c);
}

/* FIXME double check and remove editMove, unnecessary */
void Tree::doPass(bool /*sgf*/, bool /*fastLoad*/)
{
/////////////////FIXME We may have a problem at first move with handicap games ...
	StoneColor c = (current->getColor() == stoneWhite) ? stoneBlack : stoneWhite;
	
//	currentMove++;
	addMove(c, 20, 20);
#ifdef OLD
	if (!sgf)
		addMove(c, 20, 20);
	else  // Sgf reading
	{
		if (current->parent != NULL)
			c = current->parent->getColor() == stoneBlack ? stoneWhite : stoneBlack;
		else
			c = stoneBlack;
		if (!fastLoad)
			editMove(c, 20, 20);
	}
#endif //OLD
	if (current->parent != NULL)
		current->setCaptures(current->parent->getCapturesBlack(),
		current->parent->getCapturesWhite());
    emit currentMoveChanged(current);
}

bool Tree::checkMoveIsValid(StoneColor c, int x, int y)
{
    if ((x < 1 || x > boardSize || y < 1 || y > boardSize) && (x != 20 || y != 20))		//because 20,20 is pass, but ugly/unnecessary here FIXME
	{
		qWarning("Invalid position: %d/%d", x, y);
		return false;
	}

	/* special case, we're erasing a stone */
	if(c == stoneErase)
	{
        return (current->getMatrix()->getStoneAt(x,y) != stoneNone);
    }

    // If there is already a stone at (x,y) do nothing
    if ((current->getMatrix()->getStoneAt(x, y) == stoneBlack) ||
        (current->getMatrix()->getStoneAt(x, y) == stoneWhite))
        return false;

    bool koStone = ((current->getMoveNumber() > 1) && (current->parent->getMatrix()->getStoneAt(x, y) == c));

	//check for ko
	if(koStone && x == koStoneX && y == koStoneY)
		return false;

    std::vector<Group *> visited;
    std::vector<Group *>::iterator it_visited;
    int lastCaptures = current->getMatrix()->checkStoneCaptures(c,x,y,visited);
    // FIXME should not have to delete these groups here
    for (it_visited = visited.begin(); it_visited < visited.end(); ++it_visited)
    {
        delete (*it_visited);
    }

    if (lastCaptures < 0)
        return false;

	return true;
}

void Tree::addMove(StoneColor c, int x, int y)
{
    Move * lastValidMoveChecked =
            new Move(c, x, y, current->getMoveNumber() + 1, phaseOngoing, *(current->getMatrix()), true);	//clearMarks = true
    koStoneX = 0;
    koStoneY = 0;

	/* special case: pass */
	if(x == 20 && y == 20)
	{
        if (hasSon(lastValidMoveChecked))
            delete lastValidMoveChecked;
        else
            addSon(lastValidMoveChecked);
        return;
	}

    if (insertStoneFlag)
    {
        insertStone(lastValidMoveChecked);
        return;
    }

    if (c == stoneErase)
    {
        /* FIXME I don't think this is right.  First of all, it needs to be fixed in deleteNode() as well which is called as an undo
         * as well as as a tree edit.  But here, I don't think we want a new move added as a son to erase a move, I think we just
         * want to erase it... or maybe not because it makes sense to add it as a tree when its an undo in a game.
         * another point is that its not like a move in order, its like a separate tree, so addMove(stoneErase seems singularly
         * useless since we have to do something special with the tree anyway, i.e., no addSon, but maybe we go backwards and
         * delete some marker */
        lastValidMoveChecked->getMatrix()->insertStone(x, y, stoneErase);
    }

    if (hasSon(lastValidMoveChecked))
    {
        /* This happens a lot if we add a move that's already there
             * when really we're just playing along with the tree.
             * This would be the place to add any kind of tesuji testing
             * code */
        delete lastValidMoveChecked;
        return;
    }
    int lastCaptures = lastValidMoveChecked->getMatrix()->makeMove(x, y, c);
    addSon(lastValidMoveChecked);
    if (lastCaptures == 1)
        checkAddKoMark(current->getColor(), current->getX(), current->getY(), current);

	int capturesBlack, capturesWhite;
	if (current->parent != NULL)
	{
		capturesBlack = current->parent->getCapturesBlack();
		capturesWhite = current->parent->getCapturesWhite();
	}
	else
		capturesBlack = capturesWhite = 0;	

	if (c == stoneBlack)
		capturesBlack += lastCaptures;
	else if (c == stoneWhite)
        capturesWhite += lastCaptures;
    current->setCaptures(capturesBlack, capturesWhite);
    emit currentMoveChanged(current);
}

void Tree::addStoneToCurrentMove(StoneColor c, int x, int y)
{
	if (current->parent == root || current == root)
	{
		qDebug("setting that first weird move line %d", __LINE__);
		if(current->parent == root)
			qDebug("current parent root no new node %d %d", x, y);
		current->setMoveNumber(0); //TODO : make sure this is the proper way
		current->setX(-1);//-1
		current->setY(-1);//-1
		/* SGF submits handicap edit moves as root, they need to be
		 * given a color so that that color can flip over the cursor
		 * this is ugly, the cursor should be set by who's turn it
		 * is, by whatever means that's found, NOT by the color
		 * of the last move, to unify it.  Regardless, here's our
		 * solution.  Took me a while to find it because I assumed
		 * that the cursor and the move color we're linked when they're
		 * not.  Note also that this color is different from the
		 * individual colors of the matrix positions for this move */
		current->setColor(stoneBlack);
	}
	updateCurrentMatrix(c, x, y);
	//editMove(c, x, y);
}

/*
 *	Returns: true  - son of current move was found
 *           false - 
 */
bool Tree::insertStone(Move *node)
{
	if (root == NULL)
	{
		qFatal("Error: No root!");
		return false;
	}
	else
	{
		if (current == NULL)
		{
			qFatal("Error: No current node!");
			return false;
		}

		// current node has no son?
		if (current->son == NULL)
		{
			if(current == node)
			{
				qDebug("Attempting to add move as its own son!");
				return false;
			}

			current->son = node;
			node->parent = current;
			node->setTimeinfo(false);
            current = node;
			node->getMatrix()->insertStone(node->getX(), node->getY(), node->getColor());
			
			return false;
		}
		// A son found
		else
		{
			
			//we insert node between current and current->son
			node->parent = current;
			node->son = current->son;
			current->son->parent = node;
			current->son = node;

			//update all brothers to enable switching between them
			Move *t = node->son->brother;
			while (t != NULL) 
			{
				t->parent = node;
				t = t->brother;
			}
			current->parent->marker = current;
			node->marker = NULL;

			node->setTimeinfo(false);
            current = node;
			node->getMatrix()->insertStone(node->getX(), node->getY(), node->getColor());

			//update son - it is exclude from traverse search because we cannot update brothers of node->son
			node->son->setMoveNumber(node->son->getMoveNumber()+1);
			node->son->getMatrix()->insertStone(node->getX(), node->getY(), node->getColor());

			if (node->son->son != NULL)
			{
				// Traverse the tree and update every node (matrix and moveNum)
				QStack<Move*> stack;
				Move *t = NULL;
				stack.push(node->son->son);

				while (!stack.isEmpty())
				{
					t = stack.pop();
					if (t != NULL)
					{
						if (t->brother != NULL)
							stack.push(t->brother);
						if (t->son != NULL)
							stack.push(t->son);
						t->setMoveNumber(t->getMoveNumber()+1);
						t->getMatrix()->insertStone(node->getX(), node->getY(), node->getColor());
					}
				}
			}
			return true;
		}
	}
}

void Tree::undoMove(void)
{
#ifdef NOTWORKING
	previousMove();
	
	current->marker = current->son;
	current->son = NULL;
#endif //NOTWORKING
	deleteNode();
}

void Tree::checkAddKoMark(StoneColor c, int x, int y, Move * m)
{
	if(x == 20 && y == 20)	//awkward passing check FIXME
	{
		koStoneX = 0; koStoneY = 0; 	//necessary?? FIXME
		return;
	}
	if(!m)
        m = current;

	{
		StoneColor opp = (c == stoneBlack ? stoneWhite : stoneBlack);
		Matrix * trix = m->getMatrix();
		StoneColor testcolor;
		int sides = 3;
        if(x < boardSize)
		{
			testcolor = trix->getStoneAt(x + 1, y);
			if(testcolor == opp)
				sides--;
			else if(testcolor == stoneNone)
			{
				koStoneX = x + 1;
				koStoneY = y;
			}
		}
		else
			sides--;
		
		if(x > 1)
		{
			testcolor = trix->getStoneAt(x - 1, y);
			if(testcolor == opp)
				sides--;
			else if(testcolor == stoneNone)
			{
				koStoneX = x - 1;
				koStoneY = y;
			}
		}
		else
			sides--;
        if(y < boardSize)
		{
			testcolor = trix->getStoneAt(x, y + 1);
			if(testcolor == opp)
				sides--;
			else if(testcolor == stoneNone)
			{
				koStoneX = x;
				koStoneY = y + 1;
			}
		}
		else
			sides--;
		
		if(y > 1)
		{
			testcolor = trix->getStoneAt(x, y - 1);
			if(testcolor == opp)
				sides--;
			else if(testcolor == stoneNone)
			{
				koStoneX = x;
				koStoneY = y - 1;
			}
		}
		else
			sides--;
		
		if(sides != 0)
		{
			koStoneX = 0;
			koStoneY = 0;
		}
		else
		{
			/* Don't save to file */
			if(preferences.draw_ko_marker)
				trix->insertMark(koStoneX, koStoneY, markKoMarker);
		}
	}
}

int Tree::getLastCaptures(Move * m)
{
	if(!m->parent)
		return 0;
	if(m->getColor() == stoneWhite)
		return m->getCapturesWhite() - m->parent->getCapturesWhite();
	else if(m->getColor() == stoneBlack)
		return m->getCapturesBlack() - m->parent->getCapturesBlack();
	else
		return 0;
}

/*
 * this deletes the current node an all its sons
 */
void Tree::deleteNode()
{
	Move *m = getCurrent(),
		*remember = NULL,
		*remSon = NULL;
	Q_CHECK_PTR(m);
	
	if (m->parent != NULL)
	{
		remember = m->parent;
		
		// Remember son of parent if its not the move to be deleted.
		// Then check for the brothers and fix the pointer connections, if we
		// delete a node with brothers. (It gets ugly now...)
		// YUCK! I hope this works.
		if (remember->son == m)                  // This son is our move to be deleted?
		{
			if (remember->son->brother != NULL)  // This son has a brother?
				remSon = remember->son->brother; // Reset pointer
		}
		else                                     // No, the son is not our move
		{
			remSon = remember->son;
			Move *tmp = remSon, *oldTmp = tmp;
			
			do {   // Loop through all brothers until we find our move
				if (tmp == m)
				{
					if (m->brother != NULL)            // Our move has a brother?
						oldTmp->brother = m->brother;  // Then set the previous move brother
					else                               // to brother of our move
						oldTmp->brother = NULL;        // No brother found.
					break;
				}
				oldTmp = tmp;
			} while ((tmp = tmp->brother) != NULL);
		}
	}
	else if (hasPrevBrother())
	{
		remember = previousVariation();
		if (m->brother != NULL)
			remember->brother = m->brother;
		else
			remember->brother = NULL;
	}
	else if (hasNextBrother())
	{
		remember = nextVariation();
		// Urgs, remember is now root.
		setRoot(remember);
	}
	else
	{
		// Oops, first and only move. We delete everything
        init();
		return;
	}
	if(isInMainBranch(m))
		lastMoveInMainBranch = remember;
	if (m->son != NULL)
		traverseClear(m->son);  // Traverse the tree after our move (to avoid brothers)
    current = remember;
	delete m;                         // Delete our move
	//setCurrent(remember);       // Set current move to previous move
	remember->son = remSon;           // Reset son pointer, NULL
	remember->marker = NULL;          // Forget marker

    emit currentMoveChanged(current);
}


/*
 * Former BoardHandler slots
 */
void Tree::slotNavForward()
{
    nextMove();
    emit currentMoveChanged(current);
}

void Tree::slotNavBackward()
{
    previousMove();
    emit currentMoveChanged(current);
}

void Tree::slotNavFirst()
{
    setToFirstMove();  // Set move to root
    emit currentMoveChanged(current);
}

void Tree::slotNavLast()
{
    setCurrent(findLastMoveInCurrentBranch());
    emit currentMoveChanged(current);
}

void Tree::slotNavPrevComment()
{
    Move  *m = current->parent;

    while (m != NULL)
    {
        if (m->getComment() != "")
            break;
        if (m->parent == NULL)
            break;
        m = m->parent;
    }

    if (m != NULL)
    {
        setCurrent(m);
        emit currentMoveChanged(current);
    }
}

void Tree::slotNavNextComment()
{
    Move  *m = current->son;

    while (m != NULL)
    {
        if (m->getComment() != "")
            break;
        if (m->son == NULL)
            break;
        m = m->son;
    }

    if (m != NULL)
    {
        setCurrent(m);
        emit currentMoveChanged(current);
    }
}

void Tree::slotNthMove(int n)
{
    if (n < 0)
        return;

    Move *m = current,
        *old = m;

    int currentMove = m->getMoveNumber();

    while (m != NULL)
    {
        if (m->getMoveNumber() == n)
            break;
        if ((n >= currentMove && m->son == NULL && m->marker == NULL) ||
            (n < currentMove && m->parent == NULL))
            break;
        if (n > currentMove)
        {
            if (m->marker == NULL)
                m = m->son;
            else
                m = m->marker;
            m->parent->marker = m;
        }
        else
        {
            m->parent->marker = m;
            m = m->parent;
        }
    }

    if (m != NULL && m != old)
        gotoMove(m);
}

void Tree::slotNavNextVar()
{
    Move *m = nextVariation();
    if (m != NULL)
        emit currentMoveChanged(current);
}

void Tree::slotNavPrevVar()
{
    Move *m = previousVariation();
    if (m != NULL)
        emit currentMoveChanged(current);
}

void Tree::slotNavStartVar()
{
    if (current->parent == NULL)
        return;

    Move *tmp = previousMove(),
        *m = NULL;

    if (tmp == NULL)
        return;

    // Go up tree until we find a node that has > 1 sons
    while ((m = previousMove()) != NULL && getNumSons() <= 1)
    // Remember move+1, as we set current to the
    // first move after the start of the variation
        tmp = m;


    if (m == NULL)  // No such node found, so we reached root.
    {
        tmp = getRoot();
        // For convinience, if we have Move 1, go there. Looks better.
        if (tmp->son != NULL)
            tmp = nextMove();
    }

    // If found, set current to the first move inside the variation
    setCurrent(tmp);
    emit currentMoveChanged(current);
}

void Tree::slotNavNextBranch()
{
    Move *m = getCurrent(),
        *remember = m;  // Remember current in case we dont find a branch
    Q_CHECK_PTR(m);

    // We are already on a node with 2 or more sons?
    if (getNumSons() > 1)
    {
        m = nextMove();
        emit currentMoveChanged(current);
        return;
    }

    // Descent tree to last son of main variation
    while (m->son != NULL && getNumSons() <= 1)
        m = nextMove();

    if (m != NULL && m != remember)
    {
        if (m->son != NULL)
            m = nextMove();
        emit currentMoveChanged(current);
    }
    else
        setCurrent(remember);
}

/*
 * This function resumes back to the first move in the main branch
 */
void Tree::slotNavMainBranch()
{
    if (current->parent == NULL)
        return;

    Move *m = current,
        *old = m,
        *lastOddNode = NULL;

    if (m == NULL)
        return;

    while ((m = m->parent) != NULL)
    {
        if (getNumSons(m) > 1 && old != m->son)
            // Remember a node when we came from a branch
            lastOddNode = m;

        m->marker = old;
        old = m;
    }

    if (lastOddNode == NULL)
        return;

    Q_CHECK_PTR(lastOddNode);

    // Clear the marker, so we can proceed in the main branch
    lastOddNode->marker = NULL;

    setCurrent(lastOddNode);
    emit currentMoveChanged(current);
}

void Tree::gotoMove(Move *m)
{
    setCurrent(m);
    emit currentMoveChanged(current);
}

/*
 * Called after the preceding (slot nav Intersection)
 * When the intersection 'x/y' has been clicked on
 */
void Tree::findMoveByPos(int x, int y)
{
    Move *m = findMoveInMainBranch(x, y);
    if(!m)
        findMoveInCurrentBranch(x, y);

    if (m != NULL)
    {
        setCurrent(m);
    }
}

/*
 * Called by qgoboard when score button is pressed up, leaving score mode
 */
void Tree::exitScore()
{
    // Remove territory marks
    if (current->isTerritoryMarked())
    {
        current->getMatrix()->clearTerritoryMarks();
        current->setTerritoryMarked(false);
        current->setScored(false);
    }

    current->getMatrix()->absMatrix();
    emit currentMoveChanged(current);
}

/*
 * Performs all operations on the matrix of current move to display score marks
 * and score informaton on the uI
 */
void Tree::countScore(void)
{
    Matrix * current_matrix = current->getMatrix();
    current_matrix->clearTerritoryMarks();

    deadBlack = current_matrix->countDeadBlack();
    deadWhite = current_matrix->countDeadWhite();
    capturesBlack = current->getCapturesBlack();
    capturesWhite = current->getCapturesWhite();
    current_matrix->markTerritory(terrBlack,terrWhite);
    current->setTerritoryMarked(true);
    current->setScored(true);

    emit currentMoveChanged(current); // Toggles window refresh
    emit scoreChanged(terrBlack, capturesBlack, deadWhite,
                      terrWhite, capturesWhite, deadBlack);
}

void Tree::countMarked(void)
{
    Matrix * current_matrix = current->getMatrix();

    deadWhite = 0;
    deadBlack = 0;
    terrWhite = 0;
    terrBlack = 0;
    capturesBlack = current->getCapturesBlack();
    capturesWhite = current->getCapturesWhite();

    for (int i=0; i< boardSize; i++)
        for (int j=0; j< boardSize; j++)
        {

            /* When called from network code, we're just using
             * the board as server has reported it.  No stones
             * are marked as dead, but apparently ones marked as
             * territory get ghosted out */
            if(current_matrix->getMarkAt(i + 1, j + 1) == markTerrBlack)
            {
                terrBlack++;
                if (current_matrix->getStoneAt(i+1,j+1) == stoneWhite)
                    deadWhite++;
            }
            else if(current_matrix->getMarkAt(i + 1, j + 1) == markTerrWhite)
            {
                terrWhite++;
                if (current_matrix->getStoneAt(i +1,j+1) == stoneBlack)
                    deadBlack++;
            }
        }
    emit scoreChanged(terrBlack, capturesBlack, deadWhite,
                      terrWhite, capturesWhite, deadBlack);
}

/* Not totally confident that this belongs here, but
 * the score is counted here */
GameResult Tree::retrieveScore(void)
{
    GameResult g;
    g.result = GameResult::SCORE;
    /* What about different scoring types? (chinese versus japanese)
     * FIXME This basically confirms for me that this does not
     * belong here */

    float blackScore = terrBlack + capturesBlack + deadWhite;
    float whiteScore = terrWhite + capturesWhite + deadBlack + komi;
    if(whiteScore > blackScore)
    {
        g.winner_color = stoneWhite;
        g.winner_score = whiteScore;
        g.loser_score = blackScore;
    }
    else
    {
        g.winner_color = stoneBlack;
        g.winner_score = blackScore;
        g.loser_score = whiteScore;
    }
    return g;
}
