/***************************************************************************
 *   Copyright (C) 2007 by Κωνσταντίνος Σμάνης   *
 *   kon.smanis@gmail.com   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation                                              *
 *   51 Franklin Street, Fifth Floor                                       *
 *   Boston, MA 02110-1301 USA                                             *
 ***************************************************************************/

#include "qgrubeditor.h"

QGRUBEditor::QGRUBEditor( QWidget *parent ) : QMainWindow( parent )
{
//0.	INITIALISE POINTERS
	file = new FileTransactions();
	config = new Config();
//1.	TRANSLATE & LOAD THE UI
	if ( QFile::exists( config->settings.language ) )
	{
		QTranslator *translator = new QTranslator();
		translator->load( config->settings.language );
		QApplication::installTranslator( translator );
	}

	setupUi( this );
//2.	CREATE/CUSTOMISE CUSTOM OBJECTS
	customObjects();
	customActions();
	customToolbars();
	customMenus();
//3.	APPLY SETTINGS
	applySettings();

	resize( config->settings.window_size );
	restoreGeometry( config->settings.window_geometry );
	restoreState( config->settings.window_state );
//4.	CONNECTIONS
	makeConnections();
//5.	DO THE REAL WORK ( SHOW INPUT )
	scan();
	osIndexChanged( 0 );
}
QGRUBEditor::~QGRUBEditor()
{
	delete file;
	delete config;
}
//EVENTS
void QGRUBEditor::contextMenuEvent( QContextMenuEvent *event )
{
	globalContextMenu->exec( event->globalPos() );
}
void QGRUBEditor::contextMenuToolbar()
{
	toolbarContextMenu->exec( QCursor::pos() );
}
void QGRUBEditor::contextMenuEntry()
{
	if ( !listWidget_os->selectedItems().isEmpty() )
	{
		actionSet_As_Default->setEnabled( GRUBSETTINGS.default_entry() != listWidget_os->currentRow() );
		actionSet_As_Fallback->setEnabled( GRUBSETTINGS.fallback() != listWidget_os->currentRow() );
		entryContextMenu->exec( QCursor::pos() );
	}
}
void QGRUBEditor::closeEvent( QCloseEvent * )
{
	config->writeValue( "window/size", size() );
	config->writeValue( "window/geometry", saveGeometry() );
	config->writeValue( "window/state", saveState() );
}
void QGRUBEditor::rowChanged( const int row )
{
	if ( !listWidget_os->currentItem() )
		return;

	comboBox_entry->setCurrentIndex( row );

	treeWidget_property->clear();
	//TITLE
	if ( !GRUBVECTOR.at(row).title().isEmpty() )
	{
		treeWidgetItem = new QTreeWidgetItem( treeWidget_property );
		treeWidgetItem->setText( 0, tr( "Title" ) );
		treeWidgetItem->setText( 1, GRUBVECTOR.at( row ).title() );
		treeWidgetItem->setFlags( Qt::ItemIsEnabled | Qt::ItemIsSelectable );
	}
	//ROOT
	if ( !GRUBVECTOR.at( row ).root().isEmpty() )
	{
		treeWidgetItem = new QTreeWidgetItem( treeWidget_property );
		treeWidgetItem->setText( 0, tr( "Root" ) );
		treeWidgetItem->setText( 1, GRUBVECTOR.at( row ).root() );
		treeWidgetItem->setFlags( Qt::ItemIsEnabled | Qt::ItemIsSelectable );
	}
	//KERNEL
	if ( !GRUBVECTOR.at( row ).kernel().isEmpty() )
	{
		treeWidgetItem = new QTreeWidgetItem( treeWidget_property );
		treeWidgetItem->setText( 0, tr( "Kernel" ) );
		treeWidgetItem->setText( 1, GRUBVECTOR.at( row ).kernel() );
		treeWidgetItem->setFlags( Qt::ItemIsEnabled | Qt::ItemIsSelectable );
	}
	//INITRD
	if ( !GRUBVECTOR.at( row ).initrd().isEmpty() )
	{
		treeWidgetItem = new QTreeWidgetItem( treeWidget_property );
		treeWidgetItem->setText( 0, tr( "Initrd" ) );
		treeWidgetItem->setText( 1, GRUBVECTOR.at( row ).initrd() );
		treeWidgetItem->setFlags( Qt::ItemIsEnabled | Qt::ItemIsSelectable );
	}
	//CHAINLOADER
	if ( !GRUBVECTOR.at( row ).chainLoader().isEmpty() )
	{
		treeWidgetItem = new QTreeWidgetItem( treeWidget_property );
		treeWidgetItem->setText( 0, tr( "Chainloader" ) );
		treeWidgetItem->setText( 1, GRUBVECTOR.at( row ).chainLoader() );
		treeWidgetItem->setFlags( Qt::ItemIsEnabled | Qt::ItemIsSelectable );
	}
	//QUIET
	if ( GRUBVECTOR.at( row ).quiet() )
	{
		treeWidgetItem = new QTreeWidgetItem( treeWidget_property );
		treeWidgetItem->setText( 0, tr( "Quiet" ) );
		treeWidgetItem->setCheckState( 1, Qt::Checked );
		treeWidgetItem->setFlags( Qt::ItemIsEnabled | Qt::ItemIsSelectable );
	}
	//SAVEDEFAULT
	if ( GRUBVECTOR.at( row ).saveDefault() )
	{
		treeWidgetItem = new QTreeWidgetItem( treeWidget_property );
		treeWidgetItem->setText( 0, tr( "SaveDefault" ) );
		treeWidgetItem->setCheckState( 1, Qt::Checked );
		treeWidgetItem->setFlags( Qt::ItemIsEnabled | Qt::ItemIsSelectable );
	}
	//MAKEACTIVE
	if ( GRUBVECTOR.at( row ).makeActive() )
	{
		treeWidgetItem = new QTreeWidgetItem( treeWidget_property );
		treeWidgetItem->setText( 0, tr( "MakeActive" ) );
		treeWidgetItem->setCheckState( 1, Qt::Checked );
		treeWidgetItem->setFlags( Qt::ItemIsEnabled | Qt::ItemIsSelectable );
	}
	//LOCK
	if ( GRUBVECTOR.at( row ).lock() )
	{
		treeWidgetItem = new QTreeWidgetItem( treeWidget_property );
		treeWidgetItem->setText( 0, tr( "Lock" ) );
		treeWidgetItem->setCheckState( 1, Qt::Checked );
		treeWidgetItem->setFlags( Qt::ItemIsEnabled | Qt::ItemIsSelectable );
	}
}
void QGRUBEditor::osIndexChanged( const int row )
{
	switch ( row )
	{
		case 0:
			label_chainloader->setDisabled( true );
			lineEdit_chainloader->setDisabled( true );
			lineEdit_chainloader->clear();

			label_kernel->setEnabled( true );
			lineEdit_kernel->setEnabled( true );
			pushButton_browse_kernel->setEnabled( true );

			label_initrd->setEnabled( true );
			lineEdit_initrd->setEnabled( true );
			pushButton_browse_initrd->setEnabled( true );

			checkBox_quiet->setEnabled( true );
		break;
		case 1:
			label_chainloader->setEnabled( true );
			lineEdit_chainloader->setEnabled( true );

			label_kernel->setDisabled( true );
			lineEdit_kernel->setDisabled( true );
			lineEdit_kernel->clear();
			pushButton_browse_kernel->setDisabled( true );

			label_initrd->setDisabled( true );
			lineEdit_initrd->setDisabled( true );
			lineEdit_initrd->clear();
			pushButton_browse_initrd->setDisabled( true );

			checkBox_quiet->setDisabled( true );
		break;
	}
}
void QGRUBEditor::rootIndexChanged( const int row )
{
	if ( row != -1 )
		lineEdit_root->setText( DEVICEVECTOR.at( row ).grubPartition() );
}
void QGRUBEditor::rootIndexChanged_2( const int row )
{
	if ( row != -1 )
		lineEdit_root_2->setText( DEVICEVECTOR.at( row ).grubPartition() );
}
void QGRUBEditor::entryIndexChanged( const int row )
{
	if ( row != -1 )
	{
		editClear();
		lineEdit_title_2->setText( GRUBVECTOR.at( row ).title() );

		for ( int i = 0 ; i < DEVICEVECTOR.size() ; i++ )
		{
			if ( DEVICEVECTOR.at( i ).grubPartition() == GRUBVECTOR.at( row ).root() )
			{
				comboBox_root_2->setCurrentIndex( i );
				break;
			}
		}

		lineEdit_kernel_2->setText( GRUBVECTOR.at( row ).kernel() );
		lineEdit_initrd_2->setText( GRUBVECTOR.at( row ).initrd() );
		lineEdit_chainloader_2->setText( GRUBVECTOR.at( row ).chainLoader() );
		checkBox_quiet_2->setChecked( GRUBVECTOR.at( row ).quiet() );
		checkBox_savedef_2->setChecked( GRUBVECTOR.at( row ).saveDefault() );
		checkBox_makeact_2->setChecked( GRUBVECTOR.at( row ).makeActive() );
		checkBox_lock_2->setChecked( GRUBVECTOR.at( row ).lock() );
	}
}
//FUNCTIONS
//no-gui
void QGRUBEditor::customObjects()
{
	completer = new QCompleter( this );
	completer->setModel( new QDirModel( completer ) );
	completer->setCaseSensitivity( Qt::CaseSensitive );

	lineEdit_menulst = new QLineEdit( toolBar_location );
	lineEdit_menulst->setObjectName( QString::fromUtf8( "lineEdit_menulst" ) );
	lineEdit_menulst->setCompleter( completer );
	lineEdit_menulst->setFocusPolicy( Qt::ClickFocus );

	pushButton_browse_menulst = new QPushButton( toolBar_location );
	pushButton_browse_menulst->setObjectName( QString::fromUtf8( "pushButton_browse_menulst" ) );
	pushButton_browse_menulst->setText( tr( "Browse" ) );
	pushButton_browse_menulst->setIcon( QIcon( QString::fromUtf8( ":/images/browse.png" ) ) );
}
void QGRUBEditor::customActions()
{
//menu
	actionMenu->setCheckable( true );
	actionMenu->setChecked( config->settings.menu );
//iconsize & toolbutton style
	actionSmall_16x16->setCheckable( true );
	actionMedium_22x22->setCheckable( true );
	actionBig_32x32->setCheckable( true );
	actionIcon_Only->setCheckable( true );
	actionText_Only->setCheckable( true );
	actionText_Beside_Icon->setCheckable( true );
	actionText_Under_Icon->setCheckable( true );

	QActionGroup *iconSize = new QActionGroup( this );
	iconSize->addAction( actionSmall_16x16 );
	iconSize->addAction( actionMedium_22x22);
	iconSize->addAction( actionBig_32x32 );
	QActionGroup *toolbuttonStyle = new QActionGroup( this );
	toolbuttonStyle->addAction( actionIcon_Only );
	toolbuttonStyle->addAction( actionText_Only );
	toolbuttonStyle->addAction( actionText_Beside_Icon );
	toolbuttonStyle->addAction( actionText_Under_Icon );
//toolbar
	actionView_Input->setShortcut( QString( "Ctrl+V" ) );
	actionQuit->setShortcut( QString( "Ctrl+Q" ) );

	actionAdd_Entry->setShortcut( QString( "Ctrl+A" ) );
	actionEdit_Entry->setShortcut( QString( "Ctrl+E" ) );
	actionCopy_Entry->setShortcut( QString( "Ctrl+C" ) );
	actionRemove_Entry->setShortcut( QString( "Ctrl+R" ) );
	actionMove_To_Top->setShortcut( QString( "Ctrl+PgUp" ) );
	actionMove_Up->setShortcut( QString( "Ctrl+Up" ) );
	actionMove_Down->setShortcut( QString( "Ctrl+Down" ) );
	actionMove_To_Bottom->setShortcut( QString( "Ctrl+PgDown" ) );
	actionScan_File->setShortcut( QString( "F5" ) );

	actionBackup_File->setShortcut( QString( "Ctrl+B" ) );
	actionRestore_File->setShortcut( QString( "Ctrl+T" ) );
	actionDelete_Backup->setShortcut( QString( "Ctrl+D" ) );
	actionEncrypt_Password->setShortcut( QString( "Ctrl+P" ) );
	actionCreate_GRUB_Splash_Image->setShortcut( QString( "Ctrl+I" ) );

	actionSettings->setShortcut( QString( "Ctrl+S" ) );

	actionWhat_s_This->setShortcut( QString( "Shift+F1" ) );
	actionAbout_QGRUBEditor->setShortcut( QString( "F1" ) );
	actionAbout_Qt->setShortcut( QString( "F2" ) );
}
void QGRUBEditor::customToolbars()
{
	toolBar_location->addAction( actionClear );
	toolBar_location->addAction( actionReload );
	toolBar_location->addWidget( lineEdit_menulst );
	toolBar_location->addAction( actionEnter );
	toolBar_location->addWidget( pushButton_browse_menulst );
}
void QGRUBEditor::customMenus()
{
//SUBMENUS
	//View
	menuView->addAction( toolBar_file->toggleViewAction() );
	menuView->addAction( toolBar_actions->toggleViewAction() );
	menuView->addAction( toolBar_settings->toggleViewAction() );
	menuView->addAction( toolBar_help->toggleViewAction() );
	menuView->addSeparator();
	menuView->addAction( toolBar_location->toggleViewAction() );
	//Icon Size
	menuIconSize = new QMenu();
	menuIconSize->setTitle( tr( "Icon Size" ) );
	menuIconSize->addAction( actionSmall_16x16 );
	menuIconSize->addAction( actionMedium_22x22 );
	menuIconSize->addAction( actionBig_32x32 );
	//Toolbutton Style
	menuToolbuttonStyle = new QMenu();
	menuToolbuttonStyle->setTitle( tr( "Toolbutton Style" ) );
	menuToolbuttonStyle->addAction( actionIcon_Only );
	menuToolbuttonStyle->addAction( actionText_Only );
	menuToolbuttonStyle->addAction( actionText_Beside_Icon );
	menuToolbuttonStyle->addAction( actionText_Under_Icon );
//MENUS
	//global
	globalContextMenu = new QMenu();
	globalContextMenu->addAction( actionMenu );
	//toolbar
	toolbarContextMenu = new QMenu();
	toolbarContextMenu->addMenu( menuView );
	toolbarContextMenu->addMenu( menuIconSize );
	toolbarContextMenu->addMenu( menuToolbuttonStyle );
	//entry
	entryContextMenu = new QMenu();
	entryContextMenu->addAction( actionSet_As_Default );
	entryContextMenu->addAction( actionSet_As_Fallback );
}

void QGRUBEditor::makeConnections()
{
//events
	connect( toolBar_file, SIGNAL( customContextMenuRequested( QPoint ) ), this, SLOT( contextMenuToolbar() ) );
	connect( toolBar_actions, SIGNAL( customContextMenuRequested( QPoint ) ), this, SLOT( contextMenuToolbar() ) );
	connect( toolBar_tools, SIGNAL( customContextMenuRequested( QPoint ) ), this, SLOT( contextMenuToolbar() ) );
	connect( toolBar_settings, SIGNAL( customContextMenuRequested( QPoint ) ), this, SLOT( contextMenuToolbar() ) );
	connect( toolBar_help, SIGNAL( customContextMenuRequested( QPoint ) ), this, SLOT( contextMenuToolbar() ) );
	connect( toolBar_location, SIGNAL( customContextMenuRequested( QPoint ) ), this, SLOT( contextMenuToolbar() ) );

	connect( listWidget_os, SIGNAL( customContextMenuRequested( QPoint ) ), this, SLOT( contextMenuEntry() ) );

	connect( listWidget_os, SIGNAL( itemDoubleClicked( QListWidgetItem * ) ), this, SLOT( actionedit_entry() ) );
	connect( listWidget_os, SIGNAL( currentRowChanged( int ) ), this, SLOT( rowChanged( int ) ) );

	connect( comboBox_os, SIGNAL( currentIndexChanged( int ) ), this, SLOT( osIndexChanged( int ) ) );
	connect( comboBox_root, SIGNAL( currentIndexChanged( int ) ), this, SLOT( rootIndexChanged( int ) ) );
	connect( comboBox_entry, SIGNAL( currentIndexChanged( int ) ), this, SLOT( entryIndexChanged( int ) ) );
	connect( comboBox_root_2, SIGNAL( currentIndexChanged( int ) ), this, SLOT( rootIndexChanged_2( int ) ) );
//actions
	connect( actionMenu, SIGNAL( triggered() ), this, SLOT( toggleMenu() ) );

	connect( actionSmall_16x16, SIGNAL( triggered() ), this, SLOT( updateToolbarSettings() ) );
	connect( actionMedium_22x22, SIGNAL( triggered() ), this, SLOT( updateToolbarSettings() ) );
	connect( actionBig_32x32, SIGNAL( triggered() ), this, SLOT( updateToolbarSettings() ) );
	connect( actionIcon_Only, SIGNAL( triggered() ), this, SLOT( updateToolbarSettings() ) );
	connect( actionText_Only, SIGNAL( triggered() ), this, SLOT( updateToolbarSettings() ) );
	connect( actionText_Beside_Icon, SIGNAL( triggered() ), this, SLOT( updateToolbarSettings() ) );
	connect( actionText_Under_Icon, SIGNAL( triggered() ), this, SLOT( updateToolbarSettings() ) );

	connect( actionSet_As_Default, SIGNAL( triggered() ), this, SLOT( setDefault() ) );
	connect( actionSet_As_Fallback, SIGNAL( triggered() ), this, SLOT( setFallback() ) );

	connect( actionClear, SIGNAL( triggered() ), lineEdit_menulst, SLOT( clear() ) );
	connect( actionClear, SIGNAL( triggered() ), lineEdit_menulst, SLOT( setFocus() ) );
	connect( actionReload, SIGNAL( triggered() ), this, SLOT( reload() ) );
	connect( actionEnter, SIGNAL( triggered() ), this, SLOT( updateMenulst() ) );
	connect( pushButton_browse_menulst, SIGNAL( pressed() ), this, SLOT( browse_menulst() ) );

	connect( actionView_Input, SIGNAL( triggered() ), this, SLOT( actionview_input() ) );
	connect( actionQuit, SIGNAL( triggered() ), this, SLOT( close() ) );
	connect( actionAdd_Entry, SIGNAL( triggered() ), this, SLOT( actionadd_entry() ) );
	connect( actionEdit_Entry, SIGNAL( triggered() ), this, SLOT( actionedit_entry() ) );
	connect( actionCopy_Entry, SIGNAL( triggered() ), this, SLOT( actioncopy_entry() ) );
	connect( actionRemove_Entry, SIGNAL( triggered() ), this, SLOT( actionremove_entry() ) );
	connect( actionMove_To_Top, SIGNAL( triggered() ), this, SLOT( actionmove_to_top() ) );
	connect( actionMove_Up, SIGNAL( triggered() ), this, SLOT( actionmove_up() ) );
	connect( actionMove_Down, SIGNAL( triggered() ), this, SLOT( actionmove_down() ) );
	connect( actionMove_To_Bottom, SIGNAL( triggered() ), this, SLOT( actionmove_to_bottom() ) );
	connect( actionScan_File, SIGNAL( triggered() ), this, SLOT( scan() ) );
	connect( actionEncrypt_Password, SIGNAL( triggered() ), this, SLOT( actionencrypt_password() ) );
	connect( actionCreate_GRUB_Splash_Image, SIGNAL( triggered() ), this, SLOT( actioncreate_grub_splash_image() ) );
	connect( actionBackup_File, SIGNAL( triggered() ), this, SLOT( actionbackup_file() ) );
	connect( actionRestore_File, SIGNAL( triggered() ), this, SLOT( actionrestore_file() ) );
	connect( actionDelete_Backup, SIGNAL( triggered() ), this, SLOT( actiondelete_backup() ) );
	connect( actionSettings, SIGNAL( triggered() ), this, SLOT( actionsettings() ) );
	connect( actionWhat_s_This, SIGNAL( triggered() ), this, SLOT( actionwhat_s_this() ) );
	connect( actionAbout_QGRUBEditor, SIGNAL( triggered() ), this, SLOT( actionabout_qgrubeditor() ) );
	connect( actionAbout_Qt, SIGNAL( triggered() ), qApp, SLOT( aboutQt() ) );
//gui
	connect( pushButton_top, SIGNAL( pressed() ), this, SLOT(actionmove_to_top()  ) );
	connect( pushButton_up, SIGNAL( pressed() ), this, SLOT( actionmove_up() ) );
	connect( pushButton_down, SIGNAL( pressed() ), this, SLOT ( actionmove_down() ) );
	connect( pushButton_bottom, SIGNAL( pressed() ), this, SLOT( actionmove_to_bottom() ) );
	//add page
	connect( pushButton_browse_kernel, SIGNAL( pressed() ), this, SLOT( add_browse_kernel() ) );
	connect( pushButton_browse_initrd, SIGNAL( pressed() ), this, SLOT( add_browse_initrd() ) );

	connect( pushButton_OK, SIGNAL( pressed() ), this, SLOT( add_OK() ) );
	connect( pushButton_cancel, SIGNAL( pressed() ), this, SLOT( add_cancel() ) );
	connect( pushButton_apply, SIGNAL( pressed() ), this, SLOT( add_apply() ) );
	//edit page
	connect( pushButton_browse_kernel_2, SIGNAL( pressed() ), this, SLOT( edit_browse_kernel() ) );
	connect( pushButton_browse_initrd_2, SIGNAL( pressed() ), this, SLOT( edit_browse_initrd() ) );

	connect( pushButton_OK_2, SIGNAL( pressed() ), this, SLOT( edit_OK() ) );
	connect( pushButton_cancel_2, SIGNAL( pressed() ), this, SLOT( edit_cancel() ) );
	connect( pushButton_apply_2, SIGNAL( pressed() ), this, SLOT( edit_apply() ) );
	//settings
	connect( pushButton_encrypt, SIGNAL( pressed() ), this, SLOT( encryptPassword() ) );
	connect( pushButton_create, SIGNAL( pressed() ), this, SLOT( createSplashImage() ) );
	connect( pushButton_preview, SIGNAL( pressed() ), this, SLOT( preview() ) );
	connect( pushButton_browse_splash, SIGNAL( pressed() ), this, SLOT( browse_splash() ) );
}
void QGRUBEditor::makeGRUBSettingsConnections( const bool activate )
{
	if ( activate )
	{
		//menulst
		connect( lineEdit_menulst, SIGNAL( editingFinished() ), this, SLOT( updateMenulst() ) );
		//hiddenmenu
		connect( checkBox_hiddenMenu, SIGNAL( toggled( bool ) ), this, SLOT( checkBox_hiddenMenuToggled( bool ) ) );
		//timeout
		connect( checkBox_timeout, SIGNAL( toggled( bool ) ), this, SLOT( enableTimeout( bool ) ) );
		connect( spinBox_timeout, SIGNAL( valueChanged( int ) ), this, SLOT( updateTimeout( int ) ) );
		//default
		connect( checkBox_default, SIGNAL( toggled( bool ) ), this, SLOT( enableDefault( bool ) ) );
		connect( comboBox_default, SIGNAL( currentIndexChanged( int ) ), this, SLOT( updateDefault( int ) ) );
		connect( checkBox_savedefault, SIGNAL( toggled( bool ) ), this, SLOT( enableSavedefault( bool ) ) );
		//fallback
		connect( checkBox_fallback, SIGNAL( toggled( bool ) ), this, SLOT( enableFallback( bool ) ) );
		connect( comboBox_fallback, SIGNAL( currentIndexChanged( int ) ), this, SLOT( updateFallback( int ) ) );
		//password
		connect( groupBox_password, SIGNAL( toggled( bool ) ), this, SLOT( enablePassword( bool ) ) );
		connect( lineEdit_password, SIGNAL( editingFinished() ), this, SLOT( updatePassword() ) );
		//colors
		connect( groupBox_colors, SIGNAL( toggled( bool ) ), this, SLOT( updateColors() ) );
		connect( comboBox_normal_background, SIGNAL( currentIndexChanged( int ) ), this, SLOT( updateColors() ) );
		connect( comboBox_normal_foreground, SIGNAL( currentIndexChanged( int ) ), this, SLOT( updateColors() ) );
		connect( checkBox_normal_blink, SIGNAL( toggled( bool ) ), this, SLOT( updateColors() ) );
		connect( checkBox_highlighted, SIGNAL( toggled( bool ) ), this, SLOT( enableHighlight( bool ) ) );
		connect( comboBox_highlighted_background, SIGNAL( currentIndexChanged( int ) ), this, SLOT( updateColors() ) );
		connect( comboBox_highlighted_foreground, SIGNAL( currentIndexChanged( int ) ), this, SLOT( updateColors() ) );
		connect( checkBox_highlighted_blink, SIGNAL( toggled( bool ) ), this, SLOT( updateColors() ) );
		//splash
		connect( groupBox_splashimage, SIGNAL( toggled( bool ) ), this, SLOT( enableSplash( bool ) ) );
		connect( lineEdit_splash, SIGNAL( editingFinished() ), this, SLOT( updateSplash() ) );
	}
	else
	{
		//menulst
		disconnect( lineEdit_menulst, SIGNAL( editingFinished() ), this, SLOT( updateMenulst() ) );
		//hiddenmenu
		disconnect( checkBox_hiddenMenu, SIGNAL( toggled( bool ) ), this, SLOT( checkBox_hiddenMenuToggled( bool ) ) );
		//timeout
		disconnect( checkBox_timeout, SIGNAL( toggled( bool ) ), this, SLOT( enableTimeout( bool ) ) );
		disconnect( spinBox_timeout, SIGNAL( valueChanged( int ) ), this, SLOT( updateTimeout( int ) ) );
		//default
		disconnect( checkBox_default, SIGNAL( toggled( bool ) ), this, SLOT( enableDefault( bool ) ) );
		disconnect( comboBox_default, SIGNAL( currentIndexChanged( int ) ), this, SLOT( updateDefault( int ) ) );
		disconnect( checkBox_savedefault, SIGNAL( toggled( bool ) ), this, SLOT( enableSavedefault( bool ) ) );
		//fallback
		disconnect( checkBox_fallback, SIGNAL( toggled( bool ) ), this, SLOT( enableFallback( bool ) ) );
		disconnect( comboBox_fallback, SIGNAL( currentIndexChanged( int ) ), this, SLOT( updateFallback( int ) ) );
		//password
		disconnect( groupBox_password, SIGNAL( toggled( bool ) ), this, SLOT( enablePassword( bool ) ) );
		disconnect( lineEdit_password, SIGNAL( editingFinished() ), this, SLOT( updatePassword() ) );
		//colors
		disconnect( groupBox_colors, SIGNAL( toggled( bool ) ), this, SLOT( updateColors() ) );
		disconnect( comboBox_normal_background, SIGNAL( currentIndexChanged( int ) ), this, SLOT( updateColors() ) );
		disconnect( comboBox_normal_foreground, SIGNAL( currentIndexChanged( int ) ), this, SLOT( updateColors() ) );
		disconnect( checkBox_normal_blink, SIGNAL( toggled( bool ) ), this, SLOT( updateColors() ) );
		disconnect( checkBox_highlighted, SIGNAL( toggled( bool ) ), this, SLOT( enableHighlight( bool ) ) );
		disconnect( comboBox_highlighted_background, SIGNAL( currentIndexChanged( int ) ), this, SLOT( updateColors() ) );
		disconnect( comboBox_highlighted_foreground, SIGNAL( currentIndexChanged( int ) ), this, SLOT( updateColors() ) );
		disconnect( checkBox_highlighted_blink, SIGNAL( toggled( bool ) ), this, SLOT( updateColors() ) );
		//splash
		disconnect( lineEdit_splash, SIGNAL( editingFinished() ), this, SLOT( updateSplash() ) );
	}
}
void QGRUBEditor::applySettings()
{
	config->read();
//GENERAL
	//menu
	Ui_QGRUBEditor::menuBar->setVisible( config->settings.menu );
	//completion
	switch ( config->settings.completion )
	{
		case 0:
			completer->setCompletionMode( QCompleter::InlineCompletion );
		break;
		case 1:
			completer->setCompletionMode( QCompleter::PopupCompletion );
		break;
		case 2:
			completer->setCompletionMode( QCompleter::UnfilteredPopupCompletion );
		break;
	}
	//icon size
	switch ( config->settings.size )
	{
		case 16:
			actionSmall_16x16->setChecked( true );
		break;
		case 22:
			actionMedium_22x22->setChecked( true );
		break;
		case 32:
			actionBig_32x32->setChecked( true );
		break;
		default:
			config->settings.size = 32;
			actionBig_32x32->setChecked( true );
		break;
	}
	toolBar_file->setIconSize( QSize( config->settings.size, config->settings.size ) );
	toolBar_actions->setIconSize( QSize( config->settings.size, config->settings.size ) );
	toolBar_tools->setIconSize( QSize( config->settings.size, config->settings.size ) );
	toolBar_settings->setIconSize( QSize( config->settings.size, config->settings.size ) );
	toolBar_help->setIconSize( QSize( config->settings.size, config->settings.size ) );
	toolBar_location->setIconSize( QSize( config->settings.size, config->settings.size ) );
	//toolbutton style
	switch ( config->settings.style )
	{
		case 0:
			actionIcon_Only->setChecked( true );
			toolBar_file->setToolButtonStyle( Qt::ToolButtonIconOnly );
			toolBar_actions->setToolButtonStyle( Qt::ToolButtonIconOnly );
			toolBar_tools->setToolButtonStyle( Qt::ToolButtonIconOnly );
			toolBar_settings->setToolButtonStyle( Qt::ToolButtonIconOnly );
			toolBar_help->setToolButtonStyle( Qt::ToolButtonIconOnly );
			toolBar_location->setToolButtonStyle( Qt::ToolButtonIconOnly );
		break;
		case 1:
			actionText_Only->setChecked( true );
			toolBar_file->setToolButtonStyle( Qt::ToolButtonTextOnly );
			toolBar_actions->setToolButtonStyle( Qt::ToolButtonTextOnly );
			toolBar_tools->setToolButtonStyle( Qt::ToolButtonTextOnly );
			toolBar_settings->setToolButtonStyle( Qt::ToolButtonTextOnly );
			toolBar_help->setToolButtonStyle( Qt::ToolButtonTextOnly );
			toolBar_location->setToolButtonStyle( Qt::ToolButtonTextOnly );
		break;
		case 2:
			actionText_Beside_Icon->setChecked( true );
			toolBar_file->setToolButtonStyle( Qt::ToolButtonTextBesideIcon );
			toolBar_actions->setToolButtonStyle( Qt::ToolButtonTextBesideIcon );
			toolBar_tools->setToolButtonStyle( Qt::ToolButtonTextBesideIcon );
			toolBar_settings->setToolButtonStyle( Qt::ToolButtonTextBesideIcon );
			toolBar_help->setToolButtonStyle( Qt::ToolButtonTextBesideIcon );
			toolBar_location->setToolButtonStyle( Qt::ToolButtonTextBesideIcon );
		break;
		case 3:
			actionText_Under_Icon->setChecked( true );
			toolBar_file->setToolButtonStyle( Qt::ToolButtonTextUnderIcon );
			toolBar_actions->setToolButtonStyle( Qt::ToolButtonTextUnderIcon );
			toolBar_tools->setToolButtonStyle( Qt::ToolButtonTextUnderIcon );
			toolBar_settings->setToolButtonStyle( Qt::ToolButtonTextUnderIcon );
			toolBar_help->setToolButtonStyle( Qt::ToolButtonTextUnderIcon );
			toolBar_location->setToolButtonStyle( Qt::ToolButtonTextUnderIcon );
		break;
		default:
			actionIcon_Only->setChecked( true );
			toolBar_file->setToolButtonStyle( Qt::ToolButtonIconOnly );
			toolBar_actions->setToolButtonStyle( Qt::ToolButtonIconOnly );
			toolBar_tools->setToolButtonStyle( Qt::ToolButtonIconOnly );
			toolBar_settings->setToolButtonStyle( Qt::ToolButtonIconOnly );
			toolBar_help->setToolButtonStyle( Qt::ToolButtonIconOnly );
			toolBar_location->setToolButtonStyle( Qt::ToolButtonIconOnly );
		break;
	}
//PATHS
	reload();
}

int QGRUBEditor::colorConversion( const QString color )
{
	if ( color == "black" )
		return 0;
	if ( color == "blue" )
		return 1;
	if ( color == "green" )
		return 2;
	if ( color == "cyan" )
		return 3;
	if ( color == "red" )
		return 4;
	if ( color == "magenta" )
		return 5;
	if ( color == "brown" )
		return 6;
	if ( color == "light-gray" )
		return 7;
	if ( color == "dark-gray" )
		return 8;
	if ( color == "light-blue" )
		return 9;
	if ( color == "light-green" )
		return 10;
	if ( color == "light-cyan" )
		return 11;
	if ( color == "light-red" )
		return 12;
	if ( color == "light-magenta" )
		return 13;
	if ( color == "yellow" )
		return 14;
	if ( color == "white" )
		return 15;
	return -1;
}
QString QGRUBEditor::colorConversion( const int color )
{
	if ( color == 0 )
		return QString( "black" );
	if ( color == 1 )
		return QString( "blue" );
	if ( color == 2 )
		return QString( "green" );
	if ( color == 3 )
		return QString( "cyan" );
	if ( color == 4 )
		return QString( "red" );
	if ( color == 5 )
		return QString( "magenta" );
	if ( color == 6 )
		return QString( "brown" );
	if ( color == 7 )
		return QString( "light-gray" );
	if ( color == 8 )
		return QString( "dark-gray" );
	if ( color == 9 )
		return QString( "light-blue" );
	if ( color == 10 )
		return QString( "light-green" );
	if ( color == 11 )
		return QString( "light-cyan" );
	if ( color == 12 )
		return QString( "light-red" );
	if ( color == 13 )
		return QString( "light-magenta" );
	if ( color == 14 )
		return QString( "yellow" );
	if ( color == 15 )
		return QString( "white" );
	return QString();
}
QString QGRUBEditor::findPathByPartition( QString fullPath )
{
	for ( int i = 0; i < DEVICEVECTOR.size() ; i++ )
	{
		if ( DEVICEVECTOR.at( i ).mountpoint() != "/" && fullPath.startsWith( DEVICEVECTOR.at( i ).grubPartition() ) )
		{
			return fullPath.replace( 0, DEVICEVECTOR.at( i ).grubPartition().length(), DEVICEVECTOR.at( i ).mountpoint() );			
		}
	}
	for ( int i = 0; i < DEVICEVECTOR.size() ; i++ )
	{
		if ( DEVICEVECTOR.at( i ).mountpoint() == "/" && fullPath.startsWith( DEVICEVECTOR.at( i ).grubPartition() ) )
		{
			return fullPath.replace( 0, DEVICEVECTOR.at( i ).grubPartition().length(), DEVICEVECTOR.at( i ).mountpoint() );
		}
	}
	return fullPath;
}
QString QGRUBEditor::findPartitionByPath( QString fullPath )
{
	for ( int i = 0; i < DEVICEVECTOR.size() ; i++ )
	{
		if ( DEVICEVECTOR.at( i ).mountpoint() != "/" && fullPath.startsWith( DEVICEVECTOR.at( i ).mountpoint() ) )
		{
			return fullPath.replace( 0, DEVICEVECTOR.at( i ).mountpoint().length(), DEVICEVECTOR.at( i ).grubPartition() );
		}
	}
	for ( int i = 0; i < DEVICEVECTOR.size() ; i++ )
	{
		if ( DEVICEVECTOR.at( i ).mountpoint() == "/" && fullPath.startsWith( DEVICEVECTOR.at( i ).mountpoint() ) )
		{
			return fullPath.replace( 0, DEVICEVECTOR.at( i ).mountpoint().length() - 1, DEVICEVECTOR.at( i ).grubPartition() );
		}
	}
	return fullPath;
}
//gui-related
void QGRUBEditor::showInput( const int row )
{
//DEVICES
	for ( int i = 0; i < DEVICEVECTOR.size() ; i++ )
	{
		comboBox_root->addItem( DEVICEVECTOR.at( i ).mountpoint() );
		comboBox_root_2->addItem( DEVICEVECTOR.at( i ).mountpoint() );
	}

//GRUB ITEMS
	for ( int i = 0; i < GRUBVECTOR.size(); i++ )
	{
		listWidget_os->addItem( GRUBVECTOR.at( i ).title() );
		comboBox_entry->addItem( GRUBVECTOR.at( i ).title() );
		comboBox_default->addItem( GRUBVECTOR.at( i ).title() );
		comboBox_fallback->addItem( GRUBVECTOR.at( i ).title() );

		if ( GRUBSETTINGS.default_entry() == i )
		{
			listWidget_os->item( i )->setBackground( QBrush( config->settings.default_color ) );
			listWidget_os->item( i )->setText( listWidget_os->item( i )->text() + " (Default Entry)" );
		}
		else if ( GRUBSETTINGS.default_entry() == -2 )
		{
			if ( GRUBVECTOR.at( i ).saveDefault() )
			{
				listWidget_os->item( i )->setBackground( QBrush( config->settings.default_color ) );
				listWidget_os->item( i )->setText( listWidget_os->item( i )->text() + " (Default Entry)" );
			}
		}

		if ( GRUBSETTINGS.fallback() == i )
			listWidget_os->item( i )->setText( listWidget_os->item( i )->text() + " (Fallback Entry)" );
	}
	listWidget_os->setCurrentRow( row );
	listWidget_os->setFocus();
	comboBox_entry->setCurrentIndex( row );

//GRUB SETTINGS
	//splash image
	groupBox_splashimage->setChecked( !GRUBSETTINGS.splashImage().isEmpty() );
	if ( !GRUBSETTINGS.splashImage().isEmpty() )
		lineEdit_splash->setText( GRUBSETTINGS.splashImage() );
	//password
	groupBox_password->setChecked( !GRUBSETTINGS.password().isEmpty() );
	if ( !GRUBSETTINGS.password().isEmpty() )
		lineEdit_password->setText( GRUBSETTINGS.password() );
	//color
	groupBox_colors->setChecked( !GRUBSETTINGS.color().isEmpty() );
	if ( !GRUBSETTINGS.color().isEmpty() )
	{
		comboBox_normal_background->setCurrentIndex( colorConversion( GRUBSETTINGS.color().section( QRegExp( "\\s+" ), 0, 0 ).section( "/", 1 ) ) );
		comboBox_normal_foreground->setCurrentIndex( colorConversion( GRUBSETTINGS.color().section( QRegExp( "\\s+" ), 0, 0 ).section( "/", 0, 0 ).remove( "blink-" ) ) );
		if ( !GRUBSETTINGS.color().section( QRegExp( "\\s+" ), 1 ).isEmpty() )
		{
			checkBox_highlighted->setChecked( true );
			comboBox_highlighted_background->setEnabled( true );
			comboBox_highlighted_foreground->setEnabled( true );
			checkBox_highlighted_blink->setEnabled( true );

			comboBox_highlighted_background->setCurrentIndex( colorConversion( GRUBSETTINGS.color().section( QRegExp( "\\s+" ), 1 ).section( "/", 1 ) ) );
			comboBox_highlighted_foreground->setCurrentIndex( colorConversion( GRUBSETTINGS.color().section( QRegExp( "\\s+" ), 1 ).section( "/", 0, 0 ).remove( "blink-" ) ) );
		}

		checkBox_normal_blink->setChecked( GRUBSETTINGS.color().section( QRegExp( "\\s+" ), 0, 0 ).section( "/", 0, 0 ).startsWith( "blink-" ) );
		checkBox_highlighted_blink->setChecked( GRUBSETTINGS.color().section( QRegExp( "\\s+" ), 1 ).section( "/", 0, 0 ).startsWith( "blink-" ) );
	}
	//hiddenmenu
	checkBox_hiddenMenu->setChecked( GRUBSETTINGS.hiddenMenu() );
	//timeout
	checkBox_timeout->setChecked( GRUBSETTINGS.timeout() != -1 );
	spinBox_timeout->setEnabled( GRUBSETTINGS.timeout() != -1 );
	if ( GRUBSETTINGS.timeout() != -1 )
		spinBox_timeout->setValue( GRUBSETTINGS.timeout() );
	//fallback
	checkBox_fallback->setChecked( GRUBSETTINGS.fallback() != -1 );
	comboBox_fallback->setEnabled( GRUBSETTINGS.fallback() != -1 );
	if ( GRUBSETTINGS.fallback() != -1 )
		comboBox_fallback->setCurrentIndex( GRUBSETTINGS.fallback() );
	//default
	checkBox_default->setChecked( GRUBSETTINGS.default_entry() != -1 );
	comboBox_default->setEnabled( GRUBSETTINGS.default_entry() != -1 && GRUBSETTINGS.default_entry() != -2 );
	checkBox_savedefault->setEnabled( GRUBSETTINGS.default_entry() != -1 );
	if ( GRUBSETTINGS.default_entry() != -1 )
		if ( GRUBSETTINGS.default_entry() != -2 )
		{
			checkBox_savedefault->setChecked( false );
			comboBox_default->setCurrentIndex( GRUBSETTINGS.default_entry() );
		}
		else
		{
			checkBox_savedefault->setChecked( true );
		}
}
void QGRUBEditor::addClear()
{
	lineEdit_title->clear();
	comboBox_root->setCurrentIndex( 0 );
	lineEdit_kernel->clear();
	lineEdit_initrd->clear();
	lineEdit_chainloader->clear();

	checkBox_quiet->setChecked( false );
	checkBox_savedef->setChecked( false );
	checkBox_makeact->setChecked( false );
	checkBox_lock->setChecked( false );
}
void QGRUBEditor::editClear()
{
	lineEdit_title_2->clear();
	comboBox_root_2->setCurrentIndex( 0 );
	lineEdit_kernel_2->clear();
	lineEdit_initrd_2->clear();
	lineEdit_chainloader_2->clear();

	checkBox_quiet_2->setChecked( false );
	checkBox_savedef_2->setChecked( false );
	checkBox_makeact_2->setChecked( false );
	checkBox_lock_2->setChecked( false );
}
//ACTIONS
void QGRUBEditor::toggleMenu()
{
	config->settings.menu = ( !config->settings.menu );
	config->writeValue( "general/menu", config->settings.menu );
	Ui_QGRUBEditor::menuBar->setVisible( config->settings.menu );
}
void QGRUBEditor::updateToolbarSettings()
{
//icon size
	if ( actionSmall_16x16->isChecked() )
	{
		config->settings.size = 16;
	}
	else if ( actionMedium_22x22->isChecked() )
	{
		config->settings.size = 22;
	}
	else
	{
		config->settings.size = 32;
	}
//toolbutton style
	if ( actionIcon_Only->isChecked() )
	{
		config->settings.style = 0;
	}
	else if ( actionText_Only->isChecked() )
	{
		config->settings.style = 1;
	}
	else if ( actionText_Beside_Icon->isChecked() )
	{
		config->settings.style = 2;
	}
	else
	{
		config->settings.style = 3;
	}

	config->write();
	applySettings();
}
void QGRUBEditor::setDefault()
{
	GRUBSETTINGS.setDefault( listWidget_os->currentRow() );
	file->menuOutput( config->settings.menulst, &(GRUBSETTINGS), &(GRUBVECTOR) );

	scan( listWidget_os->currentRow() );
}
void QGRUBEditor::setFallback()
{
	GRUBSETTINGS.setFallback( listWidget_os->currentRow() );
	file->menuOutput( config->settings.menulst, &(GRUBSETTINGS), &(GRUBVECTOR) );

	scan( listWidget_os->currentRow() );
}

void QGRUBEditor::reload()
{
	lineEdit_menulst->setText( config->settings.menulst );
}
void QGRUBEditor::browse_menulst()
{
	QString file = QFileDialog::getOpenFileName( this, tr( "Select a GRUB menu file" ), lineEdit_menulst->text(), tr( "GRUB menu file (*.*)" ) );
	if ( !file.isEmpty() )
	{
		config->settings.menulst = file;
		config->writeValue( "paths/menulst", config->settings.menulst );
		reload();

		scan();
	}
}

void QGRUBEditor::actionview_input()
{
	Input *input = new Input( config->settings.menulst, &(DEVICEVECTOR), this );
	input->show();
	if ( input->exec() ){}
	delete input;
}
void QGRUBEditor::actionadd_entry()
{
	stackedWidget->setCurrentIndex( 1 );
	lineEdit_title->setFocus();
}
void QGRUBEditor::actionedit_entry()
{
	if ( !listWidget_os->currentItem() )
		return;

	stackedWidget->setCurrentIndex( 2 );
	comboBox_entry->setCurrentIndex( listWidget_os->currentRow() );
	lineEdit_title_2->setFocus();
}
void QGRUBEditor::actioncopy_entry()
{
	if ( !listWidget_os->currentItem() )
		return;

	file->copyEntry( config->settings.menulst, listWidget_os->currentRow(), &(GRUBSETTINGS), &(GRUBVECTOR) );
	scan( listWidget_os->count() );
}
void QGRUBEditor::actionremove_entry()
{
	if ( !listWidget_os->currentItem() )
		return;

	QMessageBox msgBox( QMessageBox::Question, QObject::tr( "Confirm Removal" ), QString( QObject::tr( "Are you sure you want to delete this entry?" ) ), QMessageBox::Yes | QMessageBox::No, 0, Qt::Dialog );
	switch ( msgBox.exec() )
	{
		case QMessageBox::Yes:
			file->removeEntry( config->settings.menulst, listWidget_os->currentRow(), &(GRUBSETTINGS), &(GRUBVECTOR) );
			( listWidget_os->currentRow() != 0 ? scan( listWidget_os->currentRow() - 1 ) : scan() );
		break;
		case QMessageBox::No:
			return;
		break;
	}
}
void QGRUBEditor::actionmove_to_top()
{
	if ( !listWidget_os->currentItem() )
		return;

	if ( listWidget_os->currentRow() != 0 )
	{
		file->move( config->settings.menulst, listWidget_os->currentRow(), 0, &(GRUBSETTINGS), &(GRUBVECTOR) );
		scan( 0 );
	}
}
void QGRUBEditor::actionmove_up()
{
	if ( !listWidget_os->currentItem() )
		return;

	if ( listWidget_os->currentRow() != 0 )
	{
		file->move( config->settings.menulst, listWidget_os->currentRow(), listWidget_os->currentRow() - 1, &(GRUBSETTINGS), &(GRUBVECTOR) );
		scan( listWidget_os->currentRow() - 1 );
	}
}
void QGRUBEditor::actionmove_down()
{
	if ( !listWidget_os->currentItem() )
		return;

	if ( listWidget_os->currentRow() != listWidget_os->count() - 1 )
	{
		file->move( config->settings.menulst, listWidget_os->currentRow(), listWidget_os->currentRow() + 1, &(GRUBSETTINGS), &(GRUBVECTOR) );
		scan( listWidget_os->currentRow() + 1 );
	}
}
void QGRUBEditor::actionmove_to_bottom()
{
	if ( !listWidget_os->currentItem() )
		return;

	if ( listWidget_os->currentRow() != listWidget_os->count() - 1 )
	{
		file->move( config->settings.menulst, listWidget_os->currentRow(), listWidget_os->count() - 1, &(GRUBSETTINGS), &(GRUBVECTOR) );
		scan( listWidget_os->count() - 1 );
	}
}
void QGRUBEditor::scan( const int row )
{
	file->menuInput( config->settings.menulst, &(GRUBSETTINGS), &(GRUBVECTOR) );
	file->mtabInput( config->settings.mtab, &(DEVICEVECTOR) );
	file->deviceInput( config->settings.devicemap, &(DEVICEVECTOR) );

	listWidget_os->clear();
	treeWidget_property->clear();
	comboBox_entry->clear();
	comboBox_root->clear();
	comboBox_root_2->clear();
	comboBox_fallback->clear();
	comboBox_default->clear();

	makeGRUBSettingsConnections( false );
	( GRUBVECTOR.size() - 1 >= row ? showInput( row ) : showInput() );
	makeGRUBSettingsConnections( true );

	if ( config->settings.autobackup && !QFile::exists( config->settings.menulst + "_original" ) )
		file->backup_original( config->settings.menulst );
}
void QGRUBEditor::actionbackup_file()
{
	BackupRestore *backup = new BackupRestore( 0, this );
	backup->show();
	if ( backup->exec() ){}
	delete backup;
}
void QGRUBEditor::actionrestore_file()
{		
	BackupRestore *restore = new BackupRestore( 1, this );
	restore->show();
	if ( restore->exec() ){}
	delete restore;

	scan();
}
void QGRUBEditor::actiondelete_backup()
{
	QString backup = QFileDialog::getOpenFileName( this, tr( "Select a GRUB menu file to delete" ), config->settings.menulst + "_backup" , tr( "GRUB menu file (*.*)" ) );
	if ( !backup.isEmpty() )
	{
		file->deleteBackup( backup );
		listWidget_os->setFocus();
	}

	scan( listWidget_os->currentRow() );
}
void QGRUBEditor::actionencrypt_password()
{
	Encrypt *encrypt = new Encrypt( this );
	encrypt->show();
	if ( encrypt->exec() ){}
	delete encrypt;
}
void QGRUBEditor::actioncreate_grub_splash_image()
{
	SplashCreator *splashcreator = new SplashCreator( this );
	splashcreator->show();
	if ( splashcreator->exec() ){}
	delete splashcreator;
}
void QGRUBEditor::actionsettings()
{
	Settings *settings = new Settings( this );
	settings->show();
	if ( settings->exec() ){}
	delete settings;

	applySettings();
	scan();
}
void QGRUBEditor::actionwhat_s_this()
{
	QWhatsThis::enterWhatsThisMode();
}
void QGRUBEditor::actionabout_qgrubeditor()
{
	About *about = new About( this );
	about->show();
	if ( about->exec() ){}
	delete about;
}
//GUI
//Settings
//menulst
void QGRUBEditor::updateMenulst()
{
	QFileInfo menulst( lineEdit_menulst->text() );

	if ( menulst.isFile() )
	{
		config->settings.menulst = lineEdit_menulst->text();
		config->writeValue( "paths/menulst", config->settings.menulst );

		scan();
	}
	else
	{
		QMessageBox::warning( this, tr( "Warning!" ), tr( "You may only select files." ), QMessageBox::Ok );
		reload();
	}
}
//hidden menu
void QGRUBEditor::checkBox_hiddenMenuToggled( const bool state )
{
	GRUBSETTINGS.setHiddenMenu( state );
	file->menuOutput( config->settings.menulst, &(GRUBSETTINGS), &(GRUBVECTOR) );
}
//timeout
void QGRUBEditor::enableTimeout( const bool state )
{
	spinBox_timeout->setEnabled( state );
	if ( state )
		updateTimeout( spinBox_timeout->value() );
	else
		updateTimeout( -1 );
}
void QGRUBEditor::updateTimeout( const int value )
{
	GRUBSETTINGS.setTimeout( value );
	file->menuOutput( config->settings.menulst, &(GRUBSETTINGS), &(GRUBVECTOR) );
}
//default
void QGRUBEditor::enableDefault( const bool state )
{
	comboBox_default->setEnabled( state );
	checkBox_savedefault->setEnabled( state );
	if ( state )
		if ( checkBox_savedefault->isChecked() )
			updateDefault( -2 );
		else
			updateDefault( comboBox_default->currentIndex() );
	else
		updateDefault( -1 );
}
void QGRUBEditor::updateDefault( const int value )
{
	if ( comboBox_default->currentIndex() != -1 )
	{
		GRUBSETTINGS.setDefault( value );
		file->menuOutput( config->settings.menulst, &(GRUBSETTINGS), &(GRUBVECTOR) );

		scan( listWidget_os->currentRow() );
	}
}
void QGRUBEditor::enableSavedefault( const bool state )
{
	if ( state )
		updateDefault( -2 );
	else
		updateDefault( comboBox_default->currentIndex() );
	comboBox_default->setEnabled( !state );
}
//fallback
void QGRUBEditor::enableFallback( const bool state )
{
	comboBox_fallback->setEnabled( state );
	if ( state )
		updateFallback( comboBox_fallback->currentIndex() );
	else
		updateFallback( -1 );
}
void QGRUBEditor::updateFallback( const int value )
{
	if ( comboBox_fallback->currentIndex() != -1 )
	{
		GRUBSETTINGS.setFallback( value );
		file->menuOutput( config->settings.menulst, &(GRUBSETTINGS), &(GRUBVECTOR) );
	}
}
//password
void QGRUBEditor::enablePassword( const bool state )
{
	if ( !state )
	{
		lineEdit_password->setText( QString() );
		updatePassword();
	}
}
void QGRUBEditor::updatePassword()
{
	GRUBSETTINGS.setPassword( lineEdit_password->text() );
	file->menuOutput( config->settings.menulst, &(GRUBSETTINGS), &(GRUBVECTOR) );
}
void QGRUBEditor::encryptPassword()
{
	actionencrypt_password();

	QClipboard *clipboard = QApplication::clipboard();
	if ( clipboard->text( QClipboard::Selection ) != QString() )
	{
		lineEdit_password->clear();
		lineEdit_password->setText( "--md5 " + clipboard->text( QClipboard::Selection ) );
		updatePassword();
	}
}
//colors
void QGRUBEditor::enableHighlight( const bool state)
{
	comboBox_highlighted_background->setEnabled( state );
	comboBox_highlighted_foreground->setEnabled( state );
	checkBox_highlighted_blink->setEnabled( state );
	updateColors();
}
void QGRUBEditor::updateColors()
{
	QString color;
	if ( groupBox_colors->isChecked() )
	{
		//normal
		if ( checkBox_normal_blink->isChecked() )
			color = QString( "blink-" + colorConversion( comboBox_normal_foreground->currentIndex() ) + "/" + colorConversion( comboBox_normal_background->currentIndex() ) );
		else
			color = QString( colorConversion( comboBox_normal_foreground->currentIndex() ) + "/" + colorConversion( comboBox_normal_background->currentIndex() ) );
		//highlighted
		if ( checkBox_highlighted->isChecked() )
		{
			if ( checkBox_highlighted_blink->isChecked() )
				color.append( QString( " blink-" + colorConversion( comboBox_highlighted_foreground->currentIndex() ) + "/" + colorConversion( comboBox_highlighted_background->currentIndex() ) ) );
			else
				color.append( QString( " " + colorConversion( comboBox_highlighted_foreground->currentIndex() ) + "/" + colorConversion( comboBox_highlighted_background->currentIndex() ) ) );
		}
	}

	GRUBSETTINGS.setColor( color );
	file->menuOutput( config->settings.menulst, &(GRUBSETTINGS), &(GRUBVECTOR) );
}
//splash image
void QGRUBEditor::enableSplash( const bool state )
{
	if ( !state )
	{
		lineEdit_splash->setText( QString() );
		updateSplash();
	}
}
void QGRUBEditor::updateSplash()
{
	GRUBSETTINGS.setSplashImage( lineEdit_splash->text() );
	file->menuOutput( config->settings.menulst, &(GRUBSETTINGS), &(GRUBVECTOR) );
}
void QGRUBEditor::createSplashImage()
{
	actioncreate_grub_splash_image();

	QClipboard *clipboard = QApplication::clipboard();
	if ( clipboard->text( QClipboard::Selection ) != QString() )
	{
		lineEdit_splash->clear();
		lineEdit_splash->setText( findPartitionByPath( clipboard->text( QClipboard::Selection ) ) );
		updateSplash();
	}
}
void QGRUBEditor::preview()
{
	if ( !lineEdit_splash->text().isEmpty() )
	{
		Preview *preview = new Preview ( findPathByPartition( lineEdit_splash->text() ), this );
		preview->show();
		if ( preview->exec() ){}
		delete preview;
	}
	else
	{
		QMessageBox::warning( this, tr( "Warning!" ), tr( "No splash image was selected!" ), QMessageBox::Ok );
	}
}
void QGRUBEditor::browse_splash()
{
	QString splashFile = QFileDialog::getOpenFileName( this, tr( "Select the splashimage" ), findPathByPartition( lineEdit_splash->text() ), tr( "GRUB splash images (*.xpm.gz)" ) );
	if ( !splashFile.isEmpty() )
	{
		splashFile = findPartitionByPath( splashFile );
		lineEdit_splash->setText( splashFile );

		GRUBSETTINGS.setSplashImage( splashFile );
		file->menuOutput( config->settings.menulst, &(GRUBSETTINGS), &(GRUBVECTOR) );
	}
}
//Entries
//Add Page
void QGRUBEditor::add_browse_kernel()
{
	QString file = QFileDialog::getOpenFileName( this, tr( "Select the kernel" ), comboBox_root->currentText(), tr( "All files (*)" ) );
	if ( !file.isEmpty() )
	{
		for ( int i = 0 ; i < DEVICEVECTOR.size() ; i++ )
		{
			if ( file.startsWith( DEVICEVECTOR.at( i ).mountpoint() ) )
				lineEdit_kernel->setText( ( findPartitionByPath( file ).startsWith( lineEdit_root->text() ) ? findPartitionByPath( file ).remove( QRegExp( "\\(hd\\d,\\d\\)" ) ) : findPartitionByPath( file ) ) + " root=UUID=" + DEVICEVECTOR.at( i ).uuid() );
		}
	}
}
void QGRUBEditor::add_browse_initrd()
{
	QString file = QFileDialog::getOpenFileName( this, tr( "Select the initrd" ), comboBox_root->currentText(), tr( "All files (*)" ) );
	if ( !file.isEmpty() )
	{
		lineEdit_initrd->setText( ( findPartitionByPath( file ).startsWith( lineEdit_root->text() ) ? findPartitionByPath( file ).remove( QRegExp( "\\(hd\\d,\\d\\)" ) ) : findPartitionByPath( file ) )  );
	}
}

void QGRUBEditor::add_OK()
{
	if ( comboBox_os->currentIndex() == 0 && !lineEdit_title->text().isEmpty() && !lineEdit_root->text().isEmpty() && !lineEdit_kernel->text().isEmpty() || comboBox_os->currentIndex() == 1 && !lineEdit_title->text().isEmpty() && !lineEdit_root->text().isEmpty() && !lineEdit_chainloader->text().isEmpty() )
	{
		GRUBItem tmp_item;
			tmp_item.setTitle( lineEdit_title->text() );
			tmp_item.setRoot( lineEdit_root->text() );
		if ( comboBox_os->currentIndex() == 0 )
		{
			tmp_item.setKernel( lineEdit_kernel->text() );
			tmp_item.setInitrd( lineEdit_initrd->text() );

			tmp_item.setQuiet( checkBox_quiet->isChecked() );
		}
		else
			tmp_item.setChainLoader( lineEdit_chainloader->text() );

			tmp_item.setSaveDefault( checkBox_savedef->isChecked() );
			tmp_item.setMakeActive( checkBox_makeact->isChecked() );
			tmp_item.setLock( checkBox_lock->isChecked() );

		file->addEntry( config->settings.menulst, tmp_item, &(GRUBSETTINGS), &(GRUBVECTOR) );
		addClear();
		stackedWidget->setCurrentIndex( 0 );
		scan( listWidget_os->count() );
	}
	else
	{
		QMessageBox::warning( this, tr( "Warning!" ), tr( "You have to fill in all fields!" ), QMessageBox::Ok );
	}
}
void QGRUBEditor::add_cancel()
{
	addClear();
	stackedWidget->setCurrentIndex( 0 );
}
void QGRUBEditor::add_apply()
{
	if ( comboBox_os->currentIndex() == 0 && !lineEdit_title->text().isEmpty() && !lineEdit_root->text().isEmpty() && !lineEdit_kernel->text().isEmpty() || comboBox_os->currentIndex() == 1 && !lineEdit_title->text().isEmpty() && !lineEdit_root->text().isEmpty() && !lineEdit_chainloader->text().isEmpty() )
	{
		GRUBItem tmp_item;
			tmp_item.setTitle( lineEdit_title->text() );
			tmp_item.setRoot( lineEdit_root->text() );
		if ( comboBox_os->currentIndex() == 0 )
		{
			tmp_item.setKernel( lineEdit_kernel->text() );
			tmp_item.setInitrd( lineEdit_initrd->text() );

			tmp_item.setQuiet( checkBox_quiet->isChecked() );
		}
		else
			tmp_item.setChainLoader( lineEdit_chainloader->text() );

			tmp_item.setSaveDefault( checkBox_savedef->isChecked() );
			tmp_item.setMakeActive( checkBox_makeact->isChecked() );
			tmp_item.setLock( checkBox_lock->isChecked() );

		file->addEntry( config->settings.menulst, tmp_item, &(GRUBSETTINGS), &(GRUBVECTOR) );
		addClear();
		scan( listWidget_os->count() );
	}
	else
	{
		QMessageBox::warning( this, tr( "Warning!" ), tr( "You have to fill in all fields!" ), QMessageBox::Ok );
	}
}
//Edit Page
void QGRUBEditor::edit_browse_kernel()
{
	QString file = QFileDialog::getOpenFileName( this, tr( "Select the kernel" ), comboBox_root_2->currentText() + lineEdit_kernel_2->text().section( QRegExp( "\\s+" ), 0 ), tr( "All files (*)" ) );
	if ( !file.isEmpty() )
	{		
		for ( int i = 0 ; i < DEVICEVECTOR.size() ; i++ )
		{
			if ( file.startsWith( DEVICEVECTOR.at( i ).mountpoint() ) )
				lineEdit_kernel_2->setText( ( findPartitionByPath( file ).startsWith( lineEdit_root_2->text() ) ? findPartitionByPath( file ).remove( QRegExp( "\\(hd\\d,\\d\\)" ) ) : findPartitionByPath( file ) ) + " root=UUID=" + DEVICEVECTOR.at( i ).uuid() );
		}
	}
}
void QGRUBEditor::edit_browse_initrd()
{
	QString file = QFileDialog::getOpenFileName( this, tr( "Select the initrd" ), comboBox_root_2->currentText() + lineEdit_initrd_2->text(), tr( "All files (*)" ) );
	if ( !file.isEmpty() )
	{
		lineEdit_initrd_2->setText( ( findPartitionByPath( file ).startsWith( lineEdit_root_2->text() ) ? findPartitionByPath( file ).remove( QRegExp( "\\(hd\\d,\\d\\)" ) ) : findPartitionByPath( file ) )  );
	}
}

void QGRUBEditor::edit_OK()
{
	GRUBItem tmp_item;
	tmp_item.setTitle( lineEdit_title_2->text() );
	tmp_item.setRoot( lineEdit_root_2->text() );
	tmp_item.setKernel( lineEdit_kernel_2->text() );
	tmp_item.setInitrd( lineEdit_initrd_2->text() );
	tmp_item.setQuiet( checkBox_quiet_2->isChecked() );
	tmp_item.setChainLoader( lineEdit_chainloader_2->text() );
	tmp_item.setSaveDefault( checkBox_savedef_2->isChecked() );
	tmp_item.setMakeActive( checkBox_makeact_2->isChecked() );
	tmp_item.setLock( checkBox_lock_2->isChecked() );

	file->editEntry( config->settings.menulst, comboBox_entry->currentIndex(), tmp_item, &(GRUBSETTINGS), &(GRUBVECTOR) );
	stackedWidget->setCurrentIndex( 0 );
	scan( comboBox_entry->currentIndex() );
}
void QGRUBEditor::edit_cancel()
{
	stackedWidget->setCurrentIndex( 0 );
	listWidget_os->setCurrentRow( comboBox_entry->currentIndex() );
}
void QGRUBEditor::edit_apply()
{
	GRUBItem tmp_item;
	tmp_item.setTitle( lineEdit_title_2->text() );
	tmp_item.setRoot( lineEdit_root_2->text() );
	tmp_item.setKernel( lineEdit_kernel_2->text() );
	tmp_item.setInitrd( lineEdit_initrd_2->text() );
	tmp_item.setQuiet( checkBox_quiet_2->isChecked() );
	tmp_item.setChainLoader( lineEdit_chainloader_2->text() );
	tmp_item.setSaveDefault( checkBox_savedef_2->isChecked() );
	tmp_item.setMakeActive( checkBox_makeact_2->isChecked() );
	tmp_item.setLock( checkBox_lock_2->isChecked() );

	file->editEntry( config->settings.menulst, comboBox_entry->currentIndex(), tmp_item, &(GRUBSETTINGS), &(GRUBVECTOR) );
	scan( comboBox_entry->currentIndex() );
}
