#!/usr/bin/env python
# File created on 27 Apr 2012
from __future__ import division

__author__ = "Greg Caporaso"
__copyright__ = "Copyright 2011, The QIIME project"
__credits__ = ["Greg Caporaso", "Jai Ram Rideout"]
__license__ = "GPL"
__version__ = "1.7.0"
__maintainer__ = "Greg Caporaso"
__email__ = "gregcaporaso@gmail.com"
__status__ = "Release"

import signal
from os.path import isdir, split, join, abspath, exists
from os import chdir, getcwd
from shutil import copytree, rmtree
from glob import glob
from site import addsitedir
from tempfile import NamedTemporaryFile
from cogent.util.misc import remove_files
from qiime.util import qiime_system_call, get_qiime_temp_dir


### Code for timing out tests that exceed a time limit
## The test case timing code included in this file is adapted from
## recipes provided at:
##  http://code.activestate.com/recipes/534115-function-timeout/
##  http://stackoverflow.com/questions/492519/timeout-on-a-python-function-call

# to use this, call initiate_timeout(allowed_seconds_per_test) in 
# TestCase.setUp() and then disable_timeout() in TestCase.tearDown()

class TimeExceededError(Exception):
    pass

def initiate_timeout(seconds=60):
    
    def timeout(signum, frame):
        raise TimeExceededError,\
         "Test failed to run in allowed time (%d seconds)." % seconds
    
    signal.signal(signal.SIGALRM, timeout)
    # set the 'alarm' to go off in seconds seconds
    signal.alarm(seconds)

def disable_timeout():
    # turn off the alarm
    signal.alarm(0)

### End code for timing out tests that exceed a time limit

class FakeFile(object):
    " A class to convert a string into a file-like object"
    def __init__(self,d=""):
        self.s = d.split('\n')
    
    def __iter__(self):
        return iter(self.s)
    
    def write(self,s):
        self.s += s
    def close(self):
        pass
    def seek(self,seek_position):
        pass


def run_script_usage_tests(qiime_test_data_dir,
                           qiime_scripts_dir,
                           working_dir,
                           verbose=False,
                           tests=None,
                           failure_log_fp=None,
                           force_overwrite=False,
                           timeout=60):
    """ Test script_usage examples when test data is present in qiime_test_data_dir
    
        qiime_test_data_dir will typically point to the qiime_test_data
        directory that is included with QIIME at the root level
        (e.g. Qiime/qiime_test_data).

        Returns a result summary string and the number of script usage
        examples (i.e. commands) that failed.
    """
    # process input filepaths and directories
    qiime_test_data_dir = abspath(qiime_test_data_dir)
    working_dir = join(working_dir,'script_usage_tests')
    if force_overwrite and exists(working_dir):
        rmtree(working_dir)
    if failure_log_fp != None:
        failure_log_fp = abspath(failure_log_fp)

    if tests == None:
        tests = [split(d)[1] for d in sorted(glob('%s/*' % qiime_test_data_dir)) if isdir(d)]
    
    if verbose:
        print 'Tests to run:\n %s' % ' '.join(tests)
    
    addsitedir(qiime_scripts_dir)
    
    failed_tests = []
    warnings = []
    total_tests = 0
    for test in tests:
        
        # import the usage examples - this is possible because we added 
        # qiime_scripts_dir to the PYTHONPATH above
        script_fn = '%s/%s.py' % (qiime_scripts_dir,test)
        script = __import__(test)
        usage_examples = script.script_info['script_usage']
        
        if verbose:
            print 'Testing %d usage examples from: %s' % (len(usage_examples),script_fn)
        
        # init the test environment
        test_input_dir = '%s/%s' % (qiime_test_data_dir,test)
        test_working_dir = '%s/%s' % (working_dir,test)
        copytree(test_input_dir,test_working_dir)
        chdir(test_working_dir)
        
        # remove pre-exisitng output files if any
        try:
            script_usage_output_to_remove = script.script_info['script_usage_output_to_remove']
        except KeyError:
            script_usage_output_to_remove = []
        for e in script_usage_output_to_remove:
            rmtree(e.replace('$PWD',getcwd()),ignore_errors=True)
            remove_files([e.replace('$PWD',getcwd())],error_on_missing=False)
        
        if verbose:
            print ' Running tests in: %s' % getcwd()
            print ' Tests:'
        
        for usage_example in usage_examples:
            if '%prog' not in usage_example[2]:
                warnings.append('%s usage examples do not all use %%prog to represent the command name. You may not be running the version of the command that you think you are!' % test)
            cmd = usage_example[2].replace('%prog',script_fn)
            if verbose:
                print '  %s' % cmd,
            
            timed_out = False
            initiate_timeout(timeout)
            try:
                stdout, stderr, return_value = qiime_system_call(cmd)
            except TimeExceededError:
                timed_out = True
            else:
                disable_timeout()
            
            total_tests += 1
            if timed_out:
                # Add a string instead of return_value - if fail_tests ever ends
                # up being returned from this function we'll want to code this as 
                # an int for consistency in the return value type.
                failed_tests.append((cmd, "", "", "None, time exceeded"))
                if verbose: print ": Timed out"
            elif return_value != 0:
                failed_tests.append((cmd, stdout, stderr, return_value))
                if verbose: print ": Failed"
            else:
                pass
                if verbose: print ": Pass"
        
        if verbose:
            print ''
            
    if failure_log_fp:
        failure_log_f = open(failure_log_fp,'w')
        if len(failed_tests) == 0:
            failure_log_f.write('All script interface tests passed.\n')
        else:
            i = 1
            for cmd, stdout, stderr, return_value in failed_tests:
                failure_log_f.write('**Failed test %d:\n%s\n\nReturn value: %s\n\nStdout:\n%s\n\nStderr:\n%s\n\n' % (i,cmd,str(return_value), stdout, stderr))
                i += 1
        failure_log_f.close()
    
    
    if warnings:
        print 'Warnings:'
        for warning in warnings:
            print ' ' + warning
        print ''
    
    result_summary = 'Ran %d commands to test %d scripts. %d of these commands failed.' % (total_tests,len(tests),len(failed_tests))
    if len(failed_tests) > 0:
        failed_scripts = set([split(e[0].split()[0])[1] for e in failed_tests])
        result_summary += '\nFailed scripts were: %s' % " ".join(failed_scripts)
    if failure_log_fp:
        result_summary += "\nFailures are summarized in %s" % failure_log_fp
    
    rmtree(working_dir)
    
    return result_summary, len(failed_tests)

def get_test_data():
    """ return a small data set with properties convenient for testing qiime
    
    Convenient properties of this test data set include:
    
     - Samples form a meaningful pattern in unifrac PCoA space (fecal, L_palm,
       and tongue samples cluster independently).
     - Sample categories differ in alpha diversity (fecal < tongue < L_palm).
     - A small reference collection is included.
     - When clustered against this reference at 97% identity, some of the 
       reads hit the reference and some do not. Some of the input reads are <60% 
       similar to any of the reference sequences.
     - Two of the samples (f2, f3) contain identical sequences.
     - One sample (not16S.1) contains seqs that are not 16s and all drop out.
     - One sample (f4) has seqs that are all identical.
    """
    result = {}
    seqs = """>f2_1271 HWI-EAS440_0386:1:30:4487:20156#0/1 orig_bc=ACCAGACGATGC new_bc=ACCAGACGATGC bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCG
>f2_1539 HWI-EAS440_0386:1:31:12039:10494#0/1 orig_bc=ACCAGACGATGC new_bc=ACCAGACGATGC bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f1_2278 HWI-EAS440_0386:1:32:3943:19113#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f2_2349 HWI-EAS440_0386:1:33:11754:2337#0/1 orig_bc=ACCAGACGATGC new_bc=ACCAGACGATGC bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f1_2727 HWI-EAS440_0386:1:33:17206:16370#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f2_2750 HWI-EAS440_0386:1:33:4173:17707#0/1 orig_bc=ACCAGACGATGC new_bc=ACCAGACGATGC bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f1_2919 HWI-EAS440_0386:1:34:7244:5240#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f1_2990 HWI-EAS440_0386:1:34:15559:8361#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f1_3856 HWI-EAS440_0386:1:36:7847:3654#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f1_4379 HWI-EAS440_0386:1:37:18509:5405#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f1_4866 HWI-EAS440_0386:1:38:15872:7019#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGATCCGAGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCGGGTTGTTAAGTCAGCGGTAAAATCGTGCGGCTCAACCGTACCGTGCCGTTGAAACTGGCAACCTTGAGTTCGGACAAGGCAGATGGAATTCGTGGTGT
>f2_4913 HWI-EAS440_0386:1:38:5446:9251#0/1 orig_bc=ACCAGACGATGC new_bc=ACCAGACGATGC bc_diffs=0
TACGTAGGGGGCAAGCGTTGTCCGGATTTACTGGGTGTAAAGGGAGCGTAGACGGTGTGGCAAGTCTGATGTGAAAGGCATGGGCTCAACCTGTGGACTGCATTGGAAACTGTCATACTTGAGTGCCGGAGGGGTAAGCGGAATTTCTAGTG
>f2_5255 HWI-EAS440_0386:1:39:7428:2824#0/1 orig_bc=ACCAGACGATGC new_bc=ACCAGACGATGC bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f2_5313 HWI-EAS440_0386:1:39:18251:4560#0/1 orig_bc=ACCAGACGATGC new_bc=ACCAGACGATGC bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f1_6581 HWI-EAS440_0386:1:41:12697:18427#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f2_6654 HWI-EAS440_0386:1:42:18390:1542#0/1 orig_bc=ACCAGACGATGC new_bc=ACCAGACGATGC bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCG
>f1_6871 HWI-EAS440_0386:1:42:12648:9905#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f1_7004 HWI-EAS440_0386:1:42:12524:15765#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f1_8307 HWI-EAS440_0386:1:45:2418:10930#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAAGATGCGAGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCGGGTTGTTAAGTCAGCGGTAAAATCGTGCGGCTCAACCGTACCGTGCCGTTGAAACTGGCAACCTTGAGTTCGGACAAGGCAGATGGAATTCGTGGTGT
>f1_8746 HWI-EAS440_0386:1:46:3313:8982#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGATCCGAGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCGGGTTGTTAAGTCAGCGGTAAAATCGTGCGGCTCAACCGTACCGTGCCGTTGAAACTGGCAACCTTGAGTTCGGACAAGGCAGATGGAATTCGTGGTGT
>f2_8835 HWI-EAS440_0386:1:46:15230:11989#0/1 orig_bc=ACCAGACGATGC new_bc=ACCAGACGATGC bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAGGGGGGTAG
>f1_10106 HWI-EAS440_0386:1:49:11503:8253#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f2_10228 HWI-EAS440_0386:1:49:10428:12536#0/1 orig_bc=ACCAGACGATGC new_bc=ACCAGACGATGC bc_diffs=0
TACGGAGGATTCGAGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCGGGTTGTTAAGTCAGCGGTAAAATCGTGCGGCTCAACCGTACCGTGCCGTTGAAACTGGCAATCTTGAGTTCGGACAAGGCAGATGG
>f2_10741 HWI-EAS440_0386:1:50:19354:13842#0/1 orig_bc=ACCAGACGATGC new_bc=ACCAGACGATGC bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f2_11370 HWI-EAS440_0386:1:52:13612:2374#0/1 orig_bc=ACCAGACGATGC new_bc=ACCAGACGATGC bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f1_11949 HWI-EAS440_0386:1:53:11869:5956#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f2_12151 HWI-EAS440_0386:1:53:15228:14924#0/1 orig_bc=ACCAGACGATGC new_bc=ACCAGACGATGC bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAGGGGGGTAG
>f2_12169 HWI-EAS440_0386:1:53:6500:15595#0/1 orig_bc=ACCAGACGATGC new_bc=ACCAGACGATGC bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f1_12885 HWI-EAS440_0386:1:55:6617:6505#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f1_13027 HWI-EAS440_0386:1:55:2587:12315#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f2_13201 HWI-EAS440_0386:1:55:8634:19786#0/1 orig_bc=ACCAGACGATGC new_bc=ACCAGACGATGC bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f2_13503 HWI-EAS440_0386:1:56:16989:11768#0/1 orig_bc=ACCAGACGATGC new_bc=ACCAGACGATGC bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f1_13697 HWI-EAS440_0386:1:56:12011:20385#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f1_14665 HWI-EAS440_0386:1:59:14413:3423#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAGGGGGGTAGAA
>f1_17256 HWI-EAS440_0386:1:66:9612:13415#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f2_17407 HWI-EAS440_0386:1:67:5154:3682#0/1 orig_bc=ACCAGACGATGC new_bc=ACCAGACGATGC bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f1_18550 HWI-EAS440_0386:1:70:6653:12933#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f1_19304 HWI-EAS440_0386:1:72:14543:4190#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGATCCGAGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCGGGTTGTTAAGTCAGCGGTAAAATCGTGCGGCTCAACCGTACCGTGCCGTTGAAACTGGCAACCTTGAGTTCGGACAAGGCAGATGGAATTCGTGGTGT
>f2_20011 HWI-EAS440_0386:1:73:5930:19960#0/1 orig_bc=ACCAGACGATGC new_bc=ACCAGACGATGC bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCG
>f2_20101 HWI-EAS440_0386:1:74:13281:8786#0/1 orig_bc=ACCAGACGATGC new_bc=ACCAGACGATGC bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f2_21396 HWI-EAS440_0386:1:79:15005:5440#0/1 orig_bc=ACCAGACGATGC new_bc=ACCAGACGATGC bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAGGGGGGTAG
>f2_23683 HWI-EAS440_0386:1:84:18937:10979#0/1 orig_bc=ACCAGACGATGC new_bc=ACCAGACGATGC bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAGGGGGGTAG
>f2_24507 HWI-EAS440_0386:1:86:4702:6973#0/1 orig_bc=ACCAGACGATGC new_bc=ACCAGACGATGC bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f1_24744 HWI-EAS440_0386:1:86:13607:15613#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAGGGGGGTAG
>p1_27798 HWI-EAS440_0386:2:37:6298:2924#0/1 orig_bc=AACGCACGCTAG new_bc=AACGCACGCTAG bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTAGATAAGTCTGAAGTTAAAGGCTGTGGCTTAACCATAGTACGCTTTGGAAACTGTTTAACTTGAGTGCAAG
>p1_28761 HWI-EAS440_0386:2:40:8922:18632#0/1 orig_bc=AACGCACGCTAG new_bc=AACGCACGCTAG bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTAGATAAGTCTGAAGTTAAAGGCTGTGGCTTAACCATAGTACGCTTTGGAAACTGTTTAACTTGAGTGCAAG
>p1_29120 HWI-EAS440_0386:2:42:1275:11504#0/1 orig_bc=AACGCACGCTAG new_bc=AACGCACGCTAG bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTAGATAAGTCTGAAGTTAAAGGCTGTGGCTTAACCATAGTACGCTTTGGAAACTGTTTAACTTGAGTGCAAG
>p1_29150 HWI-EAS440_0386:2:42:9433:13371#0/1 orig_bc=AACGCACGCTAG new_bc=AACGCACGCTAG bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTAGATAAGTCTGAAGTTAAAGGCTGTGGCTTAACCATAGTACGCTTTGGAAACTGTTTAACTTGAGTGCAAG
>p1_29350 HWI-EAS440_0386:2:43:17387:8549#0/1 orig_bc=AACGCACGCTAG new_bc=AACGCACGCTAG bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTAGATAAGTCTGAAGTTAAAGGCTGTGGCTTAACCATAGTACGCTTTGGAAACTGTTTAACTTGAGTGCAAG
>p1_29474 HWI-EAS440_0386:2:43:7240:17422#0/1 orig_bc=AACGCACGCTAG new_bc=AACGCACGCTAG bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTAGATAAGTCTGAAGTTAAAGGCTGTGGCTTAACCATAGTACGCTTTGGAAACTGTTTAACTTGAGTGCAAG
>p2_30113 HWI-EAS440_0386:2:46:8095:9920#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTAGATAAGTCTGAAGTTAAAGGCTGTGGCTTAACCATAGTACGCTTTGGAAACTGTTTAACTTGAG
>p1_30198 HWI-EAS440_0386:2:46:7650:15609#0/1 orig_bc=AACGCACGCTAG new_bc=AACGCACGCTAG bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTAGATAAGTCTGAAGTTAAAGGCTGTGGCTTAACCATAGTACGCTTTGGAAACTGTTTAACTTGAGTGCAGAAGGGG
>p2_30279 HWI-EAS440_0386:2:47:19039:4475#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTAGATAAGTCTGAAGTTAAAGGCTGTGGCTTAACCATAGTACGCTTTGGAAACTGTTTAACTTGAGTGCAAG
>p2_30549 HWI-EAS440_0386:2:48:14520:4777#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGTAGGGTGCGAGCGTTAATCGGAATTACTGGGCGTAAAGCGGGCGCAGACGGTTACTTAAGCGGGATGTGAAATCCCCGGGCTCAACCCGGGAACTGCGTTCCGAACTGGGTG
>p1_30611 HWI-EAS440_0386:2:48:11791:9189#0/1 orig_bc=AACGCACGCTAG new_bc=AACGCACGCTAG bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTAGATAAGTCTGAAGTTAAAGGCTGTGGCTTAACCATAGTACGCTTTGGAAACTGTTTAACTTGAGTGCAAG
>p2_31582 HWI-EAS440_0386:2:52:8533:3351#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCCTTGAGTACGCACAACGTTG
>p2_31615 HWI-EAS440_0386:2:52:8097:5047#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAGG
>p1_32061 HWI-EAS440_0386:2:53:12650:19528#0/1 orig_bc=AACGCACGCTAG new_bc=AACGCACGCTAG bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTAGATAAGTCTGAAGTTAAAGGCTGTGGCTTAACCATAGTACGCTTTGGAAACTGTTTAACTTGAGTGCAAG
>p2_32401 HWI-EAS440_0386:2:55:1617:7585#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTAGATAAGTCTGAAGTTAAAGGCTGTGGCTTAACCATAGTACGCTTTGGAAACTGTTTAACTTGAGTGCAAG
>p1_32437 HWI-EAS440_0386:2:55:11012:9838#0/1 orig_bc=AACGCACGCTAG new_bc=AACGCACGCTAG bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTAGATAAGTCTGAAGTTAAAGGCTGTGGCTTAACCATAGTACGCTTTGGAAACTGTTTAACTTGAGTGCAAG
>p1_32494 HWI-EAS440_0386:2:55:10467:14158#0/1 orig_bc=AACGCACGCTAG new_bc=AACGCACGCTAG bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTAGATAAGTCTGAAGTTAAAGGCTGTGGCTTAACCATAGTACGCTTTGGAAACTGTTTAACTTGAGTGCAAG
>p2_32628 HWI-EAS440_0386:2:56:18987:4105#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGTAGGGGGCAAGCGTTATCCGGATTTACTGGGTGTAAAGGGAGCGTAGACGGTGTGGCAAGTCTGATGTGAAAGGCATGGGCTCAACCTGTGGACTGCATTGGAAACTGTCATACTTGAGTGCCGGAG
>p1_32652 HWI-EAS440_0386:2:56:3581:5687#0/1 orig_bc=AACGCACGCTAG new_bc=AACGCACGCTAG bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTAGATAAGTCTGAAGTTAAAGGCTGTGGCTTAACCATAGTACGCTTTGGAAACTGTTTAACTTGAGTGCAAG
>p2_33176 HWI-EAS440_0386:2:58:7254:5181#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCCTTGAGTAC
>p2_33268 HWI-EAS440_0386:2:58:13656:11052#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGATCCGAGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGATGGATGTTTAAGTCAGTTGTGAAAGTTTGCGGCTCAACCGTAAAATTGCAGTTGATACTGGATGTCTTGAGTGCAGTTG
>p2_34392 HWI-EAS440_0386:2:62:3660:12239#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCGAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>p1_34397 HWI-EAS440_0386:2:62:8256:12723#0/1 orig_bc=AACGCACGCTAG new_bc=AACGCACGCTAG bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTAGATAAGTCTGAAGTTAAAGGCTGTGGCTTAACCATAGTACGCTTTGGAAACTGTTTAACTTGAGTGCAAG
>p2_34773 HWI-EAS440_0386:2:64:10707:4587#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTAGATAAGTCTGAAGTTAAAGGCTGTGGCTTAACCATAGTACGCTTTGGAAACTGTTTAACTTGAGTGCAGAAGGGGAGAG
>p2_34872 HWI-EAS440_0386:2:64:5256:11455#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTAGATAAGTCTGAAGTTAAAGGCTGTGGCTTAACCATAGTACGCTTTGGAAACTGTTTAACTTGAG
>p2_35125 HWI-EAS440_0386:2:65:8479:13841#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAAGGACCAGGAGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCCTTGAGTACGCACAACGTTGGC
>p2_35617 HWI-EAS440_0386:2:67:2652:17993#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTT
>p2_35810 HWI-EAS440_0386:2:68:17310:15141#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTTGATAAGTCTGAAGTTAAAGGCTGTGGCTCAACCATAGTTCGCTTTGGAAACTGTCAAACTTGAGTGCAGAAGGGGAGAG
>p1_35931 HWI-EAS440_0386:2:69:13384:7891#0/1 orig_bc=AACGCACGCTAG new_bc=AACGCACGCTAG bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTAGATAAGTCTGAAGTTAAAGGCTGTGGCTTAACCATAGTACGCTTTGGAAACTGTTTAACTTGAGTGCAAG
>p1_36300 HWI-EAS440_0386:2:70:17775:16457#0/1 orig_bc=AACGCACGCTAG new_bc=AACGCACGCTAG bc_diffs=0
TTCGTAGGGGGCAAGCGTTATCCGGATTTACTGGGTGTAAAGGGAGCGTAGACGGTGTGGCAAGTCTGATGTGAAAGGCATGGGCTCAACCTGTGGACTGCATTGGAAACTGTCATACT
>p2_36352 HWI-EAS440_0386:2:71:4992:2947#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCCTTGAGTAC
>p2_36441 HWI-EAS440_0386:2:71:7655:9093#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGTAGGGGGCAAGCGTTATCCGGATTTACTGGGTGTAAAGGGAGCGTAGACGGAATGGCAAGTCTGAAGTGAAATACCCGGGCTCAACCTGGGAACTGCTTTGGAAACTGTTGTTCTAGAGTGTTG
>p2_37291 HWI-EAS440_0386:2:75:2654:2930#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTAGATAAGTCTGAAGTTAAAGGCTGTGGCTTAACCATAGTACGCTTTGGAAACTGTTTAACTTGAGTGCAAG
>p1_37292 HWI-EAS440_0386:2:75:15500:3047#0/1 orig_bc=AACGCACGCTAG new_bc=AACGCACGCTAG bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTAGATAAGTCTGAAGTTAAAGGCTGTGGCTTAACCATAGTACGCTTTGGAAACTGTTTAACTTGAGTGCAAG
>p2_37427 HWI-EAS440_0386:2:75:17389:13227#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTAGATAAGTCTGAAGTTAAAGGCTGTGGCTTAACCATAGTACGCTTTGGAAACTGTTTAACTTGAGTGCAAG
>p1_37793 HWI-EAS440_0386:2:77:16603:2466#0/1 orig_bc=AACGCACGCTAG new_bc=AACGCACGCTAG bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTAGATAAGTCTGAAGTTAAAGGCTGTGGCTTAACCATAGTACGCTTTGGAAACTGTTTAACTTGAGTGCAAG
>p2_37880 HWI-EAS440_0386:2:77:11070:8817#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTAGATAAGTCTGAAGTTAAAGGCTGTGGCTTAACCATAGTACGCTTTGGAAACTGTTTAACTTGAGTGCAAG
>p1_37985 HWI-EAS440_0386:2:77:8401:16367#0/1 orig_bc=AACGCACGCTAG new_bc=AACGCACGCTAG bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTAGATAAGTCTGAAGTTAAAGGCTGTGGCTTAACCATAGTACGCTTTGGAAACTGTTTAACTTGAGTGCAAG
>p1_38017 HWI-EAS440_0386:2:77:13529:19101#0/1 orig_bc=AACGCACGCTAG new_bc=AACGCACGCTAG bc_diffs=0
TACGTAGGGGGCAAGCGTTATCCGGATTTACTGGGTGTAAAGGGAGCGTAGACGGAATGGCAAGTCTGAAGTGAAATACCCGGGCTCAACCTGGGAACTGCTTTGGAAACTGTTGTTCTAGGCTGTTGGAGAGGTAAGTGGAATTCCTGGTG
>p2_38493 HWI-EAS440_0386:2:80:12905:9503#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTAGATAAGTCTGAAGTTAAAGGCTGTGGCTTAACCATAGTACGCTTTGGAAACTGTTTAACTTGAGTGCAAG
>p1_38528 HWI-EAS440_0386:2:80:13500:11791#0/1 orig_bc=AACGCACGCTAG new_bc=AACGCACGCTAG bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTAGATAAGTCTGAAGTTAAAGGCTGTGGCTTAACCATAGTACGCTTTGGAAACTGTTTAACTTGAGTGCAAG
>p1_39127 HWI-EAS440_0386:2:84:11417:3032#0/1 orig_bc=AACGCACGCTAG new_bc=AACGCACGCTAG bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTAGATAAGTCTGAAGTTAAAGGCTGTGGCTTAACCATAGTACGCTTTGGAAACTGTTTAACTTGAGTGCAAG
>p2_39175 HWI-EAS440_0386:2:84:12513:6706#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAGGGGGG
>p1_39254 HWI-EAS440_0386:2:84:5143:12808#0/1 orig_bc=AACGCACGCTAG new_bc=AACGCACGCTAG bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTAGATAAGTCTGAAGTTAAAGGCTGTGGCTTAACCATAGTACGCTTTGGAAACTGTTTAACTTGAGTGCAAG
>t2_52630 HWI-EAS440_0386:5:48:16990:6975#0/1 orig_bc=ATACTATTGCGC new_bc=ATACTATTGCGC bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTTGATAAGTCTGAAGTTAAAGGCTGTGGCTCAACCATAGTTCGCTTTGGAAACTGTCAAACTTGAGTGCAGAAGGGGAGAG
>t2_52670 HWI-EAS440_0386:5:48:11227:8602#0/1 orig_bc=ATACTATTGCGC new_bc=ATACTATTGCGC bc_diffs=0
TACGGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCCTTGAGTAC
>t2_52671 HWI-EAS440_0386:5:48:12051:8637#0/1 orig_bc=ATACTATTGCGC new_bc=ATACTATTGCGC bc_diffs=0
TACGGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCCTTGAGTACGCACAACGTTGGCGG
>t2_52739 HWI-EAS440_0386:5:48:14315:11448#0/1 orig_bc=ATACTATTGCGC new_bc=ATACTATTGCGC bc_diffs=0
TACGGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCCTTGAGTACGCACAA
>t2_52841 HWI-EAS440_0386:5:48:3257:15721#0/1 orig_bc=ATACTATTGCGC new_bc=ATACTATTGCGC bc_diffs=0
TACGGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCCTTGAGTACGCACAACGTTGGCGG
>t1_52964 HWI-EAS440_0386:5:52:9778:2855#0/1 orig_bc=AGTGAGAGAAGC new_bc=AGTGAGAGAAGC bc_diffs=0
TACGGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCC
>t2_52970 HWI-EAS440_0386:5:52:6994:3157#0/1 orig_bc=ATACTATTGCGC new_bc=ATACTATTGCGC bc_diffs=0
TACGGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCCTTGAGTAC
>t1_53038 HWI-EAS440_0386:5:52:14917:5613#0/1 orig_bc=AGTGAGAGAAGC new_bc=AGTGAGAGAAGC bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTTGATAAGTCTGAAGTTAAAGGCTGTGGCTCAACCATAGTTCGCTTTGGAAACTGTCAAACTTGAGTGCAGAAGGGG
>t1_53176 HWI-EAS440_0386:5:52:6593:11349#0/1 orig_bc=AGTGAGAGAAGC new_bc=AGTGAGAGAAGC bc_diffs=0
TACGGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCCTT
>t2_53194 HWI-EAS440_0386:5:52:3945:11982#0/1 orig_bc=ATACTATTGCGC new_bc=ATACTATTGCGC bc_diffs=0
TACGGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCCTTGAGTACGC
>t2_53250 HWI-EAS440_0386:5:52:9966:14855#0/1 orig_bc=ATACTATTGCGC new_bc=ATACTATTGCGC bc_diffs=0
TACGGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCCTTGAGTAC
>t2_53290 HWI-EAS440_0386:5:52:5130:16397#0/1 orig_bc=ATACTATTGCGC new_bc=ATACTATTGCGC bc_diffs=0
TACGGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCCTTGAGTACGC
>t1_53329 HWI-EAS440_0386:5:52:14413:18238#0/1 orig_bc=AGTGAGAGAAGC new_bc=AGTGAGAGAAGC bc_diffs=0
TACGGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCCTTGAGTACGCACAACGTTGGCG
>t2_53395 HWI-EAS440_0386:5:53:4294:2974#0/1 orig_bc=ATACTATTGCGC new_bc=ATACTATTGCGC bc_diffs=0
TACGGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCCTTGAGTAC
>t2_53531 HWI-EAS440_0386:5:53:17708:9105#0/1 orig_bc=ATACTATTGCGC new_bc=ATACTATTGCGC bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTTGATAAGTCTGAAGTTAAAGGCTGTGGCTCAACCATAGTTCGCTTTGGAAACTGTCAAACTTGAGTGCAGAAGGGGAGAG
>t2_53719 HWI-EAS440_0386:5:53:15322:17370#0/1 orig_bc=ATACTATTGCGC new_bc=ATACTATTGCGC bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTAGATAAGTCTGAAGTTAAAGGCTGTGGCTTAACCATAGTACGCTTTGGAAACTGTTTAACTTGAGTGCA
>t2_53750 HWI-EAS440_0386:5:53:12367:19097#0/1 orig_bc=ATACTATTGCGC new_bc=ATACTATTGCGC bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTTGATAAGTCTGAAGTTAAAGGCTGTGGCTCAACCATAGTTCGCTTTGGAAACTGTCAAACTTGAGTGCAGAAGGGGAGAG
>t2_53764 HWI-EAS440_0386:5:54:2344:1948#0/1 orig_bc=ATACTATTGCGC new_bc=ATACTATTGCGC bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTAGATAAGTCTGAAGTTAAAGGCTGTGGCTTAACCATAGTACGCTTTGGAAACTGTTTAA
>t1_53967 HWI-EAS440_0386:5:54:10544:11545#0/1 orig_bc=AGTGAGAGAAGC new_bc=AGTGAGAGAAGC bc_diffs=0
TACGGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCCTTGAGTACGCACAACGTTGGCGGAA
>t1_54144 HWI-EAS440_0386:5:54:12690:20621#0/1 orig_bc=AGTGAGAGAAGC new_bc=AGTGAGAGAAGC bc_diffs=0
TACGGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTT
>t2_54418 HWI-EAS440_0386:5:55:17299:14062#0/1 orig_bc=ATACTATTGCGC new_bc=ATACTATTGCGC bc_diffs=0
TACGGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCCTTGAGTAC
>t1_54475 HWI-EAS440_0386:5:55:18045:16412#0/1 orig_bc=AGTGAGAGAAGC new_bc=AGTGAGAGAAGC bc_diffs=0
TACGGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCCTTGAGTACGCACAACGTTGGCGGAA
>t1_54579 HWI-EAS440_0386:5:57:5661:1897#0/1 orig_bc=AGTGAGAGAAGC new_bc=AGTGAGAGAAGC bc_diffs=0
TACGGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCC
>t2_55465 HWI-EAS440_0386:5:59:3521:5889#0/1 orig_bc=ATACTATTGCGC new_bc=ATACTATTGCGC bc_diffs=0
TACGGAAGGCCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCCTTGAGTACGCACAACGTTGGCGGAA
>t1_55525 HWI-EAS440_0386:5:59:8117:8387#0/1 orig_bc=AGTGAGAGAAGC new_bc=AGTGAGAGAAGC bc_diffs=0
TACGGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCC
>t1_55836 HWI-EAS440_0386:5:60:17033:2188#0/1 orig_bc=AGTGAGAGAAGC new_bc=AGTGAGAGAAGC bc_diffs=0
TACGGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCCTTGAGTACGCACAACGTTGGCGGAA
>t2_56059 HWI-EAS440_0386:5:60:14779:12961#0/1 orig_bc=ATACTATTGCGC new_bc=ATACTATTGCGC bc_diffs=0
TACGGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCCTTGAGTACGCACAACGTTGGCGG
>t2_56143 HWI-EAS440_0386:5:60:10116:17219#0/1 orig_bc=ATACTATTGCGC new_bc=ATACTATTGCGC bc_diffs=0
TACGGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCCTTGAGTACGCACAACGTTGGCGG
>t1_56191 HWI-EAS440_0386:5:60:12195:19960#0/1 orig_bc=AGTGAGAGAAGC new_bc=AGTGAGAGAAGC bc_diffs=0
TACGGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCCTTG
>t1_56398 HWI-EAS440_0386:5:62:1764:10337#0/1 orig_bc=AGTGAGAGAAGC new_bc=AGTGAGAGAAGC bc_diffs=0
TACGGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCCTTG
>t1_56590 HWI-EAS440_0386:5:62:14521:17922#0/1 orig_bc=AGTGAGAGAAGC new_bc=AGTGAGAGAAGC bc_diffs=0
TACGGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCCTTGAGTACGCACAACGTTGGCG
>t2_56639 HWI-EAS440_0386:5:65:15675:2221#0/1 orig_bc=ATACTATTGCGC new_bc=ATACTATTGCGC bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTAGATAAGTCTGAAGTTAAAGGCTGTGGCTTAACCATAGTACGCTTTGGAAACTGTTTAA
>t1_56713 HWI-EAS440_0386:5:65:2032:5786#0/1 orig_bc=AGTGAGAGAAGC new_bc=AGTGAGAGAAGC bc_diffs=0
TACGGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCCTT
>t1_56790 HWI-EAS440_0386:5:65:1863:8992#0/1 orig_bc=AGTGAGAGAAGC new_bc=AGTGAGAGAAGC bc_diffs=0
TACGGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCCTT
>t2_56827 HWI-EAS440_0386:5:65:12520:10388#0/1 orig_bc=ATACTATTGCGC new_bc=ATACTATTGCGC bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTAGATAAGTCTGAAGTTAAAGGCTGTGGCTTAACCATAGTACGCTTTGGAAACTGTTTAACTTGAGTGCAGAAGGGG
>t1_56940 HWI-EAS440_0386:5:65:19417:14992#0/1 orig_bc=AGTGAGAGAAGC new_bc=AGTGAGAGAAGC bc_diffs=0
TACGGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTT
>t1_57060 HWI-EAS440_0386:5:65:14396:20229#0/1 orig_bc=AGTGAGAGAAGC new_bc=AGTGAGAGAAGC bc_diffs=0
TACGGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCCTT
>t2_57101 HWI-EAS440_0386:5:71:11781:3199#0/1 orig_bc=ATACTATTGCGC new_bc=ATACTATTGCGC bc_diffs=0
TACGGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCCTTGAGTACGCACAA
>t1_57105 HWI-EAS440_0386:5:71:4972:3447#0/1 orig_bc=AGTGAGAGAAGC new_bc=AGTGAGAGAAGC bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTTGATAAGTCTGAAGTTAAAGGCTGTGGCTCAACCATAGTTCGCTTTGGAAACTGTCAAACTTGAGTGCAGAAGGGGAGAG
>t1_57179 HWI-EAS440_0386:5:71:12024:6939#0/1 orig_bc=AGTGAGAGAAGC new_bc=AGTGAGAGAAGC bc_diffs=0
TACGGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCCTTG
>t2_57248 HWI-EAS440_0386:5:71:7274:9435#0/1 orig_bc=ATACTATTGCGC new_bc=ATACTATTGCGC bc_diffs=0
TACGGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCCTTGAGTACGCACAACGTTGGCGG
>t1_57269 HWI-EAS440_0386:5:71:9902:10255#0/1 orig_bc=AGTGAGAGAAGC new_bc=AGTGAGAGAAGC bc_diffs=0
TACGGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCCTTGAGTACGCACAACGTTGGCGGAA
>t1_57351 HWI-EAS440_0386:5:71:12017:13776#0/1 orig_bc=AGTGAGAGAAGC new_bc=AGTGAGAGAAGC bc_diffs=0
TACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTAGATAAGTCTGAAGTTAAAGGCTGTGGCTTAACCATAGTACGCTTTGGAAACTGTTTAACTTGAGTGCAGAAGGGG
>t1_57363 HWI-EAS440_0386:5:71:14597:14338#0/1 orig_bc=AGTGAGAGAAGC new_bc=AGTGAGAGAAGC bc_diffs=0
TACGGAAGGTCCAGGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCTGGAGATTAAGTGTGTTGTGAAATGTAGACGCTCAACGTCTGAATTGCAGCGCATACTGGTTTCCTT
>t1_1 random.sequence1
CTTCCGCGGAGCACCTGTATGCGAGTTATACTAATATCGTACAGCACCGATAGACAGTCTTCACTTACACCAAACTCTTACCACTGTGCCCCTGCAGTTTTTCTTGATGCTACAACTAACTCATCGACAGT
>p1_2 random.sequence2
TTGTTATAAGTTTCTGATATTTAATTTCGTAAATAGTGGTTCCAACCTACCTCCGTAAAGTATACCGAGATGAAACTTCCCACGGCTCGGGCGATGATGACGTGTGGACATCGGAGCGCCGGTCCGGATGC
>f3_130 f1_2278 HWI-EAS440_0386:1:32:3943:19113#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f3_131 f1_2727 HWI-EAS440_0386:1:33:17206:16370#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f3_132 f1_2919 HWI-EAS440_0386:1:34:7244:5240#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f3_133 f1_2990 HWI-EAS440_0386:1:34:15559:8361#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f3_134 f1_3856 HWI-EAS440_0386:1:36:7847:3654#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f3_135 f1_4379 HWI-EAS440_0386:1:37:18509:5405#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f3_136 f1_4866 HWI-EAS440_0386:1:38:15872:7019#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGATCCGAGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCGGGTTGTTAAGTCAGCGGTAAAATCGTGCGGCTCAACCGTACCGTGCCGTTGAAACTGGCAACCTTGAGTTCGGACAAGGCAGATGGAATTCGTGGTGT
>f3_137 f1_6581 HWI-EAS440_0386:1:41:12697:18427#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f3_138 f1_6871 HWI-EAS440_0386:1:42:12648:9905#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f3_139 f1_7004 HWI-EAS440_0386:1:42:12524:15765#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f3_140 f1_8307 HWI-EAS440_0386:1:45:2418:10930#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAAGATGCGAGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCGGGTTGTTAAGTCAGCGGTAAAATCGTGCGGCTCAACCGTACCGTGCCGTTGAAACTGGCAACCTTGAGTTCGGACAAGGCAGATGGAATTCGTGGTGT
>f3_141 f1_8746 HWI-EAS440_0386:1:46:3313:8982#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGATCCGAGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCGGGTTGTTAAGTCAGCGGTAAAATCGTGCGGCTCAACCGTACCGTGCCGTTGAAACTGGCAACCTTGAGTTCGGACAAGGCAGATGGAATTCGTGGTGT
>f3_142 f1_10106 HWI-EAS440_0386:1:49:11503:8253#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f3_143 f1_11949 HWI-EAS440_0386:1:53:11869:5956#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f3_144 f1_12885 HWI-EAS440_0386:1:55:6617:6505#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f3_145 f1_13027 HWI-EAS440_0386:1:55:2587:12315#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f3_146 f1_13697 HWI-EAS440_0386:1:56:12011:20385#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f3_147 f1_14665 HWI-EAS440_0386:1:59:14413:3423#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAGGGGGGTAGAA
>f3_148 f1_17256 HWI-EAS440_0386:1:66:9612:13415#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f3_149 f1_18550 HWI-EAS440_0386:1:70:6653:12933#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f3_150 f1_19304 HWI-EAS440_0386:1:72:14543:4190#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGATCCGAGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCGGGTTGTTAAGTCAGCGGTAAAATCGTGCGGCTCAACCGTACCGTGCCGTTGAAACTGGCAACCTTGAGTTCGGACAAGGCAGATGGAATTCGTGGTGT
>f3_151 f1_24744 HWI-EAS440_0386:1:86:13607:15613#0/1 orig_bc=ACACTGTTCATG new_bc=ACACTGTTCATG bc_diffs=0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAGGGGGGTAG
>not16S.1_130
GAGAAGATGCGGAGTCAATCATTACGTGCCAAGGTAGTCATTACCCATCGCGGGTGCCCGTGCATGCAAGGTGTTTGGCTGGAGGCCTACTCGGGGAATATCATAACCGCAGACAGTCAATACCTATCAGA
>not16S.1_131
GAGAAGATGCGGAGTCAATCATTACGTGCCAAGGTAGTCATTACCCATCGCGGGTGCCCGTGCATGCAAGGTGTTTGGCTGGAGGCCTACTCGGGGAATATCATAACCGCAGACAGTCAATACCTATCAGA
>not16S.1_132
GAGAAGATGCGGAGTCAATCATTACGTGCCAAGGTAGTCATTACCCATCGCGGGTGCCCGTGCATGCAAGGTGTTTGGCTGGAGGCCTACTCGGGGAATATCATAACCGCAGACAGTCAATACCTATCAGA
>not16S.1_133
GAGAAGATGCGGAGTCAATCATTACGTGCCAAGGTAGTCATTACCCATCGCGGGTGCCCGTGCATGCAAGGTGTTTGGCTGGAGGCCTACTCGGGGAATATCATAACCGCAGACAGTCAATACCTATCAGA
>not16S.1_134
GAGAAGATGCGGAGTCAATCATTACGTGCCAAGGTAGTCATTACCCATCGCGGGTGCCCGTGCATGCAAGGTGTTTGGCTGGAGGCCTACTCGGGGAATATCATAACCGCAGACAGTCAATACCTATCAGA
>not16S.1_135
GAGAAGATGCGGAGTCAATCATTACGTGCCAAGGTAGTCATTACCCATCGCGGGTGCCCGTGCATGCAAGGTGTTTGGCTGGAGGCCTACTCGGGGAATATCATAACCGCAGACAGTCAATACCTATCAGA
>not16S.1_136
GAGAAGATGCGGAGTCAATCATTACGTGCCAAGGTAGTCATTACCCATCGCGGGTGCCCGTGCATGCAAGGTGTTTGGCTGGAGGCCTACTCGGGGAATATCATAACCGCAGACAGTCAATACCTATCAGA
>not16S.1_137
GAGAAGATGCGGAGTCAATCATTACGTGCCAAGGTAGTCATTACCCATCGCGGGTGCCCGTGCATGCAAGGTGTTTGGCTGGAGGCCTACTCGGGGAATATCATAACCGCAGACAGTCAATACCTATCAGA
>not16S.1_138
GAGAAGATGCGGAGTCAATCATTACGTGCCAAGGTAGTCATTACCCATCGCGGGTGCCCGTGCATGCAAGGTGTTTGGCTGGAGGCCTACTCGGGGAATATCATAACCGCAGACAGTCAATACCTATCAGA
>not16S.1_139
GAGAAGATGCGGAGTCAATCATTACGTGCCAAGGTAGTCATTACCCATCGCGGGTGCCCGTGCATGCAAGGTGTTTGGCTGGAGGCCTACTCGGGGAATATCATAACCGCAGACAGTCAATACCTATCAGA
>not16S.1_140
GAGAAGATGCGGAGTCAATCATTACGTGCCAAGGTAGTCATTACCCATCGCGGGTGCCCGTGCATGCAAGGTGTTTGGCTGGAGGCCTACTCGGGGAATATCATAACCGCAGACAGTCAATACCTATCAGA
>not16S.1_141
GAGAAGATGCGGAGTCAATCATTACGTGCCAAGGTAGTCATTACCCATCGCGGGTGCCCGTGCATGCAAGGTGTTTGGCTGGAGGCCTACTCGGGGAATATCATAACCGCAGACAGTCAATACCTATCAGA
>not16S.1_142
GAGAAGATGCGGAGTCAATCATTACGTGCCAAGGTAGTCATTACCCATCGCGGGTGCCCGTGCATGCAAGGTGTTTGGCTGGAGGCCTACTCGGGGAATATCATAACCGCAGACAGTCAATACCTATCAGA
>not16S.1_143
GAGAAGATGCGGAGTCAATCATTACGTGCCAAGGTAGTCATTACCCATCGCGGGTGCCCGTGCATGCAAGGTGTTTGGCTGGAGGCCTACTCGGGGAATATCATAACCGCAGACAGTCAATACCTATCAGA
>not16S.1_144
GAGAAGATGCGGAGTCAATCATTACGTGCCAAGGTAGTCATTACCCATCGCGGGTGCCCGTGCATGCAAGGTGTTTGGCTGGAGGCCTACTCGGGGAATATCATAACCGCAGACAGTCAATACCTATCAGA
>not16S.1_145
GAGAAGATGCGGAGTCAATCATTACGTGCCAAGGTAGTCATTACCCATCGCGGGTGCCCGTGCATGCAAGGTGTTTGGCTGGAGGCCTACTCGGGGAATATCATAACCGCAGACAGTCAATACCTATCAGA
>not16S.1_146
GAGAAGATGCGGAGTCAATCATTACGTGCCAAGGTAGTCATTACCCATCGCGGGTGCCCGTGCATGCAAGGTGTTTGGCTGGAGGCCTACTCGGGGAATATCATAACCGCAGACAGTCAATACCTATCAGA
>not16S.1_147
GAGAAGATGCGGAGTCAATCATTACGTGCCAAGGTAGTCATTACCCATCGCGGGTGCCCGTGCATGCAAGGTGTTTGGCTGGAGGCCTACTCGGGGAATATCATAACCGCAGACAGTCAATACCTATCAGA
>not16S.1_148
GAGAAGATGCGGAGTCAATCATTACGTGCCAAGGTAGTCATTACCCATCGCGGGTGCCCGTGCATGCAAGGTGTTTGGCTGGAGGCCTACTCGGGGAATATCATAACCGCAGACAGTCAATACCTATCAGA
>not16S.1_149
GAGAAGATGCGGAGTCAATCATTACGTGCCAAGGTAGTCATTACCCATCGCGGGTGCCCGTGCATGCAAGGTGTTTGGCTGGAGGCCTACTCGGGGAATATCATAACCGCAGACAGTCAATACCTATCAGA
>not16S.1_150
GTTGGTTTATGACGACGTCCGACATTACGTGCCAAGGTAGTCATTACCCATCGCGGGTGCCCGTGCATGCAAGGTGTTTGGCTGGAGGCCTACTCGGGGAATATCATAACCGCAGACAGTCAATACCTATCAGA
>not16S.1_151
GTTGGTTTATGACGACGTCCGACATTACGTGCCAAGGTAGTCATTACCCATCGCGGGTGCCCGTGCATGCAAGGTGTTTGGCTGGAGGCCTACTCGGGGAATATCATAACCGCAGACAGTCAATACCTATCAGA
>f4_0
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f4_1
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f4_2
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f4_3
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f4_4
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f4_5
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f4_6
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f4_7
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f4_8
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f4_9
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f4_10
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f4_11
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f4_12
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f4_13
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f4_14
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f4_15
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f4_16
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f4_17
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f4_18
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f4_19
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f4_20
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f4_21
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG"""

    extra_seqs = """>f5_1300
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGCCCCC
>f5_1310
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGCCCCC
>f5_1320
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGCCCCC
>f5_1330
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGCCCCC
>f5_1340
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGCCCCC
>f5_1350
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGCCCCC
>f5_1360
TACGGAGGATCCGAGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCGGGTTGTTAAGTCAGCGGTAAAATCGTGCGGCTCAACCGTACCGTGCCGTTGAAACTGGCAACCTTGAGTTCGGACAAGGCAGATGGAATTCGTGGTGT
>f5_1370
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f5_1380
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f5_1390
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f5_1400
TACGGAAGATGCGAGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCGGGTTGTTAAGTCAGCGGTAAAATCGTGCGGCTCAACCGTACCGTGCCGTTGAAACTGGCAACCTTGAGTTCGGACAAGGCAGATGGAATTCGTGGTGT
>f5_1410
TACGGAGGATCCGAGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCGGGTTGTTAAGTCAGCGGTAAAATCGTGCGGCTCAACCGTACCGTGCCGTTGAAACTGGCAACCTTGAGTTCGGACAAGGCAGATGGAATTCGTGGTGT
>f5_1420
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f5_1430
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f5_1440
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f5_1450
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f5_1460
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f5_1470
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAGGGGGGTAGAA
>f5_1480
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f5_1490
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f5_1500
TACGGAGGATCCGAGCGTTATCCGGATTTATTGGGTTTAAAGGGAGCGTAGGCGGGTTGTTAAGTCAGCGGTAAAATCGTGCGGCTCAACCGTACCGTGCCGTTGAAACTGGCAACCTTGAGTTCGGACAAGGCAGATGGAATTCGTGGTGT
>f5_1510
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAGGGGGGTAG
>f6_00
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGCCCCC
>f6_10
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGCCCCC
>f6_20
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGCCCCC
>f6_30
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGCCCCC
>f6_40
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGCCCCC
>f6_50
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f6_60
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f6_70
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f6_80
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f6_90
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f6_100
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f6_110
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f6_120
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f6_130
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f6_140
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f6_150
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f6_160
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f6_170
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f6_180
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f6_190
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f6_200
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
>f6_210
TACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAG
"""

    refseqs = """>426848
AGAGTTTGATCCTGGCTCAGGATGAACGCTAGCGGCAGGCTTAATACATGCAAGTCGAGGGGCAGCACTGGTAGCAATACCTGGTGGCGACCGGCGGACGGGTGCGTAACACGTATGCAACCTACCCTGTACAGGGGGATAGCCCGAGGAAATTCGGATTAATACCCCATACGATAAGAATCGGCATCGATTTTTATTGAAAGCTCCGGCGGTACAGGATGGGCATGCGCCCCATTAGCTAGTTGGTGAGGTAACGGCTCACCAAGGCTACGATGGGTAGGGGGCCTGAGAGGGTGATCCCCCACACTGGAACTGAGACACGGTCCAGACTCCTACGGGAGGCAGCAGTAAGGAATATTGGTCAATGGGCGCAAGCCTGAACCAGCCATGCCGCGTGCAGGAAGACTGCCATTATGGTTGTAAACTGCTTTTATATGGGAAGAAACCTCCGGACGTGTCCGGAGCTGACGGTACCATGTGAATAAGGATCGGCTAACTCCGTGCCAGCAGCCGCGGTAATACGGAGGATCCAAGCGTTATCCGGATTTATTGGGTTTAAAGGGTGCGTAGGCGGCGTGTTAAGTCAGAGGTGAAATTCGGCAGCTCAACTGTCAAATTGCCTTTGATACTGGCACACTTGAATGCGATTGAGGTAGGCGGAATGTGACATGTAGCGGTGAAATGCTTAGACATGTGACAGAACACCGATTGCGAAGGCAGCTTACCAAGTCGTTATTGACGCTGAGGCACGAAAGCGTGGGGAGCAAACAGGATTAGATACCCTGGTAGTCCACGCCGTAAACGATGATAACTCGACGTTAGCGATACACTGTTAGCGTCCAAGCGAAAGCGTTAAGTTATCCACCTGGGAAGTACGATCGCAAGGTTGAAACTCAAAGGAATTGACGGGGGCCCGCACAAGCGGTGGAGCATGTGGTTTAATTCGATGATACGCGAGGAACCTTACCAGGGCTTAAATGGGGAACGACCTTCTGGGAAACCAGAATTTCTTTTAGACGGTCCTCAAGGTGCTGCATGGTTGTCGTCAGCTCGTGCCGTGAGGTGTTGGGTTAAGTCCCGCAACGAGCGCAACCCCTACTGTTAGTTGCCAGCGGATAATGCCGGGGACTCTAGCGGAACTGCCTGTGCAAACAGAGAGGAAGGTGGGGATGACGTCAAATCATCACGGCCCTTACGTCCTGGGCTACACACGTGCTACAATGGCCGGTACAGAGGGCAGCCACTTCGTGAGAAGGAGCGAATCCTTAAAGCCGGTCTCAGTTCGGATTGTAGTCTGCAACTCGACTACATGAAGCTGGAATCGCTAGTAATCGCGTATCAGCCATGACGCGGTGAATACGTTCCCGGGCCTTGTACACACCGCCCGTCAAGCCATGGGAATTGGGAGTACCTAAAGTCGGTAACCGCAAGGAGCCGCCTAAGGTAATACCAGTGACTGGGGCTAAGTCGTAACAAGGTAGCCGTA
>42684
AGAGTTTGATCCTGGCTCAGATTGAACGCTGGCGGCATGCTTTACACATGCAAGTCGGACGGCAGCACAGAGGAGCTTGCTTCTTGGGTGGCGAGTGGCGAACGGGTGAGTGACGCATCGGAACGTACCGAGTAATGGGGGATAACTGTCCGAAAGGACAGCTAATACCGCATACGCCCTGAGGGGGAAAGCGGGGGATCTTAGGACCTCGCGTTATTCGAGCGGCCGATGTCTGATTAGCTGGTTGGCGGGGTAAAGGCCCACCAAGGCGACGATCAGTAGCGGGTCTGAGAGGATGATCCGCCACACTGGGACTGAGACACGGCCCAGACTCCTACGGGAGGCAGCAGTGGGGAATTTTGGACAATGGGCGCAAGCCTGATCCAGCCATGCCGCGTGTCTGAAGAAGGCCTTCGGGTTGTAAAGGACTTTTGTCAGGGAAGAAAAGGAACGTGTTAATACCATGTTCTGATGACGGTACCTGAAGAATAAGCACCGGCTAACTACGTGCCAGCAGCCGCGGTAATACGTAGGGTGCGAGCGTTAATCGGAATTACTGGGCGTAAAGCGGGCGCAGACGGTTACTTAAGCGGGATGTGAAATCCCCGGGCTCAACCCGGGAACTGCGTTCCGAACTGGGTGGCTAGAGTGTGTCAGAGGGGGGTAGAATTCCACGTGTAGCAGTGAAATGCGTAGAGATGTGGAGGAATACCGATGGCGAAGGCAGCCCCCTGGGATAACACTGACGTTCATGCCCGAAAGCGTGGGTAGCAAACAGGGTTAGATACCCTGGTAGTCCACGCCCTAAACGATGTCGATTAGCTGTTGGGGCACTTGATGCCTTAGTAGCGTAGCTAACGCGTGAAATCGACCGCCTGGGGAGTACGGTCGCAAGATTAAAACTCAAAGGAATTGACGGGGACCCGCACAAGCGGTGGATGATGTGGATTAATTCGATGCAACGCGAAGAACCTTACCTGGTCTTGACATGTACGGAATCTTCCAGAGACGGAAGGGTGCCTTCGGGAGCCGTAACACAGGTGCTGCATGGCTGTCGTCAGCTCGTGTCGTGAGATGTTGGGTTAAGTCCCGCAACGAGCGCAACCCTTGTCATTAGTTGCCATCACTTGGTTGGGCACTCTAATGAGACTGCCGGTGACAAACCGGAGGAAGGTGGGGATGACGTCAAGTCCTCATGGCCCTTATGACCAGGGCTTCACACGTCATACAATGGTCGGTACAGAGGGTAGCCAAGCCGCGAGGCGGAGCCAATCCCAGAAAACCGATCGTAGTCCGGATTGCACTCTGCAACTCGAGTGCATGAAGTCGGAATCGCTAGTAATCGCAGGTCAGCATACTGCGGTGAATACGTTCCCGGGTCTTGTACACACCGCCCGTCACACCATGGGAGTGGGGGATACCAGAAGCAGGTAGGCTAACCGCAAGGAGGCCGCTTGCCACGGTATGCTTCATGACTGGGGTGAAGTCGTAACAAGGTAAC
>342684
AGAGTTTGATCCTGGCTCAGGATGAACGCTAGCGGCAGGCTTAACACATGCAAGTCGAGGGGCATCGCGGGTAGCAATACCTGGCGGCGACCGGCGGAAGGGTGCGTAACGCGTGAGCGACATACCCGTGACAGGGGGATAACAGATGGAAACGTCTCCTAATACCCCATAAGATCATATATCGCATGGTATGTGATTGAAAGGTGAGAACCGGTCACGGATTGGCTCGCGTCCCATCAGGTAGACGGCGGGGCAGCGGCCCGCCGTGCCGACGACGGGTAGGGGCTCTGAGAGGAGTGACCCCCACAATGGAACTGAGACACGGTCCATACTCCTACGGGAGGCAGCAGTGAGGAATATTGGTCAATGGGCGGAAGCCTGAACCAGCCATGCCGCGTGCGGGAGGACGGCCCTATGGGTTGTAAACCGCTTTTGAGTGAGAGCAATAAGGTTCACGTGTGGACCGATGAGAGTATCATTCGAATAAGCATCGGCTAACTCCGTGCCAGCAGCCGCGGTAATACGGAGGATGCGAGCGTTATCCGGATTCATTGGGTTTAAAGGGTGCGTAGGCGGACATGTAAGTCCGAGGTGAAAGACCGGGGCCCAACCCCGGGGTTGCCTCGGATACTGTGTGTCTGGAGTGGACGTGCCGCCGGGGGAATGAGTGGTGTAGCGGTGAAATGCATAGATGTCACTCAGAACACCGATTGCGAAGGCACCTGGCGAATGTCTTACTGACGCTGAGGCACGAAAGCGTGGGGATCGAACAGGATTAGATACCCTGGTAGTCCACGCAGTAAACGATGATGGCTGTCCGTTCGCTCCGATAGGAGTGAGTAGACAAGCGAAAGCGCTAAGCCATCCACCTGGGGAGTACGGCCGCAAGGCTGAAACTCAAAGGAATTGACGGGGGCCCGCACAAGCGGAGGAACATGTGGTTTAATTCGATGATACGCGAGGAACCTTACCCGGGCTCGAACGGCAGGTGAACGATGCAGAGATGCAAAGGCCCTTCGGGGCGTCTGTCGAGGTGCTGCATGGTTGTCGTCAGCTCGTGCCGTGAGGTGTCGGCTCAAGTGCCATAACGAGCGCAACCCTTGCCTGCAGTTGCCATCGGGTAAAGCCGGGGACTCTGCAGGGACTGCCACCGCAAGGTGAGAGGAGGGGGGGGATGACGTCAAATCAGCACGGCCCTTACGTCCGGGGCGACACACGTGTTACAATGGCGGCCACAGCGGGAAGCCACCCAGTGATGGGGCGCGGATCCCAAAAAAGCCGCCTCAGTTCGGATCGGAGTCTGCAACCCGACTCCGTGAAGCTGGATTCGCTAGTAATCGCGCATCAGCCATGGCGCGGTGAATACGTTCCCGGGCCTTGTACACACCGCCCGTCAAGCCATGGGAGTCGTGGGCGCCTGAAGGCCGTGACCGCGAGGAGCGGCCTAGGGCGAACGCGGTGACTGGGGCTAAGTCGTAACAAGGTA
>295053
AGAGTTTGATCCTGGCTCAGGACGAACGCTGGCGGCGTGCCTAACACATGCAAGTCGAACGGAGATGCTCCTTCGGGAGTATCTTAGTGGCGAACGGGTGAGTAACGCGTGAGCAACCTGACCTTCACAGGGGGATAACCGCTGGAAACAGCAGCTAATACCGCATAACGTCGCAAGACCAAAGAGGGGGACCTTCGGGCCTCTTGCCATCGGATGTGCCCAGATGGGATTAGCTTGTTGGTGGGGTAACGGCTCACCAAGGCGACGATCCCTAGCTGGTCTGAGAGGATGACCAGCCACACTGGAACTGAGACACGGTCCAGACTCCTACGGGAGGCAGCAGTGGGGAATATTGCACAATGGGCGCAAGCCTGATGCAGCCATGCCGCGTGTATGAAGAAGGCCTTCGGGTTGTAAAGTACTTTCAGCGGGGAGGAAGGGAGTAAAGTTAATACCTTTGCTCATTGACGTTACCCGCAGAAGAAGCACCGGCTAACTCCGTGCCAGCAGCCGCGGTAATACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGCAGGCGGTTTGTTAAGTCAGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGATACTGGCAAGCTTGAGTCTCGTAGAGGGGGGTAGAATTCCAGGTGTAGCGGTGAAATGCGTAGAGATCTGGAGGAATACCGGTGGCGAAGGCGGCCCCCTGGACGAAGACTGACGCTCAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCTGGTAGTCCACGCCGTAAACGATGTCGACTTGGAGGTTGTGCCCTTGAGGCGTGGCTTCCGGAGCTAACGCGTTAAGTCGACCGCCTGGGGAGTACGGCCGCAAGGTTAAAACTCAAATGAATTGACGGGGGCCCGCACAAGCGGTGGAGCATGTGGTTTAATTCGATGCAACGCGAAGAACCTTACCTGGTCTTGACATCCACAGAACTTTCCAGAGATGGATTGGTGCCTTCGGGAACTGTGAGACAGGTGCTGCATGGCTGTCGTCAGCTCGTGTTGTGAAATGTTGGGTTAAGTCCCGCAACGAGCGCAACCCTTGTCCTTTGTTGCCAGCGGTCCGGCCGGGAACTCAAAGGAGACTGCCAGTGATAAACTGGAGGAAGGTGGGGATGACGTCAAGTCATCATGGCCCTTACGACCAGGGCTACACACGTGCTACAATGGCGCATACAAAGAGAAGCGACCTCGCGAGAGCAAGCGGACCTCATAAAGTGCGTCGTAGTCCGGATTGGAGTCTGCAACTCGACTCCATGAAGTCGGAATCGCTAGTAATCGTGGATCAGAATGCCACGGTGAATACGTTCCCGGGCCTTGCACACACCGCC
>879972
GACGAACGCTGGCGGCGTGCCTAATACATGCAAGTCGAACGAGATTGACCGGTGCTTGCACTGGTCAATCTAGTGGCGAACGGGTGAGTAACACGTGGGTAACCTGCCCATCAGAGGGGGATAACATTCGGAAACGGATGCTAAAACCGCATAGGTCTTCGAACCGCATGGTTTGAAGAGGAAAAGAGGCGCAAGCTTCTGCTGATGGATGGACCCGCGGTGTATTAGCTAGTTGGTGGGGTAACGGCTCACCAAGGCGACGATACATAGCCGACCTGAGAGGGTGATCGGCCACACTGGGACTGAGACACGGCCCAGACTCCTACGGGAGGCAGCAGTAGGGAATCTTCGGCAATGGACGGAAGTCTGACCGAGCAACGCCGCGTGAGTGAAGAAGGTTTTCGGATCGTAAAGCTCTGTTGTAAGAGAAGAACGAGTGTGAGAGTGGAAAGTTCACACTGTGACGGTATCTTACCAGAAAGGGACGGCTAACTACGTGCCAGCAGCCGCGGTAATACGTAGGTCCCGAGCGTTGTCCGGATTTATTGGGCGTAAAGCGAGCGCAGGCGGTTAGATAAGTCTGAAGTTAAAGGCTGTGGCTTAACCATAGTACGCTTTGGAAACTGTTTAACTTGAGTGCAAGAGGGGAGAGTGGAATTCCATGTGTAGCGGTGAAATGCGTAGATATATGGAGGAACACCGGTGGCGAAAGCGGCTCTCTGGCTTGTAACTGACGCTGAGGCTCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCTGGTAGTCCACGCCGTAAACGATGAGTGCTAGGTGTTAGACCCTTTCCGGGGTTTAGTGCCGCAGCTAACGCATTAAGCACTCCGCCTGGGGAGTACGACCGCAGGGTTGAAACTCAAAGGAATTGACGGGGGCCCGCACAAGCGGTGGAGCATGTGGTTTAATTCGAAGCAACGCGAAGAACCTTACCAGGTCTTGACATCCCTCTGACCGCTCTAGAGATAGAGCTTTCCTTCGGGACAGAGGTGACAGGTGGTGCATGGTTGTCGTCAGCTCGTGTCGTGAGATGTTGGGTTAAGTCCCGCAACGAGCGCAACCCCTATTGTTAGTTGCCATCATTCAGTTGGGCACTCTAGCGAGACTGCCGGTAATAAACCGGAGGAAGGTGGGGATGACGTCAAATCATCATGCCCCTTATGACCTGGGCTACACACGTGCTACAATGGCTGGTACAACGAGTCGCAAGCCGGTGACGGCAAGCTAATCTCTTAAAGCCAGTCTCAGTTCGGATTGTAGGCTGCAACTCGCCTACATGAAGTCGGAATCGCTAGTAATCGCGGATCAGCACGCCGCGGTGAATACGTTCCCGGGCCT
"""
    refseqs_tax = """426848	k__Bacteria; p__Bacteroidetes; c__VC2_1_Bac22; o__; f__; g__; s__
42684	k__Bacteria; p__Proteobacteria; c__Betaproteobacteria; o__Neisseriales; f__Neisseriaceae; g__; s__
342684	k__Bacteria; p__Bacteroidetes; c__Bacteroidia; o__Bacteroidales; f__; g__; s__
295053	k__Bacteria; p__Proteobacteria; c__Gammaproteobacteria; o__Enterobacteriales; f__Enterobacteriaceae; g__; s__
879972	k__Bacteria; p__Firmicutes; c__Bacilli; o__Lactobacillales; f__Streptococcaceae; g__Streptococcus; s__
"""

    refseqs_aligned = """>426848
------------------------------------------------------------------------------------------------------------AGAGTTT-GA--T-CC-T-G-GCTC-AG-GA-TGAA-C-GC--TAG-C--G-GC-A-GG--C----T-T--AATACA-T-GC-A-AGT-CGA-G-GG----------G-CAGCA---C-----------------------------TG-G-TA----------------------------------------------------GCA-A----------------------------------------------------------------------------------TAC-CT------------------G-GT--G--GC--G--AC-C-GG-C-GG-A--C-------------GGG-TGCGT-A--AC-AC-G-T-A-TG---CAA--C-CT-A--C-C-CTG--TA-C------------------------------------------------------------------A-GG----GGG-AT-AG-CCC-------------------------G-A-G-----------------------GAA-A---TTC-GGA-TTAA-TA---CC-C--C-AT-A----------C--------------------G-A----T-A--A--G-----------------AA---TC-G-----------------------------------------------------------------------------------------------------------------------G-CA-T--------------------------------------------------------------------------------------------------------------------------------------C-G-A--T-T---------------T--T-T-A-T-T-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------GAAA--G-C-T--------------------------------------------------------------------------------------------------------------------------------------CCG------------------------------------------------------------------------------------------------------------------------------------G---C-G--------------G----T-A---C-AG-G---AT---G-G-----G-CAT-GCG--C-CCC--A------TT--A--G-CT-A----G---TTGG-T-G-AG-G-T----AAC-GG-C-T-C-ACCA--A-GG-C-T--A-CG-A------------TGG-G-T------AG-GG-G-G-CCT-G-A-G---A--GG-GT--G-AT-C-CC-CCAC-A-CTGGA--A-C-TG-A-GA-C-AC-G-G-TCCAGA-CTCC-TAC-G--G-G-A-G-GC-A-GC-A-G-TA---AG-G-A-ATA-TTGGT-C-AA-T-GG--GC-GC-A----A-G-CC-T-GA-AC-CA-GCCA-TGCC-G-CG-T---G-C-A--G--GA-A-G--A--C-T-G-CCA----TT-AT---------G-G-T-T-G-T--A---AA-C-TGC--------TT-TT-A-T--A-TGG----GA-A--G---AAACCTC---CGGA----------C--GT-G-----------------T-CCG-GAG-CT-GA-CG-GT-A-C-CA-T-GT-G---------AA-----------TAAGG-ATC-GG-C-TAA---C--T-CCGT--GCCA--G-C---A--GCCG---C-GG--TA-AT--AC---GG-AG-GAT-CCA-A-G-CG-TTAT-C-CGG-AT-TT-A--T-T--GGGT-TTA----AA-GGGT-GC--G-TA-G-G-C-G------------G--C-GT-G-T-T-AA----G-T-C-A---G-AGG-TG-A-AA-TT--CG-GCA-G--------------------------------------------------------------------CT-C-AA-------------------------------------------------------------------------CT-G-T-CA-AA-T----T-G-C-C-T-T--------T--GA-T-A-C-T-G-GCA--C-A-C---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------T-T-G-A-A-T-G-----C-GA--TT-G-A------------G-GT-A-GG-C----GG--AATG-TGA-C-AT--GT-A-GCG-GTGAAA-TG-CTT-AGAC-A-TG-T-GAC--AG-A-AC-A-CC-GA--T--T--GC-GAA-G--G-C---A----G--C-T-TACCA------A-GT-CG--------------------------------------------------------------TT-A-T-T--GA--CG-----CT-GA-GG--C-A-CGA--AA-G-C--------------G-TGGG-GAG-C-A-AACA--GG-ATTA-G-ATA-C-----CC-T-G-GTA-G-T----C-CA--C-G-CCG-T-AAA--CG-ATG-AT--AA-CT---------C-GA--C--G-T-TA-G-CG-------------------------------------------------------------------------------------------ATA-CA-C----------------------------------------------------------------------------------------------------------------------------------------------------T-GT--T-A-G-C-GT-C------CA--A----GC-GAA--AG-C-G-T--T--AA-GT--T----A-TCC-ACC-T-G-GG-AAG-TA---CGA-----T-C--G-C-A-A-GGT-T--GAA-ACTC-AAA---------GGAA-TTG-ACGGG-G-G-CCCG----C-A--C-A-A-GCG-GT-G--G--AG-CA-T--GT-GGT-TT-AATT-C-G-ATG-ATAC-G-CG-A-GG-A-A-CC-TT-A-CC-AGGGC-TT-A-AA-T-G-G------------G-GAA-C-G-------------A-CC-T-T-C--TG--G-GA-A-A-C--C-A-G--A-A-T-T--T-------C----------------------------------------TTT-T------------------------------------------AG---------A---CG-GTC---C--TC---------------------------------------------------A-A-G-G-T-GCTG-CA-TGG-TT--GTC-GTC-A-GC-TC---G-TG-CC-G--TGA-GG-TGT-T-GG-G-TT-AA-GT-CCCGC-AA--------C-GAG-CGC-A-ACC-C-C-TA--CT--G-TTAG--T-T-G-C-C---AG-C-G--G----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------ATAATG--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------C----C-G------------G----G---G-A--CT---------------C-T-A-G-C-GG-A--AC-T-G-CCT--G-T------------------------------------G-----CAA----------------------------------A-C-A-GA-G-A-GG-A--AGG-T--GGGG-A-TGAC-GTC--AAAT-C---ATC-A-C-G-G-C-C-CTT----AC-G--TC-C-T-GG-GC-TA-CAC-ACGTG-C--TA--CAATG---G-CCGG-T-A--C-AGA-GG-GC--------------------------------------------------------------------------------------------------A-G-C-C-A--C-TTCG-T--G---------------------------------------A-GA-A-G-----------G--A-G-CG---A----------A--TCC-T------T---AAAGC-CG-G-T-C-T-CAG-TTC--------GGA-T-TGTAG-TC--T-GCAA-CT-C-------------------------------------------------------------------------------------------------G-ACTAC-A-T-G-AA-G-CT-GGAAT-CG-C-TA--G-TA-AT-C-G-C----GTA-TC-A-G-CC------AT--GAC-GC-G-GT-G-AAT-ACGT-T-CCCGGGCCT-TGTA----CACACCG-CCC-GTC-----A---AG--CCA-TG-GG-A--A---TT--G-GGAGT-ACC--T-AAA------G--T-CGG----------T-AA-C-C-------------------------------------------------------------G-CA-A------------------------------------------------------------------------------------------------------GG-A--G---------CC--GC--CT-AA-G-G----T-AAT-AC-CAG------------------------TG--ACT-GGGG-CT-AAG-TCGTAACAA-GGTAG-CCGT-A------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
>42684
------------------------------------------------------------------------------------------------------------AGAGTTT-GA--T-CC-T-G-GCTC-AG-AT-TGAA-C-GC--TGG-C--G-GC-A-TG--C----T-T--TACACA-T-GC-A-AGT-CGG-A-CG----------G-CAG-CA-C----------------------------AGAG-G-AG----------------------------------------------------CTT-G----------------------------------------------------------------------------------CTT-CTTG----------------G-GT--G--GC--G--AG-T-GG-C-GA-A--C-------------GGG-TGAGT-G--AC-GC-A-T-C-GG---A-A--C-GT-A--C-C-GAG--TA-A------------------------------------------------------------------T-GG----GGG-AT-AA-CTG-------------------------T-C-C-----------------------GAA-A---GGA-CAG-CTAA-TA---CC-G--C-AT-A----------C--------------------G-------------------------------------CC-C-----------------------------------------------------------------------------------------------------------------------T-GA-G--------------------------------------------------------------------------------------------------------------------------------------G-G-G---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------GAAA--G-C-G-GG-----G--GA-T--C--------------------------------------------------------------------------------------------------------------------TTA-G----------------------------------------------------------------------------------------------------------------------G-A--CC-TC--G---C-G--------------T----T-A---T-TC-G---AG---C-G-----G-CCG-ATG--T-CTG--A------TT--A--G-CT-G----G---TTGG-C-G-GG-G-T----AAA-GG-C-C-C-ACCA--A-GG-C-G--A-CG-A------------TCA-G-T------AG-CG-G-G-TCT-G-AG----A--GG-AT--G-AT-C-CG-CCAC-A-CTGGG--A-C-TG-A-GA-C-AC-G-G-CCCAGA-CTCC-TAC-G--G-G-A-G-GC-A-GC-A-G-TG---GG-G-A-ATT-TTGGA-C-AA-T-GG--GC-GC-A----A-G-CC-T-GA-TC-CA-GCCA-TGCC-G-CG-T---G-T-C--T--GA-A-G--A--A-G-G-CC-----TT-CG---------G-G-T-T-G-T--A---AA-G-GAC--------TT-TT-G-T--C-AGG----GA-A--G---AA-AAGG---AACG-TG----T--T--AA-T---A----C-----CA-T-GTT-CTG-AT-GA-CG-GT-A-C-CT-G-AA-G---------AA-----------TAAGC-ACC-GG-C-TAA---C--T-ACGT--GCCA--G-C---A--GCCG---C-GG--TA-AT--AC---GT-AG-GGT-GCG-A-G-CG-TTAA-T-CGG-AA-TT-A--C-T--GGGC-GTA----AA-GCGG-GC--G-CA-G-A-C-G------------G--T-TA-C-T-T-AA----G-C-G-G---G-ATG-TG-A-AA-TC--CC-CGG-G--------------------------------------------------------------------CT-C-AA-------------------------------------------------------------------------CC-C-G-GG-AA-C----T-G-C-G-T-T--------C--CG-A-A-C-T-G-GGT--G-G-C---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------T-A-G-A-G-T-G-----T-GT--CA-G-A------------G-GG-G-GG-T----AG--AATT-CCA-C-GT--GT-A-GCA-GTGAAA-TG-CGT-AGAG-A-TG-T-GGA--GG-A-AT-A-CC-GA--T--G--GC-GAA-G--G-C---A----G--C-C-CCCTG------G-GA-TA--------------------------------------------------------------AC-A-C-T--GA--CG-----TT-CA-TG--C-C-CGA--AA-G-C--------------G-TGGG-TAG-C-A-AACA--GG-GTTA-G-ATA-C-----CC-T-G-GTA-G-T----C-CA--C-G-CCC-T-AAA--C-GATG-TC--GA-TT---------A-GC--T--G-T-TG-G-GG-C--A--------------------------------------------------------------------------------------CTT-GA--------------------------------------------------------------------------------------------------------------------------------------------------T-G-C-CT--T-A-G-T-AG-C------GT--A----GC-TAA--CG-C-G-T--G--AA-AT--C----G-ACC-GCC-T-G-GG-GAG-TA---CGG-----T-C--G-C-A-A-GAT-T--AAA-ACTC-AAA---------GGAA-TTG-ACGGG-G-A-CCCG----C-A--C-A-A-GCG-GT-G--G--AT-GA-T--GT-GGA-TT-AATT-C-G-ATG-CAAC-G-CG-A-AG-A-A-CC-TT-A-CC-TGGTC-TT-G-AC-A-T-G--------------TAC-G-G-------------A-AT-C-T-T--CC--A-GA-G-A-C--G-G-A--A-G-G-G--T-G-----CC-------------------------------------T--TC-G------------------------------------------GG----A----G---CC-GTA---A--CA---------------------------------------------------C-A-G-G-T-GCTG-CA-TGG-CT--GTC-GTC-A-GC-TC---G-TG-TC-G--TGA-GA-TGT-T-GG-G-TT-AA-GT-CCCGC-AA--------C-GAG-CGC-A-ACC-C-T-TG--TC--A-TTAG--T-T-G-C-C---AT-C-A--C----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------TTGG----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------T----T-G------------G----G---C-A--CT---------------C-T-A-A-T-GA-G--AC-T-G-CCG--G-T------------------------------------G-A---CAA----------------------------------A-C-C-G--G-A-GG-A--AGG-T--GGGG-A-TGAC-GTC--AAGT-C---CTC-A-T-G-G-C-C-CTT----AT-G--AC-C-A-GG-GC-TT-CAC-ACGTC-A--TA--CAATG---G-TCGG-T-A--C-AGA-GG-GT--------------------------------------------------------------------------------------------------A-G-C-C-A--A-GCCG-C--G---------------------------------------A-GG-C-G-----------G--A-G-CC---A----------A--TCC-C------A-G-AAAAC-CG-A-T-C-G-TAG-TCC--------GGA-T-TGCAC-TC--T-GCAA-CT-C-------------------------------------------------------------------------------------------------G-AGTGC-A-T-G-AA-G-TC-GGAAT-CG-C-TA--G-TA-AT-C-G-C----AGG-TC-A-G-C-------AT--ACT-GC-G-GT-G-AAT-ACGT-T-CCCGGGTCT-TGTA----CACACCG-CCC-GTC-----A---CA--CCA-TG-GG-A--G---TGG-G-GG-AT-ACC--A-GAA------G--C-AGG-TA-G-G-C-T-AA-C-C-------------------------------------------------------------G-CA-A------------------------------------------------------------------------------------------------------GG-A--GG-C--C---GC-TTG--CC--ACG-G----T-ATG-CT-TCA------------------------TG--ACT-GGGG-TG-AAG-TCGTAACAA-GGTA--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
>342684
------------------------------------------------------------------------------------------------------------AGAGTTT-GA--T-CC-T-G-GCTC-AG-GA-TGAA-C-GC--TAG-C--G-GC-A-GG--C----T-T--AACACA-T-GC-A-AGT-CGA-G-GG----------G-CAT-CG-C------------------------------GG-G-TA----------------------------------------------------GCA-A---------------------------------------------------------------------------------TA-C-CT------------------G-GC--G--GC--G--AC-C-GG-C-GG-A--A-------------GGG-TGCGT-A--AC-GC-G-T-G-AG---CGA--C-AT-A--C-C-CGT--GA-C------------------------------------------------------------------A-GG----GGG-AT-AA-CAG-------------------------A-T-G-----------------------GAA-A---CGT-CTC-CTAA-TA---CC-C--C-AT-A----------A--------------------G-A--T-C-A--T--A-----------------TA---TC-G-------------------------------------------------------------------------------------------------------------------------CA-T--------------------------------------------------------------------------------------------------------------------------------------G-G-T--A-T---------------G--T-G-A-T-T-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------GAAA--G-G----------------------------------------------------------------------------------------------------------------------------------------TGA-G----------------------------------------------------------------------------------------------------------------------------------A-ACC-G--------------G----T-C---A-CG-G---AT---T-G-----G-CTC-GCG--T-CCC--A------TC--A--G-GT-A----G---ACGG-C-G-GG-G-C----AGC-GG-C-C-C-GCCG--T-GC-C-G--A-CG-A------------CGG-G-T------AG-GG-G-C-TCT-G-A-G---A--GG-AG--T-GA-C-CC-CCAC-A-ATGGA--A-C-TG-A-GA-C-AC-G-G-TCCATA-CTCC-TAC-G--G-G-A-G-GC-A-GC-A-G-TG---AG-G-A-ATA-TTGGT-C-AA-T-GG--GC-GG-A----A-G-CC-T-GA-AC-CA-GCCA-TGCC-G-CG-T---G-C-G--G--GA-G-G--A--C-G-G-CCC----TA-TG---------G-G-T-T-G-T--A---AA-C-CGC--------TT-TT-G-A--G-TGA----GA-G--C---AATAAGG---TTCA----------C--GT-G-----------------TG-GA-CCG-AT-GA-GA-GT-A-T-CA-T-TC-G---------AA-----------TAAGC-ATC-GG-C-TAA---C--T-CCGT--GCCA--G-C---A--GCCG---C-GG--TA-AT--AC---GG-AG-GAT-GCG-A-G-CG-TTAT-C-CGG-AT-TC-A--T-T--GGGT-TTA----AA-GGGT-GC--G-TA-G-G-C-G------------G--A-CA-T-G-T-AA----G-T-C-C---G-AGG-TG-A-AA-GA--CC-GGG-G--------------------------------------------------------------------CC-C-AA-------------------------------------------------------------------------CC-C-C-GG-GG-T----T-G-C-C-T-C--------G--GA-T-A-C-T-G-TGT--G-T-C---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------T-G-G-A-G-T-G-----G-AC--GT-G-C------------C-GC-C-GG-G----GG--AATG-AGT-G-GT--GT-A-GCG-GTGAAA-TG-CAT-AGAT-G-TC-A-CTC--AG-A-AC-A-CC-GA--T--T--GC-GAA-G--G-C---A----C--C-T-GGCGA------A-TG-TC--------------------------------------------------------------TT-A-C-T--GA--CG-----CT-GA-GG--C-A-CGA--AA-G-C--------------G-TGGG-GAT-C-G-AACA--GG-ATTA-G-ATA-C-----CC-T-G-GTA-G-T----C-CA--C-G-CAG-T-AAA--CG-ATG-AT--GG-CT---------G-TC--C--G-T-TC-G-CT------------------------------------------------------------------------------------------CCGATAG-G----------------------------------------------------------------------------------------------------------------------------------------------------A-GT--G-A-G-T-AG-A------CA--A----GC-GAA--AG-C-G-C--T--AA-GC--C----A-TCC-ACC-T-G-GG-GAG-TA---CGG-----C-C--G-C-A-A-GGC-T--GAA-ACTC-AAA---------GGAA-TTG-ACGGG-G-G-CCCG----C-A--C-A-A-GCG-GA-G--G--AA-CA-T--GT-GGT-TT-AATT-C-G-ATG-ATAC-G-CG-A-GG-A-A-CC-TT-A-CC-CGGGC-TC-G-AA-C-G-G------------C-AGG-T-G-------------A-AC-G-A-T--GC--A-GA-G-A-T--G-C-A--A-A-G-G----------CC-----------------------------------CTT--CG--------------------------------------------GG---------G---CG-TCT---G--TC---------------------------------------------------G-A-G-G-T-GCTG-CA-TGG-TT--GTC-GTC-A-GC-TC---G-TG-CC-G--TGA-GG-TGT-C-GG-C-TC-AA-GT-GCCAT-AA--------C-GAG-CGC-A-ACC-C-T-TG--CC--T-GCAG--T-T-G-C-C---AT-C-G--G----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------GTAAAG--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------C----C-G------------G----G---G-A--CT---------------C-T-G-C-A-GG-G--AC-T-G-CCA--C-C------------------------------------G-----CAA----------------------------------G-G-T-GA-G-A-GG-A--GGG-G--GGGG-A-TGAC-GTC--AAAT-C---AGC-A-C-G-G-C-C-CTT----AC-G--TC-C-G-GG-GC-GA-CAC-ACGTG-T--TA--CAATG---G-CGGC-C-A--C-AGC-GG-GA--------------------------------------------------------------------------------------------------A-G-C-C-A--C-CCAG-T--G---------------------------------------A-TG-G-G-----------G--C-G-CG---G----------A--TCC-C-----AA---AAAAG-CC-G-C-C-T-CAG-TTC--------GGA-T-CGGAG-TC--T-GCAA-CC-C-------------------------------------------------------------------------------------------------G-ACTCC-G-T-G-AA-G-CT-GGATT-CG-C-TA--G-TA-AT-C-G-C----GCA-TC-A-G-CC------AT--GGC-GC-G-GT-G-AAT-ACGT-T-CCCGGGCCT-TGTA----CACACCG-CCC-GTC-----A---AG--CCA-TG-GG-A--G---TCG-T-GG-GC-GCC--T-GAA------G--G-CCG----------T-GA-C-C-------------------------------------------------------------G-CG-A------------------------------------------------------------------------------------------------------GG-A--G----------C-GGC--CT--AGG-G----C-GAA-CG-CGG------------------------TG--ACT-GGGG-CT-AAG-TCGTAACAA-GGTAA-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
>295053
------------------------------------------------------------------------------------------------------------AGAGTTT-GA--T-CC-T-G-GCT-CAG-GA-CGAA-C-GC--TGG-C--G-GC-G-TG--C----C-T--AACACA-T-GC-A-AGT-CGA-A-CG----------G---------------------------------------AG-A-TG----------------------------------------------------CTC-C----------------------------------------------------------------------------------TTC-GGGA----------------G-TA--T--CT--T--AG-T-GG-C-GA-A--C-------------GGG-TGAGT-A--AC-GC-G-T-G-AG---C-A--ACCT-G--A-C-CTT--CA-C------------------------------------------------------------------A-GG----GGG-AT-AA-CCG-------------------------C-T-G-----------------------GAA-A---CAG-CAG-CTAA-TA---CC-G--C-AT-A----------A--------------------C-------------------------------------GT-C-----------------------------------------------------------------------------------------------------------------------G-CA-A--------------------------------------------------------------------------------------------------------------------------------------G-A-C---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------CAAA--G-A-G-GG-----G--GA-C--C--------------------------------------------------------------------------------------------------------------------TTC-G----------------------------------------------------------------------------------------------------------------------G-G--CC-TC--T---T-G--------------C----C-A---T-CG-G---AT---G-T-----G-CCC-AGA--T-GGG--A------TT--A--G-CT-T----G---TTGG-T-G-GG-G-T----AAC-GG-C-T-C-ACCA--A-GG-C-G--A-CG-A------------TCC-C-T------AG-CT-G-G-TCT-G-AG----A--GG-AT--G-AC-C-AG-CCAC-A-CTGGA--A-C-TG-A-GA-C-AC-G-G-TCCAGA-CTCC-TAC-G--G-G-A-G-GC-A-GC-A-G-TG---GG-G-A-ATA-TTGCA-C-AA-T-GG--GC-GC-A----A-G-CC-T-GA-TG-CA-GCCA-TGCC-G-CG-T---G-T-A--T--GA-A-G--A--A-G-G-CC-----TT-CG---------G-G-T-T-G-T--A---AA-G-TAC--------TT-TC-A-G--C-GGG----GA-G--G---AA-GGGA---GTAA-AG----T--T--AA-T---A----C-----CT-T-TGC-TCA-TT-GA-CG-TT-A-C-CC-G-CA-G---------AA-----------GAAGC-ACC-GG-C-TAA---C--T-CCGT--GCCA--G-C---A--GCCG---C-GG--TA-AT--AC---GG-AG-GGT-GCA-A-G-CG-TTAA-T-CGG-AA-TT-A--C-T--GGGC-GTA----AA-GCGC-AC--G-CA-G-G-C-G------------G--T-TT-G-T-T-AA----G-T-C-A---G-ATG-TG-A-AA-TC--CC-CGG-G--------------------------------------------------------------------CT-C-AA-------------------------------------------------------------------------CC-T-G-GG-AA-C----T-G-C-A-T-C--------T--GA-T-A-C-T-G-GCA--A-G-C---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------T-T-G-A-G-T-C-----T-CG--TA-G-A------------G-GG-G-GG-T----AG--AATT-CCA-G-GT--GT-A-GCG-GTGAAA-TG-CGT-AGAG-A-TC-T-GGA--GG-A-AT-A-CC-GG--T--G--GC-GAA-G--G-C---G----G--C-C-CCCTG------G-AC-GA--------------------------------------------------------------AG-A-C-T--GA--CG-----CT-CA-GG--T-G-CGA--AA-G-C--------------G-TGGG-GAG-C-A-AACA--GG-ATTA-G-ATA-C-----CC-T-G-GTA-G-T----C-CA--C-G-CCG-T-AAA--C-GATG-TC--GA-CT---------T-GG--A--G-G-TT-G-TG-C--C--------------------------------------------------------------------------------------CTT-GA--------------------------------------------------------------------------------------------------------------------------------------------------G-G-C-GT--G-G-C-T-TC-C------GG--A----GC-TAA--CG-C-G-T--T--AA-GT--C----G-ACC-GCC-T-G-GG-GAG-TA---CGG-----C-C--G-C-A-A-GGT-T--AAA-ACTC-AAA---------TGAA-TTG-ACGGG-G-G-CCCG----C-A--C-A-A-GCG-GT-G--G--AG-CA-T--GT-GGT-TT-AATT-C-G-ATG-CAAC-G-CG-A-AG-A-A-CC-TT-A-CC-TGGTC-TT-G-AC-A-T-C--------------CAC-A-G-------------A-AC-T-T-T--CC--A-GA-G-A-T--G-G-A--T-T-G-G--T-G-----CC-------------------------------------T--TC-G------------------------------------------GG----A----A---CT-GTG---A--GA---------------------------------------------------C-A-G-G-T-GCTG-CA-TGG-CT--GTC-GTC-A-GC-TC---G-TG-TT-G--TGA-AA-TGT-T-GG-G-TT-AA-GT-CCCGC-AA--------C-GAG-CGC-A-ACC-C-T-TG--TC--C-TTTG--T-T-G-C-C---AG-C-G--G--T-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------CCG------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------G----C----C-G------------G----G---A-A--CT---------------C-A-A-A-G-GA-G--AC-T-G-CCA--G-T------------------------------------G-A---TAA----------------------------------A-C-T-G--G-A-GG-A--AGG-T--GGGG-A-TGAC-GTC--AAGT-C---ATC-A-T-G-G-C-C-CTT----AC-G--AC-C-A-GG-GC-TA-CAC-ACGTG-C--TA--CAATG---G-CGCA-T-A--C-AAA-GA-GA--------------------------------------------------------------------------------------------------A-G-C-G-A--C-CTCG-C--G---------------------------------------A-GA-G-C-----------A--A-G-CG---G----------A--CCT-C------A-T-AAAGT-GC-G-T-C-G-TAG-TCC--------GGA-T-TGGAG-TC--T-GCAA-CT-C-------------------------------------------------------------------------------------------------G-ACTCC-A-T-G-AA-G-TC-GGAAT-CG-C-TA--G-TA-AT-C-G-T----GGA-TC-A-G--A------AT--GCC-AC-G-GT-G-AAT-ACGT-T-CCCGGGCCT-TGCA----CACACCG-CC-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
>879972
-----------------------------------------------------------------------------------------------------------------------------------------GA-CGAA-C-GC--TGG-C--G-GC-G-TG--C----C-T--AATACA-T-GC-A-AGT-CGA-A-CGA---------G--A---T--------------TG--------------ACCG-G-TG----------------------------------------------------CTT-G----------------------------------------------------------------------------------CAC-TGGTCA----------------A------TC--T--AG-T-GG-C-GA-A--C-------------GGG-TGAGT-A--AC-AC-G-T-G-GG---TAA--C-CT-G--C-C-CAT--CA-G------------------------------------------------------------------A-GG----GGG-AT-AA-CAT-------------------------T-C-G-----------------------GAA-A---CGG-ATG-CTAA-AA---CC-G--C-AT-A----------G--------------------G-T--C-T-T--C--G-----------------A----AC-C-----------------------------------------------------------------------------------------------------------------------G-CA-T--------------------------------------------------------------------------------------------------------------------------------------G-G-T----T---------------T--G-A-A-G-A------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------GGAAA--A-G-AGGC-----------------------------------------------------------------------------------------------------------------------------------GCA-A----------------------------------------------------------------------------------------------------------------------------G-CT--T---C-T-------------GC----T-G---A-TG-G---AT---G-G-----A-CCC-GCG--G-TGT--A------TT--A--G-CT-A----G---TTGG-T-G-GG-G-T----AAC-GG-C-T-C-ACCA--A-GG-C-G--A-CG-A------------TAC-A-T------AG-CC-G-A-CCT-G-AG----A--GG-GT--G-AT-C-GG-CCAC-A-CTGGG--A-C-TG-A-GA-C-AC-G-G-CCCAGA-CTCC-TAC-G--G-G-A-G-GC-A-GC-A-G-TA---GG-G-A-ATC-TTCGG-C-AA-T-GG--AC-GG-A----A-G-TC-T-GA-CC-GA-GCAA-CGCC-G-CG-T---G-A-G--T--GA-A-G--A--A-G-G-TT-----TT-CG---------G-A-T-C-G-T--A---AA-G-CTC--------TG-TT-G-T--A-AGA----GA-A--G---AACGAGT---GTGA-GA----G--T--GG-A---A----A-----GT-T-CAC-ACT-GT-GA-CG-GT-A-T-CT-T-AC-C---------AG-----------AAAGG-GAC-GG-C-TAA---C--T-ACGT--GCCA--G-C---A--GCCG---C-GG--TA-AT--AC---GT-AG-GTC-CCG-A-G-CG-TTGT-C-CGG-AT-TT-A--T-T--GGGC-GTA----AA-GCGA-GC--G-CA-G-G-C-G------------G--T-TA-G-A-T-AA----G-T-C-T---G-AAG-TT-A-AA-GG--CT-GTG-G--------------------------------------------------------------------CT-T-AA-------------------------------------------------------------------------CC-A-T-AG-TA-C------G-C-T-T-T--------G--GA-A-A-C-T-G-TTT--A-A-C---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------T-T-G-A-G-T-G-----C-AA--GA-G-G------------G-GA-G-AG-T----GG--AATT-CCA-T-GT--GT-A-GCG-GTGAAA-TG-CGT-AGAT-A-TA-T-GGA--GG-A-AC-A-CC-GG--T--G--GC-GAA-A--G-C---G----G--C-T-CTCTG------G-CT-TG--------------------------------------------------------------TA-A-C-T--GA--CG-----CT-GA-GG--C-T-CGA--AA-G-C--------------G-TGGG-GAG-C-A-AACA--GG-ATTA-G-ATA-C-----CC-T-G-GTA-G-T----C-CA--C-G-CCG-T-AAA--C-GATG-AG--TG-CT---------A-GG--T--G-T-TA-G-AC-C--C--T-----------------------------------------------------------------------------------TTC-CG--------------------------------------------------------------------------------------------------------------------------------------------------G-G-G-TT--T-A-G-T-GC-C------GC--A----GC-TAA--CG-C-A-T--T--AA-GC--A----C-TCC-GCC-T-G-GG-GAG-TA---CGA-----C-C--G-C-A-G-GGT-T--GAA-ACTC-AAA---------GGAA-TTG-ACGGG-G-G-CCCG----C-A--C-A-A-GCG-GT-G--G--AG-CA-T--GT-GGT-TT-AATT-C-G-AAG-CAAC-G-CG-A-AG-A-A-CC-TT-A-CC-AGGTC-TT-G-AC-A-TCC--------------CTC-T--------------GA-CC-G-C-T--CT--A-GA-G-A-T--A-G-A--G-C-T-T--T-------CC-------------------------------------T--TC-G------------------------------------------GG---------A---CA-GAGG--T--GA---------------------------------------------------C-A-G-G-T-GGTG-CA-TGG-TT--GTC-GTC-A-GC-TC---G-TG-TC-G--TGA-GA-TGT-T-GG-G-TT-AA-GT-CCCGC-AA--------C-GAG-CGC-A-ACC-C-C-TA--TT--G-TTAG--T-T-G-C-C---AT-C-A-----T-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------TCAG------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------T--------T-G------------G----G---C-A--CT---------------C-T-A-G-C-GA-G--AC-T-G-CCG--G-T------------------------------------A-A---TAA----------------------------------A-C-C-G--G-A-GG-A--AGG-T--GGGG-A-TGAC-GTC--AAAT-C---ATC-A-T-G-C-C-C-CTT----AT-G--AC-C-T-GG-GC-TA-CAC-ACGTG-C--TA--CAATG---G-CTGG-T-A--C-AAC-GA-GT--------------------------------------------------------------------------------------------------C-G-C-A-A--G-CCGG-T--G---------------------------------------A-CG-G-C-----------A--A-G-CT---A----------A--TCT-C------T-T-AAAGC-CA-G-T-C-T-CAG-TTC--------GGA-T-TGTAG-GC--T-GCAA-CT-C-------------------------------------------------------------------------------------------------G-CCTAC-A-T-G-AA-G-TC-GGAAT-CG-C-TA--G-TA-AT-C-G-C----GGA-TC-A-G-C-------AC--GCC-GC-G-GT-G-AAT-ACGT-T-CCCGGGCCT--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
"""

    refseqs_aligned_lanemask = """00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000110111101011001110100101101011001000010100111111010110101110111010111000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000011010110101101001000000000000011101111101001101101010101100011100101101001010111001101000000000000000000000000000000000000000000000000000000000000000000101100001110110110111000000000000000000000000010101000000000000000000000001110100011101110111101100011010010110100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000100001010001011010001100010100000101110111001011100100000011001001011010000100011110101011010100001110110101010111100101101010010110100000000000011101010000001101101010111010110000100110110010110101101111010111110010101101011010110101011111101111011101001010101011010110101011000110101011101111101011010110011011010000101011010110110110111101111010110100010101001001101010010010101011000001101100000000010101010101001000110101110000000011011010100101110000110100100000000000000000000000000000000000000000000000000000000000000000001101101010110101101000000000110000000000011111011101101011100010110111100111100101000100111100010110011011001100011011011101110101011011110101110110110100101001111011100001101111011001011010101010000000000001001011010101011000010101010001011101101011011001101110100000000000000000000000000000000000000000000000000000000000000000000110101100000000000000000000000000000000000000000000000000000000000000000000000001101010110110100001010101010100000000100110101010101011100101010000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001010101010101000001011001101010000000000001011010110100001100111101110101100110101110111111011011101111010110101110011010110101101100100100110111010010100010000100101011111000000101101100000000000000000000000000000000000000000000000000000000000000110101010011001100000110110110010101110011010100000000000000101111011101010111100110111101011101000001101010111010100001011001010111010111001011110110011011000000000101100100101011010100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000101010101101000000110010000110111001101010100100110110010000101110111010101101110110001110000010100101010101110100111011110111000000000111101110111110101011110000101001010101110110100100110110100110111011011110101011101111010110101101010110110101101111101101011010101000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001010101010111101101110110011101110101101100010110110100111011011101011010110110110111110110000000010111011101011101010110011001011110010101010000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000101001100000000000000010101010101101001101010111001010000000000000000000000000000000000001010001110000000000000000000000000000000000101010100101011010011101001111010111101110011110100011101010101010101110000110100110101011011011011101111101001100111110001011110101001011101101100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000101010101001011110100100000000000000000000000000000000000000010110101000000000001001010110001000000000010011101000000101011111011010101010111011100000000111010111110110010111101101000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010111110101010110101101111101101011001011011010101000011101101010100000001100111011010110101110111101011111111101111000011111110111011100000100011001110110110100100011101011011011100101110000001001011100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000111001100111010000101110110111000000000000000000000000110011101111011011101111111110111110111100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
"""

    md = """#SampleID	BarcodeSequence	LinkerPrimerSequence	SampleType	year	month	day	subject	days_since_epoch	Description
f1	ACACTGTTCATG	GTGCCAGCMGCCGCGGTAA	feces	2008	10	22	1	14174	fecal1
f2	ACCAGACGATGC	GTGCCAGCMGCCGCGGTAA	feces	2008	10	23	1	14175	fecal2
f3	ACCAGACGATGC	GTGCCAGCMGCCGCGGTAA	feces	2008	10	23	1	14175	identical sequences to fecal2
f4	ACCAGACGATGC	GTGCCAGCMGCCGCGGTAA	feces	2008	10	23	1	14175	all sequences identical, map to GG 295053 at 97 percent id
f5	ACCAGACGATGC	GTGCCAGCMGCCGCGGTAA	feces	2008	10	23	1	14175	derived from f3 with some changes to sequences to add one new otu
f6	ACCAGACGATGC	GTGCCAGCMGCCGCGGTAA	feces	2008	10	23	1	14175	derived from f4 with some changes to sequences to add one new otu
p1	AACGCACGCTAG	GTGCCAGCMGCCGCGGTAA	L_palm	2008	10	21	1	14173	palm1, contains one randomly generated sequence
p2	ACACTGTTCATG	GTGCCAGCMGCCGCGGTAA	L_palm	2008	10	22	1	14174	palm2
t1	AGTGAGAGAAGC	GTGCCAGCMGCCGCGGTAA	Tongue	2008	10	21	1	14173	tongue1, contains one randomly generated sequence
t2	ATACTATTGCGC	GTGCCAGCMGCCGCGGTAA	Tongue	2008	10	22	1	14174	tongue2
not16S.1	ATACTATTGCGC	GTGCCAGCMGCCGCGGTAA	Other	2008	10	22	1	14174	randomly generated sequence plus some variants, these should not map to 16S
"""
    
    biom = """{"id": "None","format": "Biological Observation Matrix 1.0.0","format_url": "http://biom-format.org","type": "OTU table","generated_by": "QIIME 1.6.0-dev","date": "2013-02-09T09:30:11.550590","matrix_type": "sparse","matrix_element_type": "int","shape": [14, 9],"data": [[0,0,20],[0,1,18],[0,2,18],[0,3,22],[0,4,4],[1,4,1],[2,0,1],[2,4,1],[2,5,1],[3,6,1],[4,4,1],[5,7,20],[6,4,1],[7,4,1],[7,5,1],[8,4,1],[8,6,2],[8,8,3],[9,7,2],[10,5,1],[11,4,9],[11,5,20],[11,6,1],[11,8,4],[12,4,3],[12,6,19],[12,8,15],[13,0,1],[13,1,4],[13,2,4]],"rows": [{"id": "295053", "metadata": {"taxonomy": ["k__Bacteria"]}},{"id": "42684", "metadata": {"taxonomy": ["k__Bacteria", "p__Proteobacteria"]}},{"id": "None11", "metadata": {"taxonomy": ["Unclassified"]}},{"id": "None10", "metadata": {"taxonomy": ["Unclassified"]}},{"id": "None7", "metadata": {"taxonomy": ["Unclassified"]}},{"id": "None6", "metadata": {"taxonomy": ["Unclassified"]}},{"id": "None5", "metadata": {"taxonomy": ["k__Bacteria"]}},{"id": "None4", "metadata": {"taxonomy": ["Unclassified"]}},{"id": "None3", "metadata": {"taxonomy": ["k__Bacteria"]}},{"id": "None2", "metadata": {"taxonomy": ["k__Bacteria"]}},{"id": "None1", "metadata": {"taxonomy": ["Unclassified"]}},{"id": "879972", "metadata": {"taxonomy": ["k__Bacteria"]}},{"id": "None9", "metadata": {"taxonomy": ["Unclassified"]}},{"id": "None8", "metadata": {"taxonomy": ["k__Bacteria"]}}],"columns": [{"id": "f2", "metadata": null},{"id": "f1", "metadata": null},{"id": "f3", "metadata": null},{"id": "f4", "metadata": null},{"id": "p2", "metadata": null},{"id": "p1", "metadata": null},{"id": "t1", "metadata": null},{"id": "not16S.1", "metadata": null},{"id": "t2", "metadata": null}]}"""
    
    tree = """(None8:0.09606,((879972:0.05039,None3:0.00778)0.980:0.15948,((None11:0.07161,None4:0.06965)0.917:0.09643,(295053:0.06096,42684:0.15599)0.910:0.08898)0.899:0.09227)0.958:0.13976,(None7:0.10435,(None5:0.02626,None9:0.00014)1.000:0.25335)0.753:0.0465);"""

    result['seqs'] = seqs.split('\n')
    result['extra_seqs'] = extra_seqs.split('\n')
    result['map'] = md.split('\n')
    result['refseqs'] = refseqs.split('\n')
    result['biom'] = biom.split('\n')
    result['tree'] = tree.split('\n')
    result['refseqs_tax'] = refseqs_tax.split('\n')
    result['refseqs_aligned'] = refseqs_aligned.split('\n')
    result['refseqs_aligned_lanemask'] = refseqs_aligned_lanemask.split('\n')
    
    return result

def write_test_data(output_dir):
    """ write small test data set files to output_dir 
    
        This function is convenient for interactive testing. If you need 
        some test files to work with, you can just run:
        
        python -c "from qiime.test import write_test_data; write_test_data('.')"
    """
    test_data = get_test_data()
    for k, v in test_data.items():
        f = open(join(output_dir,k),'w')
        f.write('\n'.join(v))
        f.close()
    
def get_test_data_fps():
    """ Returns test data as dict of filepaths. 
    
    Filepaths are created with the tempfile module, and therefore do not 
    explcitly need to be cleaned up.
    """
    test_data = get_test_data()
    temp_dir = get_qiime_temp_dir()
    result = {}
    for k, v in test_data.items():
        f = NamedTemporaryFile(prefix=k,dir=temp_dir)
        f.write('\n'.join(v))
        f.flush()
        result[k] = (f.name, f)
    return result

