/**********************************************************************************************
    Copyright (C) 2006, 2007 Oliver Eichler oliver.eichler@gmx.de,
    Michael Ritzert michael.ritzert@googlemail.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA

**********************************************************************************************/
#ifndef CCANVAS_H
#define CCANVAS_H

#include <QPixmap>
#include <QWidget>
#include "CGarminDBWpt.h"
#include "CGarminDBTrack.h"
#include "CGarminTrack.h"
#include "CGarminWpt.h"

class IMap;
class QLabel;
class CMainWindow;
class QTimer;
class QTextEdit;
class CInfoScreen;
class CDetailStatus;
class CDetailDetail;
class CGarminRoute;
class QMenu;

class CCanvas : public QWidget
{
    Q_OBJECT
        public:
        CCanvas(CMainWindow * parent);
        virtual ~CCanvas();

        enum mouse_mode_e
        {
            eMouseZoomArea       ///< use mouse to define a zoom area
            , eMouseMoveArea     ///< use mouse to move the map
            , eMouseSelectArea   ///< use mouse to select map tiles
            , eMouseNewWpt       ///< use mouse to add waypoints
            , eMouseEditWpt      ///< use mouse to select waypoints
            , eMouseMoveWpt      ///< use mouse to drag-n-drop waypoints
            , eMouseSearchOC     ///< use mouse to define a search radius for open caching
            , eMouseCutTrack     ///< use mouse to cut a track into two pieces
            , eMouseEditRte      ///< use mouse to define a new route polyline
            , eMouseMoveRte      ///< use mouse to move route points
            , eMouseDelRte       ///< use mouse to delete route points
        };

        enum move_direction_e
        {
            eMoveLeft
            , eMoveRight
            , eMoveUp
            , eMoveDown
        };

        /// change the current mouse mode
        void setMouseMode(mouse_mode_e mode);
        /// get current mouse mode
        mouse_mode_e getMouseMode(){return mouseMode;}
        /// make complete basemap visible
        void fitMapToScreen();
        /// switch display of info screen on/off
        void toggleInfo();
        /// move coordinate to the center of the viewport
        /**
            This will display a mark and the info text if info is
            not empty.
        */
        void move(double lng, double lat, const QString& info);
        void move( move_direction_e dir );

        void getCursorPos(double& lon, double& lat){lon = m_lon; lat = m_lat;}

        /// switch display of city names on/off
        void toggleCityNames();
        /// switch display of point of interest on/off
        void togglePOIs();
        /// switch display of street names on/off
        void toggleStreetNames();

        /// zoom to rectangular area (degrees)
        void zoom(const QRectF& area);
        /// zoom in/out one step
        void zoom(bool in, const QPoint& p);

        void print(QPrinter& printer);

        /// the map render object
        IMap * getMap(){return map;};

        static void drawText(const QString& str, QPainter& p, const QPoint& center);
    protected slots:
        /// increase detail level
        void incDetails();
        /// decrease detail level
        void decDetails();

        void slotIncreaseDetails();

        void slotToggleCityNames();

        void slotTogglePOIs();

        void slotToggleStreetNames();

        void slotCopyPosition();

    protected:
        /// draw map and information to paint device
        /**
            Set buffered to true for fast rendering (screen). This will
            draw the map into a pixmap buffer if it has to be updated. Else
            it will just use this buffer for faster painting.

            For printing you want to use unbuffered drawing.
        */
        void draw(QPainter& p, bool buffered);
        void drawTracks(QPainter& p);
        void drawWaypoints(QPainter& p);
        void drawSelectedMaps(QPainter& p);
        void drawZoomRect(QPainter& p);
        void drawInfoCursor(QPainter& p);
        void drawSearchResultMark(QPainter& p);
        void drawCacheSearchRadius(QPainter& p);
        void drawLiveLogPosition(QPainter& p);
        void drawRoutes(QPainter& p);
        void drawRoutePolygon(CGarminRoute * route, bool isActRoute, QPainter& p);


        void paintEvent(QPaintEvent * e);
        void resizeEvent(QResizeEvent * e);

        void mousePressEvent(QMouseEvent * e);
        void mouseMoveEvent(QMouseEvent * e);
        void mouseMoveEventCoord(QMouseEvent * e);
        void mouseMoveEventSelWpt(QMouseEvent * e);
        void mouseMoveEventSelTrk(QMouseEvent * e);
        void mouseMoveEventSelElem(QMouseEvent * e);
        void mouseReleaseEvent(QMouseEvent * e);

        void wheelEvent(QWheelEvent * e);

        /// set override cursor
        void enterEvent(QEvent * event);
        /// restore cursor
        void leaveEvent(QEvent * event);

        /// pointer to current mouse press handler
        void (CCanvas::*pfMousePressEvent)(QMouseEvent*);
        /// pointer to current mouse move handler
        void (CCanvas::*pfMouseMoveEvent)(QMouseEvent*);
        /// pointer to current mouse release handler
        void (CCanvas::*pfMouseReleaseEvent)(QMouseEvent*);

        /// current mouse mode
        mouse_mode_e mouseMode;
        /// the map render object
        IMap * map;
        /// the status bar's coord. meter
        QLabel * coord;
        /// the status bar's coord. meter
        CDetailStatus * details1;
        CDetailDetail * details2;

        /// mouse position longitude []
        double m_lon;
        /// mouse position latitude []
        double m_lat;
        /// mouse position [px]
        QPoint m_pos;

        /// enable to alter the map offset by moving the mouse
        bool moveMap;
        /// the last mouse point
        /**
            This is used as a reference to calculate the delta
            for map moving.
        */
        QPoint oldPoint;
        /// start to move map
        void mousePressMoveArea(QMouseEvent * e);
        /// move map if left button is pressed
        void mouseMoveMoveArea(QMouseEvent * e);
        /// stop to move map
        void mouseReleaseMoveArea(QMouseEvent * e);

        bool zoomMap;
        /// the zoom rectangle
        QRect rectZoom;
        /// start zoom rectangle
        void mousePressZoomArea(QMouseEvent * e);
        /// resize zoom rectangle
        void mouseMoveZoomArea(QMouseEvent * e);
        /// zoom to the area define by the zoom rectangle
        void mouseReleaseZoomArea(QMouseEvent * e);

        bool selectMap;
        /// the select rectangle
        QRect rectSelect;
        /// start select rectangle
        void mousePressSelectArea(QMouseEvent * e);
        /// resize select rectangle
        void mouseMoveSelectArea(QMouseEvent * e);
        /// select map tiles define by the select rectangle
        void mouseReleaseSelectArea(QMouseEvent * e);

        /// trigger detailed map drawing after a period of no mouse wheel event
        QTimer * timerZoomDetailDelay;

        /// the current cursor if the mouse is over the canvas widget
        QCursor cursor;

        /// display information of elements close to cursor
        CInfoScreen * infoScreen;
        /// the point on the captured line close to the cursor
        QPointF infoPoint;
        /// set true to show infoScreen
        bool showInfo;
        /// info about elements close to the cursor
        QString info;

        /// location of search item
        QPointF searchPoint;
        /// information about search item
        QString searchPointInfo;

        /// pointer to current active waypoint
        QPointer<CGarminWpt> waypoint;
        /// data base key for active waypoint
        QString wptkey;
        /// set true if waypoint is allowed to be moved
        bool wptIsMoving;
        /// old coordinates of waypoint while moving in lon/lat []
        XY wptStartDeg;
        /// old coordinates of waypoint while moving in pixel relative to viewport
        QPoint wptStartPx;

        /// start to move map
        void mousePressNewWpt(QMouseEvent * e);
        /// move map if left button is pressed
        void mouseMoveNewWpt(QMouseEvent *){};
        /// stop to move map
        void mouseReleaseNewWpt(QMouseEvent *){};
        /// start to move map
        void mousePressEditWpt(QMouseEvent * e);
        /// move map if left button is pressed
        void mouseMoveEditWpt(QMouseEvent * e);
        /// stop to move map
        void mouseReleaseEditWpt(QMouseEvent * e);
        /// start to move map
        void mousePressMoveWpt(QMouseEvent * e);
        /// move map if left button is pressed
        void mouseMoveMoveWpt(QMouseEvent * e);
        /// stop to move map
        void mouseReleaseMoveWpt(QMouseEvent * e);

        bool    defineSearch;
        QPointF searchCenter;
        quint32 searchRadius;
        quint32 searchRadiusLimit;
        double  radius;
        void mousePressSearchOC(QMouseEvent * e);
        /// move map if left button is pressed
        void mouseMoveSearchOC(QMouseEvent * e);
        /// stop to move map
        void mouseReleaseSearchOC(QMouseEvent * e);

        /// pointer to current active trackpoint
        CGarminTrack::TrkPt_t * trkpt1st;
        CGarminTrack::TrkPt_t * trkpt2nd;
        void mousePressCutTrack(QMouseEvent * e);
        /// move map if left button is pressed
        void mouseMoveCutTrack(QMouseEvent * e);
        /// stop to move map
        void mouseReleaseCutTrack(QMouseEvent * e);

        struct point_t
        {
            point_t() : valid(false), idx(-1) {}
            void reset(){valid = false; idx = -1;}
            bool    valid;
            QPoint  pt;
            XY      pos;
            int     idx;
        };

        /// the current active route polygon
        QPolygon routePolyline;
        /// the cursor route point to select a position within a route
        point_t  routeCursorPoint;

        /// the position before the actual selected position
        point_t selRoutePointPrev;
        /// the actual select position of a route
        point_t selRoutePoint;
        /// the position after the actual selected position
        point_t selRoutePointNext;

        void mousePressEditRte(QMouseEvent * e);
        void mouseMoveEditRte(QMouseEvent * e);
        void mouseReleaseEditRte(QMouseEvent * e);

        void mousePressMoveRte(QMouseEvent * e);
        void mouseMoveMoveRte(QMouseEvent * e);
        void mouseReleaseMoveRte(QMouseEvent * e);

        void mousePressDelRte(QMouseEvent * e);
        void mouseMoveDelRte(QMouseEvent * e);
        void mouseReleaseDelRte(QMouseEvent * e);

        QPixmap headingPixmap;

        QMenu * contextMenu;
};
#endif                           //CCANVAS_H
