#include <glib-object.h>
#include "streammodel.h"
#include <QDebug>
#include <QProcessEnvironment>

class StreamModelPrivate: public DeeListModel
{
    Q_DECLARE_PUBLIC(StreamModel)

public:
    StreamModelPrivate(StreamModel *model);
    ~StreamModelPrivate();
    static void onModelReady(GObject* parent, GParamSpec *pspec, StreamModelPrivate *q);
    void updateResults(StreamModelPrivate *d);

private:
    mutable StreamModel *q_ptr;
    QString streamId = NULL;
    QString serviceId = NULL;
    uint accountId = 0;
    DeeModel* m_resultsModel;
    DeeModel* m_sharedModel;
    DeeModel* m_sortedModel;
};

StreamModelPrivate::StreamModelPrivate(StreamModel *model):
    DeeListModel(model),
    q_ptr(model)
{
    QString modelName = "com.canonical.Friends.Streams";
    /* read environment variables */
    QProcessEnvironment environment = QProcessEnvironment::systemEnvironment();
    if (environment.contains(QLatin1String("FRIENDS_STREAMS_MODEL")))
    {
        modelName = environment.value(QLatin1String("FRIENDS_STREAMS_MODEL"));
    }

    qDebug() << "MODEL: " << modelName;

    m_sharedModel = dee_shared_model_new (modelName.toUtf8());
    m_resultsModel = m_sharedModel;
    m_sortedModel = dee_sequence_model_new ();
    g_signal_connect(m_sharedModel, "notify::synchronized", G_CALLBACK(onModelReady), this);
}

void
StreamModelPrivate::onModelReady(GObject* parent __attribute__ ((unused)), GParamSpec *pspec, StreamModelPrivate *d)
{
    DeeModel *model = (DeeModel*)parent;
    if (!dee_shared_model_is_synchronized ((DeeSharedModel*)model))
        return;
    DeeFilter _sort_filter;
    dee_filter_new_collator_desc (8, &_sort_filter);
    d->m_sortedModel = dee_filter_model_new (model, &_sort_filter);
    qDebug() << Q_FUNC_INFO << " " << dee_model_get_n_rows (d->m_sortedModel);
    d->updateResults(d);
}

void StreamModelPrivate::updateResults(StreamModelPrivate *d)
{
    if (d->streamId != NULL)
    {
        DeeFilter _key_filter;
        dee_filter_new_for_key_column (3, d->streamId.toUtf8().data(), &_key_filter);
        d->m_resultsModel = dee_filter_model_new (d->m_sortedModel, &_key_filter);
    } else
    {
        GRegex *regex;
        regex = g_regex_new ("^((?!reply_to).)*$", G_REGEX_FIRSTLINE, G_REGEX_MATCH_PARTIAL, NULL);
        DeeFilter _regex_filter;
        dee_filter_new_regex (3, regex, &_regex_filter);
        g_regex_unref (regex);
        d->m_resultsModel = dee_filter_model_new (d->m_sortedModel, &_regex_filter);
    }
    qDebug () << "STREAM: " << d->streamId << " ROWS: " << dee_model_get_n_rows (d->m_resultsModel);

    if (d->serviceId != NULL)
    {
        DeeFilter _service_filter;
        dee_filter_new_for_key_column (0, d->serviceId.toUtf8().data(), &_service_filter);
        d->m_resultsModel = dee_filter_model_new (d->m_resultsModel, &_service_filter);
    }
    qDebug () << "SERVICE: " << d->serviceId << " ROWS: " << dee_model_get_n_rows (d->m_resultsModel);

    if (d->accountId != 0)
    {
        DeeFilter _account_filter;
        dee_filter_new_for_any_column (1, g_variant_new_uint64(d->accountId), &_account_filter);
        d->m_resultsModel = dee_filter_model_new (d->m_resultsModel, &_account_filter);
        qDebug () << "ACCOUNT: " << d->accountId << " ROWS: " << dee_model_get_n_rows (d->m_resultsModel);
    }

    q_ptr->setModel(m_resultsModel);
    Q_EMIT q_ptr->streamChanged();
}

StreamModelPrivate::~StreamModelPrivate()
{
}

StreamModel::StreamModel(DeeListModel *parent) :
    DeeListModel(parent),
    d_ptr(new StreamModelPrivate(this))
{
}

StreamModel::~StreamModel()
{
}


void StreamModel::classBegin()
{
}

void StreamModel::componentComplete()
{
}

void StreamModel::setStream(const QString& streamId)
{
    Q_D(StreamModel);
    if (streamId == d->streamId) return;
    d->streamId = streamId;
    d->updateResults(d);
}

/*!
 * \qmlproperty string StreamModel::stream
 * If set, the model will include only this stream
 */
QString StreamModel::stream() const
{
    Q_D(const StreamModel);
    return d->streamId;
}

void StreamModel::setService(const QString& serviceId)
{
    Q_D(StreamModel);
    if (serviceId == d->serviceId) return;
    d->serviceId = serviceId;
    d->updateResults(d);
}

/*!
 * \qmlproperty string StreamModel::service
 * If set, the model will include only this service
 */
QString StreamModel::service() const
{
    Q_D(const StreamModel);
    return d->serviceId;
}

void StreamModel::setAccount(uint account)
{
    Q_D(StreamModel);
    if (account == d->accountId) return;
    d->accountId = account;
    d->updateResults(d);
    qDebug() << "Account: " << account;
}

/*!
 * \qmlproperty string StreamModel::account
 * If set, the model will include only this account
 */
uint StreamModel::account() const
{
    Q_D(const StreamModel);
    return d->accountId;
}
