// infobar.h				emacs, this is written in -*-c++-*-
//
// This program is free software. See the file COPYING for details.
// Author: Mattias Engdegrd, 1997-1999

#ifndef INFOBAR_H
#define INFOBAR_H

#include <qpainter.h>
#include <qwidget.h>
#include <qlabel.h>
#include <qpushbutton.h>
#include <qframe.h>
#include <qstring.h>
#include <qpixmap.h>

// the ClickLabel is just a QLabel that can be clicked (emits a signal)

class ClickLabel : public QLabel
{
    Q_OBJECT
public:
    ClickLabel(QWidget *parent) : QLabel(parent) {}

signals:
    void clicked();

protected:
    virtual void mousePressEvent(QMouseEvent *);
};

class BarGraph : public QFrame
{
    Q_OBJECT
public:
    BarGraph(QWidget *parent, const char *title, int divs, int bars = 1,
	     bool nolegend = FALSE);
    ~BarGraph();

    void setColor(int div, QColor color) { colors[div] = color; }
    QColor color(int div) { return colors[div]; }
    void setValue(int bar, int div, unsigned long value)
        { values(bar, div) = value; }
    void setLabel(int div, const char *name) { names[div] = name; }
    void refresh();
    void setOrientation(bool vertical);

signals:
    void clicked();

protected:
    virtual void drawContents(QPainter *p);
    virtual void resizeEvent(QResizeEvent *);
    virtual void mousePressEvent(QMouseEvent *);

private:
    void drawBar(QPainter *p, int x, int y, int w, int h, int bar);
    void drawLegend(QPainter *p, int x, int y, int w, int h, int div);
    void refreshBar(QPainter *p);
    void setBarSize();

    // accessors for values
    unsigned long &values(int bar, int div)
        { return vals[bar * ndivs + div]; }

    int nbars;			// number of parallel bars
    int ndivs;			// number of divisions of each bar
    unsigned long *vals;
    QString *names;
    QColor *colors;
    int bar_size;
    const char *caption;
    bool no_legend;
    bool vert;			// true if vertical

    static const int label_width = 50;
    static const int label_height = 16;
    static const int title_width = 30;
    static const int title_height = 16;
    static const int bar_thickness = 16;
};

// A custom-made label for neat multi-cpu status
class CpuLabel : public QFrame
{
    Q_OBJECT
public:
    CpuLabel(QWidget *parent, int cpus);

    void setValue(int cpu, int field, unsigned long value)
        { values(cpu, field) = value; }
    void setOrientation(bool vertical);
    
signals:
    void clicked();

protected:
    virtual void drawContents(QPainter *p);
    virtual void mousePressEvent(QMouseEvent *);

    int ncpus;
    unsigned long *vals;	// array of 4 * ncpus
    // accessor for values
    unsigned long &values(int cpu, int field) {return vals[cpu * 4 + field];}
    bool vert;
};

class LoadGraph : public QFrame
{
    Q_OBJECT
public:
    LoadGraph(QWidget *parent, int history_size);
    ~LoadGraph();

    void add_load_point();	// add current load as a data point
    void configure();		// respond to configuration change

    QPixmap *make_icon(int w, int h);

signals:
    void clicked();

protected:
    void drawContents(QPainter *p);
    void mousePressEvent(QMouseEvent *);

    void add_history_point(unsigned value);
    void make_graph(int w, int h);

private:
    const int hist_size;	// history buffer size
    int npoints;		// nr of history points currently remembered
    // history points are stored in fixed point, scaled by history_scale
    static const unsigned history_scale = 256;
    static const int time_dist = 60; // time between vertical lines (seconds)
    unsigned *history;		// (circular) history buffer
    unsigned peak;		// largest value in history
    int h_index;		// next history index to use
    
    bool dirty;			// history changed since pixmap was drawn
    QPixmap pm;
};

class Infobar : public QWidget
{
    Q_OBJECT
public:
    Infobar(QWidget *parent);

    void refresh();
    void update_load();
    QPixmap *load_icon(int w, int h) { return load_graph->make_icon(w, h); }
    void configure();
    void configure_geometry();
    bool swaplim_exceeded();

public slots:
    void toggle_load();
    void toggle_cpu();
    void toggle_mem();
    void toggle_swap();

signals:
    void config_change();

protected:
    void resizeEvent(QResizeEvent *);

private:
    ClickLabel *load_lbl;
    LoadGraph *load_graph;
    BarGraph *mem_bar;
    ClickLabel *mem_lbl;
    QLabel *up_lbl;
    BarGraph *cpu_bar;
    CpuLabel *cpu_lbl;
    BarGraph *swap_bar;
    ClickLabel *swap_lbl;

    bool is_vertical;

    void show_and_hide(bool, QWidget *, QWidget *);
};

#endif	// INFOBAR_H
