// CRAZY SOURCE !!!
//
// proc.cpp for Linux  
// *** the class  shuld be changed to simple functions ***  (by fasthyun)
//
// This program is free software. See the file COPYING for details.
// Author: Mattias Engdegrd, 1997-1999
// 		   Oliver
// 		   fasthyun@magicn.com 2005-


#include <stdio.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/syscall.h>
#include <dirent.h>
#include <string.h>
#include <fcntl.h>
#include <time.h>
#include <sched.h>

#include "qps.h"
#include "proc.h"
#include "svec.cpp"
#include "uidstr.h"
#include "ttystr.h"
#include "wchan.h"
#include "details.h"

#include <qmessagebox.h>

#include "proc_common.cpp"

//#define FAKE_SMP 4	// for SMP debugging on UP machines    ???(by fasthyun@magicn.com) 

#define PROCDIR  "/proc"  // will be removed !



// socket states, from <linux/net.h> and touched to avoid name collisions
enum {
	SSFREE = 0,			/* not allocated		*/
	SSUNCONNECTED,			/* unconnected to any socket	*/
	SSCONNECTING,			/* in process of connecting	*/
	SSCONNECTED,			/* connected to socket		*/
	SSDISCONNECTING			/* in process of disconnecting	*/
};

#define QPS_SCHED_AFFINITY ok
#ifdef QPS_SCHED_AFFINITY
#ifndef SYS_sched_setaffinity
#define SYS_sched_setaffinity 241
#endif
#ifndef SYS_sched_getaffinity
#define SYS_sched_getaffinity 242
#endif
// Needed for some glibc
int qps_sched_setaffinity(pid_t pid, unsigned int len, unsigned long *mask) {
      return syscall(SYS_sched_setaffinity, pid, len, mask);
};
int qps_sched_getaffinity(pid_t pid, unsigned int len, unsigned long *mask) {
      return syscall(SYS_sched_getaffinity, pid, len, mask);
};
#endif


/*
 *  Who wrote this long long comment ?  ( by fasthyun@magicn.com )
 *
When we commute from task group level to task level (or task level
to task group level) we need to completley destroy the current state of
proc[]. We exchange global <--> analytical informations. Some fields may
not be accurate immediately : %WCPU for example.
Some remarks : suppose we have a task group with TGID = 120, a thread leader 
with PID=120 and two more threads with PID=125 and PID=127.
In /proc, we only see /proc/120 and we don't see /proc/125 or /proc/127.
This does not prevent the possibility to do an open("/proc/125", ...) or an
open("/proc/127", ...). Also, path like /proc/120/task, /proc/125/task and
/proc/127/task are ok.
If the leader thread 120 terminates (it does un pthread_exit()), the directory
/proc/120 will be maintained supposing 125 or 127 are alives. 
Nevertheless, many files and directories will be empty in /proc/120. 
For example, we lose "natual" access to threads 125 and 127 through /proc/120/task.
At task group level, /proc/120/stat and /proc/120/status keep the global 
informations ; so CPU time is still incresing. But, as task is empty, we see
nothing for this task group at thread level.

*/

int Procinfo::page_k_shift;

struct {
	int  proc_id;
	char flag;
	char type;
	int  files;
} proc_info;
	
struct {
	int proc_id;
	int flag;
	char *filename;	 //  path + filename 
} list_files;


// Description: read /proc/PID/fd/*  check opened file, count opened files
//  		will be called  when every update time.
// Return Value : 
// Created by fasthyun@magicn.com
int proc_PID_fd(const int pid)  
{
	char 	path[256];
	char 	buffer[256],fname[256];
	DIR 	*d;	
	int	fdnum;
	int	len, path_len;

	sprintf(path, "/proc/%d/fd", pid);

	path_len=strlen(path);
	d = opendir(path);
	
	if(!d){
		// this happend when the process died already !
		//printf("DEBUG: PID_read_fd() *d==NULL? possible? \n");
		return FALSE;
	}

	struct dirent *e;
	while((e = readdir(d)) != 0) 
	{
		if(e->d_name[0] == '.')
			continue;		// skip "." and ".."
		
		path[path_len]='/';
		path[path_len+1]=0;
		strcat(path,e->d_name);
		
		len = readlink(path, fname, sizeof(fname) - 1); 		
		if( len > 0) 
		{	
			fname[len]=0;
			//printf("DEBUG: %s[%s]\n",path,fname);
			//if (strcmp(fname,"/dev/null")==0 ) continue;	
		}

		num_opened_files++;
		//strcpy(p, e->d_name);
		//fdnum = atoi(p);
		//read_fd(fdnum, path);
	}
	closedir(d);
	return TRUE;	

}


Procinfo::Procinfo(int proc_pid) : refcnt(1)
{
	details = 0;
	children = 0;
	fd_files = 0;
	maps = 0;

	sock_inodes = 0;
	socks_current = FALSE;
	usocks_current = FALSE;
	per_cpu_times = 0;

	environ = 0;
	envblock = 0;
	tgid=0;
	if(readproc(proc_pid) < 0)
		pid = -1;		// invalidate object, will be deleted

	selected = FALSE;
	hidekids = FALSE;
}

Procinfo::~Procinfo()
{
	if(details) {
		details->process_gone();
		details = 0;
	}
	delete environ;
	if(envblock)
		free(envblock);

	delete sock_inodes;
	delete[] per_cpu_times;

	if(maps) {
		maps->purge();
		delete maps;
	}
	if(fd_files) {
		fd_files->purge();
		delete fd_files;
	}
	delete children;
}

// miscellaneous static initializations
void Procinfo::init_static()
{
	socks.setAutoDelete(TRUE);
	usocks.setAutoDelete(TRUE);

	page_k_shift = 0;
	for(int j = getpagesize(); j > 1024; j >>= 1)
		page_k_shift++;
}

// COMMON 
// return number of bytes read if ok, -1 if failed
int Procinfo::read_file(char *name, void *buf, int max)
{
	int fd = open(name, O_RDONLY);
	if(fd < 0) return -1;
	int r = read(fd, buf, max);
	close(fd);
	return r;
}


//  Description : read /proc/1234/task/*  tasks(thread,LWP)  
//  		  add to Proc::procs[]  
//
//  Created by fasthyun@magicn.com
int Proc::readTaskByPID(int pid)
{
	char 	path[256];
	char 	*p;
	struct 	dirent *e;
	int 	thread_id;
	int 	thread_n=0;

///	printf("DEBUG:pid=%d :",pid);
	sprintf(path, "%s/%d/task",PROCDIR, pid);

	DIR *d = opendir(path);
	if(!d) return -1;


	while((e = readdir(d)) != 0) {
		if(e->d_name[0] == '.') 	continue;	// skip "." , ".."
		
		thread_id = atoi(e->d_name); // only number !!
///		printf("tid=%d ",TID);

		if(pid!=thread_id)
		{
			Procinfo *pi = new Procinfo(thread_id);

			if(pi->pid == -1)
				delete pi;		// already gone
		
			else {
				pi->generation = current_gen;
				pi->cmdline="(thread)";
				newproc(pi);
			}
				
		}

		thread_n++;
	}
	///printf("\n");
	closedir(d);				
	return thread_n;
}


// NO.2 BOTTLENECK !!!
// Description : read /proc/$PID/*  
// 		be called every UPDATE time.
//
// fasthyun@magicn.com : tooo complex  , shoud be simple and clean !
int Procinfo::readproc(int proc_pid)
{
	char path[1024];  // enough
	char buf[1024];   // enough  
	char sbuf[4096];  // should be enough to acommodate /proc/1234/stat
	char cmdbuf[MAX_CMD_LEN];
	int  cmdlen;
	QString tmp_str;
	pid = proc_pid;


	// sprintf(path, "%s/%d", "/proc", pid);
	if (flag_thread_ok && flag_show_thread)
		sprintf(path, "/proc/%d/task/%d", proc_pid,proc_pid);
	else
		sprintf(path, "/proc/%d",  proc_pid);

	if (proc_PID_fd(proc_pid)==TRUE) ;
		
	// read /proc/1234/cmdline
	strcpy(buf, path);
	strcat(buf, "/cmdline");
	if((cmdlen = read_file(buf, cmdbuf, MAX_CMD_LEN - 1)) < 0)
		return -1;
	if(cmdlen == 0) {
		cmdline = "";
	} else {
		for(int i = 0; i < cmdlen; i++)
			if(!cmdbuf[i]) cmdbuf[i] = ' ';
		int j = cmdlen - 1;
		while(j >= 0 && cmdbuf[j] == ' ')
			j--;
		cmdbuf[j + 1] = '\0';
		make_printable(cmdbuf);
		tmp_str = cmdbuf;
		cmdline = UniString(tmp_str); // for Non-ascii locale language like Korea,Japan,China
	}

	// read /proc/1234/stat
	strcpy(buf, path);
	strcat(buf, "/stat");
	int statlen;
	if((statlen = read_file(buf, sbuf, sizeof(sbuf) - 1)) <= 0) 
		return -1;
	sbuf[statlen] = '\0';
	char *p = strrchr(sbuf, ')');
	*p = '\0';			// split in two parts
	command = strchr(sbuf, '(') + 1;

	//
	// Not all values from /proc/#/stat are interesting; the ones left out
	// have been retained in comments to see where they should go, in case
	// they are needed again.
	//
	// In Linux 2.2.x, timeout has been removed, and the signal information
	// here is obsolete (/proc/#/status has real-time signal info).
	//
	// There are undocumented values after wchan, unused so far:
	// nswap		pages swapped out since process started
	// cnswap		nswap of children
	// exit_signal	(2.2.x) signal sent to parent when process exits
	// (The latter could provide a way to detect cloned processes since
	//  they usually have exit_signal != SIGCHLD, but I prefer waiting
	//  for real TIDs before implementing thread support.)
	//
	long stime, cstime;
	int i_tty;
	sscanf(p + 2, "%c %d %d %d %d %d %lu %lu %lu %lu %lu "
			"%ld %ld %ld %ld %d %d %d %*s %lu %*s %*s %*s %*s %*s %*s %*s %*s "
			"%*s %*s %*s %*s %lu",
			&state, &ppid, &pgrp, &session, &i_tty, &tpgid,
			&flags, &minflt, &cminflt, &majflt, &cmajflt,
			&utime, &stime, &cutime, &cstime, &priority, &nice,&nthreads,
			/* itrealvalue */
			&starttime,
			/* vsize */
			/* rss */
			/* rlim, startcode, endcode, startstack kstkesp kstkeip,
			   signal, blocked, sigignore, sigcatch */
			&wchan);

	tty = (dev_t)i_tty;
	utime += stime;		// we make no user/system time distinction 
	cutime += cstime;

	// read /proc/XX/statm
	strcpy(buf, path);
	strcat(buf, "/statm");
	if((statlen = read_file(buf, sbuf, sizeof(sbuf) - 1)) <= 0) 
	{
		printf("33");
		return -1;
	}

	sbuf[statlen] = '\0';
	sscanf(sbuf, "%lu %lu %lu %lu %lu %lu %lu",
			&size, &resident, &share, &trs, &lrs, &drs, &dt);
	size <<= page_k_shift;
	resident <<= page_k_shift;
	share <<= page_k_shift;
	trs <<= page_k_shift;
	lrs <<= page_k_shift;
	drs <<= page_k_shift;

	pmem = 100.0 * resident / mem_total;


	// read /proc/XX/status
	strcpy(buf, path);  // /proc/$PID
	strcat(buf, "/status");
	if((statlen = read_file(buf, sbuf, sizeof(sbuf) - 1)) <= 0) return -1;
	sbuf[statlen] = '\0';
	if(!(p = strstr(sbuf, "Tgid:")))
		return -1;
	sscanf(p, "Tgid: %d ", &tgid);
	// slpavg not supported in kernel 2.4; default value of -1
    	if((p = strstr(sbuf, "SleepAVG:")))
    		sscanf(p, "SleepAVG: %d",&slpavg);
	else 
		slpavg =-1;

	if(!(p = strstr(sbuf, "Uid:")))
		return -1;
	sscanf(p, "Uid: %d %d %d %d Gid: %d %d %d %d",
			&uid, &euid, &suid, &fsuid,
			&gid, &egid, &sgid, &fsgid);

	which_cpu = 0;
	per_cpu_times = 0;

	// SMP 
	if(num_cpus > 1) {
		per_cpu_times = new unsigned long[num_cpus];
		if (per_cpu_times == NULL){
			printf("Allocation problem - per_cpu_times\n");
			return -1;
		}
		// less than version 2.6.0  and SMP
		if(flag_24_ok) {
			strcpy(buf, path); // /proc/$PID
			strcat(buf, "/cpu"); // /proc/$PID/cpu
			if( (statlen = read_file(buf, sbuf, sizeof(sbuf) - 1)) <= 0) {
				delete per_cpu_times;
				return -1;
			}
			sbuf[statlen] = '\0';
			p = sbuf;
			for(unsigned cpu = 0; cpu < num_cpus; cpu++) {
				p = strchr(p, '\n');
				if (!p) {
					for(cpu = 0; cpu < num_cpus; cpu++)
						per_cpu_times[cpu] = 0;
					break;
				}
				p++;
				unsigned long utime, stime;
				sscanf(p, "%*s %lu %lu", &utime, &stime);
				per_cpu_times[cpu] = utime + stime;
			}
		}
	}

	gettimeofday(&tv, 0);
	policy = -1;		// will get it when needed
	rtprio = -1;		// ditto
	tms = -1;		// ditto

	num_process++;

	return pid;
}


float 	Procinfo::loadavg[] = {0.0, 0.0, 0.0};  // CPU load avg 1min,5min,15minute
int 	Procinfo::mem_total = 0;
int 	Procinfo::mem_free = 0;
int 	Procinfo::mem_buffers = 0;
int 	Procinfo::mem_cached = 0;
int 	Procinfo::swap_total = 0;
int 	Procinfo::swap_free = 0;
unsigned *Procinfo::cpu_times_vec = 0;
unsigned *Procinfo::old_cpu_times_vec = 0;
long Procinfo::boot_time = 0;
bool Procinfo::socks_current = FALSE;
QIntDict<Sockinfo> Procinfo::socks(17);
QIntDict<UnixSocket> Procinfo::usocks(17);
bool 	Procinfo::usocks_current = FALSE;
unsigned int Procinfo::num_cpus = 0;
unsigned int Procinfo::old_num_cpus = 0;
long 	Procinfo::num_process = 0;
long 	Procinfo::num_network_process = 0;
//DEL? int Procinfo::mem_shared = 0; // only linux kernel 2.4.x

// just grab the load averages
void Procinfo::read_loadavg()
{
	char path[80];
	char buf[512];
	int  n;
	strcpy(path, "/proc/loadavg");
	if((n = read_file(path, buf, sizeof(buf) - 1)) <= 0) {
		fprintf(stderr,	"qps: Cannot open /proc/loadavg  (make sure /proc is mounted)\n");
		exit(1);
	}
	buf[n] = '\0';
	sscanf(buf, "%f %f %f", &loadavg[0], &loadavg[1], &loadavg[2]);
}

// Description:  read common information  for all processes
// 		 !!! should be  simplified !!(by fasthyun@magicn.com) 
void Procinfo::read_common()
{
	char path[80];
	char buf[4096 + 1];
	

	char *p;
	int n;

	
	/* Version 2.4.x ? */
	strcpy(path,PROCDIR);
	strcat(path,"/vmstat");
	if (!stat(path, (struct stat*)buf) )
		flag_24_ok=FALSE;
	else
		flag_24_ok =TRUE;
	strcpy(path,PROCDIR);

	/* NTPL tasking ? */
	strcat(path,"/1/task");
	if (!stat(path, (struct stat*)buf) )
		flag_thread_ok = TRUE;
	else
		flag_thread_ok = FALSE;

	// read memory info
	strcpy(path, PROCDIR);
	strcat(path, "/meminfo");
	if((n = read_file(path, buf, sizeof(buf) - 1)) <= 0) return;
	buf[n] = '\0';

	// Skip the old /meminfo cruft, making this work in post-2.1.42 kernels
	// as well.  (values are now in kB)
	if( (p = strstr(buf, "MemTotal:")) != NULL )
		sscanf(p, "MemTotal: %d kB\n", &mem_total);
	if( (p = strstr(buf, "MemFree:")) != NULL )
		sscanf(p, "MemFree: %d kB\n", &mem_free);
	if( (p = strstr(buf, "Buffers:")) != NULL )
		sscanf(p, "Buffers: %d kB\n", &mem_buffers);
	if( (p = strstr(buf, "Cached:")) != NULL )
		sscanf(p, "Cached: %d kB\n", &mem_cached);

	p = strstr(buf, "SwapTotal:");
	sscanf(p, "SwapTotal: %d kB\nSwapFree: %d kB\n", &swap_total, &swap_free);

	// read system status
	strcpy(path, PROCDIR);
	strcat(path, "/stat"); // /proc/stat
	if((n = read_file(path, buf, sizeof(buf) - 1)) <= 0) return;
	buf[n] = '\0';

	old_num_cpus = num_cpus;
	if(!num_cpus) {
		// count cpus
#ifdef FAKE_SMP
		num_cpus = 4;
#else
		char *p;
		p = strstr(buf, "cpu");
		while(p < buf + sizeof(buf) - 4 && strncmp(p, "cpu", 3) == 0) {
			num_cpus++;
			if(strncmp(p, "cpu0", 4) == 0)
				num_cpus--;
			p = strchr(p, '\n');
			if(p)
				p++;
		}
#endif
		cpu_times_vec = new unsigned[CPUTIMES * num_cpus];
		old_cpu_times_vec = new unsigned[CPUTIMES * num_cpus];
	}

	for(unsigned cpu = 0; cpu < num_cpus; cpu++)
		for(int i = 0; i < CPUTIMES; i++)
			old_cpu_times(cpu, i) = cpu_times(cpu, i);

	if(num_cpus == 1) {
		unsigned iowait, irq, sftirq, nflds;
		nflds = sscanf(buf, "cpu %u %u %u %u %u %u %u",
				&cpu_times(0, CPUTIME_USER), &cpu_times(0, CPUTIME_NICE),
				&cpu_times(0, CPUTIME_SYSTEM), &cpu_times(0, CPUTIME_IDLE),
				&iowait, &irq, &sftirq);
		if( nflds > 4 ) {
			cpu_times(0, CPUTIME_SYSTEM)+= (irq+sftirq);
			cpu_times(0, CPUTIME_IDLE)+=iowait;
		}
	} else {
#ifdef FAKE_SMP
		sscanf(buf, "cpu %u %u %u %u",
				&cpu_times(0, CPUTIME_USER), &cpu_times(0, CPUTIME_NICE),
				&cpu_times(0, CPUTIME_SYSTEM), &cpu_times(0, CPUTIME_IDLE));
		for(unsigned cpu = 1; cpu < num_cpus; cpu++) {
			for(int i = 0; i < CPUTIMES; i++)
				cpu_times(cpu, i) = cpu_times(0, i);
		}
#else
		// SMP
		for(unsigned cpu = 0; cpu < num_cpus; cpu++) {
			char cpu_buf[10];
			sprintf(cpu_buf, "cpu%d", cpu);
			if((p = strstr(buf, cpu_buf)) != 0) {
		                unsigned iowait, irq, sftirq, nflds;
				nflds = sscanf(p, "%*s %u %u %u %u %u %u %u",
                                &cpu_times(cpu, CPUTIME_USER), &cpu_times(cpu, CPUTIME_NICE),
                                &cpu_times(cpu, CPUTIME_SYSTEM), &cpu_times(cpu, CPUTIME_IDLE),
                                &iowait, &irq, &sftirq);
                		if( nflds > 4 ) {
                        		cpu_times(cpu, CPUTIME_SYSTEM)+= (irq+sftirq);
                        		cpu_times(cpu, CPUTIME_IDLE)+=iowait;
				}

			} else {
				fprintf(stderr, "Error reading info for cpu %d\n", cpu);
				abort();
			}
		}
#endif
	}

	// 2.0.x kernels (at least up to 2.0.33) have an SMP bug that reports
	// cpu_time(CPUTIME_IDLE) incorrectly, since it doesn't take the number of
	// cpus into account. This is fixed in 2.1.x kernels, and since 2.0.x
	// is rather old (and unsuited for SMP anyway) we don't work around it.

	p = strstr(buf, "btime") + 6;
	sscanf(p, "%lu", &boot_time);
}


int Procinfo::get_policy()
{
	if(policy == -1)
		policy = sched_getscheduler(pid);
	return policy;
}

int Procinfo::get_rtprio()
{
	if(rtprio == -1) {
		struct sched_param p;
		if(sched_getparam(pid, &p) == 0)
			rtprio = p.sched_priority;
	}
	return rtprio;
}
double Procinfo::get_tms()
{
        struct timespec ts;
        if(sched_rr_get_interval(pid, &ts)==-1)
               	tms = -1; // should not be possible
        else {
               	tms = ts.tv_nsec;
               	tms /= 1000000000;
               	tms +=ts.tv_sec;
        }
	return tms;
}

unsigned long Procinfo::get_affcpu()
{
#ifdef QPS_SCHED_AFFINITY
	if(qps_sched_getaffinity(pid, sizeof(unsigned long), &affcpu)==-1)
        	affcpu=(unsigned long)0; 
#else
	if(sched_getaffinity(pid, sizeof(unsigned long), (cpu_set_t*)&affcpu)==-1)
        	affcpu=(unsigned long)0; 
#endif
	return affcpu;
}

// Description : read  /proc/1234/fd/* (SYMBOLIC LINK NAME)  
//
/* We need to implement support for IPV6 and sctp ? */
void Procinfo::read_fd(int fdnum, char *path)
{
	int len;
	char buf[80];
	struct stat sb;

	// The fd mode is contained in the link permission bits
	if(lstat(path, &sb) < 0)
		return;
	int mode = 0;
	if(sb.st_mode & 0400) mode |= OPEN_READ;
	if(sb.st_mode & 0200) mode |= OPEN_WRITE;

	if( (len = readlink(path, buf, sizeof(buf) - 1)) > 0) {
		buf[len] = '\0';
		unsigned long dev, ino;
		if((buf[0] == '['	// Linux 2.0 style /proc/fd
					&& sscanf(buf, "[%lx]:%lu", &dev, &ino) == 2
					&& dev == 0)
				|| sscanf(buf, "socket:[%lu]", &ino) > 0) { // Linux 2.1
			Sockinfo *si = Procinfo::socks[ino];
			char buf[80];
			if(si) {
				// a TCP or UDP socket
				sock_inodes->add(SockInode(fdnum, ino));
				sprintf(buf, "%sp socket %lu",
						si->proto == Sockinfo::TCP ? "tc" : "ud", ino);
				fd_files->add(new Fileinfo(fdnum, buf, mode));
				return;
			} else {
				// maybe a unix domain socket?
				read_usockets();
				UnixSocket *us = Procinfo::usocks[ino];
				if(us) {
					char *tp = "?", *st = "?";
					switch(us->type) {
						case SOCK_STREAM: tp = "stream"; break;
						case SOCK_DGRAM: tp = "dgram"; break;
					}
					switch(us->state) {
						case SSFREE: st = "free"; break;
						case SSUNCONNECTED: st = "unconn"; break;
						case SSCONNECTING: st = "connecting"; break;
						case SSCONNECTED: st = "connected"; break;
						case SSDISCONNECTING: st = "disconn"; break;
					}
					sprintf(buf, "unix domain socket %lu (%s, %s) ",
							ino, tp, st);
					QString s = buf;
					s.append(us->name);
					fd_files->add(new Fileinfo(fdnum, s, mode));
					return;
				}
			}
		}
		// assume fds will be read in increasing order
		fd_files->add(new Fileinfo(fdnum, buf, mode));
	}
}


// Description : read 
// return TRUE if /proc/PID/fd could be read, FALSE otherwise
// store fileinfo, and also socket inodes separately
bool Procinfo::read_fds()
{
	char path[80], *p;
	if (flag_thread_ok && flag_show_thread)
		sprintf(path,"%s/%d/task/%d/fd",PROCDIR,pid,pid);
	else
		sprintf(path, "%s/%d/fd", PROCDIR, pid);

	DIR *d = opendir(path);
	if(!d) return FALSE;

	if(!fd_files)
		fd_files = new Svec<Fileinfo*>(8);
	fd_files->clear();

	if(!sock_inodes)
		sock_inodes = new Svec<SockInode>(4);
	sock_inodes->clear();

	p = path + strlen(path) + 1;	// foolish solution...(by fasthyun@magicn.com) 
	p[-1] = '/';

	struct dirent *e;
	while((e = readdir(d)) != 0) {
		if(e->d_name[0] == '.')
			continue;		// skip . and ..
		strcpy(p, e->d_name);
		int fdnum = atoi(p);
		read_fd(fdnum, path);
	}
	closedir(d);
	return TRUE;
}


bool Procinfo::read_socket_list(Sockinfo::proto_t proto, char *pseudofile)
{
	char path[80];
	strcpy(path, PROCDIR);
	strcat(path, "/net/");
	strcat(path, pseudofile);
	FILE *f = fopen(path, "r");
	if(!f) return FALSE;

	char buf[256];
	fgets(buf, sizeof(buf), f);	// skip header
	while(fgets(buf, sizeof(buf), f) != 0) {
		Sockinfo *si = new Sockinfo;
		si->proto = proto;
		unsigned local_port, rem_port, st, tr;
		sscanf(buf + 6, "%x:%x %x:%x %x %x:%x %x:%x %x %d %d %d",
				&si->local_addr, &local_port, &si->rem_addr, &rem_port,
				&st, &si->tx_queue, &si->rx_queue,
				&tr, &si->tm_when, &si->rexmits,
				&si->uid, &si->timeout, &si->inode);
		// fix fields that aren't sizeof(int)
		si->local_port = local_port;
		si->rem_port = rem_port;
		si->st = st;
		si->tr = tr;

		socks.insert(si->inode, si);
		if(socks.count() > socks.size() * 3)
			socks.resize(socks.count());
	}
	fclose(f);
	return TRUE;
}


bool Procinfo::read_usocket_list()
{
	char path[80];
	strcpy(path, PROCDIR);
	strcat(path, "/net/unix");
	FILE *f = fopen(path, "r");
	if(!f) return FALSE;

	char buf[256];
	fgets(buf, sizeof(buf), f);	// skip header
	while(fgets(buf, sizeof(buf), f)) {
		if(buf[0])
			buf[strlen(buf) - 1] = '\0'; // chomp newline
		UnixSocket *us = new UnixSocket;
		unsigned q;
		unsigned type, state;
		int n;
		sscanf(buf, "%x: %x %x %x %x %x %ld %n",
				&q, &q, &q, &us->flags, &type, &state, &us->inode, &n);
		us->name = buf + n;
		us->type = type;
		us->state = state;
		usocks.insert(us->inode, us);
		if(usocks.count() > usocks.size() * 3)
			usocks.resize(usocks.count());
	}
	fclose(f);
	return TRUE;
}

void Procinfo::read_sockets()
{
	if(socks_current)
		return;

	socks.clear();
	if(!read_socket_list(Sockinfo::TCP, "tcp") || !read_socket_list(Sockinfo::UDP, "udp"))
		return;

	socks_current = TRUE;
}

void Procinfo::read_usockets()
{
	if(usocks_current)
		return;

	usocks.clear();
	if(!read_usocket_list())
		return;

	usocks_current = TRUE;
}

void Procinfo::invalidate_sockets()
{
	socks_current = usocks_current = FALSE;
}

// return TRUE if /proc/XX/maps could be read, FALSE otherwise
bool Procinfo::read_maps()
{
	// idea: here we could readlink /proc/XX/exe to identify the executable
	// when running 2.0.x
	char name[80];

	if (flag_thread_ok && flag_show_thread)
		sprintf(name, "%s/%d/task/%d/maps", PROCDIR, pid,pid);
	else
		sprintf(name, "%s/%d/maps", PROCDIR, pid);
	FILE *f = fopen(name, "r");
	if(!f) return FALSE;
	char line[1024];		// lines can be this long, or longer
	if(!maps)
		maps = new Svec<Mapsinfo *>;
	else
		maps->clear();

	while(fgets(line, sizeof(line), f)) {
		Mapsinfo *mi = new Mapsinfo;
		int n;
		sscanf(line, "%lx-%lx %4c %lx %x:%x %lu%n",
				&mi->from, &mi->to, mi->perm, &mi->offset,
				&mi->major, &mi->minor, &mi->inode, &n);
		if(line[n] != '\n') {
			int len = strlen(line);
			if(line[len - 1] == '\n')
				line[len - 1] = '\0';
			while(line[n] == ' ' && line[n]) n++;
			mi->filename = line + n;
		} else if((mi->major | mi->minor | mi->inode) == 0)
			mi->filename = "(anonymous)";
		maps->add(mi);
	}
	fclose(f);
	return TRUE;
}

// return TRUE if /proc/XX/environ could be read, FALSE otherwise
bool Procinfo::read_environ()
{
	int bs = 4096;		// good start
	if(envblock) free(envblock);
	envblock = (char *)malloc(bs + 1);
	char path[256];

	if(flag_thread_ok && flag_show_thread)
		sprintf(path, "%s/%d/task/%d/environ", PROCDIR, pid,pid);
	else
		sprintf(path, "%s/%d/environ", PROCDIR, pid);

	int fd = open(path, O_RDONLY);
	if(fd < 0) {
		free(envblock);
		envblock = 0;
		return FALSE;
	}
	int n;
	int ofs = 0;
	while((n = read(fd, envblock + ofs, bs - ofs)) == bs - ofs) {
		ofs = bs;
		envblock = (char *)realloc(envblock, (bs += 4096) + 1);
	}
	close(fd);
	if(n < 0) {
		free(envblock);
		envblock = 0;
		return FALSE;
	}
	n += ofs;
	envblock[n] = '\0';
	if(!environ)
		environ = new Svec<NameValue>(64);
	else
		environ->clear();
	for(int i = 0; i < n;) {
		char *p = strchr(envblock + i, '=');
		if(p)
			*p++ = '\0';
		else	// degenerate variable: treat as name with empty value
			p = envblock + i + strlen(envblock + i);
		make_printable(envblock + i);
		make_printable(p);
		environ->add(NameValue(envblock + i, p));
		i = p - envblock + strlen(p) + 1;
	}
	return TRUE;
}




Cat_dir::Cat_dir(const char *heading, const char *explain, const char *dirname,
		QString Procinfo::*member)
: Cat_string(heading, explain),
	dir(dirname),
	cache(member)
{}

QString Cat_dir::string(Procinfo *p)
{
	if((p->*cache).isNull()) {
		char path[128], buf[512];

		if (flag_thread_ok && flag_show_thread)
			sprintf(path, "%s/%d/task/%d/%s", PROCDIR, p->pid,p->pid,dir);
		else
			sprintf(path, "%s/%d/%s", PROCDIR, p->pid, dir);

		int n = readlink(path, buf, sizeof(buf) - 1);
		if(n < 0) {
			// Either a kernel process, or access denied.
			// A hyphen is visually least disturbing here.
			p->*cache = "-";
			return p->*cache;
		} else if(buf[0] != '[') {
			// linux >= 2.1.x: path name directly in link
			buf[n] = '\0';
			p->*cache = buf;
			return p->*cache;
		}

		// Either a Linux 2.0 link in [device]:inode form, or a Solaris link.
		// To resolve it, we just chdir() to it and see where we end up.
		// Perhaps we should change back later?
		if(chdir(path) < 0) {
			p->*cache = "-";    // Most likely access denied
		} else {
			// getcwd() is fairly expensive, but this is cached anyway
			if(!getcwd(buf, sizeof(buf))) {
				p->*cache = "(deleted)";
			} else 
				p->*cache = buf;
		}
	}
	return p->*cache;
}

Cat_state::Cat_state(const char *heading, const char *explain)
: Category(heading, explain)
{}

QString Cat_state::string(Procinfo *p)
{
	QString s("   ");
	int ni = p->nice;

	s[0] = p->state;
	s[1] = (p->resident == 0 && p->state != 'Z') ? 'W' : ' ';
	s[2] = (ni > 0) ? 'N' : ((ni < 0) ? '<' : ' ');
	return s;
}

Cat_policy::Cat_policy(const char *heading, const char *explain)
: Category(heading, explain)
{}

QString Cat_policy::string(Procinfo *p)
{
	QString s;
	switch(p->get_policy()) {
		case SCHED_FIFO:
			s = "FI"; break;	// first in, first out
		case SCHED_RR:
			s = "RR"; break;	// round-robin
		case SCHED_OTHER:
			s = "TS"; break;	// time-sharing
		default:
			s = "??"; break;
	}
	return s;
}

int Cat_policy::compare(Procinfo *a, Procinfo *b)
{
	return b->get_policy() - a->get_policy();
}

Cat_rtprio::Cat_rtprio(const char *heading, const char *explain)
: Category(heading, explain)
{}

QString Cat_rtprio::string(Procinfo *p)
{
	QString s;
	s.setNum(p->get_rtprio());
	return s;
}

int Cat_rtprio::compare(Procinfo *a, Procinfo *b)
{
	return b->get_rtprio() - a->get_rtprio();
}

Cat_tms::Cat_tms(const char *heading,const char *explain)
        : Category(heading,explain)
{}

QString Cat_tms::string(Procinfo *p)
{
        QString s;
	p->tms = p->get_tms();
        s.sprintf("%.3f",p->tms);
        return s;
}
int Cat_tms::compare(Procinfo *a, Procinfo *b)
{
    	return (b->get_tms() - a->get_tms())*1000 ;
}

Cat_affcpu::Cat_affcpu(const char *heading,const char *explain)
        : Category(heading,explain)
{}

QString Cat_affcpu::string(Procinfo *p)
{
        QString s;
	p->affcpu = p->get_affcpu();
        s.sprintf("%lx",p->affcpu);
        return s;
}
/*
int Cat_affcpu::compare(Procinfo *a, Procinfo *b)
{
	return (int)(b->affcpu - a->affcpu);
}
*/
Cat_time::Cat_time(const char *heading, const char *explain)
: Category(heading, explain)
{}

QString Cat_time::string(Procinfo *p)
{
	QString s;
	int ticks = p->utime;
	if(Qps::cumulative)
		ticks += p->cutime;
	int t = ticks / HZ;		// seconds
	if(t < 10) {
		int hundreds = ticks / (HZ / 100) % 100;
		s.sprintf("%1d.%02ds", t, hundreds);
	} else if(t < 100 * 60) {
		s.sprintf("%2d:%02d", t / 60, t % 60);
	} else if(t < 100 * 3600) {
		int h = t / 3600;
		t %= 3600;
		s.sprintf("%2d:%02dh", h, t / 60);
	} else {
		int d = t / 86400;
		t %= 86400;
		s.sprintf("%dd%dh", d, t / 3600);
	}
	return s;
}

int Cat_time::compare(Procinfo *a, Procinfo *b)
{
	int at = a->utime, bt = b->utime;
	if(Qps::cumulative) {
		at += a->cutime;
		bt += b->cutime;
	}
	return bt - at;
}

Cat_start::Cat_start(const char *heading, const char *explain)
: Category(heading, explain)
{}

QString Cat_start::string(Procinfo *p)
{
	time_t start = p->boot_time + p->starttime / (unsigned)HZ;
	QString s;
	char *ct = ctime(&start);
	if(p->tv.tv_sec - start < 86400) {
		ct[16] = '\0';
		s = ct + 11;
	} else {
		ct[10] = '\0';
		s = ct + 4;
	}
	return s;
}

int Cat_start::compare(Procinfo *a, Procinfo *b)
{
	unsigned long bs = b->starttime, as = a->starttime;
	return bs >= as ? (bs == as ? 0 : 1) : -1;
}

Cat_percent::Cat_percent(const char *heading, const char *explain, int w,
		float Procinfo::*member)
: Category(heading, explain), float_member(member), field_width(w)
{}

QString Cat_percent::string(Procinfo *p)
{
	QString s;
	s.sprintf("%01.2f", (double)(p->*float_member));
	return s;
}

int Cat_percent::compare(Procinfo *a, Procinfo *b)
{
	float at = a->*float_member, bt = b->*float_member;
	return at < bt ? 1 : (at > bt ? -1 : 0);
}

	Cat_tty::Cat_tty(const char *heading, const char *explain)
: Cat_string(heading, explain)
{}

QString Cat_tty::string(Procinfo *p)
{
	return Ttystr::name(p->tty);
}

Proc::Proc()
{
	char *command, *command_line;
	// Note: When adding/removing/changing the fields, the save file
	// version must be increased!
	allcats.set(F_PID, new Cat_int("PID", "Process ID", 6, &Procinfo::pid));
	allcats.set(F_TGID, new Cat_int("TGID","Task group ID",6, &Procinfo::tgid));
	allcats.set(F_PPID, new Cat_int("PPID", "Parent process ID", 6,	&Procinfo::ppid));
	allcats.set(F_PGID, new Cat_int("PGID", "Process group ID", 6,	&Procinfo::pgrp));
	allcats.set(F_SID, new Cat_int("SID", "Session ID", 6,	&Procinfo::session));
	allcats.set(F_TTY, new Cat_tty("TTY", "Terminal"));
	allcats.set(F_TPGID, new Cat_int("TPGID", "Process group ID of tty owner", 6, &Procinfo::tpgid));

	allcats.set(F_USER, new Cat_user("USER", "Owner (*=suid root, +=suid other user)"));
	allcats.set(F_GROUP, new Cat_group("GROUP", "Group name (*=sgid other)"));
	allcats.set(F_UID, new Cat_int("UID", "Real user ID", 6, &Procinfo::uid));
	allcats.set(F_EUID, new Cat_int("EUID", "Effective user ID", 6,	&Procinfo::euid));
	allcats.set(F_SUID, new Cat_int("SUID", "Saved user ID (Posix)", 6,&Procinfo::suid));
	allcats.set(F_FSUID, new Cat_int("FSUID", "File system user ID", 6,&Procinfo::fsuid));
	allcats.set(F_GID, new Cat_int("GID", "Real group ID", 6, &Procinfo::gid));
	allcats.set(F_EGID, new Cat_int("EGID", "Effective group ID", 6,&Procinfo::egid));
	allcats.set(F_SGID, new Cat_int("SGID", "Saved group ID (Posix)", 6,
				&Procinfo::sgid));
	allcats.set(F_FSGID, new Cat_int("FSGID", "File system group ID", 6,
				&Procinfo::fsgid));
	allcats.set(F_PRI, new Cat_int("PRI", "Dynamic priority", 4,
				&Procinfo::priority));
	allcats.set(F_NICE, new Cat_int("NICE",
				"Scheduling favour (higher -> less cpu time)",
				4, &Procinfo::nice));
	allcats.set(F_NLWP, new Cat_int("NLWP", "Number of tasks (threads) in task group (0 -> N/A)",
                                    5, &Procinfo::nthreads));

	allcats.set(F_PLCY, new Cat_policy("PLCY", "Scheduling policy"));
	allcats.set(F_RPRI, new Cat_rtprio("RPRI",
				"Realtime priority (0-99, more is better)"));
	allcats.set(F_TMS, new Cat_tms("TMS", "Time slice in milliseconds"));
	allcats.set(F_SLPAVG, new Cat_int("%SAVG", "Percentage average sleep time (-1 -> N/A)",4,
                                   &Procinfo::slpavg));
    	allcats.set(F_AFFCPU, new Cat_affcpu("CPUSET", "Affinity CPU mask (0 -> API not supported)"));
	allcats.set(F_MAJFLT, new Cat_uintl("MAJFLT",
				"Number of major faults (disk access)",
				8, &Procinfo::majflt));
	allcats.set(F_MINFLT, new Cat_uintl("MINFLT",
				"Number of minor faults (no disk access)",
				8, &Procinfo::minflt));
	allcats.set(F_TRS, new Cat_uintl("TRS", "Text resident set size",
				8, &Procinfo::trs));
	allcats.set(F_DRS, new Cat_uintl("DRS", "Data resident set size",
				8, &Procinfo::drs));
	allcats.set(F_SIZE, new Cat_uintl("SIZE",
				"Virtual image size of process",
				8, &Procinfo::size));
	allcats.set(F_SWAP, new Cat_swap("SWAP", "Kbytes on swap device"));
	allcats.set(F_RSS, new Cat_uintl("RSS",
				"Resident set size; program size in memory", 8, &Procinfo::resident));
	allcats.set(F_SHARE, new Cat_uintl("SHARE", "Shared memory",
				8, &Procinfo::share));
	allcats.set(F_DT, new Cat_uintl("DT",
				"Number of dirty (non-written) pages",
				7, &Procinfo::dt));
	allcats.set(F_STAT, new Cat_state("STAT", "State of the process"));
	allcats.set(F_FLAGS, new Cat_hex("FLAGS", "Process flags (hex)", 9,
				&Procinfo::flags));
	allcats.set(F_WCHAN, new Cat_wchan("WCHAN",
				"Kernel function where process is sleeping"));
	allcats.set(F_WCPU, new Cat_percent("%WCPU",
				"Weighted percentage of CPU (30 s average)",
				6, &Procinfo::wcpu));
	allcats.set(F_CPU, new Cat_percent("%CPU",
				"Percentage of CPU used since last update",
				6, &Procinfo::pcpu));
	allcats.set(F_MEM, new Cat_percent("%MEM",
				"Percentage of memory used (RSS/total mem)",
				6, &Procinfo::pmem));
	allcats.set(F_START, new Cat_start("START", "Time process started"));
	allcats.set(F_TIME, new Cat_time("TIME",
				"Total CPU time used since start"));
	allcats.set(F_CPUNUM, new Cat_int("CPU", "CPU the process is executing on",
				3, &Procinfo::which_cpu));

	command="COMMAND";	// label in linux source code --> /proc/1234/stat	
	allcats.set(F_COMM, new Cat_string(command,
				"Command that started the process",
				&Procinfo::command));
	allcats.set(F_CWD, new Cat_dir("CWD", "Current working directory",
				"cwd", &Procinfo::cwd));
	allcats.set(F_ROOT, new Cat_dir("ROOT", "Root directory of process",
				"root", &Procinfo::root));
	
	command_line="COMMAND_LINE";	//reference to /proc/1234/cmdline
	allcats.set(F_CMDLINE, new Cat_cmdline(command_line, "Command line that started the process"));

	for(int i = 0; i < allcats.size(); i++)
		allcats[i]->index = i;

	Procinfo::init_static();

	current_gen=0; // !
}

void Proc::newproc(Procinfo *p)
{
	Procinfo *oldp = procs[p->pid];
	if(oldp && flag_thread_ok && (previous_flag_show_thread != flag_show_thread) ){
		oldp->deref();
		oldp =NULL;
	}
	if(oldp) {
		// calculate pcpu (and wcpu for Linux) from previous procinfo
		int dt = (p->tv.tv_usec - oldp->tv.tv_usec) / (1000000 / HZ)
			+ (p->tv.tv_sec - oldp->tv.tv_sec) * HZ;
		int dcpu = p->utime - oldp->utime;
		p->pcpu = 100.0 * dcpu / dt;
		if(p->pcpu > 99.99) p->pcpu = 99.99;

		const float a = Procview::avg_factor;
		p->wcpu = a * oldp->wcpu + (1 - a) * p->pcpu;

		// propagate some fields to new incarnation
		p->selected = oldp->selected;
		p->details = oldp->details;
		p->hidekids = oldp->hidekids;
		oldp->details = 0;
		if(p->details)
			p->details->set_procinfo(p);

		if(Procinfo::num_cpus > 1) {
			// SMP: see which processor was used the most
			int best_cpu = -1;
			unsigned long most = 0;
			for(unsigned cpu = 0; cpu < Procinfo::num_cpus; cpu++) {
				unsigned long delta =
					p->per_cpu_times[cpu] - oldp->per_cpu_times[cpu];
				if(delta > most) {
					most = delta;
					best_cpu = cpu;
				}
				// if no cpu time has been spent, use previous value
				p->which_cpu = (best_cpu >= 0) ? best_cpu : oldp->which_cpu;
			}
		}
		oldp->deref();
	} else {
		// New process
		// %cpu first time = (cpu time since start) / (time since start)
		int jiffies_since_boot = p->tv.tv_usec / (1000000 / HZ)
			+ (p->tv.tv_sec - p->boot_time) * HZ;
		int dt = jiffies_since_boot - p->starttime;
		int dcpu = p->utime;
		p->pcpu = 100.0 * dcpu / dt;
		if(dt == 0 || p->pcpu > 99.99 || p->pcpu < 0)
			p->pcpu = 0.0;
		p->selected = FALSE;
		p->wcpu = p->pcpu;	// just a start

		if(Procinfo::num_cpus > 1) {
			// first tick: count times from 0
			unsigned long most = 0;
			for(unsigned cpu = 0; cpu < Procinfo::num_cpus; cpu++) {
				unsigned long t = p->per_cpu_times[cpu];
				if(t > most) {
					most = t;
					p->which_cpu = cpu;
				}
			}
		}
	}

	if(p->pcpu > 20) 
		p->test_stop=999;
	else p->test_stop=0;

	procs.replace(p->pid, p);
	if(procs.count() > procs.size())
		procs.resize(procs.count() * 2 - 1);
}

// BOTTLENECK !!
// Description: update the process list (fasthyun@magicn.com) 
//		read "/proc/*"
// 		be called every UPDATE time.
// 		called by Procview::refresh() 
void Proc::refresh()
{
	int pid;
	
	current_gen=!current_gen; // 0,1,0,1,0 ....  ??


	//init
	num_opened_files=0;
	Procinfo::num_process=0; 
	Procinfo::num_network_process=0; 
	Procinfo::read_common();
	//Procinfo::read_sockets();

	DIR *d = opendir(PROCDIR);
	struct dirent *e;

	while((e = readdir(d)) != 0) {
		if(e->d_name[0] >= '0' && e->d_name[0] <= '9') { // good idea !

			pid=atoi(e->d_name);
			Procinfo *pi = new Procinfo(pid);   // crazy solution ??
		
			if(flag_thread_ok && flag_show_thread)
				readTaskByPID(pid);

			if(pi->pid == -1)
				delete pi;	// already process gone(terminated)
			else {
				pi->generation = current_gen;
				newproc(pi);
				//pi->read_fds(); // too much cpu uses !!
				//if (pi->sock_inodes->size() != 0 ) 
				// Procinfo::num_network_process++;
			}
			
		}
	}
	
	closedir(d);
	if(flag_thread_ok)
		previous_flag_show_thread= flag_show_thread;

	// remove Procinfos of nonexisting processes
	for(QIntDictIterator<Procinfo> it(procs); it.current();) {
		Procinfo *p = it.current();
		if(p->generation != current_gen) {
			procs.remove(p->pid);
			p->deref();
		} else
			++it;
	}
}

Category *Proc::cat_by_name(const char *s)
{
	if(s) {
		for(int i = 0; i < allcats.size(); i++)
			if(strcmp(allcats[i]->name, s) == 0)
				return allcats[i];
	}
	return 0;
}

int  Proc::field_id_by_name(const char *s)
{
	if(s) {
		for(int i = 0; i < allcats.size(); i++)
			if(strcmp(allcats[i]->name, s) == 0)
				return i;
	}
	return -1;
}

// LINUX
int Procview::custom_fields[18] = {F_PID, F_TTY, F_USER, F_NICE,
	F_SIZE, F_RSS,
	F_STAT, F_CPU, F_START, F_TIME,
	F_CMDLINE, F_END};


int Procview::user_fields[] = {F_PID, F_TTY, F_USER, F_NICE,
	F_SIZE, F_RSS,
	F_STAT, F_CPU, F_START, F_TIME,
	F_CMDLINE, F_END};


int Procview::jobs_fields[] = {F_PID, F_TGID, F_PPID, F_PGID, F_SID, 
	F_TTY,F_TPGID, F_STAT, F_UID, F_TIME, F_CMDLINE, F_END};

int Procview::mem_fields[] = {F_PID, 
	F_TTY, F_MAJFLT, F_MINFLT,F_TRS, F_DRS,
	F_SIZE, F_SWAP, F_RSS,F_SHARE, F_DT,
	F_CMDLINE,F_END};

int Procview::sched_fields[] = {F_PID,F_TGID,F_NLWP,F_STAT,F_FLAGS,F_PLCY,
	F_PRI,F_NICE,F_TMS,F_SLPAVG,F_RPRI,F_AFFCPU,F_CPU,F_START,F_TIME,
	F_CMDLINE,F_END};

float Procview::avg_factor = 1.0;

Procview::Procview(Proc *p) : proc(p)
{
	sortcat = p->allcats[F_WCPU];
	reversed = FALSE;
	viewproc = ALL;
	viewfields = USER;
	treeview = TRUE;	// init_mode (by fasthyun)
	set_fields();
}


/*
build_tree_24 is ok for 2.4 UP and SMP
i don't know why  the new build_tree is ok for 2.4 UP, 2.6 UP and SMP
but is not ok for 2.4 SMP
*/
void Procview::build_tree_24()
{
    if(root_procs.size() > 0) {
        Procinfo *p;
        for(QIntDictIterator<Procinfo> it(proc->procs); (p = it.current()); ++it)
            if(p->children)
                p->children->clear();
        root_procs.clear();
    }
    Procinfo *p;
    for(QIntDictIterator<Procinfo> it(proc->procs); (p = it.current()); ++it) {
        if(accept_proc(p)) {
            Procinfo *parent = 0;
            if(p->ppid && (parent = proc->procs[p->ppid])
               && accept_proc(parent)) {
                if(!parent->children)
                    parent->children = new Svec<Procinfo *>(4);
                parent->children->add(p);
            } else
                root_procs.add(p);
        } else
            p->selected = FALSE;
    }
}



// 	be called by Procview::rebuild()
// 	be called every UPDATE time.
void Procview::build_tree()
{
	// printf("DEBUG: build_tree()\n");
	if(root_procs.size() > 0) {
		Procinfo *p;
		for(QIntDictIterator<Procinfo> it(proc->procs); (p = it.current()); ++it)
			if(p->children)
				p->children->clear();
		root_procs.clear();
	}
	Procinfo *p;
	for(QIntDictIterator<Procinfo> it(proc->procs); (p = it.current()); ++it) {
		if(accept_proc(p)) 
		{
			Procinfo *parent = 0;

			if(p->tgid==p->pid) // this means ,the process has no thread!!
			{

				if(p->ppid && (parent = proc->procs[p->ppid])
						&& accept_proc(parent)) {
					if(!parent->children)
						parent->children = new Svec<Procinfo *>(4);
					parent->children->add(p);
				} 
				else						
				{
					//only for init(1) process !!
					//printf("DEBUG: ppid=0  pid=%d\n",p->pid);
					root_procs.add(p);
				}
			}
			else 
			{
				Procinfo *thread_leader = 0;
				thread_leader = proc->procs[p->tgid];
				
				if(accept_proc(thread_leader)) 
				{
					if(!thread_leader->children)
						thread_leader->children = new Svec<Procinfo *>(4);
					thread_leader->children->add(p);
				}
			}


		} 
		else	p->selected = FALSE;

	}
}

//Linux: re-sort the process info
void Procview::rebuild()
{
	for(int i = 0; i < procs.size(); i++)
		procs[i]->deref(); 	// delete procs 

	procs.clear();
	if(treeview) {
		if (flag_24_ok)
			build_tree_24();
		else
			build_tree();
		parent_rows.clear();
		linearize_tree(&root_procs, 0, -1);
	} else {
		for(QIntDictIterator<Procinfo> it(proc->procs); it.current(); ++it) {
			Procinfo *p = it.current();
			if(accept_proc(p))
				procs.add(p->ref());
			else
				p->selected = FALSE;
		}
		static_sortcat = sortcat;
		procs.sort(reversed ? compare_backwards : compare);
	}
}

void Procview::set_fields()
{
	switch(viewfields) {
		case USER:
			set_fields_list(user_fields);
			break;
		case JOBS:
			set_fields_list(jobs_fields);
			break;
		case MEM:
			set_fields_list(mem_fields);
			break;
		case SCHED:
			set_fields_list(sched_fields);
			break;
		case CUSTOM:
			set_fields_list(custom_fields);
			break;
	}
}

/*
int Procview::findPID(int pid)
{
	for(int i = 0; i < proc->size(); i++);
		if ( proc[i].pid== pid) 
			return i;		
//		if(cats[i] == proc->allcats[field])
//			return i;
	return -1;
}

*/
// LINUX:
// deduce whether the currently selected fields correspond to a field list
void Procview::deduce_fields()
{
	return;

	if(viewfields != CUSTOM)
		return;

	Procview::fieldstates tags[4] = {USER, JOBS, MEM,SCHED};
	int *lists[4] = {user_fields, jobs_fields, mem_fields,sched_fields};
	for(int i = 0; i < 4; i++) {
		int *l = lists[i];
		int j;
		for(j = 0; l[j] != F_END; j++)
			if(findCol(l[j]) < 0)
				break;
		if(l[j] == F_END && j == cats.size()) {
			viewfields = tags[i];
			return;
		}
	}
}


